#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
/*
 * FileConverter.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbbackup/FileConverter.h"

#include <algorithm>
#include <iostream>
#include <cinttypes>
#include <cstdio>
#include <vector>

#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/BackupContainer.h"
#include "fdbclient/MutationList.h"
#include "flow/flow.h"
#include "flow/serialize.h"
#include "fdbclient/BuildFlags.h"
#include "flow/actorcompiler.h" // has to be last include

namespace file_converter {

void printConvertUsage() {
	std::cout << "\n"
	          << "  -r, --container Container URL.\n"
	          << "  -b, --begin BEGIN\n"
	          << "                  Begin version.\n"
	          << "  -e, --end END   End version.\n"
	          << "  --log           Enables trace file logging for the CLI session.\n"
	          << "  --logdir PATH   Specifies the output directory for trace files. If\n"
	          << "                  unspecified, defaults to the current directory. Has\n"
	          << "                  no effect unless --log is specified.\n"
	          << "  --loggroup LOG_GROUP\n"
	          << "                  Sets the LogGroup field with the specified value for all\n"
	          << "                  events in the trace output (defaults to `default').\n"
	          << "  --trace-format FORMAT\n"
	          << "                  Select the format of the trace files. xml (the default) and json are supported.\n"
	          << "                  Has no effect unless --log is specified.\n"
	          << "  --build-flags   Print build information and exit.\n"
	          << "  -h, --help      Display this help and exit.\n"
	          << "\n";

	return;
}

void printBuildInformation() {
	printf("%s", jsonBuildInformation().c_str());
}

void printLogFiles(std::string msg, const std::vector<LogFile>& files) {
	std::cout << msg << " " << files.size() << " log files\n";
	for (const auto& file : files) {
		std::cout << file.toString() << "\n";
	}
	std::cout << std::endl;
}

std::vector<LogFile> getRelevantLogFiles(const std::vector<LogFile>& files, Version begin, Version end) {
	std::vector<LogFile> filtered;
	for (const auto& file : files) {
		if (file.beginVersion <= end && file.endVersion >= begin && file.tagId >= 0 && file.fileSize > 0) {
			filtered.push_back(file);
		}
	}
	std::sort(filtered.begin(), filtered.end());

	// Remove duplicates. This is because backup workers may store the log for
	// old epochs successfully, but do not update the progress before another
	// recovery happened. As a result, next epoch will retry and creates
	// duplicated log files.
	std::vector<LogFile> sorted;
	int i = 0;
	for (int j = 1; j < filtered.size(); j++) {
		if (!filtered[i].isSubset(filtered[j])) {
			sorted.push_back(filtered[i]);
		}
		i = j;
	}
	if (i < filtered.size()) {
		sorted.push_back(filtered[i]);
	}

	return sorted;
}

struct ConvertParams {
	std::string container_url;
	Optional<std::string> proxy;
	Version begin = invalidVersion;
	Version end = invalidVersion;
	bool log_enabled = false;
	std::string log_dir, trace_format, trace_log_group;

	bool isValid() const { return begin != invalidVersion && end != invalidVersion && !container_url.empty(); }

	std::string toString() const {
		std::string s;
		s.append("ContainerURL:");
		s.append(container_url);
		if (proxy.present()) {
			s.append(" Proxy:");
			s.append(proxy.get());
		}
		s.append(" Begin:");
		s.append(format("%" PRId64, begin));
		s.append(" End:");
		s.append(format("%" PRId64, end));
		if (log_enabled) {
			if (!log_dir.empty()) {
				s.append(" LogDir:").append(log_dir);
			}
			if (!trace_format.empty()) {
				s.append(" Format:").append(trace_format);
			}
			if (!trace_log_group.empty()) {
				s.append(" LogGroup:").append(trace_log_group);
			}
		}
		return s;
	}
};

struct VersionedData {
	LogMessageVersion version;
	StringRef message; // Serialized mutation.
	Arena arena; // The arena that contains mutation.

	VersionedData() : version(invalidVersion, -1) {}
	VersionedData(LogMessageVersion v, StringRef m, Arena a) : version(v), message(m), arena(a) {}
};

struct MutationFilesReadProgress : public ReferenceCounted<MutationFilesReadProgress> {
	MutationFilesReadProgress(std::vector<LogFile>& logs, Version begin, Version end)
	  : files(logs), beginVersion(begin), endVersion(end) {}

	struct FileProgress : public ReferenceCounted<FileProgress> {
		FileProgress(Reference<IAsyncFile> f, int index) : fd(f), idx(index), offset(0), eof(false) {}

		bool operator<(const FileProgress& rhs) const {
			if (rhs.mutations.empty())
				return true;
			if (mutations.empty())
				return false;
			return mutations[0].version < rhs.mutations[0].version;
		}
		bool operator<=(const FileProgress& rhs) const {
			if (rhs.mutations.empty())
				return true;
			if (mutations.empty())
				return false;
			return mutations[0].version <= rhs.mutations[0].version;
		}
		bool empty() { return eof && mutations.empty(); }

		// Decodes the block into mutations and save them if >= minVersion and < maxVersion.
		// Returns true if new mutations has been saved.
		bool decodeBlock(const Standalone<StringRef>& buf, int len, Version minVersion, Version maxVersion) {
			StringRef block(buf.begin(), len);
			StringRefReader reader(block, restore_corrupted_data());
			int count = 0, inserted = 0;
			Version msgVersion = invalidVersion;

			try {
				// Read block header
				if (reader.consume<int32_t>() != PARTITIONED_MLOG_VERSION)
					throw restore_unsupported_file_version();

				while (1) {
					// If eof reached or first key len bytes is 0xFF then end of block was reached.
					if (reader.eof() || *reader.rptr == 0xFF)
						break;

					// Deserialize messages written in saveMutationsToFile().
					msgVersion = bigEndian64(reader.consume<Version>());
					uint32_t sub = bigEndian32(reader.consume<uint32_t>());
					int msgSize = bigEndian32(reader.consume<int>());
					const uint8_t* message = reader.consume(msgSize);

					ArenaReader rd(
					    buf.arena(), StringRef(message, msgSize), AssumeVersion(g_network->protocolVersion()));
					MutationRef m;
					rd >> m;
					count++;
					if (msgVersion >= maxVersion) {
						TraceEvent("FileDecodeEnd")
						    .detail("MaxV", maxVersion)
						    .detail("Version", msgVersion)
						    .detail("File", fd->getFilename());
						eof = true;
						break; // skip
					}
					if (msgVersion >= minVersion) {
						mutations.emplace_back(
						    LogMessageVersion(msgVersion, sub), StringRef(message, msgSize), buf.arena());
						inserted++;
					}
				}
				offset += len;

				TraceEvent("Decoded")
				    .detail("Name", fd->getFilename())
				    .detail("Count", count)
				    .detail("Insert", inserted)
				    .detail("BlockOffset", reader.rptr - buf.begin())
				    .detail("Total", mutations.size())
				    .detail("EOF", eof)
				    .detail("Version", msgVersion)
				    .detail("NewOffset", offset);
				return inserted > 0;
			} catch (Error& e) {
				TraceEvent(SevWarn, "CorruptLogFileBlock")
				    .error(e)
				    .detail("Filename", fd->getFilename())
				    .detail("BlockOffset", offset)
				    .detail("BlockLen", len)
				    .detail("ErrorRelativeOffset", reader.rptr - buf.begin())
				    .detail("ErrorAbsoluteOffset", reader.rptr - buf.begin() + offset);
				throw;
			}
		}

		Reference<IAsyncFile> fd;
		int idx; // index in the MutationFilesReadProgress::files vector
		int64_t offset; // offset of the file to be read
		bool eof; // If EOF is seen so far or endVersion is encountered. If true, the file can't be read further.
		std::vector<VersionedData> mutations; // Buffered mutations read so far
	};

	bool hasMutations() {
		for (const auto& fp : fileProgress) {
			if (!fp->empty())
				return true;
		}
		return false;
	}

	void dumpProgress(std::string msg) {
		std::cout << msg << "\n  ";
		for (const auto& fp : fileProgress) {
			std::cout << fp->fd->getFilename() << " " << fp->mutations.size() << " mutations";
			if (fp->mutations.size() > 0) {
				std::cout << ", range " << fp->mutations[0].version.toString() << " "
				          << fp->mutations.back().version.toString() << "\n";
			} else {
				std::cout << "\n\n";
			}
		}
	}

	// Sorts files according to their first mutation version and removes files without mutations.
	void sortAndRemoveEmpty() {
		std::sort(fileProgress.begin(),
		          fileProgress.end(),
		          [](const Reference<FileProgress>& a, const Reference<FileProgress>& b) { return (*a) < (*b); });
		while (!fileProgress.empty() && fileProgress.back()->empty()) {
			fileProgress.pop_back();
		}
	}

	// Requires hasMutations() return true before calling this function.
	// The caller must hold on the the arena associated with the mutation.
	Future<VersionedData> getNextMutation() { return getMutationImpl(this); }

																#line 282 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via getMutationImpl()
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class GetMutationImplActor>
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class GetMutationImplActorState {
															#line 288 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	GetMutationImplActorState(MutationFilesReadProgress* const& self) 
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : self(self)
															#line 295 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("getMutationImpl", reinterpret_cast<unsigned long>(this));

	}
	~GetMutationImplActorState() 
	{
		fdb_probe_actor_destroy("getMutationImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 281 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			ASSERT(!self->fileProgress.empty() && !self->fileProgress[0]->mutations.empty());
															#line 283 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			fp = self->fileProgress[0];
															#line 284 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			data = fp->mutations[0];
															#line 285 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			fp->mutations.erase(fp->mutations.begin());
															#line 286 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (fp->mutations.empty())
															#line 318 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			{
															#line 288 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				StrictFuture<Void> __when_expr_0 = decodeToVersion(fp, 0, self->endVersion, self->getLogFile(fp->idx));
															#line 288 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				if (static_cast<GetMutationImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 324 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetMutationImplActor*>(this)->actor_wait_state = 1;
															#line 288 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetMutationImplActor, 0, Void >*>(static_cast<GetMutationImplActor*>(this)));
															#line 329 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetMutationImplActorState();
		static_cast<GetMutationImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 291 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (fp->empty())
															#line 357 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 292 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			self->fileProgress.erase(self->fileProgress.begin());
															#line 361 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
		else
		{
															#line 295 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			for(int i = 1;i < self->fileProgress.size();i++) {
															#line 296 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				if (*self->fileProgress[i - 1] <= *self->fileProgress[i])
															#line 369 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				{
					break;
				}
															#line 299 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				std::swap(self->fileProgress[i - 1], self->fileProgress[i]);
															#line 375 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			}
		}
															#line 302 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<GetMutationImplActor*>(this)->SAV<VersionedData>::futures) { (void)(data); this->~GetMutationImplActorState(); static_cast<GetMutationImplActor*>(this)->destroy(); return 0; }
															#line 380 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<GetMutationImplActor*>(this)->SAV< VersionedData >::value()) VersionedData(std::move(data)); // state_var_RVO
		this->~GetMutationImplActorState();
		static_cast<GetMutationImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetMutationImplActor*>(this)->actor_wait_state > 0) static_cast<GetMutationImplActor*>(this)->actor_wait_state = 0;
		static_cast<GetMutationImplActor*>(this)->ActorCallback< GetMutationImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetMutationImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetMutationImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetMutationImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetMutationImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetMutationImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetMutationImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getMutationImpl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	MutationFilesReadProgress* self;
															#line 283 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<FileProgress> fp;
															#line 284 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	VersionedData data;
															#line 481 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via getMutationImpl()
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class GetMutationImplActor final : public Actor<VersionedData>, public ActorCallback< GetMutationImplActor, 0, Void >, public FastAllocated<GetMutationImplActor>, public GetMutationImplActorState<GetMutationImplActor> {
															#line 486 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<GetMutationImplActor>::operator new;
	using FastAllocated<GetMutationImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13422560246627112448UL, 14816087244873174016UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<VersionedData>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetMutationImplActor, 0, Void >;
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	GetMutationImplActor(MutationFilesReadProgress* const& self) 
															#line 503 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<VersionedData>(),
		   GetMutationImplActorState<GetMutationImplActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getMutationImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17176354734787432192UL, 10428661037991047424UL);
		ActorExecutionContextHelper __helper(static_cast<GetMutationImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getMutationImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getMutationImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetMutationImplActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] static Future<VersionedData> getMutationImpl( MutationFilesReadProgress* const& self ) {
															#line 280 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<VersionedData>(new GetMutationImplActor(self));
															#line 535 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 304 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

	LogFile& getLogFile(int index) { return files[index]; }

	Future<Void> openLogFiles(Reference<IBackupContainer> container) { return openLogFilesImpl(this, container); }

	// Opens log files in the progress and starts decoding until the beginVersion is seen.
																#line 545 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via openLogFilesImpl()
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class OpenLogFilesImplActor>
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class OpenLogFilesImplActorState {
															#line 551 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	OpenLogFilesImplActorState(MutationFilesReadProgress* const& progress,Reference<IBackupContainer> const& container) 
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : progress(progress),
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   container(container),
															#line 312 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   asyncFiles()
															#line 562 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("openLogFilesImpl", reinterpret_cast<unsigned long>(this));

	}
	~OpenLogFilesImplActorState() 
	{
		fdb_probe_actor_destroy("openLogFilesImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 313 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			for( const auto& file : progress->files ) {
															#line 314 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				asyncFiles.push_back(container->readFile(file.fileName));
															#line 579 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			}
															#line 316 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(asyncFiles);
															#line 316 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 585 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state = 1;
															#line 316 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< OpenLogFilesImplActor, 0, Void >*>(static_cast<OpenLogFilesImplActor*>(this)));
															#line 590 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~OpenLogFilesImplActorState();
		static_cast<OpenLogFilesImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 319 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::vector<Future<Void>> fileDecodes;
															#line 320 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		for(int i = 0;i < asyncFiles.size();i++) {
															#line 321 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			auto fp = makeReference<FileProgress>(asyncFiles[i].get(), i);
															#line 322 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			progress->fileProgress.push_back(fp);
															#line 323 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			fileDecodes.push_back( decodeToVersion(fp, progress->beginVersion, progress->endVersion, progress->getLogFile(i)));
															#line 621 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(fileDecodes);
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 627 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state = 2;
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenLogFilesImplActor, 1, Void >*>(static_cast<OpenLogFilesImplActor*>(this)));
															#line 632 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 319 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::vector<Future<Void>> fileDecodes;
															#line 320 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		for(int i = 0;i < asyncFiles.size();i++) {
															#line 321 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			auto fp = makeReference<FileProgress>(asyncFiles[i].get(), i);
															#line 322 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			progress->fileProgress.push_back(fp);
															#line 323 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			fileDecodes.push_back( decodeToVersion(fp, progress->beginVersion, progress->endVersion, progress->getLogFile(i)));
															#line 649 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(fileDecodes);
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 655 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state = 2;
															#line 327 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenLogFilesImplActor, 1, Void >*>(static_cast<OpenLogFilesImplActor*>(this)));
															#line 660 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state > 0) static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state = 0;
		static_cast<OpenLogFilesImplActor*>(this)->ActorCallback< OpenLogFilesImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< OpenLogFilesImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< OpenLogFilesImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< OpenLogFilesImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 329 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		progress->sortAndRemoveEmpty();
															#line 331 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<OpenLogFilesImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenLogFilesImplActorState(); static_cast<OpenLogFilesImplActor*>(this)->destroy(); return 0; }
															#line 746 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<OpenLogFilesImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenLogFilesImplActorState();
		static_cast<OpenLogFilesImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 329 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		progress->sortAndRemoveEmpty();
															#line 331 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<OpenLogFilesImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenLogFilesImplActorState(); static_cast<OpenLogFilesImplActor*>(this)->destroy(); return 0; }
															#line 760 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<OpenLogFilesImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenLogFilesImplActorState();
		static_cast<OpenLogFilesImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state > 0) static_cast<OpenLogFilesImplActor*>(this)->actor_wait_state = 0;
		static_cast<OpenLogFilesImplActor*>(this)->ActorCallback< OpenLogFilesImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< OpenLogFilesImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< OpenLogFilesImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< OpenLogFilesImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	MutationFilesReadProgress* progress;
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<IBackupContainer> container;
															#line 312 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	std::vector<Future<Reference<IAsyncFile>>> asyncFiles;
															#line 849 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via openLogFilesImpl()
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class OpenLogFilesImplActor final : public Actor<Void>, public ActorCallback< OpenLogFilesImplActor, 0, Void >, public ActorCallback< OpenLogFilesImplActor, 1, Void >, public FastAllocated<OpenLogFilesImplActor>, public OpenLogFilesImplActorState<OpenLogFilesImplActor> {
															#line 854 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<OpenLogFilesImplActor>::operator new;
	using FastAllocated<OpenLogFilesImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11471699062836317184UL, 5263880527726332928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< OpenLogFilesImplActor, 0, Void >;
friend struct ActorCallback< OpenLogFilesImplActor, 1, Void >;
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	OpenLogFilesImplActor(MutationFilesReadProgress* const& progress,Reference<IBackupContainer> const& container) 
															#line 872 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<Void>(),
		   OpenLogFilesImplActorState<OpenLogFilesImplActor>(progress, container),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("openLogFilesImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3468072685173303552UL, 17012317721691669760UL);
		ActorExecutionContextHelper __helper(static_cast<OpenLogFilesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("openLogFilesImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("openLogFilesImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< OpenLogFilesImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< OpenLogFilesImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] static Future<Void> openLogFilesImpl( MutationFilesReadProgress* const& progress, Reference<IBackupContainer> const& container ) {
															#line 310 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<Void>(new OpenLogFilesImplActor(progress, container));
															#line 905 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 333 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

	// Decodes the file until EOF or an mutation >= minVersion and saves these mutations.
	// Skip mutations >= maxVersion.
																#line 912 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via decodeToVersion()
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class DecodeToVersionActor>
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class DecodeToVersionActorState {
															#line 918 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	DecodeToVersionActorState(Reference<FileProgress> const& fp,Version const& minVersion,Version const& maxVersion,LogFile const& file) 
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : fp(fp),
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   minVersion(minVersion),
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   maxVersion(maxVersion),
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   file(file)
															#line 931 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("decodeToVersion", reinterpret_cast<unsigned long>(this));

	}
	~DecodeToVersionActorState() 
	{
		fdb_probe_actor_destroy("decodeToVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 340 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (fp->empty())
															#line 946 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			{
															#line 341 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				if (!static_cast<DecodeToVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeToVersionActorState(); static_cast<DecodeToVersionActor*>(this)->destroy(); return 0; }
															#line 950 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				new (&static_cast<DecodeToVersionActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DecodeToVersionActorState();
				static_cast<DecodeToVersionActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 343 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (!fp->mutations.empty() && fp->mutations.back().version.version >= minVersion)
															#line 958 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			{
															#line 344 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				if (!static_cast<DecodeToVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeToVersionActorState(); static_cast<DecodeToVersionActor*>(this)->destroy(); return 0; }
															#line 962 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				new (&static_cast<DecodeToVersionActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DecodeToVersionActorState();
				static_cast<DecodeToVersionActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 346 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			len = int64_t();
															#line 970 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			try {
															#line 349 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				;
															#line 974 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DecodeToVersionActorState();
		static_cast<DecodeToVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 370 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			TraceEvent(SevWarn, "CorruptedLogFileBlock") .error(e) .detail("Filename", fp->fd->getFilename()) .detail("BlockOffset", fp->offset) .detail("BlockLen", len);
															#line 375 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1006 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 368 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<DecodeToVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeToVersionActorState(); static_cast<DecodeToVersionActor*>(this)->destroy(); return 0; }
															#line 1020 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<DecodeToVersionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DecodeToVersionActorState();
		static_cast<DecodeToVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 350 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		len = std::min<int64_t>(file.blockSize, file.fileSize - fp->offset);
															#line 351 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (len == 0)
															#line 1041 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 352 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			fp->eof = true;
															#line 353 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (!static_cast<DecodeToVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeToVersionActorState(); static_cast<DecodeToVersionActor*>(this)->destroy(); return 0; }
															#line 1047 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			new (&static_cast<DecodeToVersionActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DecodeToVersionActorState();
			static_cast<DecodeToVersionActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 356 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		buf = makeString(len);
															#line 357 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<int> __when_expr_0 = fp->fd->read(mutateString(buf), len, fp->offset);
															#line 357 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<DecodeToVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1059 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<DecodeToVersionActor*>(this)->actor_wait_state = 1;
															#line 357 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DecodeToVersionActor, 0, int >*>(static_cast<DecodeToVersionActor*>(this)));
															#line 1064 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int const& rLen,int loopDepth) 
	{
															#line 358 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (len != rLen)
															#line 1086 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			return a_body1Catch2(restore_bad_read(), std::max(0, loopDepth - 1));
															#line 1090 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 361 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		TraceEvent("ReadFile") .detail("Name", fp->fd->getFilename()) .detail("Length", rLen) .detail("Offset", fp->offset);
															#line 365 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (fp->decodeBlock(buf, rLen, minVersion, maxVersion))
															#line 1096 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int && rLen,int loopDepth) 
	{
															#line 358 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (len != rLen)
															#line 1108 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			return a_body1Catch2(restore_bad_read(), std::max(0, loopDepth - 1));
															#line 1112 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 361 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		TraceEvent("ReadFile") .detail("Name", fp->fd->getFilename()) .detail("Length", rLen) .detail("Offset", fp->offset);
															#line 365 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (fp->decodeBlock(buf, rLen, minVersion, maxVersion))
															#line 1118 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(int const& rLen,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(rLen, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(int && rLen,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(rLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DecodeToVersionActor*>(this)->actor_wait_state > 0) static_cast<DecodeToVersionActor*>(this)->actor_wait_state = 0;
		static_cast<DecodeToVersionActor*>(this)->ActorCallback< DecodeToVersionActor, 0, int >::remove();

	}
	void a_callback_fire(ActorCallback< DecodeToVersionActor, 0, int >*,int const& value) 
	{
		fdb_probe_actor_enter("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeToVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DecodeToVersionActor, 0, int >*,int && value) 
	{
		fdb_probe_actor_enter("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeToVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DecodeToVersionActor, 0, int >*,Error err) 
	{
		fdb_probe_actor_enter("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeToVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeToVersion", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<FileProgress> fp;
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Version minVersion;
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Version maxVersion;
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	LogFile file;
															#line 346 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	int64_t len;
															#line 356 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Standalone<StringRef> buf;
															#line 1213 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via decodeToVersion()
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class DecodeToVersionActor final : public Actor<Void>, public ActorCallback< DecodeToVersionActor, 0, int >, public FastAllocated<DecodeToVersionActor>, public DecodeToVersionActorState<DecodeToVersionActor> {
															#line 1218 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<DecodeToVersionActor>::operator new;
	using FastAllocated<DecodeToVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16634554086298038784UL, 8501240051785447168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DecodeToVersionActor, 0, int >;
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	DecodeToVersionActor(Reference<FileProgress> const& fp,Version const& minVersion,Version const& maxVersion,LogFile const& file) 
															#line 1235 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<Void>(),
		   DecodeToVersionActorState<DecodeToVersionActor>(fp, minVersion, maxVersion, file),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("decodeToVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7632135494026246912UL, 16881309757970064640UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeToVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("decodeToVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("decodeToVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DecodeToVersionActor, 0, int >*)0, actor_cancelled()); break;
		}

	}
};
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] static Future<Void> decodeToVersion( Reference<FileProgress> const& fp, Version const& minVersion, Version const& maxVersion, LogFile const& file ) {
															#line 336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<Void>(new DecodeToVersionActor(fp, minVersion, maxVersion, file));
															#line 1267 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 378 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

	std::vector<LogFile> files;
	const Version beginVersion, endVersion;
	std::vector<Reference<FileProgress>> fileProgress;
};

// Writes a log file in the old backup format, described in backup-dataFormat.md.
// This is similar to the LogFileWriter in FileBackupAgent.actor.cpp.
struct LogFileWriter {
	LogFileWriter() : blockSize(-1) {}
	LogFileWriter(Reference<IBackupFile> f, int bsize) : file(f), blockSize(bsize) {}

	// Returns the block key, i.e., `Param1`, in the back file. The format is
	// `hash_value|commitVersion|part`.
	static Standalone<StringRef> getBlockKey(Version commitVersion, int part) {
		const int32_t version = commitVersion / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;

		BinaryWriter wr(Unversioned());
		wr << (uint8_t)hashlittle(&version, sizeof(version), 0);
		wr << bigEndian64(commitVersion);
		wr << bigEndian32(part);
		return wr.toValue();
	}

	// Start a new block if needed, then write the key and value
																#line 1296 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via writeKV_impl()
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class WriteKV_implActor>
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class WriteKV_implActorState {
															#line 1302 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	WriteKV_implActorState(LogFileWriter* const& self,Key const& k,Value const& v) 
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : self(self),
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   k(k),
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   v(v)
															#line 1313 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("writeKV_impl", reinterpret_cast<unsigned long>(this));

	}
	~WriteKV_implActorState() 
	{
		fdb_probe_actor_destroy("writeKV_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 405 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			int toWrite = sizeof(int32_t) + k.size() + sizeof(int32_t) + v.size();
															#line 406 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (self->file->size() + toWrite > self->blockEnd)
															#line 1330 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			{
															#line 408 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				int bytesLeft = self->blockEnd - self->file->size();
															#line 409 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
				if (bytesLeft > 0)
															#line 1336 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
				{
															#line 410 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
					paddingFFs = fileBackup::makePadding(bytesLeft);
															#line 411 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
					StrictFuture<Void> __when_expr_0 = self->file->append(paddingFFs.begin(), bytesLeft);
															#line 411 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
					if (static_cast<WriteKV_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1344 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<WriteKV_implActor*>(this)->actor_wait_state = 1;
															#line 411 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteKV_implActor, 0, Void >*>(static_cast<WriteKV_implActor*>(this)));
															#line 1349 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteKV_implActorState();
		static_cast<WriteKV_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 421 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->file->appendStringRefWithLen(k);
															#line 421 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1384 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteKV_implActor*>(this)->actor_wait_state = 3;
															#line 421 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteKV_implActor, 2, Void >*>(static_cast<WriteKV_implActor*>(this)));
															#line 1389 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 415 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		self->blockEnd += self->blockSize;
															#line 418 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->file->append((uint8_t*)&BACKUP_AGENT_MLOG_VERSION, sizeof(BACKUP_AGENT_MLOG_VERSION));
															#line 418 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1402 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteKV_implActor*>(this)->actor_wait_state = 2;
															#line 418 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteKV_implActor, 1, Void >*>(static_cast<WriteKV_implActor*>(this)));
															#line 1407 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state > 0) static_cast<WriteKV_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteKV_implActor*>(this)->ActorCallback< WriteKV_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteKV_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state > 0) static_cast<WriteKV_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteKV_implActor*>(this)->ActorCallback< WriteKV_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteKV_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->file->appendStringRefWithLen(v);
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1592 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteKV_implActor*>(this)->actor_wait_state = 4;
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteKV_implActor, 3, Void >*>(static_cast<WriteKV_implActor*>(this)));
															#line 1597 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->file->appendStringRefWithLen(v);
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1608 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteKV_implActor*>(this)->actor_wait_state = 4;
															#line 422 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteKV_implActor, 3, Void >*>(static_cast<WriteKV_implActor*>(this)));
															#line 1613 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state > 0) static_cast<WriteKV_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteKV_implActor*>(this)->ActorCallback< WriteKV_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteKV_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 425 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (self->file->size() > self->blockEnd)
															#line 1697 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 426 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			return a_body1Catch1(backup_bad_block_size(), loopDepth);
															#line 1701 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 428 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<WriteKV_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteKV_implActorState(); static_cast<WriteKV_implActor*>(this)->destroy(); return 0; }
															#line 1705 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<WriteKV_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteKV_implActorState();
		static_cast<WriteKV_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 425 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (self->file->size() > self->blockEnd)
															#line 1717 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 426 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			return a_body1Catch1(backup_bad_block_size(), loopDepth);
															#line 1721 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		}
															#line 428 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<WriteKV_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteKV_implActorState(); static_cast<WriteKV_implActor*>(this)->destroy(); return 0; }
															#line 1725 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<WriteKV_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteKV_implActorState();
		static_cast<WriteKV_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteKV_implActor*>(this)->actor_wait_state > 0) static_cast<WriteKV_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteKV_implActor*>(this)->ActorCallback< WriteKV_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteKV_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteKV_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	LogFileWriter* self;
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Key k;
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Value v;
															#line 410 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Value paddingFFs;
															#line 1816 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via writeKV_impl()
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class WriteKV_implActor final : public Actor<Void>, public ActorCallback< WriteKV_implActor, 0, Void >, public ActorCallback< WriteKV_implActor, 1, Void >, public ActorCallback< WriteKV_implActor, 2, Void >, public ActorCallback< WriteKV_implActor, 3, Void >, public FastAllocated<WriteKV_implActor>, public WriteKV_implActorState<WriteKV_implActor> {
															#line 1821 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<WriteKV_implActor>::operator new;
	using FastAllocated<WriteKV_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6694517222705378816UL, 6813060086912088832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteKV_implActor, 0, Void >;
friend struct ActorCallback< WriteKV_implActor, 1, Void >;
friend struct ActorCallback< WriteKV_implActor, 2, Void >;
friend struct ActorCallback< WriteKV_implActor, 3, Void >;
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	WriteKV_implActor(LogFileWriter* const& self,Key const& k,Value const& v) 
															#line 1841 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<Void>(),
		   WriteKV_implActorState<WriteKV_implActor>(self, k, v),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeKV_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12331643623674153728UL, 17746951694630343424UL);
		ActorExecutionContextHelper __helper(static_cast<WriteKV_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeKV_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeKV_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteKV_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteKV_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteKV_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteKV_implActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] static Future<Void> writeKV_impl( LogFileWriter* const& self, Key const& k, Value const& v ) {
															#line 403 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<Void>(new WriteKV_implActor(self, k, v));
															#line 1876 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 430 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

	Future<Void> writeKV(Key k, Value v) { return writeKV_impl(this, k, v); }

	// Adds a new mutation to an internal buffer and writes out when encountering
	// a new commitVersion or exceeding the block size.
																#line 1885 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via addMutation()
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class AddMutationActor>
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class AddMutationActorState {
															#line 1891 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	AddMutationActorState(LogFileWriter* const& self,Version const& commitVersion,MutationListRef const& mutations) 
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : self(self),
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   commitVersion(commitVersion),
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   mutations(mutations),
															#line 436 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   value(BinaryWriter::toValue(mutations, IncludeVersion())),
															#line 438 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   part(0)
															#line 1906 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("addMutation", reinterpret_cast<unsigned long>(this));

	}
	~AddMutationActorState() 
	{
		fdb_probe_actor_destroy("addMutation", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 439 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			;
															#line 1921 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AddMutationActorState();
		static_cast<AddMutationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 446 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<AddMutationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AddMutationActorState(); static_cast<AddMutationActor*>(this)->destroy(); return 0; }
															#line 1944 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<AddMutationActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AddMutationActorState();
		static_cast<AddMutationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 439 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!(part * CLIENT_KNOBS->MUTATION_BLOCK_SIZE < value.size()))
															#line 1963 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 440 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StringRef partBuf = value.substr( part * CLIENT_KNOBS->MUTATION_BLOCK_SIZE, std::min(value.size() - part * CLIENT_KNOBS->MUTATION_BLOCK_SIZE, CLIENT_KNOBS->MUTATION_BLOCK_SIZE));
															#line 443 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		Standalone<StringRef> key = getBlockKey(commitVersion, part);
															#line 444 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_0 = writeKV_impl(self, key, partBuf);
															#line 444 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<AddMutationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1975 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<AddMutationActor*>(this)->actor_wait_state = 1;
															#line 444 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AddMutationActor, 0, Void >*>(static_cast<AddMutationActor*>(this)));
															#line 1980 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 439 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		part++;
															#line 2002 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 439 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		part++;
															#line 2011 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AddMutationActor*>(this)->actor_wait_state > 0) static_cast<AddMutationActor*>(this)->actor_wait_state = 0;
		static_cast<AddMutationActor*>(this)->ActorCallback< AddMutationActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AddMutationActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("addMutation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AddMutationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("addMutation", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AddMutationActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("addMutation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AddMutationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("addMutation", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AddMutationActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("addMutation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AddMutationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("addMutation", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	LogFileWriter* self;
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Version commitVersion;
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	MutationListRef mutations;
															#line 436 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Standalone<StringRef> value;
															#line 438 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	int part;
															#line 2101 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via addMutation()
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class AddMutationActor final : public Actor<Void>, public ActorCallback< AddMutationActor, 0, Void >, public FastAllocated<AddMutationActor>, public AddMutationActorState<AddMutationActor> {
															#line 2106 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<AddMutationActor>::operator new;
	using FastAllocated<AddMutationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6018258094400101120UL, 8055527507645065728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AddMutationActor, 0, Void >;
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	AddMutationActor(LogFileWriter* const& self,Version const& commitVersion,MutationListRef const& mutations) 
															#line 2123 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<Void>(),
		   AddMutationActorState<AddMutationActor>(self, commitVersion, mutations),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("addMutation", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1351666754517471232UL, 8923191686972481024UL);
		ActorExecutionContextHelper __helper(static_cast<AddMutationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("addMutation");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("addMutation", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AddMutationActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] static Future<Void> addMutation( LogFileWriter* const& self, Version const& commitVersion, MutationListRef const& mutations ) {
															#line 435 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<Void>(new AddMutationActor(self, commitVersion, mutations));
															#line 2155 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 448 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

private:
	Reference<IBackupFile> file;
	int blockSize;
	int64_t blockEnd = 0;
};

															#line 2166 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
// This generated class is to be used only via convert()
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
template <class ConvertActor>
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class ConvertActorState {
															#line 2172 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	ConvertActorState(ConvertParams const& params) 
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		 : params(params),
															#line 456 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		   container(IBackupContainer::openContainer(params.container_url, params.proxy, {}))
															#line 2181 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
	{
		fdb_probe_actor_create("convert", reinterpret_cast<unsigned long>(this));

	}
	~ConvertActorState() 
	{
		fdb_probe_actor_destroy("convert", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 458 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			StrictFuture<BackupFileList> __when_expr_0 = container->dumpFileList();
															#line 458 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2198 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ConvertActor*>(this)->actor_wait_state = 1;
															#line 458 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 0, BackupFileList >*>(static_cast<ConvertActor*>(this)));
															#line 2203 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConvertActorState();
		static_cast<ConvertActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 459 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::sort(listing.logs.begin(), listing.logs.end());
															#line 460 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		TraceEvent("Container").detail("URL", params.container_url).detail("Logs", listing.logs.size());
															#line 461 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<BackupDescription> __when_expr_1 = container->describeBackup();
															#line 461 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2232 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 2;
															#line 461 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 1, BackupDescription >*>(static_cast<ConvertActor*>(this)));
															#line 2237 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(BackupFileList const& __listing,int loopDepth) 
	{
															#line 458 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		listing = __listing;
															#line 2246 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(BackupFileList && __listing,int loopDepth) 
	{
		listing = std::move(__listing);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 0, BackupFileList >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 0, BackupFileList >*,BackupFileList const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 0, BackupFileList >*,BackupFileList && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConvertActor, 0, BackupFileList >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 462 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::cout << "\n" << desc.toString() << "\n";
															#line 467 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::vector<LogFile> logs = getRelevantLogFiles(listing.logs, params.begin, params.end);
															#line 468 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		printLogFiles("Range has", logs);
															#line 470 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		progress = Reference<MutationFilesReadProgress>(new MutationFilesReadProgress(logs, params.begin, params.end));
															#line 472 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_2 = progress->openLogFiles(container);
															#line 472 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2335 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 3;
															#line 472 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 2, Void >*>(static_cast<ConvertActor*>(this)));
															#line 2340 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(BackupDescription const& __desc,int loopDepth) 
	{
															#line 461 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		desc = __desc;
															#line 2349 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(BackupDescription && __desc,int loopDepth) 
	{
		desc = std::move(__desc);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 1, BackupDescription >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 1, BackupDescription >*,BackupDescription const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 1, BackupDescription >*,BackupDescription && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConvertActor, 1, BackupDescription >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		blockSize = CLIENT_KNOBS->BACKUP_LOGFILE_BLOCK_SIZE;
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Reference<IBackupFile>> __when_expr_3 = container->writeLogFile(params.begin, params.end, blockSize);
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2432 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 4;
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*>(static_cast<ConvertActor*>(this)));
															#line 2437 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		blockSize = CLIENT_KNOBS->BACKUP_LOGFILE_BLOCK_SIZE;
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Reference<IBackupFile>> __when_expr_3 = container->writeLogFile(params.begin, params.end, blockSize);
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2450 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 4;
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*>(static_cast<ConvertActor*>(this)));
															#line 2455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConvertActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 476 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		logFile = LogFileWriter(outFile, blockSize);
															#line 477 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::cout << "Output file: " << outFile->getFileName() << "\n";
															#line 479 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		list = MutationList();
															#line 480 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		arena = Arena();
															#line 481 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		version = invalidVersion;
															#line 482 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		;
															#line 2549 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Reference<IBackupFile> const& __outFile,int loopDepth) 
	{
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		outFile = __outFile;
															#line 2558 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Reference<IBackupFile> && __outFile,int loopDepth) 
	{
		outFile = std::move(__outFile);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 3, Reference<IBackupFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*,Reference<IBackupFile> const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*,Reference<IBackupFile> && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 501 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (list.totalSize() > 0)
															#line 2637 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 502 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			StrictFuture<Void> __when_expr_6 = LogFileWriter::addMutation(&logFile, version, list);
															#line 502 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2643 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont5when1(__when_expr_6.get(), loopDepth); };
			static_cast<ConvertActor*>(this)->actor_wait_state = 7;
															#line 502 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 6, Void >*>(static_cast<ConvertActor*>(this)));
															#line 2648 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 482 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!(progress->hasMutations()))
															#line 2669 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
			return a_body1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 483 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<VersionedData> __when_expr_4 = progress->getNextMutation();
															#line 483 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2677 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 5;
															#line 483 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 4, VersionedData >*>(static_cast<ConvertActor*>(this)));
															#line 2682 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(int loopDepth) 
	{
															#line 486 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (list.totalSize() > 0 && version != data.version.version)
															#line 2704 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		{
															#line 487 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			StrictFuture<Void> __when_expr_5 = LogFileWriter::addMutation(&logFile, version, list);
															#line 487 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2710 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1cont1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ConvertActor*>(this)->actor_wait_state = 6;
															#line 487 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 5, Void >*>(static_cast<ConvertActor*>(this)));
															#line 2715 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(VersionedData const& __data,int loopDepth) 
	{
															#line 483 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		data = __data;
															#line 2729 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(VersionedData && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 4, VersionedData >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 4, VersionedData >*,VersionedData const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 4, VersionedData >*,VersionedData && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ConvertActor, 4, VersionedData >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont4loopBody1cont3(int loopDepth) 
	{
															#line 494 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		ArenaReader rd(data.arena, data.message, AssumeVersion(g_network->protocolVersion()));
															#line 495 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		MutationRef m;
															#line 496 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		rd >> m;
															#line 497 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		std::cout << data.version.toString() << " m = " << m.toString() << "\n";
															#line 498 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		list.push_back_deep(arena, m);
															#line 499 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		version = data.version.version;
															#line 2818 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 488 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		list = MutationList();
															#line 489 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		arena = Arena();
															#line 2829 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 488 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		list = MutationList();
															#line 489 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		arena = Arena();
															#line 2840 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ConvertActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 505 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		StrictFuture<Void> __when_expr_7 = outFile->finish();
															#line 505 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (static_cast<ConvertActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2926 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont6when1(__when_expr_7.get(), loopDepth); };
		static_cast<ConvertActor*>(this)->actor_wait_state = 8;
															#line 505 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ConvertActor, 7, Void >*>(static_cast<ConvertActor*>(this)));
															#line 2931 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ConvertActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<ConvertActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConvertActorState(); static_cast<ConvertActor*>(this)->destroy(); return 0; }
															#line 3027 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<ConvertActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConvertActorState();
		static_cast<ConvertActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
		if (!static_cast<ConvertActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConvertActorState(); static_cast<ConvertActor*>(this)->destroy(); return 0; }
															#line 3039 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		new (&static_cast<ConvertActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConvertActorState();
		static_cast<ConvertActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<ConvertActor*>(this)->actor_wait_state > 0) static_cast<ConvertActor*>(this)->actor_wait_state = 0;
		static_cast<ConvertActor*>(this)->ActorCallback< ConvertActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConvertActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ConvertActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ConvertActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	ConvertParams params;
															#line 456 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<IBackupContainer> container;
															#line 458 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	BackupFileList listing;
															#line 461 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	BackupDescription desc;
															#line 470 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<MutationFilesReadProgress> progress;
															#line 474 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	int blockSize;
															#line 475 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Reference<IBackupFile> outFile;
															#line 476 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	LogFileWriter logFile;
															#line 479 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	MutationList list;
															#line 480 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Arena arena;
															#line 481 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	Version version;
															#line 483 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	VersionedData data;
															#line 3146 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
};
// This generated class is to be used only via convert()
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
class ConvertActor final : public Actor<Void>, public ActorCallback< ConvertActor, 0, BackupFileList >, public ActorCallback< ConvertActor, 1, BackupDescription >, public ActorCallback< ConvertActor, 2, Void >, public ActorCallback< ConvertActor, 3, Reference<IBackupFile> >, public ActorCallback< ConvertActor, 4, VersionedData >, public ActorCallback< ConvertActor, 5, Void >, public ActorCallback< ConvertActor, 6, Void >, public ActorCallback< ConvertActor, 7, Void >, public FastAllocated<ConvertActor>, public ConvertActorState<ConvertActor> {
															#line 3151 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
public:
	using FastAllocated<ConvertActor>::operator new;
	using FastAllocated<ConvertActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11066009084372793856UL, 558030776185340416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConvertActor, 0, BackupFileList >;
friend struct ActorCallback< ConvertActor, 1, BackupDescription >;
friend struct ActorCallback< ConvertActor, 2, Void >;
friend struct ActorCallback< ConvertActor, 3, Reference<IBackupFile> >;
friend struct ActorCallback< ConvertActor, 4, VersionedData >;
friend struct ActorCallback< ConvertActor, 5, Void >;
friend struct ActorCallback< ConvertActor, 6, Void >;
friend struct ActorCallback< ConvertActor, 7, Void >;
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	ConvertActor(ConvertParams const& params) 
															#line 3175 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
		 : Actor<Void>(),
		   ConvertActorState<ConvertActor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("convert", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17730998987232764672UL, 92382927418949888UL);
		ActorExecutionContextHelper __helper(static_cast<ConvertActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("convert");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("convert", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConvertActor, 0, BackupFileList >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConvertActor, 1, BackupDescription >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ConvertActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ConvertActor, 3, Reference<IBackupFile> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ConvertActor, 4, VersionedData >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ConvertActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ConvertActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< ConvertActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
[[nodiscard]] Future<Void> convert( ConvertParams const& params ) {
															#line 455 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"
	return Future<Void>(new ConvertActor(params));
															#line 3214 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/build_output/fdbbackup/FileConverter.actor.g.cpp"
}

#line 509 "/codebuild/output/src4012789340/src/github.com/apple/foundationdb/fdbbackup/FileConverter.actor.cpp"

int parseCommandLine(ConvertParams* param, CSimpleOpt* args) {
	while (args->Next()) {
		auto lastError = args->LastError();
		switch (lastError) {
		case SO_SUCCESS:
			break;

		default:
			std::cerr << "ERROR: argument given for option: " << args->OptionText() << "\n";
			return FDB_EXIT_ERROR;
			break;
		}

		int optId = args->OptionId();
		const char* arg = args->OptionArg();
		switch (optId) {
		case OPT_HELP:
			printConvertUsage();
			return FDB_EXIT_ERROR;

		case OPT_BEGIN_VERSION:
			if (!sscanf(arg, "%" SCNd64, &param->begin)) {
				std::cerr << "ERROR: could not parse begin version " << arg << "\n";
				printConvertUsage();
				return FDB_EXIT_ERROR;
			}
			break;

		case OPT_END_VERSION:
			if (!sscanf(arg, "%" SCNd64, &param->end)) {
				std::cerr << "ERROR: could not parse end version " << arg << "\n";
				printConvertUsage();
				return FDB_EXIT_ERROR;
			}
			break;

		case OPT_CONTAINER:
			param->container_url = args->OptionArg();
			break;

		case OPT_TRACE:
			param->log_enabled = true;
			break;

		case OPT_TRACE_DIR:
			param->log_dir = args->OptionArg();
			break;

		case OPT_TRACE_FORMAT:
			if (!validateTraceFormat(args->OptionArg())) {
				std::cerr << "ERROR: Unrecognized trace format " << args->OptionArg() << "\n";
				return FDB_EXIT_ERROR;
			}
			param->trace_format = args->OptionArg();
			break;

		case OPT_TRACE_LOG_GROUP:
			param->trace_log_group = args->OptionArg();
			break;
		case OPT_BUILD_FLAGS:
			printBuildInformation();
			return FDB_EXIT_ERROR;
			break;
		}
	}
	return FDB_EXIT_SUCCESS;
}

} // namespace file_converter

int main(int argc, char** argv) {
	try {
		CSimpleOpt* args =
		    new CSimpleOpt(argc, argv, file_converter::gConverterOptions, SO_O_EXACT | SO_O_HYPHEN_TO_UNDERSCORE);
		file_converter::ConvertParams param;
		int status = file_converter::parseCommandLine(&param, args);
		std::cout << "Params: " << param.toString() << "\n";
		if (status != FDB_EXIT_SUCCESS || !param.isValid()) {
			file_converter::printConvertUsage();
			return status;
		}

		if (param.log_enabled) {
			if (param.log_dir.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE);
			} else {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE, StringRef(param.log_dir));
			}
			if (!param.trace_format.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, StringRef(param.trace_format));
			}
			if (!param.trace_log_group.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_LOG_GROUP, StringRef(param.trace_log_group));
			}
		}

		platformInit();
		Error::init();

		StringRef url(param.container_url);
		setupNetwork(0, UseMetrics::True);

		TraceEvent::setNetworkThread();
		openTraceFile({}, 10 << 20, 10 << 20, param.log_dir, "convert", param.trace_log_group);

		auto f = stopAfter(convert(param));

		runNetwork();
		return status;
	} catch (Error& e) {
		fprintf(stderr, "ERROR: %s\n", e.what());
		return FDB_EXIT_ERROR;
	} catch (std::exception& e) {
		TraceEvent(SevError, "MainError").error(unknown_error()).detail("RootException", e.what());
		return FDB_EXIT_MAIN_EXCEPTION;
	}
}