#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
/*
 * DataDistribution.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Audit.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DDSharedContext.h"
#include "fdbserver/DDTeamCollection.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TenantCache.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/MockDataDistributor.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const std::string ddServerBulkDumpFolder = "ddBulkDumpFiles";
static const std::string ddServerBulkLoadFolder = "ddBulkLoadFiles";

DataMoveType getDataMoveTypeFromDataMoveId(const UID& dataMoveId) {
	bool assigned, emptyRange;
	DataMoveType dataMoveType;
	DataMovementReason dataMoveReason;
	decodeDataMoveId(dataMoveId, assigned, emptyRange, dataMoveType, dataMoveReason);
	return dataMoveType;
}

void RelocateShard::setParentRange(KeyRange const& parent) {
	ASSERT(reason == RelocateReason::WRITE_SPLIT || reason == RelocateReason::SIZE_SPLIT);
	parent_range = parent;
}

Optional<KeyRange> RelocateShard::getParentRange() const {
	return parent_range;
}

ShardSizeBounds ShardSizeBounds::shardSizeBoundsBeforeTrack() {
	return ShardSizeBounds{ .max = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                           .iosPerKSecond = StorageMetrics::infinity,
		                                           .bytesReadPerKSecond = StorageMetrics::infinity,
		                                           .opsReadPerKSecond = StorageMetrics::infinity },
		                    .min = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = 0,
		                                           .iosPerKSecond = 0,
		                                           .bytesReadPerKSecond = 0,
		                                           .opsReadPerKSecond = 0 },
		                    .permittedError = StorageMetrics{ .bytes = -1,
		                                                      .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                                      .iosPerKSecond = StorageMetrics::infinity,
		                                                      .bytesReadPerKSecond = StorageMetrics::infinity,
		                                                      .opsReadPerKSecond = StorageMetrics::infinity } };
}

namespace {

std::set<int> const& normalDDQueueErrors() {
	static std::set<int> s{ error_code_movekeys_conflict,
		                    error_code_broken_promise,
		                    error_code_data_move_cancelled,
		                    error_code_data_move_dest_team_not_found };
	return s;
}

} // anonymous namespace

enum class DDAuditContext : uint8_t {
	INVALID = 0,
	RESUME = 1,
	LAUNCH = 2,
	RETRY = 3,
};

struct DDAudit {
	DDAudit(AuditStorageState coreState)
	  : coreState(coreState), actors(true), foundError(false), auditStorageAnyChildFailed(false), retryCount(0),
	    cancelled(false), overallCompleteDoAuditCount(0), overallIssuedDoAuditCount(0), overallSkippedDoAuditCount(0),
	    remainingBudgetForAuditTasks(SERVER_KNOBS->CONCURRENT_AUDIT_TASK_COUNT_MAX), context(DDAuditContext::INVALID) {}

	AuditStorageState coreState;
	ActorCollection actors;
	Future<Void> auditActor;
	bool foundError;
	int retryCount;
	bool auditStorageAnyChildFailed;
	bool cancelled; // use to cancel any actor beyond auditActor
	int64_t overallIssuedDoAuditCount;
	int64_t overallCompleteDoAuditCount;
	int64_t overallSkippedDoAuditCount;
	AsyncVar<int> remainingBudgetForAuditTasks;
	DDAuditContext context;
	std::unordered_set<UID> serversFinishedSSShardAudit; // dedicated to ssshard

	inline void setAuditRunActor(Future<Void> actor) { auditActor = actor; }
	inline Future<Void> getAuditRunActor() { return auditActor; }

	inline void setDDAuditContext(DDAuditContext context_) { this->context = context_; }
	inline DDAuditContext getDDAuditContext() const { return context; }

	// auditActor and actors are guaranteed to deliver a cancel signal
	void cancel() {
		auditActor.cancel();
		actors.clear(true);
		cancelled = true;
	}

	bool isCancelled() const { return cancelled; }
};

void DataMove::validateShard(const DDShardInfo& shard, KeyRangeRef range, int priority) {
	if (!valid) {
		if (shard.hasDest && shard.destId != anonymousShardId) {
			TraceEvent(SevError, "DataMoveValidationError")
			    .detail("Range", range)
			    .detail("Reason", "DataMoveMissing")
			    .detail("DestID", shard.destId)
			    .detail("ShardPrimaryDest", describe(shard.primaryDest))
			    .detail("ShardRemoteDest", describe(shard.remoteDest));
		}
		return;
	}

	ASSERT(!this->meta.ranges.empty() && this->meta.ranges.front().contains(range));

	if (!shard.hasDest) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "ShardMissingDest")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest));
		cancelled = true;
		return;
	}

	if (shard.destId != this->meta.id) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveIDMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardMoveID", shard.destId);
		cancelled = true;
		return;
	}

	if (!std::equal(
	        this->primaryDest.begin(), this->primaryDest.end(), shard.primaryDest.begin(), shard.primaryDest.end()) ||
	    !std::equal(
	        this->remoteDest.begin(), this->remoteDest.end(), shard.remoteDest.begin(), shard.remoteDest.end())) {
		TraceEvent(g_network->isSimulated() ? SevWarn : SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveDestMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest))
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		cancelled = true;
	}
}

Future<Void> StorageWiggler::onCheck() const {
	return delay(MIN_ON_CHECK_DELAY_SEC);
}

// add server to wiggling queue
void StorageWiggler::addServer(const UID& serverId, const StorageMetadataType& metadata) {
	// std::cout << "size: " << pq_handles.size() << " add " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	ASSERT(!pq_handles.contains(serverId));
	pq_handles[serverId] = wiggle_pq.emplace(metadata, serverId);
}

void StorageWiggler::removeServer(const UID& serverId) {
	// std::cout << "size: " << pq_handles.size() << " remove " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	if (contains(serverId)) { // server haven't been popped
		auto handle = pq_handles.at(serverId);
		pq_handles.erase(serverId);
		wiggle_pq.erase(handle);
	}
}

void StorageWiggler::updateMetadata(const UID& serverId, const StorageMetadataType& metadata) {
	//	std::cout << "size: " << pq_handles.size() << " update " << serverId.toString()
	//	          << " DC: " << teamCollection->isPrimary() << std::endl;
	auto handle = pq_handles.at(serverId);
	if ((*handle).first == metadata) {
		return;
	}
	wiggle_pq.update(handle, std::make_pair(metadata, serverId));
}

bool StorageWiggler::necessary(const UID& serverId, const StorageMetadataType& metadata) const {
	return metadata.wrongConfiguredForWiggle ||
	       (now() - metadata.createdTime > SERVER_KNOBS->DD_STORAGE_WIGGLE_MIN_SS_AGE_SEC);
}

Optional<UID> StorageWiggler::getNextServerId(bool necessaryOnly) {
	if (!wiggle_pq.empty()) {
		auto [metadata, id] = wiggle_pq.top();
		if (necessaryOnly && !necessary(id, metadata)) {
			return {};
		}
		wiggle_pq.pop();
		pq_handles.erase(id);
		return Optional<UID>(id);
	}
	return Optional<UID>();
}

Future<Void> StorageWiggler::resetStats() {
	metrics.reset();
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.resetStorageWiggleMetrics(tr, PrimaryRegion(teamCollection->isPrimary()), metrics);
	    });
}

Future<Void> StorageWiggler::restoreStats() {
	auto readFuture = wiggleData.storageWiggleMetrics(PrimaryRegion(teamCollection->isPrimary()))
	                      .getD(teamCollection->dbContext().getReference(), Snapshot::False, metrics);
	return store(metrics, readFuture);
}

Future<Void> StorageWiggler::startWiggle() {
	metrics.last_wiggle_start = StorageMetadataType::currentTime();
	if (shouldStartNewRound()) {
		metrics.last_round_start = metrics.last_wiggle_start;
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

Future<Void> StorageWiggler::finishWiggle() {
	metrics.last_wiggle_finish = StorageMetadataType::currentTime();
	metrics.finished_wiggle += 1;
	auto duration = metrics.last_wiggle_finish - metrics.last_wiggle_start;
	metrics.smoothed_wiggle_duration.setTotal((double)duration);

	if (shouldFinishRound()) {
		metrics.last_round_finish = metrics.last_wiggle_finish;
		metrics.finished_round += 1;
		duration = metrics.last_round_finish - metrics.last_round_start;
		metrics.smoothed_round_duration.setTotal((double)duration);
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

															#line 298 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via remoteRecovered()
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoteRecoveredActor>
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActorState {
															#line 305 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActorState(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : db(db)
															#line 312 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	~RemoteRecoveredActorState() 
	{
		fdb_probe_actor_destroy("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 297 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDTrackerStarting").log();
															#line 298 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 329 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 302 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoteRecoveredActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoteRecoveredActorState(); static_cast<RemoteRecoveredActor*>(this)->destroy(); return 0; }
															#line 352 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoteRecoveredActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 298 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(db->get().recoveryState < RecoveryState::ALL_LOGS_RECRUITED))
															#line 371 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 299 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDTrackerStarting").detail("RecoveryState", (int)db->get().recoveryState);
															#line 300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 381 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 1;
															#line 300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoteRecoveredActor, 0, Void >*>(static_cast<RemoteRecoveredActor*>(this)));
															#line 386 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state > 0) static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 0;
		static_cast<RemoteRecoveredActor*>(this)->ActorCallback< RemoteRecoveredActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoteRecoveredActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 493 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via remoteRecovered()
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActor final : public Actor<Void>, public ActorCallback< RemoteRecoveredActor, 0, Void >, public FastAllocated<RemoteRecoveredActor>, public RemoteRecoveredActorState<RemoteRecoveredActor> {
															#line 498 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoteRecoveredActor>::operator new;
	using FastAllocated<RemoteRecoveredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16914582526738826752UL, 407147850992024320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoteRecoveredActor, 0, Void >;
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActor(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 515 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoteRecoveredActorState<RemoteRecoveredActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17462346244829468672UL, 10124612030083111680UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remoteRecovered");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoteRecoveredActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> remoteRecovered( Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoteRecoveredActor(db));
															#line 548 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 304 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Ensures that the serverKeys key space is properly coalesced
// This method is only used for testing and is not implemented in a manner that is safe for large databases
															#line 555 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via debugCheckCoalescing()
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DebugCheckCoalescingActor>
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActorState {
															#line 562 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActorState(Database const& cx) 
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : cx(cx),
															#line 308 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	~DebugCheckCoalescingActorState() 
	{
		fdb_probe_actor_destroy("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 309 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 586 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 1;
															#line 311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 624 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 3;
															#line 331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 2, Void >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 653 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 312 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 314 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = int();
															#line 315 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 672 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __serverList,int loopDepth) 
	{
															#line 311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		serverList = __serverList;
															#line 681 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __serverList,int loopDepth) 
	{
		serverList = std::move(__serverList);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 328 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DoneCheckingCoalescing").log();
															#line 329 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DebugCheckCoalescingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DebugCheckCoalescingActorState(); static_cast<DebugCheckCoalescingActor*>(this)->destroy(); return 0; }
															#line 762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DebugCheckCoalescingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 315 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < serverList.size()))
															#line 781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 316 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		id = decodeServerListValue(serverList[i].value).id();
															#line 317 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, serverKeysPrefixFor(id), allKeys);
															#line 317 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 791 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 2;
															#line 317 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 796 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult const& ranges,int loopDepth) 
	{
															#line 318 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 320 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 321 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 822 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 322 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 826 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 315 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 831 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult && ranges,int loopDepth) 
	{
															#line 318 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 320 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 321 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 844 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 322 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 315 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 853 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 308 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult serverList;
															#line 314 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 316 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID id;
															#line 1030 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via debugCheckCoalescing()
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActor final : public Actor<Void>, public ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 2, Void >, public FastAllocated<DebugCheckCoalescingActor>, public DebugCheckCoalescingActorState<DebugCheckCoalescingActor> {
															#line 1035 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DebugCheckCoalescingActor>::operator new;
	using FastAllocated<DebugCheckCoalescingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18251431648966939392UL, 10474841597942208512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 2, Void >;
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActor(Database const& cx) 
															#line 1054 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DebugCheckCoalescingActorState<DebugCheckCoalescingActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13012747114799893760UL, 10690014687488268032UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("debugCheckCoalescing");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> debugCheckCoalescing( Database const& cx ) {
															#line 307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DebugCheckCoalescingActor(cx));
															#line 1089 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 335 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DataDistributor;
void runAuditStorage(
    Reference<DataDistributor> self,
    AuditStorageState auditStates,
    int retryCount,
    DDAuditContext context,
    Optional<std::unordered_set<UID>> serversFinishedSSShardAudit = Optional<std::unordered_set<UID>>());
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorageCore( Reference<DataDistributor> const& self, UID const& auditID, AuditType const& auditType, int const& currentRetryCount );

#line 347 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1105 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<UID> launchAudit( Reference<DataDistributor> const& self, KeyRange const& auditRange, AuditType const& auditType, KeyValueStoreType const& auditStorageEngineType );

#line 351 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1109 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorage( Reference<DataDistributor> const& self, TriggerAuditRequest const& req );

#line 352 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
void loadAndDispatchAudit(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit);
															#line 1114 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorageServerShard( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 354 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1118 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditStorageShardOnServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi );

#line 357 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1122 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorage( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 358 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1126 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 361 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1130 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& auditRange );

#line 364 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1134 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 367 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1138 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditOnStorageServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi, AuditStorageRequest const& req );

#line 371 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1142 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> skipAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& rangeToSkip );

#line 374 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void runBulkLoadTaskAsync(Reference<DataDistributor> self, KeyRange range, UID taskId, bool restart);
															#line 1148 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self );

#line 377 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DDBulkLoadJobManager {
	BulkLoadJobState jobState;
	BulkLoadTransportMethod jobTransportMethod;
	std::shared_ptr<BulkLoadManifestFileMap> manifestEntryMap;
	std::string manifestLocalTempFolder;
	bool allTaskSubmitted = false;

	DDBulkLoadJobManager() = default;
	DDBulkLoadJobManager(const BulkLoadJobState& jobState, const std::string& manifestLocalTempFolder)
	  : jobState(jobState), manifestLocalTempFolder(manifestLocalTempFolder), allTaskSubmitted(false) {
		manifestEntryMap = std::make_shared<BulkLoadManifestFileMap>();
	}

	bool isValid() const { return jobState.isValid(); }
};

struct DDBulkDumpJobManager {
	BulkDumpState jobState;
	std::map<Key, BulkLoadManifest> taskManifestMap;

	DDBulkDumpJobManager() = default;
	DDBulkDumpJobManager(const BulkDumpState& jobState) : jobState(jobState) {}

	bool isValid() const { return jobState.isValid(); }
};

struct DataDistributor : NonCopyable, ReferenceCounted<DataDistributor> {
public:
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
	Reference<DDSharedContext> context;
	UID ddId;
	PromiseStream<Future<Void>> addActor;

	// State initialized when bootstrap
	Reference<IDDTxnProcessor> txnProcessor;
	MoveKeysLock& lock; // reference to context->lock
	DatabaseConfiguration& configuration; // reference to context->configuration
	std::vector<Optional<Key>> primaryDcId;
	std::vector<Optional<Key>> remoteDcIds;
	Reference<InitialDataDistribution> initData;

	Reference<EventCacheHolder> initialDDEventHolder;
	Reference<EventCacheHolder> movingDataEventHolder;
	Reference<EventCacheHolder> totalDataInFlightEventHolder;
	Reference<EventCacheHolder> totalDataInFlightRemoteEventHolder;

	// Optional components that can be set after ::init(). They're optional when test, but required for DD being
	// fully-functional.
	DDTeamCollection* teamCollection;
	Reference<ShardsAffectedByTeamFailure> shardsAffectedByTeamFailure;
	// consumer is a yield stream from producer. The RelocateShard is pushed into relocationProducer and popped from
	// relocationConsumer (by DDQueue)
	PromiseStream<RelocateShard> relocationProducer, relocationConsumer;
	PromiseStream<BulkLoadShardRequest> triggerShardBulkLoading;
	Reference<PhysicalShardCollection> physicalShardCollection;
	Reference<BulkLoadTaskCollection> bulkLoadTaskCollection;

	Promise<Void> initialized;

	std::unordered_map<AuditType, std::unordered_map<UID, std::shared_ptr<DDAudit>>> audits;
	FlowLock auditStorageHaLaunchingLock;
	FlowLock auditStorageReplicaLaunchingLock;
	FlowLock auditStorageLocationMetadataLaunchingLock;
	FlowLock auditStorageSsShardLaunchingLock;
	FlowLock auditStorageRestoreLaunchingLock;
	Promise<Void> auditStorageInitialized;
	bool auditStorageInitStarted;

	Optional<Reference<TenantCache>> ddTenantCache;

	// monitor DD configuration change
	Promise<Version> configChangeWatching;
	Future<Void> onConfigChange;

	ActorCollection bulkLoadActors;
	bool bulkLoadEnabled = false;
	ParallelismLimitor bulkLoadParallelismLimitor;
	ParallelismLimitor bulkLoadEngineParallelismLimitor;
	std::string bulkLoadFolder;

	Optional<DDBulkLoadJobManager> bulkLoadJobManager;

	bool bulkDumpEnabled = false;
	ParallelismLimitor bulkDumpParallelismLimitor;
	std::string folder;
	std::string bulkDumpFolder;

	DDBulkDumpJobManager bulkDumpJobManager;

	DataDistributor(Reference<AsyncVar<ServerDBInfo> const> const& db,
	                UID id,
	                Reference<DDSharedContext> context,
	                std::string folder)
	  : dbInfo(db), context(context), ddId(id), txnProcessor(nullptr), lock(context->lock),
	    configuration(context->configuration), initialDDEventHolder(makeReference<EventCacheHolder>("InitialDD")),
	    movingDataEventHolder(makeReference<EventCacheHolder>("MovingData")),
	    totalDataInFlightEventHolder(makeReference<EventCacheHolder>("TotalDataInFlight")),
	    totalDataInFlightRemoteEventHolder(makeReference<EventCacheHolder>("TotalDataInFlightRemote")),
	    teamCollection(nullptr), bulkLoadTaskCollection(nullptr), auditStorageHaLaunchingLock(1),
	    auditStorageReplicaLaunchingLock(1), auditStorageLocationMetadataLaunchingLock(1),
	    auditStorageSsShardLaunchingLock(1), auditStorageInitStarted(false), bulkLoadActors(false),
	    bulkLoadEnabled(false), bulkLoadParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM),
	    bulkLoadEngineParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM), bulkDumpEnabled(false),
	    bulkDumpParallelismLimitor(SERVER_KNOBS->DD_BULKDUMP_PARALLELISM), folder(folder) {
		if (!folder.empty()) {
			bulkDumpFolder = abspath(joinPath(folder, ddServerBulkDumpFolder));
			// TODO(BulkDump): clear this folder in the presence of crash
			bulkLoadFolder = abspath(joinPath(folder, ddServerBulkLoadFolder));
			// TODO(BulkLoad): clear this folder in the presence of crash
		}
	}

	// bootstrap steps

	Future<Void> takeMoveKeysLock() { return store(lock, txnProcessor->takeMoveKeysLock(ddId)); }

	Future<Void> loadDatabaseConfiguration() { return store(configuration, txnProcessor->getDatabaseConfiguration()); }

	Future<Void> updateReplicaKeys() {
		return txnProcessor->updateReplicaKeys(primaryDcId, remoteDcIds, configuration);
	}

	Future<Void> loadInitialDataDistribution() {
		return store(initData,
		             txnProcessor->getInitialDataDistribution(
		                 ddId,
		                 lock,
		                 configuration.usableRegions > 1 ? remoteDcIds : std::vector<Optional<Key>>(),
		                 context->ddEnabledState.get(),
		                 SkipDDModeCheck::False));
	}

	void initDcInfo() {
		primaryDcId.clear();
		remoteDcIds.clear();
		const std::vector<RegionInfo>& regions = configuration.regions;
		if (configuration.regions.size() > 0) {
			primaryDcId.push_back(regions[0].dcId);
		}
		if (configuration.regions.size() > 1) {
			remoteDcIds.push_back(regions[1].dcId);
		}
	}

	Future<Void> waitDataDistributorEnabled() const {
		return txnProcessor->waitForDataDistributionEnabled(context->ddEnabledState.get());
	}

	// Resume in-memory audit instances and issue background audit metadata cleanup
	void resumeAuditStorage(Reference<DataDistributor> self, std::vector<AuditStorageState> auditStates) {
		for (const auto& auditState : auditStates) {
			if (auditState.getPhase() != AuditPhase::Running) {
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "WrongAuditStateToResume")
				    .detail("AuditState", auditState.toString());
				return;
			}
			if (self->audits.contains(auditState.getType()) &&
			    self->audits[auditState.getType()].contains(auditState.id)) {
				// Ignore any RUNNING auditState with an alive audit
				// instance in DD audits map
				continue;
			}
			runAuditStorage(self, auditState, 0, DDAuditContext::RESUME);
			TraceEvent(SevInfo, "AuditStorageResumed", self->ddId)
			    .detail("AuditID", auditState.id)
			    .detail("AuditType", auditState.getType())
			    .detail("AuditState", auditState.toString());
		}
		return;
	}

																#line 1324 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via initAuditStorage()
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitAuditStorageActor>
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActorState {
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActorState(Reference<DataDistributor> const& self) 
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 1337 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	~InitAuditStorageActorState() 
	{
		fdb_probe_actor_destroy("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 550 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->auditStorageInitStarted = true;
															#line 551 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			MoveKeyLockInfo lockInfo;
															#line 552 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.myOwner = self->lock.myOwner;
															#line 553 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevOwner = self->lock.prevOwner;
															#line 554 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevWrite = self->lock.prevWrite;
															#line 555 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<std::vector<AuditStorageState>> __when_expr_0 = initAuditMetadata(self->txnProcessor->context(), lockInfo, self->context->isDDEnabled(), self->ddId, SERVER_KNOBS->PERSIST_FINISH_AUDIT_COUNT);
															#line 555 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1364 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 1;
															#line 555 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*>(static_cast<InitAuditStorageActor*>(this)));
															#line 1369 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
															#line 561 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 562 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 563 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1396 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
															#line 561 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 562 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 563 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1412 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(auditStatesToResume, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(auditStatesToResume), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state > 0) static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 0;
		static_cast<InitAuditStorageActor*>(this)->ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >::remove();

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> const& value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> && value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,Error err) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1497 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via initAuditStorage()
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActor final : public Actor<Void>, public ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >, public FastAllocated<InitAuditStorageActor>, public InitAuditStorageActorState<InitAuditStorageActor> {
															#line 1502 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitAuditStorageActor>::operator new;
	using FastAllocated<InitAuditStorageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10454299909454261504UL, 10734198285716534272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >;
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActor(Reference<DataDistributor> const& self) 
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitAuditStorageActorState<InitAuditStorageActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8855224104647952640UL, 5194562846350463232UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initAuditStorage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> initAuditStorage( Reference<DataDistributor> const& self ) {
															#line 549 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitAuditStorageActor(self));
															#line 1551 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 565 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 1556 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class WaitUntilDataDistributorExitSecurityModeActor>
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActorState {
															#line 1562 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActorState(Reference<DataDistributor> const& self) 
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 567 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(self->txnProcessor->context())
															#line 1571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	~WaitUntilDataDistributorExitSecurityModeActorState() 
	{
		fdb_probe_actor_destroy("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 568 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 1586 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitUntilDataDistributorExitSecurityModeActorState();
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->DD_ENABLED_CHECK_DELAY, TaskPriority::DataDistribution);
															#line 569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1618 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 1;
															#line 569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1623 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 572 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1636 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1647 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 572 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1666 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1672 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 574 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1677 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 4;
															#line 587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& mode,int loopDepth) 
	{
															#line 575 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1796 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 576 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1800 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 578 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 579 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 581 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1814 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 582 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1818 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1828 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1833 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && mode,int loopDepth) 
	{
															#line 575 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 576 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1846 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 578 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 579 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 581 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1860 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 582 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1864 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1874 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1879 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> const& mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(mode, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> && mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(mode), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 585 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1963 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 585 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1972 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 567 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2156 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActor final : public Actor<Void>, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >, public FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>, public WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor> {
															#line 2161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator new;
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15281669097938409728UL, 6420847465047608576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >;
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActor(Reference<DataDistributor> const& self) 
															#line 2181 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4834519635010961920UL, 15563594834600371456UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitUntilDataDistributorExitSecurityMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> waitUntilDataDistributorExitSecurityMode( Reference<DataDistributor> const& self ) {
															#line 566 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new WaitUntilDataDistributorExitSecurityModeActor(self));
															#line 2216 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 591 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Initialize the required internal states of DataDistributor from system metadata. It's necessary before
	// DataDistributor start working. Doesn't include initialization of optional components, like TenantCache, DDQueue,
	// Tracker, TeamCollection. The components should call its own ::init methods.
																#line 2224 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via init()
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitActor>
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActorState {
															#line 2230 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActorState(Reference<DataDistributor> const& self) 
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 2237 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("init", reinterpret_cast<unsigned long>(this));

	}
	~InitActorState() 
	{
		fdb_probe_actor_destroy("init", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 596 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 2252 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitActorState();
		static_cast<InitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 708 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitActorState(); static_cast<InitActor*>(this)->destroy(); return 0; }
															#line 2275 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitActorState();
		static_cast<InitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 597 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->waitDataDistributorEnabled();
															#line 597 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 1;
															#line 597 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitActor, 0, Void >*>(static_cast<InitActor*>(this)));
															#line 2301 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 598 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2329 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2334 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 598 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2354 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 602 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 606 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2440 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 608 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2444 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 610 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2448 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 617 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2453 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2459 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2464 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 602 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 606 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 608 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2484 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 610 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 617 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2493 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2499 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 619 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2504 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 627 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributorExitSecurityMode").log();
															#line 629 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->loadDatabaseConfiguration();
															#line 629 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2597 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 4;
															#line 629 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< InitActor, 3, Void >*>(static_cast<InitActor*>(this)));
															#line 2602 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 630 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 631 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 635 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2702 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 637 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 639 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2708 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2714 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2719 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 630 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 631 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 635 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 637 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 639 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2738 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2749 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< InitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 643 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2837 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 643 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 645 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2860 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< InitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 647 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2944 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2948 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 655 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2954 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 663 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2958 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2962 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 669 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 673 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 695 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 701 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 2973 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 647 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2982 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2986 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 655 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2992 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 663 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2996 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 3000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 669 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 673 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 695 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 701 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 3011 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< InitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 3093 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via init()
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActor final : public Actor<Void>, public ActorCallback< InitActor, 0, Void >, public ActorCallback< InitActor, 1, Void >, public ActorCallback< InitActor, 2, Void >, public ActorCallback< InitActor, 3, Void >, public ActorCallback< InitActor, 4, Void >, public ActorCallback< InitActor, 5, Void >, public FastAllocated<InitActor>, public InitActorState<InitActor> {
															#line 3098 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitActor>::operator new;
	using FastAllocated<InitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1937488553546153728UL, 15556795589752684544UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitActor, 0, Void >;
friend struct ActorCallback< InitActor, 1, Void >;
friend struct ActorCallback< InitActor, 2, Void >;
friend struct ActorCallback< InitActor, 3, Void >;
friend struct ActorCallback< InitActor, 4, Void >;
friend struct ActorCallback< InitActor, 5, Void >;
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActor(Reference<DataDistributor> const& self) 
															#line 3120 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitActorState<InitActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1770802396446596864UL, 4776746127134105600UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< InitActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< InitActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< InitActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> init( Reference<DataDistributor> const& self ) {
															#line 595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitActor(self));
															#line 3157 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 710 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3162 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoveDataMoveTombstoneBackgroundActor>
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActorState {
															#line 3168 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActorState(Reference<DataDistributor> const& self) 
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 712 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentID()
															#line 3177 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	~RemoveDataMoveTombstoneBackgroundActorState() 
	{
		fdb_probe_actor_destroy("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 714 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				cx = openDBOnServer(self->dbInfo, TaskPriority::DefaultEndpoint, LockAware::True);
															#line 715 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr = Transaction(cx);
															#line 716 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				;
															#line 3197 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 741 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveDataMoveTombstoneBackgroundActorState(); static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->destroy(); return 0; }
															#line 3226 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3239 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3243 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 735 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "RemoveDataMoveTombstoneError", self->ddId) .errorUnsuppressed(e) .detail("CurrentDataMoveID", currentID);
															#line 3247 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 718 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 719 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 720 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			for( UID& dataMoveID : self->initData->toCleanDataMoveTombstone ) {
															#line 721 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				currentID = dataMoveID;
															#line 722 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr.clear(dataMoveKeyFor(currentID));
															#line 723 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "RemoveDataMoveTombstone", self->ddId).detail("DataMoveID", currentID);
															#line 3286 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 725 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 725 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 1;
															#line 725 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3297 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 728 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 728 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3334 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 2;
															#line 728 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3339 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 712 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID currentID;
															#line 714 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 715 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 3545 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActor final : public Actor<Void>, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >, public FastAllocated<RemoveDataMoveTombstoneBackgroundActor>, public RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor> {
															#line 3550 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator new;
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16845554789722555904UL, 3847041498509260800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >;
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >;
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActor(Reference<DataDistributor> const& self) 
															#line 3568 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929663207261195520UL, 2170269584154920960UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeDataMoveTombstoneBackground");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> removeDataMoveTombstoneBackground( Reference<DataDistributor> const& self ) {
															#line 711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoveDataMoveTombstoneBackgroundActor(self));
															#line 3601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 743 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3606 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromShards()
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromShardsActor>
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActorState {
															#line 3612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActorState(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   traceShard(traceShard)
															#line 3621 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromShardsActorState() 
	{
		fdb_probe_actor_destroy("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 746 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 3636 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 747 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for(int i = 0;i < self->initData->shards.size() - 1;i++) {
															#line 748 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					const DDShardInfo& iShard = self->initData->shards[i];
															#line 749 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					KeyRangeRef keys = KeyRangeRef(iShard.key, self->initData->shards[i + 1].key);
															#line 750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 751 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					teams.emplace_back(iShard.primarySrc, true);
															#line 752 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (self->configuration.usableRegions > 1)
															#line 3650 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 753 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.emplace_back(iShard.remoteSrc, false);
															#line 3654 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
															#line 755 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->physicalShardCollection->initPhysicalShardCollection(keys, teams, iShard.srcId.first(), 0);
															#line 3658 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundaries = std::vector<Key>();
															#line 760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadIsEnabled(self->initData->bulkLoadMode))
															#line 3665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevInfo, "DDInitCustomRangeConfigDisabledByBulkLoadMode", self->ddId);
															#line 3669 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for( auto it : self->initData->userRangeConfig->ranges() ) {
															#line 765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					auto range = it->range();
															#line 766 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					customBoundaries.push_back(range.begin);
															#line 767 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDInitCustomRangeConfig", self->ddId) .detail("Range", KeyRangeRef(range.begin, range.end)) .detail("Config", it->value());
															#line 3681 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 773 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			shard = 0;
															#line 774 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary = 0;
															#line 775 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			overreplicatedCount = 0;
															#line 776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 3692 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 850 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromShardsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromShardsActorState(); static_cast<ResumeFromShardsActor*>(this)->destroy(); return 0; }
															#line 3715 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromShardsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(shard < self->initData->shards.size() - 1))
															#line 3734 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DDShardInfo& iShard = self->initData->shards[shard];
															#line 778 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<KeyRangeRef> ranges;
															#line 780 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key beginKey = iShard.key;
															#line 781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key endKey = self->initData->shards[shard + 1].key;
															#line 782 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] <= beginKey;) {
															#line 783 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 785 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] < endKey;) {
															#line 786 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ranges.push_back(KeyRangeRef(beginKey, customBoundaries[customBoundary]));
															#line 787 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = customBoundaries[customBoundary];
															#line 788 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 790 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ranges.push_back(KeyRangeRef(beginKey, endKey));
															#line 792 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 793 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.primarySrc, true));
															#line 794 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.usableRegions > 1)
															#line 3770 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 795 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.remoteSrc, false));
															#line 3774 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 798 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int r = 0;r < ranges.size();r++) {
															#line 799 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto& keys = ranges[r];
															#line 800 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->defineShard(keys);
															#line 802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->initData->userRangeConfig->rangeContaining(keys.begin);
															#line 803 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			int customReplicas = std::max(self->configuration.storageTeamSize, it->value().replicationFactor.orDefault(0));
															#line 805 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT_WE_THINK(KeyRangeRef(it->range().begin, it->range().end).contains(keys));
															#line 807 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bool unhealthy = iShard.primarySrc.size() != customReplicas;
															#line 808 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && self->configuration.usableRegions > 1)
															#line 3792 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 809 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				unhealthy = iShard.remoteSrc.size() != customReplicas;
															#line 3796 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 811 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && iShard.primarySrc.size() > self->configuration.storageTeamSize)
															#line 3800 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 812 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (++overreplicatedCount > SERVER_KNOBS->DD_MAX_SHARDS_ON_LARGE_TEAMS)
															#line 3804 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 813 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					unhealthy = true;
															#line 3808 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 817 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (traceShard)
															#line 3813 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 818 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDInitShard", self->ddId) .detail("Keys", keys) .detail("PrimarySrc", describe(iShard.primarySrc)) .detail("RemoteSrc", describe(iShard.remoteSrc)) .detail("PrimaryDest", describe(iShard.primaryDest)) .detail("RemoteDest", describe(iShard.remoteDest)) .detail("SrcID", iShard.srcId) .detail("DestID", iShard.destId) .detail("CustomReplicas", customReplicas) .detail("StorageTeamSize", self->configuration.storageTeamSize) .detail("Unhealthy", unhealthy) .detail("Overreplicated", overreplicatedCount);
															#line 3817 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 832 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(keys, teams);
															#line 833 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if ((ddLargeTeamEnabled() && (unhealthy || r > 0)) || (iShard.hasDest && iShard.destId == anonymousShardId))
															#line 3823 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 838 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				DataMovementReason reason = DataMovementReason::RECOVER_MOVE;
															#line 839 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (unhealthy)
															#line 3829 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 840 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					reason = DataMovementReason::TEAM_UNHEALTHY;
															#line 3833 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 841 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (r > 0)
															#line 3839 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						reason = DataMovementReason::SPLIT_SHARD;
															#line 3843 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
				}
															#line 844 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(RelocateShard(keys, reason, RelocateReason::OTHER));
															#line 3848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::DataDistribution);
															#line 848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 1;
															#line 848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromShardsActor, 0, Void >*>(static_cast<ResumeFromShardsActor*>(this)));
															#line 3860 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3882 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3891 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromShardsActor*>(this)->ActorCallback< ResumeFromShardsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromShardsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool traceShard;
															#line 759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Key> customBoundaries;
															#line 773 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int shard;
															#line 774 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int customBoundary;
															#line 775 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int overreplicatedCount;
															#line 3983 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromShards()
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActor final : public Actor<Void>, public ActorCallback< ResumeFromShardsActor, 0, Void >, public FastAllocated<ResumeFromShardsActor>, public ResumeFromShardsActorState<ResumeFromShardsActor> {
															#line 3988 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromShardsActor>::operator new;
	using FastAllocated<ResumeFromShardsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7094099656605385728UL, 4452739015307430400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromShardsActor, 0, Void >;
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActor(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 4005 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromShardsActorState<ResumeFromShardsActor>(self, traceShard),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17921411212936075776UL, 15009917662088599808UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromShards");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromShardsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromShards( Reference<DataDistributor> const& self, bool const& traceShard ) {
															#line 744 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromShardsActor(self, traceShard));
															#line 4037 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 852 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// TODO: unit test needed
																#line 4043 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromDataMoves()
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromDataMovesActor>
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActorState {
															#line 4049 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart),
															#line 855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   it(self->initData->dataMoveMap.ranges().begin())
															#line 4060 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromDataMovesActorState() 
	{
		fdb_probe_actor_destroy("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 1;
															#line 857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 0, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4082 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 859 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4105 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 859 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4114 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 921 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 4198 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 922 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->addActor.send(self->removeDataMoveTombstoneBackground(self));
															#line 4202 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 924 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromDataMovesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromDataMovesActorState(); static_cast<ResumeFromDataMovesActor*>(this)->destroy(); return 0; }
															#line 4206 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromDataMovesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 859 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(it != self->initData->dataMoveMap.ranges().end()))
															#line 4225 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 860 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DataMoveMetaData& meta = it.value()->meta;
															#line 861 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		DataMoveType dataMoveType = getDataMoveTypeFromDataMoveId(meta.id);
															#line 862 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.ranges.empty())
															#line 4235 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 863 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "EmptyDataMoveRange", self->ddId).detail("DataMoveMetaData", meta.toString());
															#line 4239 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1cont1continue1(loopDepth); // continue
		}
															#line 866 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.bulkLoadTaskState.present())
															#line 4244 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 867 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 868 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.dataMoveId = meta.id;
															#line 869 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.cancelled = true;
															#line 870 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->relocationProducer.send(rs);
															#line 873 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMove", self->ddId) .detail("Reason", "DDInit") .detail("DataMove", meta.toString());
															#line 4256 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		else
		{
															#line 876 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (dataMoveType == DataMoveType::LOGICAL_BULKLOAD || dataMoveType == DataMoveType::PHYSICAL_BULKLOAD)
															#line 4263 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 879 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 880 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.dataMoveId = meta.id;
															#line 881 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.cancelled = true;
															#line 882 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(rs);
															#line 883 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMoveForWrongType", self->ddId) .detail("Reason", "WrongTypeWhenDDInit") .detail("DataMoveType", dataMoveType) .detail("DataMove", meta.toString());
															#line 4275 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1cont1loopBody1cont5(loopDepth);
			}
			else
			{
															#line 887 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (it.value()->isCancelled() || (it.value()->valid && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA))
															#line 4282 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 889 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 890 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.dataMoveId = meta.id;
															#line 891 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.cancelled = true;
															#line 892 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->relocationProducer.send(rs);
															#line 893 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent("DDInitScheduledCancelDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 4294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = a_body1cont1loopBody1cont7(loopDepth);
				}
				else
				{
															#line 894 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (it.value()->valid)
															#line 4301 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 895 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						TraceEvent(SevDebug, "DDInitFoundDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 896 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						ASSERT(meta.ranges.front() == it.range());
															#line 898 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 899 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMoveId = meta.id;
															#line 900 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMove = it.value();
															#line 901 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 902 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->primaryDest, true));
															#line 903 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (!rs.dataMove->remoteDest.empty())
															#line 4319 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 904 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->remoteDest, false));
															#line 4323 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						}
															#line 909 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->restartShardTracker.send(rs.keys);
															#line 910 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->defineShard(rs.keys);
															#line 914 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->moveShard(rs.keys, teams);
															#line 915 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->relocationProducer.send(rs);
															#line 916 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						StrictFuture<Void> __when_expr_1 = yield(TaskPriority::DataDistribution);
															#line 916 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4337 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
						static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 2;
															#line 916 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 1, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4342 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont1loopBody1cont9(loopDepth);
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1continue1(int loopDepth) 
	{
															#line 859 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4372 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 859 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4381 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRangeMap<std::shared_ptr<DataMove>>::iterator it;
															#line 4497 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromDataMoves()
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActor final : public Actor<Void>, public ActorCallback< ResumeFromDataMovesActor, 0, Void >, public ActorCallback< ResumeFromDataMovesActor, 1, Void >, public FastAllocated<ResumeFromDataMovesActor>, public ResumeFromDataMovesActorState<ResumeFromDataMovesActor> {
															#line 4502 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromDataMovesActor>::operator new;
	using FastAllocated<ResumeFromDataMovesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13008791138971174912UL, 1321524992614396416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromDataMovesActor, 0, Void >;
friend struct ActorCallback< ResumeFromDataMovesActor, 1, Void >;
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 4520 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromDataMovesActorState<ResumeFromDataMovesActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929780632836526848UL, 14642014520803479552UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromDataMoves");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromDataMoves( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromDataMovesActor(self, readyToStart));
															#line 4553 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 926 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Resume inflight relocations from the previous DD
	// TODO: The initialDataDistribution is unused once resumeRelocations,
	// DataDistributionTracker::trackInitialShards, and DDTeamCollection::init are done. In the future, we can release
	// the object to save memory usage if it turns out to be a problem.
	Future<Void> resumeRelocations() {
		ASSERT(shardsAffectedByTeamFailure); // has to be allocated
		Future<Void> shardsReady = resumeFromShards(Reference<DataDistributor>::addRef(this), g_network->isSimulated());
		return resumeFromDataMoves(Reference<DataDistributor>::addRef(this), shardsReady);
	}

	Future<Void> pollMoveKeysLock() const {
		return txnProcessor->pollMoveKeysLock(lock, context->ddEnabledState.get());
	}

	Future<bool> isDataDistributionEnabled() const {
		return txnProcessor->isDataDistributionEnabled(context->ddEnabledState.get());
	}

	Future<Void> removeKeysFromFailedServer(const UID& serverID, const std::vector<UID>& teamForDroppedRange) const {
		return txnProcessor->removeKeysFromFailedServer(
		    serverID, teamForDroppedRange, lock, context->ddEnabledState.get());
	}

	Future<Void> removeStorageServer(const UID& serverID, const Optional<UID>& tssPairID = Optional<UID>()) const {
		return txnProcessor->removeStorageServer(serverID, tssPairID, lock, context->ddEnabledState.get());
	}

	Future<Void> initDDConfigWatch();

	Future<Void> initTenantCache();
};

Future<Void> DataDistributor::initDDConfigWatch() {
	if (txnProcessor->isMocked()) {
		onConfigChange = Never();
		return Void();
	}
	onConfigChange = map(DDConfiguration().trigger.onChange(
	                         SystemDBWriteLockedNow(txnProcessor->context().getReference()), {}, configChangeWatching),
	                     [](Version v) {
		                     CODE_PROBE(true, "DataDistribution change detected");
		                     TraceEvent("DataDistributionConfigChanged").detail("ChangeVersion", v);
		                     throw dd_config_changed();
		                     return Void();
	                     });

	return success(configChangeWatching.getFuture());
}

Future<Void> DataDistributor::initTenantCache() {
	// SOMEDAY: support tenant cache in MockDD
	ASSERT(!txnProcessor->isMocked());
	ddTenantCache = makeReference<TenantCache>(txnProcessor->context(), ddId);
	return ddTenantCache.get()->build();
}

inline void addAuditToAuditMap(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit) {
	AuditType auditType = audit->coreState.getType();
	UID auditID = audit->coreState.id;
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "addAuditToAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(!self->audits[auditType].contains(auditID));
	self->audits[auditType][auditID] = audit;
	return;
}

inline std::shared_ptr<DDAudit> getAuditFromAuditMap(Reference<DataDistributor> self,
                                                     AuditType auditType,
                                                     UID auditID) {
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "getAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	return self->audits[auditType][auditID];
}

inline void removeAuditFromAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	std::shared_ptr<DDAudit> audit = self->audits[auditType][auditID];
	audit->cancel();
	self->audits[auditType].erase(auditID);
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "removeAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	return;
}

inline bool auditExistInAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	return self->audits.contains(auditType) && self->audits[auditType].contains(auditID);
}

inline bool existAuditInAuditMapForType(Reference<DataDistributor> self, AuditType auditType) {
	return self->audits.contains(auditType) && !self->audits[auditType].empty();
}

inline std::unordered_map<UID, std::shared_ptr<DDAudit>> getAuditsForType(Reference<DataDistributor> self,
                                                                          AuditType auditType) {
	ASSERT(self->audits.contains(auditType));
	return self->audits[auditType];
}

// Periodically check and log the physicalShard status; clean up empty physicalShard;
															#line 4664 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MonitorPhysicalShardStatusActor>
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActorState {
															#line 4671 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActorState(Reference<PhysicalShardCollection> const& self) 
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 4678 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	~MonitorPhysicalShardStatusActorState() 
	{
		fdb_probe_actor_destroy("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1034 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
															#line 1035 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD);
															#line 1036 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4697 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorPhysicalShardStatusActorState();
		static_cast<MonitorPhysicalShardStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1037 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->cleanUpPhysicalShardCollection();
															#line 1038 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->logPhysicalShardCollection();
															#line 1039 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->PHYSICAL_SHARD_METRICS_DELAY);
															#line 1039 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 1;
															#line 1039 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*>(static_cast<MonitorPhysicalShardStatusActor*>(this)));
															#line 4738 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state > 0) static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorPhysicalShardStatusActor*>(this)->ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<PhysicalShardCollection> self;
															#line 4832 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActor final : public Actor<Void>, public ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >, public FastAllocated<MonitorPhysicalShardStatusActor>, public MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor> {
															#line 4837 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator new;
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7545640700641765376UL, 9381464509680064256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >;
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActor(Reference<PhysicalShardCollection> const& self) 
															#line 4854 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8017156993063803648UL, 8326163650824365056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorPhysicalShardStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> monitorPhysicalShardStatus( Reference<PhysicalShardCollection> const& self ) {
															#line 1033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MonitorPhysicalShardStatusActor(self));
															#line 4887 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1042 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// This actor must be a singleton
															#line 4893 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via prepareDataMigration()
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PrepareDataMigrationActor>
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActorState {
															#line 4900 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActorState(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : req(req),
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   context(context),
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(cx)
															#line 4911 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	~PrepareDataMigrationActorState() 
	{
		fdb_probe_actor_destroy("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1049 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<Version, Tag>> __when_expr_0 = addStorageServer(cx, req.ssi);
															#line 1049 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4929 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 1;
															#line 1049 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 4934 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1065 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(context->ddEnabledState->trySetEnabled(req.requesterID));
															#line 1066 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PrepareDataMigrationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrepareDataMigrationActorState(); static_cast<PrepareDataMigrationActor*>(this)->destroy(); return 0; }
															#line 4965 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PrepareDataMigrationActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1060 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4978 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1061 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4982 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1062 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(e);
															#line 4986 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
															#line 1050 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5005 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5010 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
															#line 1050 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5023 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5028 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(verAndTag, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(verAndTag), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1058 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5112 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1058 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5121 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareBlobRestoreRequest req;
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDSharedContext> context;
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 5220 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via prepareDataMigration()
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActor final : public Actor<Void>, public ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >, public ActorCallback< PrepareDataMigrationActor, 1, Void >, public FastAllocated<PrepareDataMigrationActor>, public PrepareDataMigrationActorState<PrepareDataMigrationActor> {
															#line 5225 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PrepareDataMigrationActor>::operator new;
	using FastAllocated<PrepareDataMigrationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11140521886600785920UL, 8550759419922099200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >;
friend struct ActorCallback< PrepareDataMigrationActor, 1, Void >;
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActor(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 5243 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PrepareDataMigrationActorState<PrepareDataMigrationActor>(req, context, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12945933996581036800UL, 2706979800331646720UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("prepareDataMigration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> prepareDataMigration( PrepareBlobRestoreRequest const& req, Reference<DDSharedContext> const& context, Database const& cx ) {
															#line 1044 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PrepareDataMigrationActor(req, context, cx));
															#line 5277 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1068 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 5282 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ServeBlobMigratorRequestsActor>
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActorState {
															#line 5289 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActorState(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tracker(tracker),
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   queue(queue)
															#line 5300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	~ServeBlobMigratorRequestsActorState() 
	{
		fdb_probe_actor_destroy("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->initialized.getFuture();
															#line 1072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5317 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 1;
															#line 1072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5322 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ServeBlobMigratorRequestsActorState();
		static_cast<ServeBlobMigratorRequestsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5354 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1074 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		FutureStream<PrepareBlobRestoreRequest> __when_expr_1 = self->context->interface.prepareBlobRestoreReq.getFuture();
															#line 1074 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5447 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 2;
															#line 1074 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5452 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
															#line 1075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5461 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1076 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5465 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1079 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5469 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1082 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5474 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1084 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1086 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1088 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1091 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1095 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5496 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
															#line 1075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1076 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5510 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1079 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5514 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1082 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1084 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1086 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5525 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1088 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1091 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5533 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1095 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5541 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(req, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(req), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributionTracker> tracker;
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDQueue> queue;
															#line 5628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActor final : public Actor<Void>, public ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >, public ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >, public FastAllocated<ServeBlobMigratorRequestsActor>, public ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor> {
															#line 5633 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator new;
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5114524242437123072UL, 4376284190091228928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >;
friend struct ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >;
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActor(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 5651 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor>(self, tracker, queue),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13290839411857998080UL, 673675850101827840UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("serveBlobMigratorRequests");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> serveBlobMigratorRequests( Reference<DataDistributor> const& self, Reference<DataDistributionTracker> const& tracker, Reference<DDQueue> const& queue ) {
															#line 1069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ServeBlobMigratorRequestsActor(self, tracker, queue));
															#line 5685 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1099 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Trigger a task on range based on the current bulk load task metadata
															#line 5691 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class TriggerBulkLoadTaskActor>
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActorState {
															#line 5698 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId)
															#line 5709 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~TriggerBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1104 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 5724 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1105 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Database cx = self->txnProcessor->context();
															#line 1106 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr = Transaction(cx);
															#line 1107 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState = BulkLoadTaskState();
															#line 5758 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 1109 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1110 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1111 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1111 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5768 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1111 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5773 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1137 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5795 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1138 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistTriggerStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 5799 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1143 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1143 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5805 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1143 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5810 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1112 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5829 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5834 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1112 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5847 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5852 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1119 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1120 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1121 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1122 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5951 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1119 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1120 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1121 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1122 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5970 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5975 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6061 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6066 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1127 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6082 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1128 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1129 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1133 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1134 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6172 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1128 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1129 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1133 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1134 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6190 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1106 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1107 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState newBulkLoadTaskState;
															#line 6370 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActor final : public Actor<std::pair<BulkLoadTaskState, Version>>, public ActorCallback< TriggerBulkLoadTaskActor, 0, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 1, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 2, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 3, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 4, Void >, public FastAllocated<TriggerBulkLoadTaskActor>, public TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor> {
															#line 6375 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<TriggerBulkLoadTaskActor>::operator new;
	using FastAllocated<TriggerBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18423967338326684672UL, 10322170403536593408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BulkLoadTaskState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TriggerBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 4, Void >;
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 6396 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<std::pair<BulkLoadTaskState, Version>>(),
		   TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9998265325154424320UL, 14265371869313678848UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("triggerBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<std::pair<BulkLoadTaskState, Version>> triggerBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<std::pair<BulkLoadTaskState, Version>>(new TriggerBulkLoadTaskActor(self, taskRange, taskId));
															#line 6433 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1147 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// TODO(BulkLoad): add reason to persist
															#line 6439 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via failBulkLoadTask()
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FailBulkLoadTaskActor>
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActorState {
															#line 6446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cancelledDataMovePriority(cancelledDataMovePriority),
															#line 1153 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1154 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1155 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState()
															#line 6465 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~FailBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1156 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 6480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1186 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FailBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FailBulkLoadTaskActorState(); static_cast<FailBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6503 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FailBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1158 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1159 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1160 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1160 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6529 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1160 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 0, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6534 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1177 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 6569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1178 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistErrorStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1183 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1183 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6579 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1183 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 4, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6584 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6601 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6606 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6617 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1161 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6622 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1163 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1164 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1165 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1166 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6716 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6721 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1163 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1164 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1165 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1166 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6740 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1167 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6745 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6831 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6836 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6847 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1169 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6852 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1170 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1171 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6938 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1170 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1171 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6949 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int cancelledDataMovePriority;
															#line 1153 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1154 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1155 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 7130 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via failBulkLoadTask()
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< FailBulkLoadTaskActor, 0, Void >, public ActorCallback< FailBulkLoadTaskActor, 1, Void >, public ActorCallback< FailBulkLoadTaskActor, 2, Void >, public ActorCallback< FailBulkLoadTaskActor, 3, Void >, public ActorCallback< FailBulkLoadTaskActor, 4, Void >, public FastAllocated<FailBulkLoadTaskActor>, public FailBulkLoadTaskActorState<FailBulkLoadTaskActor> {
															#line 7135 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FailBulkLoadTaskActor>::operator new;
	using FastAllocated<FailBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3709333766515170560UL, 12037452007262210304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FailBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 4, Void >;
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 7156 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FailBulkLoadTaskActorState<FailBulkLoadTaskActor>(self, taskRange, taskId, cancelledDataMovePriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5276853651913060352UL, 273265509463260672UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("failBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> failBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId, int const& cancelledDataMovePriority ) {
															#line 1149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FailBulkLoadTaskActor(self, taskRange, taskId, cancelledDataMovePriority));
															#line 7193 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1188 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulk load task is guaranteed to be either complete or overwritten by another task
// When a bulk load task is trigged, the range traffic is turned off atomically
// If the task completes, the task re-enables the traffic atomically
															#line 7201 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via doBulkLoadTask()
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DoBulkLoadTaskActor>
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActorState {
															#line 7208 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1193 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   completeAck(),
															#line 1194 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   triggeredBulkLoadTask(),
															#line 1195 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   commitVersion(invalidVersion),
															#line 1196 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 7227 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~DoBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1197 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadEngineParallelismLimitor.canStart());
															#line 1198 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadEngineParallelismLimitor.incrementTaskCounter();
															#line 7244 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1201 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<BulkLoadTaskState, Version>> __when_expr_0 = triggerBulkLoadTask(self, range, taskId);
															#line 1201 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7250 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1201 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7255 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1279 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadEngineParallelismLimitor.decrementTaskCounter();
															#line 1280 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DoBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBulkLoadTaskActorState(); static_cast<DoBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7286 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DoBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1265 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 7299 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1266 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7303 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1268 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(e) .detail("Phase", "Error") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1274 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 7309 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1275 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7313 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1202 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1203 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1204 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1211 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1217 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1222 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7343 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7348 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1202 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1203 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1204 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1211 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1217 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1222 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7371 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7376 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(triggeredBulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(triggeredBulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1224 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ack.unretrievableError)
															#line 7460 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1225 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "See unretrievable error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7464 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1233 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_2 = failBulkLoadTask(self, range, taskId, ack.dataMovePriority);
															#line 1233 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 7470 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1233 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 2, Void >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7475 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont3Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1258 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Task complete") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck const& __ack,int loopDepth) 
	{
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ack = __ack;
															#line 7498 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck && __ack,int loopDepth) 
	{
		ack = std::move(__ack);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& failTaskError,int loopDepth=0) 
	{
		try {
															#line 1241 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_actor_cancelled)
															#line 7590 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1242 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7594 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1244 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(failTaskError) .detail("Phase", "Failed to mark task error. Error should be bulkload_task_outdated") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1251 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_movekeys_conflict)
															#line 7600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1252 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7604 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1254 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(failTaskError.code() == error_code_bulkload_task_outdated);
															#line 7608 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 1234 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7623 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 1234 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7632 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1193 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<BulkLoadAck> completeAck;
															#line 1194 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState triggeredBulkLoadTask;
															#line 1195 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Version commitVersion;
															#line 1196 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1223 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadAck ack;
															#line 7754 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via doBulkLoadTask()
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >, public ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >, public ActorCallback< DoBulkLoadTaskActor, 2, Void >, public FastAllocated<DoBulkLoadTaskActor>, public DoBulkLoadTaskActorState<DoBulkLoadTaskActor> {
															#line 7759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DoBulkLoadTaskActor>::operator new;
	using FastAllocated<DoBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10046982490172774912UL, 3070904930897812736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >;
friend struct ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >;
friend struct ActorCallback< DoBulkLoadTaskActor, 2, Void >;
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 7778 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DoBulkLoadTaskActorState<DoBulkLoadTaskActor>(self, range, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9618644142710713856UL, 6256719351634756608UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> doBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& taskId ) {
															#line 1192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DoBulkLoadTaskActor(self, range, taskId));
															#line 7813 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1282 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7818 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class EraseBulkLoadTaskActor>
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActorState {
															#line 7825 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1284 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1285 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1286 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask()
															#line 7842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~EraseBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1287 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1327 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<EraseBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EraseBulkLoadTaskActorState(); static_cast<EraseBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7880 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<EraseBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1289 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1290 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1291 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(bulkLoadTask, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Acknowledged }));
															#line 1291 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7906 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1291 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 0, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7911 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1314 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1315 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskEraseStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7950 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1320 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 7954 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1324 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1324 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7962 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1324 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 3, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7967 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7984 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7989 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8005 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8091 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8096 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8107 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1294 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8112 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1295 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1301 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1302 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8204 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1306 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8210 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1295 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1296 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1300 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1301 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1302 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8228 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1306 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8234 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1284 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1285 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1286 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 8414 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< EraseBulkLoadTaskActor, 0, Void >, public ActorCallback< EraseBulkLoadTaskActor, 1, Void >, public ActorCallback< EraseBulkLoadTaskActor, 2, Void >, public ActorCallback< EraseBulkLoadTaskActor, 3, Void >, public FastAllocated<EraseBulkLoadTaskActor>, public EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor> {
															#line 8419 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<EraseBulkLoadTaskActor>::operator new;
	using FastAllocated<EraseBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(874293924914668800UL, 8230039144714815744UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EraseBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 3, Void >;
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 8439 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(445300323391651072UL, 14768580555042305792UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("eraseBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> eraseBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1283 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new EraseBulkLoadTaskActor(self, taskRange, taskId));
															#line 8475 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1329 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 8480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadTasksActor>
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActorState {
															#line 8487 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActorState(Reference<DataDistributor> const& self) 
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginKey(normalKeys.begin),
															#line 1332 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   endKey(normalKeys.end),
															#line 1333 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeToRead(),
															#line 1334 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1335 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1336 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   i(0),
															#line 1337 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1338 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   result(),
															#line 1339 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadActors(),
															#line 1340 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range()
															#line 8514 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadTasksActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1341 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8529 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1402 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = waitForAll(bulkLoadActors);
															#line 1402 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8554 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 7;
															#line 1402 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8559 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1341 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 8575 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1343 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1344 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			result.clear();
															#line 1345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store( result, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead, SERVER_KNOBS->DD_BULKLOAD_TASK_METADATA_READ_SIZE));
															#line 1345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8588 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 1;
															#line 1345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8593 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1396 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 8628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1397 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 8632 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1399 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1399 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8638 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 6;
															#line 1399 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8643 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1348 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8660 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1348 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8671 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1394 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = result.back().key;
															#line 8755 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < result.size() - 1))
															#line 8771 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1350 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (result[i].value.empty())
															#line 8777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3continue1(loopDepth); // continue
		}
															#line 1353 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		range = Standalone(KeyRangeRef(result[i].key, result[i + 1].key));
															#line 1354 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[i].value);
															#line 1355 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 8787 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1356 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 1356 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8793 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 2;
															#line 1356 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8798 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3continue1(int loopDepth) 
	{
															#line 1349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8825 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1359 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (range != bulkLoadTaskState.getRange())
															#line 8834 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1361 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.1);
															#line 1361 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8840 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 3;
															#line 1361 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8845 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont6(int loopDepth) 
	{
															#line 1364 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.phase == BulkLoadPhase::Submitted || bulkLoadTaskState.phase == BulkLoadPhase::Triggered || bulkLoadTaskState.phase == BulkLoadPhase::Running)
															#line 8946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1368 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8950 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3loopBody1cont6loopHead1(loopDepth);
		}
		else
		{
															#line 1380 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadTaskState.phase == BulkLoadPhase::Acknowledged)
															#line 8957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1381 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskClearMetadata", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 1385 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_4 = eraseBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId());
															#line 1385 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8965 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
				static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 5;
															#line 1385 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8970 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1386 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (bulkLoadTaskState.phase == BulkLoadPhase::Error)
															#line 8977 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1387 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevWarnAlways, "DDBulkLoadTaskUnretriableError", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 8981 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 1391 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(bulkLoadTaskState.phase == BulkLoadPhase::Complete);
															#line 8987 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3loopBody1cont9(int loopDepth) 
	{
															#line 1349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 9086 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(int loopDepth) 
	{
															#line 1374 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSchedule", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("Phase", bulkLoadTaskState.phase);
															#line 1378 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadActors.push_back( doBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId()));
															#line 9097 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1369 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadEngineParallelismLimitor.canStart())
															#line 9113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1372 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->bulkLoadEngineParallelismLimitor.waitUntilCounterChanged();
															#line 1372 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9121 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont6loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 4;
															#line 1372 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 9126 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1403 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9428 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1403 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9440 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1332 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1333 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1334 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1335 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1336 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1337 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1338 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult result;
															#line 1339 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> bulkLoadActors;
															#line 1340 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 9545 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >, public FastAllocated<ScheduleBulkLoadTasksActor>, public ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor> {
															#line 9550 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator new;
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9799219688304151552UL, 5940183053805971712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >;
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActor(Reference<DataDistributor> const& self) 
															#line 9573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14856297450692458752UL, 7460955458194361088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadTasks");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self ) {
															#line 1330 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadTasksActor(self));
															#line 9612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1405 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 9617 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadTaskCoreActor>
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActorState {
															#line 9624 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 9633 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadTaskCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1407 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 1407 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9650 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 1;
															#line 1407 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 0, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9655 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadTaskCoreActorState();
		static_cast<BulkLoadTaskCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1408 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9682 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9687 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1408 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9700 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1409 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9705 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1410 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1411 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9791 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1410 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1411 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1413 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = scheduleBulkLoadTasks(self);
															#line 1413 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9896 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 3;
															#line 1413 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 2, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9901 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1423 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 1423 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9918 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 4;
															#line 1423 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 3, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9923 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1415 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9933 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1416 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9937 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1418 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadTaskCoreError", self->ddId).errorUnsuppressed(e);
															#line 1419 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 9943 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1420 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9947 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 1408 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 10152 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActor final : public Actor<Void>, public ActorCallback< BulkLoadTaskCoreActor, 0, Void >, public ActorCallback< BulkLoadTaskCoreActor, 1, Void >, public ActorCallback< BulkLoadTaskCoreActor, 2, Void >, public ActorCallback< BulkLoadTaskCoreActor, 3, Void >, public FastAllocated<BulkLoadTaskCoreActor>, public BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor> {
															#line 10157 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadTaskCoreActor>::operator new;
	using FastAllocated<BulkLoadTaskCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18257795692312393984UL, 11789809311576285952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadTaskCoreActor, 0, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 1, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 2, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 3, Void >;
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 10177 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12210363843676168704UL, 8182787864321598208UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadTaskCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadTaskCore( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 1406 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadTaskCoreActor(self, readyToStart));
															#line 10213 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1426 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10218 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadJob()
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadJobActor>
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActorState {
															#line 10225 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1428 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10236 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1430 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1430 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10253 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1430 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadJobActor, 0, Void >*>(static_cast<CheckBulkLoadJobActor*>(this)));
															#line 10258 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1431 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1432 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1433 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10285 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1434 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10289 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1436 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1437 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10295 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1438 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10299 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1440 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10303 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1441 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10307 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1443 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1431 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1432 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1433 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10327 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1434 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10331 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1436 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1437 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10337 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1438 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10341 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1440 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1441 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10349 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1443 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10353 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadJobActor*>(this)->ActorCallback< CheckBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1428 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10442 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadJob()
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActor final : public Actor<Void>, public ActorCallback< CheckBulkLoadJobActor, 0, Void >, public FastAllocated<CheckBulkLoadJobActor>, public CheckBulkLoadJobActorState<CheckBulkLoadJobActor> {
															#line 10447 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadJobActor>::operator new;
	using FastAllocated<CheckBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17040984681577773824UL, 16326153687502535936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadJobActor, 0, Void >;
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10464 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   CheckBulkLoadJobActorState<CheckBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6430116811539734272UL, 4354220408088791808UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> checkBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1427 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new CheckBulkLoadJobActor(tr, jobId));
															#line 10497 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1445 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10502 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via getBulkLoadJob()
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class GetBulkLoadJobActor>
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActorState {
															#line 10509 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1447 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10520 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~GetBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1449 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1449 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1449 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBulkLoadJobActor, 0, Void >*>(static_cast<GetBulkLoadJobActor*>(this)));
															#line 10542 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1450 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1451 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1452 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1453 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1455 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1456 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10579 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1457 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10583 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1459 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1460 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10591 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1462 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10595 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1450 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1451 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1452 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10611 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1453 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10615 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1455 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1456 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10621 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1457 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10625 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1459 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10629 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1460 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10633 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1462 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10637 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<GetBulkLoadJobActor*>(this)->ActorCallback< GetBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1447 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10726 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via getBulkLoadJob()
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActor final : public Actor<BulkLoadJobState>, public ActorCallback< GetBulkLoadJobActor, 0, Void >, public FastAllocated<GetBulkLoadJobActor>, public GetBulkLoadJobActorState<GetBulkLoadJobActor> {
															#line 10731 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<GetBulkLoadJobActor>::operator new;
	using FastAllocated<GetBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11921691501912054016UL, 10191285225711158016UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadJobState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBulkLoadJobActor, 0, Void >;
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10748 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadJobState>(),
		   GetBulkLoadJobActorState<GetBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17307579102949951488UL, 17075247084345564672UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadJobState> getBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadJobState>(new GetBulkLoadJobActor(tr, jobId));
															#line 10781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1464 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Find task metadata for a bulk load job with jobId and input range
															#line 10787 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobFindTaskActor>
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActorState {
															#line 10794 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   logId(logId),
															#line 1470 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1471 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1472 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 10813 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobFindTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1473 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 10828 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1502 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(bulkLoadTaskState); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10851 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(std::move(bulkLoadTaskState)); // state_var_RVO
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1475 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1476 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobId);
															#line 1476 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10875 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 1;
															#line 1476 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10880 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1499 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1499 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10917 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 3;
															#line 1499 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10922 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1477 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10941 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1477 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10959 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1478 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10964 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& result,int loopDepth) 
	{
															#line 1480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1481 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1482 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11052 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1483 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1485 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1486 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1487 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11068 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1495 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11074 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && result,int loopDepth) 
	{
															#line 1480 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1481 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1482 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11088 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1483 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11092 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1485 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1486 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1487 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11104 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1495 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11110 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID logId;
															#line 1470 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1471 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1472 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 11292 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActor final : public Actor<Optional<BulkLoadTaskState>>, public ActorCallback< BulkLoadJobFindTaskActor, 0, Void >, public ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >, public ActorCallback< BulkLoadJobFindTaskActor, 2, Void >, public FastAllocated<BulkLoadJobFindTaskActor>, public BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor> {
															#line 11297 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobFindTaskActor>::operator new;
	using FastAllocated<BulkLoadJobFindTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16502348125011699200UL, 16776935796608180992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<BulkLoadTaskState>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobFindTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 2, Void >;
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 11316 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Optional<BulkLoadTaskState>>(),
		   BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor>(self, range, jobId, logId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3995601487305067776UL, 1783749224615820032UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobFindTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Optional<BulkLoadTaskState>> bulkLoadJobFindTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& jobId, UID const& logId ) {
															#line 1466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Optional<BulkLoadTaskState>>(new BulkLoadJobFindTaskActor(self, range, jobId, logId));
															#line 11351 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1504 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Submit a bulkload task for the given jobId
															#line 11357 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobSubmitTaskActor>
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActorState {
															#line 11364 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(manifests),
															#line 1509 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1510 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1512 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(jobId, manifests)
															#line 11381 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobSubmitTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1513 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 11396 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1533 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV<BulkLoadTaskState>::futures) { (void)(bulkLoadTask); this->~BulkLoadJobSubmitTaskActorState(); static_cast<BulkLoadJobSubmitTaskActor*>(this)->destroy(); return 0; }
															#line 11419 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV< BulkLoadTaskState >::value()) BulkLoadTaskState(std::move(bulkLoadTask)); // state_var_RVO
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1516 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1516 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11441 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 1;
															#line 1516 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11446 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1530 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1530 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11483 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 4;
															#line 1530 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11488 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11505 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11510 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11521 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1517 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11526 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11617 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11633 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1520 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1521 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11719 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1520 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1521 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11730 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1509 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1510 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1512 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 11909 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActor final : public Actor<BulkLoadTaskState>, public ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >, public FastAllocated<BulkLoadJobSubmitTaskActor>, public BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor> {
															#line 11914 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator new;
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15856012116883390464UL, 403136944727835136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadTaskState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >;
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 11934 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadTaskState>(),
		   BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor>(self, jobId, manifests),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9698732467664213248UL, 5402256740808584704UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobSubmitTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadTaskState> bulkLoadJobSubmitTask( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadManifestSet const& manifests ) {
															#line 1506 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadTaskState>(new BulkLoadJobSubmitTaskActor(self, jobId, manifests));
															#line 11970 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1535 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// If the task reaches unretrievable error, return true; otherwise, return false.
															#line 11976 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobWaitUntilTaskCompleteOrErrorActor>
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActorState {
															#line 11983 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(bulkLoadTask)
															#line 11994 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1540 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(bulkLoadTask.isValid());
															#line 1541 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1542 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1543 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentTask = BulkLoadTaskState();
															#line 1544 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 12017 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1546 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentTask, getBulkLoadTask(&tr, bulkLoadTask.getRange(), bulkLoadTask.getTaskId(), { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running, BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }));
															#line 1546 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12050 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1546 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12055 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_JOB_MONITOR_PERIOD_SEC);
															#line 1573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1573 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12089 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1571 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12094 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1556 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12109 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1557 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12113 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1559 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12117 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1560 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1565 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12123 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1567 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12131 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1568 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12135 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1556 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12149 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1557 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12153 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1559 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12157 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1560 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1565 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12163 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1567 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12171 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1568 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12175 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1541 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1542 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1543 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState currentTask;
															#line 12459 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActor final : public Actor<Void>, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>, public BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor> {
															#line 12464 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5366535039436126976UL, 2879128695426962432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >;
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 12483 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>(self, jobId, bulkLoadTask),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16093638368250768128UL, 12615896230081500416UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobWaitUntilTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobWaitUntilTaskCompleteOrError( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadTaskState const& bulkLoadTask ) {
															#line 1537 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobWaitUntilTaskCompleteOrErrorActor(self, jobId, bulkLoadTask));
															#line 12518 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1576 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a list of manifestEntry, create a bulkload task.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 12526 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobNewTaskActor>
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActorState {
															#line 12533 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRoot(jobRoot),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestLocalTempFolder(manifestLocalTempFolder),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntries(manifestEntries),
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1588 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1589 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(),
															#line 1590 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 12560 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobNewTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1591 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!manifestEntries.empty());
															#line 12575 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1594 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(manifests, getBulkLoadManifestMetadataFromEntry( manifestEntries, manifestLocalTempFolder, jobTransportMethod, jobRoot, self->ddId));
															#line 1594 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12581 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 1;
															#line 1594 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12586 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12615 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1627 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 12632 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1630 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1636 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 12638 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1637 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 12642 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12660 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12676 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12681 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1602 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1604 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12769 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1611 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12786 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1602 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12795 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1604 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12799 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1611 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12811 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1612 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12816 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1614 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1621 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12902 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1623 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1624 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12908 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1614 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1621 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12920 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1623 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1624 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12926 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string manifestLocalTempFolder;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1587 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1588 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1589 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1590 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13042 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobNewTaskActor, 0, Void >, public ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobNewTaskActor, 2, Void >, public FastAllocated<BulkLoadJobNewTaskActor>, public BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor> {
															#line 13047 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobNewTaskActor>::operator new;
	using FastAllocated<BulkLoadJobNewTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2667257976996428800UL, 4944194679745036032UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobNewTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 2, Void >;
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActor(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 13066 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor>(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8655289167937893888UL, 2026549218517418240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobNewTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobNewTask( Reference<DataDistributor> const& self, UID const& jobId, std::string const& jobRoot, BulkLoadTransportMethod const& jobTransportMethod, std::string const& manifestLocalTempFolder, std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries, Promise<Void> const& errorOut ) {
															#line 1580 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobNewTaskActor(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut));
															#line 13101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1644 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a bulkload task range, find the task and wait until the task is complete or error.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 13109 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobMonitorTaskActor>
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActorState {
															#line 13116 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1652 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1653 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1654 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 13135 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobMonitorTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1655 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadParallelismLimitor.canStart());
															#line 1656 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.incrementTaskCounter();
															#line 13152 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1659 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Optional<BulkLoadTaskState>> __when_expr_0 = bulkLoadJobFindTask(self, taskRange, jobId, self->ddId);
															#line 1659 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13158 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 1;
															#line 1659 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13163 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1703 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13192 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1688 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13205 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1689 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 13209 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1691 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskMonitorError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1696 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1697 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 13217 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1698 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 13221 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1660 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13237 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1662 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1663 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13243 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1666 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1673 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13255 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1675 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1676 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13261 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13267 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13272 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1660 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13281 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1662 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1663 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13287 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1665 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1666 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1673 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13299 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1675 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1676 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13305 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13311 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1680 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13316 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1681 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1686 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13402 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1681 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1686 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13413 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1652 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1653 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1654 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13520 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >, public FastAllocated<BulkLoadJobMonitorTaskActor>, public BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor> {
															#line 13525 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator new;
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17493301773314551296UL, 18259701207757301504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >;
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActor(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 13543 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor>(self, jobId, taskRange, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3763012194873441280UL, 18076434267272298240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobMonitorTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobMonitorTask( Reference<DataDistributor> const& self, UID const& jobId, KeyRange const& taskRange, Promise<Void> const& errorOut ) {
															#line 1648 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobMonitorTaskActor(self, jobId, taskRange, errorOut));
															#line 13577 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1705 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 13582 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PersistBulkLoadJobTaskCountActor>
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActorState {
															#line 13589 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActorState(Reference<DataDistributor> const& self) 
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1707 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1708 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 13600 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	~PersistBulkLoadJobTaskCountActorState() 
	{
		fdb_probe_actor_destroy("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1709 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1710 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = jobState.getJobId();
															#line 1712 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = jobState.getJobRange();
															#line 1713 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(jobState.getTaskCount().present());
															#line 1714 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			taskCount = jobState.getTaskCount().get();
															#line 1715 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentJobState = BulkLoadJobState();
															#line 1716 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 13629 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1745 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13652 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1718 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1718 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13674 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 1;
															#line 1718 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13679 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1742 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1742 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13716 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 4;
															#line 1742 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13721 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1719 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13736 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1720 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13740 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1721 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1727 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13746 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1729 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1731 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13767 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1719 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13776 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1720 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13780 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1721 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1727 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13786 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1729 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13790 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1731 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1732 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13807 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13893 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13898 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13909 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1733 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13914 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1734 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1735 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 14000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1734 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1735 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 14011 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1707 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1708 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1710 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1712 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1714 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	uint64_t taskCount;
															#line 1715 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 14194 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActor final : public Actor<Void>, public ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >, public FastAllocated<PersistBulkLoadJobTaskCountActor>, public PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor> {
															#line 14199 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator new;
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4926867012188914688UL, 9682850626425615616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >;
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActor(Reference<DataDistributor> const& self) 
															#line 14219 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10651365252717723648UL, 4800485421425282816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistBulkLoadJobTaskCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> persistBulkLoadJobTaskCount( Reference<DataDistributor> const& self ) {
															#line 1706 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PersistBulkLoadJobTaskCountActor(self));
															#line 14255 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1747 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Remove the bulkload job metadata from the range map metadata.
// Add the job metadata to the history metadata with the error message.
															#line 14262 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MoveErrorBulkLoadJobToHistoryActor>
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActorState {
															#line 14269 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActorState(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage(errorMessage),
															#line 1751 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1752 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1753 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentJobState()
															#line 14284 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	~MoveErrorBulkLoadJobToHistoryActorState() 
	{
		fdb_probe_actor_destroy("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1754 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1755 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = self->bulkLoadJobManager.get().jobState.getJobId();
															#line 1756 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 14303 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1771 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveErrorBulkLoadJobToHistoryActorState(); static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->destroy(); return 0; }
															#line 14326 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1758 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1758 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14348 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 1;
															#line 1758 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14353 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1768 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1768 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14390 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1768 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14395 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14412 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14417 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14428 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1759 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14433 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14519 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14524 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14535 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1760 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14540 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1763 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14630 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14635 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1763 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14650 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14655 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14741 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14746 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14757 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14762 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 1751 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1752 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1753 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 1755 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 15028 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActor final : public Actor<Void>, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >, public FastAllocated<MoveErrorBulkLoadJobToHistoryActor>, public MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor> {
															#line 15033 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator new;
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13030494147581051648UL, 12590394935791833088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >;
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActor(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 15055 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor>(self, errorMessage),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10076677940119608320UL, 17212560479686694656UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("moveErrorBulkLoadJobToHistory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> moveErrorBulkLoadJobToHistory( Reference<DataDistributor> const& self, std::string const& errorMessage ) {
															#line 1750 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MoveErrorBulkLoadJobToHistoryActor(self, errorMessage));
															#line 15093 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1773 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Download the job manifest file from the remoteJobManifestFilePath to the localJobManifestFilePath.
// Build the bulkload manifest range map based on the localJobManifestFilePath file content.
// For any failure, we mark the job metadata as error and move the metadata to the history.
															#line 15101 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FetchBulkLoadTaskManifestEntryMapActor>
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActorState {
															#line 15108 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActorState(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   localJobManifestFilePath(localJobManifestFilePath),
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   remoteJobManifestFilePath(remoteJobManifestFilePath)
															#line 15121 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	~FetchBulkLoadTaskManifestEntryMapActorState() 
	{
		fdb_probe_actor_destroy("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1781 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid() && self->bulkLoadJobManager.get().manifestEntryMap->empty());
															#line 1783 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginTime = now();
															#line 1784 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 15140 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1786 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!fileExists(abspath(localJobManifestFilePath)))
															#line 15144 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1787 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDBulkLoadJobManagerDownloadJobManifest", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath);
															#line 1791 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					StrictFuture<Void> __when_expr_0 = downloadBulkLoadJobManifestFile( jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath, self->ddId);
															#line 1791 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15152 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 1;
															#line 1791 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15157 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1839 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBulkLoadTaskManifestEntryMapActorState(); static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->destroy(); return 0; }
															#line 15191 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1815 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 15204 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1816 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 15208 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1818 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			err = e;
															#line 1819 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1825 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string errorMessage = "Failed to build job-manifest map with error code " + std::to_string(err.code()) + ". The remote file path is " + remoteJobManifestFilePath + ". The local file path is " + localJobManifestFilePath + ". The transport method is " + convertBulkLoadTransportMethodToString(jobTransportMethod) + ".";
															#line 1830 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = moveErrorBulkLoadJobToHistory(self, errorMessage);
															#line 1830 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15220 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 3;
															#line 1830 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15225 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = getBulkLoadJobFileManifestEntryFromJobManifestFile( localJobManifestFilePath, jobRange, self->ddId, self->bulkLoadJobManager.get().manifestEntryMap);
															#line 1802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15242 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 2;
															#line 1802 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15247 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1793 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15256 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1793 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15265 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1807 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1808 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15351 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1807 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1808 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15362 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2cont1(Void const& _,int loopDepth) 
	{
															#line 1831 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1837 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15461 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont1(Void && _,int loopDepth) 
	{
															#line 1831 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1837 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15471 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string localJobManifestFilePath;
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string remoteJobManifestFilePath;
															#line 1783 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1784 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1818 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Error err;
															#line 15564 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActor final : public Actor<Void>, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >, public FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>, public FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor> {
															#line 15569 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator new;
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13422488795375689728UL, 8756064887019231744UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >;
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActor(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 15588 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor>(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7119443845485172480UL, 13876208275314761984UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBulkLoadTaskManifestEntryMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> fetchBulkLoadTaskManifestEntryMap( Reference<DataDistributor> const& self, BulkLoadTransportMethod const& jobTransportMethod, std::string const& localJobManifestFilePath, std::string const& remoteJobManifestFilePath ) {
															#line 1777 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FetchBulkLoadTaskManifestEntryMapActor(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath));
															#line 15623 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1841 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 15628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadJobActor>
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActorState {
															#line 15635 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActorState(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1843 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntry()
															#line 15646 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1844 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1845 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1846 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1847 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors = std::vector<Future<Void>>();
															#line 1848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1849 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1853 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 15673 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1938 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(actors);
															#line 1938 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15698 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 1938 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15703 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, jobState.getJobRange().end));
															#line 1855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15722 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15727 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1935 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1935 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 1935 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15769 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1858 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15786 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 1855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		res = __res;
															#line 15795 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1929 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (beginKey == jobState.getJobRange().end)
															#line 15874 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1931 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.get().allTaskSubmitted = true;
															#line 15878 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1858 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < res.size() - 1))
															#line 15896 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1862 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!res[i].value.empty())
															#line 15902 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1863 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			task = decodeBulkLoadTaskState(res[i].value);
															#line 1864 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (task.isValid())
															#line 15908 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1865 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.getJobId() != self->bulkLoadJobManager.get().jobState.getJobId())
															#line 15912 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1866 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					return a_body1loopBody1Catch1(bulkload_task_outdated(), std::max(0, loopDepth - 1));
															#line 15916 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
															#line 1869 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(self->bulkLoadJobManager.get().manifestEntryMap->find(task.getRange().begin) != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1871 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.onAnyPhase( { BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }))
															#line 15922 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1873 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(task.getRange().end == res[i + 1].key);
															#line 1874 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					beginKey = task.getRange().end;
															#line 15928 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					return a_body1loopBody1cont2continue1(loopDepth); // continue
				}
				else
				{
															#line 1877 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (task.onAnyPhase({ BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }))
															#line 15935 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 1882 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (self->bulkLoadJobManager.get().allTaskSubmitted)
															#line 15939 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 1884 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							;
															#line 15943 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
							loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
						}
						else
						{
							loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);
						}
					}
					else
					{
															#line 1898 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						UNREACHABLE();
															#line 15955 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = a_body1loopBody1cont2loopBody1cont7(loopDepth);
					}
				}
			}
			else
			{
				loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2continue1(int loopDepth) 
	{
															#line 1858 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15989 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1906 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i].key);
															#line 1907 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont8(int loopDepth) 
	{
															#line 1894 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(task.getRange().end == res[i + 1].key);
															#line 1895 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = task.getRange().end;
															#line 16029 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont2continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont9(int loopDepth) 
	{
															#line 1891 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back( bulkLoadJobMonitorTask(self, task.getJobId(), task.getRange(), errorOut));
															#line 16038 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 1885 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadParallelismLimitor.canStart())
															#line 16054 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1888 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->bulkLoadParallelismLimitor.waitUntilCounterChanged();
															#line 1888 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16062 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 1888 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16067 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2loopBody1cont13(int loopDepth) 
	{
															#line 1927 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i + 1].key);
															#line 1858 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16178 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1907 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < res[i + 1].key))
															#line 16194 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1908 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1909 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;manifestEntries.size() < SERVER_KNOBS->MANIFEST_COUNT_MAX_PER_BULKLOAD_TASK && beginKey < res[i + 1].key;) {
															#line 1911 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->bulkLoadJobManager.get().manifestEntryMap->find(beginKey);
															#line 1912 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(it != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1913 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntry = it->second;
															#line 1914 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntries.push_back(manifestEntry);
															#line 1915 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = manifestEntry.getEndKey();
															#line 16212 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1917 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!manifestEntries.empty());
															#line 1918 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobNewTask(self, jobState.getJobId(), jobState.getJobRoot(), jobState.getTransportMethod(), self->bulkLoadJobManager.get().manifestLocalTempFolder, manifestEntries, errorOut));
															#line 1925 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_TASK_SUBMISSION_INTERVAL_SEC);
															#line 1925 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16222 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 1925 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16227 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1939 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16436 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1939 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16448 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1843 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFileManifestEntry manifestEntry;
															#line 1845 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1846 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1847 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> actors;
															#line 1848 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1849 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1855 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult res;
															#line 1857 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1863 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState task;
															#line 16553 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >, public ActorCallback< ScheduleBulkLoadJobActor, 1, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 2, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 3, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 4, Void >, public FastAllocated<ScheduleBulkLoadJobActor>, public ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor> {
															#line 16558 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadJobActor>::operator new;
	using FastAllocated<ScheduleBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3605065204158547712UL, 2621083410266376192UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 4, Void >;
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActor(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 16579 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor>(self, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4684338048225246976UL, 14035353769110992896UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadJob( Reference<DataDistributor> const& self, Promise<Void> const& errorOut ) {
															#line 1842 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadJobActor(self, errorOut));
															#line 16616 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1941 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 16621 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadTaskCompleteOrErrorActor>
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActorState {
															#line 16628 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self) 
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1943 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1944 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 16639 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1945 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1947 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1948 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 1949 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 1950 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = KeyRange();
															#line 1951 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 1952 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 16668 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1997 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16691 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(true);
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1952 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 16710 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1954 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 1955 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1956 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobState.getJobId());
															#line 1956 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16723 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1956 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16728 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1994 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1994 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16765 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1994 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16770 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16787 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16792 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16803 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1957 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16808 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1958 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1959 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1960 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1961 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16898 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1963 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16902 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1967 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16910 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1968 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1973 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16916 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1975 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16920 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1976 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16924 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 1983 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16929 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1984 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1989 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16935 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1992 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16944 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1958 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1959 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1960 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1961 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16959 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1963 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16963 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1967 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16971 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1968 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1973 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16977 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1975 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16981 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1976 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16985 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 1983 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16990 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1984 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1989 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16996 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1992 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 17005 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1943 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1944 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1946 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1947 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1948 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1949 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 1950 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1951 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 17203 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActor final : public Actor<bool>, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>, public CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor> {
															#line 17208 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12903140896536481280UL, 1377355127177728256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >;
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActor(Reference<DataDistributor> const& self) 
															#line 17227 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<bool>(),
		   CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13835388531052272896UL, 2426800946393579264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<bool> checkBulkLoadTaskCompleteOrError( Reference<DataDistributor> const& self ) {
															#line 1942 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<bool>(new CheckBulkLoadTaskCompleteOrErrorActor(self));
															#line 17262 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1999 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 17267 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FinalizeBulkLoadJobActor>
															#line 2000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActorState {
															#line 17274 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActorState(Reference<DataDistributor> const& self) 
															#line 2000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2000 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2001 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 2002 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 17285 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~FinalizeBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2003 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2004 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 2005 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 2006 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 2007 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = Optional<Key>();
															#line 2008 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 2009 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 2010 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobCompleteRange = KeyRange();
															#line 2011 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = false;
															#line 2012 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			i = 0;
															#line 2013 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			allFinish = false;
															#line 2014 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 17322 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2093 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FinalizeBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinalizeBulkLoadJobActorState(); static_cast<FinalizeBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 17345 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FinalizeBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2014 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 17364 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2016 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.reset();
															#line 2017 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 2018 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId());
															#line 2018 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17377 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 2018 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17382 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2090 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 2090 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17419 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 9;
															#line 2090 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17424 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
															#line 2019 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17443 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17448 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && currentJobState,int loopDepth) 
	{
															#line 2019 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17461 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2020 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17466 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(currentJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(currentJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2021 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2022 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17552 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2021 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2022 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17563 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 2060 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (hasError)
															#line 17647 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2061 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setErrorPhase("BulkLoadTask Error.");
															#line 17651 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 2063 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setCompletePhase();
															#line 17657 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 2065 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(lastKey.present());
															#line 2066 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobCompleteRange = KeyRangeRef(jobState.getJobRange().begin, lastKey.get());
															#line 2067 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 2067 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17667 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 2067 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17672 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2022 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < bulkLoadTaskResult.size() - 1))
															#line 17688 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2023 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 2024 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 2025 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!existTask.isValid())
															#line 17698 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2026 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17702 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1loopBody1cont4continue1(loopDepth); // continue
		}
															#line 2029 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getRange() == KeyRangeRef(bulkLoadTaskResult[i].key, bulkLoadTaskResult[i + 1].key));
															#line 2034 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getJobId() == jobState.getJobId());
															#line 2035 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (existTask.phase == BulkLoadPhase::Error)
															#line 17711 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2036 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerStopClearMetadata", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2043 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = true;
															#line 17717 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);
		}
		else
		{
															#line 2045 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete && existTask.phase != BulkLoadPhase::Acknowledged)
															#line 17724 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2046 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerWrongTaskPhase", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2052 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 17730 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2055 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = setBulkLoadFinalizeTransaction(&tr, existTask.getRange(), existTask.getTaskId());
															#line 2055 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17736 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 2055 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17741 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4continue1(int loopDepth) 
	{
															#line 2022 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 17764 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
															#line 2057 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17773 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2068 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17871 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2071 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17879 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17884 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17893 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17898 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2068 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2069 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17910 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2071 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17918 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2072 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17923 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17932 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2075 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17937 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr.commit();
															#line 2077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18024 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_7.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2077 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18029 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18040 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18045 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18056 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2073 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18061 "/codebuild/output/src3173567066/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif