/*
 * Copyright 2018-2020 Yury Gribov
 *
 * The MIT License (MIT)
 *
 * Use of this source code is governed by MIT license that can be
 * found in the LICENSE.txt file.
 */

#include <dlfcn.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <mutex>

// Sanity check for ARM to avoid puzzling runtime crashes
#ifdef __arm__
# if defined __thumb__ && ! defined __THUMB_INTERWORK__
#   error "ARM trampolines need -mthumb-interwork to work in Thumb mode"
# endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

#define CHECK(cond, fmt, ...) do { \
    if(!(cond)) { \
      fprintf(stderr, "implib-gen: libfdb_c.so: " fmt "\n", ##__VA_ARGS__); \
      abort(); \
    } \
  } while(0)

#define CALL_USER_CALLBACK 1

static void *lib_handle;

static void *load_library() {
  if(lib_handle)
    return lib_handle;

  // TODO: dlopen and users callback must be protected w/ critical section (to avoid dlopening lib twice)
#if CALL_USER_CALLBACK
  extern void *fdb_shim_dlopen_callback(const char *lib_name);
  lib_handle = fdb_shim_dlopen_callback("libfdb_c.so");
  CHECK(lib_handle, "callback 'fdb_shim_dlopen_callback' failed to load library");
#else
  lib_handle = dlopen("libfdb_c.so", RTLD_LAZY | RTLD_GLOBAL);
  CHECK(lib_handle, "failed to load library: %s", dlerror());
#endif

  return lib_handle;
}

static void __attribute__((destructor)) unload_lib() {
  if(lib_handle)
    dlclose(lib_handle);
}

// TODO: convert to single 0-separated string
static const char *const sym_names[] = {
  "fdb_add_network_thread_completion_hook",
  "fdb_cluster_create_database",
  "fdb_cluster_destroy",
  "fdb_cluster_set_option",
  "fdb_create_cluster",
  "fdb_create_database",
  "fdb_create_database_from_connection_string",
  "fdb_database_blobbify_range",
  "fdb_database_blobbify_range_blocking",
  "fdb_database_create_shared_state",
  "fdb_database_create_snapshot",
  "fdb_database_create_transaction",
  "fdb_database_destroy",
  "fdb_database_flush_blob_range",
  "fdb_database_force_recovery_with_data_loss",
  "fdb_database_get_client_status",
  "fdb_database_get_main_thread_busyness",
  "fdb_database_get_server_protocol",
  "fdb_database_list_blobbified_ranges",
  "fdb_database_open_tenant",
  "fdb_database_purge_blob_granules",
  "fdb_database_reboot_worker",
  "fdb_database_set_option",
  "fdb_database_set_shared_state",
  "fdb_database_unblobbify_range",
  "fdb_database_verify_blob_range",
  "fdb_database_wait_purge_granules_complete",
  "fdb_error_predicate",
  "fdb_future_block_until_ready",
  "fdb_future_cancel",
  "fdb_future_destroy",
  "fdb_future_get_bool",
  "fdb_future_get_cluster",
  "fdb_future_get_database",
  "fdb_future_get_double",
  "fdb_future_get_error",
  "fdb_future_get_granule_summary_array",
  "fdb_future_get_int64",
  "fdb_future_get_key",
  "fdb_future_get_key_array",
  "fdb_future_get_keyrange_array",
  "fdb_future_get_keyvalue_array",
  "fdb_future_get_mappedkeyvalue_array",
  "fdb_future_get_shared_state",
  "fdb_future_get_string_array",
  "fdb_future_get_uint64",
  "fdb_future_get_value",
  "fdb_future_get_version",
  "fdb_future_is_error",
  "fdb_future_is_ready",
  "fdb_future_readbg_get_descriptions",
  "fdb_future_release_memory",
  "fdb_future_set_callback",
  "fdb_get_client_version",
  "fdb_get_error",
  "fdb_get_max_api_version",
  "fdb_network_set_option",
  "fdb_readbg_parse_delta_file",
  "fdb_readbg_parse_snapshot_file",
  "fdb_result_destroy",
  "fdb_result_get_bg_mutations_array",
  "fdb_result_get_keyvalue_array",
  "fdb_run_network",
  "fdb_select_api_version_impl",
  "fdb_setup_network",
  "fdb_stop_network",
  "fdb_tenant_blobbify_range",
  "fdb_tenant_blobbify_range_blocking",
  "fdb_tenant_create_transaction",
  "fdb_tenant_destroy",
  "fdb_tenant_flush_blob_range",
  "fdb_tenant_get_id",
  "fdb_tenant_list_blobbified_ranges",
  "fdb_tenant_purge_blob_granules",
  "fdb_tenant_unblobbify_range",
  "fdb_tenant_verify_blob_range",
  "fdb_tenant_wait_purge_granules_complete",
  "fdb_transaction_add_conflict_range",
  "fdb_transaction_atomic_op",
  "fdb_transaction_cancel",
  "fdb_transaction_clear",
  "fdb_transaction_clear_range",
  "fdb_transaction_commit",
  "fdb_transaction_destroy",
  "fdb_transaction_get",
  "fdb_transaction_get_addresses_for_key",
  "fdb_transaction_get_approximate_size",
  "fdb_transaction_get_blob_granule_ranges",
  "fdb_transaction_get_committed_version",
  "fdb_transaction_get_estimated_range_size_bytes",
  "fdb_transaction_get_key",
  "fdb_transaction_get_mapped_range",
  "fdb_transaction_get_range",
  "fdb_transaction_get_range_selector",
  "fdb_transaction_get_range_split_points",
  "fdb_transaction_get_read_version",
  "fdb_transaction_get_tag_throttled_duration",
  "fdb_transaction_get_total_cost",
  "fdb_transaction_get_versionstamp",
  "fdb_transaction_on_error",
  "fdb_transaction_read_blob_granules",
  "fdb_transaction_read_blob_granules_description",
  "fdb_transaction_read_blob_granules_finish",
  "fdb_transaction_read_blob_granules_start",
  "fdb_transaction_reset",
  "fdb_transaction_set",
  "fdb_transaction_set_option",
  "fdb_transaction_set_read_version",
  "fdb_transaction_summarize_blob_granules",
  "fdb_transaction_watch",
  "fdb_use_future_protocol_version",
  0
};

extern void *_libfdb_c_so_tramp_table[];

// Load library and resolve all symbols
static void load_and_resolve(void) {
  static std::mutex load_mutex;
  static int is_loaded = false;

  std::unique_lock<std::mutex> lock(load_mutex);
  if (is_loaded)
    return;

  void *h = 0;
  h = load_library();

  size_t i;
  for(i = 0; i + 1 < sizeof(sym_names) / sizeof(sym_names[0]); ++i)
    // Resolving some of the symbols may fail. We ignore it, because if we are loading 
    // a library of an older version it may lack certain functions
    _libfdb_c_so_tramp_table[i] = dlsym(h, sym_names[i]);

  is_loaded = true;
}

// The function is called if the table entry for the symbol is not set.
// In that case we load the library and try to resolve all symbols if that was not done yet.
// If the table entry is still missing, then the symbol is not available in the loaded library,
// which is a fatal error on which we immediately exit the process.
void _libfdb_c_so_tramp_resolve(int i) {
  assert((unsigned)i + 1 < sizeof(sym_names) / sizeof(sym_names[0]));
  load_and_resolve();
  CHECK(_libfdb_c_so_tramp_table[i], "failed to resolve symbol '%s'", sym_names[i]);
}

#ifdef __cplusplus
}  // extern "C"
#endif
