#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
/*
 * GrvProxyServer.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/ClientKnobs.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/Notified.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/Tuple.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/GrvProxyInterface.h"
#include "fdbclient/VersionVector.h"
#include "fdbserver/GrvProxyTagThrottler.h"
#include "fdbserver/GrvTransactionRateInfo.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/LogSystemDiskQueueAdapter.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbrpc/sim_validation.h"
#include "flow/IRandom.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/actorcompiler.h" // This must be the last #include.
#include "flow/genericactors.actor.h"

struct GrvProxyStats {
	CounterCollection cc;
	Counter txnRequestIn;
	Counter txnRequestOut;
	Counter txnRequestErrors;
	Counter txnStartIn;
	Counter txnStartOut;
	Counter txnStartBatch;
	Counter txnSystemPriorityStartIn;
	Counter txnSystemPriorityStartOut;
	Counter txnBatchPriorityStartIn;
	Counter txnBatchPriorityStartOut;
	Counter txnDefaultPriorityStartIn;
	Counter txnDefaultPriorityStartOut;
	Counter txnTagThrottlerIn;
	Counter txnTagThrottlerOut;
	Counter txnThrottled;
	Counter updatesFromRatekeeper;
	Counter leaseTimeouts;
	int systemGRVQueueSize;
	int defaultGRVQueueSize;
	int batchGRVQueueSize;
	int tagThrottlerGRVQueueSize;
	double transactionRateAllowed;
	double batchTransactionRateAllowed;
	double transactionLimit;
	double batchTransactionLimit;
	// how much of the GRV requests queue was processed in one attempt to hand out read version.
	double percentageOfDefaultGRVQueueProcessed;
	double percentageOfBatchGRVQueueProcessed;

	bool lastBatchQueueThrottled;
	bool lastDefaultQueueThrottled;
	double batchThrottleStartTime;
	double defaultThrottleStartTime;

	LatencySample defaultTxnGRVTimeInQueue;
	LatencySample batchTxnGRVTimeInQueue;

	// These latency bands and samples ignore latency injected by the GrvProxyTagThrottler
	LatencyBands grvLatencyBands;
	LatencySample grvLatencySample; // GRV latency metric sample of default priority
	LatencySample grvBatchLatencySample; // GRV latency metric sample of batched priority

	Future<Void> logger;

	int recentRequests;
	Deque<int> requestBuckets;
	double lastBucketBegin;
	double bucketInterval;
	Reference<Histogram> grvConfirmEpochLiveDist;
	Reference<Histogram> grvGetCommittedVersionRpcDist;

	void updateRequestBuckets() {
		while (now() - lastBucketBegin > bucketInterval) {
			lastBucketBegin += bucketInterval;
			recentRequests -= requestBuckets.front();
			requestBuckets.pop_front();
			requestBuckets.push_back(0);
		}
	}

	void addRequest(int transactionCount) {
		updateRequestBuckets();
		recentRequests += transactionCount;
		requestBuckets.back() += transactionCount;
	}

	int getRecentRequests() {
		updateRequestBuckets();
		return recentRequests /
		       (FLOW_KNOBS->BASIC_LOAD_BALANCE_UPDATE_RATE - (lastBucketBegin + bucketInterval - now()));
	}

	void update(GrvProxyTagThrottler::ReleaseTransactionsResult const& releaseStats) {
		auto const totalReleasedRequests =
		    releaseStats.batchPriorityRequestsReleased + releaseStats.defaultPriorityRequestsReleased;
		auto const totalReleasedTransactions =
		    releaseStats.batchPriorityTransactionsReleased + releaseStats.defaultPriorityTransactionsReleased;

		txnRequestIn += totalReleasedRequests;
		txnStartIn += totalReleasedTransactions;
		txnBatchPriorityStartIn += releaseStats.batchPriorityTransactionsReleased;
		txnDefaultPriorityStartIn += releaseStats.defaultPriorityTransactionsReleased;
		batchGRVQueueSize += releaseStats.batchPriorityRequestsReleased;
		defaultGRVQueueSize += releaseStats.defaultPriorityRequestsReleased;
		txnRequestErrors += releaseStats.rejectedRequests;
		txnTagThrottlerOut += totalReleasedTransactions;
		tagThrottlerGRVQueueSize -= totalReleasedRequests;
	}

	// Current stats maintained for a given grv proxy server
	explicit GrvProxyStats(UID id)
	  : cc("GrvProxyStats", id.toString()),

	    txnRequestIn("TxnRequestIn", cc), txnRequestOut("TxnRequestOut", cc), txnRequestErrors("TxnRequestErrors", cc),
	    txnStartIn("TxnStartIn", cc), txnStartOut("TxnStartOut", cc), txnStartBatch("TxnStartBatch", cc),
	    txnSystemPriorityStartIn("TxnSystemPriorityStartIn", cc),
	    txnSystemPriorityStartOut("TxnSystemPriorityStartOut", cc),
	    txnBatchPriorityStartIn("TxnBatchPriorityStartIn", cc),
	    txnBatchPriorityStartOut("TxnBatchPriorityStartOut", cc),
	    txnDefaultPriorityStartIn("TxnDefaultPriorityStartIn", cc),
	    txnDefaultPriorityStartOut("TxnDefaultPriorityStartOut", cc), txnTagThrottlerIn("TxnTagThrottlerIn", cc),
	    txnTagThrottlerOut("TxnTagThrottlerOut", cc), txnThrottled("TxnThrottled", cc),
	    updatesFromRatekeeper("UpdatesFromRatekeeper", cc), leaseTimeouts("LeaseTimeouts", cc), systemGRVQueueSize(0),
	    defaultGRVQueueSize(0), batchGRVQueueSize(0), tagThrottlerGRVQueueSize(0), transactionRateAllowed(0),
	    batchTransactionRateAllowed(0), transactionLimit(0), batchTransactionLimit(0),
	    percentageOfDefaultGRVQueueProcessed(0), percentageOfBatchGRVQueueProcessed(0), lastBatchQueueThrottled(false),
	    lastDefaultQueueThrottled(false), batchThrottleStartTime(0.0), defaultThrottleStartTime(0.0),
	    defaultTxnGRVTimeInQueue("DefaultTxnGRVTimeInQueue",
	                             id,
	                             SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                             SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    batchTxnGRVTimeInQueue("BatchTxnGRVTimeInQueue",
	                           id,
	                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    grvLatencyBands("GRVLatencyBands", id, SERVER_KNOBS->STORAGE_LOGGING_DELAY),
	    grvLatencySample("GRVLatencyMetrics",
	                     id,
	                     SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                     SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    grvBatchLatencySample("GRVBatchLatencyMetrics",
	                          id,
	                          SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                          SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    recentRequests(0), lastBucketBegin(now()),
	    bucketInterval(FLOW_KNOBS->BASIC_LOAD_BALANCE_UPDATE_RATE / FLOW_KNOBS->BASIC_LOAD_BALANCE_BUCKETS),
	    grvConfirmEpochLiveDist(
	        Histogram::getHistogram("GrvProxy"_sr, "GrvConfirmEpochLive"_sr, Histogram::Unit::milliseconds)),
	    grvGetCommittedVersionRpcDist(
	        Histogram::getHistogram("GrvProxy"_sr, "GrvGetCommittedVersionRpc"_sr, Histogram::Unit::milliseconds)) {
		// The rate at which the limit(budget) is allowed to grow.
		specialCounter(cc, "SystemGRVQueueSize", [this]() { return this->systemGRVQueueSize; });
		specialCounter(cc, "DefaultGRVQueueSize", [this]() { return this->defaultGRVQueueSize; });
		specialCounter(cc, "BatchGRVQueueSize", [this]() { return this->batchGRVQueueSize; });
		specialCounter(cc, "TagThrottlerGRVQueueSize", [this]() { return this->tagThrottlerGRVQueueSize; });
		specialCounter(
		    cc, "SystemAndDefaultTxnRateAllowed", [this]() { return int64_t(this->transactionRateAllowed); });
		specialCounter(
		    cc, "BatchTransactionRateAllowed", [this]() { return int64_t(this->batchTransactionRateAllowed); });
		specialCounter(cc, "SystemAndDefaultTxnLimit", [this]() { return int64_t(this->transactionLimit); });
		specialCounter(cc, "BatchTransactionLimit", [this]() { return int64_t(this->batchTransactionLimit); });
		specialCounter(cc, "PercentageOfDefaultGRVQueueProcessed", [this]() {
			return int64_t(100 * this->percentageOfDefaultGRVQueueProcessed);
		});
		specialCounter(cc, "PercentageOfBatchGRVQueueProcessed", [this]() {
			return int64_t(100 * this->percentageOfBatchGRVQueueProcessed);
		});

		logger = cc.traceCounters("GrvProxyMetrics", id, SERVER_KNOBS->WORKER_LOGGING_INTERVAL, "GrvProxyMetrics");
		for (int i = 0; i < FLOW_KNOBS->BASIC_LOAD_BALANCE_BUCKETS; i++) {
			requestBuckets.push_back(0);
		}
	}
};

struct GrvProxyData {
	GrvProxyInterface proxy;
	UID dbgid;

	GrvProxyStats stats;
	MasterInterface master;
	PublicRequestStream<GetReadVersionRequest> getConsistentReadVersion;
	Reference<ILogSystem> logSystem;

	Database cx;
	Reference<AsyncVar<ServerDBInfo> const> db;

	Optional<LatencyBandConfig> latencyBandConfig;
	double lastStartCommit;
	double lastCommitLatency;
	std::unique_ptr<LatencySample> versionVectorSizeOnGRVReply;
	int updateCommitRequests;
	NotifiedDouble lastCommitTime;

	Version version;
	Version minKnownCommittedVersion; // we should ask master for this version.

	GrvProxyTagThrottler tagThrottler;

	// Cache of the latest commit versions of storage servers.
	VersionVector ssVersionVectorCache;

	void updateLatencyBandConfig(Optional<LatencyBandConfig> newLatencyBandConfig) {
		if (newLatencyBandConfig.present() != latencyBandConfig.present() ||
		    (newLatencyBandConfig.present() &&
		     newLatencyBandConfig.get().grvConfig != latencyBandConfig.get().grvConfig)) {
			TraceEvent("LatencyBandGrvUpdatingConfig").detail("Present", newLatencyBandConfig.present());
			stats.grvLatencyBands.clearBands();
			if (newLatencyBandConfig.present()) {
				for (auto band : newLatencyBandConfig.get().grvConfig.bands) {
					stats.grvLatencyBands.addThreshold(band);
					tagThrottler.addLatencyBandThreshold(band);
				}
			}
		}

		latencyBandConfig = newLatencyBandConfig;
	}

	GrvProxyData(UID dbgid,
	             MasterInterface master,
	             PublicRequestStream<GetReadVersionRequest> getConsistentReadVersion,
	             Reference<AsyncVar<ServerDBInfo> const> db)
	  : dbgid(dbgid), stats(dbgid), master(master), getConsistentReadVersion(getConsistentReadVersion),
	    cx(openDBOnServer(db, TaskPriority::DefaultEndpoint, LockAware::True)), db(db), lastStartCommit(0),
	    lastCommitLatency(SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION), updateCommitRequests(0), lastCommitTime(0),
	    version(0), minKnownCommittedVersion(invalidVersion),
	    tagThrottler(CLIENT_KNOBS->PROXY_MAX_TAG_THROTTLE_DURATION) {
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR) {
			versionVectorSizeOnGRVReply =
			    std::make_unique<LatencySample>("VersionVectorSizeOnGRVReply",
			                                    dbgid,
			                                    SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
			                                    SERVER_KNOBS->LATENCY_SKETCH_ACCURACY);
		}
	}
};

															#line 265 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via healthMetricsRequestServer()
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class HealthMetricsRequestServerActor>
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class HealthMetricsRequestServerActorState {
															#line 272 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	HealthMetricsRequestServerActorState(GrvProxyInterface const& grvProxy,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply) 
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : grvProxy(grvProxy),
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   healthMetricsReply(healthMetricsReply),
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   detailedHealthMetricsReply(detailedHealthMetricsReply)
															#line 283 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this));

	}
	~HealthMetricsRequestServerActorState() 
	{
		fdb_probe_actor_destroy("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 266 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 298 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HealthMetricsRequestServerActorState();
		static_cast<HealthMetricsRequestServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 268 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		auto __when_expr_0 = grvProxy.getHealthMetrics.getFuture();
															#line 328 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<GetHealthMetricsRequest>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<GetHealthMetricsRequest>>::value, "invalid type");
															#line 267 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<HealthMetricsRequestServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 332 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
		static_cast<HealthMetricsRequestServerActor*>(this)->actor_wait_state = 1;
															#line 268 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >*>(static_cast<HealthMetricsRequestServerActor*>(this)));
															#line 337 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(GetHealthMetricsRequest const& req,int loopDepth) 
	{
															#line 269 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (req.detailed)
															#line 352 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 270 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			req.reply.send(*detailedHealthMetricsReply);
															#line 356 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 272 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			req.reply.send(*healthMetricsReply);
															#line 362 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(GetHealthMetricsRequest && req,int loopDepth) 
	{
															#line 269 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (req.detailed)
															#line 372 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 270 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			req.reply.send(*detailedHealthMetricsReply);
															#line 376 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 272 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			req.reply.send(*healthMetricsReply);
															#line 382 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HealthMetricsRequestServerActor*>(this)->actor_wait_state > 0) static_cast<HealthMetricsRequestServerActor*>(this)->actor_wait_state = 0;
		static_cast<HealthMetricsRequestServerActor*>(this)->ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >*,GetHealthMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >*,GetHealthMetricsRequest && value) 
	{
		fdb_probe_actor_enter("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface grvProxy;
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* healthMetricsReply;
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* detailedHealthMetricsReply;
															#line 457 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via healthMetricsRequestServer()
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class HealthMetricsRequestServerActor final : public Actor<Void>, public ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >, public FastAllocated<HealthMetricsRequestServerActor>, public HealthMetricsRequestServerActorState<HealthMetricsRequestServerActor> {
															#line 462 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<HealthMetricsRequestServerActor>::operator new;
	using FastAllocated<HealthMetricsRequestServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5715751466359639808UL, 7312155759107078400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >;
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	HealthMetricsRequestServerActor(GrvProxyInterface const& grvProxy,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply) 
															#line 479 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   HealthMetricsRequestServerActorState<HealthMetricsRequestServerActor>(grvProxy, healthMetricsReply, detailedHealthMetricsReply),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15006378439154462208UL, 8394861280111908096UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("healthMetricsRequestServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("healthMetricsRequestServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< HealthMetricsRequestServerActor, 0, GetHealthMetricsRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> healthMetricsRequestServer( GrvProxyInterface const& grvProxy, GetHealthMetricsReply* const& healthMetricsReply, GetHealthMetricsReply* const& detailedHealthMetricsReply ) {
															#line 263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new HealthMetricsRequestServerActor(grvProxy, healthMetricsReply, detailedHealthMetricsReply));
															#line 512 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 277 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Older FDB versions used different keys for client profiling data. This
// function performs a one-time migration of data in these keys to the new
// global configuration key space.
															#line 520 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via globalConfigMigrate()
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GlobalConfigMigrateActor>
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigMigrateActorState {
															#line 527 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigMigrateActorState(GrvProxyData* const& grvProxyData) 
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : grvProxyData(grvProxyData),
															#line 282 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   migratedKey("\xff\x02/fdbClientInfo/migrated/"_sr),
															#line 283 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   tr(makeReference<ReadYourWritesTransaction>(grvProxyData->cx))
															#line 538 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("globalConfigMigrate", reinterpret_cast<unsigned long>(this));

	}
	~GlobalConfigMigrateActorState() 
	{
		fdb_probe_actor_destroy("globalConfigMigrate", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 285 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				;
															#line 554 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GlobalConfigMigrateActorState();
		static_cast<GlobalConfigMigrateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 329 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GlobalConfigMigrateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GlobalConfigMigrateActorState(); static_cast<GlobalConfigMigrateActor*>(this)->destroy(); return 0; }
															#line 583 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GlobalConfigMigrateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GlobalConfigMigrateActorState();
		static_cast<GlobalConfigMigrateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 327 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent(SevWarnAlways, "GlobalConfigMigrationError").error(e);
															#line 596 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 286 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 624 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		try {
															#line 289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(migratedKey);
															#line 289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 630 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 1;
															#line 289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >*>(static_cast<GlobalConfigMigrateActor*>(this)));
															#line 635 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 321 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent(SevInfo, "GlobalConfigRetryableMigrationError").errorUnsuppressed(e).suppressFor(1.0);
															#line 322 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr->onError(e);
															#line 322 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 674 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 5;
															#line 322 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigMigrateActor, 4, Void >*>(static_cast<GlobalConfigMigrateActor*>(this)));
															#line 679 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 290 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (migrated.present())
															#line 694 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 295 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_1 = tr->get(Key("\xff\x02/fdbClientInfo/client_txn_sample_rate/"_sr));
															#line 295 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 702 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 2;
															#line 295 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >*>(static_cast<GlobalConfigMigrateActor*>(this)));
															#line 707 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& __migrated,int loopDepth) 
	{
															#line 289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		migrated = __migrated;
															#line 716 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && __migrated,int loopDepth) 
	{
		migrated = std::move(__migrated);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigMigrateActor*>(this)->ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = tr->get(Key("\xff\x02/fdbClientInfo/client_txn_size_limit/"_sr));
															#line 297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 797 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 3;
															#line 297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >*>(static_cast<GlobalConfigMigrateActor*>(this)));
															#line 802 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> const& __sampleRate,int loopDepth) 
	{
															#line 295 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		sampleRate = __sampleRate;
															#line 811 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> && __sampleRate,int loopDepth) 
	{
		sampleRate = std::move(__sampleRate);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigMigrateActor*>(this)->ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 300 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 302 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		tr->set(migratedKey.contents(), "1"_sr);
															#line 303 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (sampleRate.present())
															#line 894 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 304 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			const double sampleRateDbl = BinaryReader::fromStringRef<double>(sampleRate.get().contents(), Unversioned());
															#line 306 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Tuple rate = Tuple::makeTuple(sampleRateDbl);
															#line 307 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			tr->set(GlobalConfig::prefixedKey(fdbClientInfoTxnSampleRate), rate.pack());
															#line 902 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 309 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (sizeLimit.present())
															#line 906 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 310 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			const int64_t sizeLimitInt = BinaryReader::fromStringRef<int64_t>(sizeLimit.get().contents(), Unversioned());
															#line 312 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Tuple size = Tuple::makeTuple(sizeLimitInt);
															#line 313 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			tr->set(GlobalConfig::prefixedKey(fdbClientInfoTxnSizeLimit), size.pack());
															#line 914 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 316 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 316 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 920 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 4;
															#line 316 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigMigrateActor, 3, Void >*>(static_cast<GlobalConfigMigrateActor*>(this)));
															#line 925 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Optional<Value> const& __sizeLimit,int loopDepth) 
	{
															#line 297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		sizeLimit = __sizeLimit;
															#line 934 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Optional<Value> && __sizeLimit,int loopDepth) 
	{
		sizeLimit = std::move(__sizeLimit);
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigMigrateActor*>(this)->ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigMigrateActor*>(this)->ActorCallback< GlobalConfigMigrateActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GlobalConfigMigrateActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigMigrateActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigMigrateActor*>(this)->ActorCallback< GlobalConfigMigrateActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GlobalConfigMigrateActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GlobalConfigMigrateActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 282 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Key migratedKey;
															#line 283 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<Value> migrated;
															#line 295 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<Value> sampleRate;
															#line 297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<Value> sizeLimit;
															#line 1208 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via globalConfigMigrate()
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigMigrateActor final : public Actor<Void>, public ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >, public ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >, public ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >, public ActorCallback< GlobalConfigMigrateActor, 3, Void >, public ActorCallback< GlobalConfigMigrateActor, 4, Void >, public FastAllocated<GlobalConfigMigrateActor>, public GlobalConfigMigrateActorState<GlobalConfigMigrateActor> {
															#line 1213 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GlobalConfigMigrateActor>::operator new;
	using FastAllocated<GlobalConfigMigrateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7337305234134121984UL, 13977993321230647040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >;
friend struct ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >;
friend struct ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >;
friend struct ActorCallback< GlobalConfigMigrateActor, 3, Void >;
friend struct ActorCallback< GlobalConfigMigrateActor, 4, Void >;
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigMigrateActor(GrvProxyData* const& grvProxyData) 
															#line 1234 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GlobalConfigMigrateActorState<GlobalConfigMigrateActor>(grvProxyData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("globalConfigMigrate", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(154511452753742592UL, 13598310903932054784UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigMigrateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("globalConfigMigrate");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("globalConfigMigrate", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GlobalConfigMigrateActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GlobalConfigMigrateActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GlobalConfigMigrateActor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GlobalConfigMigrateActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GlobalConfigMigrateActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> globalConfigMigrate( GrvProxyData* const& grvProxyData ) {
															#line 281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GlobalConfigMigrateActor(grvProxyData));
															#line 1271 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 331 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Periodically refresh local copy of global configuration.
															#line 1277 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via globalConfigRefresh()
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GlobalConfigRefreshActor>
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigRefreshActorState {
															#line 1284 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigRefreshActorState(GrvProxyData* const& grvProxyData,Version* const& cachedVersion,RangeResult* const& cachedData) 
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : grvProxyData(grvProxyData),
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   cachedVersion(cachedVersion),
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   cachedData(cachedData),
															#line 334 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   tr(makeReference<ReadYourWritesTransaction>(grvProxyData->cx))
															#line 1297 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("globalConfigRefresh", reinterpret_cast<unsigned long>(this));

	}
	~GlobalConfigRefreshActorState() 
	{
		fdb_probe_actor_destroy("globalConfigRefresh", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 335 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 1312 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GlobalConfigRefreshActorState();
		static_cast<GlobalConfigRefreshActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 337 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 338 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			globalConfigVersionFuture = tr->get(globalConfigVersionKey);
															#line 339 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			tmpCachedDataFuture = tr->getRange(globalConfigDataKeys, CLIENT_KNOBS->TOO_MANY);
															#line 340 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = globalConfigVersionFuture;
															#line 340 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1351 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 1;
															#line 340 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >*>(static_cast<GlobalConfigRefreshActor*>(this)));
															#line 1356 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 350 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->onError(e);
															#line 350 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1380 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 3;
															#line 350 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRefreshActor, 2, Void >*>(static_cast<GlobalConfigRefreshActor*>(this)));
															#line 1385 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 341 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tmpCachedDataFuture;
															#line 341 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 2;
															#line 341 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >*>(static_cast<GlobalConfigRefreshActor*>(this)));
															#line 1407 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& __globalConfigVersion,int loopDepth) 
	{
															#line 340 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		globalConfigVersion = __globalConfigVersion;
															#line 1416 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && __globalConfigVersion,int loopDepth) 
	{
		globalConfigVersion = std::move(__globalConfigVersion);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigRefreshActor*>(this)->ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& tmpCachedData,int loopDepth) 
	{
															#line 342 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*cachedData = tmpCachedData;
															#line 343 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (globalConfigVersion.present())
															#line 1497 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 344 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Version parsedVersion;
															#line 345 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			memcpy(&parsedVersion, globalConfigVersion.get().begin(), sizeof(Version));
															#line 346 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			*cachedVersion = bigEndian64(parsedVersion);
															#line 1505 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 348 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GlobalConfigRefreshActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GlobalConfigRefreshActorState(); static_cast<GlobalConfigRefreshActor*>(this)->destroy(); return 0; }
															#line 1509 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GlobalConfigRefreshActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GlobalConfigRefreshActorState();
		static_cast<GlobalConfigRefreshActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && tmpCachedData,int loopDepth) 
	{
															#line 342 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*cachedData = tmpCachedData;
															#line 343 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (globalConfigVersion.present())
															#line 1523 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 344 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Version parsedVersion;
															#line 345 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			memcpy(&parsedVersion, globalConfigVersion.get().begin(), sizeof(Version));
															#line 346 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			*cachedVersion = bigEndian64(parsedVersion);
															#line 1531 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 348 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GlobalConfigRefreshActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GlobalConfigRefreshActorState(); static_cast<GlobalConfigRefreshActor*>(this)->destroy(); return 0; }
															#line 1535 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GlobalConfigRefreshActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GlobalConfigRefreshActorState();
		static_cast<GlobalConfigRefreshActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& tmpCachedData,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(tmpCachedData, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && tmpCachedData,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(tmpCachedData), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigRefreshActor*>(this)->ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigRefreshActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigRefreshActor*>(this)->ActorCallback< GlobalConfigRefreshActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRefreshActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GlobalConfigRefreshActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Version* cachedVersion;
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	RangeResult* cachedData;
															#line 334 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 338 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Optional<Value>> globalConfigVersionFuture;
															#line 339 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<RangeResult> tmpCachedDataFuture;
															#line 340 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<Value> globalConfigVersion;
															#line 1719 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via globalConfigRefresh()
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigRefreshActor final : public Actor<Void>, public ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >, public ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >, public ActorCallback< GlobalConfigRefreshActor, 2, Void >, public FastAllocated<GlobalConfigRefreshActor>, public GlobalConfigRefreshActorState<GlobalConfigRefreshActor> {
															#line 1724 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GlobalConfigRefreshActor>::operator new;
	using FastAllocated<GlobalConfigRefreshActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15605644905664723968UL, 9274051322853840128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >;
friend struct ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >;
friend struct ActorCallback< GlobalConfigRefreshActor, 2, Void >;
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigRefreshActor(GrvProxyData* const& grvProxyData,Version* const& cachedVersion,RangeResult* const& cachedData) 
															#line 1743 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GlobalConfigRefreshActorState<GlobalConfigRefreshActor>(grvProxyData, cachedVersion, cachedData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("globalConfigRefresh", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10827479308140659200UL, 8791014497271066624UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRefreshActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("globalConfigRefresh");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("globalConfigRefresh", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GlobalConfigRefreshActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GlobalConfigRefreshActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GlobalConfigRefreshActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> globalConfigRefresh( GrvProxyData* const& grvProxyData, Version* const& cachedVersion, RangeResult* const& cachedData ) {
															#line 333 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GlobalConfigRefreshActor(grvProxyData, cachedVersion, cachedData));
															#line 1778 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 354 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Handle common GlobalConfig transactions on the server side, because not all
// clients are allowed to read system keys. Eventually, this could become its
// own role.
															#line 1786 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via globalConfigRequestServer()
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GlobalConfigRequestServerActor>
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigRequestServerActorState {
															#line 1793 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigRequestServerActorState(GrvProxyData* const& grvProxyData,GrvProxyInterface const& grvProxy) 
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : grvProxyData(grvProxyData),
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   grvProxy(grvProxy),
															#line 359 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   actors(false),
															#line 360 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   refreshFuture(),
															#line 361 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   cachedVersion(0),
															#line 362 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   cachedData()
															#line 1810 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("globalConfigRequestServer", reinterpret_cast<unsigned long>(this));

	}
	~GlobalConfigRequestServerActorState() 
	{
		fdb_probe_actor_destroy("globalConfigRequestServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 367 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			refreshFuture = timeout(globalConfigRefresh(grvProxyData, &cachedVersion, &cachedData), SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_TIMEOUT, Void()) && delay(SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_INTERVAL);
															#line 373 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(timeout(globalConfigMigrate(grvProxyData), SERVER_KNOBS->GLOBAL_CONFIG_MIGRATE_TIMEOUT));
															#line 373 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1829 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state = 1;
															#line 373 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRequestServerActor, 0, Void >*>(static_cast<GlobalConfigRequestServerActor*>(this)));
															#line 1834 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GlobalConfigRequestServerActorState();
		static_cast<GlobalConfigRequestServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 375 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		;
															#line 1857 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 375 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		;
															#line 1866 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigRequestServerActor*>(this)->ActorCallback< GlobalConfigRequestServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GlobalConfigRequestServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 377 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		auto __when_expr_1 = grvProxy.refreshGlobalConfig.getFuture();
															#line 1957 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<GlobalConfigRefreshRequest>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<GlobalConfigRefreshRequest>>::value, "invalid type");
															#line 376 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1961 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
															#line 388 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_2 = refreshFuture;
															#line 1965 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when2(__when_expr_2.get(), loopDepth); };
															#line 394 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_3 = actors.getResult();
															#line 1969 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when3(__when_expr_3.get(), loopDepth); };
		static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state = 2;
															#line 377 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >*>(static_cast<GlobalConfigRequestServerActor*>(this)));
															#line 388 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRequestServerActor, 2, Void >*>(static_cast<GlobalConfigRequestServerActor*>(this)));
															#line 394 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GlobalConfigRequestServerActor, 3, Void >*>(static_cast<GlobalConfigRequestServerActor*>(this)));
															#line 1978 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GlobalConfigRefreshRequest const& refresh,int loopDepth) 
	{
															#line 382 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (refresh.lastKnown <= cachedVersion)
															#line 1993 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 383 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			refresh.reply.send(GlobalConfigRefreshReply{ cachedData.arena(), cachedVersion, cachedData });
															#line 1997 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 385 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			refresh.reply.sendError(future_version());
															#line 2003 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GlobalConfigRefreshRequest && refresh,int loopDepth) 
	{
															#line 382 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (refresh.lastKnown <= cachedVersion)
															#line 2013 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 383 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			refresh.reply.send(GlobalConfigRefreshReply{ cachedData.arena(), cachedVersion, cachedData });
															#line 2017 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 385 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			refresh.reply.sendError(future_version());
															#line 2023 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 389 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		refreshFuture = timeout(globalConfigRefresh(grvProxyData, &cachedVersion, &cachedData), SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_TIMEOUT, Void()) && delay(SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_INTERVAL);
															#line 2033 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 389 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		refreshFuture = timeout(globalConfigRefresh(grvProxyData, &cachedVersion, &cachedData), SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_TIMEOUT, Void()) && delay(SERVER_KNOBS->GLOBAL_CONFIG_REFRESH_INTERVAL);
															#line 2042 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when3(Void const& _,int loopDepth) 
	{
															#line 395 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		ASSERT(false);
															#line 2051 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when3(Void && _,int loopDepth) 
	{
															#line 395 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		ASSERT(false);
															#line 2060 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state > 0) static_cast<GlobalConfigRequestServerActor*>(this)->actor_wait_state = 0;
		static_cast<GlobalConfigRequestServerActor*>(this)->ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >::remove();
		static_cast<GlobalConfigRequestServerActor*>(this)->ActorCallback< GlobalConfigRequestServerActor, 2, Void >::remove();
		static_cast<GlobalConfigRequestServerActor*>(this)->ActorCallback< GlobalConfigRequestServerActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >*,GlobalConfigRefreshRequest const& value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >*,GlobalConfigRefreshRequest && value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GlobalConfigRequestServerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GlobalConfigRequestServerActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GlobalConfigRequestServerActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface grvProxy;
															#line 359 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	ActorCollection actors;
															#line 360 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> refreshFuture;
															#line 361 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Version cachedVersion;
															#line 362 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	RangeResult cachedData;
															#line 2256 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via globalConfigRequestServer()
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GlobalConfigRequestServerActor final : public Actor<Void>, public ActorCallback< GlobalConfigRequestServerActor, 0, Void >, public ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >, public ActorCallback< GlobalConfigRequestServerActor, 2, Void >, public ActorCallback< GlobalConfigRequestServerActor, 3, Void >, public FastAllocated<GlobalConfigRequestServerActor>, public GlobalConfigRequestServerActorState<GlobalConfigRequestServerActor> {
															#line 2261 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GlobalConfigRequestServerActor>::operator new;
	using FastAllocated<GlobalConfigRequestServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3056353458474764544UL, 2418745880539463936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GlobalConfigRequestServerActor, 0, Void >;
friend struct ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >;
friend struct ActorCallback< GlobalConfigRequestServerActor, 2, Void >;
friend struct ActorCallback< GlobalConfigRequestServerActor, 3, Void >;
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GlobalConfigRequestServerActor(GrvProxyData* const& grvProxyData,GrvProxyInterface const& grvProxy) 
															#line 2281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GlobalConfigRequestServerActorState<GlobalConfigRequestServerActor>(grvProxyData, grvProxy),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15730633037479306240UL, 7934117556835500544UL);
		ActorExecutionContextHelper __helper(static_cast<GlobalConfigRequestServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("globalConfigRequestServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("globalConfigRequestServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GlobalConfigRequestServerActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< GlobalConfigRequestServerActor, 1, GlobalConfigRefreshRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> globalConfigRequestServer( GrvProxyData* const& grvProxyData, GrvProxyInterface const& grvProxy ) {
															#line 358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GlobalConfigRequestServerActor(grvProxyData, grvProxy));
															#line 2315 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 400 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Get transaction rate info from RateKeeper.
															#line 2321 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via getRate()
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GetRateActor>
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GetRateActorState {
															#line 2328 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetRateActorState(UID const& myID,Reference<AsyncVar<ServerDBInfo> const> const& db,int64_t* const& inTransactionCount,int64_t* const& inBatchTransactionCount,GrvTransactionRateInfo* const& transactionRateInfo,GrvTransactionRateInfo* const& batchTransactionRateInfo,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply,TransactionTagMap<uint64_t>* const& transactionTagCounter,PrioritizedTransactionTagMap<ClientTagThrottleLimits>* const& clientThrottledTags,GrvProxyStats* const& stats,GrvProxyData* const& proxyData) 
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : myID(myID),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   db(db),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   inTransactionCount(inTransactionCount),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   inBatchTransactionCount(inBatchTransactionCount),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   transactionRateInfo(transactionRateInfo),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchTransactionRateInfo(batchTransactionRateInfo),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   healthMetricsReply(healthMetricsReply),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   detailedHealthMetricsReply(detailedHealthMetricsReply),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   transactionTagCounter(transactionTagCounter),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   clientThrottledTags(clientThrottledTags),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   stats(stats),
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   proxyData(proxyData),
															#line 414 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   nextRequestTimer(Never()),
															#line 415 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   leaseTimeout(Never()),
															#line 416 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   reply(Never()),
															#line 417 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   lastDetailedReply(0.0),
															#line 418 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   expectingDetailedReply(false)
															#line 2367 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("getRate", reinterpret_cast<unsigned long>(this));

	}
	~GetRateActorState() 
	{
		fdb_probe_actor_destroy("getRate", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 421 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (db->get().ratekeeper.present())
															#line 2382 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 422 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				nextRequestTimer = Void();
															#line 2386 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 424 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 2390 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRateActorState();
		static_cast<GetRateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 425 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 424 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GetRateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2422 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 435 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_1 = nextRequestTimer;
															#line 2426 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 449 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<GetRateInfoReply> __when_expr_2 = reply;
															#line 2430 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
															#line 476 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_3 = leaseTimeout;
															#line 2434 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when4(__when_expr_3.get(), loopDepth); };
		static_cast<GetRateActor*>(this)->actor_wait_state = 1;
															#line 425 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRateActor, 0, Void >*>(static_cast<GetRateActor*>(this)));
															#line 435 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetRateActor, 1, Void >*>(static_cast<GetRateActor*>(this)));
															#line 449 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetRateActor, 2, GetRateInfoReply >*>(static_cast<GetRateActor*>(this)));
															#line 476 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetRateActor, 3, Void >*>(static_cast<GetRateActor*>(this)));
															#line 2445 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 426 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().ratekeeper.present())
															#line 2460 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 427 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("ProxyRatekeeperChanged", myID).detail("RKID", db->get().ratekeeper.get().id());
															#line 428 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Void();
															#line 2466 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 430 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("ProxyRatekeeperDied", myID).log();
															#line 431 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Never();
															#line 432 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply = Never();
															#line 2476 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 426 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().ratekeeper.present())
															#line 2486 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 427 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("ProxyRatekeeperChanged", myID).detail("RKID", db->get().ratekeeper.get().id());
															#line 428 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Void();
															#line 2492 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 430 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("ProxyRatekeeperDied", myID).log();
															#line 431 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Never();
															#line 432 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply = Never();
															#line 2502 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 436 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = Never();
															#line 437 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		bool detailed = now() - lastDetailedReply > SERVER_KNOBS->DETAILED_METRIC_UPDATE_RATE;
															#line 439 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		reply = brokenPromiseToNever( db->get().ratekeeper.get().getRateInfo.getReply(GetRateInfoRequest(myID, *inTransactionCount, *inBatchTransactionCount, proxyData->version, *transactionTagCounter, detailed)));
															#line 446 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionTagCounter->clear();
															#line 447 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		expectingDetailedReply = detailed;
															#line 2520 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 436 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = Never();
															#line 437 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		bool detailed = now() - lastDetailedReply > SERVER_KNOBS->DETAILED_METRIC_UPDATE_RATE;
															#line 439 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		reply = brokenPromiseToNever( db->get().ratekeeper.get().getRateInfo.getReply(GetRateInfoRequest(myID, *inTransactionCount, *inBatchTransactionCount, proxyData->version, *transactionTagCounter, detailed)));
															#line 446 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionTagCounter->clear();
															#line 447 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		expectingDetailedReply = detailed;
															#line 2537 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(GetRateInfoReply const& rep,int loopDepth) 
	{
															#line 450 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		reply = Never();
															#line 452 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionRateInfo->setRate(rep.transactionRate);
															#line 453 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionRateInfo->setRate(rep.batchTransactionRate);
															#line 454 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		stats->transactionRateAllowed = rep.transactionRate;
															#line 455 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		stats->batchTransactionRateAllowed = rep.batchTransactionRate;
															#line 456 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		++stats->updatesFromRatekeeper;
															#line 459 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		leaseTimeout = delay(rep.leaseDuration);
															#line 460 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = delayJittered(rep.leaseDuration / 2);
															#line 461 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		healthMetricsReply->update(rep.healthMetrics, expectingDetailedReply, true);
															#line 462 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (expectingDetailedReply)
															#line 2564 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 463 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			detailedHealthMetricsReply->update(rep.healthMetrics, true, true);
															#line 464 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			lastDetailedReply = now();
															#line 2570 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 469 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (rep.clientThrottledTags.present())
															#line 2574 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 470 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			*clientThrottledTags = std::move(rep.clientThrottledTags.get());
															#line 2578 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 472 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (rep.proxyThrottledTags.present())
															#line 2582 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 473 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			proxyData->tagThrottler.updateRates(rep.proxyThrottledTags.get());
															#line 2586 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(GetRateInfoReply && rep,int loopDepth) 
	{
															#line 450 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		reply = Never();
															#line 452 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionRateInfo->setRate(rep.transactionRate);
															#line 453 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionRateInfo->setRate(rep.batchTransactionRate);
															#line 454 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		stats->transactionRateAllowed = rep.transactionRate;
															#line 455 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		stats->batchTransactionRateAllowed = rep.batchTransactionRate;
															#line 456 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		++stats->updatesFromRatekeeper;
															#line 459 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		leaseTimeout = delay(rep.leaseDuration);
															#line 460 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = delayJittered(rep.leaseDuration / 2);
															#line 461 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		healthMetricsReply->update(rep.healthMetrics, expectingDetailedReply, true);
															#line 462 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (expectingDetailedReply)
															#line 2614 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 463 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			detailedHealthMetricsReply->update(rep.healthMetrics, true, true);
															#line 464 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			lastDetailedReply = now();
															#line 2620 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 469 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (rep.clientThrottledTags.present())
															#line 2624 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 470 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			*clientThrottledTags = std::move(rep.clientThrottledTags.get());
															#line 2628 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 472 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (rep.proxyThrottledTags.present())
															#line 2632 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 473 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			proxyData->tagThrottler.updateRates(rep.proxyThrottledTags.get());
															#line 2636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(Void const& _,int loopDepth) 
	{
															#line 477 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionRateInfo->disable();
															#line 478 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionRateInfo->disable();
															#line 479 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		++stats->leaseTimeouts;
															#line 480 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		TraceEvent(SevWarn, "GrvProxyRateLeaseExpired", myID).suppressFor(5.0);
															#line 482 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		leaseTimeout = Never();
															#line 2654 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(Void && _,int loopDepth) 
	{
															#line 477 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionRateInfo->disable();
															#line 478 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionRateInfo->disable();
															#line 479 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		++stats->leaseTimeouts;
															#line 480 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		TraceEvent(SevWarn, "GrvProxyRateLeaseExpired", myID).suppressFor(5.0);
															#line 482 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		leaseTimeout = Never();
															#line 2671 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRateActor*>(this)->actor_wait_state > 0) static_cast<GetRateActor*>(this)->actor_wait_state = 0;
		static_cast<GetRateActor*>(this)->ActorCallback< GetRateActor, 0, Void >::remove();
		static_cast<GetRateActor*>(this)->ActorCallback< GetRateActor, 1, Void >::remove();
		static_cast<GetRateActor*>(this)->ActorCallback< GetRateActor, 2, GetRateInfoReply >::remove();
		static_cast<GetRateActor*>(this)->ActorCallback< GetRateActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetRateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 2, GetRateInfoReply >*,GetRateInfoReply const& value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 2, GetRateInfoReply >*,GetRateInfoReply && value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetRateActor, 2, GetRateInfoReply >*,Error err) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetRateActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetRateActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	UID myID;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t* inTransactionCount;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t* inBatchTransactionCount;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvTransactionRateInfo* transactionRateInfo;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvTransactionRateInfo* batchTransactionRateInfo;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* healthMetricsReply;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* detailedHealthMetricsReply;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	TransactionTagMap<uint64_t>* transactionTagCounter;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PrioritizedTransactionTagMap<ClientTagThrottleLimits>* clientThrottledTags;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyStats* stats;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* proxyData;
															#line 414 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> nextRequestTimer;
															#line 415 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> leaseTimeout;
															#line 416 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<GetRateInfoReply> reply;
															#line 417 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double lastDetailedReply;
															#line 418 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	bool expectingDetailedReply;
															#line 2947 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via getRate()
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GetRateActor final : public Actor<Void>, public ActorCallback< GetRateActor, 0, Void >, public ActorCallback< GetRateActor, 1, Void >, public ActorCallback< GetRateActor, 2, GetRateInfoReply >, public ActorCallback< GetRateActor, 3, Void >, public FastAllocated<GetRateActor>, public GetRateActorState<GetRateActor> {
															#line 2952 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GetRateActor>::operator new;
	using FastAllocated<GetRateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7409923733806015232UL, 16703826713494089984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRateActor, 0, Void >;
friend struct ActorCallback< GetRateActor, 1, Void >;
friend struct ActorCallback< GetRateActor, 2, GetRateInfoReply >;
friend struct ActorCallback< GetRateActor, 3, Void >;
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetRateActor(UID const& myID,Reference<AsyncVar<ServerDBInfo> const> const& db,int64_t* const& inTransactionCount,int64_t* const& inBatchTransactionCount,GrvTransactionRateInfo* const& transactionRateInfo,GrvTransactionRateInfo* const& batchTransactionRateInfo,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply,TransactionTagMap<uint64_t>* const& transactionTagCounter,PrioritizedTransactionTagMap<ClientTagThrottleLimits>* const& clientThrottledTags,GrvProxyStats* const& stats,GrvProxyData* const& proxyData) 
															#line 2972 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GetRateActorState<GetRateActor>(myID, db, inTransactionCount, inBatchTransactionCount, transactionRateInfo, batchTransactionRateInfo, healthMetricsReply, detailedHealthMetricsReply, transactionTagCounter, clientThrottledTags, stats, proxyData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRate", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3946064217190820608UL, 8787252396757706496UL);
		ActorExecutionContextHelper __helper(static_cast<GetRateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRate");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRate", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRateActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> getRate( UID const& myID, Reference<AsyncVar<ServerDBInfo> const> const& db, int64_t* const& inTransactionCount, int64_t* const& inBatchTransactionCount, GrvTransactionRateInfo* const& transactionRateInfo, GrvTransactionRateInfo* const& batchTransactionRateInfo, GetHealthMetricsReply* const& healthMetricsReply, GetHealthMetricsReply* const& detailedHealthMetricsReply, TransactionTagMap<uint64_t>* const& transactionTagCounter, PrioritizedTransactionTagMap<ClientTagThrottleLimits>* const& clientThrottledTags, GrvProxyStats* const& stats, GrvProxyData* const& proxyData ) {
															#line 402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GetRateActor(myID, db, inTransactionCount, inBatchTransactionCount, transactionRateInfo, batchTransactionRateInfo, healthMetricsReply, detailedHealthMetricsReply, transactionTagCounter, clientThrottledTags, stats, proxyData));
															#line 3005 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 486 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Respond with an error to the GetReadVersion request when the GRV limit is hit.
void proxyGRVThresholdExceeded(const GetReadVersionRequest* req, GrvProxyStats* stats) {
	++stats->txnRequestErrors;
	req->reply.sendError(grv_proxy_memory_limit_exceeded());
	if (req->priority == TransactionPriority::IMMEDIATE) {
		TraceEvent(g_network->isSimulated() ? SevInfo : SevWarnAlways, "ProxyGRVThresholdExceededSystem")
		    .suppressFor(60);
	} else if (req->priority == TransactionPriority::DEFAULT) {
		TraceEvent(g_network->isSimulated() ? SevInfo : SevWarnAlways, "ProxyGRVThresholdExceededDefault")
		    .suppressFor(60);
	} else {
		TraceEvent(g_network->isSimulated() ? SevInfo : SevWarnAlways, "ProxyGRVThresholdExceededBatch")
		    .suppressFor(60);
	}
}

// Drop a GetReadVersion request from a queue, by responding an error to the request.
void dropRequestFromQueue(Deque<GetReadVersionRequest>* queue, GrvProxyStats* stats) {
	proxyGRVThresholdExceeded(&queue->front(), stats);
	++stats->txnRequestOut;
	queue->pop_front();
}

// Put a GetReadVersion request into the queue corresponding to its priority.
															#line 3034 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via queueGetReadVersionRequests()
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class QueueGetReadVersionRequestsActor>
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class QueueGetReadVersionRequestsActorState {
															#line 3041 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	QueueGetReadVersionRequestsActorState(Reference<AsyncVar<ServerDBInfo> const> const& db,Deque<GetReadVersionRequest>* const& systemQueue,Deque<GetReadVersionRequest>* const& defaultQueue,Deque<GetReadVersionRequest>* const& batchQueue,FutureStream<GetReadVersionRequest> const& readVersionRequests,PromiseStream<Void> const& GRVTimer,double* const& lastGRVTime,double* const& GRVBatchTime,FutureStream<double> const& normalGRVLatency,GrvProxyStats* const& stats,GrvTransactionRateInfo* const& batchRateInfo,GrvProxyTagThrottler* const& tagThrottler) 
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : db(db),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   systemQueue(systemQueue),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   defaultQueue(defaultQueue),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchQueue(batchQueue),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   readVersionRequests(readVersionRequests),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   GRVTimer(GRVTimer),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   lastGRVTime(lastGRVTime),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   GRVBatchTime(GRVBatchTime),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   normalGRVLatency(normalGRVLatency),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   stats(stats),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchRateInfo(batchRateInfo),
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   tagThrottler(tagThrottler)
															#line 3070 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this));

	}
	~QueueGetReadVersionRequestsActorState() 
	{
		fdb_probe_actor_destroy("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 523 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			getCurrentLineage()->modify(&TransactionLineage::operation) = TransactionLineage::Operation::GetConsistentReadVersion;
															#line 525 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 3087 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QueueGetReadVersionRequestsActorState();
		static_cast<QueueGetReadVersionRequestsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 526 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		auto __when_expr_0 = readVersionRequests;
															#line 3117 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<GetReadVersionRequest>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<GetReadVersionRequest>>::value, "invalid type");
															#line 525 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<QueueGetReadVersionRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3121 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
															#line 614 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		auto __when_expr_1 = normalGRVLatency;
															#line 3125 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<double>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<double>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<QueueGetReadVersionRequestsActor*>(this)->actor_wait_state = 1;
															#line 526 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >*>(static_cast<QueueGetReadVersionRequestsActor*>(this)));
															#line 614 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >*>(static_cast<QueueGetReadVersionRequestsActor*>(this)));
															#line 3133 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(GetReadVersionRequest const& req,int loopDepth) 
	{
															#line 530 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		bool canBeQueued = true;
															#line 531 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (stats->txnRequestIn.getValue() - stats->txnRequestOut.getValue() > SERVER_KNOBS->START_TRANSACTION_MAX_QUEUE_SIZE || (g_network->isSimulated() && !g_simulator->speedUpSimulation && deterministicRandom()->random01() < 0.01))
															#line 3150 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 536 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority == TransactionPriority::BATCH)
															#line 3154 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 537 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				canBeQueued = false;
															#line 3158 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 538 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority == TransactionPriority::DEFAULT)
															#line 3164 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 539 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue->empty())
															#line 3168 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 540 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						dropRequestFromQueue(batchQueue, stats);
															#line 541 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						--stats->batchGRVQueueSize;
															#line 3174 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 543 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						canBeQueued = false;
															#line 3180 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
				}
				else
				{
															#line 546 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue->empty())
															#line 3187 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 547 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						dropRequestFromQueue(batchQueue, stats);
															#line 548 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						--stats->batchGRVQueueSize;
															#line 3193 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 549 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (!defaultQueue->empty())
															#line 3199 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 550 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							dropRequestFromQueue(defaultQueue, stats);
															#line 551 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							--stats->defaultGRVQueueSize;
															#line 3205 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
						else
						{
															#line 553 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							canBeQueued = false;
															#line 3211 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 557 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!canBeQueued)
															#line 3219 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 558 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			proxyGRVThresholdExceeded(&req, stats);
															#line 3223 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 560 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			stats->addRequest(req.transactionCount);
															#line 562 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.debugID.present())
															#line 3231 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 563 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", req.debugID.get().first(), "GrvProxyServer.queueTransactionStartRequests.Before");
															#line 3235 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 567 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (systemQueue->empty() && defaultQueue->empty() && batchQueue->empty())
															#line 3239 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 568 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				forwardPromise(GRVTimer, delayJittered(std::max(0.0, *GRVBatchTime - (now() - *lastGRVTime)), TaskPriority::ProxyGRVTimer));
															#line 3243 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 573 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority >= TransactionPriority::IMMEDIATE)
															#line 3247 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 574 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				++stats->txnRequestIn;
															#line 575 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->txnStartIn += req.transactionCount;
															#line 576 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->txnSystemPriorityStartIn += req.transactionCount;
															#line 577 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				++stats->systemGRVQueueSize;
															#line 578 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				systemQueue->push_back(req);
															#line 3259 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 579 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority >= TransactionPriority::DEFAULT)
															#line 3265 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 580 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (SERVER_KNOBS->ENFORCE_TAG_THROTTLING_ON_PROXIES && req.isTagged())
															#line 3269 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 581 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->tagThrottlerGRVQueueSize;
															#line 582 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnTagThrottlerIn += req.transactionCount;
															#line 583 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						tagThrottler->addRequest(req);
															#line 3277 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 585 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->txnRequestIn;
															#line 586 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnStartIn += req.transactionCount;
															#line 587 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnDefaultPriorityStartIn += req.transactionCount;
															#line 588 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->defaultGRVQueueSize;
															#line 589 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						defaultQueue->push_back(req);
															#line 3291 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
				}
				else
				{
															#line 593 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					int64_t proxiesCount = std::max((int)db->get().client.grvProxies.size(), 1);
															#line 594 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (batchRateInfo->getRate() <= (1.0 / proxiesCount))
															#line 3300 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 595 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						req.reply.sendError(batch_transaction_throttled());
															#line 596 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnThrottled += req.transactionCount;
															#line 3306 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 598 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (SERVER_KNOBS->ENFORCE_TAG_THROTTLING_ON_PROXIES && req.isTagged())
															#line 3312 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 599 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->tagThrottlerGRVQueueSize;
															#line 600 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnTagThrottlerIn += req.transactionCount;
															#line 601 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							tagThrottler->addRequest(req);
															#line 3320 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
						else
						{
															#line 603 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->txnRequestIn;
															#line 604 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnStartIn += req.transactionCount;
															#line 605 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnBatchPriorityStartIn += req.transactionCount;
															#line 606 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->batchGRVQueueSize;
															#line 607 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							batchQueue->push_back(req);
															#line 3334 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(GetReadVersionRequest && req,int loopDepth) 
	{
															#line 530 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		bool canBeQueued = true;
															#line 531 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (stats->txnRequestIn.getValue() - stats->txnRequestOut.getValue() > SERVER_KNOBS->START_TRANSACTION_MAX_QUEUE_SIZE || (g_network->isSimulated() && !g_simulator->speedUpSimulation && deterministicRandom()->random01() < 0.01))
															#line 3350 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 536 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority == TransactionPriority::BATCH)
															#line 3354 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 537 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				canBeQueued = false;
															#line 3358 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 538 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority == TransactionPriority::DEFAULT)
															#line 3364 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 539 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue->empty())
															#line 3368 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 540 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						dropRequestFromQueue(batchQueue, stats);
															#line 541 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						--stats->batchGRVQueueSize;
															#line 3374 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 543 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						canBeQueued = false;
															#line 3380 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
				}
				else
				{
															#line 546 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue->empty())
															#line 3387 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 547 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						dropRequestFromQueue(batchQueue, stats);
															#line 548 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						--stats->batchGRVQueueSize;
															#line 3393 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 549 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (!defaultQueue->empty())
															#line 3399 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 550 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							dropRequestFromQueue(defaultQueue, stats);
															#line 551 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							--stats->defaultGRVQueueSize;
															#line 3405 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
						else
						{
															#line 553 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							canBeQueued = false;
															#line 3411 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 557 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!canBeQueued)
															#line 3419 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 558 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			proxyGRVThresholdExceeded(&req, stats);
															#line 3423 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 560 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			stats->addRequest(req.transactionCount);
															#line 562 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.debugID.present())
															#line 3431 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 563 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", req.debugID.get().first(), "GrvProxyServer.queueTransactionStartRequests.Before");
															#line 3435 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 567 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (systemQueue->empty() && defaultQueue->empty() && batchQueue->empty())
															#line 3439 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 568 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				forwardPromise(GRVTimer, delayJittered(std::max(0.0, *GRVBatchTime - (now() - *lastGRVTime)), TaskPriority::ProxyGRVTimer));
															#line 3443 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 573 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority >= TransactionPriority::IMMEDIATE)
															#line 3447 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 574 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				++stats->txnRequestIn;
															#line 575 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->txnStartIn += req.transactionCount;
															#line 576 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->txnSystemPriorityStartIn += req.transactionCount;
															#line 577 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				++stats->systemGRVQueueSize;
															#line 578 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				systemQueue->push_back(req);
															#line 3459 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 579 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority >= TransactionPriority::DEFAULT)
															#line 3465 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 580 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (SERVER_KNOBS->ENFORCE_TAG_THROTTLING_ON_PROXIES && req.isTagged())
															#line 3469 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 581 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->tagThrottlerGRVQueueSize;
															#line 582 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnTagThrottlerIn += req.transactionCount;
															#line 583 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						tagThrottler->addRequest(req);
															#line 3477 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 585 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->txnRequestIn;
															#line 586 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnStartIn += req.transactionCount;
															#line 587 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnDefaultPriorityStartIn += req.transactionCount;
															#line 588 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						++stats->defaultGRVQueueSize;
															#line 589 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						defaultQueue->push_back(req);
															#line 3491 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
				}
				else
				{
															#line 593 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					int64_t proxiesCount = std::max((int)db->get().client.grvProxies.size(), 1);
															#line 594 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (batchRateInfo->getRate() <= (1.0 / proxiesCount))
															#line 3500 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 595 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						req.reply.sendError(batch_transaction_throttled());
															#line 596 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						stats->txnThrottled += req.transactionCount;
															#line 3506 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
															#line 598 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (SERVER_KNOBS->ENFORCE_TAG_THROTTLING_ON_PROXIES && req.isTagged())
															#line 3512 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 599 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->tagThrottlerGRVQueueSize;
															#line 600 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnTagThrottlerIn += req.transactionCount;
															#line 601 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							tagThrottler->addRequest(req);
															#line 3520 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
						else
						{
															#line 603 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->txnRequestIn;
															#line 604 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnStartIn += req.transactionCount;
															#line 605 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							stats->txnBatchPriorityStartIn += req.transactionCount;
															#line 606 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							++stats->batchGRVQueueSize;
															#line 607 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							batchQueue->push_back(req);
															#line 3534 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(double const& reply_latency,int loopDepth) 
	{
															#line 615 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double target_latency = reply_latency * SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_LATENCY_FRACTION;
															#line 616 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*GRVBatchTime = std::max( SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_MIN, std::min(SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_MAX, target_latency * SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_SMOOTHER_ALPHA + *GRVBatchTime * (1 - SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_SMOOTHER_ALPHA)));
															#line 3550 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(double && reply_latency,int loopDepth) 
	{
															#line 615 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double target_latency = reply_latency * SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_LATENCY_FRACTION;
															#line 616 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*GRVBatchTime = std::max( SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_MIN, std::min(SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_MAX, target_latency * SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_SMOOTHER_ALPHA + *GRVBatchTime * (1 - SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_SMOOTHER_ALPHA)));
															#line 3561 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QueueGetReadVersionRequestsActor*>(this)->actor_wait_state > 0) static_cast<QueueGetReadVersionRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<QueueGetReadVersionRequestsActor*>(this)->ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >::remove();
		static_cast<QueueGetReadVersionRequestsActor*>(this)->ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >::remove();

	}
	void a_callback_fire(ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >*,GetReadVersionRequest const& value) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >*,GetReadVersionRequest && value) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >*,Error err) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >*,double const& value) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >*,double && value) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >*,Error err) 
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest>* systemQueue;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest>* defaultQueue;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest>* batchQueue;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	FutureStream<GetReadVersionRequest> readVersionRequests;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<Void> GRVTimer;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double* lastGRVTime;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double* GRVBatchTime;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	FutureStream<double> normalGRVLatency;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyStats* stats;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvTransactionRateInfo* batchRateInfo;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyTagThrottler* tagThrottler;
															#line 3711 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via queueGetReadVersionRequests()
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class QueueGetReadVersionRequestsActor final : public Actor<Void>, public ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >, public ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >, public FastAllocated<QueueGetReadVersionRequestsActor>, public QueueGetReadVersionRequestsActorState<QueueGetReadVersionRequestsActor> {
															#line 3716 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<QueueGetReadVersionRequestsActor>::operator new;
	using FastAllocated<QueueGetReadVersionRequestsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1108282687949106176UL, 5240086158335921408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >;
friend struct ActorSingleCallback< QueueGetReadVersionRequestsActor, 1, double >;
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	QueueGetReadVersionRequestsActor(Reference<AsyncVar<ServerDBInfo> const> const& db,Deque<GetReadVersionRequest>* const& systemQueue,Deque<GetReadVersionRequest>* const& defaultQueue,Deque<GetReadVersionRequest>* const& batchQueue,FutureStream<GetReadVersionRequest> const& readVersionRequests,PromiseStream<Void> const& GRVTimer,double* const& lastGRVTime,double* const& GRVBatchTime,FutureStream<double> const& normalGRVLatency,GrvProxyStats* const& stats,GrvTransactionRateInfo* const& batchRateInfo,GrvProxyTagThrottler* const& tagThrottler) 
															#line 3734 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   QueueGetReadVersionRequestsActorState<QueueGetReadVersionRequestsActor>(db, systemQueue, defaultQueue, batchQueue, readVersionRequests, GRVTimer, lastGRVTime, GRVBatchTime, normalGRVLatency, stats, batchRateInfo, tagThrottler),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16228072202774818048UL, 4568169853275226880UL);
		ActorExecutionContextHelper __helper(static_cast<QueueGetReadVersionRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("queueGetReadVersionRequests");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("queueGetReadVersionRequests", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< QueueGetReadVersionRequestsActor, 0, GetReadVersionRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> queueGetReadVersionRequests( Reference<AsyncVar<ServerDBInfo> const> const& db, Deque<GetReadVersionRequest>* const& systemQueue, Deque<GetReadVersionRequest>* const& defaultQueue, Deque<GetReadVersionRequest>* const& batchQueue, FutureStream<GetReadVersionRequest> const& readVersionRequests, PromiseStream<Void> const& GRVTimer, double* const& lastGRVTime, double* const& GRVBatchTime, FutureStream<double> const& normalGRVLatency, GrvProxyStats* const& stats, GrvTransactionRateInfo* const& batchRateInfo, GrvProxyTagThrottler* const& tagThrottler ) {
															#line 511 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new QueueGetReadVersionRequestsActor(db, systemQueue, defaultQueue, batchQueue, readVersionRequests, GRVTimer, lastGRVTime, GRVBatchTime, normalGRVLatency, stats, batchRateInfo, tagThrottler));
															#line 3767 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 624 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 3772 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via updateLastCommit()
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class UpdateLastCommitActor>
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class UpdateLastCommitActorState {
															#line 3779 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	UpdateLastCommitActorState(GrvProxyData* const& self,Optional<UID> const& debugID = Optional<UID>()) 
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : self(self),
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   debugID(debugID),
															#line 626 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   confirmStart(now())
															#line 3790 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("updateLastCommit", reinterpret_cast<unsigned long>(this));

	}
	~UpdateLastCommitActorState() 
	{
		fdb_probe_actor_destroy("updateLastCommit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 627 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			self->lastStartCommit = confirmStart;
															#line 628 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			self->updateCommitRequests++;
															#line 629 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->logSystem->confirmEpochLive(debugID);
															#line 629 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<UpdateLastCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3811 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateLastCommitActor*>(this)->actor_wait_state = 1;
															#line 629 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateLastCommitActor, 0, Void >*>(static_cast<UpdateLastCommitActor*>(this)));
															#line 3816 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateLastCommitActorState();
		static_cast<UpdateLastCommitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 630 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->updateCommitRequests--;
															#line 631 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->lastCommitLatency = now() - confirmStart;
															#line 632 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->lastCommitTime = std::max(self->lastCommitTime.get(), confirmStart);
															#line 633 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<UpdateLastCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateLastCommitActorState(); static_cast<UpdateLastCommitActor*>(this)->destroy(); return 0; }
															#line 3845 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<UpdateLastCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateLastCommitActorState();
		static_cast<UpdateLastCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 630 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->updateCommitRequests--;
															#line 631 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->lastCommitLatency = now() - confirmStart;
															#line 632 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		self->lastCommitTime = std::max(self->lastCommitTime.get(), confirmStart);
															#line 633 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<UpdateLastCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateLastCommitActorState(); static_cast<UpdateLastCommitActor*>(this)->destroy(); return 0; }
															#line 3863 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<UpdateLastCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateLastCommitActorState();
		static_cast<UpdateLastCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateLastCommitActor*>(this)->actor_wait_state > 0) static_cast<UpdateLastCommitActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateLastCommitActor*>(this)->ActorCallback< UpdateLastCommitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateLastCommitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateLastCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateLastCommitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateLastCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateLastCommitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateLastCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateLastCommit", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* self;
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<UID> debugID;
															#line 626 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double confirmStart;
															#line 3952 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via updateLastCommit()
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class UpdateLastCommitActor final : public Actor<Void>, public ActorCallback< UpdateLastCommitActor, 0, Void >, public FastAllocated<UpdateLastCommitActor>, public UpdateLastCommitActorState<UpdateLastCommitActor> {
															#line 3957 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<UpdateLastCommitActor>::operator new;
	using FastAllocated<UpdateLastCommitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13149710002880036352UL, 2590705463561186304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateLastCommitActor, 0, Void >;
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	UpdateLastCommitActor(GrvProxyData* const& self,Optional<UID> const& debugID = Optional<UID>()) 
															#line 3974 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateLastCommitActorState<UpdateLastCommitActor>(self, debugID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateLastCommit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6267630262008360960UL, 2835741678179808768UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateLastCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateLastCommit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateLastCommit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateLastCommitActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> updateLastCommit( GrvProxyData* const& self, Optional<UID> const& debugID = Optional<UID>() ) {
															#line 625 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new UpdateLastCommitActor(self, debugID));
															#line 4007 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 635 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 4012 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via lastCommitUpdater()
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class LastCommitUpdaterActor>
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class LastCommitUpdaterActorState {
															#line 4019 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	LastCommitUpdaterActorState(GrvProxyData* const& self,PromiseStream<Future<Void>> const& addActor) 
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : self(self),
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   addActor(addActor)
															#line 4028 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("lastCommitUpdater", reinterpret_cast<unsigned long>(this));

	}
	~LastCommitUpdaterActorState() 
	{
		fdb_probe_actor_destroy("lastCommitUpdater", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 637 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 4043 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LastCommitUpdaterActorState();
		static_cast<LastCommitUpdaterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 638 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double interval = std::max(SERVER_KNOBS->MIN_CONFIRM_INTERVAL, (SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION - self->lastCommitLatency) / 2.0);
															#line 640 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double elapsed = now() - self->lastStartCommit;
															#line 641 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (elapsed < interval)
															#line 4077 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 642 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(interval + 0.0001 - elapsed);
															#line 642 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<LastCommitUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4083 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LastCommitUpdaterActor*>(this)->actor_wait_state = 1;
															#line 642 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LastCommitUpdaterActor, 0, Void >*>(static_cast<LastCommitUpdaterActor*>(this)));
															#line 4088 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 645 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (self->updateCommitRequests < SERVER_KNOBS->MAX_COMMIT_UPDATES)
															#line 4095 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 646 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				addActor.send(updateLastCommit(self));
															#line 4099 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 648 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevInfo : SevWarnAlways, "TooManyLastCommitUpdates") .suppressFor(1.0);
															#line 650 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				self->lastStartCommit = now();
															#line 4107 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LastCommitUpdaterActor*>(this)->actor_wait_state > 0) static_cast<LastCommitUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<LastCommitUpdaterActor*>(this)->ActorCallback< LastCommitUpdaterActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LastCommitUpdaterActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LastCommitUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LastCommitUpdaterActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LastCommitUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LastCommitUpdaterActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LastCommitUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lastCommitUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* self;
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 4211 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via lastCommitUpdater()
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class LastCommitUpdaterActor final : public Actor<Void>, public ActorCallback< LastCommitUpdaterActor, 0, Void >, public FastAllocated<LastCommitUpdaterActor>, public LastCommitUpdaterActorState<LastCommitUpdaterActor> {
															#line 4216 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<LastCommitUpdaterActor>::operator new;
	using FastAllocated<LastCommitUpdaterActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(682239041629432064UL, 11179663502508658432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LastCommitUpdaterActor, 0, Void >;
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	LastCommitUpdaterActor(GrvProxyData* const& self,PromiseStream<Future<Void>> const& addActor) 
															#line 4233 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   LastCommitUpdaterActorState<LastCommitUpdaterActor>(self, addActor),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("lastCommitUpdater", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5278625381336097792UL, 45007184844760832UL);
		ActorExecutionContextHelper __helper(static_cast<LastCommitUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("lastCommitUpdater");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("lastCommitUpdater", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LastCommitUpdaterActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> lastCommitUpdater( GrvProxyData* const& self, PromiseStream<Future<Void>> const& addActor ) {
															#line 636 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new LastCommitUpdaterActor(self, addActor));
															#line 4266 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 655 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 4271 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via getLiveCommittedVersion()
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GetLiveCommittedVersionActor>
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GetLiveCommittedVersionActorState {
															#line 4278 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetLiveCommittedVersionActorState(std::vector<SpanContext> const& spanContexts,GrvProxyData* const& grvProxyData,uint32_t const& flags,Optional<UID> const& debugID,int const& transactionCount,int const& systemTransactionCount,int const& defaultPriTransactionCount,int const& batchPriTransactionCount) 
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : spanContexts(spanContexts),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   grvProxyData(grvProxyData),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   flags(flags),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   debugID(debugID),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   transactionCount(transactionCount),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   systemTransactionCount(systemTransactionCount),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   defaultPriTransactionCount(defaultPriTransactionCount),
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchPriTransactionCount(batchPriTransactionCount),
															#line 669 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   span("GP:getLiveCommittedVersion"_loc)
															#line 4301 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this));

	}
	~GetLiveCommittedVersionActorState() 
	{
		fdb_probe_actor_destroy("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 670 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			for( const SpanContext& spanContext : spanContexts ) {
															#line 671 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				span.addLink(spanContext);
															#line 4318 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 673 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			++grvProxyData->stats.txnStartBatch;
															#line 675 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvStart = now();
															#line 676 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			replyFromMasterFuture = Future<GetRawCommittedVersionReply>();
															#line 677 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			replyFromMasterFuture = grvProxyData->master.getLiveCommittedVersion.getReply( GetRawCommittedVersionRequest(span.context, debugID, grvProxyData->ssVersionVectorCache.getMaxVersion()), TaskPriority::GetLiveCommittedVersionReply);
															#line 681 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!SERVER_KNOBS->ALWAYS_CAUSAL_READ_RISKY && !(flags & GetReadVersionRequest::FLAG_CAUSAL_READ_RISKY))
															#line 4330 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 682 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				StrictFuture<Void> __when_expr_0 = transformError(updateLastCommit(grvProxyData, debugID), broken_promise(), tlog_failed());
															#line 682 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4336 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 1;
															#line 682 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetLiveCommittedVersionActor, 0, Void >*>(static_cast<GetLiveCommittedVersionActor*>(this)));
															#line 4341 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 683 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION > 0 && now() - SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION > grvProxyData->lastCommitTime.get())
															#line 4348 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 685 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					StrictFuture<Void> __when_expr_1 = grvProxyData->lastCommitTime.whenAtLeast(now() - SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION);
															#line 685 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4354 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
					static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 2;
															#line 685 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetLiveCommittedVersionActor, 1, Void >*>(static_cast<GetLiveCommittedVersionActor*>(this)));
															#line 4359 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont4(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetLiveCommittedVersionActorState();
		static_cast<GetLiveCommittedVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 688 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvConfirmEpochLive = now();
															#line 689 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.grvConfirmEpochLiveDist->sampleSeconds(grvConfirmEpochLive - grvStart);
															#line 690 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (debugID.present())
															#line 4392 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 691 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "GrvProxyServer.getLiveCommittedVersion.confirmEpochLive");
															#line 4396 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 695 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<GetRawCommittedVersionReply> __when_expr_2 = transformError(replyFromMasterFuture, broken_promise(), master_failed());
															#line 695 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4402 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 3;
															#line 695 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >*>(static_cast<GetLiveCommittedVersionActor*>(this)));
															#line 4407 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<GetLiveCommittedVersionActor*>(this)->ActorCallback< GetLiveCommittedVersionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetLiveCommittedVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<GetLiveCommittedVersionActor*>(this)->ActorCallback< GetLiveCommittedVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetLiveCommittedVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(GetRawCommittedVersionReply const& repFromMaster,int loopDepth) 
	{
															#line 697 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->version = std::max(grvProxyData->version, repFromMaster.version);
															#line 698 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->minKnownCommittedVersion = std::max(grvProxyData->minKnownCommittedVersion, repFromMaster.minKnownCommittedVersion);
															#line 700 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR)
															#line 4600 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 702 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->ssVersionVectorCache.applyDelta(repFromMaster.ssVersionVectorDelta);
															#line 4604 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 704 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.grvGetCommittedVersionRpcDist->sampleSeconds(now() - grvConfirmEpochLive);
															#line 705 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		GetReadVersionReply rep;
															#line 706 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.version = repFromMaster.version;
															#line 707 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.locked = repFromMaster.locked;
															#line 708 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.metadataVersion = repFromMaster.metadataVersion;
															#line 709 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.processBusyTime = FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION * std::min((std::numeric_limits<int>::max() / FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION) - 1, grvProxyData->stats.getRecentRequests());
															#line 713 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.processBusyTime += FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION * (g_network->isSimulated() ? deterministicRandom()->random01() : g_network->networkInfo.metrics.lastRunLoopBusyness);
															#line 717 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (debugID.present())
															#line 4622 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 718 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "GrvProxyServer.getLiveCommittedVersion.After");
															#line 4626 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 722 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnStartOut += transactionCount;
															#line 723 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnSystemPriorityStartOut += systemTransactionCount;
															#line 724 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnDefaultPriorityStartOut += defaultPriTransactionCount;
															#line 725 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnBatchPriorityStartOut += batchPriTransactionCount;
															#line 727 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GetLiveCommittedVersionActor*>(this)->SAV<GetReadVersionReply>::futures) { (void)(rep); this->~GetLiveCommittedVersionActorState(); static_cast<GetLiveCommittedVersionActor*>(this)->destroy(); return 0; }
															#line 4638 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GetLiveCommittedVersionActor*>(this)->SAV< GetReadVersionReply >::value()) GetReadVersionReply(rep);
		this->~GetLiveCommittedVersionActorState();
		static_cast<GetLiveCommittedVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(GetRawCommittedVersionReply && repFromMaster,int loopDepth) 
	{
															#line 697 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->version = std::max(grvProxyData->version, repFromMaster.version);
															#line 698 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->minKnownCommittedVersion = std::max(grvProxyData->minKnownCommittedVersion, repFromMaster.minKnownCommittedVersion);
															#line 700 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR)
															#line 4654 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 702 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->ssVersionVectorCache.applyDelta(repFromMaster.ssVersionVectorDelta);
															#line 4658 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 704 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.grvGetCommittedVersionRpcDist->sampleSeconds(now() - grvConfirmEpochLive);
															#line 705 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		GetReadVersionReply rep;
															#line 706 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.version = repFromMaster.version;
															#line 707 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.locked = repFromMaster.locked;
															#line 708 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.metadataVersion = repFromMaster.metadataVersion;
															#line 709 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.processBusyTime = FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION * std::min((std::numeric_limits<int>::max() / FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION) - 1, grvProxyData->stats.getRecentRequests());
															#line 713 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		rep.processBusyTime += FLOW_KNOBS->BASIC_LOAD_BALANCE_COMPUTE_PRECISION * (g_network->isSimulated() ? deterministicRandom()->random01() : g_network->networkInfo.metrics.lastRunLoopBusyness);
															#line 717 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (debugID.present())
															#line 4676 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 718 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "GrvProxyServer.getLiveCommittedVersion.After");
															#line 4680 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 722 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnStartOut += transactionCount;
															#line 723 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnSystemPriorityStartOut += systemTransactionCount;
															#line 724 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnDefaultPriorityStartOut += defaultPriTransactionCount;
															#line 725 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.txnBatchPriorityStartOut += batchPriTransactionCount;
															#line 727 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GetLiveCommittedVersionActor*>(this)->SAV<GetReadVersionReply>::futures) { (void)(rep); this->~GetLiveCommittedVersionActorState(); static_cast<GetLiveCommittedVersionActor*>(this)->destroy(); return 0; }
															#line 4692 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GetLiveCommittedVersionActor*>(this)->SAV< GetReadVersionReply >::value()) GetReadVersionReply(rep);
		this->~GetLiveCommittedVersionActorState();
		static_cast<GetLiveCommittedVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(GetRawCommittedVersionReply const& repFromMaster,int loopDepth) 
	{
		loopDepth = a_body1cont6(repFromMaster, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetRawCommittedVersionReply && repFromMaster,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(repFromMaster), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<GetLiveCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<GetLiveCommittedVersionActor*>(this)->ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >*,GetRawCommittedVersionReply const& value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >*,GetRawCommittedVersionReply && value) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >*,Error err) 
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	std::vector<SpanContext> spanContexts;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	uint32_t flags;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Optional<UID> debugID;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int transactionCount;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int systemTransactionCount;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int defaultPriTransactionCount;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int batchPriTransactionCount;
															#line 669 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Span span;
															#line 675 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double grvStart;
															#line 676 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<GetRawCommittedVersionReply> replyFromMasterFuture;
															#line 688 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double grvConfirmEpochLive;
															#line 4799 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via getLiveCommittedVersion()
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GetLiveCommittedVersionActor final : public Actor<GetReadVersionReply>, public ActorCallback< GetLiveCommittedVersionActor, 0, Void >, public ActorCallback< GetLiveCommittedVersionActor, 1, Void >, public ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >, public FastAllocated<GetLiveCommittedVersionActor>, public GetLiveCommittedVersionActorState<GetLiveCommittedVersionActor> {
															#line 4804 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GetLiveCommittedVersionActor>::operator new;
	using FastAllocated<GetLiveCommittedVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1672187271937081600UL, 4571563626887803392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetReadVersionReply>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetLiveCommittedVersionActor, 0, Void >;
friend struct ActorCallback< GetLiveCommittedVersionActor, 1, Void >;
friend struct ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >;
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetLiveCommittedVersionActor(std::vector<SpanContext> const& spanContexts,GrvProxyData* const& grvProxyData,uint32_t const& flags,Optional<UID> const& debugID,int const& transactionCount,int const& systemTransactionCount,int const& defaultPriTransactionCount,int const& batchPriTransactionCount) 
															#line 4823 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<GetReadVersionReply>(),
		   GetLiveCommittedVersionActorState<GetLiveCommittedVersionActor>(spanContexts, grvProxyData, flags, debugID, transactionCount, systemTransactionCount, defaultPriTransactionCount, batchPriTransactionCount),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2565788491578482432UL, 13699237086852491520UL);
		ActorExecutionContextHelper __helper(static_cast<GetLiveCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getLiveCommittedVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getLiveCommittedVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetLiveCommittedVersionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetLiveCommittedVersionActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetLiveCommittedVersionActor, 2, GetRawCommittedVersionReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<GetReadVersionReply> getLiveCommittedVersion( std::vector<SpanContext> const& spanContexts, GrvProxyData* const& grvProxyData, uint32_t const& flags, Optional<UID> const& debugID, int const& transactionCount, int const& systemTransactionCount, int const& defaultPriTransactionCount, int const& batchPriTransactionCount ) {
															#line 656 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<GetReadVersionReply>(new GetLiveCommittedVersionActor(spanContexts, grvProxyData, flags, debugID, transactionCount, systemTransactionCount, defaultPriTransactionCount, batchPriTransactionCount));
															#line 4858 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 729 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

// Returns the current read version (or minimum known committed version if requested),
// to each request in the provided list. Also check if the request should be throttled.
// Update GRV statistics according to the request's priority.
															#line 4866 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via sendGrvReplies()
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class SendGrvRepliesActor>
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class SendGrvRepliesActorState {
															#line 4873 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	SendGrvRepliesActorState(Future<GetReadVersionReply> const& replyFuture,std::vector<GetReadVersionRequest> const& requests,GrvProxyData* const& grvProxyData,GrvProxyStats* const& stats,Version const& minKnownCommittedVersion,PrioritizedTransactionTagMap<ClientTagThrottleLimits> const& clientThrottledTags,int64_t const& midShardSize = 0) 
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : replyFuture(replyFuture),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   requests(requests),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   grvProxyData(grvProxyData),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   stats(stats),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   minKnownCommittedVersion(minKnownCommittedVersion),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   clientThrottledTags(clientThrottledTags),
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   midShardSize(midShardSize)
															#line 4892 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("sendGrvReplies", reinterpret_cast<unsigned long>(this));

	}
	~SendGrvRepliesActorState() 
	{
		fdb_probe_actor_destroy("sendGrvReplies", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 740 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<GetReadVersionReply> __when_expr_0 = replyFuture;
															#line 740 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<SendGrvRepliesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4909 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SendGrvRepliesActor*>(this)->actor_wait_state = 1;
															#line 740 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >*>(static_cast<SendGrvRepliesActor*>(this)));
															#line 4914 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SendGrvRepliesActorState();
		static_cast<SendGrvRepliesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(GetReadVersionReply const& _reply,int loopDepth) 
	{
															#line 741 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		GetReadVersionReply reply = _reply;
															#line 742 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		Version replyVersion = reply.version;
															#line 744 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double end = g_network->timer();
															#line 745 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for( GetReadVersionRequest const& request : requests ) {
															#line 746 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			double duration = end - request.requestTime() - request.proxyTagThrottledDuration;
															#line 747 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority == TransactionPriority::BATCH)
															#line 4947 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 748 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvBatchLatencySample.addMeasurement(duration);
															#line 4951 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 751 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority == TransactionPriority::DEFAULT)
															#line 4955 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 752 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvLatencySample.addMeasurement(duration);
															#line 4959 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 755 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority >= TransactionPriority::DEFAULT)
															#line 4963 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 756 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvLatencyBands.addMeasurement(duration);
															#line 4967 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 759 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.flags & GetReadVersionRequest::FLAG_USE_MIN_KNOWN_COMMITTED_VERSION)
															#line 4971 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 761 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				reply.version = minKnownCommittedVersion;
															#line 4975 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 763 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				reply.version = replyVersion;
															#line 4981 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 765 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.midShardSize = midShardSize;
															#line 766 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.tagThrottleInfo.clear();
															#line 767 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (SERVER_KNOBS->ENABLE_VERSION_VECTOR)
															#line 4989 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 768 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->ssVersionVectorCache.getDelta(request.maxVersion, reply.ssVersionVectorDelta);
															#line 769 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->versionVectorSizeOnGRVReply->addMeasurement(reply.ssVersionVectorDelta.size());
															#line 4995 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 771 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.proxyId = grvProxyData->dbgid;
															#line 772 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.proxyTagThrottledDuration = request.proxyTagThrottledDuration;
															#line 774 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.isTagged())
															#line 5003 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 775 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				auto& priorityThrottledTags = clientThrottledTags[request.priority];
															#line 776 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				for( auto tag : request.tags ) {
															#line 777 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					auto tagItr = priorityThrottledTags.find(tag.first);
															#line 778 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (tagItr != priorityThrottledTags.end())
															#line 5013 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 779 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (tagItr->second.expiration > now())
															#line 5017 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 780 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							if (tagItr->second.tpsRate == std::numeric_limits<double>::max())
															#line 5021 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							{
															#line 781 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								CODE_PROBE(true, "Auto TPS rate is unlimited");
															#line 5025 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							}
							else
							{
															#line 783 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								CODE_PROBE(true, "GRV proxy returning tag throttle");
															#line 784 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								reply.tagThrottleInfo[tag.first] = tagItr->second;
															#line 5033 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							}
						}
						else
						{
															#line 788 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							CODE_PROBE(true, "GRV proxy expiring tag throttle");
															#line 789 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							priorityThrottledTags.erase(tagItr);
															#line 5042 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
															#line 795 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (stats->lastBatchQueueThrottled)
															#line 5049 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 797 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (now() - stats->batchThrottleStartTime > CLIENT_KNOBS->GRV_SUSTAINED_THROTTLING_THRESHOLD)
															#line 5053 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 798 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkBatchThrottled = true;
															#line 5057 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 801 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (stats->lastDefaultQueueThrottled)
															#line 5062 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 803 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (now() - stats->defaultThrottleStartTime > CLIENT_KNOBS->GRV_SUSTAINED_THROTTLING_THRESHOLD)
															#line 5066 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 807 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkBatchThrottled = true;
															#line 808 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkDefaultThrottled = true;
															#line 5072 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 811 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			request.reply.send(reply);
															#line 812 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			++stats->txnRequestOut;
															#line 5079 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 815 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<SendGrvRepliesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendGrvRepliesActorState(); static_cast<SendGrvRepliesActor*>(this)->destroy(); return 0; }
															#line 5083 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<SendGrvRepliesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendGrvRepliesActorState();
		static_cast<SendGrvRepliesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(GetReadVersionReply && _reply,int loopDepth) 
	{
															#line 741 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		GetReadVersionReply reply = _reply;
															#line 742 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		Version replyVersion = reply.version;
															#line 744 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double end = g_network->timer();
															#line 745 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for( GetReadVersionRequest const& request : requests ) {
															#line 746 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			double duration = end - request.requestTime() - request.proxyTagThrottledDuration;
															#line 747 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority == TransactionPriority::BATCH)
															#line 5105 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 748 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvBatchLatencySample.addMeasurement(duration);
															#line 5109 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 751 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority == TransactionPriority::DEFAULT)
															#line 5113 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 752 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvLatencySample.addMeasurement(duration);
															#line 5117 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 755 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.priority >= TransactionPriority::DEFAULT)
															#line 5121 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 756 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				stats->grvLatencyBands.addMeasurement(duration);
															#line 5125 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 759 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.flags & GetReadVersionRequest::FLAG_USE_MIN_KNOWN_COMMITTED_VERSION)
															#line 5129 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 761 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				reply.version = minKnownCommittedVersion;
															#line 5133 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 763 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				reply.version = replyVersion;
															#line 5139 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 765 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.midShardSize = midShardSize;
															#line 766 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.tagThrottleInfo.clear();
															#line 767 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (SERVER_KNOBS->ENABLE_VERSION_VECTOR)
															#line 5147 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 768 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->ssVersionVectorCache.getDelta(request.maxVersion, reply.ssVersionVectorDelta);
															#line 769 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->versionVectorSizeOnGRVReply->addMeasurement(reply.ssVersionVectorDelta.size());
															#line 5153 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 771 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.proxyId = grvProxyData->dbgid;
															#line 772 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			reply.proxyTagThrottledDuration = request.proxyTagThrottledDuration;
															#line 774 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (request.isTagged())
															#line 5161 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 775 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				auto& priorityThrottledTags = clientThrottledTags[request.priority];
															#line 776 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				for( auto tag : request.tags ) {
															#line 777 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					auto tagItr = priorityThrottledTags.find(tag.first);
															#line 778 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (tagItr != priorityThrottledTags.end())
															#line 5171 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 779 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						if (tagItr->second.expiration > now())
															#line 5175 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						{
															#line 780 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							if (tagItr->second.tpsRate == std::numeric_limits<double>::max())
															#line 5179 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							{
															#line 781 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								CODE_PROBE(true, "Auto TPS rate is unlimited");
															#line 5183 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							}
							else
							{
															#line 783 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								CODE_PROBE(true, "GRV proxy returning tag throttle");
															#line 784 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
								reply.tagThrottleInfo[tag.first] = tagItr->second;
															#line 5191 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
							}
						}
						else
						{
															#line 788 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							CODE_PROBE(true, "GRV proxy expiring tag throttle");
															#line 789 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
							priorityThrottledTags.erase(tagItr);
															#line 5200 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
						}
					}
				}
			}
															#line 795 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (stats->lastBatchQueueThrottled)
															#line 5207 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 797 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (now() - stats->batchThrottleStartTime > CLIENT_KNOBS->GRV_SUSTAINED_THROTTLING_THRESHOLD)
															#line 5211 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 798 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkBatchThrottled = true;
															#line 5215 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 801 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (stats->lastDefaultQueueThrottled)
															#line 5220 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 803 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (now() - stats->defaultThrottleStartTime > CLIENT_KNOBS->GRV_SUSTAINED_THROTTLING_THRESHOLD)
															#line 5224 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 807 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkBatchThrottled = true;
															#line 808 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					reply.rkDefaultThrottled = true;
															#line 5230 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 811 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			request.reply.send(reply);
															#line 812 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			++stats->txnRequestOut;
															#line 5237 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 815 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<SendGrvRepliesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendGrvRepliesActorState(); static_cast<SendGrvRepliesActor*>(this)->destroy(); return 0; }
															#line 5241 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<SendGrvRepliesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendGrvRepliesActorState();
		static_cast<SendGrvRepliesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(GetReadVersionReply const& _reply,int loopDepth) 
	{
		loopDepth = a_body1cont1(_reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetReadVersionReply && _reply,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SendGrvRepliesActor*>(this)->actor_wait_state > 0) static_cast<SendGrvRepliesActor*>(this)->actor_wait_state = 0;
		static_cast<SendGrvRepliesActor*>(this)->ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >::remove();

	}
	void a_callback_fire(ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >*,GetReadVersionReply const& value) 
	{
		fdb_probe_actor_enter("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendGrvRepliesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >*,GetReadVersionReply && value) 
	{
		fdb_probe_actor_enter("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendGrvRepliesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >*,Error err) 
	{
		fdb_probe_actor_enter("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendGrvRepliesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendGrvReplies", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<GetReadVersionReply> replyFuture;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	std::vector<GetReadVersionRequest> requests;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyStats* stats;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Version minKnownCommittedVersion;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PrioritizedTransactionTagMap<ClientTagThrottleLimits> clientThrottledTags;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t midShardSize;
															#line 5338 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via sendGrvReplies()
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class SendGrvRepliesActor final : public Actor<Void>, public ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >, public FastAllocated<SendGrvRepliesActor>, public SendGrvRepliesActorState<SendGrvRepliesActor> {
															#line 5343 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<SendGrvRepliesActor>::operator new;
	using FastAllocated<SendGrvRepliesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2675694152890119424UL, 15721776699231677952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >;
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	SendGrvRepliesActor(Future<GetReadVersionReply> const& replyFuture,std::vector<GetReadVersionRequest> const& requests,GrvProxyData* const& grvProxyData,GrvProxyStats* const& stats,Version const& minKnownCommittedVersion,PrioritizedTransactionTagMap<ClientTagThrottleLimits> const& clientThrottledTags,int64_t const& midShardSize = 0) 
															#line 5360 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   SendGrvRepliesActorState<SendGrvRepliesActor>(replyFuture, requests, grvProxyData, stats, minKnownCommittedVersion, clientThrottledTags, midShardSize),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("sendGrvReplies", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6621363967670553088UL, 9891954819986612480UL);
		ActorExecutionContextHelper __helper(static_cast<SendGrvRepliesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("sendGrvReplies");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("sendGrvReplies", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SendGrvRepliesActor, 0, GetReadVersionReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> sendGrvReplies( Future<GetReadVersionReply> const& replyFuture, std::vector<GetReadVersionRequest> const& requests, GrvProxyData* const& grvProxyData, GrvProxyStats* const& stats, Version const& minKnownCommittedVersion, PrioritizedTransactionTagMap<ClientTagThrottleLimits> const& clientThrottledTags, int64_t const& midShardSize = 0 ) {
															#line 733 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new SendGrvRepliesActor(replyFuture, requests, grvProxyData, stats, minKnownCommittedVersion, clientThrottledTags, midShardSize));
															#line 5393 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 817 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 5398 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorDDMetricsChanges()
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class MonitorDDMetricsChangesActor>
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class MonitorDDMetricsChangesActorState {
															#line 5405 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	MonitorDDMetricsChangesActorState(int64_t* const& midShardSize,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : midShardSize(midShardSize),
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   db(db),
															#line 819 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   nextRequestTimer(Never()),
															#line 820 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   nextReply(Never())
															#line 5418 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this));

	}
	~MonitorDDMetricsChangesActorState() 
	{
		fdb_probe_actor_destroy("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 822 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (db->get().distributor.present())
															#line 5433 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 823 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				nextRequestTimer = Void();
															#line 5437 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 825 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 5441 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorDDMetricsChangesActorState();
		static_cast<MonitorDDMetricsChangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 828 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 827 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (static_cast<MonitorDDMetricsChangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5474 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 839 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<Void> __when_expr_1 = nextRequestTimer;
															#line 5478 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 847 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			StrictFuture<GetDataDistributorMetricsReply> __when_expr_2 = nextReply;
															#line 5482 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
			static_cast<MonitorDDMetricsChangesActor*>(this)->actor_wait_state = 1;
															#line 828 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorDDMetricsChangesActor, 0, Void >*>(static_cast<MonitorDDMetricsChangesActor*>(this)));
															#line 839 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorDDMetricsChangesActor, 1, Void >*>(static_cast<MonitorDDMetricsChangesActor*>(this)));
															#line 847 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >*>(static_cast<MonitorDDMetricsChangesActor*>(this)));
															#line 5491 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 855 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("DDMidShardSizeUpdateFail").error(e);
															#line 856 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (e.code() != error_code_timed_out && e.code() != error_code_dd_not_found)
															#line 5515 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 857 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5519 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 859 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = delay(CLIENT_KNOBS->MID_SHARD_SIZE_MAX_STALENESS);
															#line 860 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextReply = Never();
															#line 5525 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 829 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().distributor.present())
															#line 5546 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 830 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("DataDistributorChanged", db->get().id) .detail("DDID", db->get().distributor.get().id());
															#line 832 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Void();
															#line 5552 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 834 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("DataDistributorDied", db->get().id);
															#line 835 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Never();
															#line 5560 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 837 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextReply = Never();
															#line 5564 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 829 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().distributor.present())
															#line 5573 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 830 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("DataDistributorChanged", db->get().id) .detail("DDID", db->get().distributor.get().id());
															#line 832 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Void();
															#line 5579 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 834 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("DataDistributorDied", db->get().id);
															#line 835 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextRequestTimer = Never();
															#line 5587 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 837 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextReply = Never();
															#line 5591 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 840 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = Never();
															#line 841 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().distributor.present())
															#line 5602 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 842 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextReply = brokenPromiseToNever(db->get().distributor.get().dataDistributorMetrics.getReply( GetDataDistributorMetricsRequest(normalKeys, CLIENT_KNOBS->TOO_MANY, true)));
															#line 5606 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 845 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextReply = Never();
															#line 5612 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 840 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = Never();
															#line 841 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().distributor.present())
															#line 5624 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 842 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextReply = brokenPromiseToNever(db->get().distributor.get().dataDistributorMetrics.getReply( GetDataDistributorMetricsRequest(normalKeys, CLIENT_KNOBS->TOO_MANY, true)));
															#line 5628 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		else
		{
															#line 845 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			nextReply = Never();
															#line 5634 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(GetDataDistributorMetricsReply const& reply,int loopDepth) 
	{
															#line 848 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextReply = Never();
															#line 849 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		ASSERT(reply.midShardSize.present());
															#line 850 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*midShardSize = reply.midShardSize.get();
															#line 851 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = delay(CLIENT_KNOBS->MID_SHARD_SIZE_MAX_STALENESS);
															#line 5650 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(GetDataDistributorMetricsReply && reply,int loopDepth) 
	{
															#line 848 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextReply = Never();
															#line 849 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		ASSERT(reply.midShardSize.present());
															#line 850 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		*midShardSize = reply.midShardSize.get();
															#line 851 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		nextRequestTimer = delay(CLIENT_KNOBS->MID_SHARD_SIZE_MAX_STALENESS);
															#line 5665 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorDDMetricsChangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorDDMetricsChangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorDDMetricsChangesActor*>(this)->ActorCallback< MonitorDDMetricsChangesActor, 0, Void >::remove();
		static_cast<MonitorDDMetricsChangesActor*>(this)->ActorCallback< MonitorDDMetricsChangesActor, 1, Void >::remove();
		static_cast<MonitorDDMetricsChangesActor*>(this)->ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorDDMetricsChangesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorDDMetricsChangesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >*,GetDataDistributorMetricsReply const& value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >*,GetDataDistributorMetricsReply && value) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >*,Error err) 
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t* midShardSize;
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 819 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> nextRequestTimer;
															#line 820 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<GetDataDistributorMetricsReply> nextReply;
															#line 5870 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via monitorDDMetricsChanges()
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class MonitorDDMetricsChangesActor final : public Actor<Void>, public ActorCallback< MonitorDDMetricsChangesActor, 0, Void >, public ActorCallback< MonitorDDMetricsChangesActor, 1, Void >, public ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >, public FastAllocated<MonitorDDMetricsChangesActor>, public MonitorDDMetricsChangesActorState<MonitorDDMetricsChangesActor> {
															#line 5875 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<MonitorDDMetricsChangesActor>::operator new;
	using FastAllocated<MonitorDDMetricsChangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10999244823781330176UL, 7252364158516035072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorDDMetricsChangesActor, 0, Void >;
friend struct ActorCallback< MonitorDDMetricsChangesActor, 1, Void >;
friend struct ActorCallback< MonitorDDMetricsChangesActor, 2, GetDataDistributorMetricsReply >;
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	MonitorDDMetricsChangesActor(int64_t* const& midShardSize,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 5894 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorDDMetricsChangesActorState<MonitorDDMetricsChangesActor>(midShardSize, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16350006110712773376UL, 12950337145856341760UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorDDMetricsChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorDDMetricsChanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorDDMetricsChanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorDDMetricsChangesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> monitorDDMetricsChanges( int64_t* const& midShardSize, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 818 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new MonitorDDMetricsChangesActor(midShardSize, db));
															#line 5927 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 864 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 5932 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via transactionStarter()
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class TransactionStarterActor>
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class TransactionStarterActorState {
															#line 5939 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	TransactionStarterActorState(GrvProxyInterface const& proxy,Reference<AsyncVar<ServerDBInfo> const> const& db,PromiseStream<Future<Void>> const& addActor,GrvProxyData* const& grvProxyData,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply) 
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : proxy(proxy),
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   db(db),
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   addActor(addActor),
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   grvProxyData(grvProxyData),
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   healthMetricsReply(healthMetricsReply),
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   detailedHealthMetricsReply(detailedHealthMetricsReply),
															#line 871 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   lastGRVTime(0),
															#line 872 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   GRVTimer(),
															#line 873 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   GRVBatchTime(SERVER_KNOBS->START_TRANSACTION_BATCH_INTERVAL_MIN),
															#line 875 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   transactionCount(0),
															#line 876 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchTransactionCount(0),
															#line 877 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   normalRateInfo(SERVER_KNOBS->START_TRANSACTION_RATE_WINDOW, SERVER_KNOBS->START_TRANSACTION_MAX_EMPTY_QUEUE_BUDGET, 10),
															#line 880 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchRateInfo(SERVER_KNOBS->START_TRANSACTION_RATE_WINDOW, SERVER_KNOBS->START_TRANSACTION_MAX_EMPTY_QUEUE_BUDGET, 0),
															#line 884 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   systemQueue(),
															#line 885 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   defaultQueue(),
															#line 886 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   batchQueue(),
															#line 888 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   transactionTagCounter(),
															#line 889 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   clientThrottledTags(),
															#line 891 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   normalGRVLatency(),
															#line 893 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   midShardSize(SERVER_KNOBS->MIN_SHARD_BYTES)
															#line 5984 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("transactionStarter", reinterpret_cast<unsigned long>(this));

	}
	~TransactionStarterActorState() 
	{
		fdb_probe_actor_destroy("transactionStarter", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 894 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			getCurrentLineage()->modify(&TransactionLineage::operation) = TransactionLineage::Operation::GetConsistentReadVersion;
															#line 896 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(monitorDDMetricsChanges(&midShardSize, db));
															#line 898 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(getRate(proxy.id(), db, &transactionCount, &batchTransactionCount, &normalRateInfo, &batchRateInfo, healthMetricsReply, detailedHealthMetricsReply, &transactionTagCounter, &clientThrottledTags, &grvProxyData->stats, grvProxyData));
															#line 910 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(queueGetReadVersionRequests(db, &systemQueue, &defaultQueue, &batchQueue, proxy.getConsistentReadVersion.getFuture(), GRVTimer, &lastGRVTime, &GRVBatchTime, normalGRVLatency.getFuture(), &grvProxyData->stats, &batchRateInfo, &grvProxyData->tagThrottler));
															#line 923 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 6007 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TransactionStarterActorState();
		static_cast<TransactionStarterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 928 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		ASSERT(db->get().recoveryState >= RecoveryState::ACCEPTING_COMMITS);
															#line 930 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		TraceEvent("GrvProxyReadyForTxnStarts", proxy.id());
															#line 932 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		;
															#line 6034 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 923 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!(std::find(db->get().client.grvProxies.begin(), db->get().client.grvProxies.end(), proxy) == db->get().client.grvProxies.end()))
															#line 6050 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 925 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 925 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<TransactionStarterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6058 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<TransactionStarterActor*>(this)->actor_wait_state = 1;
															#line 925 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TransactionStarterActor, 0, Void >*>(static_cast<TransactionStarterActor*>(this)));
															#line 6063 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TransactionStarterActor*>(this)->actor_wait_state > 0) static_cast<TransactionStarterActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionStarterActor*>(this)->ActorCallback< TransactionStarterActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionStarterActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TransactionStarterActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TransactionStarterActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 933 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		auto __when_expr_1 = GRVTimer.getFuture();
															#line 6179 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<Void>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<Void>>::value, "invalid type");
															#line 933 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<TransactionStarterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6183 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<TransactionStarterActor*>(this)->actor_wait_state = 2;
															#line 933 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< TransactionStarterActor, 1, Void >*>(static_cast<TransactionStarterActor*>(this)));
															#line 6188 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 935 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double t = now();
															#line 936 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double elapsed = now() - lastGRVTime;
															#line 937 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		lastGRVTime = t;
															#line 940 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (elapsed == 0)
															#line 6203 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 941 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			elapsed = 1e-15;
															#line 6207 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 944 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.update(grvProxyData->tagThrottler.releaseTransactions(elapsed, batchQueue, defaultQueue));
															#line 946 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		normalRateInfo.startReleaseWindow();
															#line 947 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchRateInfo.startReleaseWindow();
															#line 949 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.transactionLimit = normalRateInfo.getLimit();
															#line 950 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.batchTransactionLimit = batchRateInfo.getLimit();
															#line 952 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int transactionsStarted[2] = { 0, 0 };
															#line 953 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int systemTransactionsStarted[2] = { 0, 0 };
															#line 954 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int defaultPriTransactionsStarted[2] = { 0, 0 };
															#line 955 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchPriTransactionsStarted[2] = { 0, 0 };
															#line 957 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		std::vector<std::vector<GetReadVersionRequest>> start( 2);
															#line 960 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		Optional<UID> debugID;
															#line 962 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int requestsToStart = 0;
															#line 964 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		uint32_t defaultQueueSize = defaultQueue.size();
															#line 965 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		uint32_t batchQueueSize = batchQueue.size();
															#line 966 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for(;requestsToStart < SERVER_KNOBS->START_TRANSACTION_MAX_REQUESTS_TO_START;) {
															#line 967 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Deque<GetReadVersionRequest>* transactionQueue;
															#line 968 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!systemQueue.empty())
															#line 6243 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 969 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				transactionQueue = &systemQueue;
															#line 6247 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 970 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (!defaultQueue.empty())
															#line 6253 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 971 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					transactionQueue = &defaultQueue;
															#line 6257 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
				else
				{
															#line 972 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue.empty())
															#line 6263 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 973 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						transactionQueue = &batchQueue;
															#line 6267 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
						break;
					}
				}
			}
															#line 978 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			auto& req = transactionQueue->front();
															#line 979 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			int tc = req.transactionCount;
															#line 981 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority < TransactionPriority::DEFAULT && !batchRateInfo.canStart(transactionsStarted[0] + transactionsStarted[1], tc))
															#line 6281 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
				break;
			}
			else
			{
															#line 984 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority < TransactionPriority::IMMEDIATE && !normalRateInfo.canStart(transactionsStarted[0] + transactionsStarted[1], tc))
															#line 6289 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
					break;
				}
			}
															#line 989 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.debugID.present())
															#line 6296 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 990 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (!debugID.present())
															#line 6300 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 991 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					debugID = nondeterministicRandom()->randomUniqueID();
															#line 6304 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 993 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				g_traceBatch.addAttach("TransactionAttachID", req.debugID.get().first(), debugID.get().first());
															#line 6308 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 996 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			transactionsStarted[req.flags & 1] += tc;
															#line 997 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			double currentTime = g_network->timer();
															#line 998 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority >= TransactionPriority::IMMEDIATE)
															#line 6316 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 999 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				systemTransactionsStarted[req.flags & 1] += tc;
															#line 1000 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				--grvProxyData->stats.systemGRVQueueSize;
															#line 6322 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 1001 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority >= TransactionPriority::DEFAULT)
															#line 6328 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 1002 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					defaultPriTransactionsStarted[req.flags & 1] += tc;
															#line 1003 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					grvProxyData->stats.defaultTxnGRVTimeInQueue.addMeasurement(currentTime - req.requestTime());
															#line 1004 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					--grvProxyData->stats.defaultGRVQueueSize;
															#line 6336 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
				else
				{
															#line 1006 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					batchPriTransactionsStarted[req.flags & 1] += tc;
															#line 1007 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					grvProxyData->stats.batchTxnGRVTimeInQueue.addMeasurement(currentTime - req.requestTime());
															#line 1008 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					--grvProxyData->stats.batchGRVQueueSize;
															#line 6346 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 1010 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			for( auto tag : req.tags ) {
															#line 1011 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				transactionTagCounter[tag.first] += tag.second;
															#line 6353 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 1013 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			start[req.flags & 1].push_back(std::move(req));
															#line 1014 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			static_assert(GetReadVersionRequest::FLAG_CAUSAL_READ_RISKY == 1, "Implementation dependent on flag value");
															#line 1015 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			transactionQueue->pop_front();
															#line 1016 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			requestsToStart++;
															#line 6363 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1018 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!batchQueue.empty())
															#line 6367 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1019 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!grvProxyData->stats.lastBatchQueueThrottled)
															#line 6371 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1020 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.lastBatchQueueThrottled = true;
															#line 1021 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.batchThrottleStartTime = now();
															#line 6377 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
		else
		{
															#line 1024 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->stats.lastBatchQueueThrottled = false;
															#line 6384 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1026 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!defaultQueue.empty())
															#line 6388 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1027 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!grvProxyData->stats.lastDefaultQueueThrottled)
															#line 6392 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1028 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.lastDefaultQueueThrottled = true;
															#line 1029 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.defaultThrottleStartTime = now();
															#line 6398 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
		else
		{
															#line 1032 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->stats.lastDefaultQueueThrottled = false;
															#line 6405 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1035 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!systemQueue.empty() || !defaultQueue.empty() || !batchQueue.empty())
															#line 6409 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1036 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			forwardPromise( GRVTimer, delayJittered(SERVER_KNOBS->START_TRANSACTION_BATCH_QUEUE_CHECK_INTERVAL, TaskPriority::ProxyGRVTimer));
															#line 6413 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1053 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int systemTotalStarted = systemTransactionsStarted[0] + systemTransactionsStarted[1];
															#line 1054 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int normalTotalStarted = defaultPriTransactionsStarted[0] + defaultPriTransactionsStarted[1];
															#line 1055 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchTotalStarted = batchPriTransactionsStarted[0] + batchPriTransactionsStarted[1];
															#line 1057 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionCount += transactionsStarted[0] + transactionsStarted[1];
															#line 1058 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionCount += batchTotalStarted;
															#line 1060 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		normalRateInfo.endReleaseWindow( systemTotalStarted + normalTotalStarted, systemQueue.empty() && defaultQueue.empty(), elapsed);
															#line 1062 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchRateInfo.endReleaseWindow(systemTotalStarted + normalTotalStarted + batchTotalStarted, systemQueue.empty() && defaultQueue.empty() && batchQueue.empty(), elapsed);
															#line 1066 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (debugID.present())
															#line 6431 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1067 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", debugID.get().first(), "GrvProxyServer.transactionStarter.AskLiveCommittedVersionFromMaster");
															#line 6435 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1072 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int defaultGRVProcessed = 0;
															#line 1073 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchGRVProcessed = 0;
															#line 1074 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for(int i = 0;i < start.size();i++) {
															#line 1075 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (start[i].size())
															#line 6445 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1076 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				std::vector<SpanContext> spanContexts;
															#line 1077 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				spanContexts.reserve(start[i].size());
															#line 1078 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				for( const GetReadVersionRequest& request : start[i] ) {
															#line 1079 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					spanContexts.push_back(request.spanContext);
															#line 6455 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 1082 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				Future<GetReadVersionReply> readVersionReply = getLiveCommittedVersion(spanContexts, grvProxyData, i, debugID, transactionsStarted[i], systemTransactionsStarted[i], defaultPriTransactionsStarted[i], batchPriTransactionsStarted[i]);
															#line 1090 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				addActor.send(sendGrvReplies(readVersionReply, start[i], grvProxyData, &grvProxyData->stats, grvProxyData->minKnownCommittedVersion, clientThrottledTags, midShardSize));
															#line 1099 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (i == 0)
															#line 6463 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 1100 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					addActor.send(timeReply(readVersionReply, normalGRVLatency));
															#line 6467 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 1102 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				defaultGRVProcessed += defaultPriTransactionsStarted[i];
															#line 1103 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				batchGRVProcessed += batchPriTransactionsStarted[i];
															#line 6473 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
															#line 1107 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.percentageOfDefaultGRVQueueProcessed = defaultQueueSize ? (double)defaultGRVProcessed / defaultQueueSize : 1;
															#line 1109 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.percentageOfBatchGRVQueueProcessed = batchQueueSize ? (double)batchGRVProcessed / batchQueueSize : 1;
															#line 6480 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 935 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double t = now();
															#line 936 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		double elapsed = now() - lastGRVTime;
															#line 937 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		lastGRVTime = t;
															#line 940 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (elapsed == 0)
															#line 6495 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 941 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			elapsed = 1e-15;
															#line 6499 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 944 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.update(grvProxyData->tagThrottler.releaseTransactions(elapsed, batchQueue, defaultQueue));
															#line 946 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		normalRateInfo.startReleaseWindow();
															#line 947 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchRateInfo.startReleaseWindow();
															#line 949 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.transactionLimit = normalRateInfo.getLimit();
															#line 950 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.batchTransactionLimit = batchRateInfo.getLimit();
															#line 952 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int transactionsStarted[2] = { 0, 0 };
															#line 953 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int systemTransactionsStarted[2] = { 0, 0 };
															#line 954 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int defaultPriTransactionsStarted[2] = { 0, 0 };
															#line 955 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchPriTransactionsStarted[2] = { 0, 0 };
															#line 957 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		std::vector<std::vector<GetReadVersionRequest>> start( 2);
															#line 960 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		Optional<UID> debugID;
															#line 962 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int requestsToStart = 0;
															#line 964 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		uint32_t defaultQueueSize = defaultQueue.size();
															#line 965 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		uint32_t batchQueueSize = batchQueue.size();
															#line 966 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for(;requestsToStart < SERVER_KNOBS->START_TRANSACTION_MAX_REQUESTS_TO_START;) {
															#line 967 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			Deque<GetReadVersionRequest>* transactionQueue;
															#line 968 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!systemQueue.empty())
															#line 6535 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 969 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				transactionQueue = &systemQueue;
															#line 6539 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 970 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (!defaultQueue.empty())
															#line 6545 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 971 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					transactionQueue = &defaultQueue;
															#line 6549 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
				else
				{
															#line 972 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					if (!batchQueue.empty())
															#line 6555 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					{
															#line 973 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
						transactionQueue = &batchQueue;
															#line 6559 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
					}
					else
					{
						break;
					}
				}
			}
															#line 978 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			auto& req = transactionQueue->front();
															#line 979 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			int tc = req.transactionCount;
															#line 981 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority < TransactionPriority::DEFAULT && !batchRateInfo.canStart(transactionsStarted[0] + transactionsStarted[1], tc))
															#line 6573 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
				break;
			}
			else
			{
															#line 984 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority < TransactionPriority::IMMEDIATE && !normalRateInfo.canStart(transactionsStarted[0] + transactionsStarted[1], tc))
															#line 6581 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
					break;
				}
			}
															#line 989 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.debugID.present())
															#line 6588 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 990 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (!debugID.present())
															#line 6592 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 991 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					debugID = nondeterministicRandom()->randomUniqueID();
															#line 6596 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 993 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				g_traceBatch.addAttach("TransactionAttachID", req.debugID.get().first(), debugID.get().first());
															#line 6600 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 996 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			transactionsStarted[req.flags & 1] += tc;
															#line 997 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			double currentTime = g_network->timer();
															#line 998 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (req.priority >= TransactionPriority::IMMEDIATE)
															#line 6608 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 999 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				systemTransactionsStarted[req.flags & 1] += tc;
															#line 1000 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				--grvProxyData->stats.systemGRVQueueSize;
															#line 6614 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			else
			{
															#line 1001 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (req.priority >= TransactionPriority::DEFAULT)
															#line 6620 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 1002 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					defaultPriTransactionsStarted[req.flags & 1] += tc;
															#line 1003 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					grvProxyData->stats.defaultTxnGRVTimeInQueue.addMeasurement(currentTime - req.requestTime());
															#line 1004 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					--grvProxyData->stats.defaultGRVQueueSize;
															#line 6628 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
				else
				{
															#line 1006 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					batchPriTransactionsStarted[req.flags & 1] += tc;
															#line 1007 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					grvProxyData->stats.batchTxnGRVTimeInQueue.addMeasurement(currentTime - req.requestTime());
															#line 1008 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					--grvProxyData->stats.batchGRVQueueSize;
															#line 6638 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
			}
															#line 1010 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			for( auto tag : req.tags ) {
															#line 1011 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				transactionTagCounter[tag.first] += tag.second;
															#line 6645 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
															#line 1013 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			start[req.flags & 1].push_back(std::move(req));
															#line 1014 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			static_assert(GetReadVersionRequest::FLAG_CAUSAL_READ_RISKY == 1, "Implementation dependent on flag value");
															#line 1015 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			transactionQueue->pop_front();
															#line 1016 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			requestsToStart++;
															#line 6655 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1018 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!batchQueue.empty())
															#line 6659 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1019 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!grvProxyData->stats.lastBatchQueueThrottled)
															#line 6663 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1020 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.lastBatchQueueThrottled = true;
															#line 1021 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.batchThrottleStartTime = now();
															#line 6669 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
		else
		{
															#line 1024 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->stats.lastBatchQueueThrottled = false;
															#line 6676 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1026 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!defaultQueue.empty())
															#line 6680 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1027 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (!grvProxyData->stats.lastDefaultQueueThrottled)
															#line 6684 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1028 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.lastDefaultQueueThrottled = true;
															#line 1029 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				grvProxyData->stats.defaultThrottleStartTime = now();
															#line 6690 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
		else
		{
															#line 1032 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData->stats.lastDefaultQueueThrottled = false;
															#line 6697 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1035 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!systemQueue.empty() || !defaultQueue.empty() || !batchQueue.empty())
															#line 6701 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1036 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			forwardPromise( GRVTimer, delayJittered(SERVER_KNOBS->START_TRANSACTION_BATCH_QUEUE_CHECK_INTERVAL, TaskPriority::ProxyGRVTimer));
															#line 6705 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1053 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int systemTotalStarted = systemTransactionsStarted[0] + systemTransactionsStarted[1];
															#line 1054 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int normalTotalStarted = defaultPriTransactionsStarted[0] + defaultPriTransactionsStarted[1];
															#line 1055 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchTotalStarted = batchPriTransactionsStarted[0] + batchPriTransactionsStarted[1];
															#line 1057 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		transactionCount += transactionsStarted[0] + transactionsStarted[1];
															#line 1058 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchTransactionCount += batchTotalStarted;
															#line 1060 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		normalRateInfo.endReleaseWindow( systemTotalStarted + normalTotalStarted, systemQueue.empty() && defaultQueue.empty(), elapsed);
															#line 1062 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		batchRateInfo.endReleaseWindow(systemTotalStarted + normalTotalStarted + batchTotalStarted, systemQueue.empty() && defaultQueue.empty() && batchQueue.empty(), elapsed);
															#line 1066 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (debugID.present())
															#line 6723 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1067 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", debugID.get().first(), "GrvProxyServer.transactionStarter.AskLiveCommittedVersionFromMaster");
															#line 6727 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1072 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int defaultGRVProcessed = 0;
															#line 1073 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		int batchGRVProcessed = 0;
															#line 1074 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		for(int i = 0;i < start.size();i++) {
															#line 1075 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (start[i].size())
															#line 6737 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1076 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				std::vector<SpanContext> spanContexts;
															#line 1077 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				spanContexts.reserve(start[i].size());
															#line 1078 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				for( const GetReadVersionRequest& request : start[i] ) {
															#line 1079 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					spanContexts.push_back(request.spanContext);
															#line 6747 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 1082 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				Future<GetReadVersionReply> readVersionReply = getLiveCommittedVersion(spanContexts, grvProxyData, i, debugID, transactionsStarted[i], systemTransactionsStarted[i], defaultPriTransactionsStarted[i], batchPriTransactionsStarted[i]);
															#line 1090 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				addActor.send(sendGrvReplies(readVersionReply, start[i], grvProxyData, &grvProxyData->stats, grvProxyData->minKnownCommittedVersion, clientThrottledTags, midShardSize));
															#line 1099 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (i == 0)
															#line 6755 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				{
															#line 1100 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
					addActor.send(timeReply(readVersionReply, normalGRVLatency));
															#line 6759 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				}
															#line 1102 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				defaultGRVProcessed += defaultPriTransactionsStarted[i];
															#line 1103 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				batchGRVProcessed += batchPriTransactionsStarted[i];
															#line 6765 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
		}
															#line 1107 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.percentageOfDefaultGRVQueueProcessed = defaultQueueSize ? (double)defaultGRVProcessed / defaultQueueSize : 1;
															#line 1109 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData->stats.percentageOfBatchGRVQueueProcessed = batchQueueSize ? (double)batchGRVProcessed / batchQueueSize : 1;
															#line 6772 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TransactionStarterActor*>(this)->actor_wait_state > 0) static_cast<TransactionStarterActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionStarterActor*>(this)->ActorSingleCallback< TransactionStarterActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< TransactionStarterActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< TransactionStarterActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< TransactionStarterActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface proxy;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData* grvProxyData;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* healthMetricsReply;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply* detailedHealthMetricsReply;
															#line 871 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double lastGRVTime;
															#line 872 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<Void> GRVTimer;
															#line 873 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	double GRVBatchTime;
															#line 875 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t transactionCount;
															#line 876 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t batchTransactionCount;
															#line 877 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvTransactionRateInfo normalRateInfo;
															#line 880 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvTransactionRateInfo batchRateInfo;
															#line 884 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest> systemQueue;
															#line 885 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest> defaultQueue;
															#line 886 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Deque<GetReadVersionRequest> batchQueue;
															#line 888 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	TransactionTagMap<uint64_t> transactionTagCounter;
															#line 889 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PrioritizedTransactionTagMap<ClientTagThrottleLimits> clientThrottledTags;
															#line 891 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<double> normalGRVLatency;
															#line 893 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	int64_t midShardSize;
															#line 6892 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via transactionStarter()
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class TransactionStarterActor final : public Actor<Void>, public ActorCallback< TransactionStarterActor, 0, Void >, public ActorSingleCallback< TransactionStarterActor, 1, Void >, public FastAllocated<TransactionStarterActor>, public TransactionStarterActorState<TransactionStarterActor> {
															#line 6897 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<TransactionStarterActor>::operator new;
	using FastAllocated<TransactionStarterActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13342759892221011200UL, 16536324234290982912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TransactionStarterActor, 0, Void >;
friend struct ActorSingleCallback< TransactionStarterActor, 1, Void >;
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	TransactionStarterActor(GrvProxyInterface const& proxy,Reference<AsyncVar<ServerDBInfo> const> const& db,PromiseStream<Future<Void>> const& addActor,GrvProxyData* const& grvProxyData,GetHealthMetricsReply* const& healthMetricsReply,GetHealthMetricsReply* const& detailedHealthMetricsReply) 
															#line 6915 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   TransactionStarterActorState<TransactionStarterActor>(proxy, db, addActor, grvProxyData, healthMetricsReply, detailedHealthMetricsReply),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("transactionStarter", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10700198538979324160UL, 3557432467917975808UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionStarterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("transactionStarter");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("transactionStarter", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TransactionStarterActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< TransactionStarterActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] static Future<Void> transactionStarter( GrvProxyInterface const& proxy, Reference<AsyncVar<ServerDBInfo> const> const& db, PromiseStream<Future<Void>> const& addActor, GrvProxyData* const& grvProxyData, GetHealthMetricsReply* const& healthMetricsReply, GetHealthMetricsReply* const& detailedHealthMetricsReply ) {
															#line 865 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new TransactionStarterActor(proxy, db, addActor, grvProxyData, healthMetricsReply, detailedHealthMetricsReply));
															#line 6949 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 1113 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 6954 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via grvProxyServerCore()
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GrvProxyServerCoreActor>
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GrvProxyServerCoreActorState {
															#line 6961 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyServerCoreActorState(GrvProxyInterface const& proxy,MasterInterface const& master,LifetimeToken const& masterLifetime,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : proxy(proxy),
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   master(master),
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   masterLifetime(masterLifetime),
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   db(db),
															#line 1118 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   grvProxyData(proxy.id(), master, proxy.getConsistentReadVersion, db),
															#line 1120 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   addActor(),
															#line 1121 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   onError(actorCollection(addActor.getFuture())),
															#line 1123 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   healthMetricsReply(),
															#line 1124 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   detailedHealthMetricsReply()
															#line 6984 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("grvProxyServerCore", reinterpret_cast<unsigned long>(this));

	}
	~GrvProxyServerCoreActorState() 
	{
		fdb_probe_actor_destroy("grvProxyServerCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1126 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(waitFailureServer(proxy.waitFailure.getFuture()));
															#line 1127 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(traceRole(Role::GRV_PROXY, proxy.id()));
															#line 1129 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("GrvProxyServerCore", proxy.id()) .detail("MasterId", master.id()) .detail("MasterLifetime", masterLifetime.toString()) .detail("RecoveryCount", db->get().recoveryCount);
															#line 1135 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 7005 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GrvProxyServerCoreActorState();
		static_cast<GrvProxyServerCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1140 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		dbInfoChange = grvProxyData.db->onChange();
															#line 1141 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData.logSystem = ILogSystem::fromServerDBInfo(proxy.id(), grvProxyData.db->get(), false, addActor);
															#line 1143 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData.updateLatencyBandConfig(grvProxyData.db->get().latencyBandConfig);
															#line 1145 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		addActor.send(transactionStarter( proxy, grvProxyData.db, addActor, &grvProxyData, &healthMetricsReply, &detailedHealthMetricsReply));
															#line 1147 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		addActor.send(healthMetricsRequestServer(proxy, &healthMetricsReply, &detailedHealthMetricsReply));
															#line 1148 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		addActor.send(globalConfigRequestServer(&grvProxyData, proxy));
															#line 1150 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (SERVER_KNOBS->REQUIRED_MIN_RECOVERY_DURATION > 0)
															#line 7040 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1151 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			addActor.send(lastCommitUpdater(&grvProxyData, addActor));
															#line 7044 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1154 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		;
															#line 7048 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1135 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!(!(masterLifetime.isEqual(grvProxyData.db->get().masterLifetime) && grvProxyData.db->get().recoveryState >= RecoveryState::ACCEPTING_COMMITS)))
															#line 7064 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1137 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_0 = grvProxyData.db->onChange();
															#line 1137 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7072 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state = 1;
															#line 1137 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GrvProxyServerCoreActor, 0, Void >*>(static_cast<GrvProxyServerCoreActor*>(this)));
															#line 7077 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state > 0) static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state = 0;
		static_cast<GrvProxyServerCoreActor*>(this)->ActorCallback< GrvProxyServerCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GrvProxyServerCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1155 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_1 = dbInfoChange;
															#line 1154 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7195 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
															#line 1165 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_2 = onError;
															#line 7199 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when2(__when_expr_2.get(), loopDepth); };
		static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state = 2;
															#line 1155 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GrvProxyServerCoreActor, 1, Void >*>(static_cast<GrvProxyServerCoreActor*>(this)));
															#line 1165 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GrvProxyServerCoreActor, 2, Void >*>(static_cast<GrvProxyServerCoreActor*>(this)));
															#line 7206 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 1156 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		dbInfoChange = grvProxyData.db->onChange();
															#line 1158 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (masterLifetime.isEqual(grvProxyData.db->get().masterLifetime) && grvProxyData.db->get().recoveryState >= RecoveryState::RECOVERY_TRANSACTION)
															#line 7223 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1160 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData.logSystem = ILogSystem::fromServerDBInfo(proxy.id(), grvProxyData.db->get(), false, addActor);
															#line 7227 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1163 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData.updateLatencyBandConfig(grvProxyData.db->get().latencyBandConfig);
															#line 7231 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 1156 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		dbInfoChange = grvProxyData.db->onChange();
															#line 1158 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (masterLifetime.isEqual(grvProxyData.db->get().masterLifetime) && grvProxyData.db->get().recoveryState >= RecoveryState::RECOVERY_TRANSACTION)
															#line 7242 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1160 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			grvProxyData.logSystem = ILogSystem::fromServerDBInfo(proxy.id(), grvProxyData.db->get(), false, addActor);
															#line 7246 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1163 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		grvProxyData.updateLatencyBandConfig(grvProxyData.db->get().latencyBandConfig);
															#line 7250 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state > 0) static_cast<GrvProxyServerCoreActor*>(this)->actor_wait_state = 0;
		static_cast<GrvProxyServerCoreActor*>(this)->ActorCallback< GrvProxyServerCoreActor, 1, Void >::remove();
		static_cast<GrvProxyServerCoreActor*>(this)->ActorCallback< GrvProxyServerCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GrvProxyServerCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GrvProxyServerCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GrvProxyServerCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface proxy;
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	MasterInterface master;
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	LifetimeToken masterLifetime;
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 1118 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyData grvProxyData;
															#line 1120 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 1121 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> onError;
															#line 1123 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply healthMetricsReply;
															#line 1124 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GetHealthMetricsReply detailedHealthMetricsReply;
															#line 1140 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> dbInfoChange;
															#line 7408 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via grvProxyServerCore()
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GrvProxyServerCoreActor final : public Actor<Void>, public ActorCallback< GrvProxyServerCoreActor, 0, Void >, public ActorCallback< GrvProxyServerCoreActor, 1, Void >, public ActorCallback< GrvProxyServerCoreActor, 2, Void >, public FastAllocated<GrvProxyServerCoreActor>, public GrvProxyServerCoreActorState<GrvProxyServerCoreActor> {
															#line 7413 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GrvProxyServerCoreActor>::operator new;
	using FastAllocated<GrvProxyServerCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18377343445714768640UL, 18204353666256388864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GrvProxyServerCoreActor, 0, Void >;
friend struct ActorCallback< GrvProxyServerCoreActor, 1, Void >;
friend struct ActorCallback< GrvProxyServerCoreActor, 2, Void >;
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyServerCoreActor(GrvProxyInterface const& proxy,MasterInterface const& master,LifetimeToken const& masterLifetime,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 7432 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GrvProxyServerCoreActorState<GrvProxyServerCoreActor>(proxy, master, masterLifetime, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("grvProxyServerCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6046041477836266752UL, 214675386595290880UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("grvProxyServerCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("grvProxyServerCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GrvProxyServerCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GrvProxyServerCoreActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> grvProxyServerCore( GrvProxyInterface const& proxy, MasterInterface const& master, LifetimeToken const& masterLifetime, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 1114 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GrvProxyServerCoreActor(proxy, master, masterLifetime, db));
															#line 7466 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 1168 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 7471 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via checkRemoved()
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class CheckRemovedActor>
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class CheckRemovedActorState {
															#line 7478 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	CheckRemovedActorState(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,GrvProxyInterface const& myInterface) 
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : db(db),
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   recoveryCount(recoveryCount),
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   myInterface(myInterface)
															#line 7489 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	~CheckRemovedActorState() 
	{
		fdb_probe_actor_destroy("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1172 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			;
															#line 7504 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckRemovedActorState();
		static_cast<CheckRemovedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1173 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (db->get().recoveryCount >= recoveryCount && std::find(db->get().client.grvProxies.begin(), db->get().client.grvProxies.end(), myInterface) == db->get().client.grvProxies.end())
															#line 7534 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		{
															#line 1176 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			return a_body1Catch1(worker_removed(), std::max(0, loopDepth - 1));
															#line 7538 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		}
															#line 1178 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 1178 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7544 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CheckRemovedActor*>(this)->actor_wait_state = 1;
															#line 1178 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckRemovedActor, 0, Void >*>(static_cast<CheckRemovedActor*>(this)));
															#line 7549 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state > 0) static_cast<CheckRemovedActor*>(this)->actor_wait_state = 0;
		static_cast<CheckRemovedActor*>(this)->ActorCallback< CheckRemovedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckRemovedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	uint64_t recoveryCount;
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface myInterface;
															#line 7647 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via checkRemoved()
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class CheckRemovedActor final : public Actor<Void>, public ActorCallback< CheckRemovedActor, 0, Void >, public FastAllocated<CheckRemovedActor>, public CheckRemovedActorState<CheckRemovedActor> {
															#line 7652 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<CheckRemovedActor>::operator new;
	using FastAllocated<CheckRemovedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17366727163919118592UL, 17867894835493071872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckRemovedActor, 0, Void >;
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	CheckRemovedActor(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,GrvProxyInterface const& myInterface) 
															#line 7669 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   CheckRemovedActorState<CheckRemovedActor>(db, recoveryCount, myInterface),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15518436168711072512UL, 6662910151609573120UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkRemoved");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckRemovedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> checkRemoved( Reference<AsyncVar<ServerDBInfo> const> const& db, uint64_t const& recoveryCount, GrvProxyInterface const& myInterface ) {
															#line 1169 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new CheckRemovedActor(db, recoveryCount, myInterface));
															#line 7702 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 1181 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"

															#line 7707 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
namespace {
// This generated class is to be used only via grvProxyServer()
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
template <class GrvProxyServerActor>
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GrvProxyServerActorState {
															#line 7714 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyServerActorState(GrvProxyInterface const& proxy,InitializeGrvProxyRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		 : proxy(proxy),
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   req(req),
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		   db(db)
															#line 7725 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
	{
		fdb_probe_actor_create("grvProxyServer", reinterpret_cast<unsigned long>(this));

	}
	~GrvProxyServerActorState() 
	{
		fdb_probe_actor_destroy("grvProxyServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1186 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				core = grvProxyServerCore(proxy, req.master, req.masterLifetime, db);
															#line 1187 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				StrictFuture<Void> __when_expr_0 = core || checkRemoved(db, req.recoveryCount, proxy);
															#line 1187 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				if (static_cast<GrvProxyServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7745 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GrvProxyServerActor*>(this)->actor_wait_state = 1;
															#line 1187 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GrvProxyServerActor, 0, Void >*>(static_cast<GrvProxyServerActor*>(this)));
															#line 7750 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GrvProxyServerActorState();
		static_cast<GrvProxyServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1201 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
		if (!static_cast<GrvProxyServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GrvProxyServerActorState(); static_cast<GrvProxyServerActor*>(this)->destroy(); return 0; }
															#line 7779 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		new (&static_cast<GrvProxyServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GrvProxyServerActorState();
		static_cast<GrvProxyServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1189 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			TraceEvent("GrvProxyTerminated", proxy.id()).errorUnsuppressed(e);
															#line 1190 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			ASSERT(e.code() != error_code_broken_promise);
															#line 1192 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			CODE_PROBE(e.code() == error_code_master_failed, "GrvProxyServer master failed");
															#line 1193 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			CODE_PROBE(e.code() == error_code_tlog_failed, "GrvProxyServer tlog failed");
															#line 1194 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
			if (e.code() != error_code_worker_removed && e.code() != error_code_tlog_stopped && e.code() != error_code_tlog_failed && e.code() != error_code_coordinators_changed && e.code() != error_code_coordinated_state_conflict && e.code() != error_code_new_coordinators_timed_out && e.code() != error_code_master_failed)
															#line 7800 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			{
															#line 1198 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7804 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GrvProxyServerActor*>(this)->actor_wait_state > 0) static_cast<GrvProxyServerActor*>(this)->actor_wait_state = 0;
		static_cast<GrvProxyServerActor*>(this)->ActorCallback< GrvProxyServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GrvProxyServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GrvProxyServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GrvProxyServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("grvProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyInterface proxy;
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	InitializeGrvProxyRequest req;
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 1186 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	Future<Void> core;
															#line 7924 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
};
// This generated class is to be used only via grvProxyServer()
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
class GrvProxyServerActor final : public Actor<Void>, public ActorCallback< GrvProxyServerActor, 0, Void >, public FastAllocated<GrvProxyServerActor>, public GrvProxyServerActorState<GrvProxyServerActor> {
															#line 7929 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
public:
	using FastAllocated<GrvProxyServerActor>::operator new;
	using FastAllocated<GrvProxyServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12502007746757981440UL, 14235842700822828032UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GrvProxyServerActor, 0, Void >;
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	GrvProxyServerActor(GrvProxyInterface const& proxy,InitializeGrvProxyRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 7946 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
		 : Actor<Void>(),
		   GrvProxyServerActorState<GrvProxyServerActor>(proxy, req, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("grvProxyServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13905335545483414272UL, 8240318245689050112UL);
		ActorExecutionContextHelper __helper(static_cast<GrvProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("grvProxyServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("grvProxyServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GrvProxyServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
[[nodiscard]] Future<Void> grvProxyServer( GrvProxyInterface const& proxy, InitializeGrvProxyRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 1182 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
	return Future<Void>(new GrvProxyServerActor(proxy, req, db));
															#line 7979 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyServer.actor.g.cpp"
}

#line 1203 "/codebuild/output/src3861208925/src/github.com/apple/foundationdb/fdbserver/GrvProxyServer.actor.cpp"
