#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
/*
 * KeyValueStoreMemory.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2026 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/Notified.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/ServerDBInfo.actor.h"
#include "fdbserver/DeltaTree.h"
#include "fdbserver/IDiskQueue.h"
#include "fdbserver/IKeyValueContainer.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/RadixTree.h"
#include "fdbserver/TransactionStoreMutationTracking.h"
#include "flow/ActorCollection.h"
#include "flow/EncryptUtils.h"
#include "flow/Knobs.h"
#include "flow/actorcompiler.h" // This must be the last #include.

#define OP_DISK_OVERHEAD (sizeof(OpHeader) + 1)
static_assert(sizeof(uint32_t) == 4);

template <typename Container>
class KeyValueStoreMemory final : public IKeyValueStore, NonCopyable {
public:
	KeyValueStoreMemory(IDiskQueue* log,
	                    Reference<AsyncVar<ServerDBInfo> const> db,
	                    UID id,
	                    int64_t memoryLimit,
	                    KeyValueStoreType storeType,
	                    bool disableSnapshot,
	                    bool replaceContent,
	                    bool exactRecovery);

	bool getReplaceContent() const override { return replaceContent; }
	// IClosable
	Future<Void> getError() const override { return log->getError(); }
	Future<Void> onClosed() const override { return log->onClosed(); }
	void dispose() override {
		recovering.cancel();
		log->dispose();
		if (reserved_buffer != nullptr) {
			delete[] reserved_buffer;
			reserved_buffer = nullptr;
		}
		delete this;
	}
	void close() override {
		recovering.cancel();
		log->close();
		if (reserved_buffer != nullptr) {
			delete[] reserved_buffer;
			reserved_buffer = nullptr;
		}
		delete this;
	}

	// IKeyValueStore
	KeyValueStoreType getType() const override { return type; }

	std::tuple<size_t, size_t, size_t> getSize() const override { return data.size(); }

	int64_t getAvailableSize() const {
		int64_t residentSize = data.sumTo(data.end()) + queue.totalSize() + // doesn't account for overhead in queue
		                       transactionSize;

		return memoryLimit - residentSize;
	}

	StorageBytes getStorageBytes() const override {
		StorageBytes diskQueueBytes = log->getStorageBytes();

		// Try to bound how many in-memory bytes we might need to write to disk if we commit() now
		int64_t uncommittedBytes = queue.totalSize() + transactionSize;

		// Check that we have enough space in memory and on disk
		int64_t freeSize = std::min(getAvailableSize(), diskQueueBytes.free / 4 - uncommittedBytes);
		int64_t availableSize = std::min(getAvailableSize(), diskQueueBytes.available / 4 - uncommittedBytes);
		int64_t totalSize = std::min(memoryLimit, diskQueueBytes.total / 4 - uncommittedBytes);

		return StorageBytes(std::max((int64_t)0, freeSize),
		                    std::max((int64_t)0, totalSize),
		                    diskQueueBytes.used,
		                    std::max((int64_t)0, availableSize));
	}

	void semiCommit() {
		transactionSize += queue.totalSize();
		if (transactionSize > 0.5 * committedDataSize) {
			transactionIsLarge = true;
			TraceEvent("KVSMemSwitchingToLargeTransactionMode", id)
			    .detail("TransactionSize", transactionSize)
			    .detail("DataSize", committedDataSize);
			CODE_PROBE(true, "KeyValueStoreMemory switching to large transaction mode");
			CODE_PROBE(committedDataSize > 1e3,
			           "KeyValueStoreMemory switching to large transaction mode with committed data");
		}

		int64_t bytesWritten = commit_queue(queue, true);
		committedWriteBytes += bytesWritten;
	}

	void set(KeyValueRef keyValue, const Arena* arena) override {
		// A commit that occurs with no available space returns Never, so we can throw out all modifications
		if (getAvailableSize() <= 0)
			return;

		if (transactionIsLarge) {
			data.insert(keyValue.key, keyValue.value);
		} else {
			queue.set(keyValue, arena);
			if (recovering.isReady() && !disableSnapshot) {
				semiCommit();
			}
		}
	}

	void clear(KeyRangeRef range, const Arena* arena) override {
		// A commit that occurs with no available space returns Never, so we can throw out all modifications
		if (getAvailableSize() <= 0)
			return;

		if (transactionIsLarge) {
			data.erase(data.lower_bound(range.begin), data.lower_bound(range.end));
		} else {
			queue.clear(range, arena);
			if (recovering.isReady() && !disableSnapshot) {
				semiCommit();
			}
		}
	}

	Future<Void> commit(bool sequential) override {
		if (getAvailableSize() <= 0) {
			TraceEvent(SevError, "KeyValueStoreMemory_OutOfSpace", id).log();
			return Never();
		}

		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndCommit(this, sequential);

		if (!disableSnapshot && replaceContent && !firstCommitWithSnapshot) {
			transactionSize += SERVER_KNOBS->REPLACE_CONTENTS_BYTES;
			committedWriteBytes += SERVER_KNOBS->REPLACE_CONTENTS_BYTES;
			semiCommit();
		}

		if (transactionIsLarge) {
			fullSnapshot(data);
			resetSnapshot = true;
			committedWriteBytes = notifiedCommittedWriteBytes.get();
			overheadWriteBytes = 0;

			if (disableSnapshot) {
				return Void();
			}
			log_op(OpCommit, StringRef(), StringRef());
		} else {
			int64_t bytesWritten = commit_queue(queue, !disableSnapshot, sequential);

			if (disableSnapshot) {
				return Void();
			}

			if (bytesWritten > 0 || committedWriteBytes > notifiedCommittedWriteBytes.get()) {
				committedWriteBytes += bytesWritten + overheadWriteBytes +
				                       OP_DISK_OVERHEAD; // OP_DISK_OVERHEAD is for the following log_op(OpCommit)
				notifiedCommittedWriteBytes.set(committedWriteBytes); // This set will cause snapshot items to be
				                                                      // written, so it must happen before the OpCommit
				log_op(OpCommit, StringRef(), StringRef());
				overheadWriteBytes = log->getCommitOverhead();
			}
		}

		auto c = log->commit();

		committedDataSize = data.sumTo(data.end());
		transactionSize = 0;
		transactionIsLarge = false;
		firstCommitWithSnapshot = false;

		addActor.send(commitAndUpdateVersions(this, c, previousSnapshotEnd));
		return c;
	}

	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadValue(this, key, options);

		auto it = data.find(key);
		if (it == data.end())
			return Optional<Value>();
		return Optional<Value>(it.getValue());
	}

	Future<Optional<Value>> readValuePrefix(KeyRef key, int maxLength, Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadValuePrefix(this, key, maxLength, options);

		auto it = data.find(key);
		if (it == data.end())
			return Optional<Value>();
		auto val = it.getValue();
		if (maxLength < val.size()) {
			return Optional<Value>(val.substr(0, maxLength));
		} else {
			return Optional<Value>(val);
		}
	}

	// If rowLimit>=0, reads first rows sorted ascending, otherwise reads last rows sorted descending
	// The total size of the returned value (less the last entry) will be less than byteLimit
	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit,
	                              int byteLimit,
	                              Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadRange(this, keys, rowLimit, byteLimit, options);

		RangeResult result;
		if (rowLimit == 0) {
			return result;
		}

		if (rowLimit > 0) {
			auto it = data.lower_bound(keys.begin);
			while (it != data.end() && rowLimit && byteLimit > 0) {
				StringRef tempKey = it.getKey(reserved_buffer);
				if (tempKey >= keys.end)
					break;

				byteLimit -= sizeof(KeyValueRef) + tempKey.size() + it.getValue().size();
				result.push_back_deep(result.arena(), KeyValueRef(tempKey, it.getValue()));
				++it;
				--rowLimit;
			}
		} else {
			rowLimit = -rowLimit;
			auto it = data.previous(data.lower_bound(keys.end));
			while (it != data.end() && rowLimit && byteLimit > 0) {
				StringRef tempKey = it.getKey(reserved_buffer);
				if (tempKey < keys.begin)
					break;

				byteLimit -= sizeof(KeyValueRef) + tempKey.size() + it.getValue().size();
				result.push_back_deep(result.arena(), KeyValueRef(tempKey, it.getValue()));
				it = data.previous(it);
				--rowLimit;
			}
		}

		result.more = rowLimit == 0 || byteLimit <= 0;
		return result;
	}

	void resyncLog() override {
		ASSERT(recovering.isReady());
		resetSnapshot = true;
		log_op(OpSnapshotAbort, StringRef(), StringRef());
	}

	void enableSnapshot() override { disableSnapshot = false; }

	int uncommittedBytes() { return queue.totalSize(); }

private:
	enum OpType {
		OpSet,
		OpClear,
		OpClearToEnd,
		OpSnapshotItem,
		OpSnapshotEnd,
		OpSnapshotAbort, // terminate an in progress snapshot in order to start a full snapshot
		OpCommit, // only in log, not in queue
		OpRollback, // only in log, not in queue
		OpSnapshotItemDelta,
		OpEncrypted_Deprecated // deprecated since we now store the encryption status in the first bit of the opType
	};

	struct OpRef {
		OpType op;
		StringRef p1, p2;
		OpRef() {}
		OpRef(Arena& a, OpRef const& o) : op(o.op), p1(a, o.p1), p2(a, o.p2) {}
		size_t expectedSize() const { return p1.expectedSize() + p2.expectedSize(); }
	};
	struct OpHeader {
		uint32_t op;
		int len1, len2;
	};

	struct OpQueue {
		OpQueue() : numBytes(0) {}

		int totalSize() const { return numBytes; }

		void clear() {
			numBytes = 0;
			operations = Standalone<VectorRef<OpRef>>();
			arenas.clear();
		}

		void rollback() { clear(); }

		void set(KeyValueRef keyValue, const Arena* arena = nullptr) {
			queue_op(OpSet, keyValue.key, keyValue.value, arena);
		}

		void clear(KeyRangeRef range, const Arena* arena = nullptr) {
			queue_op(OpClear, range.begin, range.end, arena);
		}

		void clear_to_end(StringRef fromKey, const Arena* arena = nullptr) {
			queue_op(OpClearToEnd, fromKey, StringRef(), arena);
		}

		void queue_op(OpType op, StringRef p1, StringRef p2, const Arena* arena) {
			numBytes += p1.size() + p2.size() + sizeof(OpHeader) + sizeof(OpRef);

			OpRef r;
			r.op = op;
			r.p1 = p1;
			r.p2 = p2;
			if (arena == nullptr) {
				operations.push_back_deep(operations.arena(), r);
			} else {
				operations.push_back(operations.arena(), r);
				arenas.push_back(*arena);
			}
		}

		const OpRef* begin() { return operations.begin(); }

		const OpRef* end() { return operations.end(); }

	private:
		Standalone<VectorRef<OpRef>> operations;
		uint64_t numBytes;
		std::vector<Arena> arenas;
	};
	KeyValueStoreType type;
	UID id;

	Container data;
	// reserved buffer for snapshot/fullsnapshot
	uint8_t* reserved_buffer;

	OpQueue queue; // mutations not yet commit()ted
	IDiskQueue* log;
	Reference<AsyncVar<ServerDBInfo> const> db;
	Future<Void> recovering, snapshotting;
	int64_t committedWriteBytes;
	int64_t overheadWriteBytes;
	NotifiedVersion notifiedCommittedWriteBytes;
	Key recoveredSnapshotKey; // After recovery, the next key in the currently uncompleted snapshot
	IDiskQueue::location
	    currentSnapshotEnd; // The end of the most recently completed snapshot (this snapshot cannot be discarded)
	IDiskQueue::location previousSnapshotEnd; // The end of the second most recently completed snapshot (on commit, this
	                                          // snapshot can be discarded)
	PromiseStream<Future<Void>> addActor;
	Future<Void> commitActors;

	int64_t committedDataSize;
	int64_t transactionSize;
	bool transactionIsLarge;

	bool resetSnapshot; // Set to true after a fullSnapshot is performed.  This causes the regular snapshot mechanism to
	                    // restart
	bool disableSnapshot;
	bool replaceContent;
	bool firstCommitWithSnapshot;
	int snapshotCount;

	int64_t memoryLimit; // The upper limit on the memory used by the store (excluding, possibly, some clear operations)
	std::vector<std::pair<KeyValueMapPair, uint64_t>> dataSets;

	int64_t commit_queue(OpQueue& ops, bool log, bool sequential = false) {
		int64_t total = 0, count = 0;
		IDiskQueue::location log_location = 0;

		for (auto o = ops.begin(); o != ops.end(); ++o) {
			++count;
			total += o->p1.size() + o->p2.size() + OP_DISK_OVERHEAD;
			if (o->op == OpSet) {
				if (sequential) {
					KeyValueMapPair pair(o->p1, o->p2);
					dataSets.emplace_back(pair, pair.arena.getSize() + data.getElementBytes());
				} else {
					data.insert(o->p1, o->p2);
				}
			} else if (o->op == OpClear) {
				if (sequential) {
					data.insert(dataSets);
					dataSets.clear();
				}
				data.erase(data.lower_bound(o->p1), data.lower_bound(o->p2));
			} else if (o->op == OpClearToEnd) {
				if (sequential) {
					data.insert(dataSets);
					dataSets.clear();
				}
				data.erase(data.lower_bound(o->p1), data.end());
			} else
				ASSERT(false);
			if (log)
				log_location = log_op(o->op, o->p1, o->p2);
		}
		if (sequential) {
			data.insert(dataSets);
			dataSets.clear();
		}

		bool ok = count < 1e6;
		if (!ok) {
			TraceEvent(/*ok ? SevInfo : */ SevWarnAlways, "KVSMemCommitQueue", id)
			    .detail("Bytes", total)
			    .detail("Log", log)
			    .detail("Ops", count)
			    .detail("LastLoggedLocation", log_location)
			    .detail("Details", count);
		}

		ops.clear();
		return total;
	}

	// Data format:
	// +-------------+-------------+-------------+--------+--------+-----------+
	// | opType      | len1        | len2        | param2 | param2 |   \x01    |
	// | sizeof(int) | sizeof(int) | sizeof(int) | len1   | len2   |  1 byte   |
	// +-------------+-------------+-------------+--------+--------+-----------+
	IDiskQueue::location log_op(OpType op, StringRef v1, StringRef v2) {
		uint32_t opType = (uint32_t)op;
		OpHeader h = { opType, v1.size(), v2.size() };
		log->push(StringRef((const uint8_t*)&h, sizeof(h)));
		log->push(v1);
		log->push(v2);
		IDiskQueue::location loc = log->push("\x01"_sr); // Changes here should be reflected in OP_DISK_OVERHEAD
		DEBUG_TRANSACTION_STATE_STORE("LogOp", v1, id, loc);
		return loc;
	}

																#line 470 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via readOpData()
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class ReadOpDataActor>
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class ReadOpDataActorState {
															#line 476 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	ReadOpDataActorState(KeyValueStoreMemory* const& self,OpHeader const& h,bool* const& isZeroFilled,int* const& zeroFillSize) 
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   h(h),
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   isZeroFilled(isZeroFilled),
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   zeroFillSize(zeroFillSize),
															#line 472 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   remainingBytes(h.len1 + h.len2 + 1)
															#line 491 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readOpData", reinterpret_cast<unsigned long>(this));

	}
	~ReadOpDataActorState() 
	{
		fdb_probe_actor_destroy("readOpData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Standalone<StringRef>> __when_expr_0 = self->log->readNext(remainingBytes);
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<ReadOpDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 508 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadOpDataActor*>(this)->actor_wait_state = 1;
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*>(static_cast<ReadOpDataActor*>(this)));
															#line 513 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadOpDataActorState();
		static_cast<ReadOpDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 474 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(data.size() <= remainingBytes);
															#line 475 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		*zeroFillSize = remainingBytes - data.size();
															#line 476 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (*zeroFillSize == 0)
															#line 540 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 477 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			*isZeroFilled = (data[data.size() - 1] == 0);
															#line 544 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 479 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<ReadOpDataActor*>(this)->SAV<Standalone<StringRef>>::futures) { (void)(data); this->~ReadOpDataActorState(); static_cast<ReadOpDataActor*>(this)->destroy(); return 0; }
															#line 548 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<ReadOpDataActor*>(this)->SAV< Standalone<StringRef> >::value()) Standalone<StringRef>(std::move(data)); // state_var_RVO
		this->~ReadOpDataActorState();
		static_cast<ReadOpDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		data = __data;
															#line 560 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state > 0) static_cast<ReadOpDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadOpDataActor*>(this)->ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpHeader h;
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool* isZeroFilled;
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int* zeroFillSize;
															#line 472 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int remainingBytes;
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> data;
															#line 647 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readOpData()
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class ReadOpDataActor final : public Actor<Standalone<StringRef>>, public ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >, public FastAllocated<ReadOpDataActor>, public ReadOpDataActorState<ReadOpDataActor> {
															#line 652 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadOpDataActor>::operator new;
	using FastAllocated<ReadOpDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16672218737862936320UL, 11962145171546676736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<StringRef>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >;
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	ReadOpDataActor(KeyValueStoreMemory* const& self,OpHeader const& h,bool* const& isZeroFilled,int* const& zeroFillSize) 
															#line 669 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Standalone<StringRef>>(),
		   ReadOpDataActorState<ReadOpDataActor>(self, h, isZeroFilled, zeroFillSize),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16657327943355146496UL, 2700099178469709056UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readOpData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Standalone<StringRef>> readOpData( KeyValueStoreMemory* const& self, OpHeader const& h, bool* const& isZeroFilled, int* const& zeroFillSize ) {
															#line 468 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Standalone<StringRef>>(new ReadOpDataActor(self, h, isZeroFilled, zeroFillSize));
															#line 701 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 481 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

																#line 706 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via recover()
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class RecoverActor>
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RecoverActorState {
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RecoverActorState(KeyValueStoreMemory* const& self,bool const& exactRecovery) 
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   exactRecovery(exactRecovery)
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("recover", reinterpret_cast<unsigned long>(this));

	}
	~RecoverActorState() 
	{
		fdb_probe_actor_destroy("recover", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 483 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			;
															#line 736 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 488 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedNextKey = self->recoveredSnapshotKey;
															#line 489 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedPrevSnapshotEnd = self->previousSnapshotEnd = self->log->getNextReadLocation();
															#line 491 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedSnapshotEnd = self->currentSnapshotEnd = uncommittedPrevSnapshotEnd;
															#line 493 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		zeroFillSize = 0;
															#line 494 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgSnapshotItemCount = 0;
															#line 495 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgSnapshotEndCount = 0;
															#line 496 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgMutationCount = 0;
															#line 497 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgCommitCount = 0;
															#line 498 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		startt = now();
															#line 499 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgid = self->id;
															#line 501 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		loggingDelay = delay(1.0);
															#line 503 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		recoveryQueue = OpQueue();
															#line 504 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = OpHeader();
															#line 505 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKey = Standalone<StringRef>();
															#line 506 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		isZeroFilled = bool();
															#line 508 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemRecoveryStarted", self->id).detail("SnapshotEndLocation", uncommittedSnapshotEnd);
															#line 796 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		try {
															#line 511 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			;
															#line 800 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 670 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			bool ok = e.code() == error_code_operation_cancelled || e.code() == error_code_file_not_found || e.code() == error_code_disk_adapter_reset;
															#line 672 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(ok ? SevInfo : SevError, "ErrorDuringRecovery", dbgid).errorUnsuppressed(e);
															#line 673 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (e.code() != error_code_disk_adapter_reset)
															#line 826 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 674 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 830 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
															#line 676 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->data.clear();
															#line 677 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->dataSets.clear();
															#line 836 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 641 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (zeroFillSize)
															#line 851 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 642 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (exactRecovery)
															#line 855 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 643 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				TraceEvent(SevError, "KVSMemExpectedExact", self->id).log();
															#line 644 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(false);
															#line 861 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
															#line 647 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			CODE_PROBE(true, "Fixing a partial commit at the end of the KeyValueStoreMemory log");
															#line 648 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			for(int i = 0;i < zeroFillSize;i++) {
															#line 649 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->log->push(StringRef((const uint8_t*)"", 1));
															#line 869 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
		}
															#line 655 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log_op(OpRollback, StringRef(), StringRef());
															#line 657 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->committedDataSize = self->data.sumTo(self->data.end());
															#line 659 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemRecovered", self->id) .detail("SnapshotItems", dbgSnapshotItemCount) .detail("SnapshotEnd", dbgSnapshotEndCount) .detail("Mutations", dbgMutationCount) .detail("Commits", dbgCommitCount) .detail("TimeTaken", now() - startt);
															#line 666 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->semiCommit();
															#line 668 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<RecoverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RecoverActorState(); static_cast<RecoverActor*>(this)->destroy(); return 0; }
															#line 882 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<RecoverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		{
															#line 513 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Standalone<StringRef>> __when_expr_0 = self->log->readNext(sizeof(OpHeader));
															#line 513 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 904 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 1;
															#line 513 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 0, Standalone<StringRef> >*>(static_cast<RecoverActor*>(this)));
															#line 909 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = readOpData(self, h, &isZeroFilled, &zeroFillSize);
															#line 533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 934 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 2;
															#line 533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 1, Standalone<StringRef> >*>(static_cast<RecoverActor*>(this)));
															#line 939 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Standalone<StringRef> const& data,int loopDepth) 
	{
															#line 514 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (data.size() != sizeof(OpHeader))
															#line 948 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 515 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (data.size())
															#line 952 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 516 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				CODE_PROBE( true, "zero fill partial header in KeyValueStoreMemory", probe::decoration::rare);
															#line 518 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memset(&h, 0, sizeof(OpHeader));
															#line 519 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memcpy(&h, data.begin(), data.size());
															#line 520 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				zeroFillSize = sizeof(OpHeader) - data.size() + h.len1 + h.len2 + 1;
															#line 962 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "Non-header sized data read") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 966 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 530 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = *(OpHeader*)data.begin();
															#line 531 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(h.op != OpEncrypted_Deprecated);
															#line 973 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Standalone<StringRef> && data,int loopDepth) 
	{
															#line 514 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (data.size() != sizeof(OpHeader))
															#line 982 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 515 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (data.size())
															#line 986 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 516 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				CODE_PROBE( true, "zero fill partial header in KeyValueStoreMemory", probe::decoration::rare);
															#line 518 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memset(&h, 0, sizeof(OpHeader));
															#line 519 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memcpy(&h, data.begin(), data.size());
															#line 520 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				zeroFillSize = sizeof(OpHeader) - data.size() + h.len1 + h.len2 + 1;
															#line 996 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "Non-header sized data read") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 1000 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 530 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = *(OpHeader*)data.begin();
															#line 531 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(h.op != OpEncrypted_Deprecated);
															#line 1007 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Standalone<StringRef> const& data,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(data, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Standalone<StringRef> && data,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(data), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 0, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont6(int loopDepth) 
	{
															#line 534 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (zeroFillSize > 0)
															#line 1091 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 535 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "data specified by header does not exist") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("OpCode", h.op) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 1095 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 545 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!isZeroFilled)
															#line 1100 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 546 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StringRef p1 = data.substr(0, h.len1);
															#line 547 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StringRef p2 = data.substr(h.len1, h.len2);
															#line 549 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			DEBUG_TRANSACTION_STATE_STORE("Recover", p1, self->id);
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (h.op == OpSnapshotItem || h.op == OpSnapshotItemDelta)
															#line 1110 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 559 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (h.op == OpSnapshotItemDelta)
															#line 1114 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 560 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					ASSERT(p1.size() > 1);
															#line 562 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					int borrowed = *(uint8_t*)p1.begin();
															#line 563 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					ASSERT(borrowed <= lastSnapshotKey.size());
															#line 565 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					StringRef suffix = p1.substr(1);
															#line 567 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					Arena& dataArena = *(Arena*)&data.arena();
															#line 568 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					p1 = makeString(borrowed + suffix.size(), dataArena);
															#line 570 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(p1), lastSnapshotKey.begin(), borrowed);
															#line 572 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(p1) + borrowed, suffix.begin(), suffix.size());
															#line 1132 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 574 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (p1 >= uncommittedNextKey)
															#line 1136 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 575 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					recoveryQueue.clear( KeyRangeRef(uncommittedNextKey, p1), &uncommittedNextKey .arena());
															#line 1140 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 579 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				recoveryQueue.set(KeyValueRef(p1, p2), &data.arena());
															#line 580 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uncommittedNextKey = keyAfter(p1);
															#line 581 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				++dbgSnapshotItemCount;
															#line 582 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKey = Key(p1, data.arena());
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			}
			else
			{
															#line 583 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (h.op == OpSnapshotEnd || h.op == OpSnapshotAbort)
															#line 1156 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 584 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					TraceEvent("RecSnapshotEnd", self->id) .detail("NextKey", uncommittedNextKey) .detail("Nextlocation", self->log->getNextReadLocation()) .detail("IsSnapshotEnd", h.op == OpSnapshotEnd);
															#line 589 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					if (h.op == OpSnapshotEnd)
															#line 1162 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					{
															#line 590 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						uncommittedPrevSnapshotEnd = uncommittedSnapshotEnd;
															#line 591 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						uncommittedSnapshotEnd = self->log->getNextReadLocation();
															#line 592 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						recoveryQueue.clear_to_end(uncommittedNextKey, &uncommittedNextKey.arena());
															#line 1170 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					}
															#line 595 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uncommittedNextKey = Key();
															#line 596 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					lastSnapshotKey = Key();
															#line 597 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++dbgSnapshotEndCount;
															#line 1178 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 598 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					if (h.op == OpSet)
															#line 1184 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					{
															#line 599 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						recoveryQueue.set(KeyValueRef(p1, p2), &data.arena());
															#line 600 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						++dbgMutationCount;
															#line 1190 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					}
					else
					{
															#line 601 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						if (h.op == OpClear)
															#line 1196 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
						{
															#line 602 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							recoveryQueue.clear(KeyRangeRef(p1, p2), &data.arena());
															#line 603 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							++dbgMutationCount;
															#line 1202 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
						}
						else
						{
															#line 604 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							if (h.op == OpClearToEnd)
															#line 1208 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
							{
															#line 605 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
								recoveryQueue.clear_to_end(p1, &data.arena());
															#line 1212 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
							}
							else
							{
															#line 606 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
								if (h.op == OpCommit)
															#line 1218 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
								{
															#line 607 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->commit_queue(recoveryQueue, false);
															#line 608 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									++dbgCommitCount;
															#line 609 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->recoveredSnapshotKey = uncommittedNextKey;
															#line 610 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->previousSnapshotEnd = uncommittedPrevSnapshotEnd;
															#line 611 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->currentSnapshotEnd = uncommittedSnapshotEnd;
															#line 1230 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
								}
								else
								{
															#line 612 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									if (h.op == OpRollback)
															#line 1236 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
									{
															#line 613 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										recoveryQueue.rollback();
															#line 614 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										TraceEvent("KVSMemRecSnapshotRollback", self->id).detail("NextKey", uncommittedNextKey);
															#line 615 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedNextKey = self->recoveredSnapshotKey;
															#line 616 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedPrevSnapshotEnd = self->previousSnapshotEnd;
															#line 617 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedSnapshotEnd = self->currentSnapshotEnd;
															#line 1248 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
									}
									else
									{
															#line 619 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										ASSERT(false);
															#line 1254 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
									}
								}
							}
						}
					}
				}
			}
		}
		else
		{
															#line 621 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoverySkippedZeroFill", self->id) .detail("PayloadSize", data.size()) .detail("ExpectedSize", h.len1 + h.len2 + 1) .detail("OpCode", h.op) .detail("EndsAt", self->log->getNextReadLocation());
															#line 1267 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 628 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (loggingDelay.isReady())
															#line 1271 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 629 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryLogSnap", self->id) .detail("SnapshotItems", dbgSnapshotItemCount) .detail("SnapshotEnd", dbgSnapshotEndCount) .detail("Mutations", dbgMutationCount) .detail("Commits", dbgCommitCount) .detail("EndsAt", self->log->getNextReadLocation());
															#line 635 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			loggingDelay = delay(1.0);
															#line 1277 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 638 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield();
															#line 638 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1283 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont6when1(__when_expr_2.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 3;
															#line 638 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 2, Void >*>(static_cast<RecoverActor*>(this)));
															#line 1288 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		data = __data;
															#line 1297 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1loopBody1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont7(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont7(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RecoverActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool exactRecovery;
															#line 488 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key uncommittedNextKey;
															#line 489 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location uncommittedPrevSnapshotEnd;
															#line 491 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location uncommittedSnapshotEnd;
															#line 493 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int zeroFillSize;
															#line 494 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgSnapshotItemCount;
															#line 495 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgSnapshotEndCount;
															#line 496 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgMutationCount;
															#line 497 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgCommitCount;
															#line 498 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	double startt;
															#line 499 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	UID dbgid;
															#line 501 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Future<Void> loggingDelay;
															#line 503 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpQueue recoveryQueue;
															#line 504 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpHeader h;
															#line 505 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> lastSnapshotKey;
															#line 506 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool isZeroFilled;
															#line 533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> data;
															#line 1495 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via recover()
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RecoverActor final : public Actor<Void>, public ActorCallback< RecoverActor, 0, Standalone<StringRef> >, public ActorCallback< RecoverActor, 1, Standalone<StringRef> >, public ActorCallback< RecoverActor, 2, Void >, public FastAllocated<RecoverActor>, public RecoverActorState<RecoverActor> {
															#line 1500 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<RecoverActor>::operator new;
	using FastAllocated<RecoverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6661633172976988416UL, 7801490911650264832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RecoverActor, 0, Standalone<StringRef> >;
friend struct ActorCallback< RecoverActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< RecoverActor, 2, Void >;
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RecoverActor(KeyValueStoreMemory* const& self,bool const& exactRecovery) 
															#line 1519 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RecoverActorState<RecoverActor>(self, exactRecovery),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7334007120712012544UL, 4865582193762289920UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("recover");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RecoverActor, 0, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RecoverActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RecoverActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> recover( KeyValueStoreMemory* const& self, bool const& exactRecovery ) {
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new RecoverActor(self, exactRecovery));
															#line 1553 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 681 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

	// Snapshots an entire data set
	void fullSnapshot(Container& snapshotData) {
		previousSnapshotEnd = log_op(OpSnapshotAbort, StringRef(), StringRef());
		replaceContent = false;

		// Clear everything since we are about to write the whole database
		log_op(OpClearToEnd, allKeys.begin, StringRef());

		int count = 0;
		int64_t snapshotSize = 0;
		for (auto kv = snapshotData.begin(); kv != snapshotData.end(); ++kv) {
			StringRef tempKey = kv.getKey(reserved_buffer);
			DEBUG_TRANSACTION_STATE_STORE("FullSnapshot", tempKey, id);
			log_op(OpSnapshotItem, tempKey, kv.getValue());
			snapshotSize += tempKey.size() + kv.getValue().size() + OP_DISK_OVERHEAD;
			++count;
		}

		TraceEvent("FullSnapshotEnd", id)
		    .detail("PreviousSnapshotEndLoc", previousSnapshotEnd)
		    .detail("SnapshotSize", snapshotSize)
		    .detail("SnapshotElements", count);

		currentSnapshotEnd = log_op(OpSnapshotEnd, StringRef(), StringRef());
	}

																#line 1584 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via snapshot()
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class SnapshotActor>
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class SnapshotActorState {
															#line 1590 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	SnapshotActorState(KeyValueStoreMemory* const& self) 
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self)
															#line 1597 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("snapshot", reinterpret_cast<unsigned long>(this));

	}
	~SnapshotActorState() 
	{
		fdb_probe_actor_destroy("snapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 709 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 709 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<SnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1614 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SnapshotActor*>(this)->actor_wait_state = 1;
															#line 709 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SnapshotActor, 0, Void >*>(static_cast<SnapshotActor*>(this)));
															#line 1619 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SnapshotActorState();
		static_cast<SnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKey = self->recoveredSnapshotKey;
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKeyAfter = false;
															#line 713 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotTotalWrittenBytes = 0;
															#line 714 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = 0;
															#line 715 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapItems = 0;
															#line 716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotBytes = 0;
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyA = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyB = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 723 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyUsingA = true;
															#line 725 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemStartingSnapshot", self->id).detail("StartKey", nextKey);
															#line 727 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		;
															#line 1662 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKey = self->recoveredSnapshotKey;
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKeyAfter = false;
															#line 713 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotTotalWrittenBytes = 0;
															#line 714 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = 0;
															#line 715 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapItems = 0;
															#line 716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotBytes = 0;
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyA = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyB = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 723 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyUsingA = true;
															#line 725 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemStartingSnapshot", self->id).detail("StartKey", nextKey);
															#line 727 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		;
															#line 1691 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SnapshotActor*>(this)->actor_wait_state > 0) static_cast<SnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<SnapshotActor*>(this)->ActorCallback< SnapshotActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SnapshotActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 728 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->notifiedCommittedWriteBytes.whenAtLeast(snapshotTotalWrittenBytes + 1);
															#line 728 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<SnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SnapshotActor*>(this)->actor_wait_state = 2;
															#line 728 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SnapshotActor, 1, Void >*>(static_cast<SnapshotActor*>(this)));
															#line 1789 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 730 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (self->resetSnapshot)
															#line 1798 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 731 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKey = Key();
															#line 732 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKeyAfter = false;
															#line 733 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapItems = 0;
															#line 734 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapshotBytes = 0;
															#line 735 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->resetSnapshot = false;
															#line 1810 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 738 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		auto next = nextKeyAfter ? self->data.upper_bound(nextKey) : self->data.lower_bound(nextKey);
															#line 739 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		int diff = self->notifiedCommittedWriteBytes.get() - snapshotTotalWrittenBytes;
															#line 740 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (diff > lastDiff && diff > 5e7)
															#line 1818 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 741 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(SevWarnAlways, "ManyWritesAtOnce", self->id) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotWrites", snapshotTotalWrittenBytes) .detail("Diff", diff) .detail("LastOperationWasASnapshot", nextKey == Key() && !nextKeyAfter);
															#line 1822 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 746 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = diff;
															#line 751 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		bool useDelta = false;
															#line 754 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		for(;;) {
															#line 756 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (next == self->data.end())
															#line 1832 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 759 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				useDelta = false;
															#line 761 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				auto thisSnapshotEnd = self->log_op(OpSnapshotEnd, StringRef(), StringRef());
															#line 762 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DisabledTraceEvent("SnapshotEnd", self->id) .detail("CurrentSnapshotEndLoc", self->currentSnapshotEnd) .detail("PreviousSnapshotEndLoc", self->previousSnapshotEnd) .detail("ThisSnapshotEnd", thisSnapshotEnd) .detail("Items", snapItems) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotSize", snapshotBytes);
															#line 770 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(thisSnapshotEnd >= self->currentSnapshotEnd);
															#line 771 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->previousSnapshotEnd = self->currentSnapshotEnd;
															#line 772 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->currentSnapshotEnd = thisSnapshotEnd;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (++self->snapshotCount == 2)
															#line 1848 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 775 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->replaceContent = false;
															#line 1852 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 778 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems = 0;
															#line 779 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes = 0;
															#line 780 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += OP_DISK_OVERHEAD;
															#line 783 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 1862 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					next = self->data.begin();
															#line 1866 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 787 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = Key();
															#line 788 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = false;
															#line 1874 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					break;
				}
			}
			else
			{
															#line 794 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				Key& destKey = lastSnapshotKeyUsingA ? lastSnapshotKeyA : lastSnapshotKeyB;
															#line 797 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				KeyRef tempKey = next.getKey(mutateString(destKey));
															#line 798 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int opKeySize = tempKey.size();
															#line 803 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (tempKey.begin() != destKey.begin())
															#line 1888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 804 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(destKey), tempKey.begin(), tempKey.size());
															#line 1892 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 810 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				destKey.contents() = KeyRef(destKey.begin(), tempKey.size());
															#line 812 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DEBUG_TRANSACTION_STATE_STORE("SnapshotItem", destKey.toString(), self->id);
															#line 815 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int commonPrefix;
															#line 816 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (useDelta && SERVER_KNOBS->PREFIX_COMPRESS_KVS_MEM_SNAPSHOTS)
															#line 1902 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 817 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = commonPrefixLength(lastSnapshotKeyA, lastSnapshotKeyB);
															#line 1906 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 819 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = 0;
															#line 820 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					useDelta = true;
															#line 1914 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 825 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (commonPrefix > 1)
															#line 1918 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = std::min<int>(commonPrefix, std::numeric_limits<uint8_t>::max());
															#line 831 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t& prefixLength = mutateString(destKey)[commonPrefix - 1];
															#line 832 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t backupByte = prefixLength;
															#line 833 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = commonPrefix;
															#line 835 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					opKeySize = opKeySize - commonPrefix + 1;
															#line 836 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					KeyRef opKey(&prefixLength, opKeySize);
															#line 837 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItemDelta, opKey, next.getValue());
															#line 840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = backupByte;
															#line 1936 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 842 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItem, tempKey, next.getValue());
															#line 1942 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 845 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems++;
															#line 846 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uint64_t opBytes = opKeySize + next.getValue().size() + OP_DISK_OVERHEAD;
															#line 847 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes += opBytes;
															#line 848 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += opBytes;
															#line 849 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKeyUsingA = !lastSnapshotKeyUsingA;
															#line 852 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 1956 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 853 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++next;
															#line 1960 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 856 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = destKey;
															#line 857 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = true;
															#line 1968 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					break;
				}
			}
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 730 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (self->resetSnapshot)
															#line 1981 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 731 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKey = Key();
															#line 732 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKeyAfter = false;
															#line 733 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapItems = 0;
															#line 734 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapshotBytes = 0;
															#line 735 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->resetSnapshot = false;
															#line 1993 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 738 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		auto next = nextKeyAfter ? self->data.upper_bound(nextKey) : self->data.lower_bound(nextKey);
															#line 739 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		int diff = self->notifiedCommittedWriteBytes.get() - snapshotTotalWrittenBytes;
															#line 740 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (diff > lastDiff && diff > 5e7)
															#line 2001 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		{
															#line 741 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(SevWarnAlways, "ManyWritesAtOnce", self->id) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotWrites", snapshotTotalWrittenBytes) .detail("Diff", diff) .detail("LastOperationWasASnapshot", nextKey == Key() && !nextKeyAfter);
															#line 2005 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		}
															#line 746 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = diff;
															#line 751 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		bool useDelta = false;
															#line 754 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		for(;;) {
															#line 756 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (next == self->data.end())
															#line 2015 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			{
															#line 759 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				useDelta = false;
															#line 761 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				auto thisSnapshotEnd = self->log_op(OpSnapshotEnd, StringRef(), StringRef());
															#line 762 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DisabledTraceEvent("SnapshotEnd", self->id) .detail("CurrentSnapshotEndLoc", self->currentSnapshotEnd) .detail("PreviousSnapshotEndLoc", self->previousSnapshotEnd) .detail("ThisSnapshotEnd", thisSnapshotEnd) .detail("Items", snapItems) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotSize", snapshotBytes);
															#line 770 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(thisSnapshotEnd >= self->currentSnapshotEnd);
															#line 771 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->previousSnapshotEnd = self->currentSnapshotEnd;
															#line 772 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->currentSnapshotEnd = thisSnapshotEnd;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (++self->snapshotCount == 2)
															#line 2031 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 775 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->replaceContent = false;
															#line 2035 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 778 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems = 0;
															#line 779 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes = 0;
															#line 780 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += OP_DISK_OVERHEAD;
															#line 783 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2045 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					next = self->data.begin();
															#line 2049 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 787 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = Key();
															#line 788 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = false;
															#line 2057 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					break;
				}
			}
			else
			{
															#line 794 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				Key& destKey = lastSnapshotKeyUsingA ? lastSnapshotKeyA : lastSnapshotKeyB;
															#line 797 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				KeyRef tempKey = next.getKey(mutateString(destKey));
															#line 798 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int opKeySize = tempKey.size();
															#line 803 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (tempKey.begin() != destKey.begin())
															#line 2071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 804 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(destKey), tempKey.begin(), tempKey.size());
															#line 2075 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 810 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				destKey.contents() = KeyRef(destKey.begin(), tempKey.size());
															#line 812 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DEBUG_TRANSACTION_STATE_STORE("SnapshotItem", destKey.toString(), self->id);
															#line 815 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int commonPrefix;
															#line 816 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (useDelta && SERVER_KNOBS->PREFIX_COMPRESS_KVS_MEM_SNAPSHOTS)
															#line 2085 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 817 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = commonPrefixLength(lastSnapshotKeyA, lastSnapshotKeyB);
															#line 2089 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 819 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = 0;
															#line 820 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					useDelta = true;
															#line 2097 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 825 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (commonPrefix > 1)
															#line 2101 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = std::min<int>(commonPrefix, std::numeric_limits<uint8_t>::max());
															#line 831 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t& prefixLength = mutateString(destKey)[commonPrefix - 1];
															#line 832 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t backupByte = prefixLength;
															#line 833 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = commonPrefix;
															#line 835 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					opKeySize = opKeySize - commonPrefix + 1;
															#line 836 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					KeyRef opKey(&prefixLength, opKeySize);
															#line 837 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItemDelta, opKey, next.getValue());
															#line 840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = backupByte;
															#line 2119 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 842 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItem, tempKey, next.getValue());
															#line 2125 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
															#line 845 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems++;
															#line 846 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uint64_t opBytes = opKeySize + next.getValue().size() + OP_DISK_OVERHEAD;
															#line 847 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes += opBytes;
															#line 848 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += opBytes;
															#line 849 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKeyUsingA = !lastSnapshotKeyUsingA;
															#line 852 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2139 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				{
															#line 853 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++next;
															#line 2143 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
				}
				else
				{
															#line 856 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = destKey;
															#line 857 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = true;
															#line 2151 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
					break;
				}
			}
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SnapshotActor*>(this)->actor_wait_state > 0) static_cast<SnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<SnapshotActor*>(this)->ActorCallback< SnapshotActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SnapshotActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key nextKey;
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool nextKeyAfter;
															#line 713 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	uint64_t snapshotTotalWrittenBytes;
															#line 714 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int lastDiff;
															#line 715 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int snapItems;
															#line 716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	uint64_t snapshotBytes;
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key lastSnapshotKeyA;
															#line 722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key lastSnapshotKeyB;
															#line 723 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool lastSnapshotKeyUsingA;
															#line 2255 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via snapshot()
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class SnapshotActor final : public Actor<Void>, public ActorCallback< SnapshotActor, 0, Void >, public ActorCallback< SnapshotActor, 1, Void >, public FastAllocated<SnapshotActor>, public SnapshotActorState<SnapshotActor> {
															#line 2260 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<SnapshotActor>::operator new;
	using FastAllocated<SnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3334185210556514560UL, 16020860020165297664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SnapshotActor, 0, Void >;
friend struct ActorCallback< SnapshotActor, 1, Void >;
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	SnapshotActor(KeyValueStoreMemory* const& self) 
															#line 2278 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   SnapshotActorState<SnapshotActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7446917862669081600UL, 3777547187618145280UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("snapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SnapshotActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SnapshotActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> snapshot( KeyValueStoreMemory* const& self ) {
															#line 708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new SnapshotActor(self));
															#line 2311 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 864 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

																#line 2316 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via waitAndReadValue()
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadValueActor>
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValueActorState {
															#line 2322 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValueActorState(KeyValueStoreMemory* const& self,Key const& key,Optional<ReadOptions> const& options) 
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   key(key),
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 2333 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitAndReadValue", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadValueActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2350 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadValueActor*>(this)->actor_wait_state = 1;
															#line 868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadValueActor, 0, Void >*>(static_cast<WaitAndReadValueActor*>(this)));
															#line 2355 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 869 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValueActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValue(key, options).get()); this->~WaitAndReadValueActorState(); static_cast<WaitAndReadValueActor*>(this)->destroy(); return 0; }
															#line 2378 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadValueActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValue(key, options).get());
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 869 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValueActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValue(key, options).get()); this->~WaitAndReadValueActorState(); static_cast<WaitAndReadValueActor*>(this)->destroy(); return 0; }
															#line 2390 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadValueActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValue(key, options).get());
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadValueActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadValueActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadValueActor*>(this)->ActorCallback< WaitAndReadValueActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadValueActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadValueActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadValueActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key key;
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 2479 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitAndReadValue()
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValueActor final : public Actor<Optional<Value>>, public ActorCallback< WaitAndReadValueActor, 0, Void >, public FastAllocated<WaitAndReadValueActor>, public WaitAndReadValueActorState<WaitAndReadValueActor> {
															#line 2484 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitAndReadValueActor>::operator new;
	using FastAllocated<WaitAndReadValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17926342292069857792UL, 16846540001148446976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadValueActor, 0, Void >;
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValueActor(KeyValueStoreMemory* const& self,Key const& key,Optional<ReadOptions> const& options) 
															#line 2501 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Optional<Value>>(),
		   WaitAndReadValueActorState<WaitAndReadValueActor>(self, key, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(866617973594577152UL, 12678521189896779008UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadValueActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Optional<Value>> waitAndReadValue( KeyValueStoreMemory* const& self, Key const& key, Optional<ReadOptions> const& options ) {
															#line 865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Optional<Value>>(new WaitAndReadValueActor(self, key, options));
															#line 2533 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 2537 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via waitAndReadValuePrefix()
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadValuePrefixActor>
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValuePrefixActorState {
															#line 2543 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValuePrefixActorState(KeyValueStoreMemory* const& self,Key const& key,int const& maxLength,Optional<ReadOptions> const& options) 
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   key(key),
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   maxLength(maxLength),
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 2556 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadValuePrefixActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 875 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 875 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2573 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state = 1;
															#line 875 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*>(static_cast<WaitAndReadValuePrefixActor*>(this)));
															#line 2578 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 876 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValuePrefixActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get()); this->~WaitAndReadValuePrefixActorState(); static_cast<WaitAndReadValuePrefixActor*>(this)->destroy(); return 0; }
															#line 2601 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadValuePrefixActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get());
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 876 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValuePrefixActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get()); this->~WaitAndReadValuePrefixActorState(); static_cast<WaitAndReadValuePrefixActor*>(this)->destroy(); return 0; }
															#line 2613 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadValuePrefixActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get());
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadValuePrefixActor*>(this)->ActorCallback< WaitAndReadValuePrefixActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key key;
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int maxLength;
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 2704 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitAndReadValuePrefix()
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValuePrefixActor final : public Actor<Optional<Value>>, public ActorCallback< WaitAndReadValuePrefixActor, 0, Void >, public FastAllocated<WaitAndReadValuePrefixActor>, public WaitAndReadValuePrefixActorState<WaitAndReadValuePrefixActor> {
															#line 2709 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitAndReadValuePrefixActor>::operator new;
	using FastAllocated<WaitAndReadValuePrefixActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18160865417970796288UL, 4182594956798324224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadValuePrefixActor, 0, Void >;
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValuePrefixActor(KeyValueStoreMemory* const& self,Key const& key,int const& maxLength,Optional<ReadOptions> const& options) 
															#line 2726 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Optional<Value>>(),
		   WaitAndReadValuePrefixActorState<WaitAndReadValuePrefixActor>(self, key, maxLength, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1850494756222781696UL, 8275989234370189312UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadValuePrefix");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Optional<Value>> waitAndReadValuePrefix( KeyValueStoreMemory* const& self, Key const& key, int const& maxLength, Optional<ReadOptions> const& options ) {
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Optional<Value>>(new WaitAndReadValuePrefixActor(self, key, maxLength, options));
															#line 2758 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 2762 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via waitAndReadRange()
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadRangeActor>
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadRangeActorState {
															#line 2768 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadRangeActorState(KeyValueStoreMemory* const& self,KeyRange const& keys,int const& rowLimit,int const& byteLimit,Optional<ReadOptions> const& options) 
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   keys(keys),
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   rowLimit(rowLimit),
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   byteLimit(byteLimit),
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 2783 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitAndReadRange", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadRangeActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2800 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state = 1;
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadRangeActor, 0, Void >*>(static_cast<WaitAndReadRangeActor*>(this)));
															#line 2805 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 884 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get()); this->~WaitAndReadRangeActorState(); static_cast<WaitAndReadRangeActor*>(this)->destroy(); return 0; }
															#line 2828 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get());
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 884 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get()); this->~WaitAndReadRangeActorState(); static_cast<WaitAndReadRangeActor*>(this)->destroy(); return 0; }
															#line 2840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndReadRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get());
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadRangeActor*>(this)->ActorCallback< WaitAndReadRangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyRange keys;
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int rowLimit;
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int byteLimit;
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 2933 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitAndReadRange()
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadRangeActor final : public Actor<RangeResult>, public ActorCallback< WaitAndReadRangeActor, 0, Void >, public FastAllocated<WaitAndReadRangeActor>, public WaitAndReadRangeActorState<WaitAndReadRangeActor> {
															#line 2938 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitAndReadRangeActor>::operator new;
	using FastAllocated<WaitAndReadRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10910857951963205888UL, 17739213161288503040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RangeResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadRangeActor, 0, Void >;
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadRangeActor(KeyValueStoreMemory* const& self,KeyRange const& keys,int const& rowLimit,int const& byteLimit,Optional<ReadOptions> const& options) 
															#line 2955 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<RangeResult>(),
		   WaitAndReadRangeActorState<WaitAndReadRangeActor>(self, keys, rowLimit, byteLimit, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12404310014652120320UL, 13044338866170580992UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadRangeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<RangeResult> waitAndReadRange( KeyValueStoreMemory* const& self, KeyRange const& keys, int const& rowLimit, int const& byteLimit, Optional<ReadOptions> const& options ) {
															#line 878 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<RangeResult>(new WaitAndReadRangeActor(self, keys, rowLimit, byteLimit, options));
															#line 2987 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 2991 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via waitAndCommit()
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndCommitActor>
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndCommitActorState {
															#line 2997 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndCommitActorState(KeyValueStoreMemory* const& self,bool const& sequential) 
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   sequential(sequential)
															#line 3006 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitAndCommit", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndCommitActorState() 
	{
		fdb_probe_actor_destroy("waitAndCommit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 887 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 887 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3023 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 1;
															#line 887 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 0, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3028 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->commit(sequential);
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3053 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 2;
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 1, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3058 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->commit(sequential);
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3069 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 2;
															#line 888 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 1, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3074 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state > 0) static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndCommitActor*>(this)->ActorCallback< WaitAndCommitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndCommitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 889 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitAndCommitActorState(); static_cast<WaitAndCommitActor*>(this)->destroy(); return 0; }
															#line 3158 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 889 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitAndCommitActorState(); static_cast<WaitAndCommitActor*>(this)->destroy(); return 0; }
															#line 3170 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<WaitAndCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state > 0) static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndCommitActor*>(this)->ActorCallback< WaitAndCommitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitAndCommitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool sequential;
															#line 3257 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitAndCommit()
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndCommitActor final : public Actor<Void>, public ActorCallback< WaitAndCommitActor, 0, Void >, public ActorCallback< WaitAndCommitActor, 1, Void >, public FastAllocated<WaitAndCommitActor>, public WaitAndCommitActorState<WaitAndCommitActor> {
															#line 3262 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitAndCommitActor>::operator new;
	using FastAllocated<WaitAndCommitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15639365834822076672UL, 684639862004138496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndCommitActor, 0, Void >;
friend struct ActorCallback< WaitAndCommitActor, 1, Void >;
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndCommitActor(KeyValueStoreMemory* const& self,bool const& sequential) 
															#line 3280 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   WaitAndCommitActorState<WaitAndCommitActor>(self, sequential),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2545403049674954496UL, 15925460025590017024UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndCommit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndCommitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitAndCommitActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> waitAndCommit( KeyValueStoreMemory* const& self, bool const& sequential ) {
															#line 886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new WaitAndCommitActor(self, sequential));
															#line 3313 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 3317 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
// This generated class is to be used only via commitAndUpdateVersions()
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class CommitAndUpdateVersionsActor>
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class CommitAndUpdateVersionsActorState {
															#line 3323 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	CommitAndUpdateVersionsActorState(KeyValueStoreMemory* const& self,Future<Void> const& commit,IDiskQueue::location const& location) 
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   commit(commit),
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   location(location)
															#line 3334 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this));

	}
	~CommitAndUpdateVersionsActorState() 
	{
		fdb_probe_actor_destroy("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 894 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = commit;
															#line 894 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3351 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state = 1;
															#line 894 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*>(static_cast<CommitAndUpdateVersionsActor*>(this)));
															#line 3356 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 895 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log->pop(location);
															#line 896 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<CommitAndUpdateVersionsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitAndUpdateVersionsActorState(); static_cast<CommitAndUpdateVersionsActor*>(this)->destroy(); return 0; }
															#line 3381 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<CommitAndUpdateVersionsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 895 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log->pop(location);
															#line 896 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<CommitAndUpdateVersionsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitAndUpdateVersionsActorState(); static_cast<CommitAndUpdateVersionsActor*>(this)->destroy(); return 0; }
															#line 3395 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		new (&static_cast<CommitAndUpdateVersionsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state > 0) static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state = 0;
		static_cast<CommitAndUpdateVersionsActor*>(this)->ActorCallback< CommitAndUpdateVersionsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Future<Void> commit;
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location location;
															#line 3484 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
};
// This generated class is to be used only via commitAndUpdateVersions()
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class CommitAndUpdateVersionsActor final : public Actor<Void>, public ActorCallback< CommitAndUpdateVersionsActor, 0, Void >, public FastAllocated<CommitAndUpdateVersionsActor>, public CommitAndUpdateVersionsActorState<CommitAndUpdateVersionsActor> {
															#line 3489 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
public:
	using FastAllocated<CommitAndUpdateVersionsActor>::operator new;
	using FastAllocated<CommitAndUpdateVersionsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14593787358216936704UL, 17756667018120472576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CommitAndUpdateVersionsActor, 0, Void >;
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	CommitAndUpdateVersionsActor(KeyValueStoreMemory* const& self,Future<Void> const& commit,IDiskQueue::location const& location) 
															#line 3506 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CommitAndUpdateVersionsActorState<CommitAndUpdateVersionsActor>(self, commit, location),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5732578105651983360UL, 2263904282379971584UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("commitAndUpdateVersions");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> commitAndUpdateVersions( KeyValueStoreMemory* const& self, Future<Void> const& commit, IDiskQueue::location const& location ) {
															#line 891 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new CommitAndUpdateVersionsActor(self, commit, location));
															#line 3538 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp.py_gen"
}

#line 898 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
};

template <typename Container>
KeyValueStoreMemory<Container>::KeyValueStoreMemory(IDiskQueue* log,
                                                    Reference<AsyncVar<ServerDBInfo> const> db,
                                                    UID id,
                                                    int64_t memoryLimit,
                                                    KeyValueStoreType storeType,
                                                    bool disableSnapshot,
                                                    bool replaceContent,
                                                    bool exactRecovery)
  : type(storeType), id(id), log(log), db(db), committedWriteBytes(0), overheadWriteBytes(0), currentSnapshotEnd(-1),
    previousSnapshotEnd(-1), committedDataSize(0), transactionSize(0), transactionIsLarge(false), resetSnapshot(false),
    disableSnapshot(disableSnapshot), replaceContent(replaceContent), firstCommitWithSnapshot(true), snapshotCount(0),
    memoryLimit(memoryLimit) {
	// create reserved buffer for radixtree store type
	this->reserved_buffer =
	    (storeType == KeyValueStoreType::MEMORY) ? nullptr : new uint8_t[CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT];
	if (this->reserved_buffer != nullptr)
		memset(this->reserved_buffer, 0, CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);

	recovering = recover(this, exactRecovery);
	snapshotting = snapshot(this);
	commitActors = actorCollection(addActor.getFuture());
}

IKeyValueStore* keyValueStoreMemory(std::string const& basename,
                                    UID logID,
                                    int64_t memoryLimit,
                                    std::string ext,
                                    KeyValueStoreType storeType) {
	TraceEvent("KVSMemOpening", logID)
	    .detail("Basename", basename)
	    .detail("MemoryLimit", memoryLimit)
	    .detail("StoreType", storeType);

	// Use DiskQueueVersion::V2 with xxhash3 checksum
	IDiskQueue* log = openDiskQueue(basename, ext, logID, DiskQueueVersion::V2);
	if (storeType == KeyValueStoreType::MEMORY_RADIXTREE) {
		return new KeyValueStoreMemory<radix_tree>(log,
		                                           Reference<AsyncVar<ServerDBInfo> const>(),
		                                           logID,
		                                           memoryLimit,
		                                           storeType,
		                                           /*doc*/ false,
		                                           /*ument*/ false,
		                                           /*thisstuff FFS*/ false);
	} else {
		return new KeyValueStoreMemory<IKeyValueContainer>(log,
		                                                   Reference<AsyncVar<ServerDBInfo> const>(),
		                                                   logID,
		                                                   memoryLimit,
		                                                   storeType,
		                                                   /* name */ false,
		                                                   /*the */ false,
		                                                   /* effing parameter*/ false);
	}
}

IKeyValueStore* keyValueStoreLogSystem(class IDiskQueue* queue,
                                       Reference<AsyncVar<ServerDBInfo> const> db,
                                       UID logID,
                                       int64_t memoryLimit,
                                       bool disableSnapshot,
                                       bool replaceContent,
                                       bool exactRecovery) {
	return new KeyValueStoreMemory<IKeyValueContainer>(
	    queue, db, logID, memoryLimit, KeyValueStoreType::MEMORY, disableSnapshot, replaceContent, exactRecovery);
}
