#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
/*
 * BackupAgentBase.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2026 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <iomanip>
#include <time.h>

#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbrpc/simulator.h"
#include "flow/ActorCollection.h"
#include "flow/DeterministicRandom.h"
#include "flow/network.h"

#include "flow/actorcompiler.h" // has to be last include

std::string BackupAgentBase::formatTime(int64_t epochs) {
	time_t curTime = (time_t)epochs;
	char buffer[30];
	struct tm timeinfo;
	getLocalTime(&curTime, &timeinfo);
	strftime(buffer, 30, "%Y/%m/%d.%H:%M:%S%z", &timeinfo);
	return buffer;
}

int64_t BackupAgentBase::parseTime(std::string timestamp) {
	struct tm out;
	out.tm_isdst = -1; // This field is not set by strptime. -1 tells mktime to determine whether DST is in effect

	std::string timeOnly = timestamp.substr(0, 19);

	// TODO:  Use std::get_time implementation for all platforms once supported
	// It would be nice to read the timezone using %z, but it seems not all get_time()
	// or strptime() implementations handle it correctly in all environments so we
	// will read the date and time independent of timezone at first and then adjust it.
#ifdef _WIN32
	std::istringstream s(timeOnly);
	s.imbue(std::locale(setlocale(LC_TIME, nullptr)));
	s >> std::get_time(&out, "%Y/%m/%d.%H:%M:%S");
	if (s.fail()) {
		return -1;
	}
#else
	if (strptime(timeOnly.c_str(), "%Y/%m/%d.%H:%M:%S", &out) == nullptr) {
		return -1;
	}
#endif

	// Read timezone offset in +/-HHMM format then convert to seconds
	int tzHH;
	int tzMM;
	if (sscanf(timestamp.substr(19, 5).c_str(), "%3d%2d", &tzHH, &tzMM) != 2) {
		return -1;
	}
	if (tzHH < 0) {
		tzMM = -tzMM;
	}
	// tzOffset is the number of seconds EAST of GMT
	int tzOffset = tzHH * 60 * 60 + tzMM * 60;

	// The goal is to convert the timestamp string to epoch seconds assuming the date/time was expressed in the timezone
	// at the end of the string. However, mktime() will ONLY return epoch seconds assuming the date/time is expressed in
	// local time (based on locale / environment) mktime() will set out.tm_gmtoff when available
	int64_t ts = mktime(&out);

	// localTZOffset is the number of seconds EAST of GMT
	long localTZOffset;
#ifdef _WIN32
	// _get_timezone() returns the number of seconds WEST of GMT
	if (_get_timezone(&localTZOffset) != 0) {
		return -1;
	}
	// Negate offset to match the orientation of tzOffset
	localTZOffset = -localTZOffset;
#else
	// tm.tm_gmtoff is the number of seconds EAST of GMT
	localTZOffset = out.tm_gmtoff;
#endif

	// Add back the difference between the local timezone assumed by mktime() and the intended timezone from the input
	// string
	ts += (localTZOffset - tzOffset);
	return ts;
}

const Key BackupAgentBase::keyFolderId = "config_folderid"_sr;
const Key BackupAgentBase::keyBeginVersion = "beginVersion"_sr;
const Key BackupAgentBase::keyEndVersion = "endVersion"_sr;
const Key BackupAgentBase::keyPrevBeginVersion = "prevBeginVersion"_sr;
const Key BackupAgentBase::keyConfigBackupTag = "config_backup_tag"_sr;
const Key BackupAgentBase::keyConfigLogUid = "config_log_uid"_sr;
const Key BackupAgentBase::keyConfigBackupRanges = "config_backup_ranges"_sr;
const Key BackupAgentBase::keyConfigStopWhenDoneKey = "config_stop_when_done"_sr;
const Key BackupAgentBase::keyStateStop = "state_stop"_sr;
const Key BackupAgentBase::keyStateStatus = "state_status"_sr;
const Key BackupAgentBase::keyStateLogBeginVersion = "last_begin_version"_sr;
const Key BackupAgentBase::keyLastUid = "last_uid"_sr;
const Key BackupAgentBase::keyBeginKey = "beginKey"_sr;
const Key BackupAgentBase::keyEndKey = "endKey"_sr;
const Key BackupAgentBase::keyDrVersion = "drVersion"_sr;
const Key BackupAgentBase::destUid = "destUid"_sr;
const Key BackupAgentBase::backupStartVersion = "backupStartVersion"_sr;

const Key BackupAgentBase::keyTagName = "tagname"_sr;
const Key BackupAgentBase::keyStates = "state"_sr;
const Key BackupAgentBase::keyConfig = "config"_sr;
const Key BackupAgentBase::keyErrors = "errors"_sr;
const Key BackupAgentBase::keyRanges = "ranges"_sr;
const Key BackupAgentBase::keyTasks = "tasks"_sr;
const Key BackupAgentBase::keyFutures = "futures"_sr;
const Key BackupAgentBase::keySourceStates = "source_states"_sr;
const Key BackupAgentBase::keySourceTagName = "source_tagname"_sr;

bool copyParameter(Reference<Task> source, Reference<Task> dest, Key key) {
	if (source) {
		dest->params[key] = source->params[key];
		return true;
	}

	return false;
}

Version getVersionFromString(std::string const& value) {
	Version version = invalidVersion;
	int n = 0;
	if (sscanf(value.c_str(), "%lld%n", (long long*)&version, &n) != 1 || n != value.size()) {
		TraceEvent(SevWarnAlways, "GetVersionFromString").detail("InvalidVersion", value);
		throw restore_invalid_version();
	}
	return version;
}

// Transaction log data is stored by the FoundationDB core in the
// "backupLogKeys" (i.e., \xff\x02/blog/) keyspace in a funny order for
// performance reasons.
// Returns the ranges of keys that contain the data for the given range
// of versions.
// assert CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE % blocksize = 0. Otherwise calculation of hash will be incorrect
Standalone<VectorRef<KeyRangeRef>> getLogRanges(Version beginVersion,
                                                Version endVersion,
                                                Key destUidValue,
                                                int blockSize) {
	Standalone<VectorRef<KeyRangeRef>> ret;

	Key baLogRangePrefix = destUidValue.withPrefix(backupLogKeys.begin);

	//TraceEvent("GetLogRanges").detail("DestUidValue", destUidValue).detail("Prefix", baLogRangePrefix);

	for (int64_t vblock = beginVersion / blockSize; vblock < (endVersion + blockSize - 1) / blockSize; ++vblock) {
		int64_t tb = vblock * blockSize / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
		uint64_t bv = bigEndian64(std::max(beginVersion, vblock * blockSize));
		uint64_t ev = bigEndian64(std::min(endVersion, (vblock + 1) * blockSize));
		uint32_t data = tb & 0xffffffff;
		uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);

		Key vblockPrefix = StringRef(&hash, sizeof(uint8_t)).withPrefix(baLogRangePrefix);

		ret.push_back_deep(ret.arena(),
		                   KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix),
		                               StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
	}

	return ret;
}

// Given a begin and end version, get the prefix in the database for this range
// i.e. applyLogKeys.begin/backupUid/hash(uint8)/version(64bites)/part
// returns multiple key ranges, each of length APPLY_BLOCK_SIZE
// e.g. (64, 200) -> [(64, 128), (128, 192), (192, 200)]
Standalone<VectorRef<KeyRangeRef>> getApplyRanges(Version beginVersion, Version endVersion, Key backupUid) {
	Standalone<VectorRef<KeyRangeRef>> ret;
	Key baLogRangePrefix = backupUid.withPrefix(applyLogKeys.begin);

	//TraceEvent("GetLogRanges").detail("BackupUid", backupUid).detail("Prefix", baLogRangePrefix);

	for (int64_t vblock = beginVersion / CLIENT_KNOBS->APPLY_BLOCK_SIZE;
	     vblock < (endVersion + CLIENT_KNOBS->APPLY_BLOCK_SIZE - 1) / CLIENT_KNOBS->APPLY_BLOCK_SIZE;
	     ++vblock) {
		int64_t tb = vblock * CLIENT_KNOBS->APPLY_BLOCK_SIZE / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
		uint64_t bv = bigEndian64(std::max(beginVersion, vblock * CLIENT_KNOBS->APPLY_BLOCK_SIZE));
		uint64_t ev = bigEndian64(std::min(endVersion, (vblock + 1) * CLIENT_KNOBS->APPLY_BLOCK_SIZE));
		uint32_t data = tb & 0xffffffff;
		uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);

		Key vblockPrefix = StringRef(&hash, sizeof(uint8_t)).withPrefix(baLogRangePrefix);

		ret.push_back_deep(ret.arena(),
		                   KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix),
		                               StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
	}

	return ret;
}

Key getApplyKey(Version version, Key backupUid) {
	int64_t vblock = (version - 1) / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
	uint64_t v = bigEndian64(version);
	uint32_t data = vblock & 0xffffffff;
	uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);
	Key k1 = StringRef((uint8_t*)&v, sizeof(uint64_t)).withPrefix(StringRef(&hash, sizeof(uint8_t)));
	Key k2 = k1.withPrefix(backupUid);
	return k2.withPrefix(applyLogKeys.begin);
}

// It's important to keep the hash value consistent with the one used in getLogRanges.
// Otherwise, the same version will result in different keys.
Key getLogKey(Version version, Key backupUid, int blockSize) {
	int64_t vblock = version / blockSize;
	vblock = vblock * blockSize / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
	uint64_t v = bigEndian64(version);
	uint32_t data = vblock & 0xffffffff;
	uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);
	Key k1 = StringRef((uint8_t*)&v, sizeof(uint64_t)).withPrefix(StringRef(&hash, sizeof(uint8_t)));
	Key k2 = k1.withPrefix(backupUid);
	return k2.withPrefix(backupLogKeys.begin);
}

namespace {
															#line 242 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase240()
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class FlowTestCase240Actor>
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class FlowTestCase240ActorState {
															#line 248 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowTestCase240ActorState(UnitTestParameters const& params) 
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : params(params)
															#line 255 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase240", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase240ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase240", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 241 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			std::vector<Version> versions = { 100, 841000000 };
															#line 242 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < 10;i++) {
															#line 243 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				versions.push_back(deterministicRandom()->randomInt64(0, std::numeric_limits<int32_t>::max()));
															#line 274 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 245 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key backupUid = "backupUid0"_sr;
															#line 246 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			int blockSize = deterministicRandom()->coinflip() ? CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE : 100'000;
															#line 247 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( const auto v : versions ) {
															#line 248 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key k = getLogKey(v, backupUid, blockSize);
															#line 249 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Standalone<VectorRef<KeyRangeRef>> ranges = getLogRanges(v, v + 1, backupUid, blockSize);
															#line 250 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(ranges[0].contains(k));
															#line 288 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 252 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<FlowTestCase240Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase240ActorState(); static_cast<FlowTestCase240Actor*>(this)->destroy(); return 0; }
															#line 292 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase240Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase240ActorState();
			static_cast<FlowTestCase240Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase240ActorState();
		static_cast<FlowTestCase240Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	UnitTestParameters params;
															#line 316 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase240()
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class FlowTestCase240Actor final : public Actor<Void>, public FastAllocated<FlowTestCase240Actor>, public FlowTestCase240ActorState<FlowTestCase240Actor> {
															#line 321 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase240Actor>::operator new;
	using FastAllocated<FlowTestCase240Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14167876160332092416UL, 10089627251693030400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowTestCase240Actor(UnitTestParameters const& params) 
															#line 337 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase240ActorState<FlowTestCase240Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase240", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6583051328073499392UL, 7208039936741618688UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase240Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase240");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase240", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
static Future<Void> flowTestCase240( UnitTestParameters const& params ) {
															#line 240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new FlowTestCase240Actor(params));
															#line 368 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase240, "/backup/logversion")

#line 254 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
} // namespace

Version getLogKeyVersion(Key key) {
	return bigEndian64(*(int64_t*)(key.begin() + backupLogPrefixBytes + sizeof(UID) + sizeof(uint8_t)));
}

// Given a key from one of the ranges returned by get_log_ranges,
// returns(version, part) where version is the database version number of
// the transaction log data in the value, and part is 0 for the first such
// data for a given version, 1 for the second block of data, etc.
std::pair<Version, uint32_t> decodeBKMutationLogKey(Key key) {
	return std::make_pair(
	    getLogKeyVersion(key),
	    bigEndian32(*(int32_t*)(key.begin() + backupLogPrefixBytes + sizeof(UID) + sizeof(uint8_t) + sizeof(int64_t))));
}

void _addResult(VectorRef<MutationRef>* result, int* mutationSize, Arena* arena, MutationRef logValue) {
	result->push_back_deep(*arena, logValue);
	*mutationSize += logValue.expectedSize();
}

/*
 This actor is responsible for taking an original transaction which was added to the backup mutation log (represented
 by "value" parameter), breaking it up into the individual MutationRefs (that constitute the transaction), decrypting
 each mutation (if needed) and adding/removing prefixes from the mutations. The final mutations are then added to the
 "result" vector.
 Each `value` is a param2
*/
															#line 401 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via decodeBackupLogValue()
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class DecodeBackupLogValueActor>
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class DecodeBackupLogValueActorState {
															#line 408 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DecodeBackupLogValueActorState(Arena* const& arena,VectorRef<MutationRef>* const& result,int* const& mutationSize,Standalone<StringRef> const& value,Key const& addPrefix,Key const& removePrefix,Version const& version,Reference<KeyRangeMap<Version>> const& key_version,Database const& cx,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : arena(arena),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   result(result),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   mutationSize(mutationSize),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   value(value),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   version(version),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   key_version(key_version),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   cx(cx),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(dbConfig)
															#line 435 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("decodeBackupLogValue", reinterpret_cast<unsigned long>(this));

	}
	~DecodeBackupLogValueActorState() 
	{
		fdb_probe_actor_destroy("decodeBackupLogValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 294 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset = uint64_t(0);
															#line 295 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				uint64_t protocolVersion = 0;
															#line 296 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				memcpy(&protocolVersion, value.begin(), sizeof(uint64_t));
															#line 297 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset += sizeof(uint64_t);
															#line 298 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (protocolVersion <= 0x0FDB00A200090001)
															#line 459 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 299 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					TraceEvent(SevError, "DecodeBackupLogValue") .detail("IncompatibleProtocolVersion", protocolVersion) .detail("ValueSize", value.size()) .detail("Value", value);
															#line 303 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					return a_body1Catch2(incompatible_protocol_version(), loopDepth);
															#line 465 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 306 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				totalBytes = 0;
															#line 307 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				memcpy(&totalBytes, value.begin() + offset, sizeof(uint32_t));
															#line 308 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset += sizeof(uint32_t);
															#line 309 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				consumed = 0;
															#line 311 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (totalBytes + offset > value.size())
															#line 477 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 312 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					TraceEvent(SevError, "OffsetOutOfBoundary") .detail("TotalBytes", totalBytes) .detail("Offset", offset) .detail("Version", version) .detail("ValueSize", value.size());
															#line 317 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					return a_body1Catch2(restore_missing_data(), loopDepth);
															#line 483 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 320 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				originalOffset = offset;
															#line 322 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				for(;consumed < totalBytes;) {
															#line 323 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					uint32_t type = 0;
															#line 325 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					memcpy(&type, value.begin() + offset, sizeof(uint32_t));
															#line 326 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					offset += sizeof(uint32_t);
															#line 327 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					len1 = 0;
															#line 328 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					memcpy(&len1, value.begin() + offset, sizeof(uint32_t));
															#line 329 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					offset += sizeof(uint32_t);
															#line 330 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					len2 = 0;
															#line 331 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					memcpy(&len2, value.begin() + offset, sizeof(uint32_t));
															#line 332 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					offset += sizeof(uint32_t);
															#line 334 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					ASSERT(offset + len1 + len2 <= value.size() && isValidMutationType(type));
															#line 337 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue = MutationRef();
															#line 338 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					tempArena = Arena();
															#line 339 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue.type = type;
															#line 340 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue.param1 = value.substr(offset, len1);
															#line 341 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					offset += len1;
															#line 342 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue.param2 = value.substr(offset, len2);
															#line 343 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					offset += len2;
															#line 345 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (logValue.type == MutationRef::ClearRange)
															#line 525 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 346 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						KeyRangeRef range(logValue.param1, logValue.param2);
															#line 347 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						auto ranges = key_version->intersectingRanges(range);
															#line 348 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						for( auto r : ranges ) {
															#line 349 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							if (version > r.value() && r.value() != invalidVersion)
															#line 535 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
							{
															#line 350 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
								KeyRef minKey = std::min(r.range().end, range.end);
															#line 351 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
								if (minKey == (removePrefix == StringRef() ? allKeys.end : strinc(removePrefix)))
															#line 541 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
								{
															#line 352 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									logValue.param1 = std::max(r.range().begin, range.begin);
															#line 353 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									if (removePrefix.size())
															#line 547 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									{
															#line 354 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									}
															#line 356 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									if (addPrefix.size())
															#line 555 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									{
															#line 357 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 559 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									}
															#line 359 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									logValue.param2 = addPrefix == StringRef() ? allKeys.end : strinc(addPrefix, tempArena);
															#line 360 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									_addResult(result, mutationSize, arena, logValue);
															#line 565 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
								}
								else
								{
															#line 362 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									logValue.param1 = std::max(r.range().begin, range.begin);
															#line 363 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									logValue.param2 = minKey;
															#line 364 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									if (removePrefix.size())
															#line 575 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									{
															#line 365 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 366 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param2 = logValue.param2.removePrefix(removePrefix);
															#line 581 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									}
															#line 368 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									if (addPrefix.size())
															#line 585 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									{
															#line 369 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 370 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
										logValue.param2 = logValue.param2.withPrefix(addPrefix, tempArena);
															#line 591 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
									}
															#line 372 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
									_addResult(result, mutationSize, arena, logValue);
															#line 595 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
								}
							}
						}
					}
					else
					{
															#line 377 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Version ver = key_version->rangeContaining(logValue.param1).value();
															#line 382 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						if (version > ver && ver != invalidVersion)
															#line 606 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						{
															#line 383 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							if (removePrefix.size())
															#line 610 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
							{
															#line 384 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
								logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 614 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
							}
															#line 386 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							if (addPrefix.size())
															#line 618 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
							{
															#line 387 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
								logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 622 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
							}
															#line 389 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							_addResult(result, mutationSize, arena, logValue);
															#line 626 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						}
					}
															#line 393 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					consumed += BackupAgentBase::logHeaderSize + len1 + len2;
															#line 631 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 396 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(consumed == totalBytes);
															#line 397 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (value.size() != offset)
															#line 637 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 398 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					TraceEvent(SevError, "BA_DecodeBackupLogValue") .detail("UnexpectedExtraDataSize", value.size()) .detail("Offset", offset) .detail("GroupKey", version) .detail("TotalBytes", totalBytes) .detail("Consumed", consumed) .detail("OriginalOffset", originalOffset);
															#line 405 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					return a_body1Catch2(restore_corrupted_data(), loopDepth);
															#line 643 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				loopDepth = a_body1cont20(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DecodeBackupLogValueActorState();
		static_cast<DecodeBackupLogValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 415 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<DecodeBackupLogValueActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeBackupLogValueActorState(); static_cast<DecodeBackupLogValueActor*>(this)->destroy(); return 0; }
															#line 673 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<DecodeBackupLogValueActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DecodeBackupLogValueActorState();
		static_cast<DecodeBackupLogValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 408 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(e.code() == error_code_restore_missing_data ? SevWarn : SevError, "BA_DecodeBackupLogValue") .error(e) .GetLastError() .detail("ValueSize", value.size()) .detail("Value", value);
															#line 413 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 688 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont20(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Arena* arena;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	VectorRef<MutationRef>* result;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* mutationSize;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Standalone<StringRef> value;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version version;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> key_version;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 294 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint64_t offset;
															#line 306 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t totalBytes;
															#line 309 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t consumed;
															#line 320 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int originalOffset;
															#line 327 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t len1;
															#line 330 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t len2;
															#line 337 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	MutationRef logValue;
															#line 338 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Arena tempArena;
															#line 749 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via decodeBackupLogValue()
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class DecodeBackupLogValueActor final : public Actor<Void>, public FastAllocated<DecodeBackupLogValueActor>, public DecodeBackupLogValueActorState<DecodeBackupLogValueActor> {
															#line 754 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<DecodeBackupLogValueActor>::operator new;
	using FastAllocated<DecodeBackupLogValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(752939037869005568UL, 6461855703349363456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DecodeBackupLogValueActor(Arena* const& arena,VectorRef<MutationRef>* const& result,int* const& mutationSize,Standalone<StringRef> const& value,Key const& addPrefix,Key const& removePrefix,Version const& version,Reference<KeyRangeMap<Version>> const& key_version,Database const& cx,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 770 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DecodeBackupLogValueActorState<DecodeBackupLogValueActor>(arena, result, mutationSize, value, addPrefix, removePrefix, version, key_version, cx, provisionalProxy, dbConfig),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3565411968530254592UL, 18378739926066530816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeBackupLogValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("decodeBackupLogValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] static Future<Void> decodeBackupLogValue( Arena* const& arena, VectorRef<MutationRef>* const& result, int* const& mutationSize, Standalone<StringRef> const& value, Key const& addPrefix, Key const& removePrefix, Version const& version, Reference<KeyRangeMap<Version>> const& key_version, Database const& cx, bool const& provisionalProxy, std::shared_ptr<DatabaseConfiguration> const& dbConfig ) {
															#line 282 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new DecodeBackupLogValueActor(arena, result, mutationSize, value, addPrefix, removePrefix, version, key_version, cx, provisionalProxy, dbConfig));
															#line 802 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 417 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

static double lastErrorTime = 0;

void logErrorWorker(Reference<ReadYourWritesTransaction> tr, Key keyErrors, std::string message) {
	tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
	tr->setOption(FDBTransactionOptions::LOCK_AWARE);
	if (now() - lastErrorTime > CLIENT_KNOBS->BACKUP_ERROR_DELAY) {
		TraceEvent("BA_LogError").detail("Key", keyErrors).detail("Message", message);
		lastErrorTime = now();
	}
	tr->set(keyErrors, message);
}

Future<Void> logError(Database cx, Key keyErrors, const std::string& message) {
	return runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) {
		logErrorWorker(tr, keyErrors, message);
		return Future<Void>(Void());
	});
}

Future<Void> logError(Reference<ReadYourWritesTransaction> tr, Key keyErrors, const std::string& message) {
	return logError(tr->getDatabase(), keyErrors, message);
}

															#line 830 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readCommitted()
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ReadCommittedActor>
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActorState {
															#line 837 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActorState(Database const& cx,PromiseStream<RangeResultWithVersion> const& results,Reference<FlowLock> const& lock,KeyRangeRef const& range,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   range(range),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   terminator(terminator),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   systemAccess(systemAccess),
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lockAware(lockAware),
															#line 448 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   begin(firstGreaterOrEqual(range.begin)),
															#line 449 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   end(firstGreaterOrEqual(range.end)),
															#line 450 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(cx),
															#line 451 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   releaser()
															#line 864 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	~ReadCommittedActorState() 
	{
		fdb_probe_actor_destroy("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 453 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 879 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadCommittedActorState();
		static_cast<ReadCommittedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 455 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			limits = GetRangeLimits(GetRangeLimits::ROW_LIMIT_UNLIMITED, (g_network->isSimulated() && !g_simulator->speedUpSimulation) ? CLIENT_KNOBS->BACKUP_SIMULATED_LIMIT_BYTES : CLIENT_KNOBS->BACKUP_GET_RANGE_LIMIT_BYTES);
															#line 460 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (systemAccess)
															#line 912 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 461 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 916 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 462 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (lockAware)
															#line 920 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 463 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 924 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 464 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (CLIENT_KNOBS->ENABLE_REPLICA_CONSISTENCY_CHECK_ON_BACKUP_READS)
															#line 928 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 465 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ENABLE_REPLICA_CONSISTENCY_CHECK);
															#line 466 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				int64_t requiredReplicas = CLIENT_KNOBS->BACKUP_CONSISTENCY_CHECK_REQUIRED_REPLICAS;
															#line 467 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::CONSISTENCY_CHECK_REQUIRED_REPLICAS, StringRef((uint8_t*)&requiredReplicas, sizeof(int64_t)));
															#line 936 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 472 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			releaser.release();
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = lock->take(TaskPriority::DefaultYield, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 944 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadCommittedActor*>(this)->actor_wait_state = 1;
															#line 473 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 0, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 949 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 510 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 971 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 513 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.fullReset();
															#line 975 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
			}
			else
			{
															#line 515 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 515 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 984 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
				static_cast<ReadCommittedActor*>(this)->actor_wait_state = 4;
															#line 515 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 3, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 989 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 475 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser( *lock, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr.getRange(begin, end, limits);
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1009 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadCommittedActor*>(this)->actor_wait_state = 2;
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 1, RangeResult >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1014 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 475 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser( *lock, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr.getRange(begin, end, limits);
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1027 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadCommittedActor*>(this)->actor_wait_state = 2;
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 1, RangeResult >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1032 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 482 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (values.size() > 1 && BUGGIFY)
															#line 1116 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 483 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			RangeResult copy;
															#line 485 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < values.size() / 2;i++) {
															#line 486 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				copy.push_back_deep(copy.arena(), values[i]);
															#line 1124 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 488 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			values = copy;
															#line 489 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			values.more = true;
															#line 491 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (deterministicRandom()->random01() < 0.5)
															#line 1132 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 492 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_2 = delay(6.0);
															#line 492 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1138 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_2.get(), loopDepth); };
				static_cast<ReadCommittedActor*>(this)->actor_wait_state = 3;
															#line 492 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 2, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1143 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& __values,int loopDepth) 
	{
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		values = __values;
															#line 1162 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && __values,int loopDepth) 
	{
		values = std::move(__values);
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
															#line 495 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.remaining -= values.expectedSize();
															#line 497 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(releaser.remaining >= 0);
															#line 499 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		results.send(RangeResultWithVersion(values, tr.getReadVersion().get()));
															#line 501 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (values.size() > 0)
															#line 1247 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 502 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			begin = firstGreaterThan(values.end()[-1].key);
															#line 1251 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 504 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!values.more && !limits.isReached())
															#line 1255 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 505 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 1259 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 506 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 1263 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 507 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActorState(); static_cast<ReadCommittedActor*>(this)->destroy(); return 0; }
															#line 1267 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActorState();
			static_cast<ReadCommittedActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RangeResultWithVersion> results;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeyRangeRef range;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Terminator terminator;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	AccessSystemKeys systemAccess;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	LockAware lockAware;
															#line 448 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector begin;
															#line 449 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector end;
															#line 450 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Transaction tr;
															#line 451 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock::Releaser releaser;
															#line 455 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	GetRangeLimits limits;
															#line 478 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult values;
															#line 1502 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readCommitted()
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor final : public Actor<Void>, public ActorCallback< ReadCommittedActor, 0, Void >, public ActorCallback< ReadCommittedActor, 1, RangeResult >, public ActorCallback< ReadCommittedActor, 2, Void >, public ActorCallback< ReadCommittedActor, 3, Void >, public FastAllocated<ReadCommittedActor>, public ReadCommittedActorState<ReadCommittedActor> {
															#line 1507 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadCommittedActor>::operator new;
	using FastAllocated<ReadCommittedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7774808159567143936UL, 17334921742039686912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadCommittedActor, 0, Void >;
friend struct ActorCallback< ReadCommittedActor, 1, RangeResult >;
friend struct ActorCallback< ReadCommittedActor, 2, Void >;
friend struct ActorCallback< ReadCommittedActor, 3, Void >;
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor(Database const& cx,PromiseStream<RangeResultWithVersion> const& results,Reference<FlowLock> const& lock,KeyRangeRef const& range,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 1527 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ReadCommittedActorState<ReadCommittedActor>(cx, results, lock, range, terminator, systemAccess, lockAware),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(614696038618302720UL, 17105894356619358976UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readCommitted");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadCommittedActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadCommittedActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadCommittedActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadCommittedActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> readCommitted( Database const& cx, PromiseStream<RangeResultWithVersion> const& results, Reference<FlowLock> const& lock, KeyRangeRef const& range, Terminator const& terminator, AccessSystemKeys const& systemAccess, LockAware const& lockAware ) {
															#line 441 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ReadCommittedActor(cx, results, lock, range, terminator, systemAccess, lockAware));
															#line 1563 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 520 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// read each version, potentially multiple part within the same version
															#line 1569 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readCommitted()
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ReadCommittedActor1>
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor1State {
															#line 1576 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor1State(Database const& cx,PromiseStream<RCGroup> const& results,Future<Void> const& active,Reference<FlowLock> const& lock,KeyRangeRef const& range,std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   active(active),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   range(range),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   groupBy(groupBy),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   terminator(terminator),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   systemAccess(systemAccess),
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lockAware(lockAware),
															#line 531 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   nextKey(firstGreaterOrEqual(range.begin)),
															#line 532 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   end(firstGreaterOrEqual(range.end)),
															#line 534 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rcGroup(RCGroup()),
															#line 535 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   skipGroup(ULLONG_MAX),
															#line 536 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(cx),
															#line 537 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   releaser()
															#line 1611 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	~ReadCommittedActor1State() 
	{
		fdb_probe_actor_destroy("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 539 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 1626 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadCommittedActor1State();
		static_cast<ReadCommittedActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 541 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			limits = GetRangeLimits(GetRangeLimits::ROW_LIMIT_UNLIMITED, (g_network->isSimulated() && !g_simulator->speedUpSimulation) ? CLIENT_KNOBS->BACKUP_SIMULATED_LIMIT_BYTES : CLIENT_KNOBS->BACKUP_GET_RANGE_LIMIT_BYTES);
															#line 546 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (systemAccess)
															#line 1659 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 547 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1663 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 548 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (lockAware)
															#line 1667 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 549 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1671 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(nextKey, end, limits);
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1677 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 1;
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 0, RangeResult >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1682 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 632 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 1704 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 635 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.fullReset();
															#line 1708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
			}
			else
			{
															#line 637 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 637 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1717 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 5;
															#line 637 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 4, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 555 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (rangevalue.size() > 1 && BUGGIFY)
															#line 1738 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 556 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			RangeResult copy;
															#line 558 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < rangevalue.size() / 2;i++) {
															#line 559 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				copy.push_back_deep(copy.arena(), rangevalue[i]);
															#line 1746 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 561 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rangevalue = copy;
															#line 562 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rangevalue.more = true;
															#line 564 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (deterministicRandom()->random01() < 0.01)
															#line 1754 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 565 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_1 = delay(6.0);
															#line 565 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1760 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
				static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 2;
															#line 565 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 1, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1765 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont6(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __rangevalue,int loopDepth) 
	{
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		rangevalue = __rangevalue;
															#line 1784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __rangevalue,int loopDepth) 
	{
		rangevalue = std::move(__rangevalue);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 569 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_2 = active;
															#line 569 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1865 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 3;
															#line 569 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 2, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1870 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.release();
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = lock->take(TaskPriority::DefaultYield, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1976 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 4;
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 3, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1981 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.release();
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = lock->take(TaskPriority::DefaultYield, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1994 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 4;
															#line 571 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 3, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1999 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 572 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser(*lock, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 575 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto& s : rangevalue ) {
															#line 576 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version groupKey = groupBy(s.key).first;
															#line 583 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (groupKey != skipGroup)
															#line 2089 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 584 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (rcGroup.version == -1)
															#line 2093 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 585 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.version = tr.getReadVersion().get();
															#line 586 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.groupKey = groupKey;
															#line 2099 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 587 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (rcGroup.groupKey != groupKey)
															#line 2105 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 598 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						releaser.remaining -= rcGroup.items .expectedSize();
															#line 601 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						ASSERT(releaser.remaining >= 0);
															#line 602 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						results.send(rcGroup);
															#line 603 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextKey = firstGreaterThan(rcGroup.items.end()[-1].key);
															#line 604 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						skipGroup = rcGroup.groupKey;
															#line 606 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup = RCGroup();
															#line 607 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.version = tr.getReadVersion().get();
															#line 608 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.groupKey = groupKey;
															#line 2123 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
															#line 611 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				rcGroup.items.push_back_deep(rcGroup.items.arena(), s);
															#line 2128 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
															#line 615 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!rangevalue.more)
															#line 2133 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 616 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (rcGroup.version != -1)
															#line 2137 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 617 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				releaser.remaining -= rcGroup.items .expectedSize();
															#line 620 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(releaser.remaining >= 0);
															#line 622 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.send(rcGroup);
															#line 2145 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 625 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 2149 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 626 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 2153 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 627 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActor1State(); static_cast<ReadCommittedActor1*>(this)->destroy(); return 0; }
															#line 2157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor1*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActor1State();
			static_cast<ReadCommittedActor1*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 630 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextKey = firstGreaterThan(rangevalue.end()[-1].key);
															#line 2165 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont10cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 572 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser(*lock, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 575 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto& s : rangevalue ) {
															#line 576 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version groupKey = groupBy(s.key).first;
															#line 583 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (groupKey != skipGroup)
															#line 2180 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 584 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (rcGroup.version == -1)
															#line 2184 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 585 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.version = tr.getReadVersion().get();
															#line 586 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.groupKey = groupKey;
															#line 2190 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 587 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (rcGroup.groupKey != groupKey)
															#line 2196 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 598 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						releaser.remaining -= rcGroup.items .expectedSize();
															#line 601 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						ASSERT(releaser.remaining >= 0);
															#line 602 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						results.send(rcGroup);
															#line 603 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextKey = firstGreaterThan(rcGroup.items.end()[-1].key);
															#line 604 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						skipGroup = rcGroup.groupKey;
															#line 606 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup = RCGroup();
															#line 607 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.version = tr.getReadVersion().get();
															#line 608 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.groupKey = groupKey;
															#line 2214 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
															#line 611 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				rcGroup.items.push_back_deep(rcGroup.items.arena(), s);
															#line 2219 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
															#line 615 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!rangevalue.more)
															#line 2224 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 616 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (rcGroup.version != -1)
															#line 2228 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 617 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				releaser.remaining -= rcGroup.items .expectedSize();
															#line 620 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(releaser.remaining >= 0);
															#line 622 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.send(rcGroup);
															#line 2236 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 625 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 2240 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 626 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 2244 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 627 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActor1State(); static_cast<ReadCommittedActor1*>(this)->destroy(); return 0; }
															#line 2248 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor1*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActor1State();
			static_cast<ReadCommittedActor1*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 630 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextKey = firstGreaterThan(rangevalue.end()[-1].key);
															#line 2256 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont10cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont10cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RCGroup> results;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Void> active;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeyRangeRef range;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::function<std::pair<uint64_t, uint32_t>(Key key)> groupBy;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Terminator terminator;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	AccessSystemKeys systemAccess;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	LockAware lockAware;
															#line 531 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector nextKey;
															#line 532 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector end;
															#line 534 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RCGroup rcGroup;
															#line 535 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint64_t skipGroup;
															#line 536 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Transaction tr;
															#line 537 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock::Releaser releaser;
															#line 541 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	GetRangeLimits limits;
															#line 551 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult rangevalue;
															#line 2476 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readCommitted()
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor1 final : public Actor<Void>, public ActorCallback< ReadCommittedActor1, 0, RangeResult >, public ActorCallback< ReadCommittedActor1, 1, Void >, public ActorCallback< ReadCommittedActor1, 2, Void >, public ActorCallback< ReadCommittedActor1, 3, Void >, public ActorCallback< ReadCommittedActor1, 4, Void >, public FastAllocated<ReadCommittedActor1>, public ReadCommittedActor1State<ReadCommittedActor1> {
															#line 2481 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadCommittedActor1>::operator new;
	using FastAllocated<ReadCommittedActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7774808159567143936UL, 17334921742039686912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadCommittedActor1, 0, RangeResult >;
friend struct ActorCallback< ReadCommittedActor1, 1, Void >;
friend struct ActorCallback< ReadCommittedActor1, 2, Void >;
friend struct ActorCallback< ReadCommittedActor1, 3, Void >;
friend struct ActorCallback< ReadCommittedActor1, 4, Void >;
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor1(Database const& cx,PromiseStream<RCGroup> const& results,Future<Void> const& active,Reference<FlowLock> const& lock,KeyRangeRef const& range,std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 2502 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ReadCommittedActor1State<ReadCommittedActor1>(cx, results, active, lock, range, groupBy, terminator, systemAccess, lockAware),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11877085914525018880UL, 18281691565276970752UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readCommitted");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadCommittedActor1, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadCommittedActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadCommittedActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadCommittedActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ReadCommittedActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> readCommitted( Database const& cx, PromiseStream<RCGroup> const& results, Future<Void> const& active, Reference<FlowLock> const& lock, KeyRangeRef const& range, std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy, Terminator const& terminator, AccessSystemKeys const& systemAccess, LockAware const& lockAware ) {
															#line 522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ReadCommittedActor1(cx, results, active, lock, range, groupBy, terminator, systemAccess, lockAware));
															#line 2539 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 642 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

Future<Void> readCommitted(Database cx,
                           PromiseStream<RCGroup> results,
                           Reference<FlowLock> lock,
                           KeyRangeRef range,
                           std::function<std::pair<uint64_t, uint32_t>(Key key)> groupBy) {
	return readCommitted(
	    cx, results, Void(), lock, range, groupBy, Terminator::True, AccessSystemKeys::True, LockAware::True);
}

// restore transaction has to be first in the batch, or it is the only txn in batch to make sure it never conflicts with
// others.
															#line 2555 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via sendCommitTransactionRequest()
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class SendCommitTransactionRequestActor>
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class SendCommitTransactionRequestActorState {
															#line 2562 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	SendCommitTransactionRequestActorState(CommitTransactionRequest const& req,Key const& uid,Version const& newBeginVersion,Key const& rangeBegin,NotifiedVersion* const& committedVersion,int* const& totalBytes,int* const& mutationSize,FlowLock* const& commitLock,PublicRequestStream<CommitTransactionRequest> const& commit) 
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : req(req),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   newBeginVersion(newBeginVersion),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rangeBegin(rangeBegin),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   totalBytes(totalBytes),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   mutationSize(mutationSize),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock),
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit)
															#line 2585 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this));

	}
	~SendCommitTransactionRequestActorState() 
	{
		fdb_probe_actor_destroy("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 663 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key applyBegin = uid.withPrefix(applyMutationsBeginRange.begin);
															#line 664 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key versionKey = BinaryWriter::toValue(newBeginVersion, Unversioned());
															#line 665 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key rangeEnd = getApplyKey(newBeginVersion, uid);
															#line 668 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::SetValue, applyBegin, versionKey));
															#line 669 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(applyBegin));
															#line 670 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::ClearRange, rangeBegin, rangeEnd));
															#line 671 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(rangeBegin));
															#line 676 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.read_snapshot = committedVersion->get();
															#line 677 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.flags = req.flags | CommitTransactionRequest::FLAG_IS_LOCK_AWARE;
															#line 679 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			*totalBytes += *mutationSize;
															#line 680 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = commitLock->take(TaskPriority::DefaultYield, *mutationSize);
															#line 680 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2622 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 1;
															#line 680 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 0, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2627 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 681 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Future<Void> commitAndUnlock = commitLock->releaseWhen(success(commit.getReply(req)), *mutationSize);
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = commitAndUnlock;
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 2;
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 1, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2659 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 681 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Future<Void> commitAndUnlock = commitLock->releaseWhen(success(commit.getReply(req)), *mutationSize);
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = commitAndUnlock;
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2672 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 2;
															#line 690 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 1, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2677 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state > 0) static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendCommitTransactionRequestActor*>(this)->ActorCallback< SendCommitTransactionRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 691 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<SendCommitTransactionRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendCommitTransactionRequestActorState(); static_cast<SendCommitTransactionRequestActor*>(this)->destroy(); return 0; }
															#line 2761 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<SendCommitTransactionRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 691 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<SendCommitTransactionRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendCommitTransactionRequestActorState(); static_cast<SendCommitTransactionRequestActor*>(this)->destroy(); return 0; }
															#line 2773 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<SendCommitTransactionRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state > 0) static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendCommitTransactionRequestActor*>(this)->ActorCallback< SendCommitTransactionRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newBeginVersion;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key rangeBegin;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* totalBytes;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* mutationSize;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 2874 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via sendCommitTransactionRequest()
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class SendCommitTransactionRequestActor final : public Actor<Void>, public ActorCallback< SendCommitTransactionRequestActor, 0, Void >, public ActorCallback< SendCommitTransactionRequestActor, 1, Void >, public FastAllocated<SendCommitTransactionRequestActor>, public SendCommitTransactionRequestActorState<SendCommitTransactionRequestActor> {
															#line 2879 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<SendCommitTransactionRequestActor>::operator new;
	using FastAllocated<SendCommitTransactionRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17267118725644636928UL, 4023359873865483776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SendCommitTransactionRequestActor, 0, Void >;
friend struct ActorCallback< SendCommitTransactionRequestActor, 1, Void >;
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	SendCommitTransactionRequestActor(CommitTransactionRequest const& req,Key const& uid,Version const& newBeginVersion,Key const& rangeBegin,NotifiedVersion* const& committedVersion,int* const& totalBytes,int* const& mutationSize,FlowLock* const& commitLock,PublicRequestStream<CommitTransactionRequest> const& commit) 
															#line 2897 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   SendCommitTransactionRequestActorState<SendCommitTransactionRequestActor>(req, uid, newBeginVersion, rangeBegin, committedVersion, totalBytes, mutationSize, commitLock, commit),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1297593344567152896UL, 10219936147972304640UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("sendCommitTransactionRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SendCommitTransactionRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SendCommitTransactionRequestActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> sendCommitTransactionRequest( CommitTransactionRequest const& req, Key const& uid, Version const& newBeginVersion, Key const& rangeBegin, NotifiedVersion* const& committedVersion, int* const& totalBytes, int* const& mutationSize, FlowLock* const& commitLock, PublicRequestStream<CommitTransactionRequest> const& commit ) {
															#line 654 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new SendCommitTransactionRequestActor(req, uid, newBeginVersion, rangeBegin, committedVersion, totalBytes, mutationSize, commitLock, commit));
															#line 2931 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 693 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// Decodes the backup mutation log and send the mutations to the CommitProxy.
// The mutation logs are grouped by version and passed in as a stream of RCGroup from readCommitted().
// The mutations are then decoded and sent to the CommitProxy in a batch.
															#line 2939 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via kvMutationLogToTransactions()
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class KvMutationLogToTransactionsActor>
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class KvMutationLogToTransactionsActorState {
															#line 2946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KvMutationLogToTransactionsActorState(Database const& cx,PromiseStream<RCGroup> const& results,Reference<FlowLock> const& lock,Key const& uid,Key const& addPrefix,Key const& removePrefix,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Optional<Version> const& endVersion,Key const& rangeBegin,FlowLock* const& commitLock,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rangeBegin(rangeBegin),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(dbConfig),
															#line 711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lastVersion(invalidVersion),
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endOfStream(false),
															#line 713 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   totalBytes(0)
															#line 2985 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this));

	}
	~KvMutationLogToTransactionsActorState() 
	{
		fdb_probe_actor_destroy("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 715 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 3000 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~KvMutationLogToTransactionsActorState();
		static_cast<KvMutationLogToTransactionsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		req = CommitTransactionRequest();
															#line 717 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = invalidVersion;
															#line 718 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize = 0;
															#line 719 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 3036 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 766 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_2 = sendCommitTransactionRequest( req, uid, newBeginVersion, rangeBegin, committedVersion, &totalBytes, &mutationSize, commitLock, commit);
															#line 766 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3047 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 3;
															#line 766 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3052 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			auto __when_expr_0 = results.getFuture();
															#line 3069 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			static_assert(std::is_same<decltype(__when_expr_0), FutureStream<RCGroup>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<RCGroup>>::value, "invalid type");
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3073 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.pop(), loopDepth); };
			static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 1;
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3078 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 753 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_end_of_stream)
															#line 3113 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 754 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (endVersion.present() && endVersion.get() > lastVersion && endVersion.get() > newBeginVersion)
															#line 3117 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 755 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					newBeginVersion = endVersion.get();
															#line 3121 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 757 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (newBeginVersion == invalidVersion)
															#line 3125 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 758 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3129 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
					this->~KvMutationLogToTransactionsActorState();
					static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 759 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				endOfStream = true;
															#line 3137 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 762 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 3142 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		curReq = CommitTransactionRequest();
															#line 723 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		lock->release(group.items.expectedSize());
															#line 724 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		curBatchMutationSize = 0;
															#line 726 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		BinaryWriter bw(Unversioned());
															#line 728 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < group.items.size();++i) {
															#line 731 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			bw.serializeBytes(group.items[i].value);
															#line 3166 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 734 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Standalone<StringRef> value = bw.toValue();
															#line 735 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = decodeBackupLogValue(&curReq.arena, &curReq.transaction.mutations, &curBatchMutationSize, value, addPrefix, removePrefix, group.groupKey, keyVersion, cx, provisionalProxy, dbConfig);
															#line 735 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3174 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 2;
															#line 735 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3179 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RCGroup const& __group,int loopDepth) 
	{
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		group = __group;
															#line 3188 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RCGroup && __group,int loopDepth) 
	{
		group = std::move(__group);
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >::remove();

	}
	void a_callback_fire(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,RCGroup const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,RCGroup && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 747 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < curReq.transaction.mutations.size();i++) {
															#line 748 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, curReq.transaction.mutations[i]);
															#line 3269 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 750 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize += curBatchMutationSize;
															#line 751 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = group.groupKey + 1;
															#line 3275 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 747 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < curReq.transaction.mutations.size();i++) {
															#line 748 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, curReq.transaction.mutations[i]);
															#line 3286 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 750 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize += curBatchMutationSize;
															#line 751 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = group.groupKey + 1;
															#line 3292 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorCallback< KvMutationLogToTransactionsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 768 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endOfStream)
															#line 3389 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 769 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3393 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
			this->~KvMutationLogToTransactionsActorState();
			static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 768 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endOfStream)
															#line 3407 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 769 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3411 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
			this->~KvMutationLogToTransactionsActorState();
			static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorCallback< KvMutationLogToTransactionsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RCGroup> results;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Version> endVersion;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key rangeBegin;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version lastVersion;
															#line 712 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool endOfStream;
															#line 713 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int totalBytes;
															#line 716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 717 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newBeginVersion;
															#line 718 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int mutationSize;
															#line 721 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RCGroup group;
															#line 722 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest curReq;
															#line 724 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int curBatchMutationSize;
															#line 3542 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via kvMutationLogToTransactions()
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class KvMutationLogToTransactionsActor final : public Actor<int>, public ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >, public ActorCallback< KvMutationLogToTransactionsActor, 1, Void >, public ActorCallback< KvMutationLogToTransactionsActor, 2, Void >, public FastAllocated<KvMutationLogToTransactionsActor>, public KvMutationLogToTransactionsActorState<KvMutationLogToTransactionsActor> {
															#line 3547 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<KvMutationLogToTransactionsActor>::operator new;
	using FastAllocated<KvMutationLogToTransactionsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17458249507121263104UL, 14384523177778471168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >;
friend struct ActorCallback< KvMutationLogToTransactionsActor, 1, Void >;
friend struct ActorCallback< KvMutationLogToTransactionsActor, 2, Void >;
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KvMutationLogToTransactionsActor(Database const& cx,PromiseStream<RCGroup> const& results,Reference<FlowLock> const& lock,Key const& uid,Key const& addPrefix,Key const& removePrefix,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Optional<Version> const& endVersion,Key const& rangeBegin,FlowLock* const& commitLock,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 3566 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<int>(),
		   KvMutationLogToTransactionsActorState<KvMutationLogToTransactionsActor>(cx, results, lock, uid, addPrefix, removePrefix, commit, committedVersion, endVersion, rangeBegin, commitLock, keyVersion, provisionalProxy, dbConfig),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4574830504988014848UL, 15510128346966688768UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("kvMutationLogToTransactions");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<int> kvMutationLogToTransactions( Database const& cx, PromiseStream<RCGroup> const& results, Reference<FlowLock> const& lock, Key const& uid, Key const& addPrefix, Key const& removePrefix, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, Optional<Version> const& endVersion, Key const& rangeBegin, FlowLock* const& commitLock, Reference<KeyRangeMap<Version>> const& keyVersion, bool const& provisionalProxy, std::shared_ptr<DatabaseConfiguration> const& dbConfig ) {
															#line 697 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<int>(new KvMutationLogToTransactionsActor(cx, results, lock, uid, addPrefix, removePrefix, commit, committedVersion, endVersion, rangeBegin, commitLock, keyVersion, provisionalProxy, dbConfig));
															#line 3601 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 773 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 3606 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via coalesceKeyVersionCache()
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CoalesceKeyVersionCacheActor>
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CoalesceKeyVersionCacheActorState {
															#line 3613 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CoalesceKeyVersionCacheActorState(Key const& uid,Version const& endVersion,Reference<KeyRangeMap<Version>> const& keyVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,PromiseStream<Future<Void>> const& addActor,FlowLock* const& commitLock) 
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : uid(uid),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addActor(addActor),
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock)
															#line 3632 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this));

	}
	~CoalesceKeyVersionCacheActorState() 
	{
		fdb_probe_actor_destroy("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 781 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version lastVersion = -1000;
															#line 782 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			int64_t removed = 0;
															#line 783 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req = CommitTransactionRequest();
															#line 784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			mutationSize = 0;
															#line 785 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key mapPrefix = uid.withPrefix(applyMutationsKeyVersionMapRange.begin);
															#line 787 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( auto it : keyVersion->ranges() ) {
															#line 788 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (lastVersion == -1000)
															#line 3659 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 789 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					lastVersion = it.value();
															#line 3663 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 791 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Version ver = it.value();
															#line 798 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (ver < endVersion && lastVersion < endVersion && ver != invalidVersion && lastVersion != invalidVersion)
															#line 3671 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 800 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key removeKey = it.range().begin.withPrefix(mapPrefix);
															#line 801 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key removeEnd = keyAfter(removeKey);
															#line 802 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::ClearRange, removeKey, removeEnd));
															#line 804 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						mutationSize += removeKey.size() + removeEnd.size();
															#line 805 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						removed--;
															#line 3683 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 807 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						lastVersion = ver;
															#line 3689 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
			}
															#line 812 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (removed != 0)
															#line 3695 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 813 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key countKey = uid.withPrefix(applyMutationsKeyVersionCountRange.begin);
															#line 814 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(countKey));
															#line 815 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.mutations.push_back_deep( req.arena, MutationRef(MutationRef::AddValue, countKey, StringRef((uint8_t*)&removed, 8)));
															#line 817 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.read_snapshot = committedVersion->get();
															#line 818 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.flags = req.flags | CommitTransactionRequest::FLAG_IS_LOCK_AWARE;
															#line 820 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_0 = commitLock->take(TaskPriority::DefaultYield, mutationSize);
															#line 820 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3711 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state = 1;
															#line 820 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*>(static_cast<CoalesceKeyVersionCacheActor*>(this)));
															#line 3716 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CoalesceKeyVersionCacheActorState();
		static_cast<CoalesceKeyVersionCacheActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 824 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CoalesceKeyVersionCacheActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CoalesceKeyVersionCacheActorState(); static_cast<CoalesceKeyVersionCacheActor*>(this)->destroy(); return 0; }
															#line 3744 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CoalesceKeyVersionCacheActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CoalesceKeyVersionCacheActorState();
		static_cast<CoalesceKeyVersionCacheActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 821 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		addActor.send(commitLock->releaseWhen(success(commit.getReply(req)), mutationSize));
															#line 3756 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 821 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		addActor.send(commitLock->releaseWhen(success(commit.getReply(req)), mutationSize));
															#line 3765 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state > 0) static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CoalesceKeyVersionCacheActor*>(this)->ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version endVersion;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 783 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 784 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int64_t mutationSize;
															#line 3863 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via coalesceKeyVersionCache()
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CoalesceKeyVersionCacheActor final : public Actor<Void>, public ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >, public FastAllocated<CoalesceKeyVersionCacheActor>, public CoalesceKeyVersionCacheActorState<CoalesceKeyVersionCacheActor> {
															#line 3868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CoalesceKeyVersionCacheActor>::operator new;
	using FastAllocated<CoalesceKeyVersionCacheActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17077304582983527168UL, 7279363725849483264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >;
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CoalesceKeyVersionCacheActor(Key const& uid,Version const& endVersion,Reference<KeyRangeMap<Version>> const& keyVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,PromiseStream<Future<Void>> const& addActor,FlowLock* const& commitLock) 
															#line 3885 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CoalesceKeyVersionCacheActorState<CoalesceKeyVersionCacheActor>(uid, endVersion, keyVersion, commit, committedVersion, addActor, commitLock),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18444184965993123072UL, 15771598966678214144UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("coalesceKeyVersionCache");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> coalesceKeyVersionCache( Key const& uid, Version const& endVersion, Reference<KeyRangeMap<Version>> const& keyVersion, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, PromiseStream<Future<Void>> const& addActor, FlowLock* const& commitLock ) {
															#line 774 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CoalesceKeyVersionCacheActor(uid, endVersion, keyVersion, commit, committedVersion, addActor, commitLock));
															#line 3918 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 826 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 3923 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via applyMutations()
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ApplyMutationsActor>
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ApplyMutationsActorState {
															#line 3930 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ApplyMutationsActorState(Database const& cx,Key const& uid,Key const& addPrefix,Key const& removePrefix,Version const& beginVersion,Version* const& endVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy) 
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   beginVersion(beginVersion),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 837 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(CLIENT_KNOBS->BACKUP_LOCK_BYTES),
															#line 838 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addActor(),
															#line 839 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   error(actorCollection(addActor.getFuture())),
															#line 840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   maxBytes(CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES),
															#line 841 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(std::make_shared<DatabaseConfiguration>())
															#line 3965 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("applyMutations", reinterpret_cast<unsigned long>(this));

	}
	~ApplyMutationsActorState() 
	{
		fdb_probe_actor_destroy("applyMutations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 843 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			keyVersion->insert(metadataVersionKey, 0);
															#line 3980 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			try {
															#line 846 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(*dbConfig, getDatabaseConfiguration(cx));
															#line 846 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3986 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 1;
															#line 846 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 0, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 3991 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ApplyMutationsActorState();
		static_cast<ApplyMutationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 912 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			ASSERT_WE_THINK(e.code() != error_code_transaction_rejected_range_locked);
															#line 913 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Severity sev = (e.code() == error_code_restore_missing_data || e.code() == error_code_transaction_throttled_hot_shard) ? SevWarnAlways : SevError;
															#line 917 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(sev, "ApplyMutationsError").error(e);
															#line 918 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4027 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 848 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 4041 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 848 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 4050 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 849 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4141 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 851 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = commitLock.take(TaskPriority::DefaultYield, CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 851 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4147 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 2;
															#line 851 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 1, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4152 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 858 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		int rangeCount = std::max(1, CLIENT_KNOBS->APPLY_MAX_LOCK_BYTES / maxBytes);
															#line 860 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newEndVersion = std::min(*endVersion, ((beginVersion / CLIENT_KNOBS->APPLY_BLOCK_SIZE) + rangeCount) * CLIENT_KNOBS->APPLY_BLOCK_SIZE);
															#line 867 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ranges = getApplyRanges(beginVersion, newEndVersion, uid);
															#line 868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		idx = size_t();
															#line 869 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		results = std::vector<PromiseStream<RCGroup>>();
															#line 870 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		rc = std::vector<Future<Void>>();
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		locks = std::vector<Reference<FlowLock>>();
															#line 875 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < ranges.size();++i) {
															#line 876 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			results.push_back(PromiseStream<RCGroup>());
															#line 877 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			locks.push_back(makeReference<FlowLock>( std::max(CLIENT_KNOBS->APPLY_MAX_LOCK_BYTES / ranges.size(), CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES)));
															#line 879 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rc.push_back(readCommitted(cx, results[i], locks[i], ranges[i], decodeBKMutationLogKey));
															#line 4186 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 882 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(maxBytes * CLIENT_KNOBS->APPLY_MAX_DECAY_RATE, CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES);
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		idx = 0;
															#line 4192 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		commitLock.release(CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 853 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4203 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 854 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ApplyMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ApplyMutationsActorState(); static_cast<ApplyMutationsActor*>(this)->destroy(); return 0; }
															#line 4207 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ApplyMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ApplyMutationsActorState();
			static_cast<ApplyMutationsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		commitLock.release(CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 853 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4223 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 854 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ApplyMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ApplyMutationsActorState(); static_cast<ApplyMutationsActor*>(this)->destroy(); return 0; }
															#line 4227 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ApplyMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ApplyMutationsActorState();
			static_cast<ApplyMutationsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont5(int loopDepth) 
	{
															#line 904 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = coalesceKeyVersionCache( uid, newEndVersion, keyVersion, commit, committedVersion, addActor, &commitLock);
															#line 904 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4318 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 4;
															#line 904 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 3, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4323 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(idx < ranges.size()))
															#line 4339 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 884 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<int> __when_expr_2 = kvMutationLogToTransactions(cx, results[idx], locks[idx], uid, addPrefix, removePrefix, commit, committedVersion, idx == ranges.size() - 1 ? newEndVersion : Optional<Version>(), ranges[idx].begin, &commitLock, keyVersion, provisionalProxy, dbConfig);
															#line 884 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 4347 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 3;
															#line 884 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 2, int >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4352 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1cont1(int const& bytes,int loopDepth) 
	{
															#line 899 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(CLIENT_KNOBS->APPLY_MAX_INCREASE_FACTOR * bytes, maxBytes);
															#line 900 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (error.isError())
															#line 4376 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 901 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch2(error.getError(), std::max(0, loopDepth - 2));
															#line 4380 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++idx;
															#line 4384 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1cont1(int && bytes,int loopDepth) 
	{
															#line 899 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(CLIENT_KNOBS->APPLY_MAX_INCREASE_FACTOR * bytes, maxBytes);
															#line 900 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (error.isError())
															#line 4395 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 901 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch2(error.getError(), std::max(0, loopDepth - 2));
															#line 4399 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 883 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++idx;
															#line 4403 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1when1(int const& bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1loopBody1cont1(bytes, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1when1(int && bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1loopBody1cont1(std::move(bytes), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 906 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		beginVersion = newEndVersion;
															#line 907 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (BUGGIFY)
															#line 4489 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4495 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 5;
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 4, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4500 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 906 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		beginVersion = newEndVersion;
															#line 907 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (BUGGIFY)
															#line 4516 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4522 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 5;
															#line 908 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 4, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4527 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2loopBody1cont8(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version beginVersion;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version* endVersion;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 837 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock commitLock;
															#line 838 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 839 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Void> error;
															#line 840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int maxBytes;
															#line 841 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 860 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newEndVersion;
															#line 867 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Standalone<VectorRef<KeyRangeRef>> ranges;
															#line 868 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	size_t idx;
															#line 869 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<PromiseStream<RCGroup>> results;
															#line 870 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<Future<Void>> rc;
															#line 871 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<Reference<FlowLock>> locks;
															#line 4747 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via applyMutations()
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ApplyMutationsActor final : public Actor<Void>, public ActorCallback< ApplyMutationsActor, 0, Void >, public ActorCallback< ApplyMutationsActor, 1, Void >, public ActorCallback< ApplyMutationsActor, 2, int >, public ActorCallback< ApplyMutationsActor, 3, Void >, public ActorCallback< ApplyMutationsActor, 4, Void >, public FastAllocated<ApplyMutationsActor>, public ApplyMutationsActorState<ApplyMutationsActor> {
															#line 4752 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ApplyMutationsActor>::operator new;
	using FastAllocated<ApplyMutationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5949829682976028416UL, 2798671617098887424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ApplyMutationsActor, 0, Void >;
friend struct ActorCallback< ApplyMutationsActor, 1, Void >;
friend struct ActorCallback< ApplyMutationsActor, 2, int >;
friend struct ActorCallback< ApplyMutationsActor, 3, Void >;
friend struct ActorCallback< ApplyMutationsActor, 4, Void >;
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ApplyMutationsActor(Database const& cx,Key const& uid,Key const& addPrefix,Key const& removePrefix,Version const& beginVersion,Version* const& endVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy) 
															#line 4773 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ApplyMutationsActorState<ApplyMutationsActor>(cx, uid, addPrefix, removePrefix, beginVersion, endVersion, commit, committedVersion, keyVersion, provisionalProxy),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17207310279585965312UL, 13534680049970660864UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("applyMutations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ApplyMutationsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ApplyMutationsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ApplyMutationsActor, 2, int >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ApplyMutationsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ApplyMutationsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> applyMutations( Database const& cx, Key const& uid, Key const& addPrefix, Key const& removePrefix, Version const& beginVersion, Version* const& endVersion, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, Reference<KeyRangeMap<Version>> const& keyVersion, bool const& provisionalProxy ) {
															#line 827 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ApplyMutationsActor(cx, uid, addPrefix, removePrefix, beginVersion, endVersion, commit, committedVersion, keyVersion, provisionalProxy));
															#line 4810 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 921 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 4815 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via _eraseLogData()
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class _eraseLogDataActor>
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class _eraseLogDataActorState {
															#line 4822 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	_eraseLogDataActorState(Reference<ReadYourWritesTransaction> const& tr,Key const& logUidValue,Key const& destUidValue,Optional<Version> const& endVersion,CheckBackupUID const& checkBackupUid,Version const& backupUid) 
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : tr(tr),
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   logUidValue(logUidValue),
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   destUidValue(destUidValue),
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   checkBackupUid(checkBackupUid),
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupUid(backupUid),
															#line 928 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsPath(destUidValue.withPrefix(backupLatestVersionsPrefix)),
															#line 929 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsKey(logUidValue.withPrefix(backupLatestVersionsPath))
															#line 4843 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("_eraseLogData", reinterpret_cast<unsigned long>(this));

	}
	~_eraseLogDataActorState() 
	{
		fdb_probe_actor_destroy("_eraseLogData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 931 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!destUidValue.size())
															#line 4858 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 932 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 4862 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~_eraseLogDataActorState();
				static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 935 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 936 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 938 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (checkBackupUid)
															#line 4874 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 939 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Subspace sourceStates = Subspace(databaseBackupPrefixRange.begin).get(BackupAgentBase::keySourceStates).get(logUidValue);
															#line 941 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Optional<Value>> __when_expr_0 = tr->get(sourceStates.pack(DatabaseBackupAgent::keyFolderId));
															#line 941 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4882 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 1;
															#line 941 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 4887 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_eraseLogDataActorState();
		static_cast<_eraseLogDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(backupLatestVersionsPath, strinc(backupLatestVersionsPath)), CLIENT_KNOBS->TOO_MANY);
															#line 946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4917 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 2;
															#line 946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 1, RangeResult >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 4922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> const& v,int loopDepth) 
	{
															#line 942 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (v.present() && BinaryReader::fromStringRef<Version>(v.get(), Unversioned()) > backupUid)
															#line 4931 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 943 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 4935 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && v,int loopDepth) 
	{
															#line 942 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (v.present() && BinaryReader::fromStringRef<Version>(v.get(), Unversioned()) > backupUid)
															#line 4949 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 943 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 4953 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& v,int loopDepth) 
	{
		loopDepth = a_body1cont3(v, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && v,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 950 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currBeginVersion = invalidVersion;
															#line 951 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto backupVersion : backupVersions ) {
															#line 952 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key currLogUidValue = backupVersion.key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 954 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (currLogUidValue == logUidValue)
															#line 5048 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 955 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				currBeginVersion = BinaryReader::fromStringRef<Version>(backupVersion.value, Unversioned());
															#line 5052 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				break;
			}
		}
															#line 961 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (currBeginVersion == invalidVersion)
															#line 5058 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 962 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5062 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 965 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currEndVersion = std::numeric_limits<Version>::max();
															#line 966 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endVersion.present())
															#line 5072 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 967 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			currEndVersion = std::min(currEndVersion, endVersion.get());
															#line 5076 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 970 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextSmallestVersion = currEndVersion;
															#line 971 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		bool clearLogRangesRequired = true;
															#line 974 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (backupVersions.size() > 1)
															#line 5084 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 975 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( auto backupVersion : backupVersions ) {
															#line 976 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key currLogUidValue = backupVersion.key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 977 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Version currVersion = BinaryReader::fromStringRef<Version>(backupVersion.value, Unversioned());
															#line 979 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (currLogUidValue == logUidValue)
															#line 5094 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
					continue;
				}
				else
				{
															#line 981 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (currVersion > currBeginVersion)
															#line 5102 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 982 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextSmallestVersion = std::min(currVersion, nextSmallestVersion);
															#line 5106 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 985 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						clearLogRangesRequired = false;
															#line 5112 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						break;
					}
				}
			}
		}
															#line 991 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endVersion.present() || backupVersions.size() != 1 || BUGGIFY)
															#line 5120 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 992 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!endVersion.present())
															#line 5124 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 994 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(backupLatestVersionsKey);
															#line 995 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (backupVersions.size() == 1)
															#line 5130 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 996 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					tr->clear(prefixRange(destUidValue.withPrefix(logRangesRange.begin)));
															#line 5134 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 1000 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->set(backupLatestVersionsKey, BinaryWriter::toValue<Version>(currEndVersion, Unversioned()));
															#line 5141 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 1004 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (clearLogRangesRequired)
															#line 5145 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1005 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if ((nextSmallestVersion - currBeginVersion) / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE >= std::numeric_limits<uint8_t>::max() || BUGGIFY)
															#line 5149 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1008 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Key baLogRangePrefix = destUidValue.withPrefix(backupLogKeys.begin);
															#line 1010 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					for(int h = 0;h <= std::numeric_limits<uint8_t>::max();h++) {
															#line 1011 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint64_t bv = bigEndian64(Version(0));
															#line 1012 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint64_t ev = bigEndian64(nextSmallestVersion);
															#line 1013 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint8_t h1 = h;
															#line 1014 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key vblockPrefix = StringRef(&h1, sizeof(uint8_t)).withPrefix(baLogRangePrefix);
															#line 1015 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						tr->clear(KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix), StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
															#line 5165 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
				else
				{
															#line 1019 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Standalone<VectorRef<KeyRangeRef>> ranges = getLogRanges(currBeginVersion, nextSmallestVersion, destUidValue);
															#line 1021 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					for( auto& range : ranges ) {
															#line 1022 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						tr->clear(range);
															#line 5176 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
			}
		}
		else
		{
															#line 1028 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(backupLatestVersionsPath));
															#line 1031 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(destUidValue.withPrefix(backupLogKeys.begin)));
															#line 1034 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(destUidValue.withPrefix(logRangesRange.begin)));
															#line 5189 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 1037 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!endVersion.present() && backupVersions.size() == 1)
															#line 5193 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1038 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr->getRange(KeyRangeRef(destUidLookupPrefix, strinc(destUidLookupPrefix)), CLIENT_KNOBS->TOO_MANY);
															#line 1038 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5199 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont6when1(__when_expr_2.get(), loopDepth); };
			static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 3;
															#line 1038 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 2, RangeResult >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 5204 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& __backupVersions,int loopDepth) 
	{
															#line 946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupVersions = __backupVersions;
															#line 5218 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && __backupVersions,int loopDepth) 
	{
		backupVersions = std::move(__backupVersions);
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1047 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5297 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_eraseLogDataActorState();
		static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont28(RangeResult const& existingDestUidValues,int loopDepth) 
	{
															#line 1040 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto it : existingDestUidValues ) {
															#line 1041 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (it.value == destUidValue)
															#line 5311 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1042 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(it.key);
															#line 5315 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont28(RangeResult && existingDestUidValues,int loopDepth) 
	{
															#line 1040 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto it : existingDestUidValues ) {
															#line 1041 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (it.value == destUidValue)
															#line 5328 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1042 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(it.key);
															#line 5332 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(RangeResult const& existingDestUidValues,int loopDepth) 
	{
		loopDepth = a_body1cont28(existingDestUidValues, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(RangeResult && existingDestUidValues,int loopDepth) 
	{
		loopDepth = a_body1cont28(std::move(existingDestUidValues), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key logUidValue;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key destUidValue;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Version> endVersion;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CheckBackupUID checkBackupUid;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version backupUid;
															#line 928 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsPath;
															#line 929 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsKey;
															#line 946 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult backupVersions;
															#line 950 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currBeginVersion;
															#line 965 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currEndVersion;
															#line 970 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version nextSmallestVersion;
															#line 5438 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via _eraseLogData()
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class _eraseLogDataActor final : public Actor<Void>, public ActorCallback< _eraseLogDataActor, 0, Optional<Value> >, public ActorCallback< _eraseLogDataActor, 1, RangeResult >, public ActorCallback< _eraseLogDataActor, 2, RangeResult >, public FastAllocated<_eraseLogDataActor>, public _eraseLogDataActorState<_eraseLogDataActor> {
															#line 5443 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<_eraseLogDataActor>::operator new;
	using FastAllocated<_eraseLogDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16487675066791003136UL, 12636403396316461312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _eraseLogDataActor, 0, Optional<Value> >;
friend struct ActorCallback< _eraseLogDataActor, 1, RangeResult >;
friend struct ActorCallback< _eraseLogDataActor, 2, RangeResult >;
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	_eraseLogDataActor(Reference<ReadYourWritesTransaction> const& tr,Key const& logUidValue,Key const& destUidValue,Optional<Version> const& endVersion,CheckBackupUID const& checkBackupUid,Version const& backupUid) 
															#line 5462 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   _eraseLogDataActorState<_eraseLogDataActor>(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7090378008957281792UL, 14346794352726563584UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_eraseLogData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _eraseLogDataActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _eraseLogDataActor, 2, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] static Future<Void> _eraseLogData( Reference<ReadYourWritesTransaction> const& tr, Key const& logUidValue, Key const& destUidValue, Optional<Version> const& endVersion, CheckBackupUID const& checkBackupUid, Version const& backupUid ) {
															#line 922 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new _eraseLogDataActor(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid));
															#line 5497 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1049 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

Future<Void> eraseLogData(Reference<ReadYourWritesTransaction> tr,
                          Key logUidValue,
                          Key destUidValue,
                          Optional<Version> endVersion,
                          CheckBackupUID checkBackupUid,
                          Version backupUid) {
	return _eraseLogData(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid);
}

															#line 5511 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via cleanupLogMutations()
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CleanupLogMutationsActor>
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupLogMutationsActorState {
															#line 5518 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupLogMutationsActorState(Database const& cx,Value const& destUidValue,bool const& deleteData) 
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   destUidValue(destUidValue),
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   deleteData(deleteData),
															#line 1060 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsPath(destUidValue.withPrefix(backupLatestVersionsPrefix)),
															#line 1062 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(new ReadYourWritesTransaction(cx)),
															#line 1063 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removingLogUid(),
															#line 1064 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   loggedLogUids()
															#line 5537 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("cleanupLogMutations", reinterpret_cast<unsigned long>(this));

	}
	~CleanupLogMutationsActorState() 
	{
		fdb_probe_actor_destroy("cleanupLogMutations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1066 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 5552 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CleanupLogMutationsActorState();
		static_cast<CleanupLogMutationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1068 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1069 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange( KeyRangeRef(backupLatestVersionsPath, strinc(backupLatestVersionsPath)), CLIENT_KNOBS->TOO_MANY);
															#line 1071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5589 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 1;
															#line 1071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5594 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1145 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr->onError(e);
															#line 1145 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5618 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 5;
															#line 1145 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 4, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5623 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1073 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		readVer = tr->getReadVersion().get();
															#line 1075 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		minVersion = std::numeric_limits<Version>::max();
															#line 1076 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		minVersionLogUid = Key();
															#line 1078 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupIdx = 0;
															#line 1079 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 5646 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __backupVersions,int loopDepth) 
	{
															#line 1071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupVersions = __backupVersions;
															#line 5655 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __backupVersions,int loopDepth) 
	{
		backupVersions = std::move(__backupVersions);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1117 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (deleteData)
															#line 5734 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1118 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (readVer - minVersion > CLIENT_KNOBS->MIN_CLEANUP_SECONDS * CLIENT_KNOBS->CORE_VERSIONSPERSECOND && (!removingLogUid.present() || minVersionLogUid == removingLogUid.get()))
															#line 5738 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1120 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				removingLogUid = minVersionLogUid;
															#line 1121 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_2 = eraseLogData(tr, minVersionLogUid, destUidValue);
															#line 1121 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5746 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 3;
															#line 1121 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 2, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5751 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 1125 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (removingLogUid.present() && minVersionLogUid != removingLogUid.get())
															#line 5758 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1126 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("\nWARNING: The oldest tag was possibly removed, run again without `--delete-data' to " "check.\n\n");
															#line 5762 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1129 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("\nWARNING: Did not delete data because the tag is not at least %.4f hours behind. Change " "`--min-cleanup-seconds' to adjust this threshold.\n\n", CLIENT_KNOBS->MIN_CLEANUP_SECONDS / 3600.0);
															#line 5768 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				loopDepth = a_body1loopBody1cont5(loopDepth);
			}
		}
		else
		{
															#line 1133 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (readVer - minVersion > CLIENT_KNOBS->MIN_CLEANUP_SECONDS * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)
															#line 5777 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1135 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("\nPassing `--delete-data' would delete the tag that is %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5781 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1138 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("\nPassing `--delete-data' would not delete the tag that is %.4f hours behind. Change " "`--min-cleanup-seconds' to adjust the cleanup threshold.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5787 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1079 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(backupIdx < backupVersions.size()))
															#line 5805 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1080 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currVersion = BinaryReader::fromStringRef<Version>(backupVersions[backupIdx].value, Unversioned());
															#line 1082 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currLogUid = backupVersions[backupIdx].key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 1084 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (currVersion < minVersion)
															#line 5815 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1085 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			minVersionLogUid = currLogUid;
															#line 1086 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			minVersion = currVersion;
															#line 5821 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 1089 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!loggedLogUids.count(currLogUid))
															#line 5825 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1090 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			foundDRKey = tr->get(Subspace(databaseBackupPrefixRange.begin) .get(BackupAgentBase::keySourceStates) .get(currLogUid) .pack(DatabaseBackupAgent::keyStateStatus));
															#line 1094 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			foundBackupKey = tr->get( Subspace(currLogUid.withPrefix("uid->config/"_sr).withPrefix(fileBackupPrefixRange.begin)) .pack("stateEnum"_sr));
															#line 1097 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(foundDRKey) && success(foundBackupKey);
															#line 1097 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5835 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 2;
															#line 1097 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 1, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5840 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1079 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupIdx++;
															#line 5867 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1099 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (foundDRKey.get().present() && foundBackupKey.get().present())
															#line 5876 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1100 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			printf("WARNING: Found a tag that looks like both a backup and a DR. This tag is %.4f hours " "behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5880 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1103 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (foundDRKey.get().present() && !foundBackupKey.get().present())
															#line 5886 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1104 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("Found a DR that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5890 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1106 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!foundDRKey.get().present() && foundBackupKey.get().present())
															#line 5896 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1107 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("Found a Backup that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5900 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1110 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("WARNING: Found an unknown tag that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5906 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1113 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		loggedLogUids.insert(currLogUid);
															#line 5912 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1099 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (foundDRKey.get().present() && foundBackupKey.get().present())
															#line 5921 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1100 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			printf("WARNING: Found a tag that looks like both a backup and a DR. This tag is %.4f hours " "behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5925 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1103 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (foundDRKey.get().present() && !foundBackupKey.get().present())
															#line 5931 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1104 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("Found a DR that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5935 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1106 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!foundDRKey.get().present() && foundBackupKey.get().present())
															#line 5941 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1107 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("Found a Backup that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5945 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1110 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("WARNING: Found an unknown tag that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 5951 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1113 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		loggedLogUids.insert(currLogUid);
															#line 5957 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1143 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CleanupLogMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CleanupLogMutationsActorState(); static_cast<CleanupLogMutationsActor*>(this)->destroy(); return 0; }
															#line 6041 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CleanupLogMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CleanupLogMutationsActorState();
		static_cast<CleanupLogMutationsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6061 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 4;
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 3, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6066 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6077 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 4;
															#line 1122 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 3, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6082 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 1123 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		printf("\nSuccessfully removed the tag that was %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6166 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 1123 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		printf("\nSuccessfully removed the tag that was %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6175 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Value destUidValue;
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool deleteData;
															#line 1060 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsPath;
															#line 1062 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1063 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Key> removingLogUid;
															#line 1064 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::set<Key> loggedLogUids;
															#line 1071 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult backupVersions;
															#line 1073 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version readVer;
															#line 1075 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version minVersion;
															#line 1076 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key minVersionLogUid;
															#line 1078 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int backupIdx;
															#line 1080 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currVersion;
															#line 1082 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key currLogUid;
															#line 1090 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Optional<Value>> foundDRKey;
															#line 1094 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Optional<Value>> foundBackupKey;
															#line 6374 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via cleanupLogMutations()
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupLogMutationsActor final : public Actor<Void>, public ActorCallback< CleanupLogMutationsActor, 0, RangeResult >, public ActorCallback< CleanupLogMutationsActor, 1, Void >, public ActorCallback< CleanupLogMutationsActor, 2, Void >, public ActorCallback< CleanupLogMutationsActor, 3, Void >, public ActorCallback< CleanupLogMutationsActor, 4, Void >, public FastAllocated<CleanupLogMutationsActor>, public CleanupLogMutationsActorState<CleanupLogMutationsActor> {
															#line 6379 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CleanupLogMutationsActor>::operator new;
	using FastAllocated<CleanupLogMutationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2818036558706131200UL, 16365479254239541760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CleanupLogMutationsActor, 0, RangeResult >;
friend struct ActorCallback< CleanupLogMutationsActor, 1, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 2, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 3, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 4, Void >;
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupLogMutationsActor(Database const& cx,Value const& destUidValue,bool const& deleteData) 
															#line 6400 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CleanupLogMutationsActorState<CleanupLogMutationsActor>(cx, destUidValue, deleteData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5485089562015519744UL, 10719617229371527680UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cleanupLogMutations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> cleanupLogMutations( Database const& cx, Value const& destUidValue, bool const& deleteData ) {
															#line 1059 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CleanupLogMutationsActor(cx, destUidValue, deleteData));
															#line 6437 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1149 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 6442 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via cleanupBackup()
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CleanupBackupActor>
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupBackupActorState {
															#line 6449 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupBackupActorState(Database const& cx,DeleteData const& deleteData) 
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   deleteData(deleteData),
															#line 1151 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(new ReadYourWritesTransaction(cx))
															#line 6460 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("cleanupBackup", reinterpret_cast<unsigned long>(this));

	}
	~CleanupBackupActorState() 
	{
		fdb_probe_actor_destroy("cleanupBackup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1152 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 6475 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CleanupBackupActorState();
		static_cast<CleanupBackupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1154 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1155 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(KeyRangeRef(destUidLookupPrefix, strinc(destUidLookupPrefix)), CLIENT_KNOBS->TOO_MANY);
															#line 1157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6512 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 1;
															#line 1157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 0, RangeResult >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6517 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1165 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->onError(e);
															#line 1165 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6541 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 3;
															#line 1165 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 2, Void >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6546 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1160 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		RangeForbody1loopBody1cont2Iterator0 = std::begin(destUids);
															#line 6561 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __destUids,int loopDepth) 
	{
															#line 1157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		destUids = __destUids;
															#line 6570 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __destUids,int loopDepth) 
	{
		destUids = std::move(__destUids);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1163 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CleanupBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CleanupBackupActorState(); static_cast<CleanupBackupActor*>(this)->destroy(); return 0; }
															#line 6649 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CleanupBackupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CleanupBackupActorState();
		static_cast<CleanupBackupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1160 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(RangeForbody1loopBody1cont2Iterator0 != std::end(destUids)))
															#line 6668 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1160 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		auto destUid = *RangeForbody1loopBody1cont2Iterator0;
															#line 6674 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1161 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = cleanupLogMutations(cx, destUid.value, deleteData);
															#line 1161 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6680 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 2;
															#line 1161 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 1, Void >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6685 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1160 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++RangeForbody1loopBody1cont2Iterator0;
															#line 6708 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DeleteData deleteData;
															#line 1151 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1157 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult destUids;
															#line 1160 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	decltype(std::begin(std::declval<RangeResult>())) RangeForbody1loopBody1cont2Iterator0;
															#line 6897 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via cleanupBackup()
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupBackupActor final : public Actor<Void>, public ActorCallback< CleanupBackupActor, 0, RangeResult >, public ActorCallback< CleanupBackupActor, 1, Void >, public ActorCallback< CleanupBackupActor, 2, Void >, public FastAllocated<CleanupBackupActor>, public CleanupBackupActorState<CleanupBackupActor> {
															#line 6902 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CleanupBackupActor>::operator new;
	using FastAllocated<CleanupBackupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7526748196842913792UL, 8135992260954583808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CleanupBackupActor, 0, RangeResult >;
friend struct ActorCallback< CleanupBackupActor, 1, Void >;
friend struct ActorCallback< CleanupBackupActor, 2, Void >;
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupBackupActor(Database const& cx,DeleteData const& deleteData) 
															#line 6921 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CleanupBackupActorState<CleanupBackupActor>(cx, deleteData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8531268470707185664UL, 16321355424300612352UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cleanupBackup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CleanupBackupActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CleanupBackupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CleanupBackupActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> cleanupBackup( Database const& cx, DeleteData const& deleteData ) {
															#line 1150 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CleanupBackupActor(cx, deleteData));
															#line 6956 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1169 "/codebuild/output/src2298219144/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// Convert the status text to an enumerated value
BackupAgentBase::EnumState BackupAgentBase::getState(std::string const& stateText) {
	auto enState = EnumState::STATE_ERRORED;

	if (stateText.empty()) {
		enState = EnumState::STATE_NEVERRAN;
	}

	else if (!stateText.compare("has been submitted")) {
		enState = EnumState::STATE_SUBMITTED;
	}

	else if (!stateText.compare("has been started")) {
		enState = EnumState::STATE_RUNNING;
	}

	else if (!stateText.compare("is differential")) {
		enState = EnumState::STATE_RUNNING_DIFFERENTIAL;
	}

	else if (!stateText.compare("has been completed")) {
		enState = EnumState::STATE_COMPLETED;
	}

	else if (!stateText.compare("has been aborted")) {
		enState = EnumState::STATE_ABORTED;
	}

	else if (!stateText.compare("has been partially aborted")) {
		enState = EnumState::STATE_PARTIALLY_ABORTED;
	}

	return enState;
}

const char* BackupAgentBase::getStateText(EnumState enState) {
	const char* stateText;

	switch (enState) {
	case EnumState::STATE_ERRORED:
		stateText = "has errored";
		break;
	case EnumState::STATE_NEVERRAN:
		stateText = "has never been started";
		break;
	case EnumState::STATE_SUBMITTED:
		stateText = "has been submitted";
		break;
	case EnumState::STATE_RUNNING:
		stateText = "has been started";
		break;
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
		stateText = "is differential";
		break;
	case EnumState::STATE_COMPLETED:
		stateText = "has been completed";
		break;
	case EnumState::STATE_ABORTED:
		stateText = "has been aborted";
		break;
	case EnumState::STATE_PARTIALLY_ABORTED:
		stateText = "has been partially aborted";
		break;
	default:
		stateText = "<undefined>";
		break;
	}

	return stateText;
}

const char* BackupAgentBase::getStateName(EnumState enState) {
	switch (enState) {
	case EnumState::STATE_ERRORED:
		return "Errored";
	case EnumState::STATE_NEVERRAN:
		return "NeverRan";
	case EnumState::STATE_SUBMITTED:
		return "Submitted";
		break;
	case EnumState::STATE_RUNNING:
		return "Running";
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
		return "RunningDifferentially";
	case EnumState::STATE_COMPLETED:
		return "Completed";
	case EnumState::STATE_ABORTED:
		return "Aborted";
	case EnumState::STATE_PARTIALLY_ABORTED:
		return "Aborting";
	default:
		return "<undefined>";
	}
}

bool BackupAgentBase::isRunnable(EnumState enState) {
	switch (enState) {
	case EnumState::STATE_SUBMITTED:
	case EnumState::STATE_RUNNING:
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
	case EnumState::STATE_PARTIALLY_ABORTED:
		return true;
	default:
		return false;
	}
}

Standalone<StringRef> BackupAgentBase::getCurrentTime() {
	double t = now();
	time_t curTime = t;
	char buffer[128];
	struct tm* timeinfo;
	timeinfo = localtime(&curTime);
	strftime(buffer, 128, "%Y-%m-%d-%H-%M-%S", timeinfo);

	std::string time(buffer);
	return StringRef(time + format(".%06d", (int)(1e6 * (t - curTime))));
}

std::string const BackupAgentBase::defaultTagName = "default";

void addDefaultBackupRanges(Standalone<VectorRef<KeyRangeRef>>& backupKeys) {
	backupKeys.push_back_deep(backupKeys.arena(), normalKeys);

	for (auto& r : getSystemBackupRanges()) {
		backupKeys.push_back_deep(backupKeys.arena(), r);
	}
}

VectorRef<KeyRangeRef> const& getSystemBackupRanges() {
	static Standalone<VectorRef<KeyRangeRef>> systemBackupRanges;
	return systemBackupRanges;
}

KeyRangeMap<bool> const& systemBackupMutationMask() {
	static KeyRangeMap<bool> mask;
	if (mask.size() == 1) {
		for (auto r : getSystemBackupRanges()) {
			mask.insert(r, true);
		}
	}

	return mask;
}

KeyRangeRef const& getDefaultBackupSharedRange() {
	static KeyRangeRef defaultSharedRange(""_sr, ""_sr);
	return defaultSharedRange;
}
