#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
/*
 * ManagementAPI.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cinttypes>
#include <cstddef>
#include <string>
#include <vector>

#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/GenericManagementAPI.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/RangeLock.h"
#include "flow/Error.h"
#include "fmt/format.h"
#include "fdbclient/Knobs.h"
#include "flow/Arena.h"
#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/ManagementAPI.actor.h"

#include "fdbclient/SystemData.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/CoordinationInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbrpc/simulator.h"
#include "fdbclient/StatusClient.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "fdbrpc/ReplicationPolicy.h"
#include "fdbrpc/Replication.h"
#include "fdbclient/Schemas.h"
#include "fdbrpc/SimulatorProcessInfo.h"

#include "flow/actorcompiler.h" // This must be the last #include.

bool isInteger(const std::string& s) {
	if (s.empty())
		return false;
	char* p;
	strtol(s.c_str(), &p, 10);
	return (*p == 0);
}

// Defines the mapping between configuration names (as exposed by fdbcli, buildConfiguration()) and actual configuration
// parameters
std::map<std::string, std::string> configForToken(std::string const& mode) {
	std::map<std::string, std::string> out;
	std::string p = configKeysPrefix.toString();

	if (mode == "new") {
		out[p + "initialized"] = "1";
		return out;
	}

	if (mode == "tss") {
		// Set temporary marker in config map to mark that this is a tss configuration and not a normal storage/log
		// configuration. A bit of a hack but reuses the parsing code nicely.
		out[p + "istss"] = "1";
		return out;
	}

	if (mode == "locked") {
		// Setting this key is interpreted as an instruction to use the normal version-stamp-based mechanism for locking
		// the database.
		out[databaseLockedKey.toString()] = deterministicRandom()->randomUniqueID().toString();
		return out;
	}

	size_t pos;

	// key:=value is unvalidated and unchecked
	pos = mode.find(":=");
	if (pos != std::string::npos) {
		out[p + mode.substr(0, pos)] = mode.substr(pos + 2);
		return out;
	}

	// key=value is constrained to a limited set of options and basic validation is performed
	pos = mode.find("=");
	if (pos != std::string::npos) {
		std::string key = mode.substr(0, pos);
		std::string value = mode.substr(pos + 1);

		if (key == "proxies" && isInteger(value)) {
			printf("Warning: Proxy role is being split into GRV Proxy and Commit Proxy, now prefer configuring "
			       "'grv_proxies' and 'commit_proxies' separately. Generally we should follow that 'commit_proxies'"
			       " is three times of 'grv_proxies' count and 'grv_proxies' should be not more than 4.\n");
			int proxiesCount = atoi(value.c_str());
			if (proxiesCount == -1) {
				proxiesCount = CLIENT_KNOBS->DEFAULT_AUTO_GRV_PROXIES + CLIENT_KNOBS->DEFAULT_AUTO_COMMIT_PROXIES;
				ASSERT_WE_THINK(proxiesCount >= 2);
			}

			if (proxiesCount < 2) {
				printf("Error: At least 2 proxies (1 GRV proxy and 1 Commit proxy) are required.\n");
				return out;
			}

			int grvProxyCount = std::max(1,
			                             std::min(CLIENT_KNOBS->DEFAULT_MAX_GRV_PROXIES,
			                                      proxiesCount / (CLIENT_KNOBS->DEFAULT_COMMIT_GRV_PROXIES_RATIO + 1)));
			int commitProxyCount = proxiesCount - grvProxyCount;
			ASSERT_WE_THINK(grvProxyCount >= 1 && commitProxyCount >= 1);

			out[p + "grv_proxies"] = std::to_string(grvProxyCount);
			out[p + "commit_proxies"] = std::to_string(commitProxyCount);
			printf("%d proxies are automatically converted into %d GRV proxies and %d Commit proxies.\n",
			       proxiesCount,
			       grvProxyCount,
			       commitProxyCount);

			TraceEvent("DatabaseConfigurationProxiesSpecified")
			    .detail("SpecifiedProxies", atoi(value.c_str()))
			    .detail("EffectiveSpecifiedProxies", proxiesCount)
			    .detail("ConvertedGrvProxies", grvProxyCount)
			    .detail("ConvertedCommitProxies", commitProxyCount);
		}

		if ((key == "logs" || key == "commit_proxies" || key == "grv_proxies" || key == "resolvers" ||
		     key == "remote_logs" || key == "log_routers" || key == "usable_regions" ||
		     key == "repopulate_anti_quorum" || key == "count") &&
		    isInteger(value)) {
			out[p + key] = value;
		}

		if (key == "regions") {
			json_spirit::mValue mv;
			json_spirit::read_string(value, mv);

			StatusObject regionObj;
			regionObj["regions"] = mv;
			out[p + key] =
			    BinaryWriter::toValue(regionObj, IncludeVersion(ProtocolVersion::withRegionConfiguration())).toString();
		}

		if (key == "perpetual_storage_wiggle" && isInteger(value)) {
			int ppWiggle = std::stoi(value);
			if (ppWiggle >= 2 || ppWiggle < 0) {
				printf("Error: Only 0 and 1 are valid values of perpetual_storage_wiggle at present.\n");
				return out;
			}
			out[p + key] = value;
		}
		if (key == "perpetual_storage_wiggle_locality") {
			if (!isValidPerpetualStorageWiggleLocality(value)) {
				printf("Error: perpetual_storage_wiggle_locality should be in <locality_key>:<locality_value> "
				       "format or enter 0 to disable the locality match for wiggling.\n");
				return out;
			}
			out[p + key] = value;
		}
		if (key == "storage_migration_type") {
			StorageMigrationType type;
			if (value == "disabled") {
				type = StorageMigrationType::DISABLED;
			} else if (value == "aggressive") {
				type = StorageMigrationType::AGGRESSIVE;
			} else if (value == "gradual") {
				type = StorageMigrationType::GRADUAL;
			} else {
				printf("Error: Only disabled|aggressive|gradual are valid for storage_migration_type.\n");
				return out;
			}
			out[p + key] = format("%d", type);
		}

		if (key == "blob_granules_enabled") {
			int enabled = std::stoi(value);
			if (enabled != 0 && enabled != 1) {
				printf("Error: Only 0 or 1 are valid values for blob_granules_enabled. "
				       "1 enables blob granules and 0 disables them.\n");
				return out;
			}
			out[p + key] = value;
		}

		if (key == "tenant_mode") {
			TenantMode tenantMode;
			if (value == "disabled") {
				tenantMode = TenantMode::DISABLED;
			} else if (value == "optional_experimental") {
				tenantMode = TenantMode::OPTIONAL_TENANT;
			} else if (value == "required_experimental") {
				tenantMode = TenantMode::REQUIRED;
			} else {
				printf("Error: Only disabled|optional_experimental|required_experimental are valid for tenant_mode.\n");
				return out;
			}
			out[p + key] = format("%d", tenantMode);
		}

		if (key == "encryption_at_rest_mode") {
			EncryptionAtRestMode mode;
			if (value == "disabled") {
				mode = EncryptionAtRestMode::DISABLED;
			} else if (value == "domain_aware") {
				mode = EncryptionAtRestMode::DOMAIN_AWARE;
			} else if (value == "cluster_aware") {
				mode = EncryptionAtRestMode::CLUSTER_AWARE;
			} else {
				printf("Error: Only disabled|domain_aware|cluster_aware are valid for encryption_at_rest_mode.\n");
				return out;
			}
			out[p + key] = format("%d", mode);
		}

		if (key == "exclude") {
			int p = 0;
			while (p < value.size()) {
				int end = value.find_first_of(',', p);
				if (end == value.npos) {
					end = value.size();
				}
				auto addrRef = StringRef(value).substr(p, end - p);
				AddressExclusion addr = AddressExclusion::parse(addrRef);
				if (addr.isValid()) {
					out[encodeExcludedServersKey(addr)] = "";
				} else {
					printf("Error: invalid address format: %s\n", addrRef.toString().c_str());
				}
				p = end + 1;
			}
		}

		if (key == "storage_engine" || key == "log_engine" || key == "perpetual_storage_wiggle_engine") {
			StringRef s = value;

			// Parse as engine_name[:p=v]... to handle future storage engine params
			Value engine = s.eat(":");
			std::map<Key, Value> params;
			while (!s.empty()) {
				params[s.eat("=")] = s.eat(":");
			}

			try {
				out[p + key] = format("%d", KeyValueStoreType::fromString(engine.toString()).storeType());
			} catch (Error& e) {
				printf("Error: Invalid value for %s (%s): %s\n", key.c_str(), value.c_str(), e.what());
			}
			return out;
		}

		return out;
	}

	Optional<KeyValueStoreType> logType;
	Optional<KeyValueStoreType> storeType;

	// These are legacy shorthand commands to set a specific log engine and storage engine
	// based only on the storage engine name.  Most of them assume SQLite should be the
	// log engine.
	if (mode == "ssd-1") {
		logType = KeyValueStoreType::SSD_BTREE_V1;
		storeType = KeyValueStoreType::SSD_BTREE_V1;
	} else if (mode == "ssd" || mode == "ssd-2") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_BTREE_V2;
	} else if (mode == "ssd-redwood-1") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_REDWOOD_V1;
	} else if (mode == "ssd-rocksdb-v1") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_ROCKSDB_V1;
	} else if (mode == "ssd-sharded-rocksdb") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_SHARDED_ROCKSDB;
	} else if (mode == "memory" || mode == "memory-2") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::MEMORY;
	} else if (mode == "memory-1") {
		logType = KeyValueStoreType::MEMORY;
		storeType = KeyValueStoreType::MEMORY;
	} else if (mode == "memory-radixtree" || mode == "memory-radixtree-beta") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::MEMORY_RADIXTREE;
	}
	// Add any new store types to fdbserver/workloads/ConfigureDatabase, too

	if (storeType.present()) {
		out[p + "log_engine"] = format("%d", logType.get().storeType());
		out[p + "storage_engine"] = format("%d", storeType.get().storeType());
		return out;
	}

	std::string redundancy, log_replicas;
	Reference<IReplicationPolicy> storagePolicy;
	Reference<IReplicationPolicy> tLogPolicy;

	bool redundancySpecified = true;
	if (mode == "single") {
		redundancy = "1";
		log_replicas = "1";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(new PolicyOne());

	} else if (mode == "double" || mode == "fast_recovery_double") {
		redundancy = "2";
		log_replicas = "2";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "triple" || mode == "fast_recovery_triple") {
		redundancy = "3";
		log_replicas = "3";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "three_datacenter" || mode == "multi_dc") {
		redundancy = "6";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_datacenter_fallback") {
		redundancy = "4";
		log_replicas = "4";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_data_hall") {
		redundancy = "3";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "data_hall", Reference<IReplicationPolicy>(new PolicyOne())));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_data_hall_fallback") {
		redundancy = "2";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "data_hall", Reference<IReplicationPolicy>(new PolicyOne())));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else
		redundancySpecified = false;
	if (redundancySpecified) {
		out[p + "storage_replicas"] = redundancy;
		out[p + "log_replicas"] = log_replicas;
		out[p + "log_anti_quorum"] = "0";

		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, storagePolicy);
		out[p + "storage_replication_policy"] = policyWriter.toValue().toString();

		policyWriter = BinaryWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, tLogPolicy);
		out[p + "log_replication_policy"] = policyWriter.toValue().toString();
		return out;
	}

	std::string remote_redundancy, remote_log_replicas;
	Reference<IReplicationPolicy> remoteTLogPolicy;
	bool remoteRedundancySpecified = true;
	if (mode == "remote_default") {
		remote_redundancy = "0";
		remote_log_replicas = "0";
		remoteTLogPolicy = Reference<IReplicationPolicy>();
	} else if (mode == "remote_single") {
		remote_redundancy = "1";
		remote_log_replicas = "1";
		remoteTLogPolicy = Reference<IReplicationPolicy>(new PolicyOne());
	} else if (mode == "remote_double") {
		remote_redundancy = "2";
		remote_log_replicas = "2";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "remote_triple") {
		remote_redundancy = "3";
		remote_log_replicas = "3";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "remote_three_data_hall") { // FIXME: not tested in simulation
		remote_redundancy = "3";
		remote_log_replicas = "4";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else
		remoteRedundancySpecified = false;
	if (remoteRedundancySpecified) {
		out[p + "remote_log_replicas"] = remote_log_replicas;

		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, remoteTLogPolicy);
		out[p + "remote_log_policy"] = policyWriter.toValue().toString();
		return out;
	}

	return out;
}

ConfigurationResult buildConfiguration(std::vector<StringRef> const& modeTokens,
                                       std::map<std::string, std::string>& outConf) {
	for (auto it : modeTokens) {
		std::string mode = it.toString();
		auto m = configForToken(mode);
		if (!m.size()) {
			TraceEvent(SevWarnAlways, "UnknownOption").detail("Option", mode);
			return ConfigurationResult::UNKNOWN_OPTION;
		}

		for (auto t = m.begin(); t != m.end(); ++t) {
			if (outConf.count(t->first)) {
				TraceEvent(SevWarnAlways, "ConflictingOption")
				    .detail("Option", t->first)
				    .detail("Value", t->second)
				    .detail("ExistingValue", outConf[t->first]);
				return ConfigurationResult::CONFLICTING_OPTIONS;
			}
			outConf[t->first] = t->second;
		}
	}
	auto p = configKeysPrefix.toString();
	if (!outConf.count(p + "storage_replication_policy") && outConf.count(p + "storage_replicas")) {
		int storageCount = stoi(outConf[p + "storage_replicas"]);
		Reference<IReplicationPolicy> storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(storageCount, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, storagePolicy);
		outConf[p + "storage_replication_policy"] = policyWriter.toValue().toString();
	}

	if (!outConf.count(p + "log_replication_policy") && outConf.count(p + "log_replicas")) {
		int logCount = stoi(outConf[p + "log_replicas"]);
		Reference<IReplicationPolicy> logPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(logCount, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, logPolicy);
		outConf[p + "log_replication_policy"] = policyWriter.toValue().toString();
	}
	if (outConf.count(p + "istss")) {
		// redo config parameters to be tss config instead of normal config

		// save param values from parsing as a normal config
		bool isNew = outConf.count(p + "initialized");
		Optional<std::string> count;
		Optional<std::string> storageEngine;
		if (outConf.count(p + "count")) {
			count = Optional<std::string>(outConf[p + "count"]);
		}
		if (outConf.count(p + "storage_engine")) {
			storageEngine = Optional<std::string>(outConf[p + "storage_engine"]);
		}

		// A new tss setup must have count + storage engine. An adjustment must have at least one.
		if ((isNew && (!count.present() || !storageEngine.present())) ||
		    (!isNew && !count.present() && !storageEngine.present())) {
			return ConfigurationResult::INCOMPLETE_CONFIGURATION;
		}

		// clear map and only reset tss parameters
		outConf.clear();
		if (count.present()) {
			outConf[p + "tss_count"] = count.get();
		}
		if (storageEngine.present()) {
			outConf[p + "tss_storage_engine"] = storageEngine.get();
		}
	}
	return ConfigurationResult::SUCCESS;
}

ConfigurationResult buildConfiguration(std::string const& configMode, std::map<std::string, std::string>& outConf) {
	std::vector<StringRef> modes;

	int p = 0;
	while (p < configMode.size()) {
		int end = configMode.find_first_of(' ', p);
		if (end == configMode.npos)
			end = configMode.size();
		modes.push_back(StringRef(configMode).substr(p, end - p));
		p = end + 1;
	}

	return buildConfiguration(modes, outConf);
}

bool isCompleteConfiguration(std::map<std::string, std::string> const& options) {
	std::string p = configKeysPrefix.toString();

	return options.count(p + "log_replicas") == 1 && options.count(p + "log_anti_quorum") == 1 &&
	       options.count(p + "storage_replicas") == 1 && options.count(p + "log_engine") == 1 &&
	       options.count(p + "storage_engine") == 1;
}

															#line 522 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via disableBackupWorker()
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class DisableBackupWorkerActor>
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class DisableBackupWorkerActorState {
															#line 529 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	DisableBackupWorkerActorState(Database const& cx) 
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 536 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("disableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	~DisableBackupWorkerActorState() 
	{
		fdb_probe_actor_destroy("disableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 521 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(cx);
															#line 521 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 553 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 1;
															#line 521 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 558 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
															#line 522 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!configuration.backupWorkerEnabled)
															#line 581 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 523 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyDisabled");
															#line 524 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 587 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DisableBackupWorkerActorState();
			static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=0", true);
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 597 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 602 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration && configuration,int loopDepth) 
	{
															#line 522 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!configuration.backupWorkerEnabled)
															#line 611 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 523 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyDisabled");
															#line 524 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 617 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DisableBackupWorkerActorState();
			static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=0", true);
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 627 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 632 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(configuration, loopDepth);

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration && configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(configuration), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<DisableBackupWorkerActor*>(this)->ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(ConfigurationResult const& res,int loopDepth) 
	{
															#line 527 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 528 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerDisableFailed").detail("Result", res);
															#line 529 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 722 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 726 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ConfigurationResult && res,int loopDepth) 
	{
															#line 527 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 738 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 528 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerDisableFailed").detail("Result", res);
															#line 529 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 744 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 748 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<DisableBackupWorkerActor*>(this)->ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >::remove();

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult const& value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult && value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,Error err) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 833 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via disableBackupWorker()
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class DisableBackupWorkerActor final : public Actor<Void>, public ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >, public ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >, public FastAllocated<DisableBackupWorkerActor>, public DisableBackupWorkerActorState<DisableBackupWorkerActor> {
															#line 838 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<DisableBackupWorkerActor>::operator new;
	using FastAllocated<DisableBackupWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14911531485033208576UL, 11190295745586503680UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >;
friend struct ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >;
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	DisableBackupWorkerActor(Database const& cx) 
															#line 856 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   DisableBackupWorkerActorState<DisableBackupWorkerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(814777889414831360UL, 5570830813057092096UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("disableBackupWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> disableBackupWorker( Database const& cx ) {
															#line 520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new DisableBackupWorkerActor(cx));
															#line 890 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 533 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 895 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via enableBackupWorker()
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class EnableBackupWorkerActor>
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class EnableBackupWorkerActorState {
															#line 902 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	EnableBackupWorkerActorState(Database const& cx) 
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("enableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	~EnableBackupWorkerActorState() 
	{
		fdb_probe_actor_destroy("enableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 535 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(cx);
															#line 535 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 926 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 1;
															#line 535 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 931 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
															#line 536 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (configuration.backupWorkerEnabled)
															#line 954 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 537 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyEnabled");
															#line 538 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 960 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~EnableBackupWorkerActorState();
			static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=1", true);
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 970 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 975 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration && configuration,int loopDepth) 
	{
															#line 536 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (configuration.backupWorkerEnabled)
															#line 984 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 537 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyEnabled");
															#line 538 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 990 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~EnableBackupWorkerActorState();
			static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=1", true);
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1000 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 1005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(configuration, loopDepth);

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration && configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(configuration), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<EnableBackupWorkerActor*>(this)->ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(ConfigurationResult const& res,int loopDepth) 
	{
															#line 541 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 1089 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 542 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerEnableFailed").detail("Result", res);
															#line 543 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 1095 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 545 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 1099 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ConfigurationResult && res,int loopDepth) 
	{
															#line 541 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 1111 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 542 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerEnableFailed").detail("Result", res);
															#line 543 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 1117 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 545 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 1121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<EnableBackupWorkerActor*>(this)->ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >::remove();

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult const& value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult && value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,Error err) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1206 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via enableBackupWorker()
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class EnableBackupWorkerActor final : public Actor<Void>, public ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >, public ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >, public FastAllocated<EnableBackupWorkerActor>, public EnableBackupWorkerActorState<EnableBackupWorkerActor> {
															#line 1211 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<EnableBackupWorkerActor>::operator new;
	using FastAllocated<EnableBackupWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1530202423122730752UL, 4922056719247076352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >;
friend struct ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >;
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	EnableBackupWorkerActor(Database const& cx) 
															#line 1229 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   EnableBackupWorkerActorState<EnableBackupWorkerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12529125006175537920UL, 10565899026445989632UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("enableBackupWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> enableBackupWorker( Database const& cx ) {
															#line 534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new EnableBackupWorkerActor(cx));
															#line 1263 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 547 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

/*
    - Validates encryption and tenant mode configurations
    - During cluster creation (configure new) we allow the following:
        - If encryption mode is disabled/cluster_aware then any tenant mode is allowed
        - If the encryption mode is domain_aware then the only allowed tenant mode is required
    - During cluster configuration changes the following is allowed:
        - Encryption mode cannot be changed (can only be set during creation)
        - If the encryption mode is disabled/cluster_aware then any tenant mode changes are allowed
        - If the encryption mode is domain_aware then tenant mode changes are not allowed (as the only supported mode is
          required)
*/
bool isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration> oldConfiguration,
                                       std::map<std::string, std::string> newConfig,
                                       bool creating) {
	EncryptionAtRestMode encryptMode;
	TenantMode tenantMode;
	if (creating) {
		if (newConfig.count(encryptionAtRestModeConfKey.toString()) != 0) {
			encryptMode = EncryptionAtRestMode::fromValueRef(
			    ValueRef(newConfig.find(encryptionAtRestModeConfKey.toString())->second));
			// check if the tenant mode is being set during configure new (otherwise assume tenants are disabled)
			if (newConfig.count(tenantModeConfKey.toString()) != 0) {
				tenantMode = TenantMode::fromValue(ValueRef(newConfig.find(tenantModeConfKey.toString())->second));
			}
		}
	} else {
		ASSERT(oldConfiguration.present());
		encryptMode = oldConfiguration.get().encryptionAtRestMode;
		if (newConfig.count(tenantModeConfKey.toString()) != 0) {
			tenantMode = TenantMode::fromValue(ValueRef(newConfig.find(tenantModeConfKey.toString())->second));
		} else {
			// Tenant mode and encryption mode didn't change
			return true;
		}
	}
	TraceEvent(SevDebug, "EncryptAndTenantModes")
	    .detail("EncryptMode", encryptMode.toString())
	    .detail("TenantMode", tenantMode.toString());

	if (encryptMode.mode == EncryptionAtRestMode::DOMAIN_AWARE && tenantMode != TenantMode::REQUIRED) {
		// For domain aware encryption only the required tenant mode is currently supported
		TraceEvent(SevWarnAlways, "InvalidEncryptAndTenantConfiguration")
		    .detail("EncryptMode", encryptMode.toString())
		    .detail("TenantMode", tenantMode.toString());
		return false;
	}

	return true;
}

bool isTenantModeModeConfigValid(DatabaseConfiguration oldConfiguration, DatabaseConfiguration newConfiguration) {
	TenantMode oldTenantMode = oldConfiguration.tenantMode;
	TenantMode newTenantMode = newConfiguration.tenantMode;
	TraceEvent(SevDebug, "TenantModes")
	    .detail("OldTenantMode", oldTenantMode.toString())
	    .detail("NewTenantMode", newTenantMode.toString());
	if (oldTenantMode != TenantMode::REQUIRED && newTenantMode == TenantMode::REQUIRED) {
		// TODO: Changing from optional/disabled to required tenant mode should be allowed if there is no non-tenant
		// data present
		TraceEvent(SevWarnAlways, "InvalidTenantConfiguration")
		    .detail("OldTenantMode", oldTenantMode.toString())
		    .detail("NewTenantMode", newTenantMode.toString());
		return false;
	}
	return true;
}

															#line 1335 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase615()
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class FlowTestCase615Actor>
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase615ActorState {
															#line 1342 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase615ActorState(UnitTestParameters const& params) 
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : params(params)
															#line 1349 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase615", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase615ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase615", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 616 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration oldConfig;
															#line 617 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration newConfig;
															#line 618 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<TenantMode> tenantModes = { TenantMode::DISABLED, TenantMode::OPTIONAL_TENANT, TenantMode::REQUIRED };
															#line 620 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = TenantMode::REQUIRED;
															#line 621 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 622 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 624 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 625 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 626 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 628 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 629 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = TenantMode::REQUIRED;
															#line 630 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 632 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<FlowTestCase615Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase615ActorState(); static_cast<FlowTestCase615Actor*>(this)->destroy(); return 0; }
															#line 1388 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<FlowTestCase615Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase615ActorState();
			static_cast<FlowTestCase615Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase615ActorState();
		static_cast<FlowTestCase615Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UnitTestParameters params;
															#line 1412 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase615()
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase615Actor final : public Actor<Void>, public FastAllocated<FlowTestCase615Actor>, public FlowTestCase615ActorState<FlowTestCase615Actor> {
															#line 1417 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase615Actor>::operator new;
	using FastAllocated<FlowTestCase615Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10748698848553851904UL, 7116042191642251008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase615Actor(UnitTestParameters const& params) 
															#line 1433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase615ActorState<FlowTestCase615Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase615", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10861976754903805696UL, 1184790037130897152UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase615Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase615");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase615", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
static Future<Void> flowTestCase615( UnitTestParameters const& params ) {
															#line 615 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new FlowTestCase615Actor(params));
															#line 1465 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase615, "/ManagementAPI/ChangeConfig/TenantMode")

#line 634 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// unit test for changing encryption/tenant mode config options
															#line 1472 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase636()
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class FlowTestCase636Actor>
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase636ActorState {
															#line 1479 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase636ActorState(UnitTestParameters const& params) 
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : params(params)
															#line 1486 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase636", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase636ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase636", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 637 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::map<std::string, std::string> newConfig;
															#line 638 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string encryptModeKey = encryptionAtRestModeConfKey.toString();
															#line 639 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string tenantModeKey = tenantModeConfKey.toString();
															#line 640 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<TenantMode> tenantModes = { TenantMode::DISABLED, TenantMode::OPTIONAL_TENANT, TenantMode::REQUIRED };
															#line 641 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<EncryptionAtRestMode> encryptionModes = { EncryptionAtRestMode::DISABLED, EncryptionAtRestMode::CLUSTER_AWARE, EncryptionAtRestMode::DOMAIN_AWARE };
															#line 647 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DISABLED);
															#line 648 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 649 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::CLUSTER_AWARE);
															#line 653 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 654 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 657 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DOMAIN_AWARE);
															#line 658 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT);
															#line 660 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 661 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(TenantMode::REQUIRED);
															#line 662 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 665 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(encryptModeKey);
															#line 666 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 667 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 670 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(tenantModeKey);
															#line 671 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DOMAIN_AWARE);
															#line 672 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 673 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::CLUSTER_AWARE);
															#line 674 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 677 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration oldConfig;
															#line 680 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::DISABLED;
															#line 681 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 682 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 683 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 686 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::DOMAIN_AWARE;
															#line 687 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = TenantMode::REQUIRED;
															#line 688 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT);
															#line 690 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 691 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(TenantMode::REQUIRED);
															#line 692 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 695 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::CLUSTER_AWARE;
															#line 697 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 698 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 699 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 702 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(tenantModeKey);
															#line 703 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 704 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = deterministicRandom()->randomChoice(encryptionModes);
															#line 705 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 707 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<FlowTestCase636Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase636ActorState(); static_cast<FlowTestCase636Actor*>(this)->destroy(); return 0; }
															#line 1587 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<FlowTestCase636Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase636ActorState();
			static_cast<FlowTestCase636Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase636ActorState();
		static_cast<FlowTestCase636Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UnitTestParameters params;
															#line 1611 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase636()
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase636Actor final : public Actor<Void>, public FastAllocated<FlowTestCase636Actor>, public FlowTestCase636ActorState<FlowTestCase636Actor> {
															#line 1616 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase636Actor>::operator new;
	using FastAllocated<FlowTestCase636Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6290505063100698880UL, 6946386659940540416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase636Actor(UnitTestParameters const& params) 
															#line 1632 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase636ActorState<FlowTestCase636Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase636", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1188595504874592768UL, 5058876353428624896UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase636Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase636");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase636", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
static Future<Void> flowTestCase636( UnitTestParameters const& params ) {
															#line 636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new FlowTestCase636Actor(params));
															#line 1664 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase636, "/ManagementAPI/ChangeConfig/TenantAndEncryptMode")

#line 709 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 1670 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getDatabaseConfiguration()
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDatabaseConfigurationActor>
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActorState {
															#line 1677 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActorState(Transaction* const& tr,bool const& useSystemPriority) 
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   useSystemPriority(useSystemPriority)
															#line 1686 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	~GetDatabaseConfigurationActorState() 
	{
		fdb_probe_actor_destroy("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 711 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (useSystemPriority)
															#line 1701 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 712 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1705 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 714 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(configKeys, CLIENT_KNOBS->TOO_MANY);
															#line 716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state = 1;
															#line 716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*>(static_cast<GetDatabaseConfigurationActor*>(this)));
															#line 1720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& res,int loopDepth) 
	{
															#line 717 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(res.size() < CLIENT_KNOBS->TOO_MANY);
															#line 718 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		DatabaseConfiguration config;
															#line 719 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)res);
															#line 720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActorState(); static_cast<GetDatabaseConfigurationActor*>(this)->destroy(); return 0; }
															#line 1749 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult && res,int loopDepth) 
	{
															#line 717 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(res.size() < CLIENT_KNOBS->TOO_MANY);
															#line 718 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		DatabaseConfiguration config;
															#line 719 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)res);
															#line 720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActorState(); static_cast<GetDatabaseConfigurationActor*>(this)->destroy(); return 0; }
															#line 1767 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor*>(this)->ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool useSystemPriority;
															#line 1854 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDatabaseConfiguration()
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor final : public Actor<DatabaseConfiguration>, public ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >, public FastAllocated<GetDatabaseConfigurationActor>, public GetDatabaseConfigurationActorState<GetDatabaseConfigurationActor> {
															#line 1859 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDatabaseConfigurationActor>::operator new;
	using FastAllocated<GetDatabaseConfigurationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12820167535000224000UL, 11181981197254092288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<DatabaseConfiguration>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >;
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor(Transaction* const& tr,bool const& useSystemPriority) 
															#line 1876 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<DatabaseConfiguration>(),
		   GetDatabaseConfigurationActorState<GetDatabaseConfigurationActor>(tr, useSystemPriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17873082326995364352UL, 2518798019749376UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDatabaseConfiguration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<DatabaseConfiguration> getDatabaseConfiguration( Transaction* const& tr, bool const& useSystemPriority ) {
															#line 710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<DatabaseConfiguration>(new GetDatabaseConfigurationActor(tr, useSystemPriority));
															#line 1909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 722 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 1914 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getDatabaseConfiguration()
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDatabaseConfigurationActor1>
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor1State {
															#line 1921 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor1State(Database const& cx,bool const& useSystemPriority) 
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   useSystemPriority(useSystemPriority),
															#line 724 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 1932 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	~GetDatabaseConfigurationActor1State() 
	{
		fdb_probe_actor_destroy("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 725 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 1947 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(&tr, useSystemPriority);
															#line 727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1980 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 1;
															#line 727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*>(static_cast<GetDatabaseConfigurationActor1*>(this)));
															#line 1985 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 730 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 730 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2009 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 2;
															#line 730 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*>(static_cast<GetDatabaseConfigurationActor1*>(this)));
															#line 2014 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(DatabaseConfiguration const& config,int loopDepth) 
	{
															#line 728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor1*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActor1State(); static_cast<GetDatabaseConfigurationActor1*>(this)->destroy(); return 0; }
															#line 2029 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor1*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(DatabaseConfiguration && config,int loopDepth) 
	{
															#line 728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor1*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActor1State(); static_cast<GetDatabaseConfigurationActor1*>(this)->destroy(); return 0; }
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor1*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(DatabaseConfiguration const& config,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(config, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(DatabaseConfiguration && config,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(config), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor1*>(this)->ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor1*>(this)->ActorCallback< GetDatabaseConfigurationActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool useSystemPriority;
															#line 724 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 2217 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDatabaseConfiguration()
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor1 final : public Actor<DatabaseConfiguration>, public ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >, public ActorCallback< GetDatabaseConfigurationActor1, 1, Void >, public FastAllocated<GetDatabaseConfigurationActor1>, public GetDatabaseConfigurationActor1State<GetDatabaseConfigurationActor1> {
															#line 2222 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDatabaseConfigurationActor1>::operator new;
	using FastAllocated<GetDatabaseConfigurationActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12820167535000224000UL, 11181981197254092288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<DatabaseConfiguration>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >;
friend struct ActorCallback< GetDatabaseConfigurationActor1, 1, Void >;
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor1(Database const& cx,bool const& useSystemPriority) 
															#line 2240 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<DatabaseConfiguration>(),
		   GetDatabaseConfigurationActor1State<GetDatabaseConfigurationActor1>(cx, useSystemPriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1609963244257551360UL, 6418944901107183872UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDatabaseConfiguration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<DatabaseConfiguration> getDatabaseConfiguration( Database const& cx, bool const& useSystemPriority ) {
															#line 723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<DatabaseConfiguration>(new GetDatabaseConfigurationActor1(cx, useSystemPriority));
															#line 2274 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 734 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

ConfigureAutoResult parseConfig(StatusObject const& status) {
	ConfigureAutoResult result;
	StatusObjectReader statusObj(status);

	StatusObjectReader statusObjCluster;
	if (!statusObj.get("cluster", statusObjCluster))
		return ConfigureAutoResult();

	StatusObjectReader statusObjConfig;
	if (!statusObjCluster.get("configuration", statusObjConfig))
		return ConfigureAutoResult();

	if (!statusObjConfig.get("redundancy.factor", result.old_replication))
		return ConfigureAutoResult();

	result.auto_replication = result.old_replication;

	[[maybe_unused]] int storage_replication;
	int log_replication;
	if (result.old_replication == "single") {
		result.auto_replication = "double";
		storage_replication = 2;
		log_replication = 2;
	} else if (result.old_replication == "double" || result.old_replication == "fast_recovery_double") {
		storage_replication = 2;
		log_replication = 2;
	} else if (result.old_replication == "triple" || result.old_replication == "fast_recovery_triple") {
		storage_replication = 3;
		log_replication = 3;
	} else if (result.old_replication == "three_datacenter") {
		storage_replication = 6;
		log_replication = 4;
	} else if (result.old_replication == "three_datacenter_fallback") {
		storage_replication = 4;
		log_replication = 4;
	} else if (result.old_replication == "three_data_hall") {
		storage_replication = 3;
		log_replication = 4;
	} else if (result.old_replication == "three_data_hall_fallback") {
		storage_replication = 2;
		log_replication = 4;
	} else
		return ConfigureAutoResult();

	StatusObjectReader machinesMap;
	if (!statusObjCluster.get("machines", machinesMap))
		return ConfigureAutoResult();

	std::map<std::string, std::string> machineid_dcid;
	std::set<std::string> datacenters;
	int machineCount = 0;
	for (auto mach : machinesMap.obj()) {
		StatusObjectReader machine(mach.second);
		std::string dcId;
		if (machine.get("datacenter_id", dcId)) {
			machineid_dcid[mach.first] = dcId;
			datacenters.insert(dcId);
		}
		machineCount++;
	}

	result.machines = machineCount;

	if (datacenters.size() > 1)
		return ConfigureAutoResult();

	StatusObjectReader processesMap;
	if (!statusObjCluster.get("processes", processesMap))
		return ConfigureAutoResult();

	std::set<std::string> oldMachinesWithTransaction;
	int oldTransactionProcesses = 0;
	std::map<std::string, std::vector<std::pair<NetworkAddress, ProcessClass>>> machine_processes;
	int processCount = 0;
	for (auto proc : processesMap.obj()) {
		StatusObjectReader process(proc.second);
		if (!process.has("excluded") || !process.last().get_bool()) {
			std::string addrStr;
			if (!process.get("address", addrStr))
				return ConfigureAutoResult();
			std::string class_source;
			if (!process.get("class_source", class_source))
				return ConfigureAutoResult();
			std::string class_type;
			if (!process.get("class_type", class_type))
				return ConfigureAutoResult();
			std::string machineId;
			if (!process.get("machine_id", machineId))
				return ConfigureAutoResult();

			NetworkAddress addr = NetworkAddress::parse(addrStr);
			ProcessClass processClass(class_type, class_source);

			if (processClass.classType() == ProcessClass::TransactionClass ||
			    processClass.classType() == ProcessClass::LogClass) {
				oldMachinesWithTransaction.insert(machineId);
			}

			if (processClass.classType() == ProcessClass::TransactionClass ||
			    processClass.classType() == ProcessClass::CommitProxyClass ||
			    processClass.classType() == ProcessClass::GrvProxyClass ||
			    processClass.classType() == ProcessClass::ResolutionClass ||
			    processClass.classType() == ProcessClass::StatelessClass ||
			    processClass.classType() == ProcessClass::LogClass) {
				oldTransactionProcesses++;
			}

			if (processClass.classSource() == ProcessClass::AutoSource) {
				processClass = ProcessClass(ProcessClass::UnsetClass, ProcessClass::CommandLineSource);
				result.address_class[addr] = processClass;
			}

			if (processClass.classType() != ProcessClass::TesterClass) {
				machine_processes[machineId].emplace_back(addr, processClass);
				processCount++;
			}
		}
	}

	result.processes = processCount;
	result.old_processes_with_transaction = oldTransactionProcesses;
	result.old_machines_with_transaction = oldMachinesWithTransaction.size();

	std::map<std::pair<int, std::string>, std::vector<std::pair<NetworkAddress, ProcessClass>>> count_processes;
	for (auto& it : machine_processes) {
		count_processes[std::make_pair(it.second.size(), it.first)] = it.second;
	}

	std::set<std::string> machinesWithTransaction;
	std::set<std::string> machinesWithStorage;
	int totalTransactionProcesses = 0;
	int existingProxyCount = 0;
	int existingGrvProxyCount = 0;
	int existingResolverCount = 0;
	int existingStatelessCount = 0;
	for (auto& it : machine_processes) {
		for (auto& proc : it.second) {
			if (proc.second == ProcessClass::TransactionClass || proc.second == ProcessClass::LogClass) {
				totalTransactionProcesses++;
				machinesWithTransaction.insert(it.first);
			}
			if (proc.second == ProcessClass::StatelessClass) {
				existingStatelessCount++;
			}
			if (proc.second == ProcessClass::CommitProxyClass) {
				existingProxyCount++;
			}
			if (proc.second == ProcessClass::GrvProxyClass) {
				existingGrvProxyCount++;
			}
			if (proc.second == ProcessClass::ResolutionClass) {
				existingResolverCount++;
			}
			if (proc.second == ProcessClass::StorageClass) {
				machinesWithStorage.insert(it.first);
			}
			if (proc.second == ProcessClass::UnsetClass && proc.second.classSource() == ProcessClass::DBSource) {
				machinesWithStorage.insert(it.first);
			}
		}
	}

	if (processCount < 10)
		return ConfigureAutoResult();

	result.desired_resolvers = 1;
	int resolverCount;
	if (!statusObjConfig.get("resolvers", result.old_resolvers)) {
		result.old_resolvers = CLIENT_KNOBS->DEFAULT_AUTO_RESOLVERS;
		statusObjConfig.get("auto_resolvers", result.old_resolvers);
		result.auto_resolvers = result.desired_resolvers;
		resolverCount = result.auto_resolvers;
	} else {
		result.auto_resolvers = result.old_resolvers;
		resolverCount = result.old_resolvers;
	}

	result.desired_commit_proxies = std::max(std::min(12, processCount / 15), 1);
	int proxyCount;
	if (!statusObjConfig.get("commit_proxies", result.old_commit_proxies)) {
		result.old_commit_proxies = CLIENT_KNOBS->DEFAULT_AUTO_COMMIT_PROXIES;
		statusObjConfig.get("auto_commit_proxies", result.old_commit_proxies);
		result.auto_commit_proxies = result.desired_commit_proxies;
		proxyCount = result.auto_commit_proxies;
	} else {
		result.auto_commit_proxies = result.old_commit_proxies;
		proxyCount = result.old_commit_proxies;
	}

	result.desired_grv_proxies = std::max(std::min(4, processCount / 20), 1);
	int grvProxyCount;
	if (!statusObjConfig.get("grv_proxies", result.old_grv_proxies)) {
		result.old_grv_proxies = CLIENT_KNOBS->DEFAULT_AUTO_GRV_PROXIES;
		statusObjConfig.get("auto_grv_proxies", result.old_grv_proxies);
		result.auto_grv_proxies = result.desired_grv_proxies;
		grvProxyCount = result.auto_grv_proxies;
	} else {
		result.auto_grv_proxies = result.old_grv_proxies;
		grvProxyCount = result.old_grv_proxies;
	}

	result.desired_logs = std::min(12, processCount / 20);
	result.desired_logs = std::max(result.desired_logs, log_replication + 1);
	result.desired_logs = std::min<int>(result.desired_logs, machine_processes.size());
	int logCount;
	if (!statusObjConfig.get("logs", result.old_logs)) {
		result.old_logs = CLIENT_KNOBS->DEFAULT_AUTO_LOGS;
		statusObjConfig.get("auto_logs", result.old_logs);
		result.auto_logs = result.desired_logs;
		logCount = result.auto_logs;
	} else {
		result.auto_logs = result.old_logs;
		logCount = result.old_logs;
	}

	logCount = std::max(logCount, log_replication);

	totalTransactionProcesses += std::min(existingProxyCount, proxyCount);
	totalTransactionProcesses += std::min(existingGrvProxyCount, grvProxyCount);
	totalTransactionProcesses += std::min(existingResolverCount, resolverCount);
	totalTransactionProcesses += existingStatelessCount;

	// if one process on a machine is transaction class, make them all transaction class
	for (auto& it : count_processes) {
		if (machinesWithTransaction.count(it.first.second) && !machinesWithStorage.count(it.first.second)) {
			for (auto& proc : it.second) {
				if (proc.second == ProcessClass::UnsetClass &&
				    proc.second.classSource() == ProcessClass::CommandLineSource) {
					result.address_class[proc.first] =
					    ProcessClass(ProcessClass::TransactionClass, ProcessClass::AutoSource);
					totalTransactionProcesses++;
				}
			}
		}
	}

	int desiredTotalTransactionProcesses = logCount + resolverCount + proxyCount + grvProxyCount;

	// add machines with all transaction class until we have enough processes and enough machines
	for (auto& it : count_processes) {
		if (machinesWithTransaction.size() >= logCount && totalTransactionProcesses >= desiredTotalTransactionProcesses)
			break;

		if (!machinesWithTransaction.count(it.first.second) && !machinesWithStorage.count(it.first.second)) {
			for (auto& proc : it.second) {
				if (proc.second == ProcessClass::UnsetClass &&
				    proc.second.classSource() == ProcessClass::CommandLineSource) {
					ASSERT(proc.second != ProcessClass::TransactionClass);
					result.address_class[proc.first] =
					    ProcessClass(ProcessClass::TransactionClass, ProcessClass::AutoSource);
					totalTransactionProcesses++;
					machinesWithTransaction.insert(it.first.second);
				}
			}
		}
	}

	if (machinesWithTransaction.size() < logCount || totalTransactionProcesses < desiredTotalTransactionProcesses)
		return ConfigureAutoResult();

	result.auto_processes_with_transaction = totalTransactionProcesses;
	result.auto_machines_with_transaction = machinesWithTransaction.size();

	if (3 * totalTransactionProcesses > processCount)
		return ConfigureAutoResult();

	return result;
}

															#line 2548 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getWorkers()
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetWorkersActor>
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActorState {
															#line 2555 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActorState(Transaction* const& tr) 
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processClasses(tr->getRange(processClassKeys, CLIENT_KNOBS->TOO_MANY)),
															#line 1006 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processData(tr->getRange(workerListKeys, CLIENT_KNOBS->TOO_MANY))
															#line 2566 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	~GetWorkersActorState() 
	{
		fdb_probe_actor_destroy("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1008 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(processClasses) && success(processData);
															#line 1008 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2583 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetWorkersActor*>(this)->actor_wait_state = 1;
															#line 1008 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor, 0, Void >*>(static_cast<GetWorkersActor*>(this)));
															#line 2588 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1009 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processClasses.get().more && processClasses.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1010 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processData.get().more && processData.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1012 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<Optional<Standalone<StringRef>>, ProcessClass> id_class;
															#line 1013 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processClasses.get().size();i++) {
															#line 1014 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			id_class[decodeProcessClassKey(processClasses.get()[i].key)] = decodeProcessClassValue(processClasses.get()[i].value);
															#line 2619 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<ProcessData> results;
															#line 1020 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processData.get().size();i++) {
															#line 1021 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessData data = decodeWorkerListValue(processData.get()[i].value);
															#line 1022 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessClass processClass = id_class[data.locality.processId()];
															#line 1024 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (processClass.classSource() == ProcessClass::DBSource || data.processClass.classType() == ProcessClass::UnsetClass)
															#line 2631 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				data.processClass = processClass;
															#line 2635 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1028 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (data.processClass.classType() != ProcessClass::TesterClass)
															#line 2639 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1029 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				results.push_back(data);
															#line 2643 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1032 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(results); this->~GetWorkersActorState(); static_cast<GetWorkersActor*>(this)->destroy(); return 0; }
															#line 2648 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(results);
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1009 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processClasses.get().more && processClasses.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1010 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processData.get().more && processData.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1012 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<Optional<Standalone<StringRef>>, ProcessClass> id_class;
															#line 1013 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processClasses.get().size();i++) {
															#line 1014 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			id_class[decodeProcessClassKey(processClasses.get()[i].key)] = decodeProcessClassValue(processClasses.get()[i].value);
															#line 2668 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<ProcessData> results;
															#line 1020 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processData.get().size();i++) {
															#line 1021 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessData data = decodeWorkerListValue(processData.get()[i].value);
															#line 1022 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessClass processClass = id_class[data.locality.processId()];
															#line 1024 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (processClass.classSource() == ProcessClass::DBSource || data.processClass.classType() == ProcessClass::UnsetClass)
															#line 2680 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				data.processClass = processClass;
															#line 2684 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1028 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (data.processClass.classType() != ProcessClass::TesterClass)
															#line 2688 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1029 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				results.push_back(data);
															#line 2692 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1032 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(results); this->~GetWorkersActorState(); static_cast<GetWorkersActor*>(this)->destroy(); return 0; }
															#line 2697 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(results);
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetWorkersActor*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor*>(this)->ActorCallback< GetWorkersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetWorkersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<RangeResult> processClasses;
															#line 1006 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<RangeResult> processData;
															#line 2786 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getWorkers()
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor final : public Actor<std::vector<ProcessData>>, public ActorCallback< GetWorkersActor, 0, Void >, public FastAllocated<GetWorkersActor>, public GetWorkersActorState<GetWorkersActor> {
															#line 2791 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetWorkersActor>::operator new;
	using FastAllocated<GetWorkersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16876621398767215360UL, 4669103688462601216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<ProcessData>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetWorkersActor, 0, Void >;
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor(Transaction* const& tr) 
															#line 2808 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<ProcessData>>(),
		   GetWorkersActorState<GetWorkersActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14779239272732970240UL, 16642038601798370048UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getWorkers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetWorkersActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<ProcessData>> getWorkers( Transaction* const& tr ) {
															#line 1004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<ProcessData>>(new GetWorkersActor(tr));
															#line 2841 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1034 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 2846 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getWorkers()
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetWorkersActor1>
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor1State {
															#line 2853 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor1State(Database const& cx) 
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1036 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 2862 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	~GetWorkersActor1State() 
	{
		fdb_probe_actor_destroy("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1037 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 2877 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1039 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<ProcessData>> __when_expr_0 = getWorkers(&tr);
															#line 1042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2916 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetWorkersActor1*>(this)->actor_wait_state = 1;
															#line 1042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*>(static_cast<GetWorkersActor1*>(this)));
															#line 2921 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1045 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1045 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2945 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetWorkersActor1*>(this)->actor_wait_state = 2;
															#line 1045 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor1, 1, Void >*>(static_cast<GetWorkersActor1*>(this)));
															#line 2950 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> const& workers,int loopDepth) 
	{
															#line 1043 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor1*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(workers); this->~GetWorkersActor1State(); static_cast<GetWorkersActor1*>(this)->destroy(); return 0; }
															#line 2965 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor1*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(workers);
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> && workers,int loopDepth) 
	{
															#line 1043 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor1*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(workers); this->~GetWorkersActor1State(); static_cast<GetWorkersActor1*>(this)->destroy(); return 0; }
															#line 2977 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor1*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(workers);
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(workers, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> && workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetWorkersActor1*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor1*>(this)->ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetWorkersActor1*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor1*>(this)->ActorCallback< GetWorkersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetWorkersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1036 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 3151 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getWorkers()
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor1 final : public Actor<std::vector<ProcessData>>, public ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >, public ActorCallback< GetWorkersActor1, 1, Void >, public FastAllocated<GetWorkersActor1>, public GetWorkersActor1State<GetWorkersActor1> {
															#line 3156 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetWorkersActor1>::operator new;
	using FastAllocated<GetWorkersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16876621398767215360UL, 4669103688462601216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<ProcessData>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >;
friend struct ActorCallback< GetWorkersActor1, 1, Void >;
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor1(Database const& cx) 
															#line 3174 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<ProcessData>>(),
		   GetWorkersActor1State<GetWorkersActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12240367636695030272UL, 7610739061935181824UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getWorkers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetWorkersActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<ProcessData>> getWorkers( Database const& cx ) {
															#line 1035 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<ProcessData>>(new GetWorkersActor1(cx));
															#line 3208 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1049 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 3213 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getConnectionString()
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetConnectionStringActor>
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetConnectionStringActorState {
															#line 3220 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetConnectionStringActorState(Database const& cx) 
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1051 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 3229 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getConnectionString", reinterpret_cast<unsigned long>(this));

	}
	~GetConnectionStringActorState() 
	{
		fdb_probe_actor_destroy("getConnectionString", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1052 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 3244 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 1055 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1056 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(coordinatorsKey);
															#line 1056 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 1;
															#line 1056 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*>(static_cast<GetConnectionStringActor*>(this)));
															#line 3286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1061 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1061 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3310 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 2;
															#line 1061 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetConnectionStringActor, 1, Void >*>(static_cast<GetConnectionStringActor*>(this)));
															#line 3315 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& currentKey,int loopDepth) 
	{
															#line 1057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3330 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1058 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3334 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetConnectionStringActorState();
			static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(ClusterConnectionString(currentKey.get().toString())); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3342 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(ClusterConnectionString(currentKey.get().toString()));
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && currentKey,int loopDepth) 
	{
															#line 1057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3354 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1058 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3358 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetConnectionStringActorState();
			static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(ClusterConnectionString(currentKey.get().toString())); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3366 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(ClusterConnectionString(currentKey.get().toString()));
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& currentKey,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(currentKey, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && currentKey,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(currentKey), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state > 0) static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 0;
		static_cast<GetConnectionStringActor*>(this)->ActorCallback< GetConnectionStringActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state > 0) static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 0;
		static_cast<GetConnectionStringActor*>(this)->ActorCallback< GetConnectionStringActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetConnectionStringActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1051 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 3540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getConnectionString()
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetConnectionStringActor final : public Actor<Optional<ClusterConnectionString>>, public ActorCallback< GetConnectionStringActor, 0, Optional<Value> >, public ActorCallback< GetConnectionStringActor, 1, Void >, public FastAllocated<GetConnectionStringActor>, public GetConnectionStringActorState<GetConnectionStringActor> {
															#line 3545 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetConnectionStringActor>::operator new;
	using FastAllocated<GetConnectionStringActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6085311643441436416UL, 17450864791914820096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<ClusterConnectionString>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetConnectionStringActor, 0, Optional<Value> >;
friend struct ActorCallback< GetConnectionStringActor, 1, Void >;
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetConnectionStringActor(Database const& cx) 
															#line 3563 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<ClusterConnectionString>>(),
		   GetConnectionStringActorState<GetConnectionStringActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15280877619308916480UL, 17895028133604852992UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getConnectionString");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetConnectionStringActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<ClusterConnectionString>> getConnectionString( Database const& cx ) {
															#line 1050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<ClusterConnectionString>>(new GetConnectionStringActor(cx));
															#line 3597 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1065 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

static std::vector<std::string> connectionStrings;

namespace {

															#line 3606 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getClusterConnectionStringFromStorageServer()
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetClusterConnectionStringFromStorageServerActor>
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetClusterConnectionStringFromStorageServerActorState {
															#line 3612 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetClusterConnectionStringFromStorageServerActorState(Transaction* const& tr) 
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 3619 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this));

	}
	~GetClusterConnectionStringFromStorageServerActorState() 
	{
		fdb_probe_actor_destroy("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1072 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1074 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1075 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1077 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			retryTimes = 0;
															#line 1078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 3644 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetClusterConnectionStringFromStorageServerActorState();
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1079 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (retryTimes >= CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_TIMES)
															#line 3674 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1080 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetClusterConnectionStringFromStorageServerActorState(); static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->destroy(); return 0; }
															#line 3678 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetClusterConnectionStringFromStorageServerActorState();
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_0 = tr->get(coordinatorsKey);
															#line 1083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3688 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 1;
															#line 1083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3693 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1084 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated() && currentKey.present())
															#line 3702 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1092 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int n = connectionStrings.size() > 0 ? connectionStrings.size() - 1 : 0;
															#line 1093 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < n;++i) {
															#line 1094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ASSERT(currentKey.get() != connectionStrings.at(i));
															#line 3710 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1098 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1100 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			++retryTimes;
															#line 1101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_DELAY);
															#line 1101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 2;
															#line 1101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& __currentKey,int loopDepth) 
	{
															#line 1083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		currentKey = __currentKey;
															#line 3742 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && __currentKey,int loopDepth) 
	{
		currentKey = std::move(__currentKey);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1105 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionString = ClusterConnectionString(currentKey.get().toString());
															#line 1106 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (tr->getDatabase()->getConnectionRecord() && clusterConnectionString.clusterKeyName().toString() != tr->getDatabase()->getConnectionRecord()->getConnectionString().clusterKeyName())
															#line 3823 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			++retryTimes;
															#line 1111 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_DELAY);
															#line 1111 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3831 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 3;
															#line 1111 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3836 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
															#line 1115 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(clusterConnectionString); this->~GetClusterConnectionStringFromStorageServerActorState(); static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->destroy(); return 0; }
															#line 3937 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(std::move(clusterConnectionString)); // state_var_RVO
		this->~GetClusterConnectionStringFromStorageServerActorState();
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1077 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int retryTimes;
															#line 1083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<Value> currentKey;
															#line 1105 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString clusterConnectionString;
															#line 4040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getClusterConnectionStringFromStorageServer()
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetClusterConnectionStringFromStorageServerActor final : public Actor<Optional<ClusterConnectionString>>, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >, public FastAllocated<GetClusterConnectionStringFromStorageServerActor>, public GetClusterConnectionStringFromStorageServerActorState<GetClusterConnectionStringFromStorageServerActor> {
															#line 4045 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetClusterConnectionStringFromStorageServerActor>::operator new;
	using FastAllocated<GetClusterConnectionStringFromStorageServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12662592304083599104UL, 5074462803148530432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<ClusterConnectionString>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >;
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >;
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >;
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetClusterConnectionStringFromStorageServerActor(Transaction* const& tr) 
															#line 4064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<ClusterConnectionString>>(),
		   GetClusterConnectionStringFromStorageServerActorState<GetClusterConnectionStringFromStorageServerActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(625065204163832320UL, 1122260541233559808UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getClusterConnectionStringFromStorageServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<ClusterConnectionString>> getClusterConnectionStringFromStorageServer( Transaction* const& tr ) {
															#line 1070 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<ClusterConnectionString>>(new GetClusterConnectionStringFromStorageServerActor(tr));
															#line 4098 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1118 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 4103 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via verifyConfigurationDatabaseAlive()
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class VerifyConfigurationDatabaseAliveActor>
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class VerifyConfigurationDatabaseAliveActorState {
															#line 4109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	VerifyConfigurationDatabaseAliveActorState(Database const& cx) 
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1120 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   backoff(),
															#line 1121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   configTr()
															#line 4120 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this));

	}
	~VerifyConfigurationDatabaseAliveActorState() 
	{
		fdb_probe_actor_destroy("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 4135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1126 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			configTr = ISingleThreadTransaction::create(ISingleThreadTransaction::Type::PAXOS_CONFIG, cx);
															#line 1127 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			Tuple tuple;
															#line 1128 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tuple.appendNull();
															#line 1129 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tuple << "test"_sr;
															#line 1130 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = configTr->get(tuple.pack());
															#line 1130 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4176 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 1;
															#line 1130 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4181 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1134 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerNewCoordinatorsError").error(e);
															#line 1135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (e.code() == error_code_coordinators_changed)
															#line 4205 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_1 = backoff.onError();
															#line 1136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4211 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
				static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 2;
															#line 1136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4216 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1139 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = configTr->onError(e);
															#line 1139 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4225 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when2(__when_expr_2.get(), loopDepth); };
				static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 3;
															#line 1139 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4230 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& serializedValue,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerNewCoordinatorsOnline").log();
															#line 1132 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VerifyConfigurationDatabaseAliveActorState(); static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->destroy(); return 0; }
															#line 4248 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && serializedValue,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerNewCoordinatorsOnline").log();
															#line 1132 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VerifyConfigurationDatabaseAliveActorState(); static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->destroy(); return 0; }
															#line 4262 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& serializedValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(serializedValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && serializedValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(serializedValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void const& _,int loopDepth) 
	{
															#line 1137 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		configTr->reset();
															#line 4355 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void && _,int loopDepth) 
	{
															#line 1137 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		configTr->reset();
															#line 4364 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1120 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Backoff backoff;
															#line 1121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<ISingleThreadTransaction> configTr;
															#line 4537 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via verifyConfigurationDatabaseAlive()
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class VerifyConfigurationDatabaseAliveActor final : public Actor<Void>, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >, public FastAllocated<VerifyConfigurationDatabaseAliveActor>, public VerifyConfigurationDatabaseAliveActorState<VerifyConfigurationDatabaseAliveActor> {
															#line 4542 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<VerifyConfigurationDatabaseAliveActor>::operator new;
	using FastAllocated<VerifyConfigurationDatabaseAliveActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(105873059625945344UL, 5924044052584710656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >;
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >;
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >;
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	VerifyConfigurationDatabaseAliveActor(Database const& cx) 
															#line 4561 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   VerifyConfigurationDatabaseAliveActorState<VerifyConfigurationDatabaseAliveActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3250116751694520064UL, 7074429358866028800UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("verifyConfigurationDatabaseAlive");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> verifyConfigurationDatabaseAlive( Database const& cx ) {
															#line 1119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new VerifyConfigurationDatabaseAliveActor(cx));
															#line 4595 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 4600 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via resetPreviousCoordinatorsKey()
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ResetPreviousCoordinatorsKeyActor>
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ResetPreviousCoordinatorsKeyActorState {
															#line 4606 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ResetPreviousCoordinatorsKeyActorState(Database const& cx) 
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 4613 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this));

	}
	~ResetPreviousCoordinatorsKeyActorState() 
	{
		fdb_probe_actor_destroy("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1146 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 4628 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1152 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clearTr = ISingleThreadTransaction::create(ISingleThreadTransaction::Type::RYW, cx);
															#line 4658 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		try {
															#line 1155 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			clearTr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1156 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			clearTr->clear(previousCoordinatorsKey);
															#line 1157 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = clearTr->commit();
															#line 1157 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4668 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 1;
															#line 1157 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*>(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)));
															#line 4673 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e2,int loopDepth=0) 
	{
		try {
															#line 1160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = clearTr->onError(e2);
															#line 1160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4697 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 2;
															#line 1160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*>(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)));
															#line 4702 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1158 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResetPreviousCoordinatorsKeyActorState(); static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->destroy(); return 0; }
															#line 4717 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1158 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResetPreviousCoordinatorsKeyActorState(); static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->destroy(); return 0; }
															#line 4729 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state > 0) static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state > 0) static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1152 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<ISingleThreadTransaction> clearTr;
															#line 4903 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via resetPreviousCoordinatorsKey()
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ResetPreviousCoordinatorsKeyActor final : public Actor<Void>, public ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >, public ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >, public FastAllocated<ResetPreviousCoordinatorsKeyActor>, public ResetPreviousCoordinatorsKeyActorState<ResetPreviousCoordinatorsKeyActor> {
															#line 4908 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ResetPreviousCoordinatorsKeyActor>::operator new;
	using FastAllocated<ResetPreviousCoordinatorsKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6022966134761045504UL, 5240748600984227072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >;
friend struct ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >;
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ResetPreviousCoordinatorsKeyActor(Database const& cx) 
															#line 4926 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ResetPreviousCoordinatorsKeyActorState<ResetPreviousCoordinatorsKeyActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2328696218968888064UL, 1221210460215735552UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resetPreviousCoordinatorsKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> resetPreviousCoordinatorsKey( Database const& cx ) {
															#line 1145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ResetPreviousCoordinatorsKeyActor(cx));
															#line 4959 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1164 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

} // namespace

															#line 4966 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via changeQuorumChecker()
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ChangeQuorumCheckerActor>
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumCheckerActorState {
															#line 4973 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumCheckerActorState(Transaction* const& tr,ClusterConnectionString* const& conn,std::string const& newName,bool const& disableConfigDB) 
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   conn(conn),
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   newName(newName),
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   disableConfigDB(disableConfigDB)
															#line 4986 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("changeQuorumChecker", reinterpret_cast<unsigned long>(this));

	}
	~ChangeQuorumCheckerActorState() 
	{
		fdb_probe_actor_destroy("changeQuorumChecker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1171 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerStart").detail("NewConnectionString", conn->toString());
															#line 1172 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_0 = getClusterConnectionStringFromStorageServer(tr);
															#line 1172 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 1;
															#line 1172 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5010 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1175 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clusterConnectionStringOptional.present())
															#line 5033 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1176 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5037 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumCheckerActorState();
			static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1180 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		old = clusterConnectionStringOptional.get();
															#line 1182 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (conn->hostnames.size() + conn->coords.size() == 0)
															#line 5047 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1183 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			conn->hostnames = old.hostnames;
															#line 1184 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			conn->coords = old.coords;
															#line 5053 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<NetworkAddress>> __when_expr_1 = conn->tryResolveHostnames();
															#line 1186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 2;
															#line 1186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<ClusterConnectionString> const& __clusterConnectionStringOptional,int loopDepth) 
	{
															#line 1172 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionStringOptional = __clusterConnectionStringOptional;
															#line 5073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<ClusterConnectionString> && __clusterConnectionStringOptional,int loopDepth) 
	{
		clusterConnectionStringOptional = std::move(__clusterConnectionStringOptional);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1187 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (desiredCoordinators.size() != conn->hostnames.size() + conn->coords.size())
															#line 5152 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1188 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerEarlyTermination") .detail("Reason", "One or more hostnames are unresolvable") .backtrace();
															#line 1191 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5158 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
			this->~ChangeQuorumCheckerActorState();
			static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1194 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (newName.empty())
															#line 5166 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1195 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newName = old.clusterKeyName().toString();
															#line 5170 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1197 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(conn->hostnames.begin(), conn->hostnames.end());
															#line 1198 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(conn->coords.begin(), conn->coords.end());
															#line 1199 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(old.hostnames.begin(), old.hostnames.end());
															#line 1200 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(old.coords.begin(), old.coords.end());
															#line 1201 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (conn->hostnames == old.hostnames && conn->coords == old.coords && old.clusterKeyName() == newName)
															#line 5182 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1202 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			connectionStrings.clear();
															#line 1203 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (g_network->isSimulated() && g_simulator->configDBType == ConfigDBType::DISABLED)
															#line 5188 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1204 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				disableConfigDB = true;
															#line 5192 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1206 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!disableConfigDB)
															#line 5196 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1207 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = verifyConfigurationDatabaseAlive(tr->getDatabase());
															#line 1207 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5202 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 3;
															#line 1207 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 2, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5207 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<NetworkAddress> const& __desiredCoordinators,int loopDepth) 
	{
															#line 1186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = __desiredCoordinators;
															#line 5226 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<NetworkAddress> && __desiredCoordinators,int loopDepth) 
	{
		desiredCoordinators = std::move(__desiredCoordinators);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 1221 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		conn->parseKey(newName + ':' + deterministicRandom()->randomAlphaNumeric(32));
															#line 1222 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		connectionStrings.push_back(conn->toString());
															#line 1224 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated())
															#line 5309 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1225 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int i = 0;
															#line 1226 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int protectedCount = 0;
															#line 1227 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int minimumCoordinators = (desiredCoordinators.size() / 2) + 1;
															#line 1228 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(;protectedCount < minimumCoordinators && i < desiredCoordinators.size();) {
															#line 1229 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto process = g_simulator->getProcessByAddress(desiredCoordinators[i]);
															#line 1230 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto addresses = process->addresses;
															#line 1232 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!process->isReliable())
															#line 5325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1233 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					i++;
															#line 5329 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					continue;
				}
															#line 1237 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				g_simulator->protectedAddresses.insert(process->addresses.address);
															#line 1238 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (addresses.secondaryAddress.present())
															#line 5336 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1239 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					g_simulator->protectedAddresses.insert(process->addresses.secondaryAddress.get());
															#line 5340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 1241 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("ProtectCoordinator").detail("Address", desiredCoordinators[i]).backtrace();
															#line 1242 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				protectedCount++;
															#line 1243 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				i++;
															#line 5348 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1246 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (protectedCount < minimumCoordinators)
															#line 5352 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1247 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("NotEnoughReliableCoordinators") .detail("NumReliable", protectedCount) .detail("MinimumRequired", minimumCoordinators) .detail("ConnectionString", conn->toString());
															#line 1252 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5358 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
				this->~ChangeQuorumCheckerActorState();
				static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1256 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1257 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ClientCoordinators coord(Reference<ClusterConnectionMemoryRecord>(new ClusterConnectionMemoryRecord(*conn)));
															#line 1259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1260 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1261 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (coord.clientLeaderServers[i].hostname.present())
															#line 5375 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1262 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				leaderServers.push_back(retryGetReplyFromHostname(GetLeaderRequest(coord.clusterKey, UID()), coord.clientLeaderServers[i].hostname.get(), WLTOKEN_CLIENTLEADERREG_GETLEADER, TaskPriority::CoordinationReply));
															#line 5379 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1267 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 5385 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1274 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = waitForAll(leaderServers);
															#line 1273 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5392 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
															#line 1275 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(5.0);
															#line 5396 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont5when2(__when_expr_6.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 6;
															#line 1274 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 5, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 1275 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 6, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5403 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 1209 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (BUGGIFY_WITH_PROB(0.1))
															#line 5412 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1215 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(deterministicRandom()->random01() * 10);
															#line 1215 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5418 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont8when1(__when_expr_3.get(), loopDepth); };
			static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 4;
															#line 1215 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 3, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5423 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 1217 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_4 = resetPreviousCoordinatorsKey(tr->getDatabase());
															#line 1217 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5526 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont11when1(__when_expr_4.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 5;
															#line 1217 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 4, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11cont1(Void const& _,int loopDepth) 
	{
															#line 1218 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5627 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::SAME_NETWORK_ADDRESSES);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont1(Void && _,int loopDepth) 
	{
															#line 1218 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5639 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::SAME_NETWORK_ADDRESSES);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont13(int loopDepth) 
	{
															#line 1279 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerSetCoordinatorsKey") .detail("CurrentCoordinators", old.toString()) .detail("NewCoordinators", conn->toString());
															#line 1282 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->set(coordinatorsKey, conn->toString());
															#line 1283 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(Optional<CoordinatorsResult>()); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5730 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(Optional<CoordinatorsResult>());
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when2(Void const& _,int loopDepth) 
	{
															#line 1276 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5754 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when2(Void && _,int loopDepth) 
	{
															#line 1276 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5766 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 5, Void >::remove();
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString* conn;
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string newName;
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool disableConfigDB;
															#line 1172 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<ClusterConnectionString> clusterConnectionStringOptional;
															#line 1180 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString old;
															#line 1186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> desiredCoordinators;
															#line 5909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via changeQuorumChecker()
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumCheckerActor final : public Actor<Optional<CoordinatorsResult>>, public ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >, public ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumCheckerActor, 2, Void >, public ActorCallback< ChangeQuorumCheckerActor, 3, Void >, public ActorCallback< ChangeQuorumCheckerActor, 4, Void >, public ActorCallback< ChangeQuorumCheckerActor, 5, Void >, public ActorCallback< ChangeQuorumCheckerActor, 6, Void >, public FastAllocated<ChangeQuorumCheckerActor>, public ChangeQuorumCheckerActorState<ChangeQuorumCheckerActor> {
															#line 5914 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ChangeQuorumCheckerActor>::operator new;
	using FastAllocated<ChangeQuorumCheckerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9654196069663385344UL, 11592847317697863936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<CoordinatorsResult>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 2, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 3, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 4, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 5, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 6, Void >;
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumCheckerActor(Transaction* const& tr,ClusterConnectionString* const& conn,std::string const& newName,bool const& disableConfigDB) 
															#line 5937 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<CoordinatorsResult>>(),
		   ChangeQuorumCheckerActorState<ChangeQuorumCheckerActor>(tr, conn, newName, disableConfigDB),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2226565757528493312UL, 6695731912205238272UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeQuorumChecker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<CoordinatorsResult>> changeQuorumChecker( Transaction* const& tr, ClusterConnectionString* const& conn, std::string const& newName, bool const& disableConfigDB ) {
															#line 1167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<CoordinatorsResult>>(new ChangeQuorumCheckerActor(tr, conn, newName, disableConfigDB));
															#line 5975 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1285 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 5980 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via changeQuorum()
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ChangeQuorumActor>
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumActorState {
															#line 5987 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumActorState(Database const& cx,Reference<IQuorumChange> const& change) 
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   change(change),
															#line 1287 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx),
															#line 1288 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   retries(0),
															#line 1289 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCoordinators(),
															#line 1290 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   notEnoughMachineResults(0)
															#line 6004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("changeQuorum", reinterpret_cast<unsigned long>(this));

	}
	~ChangeQuorumActorState() 
	{
		fdb_probe_actor_destroy("changeQuorum", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1292 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 6019 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_0 = getClusterConnectionStringFromStorageServer(&tr);
															#line 1294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6052 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 1;
															#line 1294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1389 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("RetryQuorumChange").error(e).detail("Retries", retries);
															#line 1390 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 1390 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 8;
															#line 1390 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 8, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6088 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1297 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clusterConnectionStringOptional.present())
															#line 6103 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1298 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6107 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldClusterConnectionString = clusterConnectionStringOptional.get();
															#line 1303 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldClusterKeyName = oldClusterConnectionString.clusterKeyName();
															#line 1305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<NetworkAddress>> __when_expr_1 = oldClusterConnectionString.tryResolveHostnames();
															#line 1305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 2;
															#line 1305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6126 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<ClusterConnectionString> const& __clusterConnectionStringOptional,int loopDepth) 
	{
															#line 1294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionStringOptional = __clusterConnectionStringOptional;
															#line 6135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<ClusterConnectionString> && __clusterConnectionStringOptional,int loopDepth) 
	{
		clusterConnectionStringOptional = std::move(__clusterConnectionStringOptional);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1306 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		result = CoordinatorsResult::SUCCESS;
															#line 1307 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!desiredCoordinators.size())
															#line 6216 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<NetworkAddress>> __when_expr_2 = change->getDesiredCoordinators(&tr, oldCoordinators, Reference<ClusterConnectionMemoryRecord>( new ClusterConnectionMemoryRecord(oldClusterConnectionString)), result);
															#line 1308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6222 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 3;
															#line 1308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6227 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::vector<NetworkAddress> const& __oldCoordinators,int loopDepth) 
	{
															#line 1305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldCoordinators = __oldCoordinators;
															#line 6241 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::vector<NetworkAddress> && __oldCoordinators,int loopDepth) 
	{
		oldCoordinators = std::move(__oldCoordinators);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 1317 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (result == CoordinatorsResult::NOT_ENOUGH_MACHINES && notEnoughMachineResults < 1)
															#line 6320 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1320 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			notEnoughMachineResults++;
															#line 1321 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(1.0);
															#line 1321 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6328 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 4;
															#line 1321 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 3, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6333 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(std::vector<NetworkAddress> const& _desiredCoordinators,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = _desiredCoordinators;
															#line 6347 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(std::vector<NetworkAddress> && _desiredCoordinators,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = _desiredCoordinators;
															#line 6356 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(std::vector<NetworkAddress> const& _desiredCoordinators,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_desiredCoordinators, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(std::vector<NetworkAddress> && _desiredCoordinators,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_desiredCoordinators), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
															#line 1325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (result != CoordinatorsResult::SUCCESS)
															#line 6440 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1326 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(result); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6444 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(std::move(result)); // state_var_RVO
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1327 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!desiredCoordinators.size())
															#line 6452 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1328 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::INVALID_NETWORK_ADDRESSES); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6456 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::INVALID_NETWORK_ADDRESSES);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1329 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(desiredCoordinators.begin(), desiredCoordinators.end());
															#line 1331 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::string newName = change->getDesiredClusterKeyName();
															#line 1332 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (newName.empty())
															#line 6468 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1333 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newName = oldClusterKeyName.toString();
															#line 6472 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1335 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldCoordinators == desiredCoordinators && oldClusterKeyName == newName)
															#line 6476 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1336 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(retries ? CoordinatorsResult::SUCCESS : CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6480 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(retries ? CoordinatorsResult::SUCCESS : CoordinatorsResult::SAME_NETWORK_ADDRESSES);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1338 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		newClusterConnectionString = ClusterConnectionString(desiredCoordinators, StringRef(newName + ':' + deterministicRandom()->randomAlphaNumeric(32)));
															#line 1340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		newClusterKeyName = newClusterConnectionString.clusterKeyName();
															#line 1342 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated())
															#line 6492 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1343 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < (desiredCoordinators.size() / 2) + 1;i++) {
															#line 1344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto process = g_simulator->getProcessByAddress(desiredCoordinators[i]);
															#line 1345 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ASSERT(process->isReliable() || process->rebooting);
															#line 1347 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				g_simulator->protectedAddresses.insert(process->addresses.address);
															#line 1348 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (process->addresses.secondaryAddress.present())
															#line 6504 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1349 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					g_simulator->protectedAddresses.insert(process->addresses.secondaryAddress.get());
															#line 6508 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 1351 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("ProtectCoordinator").detail("Address", desiredCoordinators[i]).backtrace();
															#line 6512 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1355 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("AttemptingQuorumChange") .detail("FromCS", oldClusterConnectionString.toString()) .detail("ToCS", newClusterConnectionString.toString());
															#line 1358 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		CODE_PROBE(oldClusterKeyName != newClusterKeyName, "Quorum change with new name");
															#line 1359 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		CODE_PROBE(oldClusterKeyName == newClusterKeyName, "Quorum change with unchanged name");
															#line 1361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers = std::vector<Future<Optional<LeaderInfo>>>();
															#line 1362 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		coord = ClientCoordinators(Reference<ClusterConnectionMemoryRecord>( new ClusterConnectionMemoryRecord(newClusterConnectionString)));
															#line 1365 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!change->getDesiredClusterKeyName().empty())
															#line 6527 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1366 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<CheckDescriptorMutableReply> __when_expr_4 = coord.clientLeaderServers[0].checkDescriptorMutable.getReply(CheckDescriptorMutableRequest());
															#line 1366 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6533 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_4.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 5;
															#line 1366 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6538 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 1322 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 6552 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 1322 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 6561 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
															#line 1372 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1373 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1374 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 6649 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = waitForAll(leaderServers);
															#line 1377 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6655 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont11when1(__when_expr_5.get(), loopDepth); };
															#line 1379 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(5.0);
															#line 6659 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont11when2(__when_expr_6.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 6;
															#line 1378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 5, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 1379 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 6, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6666 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont19(CheckDescriptorMutableReply const& mutabilityReply,int loopDepth) 
	{
															#line 1368 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!mutabilityReply.isMutable)
															#line 6675 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1369 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6679 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont19(CheckDescriptorMutableReply && mutabilityReply,int loopDepth) 
	{
															#line 1368 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!mutabilityReply.isMutable)
															#line 6693 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1369 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6697 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(CheckDescriptorMutableReply const& mutabilityReply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont19(mutabilityReply, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(CheckDescriptorMutableReply && mutabilityReply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont19(std::move(mutabilityReply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,CheckDescriptorMutableReply const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,CheckDescriptorMutableReply && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont11cont1(int loopDepth) 
	{
															#line 1384 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(coordinatorsKey, newClusterConnectionString.toString());
															#line 1386 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr.commit();
															#line 1386 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6790 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont11cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 7;
															#line 1386 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 7, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6795 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11when2(Void const& _,int loopDepth) 
	{
															#line 1380 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6816 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11when2(Void && _,int loopDepth) 
	{
															#line 1380 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6828 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 5, Void >::remove();
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont11cont3(Void const& _,int loopDepth) 
	{
															#line 1387 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(false);
															#line 6961 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont3(Void && _,int loopDepth) 
	{
															#line 1387 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(false);
															#line 6970 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont11cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 1391 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		++retries;
															#line 7067 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 1391 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		++retries;
															#line 7076 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IQuorumChange> change;
															#line 1287 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 1288 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int retries;
															#line 1289 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> desiredCoordinators;
															#line 1290 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int notEnoughMachineResults;
															#line 1294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<ClusterConnectionString> clusterConnectionStringOptional;
															#line 1302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString oldClusterConnectionString;
															#line 1303 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Key oldClusterKeyName;
															#line 1305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1306 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CoordinatorsResult result;
															#line 1338 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString newClusterConnectionString;
															#line 1340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Key newClusterKeyName;
															#line 1361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1362 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClientCoordinators coord;
															#line 7186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via changeQuorum()
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumActor final : public Actor<CoordinatorsResult>, public ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >, public ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumActor, 3, Void >, public ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >, public ActorCallback< ChangeQuorumActor, 5, Void >, public ActorCallback< ChangeQuorumActor, 6, Void >, public ActorCallback< ChangeQuorumActor, 7, Void >, public ActorCallback< ChangeQuorumActor, 8, Void >, public FastAllocated<ChangeQuorumActor>, public ChangeQuorumActorState<ChangeQuorumActor> {
															#line 7191 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ChangeQuorumActor>::operator new;
	using FastAllocated<ChangeQuorumActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3947960886789963008UL, 9901942826630514688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<CoordinatorsResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >;
friend struct ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumActor, 3, Void >;
friend struct ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >;
friend struct ActorCallback< ChangeQuorumActor, 5, Void >;
friend struct ActorCallback< ChangeQuorumActor, 6, Void >;
friend struct ActorCallback< ChangeQuorumActor, 7, Void >;
friend struct ActorCallback< ChangeQuorumActor, 8, Void >;
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumActor(Database const& cx,Reference<IQuorumChange> const& change) 
															#line 7216 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<CoordinatorsResult>(),
		   ChangeQuorumActorState<ChangeQuorumActor>(cx, change),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11745850326369364480UL, 4959262309748762880UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeQuorum");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ChangeQuorumActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ChangeQuorumActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ChangeQuorumActor, 7, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< ChangeQuorumActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<CoordinatorsResult> changeQuorum( Database const& cx, Reference<IQuorumChange> const& change ) {
															#line 1286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<CoordinatorsResult>(new ChangeQuorumActor(cx, change));
															#line 7256 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1395 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

struct NameQuorumChange final : IQuorumChange {
	std::string newName;
	Reference<IQuorumChange> otherChange;
	explicit NameQuorumChange(std::string const& newName, Reference<IQuorumChange> const& otherChange)
	  : newName(newName), otherChange(otherChange) {}
	Future<std::vector<NetworkAddress>> getDesiredCoordinators(Transaction* tr,
	                                                           std::vector<NetworkAddress> oldCoordinators,
	                                                           Reference<IClusterConnectionRecord> ccr,
	                                                           CoordinatorsResult& t) override {
		return otherChange->getDesiredCoordinators(tr, oldCoordinators, ccr, t);
	}
	std::string getDesiredClusterKeyName() const override { return newName; }
};
Reference<IQuorumChange> nameQuorumChange(std::string const& name, Reference<IQuorumChange> const& other) {
	return Reference<IQuorumChange>(new NameQuorumChange(name, other));
}

struct AutoQuorumChange final : IQuorumChange {
	int desired;
	explicit AutoQuorumChange(int desired) : desired(desired) {}

	Future<std::vector<NetworkAddress>> getDesiredCoordinators(Transaction* tr,
	                                                           std::vector<NetworkAddress> oldCoordinators,
	                                                           Reference<IClusterConnectionRecord> ccr,
	                                                           CoordinatorsResult& err) override {
		return getDesired(Reference<AutoQuorumChange>::addRef(this), tr, oldCoordinators, ccr, &err);
	}

																#line 7289 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getRedundancy()
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetRedundancyActor>
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetRedundancyActorState {
															#line 7295 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetRedundancyActorState(AutoQuorumChange* const& self,Transaction* const& tr) 
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1425 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fStorageReplicas(tr->get("storage_replicas"_sr.withPrefix(configKeysPrefix))),
															#line 1426 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fLogReplicas(tr->get("log_replicas"_sr.withPrefix(configKeysPrefix)))
															#line 7308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getRedundancy", reinterpret_cast<unsigned long>(this));

	}
	~GetRedundancyActorState() 
	{
		fdb_probe_actor_destroy("getRedundancy", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fStorageReplicas) && success(fLogReplicas);
															#line 1427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetRedundancyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetRedundancyActor*>(this)->actor_wait_state = 1;
															#line 1427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRedundancyActor, 0, Void >*>(static_cast<GetRedundancyActor*>(this)));
															#line 7330 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1428 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		int redundancy = std::min(atoi(fStorageReplicas.get().get().toString().c_str()), atoi(fLogReplicas.get().get().toString().c_str()));
															#line 1431 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetRedundancyActor*>(this)->SAV<int>::futures) { (void)(redundancy); this->~GetRedundancyActorState(); static_cast<GetRedundancyActor*>(this)->destroy(); return 0; }
															#line 7355 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetRedundancyActor*>(this)->SAV< int >::value()) int(redundancy);
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1428 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		int redundancy = std::min(atoi(fStorageReplicas.get().get().toString().c_str()), atoi(fLogReplicas.get().get().toString().c_str()));
															#line 1431 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetRedundancyActor*>(this)->SAV<int>::futures) { (void)(redundancy); this->~GetRedundancyActorState(); static_cast<GetRedundancyActor*>(this)->destroy(); return 0; }
															#line 7369 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetRedundancyActor*>(this)->SAV< int >::value()) int(redundancy);
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRedundancyActor*>(this)->actor_wait_state > 0) static_cast<GetRedundancyActor*>(this)->actor_wait_state = 0;
		static_cast<GetRedundancyActor*>(this)->ActorCallback< GetRedundancyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRedundancyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRedundancyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRedundancyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AutoQuorumChange* self;
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1425 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<Optional<Value>> fStorageReplicas;
															#line 1426 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<Optional<Value>> fLogReplicas;
															#line 7460 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getRedundancy()
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetRedundancyActor final : public Actor<int>, public ActorCallback< GetRedundancyActor, 0, Void >, public FastAllocated<GetRedundancyActor>, public GetRedundancyActorState<GetRedundancyActor> {
															#line 7465 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetRedundancyActor>::operator new;
	using FastAllocated<GetRedundancyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7828621118307024640UL, 16888276979238733568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRedundancyActor, 0, Void >;
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetRedundancyActor(AutoQuorumChange* const& self,Transaction* const& tr) 
															#line 7482 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<int>(),
		   GetRedundancyActorState<GetRedundancyActor>(self, tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17051584921099603200UL, 8631560012367626240UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRedundancy");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRedundancyActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<int> getRedundancy( AutoQuorumChange* const& self, Transaction* const& tr ) {
															#line 1424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<int>(new GetRedundancyActor(self, tr));
															#line 7514 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

																#line 7519 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via isAcceptable()
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IsAcceptableActor>
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IsAcceptableActorState {
															#line 7525 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IsAcceptableActorState(AutoQuorumChange* const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,int const& desiredCount,std::set<AddressExclusion>* const& excluded) 
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldCoordinators(oldCoordinators),
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   ccr(ccr),
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCount(desiredCount),
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   excluded(excluded)
															#line 7542 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("isAcceptable", reinterpret_cast<unsigned long>(this));

	}
	~IsAcceptableActorState() 
	{
		fdb_probe_actor_destroy("isAcceptable", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1440 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ClusterConnectionString cs = ccr->getConnectionString();
															#line 1441 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() != cs.hostnames.size() + cs.coords.size())
															#line 7559 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1442 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7563 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1446 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() < desiredCount)
															#line 7571 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1447 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7575 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1448 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() % 2 != 1)
															#line 7583 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1449 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7587 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1452 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& c : oldCoordinators ) {
															#line 1453 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (addressExcluded(*excluded, c))
															#line 7597 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1454 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7601 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
					this->~IsAcceptableActorState();
					static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 1459 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::sort(oldCoordinators.begin(), oldCoordinators.end());
															#line 1460 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 1;i < oldCoordinators.size();i++) {
															#line 1461 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (oldCoordinators[i - 1].ip == oldCoordinators[i].ip)
															#line 7614 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1462 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7618 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
					this->~IsAcceptableActorState();
					static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 1465 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ClientCoordinators coord(ccr);
															#line 1466 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1467 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1468 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1469 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (coord.clientLeaderServers[i].hostname.present())
															#line 7635 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1470 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					leaderServers.push_back(retryGetReplyFromHostname(GetLeaderRequest(coord.clusterKey, UID()), coord.clientLeaderServers[i].hostname.get(), WLTOKEN_CLIENTLEADERREG_GETLEADER, TaskPriority::CoordinationReply));
															#line 7639 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1475 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 7645 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
															#line 1480 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<std::vector<Optional<LeaderInfo>>>> __when_expr_0 = timeout(getAll(leaderServers), CLIENT_KNOBS->IS_ACCEPTABLE_DELAY);
															#line 1480 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IsAcceptableActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IsAcceptableActor*>(this)->actor_wait_state = 1;
															#line 1480 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*>(static_cast<IsAcceptableActor*>(this)));
															#line 7657 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<std::vector<Optional<LeaderInfo>>> const& results,int loopDepth) 
	{
															#line 1482 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!results.present())
															#line 7680 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1483 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7684 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
			this->~IsAcceptableActorState();
			static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1485 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& r : results.get() ) {
															#line 1486 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!r.present())
															#line 7694 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1487 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7698 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1491 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7707 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<std::vector<Optional<LeaderInfo>>> && results,int loopDepth) 
	{
															#line 1482 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!results.present())
															#line 7719 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1483 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7723 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
			this->~IsAcceptableActorState();
			static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1485 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& r : results.get() ) {
															#line 1486 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!r.present())
															#line 7733 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1487 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7737 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1491 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7746 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<std::vector<Optional<LeaderInfo>>> const& results,int loopDepth) 
	{
		loopDepth = a_body1cont1(results, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<std::vector<Optional<LeaderInfo>>> && results,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(results), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IsAcceptableActor*>(this)->actor_wait_state > 0) static_cast<IsAcceptableActor*>(this)->actor_wait_state = 0;
		static_cast<IsAcceptableActor*>(this)->ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >::remove();

	}
	void a_callback_fire(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Optional<std::vector<Optional<LeaderInfo>>> const& value) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Optional<std::vector<Optional<LeaderInfo>>> && value) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Error err) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AutoQuorumChange* self;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IClusterConnectionRecord> ccr;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int desiredCount;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion>* excluded;
															#line 7841 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via isAcceptable()
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IsAcceptableActor final : public Actor<bool>, public ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >, public FastAllocated<IsAcceptableActor>, public IsAcceptableActorState<IsAcceptableActor> {
															#line 7846 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IsAcceptableActor>::operator new;
	using FastAllocated<IsAcceptableActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14259566613283357952UL, 7890443424477221376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >;
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IsAcceptableActor(AutoQuorumChange* const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,int const& desiredCount,std::set<AddressExclusion>* const& excluded) 
															#line 7863 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   IsAcceptableActorState<IsAcceptableActor>(self, tr, oldCoordinators, ccr, desiredCount, excluded),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9830863436459782912UL, 6089237774462111488UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("isAcceptable");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<bool> isAcceptable( AutoQuorumChange* const& self, Transaction* const& tr, std::vector<NetworkAddress> const& oldCoordinators, Reference<IClusterConnectionRecord> const& ccr, int const& desiredCount, std::set<AddressExclusion>* const& excluded ) {
															#line 1434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new IsAcceptableActor(self, tr, oldCoordinators, ccr, desiredCount, excluded));
															#line 7895 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1493 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

																#line 7900 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getDesired()
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDesiredActor>
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDesiredActorState {
															#line 7906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDesiredActorState(Reference<AutoQuorumChange> const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,CoordinatorsResult* const& err) 
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldCoordinators(oldCoordinators),
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   ccr(ccr),
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   err(err),
															#line 1499 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCount(self->desired)
															#line 7923 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDesired", reinterpret_cast<unsigned long>(this));

	}
	~GetDesiredActorState() 
	{
		fdb_probe_actor_destroy("getDesired", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1501 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (desiredCount == -1)
															#line 7938 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1502 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<int> __when_expr_0 = getRedundancy(self.getPtr(), tr);
															#line 1502 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7944 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetDesiredActor*>(this)->actor_wait_state = 1;
															#line 1502 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 0, int >*>(static_cast<GetDesiredActor*>(this)));
															#line 7949 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDesiredActorState();
		static_cast<GetDesiredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1506 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<AddressExclusion>> __when_expr_1 = getAllExcludedServers(tr);
															#line 1506 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7979 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 2;
															#line 1506 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*>(static_cast<GetDesiredActor*>(this)));
															#line 7984 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int const& redundancy,int loopDepth) 
	{
															#line 1503 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCount = redundancy * 2 - 1;
															#line 7993 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(int && redundancy,int loopDepth) 
	{
															#line 1503 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCount = redundancy * 2 - 1;
															#line 8002 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int const& redundancy,int loopDepth) 
	{
		loopDepth = a_body1cont2(redundancy, loopDepth);

		return loopDepth;
	}
	int a_body1when1(int && redundancy,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(redundancy), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 0, int >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 0, int >*,int const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 0, int >*,int && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 0, int >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excluded = std::set<AddressExclusion>(excl.begin(), excl.end());
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<ProcessData>> __when_expr_2 = getWorkers(tr);
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8090 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 3;
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*>(static_cast<GetDesiredActor*>(this)));
															#line 8095 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excluded = std::set<AddressExclusion>(excl.begin(), excl.end());
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<ProcessData>> __when_expr_2 = getWorkers(tr);
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8108 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 3;
															#line 1509 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*>(static_cast<GetDesiredActor*>(this)));
															#line 8113 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont4(excl, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(std::vector<ProcessData> const& _workers,int loopDepth) 
	{
															#line 1510 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = _workers;
															#line 1512 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<NetworkAddress, LocalityData> addr_locality;
															#line 1513 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto w : workers ) {
															#line 1514 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			addr_locality[w.address] = w.locality;
															#line 8203 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1519 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool checkAcceptable = true;
															#line 1520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<Optional<Standalone<StringRef>>> checkDuplicates;
															#line 1521 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto addr : oldCoordinators ) {
															#line 1522 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto findResult = addr_locality.find(addr);
															#line 1523 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (findResult == addr_locality.end() || checkDuplicates.count(findResult->second.zoneId()))
															#line 8215 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1524 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				checkAcceptable = false;
															#line 8219 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				break;
			}
															#line 1527 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			checkDuplicates.insert(findResult->second.zoneId());
															#line 8224 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1530 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (checkAcceptable)
															#line 8228 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_3 = isAcceptable(self.getPtr(), tr, oldCoordinators, ccr, desiredCount, &excluded);
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetDesiredActor*>(this)->actor_wait_state = 4;
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 3, bool >*>(static_cast<GetDesiredActor*>(this)));
															#line 8239 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(std::vector<ProcessData> && _workers,int loopDepth) 
	{
															#line 1510 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = _workers;
															#line 1512 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<NetworkAddress, LocalityData> addr_locality;
															#line 1513 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto w : workers ) {
															#line 1514 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			addr_locality[w.address] = w.locality;
															#line 8259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1519 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool checkAcceptable = true;
															#line 1520 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<Optional<Standalone<StringRef>>> checkDuplicates;
															#line 1521 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto addr : oldCoordinators ) {
															#line 1522 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto findResult = addr_locality.find(addr);
															#line 1523 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (findResult == addr_locality.end() || checkDuplicates.count(findResult->second.zoneId()))
															#line 8271 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1524 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				checkAcceptable = false;
															#line 8275 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				break;
			}
															#line 1527 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			checkDuplicates.insert(findResult->second.zoneId());
															#line 8280 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1530 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (checkAcceptable)
															#line 8284 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_3 = isAcceptable(self.getPtr(), tr, oldCoordinators, ccr, desiredCount, &excluded);
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8290 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetDesiredActor*>(this)->actor_wait_state = 4;
															#line 1531 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 3, bool >*>(static_cast<GetDesiredActor*>(this)));
															#line 8295 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(std::vector<ProcessData> const& _workers,int loopDepth) 
	{
		loopDepth = a_body1cont5(_workers, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(std::vector<ProcessData> && _workers,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1538 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<NetworkAddress> chosen;
															#line 1539 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		self->addDesiredWorkers(chosen, workers, desiredCount, excluded);
															#line 1541 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (chosen.size() < desiredCount)
															#line 8388 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1542 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (chosen.empty() || chosen.size() < oldCoordinators.size())
															#line 8392 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1543 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("NotEnoughMachinesForCoordinators") .detail("EligibleWorkers", workers.size()) .detail("ChosenWorkers", chosen.size()) .detail("DesiredCoordinators", desiredCount) .detail("CurrentCoordinators", oldCoordinators.size());
															#line 1548 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				*err = CoordinatorsResult::NOT_ENOUGH_MACHINES;
															#line 1549 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(std::vector<NetworkAddress>()); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8400 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::vector<NetworkAddress>());
				this->~GetDesiredActorState();
				static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1551 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			chosen.resize((chosen.size() - 1) | 1);
															#line 8408 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1554 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(chosen); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8412 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(chosen);
		this->~GetDesiredActorState();
		static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10(bool const& ok,int loopDepth) 
	{
															#line 1532 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 8424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1533 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			*err = CoordinatorsResult::SAME_NETWORK_ADDRESSES;
															#line 1534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(oldCoordinators); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::move(oldCoordinators)); // state_var_RVO
			this->~GetDesiredActorState();
			static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(bool && ok,int loopDepth) 
	{
															#line 1532 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 8444 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1533 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			*err = CoordinatorsResult::SAME_NETWORK_ADDRESSES;
															#line 1534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(oldCoordinators); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8450 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::move(oldCoordinators)); // state_var_RVO
			this->~GetDesiredActorState();
			static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(bool const& ok,int loopDepth) 
	{
		loopDepth = a_body1cont10(ok, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(bool && ok,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(ok), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 3, bool >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 3, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 3, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 3, bool >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<AutoQuorumChange> self;
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IClusterConnectionRecord> ccr;
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CoordinatorsResult* err;
															#line 1499 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int desiredCount;
															#line 1507 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion> excluded;
															#line 1510 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<ProcessData> workers;
															#line 8551 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDesired()
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDesiredActor final : public Actor<std::vector<NetworkAddress>>, public ActorCallback< GetDesiredActor, 0, int >, public ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >, public ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >, public ActorCallback< GetDesiredActor, 3, bool >, public FastAllocated<GetDesiredActor>, public GetDesiredActorState<GetDesiredActor> {
															#line 8556 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDesiredActor>::operator new;
	using FastAllocated<GetDesiredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4107087671554650112UL, 15170645845835729408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<NetworkAddress>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDesiredActor, 0, int >;
friend struct ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >;
friend struct ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >;
friend struct ActorCallback< GetDesiredActor, 3, bool >;
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDesiredActor(Reference<AutoQuorumChange> const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,CoordinatorsResult* const& err) 
															#line 8576 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<NetworkAddress>>(),
		   GetDesiredActorState<GetDesiredActor>(self, tr, oldCoordinators, ccr, err),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3649354580216779264UL, 10244989468105284608UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDesired");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDesiredActor, 0, int >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetDesiredActor, 3, bool >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<std::vector<NetworkAddress>> getDesired( Reference<AutoQuorumChange> const& self, Transaction* const& tr, std::vector<NetworkAddress> const& oldCoordinators, Reference<IClusterConnectionRecord> const& ccr, CoordinatorsResult* const& err ) {
															#line 1494 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<NetworkAddress>>(new GetDesiredActor(self, tr, oldCoordinators, ccr, err));
															#line 8611 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1556 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

	// Select a desired set of workers such that
	// (1) the number of workers at each locality type (e.g., dcid) <= desiredCount; and
	// (2) prefer workers at a locality where less workers has been chosen than other localities: evenly distribute
	// workers.
	void addDesiredWorkers(std::vector<NetworkAddress>& chosen,
	                       const std::vector<ProcessData>& workers,
	                       int desiredCount,
	                       const std::set<AddressExclusion>& excluded) {
		std::vector<ProcessData> remainingWorkers(workers);
		deterministicRandom()->randomShuffle(remainingWorkers);

		std::partition(remainingWorkers.begin(), remainingWorkers.end(), [](const ProcessData& data) {
			return (data.processClass == ProcessClass::CoordinatorClass);
		});

		TraceEvent(SevDebug, "AutoSelectCoordinators").detail("CandidateWorkers", remainingWorkers.size());
		for (auto worker = remainingWorkers.begin(); worker != remainingWorkers.end(); worker++) {
			TraceEvent(SevDebug, "AutoSelectCoordinators")
			    .detail("Worker", worker->processClass.toString())
			    .detail("Address", worker->address.toString())
			    .detail("Locality", worker->locality.toString());
		}
		TraceEvent(SevDebug, "AutoSelectCoordinators").detail("ExcludedAddress", excluded.size());
		for (auto& excludedAddr : excluded) {
			TraceEvent(SevDebug, "AutoSelectCoordinators").detail("ExcludedAddress", excludedAddr.toString());
		}

		std::map<StringRef, int> maxCounts;
		std::map<StringRef, std::map<StringRef, int>> currentCounts;
		std::map<StringRef, int> hardLimits;

		std::vector<StringRef> fields({ "dcid"_sr, "data_hall"_sr, "zoneid"_sr, "machineid"_sr });

		for (auto field = fields.begin(); field != fields.end(); field++) {
			if (field->toString() == "zoneid") {
				hardLimits[*field] = 1;
			} else {
				hardLimits[*field] = desiredCount;
			}
		}

		while (chosen.size() < desiredCount) {
			bool found = false;
			for (auto worker = remainingWorkers.begin(); worker != remainingWorkers.end(); worker++) {
				if (addressExcluded(excluded, worker->address)) {
					continue;
				}
				// Exclude faulty node due to machine assassination
				if (g_network->isSimulated() && !g_simulator->getProcessByAddress(worker->address)->isReliable()) {
					TraceEvent("AutoSelectCoordinators").detail("SkipUnreliableWorker", worker->address.toString());
					continue;
				}
				bool valid = true;
				for (auto field = fields.begin(); field != fields.end(); field++) {
					if (maxCounts[*field] == 0) {
						maxCounts[*field] = 1;
					}
					auto value = worker->locality.get(*field).orDefault(""_sr);
					auto currentCount = currentCounts[*field][value];
					if (currentCount >= maxCounts[*field]) {
						valid = false;
						break;
					}
				}
				if (valid) {
					for (auto field = fields.begin(); field != fields.end(); field++) {
						auto value = worker->locality.get(*field).orDefault(""_sr);
						currentCounts[*field][value] += 1;
					}
					chosen.push_back(worker->address);
					remainingWorkers.erase(worker);
					found = true;
					break;
				}
			}
			if (!found) {
				bool canIncrement = false;
				for (auto field = fields.begin(); field != fields.end(); field++) {
					if (maxCounts[*field] < hardLimits[*field]) {
						maxCounts[*field] += 1;
						canIncrement = true;
						break;
					}
				}
				if (!canIncrement) {
					break;
				}
			}
		}
	}
};
Reference<IQuorumChange> autoQuorumChange(int desired) {
	return Reference<IQuorumChange>(new AutoQuorumChange(desired));
}

															#line 8711 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeServers()
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeServersActor>
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActorState {
															#line 8718 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActorState(Transaction* const& tr,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 8729 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeServersActorState() 
	{
		fdb_probe_actor_destroy("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1653 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1654 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1655 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1656 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1657 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<AddressExclusion>> __when_expr_0 = failed ? getExcludedFailedServerList(tr) : getExcludedServerList(tr);
															#line 1657 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8754 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeServersActor*>(this)->actor_wait_state = 1;
															#line 1657 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*>(static_cast<ExcludeServersActor*>(this)));
															#line 8759 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
															#line 1658 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<AddressExclusion> exclusions(excl.begin(), excl.end());
															#line 1659 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1660 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : servers ) {
															#line 1661 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusions.find(s) != exclusions.end())
															#line 8788 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1664 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1665 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 8796 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1666 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedServersKey(s), StringRef());
															#line 8800 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1668 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedServersKey(s), StringRef());
															#line 8806 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1672 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 8811 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1673 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1674 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto serversVersionKey = failed ? failedServersVersionKey : excludedServersVersionKey;
															#line 1675 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(serversVersionKey));
															#line 1676 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(serversVersionKey, excludeVersionKey);
															#line 8821 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1678 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeServersCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1682 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActorState(); static_cast<ExcludeServersActor*>(this)->destroy(); return 0; }
															#line 8827 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
															#line 1658 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<AddressExclusion> exclusions(excl.begin(), excl.end());
															#line 1659 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1660 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : servers ) {
															#line 1661 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusions.find(s) != exclusions.end())
															#line 8845 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1664 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1665 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 8853 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1666 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedServersKey(s), StringRef());
															#line 8857 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1668 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedServersKey(s), StringRef());
															#line 8863 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1672 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 8868 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1673 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1674 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto serversVersionKey = failed ? failedServersVersionKey : excludedServersVersionKey;
															#line 1675 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(serversVersionKey));
															#line 1676 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(serversVersionKey, excludeVersionKey);
															#line 8878 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1678 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeServersCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1682 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActorState(); static_cast<ExcludeServersActor*>(this)->destroy(); return 0; }
															#line 8884 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(excl, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeServersActor*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor*>(this)->ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 8973 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeServers()
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor final : public Actor<Void>, public ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >, public FastAllocated<ExcludeServersActor>, public ExcludeServersActorState<ExcludeServersActor> {
															#line 8978 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeServersActor>::operator new;
	using FastAllocated<ExcludeServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12487370146513284352UL, 8371073076974607616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >;
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor(Transaction* const& tr,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 8995 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeServersActorState<ExcludeServersActor>(tr, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15846472300878847232UL, 13853541035855110912UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeServers( Transaction* const& tr, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1652 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeServersActor(tr, servers, failed));
															#line 9028 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1684 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 9033 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeServers()
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeServersActor1>
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor1State {
															#line 9040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor1State(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 9051 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeServersActor1State() 
	{
		fdb_probe_actor_destroy("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1686 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 9066 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1687 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1688 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 9072 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1712 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1713 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 9081 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1690 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1691 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1692 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.set( SpecialKeySpace::getManagementApiCommandOptionSpecialKey(failed ? "failed" : "excluded", "force"), ValueRef());
															#line 1695 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1696 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failed").withSuffix(s.toString()) : SpecialKeySpace::getManagementApiCommandPrefix("exclude").withSuffix(s.toString());
															#line 1699 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw.set(addr, ValueRef());
															#line 9123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1701 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersSpecialKeySpaceCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed);
															#line 1704 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1704 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9131 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 1;
															#line 1704 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 0, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1707 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersError").errorUnsuppressed(e);
															#line 1708 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1708 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9162 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 2;
															#line 1708 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 1, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1705 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9182 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1705 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9194 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = excludeServers(&tr, servers, failed);
															#line 1715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 3;
															#line 1715 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 2, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9383 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1719 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersError").errorUnsuppressed(e);
															#line 1720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9409 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 5;
															#line 1720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 4, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9431 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 4;
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 3, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9436 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9447 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 4;
															#line 1716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 3, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9452 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2cont3(Void const& _,int loopDepth) 
	{
															#line 1717 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9536 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont3(Void && _,int loopDepth) 
	{
															#line 1717 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9548 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1687 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1712 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 9728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeServers()
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor1 final : public Actor<Void>, public ActorCallback< ExcludeServersActor1, 0, Void >, public ActorCallback< ExcludeServersActor1, 1, Void >, public ActorCallback< ExcludeServersActor1, 2, Void >, public ActorCallback< ExcludeServersActor1, 3, Void >, public ActorCallback< ExcludeServersActor1, 4, Void >, public FastAllocated<ExcludeServersActor1>, public ExcludeServersActor1State<ExcludeServersActor1> {
															#line 9733 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeServersActor1>::operator new;
	using FastAllocated<ExcludeServersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12487370146513284352UL, 8371073076974607616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeServersActor1, 0, Void >;
friend struct ActorCallback< ExcludeServersActor1, 1, Void >;
friend struct ActorCallback< ExcludeServersActor1, 2, Void >;
friend struct ActorCallback< ExcludeServersActor1, 3, Void >;
friend struct ActorCallback< ExcludeServersActor1, 4, Void >;
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor1(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 9754 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeServersActor1State<ExcludeServersActor1>(cx, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1716282752885344256UL, 5229742794320878592UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeServersActor1, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ExcludeServersActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ExcludeServersActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ExcludeServersActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ExcludeServersActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeServers( Database const& cx, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1685 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeServersActor1(cx, servers, failed));
															#line 9791 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1725 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// excludes localities by setting the keys in api version below 7.0
															#line 9797 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeLocalities()
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeLocalitiesActor>
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActorState {
															#line 9804 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActorState(Transaction* const& tr,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 9815 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1729 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1730 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1731 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1732 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<std::string>> __when_expr_0 = failed ? getExcludedFailedLocalityList(tr) : getExcludedLocalityList(tr);
															#line 1732 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9840 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 1732 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*>(static_cast<ExcludeLocalitiesActor*>(this)));
															#line 9845 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<std::string> const& excl,int loopDepth) 
	{
															#line 1733 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<std::string> exclusion(excl.begin(), excl.end());
															#line 1734 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1735 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& l : localities ) {
															#line 1736 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusion.find(l) != exclusion.end())
															#line 9874 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1739 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1740 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 9882 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1741 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedLocalityKey(l), StringRef());
															#line 9886 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1743 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedLocalityKey(l), StringRef());
															#line 9892 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1746 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 9897 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1747 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1748 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto localityVersionKey = failed ? failedLocalityVersionKey : excludedLocalityVersionKey;
															#line 1749 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(localityVersionKey));
															#line 1750 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(localityVersionKey, excludeVersionKey);
															#line 9907 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1752 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1756 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActorState(); static_cast<ExcludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 9913 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<std::string> && excl,int loopDepth) 
	{
															#line 1733 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<std::string> exclusion(excl.begin(), excl.end());
															#line 1734 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1735 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& l : localities ) {
															#line 1736 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusion.find(l) != exclusion.end())
															#line 9931 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1739 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1740 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 9939 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1741 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedLocalityKey(l), StringRef());
															#line 9943 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1743 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedLocalityKey(l), StringRef());
															#line 9949 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1746 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 9954 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1747 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1748 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto localityVersionKey = failed ? failedLocalityVersionKey : excludedLocalityVersionKey;
															#line 1749 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(localityVersionKey));
															#line 1750 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(localityVersionKey, excludeVersionKey);
															#line 9964 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1752 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1756 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActorState(); static_cast<ExcludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 9970 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<std::string> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(excl, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<std::string> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor*>(this)->ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,std::vector<std::string> const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,std::vector<std::string> && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::unordered_set<std::string> localities;
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 10059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeLocalities()
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor final : public Actor<Void>, public ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >, public FastAllocated<ExcludeLocalitiesActor>, public ExcludeLocalitiesActorState<ExcludeLocalitiesActor> {
															#line 10064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeLocalitiesActor>::operator new;
	using FastAllocated<ExcludeLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18112833866131571712UL, 17046801431081786624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >;
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor(Transaction* const& tr,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 10081 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeLocalitiesActorState<ExcludeLocalitiesActor>(tr, localities, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8343920748801204224UL, 15351083160916243200UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeLocalities( Transaction* const& tr, std::unordered_set<std::string> const& localities, bool const& failed ) {
															#line 1727 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeLocalitiesActor(tr, localities, failed));
															#line 10114 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1758 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Exclude the servers matching the given set of localities from use as state servers.
// excludes localities by setting the keys.
															#line 10121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeLocalities()
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeLocalitiesActor1>
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor1State {
															#line 10128 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor1State(Database const& cx,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 10139 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeLocalitiesActor1State() 
	{
		fdb_probe_actor_destroy("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1762 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 10154 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1763 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1764 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1789 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1790 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10169 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1766 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1767 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1768 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.set(SpecialKeySpace::getManagementApiCommandOptionSpecialKey( failed ? "failed_locality" : "excluded_locality", "force"), ValueRef());
															#line 1771 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( const auto& l : localities ) {
															#line 1772 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failedlocality").withSuffix(l) : SpecialKeySpace::getManagementApiCommandPrefix("excludedlocality").withSuffix(l);
															#line 1775 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw.set(addr, ValueRef());
															#line 10211 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1777 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesSpecialKeySpaceCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed);
															#line 1781 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1781 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10219 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 1;
															#line 1781 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 0, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10224 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1784 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesError").errorUnsuppressed(e);
															#line 1785 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1785 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10250 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 2;
															#line 1785 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 1, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10255 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1782 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10270 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1782 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10282 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1792 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = excludeLocalities(&tr, localities, failed);
															#line 1792 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10466 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 3;
															#line 1792 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 2, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10471 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1796 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesError").errorUnsuppressed(e);
															#line 1797 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1797 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10497 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 5;
															#line 1797 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 4, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10502 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10519 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 4;
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 3, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10524 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10535 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 4;
															#line 1793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 3, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2cont3(Void const& _,int loopDepth) 
	{
															#line 1794 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10624 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont3(Void && _,int loopDepth) 
	{
															#line 1794 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::unordered_set<std::string> localities;
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1763 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1789 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 10816 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeLocalities()
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor1 final : public Actor<Void>, public ActorCallback< ExcludeLocalitiesActor1, 0, Void >, public ActorCallback< ExcludeLocalitiesActor1, 1, Void >, public ActorCallback< ExcludeLocalitiesActor1, 2, Void >, public ActorCallback< ExcludeLocalitiesActor1, 3, Void >, public ActorCallback< ExcludeLocalitiesActor1, 4, Void >, public FastAllocated<ExcludeLocalitiesActor1>, public ExcludeLocalitiesActor1State<ExcludeLocalitiesActor1> {
															#line 10821 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeLocalitiesActor1>::operator new;
	using FastAllocated<ExcludeLocalitiesActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18112833866131571712UL, 17046801431081786624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeLocalitiesActor1, 0, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 1, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 2, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 3, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 4, Void >;
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor1(Database const& cx,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 10842 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeLocalitiesActor1State<ExcludeLocalitiesActor1>(cx, localities, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13644543871368453632UL, 7369712116135787008UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeLocalities( Database const& cx, std::unordered_set<std::string> const& localities, bool const& failed ) {
															#line 1761 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeLocalitiesActor1(cx, localities, failed));
															#line 10879 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1802 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 10884 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via includeServers()
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IncludeServersActor>
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeServersActorState {
															#line 10891 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeServersActorState(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed),
															#line 1804 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   versionKey(deterministicRandom()->randomUniqueID().toString())
															#line 10904 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("includeServers", reinterpret_cast<unsigned long>(this));

	}
	~IncludeServersActorState() 
	{
		fdb_probe_actor_destroy("includeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1805 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 10919 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1806 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1807 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10925 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1845 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1846 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10934 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1809 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1810 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1811 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1812 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!s.isValid())
															#line 10972 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1813 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 10976 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1814 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(SpecialKeySpace::getManagementApiCommandRange("failed"));
															#line 10980 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1816 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(SpecialKeySpace::getManagementApiCommandRange("exclude"));
															#line 10986 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
				else
				{
															#line 1819 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failed").withSuffix(s.toString()) : SpecialKeySpace::getManagementApiCommandPrefix("exclude").withSuffix(s.toString());
															#line 1822 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(addr);
															#line 1831 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (s.isWholeMachine())
															#line 10997 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1832 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(KeyRangeRef(addr.withSuffix(":"_sr), addr.withSuffix(";"_sr)));
															#line 11001 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
			}
															#line 1835 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersCommit").detail("Servers", describe(servers)).detail("Failed", failed);
															#line 1837 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1837 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11011 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 1;
															#line 1837 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 0, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11016 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1840 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersError").errorUnsuppressed(e);
															#line 1841 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1841 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 2;
															#line 1841 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 1, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11047 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1838 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11062 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1838 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11074 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1848 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1849 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1850 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1851 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1855 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::CAUSAL_WRITE_RISKY);
															#line 1856 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 11266 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1857 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(failedServersVersionKey));
															#line 1858 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(failedServersVersionKey, versionKey);
															#line 11272 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1860 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(excludedServersVersionKey));
															#line 1861 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(excludedServersVersionKey, versionKey);
															#line 11280 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1864 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1865 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!s.isValid())
															#line 11286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1866 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 11290 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1867 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(failedServersKeys);
															#line 11294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1869 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(excludedServersKeys);
															#line 11300 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
				else
				{
															#line 1871 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (s.isWholeMachine())
															#line 11307 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1880 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						auto addr = failed ? encodeFailedServersKey(s) : encodeExcludedServersKey(s);
															#line 1881 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(singleKeyRange(addr));
															#line 1882 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(KeyRangeRef(addr + ':', addr + char(':' + 1)));
															#line 11315 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1884 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						if (failed)
															#line 11321 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
						{
															#line 1885 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
							tr.clear(encodeFailedServersKey(s));
															#line 11325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
						}
						else
						{
															#line 1887 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
							tr.clear(encodeExcludedServersKey(s));
															#line 11331 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
						}
					}
				}
			}
															#line 1892 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersCommit").detail("Servers", describe(servers)).detail("Failed", failed);
															#line 1894 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1894 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 11342 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 3;
															#line 1894 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 2, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11347 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1897 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersError").errorUnsuppressed(e);
															#line 1898 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1898 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11373 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 4;
															#line 1898 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 3, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1895 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11393 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1895 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11405 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1804 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string versionKey;
															#line 1806 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1845 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 11587 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via includeServers()
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeServersActor final : public Actor<Void>, public ActorCallback< IncludeServersActor, 0, Void >, public ActorCallback< IncludeServersActor, 1, Void >, public ActorCallback< IncludeServersActor, 2, Void >, public ActorCallback< IncludeServersActor, 3, Void >, public FastAllocated<IncludeServersActor>, public IncludeServersActorState<IncludeServersActor> {
															#line 11592 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IncludeServersActor>::operator new;
	using FastAllocated<IncludeServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14985955504058178304UL, 7971629533755298048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IncludeServersActor, 0, Void >;
friend struct ActorCallback< IncludeServersActor, 1, Void >;
friend struct ActorCallback< IncludeServersActor, 2, Void >;
friend struct ActorCallback< IncludeServersActor, 3, Void >;
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeServersActor(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 11612 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   IncludeServersActorState<IncludeServersActor>(cx, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7035060310568136960UL, 2930223583431837696UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("includeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IncludeServersActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IncludeServersActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< IncludeServersActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< IncludeServersActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> includeServers( Database const& cx, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1803 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new IncludeServersActor(cx, servers, failed));
															#line 11648 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1903 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Remove the given localities from the exclusion list.
// include localities by clearing the keys.
															#line 11655 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via includeLocalities()
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IncludeLocalitiesActor>
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeLocalitiesActorState {
															#line 11662 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeLocalitiesActorState(Database const& cx,std::vector<std::string> const& localities,bool const& failed,bool const& includeAll) 
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed),
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   includeAll(includeAll),
															#line 1907 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   versionKey(deterministicRandom()->randomUniqueID().toString())
															#line 11677 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("includeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~IncludeLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("includeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1908 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 11692 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1910 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 11698 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1941 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1942 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 11707 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1912 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1913 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1914 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (includeAll)
															#line 11743 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1915 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (failed)
															#line 11747 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1916 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(SpecialKeySpace::getManagementApiCommandRange("failedlocality"));
															#line 11751 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1918 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(SpecialKeySpace::getManagementApiCommandRange("excludedlocality"));
															#line 11757 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
			else
			{
															#line 1921 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				for( const auto& l : localities ) {
															#line 1922 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					Key locality = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failedlocality").withSuffix(l) : SpecialKeySpace::getManagementApiCommandPrefix("excludedlocality").withSuffix(l);
															#line 1925 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(locality);
															#line 11768 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
															#line 1928 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("Failed", failed) .detail("IncludeAll", includeAll);
															#line 1933 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1933 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11777 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 1933 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 0, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 11782 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1936 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesError").errorUnsuppressed(e);
															#line 1937 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1937 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11808 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 2;
															#line 1937 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 1, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 11813 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1934 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 11828 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1934 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 11840 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1944 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1945 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1946 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1947 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1951 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::CAUSAL_WRITE_RISKY);
															#line 1952 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 12032 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1953 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(failedLocalityVersionKey));
															#line 1954 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(failedLocalityVersionKey, versionKey);
															#line 12038 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1956 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(excludedLocalityVersionKey));
															#line 1957 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(excludedLocalityVersionKey, versionKey);
															#line 12046 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1960 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (includeAll)
															#line 12050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1961 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (failed)
															#line 12054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1962 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(failedLocalityKeys);
															#line 12058 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1964 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(excludedLocalityKeys);
															#line 12064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
			else
			{
															#line 1967 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				for( const auto& l : localities ) {
															#line 1968 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 12073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1969 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(encodeFailedLocalityKey(l));
															#line 12077 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1971 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(encodeExcludedLocalityKey(l));
															#line 12083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
			}
															#line 1976 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("Failed", failed) .detail("IncludeAll", includeAll);
															#line 1981 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1981 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 12093 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 3;
															#line 1981 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 2, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 12098 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1984 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesError").errorUnsuppressed(e);
															#line 1985 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1985 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12124 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 4;
															#line 1985 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 3, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 12129 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1982 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 12144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1982 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 12156 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> localities;
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool includeAll;
															#line 1907 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string versionKey;
															#line 1909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1941 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 12340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via includeLocalities()
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeLocalitiesActor final : public Actor<Void>, public ActorCallback< IncludeLocalitiesActor, 0, Void >, public ActorCallback< IncludeLocalitiesActor, 1, Void >, public ActorCallback< IncludeLocalitiesActor, 2, Void >, public ActorCallback< IncludeLocalitiesActor, 3, Void >, public FastAllocated<IncludeLocalitiesActor>, public IncludeLocalitiesActorState<IncludeLocalitiesActor> {
															#line 12345 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IncludeLocalitiesActor>::operator new;
	using FastAllocated<IncludeLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14451772252627204352UL, 8556766290850130176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IncludeLocalitiesActor, 0, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 1, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 2, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 3, Void >;
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeLocalitiesActor(Database const& cx,std::vector<std::string> const& localities,bool const& failed,bool const& includeAll) 
															#line 12365 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   IncludeLocalitiesActorState<IncludeLocalitiesActor>(cx, localities, failed, includeAll),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3203046928570983680UL, 8290273524939964416UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("includeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> includeLocalities( Database const& cx, std::vector<std::string> const& localities, bool const& failed, bool const& includeAll ) {
															#line 1906 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new IncludeLocalitiesActor(cx, localities, failed, includeAll));
															#line 12401 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1990 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 12406 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setClass()
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetClassActor>
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetClassActorState {
															#line 12413 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetClassActorState(Database const& cx,AddressExclusion const& server,ProcessClass const& processClass) 
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   server(server),
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processClass(processClass),
															#line 1992 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 12426 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setClass", reinterpret_cast<unsigned long>(this));

	}
	~SetClassActorState() 
	{
		fdb_probe_actor_destroy("setClass", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1994 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 12441 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1996 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1997 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1998 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1999 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 2001 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<ProcessData>> __when_expr_0 = getWorkers(&tr);
															#line 2001 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12482 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetClassActor*>(this)->actor_wait_state = 1;
															#line 2001 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*>(static_cast<SetClassActor*>(this)));
															#line 12487 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2021 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2021 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12511 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<SetClassActor*>(this)->actor_wait_state = 3;
															#line 2021 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 2, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12516 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> const& workers,int loopDepth) 
	{
															#line 2003 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool foundChange = false;
															#line 2004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 2005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (server.excludes(workers[i].address))
															#line 12535 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2006 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (processClass.classType() != ProcessClass::InvalidClass)
															#line 12539 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 2007 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.set(processClassKeyFor(workers[i].locality.processId().get()), processClassValue(processClass));
															#line 12543 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 2010 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(processClassKeyFor(workers[i].locality.processId().get()));
															#line 12549 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 2011 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				foundChange = true;
															#line 12553 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2015 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (foundChange)
															#line 12558 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2016 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.set(processClassChangeKey, deterministicRandom()->randomUniqueID().toString());
															#line 12562 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12568 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetClassActor*>(this)->actor_wait_state = 2;
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 1, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12573 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> && workers,int loopDepth) 
	{
															#line 2003 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool foundChange = false;
															#line 2004 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 2005 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (server.excludes(workers[i].address))
															#line 12586 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2006 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (processClass.classType() != ProcessClass::InvalidClass)
															#line 12590 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 2007 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.set(processClassKeyFor(workers[i].locality.processId().get()), processClassValue(processClass));
															#line 12594 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 2010 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(processClassKeyFor(workers[i].locality.processId().get()));
															#line 12600 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 2011 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				foundChange = true;
															#line 12604 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2015 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (foundChange)
															#line 12609 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2016 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.set(processClassChangeKey, deterministicRandom()->randomUniqueID().toString());
															#line 12613 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12619 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetClassActor*>(this)->actor_wait_state = 2;
															#line 2018 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 1, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12624 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(workers, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> && workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 0, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 2019 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetClassActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SetClassActorState(); static_cast<SetClassActor*>(this)->destroy(); return 0; }
															#line 12708 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetClassActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 2019 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetClassActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SetClassActorState(); static_cast<SetClassActor*>(this)->destroy(); return 0; }
															#line 12720 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetClassActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetClassActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetClassActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AddressExclusion server;
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ProcessClass processClass;
															#line 1992 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 12898 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setClass()
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetClassActor final : public Actor<Void>, public ActorCallback< SetClassActor, 0, std::vector<ProcessData> >, public ActorCallback< SetClassActor, 1, Void >, public ActorCallback< SetClassActor, 2, Void >, public FastAllocated<SetClassActor>, public SetClassActorState<SetClassActor> {
															#line 12903 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetClassActor>::operator new;
	using FastAllocated<SetClassActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3367689617086481920UL, 17474752902884152576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetClassActor, 0, std::vector<ProcessData> >;
friend struct ActorCallback< SetClassActor, 1, Void >;
friend struct ActorCallback< SetClassActor, 2, Void >;
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetClassActor(Database const& cx,AddressExclusion const& server,ProcessClass const& processClass) 
															#line 12922 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   SetClassActorState<SetClassActor>(cx, server, processClass),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7738751807625442048UL, 7840563547066985984UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setClass");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetClassActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetClassActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> setClass( Database const& cx, AddressExclusion const& server, ProcessClass const& processClass ) {
															#line 1991 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new SetClassActor(cx, server, processClass));
															#line 12957 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2025 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 12962 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedServerList()
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedServerListActor>
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedServerListActorState {
															#line 12969 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedServerListActorState(Transaction* const& tr) 
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 12976 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedServerList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedServerListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedServerList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2027 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(excludedServersKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2027 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedServerListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12993 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedServerListActor*>(this)->actor_wait_state = 1;
															#line 2027 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedServerListActor, 0, RangeResult >*>(static_cast<GetExcludedServerListActor*>(this)));
															#line 12998 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedServerListActorState();
		static_cast<GetExcludedServerListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2028 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2030 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<AddressExclusion> exclusions;
															#line 2031 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(auto i = r.begin();i != r.end();++i) {
															#line 2032 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeExcludedServersKey(i->key);
															#line 2033 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (a.isValid())
															#line 13029 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2034 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.push_back(a);
															#line 13033 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2036 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2037 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedServerListActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetExcludedServerListActorState(); static_cast<GetExcludedServerListActor*>(this)->destroy(); return 0; }
															#line 13040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedServerListActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetExcludedServerListActorState();
		static_cast<GetExcludedServerListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2027 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 13052 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedServerListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedServerListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedServerListActor*>(this)->ActorCallback< GetExcludedServerListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2027 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 13131 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedServerList()
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedServerListActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetExcludedServerListActor, 0, RangeResult >, public FastAllocated<GetExcludedServerListActor>, public GetExcludedServerListActorState<GetExcludedServerListActor> {
															#line 13136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedServerListActor>::operator new;
	using FastAllocated<GetExcludedServerListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12697722419774045696UL, 7329476351157896960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedServerListActor, 0, RangeResult >;
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedServerListActor(Transaction* const& tr) 
															#line 13153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetExcludedServerListActorState<GetExcludedServerListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7528942178455984128UL, 12820870738981153792UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedServerList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedServerListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getExcludedServerList( Transaction* const& tr ) {
															#line 2026 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetExcludedServerListActor(tr));
															#line 13186 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2039 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13191 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedFailedServerList()
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedFailedServerListActor>
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedServerListActorState {
															#line 13198 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedServerListActorState(Transaction* const& tr) 
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 13205 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedFailedServerListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(failedServersKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13222 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state = 1;
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*>(static_cast<GetExcludedFailedServerListActor*>(this)));
															#line 13227 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedFailedServerListActorState();
		static_cast<GetExcludedFailedServerListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2044 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<AddressExclusion> exclusions;
															#line 2045 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(auto i = r.begin();i != r.end();++i) {
															#line 2046 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeFailedServersKey(i->key);
															#line 2047 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (a.isValid())
															#line 13258 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2048 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.push_back(a);
															#line 13262 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2050 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2051 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedFailedServerListActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetExcludedFailedServerListActorState(); static_cast<GetExcludedFailedServerListActor*>(this)->destroy(); return 0; }
															#line 13269 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedFailedServerListActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetExcludedFailedServerListActorState();
		static_cast<GetExcludedFailedServerListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 13281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedFailedServerListActor*>(this)->ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2041 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 13360 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedFailedServerList()
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedServerListActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >, public FastAllocated<GetExcludedFailedServerListActor>, public GetExcludedFailedServerListActorState<GetExcludedFailedServerListActor> {
															#line 13365 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedFailedServerListActor>::operator new;
	using FastAllocated<GetExcludedFailedServerListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1937792545326723072UL, 7171189220141997824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >;
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedServerListActor(Transaction* const& tr) 
															#line 13382 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetExcludedFailedServerListActorState<GetExcludedFailedServerListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7137961867629435392UL, 15389248370416580864UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedFailedServerList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getExcludedFailedServerList( Transaction* const& tr ) {
															#line 2040 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetExcludedFailedServerListActor(tr));
															#line 13415 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2053 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13420 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedServers()
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedServersActor>
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActorState {
															#line 13427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActorState(Transaction* const& tr) 
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2055 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(),
															#line 2057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedServers(getExcludedServerList(tr)),
															#line 2058 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedFailed(getExcludedFailedServerList(tr)),
															#line 2059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedLocalities(getAllExcludedLocalities(tr)),
															#line 2060 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fWorkers(getWorkers(tr))
															#line 13444 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedServersActorState() 
	{
		fdb_probe_actor_destroy("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fExcludedServers) && success(fExcludedFailed) && success(fExcludedLocalities);
															#line 2064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13461 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 1;
															#line 2064 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 0, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13466 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedServersActorState();
		static_cast<GetAllExcludedServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2066 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedServers = fExcludedServers.get();
															#line 2067 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedServers.begin(), excludedServers.end());
															#line 2068 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedFailed = fExcludedFailed.get();
															#line 2069 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedFailed.begin(), excludedFailed.end());
															#line 2073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excludedLocalities = fExcludedLocalities.get();
															#line 2076 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!excludedLocalities.empty())
															#line 13499 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(fWorkers);
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13505 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 2;
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 1, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13510 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2066 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedServers = fExcludedServers.get();
															#line 2067 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedServers.begin(), excludedServers.end());
															#line 2068 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedFailed = fExcludedFailed.get();
															#line 2069 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedFailed.begin(), excludedFailed.end());
															#line 2073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excludedLocalities = fExcludedLocalities.get();
															#line 2076 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!excludedLocalities.empty())
															#line 13534 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(fWorkers);
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13540 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 2;
															#line 2078 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 1, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13545 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor*>(this)->ActorCallback< GetAllExcludedServersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2090 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2091 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActorState(); static_cast<GetAllExcludedServersActor*>(this)->destroy(); return 0; }
															#line 13636 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedServersActorState();
		static_cast<GetAllExcludedServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2079 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = fWorkers.get();
															#line 2081 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& locality : excludedLocalities ) {
															#line 2082 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::set<AddressExclusion> localityAddresses = getAddressesByLocality(workers, locality);
															#line 2083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!localityAddresses.empty())
															#line 13654 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2085 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.insert(exclusions.end(), localityAddresses.begin(), localityAddresses.end());
															#line 13658 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2079 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = fWorkers.get();
															#line 2081 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& locality : excludedLocalities ) {
															#line 2082 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::set<AddressExclusion> localityAddresses = getAddressesByLocality(workers, locality);
															#line 2083 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!localityAddresses.empty())
															#line 13675 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2085 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.insert(exclusions.end(), localityAddresses.begin(), localityAddresses.end());
															#line 13679 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor*>(this)->ActorCallback< GetAllExcludedServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2055 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> exclusions;
															#line 2057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<AddressExclusion>> fExcludedServers;
															#line 2058 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<AddressExclusion>> fExcludedFailed;
															#line 2059 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fExcludedLocalities;
															#line 2060 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<ProcessData>> fWorkers;
															#line 2073 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> excludedLocalities;
															#line 2079 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<ProcessData> workers;
															#line 13777 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedServers()
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetAllExcludedServersActor, 0, Void >, public ActorCallback< GetAllExcludedServersActor, 1, Void >, public FastAllocated<GetAllExcludedServersActor>, public GetAllExcludedServersActorState<GetAllExcludedServersActor> {
															#line 13782 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedServersActor>::operator new;
	using FastAllocated<GetAllExcludedServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13903873176549393664UL, 12839964751759077632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedServersActor, 0, Void >;
friend struct ActorCallback< GetAllExcludedServersActor, 1, Void >;
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor(Transaction* const& tr) 
															#line 13800 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetAllExcludedServersActorState<GetAllExcludedServersActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17118544508197446144UL, 12587635006963120896UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedServersActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedServersActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getAllExcludedServers( Transaction* const& tr ) {
															#line 2054 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetAllExcludedServersActor(tr));
															#line 13834 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2093 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13839 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedServers()
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedServersActor1>
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor1State {
															#line 13846 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor1State(Database const& cx) 
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2095 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 13855 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedServersActor1State() 
	{
		fdb_probe_actor_destroy("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2096 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 13870 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2098 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2099 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2100 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<AddressExclusion>> __when_expr_0 = getAllExcludedServers(&tr);
															#line 2101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 1;
															#line 2101 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*>(static_cast<GetAllExcludedServersActor1*>(this)));
															#line 13914 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2104 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2104 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13938 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 2;
															#line 2104 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor1, 1, Void >*>(static_cast<GetAllExcludedServersActor1*>(this)));
															#line 13943 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<AddressExclusion> const& exclusions,int loopDepth) 
	{
															#line 2102 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor1*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActor1State(); static_cast<GetAllExcludedServersActor1*>(this)->destroy(); return 0; }
															#line 13958 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor1*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<AddressExclusion> && exclusions,int loopDepth) 
	{
															#line 2102 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor1*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActor1State(); static_cast<GetAllExcludedServersActor1*>(this)->destroy(); return 0; }
															#line 13970 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor1*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<AddressExclusion> const& exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(exclusions, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<AddressExclusion> && exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(exclusions), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor1*>(this)->ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor1*>(this)->ActorCallback< GetAllExcludedServersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2095 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 14144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedServers()
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor1 final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >, public ActorCallback< GetAllExcludedServersActor1, 1, Void >, public FastAllocated<GetAllExcludedServersActor1>, public GetAllExcludedServersActor1State<GetAllExcludedServersActor1> {
															#line 14149 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedServersActor1>::operator new;
	using FastAllocated<GetAllExcludedServersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13903873176549393664UL, 12839964751759077632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >;
friend struct ActorCallback< GetAllExcludedServersActor1, 1, Void >;
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor1(Database const& cx) 
															#line 14167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetAllExcludedServersActor1State<GetAllExcludedServersActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4564542305281620992UL, 15906870840917760000UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedServersActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getAllExcludedServers( Database const& cx ) {
															#line 2094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetAllExcludedServersActor1(cx));
															#line 14201 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2108 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14206 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedLocalityList()
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedLocalityListActor>
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedLocalityListActorState {
															#line 14213 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedLocalityListActorState(Transaction* const& tr) 
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 14220 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedLocalityListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(excludedLocalityKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14237 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state = 1;
															#line 2110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*>(static_cast<GetExcludedLocalityListActor*>(this)));
															#line 14242 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedLocalityListActorState();
		static_cast<GetExcludedLocalityListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2111 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2113 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<std::string> excludedLocalities;
															#line 2114 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& i : r ) {
															#line 2115 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeExcludedLocalityKey(i.key);
															#line 2116 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			excludedLocalities.push_back(a);
															#line 14273 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2118 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(excludedLocalities);
															#line 2119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedLocalityListActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(excludedLocalities); this->~GetExcludedLocalityListActorState(); static_cast<GetExcludedLocalityListActor*>(this)->destroy(); return 0; }
															#line 14279 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedLocalityListActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(excludedLocalities);
		this->~GetExcludedLocalityListActorState();
		static_cast<GetExcludedLocalityListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 14291 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedLocalityListActor*>(this)->ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2110 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 14370 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedLocalityList()
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedLocalityListActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >, public FastAllocated<GetExcludedLocalityListActor>, public GetExcludedLocalityListActorState<GetExcludedLocalityListActor> {
															#line 14375 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedLocalityListActor>::operator new;
	using FastAllocated<GetExcludedLocalityListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13986458352579819520UL, 6708671320427178496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >;
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedLocalityListActor(Transaction* const& tr) 
															#line 14392 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetExcludedLocalityListActorState<GetExcludedLocalityListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3296438703473487872UL, 3197736188210309376UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedLocalityList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getExcludedLocalityList( Transaction* const& tr ) {
															#line 2109 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetExcludedLocalityListActor(tr));
															#line 14425 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2121 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedFailedLocalityList()
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedFailedLocalityListActor>
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedLocalityListActorState {
															#line 14437 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedLocalityListActorState(Transaction* const& tr) 
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 14444 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedFailedLocalityListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(failedLocalityKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14461 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state = 1;
															#line 2123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*>(static_cast<GetExcludedFailedLocalityListActor*>(this)));
															#line 14466 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedFailedLocalityListActorState();
		static_cast<GetExcludedFailedLocalityListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2124 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2126 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<std::string> excludedLocalities;
															#line 2127 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& i : r ) {
															#line 2128 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeFailedLocalityKey(i.key);
															#line 2129 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			excludedLocalities.push_back(a);
															#line 14497 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2131 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(excludedLocalities);
															#line 2132 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedFailedLocalityListActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(excludedLocalities); this->~GetExcludedFailedLocalityListActorState(); static_cast<GetExcludedFailedLocalityListActor*>(this)->destroy(); return 0; }
															#line 14503 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedFailedLocalityListActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(excludedLocalities);
		this->~GetExcludedFailedLocalityListActorState();
		static_cast<GetExcludedFailedLocalityListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 14515 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedFailedLocalityListActor*>(this)->ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2123 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 14594 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedFailedLocalityList()
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedLocalityListActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >, public FastAllocated<GetExcludedFailedLocalityListActor>, public GetExcludedFailedLocalityListActorState<GetExcludedFailedLocalityListActor> {
															#line 14599 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedFailedLocalityListActor>::operator new;
	using FastAllocated<GetExcludedFailedLocalityListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12484263580634185472UL, 14408151171408026112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >;
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedLocalityListActor(Transaction* const& tr) 
															#line 14616 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetExcludedFailedLocalityListActorState<GetExcludedFailedLocalityListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11099463865640149248UL, 96070748485650944UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedFailedLocalityList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getExcludedFailedLocalityList( Transaction* const& tr ) {
															#line 2122 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetExcludedFailedLocalityListActor(tr));
															#line 14649 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2134 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14654 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedLocalitiesActor>
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActorState {
															#line 14661 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActorState(Transaction* const& tr) 
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(),
															#line 2137 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedLocalities(getExcludedLocalityList(tr)),
															#line 2138 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fFailedLocalities(getExcludedFailedLocalityList(tr))
															#line 14674 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2141 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fExcludedLocalities) && success(fFailedLocalities);
															#line 2141 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14691 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 2141 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*>(static_cast<GetAllExcludedLocalitiesActor*>(this)));
															#line 14696 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2143 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedLocalities = fExcludedLocalities.get();
															#line 2144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedLocalities.begin(), excludedLocalities.end());
															#line 2145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto failedLocalities = fFailedLocalities.get();
															#line 2146 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), failedLocalities.begin(), failedLocalities.end());
															#line 2148 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2149 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActorState(); static_cast<GetAllExcludedLocalitiesActor*>(this)->destroy(); return 0; }
															#line 14729 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2143 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedLocalities = fExcludedLocalities.get();
															#line 2144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedLocalities.begin(), excludedLocalities.end());
															#line 2145 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto failedLocalities = fFailedLocalities.get();
															#line 2146 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), failedLocalities.begin(), failedLocalities.end());
															#line 2148 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2149 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActorState(); static_cast<GetAllExcludedLocalitiesActor*>(this)->destroy(); return 0; }
															#line 14751 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor*>(this)->ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2136 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> exclusions;
															#line 2137 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fExcludedLocalities;
															#line 2138 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fFailedLocalities;
															#line 14842 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >, public FastAllocated<GetAllExcludedLocalitiesActor>, public GetAllExcludedLocalitiesActorState<GetAllExcludedLocalitiesActor> {
															#line 14847 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedLocalitiesActor>::operator new;
	using FastAllocated<GetAllExcludedLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7644299386855441152UL, 7429067320777624064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >;
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor(Transaction* const& tr) 
															#line 14864 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetAllExcludedLocalitiesActorState<GetAllExcludedLocalitiesActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15824813939058471168UL, 12546097955091880448UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getAllExcludedLocalities( Transaction* const& tr ) {
															#line 2135 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetAllExcludedLocalitiesActor(tr));
															#line 14897 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2151 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Get the list of excluded localities by reading the keys.
															#line 14903 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedLocalitiesActor1>
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor1State {
															#line 14910 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor1State(Database const& cx) 
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2154 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 14919 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedLocalitiesActor1State() 
	{
		fdb_probe_actor_destroy("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2155 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 14934 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2157 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2158 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2159 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<std::string>> __when_expr_0 = getAllExcludedLocalities(&tr);
															#line 2160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14973 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 1;
															#line 2160 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*>(static_cast<GetAllExcludedLocalitiesActor1*>(this)));
															#line 14978 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2163 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2163 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15002 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 2;
															#line 2163 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*>(static_cast<GetAllExcludedLocalitiesActor1*>(this)));
															#line 15007 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<std::string> const& exclusions,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActor1State(); static_cast<GetAllExcludedLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 15022 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(exclusions);
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<std::string> && exclusions,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActor1State(); static_cast<GetAllExcludedLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 15034 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(exclusions);
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<std::string> const& exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(exclusions, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<std::string> && exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(exclusions), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,std::vector<std::string> const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,std::vector<std::string> && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2154 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 15208 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor1 final : public Actor<std::vector<std::string>>, public ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >, public ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >, public FastAllocated<GetAllExcludedLocalitiesActor1>, public GetAllExcludedLocalitiesActor1State<GetAllExcludedLocalitiesActor1> {
															#line 15213 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedLocalitiesActor1>::operator new;
	using FastAllocated<GetAllExcludedLocalitiesActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7644299386855441152UL, 7429067320777624064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >;
friend struct ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >;
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor1(Database const& cx) 
															#line 15231 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetAllExcludedLocalitiesActor1State<GetAllExcludedLocalitiesActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3977877330550522880UL, 15060849589996873216UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getAllExcludedLocalities( Database const& cx ) {
															#line 2153 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetAllExcludedLocalitiesActor1(cx));
															#line 15265 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2167 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Decodes the locality string to a pair of locality prefix and its value.
// The prefix could be dcid, processid, machineid, processid.
std::pair<std::string, std::string> decodeLocality(const std::string& locality) {
	StringRef localityRef((const uint8_t*)(locality.c_str()), locality.size());

	std::string localityKeyValue = localityRef.removePrefix(LocalityData::ExcludeLocalityPrefix).toString();
	int split = localityKeyValue.find(':');
	if (split != std::string::npos) {
		return std::make_pair(localityKeyValue.substr(0, split), localityKeyValue.substr(split + 1));
	}

	return std::make_pair("", "");
}

// Returns the list of IPAddresses of the servers that match the given locality.
// Example: locality="dcid:primary" returns all the ip addresses of the servers in the primary dc.
std::set<AddressExclusion> getServerAddressesByLocality(
    const std::map<std::string, StorageServerInterface> server_interfaces,
    const std::string& locality) {
	std::pair<std::string, std::string> locality_key_value = decodeLocality(locality);
	std::set<AddressExclusion> locality_addresses;

	for (auto& server : server_interfaces) {
		auto locality_value = server.second.locality.get(locality_key_value.first);
		if (!locality_value.present()) {
			continue;
		}

		if (locality_value.get() != locality_key_value.second) {
			continue;
		}

		auto primary_address = server.second.address();
		locality_addresses.insert(AddressExclusion(primary_address.ip, primary_address.port));
		if (server.second.secondaryAddress().present()) {
			auto secondary_address = server.second.secondaryAddress().get();
			locality_addresses.insert(AddressExclusion(secondary_address.ip, secondary_address.port));
		}
	}

	return locality_addresses;
}

// Returns the list of IPAddresses of the workers that match the given locality.
// Example: locality="locality_dcid:primary" returns all the ip addresses of the workers in the primary dc.
std::set<AddressExclusion> getAddressesByLocality(const std::vector<ProcessData>& workers,
                                                  const std::string& locality) {
	std::pair<std::string, std::string> locality_key_value = decodeLocality(locality);
	std::set<AddressExclusion> locality_addresses;

	for (int i = 0; i < workers.size(); i++) {
		auto locality_value = workers[i].locality.get(locality_key_value.first);
		if (!locality_value.present()) {
			continue;
		}

		if (locality_value.get() != locality_key_value.second) {
			continue;
		}

		locality_addresses.insert(AddressExclusion(workers[i].address.ip, workers[i].address.port));
	}

	return locality_addresses;
}

															#line 15336 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via printHealthyZone()
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class PrintHealthyZoneActor>
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class PrintHealthyZoneActorState {
															#line 15343 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	PrintHealthyZoneActorState(Database const& cx) 
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2235 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 15352 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("printHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~PrintHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("printHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2236 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 15367 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2238 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 2239 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2240 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2240 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15404 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2240 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*>(static_cast<PrintHealthyZoneActor*>(this)));
															#line 15409 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2254 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2254 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2254 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrintHealthyZoneActor, 1, Void >*>(static_cast<PrintHealthyZoneActor*>(this)));
															#line 15438 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15453 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2242 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			printf("Data distribution has been disabled for all storage server failures in this cluster and thus " "maintenance mode is not active.\n");
															#line 15457 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
		else
		{
															#line 2244 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!val.present() || decodeHealthyZoneValue(val.get()).second <= tr.getReadVersion().get())
															#line 15463 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2245 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("No ongoing maintenance.\n");
															#line 15467 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 2247 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto healthyZone = decodeHealthyZoneValue(val.get());
															#line 2248 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				fmt::print("Maintenance for zone {0} will continue for {1} seconds.\n", healthyZone.first.toString(), (healthyZone.second - tr.getReadVersion().get()) / CLIENT_KNOBS->CORE_VERSIONSPERSECOND);
															#line 15475 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2252 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<PrintHealthyZoneActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrintHealthyZoneActorState(); static_cast<PrintHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15480 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<PrintHealthyZoneActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15492 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2242 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			printf("Data distribution has been disabled for all storage server failures in this cluster and thus " "maintenance mode is not active.\n");
															#line 15496 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
		else
		{
															#line 2244 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!val.present() || decodeHealthyZoneValue(val.get()).second <= tr.getReadVersion().get())
															#line 15502 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2245 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("No ongoing maintenance.\n");
															#line 15506 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 2247 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto healthyZone = decodeHealthyZoneValue(val.get());
															#line 2248 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				fmt::print("Maintenance for zone {0} will continue for {1} seconds.\n", healthyZone.first.toString(), (healthyZone.second - tr.getReadVersion().get()) / CLIENT_KNOBS->CORE_VERSIONSPERSECOND);
															#line 15514 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2252 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<PrintHealthyZoneActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrintHealthyZoneActorState(); static_cast<PrintHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15519 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<PrintHealthyZoneActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<PrintHealthyZoneActor*>(this)->ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<PrintHealthyZoneActor*>(this)->ActorCallback< PrintHealthyZoneActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2235 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 15693 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via printHealthyZone()
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class PrintHealthyZoneActor final : public Actor<Void>, public ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< PrintHealthyZoneActor, 1, Void >, public FastAllocated<PrintHealthyZoneActor>, public PrintHealthyZoneActorState<PrintHealthyZoneActor> {
															#line 15698 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<PrintHealthyZoneActor>::operator new;
	using FastAllocated<PrintHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18021955742278569984UL, 1602703512969712640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< PrintHealthyZoneActor, 1, Void >;
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	PrintHealthyZoneActor(Database const& cx) 
															#line 15716 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   PrintHealthyZoneActorState<PrintHealthyZoneActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3507524340034560UL, 4848501524512482048UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("printHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PrintHealthyZoneActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> printHealthyZone( Database const& cx ) {
															#line 2234 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new PrintHealthyZoneActor(cx));
															#line 15750 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2258 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 15755 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via clearHealthyZone()
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ClearHealthyZoneActor>
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ClearHealthyZoneActorState {
															#line 15762 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClearHealthyZoneActorState(Database const& cx,bool const& printWarning,bool const& clearSSFailureZoneString) 
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   printWarning(printWarning),
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   clearSSFailureZoneString(clearSSFailureZoneString),
															#line 2260 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 15775 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("clearHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~ClearHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("clearHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2261 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ClearHealthyZone").detail("ClearSSFailureZoneString", clearSSFailureZoneString);
															#line 2262 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 15792 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2264 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2265 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2266 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2267 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2267 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15831 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2267 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15836 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15860 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 2, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15865 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2268 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clearSSFailureZoneString && val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15880 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2270 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 15884 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2271 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 15888 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2274 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15892 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~ClearHealthyZoneActorState();
			static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2277 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(healthyZoneKey);
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15904 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 1, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15909 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2268 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clearSSFailureZoneString && val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15918 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2270 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 15922 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2271 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 15926 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2274 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15930 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~ClearHealthyZoneActorState();
			static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2277 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(healthyZoneKey);
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15942 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2278 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 1, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15947 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 2279 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16031 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 2279 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16043 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool printWarning;
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool clearSSFailureZoneString;
															#line 2260 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 16221 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via clearHealthyZone()
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ClearHealthyZoneActor final : public Actor<bool>, public ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< ClearHealthyZoneActor, 1, Void >, public ActorCallback< ClearHealthyZoneActor, 2, Void >, public FastAllocated<ClearHealthyZoneActor>, public ClearHealthyZoneActorState<ClearHealthyZoneActor> {
															#line 16226 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ClearHealthyZoneActor>::operator new;
	using FastAllocated<ClearHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15618599556202930432UL, 15987040098830664704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< ClearHealthyZoneActor, 1, Void >;
friend struct ActorCallback< ClearHealthyZoneActor, 2, Void >;
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClearHealthyZoneActor(Database const& cx,bool const& printWarning,bool const& clearSSFailureZoneString) 
															#line 16245 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   ClearHealthyZoneActorState<ClearHealthyZoneActor>(cx, printWarning, clearSSFailureZoneString),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(868363240144782336UL, 16824621560516897536UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clearHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> clearHealthyZone( Database const& cx, bool const& printWarning, bool const& clearSSFailureZoneString ) {
															#line 2259 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new ClearHealthyZoneActor(cx, printWarning, clearSSFailureZoneString));
															#line 16280 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2285 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 16285 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setHealthyZone()
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetHealthyZoneActor>
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetHealthyZoneActorState {
															#line 16292 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetHealthyZoneActorState(Database const& cx,StringRef const& zoneId,double const& seconds,bool const& printWarning) 
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   zoneId(zoneId),
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   seconds(seconds),
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   printWarning(printWarning),
															#line 2287 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 16307 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~SetHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("setHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2288 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("SetHealthyZone").detail("Zone", zoneId).detail("DurationSeconds", seconds);
															#line 2289 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 16324 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2291 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2292 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2293 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16363 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16368 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 2308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16392 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 4;
															#line 2308 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 3, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16397 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2295 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 16412 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2296 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 16416 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2297 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 16420 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2300 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~SetHealthyZoneActorState();
			static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Version> __when_expr_1 = tr.getReadVersion();
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16434 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 1, Version >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16439 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2295 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 16448 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2296 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 16452 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2297 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 16456 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2300 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16460 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~SetHealthyZoneActorState();
			static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Version> __when_expr_1 = tr.getReadVersion();
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16470 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2302 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 1, Version >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16475 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Version const& readVersion,int loopDepth) 
	{
															#line 2303 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(healthyZoneKey, healthyZoneValue(zoneId, readVersion + (seconds * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)));
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16563 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 2, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16568 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Version && readVersion,int loopDepth) 
	{
															#line 2303 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(healthyZoneKey, healthyZoneValue(zoneId, readVersion + (seconds * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)));
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16581 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2305 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 2, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16586 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Version const& readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(readVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Version && readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(readVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2306 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16670 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2306 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16682 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	StringRef zoneId;
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	double seconds;
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool printWarning;
															#line 2287 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 16862 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setHealthyZone()
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetHealthyZoneActor final : public Actor<bool>, public ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< SetHealthyZoneActor, 1, Version >, public ActorCallback< SetHealthyZoneActor, 2, Void >, public ActorCallback< SetHealthyZoneActor, 3, Void >, public FastAllocated<SetHealthyZoneActor>, public SetHealthyZoneActorState<SetHealthyZoneActor> {
															#line 16867 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetHealthyZoneActor>::operator new;
	using FastAllocated<SetHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15870478742953865728UL, 17906251069754494720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< SetHealthyZoneActor, 1, Version >;
friend struct ActorCallback< SetHealthyZoneActor, 2, Void >;
friend struct ActorCallback< SetHealthyZoneActor, 3, Void >;
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetHealthyZoneActor(Database const& cx,StringRef const& zoneId,double const& seconds,bool const& printWarning) 
															#line 16887 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   SetHealthyZoneActorState<SetHealthyZoneActor>(cx, zoneId, seconds, printWarning),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14961976220816392704UL, 7810201303269410048UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 1, Version >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> setHealthyZone( Database const& cx, StringRef const& zoneId, double const& seconds, bool const& printWarning ) {
															#line 2286 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new SetHealthyZoneActor(cx, zoneId, seconds, printWarning));
															#line 16923 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2312 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 16928 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setDDMode()
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetDDModeActor>
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetDDModeActorState {
															#line 16935 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetDDModeActorState(Database const& cx,int const& mode) 
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   mode(mode),
															#line 2314 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx),
															#line 2315 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldMode(-1),
															#line 2316 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   wr(Unversioned())
															#line 16950 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setDDMode", reinterpret_cast<unsigned long>(this));

	}
	~SetDDModeActorState() 
	{
		fdb_probe_actor_destroy("setDDMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2317 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			wr << mode;
															#line 2319 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 16967 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2321 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2322 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2323 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2324 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(dataDistributionModeKey);
															#line 2324 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17006 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 1;
															#line 2324 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 0, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17011 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2355 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("SetDDModeRetrying").error(e);
															#line 2356 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 2356 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17037 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 4;
															#line 2356 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 3, Void >*>(static_cast<SetDDModeActor*>(this)));
															#line 17042 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& old,int loopDepth) 
	{
															#line 2325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldMode < 0)
															#line 17057 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2326 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldMode = 1;
															#line 2327 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (old.present())
															#line 17063 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2328 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				BinaryReader rd(old.get(), Unversioned());
															#line 2329 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				rd >> oldMode;
															#line 17069 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2332 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrMyOwner(Unversioned());
															#line 2333 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrMyOwner << dataDistributionModeLock;
															#line 2334 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockOwnerKey, wrMyOwner.toValue());
															#line 2335 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrLastWrite(Unversioned());
															#line 2336 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrLastWrite << deterministicRandom()->randomUniqueID();
															#line 2337 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockWriteKey, wrLastWrite.toValue());
															#line 2339 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(dataDistributionModeKey, wr.toValue());
															#line 2340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (mode)
															#line 17088 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(healthyZoneKey);
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17094 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 2;
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 1, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17099 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && old,int loopDepth) 
	{
															#line 2325 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldMode < 0)
															#line 17113 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2326 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldMode = 1;
															#line 2327 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (old.present())
															#line 17119 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2328 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				BinaryReader rd(old.get(), Unversioned());
															#line 2329 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				rd >> oldMode;
															#line 17125 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2332 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrMyOwner(Unversioned());
															#line 2333 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrMyOwner << dataDistributionModeLock;
															#line 2334 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockOwnerKey, wrMyOwner.toValue());
															#line 2335 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrLastWrite(Unversioned());
															#line 2336 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrLastWrite << deterministicRandom()->randomUniqueID();
															#line 2337 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockWriteKey, wrLastWrite.toValue());
															#line 2339 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(dataDistributionModeKey, wr.toValue());
															#line 2340 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (mode)
															#line 17144 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(healthyZoneKey);
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17150 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 2;
															#line 2344 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 1, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17155 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& old,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(old, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && old,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(old), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 2352 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2352 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17246 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetDDModeActor*>(this)->actor_wait_state = 3;
															#line 2352 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 2, Void >*>(static_cast<SetDDModeActor*>(this)));
															#line 17251 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Optional<Value> const& currentHealthyZoneValue,int loopDepth) 
	{
															#line 2345 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (currentHealthyZoneValue.present() && decodeHealthyZoneValue(currentHealthyZoneValue.get()).first == ignoreSSFailuresZoneString)
															#line 17260 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2348 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.clear(healthyZoneKey);
															#line 17264 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2350 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(rebalanceDDIgnoreKey);
															#line 17268 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Optional<Value> && currentHealthyZoneValue,int loopDepth) 
	{
															#line 2345 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (currentHealthyZoneValue.present() && decodeHealthyZoneValue(currentHealthyZoneValue.get()).first == ignoreSSFailuresZoneString)
															#line 17277 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2348 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.clear(healthyZoneKey);
															#line 17281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2350 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(rebalanceDDIgnoreKey);
															#line 17285 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> const& currentHealthyZoneValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(currentHealthyZoneValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> && currentHealthyZoneValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(currentHealthyZoneValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 2353 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetDDModeActor*>(this)->SAV<int>::futures) { (void)(oldMode); this->~SetDDModeActorState(); static_cast<SetDDModeActor*>(this)->destroy(); return 0; }
															#line 17369 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetDDModeActor*>(this)->SAV< int >::value()) int(std::move(oldMode)); // state_var_RVO
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 2353 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetDDModeActor*>(this)->SAV<int>::futures) { (void)(oldMode); this->~SetDDModeActorState(); static_cast<SetDDModeActor*>(this)->destroy(); return 0; }
															#line 17381 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetDDModeActor*>(this)->SAV< int >::value()) int(std::move(oldMode)); // state_var_RVO
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int mode;
															#line 2314 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 2315 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int oldMode;
															#line 2316 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	BinaryWriter wr;
															#line 17561 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setDDMode()
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetDDModeActor final : public Actor<int>, public ActorCallback< SetDDModeActor, 0, Optional<Value> >, public ActorCallback< SetDDModeActor, 1, Optional<Value> >, public ActorCallback< SetDDModeActor, 2, Void >, public ActorCallback< SetDDModeActor, 3, Void >, public FastAllocated<SetDDModeActor>, public SetDDModeActorState<SetDDModeActor> {
															#line 17566 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetDDModeActor>::operator new;
	using FastAllocated<SetDDModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8090375644941392640UL, 6963524998233662464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetDDModeActor, 0, Optional<Value> >;
friend struct ActorCallback< SetDDModeActor, 1, Optional<Value> >;
friend struct ActorCallback< SetDDModeActor, 2, Void >;
friend struct ActorCallback< SetDDModeActor, 3, Void >;
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetDDModeActor(Database const& cx,int const& mode) 
															#line 17586 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<int>(),
		   SetDDModeActorState<SetDDModeActor>(cx, mode),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3750982474694156544UL, 11167414918040745728UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setDDMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetDDModeActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetDDModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetDDModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SetDDModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<int> setDDMode( Database const& cx, int const& mode ) {
															#line 2313 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<int>(new SetDDModeActor(cx, mode));
															#line 17622 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2360 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 17627 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via checkForExcludingServersTxActor()
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class CheckForExcludingServersTxActorActor>
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersTxActorActorState {
															#line 17634 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersTxActorActorState(ReadYourWritesTransaction* const& tr,std::set<AddressExclusion>* const& exclusions,std::set<NetworkAddress>* const& inProgressExclusion) 
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(exclusions),
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   inProgressExclusion(inProgressExclusion)
															#line 17645 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this));

	}
	~CheckForExcludingServersTxActorActorState() 
	{
		fdb_probe_actor_destroy("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2365 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(inProgressExclusion->size() == 0);
															#line 2366 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!exclusions->size())
															#line 17662 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2367 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckForExcludingServersTxActorActorState(); static_cast<CheckForExcludingServersTxActorActor*>(this)->destroy(); return 0; }
															#line 17666 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV< bool >::value()) bool(true);
				this->~CheckForExcludingServersTxActorActorState();
				static_cast<CheckForExcludingServersTxActorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2369 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2370 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2371 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2377 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2377 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17682 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 1;
															#line 2377 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17687 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckForExcludingServersTxActorActorState();
		static_cast<CheckForExcludingServersTxActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& serverList,int loopDepth) 
	{
															#line 2378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2380 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ok = true;
															#line 2381 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : serverList ) {
															#line 2382 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto addresses = decodeServerListValue(s.value).getKeyValues.getEndpoint().addresses;
															#line 2383 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addressExcluded(*exclusions, addresses.address))
															#line 17718 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2384 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2385 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.address);
															#line 17724 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2387 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addresses.secondaryAddress.present() && addressExcluded(*exclusions, addresses.secondaryAddress.get()))
															#line 17728 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2388 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2389 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.secondaryAddress.get());
															#line 17734 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2393 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 17739 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Standalone<StringRef>>> __when_expr_1 = tr->get(logsKey);
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17745 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 2;
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17750 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(RangeResult && serverList,int loopDepth) 
	{
															#line 2378 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2380 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ok = true;
															#line 2381 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : serverList ) {
															#line 2382 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto addresses = decodeServerListValue(s.value).getKeyValues.getEndpoint().addresses;
															#line 2383 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addressExcluded(*exclusions, addresses.address))
															#line 17772 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2384 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2385 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.address);
															#line 17778 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2387 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addresses.secondaryAddress.present() && addressExcluded(*exclusions, addresses.secondaryAddress.get()))
															#line 17782 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2388 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2389 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.secondaryAddress.get());
															#line 17788 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2393 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 17793 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Standalone<StringRef>>> __when_expr_1 = tr->get(logsKey);
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17799 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 2;
															#line 2394 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17804 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& serverList,int loopDepth) 
	{
		loopDepth = a_body1cont1(serverList, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && serverList,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(serverList), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersTxActorActor*>(this)->ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2411 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV<bool>::futures) { (void)(ok); this->~CheckForExcludingServersTxActorActorState(); static_cast<CheckForExcludingServersTxActorActor*>(this)->destroy(); return 0; }
															#line 17893 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV< bool >::value()) bool(std::move(ok)); // state_var_RVO
		this->~CheckForExcludingServersTxActorActorState();
		static_cast<CheckForExcludingServersTxActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Optional<Standalone<StringRef>> const& value,int loopDepth) 
	{
															#line 2395 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(value.present());
															#line 2396 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto logs = decodeLogsValue(value.get());
															#line 2397 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.first ) {
															#line 2398 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17911 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2399 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2400 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17917 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2403 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.second ) {
															#line 2404 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17924 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2405 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2406 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17930 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Optional<Standalone<StringRef>> && value,int loopDepth) 
	{
															#line 2395 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(value.present());
															#line 2396 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto logs = decodeLogsValue(value.get());
															#line 2397 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.first ) {
															#line 2398 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17947 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2399 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2400 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17953 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2403 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.second ) {
															#line 2404 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17960 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2405 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2406 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17966 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Standalone<StringRef>> const& value,int loopDepth) 
	{
		loopDepth = a_body1cont7(value, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Standalone<StringRef>> && value,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersTxActorActor*>(this)->ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction* tr;
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion>* exclusions;
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<NetworkAddress>* inProgressExclusion;
															#line 2380 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool ok;
															#line 18056 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via checkForExcludingServersTxActor()
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersTxActorActor final : public Actor<bool>, public ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >, public ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >, public FastAllocated<CheckForExcludingServersTxActorActor>, public CheckForExcludingServersTxActorActorState<CheckForExcludingServersTxActorActor> {
															#line 18061 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<CheckForExcludingServersTxActorActor>::operator new;
	using FastAllocated<CheckForExcludingServersTxActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14649292037843463680UL, 5440771375749529856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >;
friend struct ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >;
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersTxActorActor(ReadYourWritesTransaction* const& tr,std::set<AddressExclusion>* const& exclusions,std::set<NetworkAddress>* const& inProgressExclusion) 
															#line 18079 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   CheckForExcludingServersTxActorActorState<CheckForExcludingServersTxActorActor>(tr, exclusions, inProgressExclusion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(835331421418836480UL, 17258140674795182080UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkForExcludingServersTxActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> checkForExcludingServersTxActor( ReadYourWritesTransaction* const& tr, std::set<AddressExclusion>* const& exclusions, std::set<NetworkAddress>* const& inProgressExclusion ) {
															#line 2361 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new CheckForExcludingServersTxActorActor(tr, exclusions, inProgressExclusion));
															#line 18113 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2413 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 18118 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via checkForExcludingServers()
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class CheckForExcludingServersActor>
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersActorState {
															#line 18125 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersActorState(Database const& cx,std::vector<AddressExclusion> const& excl,bool const& waitForAllExcluded) 
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   excl(excl),
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   waitForAllExcluded(waitForAllExcluded),
															#line 2417 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(excl.begin(), excl.end()),
															#line 2418 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   inProgressExclusion()
															#line 18140 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("checkForExcludingServers", reinterpret_cast<unsigned long>(this));

	}
	~CheckForExcludingServersActorState() 
	{
		fdb_probe_actor_destroy("checkForExcludingServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2420 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 18155 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckForExcludingServersActorState();
		static_cast<CheckForExcludingServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2436 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18178 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
		this->~CheckForExcludingServersActorState();
		static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2421 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr = ReadYourWritesTransaction(cx);
															#line 2422 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		inProgressExclusion.clear();
															#line 18199 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		try {
															#line 2424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_0 = checkForExcludingServersTxActor(&tr, &exclusions, &inProgressExclusion);
															#line 2424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18205 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 1;
															#line 2424 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 0, bool >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18210 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2432 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("CheckForExcludingServersError").error(e);
															#line 2433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18249 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 3;
															#line 2433 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 2, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18254 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(bool const& ok,int loopDepth) 
	{
															#line 2425 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 18269 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2426 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18273 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
			this->~CheckForExcludingServersActorState();
			static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!waitForAllExcluded)
															#line 18281 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayJittered(1.0);
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18289 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 2;
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 1, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18294 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(bool && ok,int loopDepth) 
	{
															#line 2425 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 18303 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2426 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18307 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
			this->~CheckForExcludingServersActorState();
			static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2427 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!waitForAllExcluded)
															#line 18315 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayJittered(1.0);
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18323 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 2;
															#line 2430 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 1, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18328 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/build_output/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(bool const& ok,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(ok, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool && ok,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(ok), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> excl;
															#line 2414 "/codebuild/output/src2270659404/src/github.com/apple/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool waitForAllExcluded;
													