#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
/*
 * WorkerInterface.actor.h
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once
#if defined(NO_INTELLISENSE) && !defined(FDBSERVER_WORKERINTERFACE_ACTOR_G_H)
#define FDBSERVER_WORKERINTERFACE_ACTOR_G_H
#include "fdbserver/WorkerInterface.actor.g.h"
#elif !defined(FDBSERVER_WORKERINTERFACE_ACTOR_H)
#define FDBSERVER_WORKERINTERFACE_ACTOR_H

#include "fdbserver/BackupInterface.h"
#include "fdbserver/DataDistributorInterface.h"
#include "fdbclient/EncryptKeyProxyInterface.h"
#include "fdbserver/MasterInterface.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/RatekeeperInterface.h"
#include "fdbclient/ConsistencyScanInterface.actor.h"
#include "fdbserver/ResolverInterface.h"
#include "fdbclient/ClientBooleanParams.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbclient/FDBTypes.h"
#include "fdbserver/LogSystemConfig.h"
#include "fdbrpc/MultiInterface.h"
#include "fdbclient/ClientWorkerInterface.h"
#include "fdbserver/RecoveryState.h"
#include "fdbserver/ConfigBroadcastInterface.h"
#include "flow/actorcompiler.h"

struct WorkerInterface {
	constexpr static FileIdentifier file_identifier = 14712718;
	ClientWorkerInterface clientInterface;
	LocalityData locality;
	RequestStream<struct InitializeTLogRequest> tLog;
	RequestStream<struct RecruitMasterRequest> master;
	RequestStream<struct InitializeCommitProxyRequest> commitProxy;
	RequestStream<struct InitializeGrvProxyRequest> grvProxy;
	RequestStream<struct InitializeDataDistributorRequest> dataDistributor;
	RequestStream<struct InitializeRatekeeperRequest> ratekeeper;
	RequestStream<struct InitializeConsistencyScanRequest> consistencyScan;
	RequestStream<struct InitializeResolverRequest> resolver;
	RequestStream<struct InitializeStorageRequest> storage;
	RequestStream<struct InitializeLogRouterRequest> logRouter;
	RequestStream<struct InitializeBackupRequest> backup;
	RequestStream<struct InitializeEncryptKeyProxyRequest> encryptKeyProxy;

	RequestStream<struct LoadedPingRequest> debugPing;
	RequestStream<struct CoordinationPingMessage> coordinationPing;
	RequestStream<ReplyPromise<Void>> waitFailure;
	RequestStream<struct SetMetricsLogRateRequest> setMetricsRate;
	RequestStream<struct EventLogRequest> eventLogRequest;
	RequestStream<struct TraceBatchDumpRequest> traceBatchDumpRequest;
	RequestStream<struct DiskStoreRequest> diskStoreRequest;
	RequestStream<struct ExecuteRequest> execReq;
	RequestStream<struct WorkerSnapRequest> workerSnapReq;
	RequestStream<struct UpdateServerDBInfoRequest> updateServerDBInfo;

	TesterInterface testerInterface;

	UID id() const { return tLog.getEndpoint().token; }
	NetworkAddress address() const { return tLog.getEndpoint().getPrimaryAddress(); }
	NetworkAddress stableAddress() const { return tLog.getEndpoint().getStableAddress(); }
	Optional<NetworkAddress> secondaryAddress() const { return tLog.getEndpoint().addresses.secondaryAddress; }
	NetworkAddressList addresses() const { return tLog.getEndpoint().addresses; }

	WorkerInterface() {}
	WorkerInterface(const LocalityData& locality) : locality(locality) {}

	void initEndpoints() {
		clientInterface.initEndpoints();
		tLog.getEndpoint(TaskPriority::Worker);
		master.getEndpoint(TaskPriority::Worker);
		commitProxy.getEndpoint(TaskPriority::Worker);
		grvProxy.getEndpoint(TaskPriority::Worker);
		resolver.getEndpoint(TaskPriority::Worker);
		logRouter.getEndpoint(TaskPriority::Worker);
		debugPing.getEndpoint(TaskPriority::Worker);
		coordinationPing.getEndpoint(TaskPriority::Worker);
		updateServerDBInfo.getEndpoint(TaskPriority::Worker);
		eventLogRequest.getEndpoint(TaskPriority::Worker);
	}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           clientInterface,
		           locality,
		           tLog,
		           master,
		           commitProxy,
		           grvProxy,
		           dataDistributor,
		           ratekeeper,
		           consistencyScan,
		           resolver,
		           storage,
		           logRouter,
		           debugPing,
		           coordinationPing,
		           waitFailure,
		           setMetricsRate,
		           eventLogRequest,
		           traceBatchDumpRequest,
		           testerInterface,
		           diskStoreRequest,
		           execReq,
		           workerSnapReq,
		           backup,
		           encryptKeyProxy,
		           updateServerDBInfo);
	}
};

struct WorkerDetails {
	constexpr static FileIdentifier file_identifier = 9973980;
	WorkerInterface interf;
	ProcessClass processClass;
	bool degraded;
	bool recoveredDiskFiles;

	WorkerDetails() : degraded(false), recoveredDiskFiles(false) {}
	WorkerDetails(const WorkerInterface& interf, ProcessClass processClass, bool degraded, bool recoveredDiskFiles)
	  : interf(interf), processClass(processClass), degraded(degraded), recoveredDiskFiles(recoveredDiskFiles) {}

	bool operator<(const WorkerDetails& r) const { return interf.id() < r.interf.id(); }

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, interf, processClass, degraded, recoveredDiskFiles);
	}
};

// This interface and its serialization depend on slicing, since the client will deserialize only the first part of this
// structure
struct SWIFT_CXX_IMPORT_OWNED ClusterControllerFullInterface {
	constexpr static FileIdentifier file_identifier = ClusterControllerClientInterface::file_identifier;
	ClusterInterface clientInterface;
	RequestStream<struct RecruitFromConfigurationRequest> recruitFromConfiguration;
	RequestStream<struct RecruitRemoteFromConfigurationRequest> recruitRemoteFromConfiguration;
	RequestStream<struct RecruitStorageRequest> recruitStorage;
	RequestStream<struct RegisterWorkerRequest> registerWorker;
	RequestStream<struct GetWorkersRequest> getWorkers;
	RequestStream<struct RegisterMasterRequest> registerMaster;
	RequestStream<struct GetServerDBInfoRequest>
	    getServerDBInfo; // only used by testers; the cluster controller will send the serverDBInfo to workers
	RequestStream<struct UpdateWorkerHealthRequest> updateWorkerHealth;
	RequestStream<struct TLogRejoinRequest>
	    tlogRejoin; // sent by tlog (whether or not rebooted) to communicate with a new controller
	RequestStream<struct BackupWorkerDoneRequest> notifyBackupWorkerDone;
	RequestStream<struct ChangeCoordinatorsRequest> changeCoordinators;
	RequestStream<struct GetEncryptionAtRestModeRequest> getEncryptionAtRestMode;

	UID id() const { return clientInterface.id(); }
	bool operator==(ClusterControllerFullInterface const& r) const { return id() == r.id(); }
	bool operator!=(ClusterControllerFullInterface const& r) const { return id() != r.id(); }

	NetworkAddress address() const { return clientInterface.address(); }

	bool hasMessage() const {
		return clientInterface.hasMessage() || recruitFromConfiguration.getFuture().isReady() ||
		       recruitRemoteFromConfiguration.getFuture().isReady() || recruitStorage.getFuture().isReady() ||
		       registerWorker.getFuture().isReady() || getWorkers.getFuture().isReady() ||
		       registerMaster.getFuture().isReady() || getServerDBInfo.getFuture().isReady() ||
		       updateWorkerHealth.getFuture().isReady() || tlogRejoin.getFuture().isReady() ||
		       notifyBackupWorkerDone.getFuture().isReady() || changeCoordinators.getFuture().isReady() ||
		       getEncryptionAtRestMode.getFuture().isReady();
	}

	void initEndpoints() {
		clientInterface.initEndpoints();
		recruitFromConfiguration.getEndpoint(TaskPriority::ClusterControllerRecruit);
		recruitRemoteFromConfiguration.getEndpoint(TaskPriority::ClusterControllerRecruit);
		recruitStorage.getEndpoint(TaskPriority::ClusterController);
		registerWorker.getEndpoint(TaskPriority::ClusterControllerWorker);
		getWorkers.getEndpoint(TaskPriority::ClusterController);
		registerMaster.getEndpoint(TaskPriority::ClusterControllerRegister);
		getServerDBInfo.getEndpoint(TaskPriority::ClusterController);
		updateWorkerHealth.getEndpoint(TaskPriority::ClusterController);
		tlogRejoin.getEndpoint(TaskPriority::MasterTLogRejoin);
		notifyBackupWorkerDone.getEndpoint(TaskPriority::ClusterController);
		changeCoordinators.getEndpoint(TaskPriority::DefaultEndpoint);
		getEncryptionAtRestMode.getEndpoint(TaskPriority::ClusterController);
	}

	template <class Ar>
	void serialize(Ar& ar) {
		if constexpr (!is_fb_function<Ar>) {
			ASSERT(ar.protocolVersion().isValid());
		}
		serializer(ar,
		           clientInterface,
		           recruitFromConfiguration,
		           recruitRemoteFromConfiguration,
		           recruitStorage,
		           registerWorker,
		           getWorkers,
		           registerMaster,
		           getServerDBInfo,
		           updateWorkerHealth,
		           tlogRejoin,
		           notifyBackupWorkerDone,
		           changeCoordinators,
		           getEncryptionAtRestMode);
	}
};

using AsyncVar_Optional_ClusterControllerFullInterface = AsyncVar<Optional<ClusterControllerFullInterface>>;

struct RegisterWorkerReply {
	constexpr static FileIdentifier file_identifier = 16475696;
	ProcessClass processClass;
	ClusterControllerPriorityInfo priorityInfo;

	RegisterWorkerReply()
	  : priorityInfo(ProcessClass::UnsetFit, false, ClusterControllerPriorityInfo::FitnessUnknown) {}
	RegisterWorkerReply(ProcessClass processClass, ClusterControllerPriorityInfo priorityInfo)
	  : processClass(processClass), priorityInfo(priorityInfo) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, processClass, priorityInfo);
	}
};

struct RegisterMasterRequest {
	constexpr static FileIdentifier file_identifier = 10773445;
	UID id;
	LocalityData mi;
	LogSystemConfig logSystemConfig;
	std::vector<CommitProxyInterface> commitProxies;
	std::vector<GrvProxyInterface> grvProxies;
	std::vector<ResolverInterface> resolvers;
	DBRecoveryCount recoveryCount;
	int64_t registrationCount;
	Optional<DatabaseConfiguration> configuration;
	std::vector<UID> priorCommittedLogServers;
	RecoveryState recoveryState;
	bool recoveryStalled;

	ReplyPromise<Void> reply;

	RegisterMasterRequest() : logSystemConfig(0) {}

	template <class Ar>
	void serialize(Ar& ar) {
		if constexpr (!is_fb_function<Ar>) {
			ASSERT(ar.protocolVersion().isValid());
		}
		serializer(ar,
		           id,
		           mi,
		           logSystemConfig,
		           commitProxies,
		           grvProxies,
		           resolvers,
		           recoveryCount,
		           registrationCount,
		           configuration,
		           priorCommittedLogServers,
		           recoveryState,
		           recoveryStalled,
		           reply);
	}
};

// Instantiated in worker.actor.cpp
extern template class RequestStream<RegisterMasterRequest, false>;
extern template struct NetNotifiedQueue<RegisterMasterRequest, false>;

struct RecruitFromConfigurationReply {
	constexpr static FileIdentifier file_identifier = 2224085;
	std::vector<WorkerInterface> backupWorkers;
	std::vector<WorkerInterface> tLogs;
	std::vector<WorkerInterface> satelliteTLogs;
	std::vector<WorkerInterface> commitProxies;
	std::vector<WorkerInterface> grvProxies;
	std::vector<WorkerInterface> resolvers;
	std::vector<WorkerInterface> storageServers;
	std::vector<WorkerInterface> oldLogRouters; // During recovery, log routers for older generations will be recruited.
	Optional<Key> dcId; // dcId is where master is recruited. It prefers to be in configuration.primaryDcId, but
	                    // it can be recruited from configuration.secondaryDc: The dcId will be the secondaryDcId and
	                    // this generation's primaryDC in memory is different from configuration.primaryDcId.
	bool satelliteFallback;

	RecruitFromConfigurationReply() : satelliteFallback(false) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           tLogs,
		           satelliteTLogs,
		           commitProxies,
		           grvProxies,
		           resolvers,
		           storageServers,
		           oldLogRouters,
		           dcId,
		           satelliteFallback,
		           backupWorkers);
	}
};

struct RecruitFromConfigurationRequest {
	constexpr static FileIdentifier file_identifier = 2023046;
	DatabaseConfiguration configuration;
	bool recruitSeedServers;
	int maxOldLogRouters;
	ReplyPromise<RecruitFromConfigurationReply> reply;

	RecruitFromConfigurationRequest() {}
	explicit RecruitFromConfigurationRequest(DatabaseConfiguration const& configuration,
	                                         bool recruitSeedServers,
	                                         int maxOldLogRouters)
	  : configuration(configuration), recruitSeedServers(recruitSeedServers), maxOldLogRouters(maxOldLogRouters) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, configuration, recruitSeedServers, maxOldLogRouters, reply);
	}
};

struct RecruitRemoteFromConfigurationReply {
	constexpr static FileIdentifier file_identifier = 9091392;
	std::vector<WorkerInterface> remoteTLogs;
	std::vector<WorkerInterface> logRouters;
	Optional<UID> dbgId;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, remoteTLogs, logRouters, dbgId);
	}
};

struct RecruitRemoteFromConfigurationRequest {
	constexpr static FileIdentifier file_identifier = 3235995;
	DatabaseConfiguration configuration;
	Optional<Key> dcId;
	int logRouterCount;
	std::vector<UID> exclusionWorkerIds;
	Optional<UID> dbgId;
	ReplyPromise<RecruitRemoteFromConfigurationReply> reply;

	RecruitRemoteFromConfigurationRequest() {}
	RecruitRemoteFromConfigurationRequest(DatabaseConfiguration const& configuration,
	                                      Optional<Key> const& dcId,
	                                      int logRouterCount,
	                                      const std::vector<UID>& exclusionWorkerIds)
	  : configuration(configuration), dcId(dcId), logRouterCount(logRouterCount),
	    exclusionWorkerIds(exclusionWorkerIds) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, configuration, dcId, logRouterCount, exclusionWorkerIds, dbgId, reply);
	}
};

struct RecruitStorageReply {
	constexpr static FileIdentifier file_identifier = 15877089;
	WorkerInterface worker;
	ProcessClass processClass;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, worker, processClass);
	}
};

struct RecruitStorageRequest {
	constexpr static FileIdentifier file_identifier = 905920;
	std::vector<Optional<Standalone<StringRef>>> excludeMachines; //< Don't recruit any of these machines
	std::vector<AddressExclusion> excludeAddresses; //< Don't recruit any of these addresses
	std::vector<Optional<Standalone<StringRef>>> includeDCs;
	bool criticalRecruitment; //< True if machine classes are to be ignored
	ReplyPromise<RecruitStorageReply> reply;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, excludeMachines, excludeAddresses, includeDCs, criticalRecruitment, reply);
	}
};

struct RegisterWorkerRequest {
	constexpr static FileIdentifier file_identifier = 14332605;
	WorkerInterface wi;
	ProcessClass initialClass;
	ProcessClass processClass;
	ClusterControllerPriorityInfo priorityInfo;
	Generation generation;
	Optional<DataDistributorInterface> distributorInterf;
	Optional<RatekeeperInterface> ratekeeperInterf;
	Optional<EncryptKeyProxyInterface> encryptKeyProxyInterf;
	Optional<ConsistencyScanInterface> consistencyScanInterf;
	Standalone<VectorRef<StringRef>> issues;
	std::vector<NetworkAddress> incompatiblePeers;
	ReplyPromise<RegisterWorkerReply> reply;
	bool degraded;
	Optional<Version> lastSeenKnobVersion;
	Optional<ConfigClassSet> knobConfigClassSet;
	bool requestDbInfo;
	bool recoveredDiskFiles;
	ConfigBroadcastInterface configBroadcastInterface;
	Optional<UID> clusterId;

	RegisterWorkerRequest()
	  : priorityInfo(ProcessClass::UnsetFit, false, ClusterControllerPriorityInfo::FitnessUnknown), degraded(false) {}
	RegisterWorkerRequest(WorkerInterface wi,
	                      ProcessClass initialClass,
	                      ProcessClass processClass,
	                      ClusterControllerPriorityInfo priorityInfo,
	                      Generation generation,
	                      Optional<DataDistributorInterface> ddInterf,
	                      Optional<RatekeeperInterface> rkInterf,
	                      Optional<EncryptKeyProxyInterface> ekpInterf,
	                      Optional<ConsistencyScanInterface> csInterf,
	                      bool degraded,
	                      Optional<Version> lastSeenKnobVersion,
	                      Optional<ConfigClassSet> knobConfigClassSet,
	                      bool recoveredDiskFiles,
	                      ConfigBroadcastInterface configBroadcastInterface,
	                      Optional<UID> clusterId)
	  : wi(wi), initialClass(initialClass), processClass(processClass), priorityInfo(priorityInfo),
	    generation(generation), distributorInterf(ddInterf), ratekeeperInterf(rkInterf),
	    encryptKeyProxyInterf(ekpInterf), consistencyScanInterf(csInterf), degraded(degraded),
	    lastSeenKnobVersion(lastSeenKnobVersion), knobConfigClassSet(knobConfigClassSet), requestDbInfo(false),
	    recoveredDiskFiles(recoveredDiskFiles), configBroadcastInterface(configBroadcastInterface),
	    clusterId(clusterId) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           wi,
		           initialClass,
		           processClass,
		           priorityInfo,
		           generation,
		           distributorInterf,
		           ratekeeperInterf,
		           encryptKeyProxyInterf,
		           consistencyScanInterf,
		           issues,
		           incompatiblePeers,
		           reply,
		           degraded,
		           lastSeenKnobVersion,
		           knobConfigClassSet,
		           requestDbInfo,
		           recoveredDiskFiles,
		           configBroadcastInterface,
		           clusterId);
	}
};

struct GetWorkersRequest {
	constexpr static FileIdentifier file_identifier = 1254174;
	enum { TESTER_CLASS_ONLY = 0x1, NON_EXCLUDED_PROCESSES_ONLY = 0x2 };

	int flags;
	ReplyPromise<std::vector<WorkerDetails>> reply;

	GetWorkersRequest() : flags(0) {}
	explicit GetWorkersRequest(int fl) : flags(fl) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, flags, reply);
	}
};

struct UpdateWorkerHealthRequest {
	constexpr static FileIdentifier file_identifier = 5789927;
	NetworkAddress address;
	std::vector<NetworkAddress> degradedPeers;
	std::vector<NetworkAddress> disconnectedPeers;
	std::vector<NetworkAddress> recoveredPeers;

	template <class Ar>
	void serialize(Ar& ar) {
		if constexpr (!is_fb_function<Ar>) {
			ASSERT(ar.protocolVersion().isValid());
		}
		serializer(ar, address, degradedPeers, disconnectedPeers, recoveredPeers);
	}
};

struct TLogRejoinReply {
	constexpr static FileIdentifier file_identifier = 11;

	// false means someone else registered, so we should re-register.  true means this master is recovered, so don't
	// send again to the same master.
	bool masterIsRecovered;
	TLogRejoinReply() = default;
	explicit TLogRejoinReply(bool masterIsRecovered) : masterIsRecovered(masterIsRecovered) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, masterIsRecovered);
	}
};

struct TLogRejoinRequest {
	constexpr static FileIdentifier file_identifier = 15692200;
	TLogInterface myInterface;
	ReplyPromise<TLogRejoinReply> reply;

	TLogRejoinRequest() {}
	explicit TLogRejoinRequest(const TLogInterface& interf) : myInterface(interf) {}
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, myInterface, reply);
	}
};

struct BackupWorkerDoneRequest {
	constexpr static FileIdentifier file_identifier = 8736351;
	UID workerUID;
	LogEpoch backupEpoch;
	ReplyPromise<Void> reply;

	BackupWorkerDoneRequest() : workerUID(), backupEpoch(-1) {}
	BackupWorkerDoneRequest(UID id, LogEpoch epoch) : workerUID(id), backupEpoch(epoch) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, workerUID, backupEpoch, reply);
	}
};

struct GetEncryptionAtRestModeResponse {
	constexpr static FileIdentifier file_identifier = 2932156;
	uint32_t mode;

	GetEncryptionAtRestModeResponse() : mode(EncryptionAtRestMode::Mode::DISABLED) {}
	GetEncryptionAtRestModeResponse(uint32_t m) : mode(m) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, mode);
	}
};

struct GetEncryptionAtRestModeRequest {
	constexpr static FileIdentifier file_identifier = 2670826;
	UID tlogId;
	ReplyPromise<GetEncryptionAtRestModeResponse> reply;

	GetEncryptionAtRestModeRequest() {}
	GetEncryptionAtRestModeRequest(UID tId) : tlogId(tId) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, tlogId, reply);
	}
};

struct InitializeTLogRequest {
	constexpr static FileIdentifier file_identifier = 15604392;
	UID recruitmentID;
	LogSystemConfig recoverFrom;
	Version recoverAt;
	Version knownCommittedVersion;
	LogEpoch epoch;
	std::vector<Tag> recoverTags;
	std::vector<Tag> allTags;
	TLogVersion logVersion;
	KeyValueStoreType storeType;
	TLogSpillType spillType;
	Tag remoteTag;
	int8_t locality;
	bool isPrimary;
	Version startVersion;
	int logRouterTags;
	int txsTags;
	Version recoveryTransactionVersion;
	std::vector<Version> oldGenerationRecoverAtVersions;

	ReplyPromise<struct TLogInterface> reply;

	InitializeTLogRequest() : recoverFrom(0) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           recruitmentID,
		           recoverFrom,
		           recoverAt,
		           knownCommittedVersion,
		           epoch,
		           recoverTags,
		           allTags,
		           storeType,
		           remoteTag,
		           locality,
		           isPrimary,
		           startVersion,
		           logRouterTags,
		           reply,
		           logVersion,
		           spillType,
		           txsTags,
		           recoveryTransactionVersion,
		           oldGenerationRecoverAtVersions);
	}
};

struct InitializeLogRouterRequest {
	constexpr static FileIdentifier file_identifier = 2976228;
	uint64_t recoveryCount;
	Tag routerTag;
	Version startVersion;
	std::vector<LocalityData> tLogLocalities;
	Reference<IReplicationPolicy> tLogPolicy;
	int8_t locality;
	ReplyPromise<struct TLogInterface> reply;
	Optional<Version> recoverAt = Optional<Version>();
	// @todo investigate whether we really need to propagate the known locked
	// tLog list as part of this request (or the log router can obtain this
	// information from the logSystem).
	Optional<std::map<uint8_t, std::vector<uint16_t>>> knownLockedTLogIds =
	    Optional<std::map<uint8_t, std::vector<uint16_t>>>();
	bool allowDropInSim; // Simulation-only field for fault injection testing. When true, allows the worker to
	                     // selectively drop responses to initialization messages to test recovery behavior under
	                     // partial failures. Must only be true in simulation.

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           recoveryCount,
		           routerTag,
		           startVersion,
		           tLogLocalities,
		           tLogPolicy,
		           locality,
		           reply,
		           recoverAt,
		           knownLockedTLogIds,
		           allowDropInSim);
	}
};

struct InitializeBackupReply {
	constexpr static FileIdentifier file_identifier = 13511909;
	struct BackupInterface interf;
	LogEpoch backupEpoch;

	InitializeBackupReply() = default;
	InitializeBackupReply(BackupInterface bi, LogEpoch e) : interf(bi), backupEpoch(e) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, interf, backupEpoch);
	}
};

struct InitializeBackupRequest {
	constexpr static FileIdentifier file_identifier = 1245415;
	UID reqId;
	LogEpoch recruitedEpoch; // The epoch the worker is recruited.
	LogEpoch backupEpoch; // The epoch the worker should work on. If different from the recruitedEpoch, then it refers
	                      // to some previous epoch with unfinished work.
	Tag routerTag;
	int totalTags;
	Version startVersion;
	Optional<Version> endVersion;
	ReplyPromise<struct InitializeBackupReply> reply;

	InitializeBackupRequest() = default;
	explicit InitializeBackupRequest(UID id) : reqId(id) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reqId, recruitedEpoch, backupEpoch, routerTag, totalTags, startVersion, endVersion, reply);
	}
};

// FIXME: Rename to InitializeMasterRequest, etc
struct RecruitMasterRequest {
	constexpr static FileIdentifier file_identifier = 12684574;
	LifetimeToken lifetime;
	bool forceRecovery;
	ReplyPromise<struct MasterInterface> reply;

	template <class Ar>
	void serialize(Ar& ar) {
		if constexpr (!is_fb_function<Ar>) {
			ASSERT(ar.protocolVersion().isValid());
		}
		serializer(ar, lifetime, forceRecovery, reply);
	}
};

// Instantiated in worker.actor.cpp
extern template class RequestStream<RecruitMasterRequest, false>;
extern template struct NetNotifiedQueue<RecruitMasterRequest, false>;

struct InitializeCommitProxyRequest {
	constexpr static FileIdentifier file_identifier = 10344153;
	MasterInterface master;
	LifetimeToken masterLifetime;
	uint64_t recoveryCount;
	Version recoveryTransactionVersion;
	bool firstProxy;
	ReplyPromise<CommitProxyInterface> reply;
	EncryptionAtRestMode encryptMode;
	uint16_t commitProxyIndex;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar,
		           master,
		           masterLifetime,
		           recoveryCount,
		           recoveryTransactionVersion,
		           firstProxy,
		           reply,
		           encryptMode,
		           commitProxyIndex);
	}
};

// Instantiated in worker.actor.cpp
extern template class RequestStream<InitializeCommitProxyRequest, false>;
extern template struct NetNotifiedQueue<InitializeCommitProxyRequest, false>;

struct InitializeGrvProxyRequest {
	constexpr static FileIdentifier file_identifier = 8265613;
	MasterInterface master;
	LifetimeToken masterLifetime;
	uint64_t recoveryCount;
	ReplyPromise<GrvProxyInterface> reply;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, master, masterLifetime, recoveryCount, reply);
	}
};

// Instantiated in worker.actor.cpp
extern template class RequestStream<InitializeGrvProxyRequest, false>;
extern template struct NetNotifiedQueue<InitializeGrvProxyRequest, false>;

struct InitializeDataDistributorRequest {
	constexpr static FileIdentifier file_identifier = 8858952;
	UID reqId;
	ReplyPromise<DataDistributorInterface> reply;

	InitializeDataDistributorRequest() {}
	explicit InitializeDataDistributorRequest(UID uid) : reqId(uid) {}
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reqId, reply);
	}
};

struct InitializeRatekeeperRequest {
	constexpr static FileIdentifier file_identifier = 6416816;
	UID reqId;
	ReplyPromise<RatekeeperInterface> reply;

	InitializeRatekeeperRequest() {}
	explicit InitializeRatekeeperRequest(UID uid) : reqId(uid) {}
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reqId, reply);
	}
};

struct InitializeConsistencyScanRequest {
	constexpr static FileIdentifier file_identifier = 3104275;
	UID reqId;
	ReplyPromise<ConsistencyScanInterface> reply;

	InitializeConsistencyScanRequest() {}
	explicit InitializeConsistencyScanRequest(UID uid) : reqId(uid) {}
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reqId, reply);
	}
};

struct InitializeResolverRequest {
	constexpr static FileIdentifier file_identifier = 7413317;
	LifetimeToken masterLifetime;
	uint64_t recoveryCount;
	int commitProxyCount;
	int resolverCount;
	UID masterId; // master's UID
	ReplyPromise<ResolverInterface> reply;
	EncryptionAtRestMode encryptMode;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, masterLifetime, recoveryCount, commitProxyCount, resolverCount, masterId, reply, encryptMode);
	}
};

struct InitializeStorageReply {
	constexpr static FileIdentifier file_identifier = 10390645;
	StorageServerInterface interf;
	Version addedVersion;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, interf, addedVersion);
	}
};

struct InitializeStorageRequest {
	constexpr static FileIdentifier file_identifier = 16665642;
	Tag seedTag; //< If this server will be passed to seedShardServers, this will be a tag, otherwise it is invalidTag
	UID reqId;
	UID interfaceId;
	KeyValueStoreType storeType;
	Optional<std::pair<UID, Version>>
	    tssPairIDAndVersion; // Only set if recruiting a tss. Will be the UID and Version of its SS pair.
	Version initialClusterVersion;
	ReplyPromise<InitializeStorageReply> reply;
	EncryptionAtRestMode encryptMode;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(
		    ar, seedTag, reqId, interfaceId, storeType, reply, tssPairIDAndVersion, initialClusterVersion, encryptMode);
	}
};

struct InitializeEncryptKeyProxyRequest {
	constexpr static FileIdentifier file_identifier = 4180191;
	UID reqId;
	UID interfaceId;
	ReplyPromise<EncryptKeyProxyInterface> reply;
	EncryptionAtRestMode encryptMode;

	InitializeEncryptKeyProxyRequest() {}
	explicit InitializeEncryptKeyProxyRequest(UID uid) : reqId(uid) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reqId, interfaceId, reply, encryptMode);
	}
};

struct TraceBatchDumpRequest {
	constexpr static FileIdentifier file_identifier = 8184121;
	ReplyPromise<Void> reply;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reply);
	}
};

struct ExecuteRequest {
	constexpr static FileIdentifier file_identifier = 8184128;
	ReplyPromise<Void> reply;

	Arena arena;
	StringRef execPayload;

	ExecuteRequest(StringRef execPayload) : execPayload(execPayload) {}

	ExecuteRequest() : execPayload() {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reply, execPayload, arena);
	}
};

// FIXME: describe purpose
struct WorkerSnapRequest {
	constexpr static FileIdentifier file_identifier = 8194122;
	ReplyPromise<Void> reply;
	Arena arena;
	StringRef snapPayload;
	UID snapUID;
	StringRef role;

	WorkerSnapRequest(StringRef snapPayload, UID snapUID, StringRef role)
	  : snapPayload(snapPayload), snapUID(snapUID), role(role) {}
	WorkerSnapRequest() = default;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reply, snapPayload, snapUID, role, arena);
	}
};

struct LoadedReply {
	constexpr static FileIdentifier file_identifier = 9956350;
	Standalone<StringRef> payload;
	UID id;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, payload, id);
	}
};

struct LoadedPingRequest {
	constexpr static FileIdentifier file_identifier = 4590979;
	UID id;
	bool loadReply;
	Standalone<StringRef> payload;
	ReplyPromise<LoadedReply> reply;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, id, loadReply, payload, reply);
	}
};

struct CoordinationPingMessage {
	constexpr static FileIdentifier file_identifier = 9982747;
	UID clusterControllerId;
	int64_t timeStep;

	CoordinationPingMessage() : timeStep(0) {}
	CoordinationPingMessage(UID ccId, uint64_t step) : clusterControllerId(ccId), timeStep(step) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, clusterControllerId, timeStep);
	}
};

struct SetMetricsLogRateRequest {
	constexpr static FileIdentifier file_identifier = 4245995;
	uint32_t metricsLogsPerSecond;

	SetMetricsLogRateRequest() : metricsLogsPerSecond(1) {}
	explicit SetMetricsLogRateRequest(uint32_t logsPerSecond) : metricsLogsPerSecond(logsPerSecond) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, metricsLogsPerSecond);
	}
};

struct EventLogRequest {
	constexpr static FileIdentifier file_identifier = 122319;
	bool getLastError;
	Standalone<StringRef> eventName;
	ReplyPromise<TraceEventFields> reply;

	EventLogRequest() : getLastError(true) {}
	explicit EventLogRequest(Standalone<StringRef> eventName) : getLastError(false), eventName(eventName) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, getLastError, eventName, reply);
	}
};

struct DebugEntryRef {
	double time;
	NetworkAddress address;
	StringRef context;
	Version version;
	MutationRef mutation;
	DebugEntryRef() {}
	DebugEntryRef(const char* c, Version v, MutationRef const& m)
	  : time(now()), address(g_network->getLocalAddress()), context((const uint8_t*)c, strlen(c)), version(v),
	    mutation(m) {}
	DebugEntryRef(Arena& a, DebugEntryRef const& d)
	  : time(d.time), address(d.address), context(d.context), version(d.version), mutation(a, d.mutation) {}

	size_t expectedSize() const { return context.expectedSize() + mutation.expectedSize(); }

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, time, address, context, version, mutation);
	}
};

struct DiskStoreRequest {
	constexpr static FileIdentifier file_identifier = 1986262;
	bool includePartialStores;
	ReplyPromise<Standalone<VectorRef<UID>>> reply;

	DiskStoreRequest(bool includePartialStores = false) : includePartialStores(includePartialStores) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, includePartialStores, reply);
	}
};

struct Role {
	static const Role WORKER;
	static const Role STORAGE_SERVER;
	static const Role TESTING_STORAGE_SERVER;
	static const Role TRANSACTION_LOG;
	static const Role SHARED_TRANSACTION_LOG;
	static const Role COMMIT_PROXY;
	static const Role GRV_PROXY;
	static const Role MASTER;
	static const Role RESOLVER;
	static const Role CLUSTER_CONTROLLER;
	static const Role TESTER;
	static const Role LOG_ROUTER;
	static const Role DATA_DISTRIBUTOR;
	static const Role RATEKEEPER;
	static const Role STORAGE_CACHE;
	static const Role COORDINATOR;
	static const Role BACKUP;
	static const Role ENCRYPT_KEY_PROXY;
	static const Role CONSISTENCYSCAN;

	std::string roleName;
	std::string abbreviation;
	bool includeInTraceRoles;

	static const Role& get(ProcessClass::ClusterRole role) {
		switch (role) {
		case ProcessClass::Storage:
			return STORAGE_SERVER;
		case ProcessClass::TLog:
			return TRANSACTION_LOG;
		case ProcessClass::CommitProxy:
			return COMMIT_PROXY;
		case ProcessClass::GrvProxy:
			return GRV_PROXY;
		case ProcessClass::Master:
			return MASTER;
		case ProcessClass::Resolver:
			return RESOLVER;
		case ProcessClass::LogRouter:
			return LOG_ROUTER;
		case ProcessClass::ClusterController:
			return CLUSTER_CONTROLLER;
		case ProcessClass::DataDistributor:
			return DATA_DISTRIBUTOR;
		case ProcessClass::Ratekeeper:
			return RATEKEEPER;
		case ProcessClass::StorageCache:
			return STORAGE_CACHE;
		case ProcessClass::Backup:
			return BACKUP;
		case ProcessClass::EncryptKeyProxy:
			return ENCRYPT_KEY_PROXY;
		case ProcessClass::ConsistencyScan:
			return CONSISTENCYSCAN;
		case ProcessClass::Worker:
			return WORKER;
		case ProcessClass::NoRole:
		default:
			ASSERT(false);
			throw internal_error();
		}
	}

	bool operator==(const Role& r) const { return roleName == r.roleName; }
	bool operator!=(const Role& r) const { return !(*this == r); }

private:
	Role(std::string roleName, std::string abbreviation, bool includeInTraceRoles = true)
	  : roleName(roleName), abbreviation(abbreviation), includeInTraceRoles(includeInTraceRoles) {
		ASSERT(abbreviation.size() == 2); // Having a fixed size makes log queries more straightforward
	}
};

void startRole(const Role& role,
               UID roleId,
               UID workerId,
               const std::map<std::string, std::string>& details = std::map<std::string, std::string>(),
               const std::string& origination = "Recruited");
void endRole(const Role& role, UID id, std::string reason, bool ok = true, Error e = Error());

															#line 1088 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> traceRole( Role const& role, UID const& roleId );

#line 1087 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

struct ServerDBInfo;

class Database openDBOnServer(Reference<AsyncVar<ServerDBInfo> const> const& db,
                              TaskPriority taskID = TaskPriority::DefaultEndpoint,
                              LockAware = LockAware::False,
                              EnableLocalityLoadBalance = EnableLocalityLoadBalance::True);
															#line 1099 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> extractClusterInterface( Reference<AsyncVar<Optional<struct ClusterControllerFullInterface>> const> const& in, Reference<AsyncVar<Optional<struct ClusterInterface>>> const& out );

#line 1097 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1104 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> fdbd( Reference<IClusterConnectionRecord> const& ccr, LocalityData const& localities, ProcessClass const& processClass, std::string const& dataFolder, std::string const& coordFolder, int64_t const& memoryLimit, std::string const& metricsConnFile, std::string const& metricsPrefix, int64_t const& memoryProfilingThreshold, std::string const& whitelistBinPaths, std::string const& configPath, std::map<std::string, std::string> const& manualKnobOverrides, ConfigDBType const& configDBType, bool const& consistencyCheckUrgentMode );

#line 1112 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1109 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> clusterController( Reference<IClusterConnectionRecord> const& ccr, Reference<AsyncVar<Optional<ClusterControllerFullInterface>>> const& currentCC, Reference<AsyncVar<ClusterControllerPriorityInfo>> const& asyncPriorityInfo, LocalityData const& locality, ConfigDBType const& configDBType, Reference<AsyncVar<Optional<UID>>> const& clusterId );

#line 1119 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

// These servers are started by workerServer
class IKeyValueStore;
class ServerCoordinators;
class IDiskQueue;

															#line 1119 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> encryptKeyProxyServer( EncryptKeyProxyInterface const& ei, Reference<AsyncVar<ServerDBInfo>> const& db, EncryptionAtRestMode const& encryptMode );

#line 1128 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1124 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> storageServer( IKeyValueStore* const& persistentData, StorageServerInterface const& ssi, Tag const& seedTag, Version const& startVersion, Version const& tssSeedVersion, ReplyPromise<InitializeStorageReply> const& recruitReply, Reference<AsyncVar<ServerDBInfo> const> const& db, std::string const& folder, Reference<GetEncryptCipherKeysMonitor> const& encryptionMonitor );

#line 1138 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1128 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> storageServer( IKeyValueStore* const& persistentData, StorageServerInterface const& ssi, Reference<AsyncVar<ServerDBInfo> const> const& db, std::string const& folder, Promise<Void> const& recovered, Reference<IClusterConnectionRecord> const& connRecord, Reference<GetEncryptCipherKeysMonitor> const& encryptionMonitor );

#line 1145 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
// changes pssi->id() to be the recovered ID
															#line 1133 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> masterServer( MasterInterface const& mi, Reference<AsyncVar<ServerDBInfo> const> const& db, Reference<AsyncVar<Optional<ClusterControllerFullInterface>> const> const& ccInterface, ServerCoordinators const& serverCoordinators, LifetimeToken const& lifetime, bool const& forceRecovery );

#line 1152 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1137 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> commitProxyServer( CommitProxyInterface const& proxy, InitializeCommitProxyRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db, std::string const& whitelistBinPaths );

#line 1156 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1141 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> grvProxyServer( GrvProxyInterface const& proxy, InitializeGrvProxyRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db );

#line 1159 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1145 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> tLog( IKeyValueStore* const& persistentData, IDiskQueue* const& persistentQueue, Reference<AsyncVar<ServerDBInfo> const> const& db, LocalityData const& locality, PromiseStream<InitializeTLogRequest> const& tlogRequests, UID const& tlogId, UID const& workerID, bool const& restoreFromDisk, Promise<Void> const& oldLog, Promise<Void> const& recovered, std::string const& folder, Reference<AsyncVar<bool>> const& degraded, Reference<AsyncVar<UID>> const& activeSharedTLog, Reference<AsyncVar<bool>> const& enablePrimaryTxnSystemHealthCheck );

#line 1173 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1149 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> resolver( ResolverInterface const& resolver, InitializeResolverRequest const& initReq, Reference<AsyncVar<ServerDBInfo> const> const& db );

#line 1176 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
Future<Void> logRouter(TLogInterface interf,
                       InitializeLogRouterRequest req,
                       Reference<AsyncVar<ServerDBInfo> const> db);
Future<Void> dataDistributor(DataDistributorInterface ddi,
                             Reference<AsyncVar<ServerDBInfo> const> db,
                             std::string folder);
															#line 1159 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> ratekeeper( RatekeeperInterface const& rki, Reference<AsyncVar<ServerDBInfo> const> const& db );

#line 1183 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1163 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> consistencyScan( ConsistencyScanInterface const& csInterf, Reference<AsyncVar<ServerDBInfo> const> const& dbInfo );

#line 1184 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1168 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> storageCacheServer( StorageServerInterface const& interf, uint16_t const& id, Reference<AsyncVar<ServerDBInfo> const> const& db );

#line 1188 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1172 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
[[nodiscard]] Future<Void> backupWorker( BackupInterface const& bi, InitializeBackupRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db );

#line 1191 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

void registerThreadForProfiling();

// Returns true if `address` is used in the db (indicated by `dbInfo`) transaction system and in the db's primary
// satellite DC.
bool addressInDbAndPrimarySatelliteDc(const NetworkAddress& address, Reference<AsyncVar<ServerDBInfo> const> dbInfo);

// Returns true if `address` is used in the db (indicated by `dbInfo`) transaction system and in the db's remote DC.
bool addressInDbAndRemoteDc(
    const NetworkAddress& address,
    Reference<AsyncVar<ServerDBInfo> const> dbInfo,
    Optional<std::vector<NetworkAddress>> storageServers = Optional<std::vector<NetworkAddress>>{});

void updateCpuProfiler(ProfilerRequest req);

typedef decltype(&tLog) TLogFn;

extern bool isSimulatorProcessUnreliable();

															#line 1195 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
namespace {
// This generated class is to be used only via ioTimeoutError()
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T, class IoTimeoutErrorActor>
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoTimeoutErrorActorState {
															#line 1202 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoTimeoutErrorActorState(Future<T> const& what,double const& time,const char* const& context = nullptr) 
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		 : what(what),
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   time(time),
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   context(context),
															#line 1214 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   orig(now()),
															#line 1215 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   trace(platform::get_backtrace())
															#line 1217 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
	{
		fdb_probe_actor_create("ioTimeoutError", reinterpret_cast<unsigned long>(this));

	}
	~IoTimeoutErrorActorState() 
	{
		fdb_probe_actor_destroy("ioTimeoutError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1216 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (g_network->isSimulated() && !g_simulator->speedUpSimulation)
															#line 1232 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			{
															#line 1217 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				time += std::max(0.0, FLOW_KNOBS->SIM_SPEEDUP_AFTER_SECONDS - now());
															#line 1236 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			}
															#line 1219 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			Future<Void> end = lowPriorityDelay(time);
															#line 1221 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			StrictFuture<T> __when_expr_0 = what;
															#line 1220 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (static_cast<IoTimeoutErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1244 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 1224 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			StrictFuture<Void> __when_expr_1 = end;
															#line 1248 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<IoTimeoutErrorActor*>(this)->actor_wait_state = 1;
															#line 1221 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IoTimeoutErrorActor, 0, T >*>(static_cast<IoTimeoutErrorActor*>(this)));
															#line 1224 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IoTimeoutErrorActor, 1, Void >*>(static_cast<IoTimeoutErrorActor*>(this)));
															#line 1255 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IoTimeoutErrorActorState();
		static_cast<IoTimeoutErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
															#line 1222 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoTimeoutErrorActorState(); static_cast<IoTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 1278 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoTimeoutErrorActorState();
		static_cast<IoTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
															#line 1222 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoTimeoutErrorActorState(); static_cast<IoTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 1290 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoTimeoutErrorActorState();
		static_cast<IoTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 1225 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1226 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 1304 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1227 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 1308 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1229 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1230 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1231 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 1316 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1232 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 1320 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1234 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.detail("OrigTime", orig).detail("OrigTrace", trace).log();
															#line 1235 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 1326 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 1225 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1226 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 1336 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1227 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 1340 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1229 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1230 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1231 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 1348 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1232 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 1352 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1234 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.detail("OrigTime", orig).detail("OrigTrace", trace).log();
															#line 1235 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 1358 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IoTimeoutErrorActor*>(this)->actor_wait_state > 0) static_cast<IoTimeoutErrorActor*>(this)->actor_wait_state = 0;
		static_cast<IoTimeoutErrorActor*>(this)->ActorCallback< IoTimeoutErrorActor, 0, T >::remove();
		static_cast<IoTimeoutErrorActor*>(this)->ActorCallback< IoTimeoutErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IoTimeoutErrorActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IoTimeoutErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	Future<T> what;
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	double time;
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	const char* context;
															#line 1214 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	double orig;
															#line 1215 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	std::string trace;
															#line 1493 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
};
// This generated class is to be used only via ioTimeoutError()
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoTimeoutErrorActor final : public Actor<T>, public ActorCallback< IoTimeoutErrorActor<T>, 0, T >, public ActorCallback< IoTimeoutErrorActor<T>, 1, Void >, public FastAllocated<IoTimeoutErrorActor<T>>, public IoTimeoutErrorActorState<T, IoTimeoutErrorActor<T>> {
															#line 1500 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
	using FastAllocated<IoTimeoutErrorActor<T>>::operator new;
	using FastAllocated<IoTimeoutErrorActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15941304087807223808UL, 11838612762633837312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IoTimeoutErrorActor<T>, 0, T >;
friend struct ActorCallback< IoTimeoutErrorActor<T>, 1, Void >;
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoTimeoutErrorActor(Future<T> const& what,double const& time,const char* const& context = nullptr) 
															#line 1518 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		 : Actor<T>(),
		   IoTimeoutErrorActorState<T, IoTimeoutErrorActor<T>>(what, time, context),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("ioTimeoutError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6479263997658130944UL, 15483549950528013824UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("ioTimeoutError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("ioTimeoutError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IoTimeoutErrorActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
[[nodiscard]] Future<T> ioTimeoutError( Future<T> const& what, double const& time, const char* const& context = nullptr ) {
															#line 1210 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	return Future<T>(new IoTimeoutErrorActor<T>(what, time, context));
															#line 1553 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
}

#line 1239 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1558 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
namespace {
// This generated class is to be used only via ioTimeoutErrorIfCleared()
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T, class IoTimeoutErrorIfClearedActor>
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoTimeoutErrorIfClearedActorState {
															#line 1565 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoTimeoutErrorIfClearedActorState(Future<T> const& what,double const& time,Reference<AsyncVar<bool>> const& condition,const char* const& context = nullptr) 
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		 : what(what),
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   time(time),
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   condition(condition),
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   context(context)
															#line 1578 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
	{
		fdb_probe_actor_create("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this));

	}
	~IoTimeoutErrorIfClearedActorState() 
	{
		fdb_probe_actor_destroy("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1247 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (g_network->isSimulated() && !g_simulator->speedUpSimulation)
															#line 1593 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			{
															#line 1248 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				time += std::max(0.0, FLOW_KNOBS->SIM_SPEEDUP_AFTER_SECONDS - now());
															#line 1597 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			}
															#line 1250 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			Future<Void> end = lowPriorityDelayAfterCleared(condition, time);
															#line 1252 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			StrictFuture<T> __when_expr_0 = what;
															#line 1251 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (static_cast<IoTimeoutErrorIfClearedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1605 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 1255 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			StrictFuture<Void> __when_expr_1 = end;
															#line 1609 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<IoTimeoutErrorIfClearedActor*>(this)->actor_wait_state = 1;
															#line 1252 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IoTimeoutErrorIfClearedActor, 0, T >*>(static_cast<IoTimeoutErrorIfClearedActor*>(this)));
															#line 1255 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IoTimeoutErrorIfClearedActor, 1, Void >*>(static_cast<IoTimeoutErrorIfClearedActor*>(this)));
															#line 1616 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IoTimeoutErrorIfClearedActorState();
		static_cast<IoTimeoutErrorIfClearedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
															#line 1253 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoTimeoutErrorIfClearedActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoTimeoutErrorIfClearedActorState(); static_cast<IoTimeoutErrorIfClearedActor*>(this)->destroy(); return 0; }
															#line 1639 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoTimeoutErrorIfClearedActor*>(this)->SAV< T >::value()) T(t);
		this->~IoTimeoutErrorIfClearedActorState();
		static_cast<IoTimeoutErrorIfClearedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
															#line 1253 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoTimeoutErrorIfClearedActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoTimeoutErrorIfClearedActorState(); static_cast<IoTimeoutErrorIfClearedActor*>(this)->destroy(); return 0; }
															#line 1651 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoTimeoutErrorIfClearedActor*>(this)->SAV< T >::value()) T(t);
		this->~IoTimeoutErrorIfClearedActorState();
		static_cast<IoTimeoutErrorIfClearedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 1256 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1257 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 1665 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1258 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 1669 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1260 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1261 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1262 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 1677 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1263 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 1681 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1265 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.log();
															#line 1266 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 1687 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 1256 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1257 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 1697 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1258 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 1701 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1260 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1261 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1262 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 1709 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1263 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 1713 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1265 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.log();
															#line 1266 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 1719 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IoTimeoutErrorIfClearedActor*>(this)->actor_wait_state > 0) static_cast<IoTimeoutErrorIfClearedActor*>(this)->actor_wait_state = 0;
		static_cast<IoTimeoutErrorIfClearedActor*>(this)->ActorCallback< IoTimeoutErrorIfClearedActor, 0, T >::remove();
		static_cast<IoTimeoutErrorIfClearedActor*>(this)->ActorCallback< IoTimeoutErrorIfClearedActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorIfClearedActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorIfClearedActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IoTimeoutErrorIfClearedActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorIfClearedActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IoTimeoutErrorIfClearedActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IoTimeoutErrorIfClearedActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	Future<T> what;
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	double time;
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	Reference<AsyncVar<bool>> condition;
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	const char* context;
															#line 1852 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
};
// This generated class is to be used only via ioTimeoutErrorIfCleared()
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoTimeoutErrorIfClearedActor final : public Actor<T>, public ActorCallback< IoTimeoutErrorIfClearedActor<T>, 0, T >, public ActorCallback< IoTimeoutErrorIfClearedActor<T>, 1, Void >, public FastAllocated<IoTimeoutErrorIfClearedActor<T>>, public IoTimeoutErrorIfClearedActorState<T, IoTimeoutErrorIfClearedActor<T>> {
															#line 1859 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
	using FastAllocated<IoTimeoutErrorIfClearedActor<T>>::operator new;
	using FastAllocated<IoTimeoutErrorIfClearedActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17776586298800908800UL, 3587126548790489088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IoTimeoutErrorIfClearedActor<T>, 0, T >;
friend struct ActorCallback< IoTimeoutErrorIfClearedActor<T>, 1, Void >;
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoTimeoutErrorIfClearedActor(Future<T> const& what,double const& time,Reference<AsyncVar<bool>> const& condition,const char* const& context = nullptr) 
															#line 1877 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		 : Actor<T>(),
		   IoTimeoutErrorIfClearedActorState<T, IoTimeoutErrorIfClearedActor<T>>(what, time, condition, context),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14889594600743255040UL, 7700329930618080000UL);
		ActorExecutionContextHelper __helper(static_cast<IoTimeoutErrorIfClearedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("ioTimeoutErrorIfCleared");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("ioTimeoutErrorIfCleared", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IoTimeoutErrorIfClearedActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
[[nodiscard]] Future<T> ioTimeoutErrorIfCleared( Future<T> const& what, double const& time, Reference<AsyncVar<bool>> const& condition, const char* const& context = nullptr ) {
															#line 1240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	return Future<T>(new IoTimeoutErrorIfClearedActor<T>(what, time, condition, context));
															#line 1912 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
}

#line 1270 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

															#line 1917 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
namespace {
// This generated class is to be used only via ioDegradedOrTimeoutError()
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T, class IoDegradedOrTimeoutErrorActor>
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoDegradedOrTimeoutErrorActorState {
															#line 1924 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoDegradedOrTimeoutErrorActorState(Future<T> const& what,double const& errTime,Reference<AsyncVar<bool>> const& degraded,double const& degradedTime,const char* const& context = nullptr) 
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		 : what(what),
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   errTime(errTime),
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   degraded(degraded),
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   degradedTime(degradedTime),
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		   context(context)
															#line 1939 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
	{
		fdb_probe_actor_create("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this));

	}
	~IoDegradedOrTimeoutErrorActorState() 
	{
		fdb_probe_actor_destroy("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1279 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (g_network->isSimulated() && !g_simulator->speedUpSimulation)
															#line 1954 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			{
															#line 1280 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				double timeShift = std::max(0.0, FLOW_KNOBS->SIM_SPEEDUP_AFTER_SECONDS - now());
															#line 1281 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				errTime += timeShift;
															#line 1282 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				degradedTime += timeShift;
															#line 1962 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			}
															#line 1285 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			if (degradedTime < errTime)
															#line 1966 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
			{
															#line 1286 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				Future<Void> degradedEnd = lowPriorityDelay(degradedTime);
															#line 1288 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				StrictFuture<T> __when_expr_0 = what;
															#line 1287 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				if (static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1974 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 1291 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				StrictFuture<Void> __when_expr_1 = degradedEnd;
															#line 1978 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
				static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state = 1;
															#line 1288 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IoDegradedOrTimeoutErrorActor, 0, T >*>(static_cast<IoDegradedOrTimeoutErrorActor*>(this)));
															#line 1291 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IoDegradedOrTimeoutErrorActor, 1, Void >*>(static_cast<IoDegradedOrTimeoutErrorActor*>(this)));
															#line 1985 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IoDegradedOrTimeoutErrorActorState();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1299 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Future<Void> end = lowPriorityDelay(errTime - degradedTime);
															#line 1301 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		StrictFuture<T> __when_expr_2 = what;
															#line 1300 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2017 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
															#line 1304 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		StrictFuture<Void> __when_expr_3 = end;
															#line 2021 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when2(__when_expr_3.get(), loopDepth); };
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state = 2;
															#line 1301 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< IoDegradedOrTimeoutErrorActor, 2, T >*>(static_cast<IoDegradedOrTimeoutErrorActor*>(this)));
															#line 1304 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< IoDegradedOrTimeoutErrorActor, 3, Void >*>(static_cast<IoDegradedOrTimeoutErrorActor*>(this)));
															#line 2028 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
															#line 1289 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoDegradedOrTimeoutErrorActorState(); static_cast<IoDegradedOrTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 2043 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoDegradedOrTimeoutErrorActorState();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
															#line 1289 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoDegradedOrTimeoutErrorActorState(); static_cast<IoDegradedOrTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 2055 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoDegradedOrTimeoutErrorActorState();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 1292 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		CODE_PROBE(true, "TLog degraded", probe::func::deduplicate);
															#line 1293 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent(SevWarnAlways, "IoDegraded").log();
															#line 1294 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		degraded->set(true);
															#line 2071 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 1292 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		CODE_PROBE(true, "TLog degraded", probe::func::deduplicate);
															#line 1293 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent(SevWarnAlways, "IoDegraded").log();
															#line 1294 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		degraded->set(true);
															#line 2084 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state > 0) static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state = 0;
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->ActorCallback< IoDegradedOrTimeoutErrorActor, 0, T >::remove();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->ActorCallback< IoDegradedOrTimeoutErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IoDegradedOrTimeoutErrorActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IoDegradedOrTimeoutErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1when1(T const& t,int loopDepth) 
	{
															#line 1302 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoDegradedOrTimeoutErrorActorState(); static_cast<IoDegradedOrTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 2214 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoDegradedOrTimeoutErrorActorState();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(T && t,int loopDepth) 
	{
															#line 1302 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (!static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV<T>::futures) { (void)(t); this->~IoDegradedOrTimeoutErrorActorState(); static_cast<IoDegradedOrTimeoutErrorActor*>(this)->destroy(); return 0; }
															#line 2226 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		new (&static_cast<IoDegradedOrTimeoutErrorActor*>(this)->SAV< T >::value()) T(t);
		this->~IoDegradedOrTimeoutErrorActorState();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when2(Void const& _,int loopDepth) 
	{
															#line 1305 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1306 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 2240 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1307 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 2244 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1309 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1310 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1311 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 2252 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1312 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 2256 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1314 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.log();
															#line 1315 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 2262 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	int a_body1cont1when2(Void && _,int loopDepth) 
	{
															#line 1305 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		Error err = io_timeout();
															#line 1306 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (isSimulatorProcessUnreliable())
															#line 2272 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1307 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			err = err.asInjectedFault();
															#line 2276 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1309 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		TraceEvent e(SevError, "IoTimeoutError");
															#line 1310 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.error(err);
															#line 1311 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		if (context != nullptr)
															#line 2284 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		{
															#line 1312 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
			e.detail("Context", context);
															#line 2288 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		}
															#line 1314 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		e.log();
															#line 1315 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
		return a_body1Catch1(err, loopDepth);
															#line 2294 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state > 0) static_cast<IoDegradedOrTimeoutErrorActor*>(this)->actor_wait_state = 0;
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->ActorCallback< IoDegradedOrTimeoutErrorActor, 2, T >::remove();
		static_cast<IoDegradedOrTimeoutErrorActor*>(this)->ActorCallback< IoDegradedOrTimeoutErrorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 2, T >*,T const& value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 2, T >*,T && value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< IoDegradedOrTimeoutErrorActor, 2, T >*,Error err) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< IoDegradedOrTimeoutErrorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< IoDegradedOrTimeoutErrorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	Future<T> what;
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	double errTime;
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	Reference<AsyncVar<bool>> degraded;
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	double degradedTime;
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	const char* context;
															#line 2429 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
};
// This generated class is to be used only via ioDegradedOrTimeoutError()
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
class IoDegradedOrTimeoutErrorActor final : public Actor<T>, public ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 0, T >, public ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 1, Void >, public ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 2, T >, public ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 3, Void >, public FastAllocated<IoDegradedOrTimeoutErrorActor<T>>, public IoDegradedOrTimeoutErrorActorState<T, IoDegradedOrTimeoutErrorActor<T>> {
															#line 2436 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
public:
	using FastAllocated<IoDegradedOrTimeoutErrorActor<T>>::operator new;
	using FastAllocated<IoDegradedOrTimeoutErrorActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17872999569605358592UL, 9247078514108283904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 0, T >;
friend struct ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 1, Void >;
friend struct ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 2, T >;
friend struct ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 3, Void >;
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	IoDegradedOrTimeoutErrorActor(Future<T> const& what,double const& errTime,Reference<AsyncVar<bool>> const& degraded,double const& degradedTime,const char* const& context = nullptr) 
															#line 2456 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
		 : Actor<T>(),
		   IoDegradedOrTimeoutErrorActorState<T, IoDegradedOrTimeoutErrorActor<T>>(what, errTime, degraded, degradedTime, context),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12926056991623421184UL, 13749325704061913856UL);
		ActorExecutionContextHelper __helper(static_cast<IoDegradedOrTimeoutErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("ioDegradedOrTimeoutError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("ioDegradedOrTimeoutError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 0, T >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IoDegradedOrTimeoutErrorActor<T>, 2, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
template <class T>
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
[[nodiscard]] Future<T> ioDegradedOrTimeoutError( Future<T> const& what, double const& errTime, Reference<AsyncVar<bool>> const& degraded, double const& degradedTime, const char* const& context = nullptr ) {
															#line 1271 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"
	return Future<T>(new IoDegradedOrTimeoutErrorActor<T>(what, errTime, degraded, degradedTime, context));
															#line 2492 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/WorkerInterface.actor.g.h"
}

#line 1319 "/codebuild/output/src4111079991/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/WorkerInterface.actor.h"

#include "fdbserver/ServerDBInfo.h"
#include "flow/unactorcompiler.h"
#endif
