#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
/*
 * AuthzTlsTest.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _WIN32

#include <algorithm>
#include <array>
#include <cstring>
#include <ctime>
#include <iostream>
#include <string_view>
#include <thread>
#include <type_traits>

#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>

#include <fmt/core.h>

#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/FlowTransport.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/MkCert.h"
#include "flow/ScopeExit.h"
#include "flow/TLSConfig.actor.h"

#include "flow/actorcompiler.h" // This must be the last #include.

using namespace std::literals::string_view_literals;

enum Role : uint8_t { MAIN, CLIENT, SERVER, UNDETERMINED, LAST };

constexpr std::array<std::string_view, Role::LAST> ROLE_STRING{ "MAIN"sv, "CLIENT"sv, "SERVER"sv, "UNDETERMINED"sv };

Role role = Role::MAIN;

template <>
struct fmt::formatter<Role> : fmt::formatter<std::string> {
	auto format(Role role, fmt::format_context& ctx) const {
		return fmt::format_to(ctx.out(), "{:^10}", ROLE_STRING[static_cast<int>(role)]);
	}
};

template <class... Args>
void logRaw(const fmt::format_string<Args...>& fmt_str, Args&&... args) {
	std::cout << fmt::format(fmt_str, std::forward<Args>(args)...);
	std::cout.flush();
}

template <class... Args>
void log(const fmt::format_string<Args...>& fmt_str, Args&&... args) {
	// NOTE: The fmt::formatter<Role> can do the padding, but not this fmt::format expression
	std::cout << fmt::format("[{}] ", role);
	logRaw(fmt_str, std::forward<Args>(args)...);
	std::cout << std::endl;
}

enum ChainLength : int { NO_TLS = -1 };

template <>
struct fmt::formatter<ChainLength> : fmt::formatter<std::string> {
	auto format(ChainLength value, fmt::format_context& ctx) const {
		if (value == NO_TLS)
			return fmt::format_to(ctx.out(), "NO_TLS");
		else
			return fmt::format_to(ctx.out(), "{}", static_cast<std::underlying_type_t<ChainLength>>(value));
	}
};

template <>
struct fmt::formatter<std::vector<std::pair<ChainLength, ChainLength>>> : fmt::formatter<std::string> {
	auto format(const std::vector<std::pair<ChainLength, ChainLength>>& entries, fmt::format_context& ctx) const {
		fmt::format_to(ctx.out(), "[");
		bool first = true;
		for (const auto& entry : entries) {
			fmt::format_to(ctx.out(), "{}{{ {}, {} }}", (first ? "" : ", "), entry.first, entry.second);
			first = false;
		}
		return fmt::format_to(ctx.out(), "]");
	}
};

std::string drainPipe(const int pipeFd) {
	int readRc = 0;
	std::string ret;
	char buf[PIPE_BUF];
	while ((readRc = ::read(pipeFd, buf, PIPE_BUF)) > 0) {
		ret.append(buf, readRc);
	}
	if (readRc != 0) {
		log("Unexpected error draining pipe: {}", strerror(errno));
		throw std::runtime_error("pipe read error");
	}
	return ret;
}

struct TLSCreds {
	bool noTls = false;
	std::string certBytes;
	std::string keyBytes;
	std::string caBytes;
};

TLSCreds makeCreds(const ChainLength chainLen, const mkcert::ESide side) {
	if (chainLen == 0 || chainLen == NO_TLS) {
		return TLSCreds{ chainLen == NO_TLS, "", "", "" };
	}
	auto arena = Arena();
	auto ret = TLSCreds{};
	auto specs = mkcert::makeCertChainSpec(arena, std::labs(chainLen), side);
	if (chainLen < 0) {
		specs[0].offsetNotBefore = -60l * 60 * 24 * 365;
		specs[0].offsetNotAfter = -10l; // cert that expired 10 seconds ago
	}
	auto chain = mkcert::makeCertChain(arena, specs, {} /* create root CA cert from spec*/);
	if (chain.size() == 1) {
		ret.certBytes = concatCertChain(arena, chain).toString();
	} else {
		auto nonRootChain = chain;
		nonRootChain.pop_back();
		ret.certBytes = concatCertChain(arena, nonRootChain).toString();
	}
	ret.caBytes = chain.back().certPem.toString();
	ret.keyBytes = chain.front().privateKeyPem.toString();
	return ret;
}

enum class Result : int { ERROR = 0, TRUSTED, UNTRUSTED, TIMEOUT, LAST };

constexpr std::array<std::string_view, static_cast<size_t>(Result::LAST)> RESULT_STRING{ "ERROR",
	                                                                                     "TRUSTED",
	                                                                                     "UNTRUSTED",
	                                                                                     "TIMEOUT" };
template <>
struct fmt::formatter<Result> : fmt::formatter<std::string> {
	auto format(const Result& r, fmt::format_context& ctx) const {
		return fmt::format_to(ctx.out(), "{}", RESULT_STRING[static_cast<int>(r)]);
	}
};

															#line 162 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
namespace {
// This generated class is to be used only via stopNetworkAfter()
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
template <class T, class StopNetworkAfterActor>
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
class StopNetworkAfterActorState {
															#line 169 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
public:
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	StopNetworkAfterActorState(Future<T> const& what) 
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		 : what(what)
															#line 176 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
	{
		fdb_probe_actor_create("stopNetworkAfter", reinterpret_cast<unsigned long>(this));

	}
	~StopNetworkAfterActorState() 
	{
		fdb_probe_actor_destroy("stopNetworkAfter", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 162 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
			StrictFuture<T> __when_expr_0 = what;
															#line 162 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
			if (static_cast<StopNetworkAfterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 193 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<StopNetworkAfterActor*>(this)->actor_wait_state = 1;
															#line 162 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StopNetworkAfterActor, 0, T >*>(static_cast<StopNetworkAfterActor*>(this)));
															#line 198 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(T const& t,int loopDepth) 
	{
															#line 163 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		g_network->stop();
															#line 164 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		if (!static_cast<StopNetworkAfterActor*>(this)->SAV<T>::futures) { (void)(t); this->~StopNetworkAfterActorState(); static_cast<StopNetworkAfterActor*>(this)->destroy(); return 0; }
															#line 223 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		new (&static_cast<StopNetworkAfterActor*>(this)->SAV< T >::value()) T(t);
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(T && t,int loopDepth) 
	{
															#line 163 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		g_network->stop();
															#line 164 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		if (!static_cast<StopNetworkAfterActor*>(this)->SAV<T>::futures) { (void)(t); this->~StopNetworkAfterActorState(); static_cast<StopNetworkAfterActor*>(this)->destroy(); return 0; }
															#line 237 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		new (&static_cast<StopNetworkAfterActor*>(this)->SAV< T >::value()) T(t);
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
		loopDepth = a_body1cont1(t, loopDepth);

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(t), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StopNetworkAfterActor*>(this)->actor_wait_state > 0) static_cast<StopNetworkAfterActor*>(this)->actor_wait_state = 0;
		static_cast<StopNetworkAfterActor*>(this)->ActorCallback< StopNetworkAfterActor, 0, T >::remove();

	}
	void a_callback_fire(ActorCallback< StopNetworkAfterActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StopNetworkAfterActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StopNetworkAfterActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	Future<T> what;
															#line 322 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
};
// This generated class is to be used only via stopNetworkAfter()
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
template <class T>
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
class StopNetworkAfterActor final : public Actor<T>, public ActorCallback< StopNetworkAfterActor<T>, 0, T >, public FastAllocated<StopNetworkAfterActor<T>>, public StopNetworkAfterActorState<T, StopNetworkAfterActor<T>> {
															#line 329 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
public:
	using FastAllocated<StopNetworkAfterActor<T>>::operator new;
	using FastAllocated<StopNetworkAfterActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14481873957495672064UL, 8500758776770739968UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StopNetworkAfterActor<T>, 0, T >;
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	StopNetworkAfterActor(Future<T> const& what) 
															#line 346 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		 : Actor<T>(),
		   StopNetworkAfterActorState<T, StopNetworkAfterActor<T>>(what),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5440916104915410176UL, 16460411745428950528UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("stopNetworkAfter");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StopNetworkAfterActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
template <class T>
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
[[nodiscard]] Future<T> stopNetworkAfter( Future<T> const& what ) {
															#line 160 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	return Future<T>(new StopNetworkAfterActor<T>(what));
															#line 381 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
}

#line 166 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"

// Reflective struct containing information about the requester from a server PoV
struct SessionInfo {
	constexpr static FileIdentifier file_identifier = 1578312;
	bool isPeerTrusted = false;
	NetworkAddress peerAddress;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, isPeerTrusted, peerAddress);
	}
};

struct SessionProbeRequest {
	constexpr static FileIdentifier file_identifier = 1559713;
	ReplyPromise<SessionInfo> reply{ PeerCompatibilityPolicy{ RequirePeer::AtLeast,
		                                                      ProtocolVersion::withStableInterfaces() } };

	bool verify() const { return true; }

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reply);
	}
};

struct SessionProbeReceiver final : NetworkMessageReceiver {
	SessionProbeReceiver() {}
	void receive(ArenaObjectReader& reader) override {
		SessionProbeRequest req;
		reader.deserialize(req);
		SessionInfo res;
		res.isPeerTrusted = FlowTransport::transport().currentDeliveryPeerIsTrusted();
		res.peerAddress = FlowTransport::transport().currentDeliveryPeerAddress();
		req.reply.send(res);
	}
	PeerCompatibilityPolicy peerCompatibilityPolicy() const override {
		return PeerCompatibilityPolicy{ RequirePeer::AtLeast, ProtocolVersion::withStableInterfaces() };
	}
	bool isPublic() const override { return true; }
};

void runServer(const Endpoint& endpoint, int addrPipe, int completionPipe) {
	auto realAddr = FlowTransport::transport().getLocalAddresses().address;
	log("Listening at {}", realAddr.toString());
	log("Endpoint token is {}", endpoint.token.toString());
	static_assert(std::is_trivially_destructible_v<NetworkAddress>,
	              "NetworkAddress cannot be directly put on wire; need proper (de-)serialization");
	// below writes/reads would block, but this is good enough for a test.
	if (sizeof(realAddr) != ::write(addrPipe, &realAddr, sizeof(realAddr))) {
		log("Failed to write server addr to pipe: {}", strerror(errno));
		return;
	}
	if (sizeof(endpoint.token) != ::write(addrPipe, &endpoint.token, sizeof(endpoint.token))) {
		log("Failed to write server endpoint to pipe: {}", strerror(errno));
		return;
	}
	auto done = false;
	if (sizeof(done) != ::read(completionPipe, &done, sizeof(done))) {
		log("Failed to read completion flag from pipe: {}", strerror(errno));
		return;
	}
	return;
}

															#line 450 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
namespace {
// This generated class is to be used only via waitAndPrintResponse()
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
template <class WaitAndPrintResponseActor>
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
class WaitAndPrintResponseActorState {
															#line 457 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
public:
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	WaitAndPrintResponseActorState(Future<SessionInfo> const& response,Result* const& rc) 
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		 : response(response),
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		   rc(rc)
															#line 466 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
	{
		fdb_probe_actor_create("waitAndPrintResponse", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndPrintResponseActorState() 
	{
		fdb_probe_actor_destroy("waitAndPrintResponse", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 233 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				StrictFuture<SessionInfo> __when_expr_0 = response;
															#line 233 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				if (static_cast<WaitAndPrintResponseActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 484 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<WaitAndPrintResponseActor*>(this)->actor_wait_state = 1;
															#line 233 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >*>(static_cast<WaitAndPrintResponseActor*>(this)));
															#line 489 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndPrintResponseActorState();
		static_cast<WaitAndPrintResponseActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 245 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		if (!static_cast<WaitAndPrintResponseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitAndPrintResponseActorState(); static_cast<WaitAndPrintResponseActor*>(this)->destroy(); return 0; }
															#line 518 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		new (&static_cast<WaitAndPrintResponseActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitAndPrintResponseActorState();
		static_cast<WaitAndPrintResponseActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& err,int loopDepth=0) 
	{
		try {
															#line 237 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
			if (err.code() != error_code_operation_cancelled)
															#line 531 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
			{
															#line 238 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				log("Unexpected error: {}", err.what());
															#line 239 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				*rc = Result::ERROR;
															#line 537 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
			}
			else
			{
															#line 241 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				log("Timed out");
															#line 242 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
				*rc = Result::TIMEOUT;
															#line 545 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(SessionInfo const& info,int loopDepth) 
	{
															#line 234 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		log("Probe response: trusted={} peerAddress={}", info.isPeerTrusted, info.peerAddress.toString());
															#line 235 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		*rc = info.isPeerTrusted ? Result::TRUSTED : Result::UNTRUSTED;
															#line 563 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(SessionInfo && info,int loopDepth) 
	{
															#line 234 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		log("Probe response: trusted={} peerAddress={}", info.isPeerTrusted, info.peerAddress.toString());
															#line 235 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
		*rc = info.isPeerTrusted ? Result::TRUSTED : Result::UNTRUSTED;
															#line 574 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(SessionInfo const& info,int loopDepth) 
	{
		loopDepth = a_body1cont2(info, loopDepth);

		return loopDepth;
	}
	int a_body1when1(SessionInfo && info,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(info), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndPrintResponseActor*>(this)->actor_wait_state > 0) static_cast<WaitAndPrintResponseActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndPrintResponseActor*>(this)->ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >*,SessionInfo const& value) 
	{
		fdb_probe_actor_enter("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndPrintResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >*,SessionInfo && value) 
	{
		fdb_probe_actor_enter("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndPrintResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndPrintResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	Future<SessionInfo> response;
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	Result* rc;
															#line 671 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
};
// This generated class is to be used only via waitAndPrintResponse()
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
class WaitAndPrintResponseActor final : public Actor<Void>, public ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >, public FastAllocated<WaitAndPrintResponseActor>, public WaitAndPrintResponseActorState<WaitAndPrintResponseActor> {
															#line 676 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
public:
	using FastAllocated<WaitAndPrintResponseActor>::operator new;
	using FastAllocated<WaitAndPrintResponseActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11317746253205249280UL, 872148005908232704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >;
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	WaitAndPrintResponseActor(Future<SessionInfo> const& response,Result* const& rc) 
															#line 693 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
		 : Actor<Void>(),
		   WaitAndPrintResponseActorState<WaitAndPrintResponseActor>(response, rc),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(227023383426748672UL, 6405512955091383296UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndPrintResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndPrintResponse");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndPrintResponse", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndPrintResponseActor, 0, SessionInfo >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
[[nodiscard]] Future<Void> waitAndPrintResponse( Future<SessionInfo> const& response, Result* const& rc ) {
															#line 231 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"
	return Future<Void>(new WaitAndPrintResponseActor(response, rc));
															#line 726 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/build_output/fdbrpc/tests/AuthzTlsTest.actor.g.cpp"
}

#line 247 "/codebuild/output/src2230694009/src/github.com/apple/foundationdb/fdbrpc/tests/AuthzTlsTest.actor.cpp"

// int runAsServer(TLSCreds creds, int addrPipe, int completionPipe, Result expect) {}

template <bool IsServer>
int runHost(TLSCreds creds, int addrPipe, int completionPipe, Result expect) {
	auto tlsConfig = TLSConfig(IsServer ? TLSEndpointType::SERVER : TLSEndpointType::CLIENT);
	bool const noTls = creds.noTls;
	if (!noTls) {
		tlsConfig.setCertificateBytes(creds.certBytes);
		tlsConfig.setCABytes(creds.caBytes);
		tlsConfig.setKeyBytes(creds.keyBytes);
	}
	g_network = newNet2(tlsConfig);
	openTraceFile({}, 10 << 20, 10 << 20, ".", IsServer ? "authz_tls_unittest_server" : "authz_tls_unittest_client");
	FlowTransport::createInstance(!IsServer, 1, WLTOKEN_RESERVED_COUNT);
	auto& transport = FlowTransport::transport();
	if constexpr (IsServer) {
		auto addr = NetworkAddress::parse(noTls ? "127.0.0.1:0" : "127.0.0.1:0:tls");
		auto endpoint = Endpoint();
		auto receiver = SessionProbeReceiver();
		auto listenFuture = transport.bind(addr, addr);
		transport.addEndpoint(endpoint, &receiver, TaskPriority::ReadSocket);
		auto thread = std::thread([]() {
			g_network->run();
			flushTraceFileVoid();
		});
		runServer(endpoint, addrPipe, completionPipe);
		auto cleanupGuard = ScopeExit([&thread]() {
			g_network->stop();
			thread.join();
		});
		return 0;
	} else {
		auto dest = Endpoint();
		auto& serverAddr = dest.addresses.address;
		if (sizeof(serverAddr) != ::read(addrPipe, &serverAddr, sizeof(serverAddr))) {
			log("Failed to read server addr from pipe: {}", strerror(errno));
			return 1;
		}
		if (noTls)
			serverAddr.flags &= ~NetworkAddress::FLAG_TLS;
		else
			serverAddr.flags |= NetworkAddress::FLAG_TLS;
		auto& token = dest.token;
		if (sizeof(token) != ::read(addrPipe, &token, sizeof(token))) {
			log("Failed to read server endpoint token from pipe: {}", strerror(errno));
			return 2;
		}
		log("Server address is {}{}", serverAddr.toString(), noTls ? " (TLS suffix removed)" : "");
		log("Server endpoint token is {}", token.toString());
		auto sessionProbeReq = SessionProbeRequest{};
		transport.sendUnreliable(SerializeSource(sessionProbeReq), dest, true /*openConnection*/);
		log("Request is sent");
		auto rc = 0;
		auto result = Result::ERROR;
		{
			auto timeout = delay(expect == Result::TIMEOUT ? 0.5 : 5);
			auto complete = waitAndPrintResponse(sessionProbeReq.reply.getFuture(), &result);
			auto f = stopNetworkAfter(complete || timeout);
			g_network->run();
		}
		auto done = true;
		if (sizeof(done) != ::write(completionPipe, &done, sizeof(done))) {
			log("Failed to signal server to terminate: {}", strerror(errno));
			rc = 4;
		}
		if (rc == 0) {
			if (expect != result) {
				log("Test failed: expected {}, got {}", expect, result);
				rc = 5;
			} else {
				log("Response OK: got {} as expected", result);
			}
		}
		return rc;
	}
}

Result getExpectedResult(ChainLength serverChainLen, ChainLength clientChainLen) {
	auto expect = Result::ERROR;
	if (serverChainLen > 0) {
		if (clientChainLen == NO_TLS || clientChainLen < 0) {
			expect = Result::TIMEOUT;
		} else if (clientChainLen > 0) {
			expect = Result::TRUSTED;
		} else if (clientChainLen == 0) {
			expect = Result::UNTRUSTED;
		}
	} else if (serverChainLen == NO_TLS && clientChainLen == NO_TLS) {
		expect = Result::TRUSTED;
	} else {
		expect = Result::TIMEOUT;
	}
	return expect;
}

std::pair<bool, std::string> waitPidStatusInterpreter(const char* procName, const int status) {
	std::string prefix = fmt::format("{} subprocess ", procName);
	std::string message;
	if (WIFEXITED(status)) {
		const auto exitStatus = WEXITSTATUS(status);
		if (exitStatus == 0) {
			return { true, fmt::format("{} waitpid() OK", prefix) };
		}
		message = fmt::format("{} exited with status {}", prefix, exitStatus);
	} else if (WIFSIGNALED(status)) {
		const auto signal = WTERMSIG(status);
		message = fmt::format("{} killed by signal {} - {}", prefix, signal, strsignal(signal));
#ifdef WCOREDUMP
		const auto coreDumped = WCOREDUMP(status);
		if (coreDumped)
			message.append(std::string_view(" (core dumped)"));
#endif // WCOREDUMP
	} else if (WIFSTOPPED(status)) {
		const auto signal = WSTOPSIG(status);
		message = fmt::format("{} stopped by signal {} - {}", prefix, signal, strsignal(signal));
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wparentheses"
	} else if (WIFCONTINUED(status)) {
#pragma clang diagnostic pop
		message = fmt::format("{} continued by signal SIGCONT", prefix);
	}

	if (message.empty()) {
		message = fmt::format("{} Unrecognized status {} (Check man 2 waitpid for more details)", prefix, status);
	}

	return { false, message };
}

bool waitPid(pid_t subProcPid, const char* procName) {
	auto status = int{};
	auto pid = ::waitpid(subProcPid, &status, 0);

	if (pid < 0) {
		log("{} subprocess waitpid() failed with {}", procName, strerror(errno));

		return false;
	} else {
		auto [ok, message] = waitPidStatusInterpreter(procName, status);
		log("{}", message);

		return ok;
	}
}

int runTlsTest(ChainLength serverChainLen, ChainLength clientChainLen) {
	log("==== BEGIN TESTCASE ====");
	auto const expect = getExpectedResult(serverChainLen, clientChainLen);
	using namespace std::literals::string_literals;
	log("Cert chain length: server={} client={}", serverChainLen, clientChainLen);
	auto arena = Arena();
	auto serverCreds = makeCreds(serverChainLen, mkcert::ESide::Server);
	auto clientCreds = makeCreds(clientChainLen, mkcert::ESide::Client);
	// make server and client trust each other
	std::swap(serverCreds.caBytes, clientCreds.caBytes);
	auto clientPid = pid_t{};
	auto serverPid = pid_t{};
	int addrPipe[2], completionPipe[2], serverStdoutPipe[2], clientStdoutPipe[2];
	if (::pipe(addrPipe) || ::pipe(completionPipe) || ::pipe(serverStdoutPipe) || ::pipe(clientStdoutPipe)) {
		log("Pipe open failed: {}", strerror(errno));
		return 1;
	}
	auto ok = true;
	{
		serverPid = fork();
		if (serverPid == -1) {
			log("fork() for server subprocess failed: {}", strerror(errno));
			return 1;
		} else if (serverPid == 0) {
			role = Role::SERVER;
			// server subprocess
			::close(addrPipe[0]); // close address-in pipe (server writes its own address for client)
			::close(
			    completionPipe[1]); // close completion-flag-out pipe (server awaits/reads completion flag from client)
			::close(clientStdoutPipe[0]);
			::close(clientStdoutPipe[1]);
			::close(serverStdoutPipe[0]);
			auto pipeCleanup = ScopeExit([&addrPipe, &completionPipe]() {
				::close(addrPipe[1]);
				::close(completionPipe[0]);
			});
			if (-1 == ::dup2(serverStdoutPipe[1], STDOUT_FILENO)) {
				log("Failed to redirect server stdout to pipe: {}", strerror(errno));
				::close(serverStdoutPipe[1]);
				return 1;
			}
			_exit(runHost<true>(std::move(serverCreds), addrPipe[1], completionPipe[0], expect));
		}
		auto serverProcCleanup = ScopeExit([&ok, serverPid]() {
			if (!waitPid(serverPid, "Server"))
				ok = false;
		});
		clientPid = fork();
		if (clientPid == -1) {
			log("fork() for client subprocess failed: {}", strerror(errno));
			return 1;
		} else if (clientPid == 0) {
			role = Role::CLIENT;
			::close(addrPipe[1]);
			::close(completionPipe[0]);
			::close(serverStdoutPipe[0]);
			::close(serverStdoutPipe[1]);
			::close(clientStdoutPipe[0]);
			auto pipeCleanup = ScopeExit([&addrPipe, &completionPipe]() {
				::close(addrPipe[0]);
				::close(completionPipe[1]);
			});
			if (-1 == ::dup2(clientStdoutPipe[1], STDOUT_FILENO)) {
				log("Failed to redirect client stdout to pipe: {}", strerror(errno));
				::close(clientStdoutPipe[1]);
				return 1;
			}
			_exit(runHost<false>(std::move(clientCreds), addrPipe[0], completionPipe[1], expect));
		}
		auto clientProcCleanup = ScopeExit([&ok, clientPid]() {
			if (!waitPid(clientPid, "Client"))
				ok = false;
		});
	}
	// main process
	::close(addrPipe[0]);
	::close(addrPipe[1]);
	::close(completionPipe[0]);
	::close(completionPipe[1]);
	::close(serverStdoutPipe[1]);
	::close(clientStdoutPipe[1]);
	auto pipeCleanup = ScopeExit([&]() {
		::close(serverStdoutPipe[0]);
		::close(clientStdoutPipe[0]);
	});
	std::string const clientStdout = drainPipe(clientStdoutPipe[0]);
	log("/// Begin Client STDOUT ///");
	logRaw(fmt::runtime(clientStdout));
	log("/// End Client STDOUT ///");
	std::string const serverStdout = drainPipe(serverStdoutPipe[0]);
	log("/// Begin Server STDOUT ///");
	logRaw(fmt::runtime(serverStdout));
	log("/// End Server STDOUT ///");
	log(fmt::runtime(ok ? "OK" : "FAILED"));
	return !ok;
}

int main(int argc, char** argv) {
	unsigned seed = std::time(nullptr);
	if (argc > 1)
		seed = std::stoul(argv[1]);
	std::srand(seed);
	log("Seed: {}", seed);
	auto categoryToValue = [](int category) -> ChainLength {
		if (category == 2 || category == -2) {
			return static_cast<ChainLength>(category + std::rand() % 3);
		} else {
			return static_cast<ChainLength>(category);
		}
	};
	std::vector<std::pair<ChainLength, ChainLength>> inputs;
	std::vector<int> categories{ 0, NO_TLS, 1, -1, 2, -2 };
	for (auto lhs : categories) {
		for (auto rhs : categories) {
			auto input = std::pair(categoryToValue(lhs), categoryToValue(rhs));
			inputs.push_back(input);
		}
	}
	std::vector<std::pair<ChainLength, ChainLength>> failed;
	for (auto input : inputs) {
		auto [serverChainLen, clientChainLen] = input;
		if (runTlsTest(serverChainLen, clientChainLen))
			failed.push_back({ serverChainLen, clientChainLen });
	}
	if (!failed.empty()) {
		log("Test Failed: {}/{} cases: {}", failed.size(), inputs.size(), failed);
		return 1;
	} else {
		log("Test OK: {}/{} cases passed", inputs.size(), inputs.size());
		return 0;
	}
}
#else // _WIN32

#include <iostream>

int main() {
	std::cerr << "TLS test is not supported in Windows" << std::endl;
	return -1;
}

#endif // _WIN32
