#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
/*
 * FlowTransport.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbrpc/FlowTransport.h"
#include "flow/Arena.h"
#include "flow/IThreadPool.h"
#include "flow/Knobs.h"
#include "flow/NetworkAddress.h"
#include "flow/network.h"

#include <cstdint>
#include <fstream>
#include <string>
#include <unordered_map>
#if VALGRIND
#include <memcheck.h>
#endif

#include <boost/unordered_map.hpp>

#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/FailureMonitor.h"
#include "fdbrpc/HealthMonitor.h"
#include "fdbrpc/JsonWebKeySet.h"
#include "fdbrpc/genericactors.actor.h"
#include "fdbrpc/IPAllowList.h"
#include "fdbrpc/simulator.h"
#include "flow/ActorCollection.h"
#include "flow/Error.h"
#include "flow/flow.h"
#include "flow/Net2Packet.h"
#include "flow/TDMetric.actor.h"
#include "flow/ObjectSerializer.h"
#include "flow/Platform.h"
#include "flow/ProtocolVersion.h"
#include "flow/UnitTest.h"
#include "flow/WatchFile.actor.h"
#include "flow/IConnection.h"
#define XXH_INLINE_ALL
#include "flow/xxhash.h"
#include "flow/actorcompiler.h" // This must be the last #include.

void removeCachedDNS(const std::string& host, const std::string& service) {
	INetworkConnections::net()->removeCachedDNS(host, service);
}

// FIXME: explain the purpose of these variables, their update
// discipline, and the API calls that use them.

namespace {

NetworkAddressList g_currentDeliveryPeerAddress = NetworkAddressList();
bool g_currentDeliverPeerAddressTrusted = false;
Future<Void> g_currentDeliveryPeerDisconnect;

} // namespace

// FIXME: stop referring to messages as "packets".  Packets are known
// to the network infrastructure (routers, etc) and are layed out in
// compliance with widely adopted industry standards such as TCP and
// IP.  Messages, on the other hand, are generally not understoody
// beyond the boundaries of the application that defines them.
// Certainly this applies to FDB messages.  So we should refer to FDB
// messages as "messages".
constexpr int PACKET_LEN_WIDTH = sizeof(uint32_t);

// FIXME: explain what this is for
const uint64_t TOKEN_STREAM_FLAG = 1;

FDB_BOOLEAN_PARAM(InReadSocket);
FDB_BOOLEAN_PARAM(IsStableConnection);

class EndpointMap : NonCopyable {
public:
	// Reserve space for this many wellKnownEndpoints
	explicit EndpointMap(int wellKnownEndpointCount);
	void insertWellKnown(NetworkMessageReceiver* r, const Endpoint::Token& token, TaskPriority priority);
	void insert(NetworkMessageReceiver* r, Endpoint::Token& token, TaskPriority priority);
	const Endpoint& insert(NetworkAddressList localAddresses,
	                       std::vector<std::pair<FlowReceiver*, TaskPriority>> const& streams);
	NetworkMessageReceiver* get(Endpoint::Token const& token);
	TaskPriority getPriority(Endpoint::Token const& token);
	void remove(Endpoint::Token const& token, NetworkMessageReceiver* r);

private:
	void realloc();

	struct Entry {
		union {
			uint64_t
			    uid[2]; // priority packed into lower 32 bits; actual lower 32 bits of token are the index in data[]
			uint32_t nextFree;
		};
		NetworkMessageReceiver* receiver = nullptr;
		Endpoint::Token& token() { return *(Endpoint::Token*)uid; }
	};
	int wellKnownEndpointCount;
	std::vector<Entry> data;
	uint32_t firstFree;
};

EndpointMap::EndpointMap(int wellKnownEndpointCount)
  : wellKnownEndpointCount(wellKnownEndpointCount), data(wellKnownEndpointCount), firstFree(-1) {}

void EndpointMap::realloc() {
	int oldSize = data.size();
	data.resize(std::max(128, oldSize * 2));
	for (int i = oldSize; i < data.size(); i++) {
		data[i].receiver = 0;
		data[i].nextFree = i + 1;
	}
	data[data.size() - 1].nextFree = firstFree;
	firstFree = oldSize;
}

void EndpointMap::insertWellKnown(NetworkMessageReceiver* r, const Endpoint::Token& token, TaskPriority priority) {
	int index = token.second();
	ASSERT(index <= wellKnownEndpointCount);
	ASSERT(data[index].receiver == nullptr);
	data[index].receiver = r;
	data[index].token() =
	    Endpoint::Token(token.first(), (token.second() & 0xffffffff00000000LL) | static_cast<uint32_t>(priority));
}

void EndpointMap::insert(NetworkMessageReceiver* r, Endpoint::Token& token, TaskPriority priority) {
	if (firstFree == uint32_t(-1))
		realloc();
	int index = firstFree;
	firstFree = data[index].nextFree;
	token = Endpoint::Token(token.first(), (token.second() & 0xffffffff00000000LL) | index);
	data[index].token() =
	    Endpoint::Token(token.first(), (token.second() & 0xffffffff00000000LL) | static_cast<uint32_t>(priority));
	data[index].receiver = r;
}

const Endpoint& EndpointMap::insert(NetworkAddressList localAddresses,
                                    std::vector<std::pair<FlowReceiver*, TaskPriority>> const& streams) {
	int adjacentFree = 0;
	int adjacentStart = -1;
	firstFree = -1;
	for (int i = wellKnownEndpointCount; i < data.size(); i++) {
		if (data[i].receiver) {
			adjacentFree = 0;
		} else {
			data[i].nextFree = firstFree;
			firstFree = i;
			if (adjacentStart == -1 && ++adjacentFree == streams.size()) {
				adjacentStart = i + 1 - adjacentFree;
				firstFree = data[adjacentStart].nextFree;
			}
		}
	}
	if (adjacentStart == -1) {
		data.resize(data.size() + streams.size() - adjacentFree);
		adjacentStart = data.size() - streams.size();
		if (adjacentFree > 0) {
			firstFree = data[adjacentStart].nextFree;
		}
	}

	UID base = deterministicRandom()->randomUniqueID();
	for (uint64_t i = 0; i < streams.size(); i++) {
		int index = adjacentStart + i;
		uint64_t first = (base.first() + (i << 32)) | TOKEN_STREAM_FLAG;
		streams[i].first->setEndpoint(
		    Endpoint(localAddresses, UID(first, (base.second() & 0xffffffff00000000LL) | index)));
		data[index].token() =
		    Endpoint::Token(first, (base.second() & 0xffffffff00000000LL) | static_cast<uint32_t>(streams[i].second));
		data[index].receiver = (NetworkMessageReceiver*)streams[i].first;
	}

	return streams[0].first->getEndpoint(TaskPriority::DefaultEndpoint);
}

NetworkMessageReceiver* EndpointMap::get(Endpoint::Token const& token) {
	uint32_t index = token.second();
	if (index < wellKnownEndpointCount && data[index].receiver == nullptr) {
		TraceEvent(SevWarnAlways, "WellKnownEndpointNotAdded")
		    .detail("Token", token)
		    .detail("Index", index)
		    .backtrace();
	}
	if (index < data.size() && data[index].token().first() == token.first() &&
	    ((data[index].token().second() & 0xffffffff00000000LL) | index) == token.second())
		return data[index].receiver;
	return 0;
}

TaskPriority EndpointMap::getPriority(Endpoint::Token const& token) {
	uint32_t index = token.second();
	if (index < data.size() && data[index].token().first() == token.first() &&
	    ((data[index].token().second() & 0xffffffff00000000LL) | index) == token.second()) {
		auto res = static_cast<TaskPriority>(data[index].token().second());
		// we don't allow this priority to be "misused" for other stuff as we won't even
		// attempt to find an endpoint if UnknownEndpoint is returned here
		ASSERT(res != TaskPriority::UnknownEndpoint);
		return res;
	}
	return TaskPriority::UnknownEndpoint;
}

void EndpointMap::remove(Endpoint::Token const& token, NetworkMessageReceiver* r) {
	uint32_t index = token.second();
	if (index < wellKnownEndpointCount) {
		data[index].receiver = nullptr;
	} else if (index < data.size() && data[index].token().first() == token.first() &&
	           ((data[index].token().second() & 0xffffffff00000000LL) | index) == token.second() &&
	           data[index].receiver == r) {
		data[index].receiver = 0;
		data[index].nextFree = firstFree;
		firstFree = index;
	}
}

struct EndpointNotFoundReceiver final : NetworkMessageReceiver {
	EndpointNotFoundReceiver(EndpointMap& endpoints) {
		endpoints.insertWellKnown(
		    this, Endpoint::wellKnownToken(WLTOKEN_ENDPOINT_NOT_FOUND), TaskPriority::DefaultEndpoint);
	}

	void receive(ArenaObjectReader& reader) override {
		// Remote machine tells us it doesn't have endpoint e
		UID token;
		reader.deserialize(token);
		Endpoint e = FlowTransport::transport().loadedEndpoint(token);
		IFailureMonitor::failureMonitor().endpointNotFound(e);
	}
	bool isPublic() const override { return true; }
};

struct PingRequest {
	constexpr static FileIdentifier file_identifier = 4707015;
	ReplyPromise<Void> reply{ PeerCompatibilityPolicy{ RequirePeer::AtLeast,
		                                               ProtocolVersion::withStableInterfaces() } };
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, reply);
	}
};

struct PingReceiver final : NetworkMessageReceiver {
	PingReceiver(EndpointMap& endpoints) {
		endpoints.insertWellKnown(this, Endpoint::wellKnownToken(WLTOKEN_PING_PACKET), TaskPriority::ReadSocket);
	}
	void receive(ArenaObjectReader& reader) override {
		PingRequest req;
		reader.deserialize(req);
		req.reply.send(Void());
	}
	PeerCompatibilityPolicy peerCompatibilityPolicy() const override {
		return PeerCompatibilityPolicy{ RequirePeer::AtLeast, ProtocolVersion::withStableInterfaces() };
	}
	bool isPublic() const override { return true; }
};

struct UnauthorizedEndpointReceiver final : NetworkMessageReceiver {
	UnauthorizedEndpointReceiver(EndpointMap& endpoints) {
		endpoints.insertWellKnown(
		    this, Endpoint::wellKnownToken(WLTOKEN_UNAUTHORIZED_ENDPOINT), TaskPriority::ReadSocket);
	}

	void receive(ArenaObjectReader& reader) override {
		UID token;
		reader.deserialize(token);
		Endpoint e = FlowTransport::transport().loadedEndpoint(token);
		IFailureMonitor::failureMonitor().unauthorizedEndpoint(e);
	}
	bool isPublic() const override { return true; }
};

// NetworkAddressCachedString retains a cached Standalone<StringRef> of
// a NetworkAddressList.address.toString() value. This cached value is useful
// for features in the hot path (i.e. Tracing), which need the String formatted value
// frequently and do not wish to pay the formatting cost. If the underlying NetworkAddressList
// needs to change, do not attempt to update it directly, use the setNetworkAddress API as it
// will ensure the new toString() cached value is updated.
class NetworkAddressCachedString {
public:
	NetworkAddressCachedString() { setAddressList(NetworkAddressList()); }
	NetworkAddressCachedString(NetworkAddressList const& list) { setAddressList(list); }
	NetworkAddressList const& getAddressList() const { return addressList; }
	void setAddressList(NetworkAddressList const& list) {
		cachedStr = Standalone<StringRef>(StringRef(list.address.toString()));
		addressList = list;
	}
	void setNetworkAddress(NetworkAddress const& addr) {
		addressList.address = addr;
		setAddressList(addressList); // force the recaching of the string.
	}
	Standalone<StringRef> getLocalAddressAsString() const { return cachedStr; }
	operator NetworkAddressList const&() { return addressList; }

private:
	NetworkAddressList addressList;
	Standalone<StringRef> cachedStr;
};

class TransportData {
public:
	TransportData(uint64_t transportId, int maxWellKnownEndpoints, IPAllowList const* allowList);

	~TransportData();

	void initMetrics() {
		bytesSent.init("Net2.BytesSent"_sr);
		countPacketsReceived.init("Net2.CountPacketsReceived"_sr);
		countPacketsGenerated.init("Net2.CountPacketsGenerated"_sr);
		countConnEstablished.init("Net2.CountConnEstablished"_sr);
		countConnClosedWithError.init("Net2.CountConnClosedWithError"_sr);
		countConnClosedWithoutError.init("Net2.CountConnClosedWithoutError"_sr);
	}

	Reference<struct Peer> getPeer(NetworkAddress const& address);
	Reference<struct Peer> getOrOpenPeer(NetworkAddress const& address, bool startConnectionKeeper = true);

	// Returns true if given network address 'address' is one of the address we are listening on.
	bool isLocalAddress(const NetworkAddress& address) const;
	void applyPublicKeySet(StringRef jwkSetString);

	NetworkAddressCachedString localAddresses;
	std::vector<Future<Void>> listeners;
	std::unordered_map<NetworkAddress, Reference<struct Peer>> peers;
	// FIXME: explain what the std::pair<double, double> represent:
	std::unordered_map<NetworkAddress, std::pair<double, double>> closedPeers;
	HealthMonitor healthMonitor;
	std::set<NetworkAddress> orderedAddresses;
	Reference<AsyncVar<bool>> degraded;

	EndpointMap endpoints;
	EndpointNotFoundReceiver endpointNotFoundReceiver{ endpoints };
	PingReceiver pingReceiver{ endpoints };
	UnauthorizedEndpointReceiver unauthorizedEndpointReceiver{ endpoints };

	Int64MetricHandle bytesSent;
	Int64MetricHandle countPacketsReceived;
	Int64MetricHandle countPacketsGenerated;
	Int64MetricHandle countConnEstablished;
	Int64MetricHandle countConnClosedWithError;
	Int64MetricHandle countConnClosedWithoutError;

	// FIXME: explain what the std::pair<int64_t, double> represent:
	std::map<NetworkAddress, std::pair<uint64_t, double>> incompatiblePeers;
	AsyncTrigger incompatiblePeersChanged;
	uint32_t numIncompatibleConnections;
	// FIXME: explain the map
	std::map<uint64_t, double> multiVersionConnections;
	double lastIncompatibleMessage;
	uint64_t transportId;
	IPAllowList allowList;

	Future<Void> multiVersionCleanup;
	Future<Void> pingLogger;
	Future<Void> publicKeyFileWatch;

	std::unordered_map<Standalone<StringRef>, PublicKey> publicKeys;

	struct ConnectionHistoryEntry {
		// FIXME: explain why int64_t and not double is used to represent time
		int64_t time;
		NetworkAddress addr;
		bool failed;
	};
	std::deque<ConnectionHistoryEntry> connectionHistory;
	Future<Void> connectionHistoryLoggerF;
	Reference<IThreadPool> connectionLogWriterThread;
};

struct ConnectionLogWriter : IThreadPoolReceiver {
	const std::string baseDir;
	std::string fileName;
	std::fstream file;

	ConnectionLogWriter(const std::string baseDir) : baseDir(baseDir) {}

	virtual ~ConnectionLogWriter() {
		if (file.is_open())
			file.close();
	}

	struct AppendAction : TypedAction<ConnectionLogWriter, AppendAction> {
		std::string localAddr;
		std::deque<TransportData::ConnectionHistoryEntry> entries;
		AppendAction(std::string localAddr, std::deque<TransportData::ConnectionHistoryEntry>&& entries)
		  : localAddr(localAddr), entries(std::move(entries)) {}

		double getTimeEstimate() const { return 2; }
	};

	std::string newFileName() const { return baseDir + "fdb-connection-log-" + time_str() + ".csv"; }

	void init() { fileName = newFileName(); }

	std::string time_str() const { return std::to_string(now()); }

	void openOrRoll() {
		if (!file.is_open()) {
			TraceEvent("OpenConnectionLog").detail("FileName", fileName);
			file = std::fstream(fileName, std::ios::in | std::ios::out | std::ios::app);
		}

		if (!file.is_open()) {
			TraceEvent(SevError, "ErrorOpenConnectionLog").detail("FileName", fileName);
			throw io_error();
		}

		if (file.tellg() > 100 * 1024 * 1024 /* 100 MB */) {
			file.close();
			fileName = newFileName();
			TraceEvent("RollConnectionLog").detail("FileName", fileName);
			openOrRoll();
		}
	}

	void action(AppendAction& a) {
		openOrRoll();

		std::string output;
		for (const auto& entry : a.entries) {
			output += std::to_string(entry.time) + ",";
			output += a.localAddr + ",";
			output += entry.failed ? "failed," : "success,";
			output += entry.addr.toString() + "\n";
		}
		file << output;
		file.flush();
	}
};

															#line 448 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionHistoryLogger()
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionHistoryLoggerActor>
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionHistoryLoggerActorState {
															#line 455 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionHistoryLoggerActorState(TransportData* const& self) 
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self)
															#line 462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionHistoryLogger", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionHistoryLoggerActorState() 
	{
		fdb_probe_actor_destroy("connectionHistoryLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 447 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!FLOW_KNOBS->LOG_CONNECTION_ATTEMPTS_ENABLED)
															#line 477 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 448 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (!static_cast<ConnectionHistoryLoggerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectionHistoryLoggerActorState(); static_cast<ConnectionHistoryLoggerActor*>(this)->destroy(); return 0; }
															#line 481 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				new (&static_cast<ConnectionHistoryLoggerActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ConnectionHistoryLoggerActorState();
				static_cast<ConnectionHistoryLoggerActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 451 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			next = Void();
															#line 454 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (g_network->isSimulated())
															#line 491 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 455 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->connectionLogWriterThread = Reference<IThreadPool>(new DummyThreadPool());
															#line 495 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 457 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->connectionLogWriterThread = createGenericThreadPool();
															#line 501 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 460 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->connectionLogWriterThread->addThread(new ConnectionLogWriter(FLOW_KNOBS->CONNECTION_LOG_DIRECTORY));
															#line 461 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 507 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionHistoryLoggerActorState();
		static_cast<ConnectionHistoryLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = next;
															#line 462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionHistoryLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 539 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ConnectionHistoryLoggerActor*>(this)->actor_wait_state = 1;
															#line 462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionHistoryLoggerActor, 0, Void >*>(static_cast<ConnectionHistoryLoggerActor*>(this)));
															#line 544 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 463 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		next = delay(FLOW_KNOBS->LOG_CONNECTION_INTERVAL_SECS);
															#line 464 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->connectionHistory.size() == 0)
															#line 555 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 467 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		std::string localAddr = FlowTransport::getGlobalLocalAddress().toString();
															#line 468 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		auto action = new ConnectionLogWriter::AppendAction(localAddr, std::move(self->connectionHistory));
															#line 469 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(action != nullptr);
															#line 470 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->connectionLogWriterThread->post(action);
															#line 471 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(self->connectionHistory.size() == 0);
															#line 569 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 463 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		next = delay(FLOW_KNOBS->LOG_CONNECTION_INTERVAL_SECS);
															#line 464 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->connectionHistory.size() == 0)
															#line 580 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 467 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		std::string localAddr = FlowTransport::getGlobalLocalAddress().toString();
															#line 468 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		auto action = new ConnectionLogWriter::AppendAction(localAddr, std::move(self->connectionHistory));
															#line 469 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(action != nullptr);
															#line 470 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->connectionLogWriterThread->post(action);
															#line 471 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(self->connectionHistory.size() == 0);
															#line 594 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionHistoryLoggerActor*>(this)->actor_wait_state > 0) static_cast<ConnectionHistoryLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionHistoryLoggerActor*>(this)->ActorCallback< ConnectionHistoryLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionHistoryLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionHistoryLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionHistoryLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionHistoryLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionHistoryLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionHistoryLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 451 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> next;
															#line 678 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionHistoryLogger()
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionHistoryLoggerActor final : public Actor<Void>, public ActorCallback< ConnectionHistoryLoggerActor, 0, Void >, public FastAllocated<ConnectionHistoryLoggerActor>, public ConnectionHistoryLoggerActorState<ConnectionHistoryLoggerActor> {
															#line 683 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionHistoryLoggerActor>::operator new;
	using FastAllocated<ConnectionHistoryLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4588699612634316544UL, 2551173952702600960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionHistoryLoggerActor, 0, Void >;
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionHistoryLoggerActor(TransportData* const& self) 
															#line 700 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionHistoryLoggerActorState<ConnectionHistoryLoggerActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8967533971462996992UL, 8133420289736437504UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionHistoryLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionHistoryLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionHistoryLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionHistoryLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> connectionHistoryLogger( TransportData* const& self ) {
															#line 446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionHistoryLoggerActor(self));
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 474 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 738 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via pingLatencyLogger()
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class PingLatencyLoggerActor>
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class PingLatencyLoggerActorState {
															#line 745 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	PingLatencyLoggerActorState(TransportData* const& self) 
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 476 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   lastAddress(NetworkAddress())
															#line 754 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("pingLatencyLogger", reinterpret_cast<unsigned long>(this));

	}
	~PingLatencyLoggerActorState() 
	{
		fdb_probe_actor_destroy("pingLatencyLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 477 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 769 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PingLatencyLoggerActorState();
		static_cast<PingLatencyLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 478 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->orderedAddresses.size())
															#line 799 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 479 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			auto it = self->orderedAddresses.upper_bound(lastAddress);
															#line 480 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (it == self->orderedAddresses.end())
															#line 805 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 481 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it = self->orderedAddresses.begin();
															#line 809 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 483 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastAddress = *it;
															#line 484 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			auto peer = self->getPeer(lastAddress);
															#line 485 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!peer)
															#line 817 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 486 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent(SevWarnAlways, "MissingNetworkAddress") .suppressFor(10.0) .detail("PeerAddr", lastAddress) .detail("PeerAddress", lastAddress);
															#line 821 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 491 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer->lastLoggedTime <= 0.0)
															#line 825 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 492 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->lastLoggedTime = peer->lastConnectTime;
															#line 829 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 495 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer && (peer->pingLatencies.getPopulationSize() >= 10 || peer->connectFailedCount > 0 || peer->timeoutCount > 0))
															#line 833 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 497 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent("PingLatency") .suppressFor(30.0) .detail("Elapsed", now() - peer->lastLoggedTime) .detail("PeerAddr", lastAddress) .detail("PeerAddress", lastAddress) .detail("MinLatency", peer->pingLatencies.min()) .detail("MaxLatency", peer->pingLatencies.max()) .detail("MeanLatency", peer->pingLatencies.mean()) .detail("MedianLatency", peer->pingLatencies.median()) .detail("P90Latency", peer->pingLatencies.percentile(0.90)) .detail("Count", peer->pingLatencies.getPopulationSize()) .detail("BytesReceived", peer->bytesReceived - peer->lastLoggedBytesReceived) .detail("BytesSent", peer->bytesSent - peer->lastLoggedBytesSent) .detail("TimeoutCount", peer->timeoutCount) .detail("ConnectOutgoingCount", peer->connectOutgoingCount) .detail("ConnectIncomingCount", peer->connectIncomingCount) .detail("ConnectFailedCount", peer->connectFailedCount) .detail("ConnectMinLatency", peer->connectLatencies.min()) .detail("ConnectMaxLatency", peer->connectLatencies.max()) .detail("ConnectMeanLatency", peer->connectLatencies.mean()) .detail("ConnectMedianLatency", peer->connectLatencies.median()) .detail("ConnectP90Latency", peer->connectLatencies.percentile(0.90));
															#line 519 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->lastLoggedTime = now();
															#line 520 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->connectOutgoingCount = 0;
															#line 521 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->connectIncomingCount = 0;
															#line 522 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->connectFailedCount = 0;
															#line 523 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->pingLatencies.clear();
															#line 524 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->connectLatencies.clear();
															#line 525 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->lastLoggedBytesReceived = peer->bytesReceived;
															#line 526 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->lastLoggedBytesSent = peer->bytesSent;
															#line 527 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->timeoutCount = 0;
															#line 528 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->PING_LOGGING_INTERVAL);
															#line 528 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 859 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 1;
															#line 528 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PingLatencyLoggerActor, 0, Void >*>(static_cast<PingLatencyLoggerActor*>(this)));
															#line 864 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 529 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (it == self->orderedAddresses.begin())
															#line 871 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 530 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					StrictFuture<Void> __when_expr_1 = delay(FLOW_KNOBS->PING_LOGGING_INTERVAL);
															#line 530 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 877 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
					static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 2;
															#line 530 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PingLatencyLoggerActor, 1, Void >*>(static_cast<PingLatencyLoggerActor*>(this)));
															#line 882 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1loopBody1cont7(loopDepth);
				}
			}
		}
		else
		{
															#line 533 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(FLOW_KNOBS->PING_LOGGING_INTERVAL);
															#line 533 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 897 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
			static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 3;
															#line 533 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PingLatencyLoggerActor, 2, Void >*>(static_cast<PingLatencyLoggerActor*>(this)));
															#line 902 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state > 0) static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<PingLatencyLoggerActor*>(this)->ActorCallback< PingLatencyLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PingLatencyLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state > 0) static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<PingLatencyLoggerActor*>(this)->ActorCallback< PingLatencyLoggerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PingLatencyLoggerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state > 0) static_cast<PingLatencyLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<PingLatencyLoggerActor*>(this)->ActorCallback< PingLatencyLoggerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PingLatencyLoggerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PingLatencyLoggerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 476 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	NetworkAddress lastAddress;
															#line 1191 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via pingLatencyLogger()
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class PingLatencyLoggerActor final : public Actor<Void>, public ActorCallback< PingLatencyLoggerActor, 0, Void >, public ActorCallback< PingLatencyLoggerActor, 1, Void >, public ActorCallback< PingLatencyLoggerActor, 2, Void >, public FastAllocated<PingLatencyLoggerActor>, public PingLatencyLoggerActorState<PingLatencyLoggerActor> {
															#line 1196 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<PingLatencyLoggerActor>::operator new;
	using FastAllocated<PingLatencyLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14333132819586075648UL, 3094021053257746176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PingLatencyLoggerActor, 0, Void >;
friend struct ActorCallback< PingLatencyLoggerActor, 1, Void >;
friend struct ActorCallback< PingLatencyLoggerActor, 2, Void >;
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	PingLatencyLoggerActor(TransportData* const& self) 
															#line 1215 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   PingLatencyLoggerActorState<PingLatencyLoggerActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("pingLatencyLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17065004379880444160UL, 12961548686568209920UL);
		ActorExecutionContextHelper __helper(static_cast<PingLatencyLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("pingLatencyLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("pingLatencyLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PingLatencyLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PingLatencyLoggerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PingLatencyLoggerActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> pingLatencyLogger( TransportData* const& self ) {
															#line 475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new PingLatencyLoggerActor(self));
															#line 1250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 537 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

// FIXME: why doesn't this just pass a IPAllowList for this to copy, rather than a pointer?
TransportData::TransportData(uint64_t transportId, int maxWellKnownEndpoints, IPAllowList const* allowList)
  : endpoints(maxWellKnownEndpoints), endpointNotFoundReceiver(endpoints), pingReceiver(endpoints),
    numIncompatibleConnections(0), lastIncompatibleMessage(0), transportId(transportId),
    allowList(allowList == nullptr ? IPAllowList() : *allowList) {
	degraded = makeReference<AsyncVar<bool>>(false);
	pingLogger = pingLatencyLogger(this);

	connectionHistoryLoggerF = connectionHistoryLogger(this);
}

#define CONNECT_PACKET_V0 0x0FDB00A444020001LL
#define CONNECT_PACKET_V0_SIZE 14

#pragma pack(push, 1)
struct ConnectPacket {
	// The value does not include the size of `connectPacketLength` itself,
	// but only the other fields of this structure.
	// FIXME: document the wire protocol in more detail than just the prior sentence.
	uint32_t connectPacketLength = 0;
	ProtocolVersion protocolVersion; // Expect currentProtocolVersion

	uint16_t canonicalRemotePort = 0; // Port number to reconnect to the originating process
	uint64_t connectionId = 0; // Multi-version clients will use the same Id for both connections, other connections
	                           // will set this to zero. Added at protocol Version 0x0FDB00A444020001.

	// IP Address to reconnect to the originating process. Only one of these must be populated.
	uint32_t canonicalRemoteIp4 = 0;

	enum ConnectPacketFlags { FLAG_IPV6 = 1 };
	uint16_t flags = 0;
	uint8_t canonicalRemoteIp6[16] = { 0 };

	ConnectPacket() = default;

	IPAddress canonicalRemoteIp() const {
		if (isIPv6()) {
			IPAddress::IPAddressStore store;
			memcpy(store.data(), canonicalRemoteIp6, sizeof(canonicalRemoteIp6));
			return IPAddress(store);
		} else {
			return IPAddress(canonicalRemoteIp4);
		}
	}

	void setCanonicalRemoteIp(const IPAddress& ip) {
		if (ip.isV6()) {
			flags = flags | FLAG_IPV6;
			memcpy(&canonicalRemoteIp6, ip.toV6().data(), 16);
		} else {
			flags = flags & ~FLAG_IPV6;
			canonicalRemoteIp4 = ip.toV4();
		}
	}

	bool isIPv6() const { return flags & FLAG_IPV6; }

	uint32_t totalPacketSize() const { return connectPacketLength + sizeof(connectPacketLength); }

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, connectPacketLength);
		if (connectPacketLength > sizeof(ConnectPacket) - sizeof(connectPacketLength)) {
			ASSERT(!g_network->isSimulated());
			TraceEvent("SerializationFailed").backtrace();
			throw serialization_failed();
		}

		serializer(ar, protocolVersion, canonicalRemotePort, connectionId, canonicalRemoteIp4);
		// We can send everything in serialized packet, since the current version of ConnectPacket
		// is backward compatible with CONNECT_PACKET_V0.
		serializer(ar, flags);
		ar.serializeBytes(&canonicalRemoteIp6, sizeof(canonicalRemoteIp6));
	}
};

#pragma pack(pop)

															#line 1333 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
[[nodiscard]] static Future<Void> connectionReader( TransportData* const& transport, Reference<IConnection> const& conn, Reference<struct Peer> const& peer, Promise<Reference<struct Peer>> const& onConnected );

#line 620 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

static void sendLocal(TransportData* self, ISerializeSource const& what, const Endpoint& destination);
static ReliablePacket* sendPacket(TransportData* self,
                                  Reference<Peer> peer,
                                  ISerializeSource const& what,
                                  const Endpoint& destination,
                                  bool reliable);

															#line 1345 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionMonitor()
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionMonitorActor>
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionMonitorActorState {
															#line 1352 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionMonitorActorState(Reference<Peer> const& peer) 
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : peer(peer),
															#line 629 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   remotePingEndpoint({ peer->destination }, Endpoint::wellKnownToken(WLTOKEN_PING_PACKET))
															#line 1361 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionMonitor", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionMonitorActorState() 
	{
		fdb_probe_actor_destroy("connectionMonitor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 631 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			peer->lastDataPacketSentTime = now();
															#line 632 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 1378 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionMonitorActorState();
		static_cast<ConnectionMonitorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 633 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!FlowTransport::isClient() && !peer->destination.isPublic() && peer->compatible)
															#line 1408 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 654 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastRefreshed = now();
															#line 655 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastBytesReceived = peer->bytesReceived;
															#line 656 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 1416 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 676 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::ReadSocket);
															#line 676 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1432 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 2;
															#line 676 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 1, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 1437 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->CONNECTION_MONITOR_LOOP_TIME, TaskPriority::ReadSocket);
															#line 657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1461 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 1;
															#line 657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 0, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 1466 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 658 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (lastBytesReceived < peer->bytesReceived)
															#line 1488 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 659 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastRefreshed = now();
															#line 660 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastBytesReceived = peer->bytesReceived;
															#line 1494 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		else
		{
															#line 661 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (lastRefreshed < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT * FLOW_KNOBS->CONNECTION_MONITOR_INCOMING_IDLE_MULTIPLIER)
															#line 1500 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
				return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 658 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (lastBytesReceived < peer->bytesReceived)
															#line 1513 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 659 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastRefreshed = now();
															#line 660 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			lastBytesReceived = peer->bytesReceived;
															#line 1519 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		else
		{
															#line 661 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (lastRefreshed < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT * FLOW_KNOBS->CONNECTION_MONITOR_INCOMING_IDLE_MULTIPLIER)
															#line 1525 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
				return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state > 0) static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 678 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (peer->reliable.empty() && peer->unsent.empty() && peer->outstandingReplies == 0)
															#line 1613 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 679 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer->peerReferences == 0 && (peer->lastDataPacketSentTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_UNREFERENCED_CLOSE_DELAY))
															#line 1617 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 682 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				return a_body1Catch1(connection_unreferenced(), std::max(0, loopDepth - 1));
															#line 1621 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 683 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (FlowTransport::isClient() && peer->compatible && peer->destination.isPublic() && (peer->lastConnectTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT) && (peer->lastDataPacketSentTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT))
															#line 1627 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 687 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					return a_body1Catch1(connection_idle(), std::max(0, loopDepth - 1));
															#line 1631 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
			}
		}
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayJittered(FLOW_KNOBS->CONNECTION_MONITOR_LOOP_TIME, TaskPriority::ReadSocket);
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1639 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 3;
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 2, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 1644 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 678 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (peer->reliable.empty() && peer->unsent.empty() && peer->outstandingReplies == 0)
															#line 1653 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 679 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer->peerReferences == 0 && (peer->lastDataPacketSentTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_UNREFERENCED_CLOSE_DELAY))
															#line 1657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 682 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				return a_body1Catch1(connection_unreferenced(), std::max(0, loopDepth - 1));
															#line 1661 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 683 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (FlowTransport::isClient() && peer->compatible && peer->destination.isPublic() && (peer->lastConnectTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT) && (peer->lastDataPacketSentTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_IDLE_TIMEOUT))
															#line 1667 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 687 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					return a_body1Catch1(connection_idle(), std::max(0, loopDepth - 1));
															#line 1671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
			}
		}
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayJittered(FLOW_KNOBS->CONNECTION_MONITOR_LOOP_TIME, TaskPriority::ReadSocket);
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1679 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 3;
															#line 691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 2, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 1684 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state > 0) static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 694 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		pingRequest = PingRequest();
															#line 695 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		FlowTransport::transport().sendUnreliable(SerializeSource<PingRequest>(pingRequest), remotePingEndpoint, true);
															#line 696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startingBytes = peer->bytesReceived;
															#line 697 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		timeouts = 0;
															#line 698 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startTime = now();
															#line 699 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 1778 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 694 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		pingRequest = PingRequest();
															#line 695 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		FlowTransport::transport().sendUnreliable(SerializeSource<PingRequest>(pingRequest), remotePingEndpoint, true);
															#line 696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startingBytes = peer->bytesReceived;
															#line 697 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		timeouts = 0;
															#line 698 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startTime = now();
															#line 699 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 1797 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state > 0) static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 701 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 700 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1896 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_3.get(), loopDepth); };
															#line 719 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_4 = pingRequest.reply.getFuture();
															#line 1900 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont4loopBody1when2(__when_expr_4.get(), loopDepth); };
															#line 725 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_5 = peer->resetPing.onTrigger();
															#line 1904 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont4loopBody1when3(__when_expr_5.get(), loopDepth); };
		static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 4;
															#line 701 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 3, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 719 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 4, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 725 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ConnectionMonitorActor, 5, Void >*>(static_cast<ConnectionMonitorActor*>(this)));
															#line 1913 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		peer->timeoutCount++;
															#line 703 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (startingBytes == peer->bytesReceived)
															#line 1943 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 704 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer->destination.isPublic())
															#line 1947 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 705 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->pingLatencies.addSample(now() - startTime);
															#line 1951 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 707 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent("ConnectionTimeout").suppressFor(1.0).detail("WithAddr", peer->destination);
															#line 708 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1Catch1(connection_failed(), std::max(0, loopDepth - 2));
															#line 1957 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (timeouts > 1)
															#line 1961 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 711 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevWarnAlways, "ConnectionSlowPing") .suppressFor(1.0) .detail("WithAddr", peer->destination) .detail("Timeouts", timeouts);
															#line 1965 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 716 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startingBytes = peer->bytesReceived;
															#line 717 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		timeouts++;
															#line 1971 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
															#line 702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		peer->timeoutCount++;
															#line 703 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (startingBytes == peer->bytesReceived)
															#line 1982 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 704 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (peer->destination.isPublic())
															#line 1986 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 705 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->pingLatencies.addSample(now() - startTime);
															#line 1990 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 707 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent("ConnectionTimeout").suppressFor(1.0).detail("WithAddr", peer->destination);
															#line 708 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1Catch1(connection_failed(), std::max(0, loopDepth - 2));
															#line 1996 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (timeouts > 1)
															#line 2000 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 711 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevWarnAlways, "ConnectionSlowPing") .suppressFor(1.0) .detail("WithAddr", peer->destination) .detail("Timeouts", timeouts);
															#line 2004 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 716 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		startingBytes = peer->bytesReceived;
															#line 717 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		timeouts++;
															#line 2010 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 720 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (peer->destination.isPublic())
															#line 2019 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 721 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			peer->pingLatencies.addSample(now() - startTime);
															#line 2023 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when2(Void && _,int loopDepth) 
	{
															#line 720 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (peer->destination.isPublic())
															#line 2033 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 721 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			peer->pingLatencies.addSample(now() - startTime);
															#line 2037 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when3(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when3(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ConnectionMonitorActor*>(this)->actor_wait_state > 0) static_cast<ConnectionMonitorActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 3, Void >::remove();
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 4, Void >::remove();
		static_cast<ConnectionMonitorActor*>(this)->ActorCallback< ConnectionMonitorActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ConnectionMonitorActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ConnectionMonitorActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<Peer> peer;
															#line 629 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Endpoint remotePingEndpoint;
															#line 654 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	double lastRefreshed;
															#line 655 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int64_t lastBytesReceived;
															#line 694 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	PingRequest pingRequest;
															#line 696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int64_t startingBytes;
															#line 697 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int timeouts;
															#line 698 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	double startTime;
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionMonitor()
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionMonitorActor final : public Actor<Void>, public ActorCallback< ConnectionMonitorActor, 0, Void >, public ActorCallback< ConnectionMonitorActor, 1, Void >, public ActorCallback< ConnectionMonitorActor, 2, Void >, public ActorCallback< ConnectionMonitorActor, 3, Void >, public ActorCallback< ConnectionMonitorActor, 4, Void >, public ActorCallback< ConnectionMonitorActor, 5, Void >, public FastAllocated<ConnectionMonitorActor>, public ConnectionMonitorActorState<ConnectionMonitorActor> {
															#line 2255 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionMonitorActor>::operator new;
	using FastAllocated<ConnectionMonitorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6532383879370987264UL, 14037690648462347520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionMonitorActor, 0, Void >;
friend struct ActorCallback< ConnectionMonitorActor, 1, Void >;
friend struct ActorCallback< ConnectionMonitorActor, 2, Void >;
friend struct ActorCallback< ConnectionMonitorActor, 3, Void >;
friend struct ActorCallback< ConnectionMonitorActor, 4, Void >;
friend struct ActorCallback< ConnectionMonitorActor, 5, Void >;
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionMonitorActor(Reference<Peer> const& peer) 
															#line 2277 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionMonitorActorState<ConnectionMonitorActor>(peer),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionMonitor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11834707007208155904UL, 2727031628447019520UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionMonitorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionMonitor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionMonitor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionMonitorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectionMonitorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ConnectionMonitorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ConnectionMonitorActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> connectionMonitor( Reference<Peer> const& peer ) {
															#line 628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionMonitorActor(peer));
															#line 2313 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 732 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 2318 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionWriter()
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionWriterActor>
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionWriterActorState {
															#line 2325 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionWriterActorState(Reference<Peer> const& self,Reference<IConnection> const& conn) 
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   conn(conn),
															#line 734 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   lastWriteTime(now())
															#line 2336 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionWriter", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionWriterActorState() 
	{
		fdb_probe_actor_destroy("connectionWriter", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 735 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 2351 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionWriterActorState();
		static_cast<ConnectionWriterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 736 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delayJittered( std::max<double>(FLOW_KNOBS->MIN_COALESCE_DELAY, FLOW_KNOBS->MAX_COALESCE_DELAY - (now() - lastWriteTime)), TaskPriority::WriteSocket);
															#line 736 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2383 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 1;
															#line 736 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionWriterActor, 0, Void >*>(static_cast<ConnectionWriterActor*>(this)));
															#line 2388 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 741 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 2397 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 741 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 2406 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state > 0) static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionWriterActor*>(this)->ActorCallback< ConnectionWriterActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionWriterActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 762 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 2490 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 742 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		lastWriteTime = now();
															#line 744 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		int sent = conn->write(self->unsent.getUnsent(), FLOW_KNOBS->MAX_PACKET_SEND_BYTES);
															#line 745 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (sent)
															#line 2510 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 746 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->bytesSent += sent;
															#line 747 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->transport->bytesSent += sent;
															#line 748 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->unsent.sent(sent);
															#line 2518 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 751 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->unsent.empty())
															#line 2522 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 755 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		CODE_PROBE( true, "We didn't write everything, so apparently the write buffer is full.  Wait for it to be nonfull");
															#line 757 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_1 = conn->onWritable();
															#line 757 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2532 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 2;
															#line 757 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionWriterActor, 1, Void >*>(static_cast<ConnectionWriterActor*>(this)));
															#line 2537 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::WriteSocket);
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2561 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 3;
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionWriterActor, 2, Void >*>(static_cast<ConnectionWriterActor*>(this)));
															#line 2566 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::WriteSocket);
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2577 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 3;
															#line 758 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionWriterActor, 2, Void >*>(static_cast<ConnectionWriterActor*>(this)));
															#line 2582 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state > 0) static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionWriterActor*>(this)->ActorCallback< ConnectionWriterActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectionWriterActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state > 0) static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionWriterActor*>(this)->ActorCallback< ConnectionWriterActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectionWriterActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 762 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!(self->unsent.empty()))
															#line 2766 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 763 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->dataToSend.onTrigger();
															#line 763 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2774 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 4;
															#line 763 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionWriterActor, 3, Void >*>(static_cast<ConnectionWriterActor*>(this)));
															#line 2779 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ConnectionWriterActor*>(this)->actor_wait_state > 0) static_cast<ConnectionWriterActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionWriterActor*>(this)->ActorCallback< ConnectionWriterActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionWriterActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConnectionWriterActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<Peer> self;
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IConnection> conn;
															#line 734 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	double lastWriteTime;
															#line 2890 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionWriter()
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionWriterActor final : public Actor<Void>, public ActorCallback< ConnectionWriterActor, 0, Void >, public ActorCallback< ConnectionWriterActor, 1, Void >, public ActorCallback< ConnectionWriterActor, 2, Void >, public ActorCallback< ConnectionWriterActor, 3, Void >, public FastAllocated<ConnectionWriterActor>, public ConnectionWriterActorState<ConnectionWriterActor> {
															#line 2895 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionWriterActor>::operator new;
	using FastAllocated<ConnectionWriterActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13185011669886928128UL, 1805011452630795520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionWriterActor, 0, Void >;
friend struct ActorCallback< ConnectionWriterActor, 1, Void >;
friend struct ActorCallback< ConnectionWriterActor, 2, Void >;
friend struct ActorCallback< ConnectionWriterActor, 3, Void >;
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionWriterActor(Reference<Peer> const& self,Reference<IConnection> const& conn) 
															#line 2915 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionWriterActorState<ConnectionWriterActor>(self, conn),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionWriter", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17093672846567471872UL, 15000040872837836032UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionWriterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionWriter");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionWriter", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionWriterActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectionWriterActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ConnectionWriterActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ConnectionWriterActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> connectionWriter( Reference<Peer> const& self, Reference<IConnection> const& conn ) {
															#line 733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionWriterActor(self, conn));
															#line 2951 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 766 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 2956 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via delayedHealthUpdate()
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class DelayedHealthUpdateActor>
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class DelayedHealthUpdateActorState {
															#line 2963 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	DelayedHealthUpdateActorState(NetworkAddress const& address,bool* const& tooManyConnectionsClosed) 
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : address(address),
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   tooManyConnectionsClosed(tooManyConnectionsClosed),
															#line 768 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   start(now())
															#line 2974 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("delayedHealthUpdate", reinterpret_cast<unsigned long>(this));

	}
	~DelayedHealthUpdateActorState() 
	{
		fdb_probe_actor_destroy("delayedHealthUpdate", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 769 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 2989 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DelayedHealthUpdateActorState();
		static_cast<DelayedHealthUpdateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 786 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!static_cast<DelayedHealthUpdateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelayedHealthUpdateActorState(); static_cast<DelayedHealthUpdateActor*>(this)->destroy(); return 0; }
															#line 3012 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		new (&static_cast<DelayedHealthUpdateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DelayedHealthUpdateActorState();
		static_cast<DelayedHealthUpdateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 770 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (FLOW_KNOBS->HEALTH_MONITOR_MARK_FAILED_UNSTABLE_CONNECTIONS && FlowTransport::transport().healthMonitor()->tooManyConnectionsClosed(address) && address.isPublic())
															#line 3031 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 772 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delayJittered(FLOW_KNOBS->MAX_RECONNECTION_TIME * 2.0);
															#line 772 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<DelayedHealthUpdateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3037 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DelayedHealthUpdateActor*>(this)->actor_wait_state = 1;
															#line 772 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DelayedHealthUpdateActor, 0, Void >*>(static_cast<DelayedHealthUpdateActor*>(this)));
															#line 3042 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 774 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (*tooManyConnectionsClosed)
															#line 3049 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 775 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent("TooManyConnectionsClosedMarkAvailable") .detail("Dest", address) .detail("StartTime", start) .detail("TimeElapsed", now() - start) .detail("ClosedCount", FlowTransport::transport().healthMonitor()->closedConnectionsCount(address));
															#line 780 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				*tooManyConnectionsClosed = false;
															#line 3055 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 782 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			IFailureMonitor::failureMonitor().setStatus(address, FailureStatus(false));
															#line 3059 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DelayedHealthUpdateActor*>(this)->actor_wait_state > 0) static_cast<DelayedHealthUpdateActor*>(this)->actor_wait_state = 0;
		static_cast<DelayedHealthUpdateActor*>(this)->ActorCallback< DelayedHealthUpdateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DelayedHealthUpdateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelayedHealthUpdateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DelayedHealthUpdateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelayedHealthUpdateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DelayedHealthUpdateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelayedHealthUpdateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	NetworkAddress address;
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool* tooManyConnectionsClosed;
															#line 768 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	double start;
															#line 3177 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via delayedHealthUpdate()
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class DelayedHealthUpdateActor final : public Actor<Void>, public ActorCallback< DelayedHealthUpdateActor, 0, Void >, public FastAllocated<DelayedHealthUpdateActor>, public DelayedHealthUpdateActorState<DelayedHealthUpdateActor> {
															#line 3182 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<DelayedHealthUpdateActor>::operator new;
	using FastAllocated<DelayedHealthUpdateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3358835115068293632UL, 1045251005394473728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DelayedHealthUpdateActor, 0, Void >;
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	DelayedHealthUpdateActor(NetworkAddress const& address,bool* const& tooManyConnectionsClosed) 
															#line 3199 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DelayedHealthUpdateActorState<DelayedHealthUpdateActor>(address, tooManyConnectionsClosed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13660702714924964352UL, 9796935992414942720UL);
		ActorExecutionContextHelper __helper(static_cast<DelayedHealthUpdateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("delayedHealthUpdate");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("delayedHealthUpdate", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DelayedHealthUpdateActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> delayedHealthUpdate( NetworkAddress const& address, bool* const& tooManyConnectionsClosed ) {
															#line 767 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new DelayedHealthUpdateActor(address, tooManyConnectionsClosed));
															#line 3232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 3237 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionKeeper()
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionKeeperActor>
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionKeeperActorState {
															#line 3244 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionKeeperActorState(Reference<Peer> const& self,Reference<IConnection> const& conn = Reference<IConnection>(),Future<Void> const& reader = Void()) 
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   conn(conn),
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   reader(reader)
															#line 3255 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionKeeper", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionKeeperActorState() 
	{
		fdb_probe_actor_destroy("connectionKeeper", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 792 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevDebug, "ConnectionKeeper", conn ? conn->getDebugID() : UID()) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination) .detail("ConnSet", (bool)conn);
															#line 796 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			ASSERT_WE_THINK(FlowTransport::transport().getLocalAddress() != self->destination);
															#line 798 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			delayedHealthUpdateF = Future<Void>();
															#line 799 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			firstConnFailedTime = Optional<double>();
															#line 800 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			retryConnect = false;
															#line 801 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			tooManyConnectionsClosed = false;
															#line 803 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 3282 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionKeeperActorState();
		static_cast<ConnectionKeeperActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 805 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			delayedHealthUpdateF = Future<Void>();
															#line 807 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!conn)
															#line 3315 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 808 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->outgoingConnectionIdle = true;
															#line 810 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				;
															#line 3321 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1loopHead1(loopDepth);
			}
			else
			{
															#line 898 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->outgoingConnectionIdle = false;
															#line 899 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->lastConnectTime = now();
															#line 3330 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1cont2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 926 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->connected = false;
															#line 927 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			delayedHealthUpdateF.cancel();
															#line 928 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (now() - self->lastConnectTime > FLOW_KNOBS->RECONNECTION_RESET_TIME)
															#line 3357 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 929 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->reconnectionDelay = FLOW_KNOBS->INITIAL_RECONNECTION_TIME;
															#line 3361 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 931 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->reconnectionDelay = std::min(FLOW_KNOBS->MAX_RECONNECTION_TIME, self->reconnectionDelay * FLOW_KNOBS->RECONNECTION_TIME_GROWTH_RATE);
															#line 3367 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 935 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (firstConnFailedTime.present())
															#line 3371 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 936 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (now() - firstConnFailedTime.get() > FLOW_KNOBS->PEER_UNAVAILABLE_FOR_LONG_TIME_TIMEOUT)
															#line 3375 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 937 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					TraceEvent(SevWarnAlways, "PeerUnavailableForLongTime", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 941 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					firstConnFailedTime = now() - FLOW_KNOBS->PEER_UNAVAILABLE_FOR_LONG_TIME_TIMEOUT / 2.0;
															#line 3381 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 944 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				firstConnFailedTime = now();
															#line 3388 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 949 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			retryConnect = true;
															#line 950 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() == error_code_connection_failed)
															#line 3394 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 951 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (!self->destination.isPublic())
															#line 3398 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 953 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					IFailureMonitor::failureMonitor().setStatus(self->destination, FailureStatus(true));
															#line 3402 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
				else
				{
															#line 954 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (now() - firstConnFailedTime.get() > FLOW_KNOBS->FAILURE_DETECTION_DELAY)
															#line 3408 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 955 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						IFailureMonitor::failureMonitor().setStatus(self->destination, FailureStatus(true));
															#line 3412 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
				}
			}
															#line 959 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->discardUnreliablePackets();
															#line 960 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			reader = Future<Void>();
															#line 961 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			bool ok = e.code() == error_code_connection_failed || e.code() == error_code_actor_cancelled || e.code() == error_code_connection_unreferenced || e.code() == error_code_connection_idle || (g_network->isSimulated() && e.code() == error_code_checksum_failed);
															#line 965 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (self->compatible)
															#line 3424 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 966 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent(ok ? SevInfo : SevWarnAlways, "ConnectionClosed", conn ? conn->getDebugID() : UID()) .errorUnsuppressed(e) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 3428 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 972 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent( ok ? SevInfo : SevWarnAlways, "IncompatibleConnectionClosed", conn ? conn->getDebugID() : UID()) .errorUnsuppressed(e) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 980 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->compatible = true;
															#line 3436 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 983 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (self->destination.isPublic() && IFailureMonitor::failureMonitor().getState(self->destination).isAvailable() && !FlowTransport::isClient())
															#line 3440 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 986 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				auto& it = self->transport->closedPeers[self->destination];
															#line 987 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (now() - it.second > FLOW_KNOBS->TOO_MANY_CONNECTIONS_CLOSED_RESET_DELAY)
															#line 3446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 988 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					it.first = now();
															#line 3450 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
				else
				{
															#line 989 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (now() - it.first > FLOW_KNOBS->TOO_MANY_CONNECTIONS_CLOSED_TIMEOUT)
															#line 3456 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 990 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						TraceEvent(SevWarnAlways, "TooManyConnectionsClosed", conn ? conn->getDebugID() : UID()) .suppressFor(5.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 994 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						self->transport->degraded->set(true);
															#line 3462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
				}
															#line 996 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it.second = now();
															#line 3467 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 999 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (conn)
															#line 3471 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1000 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (self->destination.isPublic() && e.code() == error_code_connection_failed)
															#line 3475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1001 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					FlowTransport::transport().healthMonitor()->reportPeerClosed(self->destination);
															#line 1002 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (FLOW_KNOBS->HEALTH_MONITOR_MARK_FAILED_UNSTABLE_CONNECTIONS && FlowTransport::transport().healthMonitor()->tooManyConnectionsClosed(self->destination) && self->destination.isPublic())
															#line 3481 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1005 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						TraceEvent("TooManyConnectionsClosedMarkFailed") .detail("Dest", self->destination) .detail( "ClosedCount", FlowTransport::transport().healthMonitor()->closedConnectionsCount(self->destination));
															#line 1010 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						tooManyConnectionsClosed = true;
															#line 1011 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						IFailureMonitor::failureMonitor().setStatus(self->destination, FailureStatus(true));
															#line 3489 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
				}
															#line 1015 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				conn->close();
															#line 1016 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				conn = Reference<IConnection>();
															#line 1021 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (e.code() != error_code_incompatible_protocol_version)
															#line 3498 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1022 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					self->protocolVersion->set(Optional<ProtocolVersion>());
															#line 3502 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
			}
															#line 1027 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			IFailureMonitor::failureMonitor().notifyDisconnect(self->destination);
															#line 1028 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			Promise<Void> disconnect = self->disconnect;
															#line 1029 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->disconnect = Promise<Void>();
															#line 1030 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			disconnect.send(Void());
															#line 1032 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3515 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1033 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 3519 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 1036 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (self->peerReferences <= 0 && self->reliable.empty() && self->unsent.empty() && self->outstandingReplies == 0)
															#line 3523 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1038 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent("PeerDestroy") .errorUnsuppressed(e) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 1043 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->connect.cancel();
															#line 1044 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->transport->peers.erase(self->destination);
															#line 1045 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->transport->orderedAddresses.erase(self->destination);
															#line 1046 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (!static_cast<ConnectionKeeperActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectionKeeperActorState(); static_cast<ConnectionKeeperActor*>(this)->destroy(); return 0; }
															#line 3535 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				new (&static_cast<ConnectionKeeperActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ConnectionKeeperActorState();
				static_cast<ConnectionKeeperActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 902 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		firstConnFailedTime.reset();
															#line 3555 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		try {
															#line 904 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->transport->countConnEstablished++;
															#line 905 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!delayedHealthUpdateF.isValid())
															#line 3561 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 906 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				delayedHealthUpdateF = delayedHealthUpdate(self->destination, &tooManyConnectionsClosed);
															#line 3565 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 907 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->connected = true;
															#line 908 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_8 = connectionWriter(self, conn) || reader || connectionMonitor(self) || self->resetConnection.onTrigger();
															#line 908 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 3573 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1loopBody1cont2Catch1(__when_expr_8.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_8.get(), loopDepth); };
			static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 6;
															#line 908 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 8, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 3578 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 827 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(self->destination.isPublic());
															#line 828 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->outgoingConnectionIdle = false;
															#line 829 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayJittered(std::max(0.0, self->lastConnectTime + self->reconnectionDelay - now()));
															#line 829 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3599 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 2;
															#line 829 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 2, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 3604 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 810 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!(self->unsent.empty()))
															#line 3620 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 812 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		Future<Void> retryConnectF = Never();
															#line 813 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (retryConnect)
															#line 3628 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 814 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			retryConnectF = IFailureMonitor::failureMonitor().getState(self->destination).isAvailable() ? delay(FLOW_KNOBS->FAILURE_DETECTION_DELAY) : delay(FLOW_KNOBS->SERVER_REQUEST_INTERVAL);
															#line 3632 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 820 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->dataToSend.onTrigger();
															#line 819 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3638 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 821 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_1 = retryConnectF;
															#line 3642 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 1;
															#line 820 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 0, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 821 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 1, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 3649 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 0, Void >::remove();
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 832 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->lastConnectTime = now();
															#line 834 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		TraceEvent("ConnectingTo", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination) .detail("PeerReferences", self->peerReferences) .detail("FailureStatus", IFailureMonitor::failureMonitor().getState(self->destination).isAvailable() ? "OK" : "FAILED");
															#line 842 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		++self->connectOutgoingCount;
															#line 3826 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		try {
															#line 845 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Reference<IConnection>> __when_expr_3 = INetworkConnections::net()->connect(self->destination);
															#line 844 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 3832 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont4Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
															#line 881 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 3836 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont4Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont4when2(__when_expr_4.get(), loopDepth); };
			static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 3;
															#line 845 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 881 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 4, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 3843 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 832 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->lastConnectTime = now();
															#line 834 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		TraceEvent("ConnectingTo", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination) .detail("PeerReferences", self->peerReferences) .detail("FailureStatus", IFailureMonitor::failureMonitor().getState(self->destination).isAvailable() ? "OK" : "FAILED");
															#line 842 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		++self->connectOutgoingCount;
															#line 3862 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		try {
															#line 845 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Reference<IConnection>> __when_expr_3 = INetworkConnections::net()->connect(self->destination);
															#line 844 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 3868 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont4Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
															#line 881 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 3872 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont4Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont4when2(__when_expr_4.get(), loopDepth); };
			static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 3;
															#line 845 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 881 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 4, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 3879 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 886 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			++self->connectFailedCount;
															#line 887 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() != error_code_connection_failed)
															#line 3978 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 888 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				return a_body1loopBody1Catch1(e, loopDepth);
															#line 3982 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 890 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent("ConnectionTimedOut", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 895 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1loopBody1Catch1(e, loopDepth);
															#line 3988 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Reference<IConnection> const& _conn,int loopDepth) 
	{
															#line 847 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn = _conn;
															#line 848 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countOutgoingConnectionCreated = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/OutgoingConnectionCreated");
															#line 850 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countOutgoingConnectionCreated->increment(1);
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_5 = conn->connectHandshake();
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 4016 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont4Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont4when1when1(__when_expr_5.get(), loopDepth); };
		static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 4;
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 5, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 4021 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Reference<IConnection> && _conn,int loopDepth) 
	{
															#line 847 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn = _conn;
															#line 848 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countOutgoingConnectionCreated = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/OutgoingConnectionCreated");
															#line 850 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countOutgoingConnectionCreated->increment(1);
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_5 = conn->connectHandshake();
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 4038 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont4Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont4when1when1(__when_expr_5.get(), loopDepth); };
		static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 4;
															#line 851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 5, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 4043 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when2(Void const& _,int loopDepth) 
	{
															#line 882 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1loopBody1cont4Catch1(connection_failed(), loopDepth);
															#line 4052 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1loopBody1cont4when2(Void && _,int loopDepth) 
	{
															#line 882 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1loopBody1cont4Catch1(connection_failed(), loopDepth);
															#line 4060 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >::remove();
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 4, Void >::remove();

	}
	int a_body1loopBody1cont4when1cont1(Void const& _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countOutgoingConnectionHandshakeComplete = SimpleCounter<int64_t>::makeCounter( "/Transport/TLS/OutgoingConnectionHandshakeComplete");
															#line 855 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countOutgoingConnectionHandshakeComplete->increment(1);
															#line 856 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->connectLatencies.addSample(now() - self->lastConnectTime);
															#line 857 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (FlowTransport::isClient())
															#line 4081 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 858 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			IFailureMonitor::failureMonitor().setStatus(self->destination, FailureStatus(false));
															#line 4085 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 860 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->unsent.empty())
															#line 4089 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 861 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			delayedHealthUpdateF = delayedHealthUpdate(self->destination, &tooManyConnectionsClosed);
															#line 864 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_6 = delayedHealthUpdateF;
															#line 863 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 4097 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1cont4Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont4when1cont1when1(__when_expr_6.get(), loopDepth); };
															#line 870 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_7 = self->dataToSend.onTrigger();
															#line 4101 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1cont4Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont4when1cont1when2(__when_expr_7.get(), loopDepth); };
			static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 5;
															#line 864 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 6, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 870 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 7, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 4108 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4when1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont1(Void && _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countOutgoingConnectionHandshakeComplete = SimpleCounter<int64_t>::makeCounter( "/Transport/TLS/OutgoingConnectionHandshakeComplete");
															#line 855 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countOutgoingConnectionHandshakeComplete->increment(1);
															#line 856 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->connectLatencies.addSample(now() - self->lastConnectTime);
															#line 857 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (FlowTransport::isClient())
															#line 4128 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 858 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			IFailureMonitor::failureMonitor().setStatus(self->destination, FailureStatus(false));
															#line 4132 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 860 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (self->unsent.empty())
															#line 4136 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 861 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			delayedHealthUpdateF = delayedHealthUpdate(self->destination, &tooManyConnectionsClosed);
															#line 864 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_6 = delayedHealthUpdateF;
															#line 863 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont4Catch1(actor_cancelled(), loopDepth);
															#line 4144 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1cont4Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont4when1cont1when1(__when_expr_6.get(), loopDepth); };
															#line 870 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_7 = self->dataToSend.onTrigger();
															#line 4148 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1cont4Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont4when1cont1when2(__when_expr_7.get(), loopDepth); };
			static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 5;
															#line 864 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 6, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 870 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ConnectionKeeperActor, 7, Void >*>(static_cast<ConnectionKeeperActor*>(this)));
															#line 4155 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4when1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4when1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4when1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4when1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont4when1cont2(int loopDepth) 
	{
															#line 874 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		TraceEvent("ConnectionExchangingConnectPacket", conn->getDebugID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 878 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->prependConnectPacket();
															#line 879 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		reader = connectionReader(self->transport, conn, self, Promise<Reference<Peer>>());
															#line 4248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4when1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont1when1(Void const& _,int loopDepth) 
	{
															#line 865 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn->close();
															#line 866 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn = Reference<IConnection>();
															#line 867 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		retryConnect = false;
															#line 4267 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont1when1(Void && _,int loopDepth) 
	{
															#line 865 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn->close();
															#line 866 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		conn = Reference<IConnection>();
															#line 867 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		retryConnect = false;
															#line 4280 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4when1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4when1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 6, Void >::remove();
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont4Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 916 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() == error_code_connection_failed || e.code() == error_code_actor_cancelled || e.code() == error_code_connection_unreferenced || (g_network->isSimulated() && e.code() == error_code_checksum_failed))
															#line 4550 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 919 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->transport->countConnClosedWithoutError++;
															#line 4554 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 921 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->transport->countConnClosedWithError++;
															#line 4560 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 923 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1loopBody1Catch1(e, loopDepth);
															#line 4564 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 910 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		TraceEvent("ConnectionReset", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 914 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1loopBody1cont2Catch1(connection_failed(), loopDepth);
															#line 4580 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 910 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		TraceEvent("ConnectionReset", conn ? conn->getDebugID() : UID()) .suppressFor(1.0) .detail("PeerAddr", self->destination) .detail("PeerAddress", self->destination);
															#line 914 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1loopBody1cont2Catch1(connection_failed(), loopDepth);
															#line 4590 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ConnectionKeeperActor*>(this)->actor_wait_state > 0) static_cast<ConnectionKeeperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionKeeperActor*>(this)->ActorCallback< ConnectionKeeperActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< ConnectionKeeperActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< ConnectionKeeperActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<Peer> self;
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IConnection> conn;
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> reader;
															#line 798 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> delayedHealthUpdateF;
															#line 799 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Optional<double> firstConnFailedTime;
															#line 800 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int retryConnect;
															#line 801 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool tooManyConnectionsClosed;
															#line 4683 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionKeeper()
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionKeeperActor final : public Actor<Void>, public ActorCallback< ConnectionKeeperActor, 0, Void >, public ActorCallback< ConnectionKeeperActor, 1, Void >, public ActorCallback< ConnectionKeeperActor, 2, Void >, public ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >, public ActorCallback< ConnectionKeeperActor, 5, Void >, public ActorCallback< ConnectionKeeperActor, 6, Void >, public ActorCallback< ConnectionKeeperActor, 7, Void >, public ActorCallback< ConnectionKeeperActor, 4, Void >, public ActorCallback< ConnectionKeeperActor, 8, Void >, public FastAllocated<ConnectionKeeperActor>, public ConnectionKeeperActorState<ConnectionKeeperActor> {
															#line 4688 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionKeeperActor>::operator new;
	using FastAllocated<ConnectionKeeperActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1660973955874589696UL, 1400972569220235520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionKeeperActor, 0, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 1, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 2, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >;
friend struct ActorCallback< ConnectionKeeperActor, 5, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 6, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 7, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 4, Void >;
friend struct ActorCallback< ConnectionKeeperActor, 8, Void >;
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionKeeperActor(Reference<Peer> const& self,Reference<IConnection> const& conn = Reference<IConnection>(),Future<Void> const& reader = Void()) 
															#line 4713 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionKeeperActorState<ConnectionKeeperActor>(self, conn, reader),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionKeeper", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5184200756758760448UL, 10907201916319375872UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionKeeperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionKeeper");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionKeeper", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 3, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 5, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 6, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ConnectionKeeperActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] Future<Void> connectionKeeper( Reference<Peer> const& self, Reference<IConnection> const& conn = Reference<IConnection>(), Future<Void> const& reader = Void() ) {
															#line 789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionKeeperActor(self, conn, reader));
															#line 4751 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1051 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

Peer::Peer(TransportData* transport, NetworkAddress const& destination)
  : transport(transport), destination(destination), compatible(true), connected(false), outgoingConnectionIdle(true),
    lastConnectTime(0.0), reconnectionDelay(FLOW_KNOBS->INITIAL_RECONNECTION_TIME), peerReferences(-1),
    bytesReceived(0), bytesSent(0), lastDataPacketSentTime(now()), outstandingReplies(0),
    pingLatencies(destination.isPublic() ? FLOW_KNOBS->PING_SKETCH_ACCURACY : 0.1), lastLoggedTime(0.0),
    lastLoggedBytesReceived(0), lastLoggedBytesSent(0), timeoutCount(0),
    protocolVersion(Reference<AsyncVar<Optional<ProtocolVersion>>>(new AsyncVar<Optional<ProtocolVersion>>())),
    connectOutgoingCount(0), connectIncomingCount(0), connectFailedCount(0),
    connectLatencies(destination.isPublic() ? FLOW_KNOBS->PING_SKETCH_ACCURACY : 0.1) {
	IFailureMonitor::failureMonitor().setStatus(destination, FailureStatus(false));
}

void Peer::send(PacketBuffer* pb, ReliablePacket* rp, bool firstUnsent) {
	unsent.setWriteBuffer(pb);
	if (rp)
		reliable.insert(rp);
	if (firstUnsent)
		dataToSend.trigger();
}

void Peer::prependConnectPacket() {
	// Send the ConnectPacket expected at the beginning of a new connection
	ConnectPacket pkt;
	if (transport->localAddresses.getAddressList().address.isTLS() == destination.isTLS()) {
		pkt.canonicalRemotePort = transport->localAddresses.getAddressList().address.port;
		pkt.setCanonicalRemoteIp(transport->localAddresses.getAddressList().address.ip);
	} else if (transport->localAddresses.getAddressList().secondaryAddress.present()) {
		pkt.canonicalRemotePort = transport->localAddresses.getAddressList().secondaryAddress.get().port;
		pkt.setCanonicalRemoteIp(transport->localAddresses.getAddressList().secondaryAddress.get().ip);
	} else {
		// a "mixed" TLS/non-TLS connection is like a client/server connection - there's no way to reverse it
		pkt.canonicalRemotePort = 0;
		pkt.setCanonicalRemoteIp(IPAddress(0));
	}

	pkt.connectPacketLength = sizeof(pkt) - sizeof(pkt.connectPacketLength);
	pkt.protocolVersion = g_network->protocolVersion();
	pkt.protocolVersion.addObjectSerializerFlag();
	pkt.connectionId = transport->transportId;

	PacketBuffer *pb_first = PacketBuffer::create(), *pb_end = nullptr;
	PacketWriter wr(pb_first, nullptr, Unversioned());
	pkt.serialize(wr);
	pb_end = wr.finish();
#if VALGRIND
	SendBuffer* checkbuf = pb_first;
	while (checkbuf) {
		int size = checkbuf->bytes_written;
		const uint8_t* data = checkbuf->data();
		VALGRIND_CHECK_MEM_IS_DEFINED(data, size);
		checkbuf = checkbuf->next;
	}
#endif
	unsent.prependWriteBuffer(pb_first, pb_end);
}

void Peer::discardUnreliablePackets() {
	// Throw away the current unsent list, dropping the reference count on each PacketBuffer that accounts for presence
	// in the unsent list
	unsent.discardAll();

	// If there are reliable packets, compact reliable packets into a new unsent range
	if (!reliable.empty()) {
		PacketBuffer* pb = unsent.getWriteBuffer();
		pb = reliable.compact(pb, nullptr);
		unsent.setWriteBuffer(pb);
	}
}

void Peer::onIncomingConnection(Reference<Peer> self, Reference<IConnection> conn, Future<Void> reader) {
	// In case two processes are trying to connect to each other simultaneously, the process with the larger canonical
	// NetworkAddress gets to keep its outgoing connection.
	++self->connectIncomingCount;
	if (!destination.isPublic() && !outgoingConnectionIdle)
		throw address_in_use();
	NetworkAddress compatibleAddr = transport->localAddresses.getAddressList().address;
	if (transport->localAddresses.getAddressList().secondaryAddress.present() &&
	    transport->localAddresses.getAddressList().secondaryAddress.get().isTLS() == destination.isTLS()) {
		compatibleAddr = transport->localAddresses.getAddressList().secondaryAddress.get();
	}

	if (!destination.isPublic() || outgoingConnectionIdle || destination > compatibleAddr ||
	    (lastConnectTime > 1.0 && now() - lastConnectTime > FLOW_KNOBS->ALWAYS_ACCEPT_DELAY)) {
		// Keep the new connection
		TraceEvent("IncomingConnection"_audit, conn->getDebugID())
		    .suppressFor(1.0)
		    .detail("FromAddr", conn->getPeerAddress())
		    .detail("CanonicalAddr", destination)
		    .detail("IsPublic", destination.isPublic())
		    .detail("Trusted", self->transport->allowList(conn->getPeerAddress().ip) && conn->hasTrustedPeer());

		connect.cancel();
		prependConnectPacket();
		connect = connectionKeeper(self, conn, reader);
	} else {
		// Keep our prior connection
		TraceEvent("RedundantConnection", conn->getDebugID())
		    .suppressFor(1.0)
		    .detail("FromAddr", conn->getPeerAddress().toString())
		    .detail("CanonicalAddr", destination)
		    .detail("LocalAddr", compatibleAddr);

		reader.cancel();
		conn->close();
	}
}

TransportData::~TransportData() {
	for (auto& p : peers) {
		p.second->connect.cancel();
	}
}

static bool checkCompatible(const PeerCompatibilityPolicy& policy, ProtocolVersion version) {
	switch (policy.requirement) {
	case RequirePeer::Exactly:
		return version.version() == policy.version.version();
	case RequirePeer::AtLeast:
		return version.version() >= policy.version.version();
	default:
		ASSERT(false);
		return false;
	}
}

// This actor looks up the task associated with an endpoint
// and sends the message to it. The actual deserialization will
// be done by that task (see NetworkMessageReceiver).
															#line 4884 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via deliver()
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class DeliverActor>
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class DeliverActorState {
															#line 4891 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	DeliverActorState(TransportData* const& self,Endpoint const& destination,TaskPriority const& priority,ArenaReader const& reader,NetworkAddress const& peerAddress,bool const& isTrustedPeer,InReadSocket const& inReadSocket,Future<Void> const& disconnect) 
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   destination(destination),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   priority(priority),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   reader(reader),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   peerAddress(peerAddress),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   isTrustedPeer(isTrustedPeer),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   inReadSocket(inReadSocket),
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   disconnect(disconnect)
															#line 4912 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("deliver", reinterpret_cast<unsigned long>(this));

	}
	~DeliverActorState() 
	{
		fdb_probe_actor_destroy("deliver", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1193 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (priority < TaskPriority::ReadSocket || !inReadSocket)
															#line 4927 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1194 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				StrictFuture<Void> __when_expr_0 = orderedDelay(0, priority);
															#line 1194 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 4933 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				static_cast<DeliverActor*>(this)->actor_wait_state = 1;
															#line 1194 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeliverActor, 0, Void >*>(static_cast<DeliverActor*>(this)));
															#line 4937 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 1196 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_network->setCurrentTask(priority);
															#line 4944 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<DeliverActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1199 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		auto receiver = self->endpoints.get(destination.token);
															#line 1200 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (receiver && (isTrustedPeer || receiver->isPublic()))
															#line 4969 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1201 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!checkCompatible(receiver->peerCompatibilityPolicy(), reader.protocolVersion()))
															#line 4973 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1202 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				delete static_cast<DeliverActor*>(this);
															#line 4977 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				return 0;
			}
			try {
															#line 1205 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ASSERT(g_currentDeliveryPeerAddress == NetworkAddressList());
															#line 1206 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ASSERT(!g_currentDeliverPeerAddressTrusted);
															#line 1207 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliveryPeerAddress = destination.addresses;
															#line 1208 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliverPeerAddressTrusted = isTrustedPeer;
															#line 1209 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliveryPeerDisconnect = disconnect;
															#line 1210 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				StringRef data = reader.arenaReadAll();
															#line 1211 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ASSERT(data.size() > 8);
															#line 1212 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ArenaObjectReader objReader(reader.arena(), reader.arenaReadAll(), AssumeVersion(reader.protocolVersion()));
															#line 1213 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				receiver->receive(objReader);
															#line 1214 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliveryPeerAddress = NetworkAddressList();
															#line 1215 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliverPeerAddressTrusted = false;
															#line 1216 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				g_currentDeliveryPeerDisconnect = Future<Void>();
															#line 5005 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1cont8(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1231 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (destination.token.first() & TOKEN_STREAM_FLAG)
															#line 5018 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1233 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (receiver)
															#line 5022 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1234 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					TraceEvent(SevWarnAlways, "AttemptedRPCToPrivatePrevented"_audit) .detail("From", peerAddress) .detail("Token", destination.token) .detail("Receiver", typeid(*receiver).name());
															#line 1238 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					ASSERT(!self->isLocalAddress(destination.getPrimaryAddress()));
															#line 1239 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					Reference<Peer> peer = self->getOrOpenPeer(destination.getPrimaryAddress());
															#line 1240 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					sendPacket(self, peer, SerializeSource<UID>(destination.token), Endpoint::wellKnown(destination.addresses, WLTOKEN_UNAUTHORIZED_ENDPOINT), false);
															#line 5032 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1246 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (destination.token.first() != -1)
															#line 5038 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1247 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						if (self->isLocalAddress(destination.getPrimaryAddress()))
															#line 5042 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						{
															#line 1248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							sendLocal(self, SerializeSource<UID>(destination.token), Endpoint::wellKnown(destination.addresses, WLTOKEN_ENDPOINT_NOT_FOUND));
															#line 5046 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
						else
						{
															#line 1252 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							Reference<Peer> peer = self->getOrOpenPeer(destination.getPrimaryAddress());
															#line 1253 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							sendPacket(self, peer, SerializeSource<UID>(destination.token), Endpoint::wellKnown(destination.addresses, WLTOKEN_ENDPOINT_NOT_FOUND), false);
															#line 5054 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
					}
				}
			}
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeliverActor*>(this)->actor_wait_state > 0) static_cast<DeliverActor*>(this)->actor_wait_state = 0;
		static_cast<DeliverActor*>(this)->ActorCallback< DeliverActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeliverActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deliver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeliverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deliver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeliverActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deliver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeliverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deliver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeliverActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deliver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeliverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deliver", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1218 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			g_currentDeliveryPeerAddress = NetworkAddressList();
															#line 1219 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			g_currentDeliverPeerAddressTrusted = false;
															#line 1220 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			g_currentDeliveryPeerDisconnect = Future<Void>();
															#line 1221 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevError, "ReceiverError") .error(e) .detail("Token", destination.token.toString()) .detail("Peer", destination.getPrimaryAddress()) .detail("PeerAddress", destination.getPrimaryAddress());
															#line 1226 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!FlowTransport::isClient())
															#line 5176 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1227 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				flushAndExit(FDB_EXIT_ERROR);
															#line 5180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 1229 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 5184 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont16(int loopDepth) 
	{
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		delete static_cast<DeliverActor*>(this);
															#line 5211 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		return 0;

		return loopDepth;
	}
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Endpoint destination;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TaskPriority priority;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ArenaReader reader;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	NetworkAddress peerAddress;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool isTrustedPeer;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	InReadSocket inReadSocket;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> disconnect;
															#line 5232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via deliver()
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class DeliverActor final : public Actor<void>, public ActorCallback< DeliverActor, 0, Void >, public FastAllocated<DeliverActor>, public DeliverActorState<DeliverActor> {
															#line 5237 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<DeliverActor>::operator new;
	using FastAllocated<DeliverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16376429846371329536UL, 14364326376692589568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< DeliverActor, 0, Void >;
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	DeliverActor(TransportData* const& self,Endpoint const& destination,TaskPriority const& priority,ArenaReader const& reader,NetworkAddress const& peerAddress,bool const& isTrustedPeer,InReadSocket const& inReadSocket,Future<Void> const& disconnect) 
															#line 5254 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<void>(),
		   DeliverActorState<DeliverActor>(self, destination, priority, reader, peerAddress, isTrustedPeer, inReadSocket, disconnect),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deliver", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5177313518531985152UL, 17881294458268590080UL);
		ActorExecutionContextHelper __helper(static_cast<DeliverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deliver");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deliver", reinterpret_cast<unsigned long>(this), -1);

	}
};
} // namespace
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
static void deliver( TransportData* const& self, Endpoint const& destination, TaskPriority const& priority, ArenaReader const& reader, NetworkAddress const& peerAddress, bool const& isTrustedPeer, InReadSocket const& inReadSocket, Future<Void> const& disconnect ) {
															#line 1180 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	new DeliverActor(self, destination, priority, reader, peerAddress, isTrustedPeer, inReadSocket, disconnect);
															#line 5278 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1263 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

static void scanPackets(TransportData* transport,
                        uint8_t*& unprocessed_begin, // FIXME: why isn't this called `start`?
                        const uint8_t* e, // FIXME: why isn't this called `end`?
                        Arena& arena,
                        NetworkAddress const& peerAddress,
                        bool isTrustedPeer,
                        ProtocolVersion peerProtocolVersion,
                        Future<Void> disconnect,
                        IsStableConnection isStableConnection) {
	// Find each complete packet in the given byte range and queue a ready task to deliver it.
	// Remove the complete packets from the range by increasing unprocessed_begin.
	// There won't be more than 64K of data plus one packet, so this shouldn't take a long time.
	// FIXME: explain how we are sure that it's not more than 64K plus one packet
	uint8_t* p = unprocessed_begin;

	const bool checksumEnabled = !peerAddress.isTLS();
	loop {
		uint32_t packetLen;
		XXH64_hash_t packetChecksum;

		// Note that these potentially unaligned loads work on x86 and newer ARM but will blow up
		// on many other architectures.  Whether this will be a problem in the future
		// is hard to tell.

		// Read packet length if size is sufficient or stop
		if (e - p < PACKET_LEN_WIDTH)
			break;
		packetLen = *(uint32_t*)p;
		p += PACKET_LEN_WIDTH;

		// Read checksum if present
		if (checksumEnabled) {
			// Read checksum if size is sufficient or stop
			if (e - p < sizeof(packetChecksum))
				break;
			packetChecksum = *(XXH64_hash_t*)p;
			p += sizeof(packetChecksum);
		}

		if (packetLen > FLOW_KNOBS->PACKET_LIMIT) {
			TraceEvent(SevError, "PacketLimitExceeded")
			    .detail("FromPeer", peerAddress.toString())
			    .detail("Length", (int)packetLen);
			throw platform_error();
		}

		if (e - p < packetLen)
			break;

		if (packetLen < sizeof(UID)) {
			if (g_network->isSimulated()) {
				// Same as ASSERT(false), but prints packet length:
				ASSERT_GE(packetLen, sizeof(UID));
			} else {
				TraceEvent(SevError, "PacketTooSmall")
				    .detail("FromPeer", peerAddress.toString())
				    .detail("Length", packetLen);
				throw platform_error();
			}
		}

		if (checksumEnabled) {
			bool isBuggifyEnabled = false;
			if (g_network->isSimulated() && !isStableConnection &&
			    g_network->now() - g_simulator->lastConnectionFailure >
			        g_simulator->connectionFailuresDisableDuration &&
			    BUGGIFY_WITH_PROB(0.0001)) {
				g_simulator->lastConnectionFailure = g_network->now();
				isBuggifyEnabled = true;
				TraceEvent(SevInfo, "BitsFlip").log();
				int flipBits = 32 - (int)floor(log2(deterministicRandom()->randomUInt32()));

				uint32_t firstFlipByteLocation = deterministicRandom()->randomUInt32() % packetLen;
				int firstFlipBitLocation = deterministicRandom()->randomInt(0, 8);
				*(p + firstFlipByteLocation) ^= 1 << firstFlipBitLocation;
				flipBits--;

				for (int i = 0; i < flipBits; i++) {
					uint32_t byteLocation = deterministicRandom()->randomUInt32() % packetLen;
					int bitLocation = deterministicRandom()->randomInt(0, 8);
					if (byteLocation != firstFlipByteLocation || bitLocation != firstFlipBitLocation) {
						*(p + byteLocation) ^= 1 << bitLocation;
					}
				}
			}

			XXH64_hash_t calculatedChecksum = XXH3_64bits(p, packetLen);
			if (calculatedChecksum != packetChecksum) {
				if (isBuggifyEnabled) {
					TraceEvent(SevInfo, "ChecksumMismatchExp")
					    .detail("PacketChecksum", packetChecksum)
					    .detail("CalculatedChecksum", calculatedChecksum)
					    .detail("PeerAddress", peerAddress.toString());
				} else {
					TraceEvent(SevWarnAlways, "ChecksumMismatchUnexp")
					    .detail("PacketChecksum", packetChecksum)
					    .detail("CalculatedChecksum", calculatedChecksum);
				}
				throw checksum_failed();
			} else {
				if (isBuggifyEnabled) {
					TraceEvent(SevError, "ChecksumMatchUnexp")
					    .detail("PacketChecksum", packetChecksum)
					    .detail("CalculatedChecksum", calculatedChecksum);
				}
			}
		}

#if VALGRIND
		VALGRIND_CHECK_MEM_IS_DEFINED(p, packetLen);
#endif
		// remove object serializer flag to account for flat buffer
		peerProtocolVersion.removeObjectSerializerFlag();
		ArenaReader reader(arena, StringRef(p, packetLen), AssumeVersion(peerProtocolVersion));
		UID token;
		reader >> token;

		++transport->countPacketsReceived;

		if (packetLen > FLOW_KNOBS->PACKET_WARNING) {
			TraceEvent(SevWarn, "LargePacketReceived")
			    .suppressFor(1.0)
			    .detail("FromPeer", peerAddress.toString())
			    .detail("Length", (int)packetLen)
			    .detail("Token", token);
		}

		ASSERT(!reader.empty());
		TaskPriority priority = transport->endpoints.getPriority(token);
		// we ignore packets to unknown endpoints if they're not going to a stream anyways, so we can just
		// return here. The main place where this seems to happen is if a ReplyPromise is not waited on
		// long enough.
		// It would be slightly more elegant/readable to put this if-block into the deliver actor, but if
		// we have many messages to UnknownEndpoint we want to optimize earlier. As deliver is an actor it
		// will allocate some state on the heap and this prevents it from doing that.
		if (priority != TaskPriority::UnknownEndpoint || (token.first() & TOKEN_STREAM_FLAG) != 0) {
			deliver(transport,
			        Endpoint({ peerAddress }, token),
			        priority,
			        std::move(reader),
			        peerAddress,
			        isTrustedPeer,
			        InReadSocket::True,
			        disconnect);
		}

		unprocessed_begin = p = p + packetLen;
	}
}

// Given unprocessed buffer [begin, end), check if next packet size is known and return
// enough size for the next packet, whose format is: {size, optional_checksum, data} +
// next_packet_size.
static int getNewBufferSize(const uint8_t* begin,
                            const uint8_t* end,
                            const NetworkAddress& peerAddress,
                            ProtocolVersion peerProtocolVersion) {
	const int len = end - begin;
	if (len < PACKET_LEN_WIDTH) {
		return FLOW_KNOBS->MIN_PACKET_BUFFER_BYTES;
	}
	const uint32_t packetLen = *(uint32_t*)begin;
	if (packetLen > FLOW_KNOBS->PACKET_LIMIT) {
		TraceEvent(SevError, "PacketLimitExceeded")
		    .detail("FromPeer", peerAddress.toString())
		    .detail("Length", (int)packetLen);
		throw platform_error();
	}
	return std::max<uint32_t>(FLOW_KNOBS->MIN_PACKET_BUFFER_BYTES,
	                          packetLen + sizeof(uint32_t) * (peerAddress.isTLS() ? 2 : 3));
}

// This actor exists whenever there is an open or opening connection, whether incoming or outgoing
// For incoming connections conn is set and peer is initially nullptr; for outgoing connections it is the reverse
															#line 5457 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionReader()
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionReaderActor>
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionReaderActorState {
															#line 5464 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionReaderActorState(TransportData* const& transport,Reference<IConnection> const& conn,Reference<Peer> const& peer,Promise<Reference<Peer>> const& onConnected) 
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : transport(transport),
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   conn(conn),
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   peer(peer),
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   onConnected(onConnected),
															#line 1443 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   arena(),
															#line 1444 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   unprocessed_begin(nullptr),
															#line 1445 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   unprocessed_end(nullptr),
															#line 1446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   buffer_end(nullptr),
															#line 1447 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   expectConnectPacket(true),
															#line 1448 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   compatible(false),
															#line 1449 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   incompatiblePeerCounted(false),
															#line 1450 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   peerAddress(),
															#line 1451 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   peerProtocolVersion(),
															#line 1452 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   trusted(transport->allowList(conn->getPeerAddress().ip) && conn->hasTrustedPeer())
															#line 5497 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionReader", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionReaderActorState() 
	{
		fdb_probe_actor_destroy("connectionReader", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1453 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			peerAddress = conn->getPeerAddress();
															#line 1455 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!peer)
															#line 5514 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1456 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ASSERT(!peerAddress.isPublic());
															#line 5518 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			try {
															#line 1459 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				;
															#line 5523 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionReaderActorState();
		static_cast<ConnectionReaderActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1640 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (incompatiblePeerCounted)
															#line 5553 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1641 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ASSERT(peer && peer->transport->numIncompatibleConnections > 0);
															#line 1642 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->transport->numIncompatibleConnections--;
															#line 5559 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 1644 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 5563 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1460 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 5584 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1635 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_3 = conn->onReadable();
															#line 1635 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5595 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 4;
															#line 1635 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 3, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 5600 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1461 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		readAllBytes = buffer_end - unprocessed_end;
															#line 1462 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (readAllBytes < FLOW_KNOBS->MIN_PACKET_BUFFER_FREE_BYTES)
															#line 5618 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1463 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			Arena newArena;
															#line 1464 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			const int unproc_len = unprocessed_end - unprocessed_begin;
															#line 1465 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			const int len = getNewBufferSize(unprocessed_begin, unprocessed_end, peerAddress, peerProtocolVersion);
															#line 1467 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			uint8_t* const newBuffer = new (newArena) uint8_t[len];
															#line 1468 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (unproc_len > 0)
															#line 5630 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1469 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				memcpy(newBuffer, unprocessed_begin, unproc_len);
															#line 5634 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 1471 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			arena = newArena;
															#line 1472 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			unprocessed_begin = newBuffer;
															#line 1473 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			unprocessed_end = newBuffer + unproc_len;
															#line 1474 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			buffer_end = newBuffer + len;
															#line 1475 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			readAllBytes = buffer_end - unprocessed_end;
															#line 5646 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1478 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		totalReadBytes = 0;
															#line 1479 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 5652 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1490 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (peer)
															#line 5674 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1491 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			peer->bytesReceived += totalReadBytes;
															#line 5678 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1493 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (totalReadBytes == 0)
															#line 5682 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1495 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		readWillBlock = totalReadBytes != readAllBytes;
															#line 1497 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (expectConnectPacket && unprocessed_end - unprocessed_begin >= CONNECT_PACKET_V0_SIZE)
															#line 5690 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1502 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			int32_t connectPacketSize = ((ConnectPacket*)unprocessed_begin)->totalPacketSize();
															#line 1503 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (unprocessed_end - unprocessed_begin >= connectPacketSize)
															#line 5696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1504 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				auto protocolVersion = ((ConnectPacket*)unprocessed_begin)->protocolVersion;
															#line 1505 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				BinaryReader pktReader(unprocessed_begin, connectPacketSize, AssumeVersion(protocolVersion));
															#line 1506 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				ConnectPacket pkt;
															#line 1507 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				serializer(pktReader, pkt);
															#line 1509 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				uint64_t connectionId = pkt.connectionId;
															#line 1510 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (!pkt.protocolVersion.hasObjectSerializerFlag() || !pkt.protocolVersion.isCompatible(g_network->protocolVersion()))
															#line 5710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1512 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					NetworkAddress addr = pkt.canonicalRemotePort ? NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort) : conn->getPeerAddress();
															#line 1515 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (connectionId != 1)
															#line 5716 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1516 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						addr.port = 0;
															#line 5720 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
															#line 1518 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (!transport->multiVersionConnections.count(connectionId))
															#line 5724 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1519 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						if (now() - transport->lastIncompatibleMessage > FLOW_KNOBS->CONNECTION_REJECTED_MESSAGE_DELAY)
															#line 5728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						{
															#line 1521 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							TraceEvent(SevWarn, "ConnectionRejected", conn->getDebugID()) .detail("Reason", "IncompatibleProtocolVersion") .detail("LocalVersion", g_network->protocolVersion()) .detail("RejectedVersion", pkt.protocolVersion) .detail("Peer", pkt.canonicalRemotePort ? NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort) : conn->getPeerAddress()) .detail("PeerAddress", pkt.canonicalRemotePort ? NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort) : conn->getPeerAddress()) .detail("ConnectionId", connectionId);
															#line 1534 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							transport->lastIncompatibleMessage = now();
															#line 5734 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
															#line 1536 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						if (!transport->incompatiblePeers.count(addr))
															#line 5738 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						{
															#line 1537 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							transport->incompatiblePeers[addr] = std::make_pair(connectionId, now());
															#line 5742 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
					}
					else
					{
															#line 1539 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						if (connectionId > 1)
															#line 5749 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						{
															#line 1540 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							transport->multiVersionConnections[connectionId] = now() + FLOW_KNOBS->CONNECTION_ID_TIMEOUT;
															#line 5753 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
					}
															#line 1543 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					compatible = false;
															#line 1544 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					static SimpleCounter<int64_t>* countConnectionIncompatible = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/ConnectionIncompatible");
															#line 1546 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					countConnectionIncompatible->increment(1);
															#line 1547 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (!protocolVersion.hasInexpensiveMultiVersionClient())
															#line 5764 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1548 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						if (peer)
															#line 5768 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						{
															#line 1549 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
							peer->protocolVersion->set(protocolVersion);
															#line 5772 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
						}
															#line 1551 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						static SimpleCounter<int64_t>* countConnectionIncompatibleWithVeryOldClient = SimpleCounter<int64_t>::makeCounter( "/Transport/TLS/ConnectionIncompatibleWithVeryOldClient");
															#line 1554 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						countConnectionIncompatibleWithVeryOldClient->increment(1);
															#line 1557 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						return a_body1Catch2(incompatible_protocol_version(), std::max(0, loopDepth - 2));
															#line 5780 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
				}
				else
				{
															#line 1560 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					compatible = true;
															#line 1561 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					TraceEvent("ConnectionEstablished", conn->getDebugID()) .suppressFor(1.0) .detail("Peer", conn->getPeerAddress()) .detail("PeerAddress", conn->getPeerAddress()) .detail("ConnectionId", connectionId);
															#line 5789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
															#line 1568 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (connectionId > 1)
															#line 5793 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1569 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					transport->multiVersionConnections[connectionId] = now() + FLOW_KNOBS->CONNECTION_ID_TIMEOUT;
															#line 5797 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
															#line 1572 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				unprocessed_begin += connectPacketSize;
															#line 1573 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				expectConnectPacket = false;
															#line 1575 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (peer)
															#line 5805 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1576 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					peerProtocolVersion = protocolVersion;
															#line 1578 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					TraceEvent("ConnectedOutgoing") .suppressFor(1.0) .detail("PeerAddr", NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort)) .detail("PeerAddress", NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort));
															#line 1583 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					peer->compatible = compatible;
															#line 1584 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (!compatible)
															#line 5815 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1585 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						peer->transport->numIncompatibleConnections++;
															#line 1586 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						incompatiblePeerCounted = true;
															#line 5821 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
															#line 1588 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					ASSERT(pkt.canonicalRemotePort == peerAddress.port);
															#line 1589 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					onConnected.send(peer);
															#line 5827 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					loopDepth = a_body1loopBody1loopBody1cont8(loopDepth);
				}
				else
				{
															#line 1591 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					peerProtocolVersion = protocolVersion;
															#line 1592 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (pkt.canonicalRemotePort)
															#line 5836 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1593 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						peerAddress = NetworkAddress(pkt.canonicalRemoteIp(), pkt.canonicalRemotePort, true, peerAddress.isTLS(), NetworkAddressFromHostname(peerAddress.fromHostname));
															#line 5840 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
															#line 1599 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					peer = transport->getOrOpenPeer(peerAddress, false);
															#line 1600 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					peer->compatible = compatible;
															#line 1601 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (!compatible)
															#line 5848 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					{
															#line 1602 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						peer->transport->numIncompatibleConnections++;
															#line 1603 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
						incompatiblePeerCounted = true;
															#line 5854 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					}
															#line 1605 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					onConnected.send(peer);
															#line 1606 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					StrictFuture<Void> __when_expr_1 = delay(0);
															#line 1606 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 5862 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 2;
															#line 1606 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 1, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 5867 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
			}
			else
			{
				loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1(int loopDepth) 
	{
															#line 1480 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		const int len = std::min<int>(buffer_end - unprocessed_end, FLOW_KNOBS->MAX_PACKET_SEND_BYTES);
															#line 1481 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (len == 0)
															#line 5896 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1483 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		readBytes = conn->read(unprocessed_end, unprocessed_end + len);
															#line 1484 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (readBytes == 0)
															#line 5904 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1486 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::ReadSocket);
															#line 1486 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 3));
															#line 5912 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 3)); else return a_body1loopBody1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 1;
															#line 1486 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 0, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 5917 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1487 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		totalReadBytes += readBytes;
															#line 1488 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		unprocessed_end += readBytes;
															#line 5941 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1487 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		totalReadBytes += readBytes;
															#line 1488 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		unprocessed_end += readBytes;
															#line 5952 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state > 0) static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionReaderActor*>(this)->ActorCallback< ConnectionReaderActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionReaderActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont4(int loopDepth) 
	{
															#line 1612 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!expectConnectPacket)
															#line 6036 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1613 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (compatible || peerProtocolVersion.hasStableInterfaces())
															#line 6040 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1614 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				scanPackets(transport, unprocessed_begin, unprocessed_end, arena, peerAddress, trusted, peerProtocolVersion, peer->disconnect.getFuture(), IsStableConnection(g_network->isSimulated() && conn->isStableConnection()));
															#line 6044 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1624 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				unprocessed_begin = unprocessed_end;
															#line 1625 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				peer->resetPing.trigger();
															#line 6052 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
		}
															#line 1629 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (readWillBlock)
															#line 6057 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1632 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::ReadSocket);
															#line 1632 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6065 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 3;
															#line 1632 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 2, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 6070 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont8(int loopDepth) 
	{
															#line 1608 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		peer->protocolVersion->set(peerProtocolVersion);
															#line 6085 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state > 0) static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionReaderActor*>(this)->ActorCallback< ConnectionReaderActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectionReaderActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont26(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont26(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont26(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont26(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state > 0) static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionReaderActor*>(this)->ActorCallback< ConnectionReaderActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectionReaderActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0, TaskPriority::ReadSocket);
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6270 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_4.get(), loopDepth); };
		static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 5;
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 4, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 6275 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0, TaskPriority::ReadSocket);
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6286 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_4.get(), loopDepth); };
		static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 5;
															#line 1636 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ConnectionReaderActor, 4, Void >*>(static_cast<ConnectionReaderActor*>(this)));
															#line 6291 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state > 0) static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionReaderActor*>(this)->ActorCallback< ConnectionReaderActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConnectionReaderActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ConnectionReaderActor*>(this)->actor_wait_state > 0) static_cast<ConnectionReaderActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionReaderActor*>(this)->ActorCallback< ConnectionReaderActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ConnectionReaderActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ConnectionReaderActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* transport;
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IConnection> conn;
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<Peer> peer;
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Promise<Reference<Peer>> onConnected;
															#line 1443 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Arena arena;
															#line 1444 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	uint8_t* unprocessed_begin;
															#line 1445 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	uint8_t* unprocessed_end;
															#line 1446 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	uint8_t* buffer_end;
															#line 1447 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool expectConnectPacket;
															#line 1448 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool compatible;
															#line 1449 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool incompatiblePeerCounted;
															#line 1450 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	NetworkAddress peerAddress;
															#line 1451 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ProtocolVersion peerProtocolVersion;
															#line 1452 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool trusted;
															#line 1461 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int readAllBytes;
															#line 1478 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int totalReadBytes;
															#line 1483 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int readBytes;
															#line 1495 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	bool readWillBlock;
															#line 6494 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionReader()
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionReaderActor final : public Actor<Void>, public ActorCallback< ConnectionReaderActor, 0, Void >, public ActorCallback< ConnectionReaderActor, 1, Void >, public ActorCallback< ConnectionReaderActor, 2, Void >, public ActorCallback< ConnectionReaderActor, 3, Void >, public ActorCallback< ConnectionReaderActor, 4, Void >, public FastAllocated<ConnectionReaderActor>, public ConnectionReaderActorState<ConnectionReaderActor> {
															#line 6499 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionReaderActor>::operator new;
	using FastAllocated<ConnectionReaderActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16879929266828224768UL, 8044274127692413184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionReaderActor, 0, Void >;
friend struct ActorCallback< ConnectionReaderActor, 1, Void >;
friend struct ActorCallback< ConnectionReaderActor, 2, Void >;
friend struct ActorCallback< ConnectionReaderActor, 3, Void >;
friend struct ActorCallback< ConnectionReaderActor, 4, Void >;
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionReaderActor(TransportData* const& transport,Reference<IConnection> const& conn,Reference<Peer> const& peer,Promise<Reference<Peer>> const& onConnected) 
															#line 6520 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionReaderActorState<ConnectionReaderActor>(transport, conn, peer, onConnected),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionReader", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11457441682608471808UL, 4039782478752333568UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionReaderActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionReader");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionReader", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionReaderActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectionReaderActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ConnectionReaderActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ConnectionReaderActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ConnectionReaderActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] static Future<Void> connectionReader( TransportData* const& transport, Reference<IConnection> const& conn, Reference<Peer> const& peer, Promise<Reference<Peer>> const& onConnected ) {
															#line 1438 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionReaderActor(transport, conn, peer, onConnected));
															#line 6557 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1647 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 6562 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via connectionIncoming()
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ConnectionIncomingActor>
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionIncomingActorState {
															#line 6569 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionIncomingActorState(TransportData* const& self,Reference<IConnection> const& conn) 
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   conn(conn),
															#line 1649 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   entry()
															#line 6580 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("connectionIncoming", reinterpret_cast<unsigned long>(this));

	}
	~ConnectionIncomingActorState() 
	{
		fdb_probe_actor_destroy("connectionIncoming", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1650 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			entry.time = now();
															#line 1651 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			entry.addr = conn->getPeerAddress();
															#line 6597 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			try {
															#line 1653 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				StrictFuture<Void> __when_expr_0 = conn->acceptHandshake();
															#line 1653 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (static_cast<ConnectionIncomingActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6603 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ConnectionIncomingActor*>(this)->actor_wait_state = 1;
															#line 1653 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 0, Void >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 6608 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectionIncomingActorState();
		static_cast<ConnectionIncomingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1699 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!static_cast<ConnectionIncomingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectionIncomingActorState(); static_cast<ConnectionIncomingActor*>(this)->destroy(); return 0; }
															#line 6637 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		new (&static_cast<ConnectionIncomingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConnectionIncomingActorState();
		static_cast<ConnectionIncomingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1683 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 6650 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1684 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent("IncomingConnectionError", conn->getDebugID()) .errorUnsuppressed(e) .suppressFor(1.0) .detail("FromAddress", conn->getPeerAddress());
															#line 1688 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				static SimpleCounter<int64_t>* countIncomingConnectionFailed = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionFailed");
															#line 1690 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				countIncomingConnectionFailed->increment(1);
															#line 1691 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (FLOW_KNOBS->LOG_CONNECTION_ATTEMPTS_ENABLED)
															#line 6660 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1692 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					entry.failed = true;
															#line 1693 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					self->connectionHistory.push_back(entry);
															#line 6666 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
			}
															#line 1696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			conn->close();
															#line 6671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1654 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionHandshakeAccepted = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionHandshakeAccepted");
															#line 1656 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionHandshakeAccepted->increment(1);
															#line 1657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		onConnected = Promise<Reference<Peer>>();
															#line 1658 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		reader = connectionReader(self, conn, Reference<Peer>(), onConnected);
															#line 1659 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (FLOW_KNOBS->LOG_CONNECTION_ATTEMPTS_ENABLED)
															#line 6694 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1660 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			entry.failed = false;
															#line 1661 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->connectionHistory.push_back(entry);
															#line 6700 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1664 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_1 = reader;
															#line 1663 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionIncomingActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6706 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 1668 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Reference<Peer>> __when_expr_2 = onConnected.getFuture();
															#line 6710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont2when2(__when_expr_2.get(), loopDepth); };
															#line 1671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delayJittered(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 6714 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont2when3(__when_expr_3.get(), loopDepth); };
		static_cast<ConnectionIncomingActor*>(this)->actor_wait_state = 2;
															#line 1664 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 1, Void >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 1668 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 1671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 3, Void >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 6723 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1654 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionHandshakeAccepted = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionHandshakeAccepted");
															#line 1656 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionHandshakeAccepted->increment(1);
															#line 1657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		onConnected = Promise<Reference<Peer>>();
															#line 1658 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		reader = connectionReader(self, conn, Reference<Peer>(), onConnected);
															#line 1659 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (FLOW_KNOBS->LOG_CONNECTION_ATTEMPTS_ENABLED)
															#line 6740 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1660 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			entry.failed = false;
															#line 1661 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->connectionHistory.push_back(entry);
															#line 6746 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1664 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_1 = reader;
															#line 1663 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ConnectionIncomingActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6752 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 1668 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Reference<Peer>> __when_expr_2 = onConnected.getFuture();
															#line 6756 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont2when2(__when_expr_2.get(), loopDepth); };
															#line 1671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delayJittered(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 6760 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont2when3(__when_expr_3.get(), loopDepth); };
		static_cast<ConnectionIncomingActor*>(this)->actor_wait_state = 2;
															#line 1664 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 1, Void >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 1668 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 1671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ConnectionIncomingActor, 3, Void >*>(static_cast<ConnectionIncomingActor*>(this)));
															#line 6769 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectionIncomingActor*>(this)->actor_wait_state > 0) static_cast<ConnectionIncomingActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionIncomingActor*>(this)->ActorCallback< ConnectionIncomingActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectionIncomingActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1679 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionConnected = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionConnected");
															#line 1681 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionConnected->increment(1);
															#line 6855 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
															#line 1665 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(false);
															#line 1666 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!static_cast<ConnectionIncomingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectionIncomingActorState(); static_cast<ConnectionIncomingActor*>(this)->destroy(); return 0; }
															#line 6866 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		new (&static_cast<ConnectionIncomingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConnectionIncomingActorState();
		static_cast<ConnectionIncomingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
															#line 1665 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		ASSERT(false);
															#line 1666 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (!static_cast<ConnectionIncomingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectionIncomingActorState(); static_cast<ConnectionIncomingActor*>(this)->destroy(); return 0; }
															#line 6880 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		new (&static_cast<ConnectionIncomingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConnectionIncomingActorState();
		static_cast<ConnectionIncomingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when2(Reference<Peer> const& p,int loopDepth) 
	{
															#line 1669 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		p->onIncomingConnection(p, conn, reader);
															#line 6892 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Reference<Peer> && p,int loopDepth) 
	{
															#line 1669 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		p->onIncomingConnection(p, conn, reader);
															#line 6901 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when3(Void const& _,int loopDepth) 
	{
															#line 1672 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		CODE_PROBE(true, "Incoming connection timed out");
															#line 1673 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionTimedout = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionTimedout");
															#line 1675 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionTimedout->increment(1);
															#line 1676 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1Catch2(timed_out(), loopDepth);
															#line 6916 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont2when3(Void && _,int loopDepth) 
	{
															#line 1672 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		CODE_PROBE(true, "Incoming connection timed out");
															#line 1673 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionTimedout = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionTimedout");
															#line 1675 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionTimedout->increment(1);
															#line 1676 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		return a_body1Catch2(timed_out(), loopDepth);
															#line 6930 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectionIncomingActor*>(this)->actor_wait_state > 0) static_cast<ConnectionIncomingActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectionIncomingActor*>(this)->ActorCallback< ConnectionIncomingActor, 1, Void >::remove();
		static_cast<ConnectionIncomingActor*>(this)->ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >::remove();
		static_cast<ConnectionIncomingActor*>(this)->ActorCallback< ConnectionIncomingActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectionIncomingActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >*,Reference<Peer> const& value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >*,Reference<Peer> && value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >*,Error err) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ConnectionIncomingActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ConnectionIncomingActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IConnection> conn;
															#line 1649 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData::ConnectionHistoryEntry entry;
															#line 1657 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Promise<Reference<Peer>> onConnected;
															#line 1658 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> reader;
															#line 7136 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via connectionIncoming()
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ConnectionIncomingActor final : public Actor<Void>, public ActorCallback< ConnectionIncomingActor, 0, Void >, public ActorCallback< ConnectionIncomingActor, 1, Void >, public ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >, public ActorCallback< ConnectionIncomingActor, 3, Void >, public FastAllocated<ConnectionIncomingActor>, public ConnectionIncomingActorState<ConnectionIncomingActor> {
															#line 7141 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ConnectionIncomingActor>::operator new;
	using FastAllocated<ConnectionIncomingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15942323201420345088UL, 12726502748232903680UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectionIncomingActor, 0, Void >;
friend struct ActorCallback< ConnectionIncomingActor, 1, Void >;
friend struct ActorCallback< ConnectionIncomingActor, 2, Reference<Peer> >;
friend struct ActorCallback< ConnectionIncomingActor, 3, Void >;
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ConnectionIncomingActor(TransportData* const& self,Reference<IConnection> const& conn) 
															#line 7161 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ConnectionIncomingActorState<ConnectionIncomingActor>(self, conn),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectionIncoming", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10922312795653603584UL, 3326327758256808448UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectionIncomingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectionIncoming");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectionIncoming", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectionIncomingActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectionIncomingActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] static Future<Void> connectionIncoming( TransportData* const& self, Reference<IConnection> const& conn ) {
															#line 1648 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ConnectionIncomingActor(self, conn));
															#line 7195 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1701 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

															#line 7200 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via listen()
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class ListenActor>
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ListenActorState {
															#line 7207 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ListenActorState(TransportData* const& self,NetworkAddress const& listenAddr) 
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self),
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   listenAddr(listenAddr),
															#line 1703 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   incoming(),
															#line 1705 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   listener(INetworkConnections::net()->listen(listenAddr))
															#line 7220 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("listen", reinterpret_cast<unsigned long>(this));

	}
	~ListenActorState() 
	{
		fdb_probe_actor_destroy("listen", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1706 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (!g_network->isSimulated() && self->localAddresses.getAddressList().address.port == 0)
															#line 7235 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1707 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				TraceEvent(SevInfo, "UpdatingListenAddress") .detail("AssignedListenAddress", listener->getListenAddress().toString());
															#line 1709 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				self->localAddresses.setNetworkAddress(listener->getListenAddress());
															#line 1710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				setTraceLocalAddress(listener->getListenAddress());
															#line 7243 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 1712 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			connectionCount = 0;
															#line 7247 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			try {
															#line 1714 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				;
															#line 7251 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListenActorState();
		static_cast<ListenActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1732 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevError, "ListenError").error(e);
															#line 1733 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7283 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1715 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Reference<IConnection>> __when_expr_0 = listener->accept();
															#line 1715 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<ListenActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7306 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ListenActor*>(this)->actor_wait_state = 1;
															#line 1715 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListenActor, 0, Reference<IConnection> >*>(static_cast<ListenActor*>(this)));
															#line 7311 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Reference<IConnection> const& conn,int loopDepth) 
	{
															#line 1716 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionCreated = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionCreated");
															#line 1718 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionCreated->increment(1);
															#line 1719 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (conn)
															#line 7324 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1720 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent("ConnectionFrom", conn->getDebugID()) .suppressFor(1.0) .detail("FromAddress", conn->getPeerAddress()) .detail("ListenAddress", listenAddr.toString());
															#line 1724 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			incoming.add(connectionIncoming(self, conn));
															#line 7330 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1726 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		connectionCount++;
															#line 1727 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (connectionCount % (FLOW_KNOBS->ACCEPT_BATCH_SIZE) == 0)
															#line 7336 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::AcceptSocket);
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ListenActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7342 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ListenActor*>(this)->actor_wait_state = 2;
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListenActor, 1, Void >*>(static_cast<ListenActor*>(this)));
															#line 7347 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Reference<IConnection> && conn,int loopDepth) 
	{
															#line 1716 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		static SimpleCounter<int64_t>* countIncomingConnectionCreated = SimpleCounter<int64_t>::makeCounter("/Transport/TLS/IncomingConnectionCreated");
															#line 1718 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		countIncomingConnectionCreated->increment(1);
															#line 1719 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (conn)
															#line 7365 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1720 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent("ConnectionFrom", conn->getDebugID()) .suppressFor(1.0) .detail("FromAddress", conn->getPeerAddress()) .detail("ListenAddress", listenAddr.toString());
															#line 1724 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			incoming.add(connectionIncoming(self, conn));
															#line 7371 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 1726 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		connectionCount++;
															#line 1727 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (connectionCount % (FLOW_KNOBS->ACCEPT_BATCH_SIZE) == 0)
															#line 7377 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::AcceptSocket);
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<ListenActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7383 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ListenActor*>(this)->actor_wait_state = 2;
															#line 1728 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListenActor, 1, Void >*>(static_cast<ListenActor*>(this)));
															#line 7388 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<IConnection> const& conn,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(conn, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<IConnection> && conn,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(conn), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListenActor*>(this)->actor_wait_state > 0) static_cast<ListenActor*>(this)->actor_wait_state = 0;
		static_cast<ListenActor*>(this)->ActorCallback< ListenActor, 0, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< ListenActor, 0, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListenActor, 0, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListenActor, 0, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ListenActor*>(this)->actor_wait_state > 0) static_cast<ListenActor*>(this)->actor_wait_state = 0;
		static_cast<ListenActor*>(this)->ActorCallback< ListenActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListenActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ListenActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ListenActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	NetworkAddress listenAddr;
															#line 1703 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ActorCollectionNoErrors incoming;
															#line 1705 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IListener> listener;
															#line 1712 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	uint64_t connectionCount;
															#line 7576 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via listen()
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class ListenActor final : public Actor<Void>, public ActorCallback< ListenActor, 0, Reference<IConnection> >, public ActorCallback< ListenActor, 1, Void >, public FastAllocated<ListenActor>, public ListenActorState<ListenActor> {
															#line 7581 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<ListenActor>::operator new;
	using FastAllocated<ListenActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5617709907636958464UL, 9030723251977371904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListenActor, 0, Reference<IConnection> >;
friend struct ActorCallback< ListenActor, 1, Void >;
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	ListenActor(TransportData* const& self,NetworkAddress const& listenAddr) 
															#line 7599 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ListenActorState<ListenActor>(self, listenAddr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listen", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9989528927518907392UL, 7817939837473479168UL);
		ActorExecutionContextHelper __helper(static_cast<ListenActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listen");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listen", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListenActor, 0, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ListenActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] static Future<Void> listen( TransportData* const& self, NetworkAddress const& listenAddr ) {
															#line 1702 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new ListenActor(self, listenAddr));
															#line 7633 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1736 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

Reference<Peer> TransportData::getPeer(NetworkAddress const& address) {
	auto peer = peers.find(address);
	if (peer != peers.end()) {
		return peer->second;
	}
	return Reference<Peer>();
}

Reference<Peer> TransportData::getOrOpenPeer(NetworkAddress const& address, bool startConnectionKeeper) {
	auto peer = getPeer(address);
	if (!peer) {
		peer = makeReference<Peer>(this, address);
		if (startConnectionKeeper && !isLocalAddress(address)) {
			peer->connect = connectionKeeper(peer);
		}
		peers[address] = peer;
		if (address.isPublic()) {
			orderedAddresses.insert(address);
		}
	}

	return peer;
}

bool TransportData::isLocalAddress(const NetworkAddress& address) const {
	return address == localAddresses.getAddressList().address ||
	       (localAddresses.getAddressList().secondaryAddress.present() &&
	        address == localAddresses.getAddressList().secondaryAddress.get());
}

void TransportData::applyPublicKeySet(StringRef jwkSetString) {
	auto jwks = JsonWebKeySet::parse(jwkSetString, {});
	if (!jwks.present())
		throw pkey_decode_error();
	const auto& keySet = jwks.get().keys;
	publicKeys.clear();
	int numPrivateKeys = 0;
	for (auto [keyName, key] : keySet) {
		// ignore private keys
		if (key.isPublic()) {
			publicKeys[keyName] = key.getPublic();
		} else {
			numPrivateKeys++;
		}
	}
	TraceEvent(SevInfo, "AuthzPublicKeySetApply"_audit).detail("NumPublicKeys", publicKeys.size());
	if (numPrivateKeys > 0) {
		TraceEvent(SevWarnAlways, "AuthzPublicKeySetContainsPrivateKeys").detail("NumPrivateKeys", numPrivateKeys);
	}
}

															#line 7689 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via multiVersionCleanupWorker()
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class MultiVersionCleanupWorkerActor>
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class MultiVersionCleanupWorkerActorState {
															#line 7696 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	MultiVersionCleanupWorkerActorState(TransportData* const& self) 
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : self(self)
															#line 7703 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this));

	}
	~MultiVersionCleanupWorkerActorState() 
	{
		fdb_probe_actor_destroy("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1789 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 7718 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MultiVersionCleanupWorkerActorState();
		static_cast<MultiVersionCleanupWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1790 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->CONNECTION_CLEANUP_DELAY);
															#line 1790 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<MultiVersionCleanupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7750 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MultiVersionCleanupWorkerActor*>(this)->actor_wait_state = 1;
															#line 1790 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >*>(static_cast<MultiVersionCleanupWorkerActor*>(this)));
															#line 7755 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1791 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		bool foundIncompatible = false;
															#line 1792 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		for(auto it = self->incompatiblePeers.begin();it != self->incompatiblePeers.end();) {
															#line 1793 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (self->multiVersionConnections.count(it->second.first))
															#line 7768 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1794 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it = self->incompatiblePeers.erase(it);
															#line 7772 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1796 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (now() - it->second.second > FLOW_KNOBS->INCOMPATIBLE_PEER_DELAY_BEFORE_LOGGING)
															#line 7778 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1797 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					foundIncompatible = true;
															#line 7782 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
															#line 1799 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it++;
															#line 7786 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
		}
															#line 1803 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		for(auto it = self->multiVersionConnections.begin();it != self->multiVersionConnections.end();) {
															#line 1804 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (it->second < now())
															#line 7793 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1805 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it = self->multiVersionConnections.erase(it);
															#line 7797 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1807 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it++;
															#line 7803 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
		}
															#line 1811 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (foundIncompatible)
															#line 7808 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1812 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->incompatiblePeersChanged.trigger();
															#line 7812 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1791 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		bool foundIncompatible = false;
															#line 1792 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		for(auto it = self->incompatiblePeers.begin();it != self->incompatiblePeers.end();) {
															#line 1793 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (self->multiVersionConnections.count(it->second.first))
															#line 7826 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1794 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it = self->incompatiblePeers.erase(it);
															#line 7830 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1796 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				if (now() - it->second.second > FLOW_KNOBS->INCOMPATIBLE_PEER_DELAY_BEFORE_LOGGING)
															#line 7836 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				{
															#line 1797 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
					foundIncompatible = true;
															#line 7840 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
				}
															#line 1799 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it++;
															#line 7844 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
		}
															#line 1803 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		for(auto it = self->multiVersionConnections.begin();it != self->multiVersionConnections.end();) {
															#line 1804 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (it->second < now())
															#line 7851 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 1805 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it = self->multiVersionConnections.erase(it);
															#line 7855 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1807 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				it++;
															#line 7861 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
		}
															#line 1811 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (foundIncompatible)
															#line 7866 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 1812 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			self->incompatiblePeersChanged.trigger();
															#line 7870 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MultiVersionCleanupWorkerActor*>(this)->actor_wait_state > 0) static_cast<MultiVersionCleanupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<MultiVersionCleanupWorkerActor*>(this)->ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MultiVersionCleanupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MultiVersionCleanupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MultiVersionCleanupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 7953 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via multiVersionCleanupWorker()
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class MultiVersionCleanupWorkerActor final : public Actor<Void>, public ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >, public FastAllocated<MultiVersionCleanupWorkerActor>, public MultiVersionCleanupWorkerActorState<MultiVersionCleanupWorkerActor> {
															#line 7958 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<MultiVersionCleanupWorkerActor>::operator new;
	using FastAllocated<MultiVersionCleanupWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14436819707514823936UL, 3863680309249250048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >;
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	MultiVersionCleanupWorkerActor(TransportData* const& self) 
															#line 7975 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   MultiVersionCleanupWorkerActorState<MultiVersionCleanupWorkerActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8058411969194286848UL, 14729391977982034432UL);
		ActorExecutionContextHelper __helper(static_cast<MultiVersionCleanupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("multiVersionCleanupWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("multiVersionCleanupWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MultiVersionCleanupWorkerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] static Future<Void> multiVersionCleanupWorker( TransportData* const& self ) {
															#line 1788 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new MultiVersionCleanupWorkerActor(self));
															#line 8008 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 1816 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

FlowTransport::FlowTransport(uint64_t transportId, int maxWellKnownEndpoints, IPAllowList const* allowList)
  : self(new TransportData(transportId, maxWellKnownEndpoints, allowList)) {
	self->multiVersionCleanup = multiVersionCleanupWorker(self);
	if (g_network->isSimulated()) {
		for (auto const& p : g_simulator->authKeys) {
			self->publicKeys.emplace(p.first, p.second.toPublic());
		}
	}
}

FlowTransport::~FlowTransport() {
	delete self;
}

void FlowTransport::initMetrics() {
	self->initMetrics();
}

NetworkAddressList FlowTransport::getLocalAddresses() const {
	return self->localAddresses.getAddressList();
}

NetworkAddress FlowTransport::getLocalAddress() const {
	return self->localAddresses.getAddressList().address;
}

Standalone<StringRef> FlowTransport::getLocalAddressAsString() const {
	return self->localAddresses.getLocalAddressAsString();
}

const std::unordered_map<NetworkAddress, Reference<Peer>>& FlowTransport::getAllPeers() const {
	return self->peers;
}

std::map<NetworkAddress, std::pair<uint64_t, double>>* FlowTransport::getIncompatiblePeers() {
	for (auto it = self->incompatiblePeers.begin(); it != self->incompatiblePeers.end();) {
		if (self->multiVersionConnections.count(it->second.first)) {
			it = self->incompatiblePeers.erase(it);
		} else {
			it++;
		}
	}
	return &self->incompatiblePeers;
}

Future<Void> FlowTransport::onIncompatibleChanged() {
	return self->incompatiblePeersChanged.onTrigger();
}

Future<Void> FlowTransport::bind(NetworkAddress publicAddress, NetworkAddress listenAddress) {
	ASSERT(publicAddress.isPublic());
	if (self->localAddresses.getAddressList().address == NetworkAddress()) {
		self->localAddresses.setNetworkAddress(publicAddress);
	} else {
		auto addrList = self->localAddresses.getAddressList();
		addrList.secondaryAddress = publicAddress;
		self->localAddresses.setAddressList(addrList);
	}
	// reformatLocalAddress()
	TraceEvent("Binding").detail("PublicAddress", publicAddress).detail("ListenAddress", listenAddress);

	Future<Void> listenF = listen(self, listenAddress);
	self->listeners.push_back(listenF);
	return listenF;
}

Endpoint FlowTransport::loadedEndpoint(const UID& token) {
	return Endpoint(g_currentDeliveryPeerAddress, token);
}

Future<Void> FlowTransport::loadedDisconnect() {
	return g_currentDeliveryPeerDisconnect;
}

void FlowTransport::addPeerReference(const Endpoint& endpoint, bool isStream) {
	if (!isStream || !endpoint.getPrimaryAddress().isValid() || !endpoint.getPrimaryAddress().isPublic())
		return;

	Reference<Peer> peer = self->getOrOpenPeer(endpoint.getPrimaryAddress());
	if (peer->peerReferences == -1) {
		peer->peerReferences = 1;
	} else {
		peer->peerReferences++;
	}
}

void FlowTransport::removePeerReference(const Endpoint& endpoint, bool isStream) {
	if (!isStream || !endpoint.getPrimaryAddress().isValid() || !endpoint.getPrimaryAddress().isPublic())
		return;
	Reference<Peer> peer = self->getPeer(endpoint.getPrimaryAddress());
	if (peer) {
		peer->peerReferences--;
		if (peer->peerReferences < 0) {
			TraceEvent(SevError, "InvalidPeerReferences")
			    .detail("References", peer->peerReferences)
			    .detail("Address", endpoint.getPrimaryAddress())
			    .detail("Token", endpoint.token);
		}
		if (peer->peerReferences == 0 && peer->reliable.empty() && peer->unsent.empty() &&
		    peer->outstandingReplies == 0 &&
		    peer->lastDataPacketSentTime < now() - FLOW_KNOBS->CONNECTION_MONITOR_UNREFERENCED_CLOSE_DELAY) {
			peer->resetPing.trigger();
		}
	}
}

void FlowTransport::addEndpoint(Endpoint& endpoint, NetworkMessageReceiver* receiver, TaskPriority taskID) {
	endpoint.token = deterministicRandom()->randomUniqueID();
	if (receiver->isStream()) {
		endpoint.addresses = self->localAddresses.getAddressList();
		endpoint.token = UID(endpoint.token.first() | TOKEN_STREAM_FLAG, endpoint.token.second());
	} else {
		endpoint.addresses = NetworkAddressList();
		endpoint.token = UID(endpoint.token.first() & ~TOKEN_STREAM_FLAG, endpoint.token.second());
	}
	self->endpoints.insert(receiver, endpoint.token, taskID);
}

void FlowTransport::addEndpoints(std::vector<std::pair<FlowReceiver*, TaskPriority>> const& streams) {
	self->endpoints.insert(self->localAddresses.getAddressList(), streams);
}

void FlowTransport::removeEndpoint(const Endpoint& endpoint, NetworkMessageReceiver* receiver) {
	self->endpoints.remove(endpoint.token, receiver);
}

void FlowTransport::addWellKnownEndpoint(Endpoint& endpoint, NetworkMessageReceiver* receiver, TaskPriority taskID) {
	endpoint.addresses = self->localAddresses.getAddressList();
	ASSERT(receiver->isStream());
	self->endpoints.insertWellKnown(receiver, endpoint.token, taskID);
}

static void sendLocal(TransportData* self, ISerializeSource const& what, const Endpoint& destination) {
	CODE_PROBE(true, "\"Loopback\" delivery");
	// SOMEDAY: Would it be better to avoid (de)serialization by doing this check in flow?

	Standalone<StringRef> copy;
	ObjectWriter wr(AssumeVersion(g_network->protocolVersion()));
	what.serializeObjectWriter(wr);
	copy = wr.toStringRef();
#if VALGRIND
	VALGRIND_CHECK_MEM_IS_DEFINED(copy.begin(), copy.size());
#endif

	ASSERT(copy.size() > 0);
	TaskPriority priority = self->endpoints.getPriority(destination.token);
	if (priority != TaskPriority::UnknownEndpoint || (destination.token.first() & TOKEN_STREAM_FLAG) != 0) {
		deliver(self,
		        destination,
		        priority,
		        ArenaReader(copy.arena(), copy, AssumeVersion(currentProtocolVersion())),
		        NetworkAddress(),
		        true,
		        InReadSocket::False,
		        Never());
	}
}

static ReliablePacket* sendPacket(TransportData* self,
                                  Reference<Peer> peer,
                                  ISerializeSource const& what,
                                  const Endpoint& destination,
                                  bool reliable) {
	const bool checksumEnabled = !destination.getPrimaryAddress().isTLS();
	++self->countPacketsGenerated;

	// If there isn't an open connection, a public address, or the peer isn't compatible, we can't send
	if (!peer || (peer->outgoingConnectionIdle && !destination.getPrimaryAddress().isPublic()) ||
	    (!peer->compatible && destination.token != Endpoint::wellKnownToken(WLTOKEN_PING_PACKET))) {
		CODE_PROBE(true, "Can't send to private address without a compatible open connection");
		return nullptr;
	}

	bool firstUnsent = peer->unsent.empty();

	PacketBuffer* pb = peer->unsent.getWriteBuffer();
	ReliablePacket* rp = reliable ? new ReliablePacket : 0;

	int prevBytesWritten = pb->bytes_written;
	PacketBuffer* checksumPb = pb;

	PacketWriter wr(pb,
	                rp,
	                AssumeVersion(g_network->protocolVersion())); // SOMEDAY: Can we downgrade to talk to older peers?

	// Reserve some space for packet length and checksum, write them after serializing data
	SplitBuffer packetInfoBuffer;
	uint32_t len;

	// This is technically abstraction breaking but avoids XXH3_createState() and XXH3_freeState() which are just
	// malloc/free
	XXH3_state_t checksumState;
	// Checksum will be calculated with buffer API if contiguous, else using stream API.  Mode is tracked here.
	bool checksumStream = false;
	XXH64_hash_t checksum;

	int packetInfoSize = PACKET_LEN_WIDTH;
	if (checksumEnabled) {
		packetInfoSize += sizeof(checksum);
	}

	wr.writeAhead(packetInfoSize, &packetInfoBuffer);
	wr << destination.token;
	what.serializePacketWriter(wr);
	pb = wr.finish();
	len = wr.size() - packetInfoSize;

	if (checksumEnabled) {
		// Find the correct place to start calculating checksum
		uint32_t checksumUnprocessedLength = len;
		prevBytesWritten += packetInfoSize;
		if (prevBytesWritten >= checksumPb->bytes_written) {
			prevBytesWritten -= checksumPb->bytes_written;
			checksumPb = checksumPb->nextPacketBuffer();
		}

		// Checksum calculation
		while (checksumUnprocessedLength > 0) {
			uint32_t processLength =
			    std::min(checksumUnprocessedLength, (uint32_t)(checksumPb->bytes_written - prevBytesWritten));

			// If not in checksum stream mode yet
			if (!checksumStream) {
				// If there is nothing left to process then calculate checksum directly
				if (processLength == checksumUnprocessedLength) {
					checksum = XXH3_64bits(checksumPb->data() + prevBytesWritten, processLength);
				} else {
					// Otherwise, initialize checksum state and switch to stream mode
					if (XXH3_64bits_reset(&checksumState) != XXH_OK) {
						throw internal_error();
					}
					checksumStream = true;
				}
			}

			// If in checksum stream mode, update the checksum state
			if (checksumStream) {
				if (XXH3_64bits_update(&checksumState, checksumPb->data() + prevBytesWritten, processLength) !=
				    XXH_OK) {
					throw internal_error();
				}
			}

			checksumUnprocessedLength -= processLength;
			checksumPb = checksumPb->nextPacketBuffer();
			prevBytesWritten = 0;
		}

		// If in checksum stream mode, get the final checksum
		if (checksumStream) {
			checksum = XXH3_64bits_digest(&checksumState);
		}
	}

	// Write packet length and checksum into packet buffer
	packetInfoBuffer.write(&len, sizeof(len));
	if (checksumEnabled) {
		packetInfoBuffer.write(&checksum, sizeof(checksum), sizeof(len));
	}

	if (len > FLOW_KNOBS->PACKET_LIMIT) {
		TraceEvent(SevError, "PacketLimitExceeded")
		    .detail("ToPeer", destination.getPrimaryAddress())
		    .detail("Length", (int)len);
		// throw platform_error();  // FIXME: How to recover from this situation?
	} else if (len > FLOW_KNOBS->PACKET_WARNING) {
		TraceEvent(SevWarn, "LargePacketSent")
		    .suppressFor(1.0)
		    .detail("ToPeer", destination.getPrimaryAddress())
		    .detail("Length", (int)len)
		    .detail("Token", destination.token)
		    .backtrace();
	}

#if VALGRIND
	SendBuffer* checkbuf = pb;
	while (checkbuf) {
		int size = checkbuf->bytes_written;
		const uint8_t* data = checkbuf->data();
		VALGRIND_CHECK_MEM_IS_DEFINED(data, size);
		checkbuf = checkbuf->next;
	}
#endif

	peer->send(pb, rp, firstUnsent);
	if (destination.token != Endpoint::wellKnownToken(WLTOKEN_PING_PACKET)) {
		peer->lastDataPacketSentTime = now();
	}
	return rp;
}

ReliablePacket* FlowTransport::sendReliable(ISerializeSource const& what, const Endpoint& destination) {
	if (self->isLocalAddress(destination.getPrimaryAddress())) {
		sendLocal(self, what, destination);
		return nullptr;
	}
	Reference<Peer> peer = self->getOrOpenPeer(destination.getPrimaryAddress());
	return sendPacket(self, peer, what, destination, true);
}

void FlowTransport::cancelReliable(ReliablePacket* p) {
	if (p)
		p->remove();
	// SOMEDAY: Call reliable.compact() if a lot of memory is wasted in PacketBuffers by formerly reliable packets mixed
	// with a few reliable ones.  Don't forget to delref the new PacketBuffers since they are unsent.
}

Reference<Peer> FlowTransport::sendUnreliable(ISerializeSource const& what,
                                              const Endpoint& destination,
                                              bool openConnection) {
	if (self->isLocalAddress(destination.getPrimaryAddress())) {
		sendLocal(self, what, destination);
		return Reference<Peer>();
	}
	Reference<Peer> peer;
	if (openConnection) {
		peer = self->getOrOpenPeer(destination.getPrimaryAddress());
	} else {
		peer = self->getPeer(destination.getPrimaryAddress());
	}

	sendPacket(self, peer, what, destination, false);
	return peer;
}

Reference<AsyncVar<bool>> FlowTransport::getDegraded() {
	return self->degraded;
}

// Returns the protocol version of the peer at the specified address. The result is returned as an AsyncVar that
// can be used to monitor for changes of a peer's protocol. The protocol version will be unset in the event that
// there is no connection established to the peer.
//
// Note that this function does not establish a connection to the peer. In order to obtain a peer's protocol
// version, some other mechanism should be used to connect to that peer.
Optional<Reference<AsyncVar<Optional<ProtocolVersion>> const>> FlowTransport::getPeerProtocolAsyncVar(
    NetworkAddress addr) {
	auto itr = self->peers.find(addr);
	if (itr != self->peers.end()) {
		return itr->second->protocolVersion;
	} else {
		return Optional<Reference<AsyncVar<Optional<ProtocolVersion>> const>>();
	}
}

void FlowTransport::resetConnection(NetworkAddress address) {
	auto peer = self->getPeer(address);
	if (peer) {
		peer->resetConnection.trigger();
	}
}

bool FlowTransport::incompatibleOutgoingConnectionsPresent() {
	return self->numIncompatibleConnections > 0;
}

void FlowTransport::createInstance(bool isClient,
                                   uint64_t transportId,
                                   int maxWellKnownEndpoints,
                                   IPAllowList const* allowList) {
	g_network->setGlobal(INetwork::enFlowTransport,
	                     (flowGlobalType) new FlowTransport(transportId, maxWellKnownEndpoints, allowList));
	g_network->setGlobal(INetwork::enNetworkAddressFunc, (flowGlobalType)&FlowTransport::getGlobalLocalAddress);
	g_network->setGlobal(INetwork::enNetworkAddressesFunc, (flowGlobalType)&FlowTransport::getGlobalLocalAddresses);
	g_network->setGlobal(INetwork::enFailureMonitor, (flowGlobalType) new SimpleFailureMonitor());
	g_network->setGlobal(INetwork::enClientFailureMonitor, isClient ? (flowGlobalType)1 : nullptr);
}

HealthMonitor* FlowTransport::healthMonitor() {
	return &self->healthMonitor;
}

Optional<PublicKey> FlowTransport::getPublicKeyByName(StringRef name) const {
	auto iter = self->publicKeys.find(name);
	if (iter != self->publicKeys.end()) {
		return iter->second;
	}
	return {};
}

NetworkAddress FlowTransport::currentDeliveryPeerAddress() const {
	return g_currentDeliveryPeerAddress.address;
}

bool FlowTransport::currentDeliveryPeerIsTrusted() const {
	return g_currentDeliverPeerAddressTrusted;
}

void FlowTransport::addPublicKey(StringRef name, PublicKey key) {
	self->publicKeys[name] = key;
}

void FlowTransport::removePublicKey(StringRef name) {
	self->publicKeys.erase(name);
}

void FlowTransport::removeAllPublicKeys() {
	self->publicKeys.clear();
}

void FlowTransport::loadPublicKeyFile(const std::string& filePath) {
	if (!fileExists(filePath)) {
		throw file_not_found();
	}
	int64_t const len = fileSize(filePath);
	if (len <= 0) {
		TraceEvent(SevWarn, "AuthzPublicKeySetEmpty").detail("Path", filePath);
	} else if (len > FLOW_KNOBS->PUBLIC_KEY_FILE_MAX_SIZE) {
		throw file_too_large();
	} else {
		auto json = readFileBytes(filePath, len);
		self->applyPublicKeySet(StringRef(json));
	}
}

															#line 8428 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via watchPublicKeyJwksFile()
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
template <class WatchPublicKeyJwksFileActor>
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class WatchPublicKeyJwksFileActorState {
															#line 8435 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	WatchPublicKeyJwksFileActorState(std::string const& filePath,TransportData* const& self) 
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		 : filePath(filePath),
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   self(self),
															#line 2233 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   fileChanged(),
															#line 2234 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   fileWatch(),
															#line 2235 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		   errorCount(0)
															#line 8450 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this));

	}
	~WatchPublicKeyJwksFileActorState() 
	{
		fdb_probe_actor_destroy("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2238 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			;
															#line 8465 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchPublicKeyJwksFileActorState();
		static_cast<WatchPublicKeyJwksFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2243 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		const int& intervalSeconds = FLOW_KNOBS->PUBLIC_KEY_FILE_REFRESH_INTERVAL_SECONDS;
															#line 2244 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		fileWatch = watchFileForChanges(filePath, &fileChanged, &intervalSeconds, "AuthzPublicKeySetRefreshStatError");
															#line 2245 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		;
															#line 8492 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2239 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (IAsyncFileSystem::filesystem())
															#line 8508 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2241 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 2241 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8516 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 1;
															#line 2241 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8521 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state > 0) static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 0;
		static_cast<WatchPublicKeyJwksFileActor*>(this)->ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2247 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			StrictFuture<Void> __when_expr_1 = fileChanged.onTrigger();
															#line 2247 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8640 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 2;
															#line 2247 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8645 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2262 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 8667 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			{
															#line 2263 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 8671 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			}
															#line 2266 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			errorCount++;
															#line 2267 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevWarn, "AuthzPublicKeySetRefreshError"_audit).error(e).detail("ErrorCount", errorCount);
															#line 8677 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Reference<IAsyncFile>> __when_expr_2 = IAsyncFileSystem::filesystem()->open( filePath, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0);
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8694 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 3;
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8699 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Reference<IAsyncFile>> __when_expr_2 = IAsyncFileSystem::filesystem()->open( filePath, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0);
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8710 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 3;
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8715 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state > 0) static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 0;
		static_cast<WatchPublicKeyJwksFileActor*>(this)->ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<int64_t> __when_expr_3 = file->size();
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8801 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 4;
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8806 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		file = __file;
															#line 8815 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state > 0) static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 0;
		static_cast<WatchPublicKeyJwksFileActor*>(this)->ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
															#line 2251 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		json = std::string(filesize, '\0');
															#line 2252 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (filesize > FLOW_KNOBS->PUBLIC_KEY_FILE_MAX_SIZE)
															#line 8896 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 2253 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			return a_body1cont1loopBody1Catch1(file_too_large(), loopDepth);
															#line 8900 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		}
															#line 2254 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (filesize <= 0)
															#line 8904 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		{
															#line 2255 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
			TraceEvent(SevWarn, "AuthzPublicKeySetEmpty").suppressFor(60);
															#line 8908 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
			return a_body1cont1loopHead1(loopDepth); // continue
		}
															#line 2258 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		StrictFuture<Void> __when_expr_4 = success(file->read(&json[0], filesize, 0));
															#line 2258 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8915 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 5;
															#line 2258 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >*>(static_cast<WatchPublicKeyJwksFileActor*>(this)));
															#line 8920 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(int64_t const& __filesize,int loopDepth) 
	{
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		filesize = __filesize;
															#line 8929 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(int64_t && __filesize,int loopDepth) 
	{
		filesize = std::move(__filesize);
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state > 0) static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 0;
		static_cast<WatchPublicKeyJwksFileActor*>(this)->ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2259 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->applyPublicKeySet(StringRef(json));
															#line 2260 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		errorCount = 0;
															#line 9010 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2259 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		self->applyPublicKeySet(StringRef(json));
															#line 2260 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
		errorCount = 0;
															#line 9021 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state > 0) static_cast<WatchPublicKeyJwksFileActor*>(this)->actor_wait_state = 0;
		static_cast<WatchPublicKeyJwksFileActor*>(this)->ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	std::string filePath;
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	TransportData* self;
															#line 2233 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	AsyncTrigger fileChanged;
															#line 2234 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Future<Void> fileWatch;
															#line 2235 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	unsigned errorCount;
															#line 2248 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	Reference<IAsyncFile> file;
															#line 2250 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	int64_t filesize;
															#line 2251 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	std::string json;
															#line 9130 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
};
// This generated class is to be used only via watchPublicKeyJwksFile()
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
class WatchPublicKeyJwksFileActor final : public Actor<Void>, public ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >, public ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >, public ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >, public ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >, public ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >, public FastAllocated<WatchPublicKeyJwksFileActor>, public WatchPublicKeyJwksFileActorState<WatchPublicKeyJwksFileActor> {
															#line 9135 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
public:
	using FastAllocated<WatchPublicKeyJwksFileActor>::operator new;
	using FastAllocated<WatchPublicKeyJwksFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5636803517330439936UL, 11768327867568834304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >;
friend struct ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >;
friend struct ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >;
friend struct ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >;
friend struct ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >;
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	WatchPublicKeyJwksFileActor(std::string const& filePath,TransportData* const& self) 
															#line 9156 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   WatchPublicKeyJwksFileActorState<WatchPublicKeyJwksFileActor>(filePath, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3000520082504806400UL, 8913624740779333376UL);
		ActorExecutionContextHelper __helper(static_cast<WatchPublicKeyJwksFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchPublicKeyJwksFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchPublicKeyJwksFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchPublicKeyJwksFileActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WatchPublicKeyJwksFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WatchPublicKeyJwksFileActor, 2, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WatchPublicKeyJwksFileActor, 3, int64_t >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WatchPublicKeyJwksFileActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
[[nodiscard]] static Future<Void> watchPublicKeyJwksFile( std::string const& filePath, TransportData* const& self ) {
															#line 2232 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"
	return Future<Void>(new WatchPublicKeyJwksFileActor(filePath, self));
															#line 9193 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/build_output/fdbrpc/FlowTransport.actor.g.cpp.py_gen"
}

#line 2271 "/codebuild/output/src3175872744/src/github.com/apple/foundationdb/fdbrpc/FlowTransport.actor.cpp"

void FlowTransport::watchPublicKeyFile(const std::string& publicKeyFilePath) {
	self->publicKeyFileWatch = watchPublicKeyJwksFile(publicKeyFilePath, self);
}
