#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
/*
 * MockS3Server.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2025 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/MockS3Server.h"

#include "fdbrpc/HTTP.h"
#include "fdbrpc/simulator.h"
#include "flow/Trace.h"
#include "flow/IRandom.h"
#include "flow/serialize.h"

#include <string>
#include <map>
#include <sstream>
#include <regex>
#include <utility>
#include <iostream>
#include <algorithm>

#include "flow/IAsyncFile.h" // For IAsyncFileSystem
#include "flow/Platform.h" // For platform::createDirectory

#include "rapidjson/document.h"
#include "rapidjson/writer.h"
#include "rapidjson/stringbuffer.h"
#include "rapidjson/prettywriter.h"

#include "flow/actorcompiler.h" // This must be the last #include.

/*
 * ACTOR STATE VARIABLE INITIALIZATION REQUIREMENT
 *
 * ACTORs with early returns (before any wait()) crash with canBeSet() assertion if no state
 * variable is declared before the return. The actor compiler generates a member initialization
 * list (": member(value)") in the state class constructor only when it sees state variables.
 * This initialization list ensures the Actor<T> base class and its internal Promise are fully
 * initialized before any code runs. Without it, early returns try to use an uninitialized Promise.
 *
 * FIX: Declare at least one state variable BEFORE any early return. Declaration alone is enough.
 *
 * CORRECT:
 *   ACTOR Future<Void> someActor(...) {
 *       state std::string data;                // Triggers member init list (requires default ctor)
 *       if (earlyExitCondition) return Void(); // Safe - Promise is initialized
 *       data = computeValue();                 // Can initialize later
 *   }
 *
 *   // Or if no default constructor: state MyType x(params); and initialize at declaration
 *
 * WRONG (canBeSet() crash):
 *   ACTOR Future<Void> someActor(...) {
 *       if (earlyExitCondition) return Void(); // CRASH - no member init list generated yet
 *       state std::string data;                // Too late - compiler didn't see it early enough
 *   }
 */

// MockS3 persistence file extensions
namespace {
constexpr const char* OBJECT_DATA_SUFFIX = ".data";
constexpr const char* OBJECT_META_SUFFIX = ".meta.json";
constexpr const char* MULTIPART_STATE_SUFFIX = ".state.json";
constexpr size_t OBJECT_META_SUFFIX_LEN = 10; // strlen(".meta.json")
constexpr size_t MULTIPART_STATE_SUFFIX_LEN = 11; // strlen(".state.json")
} // namespace

// Global storage for MockS3 (shared across all simulated processes)
struct MockS3GlobalStorage {
	struct ObjectData {
		std::string content;
		HTTP::Headers headers;
		std::map<std::string, std::string> tags;
		std::string etag;
		double lastModified;

		ObjectData() : lastModified(now()) {}
		ObjectData(const std::string& data) : content(data), lastModified(now()) { etag = generateETag(data); }

		static std::string generateETag(const std::string& content) {
			return "\"" + HTTP::computeMD5Sum(content) + "\"";
		}
	};

	struct MultipartUpload {
		std::string uploadId;
		std::string bucket;
		std::string object;
		std::map<int, std::pair<std::string, std::string>> parts; // partNum -> {etag, content}
		HTTP::Headers metadata;
		double initiated;

		MultipartUpload() = default;
		MultipartUpload(const std::string& b, const std::string& o) : bucket(b), object(o), initiated(now()) {
			uploadId = deterministicRandom()->randomUniqueID().toString();
		}
	};

	std::map<std::string, std::map<std::string, ObjectData>> buckets;
	std::map<std::string, MultipartUpload> multipartUploads;

	// Persistence configuration
	std::string persistenceDir;
	bool persistenceEnabled = false;
	bool persistenceLoaded = false;

	// Note: In FDB simulation, function-local statics are SHARED across all simulated processes
	// because they all run on the same OS thread. This is exactly what we want for MockS3 storage.
	MockS3GlobalStorage() { TraceEvent("MockS3GlobalStorageCreated").detail("Address", format("%p", this)); }

	// Clear all stored data - called at the start of each simulation test to prevent
	// data accumulation across multiple tests
	void clearStorage() {
		buckets.clear();
		multipartUploads.clear();
		TraceEvent("MockS3GlobalStorageCleared").detail("Address", format("%p", this));
	}

	// Enable persistence to specified directory
	// Note: When using simulation filesystem, directories are created automatically by file open()
	// and each simulated machine has its own isolated directory structure
	void enablePersistence(const std::string& dir) {
		persistenceDir = dir;
		persistenceEnabled = true;
		persistenceLoaded = false;

		TraceEvent("MockS3PersistenceEnabled")
		    .detail("Directory", dir)
		    .detail("Address", format("%p", this))
		    .detail("UsingSimulationFS", g_network->isSimulated());
	}

	// Get paths for persistence files
	// Note: Object names with slashes map directly to filesystem directory structure
	std::string getObjectDataPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_DATA_SUFFIX;
	}

	std::string getObjectMetaPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_META_SUFFIX;
	}

	std::string getMultipartStatePath(const std::string& uploadId) const {
		return persistenceDir + "/multipart/" + uploadId + MULTIPART_STATE_SUFFIX;
	}

	std::string getMultipartPartPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
	}

	std::string getMultipartPartMetaPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum) + OBJECT_META_SUFFIX;
	}
};

// Accessor function - uses function-local static for lazy initialization
// In simulation, this static is shared across all simulated processes (same OS thread)
static MockS3GlobalStorage& getGlobalStorage() {
	static MockS3GlobalStorage storage;
	return storage;
}

// Helper: Create all parent directories for a file path
// Uses platform::createDirectory which handles recursive creation and EEXIST errors
static void createParentDirectories(const std::string& filePath) {
	size_t lastSlash = filePath.find_last_of('/');
	if (lastSlash != std::string::npos && lastSlash > 0) {
		std::string parentDir = filePath.substr(0, lastSlash);
		platform::createDirectory(parentDir); // Handles recursive creation and EEXIST
	}
}

// ACTOR: Atomic file write using simulation filesystem without chaos injection
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
// (see AsyncFileChaos.h:40). OPEN_NO_AIO controls AsyncFileNonDurable behavior.
															#line 193 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via atomicWriteFile()
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class AtomicWriteFileActor>
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActorState {
															#line 200 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActorState(std::string const& path,std::string const& content) 
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   content(content)
															#line 209 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	~AtomicWriteFileActorState() 
	{
		fdb_probe_actor_destroy("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 194 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				createParentDirectories(path);
															#line 198 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_ATOMIC_WRITE_AND_CREATE | IAsyncFile::OPEN_CREATE | IAsyncFile::OPEN_READWRITE | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 198 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 229 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 1;
															#line 198 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 234 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 212 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<AtomicWriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AtomicWriteFileActorState(); static_cast<AtomicWriteFileActor*>(this)->destroy(); return 0; }
															#line 263 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<AtomicWriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 210 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceWriteException").error(e).detail("Path", path);
															#line 276 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 204 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = file->write(content.data(), content.size(), 0);
															#line 204 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 293 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 2;
															#line 204 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 1, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 298 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 198 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 307 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 388 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 393 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 404 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 205 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 409 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 208 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 495 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 208 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 506 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 198 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 605 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via atomicWriteFile()
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActor final : public Actor<Void>, public ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< AtomicWriteFileActor, 1, Void >, public ActorCallback< AtomicWriteFileActor, 2, Void >, public FastAllocated<AtomicWriteFileActor>, public AtomicWriteFileActorState<AtomicWriteFileActor> {
															#line 610 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<AtomicWriteFileActor>::operator new;
	using FastAllocated<AtomicWriteFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3708685722167724032UL, 13128651616387471104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< AtomicWriteFileActor, 1, Void >;
friend struct ActorCallback< AtomicWriteFileActor, 2, Void >;
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActor(std::string const& path,std::string const& content) 
															#line 629 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   AtomicWriteFileActorState<AtomicWriteFileActor>(path, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1551474338982279168UL, 5682969234323159552UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("atomicWriteFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> atomicWriteFile( std::string const& path, std::string const& content ) {
															#line 191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new AtomicWriteFileActor(path, content));
															#line 664 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 214 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Read file content using simulation filesystem without chaos
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
															#line 671 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via readFileContent()
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class ReadFileContentActor>
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActorState {
															#line 678 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActorState(std::string const& path) 
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 218 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   exists(fileExists(path))
															#line 687 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	~ReadFileContentActorState() 
	{
		fdb_probe_actor_destroy("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 221 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!exists)
															#line 703 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 222 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 707 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
					this->~ReadFileContentActorState();
					static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 717 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadFileContentActor*>(this)->actor_wait_state = 1;
															#line 225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*>(static_cast<ReadFileContentActor*>(this)));
															#line 722 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 243 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadException").error(e).detail("Path", path);
															#line 244 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 754 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
			this->~ReadFileContentActorState();
			static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 227 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = file->size();
															#line 227 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 774 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 2;
															#line 227 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 1, int64_t >*>(static_cast<ReadFileContentActor*>(this)));
															#line 779 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 229 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = std::string();
															#line 230 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content.resize(fileSize);
															#line 231 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int> __when_expr_2 = file->read((uint8_t*)content.data(), fileSize, 0);
															#line 231 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 873 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 3;
															#line 231 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 2, int >*>(static_cast<ReadFileContentActor*>(this)));
															#line 878 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __fileSize,int loopDepth) 
	{
															#line 227 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileSize = __fileSize;
															#line 887 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __fileSize,int loopDepth) 
	{
		fileSize = std::move(__fileSize);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int const& bytesRead,int loopDepth) 
	{
															#line 232 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 234 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 968 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 235 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 972 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 241 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 976 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int && bytesRead,int loopDepth) 
	{
															#line 232 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 234 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 990 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 235 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 994 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 241 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 998 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 218 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	bool exists;
															#line 225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 227 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int64_t fileSize;
															#line 229 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1091 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via readFileContent()
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActor final : public Actor<std::string>, public ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >, public ActorCallback< ReadFileContentActor, 1, int64_t >, public ActorCallback< ReadFileContentActor, 2, int >, public FastAllocated<ReadFileContentActor>, public ReadFileContentActorState<ReadFileContentActor> {
															#line 1096 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<ReadFileContentActor>::operator new;
	using FastAllocated<ReadFileContentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15819552454592647936UL, 12556914020460102144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ReadFileContentActor, 1, int64_t >;
friend struct ActorCallback< ReadFileContentActor, 2, int >;
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActor(std::string const& path) 
															#line 1115 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<std::string>(),
		   ReadFileContentActorState<ReadFileContentActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7623102229524567296UL, 7126613595367428096UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFileContent");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadFileContentActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadFileContentActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<std::string> readFileContent( std::string const& path ) {
															#line 217 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<std::string>(new ReadFileContentActor(path));
															#line 1150 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 247 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete file using simulation filesystem
// Wraps deleteFile with trace events and error handling for MockS3 persistence cleanup
															#line 1157 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedFile()
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedFileActor>
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActorState {
															#line 1164 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActorState(std::string const& path) 
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path)
															#line 1171 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedFileActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 252 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = IAsyncFileSystem::filesystem()->deleteFile(path, true);
															#line 252 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1189 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 1;
															#line 252 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedFileActor, 0, Void >*>(static_cast<DeletePersistedFileActor*>(this)));
															#line 1194 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 257 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedFileActorState(); static_cast<DeletePersistedFileActor*>(this)->destroy(); return 0; }
															#line 1223 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 255 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceDeleteException").error(e).detail("Path", path);
															#line 1236 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 253 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1251 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 253 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1260 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedFileActor*>(this)->ActorCallback< DeletePersistedFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 1355 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedFile()
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActor final : public Actor<Void>, public ActorCallback< DeletePersistedFileActor, 0, Void >, public FastAllocated<DeletePersistedFileActor>, public DeletePersistedFileActorState<DeletePersistedFileActor> {
															#line 1360 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedFileActor>::operator new;
	using FastAllocated<DeletePersistedFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14074362365298153216UL, 13991377332325775616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedFileActor, 0, Void >;
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActor(std::string const& path) 
															#line 1377 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedFileActorState<DeletePersistedFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8878611606913210624UL, 16427746229923233536UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedFile( std::string const& path ) {
															#line 250 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedFileActor(path));
															#line 1410 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 259 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// JSON Serialization using rapidjson
static std::string serializeObjectMeta(const MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("etag", Value(obj.etag.c_str(), allocator), allocator);
	doc.AddMember("lastModified", obj.lastModified, allocator);

	Value tagsObj(kObjectType);
	for (const auto& tag : obj.tags) {
		tagsObj.AddMember(Value(tag.first.c_str(), allocator), Value(tag.second.c_str(), allocator), allocator);
	}
	doc.AddMember("tags", tagsObj, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

// JSON Deserialization using rapidjson
static void deserializeObjectMeta(const std::string& jsonStr, MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("etag") && doc["etag"].IsString())
		obj.etag = doc["etag"].GetString();
	if (doc.HasMember("lastModified") && doc["lastModified"].IsNumber())
		obj.lastModified = doc["lastModified"].GetDouble();
	if (doc.HasMember("tags") && doc["tags"].IsObject()) {
		for (auto& m : doc["tags"].GetObject()) {
			if (m.value.IsString())
				obj.tags[m.name.GetString()] = m.value.GetString();
		}
	}
}

static std::string serializeMultipartState(const MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("uploadId", Value(upload.uploadId.c_str(), allocator), allocator);
	doc.AddMember("bucket", Value(upload.bucket.c_str(), allocator), allocator);
	doc.AddMember("object", Value(upload.object.c_str(), allocator), allocator);
	doc.AddMember("initiated", upload.initiated, allocator);

	Value partsArray(kArrayType);
	for (const auto& part : upload.parts) {
		Value partObj(kObjectType);
		partObj.AddMember("partNum", part.first, allocator);
		partObj.AddMember("etag", Value(part.second.first.c_str(), allocator), allocator);
		partsArray.PushBack(partObj, allocator);
	}
	doc.AddMember("parts", partsArray, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

static void deserializeMultipartState(const std::string& jsonStr, MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("uploadId") && doc["uploadId"].IsString())
		upload.uploadId = doc["uploadId"].GetString();
	if (doc.HasMember("bucket") && doc["bucket"].IsString())
		upload.bucket = doc["bucket"].GetString();
	if (doc.HasMember("object") && doc["object"].IsString())
		upload.object = doc["object"].GetString();
	if (doc.HasMember("initiated") && doc["initiated"].IsNumber())
		upload.initiated = doc["initiated"].GetDouble();
	if (doc.HasMember("parts") && doc["parts"].IsArray()) {
		for (auto& partVal : doc["parts"].GetArray()) {
			if (partVal.HasMember("partNum") && partVal["partNum"].IsInt() && partVal.HasMember("etag") &&
			    partVal["etag"].IsString()) {
				int partNum = partVal["partNum"].GetInt();
				std::string etag = partVal["etag"].GetString();
				upload.parts[partNum] = { etag, "" }; // Content loaded separately from .part.N files
			}
		}
	}
}

// Forward declarations for state loading functions
															#line 1507 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir );

#line 353 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1511 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir );

#line 354 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1515 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  );

#line 355 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
Future<Void> loadMockS3PersistedStateFuture();

static std::string serializePartMeta(const std::string& etag) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();
	doc.AddMember("etag", Value(etag.c_str(), allocator), allocator);

	StringBuffer buffer;
	Writer<StringBuffer> writer(buffer); // Use Writer instead of PrettyWriter for compact output
	doc.Accept(writer);
	return buffer.GetString();
}

// ACTOR: Persist object data and metadata
															#line 1535 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistObject()
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistObjectActor>
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActorState {
															#line 1542 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActorState(std::string const& bucket,std::string const& object) 
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 1551 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistObject", reinterpret_cast<unsigned long>(this));

	}
	~PersistObjectActorState() 
	{
		fdb_probe_actor_destroy("persistObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 372 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 373 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 375 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = storage.buckets.find(bucket);
															#line 376 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter == storage.buckets.end())
															#line 1572 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 377 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1576 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 380 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto objectIter = bucketIter->second.find(object);
															#line 381 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (objectIter == bucketIter->second.end())
															#line 1586 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 382 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1590 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 386 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			content = objectIter->second.content;
															#line 387 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaJson = serializeObjectMeta(objectIter->second);
															#line 1600 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 392 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				dataPath = storage.getObjectDataPath(bucket, object);
															#line 393 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				metaPath = storage.getObjectMetaPath(bucket, object);
															#line 397 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(dataPath) || fileExists(dataPath + ".part"))
															#line 1608 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 398 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3ObjectAlreadyPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath);
															#line 402 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1614 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistObjectActorState();
					static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 405 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistingObject") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath) .detail("Size", content.size());
															#line 412 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(dataPath, content);
															#line 412 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1626 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistObjectActor*>(this)->actor_wait_state = 1;
															#line 412 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 0, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1631 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 425 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1660 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 422 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3PersistObjectFailed").error(e).detail("Bucket", bucket).detail("Object", object);
															#line 1673 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1690 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1695 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1706 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1711 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1795 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1804 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 386 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 387 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 392 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 393 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistObject()
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActor final : public Actor<Void>, public ActorCallback< PersistObjectActor, 0, Void >, public ActorCallback< PersistObjectActor, 1, Void >, public FastAllocated<PersistObjectActor>, public PersistObjectActorState<PersistObjectActor> {
															#line 1914 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistObjectActor>::operator new;
	using FastAllocated<PersistObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4385135596665286144UL, 2390600776999120896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistObjectActor, 0, Void >;
friend struct ActorCallback< PersistObjectActor, 1, Void >;
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActor(std::string const& bucket,std::string const& object) 
															#line 1932 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistObjectActorState<PersistObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1247695763033469440UL, 1096140585632004864UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistObject( std::string const& bucket, std::string const& object ) {
															#line 371 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistObjectActor(bucket, object));
															#line 1966 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 427 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Persist multipart upload state
															#line 1972 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistMultipartState()
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistMultipartStateActor>
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActorState {
															#line 1979 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActorState(std::string const& uploadId) 
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 430 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 431 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   parts()
															#line 1990 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	~PersistMultipartStateActorState() 
	{
		fdb_probe_actor_destroy("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 433 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 434 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 436 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 437 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == storage.multipartUploads.end())
															#line 2011 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 438 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2015 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 441 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			const auto& upload = uploadIter->second;
															#line 442 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = storage.persistenceDir;
															#line 443 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			parts = upload.parts;
															#line 2027 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 448 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 452 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(statePath) || fileExists(statePath + ".part"))
															#line 2033 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 453 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartAlreadyPersisted").detail("UploadId", uploadId).detail("StatePath", statePath);
															#line 454 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2039 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistMultipartStateActorState();
					static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 457 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string stateJson = serializeMultipartState(upload);
															#line 458 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(statePath, stateJson);
															#line 458 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2051 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 1;
															#line 458 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 0, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2056 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 482 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2085 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 479 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 2098 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 461 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 462 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2115 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 461 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 462 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2126 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 477 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPersisted").detail("UploadId", uploadId).detail("PartsCount", parts.size());
															#line 2210 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 462 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partIter != parts.end()))
															#line 2226 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 463 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = partIter->first;
															#line 464 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		etag = partIter->second.first;
															#line 465 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = partIter->second.second;
															#line 467 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 468 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(partPath, partData);
															#line 468 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2242 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 2;
															#line 468 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 1, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2247 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 470 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2275 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2280 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 470 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2295 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2300 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2384 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2393 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 430 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 431 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>> parts;
															#line 461 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>>::iterator partIter;
															#line 463 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 464 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 465 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 467 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 470 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 2506 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistMultipartState()
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActor final : public Actor<Void>, public ActorCallback< PersistMultipartStateActor, 0, Void >, public ActorCallback< PersistMultipartStateActor, 1, Void >, public ActorCallback< PersistMultipartStateActor, 2, Void >, public FastAllocated<PersistMultipartStateActor>, public PersistMultipartStateActorState<PersistMultipartStateActor> {
															#line 2511 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistMultipartStateActor>::operator new;
	using FastAllocated<PersistMultipartStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13934349301303654400UL, 4116443332706739456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMultipartStateActor, 0, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 1, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 2, Void >;
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActor(std::string const& uploadId) 
															#line 2530 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistMultipartStateActorState<PersistMultipartStateActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9066204211401481728UL, 7216261254142226944UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMultipartState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistMultipartState( std::string const& uploadId ) {
															#line 429 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistMultipartStateActor(uploadId));
															#line 2565 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 484 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted object
															#line 2571 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedObject()
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedObjectActor>
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActorState {
															#line 2578 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActorState(std::string const& bucket,std::string const& object) 
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 487 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   dataPath(),
															#line 488 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   metaPath()
															#line 2591 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedObjectActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 490 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 491 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 493 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = storage.getObjectDataPath(bucket, object);
															#line 494 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = storage.getObjectMetaPath(bucket, object);
															#line 2612 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 497 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(dataPath);
															#line 497 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2618 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 1;
															#line 497 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 0, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2623 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 508 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedObjectActorState(); static_cast<DeletePersistedObjectActor*>(this)->destroy(); return 0; }
															#line 2652 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 502 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedObjectFailed") .error(e) .detail("Bucket", bucket) .detail("Object", object);
															#line 2665 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2682 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2687 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2698 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 498 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2703 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 500 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 500 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2796 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 487 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 488 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 2897 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedObject()
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActor final : public Actor<Void>, public ActorCallback< DeletePersistedObjectActor, 0, Void >, public ActorCallback< DeletePersistedObjectActor, 1, Void >, public FastAllocated<DeletePersistedObjectActor>, public DeletePersistedObjectActorState<DeletePersistedObjectActor> {
															#line 2902 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedObjectActor>::operator new;
	using FastAllocated<DeletePersistedObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6876170406032733440UL, 10735752243955669504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedObjectActor, 0, Void >;
friend struct ActorCallback< DeletePersistedObjectActor, 1, Void >;
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActor(std::string const& bucket,std::string const& object) 
															#line 2920 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedObjectActorState<DeletePersistedObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5402173951988437248UL, 4607022702540543488UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedObject( std::string const& bucket, std::string const& object ) {
															#line 486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedObjectActor(bucket, object));
															#line 2954 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 510 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted multipart upload
															#line 2960 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedMultipart()
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedMultipartActor>
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActorState {
															#line 2967 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActorState(std::string const& uploadId) 
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 513 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   maxPart(),
															#line 514 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 515 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNum(),
															#line 516 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partPath(),
															#line 517 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partMetaPath()
															#line 2984 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedMultipartActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 519 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 520 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 3001 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 524 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 525 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				maxPart = 100;
															#line 526 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (uploadIter != storage.multipartUploads.end())
															#line 3009 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 527 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					for( const auto& part : uploadIter->second.parts ) {
															#line 528 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						maxPart = std::max(maxPart, part.first);
															#line 3015 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					}
				}
															#line 533 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				persistenceDir = storage.persistenceDir;
															#line 536 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 537 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(statePath);
															#line 537 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3026 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 1;
															#line 537 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 0, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3031 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 561 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedMultipartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedMultipartActorState(); static_cast<DeletePersistedMultipartActor*>(this)->destroy(); return 0; }
															#line 3060 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedMultipartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 558 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 3073 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 542 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3090 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 542 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 556 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartDeleted").detail("UploadId", uploadId);
															#line 3185 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 542 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxPart + 10))
															#line 3201 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 543 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 544 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 545 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(partPath);
															#line 545 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3213 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 2;
															#line 545 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 1, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3218 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3242 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3247 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3258 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3263 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 551 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3349 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3355 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3360 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 551 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3376 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3382 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3387 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 513 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxPart;
															#line 514 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 515 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 516 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 517 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 3590 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedMultipart()
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActor final : public Actor<Void>, public ActorCallback< DeletePersistedMultipartActor, 0, Void >, public ActorCallback< DeletePersistedMultipartActor, 1, Void >, public ActorCallback< DeletePersistedMultipartActor, 2, Void >, public ActorCallback< DeletePersistedMultipartActor, 3, Void >, public FastAllocated<DeletePersistedMultipartActor>, public DeletePersistedMultipartActorState<DeletePersistedMultipartActor> {
															#line 3595 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedMultipartActor>::operator new;
	using FastAllocated<DeletePersistedMultipartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2987286003197211904UL, 6107226078382917632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedMultipartActor, 0, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 1, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 2, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 3, Void >;
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActor(std::string const& uploadId) 
															#line 3615 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedMultipartActorState<DeletePersistedMultipartActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15695025802244908800UL, 1209722574274508288UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedMultipart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedMultipart( std::string const& uploadId ) {
															#line 512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedMultipartActor(uploadId));
															#line 3651 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 563 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Mock S3 Server Implementation for deterministic testing
class MockS3ServerImpl {
public:
	using ObjectData = MockS3GlobalStorage::ObjectData;
	using MultipartUpload = MockS3GlobalStorage::MultipartUpload;

	MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Constructor").detail("Address", format("%p", this)); }

	~MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Destructor").detail("Address", format("%p", this)); }

	// S3 Operation Handlers
																#line 3667 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleRequest()
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleRequestActor>
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActorState {
															#line 3673 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response)
															#line 3684 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	~HandleRequestActorState() 
	{
		fdb_probe_actor_destroy("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 579 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3Request") .detail("Method", req->verb) .detail("Resource", req->resource) .detail("ContentLength", req->data.contentLen) .detail("Headers", req->data.headers.size()) .detail("UserAgent", req->data.headers.find("User-Agent") != req->data.headers.end() ? req->data.headers.at("User-Agent") : "N/A") .detail("Host", req->data.headers.find("Host") != req->data.headers.end() ? req->data.headers.at("Host") : "N/A");
															#line 3699 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 592 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string bucket, object;
															#line 593 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::map<std::string, std::string> queryParams;
															#line 594 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->parseS3Request(req->resource, bucket, object, queryParams);
															#line 596 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ParsedRequest") .detail("Bucket", bucket) .detail("Object", object) .detail("QueryParamCount", queryParams.size());
															#line 602 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (queryParams.count("uploads"))
															#line 3711 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 603 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = self->handleMultipartStart(self, req, response, bucket, object);
															#line 603 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3717 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleRequestActor*>(this)->actor_wait_state = 1;
															#line 603 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 0, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3722 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 604 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (queryParams.count("uploadId"))
															#line 3729 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					{
															#line 605 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("partNumber"))
															#line 3733 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 606 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							StrictFuture<Void> __when_expr_1 = self->handleUploadPart(self, req, response, bucket, object, queryParams);
															#line 606 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3739 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
							static_cast<HandleRequestActor*>(this)->actor_wait_state = 2;
															#line 606 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 1, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 607 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "POST")
															#line 3751 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 608 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_2 = self->handleMultipartComplete(self, req, response, bucket, object, queryParams);
															#line 608 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3757 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 3;
															#line 608 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 2, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3762 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 609 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "DELETE")
															#line 3769 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 610 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_3 = self->handleMultipartAbort(self, req, response, bucket, object, queryParams);
															#line 610 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3775 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1when4(__when_expr_3.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 4;
															#line 610 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 3, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3780 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 612 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "InvalidRequest", "Unknown multipart operation");
															#line 3787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont9(loopDepth);
								}
							}
						}
					}
					else
					{
															#line 615 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("tagging"))
															#line 3797 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 616 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "PUT")
															#line 3801 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 617 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_4 = self->handlePutObjectTags(self, req, response, bucket, object);
															#line 617 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1when5(__when_expr_4.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 5;
															#line 617 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 4, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3812 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 618 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "GET")
															#line 3819 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 619 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_5 = self->handleGetObjectTags(self, req, response, bucket, object);
															#line 619 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3825 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), loopDepth); else return a_body1when6(__when_expr_5.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 6;
															#line 619 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 5, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3830 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 621 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not allowed for tagging");
															#line 3837 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont15(loopDepth);
								}
							}
						}
						else
						{
															#line 626 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (queryParams.count("list-type") || (req->verb == "GET" && object.empty()))
															#line 3846 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 628 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_6 = self->handleListObjects(self, req, response, bucket, queryParams);
															#line 628 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3852 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), loopDepth); else return a_body1when7(__when_expr_6.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 7;
															#line 628 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 6, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3857 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 629 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (object.empty())
															#line 3864 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 631 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "HEAD")
															#line 3868 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 632 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_7 = self->handleHeadBucket(self, req, response, bucket);
															#line 632 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3874 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), loopDepth); else return a_body1when8(__when_expr_7.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 8;
															#line 632 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 7, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3879 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 633 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "PUT")
															#line 3886 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 634 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_8 = self->handlePutBucket(self, req, response, bucket);
															#line 634 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3892 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), loopDepth); else return a_body1when9(__when_expr_8.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 9;
															#line 634 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 8, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3897 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 636 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Bucket operation not supported");
															#line 3904 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = a_body1cont23(loopDepth);
										}
									}
								}
								else
								{
															#line 643 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "PUT")
															#line 3913 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 644 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_9 = self->handlePutObject(self, req, response, bucket, object);
															#line 644 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3919 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), loopDepth); else return a_body1when10(__when_expr_9.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 10;
															#line 644 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 9, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3924 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 645 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "GET")
															#line 3931 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 646 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_10 = self->handleGetObject(self, req, response, bucket, object);
															#line 646 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3937 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), loopDepth); else return a_body1when11(__when_expr_10.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 11;
															#line 646 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 10, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3942 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 647 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (req->verb == "DELETE")
															#line 3949 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											{
															#line 648 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												StrictFuture<Void> __when_expr_11 = self->handleDeleteObject(self, req, response, bucket, object);
															#line 648 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3955 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch2(__when_expr_11.getError(), loopDepth); else return a_body1when12(__when_expr_11.get(), loopDepth); };
												static_cast<HandleRequestActor*>(this)->actor_wait_state = 12;
															#line 648 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 11, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3960 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												loopDepth = 0;
											}
											else
											{
															#line 649 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (req->verb == "HEAD")
															#line 3967 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												{
															#line 650 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													StrictFuture<Void> __when_expr_12 = self->handleHeadObject(self, req, response, bucket, object);
															#line 650 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3973 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch2(__when_expr_12.getError(), loopDepth); else return a_body1when13(__when_expr_12.get(), loopDepth); };
													static_cast<HandleRequestActor*>(this)->actor_wait_state = 13;
															#line 650 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 12, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3978 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = 0;
												}
												else
												{
															#line 652 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not supported");
															#line 3985 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = a_body1cont32(loopDepth);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 662 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleRequestActorState(); static_cast<HandleRequestActor*>(this)->destroy(); return 0; }
															#line 4023 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 658 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3RequestError").error(e).detail("Resource", req->resource);
															#line 659 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			self->sendError(response, 500, "InternalError", "Internal server error");
															#line 4038 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont15(int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont18(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19(_, loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont20(int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont21(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont23(int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont24(_, loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont24(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont26(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont28(int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont32(int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont33(_, loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont33(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont35(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 5295 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleRequest()
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActor final : public Actor<Void>, public ActorCallback< HandleRequestActor, 0, Void >, public ActorCallback< HandleRequestActor, 1, Void >, public ActorCallback< HandleRequestActor, 2, Void >, public ActorCallback< HandleRequestActor, 3, Void >, public ActorCallback< HandleRequestActor, 4, Void >, public ActorCallback< HandleRequestActor, 5, Void >, public ActorCallback< HandleRequestActor, 6, Void >, public ActorCallback< HandleRequestActor, 7, Void >, public ActorCallback< HandleRequestActor, 8, Void >, public ActorCallback< HandleRequestActor, 9, Void >, public ActorCallback< HandleRequestActor, 10, Void >, public ActorCallback< HandleRequestActor, 11, Void >, public ActorCallback< HandleRequestActor, 12, Void >, public FastAllocated<HandleRequestActor>, public HandleRequestActorState<HandleRequestActor> {
															#line 5300 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleRequestActor>::operator new;
	using FastAllocated<HandleRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(631352682771585280UL, 6348766963303882240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleRequestActor, 0, Void >;
friend struct ActorCallback< HandleRequestActor, 1, Void >;
friend struct ActorCallback< HandleRequestActor, 2, Void >;
friend struct ActorCallback< HandleRequestActor, 3, Void >;
friend struct ActorCallback< HandleRequestActor, 4, Void >;
friend struct ActorCallback< HandleRequestActor, 5, Void >;
friend struct ActorCallback< HandleRequestActor, 6, Void >;
friend struct ActorCallback< HandleRequestActor, 7, Void >;
friend struct ActorCallback< HandleRequestActor, 8, Void >;
friend struct ActorCallback< HandleRequestActor, 9, Void >;
friend struct ActorCallback< HandleRequestActor, 10, Void >;
friend struct ActorCallback< HandleRequestActor, 11, Void >;
friend struct ActorCallback< HandleRequestActor, 12, Void >;
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 5329 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleRequestActorState<HandleRequestActor>(self, req, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3085149730912622080UL, 127431907239638784UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleRequestActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleRequestActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleRequestActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< HandleRequestActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< HandleRequestActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< HandleRequestActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< HandleRequestActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< HandleRequestActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< HandleRequestActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< HandleRequestActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< HandleRequestActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< HandleRequestActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleRequest( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleRequestActor(self, req, response));
															#line 5373 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 664 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	void parseS3Request(const std::string& resource,
	                    std::string& bucket,
	                    std::string& object,
	                    std::map<std::string, std::string>& queryParams) {

		// Split resource into path and query string
		size_t queryPos = resource.find('?');
		std::string path = (queryPos != std::string::npos) ? resource.substr(0, queryPos) : resource;
		std::string query = (queryPos != std::string::npos) ? resource.substr(queryPos + 1) : "";

		// Parse path: /bucket/object (like real S3)
		if (path.size() > 1) {
			path = path.substr(1); // Remove leading /
			size_t slashPos = path.find('/');
			if (slashPos != std::string::npos) {
				bucket = path.substr(0, slashPos);
				object = path.substr(slashPos + 1);
			} else {
				bucket = path;
				object = "";
			}
		}

		// Parse query parameters
		if (!query.empty()) {
			std::regex paramRegex("([^&=]+)=?([^&]*)");
			std::sregex_iterator iter(query.begin(), query.end(), paramRegex);
			std::sregex_iterator end;

			for (; iter != end; ++iter) {
				std::string key = iter->str(1);
				std::string value = iter->str(2);
				// URL decode the parameter value
				queryParams[key] = HTTP::urlDecode(value);
			}
		}

		// MockS3Server handles S3 HTTP requests where bucket is always the first path component
		// For bucket operations: HEAD /bucket_name
		// For object operations: HEAD /bucket_name/object_path
		if (bucket.empty()) {
			TraceEvent(SevWarn, "MockS3MissingBucketInPath").detail("Resource", resource).detail("QueryString", query);
			throw backup_invalid_url();
		}

		TraceEvent("MockS3ParsedPath")
		    .detail("OriginalResource", resource)
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("QueryString", query);
	}

	// Parse HTTP Range header: "bytes=start-end"
	// Returns true if parsing succeeded, false otherwise
	// Sets rangeStart and rangeEnd to the parsed values
	static bool parseRangeHeader(const std::string& rangeHeader, int64_t& rangeStart, int64_t& rangeEnd) {
		if (rangeHeader.empty()) {
			return false;
		}

		// Check for "bytes=" prefix
		if (rangeHeader.substr(0, 6) != "bytes=") {
			return false;
		}

		std::string range = rangeHeader.substr(6);
		size_t dashPos = range.find('-');
		if (dashPos == std::string::npos) {
			return false;
		}

		try {
			rangeStart = std::stoll(range.substr(0, dashPos));
			std::string endStr = range.substr(dashPos + 1);
			if (endStr.empty()) {
				// Open-ended range (e.g., "bytes=100-")
				rangeEnd = -1; // Indicates open-ended
			} else {
				rangeEnd = std::stoll(endStr);
			}
			return true;
		} catch (...) {
			return false;
		}
	}

	// Multipart Upload Operations
																#line 5465 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartStart()
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartStartActor>
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActorState {
															#line 5471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 5486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartStartActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 758 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartStart").detail("Bucket", bucket).detail("Object", object);
															#line 763 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string existingUploadId;
															#line 764 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( const auto& pair : getGlobalStorage().multipartUploads ) {
															#line 765 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (pair.second.bucket == bucket && pair.second.object == object)
															#line 5507 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 766 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					existingUploadId = pair.first;
															#line 767 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartStartIdempotent") .detail("Bucket", bucket) .detail("Object", object) .detail("ExistingUploadId", existingUploadId);
															#line 5513 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					break;
				}
			}
															#line 775 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = std::string();
															#line 776 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!existingUploadId.empty())
															#line 5521 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 777 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = existingUploadId;
															#line 5525 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont1(loopDepth);
			}
			else
			{
															#line 780 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				MultipartUpload upload(bucket, object);
															#line 781 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = upload.uploadId;
															#line 782 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 783 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3MultipartStarted").detail("UploadId", uploadId);
															#line 786 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (getGlobalStorage().persistenceEnabled)
															#line 5540 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5546 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 1;
															#line 787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartStartActor, 0, Void >*>(static_cast<HandleMultipartStartActor*>(this)));
															#line 5551 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont5(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 792 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 802 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 804 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5584 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartStartActor*>(this)->ActorCallback< HandleMultipartStartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartStartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 775 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 5697 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartStart()
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActor final : public Actor<Void>, public ActorCallback< HandleMultipartStartActor, 0, Void >, public FastAllocated<HandleMultipartStartActor>, public HandleMultipartStartActorState<HandleMultipartStartActor> {
															#line 5702 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartStartActor>::operator new;
	using FastAllocated<HandleMultipartStartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1334039173741074176UL, 13296020425517021440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartStartActor, 0, Void >;
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 5719 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartStartActorState<HandleMultipartStartActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10763588338010117376UL, 16830117502434504448UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartStart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartStartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartStart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartStartActor(self, req, response, bucket, object));
															#line 5751 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 806 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 5756 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleUploadPart()
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleUploadPartActor>
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActorState {
															#line 5762 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 814 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId")),
															#line 815 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNumber(std::stoi(queryParams.at("partNumber")))
															#line 5783 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	~HandleUploadPartActorState() 
	{
		fdb_probe_actor_destroy("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 817 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3UploadPart") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ContentLength", req->data.contentLen) .detail("ActualContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.size() > 0 ? req->data.content.substr(0, std::min((size_t)20, req->data.content.size())) : "EMPTY");
															#line 827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 828 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 5802 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 829 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 830 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5808 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleUploadPartActorState();
				static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 834 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = ObjectData::generateETag(req->data.content);
															#line 835 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadIter->second.parts[partNumber] = { etag, req->data.content };
															#line 838 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 5820 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 839 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 839 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5826 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 1;
															#line 839 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleUploadPartActor, 0, Void >*>(static_cast<HandleUploadPartActor*>(this)));
															#line 5831 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 844 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 845 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 846 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 848 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 853 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5869 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state > 0) static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleUploadPartActor*>(this)->ActorCallback< HandleUploadPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleUploadPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 814 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 815 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNumber;
															#line 834 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 5982 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleUploadPart()
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActor final : public Actor<Void>, public ActorCallback< HandleUploadPartActor, 0, Void >, public FastAllocated<HandleUploadPartActor>, public HandleUploadPartActorState<HandleUploadPartActor> {
															#line 5987 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleUploadPartActor>::operator new;
	using FastAllocated<HandleUploadPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2479447467298230784UL, 15917489224360144896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleUploadPartActor, 0, Void >;
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleUploadPartActorState<HandleUploadPartActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2867735766649073408UL, 15512396953593662720UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleUploadPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleUploadPartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleUploadPart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleUploadPartActor(self, req, response, bucket, object, queryParams));
															#line 6036 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 855 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartComplete()
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartCompleteActor>
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActorState {
															#line 6047 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 863 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6066 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartCompleteActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 865 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartComplete").detail("UploadId", uploadId);
															#line 867 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 868 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6085 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 869 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 870 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6091 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartCompleteActorState();
				static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 874 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			combinedContent = std::string();
															#line 875 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( auto& part : uploadIter->second.parts ) {
															#line 876 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				combinedContent += part.second.second;
															#line 6103 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 879 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartDebug") .detail("UploadId", uploadId) .detail("PartsCount", uploadIter->second.parts.size()) .detail("CombinedSize", combinedContent.size()) .detail("CombinedPreview", combinedContent.size() > 0 ? combinedContent.substr(0, std::min((size_t)20, combinedContent.size())) : "EMPTY");
															#line 888 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(combinedContent);
															#line 889 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 891 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartFinalObject") .detail("UploadId", uploadId) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size()) .detail("StoredPreview", getGlobalStorage().buckets[bucket][object].content.size() > 0 ? getGlobalStorage().buckets[bucket][object].content.substr( 0, std::min((size_t)20, getGlobalStorage().buckets[bucket][object].content.size())) : "EMPTY");
															#line 901 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6115 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 902 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 902 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6121 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 1;
															#line 902 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 0, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6126 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 906 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 907 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (getGlobalStorage().persistenceEnabled)
															#line 6156 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 908 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 908 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6162 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 908 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6167 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 912 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 922 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 924 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 926 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6274 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 863 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 874 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string combinedContent;
															#line 6385 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartComplete()
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActor final : public Actor<Void>, public ActorCallback< HandleMultipartCompleteActor, 0, Void >, public ActorCallback< HandleMultipartCompleteActor, 1, Void >, public FastAllocated<HandleMultipartCompleteActor>, public HandleMultipartCompleteActorState<HandleMultipartCompleteActor> {
															#line 6390 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartCompleteActor>::operator new;
	using FastAllocated<HandleMultipartCompleteActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16901264196580430592UL, 8399051950875518720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartCompleteActor, 0, Void >;
friend struct ActorCallback< HandleMultipartCompleteActor, 1, Void >;
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6408 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartCompleteActorState<HandleMultipartCompleteActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7470605948036193280UL, 16274542347142593024UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartComplete");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartComplete( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartCompleteActor(self, req, response, bucket, object, queryParams));
															#line 6441 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 928 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6446 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartAbort()
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartAbortActor>
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActorState {
															#line 6452 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 936 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartAbortActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 938 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartAbort").detail("UploadId", uploadId);
															#line 940 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6490 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 942 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 943 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6496 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartAbortActorState();
				static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 947 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().multipartUploads.erase(uploadId);
															#line 948 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6506 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 949 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedMultipart(uploadId);
															#line 949 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 1;
															#line 949 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartAbortActor, 0, Void >*>(static_cast<HandleMultipartAbortActor*>(this)));
															#line 6517 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 952 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 953 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 954 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 956 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 958 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6553 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartAbortActor*>(this)->ActorCallback< HandleMultipartAbortActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 936 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 6662 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartAbort()
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActor final : public Actor<Void>, public ActorCallback< HandleMultipartAbortActor, 0, Void >, public FastAllocated<HandleMultipartAbortActor>, public HandleMultipartAbortActorState<HandleMultipartAbortActor> {
															#line 6667 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartAbortActor>::operator new;
	using FastAllocated<HandleMultipartAbortActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16614988012158174208UL, 14258444816572324352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartAbortActor, 0, Void >;
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6684 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartAbortActorState<HandleMultipartAbortActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3855017746569260032UL, 6136799802664629248UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartAbort");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartAbortActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartAbort( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartAbortActor(self, req, response, bucket, object, queryParams));
															#line 6716 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 960 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	// Object Tagging Operations
	static Future<Void> handlePutObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3PutObjectTags")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagsXML", req->data.content);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Parse tags XML (simplified parser)
		std::map<std::string, std::string> tags = self->parseTagsXML(req->data.content);
		objectIter->second.tags = tags;

		response->code = 200;
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3ObjectTagsSet")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", tags.size());

		return Void();
	}

	static Future<Void> handleGetObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3GetObjectTags").detail("Bucket", bucket).detail("Object", object);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Generate tags XML response
		std::string xml = self->generateTagsXML(objectIter->second.tags);
		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ObjectTagsRetrieved")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", objectIter->second.tags.size());

		return Void();
	}

	// Basic Object Operations
																#line 6794 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handlePutObject()
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandlePutObjectActor>
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActorState {
															#line 6800 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 6815 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	~HandlePutObjectActorState() 
	{
		fdb_probe_actor_destroy("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1040 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Debug") .detail("Bucket", bucket) .detail("Object", object) .detail("ContentLength", req->data.contentLen) .detail("ContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.substr(0, std::min(100, (int)req->data.content.size())));
															#line 1047 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(req->data.content);
															#line 1048 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = obj.etag;
															#line 1049 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 1051 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Stored") .detail("Bucket", bucket) .detail("Object", object) .detail("ETag", etag) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size());
															#line 1058 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6840 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1059 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 1059 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6846 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 1;
															#line 1059 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandlePutObjectActor, 0, Void >*>(static_cast<HandlePutObjectActor*>(this)));
															#line 6851 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1062 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1064 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1067 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6887 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state > 0) static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandlePutObjectActor*>(this)->ActorCallback< HandlePutObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandlePutObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 1048 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 6994 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handlePutObject()
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActor final : public Actor<Void>, public ActorCallback< HandlePutObjectActor, 0, Void >, public FastAllocated<HandlePutObjectActor>, public HandlePutObjectActorState<HandlePutObjectActor> {
															#line 6999 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandlePutObjectActor>::operator new;
	using FastAllocated<HandlePutObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14569255047761087744UL, 5976702089129408512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandlePutObjectActor, 0, Void >;
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7016 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandlePutObjectActorState<HandlePutObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4560658774791891968UL, 15471708591759102464UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handlePutObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandlePutObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handlePutObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandlePutObjectActor(self, req, response, bucket, object));
															#line 7048 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1076 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleGetObject(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket,
	                                    std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		std::string content = objectIter->second.content;
		std::string etag = objectIter->second.etag;
		std::string contentMD5 = HTTP::computeMD5Sum(content);

		// Handle HTTP Range header for partial content requests
		// This is essential for AsyncFileEncrypted to read encrypted blocks correctly
		int64_t rangeStart = 0;
		int64_t rangeEnd = static_cast<int64_t>(content.size() - 1);
		bool isRangeRequest = false;

		auto rangeHeader = req->data.headers.find("Range");
		if (rangeHeader != req->data.headers.end()) {
			int64_t parsedStart, parsedEnd;
			if (parseRangeHeader(rangeHeader->second, parsedStart, parsedEnd)) {
				rangeStart = parsedStart;
				if (parsedEnd == -1) {
					// Open-ended range (e.g., "bytes=100-")
					rangeEnd = static_cast<int64_t>(content.size() - 1);
				} else {
					rangeEnd = parsedEnd;
				}
				// Clamp range to actual content size
				int64_t contentSize = static_cast<int64_t>(content.size() - 1);
				rangeEnd = std::min(rangeEnd, contentSize);
				rangeStart = std::min(rangeStart, contentSize);
				if (rangeStart <= rangeEnd) {
					isRangeRequest = true;
				}
			}
		}

		// Extract the requested range
		std::string responseContent;
		if (isRangeRequest && rangeStart <= rangeEnd) {
			responseContent =
			    content.substr(static_cast<size_t>(rangeStart), static_cast<size_t>(rangeEnd - rangeStart + 1));
			response->code = 206; // Partial Content
			response->data.headers["Content-Range"] =
			    format("bytes %lld-%lld/%zu", rangeStart, rangeEnd, content.size());
			// For range requests, calculate MD5 of the partial content, not full content
			contentMD5 = HTTP::computeMD5Sum(responseContent);
		} else {
			responseContent = content;
			response->code = 200;
		}

		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Type"] = "binary/octet-stream";
		response->data.headers["Content-MD5"] = contentMD5;

		// Write content to response
		response->data.contentLen = responseContent.size();
		response->data.headers["Content-Length"] = std::to_string(responseContent.size());
		response->data.content->discardAll(); // Clear existing content

		if (!responseContent.empty()) {
			// Use the correct approach: getWriteBuffer from the UnsentPacketQueue
			PacketBuffer* buffer = response->data.content->getWriteBuffer(responseContent.size());
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(responseContent);
			pw.finish();
		}

		return Void();
	}

																#line 7137 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleDeleteObject()
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleDeleteObjectActor>
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActorState {
															#line 7143 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 7158 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	~HandleDeleteObjectActorState() 
	{
		fdb_probe_actor_destroy("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1167 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3DeleteObject").detail("Bucket", bucket).detail("Object", object);
															#line 1169 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = getGlobalStorage().buckets.find(bucket);
															#line 1170 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter != getGlobalStorage().buckets.end())
															#line 7177 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1171 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIter->second.erase(object);
															#line 7181 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 1175 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 7185 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1176 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedObject(bucket, object);
															#line 1176 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7191 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 1;
															#line 1176 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleDeleteObjectActor, 0, Void >*>(static_cast<HandleDeleteObjectActor*>(this)));
															#line 7196 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1179 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1180 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1181 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1183 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1185 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7232 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state > 0) static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandleDeleteObjectActor*>(this)->ActorCallback< HandleDeleteObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 7337 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleDeleteObject()
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActor final : public Actor<Void>, public ActorCallback< HandleDeleteObjectActor, 0, Void >, public FastAllocated<HandleDeleteObjectActor>, public HandleDeleteObjectActorState<HandleDeleteObjectActor> {
															#line 7342 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleDeleteObjectActor>::operator new;
	using FastAllocated<HandleDeleteObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11755301209539626240UL, 5089032088794014976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleDeleteObjectActor, 0, Void >;
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7359 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleDeleteObjectActorState<HandleDeleteObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7788094060450111232UL, 5901935141690097664UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleDeleteObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleDeleteObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleDeleteObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1161 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleDeleteObjectActor(self, req, response, bucket, object));
															#line 7391 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1187 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleHeadObject(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket,
	                                     std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			TraceEvent("MockS3HeadObjectNoBucket")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("AvailableBuckets", getGlobalStorage().buckets.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			TraceEvent("MockS3HeadObjectNoObject")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("ObjectsInBucket", bucketIter->second.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		const ObjectData& obj = objectIter->second;
		std::string etag = obj.etag;
		size_t contentSize = obj.content.size();
		std::string preview = contentSize > 0 ? obj.content.substr(0, std::min((size_t)20, contentSize)) : "EMPTY";

		TraceEvent("MockS3HeadObjectFound")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("Size", contentSize)
		    .detail("Preview", preview);

		response->code = 200;
		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Length"] = std::to_string(contentSize);
		response->data.headers["Content-Type"] = "binary/octet-stream";
		// HEAD requests need contentLen set to actual size for headers
		response->data.contentLen = contentSize; // This controls ResponseContentSize in HTTP logs

		return Void();
	}

	// S3 ListObjects Operation
	static Future<Void> handleListObjects(MockS3ServerImpl* self,
	                                      Reference<HTTP::IncomingRequest> req,
	                                      Reference<HTTP::OutgoingResponse> response,
	                                      std::string bucket,
	                                      std::map<std::string, std::string> queryParams) {

		TraceEvent("MockS3ListObjects").detail("Bucket", bucket).detail("QueryParamCount", queryParams.size());

		// Get query parameters for listing
		std::string prefix = queryParams.count("prefix") ? queryParams.at("prefix") : "";
		std::string delimiter = queryParams.count("delimiter") ? queryParams.at("delimiter") : "";
		std::string marker = queryParams.count("marker") ? queryParams.at("marker") : "";
		std::string continuationToken =
		    queryParams.count("continuation-token") ? queryParams.at("continuation-token") : "";
		int maxKeys = queryParams.count("max-keys") ? std::stoi(queryParams.at("max-keys")) : 1000;

		TraceEvent("MockS3ListObjectsDebug")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("Delimiter", delimiter)
		    .detail("Marker", marker)
		    .detail("ContinuationToken", continuationToken)
		    .detail("MaxKeys", maxKeys);

		// Find bucket
		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		// Collect all matching objects first
		std::vector<std::pair<std::string, const ObjectData*>> matchingObjects;
		for (const auto& objectPair : bucketIter->second) {
			const std::string& objectName = objectPair.first;
			const ObjectData& objectData = objectPair.second;

			// Apply prefix filter
			if (!prefix.empty() && objectName.find(prefix) != 0) {
				continue;
			}

			matchingObjects.push_back({ objectName, &objectData });
		}

		// Sort objects by name for consistent pagination
		std::sort(matchingObjects.begin(), matchingObjects.end());

		// Find starting point for pagination
		size_t startIndex = 0;
		if (!marker.empty()) {
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > marker) {
					startIndex = i;
					break;
				}
			}
		} else if (!continuationToken.empty()) {
			// Simple continuation token implementation (just use the last object name)
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > continuationToken) {
					startIndex = i;
					break;
				}
			}
		}

		// Build list of objects for this page
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<ListBucketResult>\n";
		xml += "<Name>" + bucket + "</Name>\n";
		xml += "<Prefix>" + prefix + "</Prefix>\n";
		xml += "<MaxKeys>" + std::to_string(maxKeys) + "</MaxKeys>\n";

		if (!marker.empty()) {
			xml += "<Marker>" + marker + "</Marker>\n";
		}

		int count = 0;
		std::string lastKey;
		size_t totalMatching = matchingObjects.size();

		for (size_t i = startIndex; i < matchingObjects.size() && count < maxKeys; i++) {
			const std::string& objectName = matchingObjects[i].first;
			const ObjectData* objectData = matchingObjects[i].second;

			xml += "<Contents>\n";
			xml += "<Key>" + objectName + "</Key>\n";
			xml += "<LastModified>" + std::to_string((int64_t)objectData->lastModified) + "</LastModified>\n";
			xml += "<ETag>" + objectData->etag + "</ETag>\n";
			xml += "<Size>" + std::to_string(objectData->content.size()) + "</Size>\n";
			xml += "<StorageClass>STANDARD</StorageClass>\n";
			xml += "</Contents>\n";

			lastKey = objectName;
			count++;
		}

		// Determine if there are more results
		bool isTruncated = (startIndex + count) < totalMatching;
		xml += "<IsTruncated>" + std::string(isTruncated ? "true" : "false") + "</IsTruncated>\n";

		if (isTruncated && !lastKey.empty()) {
			xml += "<NextMarker>" + lastKey + "</NextMarker>\n";
		}

		xml += "</ListBucketResult>";

		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ListObjectsCompleted")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("ObjectCount", count)
		    .detail("StartIndex", startIndex)
		    .detail("TotalMatching", totalMatching)
		    .detail("IsTruncated", isTruncated)
		    .detail("NextMarker", isTruncated ? lastKey : "");

		return Void();
	}

	// S3 Bucket Operations
	static Future<Void> handleHeadBucket(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket) {

		TraceEvent("MockS3HeadBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation like real S3)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketHead").detail("Bucket", bucket);

		return Void();
	}

	static Future<Void> handlePutBucket(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket) {

		TraceEvent("MockS3PutBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketCreated").detail("Bucket", bucket);

		return Void();
	}

	// Utility Methods

	void sendError(Reference<HTTP::OutgoingResponse> response,
	               int code,
	               const std::string& errorCode,
	               const std::string& message) {

		TraceEvent("MockS3Error").detail("Code", code).detail("ErrorCode", errorCode).detail("Message", message);

		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		                         "<Error>\n"
		                         "  <Code>%s</Code>\n"
		                         "  <Message>%s</Message>\n"
		                         "</Error>",
		                         errorCode.c_str(),
		                         message.c_str());

		sendXMLResponse(response, code, xml);
	}

	void sendXMLResponse(Reference<HTTP::OutgoingResponse> response, int code, const std::string& xml) {
		TraceEvent("MockS3SendXMLResponse_Start")
		    .detail("Code", code)
		    .detail("XMLSize", xml.size())
		    .detail("XMLPreview", xml.size() > 0 ? xml.substr(0, std::min((size_t)50, xml.size())) : "EMPTY");

		response->code = code;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.headers["Content-Length"] = std::to_string(xml.size());
		response->data.headers["Content-MD5"] = HTTP::computeMD5Sum(xml);

		// Actually put the XML content into the response
		if (xml.empty()) {
			response->data.contentLen = 0;
			TraceEvent("MockS3SendXMLResponse_Empty").detail("ResponseCode", response->code);
		} else {
			// Use the existing content queue instead of creating a new one
			// This prevents memory management issues and potential canBeSet() failures
			size_t contentSize = xml.size();
			response->data.contentLen = contentSize;

			// Clear any existing content and write the XML
			response->data.content->discardAll();
			PacketBuffer* buffer = response->data.content->getWriteBuffer(contentSize);
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(xml);
			pw.finish();
		}

		TraceEvent("MockS3SendXMLResponse_Complete")
		    .detail("FinalCode", response->code)
		    .detail("FinalContentLen", response->data.contentLen)
		    .detail("XMLSize", xml.size());
	}

	std::map<std::string, std::string> parseTagsXML(const std::string& xml) {
		std::map<std::string, std::string> tags;

		// Simplified XML parsing for tags - this would need a proper XML parser in production
		std::regex tagRegex("<Tag><Key>([^<]+)</Key><Value>([^<]*)</Value></Tag>");
		std::sregex_iterator iter(xml.begin(), xml.end(), tagRegex);
		std::sregex_iterator end;

		for (; iter != end; ++iter) {
			std::string key = iter->str(1);
			std::string value = iter->str(2);
			tags[key] = value;

			TraceEvent("MockS3ParsedTag").detail("Key", key).detail("Value", value);
		}

		return tags;
	}

	std::string generateTagsXML(const std::map<std::string, std::string>& tags) {
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Tagging><TagSet>";

		for (const auto& tag : tags) {
			xml += "<Tag><Key>" + tag.first + "</Key><Value>" + tag.second + "</Value></Tag>";
		}

		xml += "</TagSet></Tagging>";
		return xml;
	}
};

// Global registry to track registered servers and avoid conflicts
static std::map<std::string, bool> registeredServers;

// Clear global storage state for clean test runs
static void clearSingletonState() {
	getGlobalStorage().buckets.clear();
	getGlobalStorage().multipartUploads.clear();
	TraceEvent("MockS3ServerImpl_StateCleared");
}

// Process a Mock S3 request directly (for wrapping/chaos injection)
Future<Void> processMockS3Request(Reference<HTTP::IncomingRequest> req, Reference<HTTP::OutgoingResponse> response) {
	static MockS3ServerImpl serverInstance;
	return MockS3ServerImpl::handleRequest(&serverInstance, req, response);
}

// Request Handler Implementation - Each handler instance works with global storage
Future<Void> MockS3RequestHandler::handleRequest(Reference<HTTP::IncomingRequest> req,
                                                 Reference<HTTP::OutgoingResponse> response) {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard")
		    .detail("Method", "handleRequest")
		    .detail("Resource", req->resource);
		return Void();
	}

	return processMockS3Request(req, response);
}

Reference<HTTP::IRequestHandler> MockS3RequestHandler::clone() {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard").detail("Method", "clone");
		// Return nullptr - caller must handle this gracefully
		return Reference<HTTP::IRequestHandler>();
	}
	return makeReference<MockS3RequestHandler>();
}

// Safe server registration that prevents conflicts (internal implementation)
															#line 7737 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class RegisterMockS3Server_implActor>
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActorState {
															#line 7744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActorState(std::string const& ip,std::string const& port) 
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : ip(ip),
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   port(port),
															#line 1530 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   serverKey(ip + ":" + port)
															#line 7755 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	~RegisterMockS3Server_implActorState() 
	{
		fdb_probe_actor_destroy("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1533 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Start") .detail("IP", ip) .detail("Port", port) .detail("ServerKey", serverKey) .detail("IsSimulated", g_network->isSimulated()) .detail("AlreadyRegistered", registeredServers.count(serverKey) > 0);
															#line 1542 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (registeredServers.count(serverKey))
															#line 7772 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1543 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3ServerAlreadyRegistered").detail("Address", serverKey);
															#line 1544 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7778 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~RegisterMockS3Server_implActorState();
				static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1550 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!getGlobalStorage().persistenceEnabled)
															#line 7787 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 1551 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					std::string persistenceDir = "simfdb/mocks3";
															#line 1552 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					enableMockS3Persistence(persistenceDir);
															#line 1553 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", persistenceDir);
															#line 1558 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 1558 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7799 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 1;
															#line 1558 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 0, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7804 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont3(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1583 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7838 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerRegistrationFailed") .error(e) .detail("Address", serverKey) .detail("ErrorCode", e.code()) .detail("ErrorName", e.name());
															#line 1580 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7853 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1561 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Calling registerSimHTTPServer") .detail("Address", serverKey);
															#line 1565 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = g_simulator->registerSimHTTPServer(ip, port, makeReference<MockS3RequestHandler>());
															#line 1565 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7871 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1565 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7876 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1566 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1568 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1570 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 7976 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1566 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1568 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1570 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 7989 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string ip;
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string port;
															#line 1530 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 8088 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActor final : public Actor<Void>, public ActorCallback< RegisterMockS3Server_implActor, 0, Void >, public ActorCallback< RegisterMockS3Server_implActor, 1, Void >, public FastAllocated<RegisterMockS3Server_implActor>, public RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor> {
															#line 8093 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<RegisterMockS3Server_implActor>::operator new;
	using FastAllocated<RegisterMockS3Server_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6492106687497475072UL, 940949132257192960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterMockS3Server_implActor, 0, Void >;
friend struct ActorCallback< RegisterMockS3Server_implActor, 1, Void >;
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActor(std::string const& ip,std::string const& port) 
															#line 8111 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor>(ip, port),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13948469852865528320UL, 9795983047084281344UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerMockS3Server_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> registerMockS3Server_impl( std::string const& ip, std::string const& port ) {
															#line 1529 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new RegisterMockS3Server_implActor(ip, port));
															#line 8145 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1585 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Public Interface Implementation
															#line 8151 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3Server()
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerActor>
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActorState {
															#line 8158 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActorState(NetworkAddress const& listenAddress) 
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress)
															#line 8165 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerActorState() 
	{
		fdb_probe_actor_destroy("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1588 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerStarting").detail("ListenAddress", listenAddress.toString());
															#line 8180 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1591 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRegistering") .detail("IP", listenAddress.ip.toString()) .detail("Port", std::to_string(listenAddress.port)) .detail("IsSimulated", g_network->isSimulated());
															#line 1597 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = registerMockS3Server_impl(listenAddress.ip.toString(), std::to_string(listenAddress.port));
															#line 1597 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 8188 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 1;
															#line 1597 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerActor, 0, Void >*>(static_cast<StartMockS3ServerActor*>(this)));
															#line 8193 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1608 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerActorState(); static_cast<StartMockS3ServerActor*>(this)->destroy(); return 0; }
															#line 8222 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1604 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerStartError").error(e).detail("ListenAddress", listenAddress.toString());
															#line 1605 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8237 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1599 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8251 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1599 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8260 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerActor*>(this)->ActorCallback< StartMockS3ServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 8355 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3Server()
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerActor, 0, Void >, public FastAllocated<StartMockS3ServerActor>, public StartMockS3ServerActorState<StartMockS3ServerActor> {
															#line 8360 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerActor>::operator new;
	using FastAllocated<StartMockS3ServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15398603176224934400UL, 16545409720808957184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerActor, 0, Void >;
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActor(NetworkAddress const& listenAddress) 
															#line 8377 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerActorState<StartMockS3ServerActor>(listenAddress),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16442790091880884736UL, 12597676879247326208UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3Server");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3Server( NetworkAddress const& listenAddress ) {
															#line 1587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerActor(listenAddress));
															#line 8410 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1610 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Clear all MockS3 global storage - called at the start of each simulation test
void clearMockS3Storage() {
	getGlobalStorage().clearStorage();
	// Note: Do NOT clear chaos server registry here - it must persist across tests
	// like the simulator's httpHandlers map, to prevent duplicate registration attempts
}

// Enable persistence for MockS3 storage
void enableMockS3Persistence(const std::string& persistenceDir) {
	getGlobalStorage().enablePersistence(persistenceDir);
	TraceEvent("MockS3PersistenceConfigured").detail("Directory", persistenceDir);
}

// Check if MockS3 persistence is currently enabled
bool isMockS3PersistenceEnabled() {
	return getGlobalStorage().persistenceEnabled;
}

// ACTOR: Load persisted objects from disk
															#line 8434 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedObjects()
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedObjectsActor>
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActorState {
															#line 8441 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActorState(std::string const& persistenceDir) 
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1631 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   objectsDir(persistenceDir + "/objects")
															#line 8450 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedObjectsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1633 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(objectsDir))
															#line 8465 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1634 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadObjects").detail("Status", "NoObjectsDir");
															#line 1635 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8471 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedObjectsActorState();
				static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1640 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				buckets = platform::listFiles(objectsDir, "");
															#line 1642 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(buckets.begin(), buckets.end());
															#line 1643 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 1645 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 8486 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1700 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8515 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1697 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadObjectsFailed").error(e);
															#line 8528 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1695 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectsLoaded").detail("BucketsCount", getGlobalStorage().buckets.size());
															#line 8543 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1645 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(bucketIdx < buckets.size()))
															#line 8559 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1646 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucket = buckets[bucketIdx];
															#line 1647 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bucket == "." || bucket == "..")
															#line 8567 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1650 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketDir = objectsDir + "/" + bucket;
															#line 1651 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!directoryExists(bucketDir))
															#line 8575 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1655 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		files = platform::listFiles(bucketDir, "");
															#line 1656 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::sort(files.begin(), files.end());
															#line 1657 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 1659 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 8587 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1645 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8609 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1645 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8618 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1659 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 8634 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1660 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1663 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > OBJECT_META_SUFFIX_LEN && fileName.substr(fileName.size() - OBJECT_META_SUFFIX_LEN) == OBJECT_META_SUFFIX)
															#line 8642 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1666 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			objectName = fileName.substr(0, fileName.size() - OBJECT_META_SUFFIX_LEN);
															#line 1667 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = bucketDir + "/" + objectName + OBJECT_DATA_SUFFIX;
															#line 1668 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = bucketDir + "/" + fileName;
															#line 1670 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(dataPath))
															#line 8652 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1671 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3LoadObjectSkipped") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Reason", "NoDataFile");
															#line 8656 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				return a_body1loopBody1continue1(loopDepth); // continue
			}
															#line 1679 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(dataPath);
															#line 1679 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8663 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 1;
															#line 1679 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 0, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8668 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1continue1(int loopDepth) 
	{
															#line 1659 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8695 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1659 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 1680 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(metaPath);
															#line 1680 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8715 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 2;
															#line 1680 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 1, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8720 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string const& __content,int loopDepth) 
	{
															#line 1679 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = __content;
															#line 8729 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string && __content,int loopDepth) 
	{
		content = std::move(__content);
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
															#line 1683 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		MockS3GlobalStorage::ObjectData obj(content);
															#line 1684 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeObjectMeta(metaJson, obj);
															#line 1685 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().buckets[bucket][objectName] = std::move(obj);
															#line 1687 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectRestored") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Size", content.size());
															#line 8814 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string const& __metaJson,int loopDepth) 
	{
															#line 1680 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		metaJson = __metaJson;
															#line 8823 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string && __metaJson,int loopDepth) 
	{
		metaJson = std::move(__metaJson);
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1631 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectsDir;
															#line 1640 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> buckets;
															#line 1643 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int bucketIdx;
															#line 1646 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1650 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucketDir;
															#line 1655 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1657 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1660 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1666 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectName;
															#line 1667 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 1668 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1679 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1680 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 8939 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedObjects()
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActor final : public Actor<Void>, public ActorCallback< LoadPersistedObjectsActor, 0, std::string >, public ActorCallback< LoadPersistedObjectsActor, 1, std::string >, public FastAllocated<LoadPersistedObjectsActor>, public LoadPersistedObjectsActorState<LoadPersistedObjectsActor> {
															#line 8944 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedObjectsActor>::operator new;
	using FastAllocated<LoadPersistedObjectsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10417068698988429824UL, 11708838803047522816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedObjectsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedObjectsActor, 1, std::string >;
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActor(std::string const& persistenceDir) 
															#line 8962 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedObjectsActorState<LoadPersistedObjectsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15447654064422368256UL, 7597335790883620352UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedObjects");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 1, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir ) {
															#line 1630 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedObjectsActor(persistenceDir));
															#line 8996 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1702 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load persisted multipart uploads from disk
															#line 9002 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedMultipartUploadsActor>
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActorState {
															#line 9009 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActorState(std::string const& persistenceDir) 
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1705 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   multipartDir(persistenceDir + "/multipart")
															#line 9018 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedMultipartUploadsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1707 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(multipartDir))
															#line 9033 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1708 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadMultipart").detail("Status", "NoMultipartDir");
															#line 1709 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9039 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedMultipartUploadsActorState();
				static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1714 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				files = platform::listFiles(multipartDir, "");
															#line 1715 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(files.begin(), files.end());
															#line 1716 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 1718 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 9054 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1784 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9083 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1781 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartFailed").error(e);
															#line 9096 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1779 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadsLoaded").detail("UploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9111 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1718 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 9127 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1719 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1722 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > MULTIPART_STATE_SUFFIX_LEN && fileName.substr(fileName.size() - MULTIPART_STATE_SUFFIX_LEN) == MULTIPART_STATE_SUFFIX)
															#line 9135 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1724 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = fileName.substr(0, fileName.size() - MULTIPART_STATE_SUFFIX_LEN);
															#line 1725 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			statePath = multipartDir + "/" + fileName;
															#line 1728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(statePath);
															#line 1728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9145 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 1;
															#line 1728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9150 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1718 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9177 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1718 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9186 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1729 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (stateJson.empty())
															#line 9195 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1730 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartSkipped") .detail("UploadId", uploadId) .detail("Reason", "EmptyStateFile");
															#line 9199 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			return a_body1continue1(loopDepth); // continue
		}
															#line 1737 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload = MockS3GlobalStorage::MultipartUpload("", "");
															#line 1738 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.uploadId = uploadId;
															#line 1739 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeMultipartState(stateJson, upload);
															#line 1742 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 1743 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		maxAttempts = 10000;
															#line 1744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 9214 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string const& __stateJson,int loopDepth) 
	{
															#line 1728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		stateJson = __stateJson;
															#line 9223 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string && __stateJson,int loopDepth) 
	{
		stateJson = std::move(__stateJson);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1769 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadRestored") .detail("UploadId", uploadId) .detail("Bucket", upload.bucket) .detail("Object", upload.object) .detail("PartsCount", upload.parts.size());
															#line 1775 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 9304 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxAttempts))
															#line 9320 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1745 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = multipartDir + "/" + uploadId + ".part." + std::to_string(partNum);
															#line 1746 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 1748 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!fileExists(partPath) || !fileExists(partMetaPath))
															#line 9330 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(partPath);
															#line 1752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9338 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 2;
															#line 1752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9343 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1753 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_2 = readFileContent(partMetaPath);
															#line 1753 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9367 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 3;
															#line 1753 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9372 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string const& __partData,int loopDepth) 
	{
															#line 1752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = __partData;
															#line 9381 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string && __partData,int loopDepth) 
	{
		partData = std::move(__partData);
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont4(int loopDepth) 
	{
															#line 1756 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		using namespace rapidjson;
															#line 1757 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		Document doc;
															#line 1758 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		doc.Parse(partMetaJson.c_str());
															#line 1759 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string etag = doc.HasMember("etag") && doc["etag"].IsString() ? doc["etag"].GetString() : "";
															#line 1760 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.parts[partNum] = { etag, partData };
															#line 1762 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPartRestored") .detail("UploadId", uploadId) .detail("PartNumber", partNum) .detail("Size", partData.size());
															#line 1744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 9472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string const& __partMetaJson,int loopDepth) 
	{
															#line 1753 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = __partMetaJson;
															#line 9481 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string && __partMetaJson,int loopDepth) 
	{
		partMetaJson = std::move(__partMetaJson);
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1705 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string multipartDir;
															#line 1714 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1716 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1719 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1724 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 1725 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string statePath;
															#line 1728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string stateJson;
															#line 1737 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3GlobalStorage::MultipartUpload upload;
															#line 1742 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 1743 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxAttempts;
															#line 1745 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 1746 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 1752 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 1753 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 9599 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActor final : public Actor<Void>, public ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >, public FastAllocated<LoadPersistedMultipartUploadsActor>, public LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor> {
															#line 9604 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator new;
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1352991564088874496UL, 12105357160635106304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >;
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActor(std::string const& persistenceDir) 
															#line 9623 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17523876871774881024UL, 15657059887496463360UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedMultipartUploads");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir ) {
															#line 1704 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedMultipartUploadsActor(persistenceDir));
															#line 9658 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1786 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load all persisted state from disk
															#line 9664 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadMockS3PersistedStateImplActor>
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActorState {
															#line 9671 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActorState() 
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1789 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir()
															#line 9678 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	~LoadMockS3PersistedStateImplActorState() 
	{
		fdb_probe_actor_destroy("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1791 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled || getGlobalStorage().persistenceLoaded)
															#line 9693 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1792 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9697 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadMockS3PersistedStateImplActorState();
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1795 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = getGlobalStorage().persistenceDir;
															#line 1796 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3LoadPersistedStateStart").detail("PersistenceDir", persistenceDir);
															#line 9707 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1800 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadPersistedObjects(persistenceDir);
															#line 1800 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9713 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 1;
															#line 1800 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9718 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1815 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9747 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1811 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3LoadPersistedStateFailed").error(e);
															#line 1812 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9762 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9778 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9783 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9794 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1803 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9799 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1805 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9885 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1805 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1807 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9896 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1789 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 9991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActor final : public Actor<Void>, public ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >, public ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >, public FastAllocated<LoadMockS3PersistedStateImplActor>, public LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor> {
															#line 9996 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator new;
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(191691016996159232UL, 1501146699199021312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >;
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >;
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActor() 
															#line 10014 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10516119037031120128UL, 15944437332465256704UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadMockS3PersistedStateImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  ) {
															#line 1788 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadMockS3PersistedStateImplActor());
															#line 10048 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1817 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Load persisted state from disk (called at server startup) - returns Future for use in ACTOR context
Future<Void> loadMockS3PersistedStateFuture() {
	if (getGlobalStorage().persistenceEnabled && !getGlobalStorage().persistenceLoaded) {
		return loadMockS3PersistedStateImpl();
	}
	return Void();
}

// Unit Tests for MockS3Server
															#line 10062 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1827()
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1827Actor>
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1827ActorState {
															#line 10069 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1827ActorState(UnitTestParameters const& params) 
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10076 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1827", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1827ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1827", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1829 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1830 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1831 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1832 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1834 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1836 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1837 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1838 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1840 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1827Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1827ActorState(); static_cast<FlowTestCase1827Actor*>(this)->destroy(); return 0; }
															#line 10107 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1827Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1827ActorState();
			static_cast<FlowTestCase1827Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1827ActorState();
		static_cast<FlowTestCase1827Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10131 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1827()
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1827Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1827Actor>, public FlowTestCase1827ActorState<FlowTestCase1827Actor> {
															#line 10136 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1827Actor>::operator new;
	using FastAllocated<FlowTestCase1827Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6594492823559262720UL, 7282297542377186560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1827Actor(UnitTestParameters const& params) 
															#line 10152 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1827ActorState<FlowTestCase1827Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1827", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5741044775903930368UL, 4146515168313127424UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1827Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1827");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1827", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1827( UnitTestParameters const& params ) {
															#line 1827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1827Actor(params));
															#line 10184 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1827, "/MockS3Server/parseS3Request/ValidBucketParameter")

#line 1842 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10190 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1843()
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1843Actor>
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1843ActorState {
															#line 10197 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1843ActorState(UnitTestParameters const& params) 
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10204 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1843", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1843ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1843", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1845 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1846 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/?region=us-east-1";
															#line 1847 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1848 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10225 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1851 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1852 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10231 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1843ActorState();
		static_cast<FlowTestCase1843Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1857 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1843Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1843ActorState(); static_cast<FlowTestCase1843Actor*>(this)->destroy(); return 0; }
															#line 10260 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1843Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1843ActorState();
		static_cast<FlowTestCase1843Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1854 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10273 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10299 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1843()
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1843Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1843Actor>, public FlowTestCase1843ActorState<FlowTestCase1843Actor> {
															#line 10304 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1843Actor>::operator new;
	using FastAllocated<FlowTestCase1843Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8952072803928660480UL, 15339980749684809984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1843Actor(UnitTestParameters const& params) 
															#line 10320 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1843ActorState<FlowTestCase1843Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1843", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4449689845564738304UL, 14679834532682452736UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1843Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1843");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1843", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1843( UnitTestParameters const& params ) {
															#line 1843 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1843Actor(params));
															#line 10352 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1843, "/MockS3Server/parseS3Request/MissingBucketParameter")

#line 1859 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10358 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1860()
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1860Actor>
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1860ActorState {
															#line 10365 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1860ActorState(UnitTestParameters const& params) 
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10372 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1860", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1860ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1860", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1862 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1863 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/";
															#line 1864 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1865 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10393 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1868 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1869 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10399 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1860ActorState();
		static_cast<FlowTestCase1860Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1874 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1860Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1860ActorState(); static_cast<FlowTestCase1860Actor*>(this)->destroy(); return 0; }
															#line 10428 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1860Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1860ActorState();
		static_cast<FlowTestCase1860Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1871 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10441 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10467 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1860()
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1860Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1860Actor>, public FlowTestCase1860ActorState<FlowTestCase1860Actor> {
															#line 10472 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1860Actor>::operator new;
	using FastAllocated<FlowTestCase1860Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12519557153300169216UL, 17874954403895374592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1860Actor(UnitTestParameters const& params) 
															#line 10488 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1860ActorState<FlowTestCase1860Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1860", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9490600770812761088UL, 2532780761947624704UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1860Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1860");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1860", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1860( UnitTestParameters const& params ) {
															#line 1860 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1860Actor(params));
															#line 10520 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1860, "/MockS3Server/parseS3Request/EmptyQueryString")

#line 1876 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10526 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1877()
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1877Actor>
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1877ActorState {
															#line 10533 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1877ActorState(UnitTestParameters const& params) 
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10540 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1877", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1877ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1877", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1879 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1880 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/testobject?region=us-east-1";
															#line 1881 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1882 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1884 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1886 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1887 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "testobject");
															#line 1888 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1890 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1877Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1877ActorState(); static_cast<FlowTestCase1877Actor*>(this)->destroy(); return 0; }
															#line 10571 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1877Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1877ActorState();
			static_cast<FlowTestCase1877Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1877ActorState();
		static_cast<FlowTestCase1877Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10595 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1877()
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1877Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1877Actor>, public FlowTestCase1877ActorState<FlowTestCase1877Actor> {
															#line 10600 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1877Actor>::operator new;
	using FastAllocated<FlowTestCase1877Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2826948334606723840UL, 11735580979795992320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1877Actor(UnitTestParameters const& params) 
															#line 10616 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1877ActorState<FlowTestCase1877Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1877", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1693207587077163776UL, 1729170723089437696UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1877Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1877");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1877", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1877( UnitTestParameters const& params ) {
															#line 1877 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1877Actor(params));
															#line 10648 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1877, "/MockS3Server/parseS3Request/BucketParameterOverride")

#line 1892 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10654 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1893()
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1893Actor>
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1893ActorState {
															#line 10661 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1893ActorState(UnitTestParameters const& params) 
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10668 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1893", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1893ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1893", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1895 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1896 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/folder/subfolder/file.txt?region=us-east-1";
															#line 1897 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1898 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1900 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1902 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1903 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "folder/subfolder/file.txt");
															#line 1904 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1906 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1893Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1893ActorState(); static_cast<FlowTestCase1893Actor*>(this)->destroy(); return 0; }
															#line 10699 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1893Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1893ActorState();
			static_cast<FlowTestCase1893Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1893ActorState();
		static_cast<FlowTestCase1893Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10723 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1893()
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1893Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1893Actor>, public FlowTestCase1893ActorState<FlowTestCase1893Actor> {
															#line 10728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1893Actor>::operator new;
	using FastAllocated<FlowTestCase1893Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2018559741104399360UL, 11857151587847847936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1893Actor(UnitTestParameters const& params) 
															#line 10744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1893ActorState<FlowTestCase1893Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1893", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3218957982273941248UL, 11196005967260126208UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1893Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1893");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1893", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1893( UnitTestParameters const& params ) {
															#line 1893 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1893Actor(params));
															#line 10776 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1893, "/MockS3Server/parseS3Request/ComplexPath")

#line 1908 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10782 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1909()
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1909Actor>
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1909ActorState {
															#line 10789 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1909ActorState(UnitTestParameters const& params) 
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10796 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1909", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1909ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1909", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1911 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1912 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&param=value%3Dtest";
															#line 1913 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1914 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1916 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1918 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1919 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1920 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["param"] == "value=test");
															#line 1922 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1909Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1909ActorState(); static_cast<FlowTestCase1909Actor*>(this)->destroy(); return 0; }
															#line 10827 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1909Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1909ActorState();
			static_cast<FlowTestCase1909Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1909ActorState();
		static_cast<FlowTestCase1909Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10851 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1909()
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1909Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1909Actor>, public FlowTestCase1909ActorState<FlowTestCase1909Actor> {
															#line 10856 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1909Actor>::operator new;
	using FastAllocated<FlowTestCase1909Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17912513139841184512UL, 14792809178958659072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1909Actor(UnitTestParameters const& params) 
															#line 10872 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1909ActorState<FlowTestCase1909Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1909", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6394279864643024896UL, 11188941948135953920UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1909Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1909");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1909", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1909( UnitTestParameters const& params ) {
															#line 1909 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1909Actor(params));
															#line 10904 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1909, "/MockS3Server/parseS3Request/URLEncodedParameters")

#line 1924 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10910 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1925()
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1925Actor>
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1925ActorState {
															#line 10917 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1925ActorState(UnitTestParameters const& params) 
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10924 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1925", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1925ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1925", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1927 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1928 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1929 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1930 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1932 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1934 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1935 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1936 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1938 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1925Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1925ActorState(); static_cast<FlowTestCase1925Actor*>(this)->destroy(); return 0; }
															#line 10955 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1925Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1925ActorState();
			static_cast<FlowTestCase1925Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1925ActorState();
		static_cast<FlowTestCase1925Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10979 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1925()
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1925Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1925Actor>, public FlowTestCase1925ActorState<FlowTestCase1925Actor> {
															#line 10984 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1925Actor>::operator new;
	using FastAllocated<FlowTestCase1925Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8918781742842249984UL, 3179846662043881472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1925Actor(UnitTestParameters const& params) 
															#line 11000 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1925ActorState<FlowTestCase1925Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1925", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3304802687758828288UL, 6227703347189398016UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1925Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1925");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1925", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1925( UnitTestParameters const& params ) {
															#line 1925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1925Actor(params));
															#line 11032 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1925, "/MockS3Server/parseS3Request/EmptyPath")

#line 1940 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11038 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1941()
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1941Actor>
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1941ActorState {
															#line 11045 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1941ActorState(UnitTestParameters const& params) 
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1941", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1941ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1941", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1943 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1944 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1945 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1946 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1948 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1950 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1951 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1952 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1954 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1941Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1941ActorState(); static_cast<FlowTestCase1941Actor*>(this)->destroy(); return 0; }
															#line 11083 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1941Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1941ActorState();
			static_cast<FlowTestCase1941Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1941ActorState();
		static_cast<FlowTestCase1941Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11107 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1941()
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1941Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1941Actor>, public FlowTestCase1941ActorState<FlowTestCase1941Actor> {
															#line 11112 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1941Actor>::operator new;
	using FastAllocated<FlowTestCase1941Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13539224356245233152UL, 12411106718390699264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1941Actor(UnitTestParameters const& params) 
															#line 11128 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1941ActorState<FlowTestCase1941Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1941", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13752984186205025536UL, 14824459033514452224UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1941Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1941");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1941", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1941( UnitTestParameters const& params ) {
															#line 1941 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1941Actor(params));
															#line 11160 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1941, "/MockS3Server/parseS3Request/OnlyBucketInPath")

#line 1956 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11166 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1957()
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1957Actor>
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1957ActorState {
															#line 11173 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1957ActorState(UnitTestParameters const& params) 
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11180 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1957", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1957ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1957", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1959 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1960 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&version=1&encoding=utf8";
															#line 1961 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1962 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1964 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1966 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1967 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1968 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["version"] == "1");
															#line 1969 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["encoding"] == "utf8");
															#line 1970 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams.size() == 3);
															#line 1972 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1957Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1957ActorState(); static_cast<FlowTestCase1957Actor*>(this)->destroy(); return 0; }
															#line 11215 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1957Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1957ActorState();
			static_cast<FlowTestCase1957Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1957ActorState();
		static_cast<FlowTestCase1957Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11239 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1957()
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1957Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1957Actor>, public FlowTestCase1957ActorState<FlowTestCase1957Actor> {
															#line 11244 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1957Actor>::operator new;
	using FastAllocated<FlowTestCase1957Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5844456273791564544UL, 773830881379617280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1957Actor(UnitTestParameters const& params) 
															#line 11260 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1957ActorState<FlowTestCase1957Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1957", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17946766161398388992UL, 18303436092825378816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1957Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1957");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1957", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1957( UnitTestParameters const& params ) {
															#line 1957 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1957Actor(params));
															#line 11292 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1957, "/MockS3Server/parseS3Request/MultipleParameters")

#line 1974 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11298 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1975()
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1975Actor>
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1975ActorState {
															#line 11305 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1975ActorState(UnitTestParameters const& params) 
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11312 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1975", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1975ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1975", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1977 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1978 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?flag&region=us-east-1";
															#line 1979 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1980 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1982 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1984 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1985 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["flag"] == "");
															#line 1986 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1988 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1975Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1975ActorState(); static_cast<FlowTestCase1975Actor*>(this)->destroy(); return 0; }
															#line 11343 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1975Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1975ActorState();
			static_cast<FlowTestCase1975Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1975ActorState();
		static_cast<FlowTestCase1975Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11367 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1975()
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1975Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1975Actor>, public FlowTestCase1975ActorState<FlowTestCase1975Actor> {
															#line 11372 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1975Actor>::operator new;
	using FastAllocated<FlowTestCase1975Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2987729499320253184UL, 11651811026733924096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1975Actor(UnitTestParameters const& params) 
															#line 11388 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1975ActorState<FlowTestCase1975Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1975", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6703790926434346240UL, 3185348691346280704UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1975Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1975");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1975", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1975( UnitTestParameters const& params ) {
															#line 1975 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1975Actor(params));
															#line 11420 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1975, "/MockS3Server/parseS3Request/ParametersWithoutValues")

#line 1990 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11426 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1991()
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1991Actor>
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1991ActorState {
															#line 11433 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1991ActorState(UnitTestParameters const& params) 
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11440 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1991", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1991ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1991", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1992 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0-99";
															#line 1993 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 1995 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 1997 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 1998 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 0);
															#line 1999 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 99);
															#line 2001 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1991Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1991ActorState(); static_cast<FlowTestCase1991Actor*>(this)->destroy(); return 0; }
															#line 11467 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1991Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1991ActorState();
			static_cast<FlowTestCase1991Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1991ActorState();
		static_cast<FlowTestCase1991Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11491 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1991()
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1991Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1991Actor>, public FlowTestCase1991ActorState<FlowTestCase1991Actor> {
															#line 11496 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1991Actor>::operator new;
	using FastAllocated<FlowTestCase1991Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15031680863735786496UL, 17313618860186590464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1991Actor(UnitTestParameters const& params) 
															#line 11512 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1991ActorState<FlowTestCase1991Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1991", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3613356184874387200UL, 3378451978351106560UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1991Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1991");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1991", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1991( UnitTestParameters const& params ) {
															#line 1991 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1991Actor(params));
															#line 11544 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1991, "/MockS3Server/RangeHeader/SimpleByteRange")

#line 2003 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11550 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2004()
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2004Actor>
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2004ActorState {
															#line 11557 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2004ActorState(UnitTestParameters const& params) 
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11564 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2004", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2004ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2004", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2005 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=100-199";
															#line 2006 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2008 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2010 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2011 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 100);
															#line 2012 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 199);
															#line 2014 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2004Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2004ActorState(); static_cast<FlowTestCase2004Actor*>(this)->destroy(); return 0; }
															#line 11591 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2004Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2004ActorState();
			static_cast<FlowTestCase2004Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2004ActorState();
		static_cast<FlowTestCase2004Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11615 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2004()
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2004Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2004Actor>, public FlowTestCase2004ActorState<FlowTestCase2004Actor> {
															#line 11620 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2004Actor>::operator new;
	using FastAllocated<FlowTestCase2004Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17473284937188553216UL, 7808402722411376384UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2004Actor(UnitTestParameters const& params) 
															#line 11636 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2004ActorState<FlowTestCase2004Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2004", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(215920999306051328UL, 4539828599965722880UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2004Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2004");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2004", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2004( UnitTestParameters const& params ) {
															#line 2004 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2004Actor(params));
															#line 11668 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2004, "/MockS3Server/RangeHeader/MiddleRange")

#line 2016 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11674 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2017()
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2017Actor>
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2017ActorState {
															#line 11681 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2017ActorState(UnitTestParameters const& params) 
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11688 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2017", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2017ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2017", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2018 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=1000000-1999999";
															#line 2019 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2021 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2023 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2024 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 1000000);
															#line 2025 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 1999999);
															#line 2027 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2017Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2017ActorState(); static_cast<FlowTestCase2017Actor*>(this)->destroy(); return 0; }
															#line 11715 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2017Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2017ActorState();
			static_cast<FlowTestCase2017Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2017ActorState();
		static_cast<FlowTestCase2017Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11739 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2017()
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2017Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2017Actor>, public FlowTestCase2017ActorState<FlowTestCase2017Actor> {
															#line 11744 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2017Actor>::operator new;
	using FastAllocated<FlowTestCase2017Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5150219495233887488UL, 10942379155695991552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2017Actor(UnitTestParameters const& params) 
															#line 11760 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2017ActorState<FlowTestCase2017Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2017", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7829283054101161984UL, 9524968920067109632UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2017Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2017");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2017", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2017( UnitTestParameters const& params ) {
															#line 2017 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2017Actor(params));
															#line 11792 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2017, "/MockS3Server/RangeHeader/LargeOffsets")

#line 2029 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11798 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2030()
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2030Actor>
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2030ActorState {
															#line 11805 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2030ActorState(UnitTestParameters const& params) 
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11812 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2030", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2030ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2030", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2031 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "invalid-range";
															#line 2032 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2034 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2036 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2038 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2030Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2030ActorState(); static_cast<FlowTestCase2030Actor*>(this)->destroy(); return 0; }
															#line 11835 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2030Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2030ActorState();
			static_cast<FlowTestCase2030Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2030ActorState();
		static_cast<FlowTestCase2030Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11859 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2030()
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2030Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2030Actor>, public FlowTestCase2030ActorState<FlowTestCase2030Actor> {
															#line 11864 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2030Actor>::operator new;
	using FastAllocated<FlowTestCase2030Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3383621608479908352UL, 593713565960677120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2030Actor(UnitTestParameters const& params) 
															#line 11880 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2030ActorState<FlowTestCase2030Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2030", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4130846616586569984UL, 5934579303661222144UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2030Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2030");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2030", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2030( UnitTestParameters const& params ) {
															#line 2030 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2030Actor(params));
															#line 11912 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2030, "/MockS3Server/RangeHeader/InvalidFormat")

#line 2040 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11918 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2041()
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2041Actor>
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2041ActorState {
															#line 11925 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2041ActorState(UnitTestParameters const& params) 
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11932 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2041", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2041ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2041", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2042 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "0-99";
															#line 2043 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2045 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2047 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2049 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2041Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2041ActorState(); static_cast<FlowTestCase2041Actor*>(this)->destroy(); return 0; }
															#line 11955 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2041Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2041ActorState();
			static_cast<FlowTestCase2041Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2041ActorState();
		static_cast<FlowTestCase2041Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11979 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2041()
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2041Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2041Actor>, public FlowTestCase2041ActorState<FlowTestCase2041Actor> {
															#line 11984 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2041Actor>::operator new;
	using FastAllocated<FlowTestCase2041Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6310362826683999232UL, 1819248766335118592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2041Actor(UnitTestParameters const& params) 
															#line 12000 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2041ActorState<FlowTestCase2041Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2041", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17417462349685157120UL, 16743617727154502144UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2041Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2041");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2041", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2041( UnitTestParameters const& params ) {
															#line 2041 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2041Actor(params));
															#line 12032 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2041, "/MockS3Server/RangeHeader/MissingBytesPrefix")

#line 2051 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12038 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2052()
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2052Actor>
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2052ActorState {
															#line 12045 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2052ActorState(UnitTestParameters const& params) 
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2052", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2052ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2052", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2053 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0";
															#line 2054 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2056 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2058 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2060 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2052Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2052ActorState(); static_cast<FlowTestCase2052Actor*>(this)->destroy(); return 0; }
															#line 12075 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2052Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2052ActorState();
			static_cast<FlowTestCase2052Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2052ActorState();
		static_cast<FlowTestCase2052Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12099 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2052()
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2052Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2052Actor>, public FlowTestCase2052ActorState<FlowTestCase2052Actor> {
															#line 12104 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2052Actor>::operator new;
	using FastAllocated<FlowTestCase2052Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7662264358606175488UL, 15526653563918663168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2052Actor(UnitTestParameters const& params) 
															#line 12120 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2052ActorState<FlowTestCase2052Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2052", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7521430964007951104UL, 13810774539585666048UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2052Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2052");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2052", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2052( UnitTestParameters const& params ) {
															#line 2052 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2052Actor(params));
															#line 12152 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2052, "/MockS3Server/RangeHeader/MissingDash")

#line 2062 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12158 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2063()
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2063Actor>
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2063ActorState {
															#line 12165 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2063ActorState(UnitTestParameters const& params) 
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12172 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2063", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2063ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2063", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2064 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "";
															#line 2065 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2067 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2069 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2071 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2063Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2063ActorState(); static_cast<FlowTestCase2063Actor*>(this)->destroy(); return 0; }
															#line 12195 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2063Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2063ActorState();
			static_cast<FlowTestCase2063Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2063ActorState();
		static_cast<FlowTestCase2063Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12219 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2063()
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2063Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2063Actor>, public FlowTestCase2063ActorState<FlowTestCase2063Actor> {
															#line 12224 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2063Actor>::operator new;
	using FastAllocated<FlowTestCase2063Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12111848244824571392UL, 18421197757947364096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2063Actor(UnitTestParameters const& params) 
															#line 12240 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2063ActorState<FlowTestCase2063Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2063", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10024756261257064704UL, 6150053523691575552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2063Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2063");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2063", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2063( UnitTestParameters const& params ) {
															#line 2063 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2063Actor(params));
															#line 12272 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2063, "/MockS3Server/RangeHeader/EmptyString")

#line 2073 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12278 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2074()
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2074Actor>
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2074ActorState {
															#line 12285 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2074ActorState(UnitTestParameters const& params) 
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12292 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2074", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2074ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2074", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2075 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=-100-200";
															#line 2076 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2078 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2081 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2083 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2074Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2074ActorState(); static_cast<FlowTestCase2074Actor*>(this)->destroy(); return 0; }
															#line 12315 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2074Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2074ActorState();
			static_cast<FlowTestCase2074Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2074ActorState();
		static_cast<FlowTestCase2074Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12339 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2074()
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2074Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2074Actor>, public FlowTestCase2074ActorState<FlowTestCase2074Actor> {
															#line 12344 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2074Actor>::operator new;
	using FastAllocated<FlowTestCase2074Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11003674111181091072UL, 1226926955141888256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2074Actor(UnitTestParameters const& params) 
															#line 12360 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2074ActorState<FlowTestCase2074Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2074", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5313871499119712256UL, 4540502925954524160UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2074Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2074");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2074", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2074( UnitTestParameters const& params ) {
															#line 2074 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2074Actor(params));
															#line 12392 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2074, "/MockS3Server/RangeHeader/NegativeStart")

#line 2085 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12398 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2086()
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2086Actor>
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2086ActorState {
															#line 12405 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2086ActorState(UnitTestParameters const& params) 
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12412 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2086", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2086ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2086", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2087 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=200-100";
															#line 2088 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2090 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2093 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2094 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 200);
															#line 2095 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 100);
															#line 2097 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2086Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2086ActorState(); static_cast<FlowTestCase2086Actor*>(this)->destroy(); return 0; }
															#line 12439 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2086Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2086ActorState();
			static_cast<FlowTestCase2086Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2086ActorState();
		static_cast<FlowTestCase2086Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12463 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2086()
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2086Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2086Actor>, public FlowTestCase2086ActorState<FlowTestCase2086Actor> {
															#line 12468 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2086Actor>::operator new;
	using FastAllocated<FlowTestCase2086Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10706843918986801664UL, 104039308447221760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2086Actor(UnitTestParameters const& params) 
															#line 12484 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2086ActorState<FlowTestCase2086Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2086", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5120362537455785216UL, 6067354855254433024UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2086Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2086");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2086", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2086( UnitTestParameters const& params ) {
															#line 2086 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2086Actor(params));
															#line 12516 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2086, "/MockS3Server/RangeHeader/StartGreaterThanEnd")

#line 2099 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Real HTTP Server Implementation for ctests
															#line 12523 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerReal_implActor>
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActorState {
															#line 12530 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActorState(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress),
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(persistenceDir)
															#line 12539 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerReal_implActorState() 
	{
		fdb_probe_actor_destroy("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2102 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerRealStarting").detail("ListenAddress", listenAddress.toString());
															#line 2105 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 12556 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 2107 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (persistenceDir.empty())
															#line 12560 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 2108 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					persistenceDir = "simfdb/mocks3";
															#line 12564 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				}
															#line 2110 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(persistenceDir);
															#line 2111 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRealPersistenceEnabled") .detail("ListenAddress", listenAddress.toString()) .detail("PersistenceDir", persistenceDir);
															#line 2116 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 2116 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12574 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 1;
															#line 2116 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12579 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2119 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server = makeReference<HTTP::SimServerContext>();
															#line 2120 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server->registerNewServer(listenAddress, makeReference<MockS3RequestHandler>());
															#line 2122 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRealStarted") .detail("ListenAddress", listenAddress.toString()) .detail("ServerPtr", format("%p", server.getPtr()));
															#line 2127 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = Never();
															#line 2127 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12615 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 2;
															#line 2127 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12620 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2128 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12716 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2128 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12728 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 2119 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::SimServerContext> server;
															#line 12817 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerReal_implActor, 0, Void >, public ActorCallback< StartMockS3ServerReal_implActor, 1, Void >, public FastAllocated<StartMockS3ServerReal_implActor>, public StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor> {
															#line 12822 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerReal_implActor>::operator new;
	using FastAllocated<StartMockS3ServerReal_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4377143713612343808UL, 6087085591745396992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerReal_implActor, 0, Void >;
friend struct ActorCallback< StartMockS3ServerReal_implActor, 1, Void >;
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActor(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 12840 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor>(listenAddress, persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15792334911365886208UL, 3052106341508935936UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3ServerReal_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3ServerReal_impl( NetworkAddress const& listenAddress, std::string const& persistenceDir ) {
															#line 2101 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerReal_implActor(listenAddress, persistenceDir));
															#line 12874 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 2130 "/codebuild/output/src1514243894/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

Future<Void> startMockS3ServerReal(const NetworkAddress& listenAddress, const std::string& persistenceDir) {
	return startMockS3ServerReal_impl(listenAddress, persistenceDir);
}

// Wrapper for registerMockS3Server (calls the ACTOR implementation)
Future<Void> registerMockS3Server(std::string ip, std::string port) {
	return registerMockS3Server_impl(ip, port);
}
