#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
/*
 * FuzzApiCorrectness.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <limits.h>
#include <mutex>
#include <functional>
#include <sstream>

#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbclient/GenericManagementAPI.actor.h"
#include "fdbclient/TenantManagement.actor.h"
#include "fdbclient/ThreadSafeTransaction.h"
#include "flow/ActorCollection.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "flow/Arena.h"
#include "flow/actorcompiler.h" // This must be the last #include.

namespace ph = std::placeholders;

// This allows us to dictate which exceptions we SHOULD get.
// We can use this to suppress expected exceptions, and take action
// if we don't get an exception wqe should have gotten.
struct ExceptionContract {
	enum occurance_t { Never = 0, Possible = 1, Always = 2 };

	std::string func;
	std::map<int, occurance_t> expected;
	std::function<void(TraceEvent&)> augment;

	ExceptionContract(const char* func_, const std::function<void(TraceEvent&)>& augment_)
	  : func(func_), augment(augment_) {}
	ExceptionContract& operator=(const std::map<int, occurance_t>& e) {
		expected = e;
		return *this;
	}

	static occurance_t possibleButRequiredIf(bool in) { return in ? Always : Possible; }
	static occurance_t requiredIf(bool in) { return in ? Always : Never; }
	static occurance_t possibleIf(bool in) { return in ? Possible : Never; }

	void handleException(const Error& e, Reference<ITransaction> tr) const {
		// We should always ignore these.
		if (e.code() == error_code_used_during_commit || e.code() == error_code_transaction_too_old ||
		    e.code() == error_code_future_version || e.code() == error_code_transaction_cancelled ||
		    e.code() == error_code_key_too_large || e.code() == error_code_value_too_large ||
		    e.code() == error_code_process_behind || e.code() == error_code_batch_transaction_throttled ||
		    e.code() == error_code_tag_throttled || e.code() == error_code_grv_proxy_memory_limit_exceeded) {
			return;
		}

		auto i = expected.find(e.code());
		if (i != expected.end() && i->second != Never) {
			Severity s = (i->second == Possible) ? SevWarn : SevInfo;
			TraceEvent evt(s, func.c_str());
			evt.error(e)
			    .detail("Thrown", true)
			    .detail("Expected", i->second == Possible ? "possible" : "always")
			    .detail("Tenant", tr->getTenant())
			    .backtrace();
			if (augment)
				augment(evt);
			return;
		}

		TraceEvent evt(SevError, func.c_str());
		evt.error(e).detail("Thrown", true).detail("Expected", "never").detail("Tenant", tr->getTenant()).backtrace();
		if (augment)
			augment(evt);
		throw e;
	}

	// Return true if we should have thrown, but didn't.
	void handleNotThrown(Reference<ITransaction> tr) const {
		for (auto i : expected) {
			if (i.second == Always) {
				TraceEvent evt(SevError, func.c_str());
				evt.error(Error::fromUnvalidatedCode(i.first))
				    .detail("Thrown", false)
				    .detail("Expected", "always")
				    .detail("Tenant", tr->getTenant())
				    .backtrace();
				if (augment)
					augment(evt);
			}
		}
	}
};

struct FuzzApiCorrectnessWorkload : TestWorkload {
	static constexpr auto NAME = "FuzzApiCorrectness";
	static std::once_flag onceFlag;
	static std::vector<std::function<
	    Future<Void>(unsigned int const&, FuzzApiCorrectnessWorkload* const&, Reference<ITransaction> const&)>>
	    testCases;

	double testDuration;
	int numOps;
	bool rarelyCommit, adjacentKeys;
	int minNode, nodes;
	std::pair<int, int> valueSizeRange;
	int maxClearSize;
	double initialKeyDensity;
	bool useSystemKeys;
	bool writeSystemKeys = false; // whether we really write to a system key in the workload
	KeyRange conflictRange;
	unsigned int operationId;
	int64_t maximumTotalData;
	bool specialKeysRelaxed;
	bool specialKeysWritesEnabled;

	bool success;

	Reference<IDatabase> db;

	std::vector<Reference<ITenant>> tenants;
	std::set<TenantName> createdTenants;
	int numTenants;
	int numTenantGroups;
	int minTenantNum = -1;

	bool illegalTenantAccess = false;

	// Map from tenant number to key prefix
	std::map<int, std::string> keyPrefixes;

	FuzzApiCorrectnessWorkload(WorkloadContext const& wcx) : TestWorkload(wcx), operationId(0), success(true) {
		std::call_once(onceFlag, [&]() { addTestCases(); });

		testDuration = getOption(options, "testDuration"_sr, 60.0);
		numOps = getOption(options, "numOps"_sr, 21);
		rarelyCommit = getOption(options, "rarelyCommit"_sr, false);
		maximumTotalData = getOption(options, "maximumTotalData"_sr, 15e6);
		minNode = getOption(options, "minNode"_sr, 0);
		adjacentKeys = deterministicRandom()->coinflip();
		useSystemKeys = deterministicRandom()->coinflip();
		initialKeyDensity = deterministicRandom()->random01(); // This fraction of keys are present before the first
		                                                       // transaction (and after an unknown result)
		specialKeysRelaxed = deterministicRandom()->coinflip();
		// Only enable special keys writes when allowed to access system keys
		specialKeysWritesEnabled = useSystemKeys && deterministicRandom()->coinflip();

		int maxTenants = getOption(options, "numTenants"_sr, 4);
		numTenants = deterministicRandom()->randomInt(0, maxTenants + 1);

		int maxTenantGroups = getOption(options, "numTenantGroups"_sr, numTenants);
		numTenantGroups = deterministicRandom()->randomInt(0, maxTenantGroups + 1);

		// See https://github.com/apple/foundationdb/issues/2424
		if (BUGGIFY) {
			enableClientBuggify();
		}

		if (adjacentKeys) {
			nodes = std::min<int64_t>(deterministicRandom()->randomInt(1, 4 << deterministicRandom()->randomInt(0, 14)),
			                          CLIENT_KNOBS->KEY_SIZE_LIMIT * 1.2);
		} else {
			nodes = deterministicRandom()->randomInt(1, 4 << deterministicRandom()->randomInt(0, 20));
		}

		int newNodes =
		    std::min<int>(nodes, maximumTotalData / (getKeyForIndex(-1, nodes).size() + valueSizeRange.second));
		minNode = std::max(minNode, nodes - newNodes);
		nodes = newNodes;

		if (useSystemKeys && deterministicRandom()->coinflip()) {
			keyPrefixes[-1] = "\xff\x01";
			writeSystemKeys = true;
		}

		maxClearSize = 1 << deterministicRandom()->randomInt(0, 20);
		conflictRange = KeyRangeRef("\xfe"_sr, "\xfe\x00"_sr);
		TraceEvent("FuzzApiCorrectnessConfiguration")
		    .detail("Nodes", nodes)
		    .detail("NumTenants", numTenants)
		    .detail("InitialKeyDensity", initialKeyDensity)
		    .detail("AdjacentKeys", adjacentKeys)
		    .detail("ValueSizeMin", valueSizeRange.first)
		    .detail("ValueSizeRange", valueSizeRange.second)
		    .detail("MaxClearSize", maxClearSize)
		    .detail("UseSystemKeys", useSystemKeys)
		    .detail("SpecialKeysRelaxed", specialKeysRelaxed)
		    .detail("SpecialKeysWritesEnabled", specialKeysWritesEnabled);

		TraceEvent("RemapEventSeverity")
		    .detail("TargetEvent", "LargePacketSent")
		    .detail("OriginalSeverity", SevWarnAlways)
		    .detail("NewSeverity", SevInfo);
		TraceEvent("RemapEventSeverity")
		    .detail("TargetEvent", "LargePacketReceived")
		    .detail("OriginalSeverity", SevWarnAlways)
		    .detail("NewSeverity", SevInfo);
		TraceEvent("RemapEventSeverity")
		    .detail("TargetEvent", "LargeTransaction")
		    .detail("OriginalSeverity", SevWarnAlways)
		    .detail("NewSeverity", SevInfo);
	}

	static TenantName getTenant(int num) { return TenantNameRef(format("tenant_%d", num)); }
	Optional<TenantGroupName> getTenantGroup(int num) {
		int groupNum = num % (numTenantGroups + 1);
		if (groupNum == numTenantGroups - 1) {
			return Optional<TenantGroupName>();
		} else {
			return TenantGroupNameRef(format("tenantgroup_%d", groupNum));
		}
	}
	bool canUseTenant(Optional<TenantName> tenant) {
		return !tenant.present() || createdTenants.contains(tenant.get());
	}

	Future<Void> setup(Database const& cx) override {
		if (clientId == 0) {
			return _setup(cx, this);
		}

		return Void();
	}

																#line 242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via _setup()
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class _setupActor>
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class _setupActorState {
															#line 248 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	_setupActorState(Database const& cx,FuzzApiCorrectnessWorkload* const& self) 
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : cx(cx),
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   self(self)
															#line 257 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 241 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Reference<IDatabase>> __when_expr_0 = unsafeThreadFutureToFuture(ThreadSafeDatabase::createFromExistingDatabase(cx));
															#line 241 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 274 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 241 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Reference<IDatabase> >*>(static_cast<_setupActor*>(this)));
															#line 279 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IDatabase> const& db,int loopDepth) 
	{
															#line 242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		self->db = db;
															#line 244 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		std::vector<Future<Void>> tenantFutures;
															#line 246 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		for(int i = 0;i < self->numTenants;++i) {
															#line 247 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantName tenantName = getTenant(i);
															#line 248 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantMapEntry entry;
															#line 249 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			entry.tenantGroup = self->getTenantGroup(i);
															#line 250 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tenantFutures.push_back(::success(TenantAPI::createTenant(cx.getReference(), tenantName, entry)));
															#line 251 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->createdTenants.insert(tenantName);
															#line 316 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(tenantFutures);
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 322 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, Void >*>(static_cast<_setupActor*>(this)));
															#line 327 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IDatabase> && db,int loopDepth) 
	{
															#line 242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		self->db = db;
															#line 244 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		std::vector<Future<Void>> tenantFutures;
															#line 246 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		for(int i = 0;i < self->numTenants;++i) {
															#line 247 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantName tenantName = getTenant(i);
															#line 248 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantMapEntry entry;
															#line 249 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			entry.tenantGroup = self->getTenantGroup(i);
															#line 250 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tenantFutures.push_back(::success(TenantAPI::createTenant(cx.getReference(), tenantName, entry)));
															#line 251 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->createdTenants.insert(tenantName);
															#line 350 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(tenantFutures);
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 356 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, Void >*>(static_cast<_setupActor*>(this)));
															#line 361 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IDatabase> const& db,int loopDepth) 
	{
		loopDepth = a_body1cont1(db, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IDatabase> && db,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(db), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Reference<IDatabase> >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Reference<IDatabase> >*,Reference<IDatabase> const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Reference<IDatabase> >*,Reference<IDatabase> && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Reference<IDatabase> >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 256 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		for(int i = 0;i < self->numTenants + 1;++i) {
															#line 257 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantName tenantName = getTenant(i);
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->tenants.push_back(self->db->openTenant(tenantName));
															#line 449 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<DatabaseConfiguration> __when_expr_2 = getDatabaseConfiguration(cx);
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 455 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 3;
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 2, DatabaseConfiguration >*>(static_cast<_setupActor*>(this)));
															#line 460 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 256 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		for(int i = 0;i < self->numTenants + 1;++i) {
															#line 257 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TenantName tenantName = getTenant(i);
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->tenants.push_back(self->db->openTenant(tenantName));
															#line 473 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<DatabaseConfiguration> __when_expr_2 = getDatabaseConfiguration(cx);
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 479 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 3;
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 2, DatabaseConfiguration >*>(static_cast<_setupActor*>(this)));
															#line 484 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _setupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(DatabaseConfiguration const& config,int loopDepth) 
	{
															#line 264 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (config.encryptionAtRestMode.mode == EncryptionAtRestMode::DOMAIN_AWARE)
															#line 568 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 265 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->minTenantNum = 0;
															#line 572 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 268 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 576 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(DatabaseConfiguration && config,int loopDepth) 
	{
															#line 264 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (config.encryptionAtRestMode.mode == EncryptionAtRestMode::DOMAIN_AWARE)
															#line 588 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 265 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->minTenantNum = 0;
															#line 592 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 268 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 596 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(DatabaseConfiguration const& config,int loopDepth) 
	{
		loopDepth = a_body1cont4(config, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(DatabaseConfiguration && config,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(config), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 2, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _setupActor, 2, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Database cx;
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	FuzzApiCorrectnessWorkload* self;
															#line 683 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via _setup()
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Reference<IDatabase> >, public ActorCallback< _setupActor, 1, Void >, public ActorCallback< _setupActor, 2, DatabaseConfiguration >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 688 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3661502387280480768UL, 15241216677023234816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Reference<IDatabase> >;
friend struct ActorCallback< _setupActor, 1, Void >;
friend struct ActorCallback< _setupActor, 2, DatabaseConfiguration >;
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	_setupActor(Database const& cx,FuzzApiCorrectnessWorkload* const& self) 
															#line 707 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Reference<IDatabase> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _setupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _setupActor, 2, DatabaseConfiguration >*)0, actor_cancelled()); break;
		}

	}
};
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] Future<Void> _setup( Database const& cx, FuzzApiCorrectnessWorkload* const& self ) {
															#line 240 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<Void>(new _setupActor(cx, self));
															#line 741 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

	Future<Void> start(Database const& cx) override {
		if (clientId == 0) {
			return loadAndRun(this, cx);
		}
		return Void();
	}

	Future<bool> check(Database const& cx) override {
		if (!writeSystemKeys) { // there must be illegal access during data load
			return illegalTenantAccess;
		}
		return success;
	}

	Key getKeyForIndex(int tenantNum, int idx) {
		idx += minNode;
		if (adjacentKeys) {
			return Key(keyPrefixes[tenantNum] + std::string(idx, '\x00'));
		} else {
			return Key(keyPrefixes[tenantNum] + format("%010d", idx));
		}
	}

	KeyRef getMaxKey(Reference<ITransaction> tr) const {
		if (useSystemKeys && !tr->getTenant().present()) {
			return systemKeys.end;
		} else {
			return normalKeys.end;
		}
	}

	Value getRandomValue() const {
		return Value(
		    std::string(deterministicRandom()->randomInt(valueSizeRange.first, valueSizeRange.second + 1), 'x'));
	}

	void getMetrics(std::vector<PerfMetric>& m) override {
		// m.push_back( transactions.getMetric() );
		// m.push_back( retries.getMetric() );
	}

	// Prevent a write only transaction whose commit was previously cancelled from being reordered after this
	// transaction
																#line 789 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via writeBarrier()
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class WriteBarrierActor>
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class WriteBarrierActorState {
															#line 795 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	WriteBarrierActorState(Reference<IDatabase> const& db) 
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : db(db),
															#line 315 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   tr(db->createTransaction())
															#line 804 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("writeBarrier", reinterpret_cast<unsigned long>(this));

	}
	~WriteBarrierActorState() 
	{
		fdb_probe_actor_destroy("writeBarrier", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 316 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			;
															#line 819 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteBarrierActorState();
		static_cast<WriteBarrierActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 318 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 321 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->addWriteConflictRange(allKeys);
															#line 322 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->clear(normalKeys);
															#line 323 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_0 = unsafeThreadFutureToFuture(tr->commit());
															#line 323 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<WriteBarrierActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 858 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteBarrierActor*>(this)->actor_wait_state = 1;
															#line 323 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteBarrierActor, 0, Void >*>(static_cast<WriteBarrierActor*>(this)));
															#line 863 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 326 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_1 = unsafeThreadFutureToFuture(tr->onError(e));
															#line 326 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<WriteBarrierActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 887 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WriteBarrierActor*>(this)->actor_wait_state = 2;
															#line 326 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteBarrierActor, 1, Void >*>(static_cast<WriteBarrierActor*>(this)));
															#line 892 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 324 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<WriteBarrierActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteBarrierActorState(); static_cast<WriteBarrierActor*>(this)->destroy(); return 0; }
															#line 907 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<WriteBarrierActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteBarrierActorState();
		static_cast<WriteBarrierActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 324 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<WriteBarrierActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteBarrierActorState(); static_cast<WriteBarrierActor*>(this)->destroy(); return 0; }
															#line 919 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<WriteBarrierActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteBarrierActorState();
		static_cast<WriteBarrierActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteBarrierActor*>(this)->actor_wait_state > 0) static_cast<WriteBarrierActor*>(this)->actor_wait_state = 0;
		static_cast<WriteBarrierActor*>(this)->ActorCallback< WriteBarrierActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteBarrierActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteBarrierActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteBarrierActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteBarrierActor*>(this)->actor_wait_state > 0) static_cast<WriteBarrierActor*>(this)->actor_wait_state = 0;
		static_cast<WriteBarrierActor*>(this)->ActorCallback< WriteBarrierActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteBarrierActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteBarrierActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteBarrierActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<IDatabase> db;
															#line 315 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<ITransaction> tr;
															#line 1093 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via writeBarrier()
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class WriteBarrierActor final : public Actor<Void>, public ActorCallback< WriteBarrierActor, 0, Void >, public ActorCallback< WriteBarrierActor, 1, Void >, public FastAllocated<WriteBarrierActor>, public WriteBarrierActorState<WriteBarrierActor> {
															#line 1098 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<WriteBarrierActor>::operator new;
	using FastAllocated<WriteBarrierActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10974531662980778752UL, 8915444039998981632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteBarrierActor, 0, Void >;
friend struct ActorCallback< WriteBarrierActor, 1, Void >;
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	WriteBarrierActor(Reference<IDatabase> const& db) 
															#line 1116 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   WriteBarrierActorState<WriteBarrierActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeBarrier", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12408030041787207680UL, 1824477430591930112UL);
		ActorExecutionContextHelper __helper(static_cast<WriteBarrierActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeBarrier");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeBarrier", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteBarrierActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteBarrierActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] Future<Void> writeBarrier( Reference<IDatabase> const& db ) {
															#line 314 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<Void>(new WriteBarrierActor(db));
															#line 1149 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 330 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

																#line 1154 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via loadAndRun()
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class LoadAndRunActor>
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class LoadAndRunActorState {
															#line 1160 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	LoadAndRunActorState(FuzzApiCorrectnessWorkload* const& self,Database const& cx) 
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : self(self),
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   cx(cx),
															#line 332 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   startTime(now()),
															#line 333 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   nodesPerTenant(std::max<int>(1, self->nodes / (self->numTenants + 1))),
															#line 334 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   keysPerBatch(std::min<int64_t>(1000, 1 + CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT / 2 / (self->getKeyForIndex(-1, nodesPerTenant).size() + self->valueSizeRange.second)))
															#line 1175 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("loadAndRun", reinterpret_cast<unsigned long>(this));

	}
	~LoadAndRunActorState() 
	{
		fdb_probe_actor_destroy("loadAndRun", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				;
															#line 1191 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadAndRunActorState();
		static_cast<LoadAndRunActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 413 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TraceEvent("FuzzLoadAndRunError").error(e).backtrace();
															#line 414 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1223 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 340 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		tenantNum = self->minTenantNum;
															#line 341 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 1246 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 400 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 1255 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 341 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!(tenantNum < self->numTenants))
															#line 1271 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 342 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		i = 0;
															#line 343 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->writeBarrier(self->db);
															#line 343 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<LoadAndRunActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1281 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<LoadAndRunActor*>(this)->actor_wait_state = 1;
															#line 343 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadAndRunActor, 0, Void >*>(static_cast<LoadAndRunActor*>(this)));
															#line 1286 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 1308 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 1317 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadAndRunActor*>(this)->actor_wait_state > 0) static_cast<LoadAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<LoadAndRunActor*>(this)->ActorCallback< LoadAndRunActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadAndRunActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 341 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		++tenantNum;
															#line 1401 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!(i < nodesPerTenant))
															#line 1417 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1loopBody1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 345 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		tr = tenantNum < 0 ? self->db->createTransaction() : self->tenants[tenantNum]->createTransaction();
															#line 348 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 1425 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 393 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (self->illegalTenantAccess)
															#line 1447 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1loopBody1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		i += keysPerBatch;
															#line 1453 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1(int loopDepth) 
	{
															#line 349 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (now() - startTime > self->testDuration)
															#line 1469 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 350 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (!static_cast<LoadAndRunActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadAndRunActorState(); static_cast<LoadAndRunActor*>(this)->destroy(); return 0; }
															#line 1473 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			new (&static_cast<LoadAndRunActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~LoadAndRunActorState();
			static_cast<LoadAndRunActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		try {
															#line 352 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (self->useSystemKeys && tenantNum == -1)
															#line 1482 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 353 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 355 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (self->specialKeysRelaxed)
															#line 1490 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 356 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_RELAXED);
															#line 1494 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 357 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (self->specialKeysWritesEnabled)
															#line 1498 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 358 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1502 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 360 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (i == 0)
															#line 1506 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 361 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr->clear(normalKeys);
															#line 1510 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 364 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			int end = std::min(nodesPerTenant, i + keysPerBatch);
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->clear(KeyRangeRef(self->getKeyForIndex(tenantNum, i), self->getKeyForIndex(tenantNum, end)));
															#line 368 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			for(int j = i;j < end;j++) {
															#line 369 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				if (deterministicRandom()->random01() < self->initialKeyDensity)
															#line 1520 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				{
															#line 370 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
					Key key = self->getKeyForIndex(tenantNum, j);
															#line 371 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
					if (key.size() <= getMaxWriteKeySize(key, false))
															#line 1526 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
					{
															#line 372 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
						Value value = self->getRandomValue();
															#line 373 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
						value = value.substr( 0, std::min<int>(value.size(), CLIENT_KNOBS->VALUE_SIZE_LIMIT));
															#line 375 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
						tr->set(key, value);
															#line 1534 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
					}
				}
			}
															#line 379 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_1 = unsafeThreadFutureToFuture(tr->commit());
															#line 379 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<LoadAndRunActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont1loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<LoadAndRunActor*>(this)->actor_wait_state = 2;
															#line 379 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadAndRunActor, 1, Void >*>(static_cast<LoadAndRunActor*>(this)));
															#line 1547 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1loopBody1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 3));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 3));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopBody1cont1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 383 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (e.code() == error_code_illegal_tenant_access)
															#line 1582 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 384 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				ASSERT(!self->writeSystemKeys);
															#line 385 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				ASSERT_EQ(tenantNum, -1);
															#line 386 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				self->illegalTenantAccess = true;
															#line 1590 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				return a_body1loopBody1loopBody1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 389 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_2 = unsafeThreadFutureToFuture(tr->onError(e));
															#line 389 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<LoadAndRunActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 4));
															#line 1597 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 4)); else return a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<LoadAndRunActor*>(this)->actor_wait_state = 3;
															#line 389 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadAndRunActor, 2, Void >*>(static_cast<LoadAndRunActor*>(this)));
															#line 1602 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 4));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 4));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1loopBody1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
		return a_body1loopBody1loopBody1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadAndRunActor*>(this)->actor_wait_state > 0) static_cast<LoadAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<LoadAndRunActor*>(this)->ActorCallback< LoadAndRunActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadAndRunActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadAndRunActor*>(this)->actor_wait_state > 0) static_cast<LoadAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<LoadAndRunActor*>(this)->ActorCallback< LoadAndRunActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadAndRunActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 402 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_3 = self->randomTransaction(self, cx) && delay(self->numOps * .001);
															#line 402 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<LoadAndRunActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1807 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont1loopBody1when1(__when_expr_3.get(), loopDepth); };
			static_cast<LoadAndRunActor*>(this)->actor_wait_state = 4;
															#line 402 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< LoadAndRunActor, 3, Void >*>(static_cast<LoadAndRunActor*>(this)));
															#line 1812 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 408 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (now() - startTime > self->testDuration)
															#line 1840 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 409 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (!static_cast<LoadAndRunActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadAndRunActorState(); static_cast<LoadAndRunActor*>(this)->destroy(); return 0; }
															#line 1844 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			new (&static_cast<LoadAndRunActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~LoadAndRunActorState();
			static_cast<LoadAndRunActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 404 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (e.code() != error_code_not_committed)
															#line 1859 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 405 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				return a_body1Catch2(e, std::max(0, loopDepth - 2));
															#line 1863 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<LoadAndRunActor*>(this)->actor_wait_state > 0) static_cast<LoadAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<LoadAndRunActor*>(this)->ActorCallback< LoadAndRunActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< LoadAndRunActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< LoadAndRunActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	FuzzApiCorrectnessWorkload* self;
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Database cx;
															#line 332 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	double startTime;
															#line 333 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int nodesPerTenant;
															#line 334 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int keysPerBatch;
															#line 340 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int tenantNum;
															#line 342 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int i;
															#line 345 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<ITransaction> tr;
															#line 1991 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via loadAndRun()
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class LoadAndRunActor final : public Actor<Void>, public ActorCallback< LoadAndRunActor, 0, Void >, public ActorCallback< LoadAndRunActor, 1, Void >, public ActorCallback< LoadAndRunActor, 2, Void >, public ActorCallback< LoadAndRunActor, 3, Void >, public FastAllocated<LoadAndRunActor>, public LoadAndRunActorState<LoadAndRunActor> {
															#line 1996 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<LoadAndRunActor>::operator new;
	using FastAllocated<LoadAndRunActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1577889896758890240UL, 11807289741972219136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadAndRunActor, 0, Void >;
friend struct ActorCallback< LoadAndRunActor, 1, Void >;
friend struct ActorCallback< LoadAndRunActor, 2, Void >;
friend struct ActorCallback< LoadAndRunActor, 3, Void >;
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	LoadAndRunActor(FuzzApiCorrectnessWorkload* const& self,Database const& cx) 
															#line 2016 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   LoadAndRunActorState<LoadAndRunActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadAndRun", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1948779402819391744UL, 14849708037289523712UL);
		ActorExecutionContextHelper __helper(static_cast<LoadAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadAndRun");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadAndRun", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadAndRunActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadAndRunActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadAndRunActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< LoadAndRunActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] Future<Void> loadAndRun( FuzzApiCorrectnessWorkload* const& self, Database const& cx ) {
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<Void>(new LoadAndRunActor(self, cx));
															#line 2051 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 417 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

																#line 2056 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via randomTransaction()
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class RandomTransactionActor>
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RandomTransactionActorState {
															#line 2062 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RandomTransactionActorState(FuzzApiCorrectnessWorkload* const& self,Database const& cx) 
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : self(self),
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   cx(cx),
															#line 419 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   tr(),
															#line 420 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   readYourWritesDisabled(deterministicRandom()->coinflip()),
															#line 421 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   readAheadDisabled(deterministicRandom()->coinflip()),
															#line 422 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   operations(),
															#line 423 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   waitLocation(0),
															#line 425 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   tenantNum(deterministicRandom()->randomInt(self->minTenantNum, self->tenants.size()))
															#line 2083 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("randomTransaction", reinterpret_cast<unsigned long>(this));

	}
	~RandomTransactionActorState() 
	{
		fdb_probe_actor_destroy("randomTransaction", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 426 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (tenantNum == -1)
															#line 2098 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 427 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr = self->db->createTransaction();
															#line 2102 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
			else
			{
															#line 429 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				tr = self->tenants[tenantNum]->createTransaction();
															#line 2108 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 432 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			rawAccess = tenantNum == -1 && deterministicRandom()->coinflip();
															#line 434 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			;
															#line 2114 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RandomTransactionActorState();
		static_cast<RandomTransactionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 516 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RandomTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RandomTransactionActorState(); static_cast<RandomTransactionActor*>(this)->destroy(); return 0; }
															#line 2137 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RandomTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RandomTransactionActorState();
		static_cast<RandomTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 435 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		cancelled = false;
															#line 436 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (readYourWritesDisabled)
															#line 2158 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 437 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_YOUR_WRITES_DISABLE);
															#line 2162 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 438 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (readAheadDisabled)
															#line 2166 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 439 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_AHEAD_DISABLE);
															#line 2170 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 440 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (self->useSystemKeys && tenantNum == -1)
															#line 2174 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 441 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2178 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 443 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (self->specialKeysRelaxed)
															#line 2182 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 444 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_RELAXED);
															#line 2186 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 446 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (self->specialKeysWritesEnabled)
															#line 2190 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 447 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 2194 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 449 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (rawAccess)
															#line 2198 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 450 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			tr->setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 2202 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 452 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		tr->addWriteConflictRange(self->conflictRange);
															#line 2206 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		try {
															#line 455 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			numWaits = deterministicRandom()->randomInt(1, 5);
															#line 456 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			i = 0;
															#line 460 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			;
															#line 2214 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 495 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			operations.clear();
															#line 496 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			waitLocation = 0;
															#line 498 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (e.code() == error_code_not_committed || e.code() == error_code_commit_unknown_result || cancelled)
															#line 2253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 499 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				return a_body1Catch1(not_committed(), std::max(0, loopDepth - 1));
															#line 2257 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
			else
			{
															#line 500 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				if (e.code() == error_code_illegal_tenant_access)
															#line 2263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				{
															#line 501 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
					ASSERT_EQ(tenantNum, -1);
															#line 502 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
					ASSERT_EQ(cx->getTenantMode(), TenantMode::REQUIRED);
															#line 503 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
					if (!static_cast<RandomTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RandomTransactionActorState(); static_cast<RandomTransactionActor*>(this)->destroy(); return 0; }
															#line 2271 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
					new (&static_cast<RandomTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~RandomTransactionActorState();
					static_cast<RandomTransactionActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
			try {
															#line 507 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_4 = unsafeThreadFutureToFuture(tr->onError(e));
															#line 507 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1Catch1(actor_cancelled(), loopDepth);
															#line 2283 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<RandomTransactionActor*>(this)->actor_wait_state = 5;
															#line 507 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 4, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2288 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1Catch1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1Catch1Catch1(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
															#line 484 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(operations);
															#line 484 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2311 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_2.get(), loopDepth); };
		static_cast<RandomTransactionActor*>(this)->actor_wait_state = 3;
															#line 484 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 2, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2316 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 460 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!(i < numWaits))
															#line 2332 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 461 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		numOps = deterministicRandom()->randomInt(1, self->numOps);
															#line 462 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		j = 0;
															#line 463 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		;
															#line 2342 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 477 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (waitLocation < operations.size())
															#line 2364 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 478 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			int waitOp = deterministicRandom()->randomInt(waitLocation, operations.size());
															#line 479 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_0 = operations[waitOp];
															#line 479 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2372 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RandomTransactionActor*>(this)->actor_wait_state = 1;
															#line 479 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 0, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2377 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1(int loopDepth) 
	{
															#line 463 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!(j < numOps))
															#line 2398 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 464 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		operationType = deterministicRandom()->randomInt(0, testCases.size());
															#line 465 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		printf("%d: Selected Operation %d\n", self->operationId + 1, operationType);
															#line 2406 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		try {
															#line 467 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			operations.push_back(testCases[operationType](++self->operationId, self, tr));
															#line 2410 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopBody1loopBody1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 463 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		j++;
															#line 2438 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 469 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			TraceEvent(SevWarn, "IgnoredOperation") .error(e) .detail("Operation", operationType) .detail("Id", self->operationId);
															#line 2448 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopBody1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 460 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		i++;
															#line 2476 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0.000001);
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2487 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<RandomTransactionActor*>(this)->actor_wait_state = 2;
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 1, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2492 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0.000001);
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2503 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<RandomTransactionActor*>(this)->actor_wait_state = 2;
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 1, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2508 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state > 0) static_cast<RandomTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RandomTransactionActor*>(this)->ActorCallback< RandomTransactionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RandomTransactionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 481 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		waitLocation = operations.size();
															#line 2592 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 481 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		waitLocation = operations.size();
															#line 2601 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state > 0) static_cast<RandomTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RandomTransactionActor*>(this)->ActorCallback< RandomTransactionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RandomTransactionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		try {
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_3 = timeoutError(unsafeThreadFutureToFuture(tr->commit()), 30);
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont9Catch1(actor_cancelled(), loopDepth);
															#line 2688 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont9Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
			static_cast<RandomTransactionActor*>(this)->actor_wait_state = 4;
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 3, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2693 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont9Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont9Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		try {
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_3 = timeoutError(unsafeThreadFutureToFuture(tr->commit()), 30);
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (static_cast<RandomTransactionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont9Catch1(actor_cancelled(), loopDepth);
															#line 2711 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont9Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
			static_cast<RandomTransactionActor*>(this)->actor_wait_state = 4;
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RandomTransactionActor, 3, Void >*>(static_cast<RandomTransactionActor*>(this)));
															#line 2716 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont9Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont9Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state > 0) static_cast<RandomTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RandomTransactionActor*>(this)->ActorCallback< RandomTransactionActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RandomTransactionActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont9Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 488 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (e.code() == error_code_client_invalid_operation || e.code() == error_code_transaction_too_large || e.code() == error_code_invalid_option)
															#line 2813 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 490 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				return a_body1loopBody1Catch1(not_committed(), loopDepth);
															#line 2817 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state > 0) static_cast<RandomTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RandomTransactionActor*>(this)->ActorCallback< RandomTransactionActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont9Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont9Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RandomTransactionActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont9Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 509 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			if (e.code() == error_code_transaction_timed_out)
															#line 2940 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			{
															#line 510 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				return a_body1Catch1(not_committed(), std::max(0, loopDepth - 1));
															#line 2944 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			}
															#line 512 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 2948 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<RandomTransactionActor*>(this)->actor_wait_state > 0) static_cast<RandomTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RandomTransactionActor*>(this)->ActorCallback< RandomTransactionActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< RandomTransactionActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< RandomTransactionActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1Catch1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	FuzzApiCorrectnessWorkload* self;
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Database cx;
															#line 419 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<ITransaction> tr;
															#line 420 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	bool readYourWritesDisabled;
															#line 421 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	bool readAheadDisabled;
															#line 422 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	std::vector<Future<Void>> operations;
															#line 423 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int waitLocation;
															#line 425 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int tenantNum;
															#line 432 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	bool rawAccess;
															#line 435 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	bool cancelled;
															#line 455 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int numWaits;
															#line 456 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int i;
															#line 461 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int numOps;
															#line 462 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int j;
															#line 464 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	int operationType;
															#line 3088 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via randomTransaction()
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RandomTransactionActor final : public Actor<Void>, public ActorCallback< RandomTransactionActor, 0, Void >, public ActorCallback< RandomTransactionActor, 1, Void >, public ActorCallback< RandomTransactionActor, 2, Void >, public ActorCallback< RandomTransactionActor, 3, Void >, public ActorCallback< RandomTransactionActor, 4, Void >, public FastAllocated<RandomTransactionActor>, public RandomTransactionActorState<RandomTransactionActor> {
															#line 3093 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<RandomTransactionActor>::operator new;
	using FastAllocated<RandomTransactionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3278695473915294976UL, 4759664778538152704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RandomTransactionActor, 0, Void >;
friend struct ActorCallback< RandomTransactionActor, 1, Void >;
friend struct ActorCallback< RandomTransactionActor, 2, Void >;
friend struct ActorCallback< RandomTransactionActor, 3, Void >;
friend struct ActorCallback< RandomTransactionActor, 4, Void >;
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RandomTransactionActor(FuzzApiCorrectnessWorkload* const& self,Database const& cx) 
															#line 3114 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   RandomTransactionActorState<RandomTransactionActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("randomTransaction", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15887765483757040128UL, 8877054762753181952UL);
		ActorExecutionContextHelper __helper(static_cast<RandomTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("randomTransaction");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("randomTransaction", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RandomTransactionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RandomTransactionActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RandomTransactionActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RandomTransactionActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< RandomTransactionActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] Future<Void> randomTransaction( FuzzApiCorrectnessWorkload* const& self, Database const& cx ) {
															#line 418 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<Void>(new RandomTransactionActor(self, cx));
															#line 3150 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 518 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

	template <typename Subclass, typename T>
	struct BaseTest {
		typedef T value_type;

																	#line 3159 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via runTest()
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class RunTestActor>
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RunTestActorState {
															#line 3165 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RunTestActorState(unsigned int const& id,FuzzApiCorrectnessWorkload* const& wl,Reference<ITransaction> const& tr) 
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : id(id),
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   wl(wl),
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   tr(tr),
															#line 524 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   self(id, wl, tr)
															#line 3178 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("runTest", reinterpret_cast<unsigned long>(this));

	}
	~RunTestActorState() 
	{
		fdb_probe_actor_destroy("runTest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 527 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				StrictFuture<value_type> __when_expr_0 = timeoutError(BaseTest::runTest2(tr, &self), 1000);
															#line 527 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				if (static_cast<RunTestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3196 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RunTestActor*>(this)->actor_wait_state = 1;
															#line 527 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunTestActor, 0, value_type >*>(static_cast<RunTestActor*>(this)));
															#line 3201 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunTestActorState();
		static_cast<RunTestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 533 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RunTestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RunTestActorState(); static_cast<RunTestActor*>(this)->destroy(); return 0; }
															#line 3230 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RunTestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RunTestActorState();
		static_cast<RunTestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 531 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self.contract.handleException(e, tr);
															#line 3243 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(value_type const& result,int loopDepth) 
	{
															#line 528 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		self.contract.handleNotThrown(tr);
															#line 529 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RunTestActor*>(this)->SAV<Void>::futures) { (void)(self.errorCheck(tr, result)); this->~RunTestActorState(); static_cast<RunTestActor*>(this)->destroy(); return 0; }
															#line 3260 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RunTestActor*>(this)->SAV< Void >::value()) Void(self.errorCheck(tr, result));
		this->~RunTestActorState();
		static_cast<RunTestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(value_type && result,int loopDepth) 
	{
															#line 528 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		self.contract.handleNotThrown(tr);
															#line 529 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RunTestActor*>(this)->SAV<Void>::futures) { (void)(self.errorCheck(tr, result)); this->~RunTestActorState(); static_cast<RunTestActor*>(this)->destroy(); return 0; }
															#line 3274 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RunTestActor*>(this)->SAV< Void >::value()) Void(self.errorCheck(tr, result));
		this->~RunTestActorState();
		static_cast<RunTestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(value_type const& result,int loopDepth) 
	{
		loopDepth = a_body1cont2(result, loopDepth);

		return loopDepth;
	}
	int a_body1when1(value_type && result,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunTestActor*>(this)->actor_wait_state > 0) static_cast<RunTestActor*>(this)->actor_wait_state = 0;
		static_cast<RunTestActor*>(this)->ActorCallback< RunTestActor, 0, value_type >::remove();

	}
	void a_callback_fire(ActorCallback< RunTestActor, 0, value_type >*,value_type const& value) 
	{
		fdb_probe_actor_enter("runTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunTestActor, 0, value_type >*,value_type && value) 
	{
		fdb_probe_actor_enter("runTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunTestActor, 0, value_type >*,Error err) 
	{
		fdb_probe_actor_enter("runTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	unsigned int id;
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	FuzzApiCorrectnessWorkload* wl;
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<ITransaction> tr;
															#line 524 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Subclass self;
															#line 3365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via runTest()
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RunTestActor final : public Actor<Void>, public ActorCallback< RunTestActor, 0, value_type >, public FastAllocated<RunTestActor>, public RunTestActorState<RunTestActor> {
															#line 3370 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<RunTestActor>::operator new;
	using FastAllocated<RunTestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11930809960289729536UL, 5192329930465471744UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunTestActor, 0, value_type >;
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RunTestActor(unsigned int const& id,FuzzApiCorrectnessWorkload* const& wl,Reference<ITransaction> const& tr) 
															#line 3387 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   RunTestActorState<RunTestActor>(id, wl, tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runTest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12013671372596341504UL, 10273208522725205248UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runTest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runTest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunTestActor, 0, value_type >*)0, actor_cancelled()); break;
		}

	}
};
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] static Future<Void> runTest( unsigned int const& id, FuzzApiCorrectnessWorkload* const& wl, Reference<ITransaction> const& tr ) {
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<Void>(new RunTestActor(id, wl, tr));
															#line 3419 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 535 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

																	#line 3424 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
// This generated class is to be used only via runTest2()
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
template <class RunTest2Actor>
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RunTest2ActorState {
															#line 3430 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RunTest2ActorState(Reference<ITransaction> const& tr,Subclass* const& self) 
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		 : tr(tr),
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   self(self),
															#line 537 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   future(unsafeThreadFutureToFuture(self->createFuture(tr))),
															#line 541 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		   i()
															#line 3443 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("runTest2", reinterpret_cast<unsigned long>(this));

	}
	~RunTest2ActorState() 
	{
		fdb_probe_actor_destroy("runTest2", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			i = self->pre_steps.begin();
															#line 3458 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunTest2ActorState();
		static_cast<RunTest2Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 546 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<value_type> __when_expr_1 = future;
															#line 546 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<RunTest2Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3483 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunTest2Actor*>(this)->actor_wait_state = 2;
															#line 546 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunTest2Actor, 1, value_type >*>(static_cast<RunTest2Actor*>(this)));
															#line 3488 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!(i != self->pre_steps.end()))
															#line 3504 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 543 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_0 = unsafeThreadFutureToFuture(*i);
															#line 543 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (static_cast<RunTest2Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3512 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunTest2Actor*>(this)->actor_wait_state = 1;
															#line 543 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunTest2Actor, 0, Void >*>(static_cast<RunTest2Actor*>(this)));
															#line 3517 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		++i;
															#line 3539 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		++i;
															#line 3548 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunTest2Actor*>(this)->actor_wait_state > 0) static_cast<RunTest2Actor*>(this)->actor_wait_state = 0;
		static_cast<RunTest2Actor*>(this)->ActorCallback< RunTest2Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunTest2Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunTest2Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunTest2Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(value_type const& result,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (future.isError())
															#line 3632 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 548 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->contract.handleException(future.getError(), tr);
															#line 3636 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
		else
		{
															#line 550 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			ASSERT(future.isValid());
															#line 3642 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 553 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RunTest2Actor*>(this)->SAV<value_type>::futures) { (void)(result); this->~RunTest2ActorState(); static_cast<RunTest2Actor*>(this)->destroy(); return 0; }
															#line 3646 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RunTest2Actor*>(this)->SAV< value_type >::value()) value_type(result);
		this->~RunTest2ActorState();
		static_cast<RunTest2Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(value_type && result,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (future.isError())
															#line 3658 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		{
															#line 548 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			self->contract.handleException(future.getError(), tr);
															#line 3662 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
		else
		{
															#line 550 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
			ASSERT(future.isValid());
															#line 3668 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		}
															#line 553 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
		if (!static_cast<RunTest2Actor*>(this)->SAV<value_type>::futures) { (void)(result); this->~RunTest2ActorState(); static_cast<RunTest2Actor*>(this)->destroy(); return 0; }
															#line 3672 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		new (&static_cast<RunTest2Actor*>(this)->SAV< value_type >::value()) value_type(result);
		this->~RunTest2ActorState();
		static_cast<RunTest2Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(value_type const& result,int loopDepth) 
	{
		loopDepth = a_body1cont2(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(value_type && result,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunTest2Actor*>(this)->actor_wait_state > 0) static_cast<RunTest2Actor*>(this)->actor_wait_state = 0;
		static_cast<RunTest2Actor*>(this)->ActorCallback< RunTest2Actor, 1, value_type >::remove();

	}
	void a_callback_fire(ActorCallback< RunTest2Actor, 1, value_type >*,value_type const& value) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunTest2Actor, 1, value_type >*,value_type && value) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunTest2Actor, 1, value_type >*,Error err) 
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Reference<ITransaction> tr;
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Subclass* self;
															#line 537 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	Future<value_type> future;
															#line 541 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	std::vector<ThreadFuture<Void>>::iterator i;
															#line 3763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
};
// This generated class is to be used only via runTest2()
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
class RunTest2Actor final : public Actor<value_type>, public ActorCallback< RunTest2Actor, 0, Void >, public ActorCallback< RunTest2Actor, 1, value_type >, public FastAllocated<RunTest2Actor>, public RunTest2ActorState<RunTest2Actor> {
															#line 3768 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
public:
	using FastAllocated<RunTest2Actor>::operator new;
	using FastAllocated<RunTest2Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15138056942005012224UL, 3853776565819548416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<value_type>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunTest2Actor, 0, Void >;
friend struct ActorCallback< RunTest2Actor, 1, value_type >;
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	RunTest2Actor(Reference<ITransaction> const& tr,Subclass* const& self) 
															#line 3786 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
		 : Actor<value_type>(),
		   RunTest2ActorState<RunTest2Actor>(tr, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runTest2", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5572249146214006272UL, 2926611344581904128UL);
		ActorExecutionContextHelper __helper(static_cast<RunTest2Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runTest2");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runTest2", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunTest2Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunTest2Actor, 1, value_type >*)0, actor_cancelled()); break;
		}

	}
};
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
[[nodiscard]] static Future<value_type> runTest2( Reference<ITransaction> const& tr, Subclass* const& self ) {
															#line 536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"
	return Future<value_type>(new RunTest2Actor(tr, self));
															#line 3819 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/FuzzApiCorrectness.actor.g.cpp"
}

#line 555 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/workloads/FuzzApiCorrectness.actor.cpp"

		virtual ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) = 0;
		virtual Void errorCheck(Reference<ITransaction> tr, value_type result) { return Void(); }
		virtual void augmentTrace(TraceEvent& e) const { e.detail("Id", id); }

	protected:
		unsigned int id;
		FuzzApiCorrectnessWorkload* workload;
		ExceptionContract contract;
		std::vector<ThreadFuture<Void>> pre_steps;

		BaseTest(unsigned int id_, FuzzApiCorrectnessWorkload* wl, const char* func)
		  : id(id_), workload(wl), contract(func, std::bind(&BaseTest::augmentTrace, this, ph::_1)) {}

		static Key makeKey() {
			double ksrv = deterministicRandom()->random01();

			// 25% of the time it's empty, 25% it's above range.
			if (ksrv < 0.25)
				return Key();

			int64_t key_size;
			if (ksrv < 0.5)
				key_size = deterministicRandom()->randomInt64(1, CLIENT_KNOBS->KEY_SIZE_LIMIT + 1) +
				           CLIENT_KNOBS->KEY_SIZE_LIMIT;
			else
				key_size = deterministicRandom()->randomInt64(1, CLIENT_KNOBS->KEY_SIZE_LIMIT + 1);

			std::string skey;
			skey.reserve(key_size);
			for (size_t j = 0; j < key_size; ++j)
				skey.append(1, (char)deterministicRandom()->randomInt(0, 256));

			// 15% (= 20% * 75%) of the time generating keys after \xff\xff to test special keys code
			if (deterministicRandom()->random01() < 0.2)
				return Key(skey).withPrefix(specialKeys.begin);
			else
				return Key(skey);
		}

		static Value makeValue() {
			double vrv = deterministicRandom()->random01();

			// 25% of the time it's empty, 25% it's above range.
			if (vrv < 0.25)
				return Value();

			int64_t value_size;
			if (vrv < 0.5)
				value_size = deterministicRandom()->randomInt64(1, CLIENT_KNOBS->VALUE_SIZE_LIMIT + 1) +
				             CLIENT_KNOBS->VALUE_SIZE_LIMIT;
			else
				value_size = deterministicRandom()->randomInt64(1, CLIENT_KNOBS->VALUE_SIZE_LIMIT + 1);

			std::string svalue;
			svalue.reserve(value_size);
			for (size_t j = 0; j < value_size; ++j)
				svalue.append(1, (char)deterministicRandom()->randomInt(0, 256));

			return Value(svalue);
		}

		static KeySelector makeKeySel() {
			// 40% of the time no offset, 30% it's a positive or negative.
			double orv = deterministicRandom()->random01();
			int offs = 0;
			if (orv >= 0.4) {
				if (orv < 0.7)
					offs = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					offs = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}
			return KeySelectorRef(makeKey(), deterministicRandom()->coinflip(), offs);
		}

		static GetRangeLimits makeRangeLimits() {
			double lrv = deterministicRandom()->random01();
			int rowlimit = 0;
			if (lrv >= 0.2) {
				if (lrv < 0.4)
					rowlimit = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					rowlimit = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}

			if (deterministicRandom()->coinflip())
				return GetRangeLimits(rowlimit);

			lrv = deterministicRandom()->random01();
			int bytelimit = 0;
			if (lrv >= 0.2) {
				if (lrv < 0.4)
					bytelimit = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					bytelimit = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}

			// Try again if both are 0.
			if (!rowlimit && !bytelimit)
				return makeRangeLimits();

			return GetRangeLimits(rowlimit, bytelimit);
		}

		static bool isOverlapping(const Key& begin, const Key& end, const KeyRangeRef& range) {
			return ((range.begin >= begin && range.begin < end) || (range.end >= begin && range.end < end) ||
			        (begin >= range.begin && begin < range.end) || (end >= range.begin && end < range.end));
		}
		static bool isOverlapping(const Key& begin, const Key& end, const KeyRef& key) {
			return (key >= begin && key < end);
		}

		static std::string slashToEnd(const KeyRef& key) { return key.toString().replace(key.size() - 1, 1, 1, '0'); }

		static KeyRangeRef& getServerKeys() {
			static std::string serverKeysEnd = slashToEnd(serverKeysPrefix);
			static KeyRangeRef serverKeys(serverKeysPrefix, StringRef(serverKeysEnd));
			return serverKeys;
		}

		static KeyRangeRef& getGlobalKeys() {
			static std::string globalKeysPrefix2 = globalKeysPrefix.toString() + "/";
			static std::string globalKeysEnd = slashToEnd(globalKeysPrefix2);
			static KeyRangeRef globalKeys(globalKeysPrefix2, StringRef(globalKeysEnd));
			return globalKeys;
		}

		static bool isProtectedKey(const Key& begin, const Key& end) {
			if (end < begin)
				return false;

			return (isOverlapping(begin, end, keyServersKeys) || isOverlapping(begin, end, serverListKeys) ||
			        isOverlapping(begin, end, processClassKeys) || isOverlapping(begin, end, configKeys) ||
			        isOverlapping(begin, end, workerListKeys) || isOverlapping(begin, end, backupLogKeys) ||
			        isOverlapping(begin, end, getServerKeys()) || isOverlapping(begin, end, getGlobalKeys()) ||
			        isOverlapping(begin, end, coordinatorsKey) || isOverlapping(begin, end, backupEnabledKey) ||
			        isOverlapping(begin, end, backupVersionKey));
		}

		static bool isProtectedKey(const Key& k) {
			return (isOverlapping(keyServersKeys.begin, keyServersKeys.end, k) ||
			        isOverlapping(serverListKeys.begin, serverListKeys.end, k) ||
			        isOverlapping(processClassKeys.begin, processClassKeys.end, k) ||
			        isOverlapping(configKeys.begin, configKeys.end, k) ||
			        isOverlapping(workerListKeys.begin, workerListKeys.end, k) ||
			        isOverlapping(backupLogKeys.begin, backupLogKeys.end, k) ||
			        isOverlapping(getServerKeys().begin, getServerKeys().end, k) ||
			        isOverlapping(getGlobalKeys().begin, getGlobalKeys().end, k) || coordinatorsKey == k ||
			        backupEnabledKey == k || backupVersionKey == k);
		}
	};

	template <typename Subclass>
	struct BaseTestCallback : BaseTest<Subclass, Void> {
		typedef typename BaseTest<Subclass, Void>::value_type value_type;

		BaseTestCallback(unsigned int id, FuzzApiCorrectnessWorkload* wl, const char* func)
		  : BaseTest<Subclass, Void>(id, wl, func) {}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			callback(tr);
			return tr.castTo<ThreadSafeTransaction>()->checkDeferredError();
		}

		Void errorCheck(Reference<ITransaction> tr, value_type result) override {
			callbackErrorCheck(tr);
			return Void();
		}

		virtual void callback(Reference<ITransaction> tr) = 0;
		virtual void callbackErrorCheck(Reference<ITransaction> tr) {}
	};

	struct TestSetVersion : public BaseTest<TestSetVersion, Version> {
		typedef BaseTest<TestSetVersion, Version> base_type;
		Version v;

		TestSetVersion(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestSetVersion") {
			if (deterministicRandom()->coinflip())
				v = deterministicRandom()->randomInt64(INT64_MIN, 0);
			else
				v = deterministicRandom()->randomInt64(0, INT64_MAX);

			contract = { std::make_pair(error_code_read_version_already_set, ExceptionContract::Possible),
				         std::make_pair(error_code_version_invalid, ExceptionContract::requiredIf(v <= 0)) };
		}

		ThreadFuture<Version> createFuture(Reference<ITransaction> tr) override {
			tr->setVersion(v);
			pre_steps.push_back(tr.castTo<ThreadSafeTransaction>()->checkDeferredError());
			return tr->getReadVersion();
		}

		Void errorCheck(Reference<ITransaction> tr, value_type result) override {
			ASSERT(v == result);
			return Void();
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Version", v);
		}
	};

	struct TestGet : public BaseTest<TestGet, Optional<Value>> {
		typedef BaseTest<TestGet, Optional<Value>> base_type;
		Key key;

		TestGet(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGet") {
			key = makeKey();
			contract = {
				std::make_pair(
				    error_code_key_outside_legal_range,
				    ExceptionContract::requiredIf((key >= workload->getMaxKey(tr)) && !specialKeys.contains(key))),
				std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				std::make_pair(
				    error_code_special_keys_no_module_found,
				    ExceptionContract::possibleIf(specialKeys.contains(key) && !workload->specialKeysRelaxed)),
				// Read this particular special key may throw timed_out
				std::make_pair(error_code_timed_out, ExceptionContract::possibleIf(key == "\xff\xff/status/json"_sr)),
				// Read this particular special key may throw special_keys_api_failure
				std::make_pair(
				    error_code_special_keys_api_failure,
				    ExceptionContract::possibleIf(
				        key == "auto_coordinators"_sr.withPrefix(
				                   SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin))),
				std::make_pair(error_code_tenant_not_found,
				               ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))),
				std::make_pair(error_code_invalid_option,
				               ExceptionContract::possibleIf(tr->getTenant().present() && specialKeys.contains(key))),
				std::make_pair(error_code_illegal_tenant_access,
				               ExceptionContract::possibleIf(tr->getTenant().present() && specialKeys.contains(key)))
			};
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->get(key, deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key);
			e.detail("Size", key.size());
		}
	};

	struct TestGetKey : public BaseTest<TestGetKey, Key> {
		typedef BaseTest<TestGetKey, Key> base_type;
		KeySelector keysel;

		TestGetKey(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetKey") {
			keysel = makeKeySel();
			contract = { std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((keysel.getKey() > workload->getMaxKey(tr)))),
				         std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				         std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				         std::make_pair(error_code_tenant_not_found,
				                        ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))) };
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getKey(keysel, deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("KeySel", keysel);
		}
	};

	struct TestGetRange0 : public BaseTest<TestGetRange0, RangeResult> {
		typedef BaseTest<TestGetRange0, RangeResult> base_type;
		KeySelector keysel1, keysel2;
		int limit;

		TestGetRange0(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetRange0") {
			keysel1 = makeKeySel();
			keysel2 = makeKeySel();
			limit = 0;

			double lrv = deterministicRandom()->random01();
			if (lrv > 0.20) {
				if (lrv < 0.4)
					limit = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					limit = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}

			bool isSpecialKeyRange = specialKeys.contains(keysel1.getKey()) && specialKeys.begin <= keysel2.getKey() &&
			                         keysel2.getKey() <= specialKeys.end;

			contract = {
				std::make_pair(error_code_range_limits_invalid, ExceptionContract::possibleButRequiredIf(limit < 0)),
				std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				std::make_pair(error_code_key_outside_legal_range,
				               ExceptionContract::requiredIf(((keysel1.getKey() > workload->getMaxKey(tr)) ||
				                                              (keysel2.getKey() > workload->getMaxKey(tr))) &&
				                                             !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_cross_module_read,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(error_code_special_keys_no_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				// Read some special keys, e.g. status/json, can throw timed_out
				std::make_pair(error_code_timed_out, ExceptionContract::possibleIf(isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_api_failure, ExceptionContract::possibleIf(isSpecialKeyRange)),
				std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				std::make_pair(error_code_tenant_not_found,
				               ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))),
				std::make_pair(error_code_invalid_option,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange)),
				std::make_pair(error_code_illegal_tenant_access,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange))
			};
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getRange(
			    keysel1, keysel2, limit, deterministicRandom()->coinflip(), deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("KeySel1", keysel1).detail("KeySel2", keysel2).detail("Limit", limit);
		}
	};

	struct TestGetRange1 : public BaseTest<TestGetRange1, RangeResult> {
		typedef BaseTest<TestGetRange1, RangeResult> base_type;
		KeySelector keysel1, keysel2;
		GetRangeLimits limits;

		TestGetRange1(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetRange1") {
			keysel1 = makeKeySel();
			keysel2 = makeKeySel();
			limits = makeRangeLimits();

			bool isSpecialKeyRange = specialKeys.contains(keysel1.getKey()) && specialKeys.begin <= keysel2.getKey() &&
			                         keysel2.getKey() <= specialKeys.end;

			contract = {
				std::make_pair(error_code_range_limits_invalid,
				               ExceptionContract::possibleButRequiredIf(!limits.isReached() && !limits.isValid())),
				std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				std::make_pair(error_code_key_outside_legal_range,
				               ExceptionContract::requiredIf(((keysel1.getKey() > workload->getMaxKey(tr)) ||
				                                              (keysel2.getKey() > workload->getMaxKey(tr))) &&
				                                             !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_cross_module_read,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(error_code_special_keys_no_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(error_code_timed_out, ExceptionContract::possibleIf(isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_api_failure, ExceptionContract::possibleIf(isSpecialKeyRange)),
				std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				std::make_pair(error_code_tenant_not_found,
				               ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))),
				std::make_pair(error_code_illegal_tenant_access,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange))
			};
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getRange(
			    keysel1, keysel2, limits, deterministicRandom()->coinflip(), deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("KeySel1", keysel1).detail("KeySel2", keysel2);
			std::stringstream ss;
			ss << "(" << limits.rows << ", " << limits.minRows << ", " << limits.bytes << ")";
			e.detail("Limits", ss.str());
		}
	};

	struct TestGetRange2 : public BaseTest<TestGetRange2, RangeResult> {
		typedef BaseTest<TestGetRange2, RangeResult> base_type;
		Key key1, key2;
		int limit;

		TestGetRange2(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetRange2") {
			key1 = makeKey();
			key2 = makeKey();
			limit = 0;

			double lrv = deterministicRandom()->random01();
			if (lrv > 0.20) {
				if (lrv < 0.4)
					limit = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					limit = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}

			bool isSpecialKeyRange = specialKeys.contains(key1) && specialKeys.begin <= key2 && key2 <= specialKeys.end;
			// Read this particular special key may throw special_keys_api_failure
			Key autoCoordinatorSpecialKey = "auto_coordinators"_sr.withPrefix(
			    SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin);
			KeyRangeRef actorLineageRange = SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTORLINEAGE);
			// Read this particular special key may throw timed_out
			Key statusJsonSpecialKey = "\xff\xff/status/json"_sr;

			contract = {
				std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				std::make_pair(error_code_range_limits_invalid, ExceptionContract::possibleButRequiredIf(limit < 0)),
				std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				std::make_pair(
				    error_code_key_outside_legal_range,
				    ExceptionContract::requiredIf(
				        ((key1 > workload->getMaxKey(tr)) || (key2 > workload->getMaxKey(tr))) && !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_cross_module_read,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(error_code_special_keys_no_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(
				    error_code_timed_out,
				    ExceptionContract::possibleIf(key1 <= statusJsonSpecialKey && statusJsonSpecialKey < key2)),
				std::make_pair(error_code_special_keys_api_failure,
				               ExceptionContract::possibleIf(
				                   (key1 <= autoCoordinatorSpecialKey && autoCoordinatorSpecialKey < key2) ||
				                   actorLineageRange.intersects(KeyRangeRef(key1, key2)))),
				std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				std::make_pair(error_code_tenant_not_found,
				               ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))),
				std::make_pair(error_code_invalid_option,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange)),
				std::make_pair(error_code_illegal_tenant_access,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange))
			};
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getRange(
			    KeyRangeRef(key1, key2), limit, deterministicRandom()->coinflip(), deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2).detail("Limit", limit);
		}
	};

	struct TestGetRange3 : public BaseTest<TestGetRange3, RangeResult> {
		typedef BaseTest<TestGetRange3, RangeResult> base_type;
		Key key1, key2;
		GetRangeLimits limits;

		TestGetRange3(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetRange3") {
			key1 = makeKey();
			key2 = makeKey();
			limits = makeRangeLimits();

			bool isSpecialKeyRange = specialKeys.contains(key1) && specialKeys.begin <= key2 && key2 <= specialKeys.end;
			Key autoCoordinatorSpecialKey = "auto_coordinators"_sr.withPrefix(
			    SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin);
			KeyRangeRef actorLineageRange = SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTORLINEAGE);
			Key statusJsonSpecialKey = "\xff\xff/status/json"_sr;

			contract = {
				std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				std::make_pair(error_code_range_limits_invalid,
				               ExceptionContract::possibleButRequiredIf(!limits.isReached() && !limits.isValid())),
				std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				std::make_pair(
				    error_code_key_outside_legal_range,
				    ExceptionContract::requiredIf(
				        ((key1 > workload->getMaxKey(tr)) || (key2 > workload->getMaxKey(tr))) && !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_cross_module_read,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(error_code_special_keys_no_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && !workload->specialKeysRelaxed)),
				std::make_pair(
				    error_code_timed_out,
				    ExceptionContract::possibleIf(key1 <= statusJsonSpecialKey && statusJsonSpecialKey < key2)),
				std::make_pair(error_code_special_keys_api_failure,
				               ExceptionContract::possibleIf(
				                   (key1 <= autoCoordinatorSpecialKey && autoCoordinatorSpecialKey < key2) ||
				                   actorLineageRange.intersects(KeyRangeRef(key1, key2)))),
				std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				std::make_pair(error_code_tenant_not_found,
				               ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))),
				std::make_pair(error_code_invalid_option,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange)),
				std::make_pair(error_code_illegal_tenant_access,
				               ExceptionContract::possibleIf(tr->getTenant().present() && isSpecialKeyRange))
			};
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getRange(
			    KeyRangeRef(key1, key2), limits, deterministicRandom()->coinflip(), deterministicRandom()->coinflip());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2);
			std::stringstream ss;
			ss << "(" << limits.rows << ", " << limits.minRows << ", " << limits.bytes << ")";
			e.detail("Limits", ss.str());
		}
	};

	struct TestGetAddressesForKey : public BaseTest<TestGetAddressesForKey, Standalone<VectorRef<const char*>>> {
		typedef BaseTest<TestGetAddressesForKey, Standalone<VectorRef<const char*>>> base_type;
		Key key;

		TestGetAddressesForKey(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestGetAddressesForKey") {
			key = makeKey();
			contract = { std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				         std::make_pair(error_code_tenant_not_found,
				                        ExceptionContract::requiredIf(!workload->canUseTenant(tr->getTenant()))) };
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override {
			return tr->getAddressesForKey(key);
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key);
		}
	};

	struct TestAddReadConflictRange : public BaseTestCallback<TestAddReadConflictRange> {
		typedef BaseTest<TestAddReadConflictRange, Void> base_type;
		Key key1, key2;

		TestAddReadConflictRange(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestAddReadConflictRange") {
			key1 = makeKey();
			key2 = makeKey();
			contract = { std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				         std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((key1 > workload->getMaxKey(tr)) ||
				                                                      (key2 > workload->getMaxKey(tr)))) };
		}

		void callback(Reference<ITransaction> tr) override { tr->addReadConflictRange(KeyRangeRef(key1, key2)); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2);
		}
	};

	struct TestAtomicOp : public BaseTestCallback<TestAtomicOp> {
		typedef BaseTest<TestAtomicOp, Void> base_type;
		Key key;
		Value value;
		uint8_t op;
		int32_t pos;

		TestAtomicOp(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestAtomicOp") {
			key = makeKey();
			while (isProtectedKey(key)) {
				key = makeKey();
			}
			value = makeValue();
			double arv = deterministicRandom()->random01();
			if (arv < 0.25) {
				int val = UINT8_MAX;
				for (auto i = FDBMutationTypes::optionInfo.begin(); i != FDBMutationTypes::optionInfo.end(); ++i)
					if (i->first < val)
						val = i->first;
				op = deterministicRandom()->randomInt(0, val);
			} else if (arv < 0.50) {
				int val = 0;
				for (auto i = FDBMutationTypes::optionInfo.begin(); i != FDBMutationTypes::optionInfo.end(); ++i)
					if (i->first > val)
						val = i->first;
				op = deterministicRandom()->randomInt(val + 1, UINT8_MAX);
			} else {
				int minval = UINT8_MAX, maxval = 0;
				for (auto i = FDBMutationTypes::optionInfo.begin(); i != FDBMutationTypes::optionInfo.end(); ++i) {
					if (i->first < minval)
						minval = i->first;
					if (i->first > maxval)
						maxval = i->first;
				}
				op = deterministicRandom()->randomInt(minval, maxval + 1);
			}

			pos = -1;
			if (op == MutationRef::SetVersionstampedKey && key.size() >= 4) {
				pos = littleEndian32(*(int32_t*)&key.end()[-4]);
			}
			if (op == MutationRef::SetVersionstampedValue && value.size() >= 4) {
				pos = littleEndian32(*(int32_t*)&value.end()[-4]);
			}

			contract = { std::make_pair(error_code_key_too_large,
				                        key.size() > getMaxWriteKeySize(key, true)    ? ExceptionContract::Always
				                        : key.size() > getMaxWriteKeySize(key, false) ? ExceptionContract::Possible
				                                                                      : ExceptionContract::Never),
				         std::make_pair(error_code_value_too_large,
				                        ExceptionContract::requiredIf(value.size() > CLIENT_KNOBS->VALUE_SIZE_LIMIT)),
				         std::make_pair(error_code_invalid_mutation_type,
				                        ExceptionContract::requiredIf(!isValidMutationType(op) ||
				                                                      !isAtomicOp((MutationRef::Type)op))),
				         std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((key >= workload->getMaxKey(tr)))),
				         std::make_pair(error_code_client_invalid_operation,
				                        ExceptionContract::requiredIf((op == MutationRef::SetVersionstampedKey &&
				                                                       (pos < 0 || pos + 10 > key.size() - 4)) ||
				                                                      (op == MutationRef::SetVersionstampedValue &&
				                                                       (pos < 0 || pos + 10 > value.size() - 4)))) };
		}

		void callback(Reference<ITransaction> tr) override { tr->atomicOp(key, value, (FDBMutationTypes::Option)op); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key).detail("Value", value).detail("Op", op).detail("Pos", pos);
		}
	};

	struct TestSet : public BaseTestCallback<TestSet> {
		typedef BaseTest<TestSet, Void> base_type;
		Key key;
		Value value;

		TestSet(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestSet") {
			key = makeKey();
			while (isProtectedKey(key)) {
				key = makeKey();
			}
			value = makeValue();
			contract = { std::make_pair(error_code_key_too_large,
				                        key.size() > getMaxWriteKeySize(key, true)    ? ExceptionContract::Always
				                        : key.size() > getMaxWriteKeySize(key, false) ? ExceptionContract::Possible
				                                                                      : ExceptionContract::Never),
				         std::make_pair(error_code_value_too_large,
				                        ExceptionContract::requiredIf(value.size() > CLIENT_KNOBS->VALUE_SIZE_LIMIT)),
				         std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((key >= workload->getMaxKey(tr)) &&
				                                                      !specialKeys.contains(key))),
				         std::make_pair(error_code_special_keys_write_disabled,
				                        ExceptionContract::requiredIf(specialKeys.contains(key) &&
				                                                      !workload->specialKeysWritesEnabled)),
				         std::make_pair(error_code_special_keys_no_write_module_found,
				                        ExceptionContract::possibleIf(specialKeys.contains(key) &&
				                                                      workload->specialKeysWritesEnabled)) };
		}

		void callback(Reference<ITransaction> tr) override { tr->set(key, value); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key).detail("Value", value);
		}
	};

	struct TestClear0 : public BaseTestCallback<TestClear0> {
		typedef BaseTest<TestClear0, Void> base_type;
		Key key1, key2;

		TestClear0(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestClear0") {
			key1 = makeKey();
			key2 = makeKey();
			while (isProtectedKey(key1, key2)) {
				key1 = makeKey();
				key2 = makeKey();
			}

			bool isSpecialKeyRange = specialKeys.contains(key1) && key2 <= specialKeys.end;

			contract = {
				std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				std::make_pair(
				    error_code_key_outside_legal_range,
				    ExceptionContract::requiredIf(
				        ((key1 > workload->getMaxKey(tr)) || (key2 > workload->getMaxKey(tr))) && !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_write_disabled,
				               ExceptionContract::requiredIf(isSpecialKeyRange && !workload->specialKeysWritesEnabled)),
				std::make_pair(error_code_special_keys_cross_module_clear,
				               ExceptionContract::possibleIf(isSpecialKeyRange && workload->specialKeysWritesEnabled)),
				std::make_pair(error_code_special_keys_no_write_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && workload->specialKeysWritesEnabled))
			};
		}

		void callback(Reference<ITransaction> tr) override { tr->clear(key1, key2); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2);
		}
	};

	struct TestClear1 : public BaseTestCallback<TestClear1> {
		typedef BaseTest<TestClear1, Void> base_type;
		Key key1, key2;

		TestClear1(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestClear1") {
			key1 = makeKey();
			key2 = makeKey();
			while (isProtectedKey(key1, key2)) {
				key1 = makeKey();
				key2 = makeKey();
			}

			bool isSpecialKeyRange = specialKeys.contains(key1) && key2 <= specialKeys.end;

			contract = {
				std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				std::make_pair(
				    error_code_key_outside_legal_range,
				    ExceptionContract::requiredIf(
				        ((key1 > workload->getMaxKey(tr)) || (key2 > workload->getMaxKey(tr))) && !isSpecialKeyRange)),
				std::make_pair(error_code_special_keys_write_disabled,
				               ExceptionContract::requiredIf(isSpecialKeyRange && !workload->specialKeysWritesEnabled)),
				std::make_pair(error_code_special_keys_cross_module_clear,
				               ExceptionContract::possibleIf(isSpecialKeyRange && workload->specialKeysWritesEnabled)),
				std::make_pair(error_code_special_keys_no_write_module_found,
				               ExceptionContract::possibleIf(isSpecialKeyRange && workload->specialKeysWritesEnabled))
			};
		}

		void callback(Reference<ITransaction> tr) override { tr->clear(KeyRangeRef(key1, key2)); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2);
		}
	};

	struct TestClear2 : public BaseTestCallback<TestClear2> {
		typedef BaseTest<TestClear2, Void> base_type;
		Key key;

		TestClear2(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestClear2") {
			key = makeKey();
			while (isProtectedKey(key)) {
				key = makeKey();
			}
			contract = { std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf(key >= workload->getMaxKey(tr))),
				         std::make_pair(error_code_special_keys_write_disabled,
				                        ExceptionContract::requiredIf(specialKeys.contains(key) &&
				                                                      !workload->specialKeysWritesEnabled)),
				         std::make_pair(error_code_special_keys_no_write_module_found,
				                        ExceptionContract::possibleIf(specialKeys.contains(key) &&
				                                                      workload->specialKeysWritesEnabled)) };
		}

		void callback(Reference<ITransaction> tr) override { tr->clear(key); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key);
		}
	};

	struct TestWatch : public BaseTest<TestWatch, Void> {
		typedef BaseTest<TestWatch, Void> base_type;
		Key key;

		TestWatch(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTest(id, workload, "TestWatch") {
			key = makeKey();
			printf("Watching: %d %s\n", key.size(), printable(key.substr(0, std::min(key.size(), 20))).c_str());
			contract = { std::make_pair(error_code_key_too_large,
				                        key.size() > getMaxWriteKeySize(key, true)    ? ExceptionContract::Always
				                        : key.size() > getMaxWriteKeySize(key, false) ? ExceptionContract::Possible
				                                                                      : ExceptionContract::Never),
				         std::make_pair(error_code_watches_disabled, ExceptionContract::Possible),
				         std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((key >= workload->getMaxKey(tr)))),
				         std::make_pair(error_code_client_invalid_operation, ExceptionContract::Possible),
				         std::make_pair(error_code_timed_out, ExceptionContract::Possible),
				         std::make_pair(error_code_accessed_unreadable, ExceptionContract::Possible),
				         std::make_pair(error_code_tenant_not_found,
				                        ExceptionContract::possibleIf(!workload->canUseTenant(tr->getTenant()))) };
		}

		ThreadFuture<value_type> createFuture(Reference<ITransaction> tr) override { return tr->watch(key); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key", key);
		}
	};

	struct TestAddWriteConflictRange : public BaseTestCallback<TestAddWriteConflictRange> {
		typedef BaseTest<TestAddWriteConflictRange, Void> base_type;
		Key key1, key2;

		TestAddWriteConflictRange(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestAddWriteConflictRange") {
			key1 = makeKey();
			key2 = makeKey();
			contract = { std::make_pair(error_code_inverted_range, ExceptionContract::requiredIf(key1 > key2)),
				         std::make_pair(error_code_key_outside_legal_range,
				                        ExceptionContract::requiredIf((key1 > workload->getMaxKey(tr)) ||
				                                                      (key2 > workload->getMaxKey(tr)))) };
		}

		void callback(Reference<ITransaction> tr) override { tr->addWriteConflictRange(KeyRangeRef(key1, key2)); }

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Key1", key1).detail("Key2", key2);
		}
	};

	struct TestSetOption : public BaseTestCallback<TestSetOption> {
		typedef BaseTest<TestSetOption, Void> base_type;
		int op;
		Optional<Standalone<StringRef>> val;

		TestSetOption(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestSetOption") {
			double arv = deterministicRandom()->random01();
			if (arv < 0.25) {
				int val = INT_MAX;
				for (auto i = FDBTransactionOptions::optionInfo.begin(); i != FDBTransactionOptions::optionInfo.end();
				     ++i)
					if (i->first < val)
						val = i->first;
				op = deterministicRandom()->randomInt(INT_MIN, val);
			} else if (arv < 0.50) {
				int val = INT_MIN;
				for (auto i = FDBTransactionOptions::optionInfo.begin(); i != FDBTransactionOptions::optionInfo.end();
				     ++i)
					if (i->first > val)
						val = i->first;
				op = deterministicRandom()->randomInt(val + 1, INT_MAX);
			} else {
				int minval = INT_MAX, maxval = INT_MIN;
				for (auto i = FDBTransactionOptions::optionInfo.begin(); i != FDBTransactionOptions::optionInfo.end();
				     ++i) {
					if (i->first < minval)
						minval = i->first;
					if (i->first > maxval)
						maxval = i->first;
				}
				op = deterministicRandom()->randomInt(minval, maxval + 1);
			}

			// do not test the following options since they are actually used by the workload
			if (op == FDBTransactionOptions::ACCESS_SYSTEM_KEYS || op == FDBTransactionOptions::READ_SYSTEM_KEYS ||
			    op == FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES ||
			    op == FDBTransactionOptions::RAW_ACCESS) {
				op = -1;
			}

			// do not test the option since it's already used by the workload
			if (op == FDBTransactionOptions::SPECIAL_KEY_SPACE_RELAXED)
				op = -1;
			// disable for now(see issue#3934, pr#3930)
			if (op == FDBTransactionOptions::CHECK_WRITES_ENABLE)
				op = -1;

			double orv = deterministicRandom()->random01();
			if (orv >= 0.25) {
				int64_t value_size = deterministicRandom()->randomInt64(0, CLIENT_KNOBS->VALUE_SIZE_LIMIT + 1) +
				                     CLIENT_KNOBS->VALUE_SIZE_LIMIT;

				std::string svalue;
				svalue.reserve(value_size);
				for (size_t j = 0; j < value_size; ++j)
					svalue.append(1, (char)deterministicRandom()->randomInt(0, 256));

				val = Standalone<StringRef>(StringRef(svalue));
			}

			contract = { std::make_pair(error_code_invalid_option, ExceptionContract::Possible),
				         std::make_pair(error_code_invalid_option_value, ExceptionContract::Possible),
				         std::make_pair(error_code_tag_too_long, ExceptionContract::Possible),
				         std::make_pair(error_code_too_many_tags, ExceptionContract::Possible),
				         std::make_pair(
				             error_code_client_invalid_operation,
				             ExceptionContract::possibleIf(
				                 (FDBTransactionOptions::Option)op == FDBTransactionOptions::READ_YOUR_WRITES_DISABLE ||
				                 (FDBTransactionOptions::Option)op == FDBTransactionOptions::LOG_TRANSACTION)),
				         std::make_pair(
				             error_code_read_version_already_set,
				             ExceptionContract::possibleIf((FDBTransactionOptions::Option)op ==
				                                           FDBTransactionOptions::INITIALIZE_NEW_DATABASE)) };
		}

		void callback(Reference<ITransaction> tr) override {
			tr->setOption((FDBTransactionOptions::Option)op, val.castTo<StringRef>());
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("Op", op).detail("Val", val);
		}
	};

	struct TestOnError : public BaseTestCallback<TestOnError> {
		typedef BaseTest<TestOnError, Void> base_type;
		int errorcode;

		TestOnError(unsigned int id, FuzzApiCorrectnessWorkload* workload, Reference<ITransaction> tr)
		  : BaseTestCallback(id, workload, "TestOnError") {
			errorcode = 0;
			double erv = deterministicRandom()->random01();
			if (erv >= 0.2) {
				if (erv < 0.6)
					errorcode = deterministicRandom()->randomInt(INT_MIN, 0);
				else
					errorcode = deterministicRandom()->randomInt(0, INT_MAX) + 1;
			}
		}

		void callback(Reference<ITransaction> tr) override {
			tr->onError(Error::fromUnvalidatedCode(errorcode));
			// This is necessary here, as onError will have reset this
			// value, we will be looking at the wrong thing.
			if (workload->useSystemKeys && !tr->getTenant().present())
				tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
		}

		void augmentTrace(TraceEvent& e) const override {
			base_type::augmentTrace(e);
			e.detail("ErrorCode", errorcode);
		}
	};

	static void addTestCases() {
		testCases.push_back(std::bind(&TestSetVersion::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGet::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetKey::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetRange0::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetRange1::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetRange2::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetRange3::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestGetAddressesForKey::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestAddReadConflictRange::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestAtomicOp::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestSet::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestClear0::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestClear1::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestClear2::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestWatch::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestAddWriteConflictRange::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestSetOption::runTest, ph::_1, ph::_2, ph::_3));
		testCases.push_back(std::bind(&TestOnError::runTest, ph::_1, ph::_2, ph::_3));
	}
};

std::once_flag FuzzApiCorrectnessWorkload::onceFlag;
std::vector<std::function<
    Future<Void>(unsigned int const&, FuzzApiCorrectnessWorkload* const&, Reference<ITransaction> const&)>>
    FuzzApiCorrectnessWorkload::testCases;
WorkloadFactory<FuzzApiCorrectnessWorkload> FuzzApiCorrectnessWorkloadFactory;
