#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
/*
 * ConfigBroadcaster.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>

#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbserver/ConfigBroadcaster.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/IConfigConsumer.h"
#include "flow/UnitTest.h"
#include "flow/actorcompiler.h" // must be last include

namespace {

bool matchesConfigClass(ConfigClassSet const& configClassSet, Optional<KeyRef> configClass) {
	return !configClass.present() || configClassSet.contains(configClass.get());
}

// Helper functions for STL containers, with flow-friendly error handling
template <class MapContainer, class K>
auto get(MapContainer& m, K const& k) -> decltype(m.at(k)) {
	auto it = m.find(k);
	ASSERT(it != m.end());
	return it->second;
}
template <class Container, class K>
void remove(Container& container, K const& k) {
	auto it = container.find(k);
	ASSERT(it != container.end());
	container.erase(it);
}

} // namespace

class ConfigBroadcasterImpl {
	// Holds information about each client connected to the broadcaster.
	struct BroadcastClientDetails {
		// Triggered when the worker dies.
		Future<Void> watcher;
		ConfigClassSet configClassSet;
		Version lastSeenVersion;
		ConfigBroadcastInterface broadcastInterface;

		bool operator==(BroadcastClientDetails const& rhs) const {
			return configClassSet == rhs.configClassSet && lastSeenVersion == rhs.lastSeenVersion &&
			       broadcastInterface == rhs.broadcastInterface;
		}
		bool operator!=(BroadcastClientDetails const& rhs) const { return !(*this == rhs); }

		BroadcastClientDetails() = default;
		BroadcastClientDetails(Future<Void> watcher,
		                       ConfigClassSet const& configClassSet,
		                       Version lastSeenVersion,
		                       ConfigBroadcastInterface broadcastInterface)
		  : watcher(watcher), configClassSet(configClassSet), lastSeenVersion(lastSeenVersion),
		    broadcastInterface(broadcastInterface) {}
	};

	ConfigDBType configDBType;
	std::map<ConfigKey, KnobValue> snapshot;
	std::deque<VersionedConfigMutation> mutationHistory;
	std::deque<VersionedConfigCommitAnnotation> annotationHistory;
	Version lastCompactedVersion;
	Version largestLiveVersion;
	Version mostRecentVersion;
	CoordinatorsHash coordinatorsHash;
	std::unique_ptr<IConfigConsumer> consumer;
	Future<Void> consumerFuture;
	ActorCollection actors{ false };
	std::unordered_map<NetworkAddress, Future<Void>> registrationActors;
	std::map<UID, BroadcastClientDetails> clients;
	std::map<UID, Future<Void>> clientFailures;

	// State related to changing coordinators

	// Used to read a snapshot from the previous coordinators after a change
	// coordinators command.
	Future<Optional<Value>> previousCoordinatorsFuture;
	std::unique_ptr<IConfigConsumer> previousCoordinatorsConsumer;
	Future<Void> previousCoordinatorsSnapshotFuture;
	Version largestConfigNodeVersion{ ::invalidVersion };

	UID id;
	CounterCollection cc;
	Counter compactRequest;
	Counter successfulChangeRequest;
	Counter failedChangeRequest;
	Counter snapshotRequest;
	Future<Void> logger;

	int coordinators = 0;
	std::unordered_set<NetworkAddress> registeredConfigNodes;
	std::unordered_set<NetworkAddress> activeConfigNodes;
	std::unordered_set<NetworkAddress> registrationResponses;
	std::unordered_set<NetworkAddress> registrationResponsesUnregistered;
	bool disallowUnregistered = false;
	Promise<Void> newConfigNodesAllowed;

	Future<Void> pushSnapshot(Version snapshotVersion, BroadcastClientDetails const& client) {
		if (client.lastSeenVersion >= snapshotVersion) {
			return Void();
		}

		++snapshotRequest;
		ConfigBroadcastSnapshotRequest request;
		for (const auto& [key, value] : snapshot) {
			if (matchesConfigClass(client.configClassSet, key.configClass)) {
				request.snapshot[key] = value;
			}
		}
		request.version = snapshotVersion;
		// TODO: Don't need a delay if there are no atomic changes.
		// Delay restarting the cluster controller to allow messages to be sent to workers.
		request.restartDelay = client.broadcastInterface.address() == g_network->getLocalAddress()
		                           ? SERVER_KNOBS->BROADCASTER_SELF_UPDATE_DELAY
		                           : 0.0;
		TraceEvent(SevDebug, "ConfigBroadcasterSnapshotRequest", id)
		    .detail("Size", request.snapshot.size())
		    .detail("Version", request.version);
		return success(brokenPromiseToNever(client.broadcastInterface.snapshot.getReply(request)));
	}

	template <class Changes>
	Future<Void> pushChanges(BroadcastClientDetails& client, Changes const& changes) {
		// Skip if client has already seen the latest version.
		if (client.lastSeenVersion >= mostRecentVersion) {
			return Void();
		}

		ConfigBroadcastChangesRequest req;
		for (const auto& versionedMutation : changes) {
			if (versionedMutation.version > client.lastSeenVersion &&
			    matchesConfigClass(client.configClassSet, versionedMutation.mutation.getConfigClass())) {
				TraceEvent te(SevDebug, "ConfigBroadcasterSendingChangeMutation", id);
				te.detail("Version", versionedMutation.version)
				    .detail("ReqLastSeenVersion", client.lastSeenVersion)
				    .detail("ConfigClass", versionedMutation.mutation.getConfigClass())
				    .detail("KnobName", versionedMutation.mutation.getKnobName())
				    .detail("ClientID", client.broadcastInterface.id());
				if (versionedMutation.mutation.isSet()) {
					te.detail("Op", "Set").detail("KnobValue", versionedMutation.mutation.getValue().toString());
				} else {
					te.detail("Op", "Clear");
				}

				req.changes.push_back_deep(req.changes.arena(), versionedMutation);
			}
		}

		if (req.changes.size() == 0) {
			return Void();
		}

		client.lastSeenVersion = mostRecentVersion;
		req.mostRecentVersion = mostRecentVersion;
		// TODO: Don't need a delay if there are no atomic changes.
		// Delay restarting the cluster controller to allow messages to be sent to workers.
		req.restartDelay = client.broadcastInterface.address() == g_network->getLocalAddress()
		                       ? SERVER_KNOBS->BROADCASTER_SELF_UPDATE_DELAY
		                       : 0.0;
		++successfulChangeRequest;
		return success(client.broadcastInterface.changes.getReply(req));
	}

	ConfigBroadcasterImpl()
	  : lastCompactedVersion(0), largestLiveVersion(0), mostRecentVersion(0),
	    id(deterministicRandom()->randomUniqueID()), cc("ConfigBroadcaster"), compactRequest("CompactRequest", cc),
	    successfulChangeRequest("SuccessfulChangeRequest", cc), failedChangeRequest("FailedChangeRequest", cc),
	    snapshotRequest("SnapshotRequest", cc) {
		logger = cc.traceCounters(
		    "ConfigBroadcasterMetrics", id, SERVER_KNOBS->WORKER_LOGGING_INTERVAL, "ConfigBroadcasterMetrics");
	}

	void addChanges(Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
	                Version mostRecentVersion,
	                Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations) {
		this->mostRecentVersion = mostRecentVersion;
		mutationHistory.insert(mutationHistory.end(), changes.begin(), changes.end());
		annotationHistory.insert(annotationHistory.end(), annotations.begin(), annotations.end());
		for (const auto& change : changes) {
			const auto& mutation = change.mutation;
			if (mutation.isSet()) {
				snapshot[mutation.getKey()] = mutation.getValue();
			} else {
				snapshot.erase(mutation.getKey());
			}
		}

		for (auto& [id, client] : clients) {
			actors.add(brokenPromiseToNever(pushChanges(client, changes)));
		}
	}

																#line 213 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
// This generated class is to be used only via pushSnapshotAndChanges()
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
template <class PushSnapshotAndChangesActor>
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class PushSnapshotAndChangesActorState {
															#line 219 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	PushSnapshotAndChangesActorState(ConfigBroadcasterImpl* const& self,Version const& snapshotVersion) 
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		 : self(self),
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   snapshotVersion(snapshotVersion)
															#line 228 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
	{
		fdb_probe_actor_create("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this));

	}
	~PushSnapshotAndChangesActorState() 
	{
		fdb_probe_actor_destroy("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 212 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			std::vector<Future<Void>> futures;
															#line 213 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			for( const auto& [id, client] : self->clients ) {
															#line 214 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				futures.push_back(brokenPromiseToNever(self->pushSnapshot(snapshotVersion, client)));
															#line 247 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			}
															#line 216 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(futures);
															#line 216 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<PushSnapshotAndChangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 253 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PushSnapshotAndChangesActor*>(this)->actor_wait_state = 1;
															#line 216 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PushSnapshotAndChangesActor, 0, Void >*>(static_cast<PushSnapshotAndChangesActor*>(this)));
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PushSnapshotAndChangesActorState();
		static_cast<PushSnapshotAndChangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 217 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<PushSnapshotAndChangesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PushSnapshotAndChangesActorState(); static_cast<PushSnapshotAndChangesActor*>(this)->destroy(); return 0; }
															#line 281 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<PushSnapshotAndChangesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PushSnapshotAndChangesActorState();
		static_cast<PushSnapshotAndChangesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 217 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<PushSnapshotAndChangesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PushSnapshotAndChangesActorState(); static_cast<PushSnapshotAndChangesActor*>(this)->destroy(); return 0; }
															#line 293 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<PushSnapshotAndChangesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PushSnapshotAndChangesActorState();
		static_cast<PushSnapshotAndChangesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PushSnapshotAndChangesActor*>(this)->actor_wait_state > 0) static_cast<PushSnapshotAndChangesActor*>(this)->actor_wait_state = 0;
		static_cast<PushSnapshotAndChangesActor*>(this)->ActorCallback< PushSnapshotAndChangesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PushSnapshotAndChangesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PushSnapshotAndChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PushSnapshotAndChangesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PushSnapshotAndChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PushSnapshotAndChangesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PushSnapshotAndChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcasterImpl* self;
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	Version snapshotVersion;
															#line 380 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
};
// This generated class is to be used only via pushSnapshotAndChanges()
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class PushSnapshotAndChangesActor final : public Actor<Void>, public ActorCallback< PushSnapshotAndChangesActor, 0, Void >, public FastAllocated<PushSnapshotAndChangesActor>, public PushSnapshotAndChangesActorState<PushSnapshotAndChangesActor> {
															#line 385 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
	using FastAllocated<PushSnapshotAndChangesActor>::operator new;
	using FastAllocated<PushSnapshotAndChangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(665621622669583872UL, 5402410552692811264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PushSnapshotAndChangesActor, 0, Void >;
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	PushSnapshotAndChangesActor(ConfigBroadcasterImpl* const& self,Version const& snapshotVersion) 
															#line 402 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		 : Actor<Void>(),
		   PushSnapshotAndChangesActorState<PushSnapshotAndChangesActor>(self, snapshotVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(315280530558573312UL, 1591422952291461376UL);
		ActorExecutionContextHelper __helper(static_cast<PushSnapshotAndChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("pushSnapshotAndChanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("pushSnapshotAndChanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PushSnapshotAndChangesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
[[nodiscard]] static Future<Void> pushSnapshotAndChanges( ConfigBroadcasterImpl* const& self, Version const& snapshotVersion ) {
															#line 211 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	return Future<Void>(new PushSnapshotAndChangesActor(self, snapshotVersion));
															#line 434 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
}

#line 219 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"

																#line 439 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
// This generated class is to be used only via waitForFailure()
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
template <class WaitForFailureActor>
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class WaitForFailureActorState {
															#line 445 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	WaitForFailureActorState(ConfigBroadcasterImpl* const& self,Future<Void> const& watcher,UID const& clientUID,NetworkAddress const& clientAddress,bool const& isCoordinator) 
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		 : self(self),
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   watcher(watcher),
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   clientUID(clientUID),
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   clientAddress(clientAddress),
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   isCoordinator(isCoordinator)
															#line 460 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForFailure", reinterpret_cast<unsigned long>(this));

	}
	~WaitForFailureActorState() 
	{
		fdb_probe_actor_destroy("waitForFailure", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 225 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_0 = watcher;
															#line 225 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<WaitForFailureActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 477 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitForFailureActor*>(this)->actor_wait_state = 1;
															#line 225 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForFailureActor, 0, Void >*>(static_cast<WaitForFailureActor*>(this)));
															#line 482 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForFailureActorState();
		static_cast<WaitForFailureActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 226 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent(SevDebug, "ConfigBroadcastClientDied", self->id) .detail("ClientID", clientUID) .detail("Address", clientAddress) .detail("IsUnregistered", self->registrationResponsesUnregistered.find(clientAddress) != self->registrationResponsesUnregistered.end()) .detail("IsActive", self->activeConfigNodes.find(clientAddress) != self->activeConfigNodes.end());
															#line 233 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->clients.erase(clientUID);
															#line 234 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->clientFailures.erase(clientUID);
															#line 235 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (isCoordinator)
															#line 511 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 236 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->registrationResponses.erase(clientAddress);
															#line 237 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (self->activeConfigNodes.find(clientAddress) != self->activeConfigNodes.end())
															#line 517 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 238 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->activeConfigNodes.erase(clientAddress);
															#line 239 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (self->registrationResponsesUnregistered.find(clientAddress) != self->registrationResponsesUnregistered.end())
															#line 523 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 241 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->registrationResponsesUnregistered.erase(clientAddress);
															#line 242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->disallowUnregistered = false;
															#line 244 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					if (self->newConfigNodesAllowed.isSet())
															#line 531 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
					{
															#line 245 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
						self->newConfigNodesAllowed.reset();
															#line 535 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
					}
				}
			}
		}
															#line 250 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<WaitForFailureActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitForFailureActorState(); static_cast<WaitForFailureActor*>(this)->destroy(); return 0; }
															#line 542 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<WaitForFailureActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitForFailureActorState();
		static_cast<WaitForFailureActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 226 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent(SevDebug, "ConfigBroadcastClientDied", self->id) .detail("ClientID", clientUID) .detail("Address", clientAddress) .detail("IsUnregistered", self->registrationResponsesUnregistered.find(clientAddress) != self->registrationResponsesUnregistered.end()) .detail("IsActive", self->activeConfigNodes.find(clientAddress) != self->activeConfigNodes.end());
															#line 233 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->clients.erase(clientUID);
															#line 234 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->clientFailures.erase(clientUID);
															#line 235 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (isCoordinator)
															#line 560 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 236 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->registrationResponses.erase(clientAddress);
															#line 237 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (self->activeConfigNodes.find(clientAddress) != self->activeConfigNodes.end())
															#line 566 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 238 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->activeConfigNodes.erase(clientAddress);
															#line 239 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (self->registrationResponsesUnregistered.find(clientAddress) != self->registrationResponsesUnregistered.end())
															#line 572 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 241 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->registrationResponsesUnregistered.erase(clientAddress);
															#line 242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->disallowUnregistered = false;
															#line 244 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					if (self->newConfigNodesAllowed.isSet())
															#line 580 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
					{
															#line 245 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
						self->newConfigNodesAllowed.reset();
															#line 584 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
					}
				}
			}
		}
															#line 250 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<WaitForFailureActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitForFailureActorState(); static_cast<WaitForFailureActor*>(this)->destroy(); return 0; }
															#line 591 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<WaitForFailureActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitForFailureActorState();
		static_cast<WaitForFailureActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForFailureActor*>(this)->actor_wait_state > 0) static_cast<WaitForFailureActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFailureActor*>(this)->ActorCallback< WaitForFailureActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFailureActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForFailureActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForFailureActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFailure", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcasterImpl* self;
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	Future<Void> watcher;
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	UID clientUID;
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	NetworkAddress clientAddress;
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	bool isCoordinator;
															#line 684 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
};
// This generated class is to be used only via waitForFailure()
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class WaitForFailureActor final : public Actor<Void>, public ActorCallback< WaitForFailureActor, 0, Void >, public FastAllocated<WaitForFailureActor>, public WaitForFailureActorState<WaitForFailureActor> {
															#line 689 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
	using FastAllocated<WaitForFailureActor>::operator new;
	using FastAllocated<WaitForFailureActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5921495865670666240UL, 11975344338996323840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForFailureActor, 0, Void >;
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	WaitForFailureActor(ConfigBroadcasterImpl* const& self,Future<Void> const& watcher,UID const& clientUID,NetworkAddress const& clientAddress,bool const& isCoordinator) 
															#line 706 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		 : Actor<Void>(),
		   WaitForFailureActorState<WaitForFailureActor>(self, watcher, clientUID, clientAddress, isCoordinator),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForFailure", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16431399585899188480UL, 7412607704015144448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForFailure");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForFailure", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForFailureActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
[[nodiscard]] static Future<Void> waitForFailure( ConfigBroadcasterImpl* const& self, Future<Void> const& watcher, UID const& clientUID, NetworkAddress const& clientAddress, bool const& isCoordinator ) {
															#line 220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	return Future<Void>(new WaitForFailureActor(self, watcher, clientUID, clientAddress, isCoordinator));
															#line 738 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
}

#line 252 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"

	// Determines whether the registering ConfigNode is allowed to start
	// serving configuration database requests and snapshot data. In order to
	// ensure strict serializability, some nodes may be temporarily restricted
	// from participation until the other nodes in the system are brought up to
	// date.
																#line 748 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
// This generated class is to be used only via registerNodeInternal()
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
template <class RegisterNodeInternalActor>
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class RegisterNodeInternalActorState {
															#line 754 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	RegisterNodeInternalActorState(ConfigBroadcaster* const& broadcaster,ConfigBroadcasterImpl* const& self,ConfigBroadcastInterface const& configBroadcastInterface) 
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		 : broadcaster(broadcaster),
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   self(self),
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   configBroadcastInterface(configBroadcastInterface)
															#line 765 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
	{
		fdb_probe_actor_create("registerNodeInternal", reinterpret_cast<unsigned long>(this));

	}
	~RegisterNodeInternalActorState() 
	{
		fdb_probe_actor_destroy("registerNodeInternal", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 261 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (self->configDBType == ConfigDBType::SIMPLE)
															#line 780 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 262 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->consumerFuture = self->consumer->consume(*broadcaster);
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				StrictFuture<Void> __when_expr_0 = success(brokenPromiseToNever( configBroadcastInterface.ready.getReply(ConfigBroadcastReadyRequest{ 0, {}, -1, -1 })));
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 788 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 1;
															#line 263 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 0, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 793 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterNodeInternalActorState();
		static_cast<RegisterNodeInternalActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 268 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		address = configBroadcastInterface.address();
															#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		StrictFuture<ConfigBroadcastRegisteredReply> __when_expr_1 = brokenPromiseToNever(configBroadcastInterface.registered.getReply(ConfigBroadcastRegisteredRequest{}));
															#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 825 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 2;
															#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 830 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 265 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<RegisterNodeInternalActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeInternalActorState(); static_cast<RegisterNodeInternalActor*>(this)->destroy(); return 0; }
															#line 839 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<RegisterNodeInternalActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterNodeInternalActorState();
		static_cast<RegisterNodeInternalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 265 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<RegisterNodeInternalActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeInternalActorState(); static_cast<RegisterNodeInternalActor*>(this)->destroy(); return 0; }
															#line 851 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<RegisterNodeInternalActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterNodeInternalActorState();
		static_cast<RegisterNodeInternalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 272 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->largestConfigNodeVersion = std::max(self->largestConfigNodeVersion, reply.lastSeenVersion);
															#line 273 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		registered = reply.registered;
															#line 274 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent("ConfigBroadcasterRegisterNodeReceivedRegistrationReply", self->id) .detail("Address", address) .detail("Registered", registered) .detail("DisallowUnregistered", self->disallowUnregistered) .detail("LastSeenVersion", reply.lastSeenVersion);
															#line 280 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (self->activeConfigNodes.find(address) != self->activeConfigNodes.end())
															#line 944 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 281 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->activeConfigNodes.erase(address);
															#line 282 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (self->registrationResponsesUnregistered.find(address) != self->registrationResponsesUnregistered.end())
															#line 950 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 284 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->registrationResponsesUnregistered.erase(address);
															#line 288 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->disallowUnregistered = false;
															#line 297 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (self->newConfigNodesAllowed.isSet())
															#line 958 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 298 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->newConfigNodesAllowed.reset();
															#line 962 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
			}
		}
															#line 302 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		int responsesRemaining = self->coordinators - (int)self->registrationResponses.size();
															#line 303 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		int nodesTillQuorum = self->coordinators / 2 + 1 - (int)self->activeConfigNodes.size();
															#line 305 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (registered)
															#line 972 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 306 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->registeredConfigNodes.insert(address);
															#line 307 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->activeConfigNodes.insert(address);
															#line 308 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->disallowUnregistered = true;
															#line 980 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		}
		else
		{
															#line 309 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if ((self->activeConfigNodes.size() < self->coordinators / 2 + 1 && !self->disallowUnregistered) || (self->registrationResponsesUnregistered.size() < self->coordinators / 2 && responsesRemaining <= nodesTillQuorum))
															#line 986 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 323 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->activeConfigNodes.insert(address);
															#line 324 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->registrationResponsesUnregistered.insert(address);
															#line 325 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if ((self->activeConfigNodes.size() >= self->coordinators / 2 + 1 || self->registrationResponsesUnregistered.size() >= self->coordinators / 2 + 1) && self->newConfigNodesAllowed.canBeSet())
															#line 994 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 328 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->newConfigNodesAllowed.send(Void());
															#line 998 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
			}
			else
			{
															#line 331 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->disallowUnregistered = true;
															#line 1005 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			}
		}
															#line 333 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		self->registrationResponses.insert(address);
															#line 338 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield();
															#line 338 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1014 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 3;
															#line 338 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 2, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1019 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(ConfigBroadcastRegisteredReply const& __reply,int loopDepth) 
	{
															#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		reply = __reply;
															#line 1028 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ConfigBroadcastRegisteredReply && __reply,int loopDepth) 
	{
		reply = std::move(__reply);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >*,ConfigBroadcastRegisteredReply const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >*,ConfigBroadcastRegisteredReply && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_3 = self->previousCoordinatorsFuture;
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1109 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 4;
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1114 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_3 = self->previousCoordinatorsFuture;
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1125 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 4;
															#line 339 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1130 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont14(Optional<Value> const& previousCoordinators,int loopDepth) 
	{
															#line 340 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent("ConfigBroadcasterRegisterNodeReadPreviousCoordinators", self->id) .detail("PreviousCoordinators", previousCoordinators) .detail("HasStartedConsumer", self->previousCoordinatorsSnapshotFuture.isValid());
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (previousCoordinators.present())
															#line 1216 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 345 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (!self->previousCoordinatorsSnapshotFuture.isValid())
															#line 1220 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 349 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				size_t previousCoordinatorsHash = std::hash<std::string>()(previousCoordinators.get().toString());
															#line 350 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (previousCoordinatorsHash != self->coordinatorsHash)
															#line 1226 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 351 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					ServerCoordinators previousCoordinatorsData(Reference<IClusterConnectionRecord>( new ClusterConnectionMemoryRecord(previousCoordinators.get().toString())));
															#line 353 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					TraceEvent("ConfigBroadcasterRegisterNodeStartingConsumer", self->id).log();
															#line 354 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsConsumer = IConfigConsumer::createPaxos( previousCoordinatorsData, 0.5, SERVER_KNOBS->COMPACTION_INTERVAL, true);
															#line 356 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsSnapshotFuture = self->previousCoordinatorsConsumer->readSnapshot(*broadcaster);
															#line 1236 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
				else
				{
															#line 362 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsSnapshotFuture = Void();
															#line 1242 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
			}
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_4 = self->previousCoordinatorsSnapshotFuture;
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1249 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont14when1(__when_expr_4.get(), loopDepth); };
			static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 5;
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 4, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1254 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14(Optional<Value> && previousCoordinators,int loopDepth) 
	{
															#line 340 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent("ConfigBroadcasterRegisterNodeReadPreviousCoordinators", self->id) .detail("PreviousCoordinators", previousCoordinators) .detail("HasStartedConsumer", self->previousCoordinatorsSnapshotFuture.isValid());
															#line 344 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (previousCoordinators.present())
															#line 1270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 345 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (!self->previousCoordinatorsSnapshotFuture.isValid())
															#line 1274 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 349 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				size_t previousCoordinatorsHash = std::hash<std::string>()(previousCoordinators.get().toString());
															#line 350 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (previousCoordinatorsHash != self->coordinatorsHash)
															#line 1280 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 351 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					ServerCoordinators previousCoordinatorsData(Reference<IClusterConnectionRecord>( new ClusterConnectionMemoryRecord(previousCoordinators.get().toString())));
															#line 353 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					TraceEvent("ConfigBroadcasterRegisterNodeStartingConsumer", self->id).log();
															#line 354 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsConsumer = IConfigConsumer::createPaxos( previousCoordinatorsData, 0.5, SERVER_KNOBS->COMPACTION_INTERVAL, true);
															#line 356 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsSnapshotFuture = self->previousCoordinatorsConsumer->readSnapshot(*broadcaster);
															#line 1290 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
				else
				{
															#line 362 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					self->previousCoordinatorsSnapshotFuture = Void();
															#line 1296 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
			}
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_4 = self->previousCoordinatorsSnapshotFuture;
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1303 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont14when1(__when_expr_4.get(), loopDepth); };
			static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 5;
															#line 365 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 4, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1308 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> const& previousCoordinators,int loopDepth) 
	{
		loopDepth = a_body1cont14(previousCoordinators, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> && previousCoordinators,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(previousCoordinators), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont14cont1(int loopDepth) 
	{
															#line 368 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		sendSnapshot = self->previousCoordinatorsConsumer && reply.lastSeenVersion <= self->mostRecentVersion;
															#line 411 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (self->previousCoordinatorsConsumer && self->largestConfigNodeVersion > self->mostRecentVersion && self->registeredConfigNodes.size() >= self->coordinators / 2 + 1)
															#line 1399 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 413 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			sendSnapshot = false;
															#line 1403 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		}
															#line 424 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent("ConfigBroadcasterRegisterNodeDetermineEligibility", self->id) .detail("Registered", registered) .detail("SendSnapshot", sendSnapshot);
															#line 427 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!registered && !sendSnapshot)
															#line 1409 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 428 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_5 = self->newConfigNodesAllowed.getFuture();
															#line 428 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1415 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont14cont1when1(__when_expr_5.get(), loopDepth); };
			static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 6;
															#line 428 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 5, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1420 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont14cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont14when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont14when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont14cont6(int loopDepth) 
	{
															#line 431 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		sendSnapshot = sendSnapshot || (!registered && self->snapshot.size() > 0);
															#line 432 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		TraceEvent("ConfigBroadcasterRegisterNodeSendingReadyRequest", self->id) .detail("ConfigNodeAddress", address) .detail("SendSnapshot", sendSnapshot) .detail("SnapshotVersion", self->mostRecentVersion) .detail("SnapshotSize", self->snapshot.size()) .detail("LargestLiveVersion", self->largestLiveVersion);
															#line 438 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (sendSnapshot)
															#line 1525 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 439 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			Version liveVersion = std::max(self->largestLiveVersion, self->mostRecentVersion);
															#line 440 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_6 = success(brokenPromiseToNever(configBroadcastInterface.ready.getReply(ConfigBroadcastReadyRequest{ self->coordinatorsHash, self->snapshot, self->mostRecentVersion, liveVersion })));
															#line 440 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1533 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont14cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 7;
															#line 440 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 6, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1538 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 443 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_7 = success(brokenPromiseToNever(configBroadcastInterface.ready.getReply( ConfigBroadcastReadyRequest{ self->coordinatorsHash, {}, -1, -1 })));
															#line 443 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1547 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont14cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 8;
															#line 443 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeInternalActor, 7, Void >*>(static_cast<RegisterNodeInternalActor*>(this)));
															#line 1552 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont14cont9(int loopDepth) 
	{
															#line 448 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!self->consumerFuture.isValid())
															#line 1649 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		{
															#line 449 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (sendSnapshot)
															#line 1653 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 450 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				self->consumer->allowSpecialCaseRollforward();
															#line 1657 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			}
															#line 452 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			self->consumerFuture = self->consumer->consume(*broadcaster);
															#line 1661 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		}
															#line 454 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<RegisterNodeInternalActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeInternalActorState(); static_cast<RegisterNodeInternalActor*>(this)->destroy(); return 0; }
															#line 1665 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<RegisterNodeInternalActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterNodeInternalActorState();
		static_cast<RegisterNodeInternalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont14cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont14cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeInternalActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeInternalActor*>(this)->ActorCallback< RegisterNodeInternalActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RegisterNodeInternalActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< RegisterNodeInternalActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcaster* broadcaster;
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcasterImpl* self;
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcastInterface configBroadcastInterface;
															#line 268 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	NetworkAddress address;
															#line 270 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcastRegisteredReply reply;
															#line 273 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	bool registered;
															#line 368 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	bool sendSnapshot;
															#line 1861 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
};
// This generated class is to be used only via registerNodeInternal()
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class RegisterNodeInternalActor final : public Actor<Void>, public ActorCallback< RegisterNodeInternalActor, 0, Void >, public ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >, public ActorCallback< RegisterNodeInternalActor, 2, Void >, public ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >, public ActorCallback< RegisterNodeInternalActor, 4, Void >, public ActorCallback< RegisterNodeInternalActor, 5, Void >, public ActorCallback< RegisterNodeInternalActor, 6, Void >, public ActorCallback< RegisterNodeInternalActor, 7, Void >, public FastAllocated<RegisterNodeInternalActor>, public RegisterNodeInternalActorState<RegisterNodeInternalActor> {
															#line 1866 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
	using FastAllocated<RegisterNodeInternalActor>::operator new;
	using FastAllocated<RegisterNodeInternalActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3448010923998316032UL, 14510536287853214464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterNodeInternalActor, 0, Void >;
friend struct ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >;
friend struct ActorCallback< RegisterNodeInternalActor, 2, Void >;
friend struct ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >;
friend struct ActorCallback< RegisterNodeInternalActor, 4, Void >;
friend struct ActorCallback< RegisterNodeInternalActor, 5, Void >;
friend struct ActorCallback< RegisterNodeInternalActor, 6, Void >;
friend struct ActorCallback< RegisterNodeInternalActor, 7, Void >;
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	RegisterNodeInternalActor(ConfigBroadcaster* const& broadcaster,ConfigBroadcasterImpl* const& self,ConfigBroadcastInterface const& configBroadcastInterface) 
															#line 1890 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		 : Actor<Void>(),
		   RegisterNodeInternalActorState<RegisterNodeInternalActor>(broadcaster, self, configBroadcastInterface),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerNodeInternal", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8354995016968414208UL, 8310047142978139136UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeInternalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerNodeInternal");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerNodeInternal", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 1, ConfigBroadcastRegisteredReply >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 3, Optional<Value> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< RegisterNodeInternalActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
[[nodiscard]] static Future<Void> registerNodeInternal( ConfigBroadcaster* const& broadcaster, ConfigBroadcasterImpl* const& self, ConfigBroadcastInterface const& configBroadcastInterface ) {
															#line 258 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	return Future<Void>(new RegisterNodeInternalActor(broadcaster, self, configBroadcastInterface));
															#line 1929 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
}

#line 456 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"

																#line 1934 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
// This generated class is to be used only via registerNode()
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
template <class RegisterNodeActor>
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class RegisterNodeActorState {
															#line 1940 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	RegisterNodeActorState(ConfigBroadcaster* const& self,ConfigBroadcasterImpl* const& impl,ConfigBroadcastInterface const& broadcastInterface,Version const& lastSeenVersion,ConfigClassSet const& configClassSet,Future<Void> const& watcher,bool const& isCoordinator) 
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		 : self(self),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   impl(impl),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   broadcastInterface(broadcastInterface),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   lastSeenVersion(lastSeenVersion),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   configClassSet(configClassSet),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   watcher(watcher),
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   isCoordinator(isCoordinator),
															#line 464 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		   client(watcher, std::move(configClassSet), lastSeenVersion, std::move(broadcastInterface))
															#line 1961 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
	{
		fdb_probe_actor_create("registerNode", reinterpret_cast<unsigned long>(this));

	}
	~RegisterNodeActorState() 
	{
		fdb_probe_actor_destroy("registerNode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 467 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (impl->clients.contains(broadcastInterface.id()))
															#line 1976 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 469 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (!static_cast<RegisterNodeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeActorState(); static_cast<RegisterNodeActor*>(this)->destroy(); return 0; }
															#line 1980 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				new (&static_cast<RegisterNodeActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~RegisterNodeActorState();
				static_cast<RegisterNodeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 472 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			TraceEvent(SevDebug, "ConfigBroadcasterRegisteringWorker", impl->id) .detail("ClientID", broadcastInterface.id()) .detail("MostRecentVersion", impl->mostRecentVersion) .detail("IsCoordinator", isCoordinator);
															#line 477 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (isCoordinator)
															#line 1990 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 480 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				impl->registrationActors[broadcastInterface.address()] = registerNodeInternal(self, impl, broadcastInterface);
															#line 1994 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			}
															#line 485 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_0 = impl->pushSnapshot(impl->mostRecentVersion, client);
															#line 485 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<RegisterNodeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2000 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RegisterNodeActor*>(this)->actor_wait_state = 1;
															#line 485 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterNodeActor, 0, Void >*>(static_cast<RegisterNodeActor*>(this)));
															#line 2005 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterNodeActorState();
		static_cast<RegisterNodeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		impl->clients[broadcastInterface.id()] = client;
															#line 487 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		impl->clientFailures[broadcastInterface.id()] = waitForFailure(impl, watcher, broadcastInterface.id(), broadcastInterface.address(), isCoordinator);
															#line 489 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<RegisterNodeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeActorState(); static_cast<RegisterNodeActor*>(this)->destroy(); return 0; }
															#line 2032 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<RegisterNodeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterNodeActorState();
		static_cast<RegisterNodeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 486 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		impl->clients[broadcastInterface.id()] = client;
															#line 487 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		impl->clientFailures[broadcastInterface.id()] = waitForFailure(impl, watcher, broadcastInterface.id(), broadcastInterface.address(), isCoordinator);
															#line 489 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<RegisterNodeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterNodeActorState(); static_cast<RegisterNodeActor*>(this)->destroy(); return 0; }
															#line 2048 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<RegisterNodeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterNodeActorState();
		static_cast<RegisterNodeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterNodeActor*>(this)->actor_wait_state > 0) static_cast<RegisterNodeActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterNodeActor*>(this)->ActorCallback< RegisterNodeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterNodeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerNode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterNodeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerNode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterNodeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerNode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerNode", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcaster* self;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcasterImpl* impl;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigBroadcastInterface broadcastInterface;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	Version lastSeenVersion;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ConfigClassSet configClassSet;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	Future<Void> watcher;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	bool isCoordinator;
															#line 464 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	BroadcastClientDetails client;
															#line 2147 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
};
// This generated class is to be used only via registerNode()
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class RegisterNodeActor final : public Actor<Void>, public ActorCallback< RegisterNodeActor, 0, Void >, public FastAllocated<RegisterNodeActor>, public RegisterNodeActorState<RegisterNodeActor> {
															#line 2152 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
	using FastAllocated<RegisterNodeActor>::operator new;
	using FastAllocated<RegisterNodeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7188048280736166400UL, 9419924932264730880UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterNodeActor, 0, Void >;
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	RegisterNodeActor(ConfigBroadcaster* const& self,ConfigBroadcasterImpl* const& impl,ConfigBroadcastInterface const& broadcastInterface,Version const& lastSeenVersion,ConfigClassSet const& configClassSet,Future<Void> const& watcher,bool const& isCoordinator) 
															#line 2169 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		 : Actor<Void>(),
		   RegisterNodeActorState<RegisterNodeActor>(self, impl, broadcastInterface, lastSeenVersion, configClassSet, watcher, isCoordinator),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerNode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(436462263414743296UL, 10319653636521590016UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterNodeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerNode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerNode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterNodeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
[[nodiscard]] static Future<Void> registerNode( ConfigBroadcaster* const& self, ConfigBroadcasterImpl* const& impl, ConfigBroadcastInterface const& broadcastInterface, Version const& lastSeenVersion, ConfigClassSet const& configClassSet, Future<Void> const& watcher, bool const& isCoordinator ) {
															#line 457 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	return Future<Void>(new RegisterNodeActor(self, impl, broadcastInterface, lastSeenVersion, configClassSet, watcher, isCoordinator));
															#line 2201 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
}

#line 491 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"

public:
	Future<Void> registerNode(ConfigBroadcaster& self,
	                          ConfigBroadcastInterface const& broadcastInterface,
	                          Version lastSeenVersion,
	                          ConfigClassSet configClassSet,
	                          Future<Void> watcher,
	                          bool isCoordinator) {
		return registerNode(&self, this, broadcastInterface, lastSeenVersion, configClassSet, watcher, isCoordinator);
	}

	// Updates the broadcasters knowledge of which replicas are fully up to
	// date, based on data gathered by the consumer.
	void updateKnownReplicas(std::vector<ConfigFollowerInterface> const& readReplicas) {
		if (!newConfigNodesAllowed.canBeSet()) {
			return;
		}

		for (const auto& cfi : readReplicas) {
			this->activeConfigNodes.insert(cfi.address());
		}
		if (activeConfigNodes.size() >= coordinators / 2 + 1) {
			disallowUnregistered = true;
			newConfigNodesAllowed.send(Void());
		}
	}

	void applyChanges(Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
	                  Version mostRecentVersion,
	                  Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations,
	                  std::vector<ConfigFollowerInterface> const& readReplicas) {
		if (mostRecentVersion >= 0) {
			TraceEvent(SevInfo, "ConfigBroadcasterApplyingChanges", id)
			    .detail("ChangesSize", changes.size())
			    .detail("AnnotationsSize", annotations.size())
			    .detail("CurrentMostRecentVersion", this->mostRecentVersion)
			    .detail("NewMostRecentVersion", mostRecentVersion)
			    .detail("ActiveReplicas", readReplicas.size());
			addChanges(changes, mostRecentVersion, annotations);
		}

		updateKnownReplicas(readReplicas);
	}

	template <class Snapshot>
	void applySnapshotAndChanges(Snapshot&& snapshot,
	                             Version snapshotVersion,
	                             Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
	                             Version changesVersion,
	                             Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations,
	                             std::vector<ConfigFollowerInterface> const& readReplicas,
	                             Version largestLiveVersion,
	                             bool fromPreviousCoordinators) {
		TraceEvent(SevInfo, "ConfigBroadcasterApplyingSnapshotAndChanges", id)
		    .detail("CurrentMostRecentVersion", this->mostRecentVersion)
		    .detail("SnapshotSize", snapshot.size())
		    .detail("SnapshotVersion", snapshotVersion)
		    .detail("ChangesSize", changes.size())
		    .detail("ChangesVersion", changesVersion)
		    .detail("AnnotationsSize", annotations.size())
		    .detail("ActiveReplicas", readReplicas.size())
		    .detail("LargestLiveVersion", largestLiveVersion)
		    .detail("FromPreviousCoordinators", fromPreviousCoordinators);
		// Avoid updating state if the snapshot contains no mutations, or if it
		// contains old mutations. This can happen when the set of coordinators
		// is changed, and a new coordinator comes online that has not yet had
		// the current configuration database pushed to it, or when a new
		// coordinator contains state from an old configuration database
		// generation.
		if ((snapshot.size() != 0 || changes.size() != 0) &&
		    (snapshotVersion > this->mostRecentVersion || changesVersion > this->mostRecentVersion)) {
			this->snapshot = std::forward<Snapshot>(snapshot);
			this->lastCompactedVersion = snapshotVersion;
			this->largestLiveVersion = std::max(this->largestLiveVersion, largestLiveVersion);
			addChanges(changes, changesVersion, annotations);
			actors.add(pushSnapshotAndChanges(this, snapshotVersion));
		}

		if (!fromPreviousCoordinators) {
			updateKnownReplicas(readReplicas);
		}
	}

	ConfigBroadcasterImpl(ConfigFollowerInterface const& cfi) : ConfigBroadcasterImpl() {
		configDBType = ConfigDBType::SIMPLE;
		coordinators = 1;
		consumer = IConfigConsumer::createTestSimple(cfi, 0.5, Optional<double>{});
		TraceEvent(SevDebug, "ConfigBroadcasterStartingConsumer", id).detail("Consumer", consumer->getID());
	}

	ConfigBroadcasterImpl(ServerCoordinators const& coordinators,
	                      ConfigDBType configDBType,
	                      Future<Optional<Value>> previousCoordinatorsFuture)
	  : ConfigBroadcasterImpl() {
		this->configDBType = configDBType;
		this->coordinators = coordinators.configServers.size();
		if (configDBType != ConfigDBType::DISABLED) {
			if (configDBType == ConfigDBType::SIMPLE) {
				consumer = IConfigConsumer::createSimple(coordinators, 0.5, SERVER_KNOBS->COMPACTION_INTERVAL);
			} else {
				this->previousCoordinatorsFuture = previousCoordinatorsFuture;
				consumer = IConfigConsumer::createPaxos(coordinators, 0.5, SERVER_KNOBS->COMPACTION_INTERVAL);
			}

			coordinatorsHash = std::hash<std::string>()(coordinators.ccr->getConnectionString().toString());

			TraceEvent(SevInfo, "ConfigBroadcasterStartingConsumer", id)
			    .detail("Consumer", consumer->getID())
			    .detail("UsingSimpleConsumer", configDBType == ConfigDBType::SIMPLE)
			    .detail("CoordinatorsCount", this->coordinators)
			    .detail("CoordinatorsHash", coordinatorsHash)
			    .detail("CompactionInterval", SERVER_KNOBS->COMPACTION_INTERVAL);
		}
	}

	JsonBuilderObject getStatus() const {
		JsonBuilderObject result;
		JsonBuilderArray mutationsArray;
		for (const auto& versionedMutation : mutationHistory) {
			JsonBuilderObject mutationObject;
			mutationObject["version"] = versionedMutation.version;
			const auto& mutation = versionedMutation.mutation;
			mutationObject["type"] = mutation.isSet() ? "set" : "clear";
			mutationObject["config_class"] = mutation.getConfigClass().orDefault("<global>"_sr);
			mutationObject["knob_name"] = mutation.getKnobName();
			if (mutation.isSet()) {
				mutationObject["knob_value"] = mutation.getValue().toString();
			}
			mutationsArray.push_back(std::move(mutationObject));
		}
		result["mutations"] = std::move(mutationsArray);
		JsonBuilderArray commitsArray;
		for (const auto& versionedAnnotation : annotationHistory) {
			JsonBuilderObject commitObject;
			commitObject["version"] = versionedAnnotation.version;
			commitObject["description"] = versionedAnnotation.annotation.description;
			commitObject["timestamp"] = versionedAnnotation.annotation.timestamp;
			commitsArray.push_back(std::move(commitObject));
		}
		result["commits"] = std::move(commitsArray);
		JsonBuilderObject snapshotObject;
		std::map<Optional<Key>, std::vector<std::pair<Key, Value>>> snapshotMap;
		for (const auto& [configKey, value] : snapshot) {
			snapshotMap[configKey.configClass.castTo<Key>()].emplace_back(configKey.knobName, value.toString());
		}
		for (const auto& [configClass, kvs] : snapshotMap) {
			JsonBuilderObject kvsObject;
			for (const auto& [knobName, knobValue] : kvs) {
				kvsObject[knobName] = knobValue;
			}
			snapshotObject[configClass.orDefault("<global>"_sr)] = std::move(kvsObject);
		}
		result["snapshot"] = std::move(snapshotObject);
		result["last_compacted_version"] = lastCompactedVersion;
		result["most_recent_version"] = mostRecentVersion;
		return result;
	}

	void compact(Version compactionVersion) {
		{
			auto it = std::find_if(mutationHistory.begin(), mutationHistory.end(), [compactionVersion](const auto& vm) {
				return vm.version > compactionVersion;
			});
			mutationHistory.erase(mutationHistory.begin(), it);
		}
		{
			auto it = std::find_if(annotationHistory.begin(),
			                       annotationHistory.end(),
			                       [compactionVersion](const auto& va) { return va.version > compactionVersion; });
			annotationHistory.erase(annotationHistory.begin(), it);
		}
		lastCompactedVersion = compactionVersion;
	}

	Future<Void> getError() const { return consumerFuture || actors.getResult(); }

	Future<Void> getClientFailure(UID clientUID) const { return clientFailures.find(clientUID)->second; }

	UID getID() const { return id; }

	static void runPendingRequestStoreTest(bool includeGlobalMutation, int expectedMatches);
};

ConfigBroadcaster::ConfigBroadcaster() {}

ConfigBroadcaster::ConfigBroadcaster(ConfigFollowerInterface const& cfi)
  : impl(PImpl<ConfigBroadcasterImpl>::create(cfi)) {}

ConfigBroadcaster::ConfigBroadcaster(ServerCoordinators const& coordinators,
                                     ConfigDBType configDBType,
                                     Future<Optional<Value>> previousCoordinatorsFuture)
  : impl(PImpl<ConfigBroadcasterImpl>::create(coordinators, configDBType, previousCoordinatorsFuture)) {}

ConfigBroadcaster::ConfigBroadcaster(ConfigBroadcaster&&) = default;

ConfigBroadcaster& ConfigBroadcaster::operator=(ConfigBroadcaster&&) = default;

ConfigBroadcaster::~ConfigBroadcaster() = default;

Future<Void> ConfigBroadcaster::registerNode(ConfigBroadcastInterface const& broadcastInterface,
                                             Version lastSeenVersion,
                                             ConfigClassSet const& configClassSet,
                                             Future<Void> watcher,
                                             bool isCoordinator) {
	return impl->registerNode(*this, broadcastInterface, lastSeenVersion, configClassSet, watcher, isCoordinator);
}

void ConfigBroadcaster::applyChanges(Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
                                     Version mostRecentVersion,
                                     Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations,
                                     std::vector<ConfigFollowerInterface> const& readReplicas) {
	impl->applyChanges(changes, mostRecentVersion, annotations, readReplicas);
}

void ConfigBroadcaster::applySnapshotAndChanges(
    std::map<ConfigKey, KnobValue> const& snapshot,
    Version snapshotVersion,
    Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
    Version changesVersion,
    Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations,
    std::vector<ConfigFollowerInterface> const& readReplicas,
    Version largestLiveVersion,
    bool fromPreviousCoordinators) {
	impl->applySnapshotAndChanges(snapshot,
	                              snapshotVersion,
	                              changes,
	                              changesVersion,
	                              annotations,
	                              readReplicas,
	                              largestLiveVersion,
	                              fromPreviousCoordinators);
}

void ConfigBroadcaster::applySnapshotAndChanges(
    std::map<ConfigKey, KnobValue>&& snapshot,
    Version snapshotVersion,
    Standalone<VectorRef<VersionedConfigMutationRef>> const& changes,
    Version changesVersion,
    Standalone<VectorRef<VersionedConfigCommitAnnotationRef>> const& annotations,
    std::vector<ConfigFollowerInterface> const& readReplicas,
    Version largestLiveVersion,
    bool fromPreviousCoordinators) {
	impl->applySnapshotAndChanges(std::move(snapshot),
	                              snapshotVersion,
	                              changes,
	                              changesVersion,
	                              annotations,
	                              readReplicas,
	                              largestLiveVersion,
	                              fromPreviousCoordinators);
}

Future<Void> ConfigBroadcaster::getError() const {
	return impl->getError();
}

Future<Void> ConfigBroadcaster::getClientFailure(UID clientUID) const {
	return impl->getClientFailure(clientUID);
}

UID ConfigBroadcaster::getID() const {
	return impl->getID();
}

JsonBuilderObject ConfigBroadcaster::getStatus() const {
	return impl->getStatus();
}

void ConfigBroadcaster::compact(Version compactionVersion) {
	impl->compact(compactionVersion);
}

															#line 2477 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
namespace {
// This generated class is to be used only via lockConfigNodesImpl()
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
template <class LockConfigNodesImplActor>
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class LockConfigNodesImplActorState {
															#line 2484 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	LockConfigNodesImplActorState(ServerCoordinators const& coordinators) 
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		 : coordinators(coordinators)
															#line 2491 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
	{
		fdb_probe_actor_create("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this));

	}
	~LockConfigNodesImplActorState() 
	{
		fdb_probe_actor_destroy("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 764 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (coordinators.configServers.empty())
															#line 2506 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			{
															#line 765 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (!static_cast<LockConfigNodesImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockConfigNodesImplActorState(); static_cast<LockConfigNodesImplActor*>(this)->destroy(); return 0; }
															#line 2510 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				new (&static_cast<LockConfigNodesImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LockConfigNodesImplActorState();
				static_cast<LockConfigNodesImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 768 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			CoordinatorsHash coordinatorsHash = std::hash<std::string>()(coordinators.ccr->getConnectionString().toString());
															#line 770 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			std::vector<Future<Void>> lockRequests;
															#line 771 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			lockRequests.reserve(coordinators.configServers.size());
															#line 772 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			for(int i = 0;i < coordinators.configServers.size();i++) {
															#line 773 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
				if (coordinators.configServers[i].hostname.present())
															#line 2526 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				{
															#line 774 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					lockRequests.push_back(retryGetReplyFromHostname(ConfigFollowerLockRequest{ coordinatorsHash }, coordinators.configServers[i].hostname.get(), WLTOKEN_CONFIGFOLLOWER_LOCK));
															#line 2530 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
				else
				{
															#line 778 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
					lockRequests.push_back( retryBrokenPromise(coordinators.configServers[i].lock, ConfigFollowerLockRequest{ coordinatorsHash }));
															#line 2536 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
				}
			}
															#line 782 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			int quorum_size = lockRequests.size() / 2 + 1;
															#line 783 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			StrictFuture<Void> __when_expr_0 = quorum(lockRequests, quorum_size);
															#line 783 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			if (static_cast<LockConfigNodesImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2545 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LockConfigNodesImplActor*>(this)->actor_wait_state = 1;
															#line 783 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LockConfigNodesImplActor, 0, Void >*>(static_cast<LockConfigNodesImplActor*>(this)));
															#line 2550 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LockConfigNodesImplActorState();
		static_cast<LockConfigNodesImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 784 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<LockConfigNodesImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockConfigNodesImplActorState(); static_cast<LockConfigNodesImplActor*>(this)->destroy(); return 0; }
															#line 2573 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<LockConfigNodesImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LockConfigNodesImplActorState();
		static_cast<LockConfigNodesImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 784 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
		if (!static_cast<LockConfigNodesImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockConfigNodesImplActorState(); static_cast<LockConfigNodesImplActor*>(this)->destroy(); return 0; }
															#line 2585 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		new (&static_cast<LockConfigNodesImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LockConfigNodesImplActorState();
		static_cast<LockConfigNodesImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LockConfigNodesImplActor*>(this)->actor_wait_state > 0) static_cast<LockConfigNodesImplActor*>(this)->actor_wait_state = 0;
		static_cast<LockConfigNodesImplActor*>(this)->ActorCallback< LockConfigNodesImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LockConfigNodesImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LockConfigNodesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LockConfigNodesImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LockConfigNodesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LockConfigNodesImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LockConfigNodesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	ServerCoordinators coordinators;
															#line 2670 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
};
// This generated class is to be used only via lockConfigNodesImpl()
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
class LockConfigNodesImplActor final : public Actor<Void>, public ActorCallback< LockConfigNodesImplActor, 0, Void >, public FastAllocated<LockConfigNodesImplActor>, public LockConfigNodesImplActorState<LockConfigNodesImplActor> {
															#line 2675 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
public:
	using FastAllocated<LockConfigNodesImplActor>::operator new;
	using FastAllocated<LockConfigNodesImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9275625878409219840UL, 15691918162450947072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LockConfigNodesImplActor, 0, Void >;
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	LockConfigNodesImplActor(ServerCoordinators const& coordinators) 
															#line 2692 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
		 : Actor<Void>(),
		   LockConfigNodesImplActorState<LockConfigNodesImplActor>(coordinators),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17562479931123276544UL, 3774863073311192320UL);
		ActorExecutionContextHelper __helper(static_cast<LockConfigNodesImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("lockConfigNodesImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("lockConfigNodesImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LockConfigNodesImplActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
[[nodiscard]] static Future<Void> lockConfigNodesImpl( ServerCoordinators const& coordinators ) {
															#line 763 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"
	return Future<Void>(new LockConfigNodesImplActor(coordinators));
															#line 2725 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/build_output/fdbserver/ConfigBroadcaster.actor.g.cpp"
}

#line 786 "/codebuild/output/src1466137952/src/github.com/apple/foundationdb/fdbserver/ConfigBroadcaster.actor.cpp"

Future<Void> ConfigBroadcaster::lockConfigNodes(ServerCoordinators coordinators) {
	return lockConfigNodesImpl(coordinators);
}
