#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
/*
 * storageserver.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cinttypes>
#include <functional>
#include <iterator>
#include <limits>
#include <memory>
#include <type_traits>
#include <unordered_map>

#include "fdbclient/BlobCipher.h"
#include "fdbclient/BlobGranuleCommon.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/Knobs.h"
#include "fdbrpc/TenantInfo.h"
#include "flow/ApiVersion.h"
#include "flow/Buggify.h"
#include "flow/Platform.h"
#include "flow/network.h"
#include "fmt/format.h"
#include "fdbclient/Audit.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/LoadBalance.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/PriorityMultiLock.actor.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/Trace.h"
#include "fdbclient/Tracing.h"
#include "flow/Util.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BlobConnectionProvider.h"
#include "fdbclient/BlobGranuleReader.actor.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"
#include "fdbclient/StatusClient.h"
#include "fdbclient/StorageServerShard.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/Tenant.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/Tuple.h"
#include "fdbclient/VersionedMap.h"
#include "fdbrpc/sim_validation.h"
#include "fdbrpc/Smoother.h"
#include "fdbrpc/Stats.h"
#include "fdbserver/AccumulativeChecksumUtil.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/LatencyBandConfig.h"
#include "fdbserver/LogProtocolMessage.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/MutationTracking.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "fdbserver/Ratekeeper.h"
#include "fdbserver/RecoveryState.h"
#include "fdbserver/RocksDBCheckpointUtils.actor.h"
#include "fdbserver/ServerCheckpoint.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/SpanContextMessage.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TransactionTagCounter.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbserver/BlobGranuleServerCommon.actor.h"
#include "fdbserver/StorageCorruptionBug.h"
#include "fdbserver/StorageServerUtils.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/TDMetric.actor.h"
#include "flow/Trace.h"
#include "flow/Util.h"
#include "flow/genericactors.actor.h"
#include "fdbserver/FDBRocksDBVersion.h"

#include "flow/actorcompiler.h" // This must be the last #include.

#ifndef __INTEL_COMPILER
#pragma region Data Structures
#endif

#define SHORT_CIRCUT_ACTUAL_STORAGE 0

namespace {
enum ChangeServerKeysContext { CSK_UPDATE, CSK_RESTORE, CSK_ASSIGN_EMPTY, CSK_FALL_BACK };

std::string changeServerKeysContextName(const ChangeServerKeysContext& context) {
	switch (context) {
	case CSK_UPDATE:
		return "Update";
	case CSK_RESTORE:
		return "Restore";
	case CSK_ASSIGN_EMPTY:
		return "AssignEmpty";
	case CSK_FALL_BACK:
		return "FallBackToFetchKeys";
	default:
		ASSERT(false);
	}
	return "UnknownContext";
}

bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_transaction_too_old:
	case error_code_future_version:
	case error_code_wrong_shard_server:
	case error_code_process_behind:
	case error_code_watch_cancelled:
	case error_code_unknown_change_feed:
	case error_code_server_overloaded:
	case error_code_change_feed_popped:
	case error_code_tenant_name_required:
	case error_code_tenant_removed:
	case error_code_tenant_not_found:
	case error_code_tenant_locked:
	// getMappedRange related exceptions that are not retriable:
	case error_code_mapper_bad_index:
	case error_code_mapper_no_such_key:
	case error_code_mapper_bad_range_decriptor:
	case error_code_quick_get_key_values_has_more:
	case error_code_quick_get_value_miss:
	case error_code_quick_get_key_values_miss:
	case error_code_get_mapped_key_values_has_more:
	case error_code_key_not_tuple:
	case error_code_value_not_tuple:
	case error_code_mapper_not_tuple:
		// case error_code_all_alternatives_failed:
		return true;
	default:
		return false;
	}
}

} // namespace

#define PERSIST_PREFIX "\xff\xff"

FDB_BOOLEAN_PARAM(UnlimitedCommitBytes);
FDB_BOOLEAN_PARAM(MoveInFailed);
FDB_BOOLEAN_PARAM(MoveInUpdatesSpilled);

// Immutable
static const KeyValueRef persistFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/4"_sr);
static const KeyValueRef persistShardAwareFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/5"_sr);
static const KeyRangeRef persistFormatReadableRange("FoundationDB/StorageServer/1/2"_sr,
                                                    "FoundationDB/StorageServer/1/6"_sr);
static const KeyRef persistID = PERSIST_PREFIX "ID"_sr;
static const KeyRef persistTssPairID = PERSIST_PREFIX "tssPairID"_sr;
static const KeyRef persistSSPairID = PERSIST_PREFIX "ssWithTSSPairID"_sr;
static const KeyRef persistTssQuarantine = PERSIST_PREFIX "tssQ"_sr;

// (Potentially) change with the durable version or when fetchKeys completes
static const KeyRef persistVersion = PERSIST_PREFIX "Version"_sr;
static const KeyRangeRef persistShardAssignedKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAssigned/"_sr, PERSIST_PREFIX "ShardAssigned0"_sr);
static const KeyRangeRef persistShardAvailableKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAvailable/"_sr, PERSIST_PREFIX "ShardAvailable0"_sr);
static const KeyRangeRef persistByteSampleKeys = KeyRangeRef(PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS0"_sr);
static const KeyRangeRef persistByteSampleSampleKeys =
    KeyRangeRef(PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS0"_sr);
static const KeyRef persistLogProtocol = PERSIST_PREFIX "LogProtocol"_sr;
static const KeyRef persistPrimaryLocality = PERSIST_PREFIX "PrimaryLocality"_sr;
static const KeyRangeRef persistChangeFeedKeys = KeyRangeRef(PERSIST_PREFIX "CF/"_sr, PERSIST_PREFIX "CF0"_sr);
static const KeyRangeRef persistTenantMapKeys = KeyRangeRef(PERSIST_PREFIX "TM/"_sr, PERSIST_PREFIX "TM0"_sr);
// data keys are unmangled (but never start with PERSIST_PREFIX because they are always in allKeys)

static const KeyRangeRef persistStorageServerShardKeys =
    KeyRangeRef(PERSIST_PREFIX "StorageServerShard/"_sr, PERSIST_PREFIX "StorageServerShard0"_sr);

// Checkpoint related prefixes.
static const KeyRangeRef persistCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "Checkpoint/"_sr, PERSIST_PREFIX "Checkpoint0"_sr);
static const KeyRangeRef persistPendingCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "PendingCheckpoint/"_sr, PERSIST_PREFIX "PendingCheckpoint0"_sr);
static const std::string serverCheckpointFolder = "serverCheckpoints";
static const std::string checkpointBytesSampleTempFolder = "/metadata_temp";
static const std::string fetchedCheckpointFolder = "fetchedCheckpoints";
static const std::string serverBulkDumpFolder = "bulkDumpFiles";
static const std::string serverBulkLoadFolder = "bulkLoadFiles";
static const std::string invalidTeamId = "InvalidTeam";

static const KeyRangeRef persistBulkLoadTaskKeys =
    KeyRangeRef(PERSIST_PREFIX "BulkLoadTask/"_sr, PERSIST_PREFIX "BulkLoadTask0"_sr);

// Accumulative checksum related prefix
static const KeyRangeRef persistAccumulativeChecksumKeys =
    KeyRangeRef(PERSIST_PREFIX "AccumulativeChecksum/"_sr, PERSIST_PREFIX "AccumulativeChecksum0"_sr);

inline Key encodePersistAccumulativeChecksumKey(uint16_t acsIndex) {
	BinaryWriter wr(Unversioned());
	wr.serializeBytes(persistAccumulativeChecksumKeys.begin);
	wr << bigEndian16(acsIndex);
	return wr.toValue();
}

inline uint16_t decodePersistAccumulativeChecksumKey(const Key& key) {
	uint16_t acsIndex;
	BinaryReader rd(key.removePrefix(persistAccumulativeChecksumKeys.begin), Unversioned());
	rd >> acsIndex;
	return bigEndian16(acsIndex);
}

// MoveInUpdates caches new updates of a move-in shard, before that shard is ready to accept writes.
struct MoveInUpdates {
	MoveInUpdates() : spilled(MoveInUpdatesSpilled::False) {}
	MoveInUpdates(UID id,
	              Version version,
	              struct StorageServer* data,
	              IKeyValueStore* store,
	              MoveInUpdatesSpilled spilled);

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 bool allowSpill);

	bool hasNext() const;

	std::vector<Standalone<VerUpdateRef>> next(const int byteLimit);
	const std::deque<Standalone<VerUpdateRef>>& getUpdatesQueue() const { return this->updates; }

	UID id;
	Version lastRepliedVersion;
	std::deque<Standalone<VerUpdateRef>> updates;
	std::vector<Standalone<VerUpdateRef>> spillBuffer;
	struct StorageServer* data;
	IKeyValueStore* store;
	KeyRange range;
	bool fail;
	MoveInUpdatesSpilled spilled;
	size_t size;
	Future<Void> loadFuture;
	Severity logSev;

private:
																#line 284 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
[[nodiscard]] static Future<Void> loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end );
template <class> friend class MoveInUpdates_LoadUpdatesActorState;

#line 283 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

	Key getPersistKey(const Version version, const int idx) const;
};

															#line 293 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
// This generated class is to be used only via loadUpdates()
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class MoveInUpdates_LoadUpdatesActor>
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActorState {
															#line 299 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActorState(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   begin(begin),
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   end(end)
															#line 310 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	~MoveInUpdates_LoadUpdatesActorState() 
	{
		fdb_probe_actor_destroy("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 288 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spilled);
															#line 289 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (begin >= end)
															#line 327 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 290 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				self->spilled = MoveInUpdatesSpilled::False;
															#line 291 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 333 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MoveInUpdates_LoadUpdatesActorState();
				static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 294 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Key beginKey = persistUpdatesKey(self->id, begin), endKey = persistUpdatesKey(self->id, end);
															#line 295 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadBegin", self->id) .detail("BeginVersion", begin) .detail("EndVersion", end) .detail("BeginKey", beginKey) .detail("EndKey", endKey);
															#line 300 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(beginKey < endKey);
															#line 301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = self->store->readRange(KeyRangeRef(beginKey, endKey), SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT, SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT);
															#line 301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 349 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 1;
															#line 301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*>(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)));
															#line 354 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& res,int loopDepth) 
	{
															#line 304 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 305 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 306 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 307 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 310 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 389 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 316 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 393 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 317 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 318 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 399 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 403 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 322 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 408 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 325 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 416 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 328 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 329 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 335 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 424 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult && res,int loopDepth) 
	{
															#line 304 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 305 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 306 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 307 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 310 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 448 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 316 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 452 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 317 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 318 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 458 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 462 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 322 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 467 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 325 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 475 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 328 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 329 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 335 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 483 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state > 0) static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 0;
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates* self;
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version begin;
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version end;
															#line 572 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via loadUpdates()
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActor final : public Actor<Void>, public ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >, public FastAllocated<MoveInUpdates_LoadUpdatesActor>, public MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor> {
															#line 577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator new;
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12538603962767098368UL, 771811324742847232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >;
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActor(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 594 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor>(self, begin, end),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5561495993262577152UL, 4925366652637659904UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadUpdates");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> MoveInUpdates::loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end ) {
															#line 287 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new MoveInUpdates_LoadUpdatesActor(self, begin, end));
															#line 626 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 337 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

bool MoveInUpdates::hasNext() const {
	return this->spilled || (!this->updates.empty() && this->updates.back().version > this->lastRepliedVersion);
}

// MoveInShard corresponds to a move-in physical shard, a class representation of MoveInShardMetaData.
struct MoveInShard {
	std::shared_ptr<MoveInShardMetaData> meta;
	struct StorageServer* server;
	std::shared_ptr<MoveInUpdates> updates;
	bool isRestored;
	Version transferredVersion;
	ConductBulkLoad conductBulkLoad = ConductBulkLoad::False;

	Future<Void> fetchClient; // holds FetchShard() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;

	Severity logSev = static_cast<Severity>(SERVER_KNOBS->PHYSICAL_SHARD_MOVE_LOG_SEVERITY);

	MoveInShard() = default;
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad,
	            MoveInPhase phase);
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad);
	MoveInShard(StorageServer* server, MoveInShardMetaData meta);
	~MoveInShard();

	UID id() const { return this->meta->id; }
	UID dataMoveId() const { return this->meta->dataMoveId; }
	void setPhase(const MoveInPhase& phase) { this->meta->setPhase(phase); }
	MoveInPhase getPhase() const { return this->meta->getPhase(); }
	const std::vector<KeyRange>& ranges() const { return this->meta->ranges; }
	const std::vector<CheckpointMetaData>& checkpoints() const { return this->meta->checkpoints; }
	std::string destShardIdString() const { return this->meta->destShardIdString(); }
	void addRange(const KeyRangeRef range);
	void removeRange(const KeyRangeRef range);
	void cancel(const MoveInFailed failed = MoveInFailed::False);
	bool isDataTransferred() const { return meta->getPhase() >= MoveInPhase::ApplyingUpdates; }
	bool isDataAndCFTransferred() const { throw not_implemented(); }
	bool failed() const { return this->getPhase() == MoveInPhase::Cancel || this->getPhase() == MoveInPhase::Error; }
	void setHighWatermark(const Version version) { this->meta->highWatermark = version; }
	Version getHighWatermark() const { return this->meta->highWatermark; }

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	KeyRangeRef getAffectedRange(const MutationRef& mutation) const;

	std::string toString() const { return meta != nullptr ? meta->toString() : "Empty"; }
};

struct AddingShard : NonCopyable {
	KeyRange keys;
	Future<Void> fetchClient; // holds FetchKeys() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;
	DataMovementReason reason;
	SSBulkLoadMetadata ssBulkLoadMetadata;
	std::string teamId;

	// During the Fetching phase, it saves newer mutations whose version is greater or equal to fetchClient's
	// fetchVersion, while the shard is still busy catching up with fetchClient. It applies these updates after fetching
	// completes.
	std::deque<Standalone<VerUpdateRef>> updates;

	struct StorageServer* server;
	Version transferredVersion;
	Version fetchVersion;

	// To learn more details of the phase transitions, see function fetchKeys(). The phases below are sorted in
	// chronological order and do not go back.
	enum Phase {
		WaitPrevious,
		// During Fetching phase, it fetches data before fetchVersion and write it to storage, then let updater know it
		// is ready to update the deferred updates` (see the comment of member variable `updates` above).
		Fetching,
		// During the FetchingCF phase, the shard data is transferred but the remaining change feed data is still being
		// transferred. This is equivalent to the waiting phase for non-changefeed data.
		FetchingCF,
		// During Waiting phase, it sends updater the deferred updates, and wait until they are durable.
		Waiting
		// The shard's state is changed from adding to readWrite then.
	};

	Phase phase;

	AddingShard(StorageServer* server,
	            KeyRangeRef const& keys,
	            DataMovementReason reason,
	            const SSBulkLoadMetadata& ssBulkLoadMetadata);

	// When fetchKeys "partially completes" (splits an adding shard in two), this is used to construct the left half
	AddingShard(AddingShard* prev, KeyRange const& keys)
	  : keys(keys), fetchClient(prev->fetchClient), server(prev->server), transferredVersion(prev->transferredVersion),
	    fetchVersion(prev->fetchVersion), phase(prev->phase), reason(prev->reason),
	    ssBulkLoadMetadata(prev->ssBulkLoadMetadata) {}
	~AddingShard() {
		if (!fetchComplete.isSet())
			fetchComplete.send(Void());
		if (!readWrite.isSet())
			readWrite.send(Void());
	}

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	bool isDataTransferred() const { return phase >= FetchingCF; }
	bool isDataAndCFTransferred() const { return phase >= Waiting; }

	SSBulkLoadMetadata getSSBulkLoadMetadata() const { return ssBulkLoadMetadata; }
};

class ShardInfo : public ReferenceCounted<ShardInfo>, NonCopyable {
private:
	ShardInfo(KeyRange keys, std::unique_ptr<AddingShard>&& adding, StorageServer* readWrite)
	  : adding(std::move(adding)), readWrite(readWrite), keys(keys), shardId(0LL), desiredShardId(0LL), version(0) {}
	ShardInfo(KeyRange keys, std::shared_ptr<MoveInShard> moveInShard)
	  : adding(nullptr), readWrite(nullptr), moveInShard(moveInShard), keys(keys),
	    shardId(moveInShard->meta->destShardId()), desiredShardId(moveInShard->meta->destShardId()),
	    version(moveInShard->meta->createVersion) {}

	// A shard has 4 mutual exclusive states: adding, moveInShard, readWrite and notAssigned.
	std::unique_ptr<AddingShard> adding;
	struct StorageServer* readWrite;
	std::shared_ptr<MoveInShard> moveInShard; // The shard is being moved in via physical-shard-move.
	KeyRange keys;
	uint64_t changeCounter;
	uint64_t shardId;
	uint64_t desiredShardId;
	std::string teamId = invalidTeamId;
	Version version;

public:
	static ShardInfo* newNotAssigned(KeyRange keys) { return new ShardInfo(keys, nullptr, nullptr); }
	static ShardInfo* newReadWrite(KeyRange keys, StorageServer* data) { return new ShardInfo(keys, nullptr, data); }
	static ShardInfo* newAdding(StorageServer* data,
	                            KeyRange keys,
	                            DataMovementReason reason,
	                            const SSBulkLoadMetadata& ssBulkLoadMetadata) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(data, keys, reason, ssBulkLoadMetadata), nullptr);
	}
	static ShardInfo* addingSplitLeft(KeyRange keys, AddingShard* oldShard) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(oldShard, keys), nullptr);
	}

	static ShardInfo* newShard(StorageServer* data, const StorageServerShard& shard);

	static bool canMerge(const ShardInfo* l, const ShardInfo* r) {
		if (l == nullptr || r == nullptr || l->keys.end != r->keys.begin || l->version == invalidVersion ||
		    r->version == invalidVersion) {
			return false;
		}
		if (l->getShardId() != r->getShardId() || l->getDesiredShardId() != r->getDesiredShardId()) {
			return false;
		}
		return (l->isReadable() && r->isReadable()) || (!l->assigned() && !r->assigned());
	}

	StorageServerShard toStorageServerShard() const {
		StorageServerShard::ShardState st = StorageServerShard::NotAssigned;
		Optional<UID> moveInShardId;
		if (this->isReadable()) {
			st = StorageServerShard::ReadWrite;
		} else if (!this->assigned()) {
			st = StorageServerShard::NotAssigned;
		} else if (this->getAddingShard()) {
			st = this->getAddingShard()->phase == AddingShard::Waiting ? StorageServerShard::ReadWritePending
			                                                           : StorageServerShard::Adding;
		} else {
			ASSERT(this->getMoveInShard());
			const MoveInPhase phase = this->getMoveInShard()->getPhase();
			if (phase < MoveInPhase::ReadWritePending) {
				st = StorageServerShard::MovingIn;
			} else if (phase == MoveInPhase::ReadWritePending) {
				st = StorageServerShard::ReadWritePending;
			} else if (phase == MoveInPhase::Complete) {
				st = StorageServerShard::ReadWrite;
			} else {
				st = StorageServerShard::MovingIn;
			}
			// Clear moveInShardId if the data move is complete.
			if (phase != MoveInPhase::ReadWritePending && phase != MoveInPhase::Complete) {
				moveInShardId = this->getMoveInShard()->id();
			}
		}
		return StorageServerShard(this->keys, this->version, this->shardId, this->desiredShardId, st, moveInShardId);
	}

	// Copies necessary information from `shard`.
	void populateShard(const StorageServerShard& shard) {
		this->version = shard.version;
		this->shardId = shard.id;
		this->desiredShardId = shard.desiredId;
	}

	// Returns true if the current shard is merged with `other`.
	bool mergeWith(const ShardInfo* other) {
		if (!canMerge(this, other)) {
			return false;
		}
		this->keys = KeyRangeRef(this->keys.begin, other->range().end);
		this->version = std::max(this->version, other->getVersion());
		return true;
	}

	void validate() const {
		// TODO: Complete this.
	}

	bool isReadable() const { return readWrite != nullptr; }
	bool notAssigned() const { return !readWrite && !adding && !moveInShard; }
	bool assigned() const { return readWrite || adding || moveInShard; }

	KeyRange range() const { return keys; }
	uint64_t getShardId() const { return shardId; }
	uint64_t getDesiredShardId() const { return desiredShardId; }
	uint64_t getChangeCounter() const { return changeCounter; }
	AddingShard* getAddingShard() const { return adding.get(); }
	std::shared_ptr<MoveInShard> getMoveInShard() const { return moveInShard; }
	Version getVersion() const { return version; }
	std::string getTeamId() const { return teamId; }

	void setChangeCounter(uint64_t shardChangeCounter) { changeCounter = shardChangeCounter; }
	void setShardId(uint64_t id) { shardId = id; }
	void setDesiredShardId(uint64_t id) { desiredShardId = id; }

	bool isInVersionedData() const {
		return readWrite || (adding && adding->isDataTransferred()) ||
		       (moveInShard && moveInShard->isDataTransferred());
	}
	bool isCFInVersionedData() const { return readWrite || (adding && adding->isDataAndCFTransferred()); }
	bool isReadWritePending() const {
		return isCFInVersionedData() || (moveInShard && (moveInShard->getPhase() == MoveInPhase::ReadWritePending ||
		                                                 moveInShard->getPhase() == MoveInPhase::Complete));
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);
	bool isFetched() const {
		return readWrite || (adding && adding->fetchComplete.isSet()) ||
		       (moveInShard && moveInShard->fetchComplete.isSet());
	}

	bool isFetching() const {
		if (adding) {
			return !adding->fetchComplete.isSet();
		}
		if (moveInShard) {
			return !moveInShard->fetchComplete.isSet();
		}
		return false;
	}

	std::string debugDescribeState() const {
		if (notAssigned()) {
			return "NotAssigned";
		} else if (adding && !adding->isDataAndCFTransferred()) {
			return "AddingFetchingCF";
		} else if (adding && !adding->isDataTransferred()) {
			return "AddingFetching";
		} else if (adding) {
			return "AddingTransferred";
		} else if (moveInShard) {
			return moveInShard->meta->toString();
		} else {
			return "ReadWrite";
		}
	}
};

struct StorageServerDisk {
	explicit StorageServerDisk(struct StorageServer* data, IKeyValueStore* storage) : data(data), storage(storage) {}

	IKeyValueStore* getKeyValueStore() const { return this->storage; }

	void makeNewStorageServerDurable(const bool shardAware);
	bool makeVersionMutationsDurable(Version& prevStorageVersion,
	                                 Version newStorageVersion,
	                                 int64_t& bytesLeft,
	                                 UnlimitedCommitBytes unlimitedCommitBytes,
	                                 int64_t& clearRangesLeft,
	                                 const UID& ssId,
	                                 bool verbose = false);
	void makeVersionDurable(Version version);
	void makeAccumulativeChecksumDurable(const AccumulativeChecksumState& acsState);
	void clearAccumulativeChecksumState(const AccumulativeChecksumState& acsState);
	void makeTssQuarantineDurable();
	Future<bool> restoreDurableState();

	void changeLogProtocol(Version version, ProtocolVersion protocol);

	void writeMutation(MutationRef mutation);
	void writeKeyValue(KeyValueRef kv);
	void clearRange(KeyRangeRef keys);

	Future<Void> addRange(KeyRangeRef range, std::string id) {
		return storage->addRange(range, id, !SERVER_KNOBS->SHARDED_ROCKSDB_DELAY_COMPACTION_FOR_DATA_MOVE);
	}

	std::vector<std::string> removeRange(KeyRangeRef range) { return storage->removeRange(range); }

	void markRangeAsActive(KeyRangeRef range) { storage->markRangeAsActive(range); }

	Future<Void> replaceRange(KeyRange range, Standalone<VectorRef<KeyValueRef>> data) {
		return storage->replaceRange(range, data);
	}

	void persistRangeMapping(KeyRangeRef range, bool isAdd) { storage->persistRangeMapping(range, isAdd); }

	CoalescedKeyRangeMap<std::string> getExistingRanges() { return storage->getExistingRanges(); }

	Future<Void> getError() { return storage->getError(); }
	Future<Void> init() { return storage->init(); }
	Future<Void> canCommit() { return storage->canCommit(); }
	Future<Void> commit() { return storage->commit(); }

	void logRecentRocksDBBackgroundWorkStats(UID ssId, std::string logReason) {
		return storage->logRecentRocksDBBackgroundWorkStats(ssId, logReason);
	}

	// SOMEDAY: Put readNextKeyInclusive in IKeyValueStore
	// Read the key that is equal or greater then 'key' from the storage engine.
	// For example, readNextKeyInclusive("a") should return:
	//  - "a", if key "a" exist
	//  - "b", if key "a" doesn't exist, and "b" is the next existing key in total order
	//  - allKeys.end, if keyrange [a, allKeys.end) is empty
	Future<Key> readNextKeyInclusive(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return readFirstKey(storage, KeyRangeRef(key, allKeys.end), options);
	}
	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValue(key, options);
	}
	Future<Optional<Value>> readValuePrefix(KeyRef key,
	                                        int maxLength,
	                                        Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValuePrefix(key, maxLength, options);
	}
	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit = 1 << 30,
	                              int byteLimit = 1 << 30,
	                              Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return storage->readRange(keys, rowLimit, byteLimit, options);
	}

	Future<CheckpointMetaData> checkpoint(const CheckpointRequest& request) { return storage->checkpoint(request); }

	Future<Void> restore(const std::vector<CheckpointMetaData>& checkpoints) { return storage->restore(checkpoints); }

	Future<Void> restore(const std::string& shardId,
	                     const std::vector<KeyRange>& ranges,
	                     const std::vector<CheckpointMetaData>& checkpoints) {
		return storage->restore(shardId, ranges, checkpoints);
	}

	Future<Void> deleteCheckpoint(const CheckpointMetaData& checkpoint) {
		return storage->deleteCheckpoint(checkpoint);
	}

	KeyValueStoreType getKeyValueStoreType() const { return storage->getType(); }
	StorageBytes getStorageBytes() const { return storage->getStorageBytes(); }
	std::tuple<size_t, size_t, size_t> getSize() const { return storage->getSize(); }

	Future<EncryptionAtRestMode> encryptionMode() { return storage->encryptionMode(); }

	// The following are pointers to the Counters in StorageServer::counters of the same names.
	Counter* kvCommitLogicalBytes;
	Counter* kvClearRanges;
	Counter* kvClearSingleKey;
	Counter* kvGets;
	Counter* kvScans;
	Counter* kvCommits;

private:
	struct StorageServer* data;
	IKeyValueStore* storage;
	void writeMutations(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);
	void writeMutationsBuggy(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);

																#line 1025 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
// This generated class is to be used only via readFirstKey()
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadFirstKeyActor>
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActorState {
															#line 1031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActorState(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : storage(storage),
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 1042 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	~ReadFirstKeyActorState() 
	{
		fdb_probe_actor_destroy("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = storage->readRange(range, 1, 1 << 30, options);
															#line 733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1059 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 1;
															#line 733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFirstKeyActor, 0, RangeResult >*>(static_cast<ReadFirstKeyActor*>(this)));
															#line 1064 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFirstKeyActorState();
		static_cast<ReadFirstKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& r,int loopDepth) 
	{
															#line 734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1087 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 735 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1091 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1101 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1cont1(RangeResult && r,int loopDepth) 
	{
															#line 734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1114 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 735 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1118 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1128 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& r,int loopDepth) 
	{
		loopDepth = a_body1cont1(r, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && r,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state > 0) static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFirstKeyActor*>(this)->ActorCallback< ReadFirstKeyActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	IKeyValueStore* storage;
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRangeRef range;
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 1218 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via readFirstKey()
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActor final : public Actor<Key>, public ActorCallback< ReadFirstKeyActor, 0, RangeResult >, public FastAllocated<ReadFirstKeyActor>, public ReadFirstKeyActorState<ReadFirstKeyActor> {
															#line 1223 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ReadFirstKeyActor>::operator new;
	using FastAllocated<ReadFirstKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16837006158351742720UL, 11707585986744784640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFirstKeyActor, 0, RangeResult >;
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActor(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 1240 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Key>(),
		   ReadFirstKeyActorState<ReadFirstKeyActor>(storage, range, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4418144802222492160UL, 5296783460543853568UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFirstKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFirstKeyActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] static Future<Key> readFirstKey( IKeyValueStore* const& storage, KeyRangeRef const& range, Optional<ReadOptions> const& options ) {
															#line 732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new ReadFirstKeyActor(storage, range, options));
															#line 1272 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 739 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
};

struct UpdateEagerReadInfo {
	std::vector<KeyRef> keyBegin;
	std::vector<Key> keyEnd; // these are for ClearRange

	std::vector<std::pair<KeyRef, int>> keys;
	std::vector<Optional<Value>> value;

	Arena arena;
	bool enableClearRangeEagerReads;

	UpdateEagerReadInfo(bool enableClearRangeEagerReads) : enableClearRangeEagerReads(enableClearRangeEagerReads) {}

	void addMutations(VectorRef<MutationRef> const& mutations) {
		for (auto& m : mutations)
			addMutation(m);
	}

	void addMutation(MutationRef const& m) {
		// SOMEDAY: Theoretically we can avoid a read if there is an earlier overlapping ClearRange
		if (m.type == MutationRef::ClearRange && !m.param2.startsWith(systemKeys.end) && enableClearRangeEagerReads)
			keyBegin.push_back(m.param2);
		else if (m.type == MutationRef::CompareAndClear) {
			if (enableClearRangeEagerReads)
				keyBegin.push_back(keyAfter(m.param1, arena));
			if (keys.size() > 0 && keys.back().first == m.param1) {
				// Don't issue a second read, if the last read was equal to the current key.
				// CompareAndClear is likely to be used after another atomic operation on same key.
				keys.back().second = std::max(keys.back().second, m.param2.size() + 1);
			} else {
				keys.emplace_back(m.param1, m.param2.size() + 1);
			}
		} else if ((m.type == MutationRef::AppendIfFits) || (m.type == MutationRef::ByteMin) ||
		           (m.type == MutationRef::ByteMax))
			keys.emplace_back(m.param1, CLIENT_KNOBS->VALUE_SIZE_LIMIT);
		else if (isAtomicOp((MutationRef::Type)m.type))
			keys.emplace_back(m.param1, m.param2.size());
	}

	void finishKeyBegin() {
		if (enableClearRangeEagerReads) {
			std::sort(keyBegin.begin(), keyBegin.end());
			keyBegin.resize(std::unique(keyBegin.begin(), keyBegin.end()) - keyBegin.begin());
		}
		std::sort(keys.begin(), keys.end(), [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			return (lhs.first < rhs.first) || (lhs.first == rhs.first && lhs.second > rhs.second);
		});
		keys.resize(std::unique(keys.begin(),
		                        keys.end(),
		                        [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                        return lhs.first == rhs.first;
		                        }) -
		            keys.begin());
		// value gets populated in doEagerReads
	}

	Optional<Value>& getValue(KeyRef key) {
		int i = std::lower_bound(keys.begin(),
		                         keys.end(),
		                         std::pair<KeyRef, int>(key, 0),
		                         [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                         return lhs.first < rhs.first;
		                         }) -
		        keys.begin();
		ASSERT(i < keys.size() && keys[i].first == key);
		return value[i];
	}

	KeyRef getKeyEnd(KeyRef key) {
		int i = std::lower_bound(keyBegin.begin(), keyBegin.end(), key) - keyBegin.begin();
		ASSERT(i < keyBegin.size() && keyBegin[i] == key);
		return keyEnd[i];
	}
};

const int VERSION_OVERHEAD =
    64 + sizeof(Version) + sizeof(Standalone<VerUpdateRef>) + // mutationLog, 64b overhead for map
    2 * (64 + sizeof(Version) +
         sizeof(Reference<VersionedMap<KeyRef, ValueOrClearToRef>::PTreeT>)); // versioned map [ x2 for
                                                                              // createNewVersion(version+1) ], 64b
                                                                              // overhead for map

static int mvccStorageBytes(MutationRef const& m) {
	return mvccStorageBytes(m.param1.size() + m.param2.size());
}

struct FetchInjectionInfo {
	Arena arena;
	std::vector<VerUpdateRef> changes;
};

struct ChangeFeedInfo : ReferenceCounted<ChangeFeedInfo> {
	std::deque<Standalone<EncryptedMutationsAndVersionRef>> mutations;
	Version fetchVersion = invalidVersion; // The version that commits from a fetch have been written to storage, but
	                                       // have not yet been committed as part of updateStorage.
	Version storageVersion = invalidVersion; // The version between the storage version and the durable version are
	                                         // being written to disk as part of the current commit in updateStorage.
	Version durableVersion = invalidVersion; // All versions before the durable version are durable on disk
	Version metadataVersion = invalidVersion; // Last update to the change feed metadata. Used for reasoning about
	                                          // fetched metadata vs local metadata
	Version emptyVersion = 0; // The change feed does not have any mutations before emptyVersion
	KeyRange range;
	Key id;
	AsyncTrigger newMutations;
	NotifiedVersion durableFetchVersion;
	// A stopped change feed no longer adds new mutations, but is still queryable.
	// stopVersion = MAX_VERSION means the feed has not been stopped
	Version stopVersion = MAX_VERSION;

	// We need to track the version the change feed metadata was created by private mutation, so that if it is rolled
	// back, we can avoid notifying other SS of change feeds that don't durably exist
	Version metadataCreateVersion = invalidVersion;

	FlowLock fetchLock = FlowLock(1);

	bool removing = false;
	bool destroyed = false;

	KeyRangeMap<std::unordered_map<UID, Promise<Void>>> moveTriggers;

	void triggerOnMove(KeyRange range, UID streamUID, Promise<Void> p) {
		auto toInsert = moveTriggers.modify(range);
		for (auto triggerRange = toInsert.begin(); triggerRange != toInsert.end(); ++triggerRange) {
			triggerRange->value().insert({ streamUID, p });
		}
	}

	void moved(KeyRange range) {
		auto toTrigger = moveTriggers.intersectingRanges(range);
		for (auto& triggerRange : toTrigger) {
			for (auto& triggerStream : triggerRange.cvalue()) {
				if (triggerStream.second.canBeSet()) {
					triggerStream.second.send(Void());
				}
			}
		}
		// coalesce doesn't work with promises
		moveTriggers.insert(range, std::unordered_map<UID, Promise<Void>>());
	}

	void removeOnMoveTrigger(KeyRange range, UID streamUID) {
		auto toRemove = moveTriggers.modify(range);
		for (auto triggerRange = toRemove.begin(); triggerRange != toRemove.end(); ++triggerRange) {
			auto streamToRemove = triggerRange->value().find(streamUID);
			if (streamToRemove == triggerRange->cvalue().end()) {
				ASSERT(destroyed);
			} else {
				triggerRange->value().erase(streamToRemove);
			}
		}
		// TODO: may be more cleanup possible here
	}

	void destroy(Version destroyVersion) {
		updateMetadataVersion(destroyVersion);
		removing = true;
		destroyed = true;
		moved(range);
		newMutations.trigger();
	}

	bool updateMetadataVersion(Version version) {
		// don't update metadata version if removing, so that metadata version remains the moved away version
		if (!removing && version > metadataVersion) {
			metadataVersion = version;
			return true;
		}
		return false;
	}
};

class ServerWatchMetadata : public ReferenceCounted<ServerWatchMetadata> {
public:
	Key key;
	Optional<Value> value;
	Version version;
	Future<Version> watch_impl;
	Promise<Version> versionPromise;
	Optional<TagSet> tags;
	Optional<UID> debugID;
	int64_t tenantId;

	ServerWatchMetadata(Key key,
	                    Optional<Value> value,
	                    Version version,
	                    Optional<TagSet> tags,
	                    Optional<UID> debugID,
	                    int64_t tenantId)
	  : key(key), value(value), version(version), tags(tags), debugID(debugID), tenantId(tenantId) {}
};

struct BusiestWriteTagContext {
	const std::string busiestWriteTagTrackingKey;
	UID ratekeeperID;
	Reference<EventCacheHolder> busiestWriteTagEventHolder;
	double lastUpdateTime;

	BusiestWriteTagContext(const UID& thisServerID)
	  : busiestWriteTagTrackingKey(thisServerID.toString() + "/BusiestWriteTag"), ratekeeperID(UID()),
	    busiestWriteTagEventHolder(makeReference<EventCacheHolder>(busiestWriteTagTrackingKey)), lastUpdateTime(-1) {}
};

struct TenantSSInfo {
	constexpr static FileIdentifier file_identifier = 3253114;
	TenantAPI::TenantLockState lockState;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, lockState);
	}
};

struct SSBulkLoadMetrics {
public:
	SSBulkLoadMetrics() : ongoingTasks(0) {}
	void addTask() { ongoingTasks++; }
	void removeTask() { ongoingTasks--; }
	int getOngoingTasks() { return ongoingTasks; }

private:
	int ongoingTasks = 0;
};

struct StorageServer : public IStorageMetricsService {
	typedef VersionedMap<KeyRef, ValueOrClearToRef> VersionedData;

private:
	// versionedData contains sets and clears.

	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// ~ Clears are maximal: If versionedData.at(v) contains a clear [b,e) then
	//      there is a key data[e]@v, or e==allKeys.end, or a shard boundary or former boundary at e

	// * Reads are possible: When k is in a readable shard, for any v in [storageVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v    (storage[k] might be @ any version in
	//      [durableVersion, storageVersion])

	// * Transferred shards are partially readable: When k is in an adding, transferred shard, for any v in
	// [transferredVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v

	// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
	// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
	//      contain later versions if applyUpdate is on the stack.

	// * Old shards are erased: versionedData.atLatest() has entries (sets or intersecting clears) only for keys in
	// readable or adding,transferred shards.
	//   Earlier versions may have extra entries for shards that *were* readable or adding,transferred when those
	//   versions were the latest, but they eventually are forgotten.

	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion(), but
	// views
	//   at older versions may contain older items which are also in storage (this is OK because of idempotency)

	VersionedData versionedData;
	std::map<Version, Standalone<VerUpdateRef>> mutationLog; // versions (durableVersion, version]

	using WatchMapKey = std::pair<int64_t, Key>;
	using WatchMapKeyHasher = boost::hash<WatchMapKey>;
	using WatchMapValue = Reference<ServerWatchMetadata>;
	using WatchMap_t = std::unordered_map<WatchMapKey, WatchMapValue, WatchMapKeyHasher>;
	WatchMap_t watchMap; // keep track of server watches

public:
	struct PendingNewShard {
		PendingNewShard(uint64_t shardId, KeyRangeRef range) : shardId(format("%016llx", shardId)), range(range) {}

		std::string toString() const {
			return fmt::format("PendingNewShard: [ShardID]: {} [Range]: {}",
			                   this->shardId,
			                   Traceable<KeyRangeRef>::toString(this->range));
		}

		std::string shardId;
		KeyRange range;
	};

	std::map<Version, std::vector<CheckpointMetaData>> pendingCheckpoints; // Pending checkpoint requests
	std::unordered_map<UID, CheckpointMetaData> checkpoints; // Existing and deleting checkpoints
	std::unordered_map<UID, ICheckpointReader*> liveCheckpointReaders; // Active checkpoint readers
	VersionedMap<int64_t, TenantSSInfo> tenantMap;
	std::map<Version, std::vector<PendingNewShard>>
	    pendingAddRanges; // Pending requests to add ranges to physical shards
	std::map<Version, std::vector<KeyRange>>
	    pendingRemoveRanges; // Pending requests to remove ranges from physical shards
	std::deque<std::pair<Standalone<StringRef>, Standalone<StringRef>>> constructedData;

	bool shardAware; // True if the storage server is aware of the physical shards.

	LocalityData locality; // Storage server's locality information.

	// Histograms
	struct FetchKeysHistograms {
		const Reference<Histogram> latency;
		const Reference<Histogram> bytes;
		const Reference<Histogram> bandwidth;
		const Reference<Histogram> bytesPerCommit;

		FetchKeysHistograms()
		  : latency(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                    FETCH_KEYS_LATENCY_HISTOGRAM,
		                                    Histogram::Unit::milliseconds)),
		    bytes(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                  FETCH_KEYS_BYTES_HISTOGRAM,
		                                  Histogram::Unit::bytes)),
		    bandwidth(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                      FETCH_KEYS_BYTES_PER_SECOND_HISTOGRAM,
		                                      Histogram::Unit::bytes_per_second)),
		    bytesPerCommit(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                           FETCH_KEYS_BYTES_PER_COMMIT_HISTOGRAM,
		                                           Histogram::Unit::bytes)) {}
	} fetchKeysHistograms;

	Reference<Histogram> tlogCursorReadsLatencyHistogram;
	Reference<Histogram> ssVersionLockLatencyHistogram;
	Reference<Histogram> eagerReadsLatencyHistogram;
	Reference<Histogram> fetchKeysPTreeUpdatesLatencyHistogram;
	Reference<Histogram> tLogMsgsPTreeUpdatesLatencyHistogram;
	Reference<Histogram> storageUpdatesDurableLatencyHistogram;
	Reference<Histogram> storageCommitLatencyHistogram;
	Reference<Histogram> ssDurableVersionUpdateLatencyHistogram;
	// Histograms of requests sent to KVS.
	Reference<Histogram> readRangeBytesReturnedHistogram;
	Reference<Histogram> readRangeBytesLimitHistogram;
	Reference<Histogram> readRangeKVPairsReturnedHistogram;

	// watch map operations
	Reference<ServerWatchMetadata> getWatchMetadata(KeyRef key, int64_t tenantId) const;
	KeyRef setWatchMetadata(Reference<ServerWatchMetadata> metadata);
	void deleteWatchMetadata(KeyRef key, int64_t tenantId);
	void clearWatchMetadata();

	// tenant map operations
	void insertTenant(TenantMapEntry const& tenant, Version version, bool persist);
	void clearTenants(StringRef startTenant, StringRef endTenant, Version version);

	void checkTenantEntry(Version version, TenantInfo tenant, bool lockAware);

	std::vector<StorageServerShard> getStorageServerShards(KeyRangeRef range);
	std::shared_ptr<MoveInShard> getMoveInShard(const UID& dataMoveId,
	                                            const Version version,
	                                            const ConductBulkLoad conductBulkLoad);

	class CurrentRunningFetchKeys {
		std::unordered_map<UID, double> startTimeMap;
		std::unordered_map<UID, KeyRange> keyRangeMap;

		static const StringRef emptyString;
		static const KeyRangeRef emptyKeyRange;

	public:
		void recordStart(const UID id, const KeyRange& keyRange) {
			startTimeMap[id] = now();
			keyRangeMap[id] = keyRange;
		}

		void recordFinish(const UID id) {
			startTimeMap.erase(id);
			keyRangeMap.erase(id);
		}

		std::pair<double, KeyRange> longestTime() const {
			if (numRunning() == 0) {
				return { -1, emptyKeyRange };
			}

			const double currentTime = now();
			double longest = 0;
			UID UIDofLongest;
			for (const auto& kv : startTimeMap) {
				const double currentRunningTime = currentTime - kv.second;
				if (longest <= currentRunningTime) {
					longest = currentRunningTime;
					UIDofLongest = kv.first;
				}
			}
			if (BUGGIFY) {
				UIDofLongest = deterministicRandom()->randomUniqueID();
			}
			auto it = keyRangeMap.find(UIDofLongest);
			if (it != keyRangeMap.end()) {
				return { longest, it->second };
			}
			return { -1, emptyKeyRange };
		}

		int numRunning() const { return startTimeMap.size(); }
	} currentRunningFetchKeys;

	Tag tag;
	std::vector<std::pair<Version, Tag>> history;
	std::vector<std::pair<Version, Tag>> allHistory;
	Version poppedAllAfter;
	std::map<Version, Arena>
	    freeable; // for each version, an Arena that must be held until that version is < oldestVersion
	Arena lastArena;
	double cpuUsage;
	double diskUsage;

	std::map<Version, Standalone<VerUpdateRef>> const& getMutationLog() const { return mutationLog; }
	std::map<Version, Standalone<VerUpdateRef>>& getMutableMutationLog() { return mutationLog; }
	VersionedData const& data() const { return versionedData; }
	VersionedData& mutableData() { return versionedData; }

	mutable double old_rate = 1.0;
	double currentRate() const {
		auto versionLag = version.get() - durableVersion.get();
		double res;
		if (versionLag >= SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX) {
			res = 0.0;
		} else if (versionLag > SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) {
			res =
			    1.0 -
			    (double(versionLag - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) /
			     double(SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX));
		} else {
			res = 1.0;
		}
		if (res != old_rate) {
			TraceEvent(SevDebug, "LocalRatekeeperChange", thisServerID)
			    .detail("Old", old_rate)
			    .detail("New", res)
			    .detail("NonDurableVersions", versionLag);
			old_rate = res;
		}
		return res;
	}

	void addMutationToMutationLogOrStorage(
	    Version ver,
	    MutationRef m); // Appends m to mutationLog@ver, or to storage if ver==invalidVersion

	// Update the byteSample, and write the updates to the mutation log@ver, or to storage if ver==invalidVersion
	void byteSampleApplyMutation(MutationRef const& m, Version ver);
	void byteSampleApplySet(KeyValueRef kv, Version ver);
	void byteSampleApplyClear(KeyRangeRef range, Version ver);

	void popVersion(Version v, bool popAllTags = false) {
		if (logSystem && !isTss()) {
			if (v > poppedAllAfter) {
				popAllTags = true;
				poppedAllAfter = std::numeric_limits<Version>::max();
			}

			std::vector<std::pair<Version, Tag>>* hist = &history;
			std::vector<std::pair<Version, Tag>> allHistoryCopy;
			if (popAllTags) {
				allHistoryCopy = allHistory;
				hist = &allHistoryCopy;
			}

			while (hist->size() && v > hist->back().first) {
				logSystem->pop(v, hist->back().second);
				hist->pop_back();
			}
			if (hist->size()) {
				logSystem->pop(v, hist->back().second);
			} else {
				logSystem->pop(v, tag);
			}
		}
	}

	Standalone<VerUpdateRef>& addVersionToMutationLog(Version v) {
		// return existing version...
		auto m = mutationLog.find(v);
		if (m != mutationLog.end())
			return m->second;

		// ...or create a new one
		auto& u = mutationLog[v];
		u.version = v;
		if (lastArena.getSize() >= 65536)
			lastArena = Arena(4096);
		u.arena() = lastArena;
		counters.bytesInput += VERSION_OVERHEAD;
		return u;
	}

	MutationRef addMutationToMutationLog(Standalone<VerUpdateRef>& mLV, MutationRef const& m) {
		byteSampleApplyMutation(m, mLV.version);
		counters.bytesInput += mvccStorageBytes(m);
		return mLV.push_back_deep(mLV.arena(), m);
	}

	void setTssPair(UID pairId) {
		tssPairID = Optional<UID>(pairId);

		// Set up tss fault injection here, only if we are in simulated mode and with fault injection.
		// With fault injection enabled, the tss will start acting normal for a bit, then after the specified delay
		// start behaving incorrectly.
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    g_simulator->tssMode >= ISimulator::TSSMode::EnabledAddDelay) {
			tssFaultInjectTime = now() + deterministicRandom()->randomInt(60, 300);
			TraceEvent(SevWarnAlways, "TSSInjectFaultEnabled", thisServerID)
			    .detail("Mode", g_simulator->tssMode)
			    .detail("At", tssFaultInjectTime.get());
		}
	}

	// If a TSS is "in quarantine", it means it has incorrect data. It is effectively in a "zombie" state where it
	// rejects all read requests and ignores all non-private mutations and data movements, but otherwise is still part
	// of the cluster. The purpose of this state is to "freeze" the TSS state after a mismatch so a human operator can
	// investigate, but preventing a new storage process from replacing the TSS on the worker. It will still get removed
	// from the cluster if it falls behind on the mutation stream, or if its tss pair gets removed and its tag is no
	// longer valid.
	bool isTSSInQuarantine() const { return tssPairID.present() && tssInQuarantine; }

	void startTssQuarantine() {
		if (!tssInQuarantine) {
			// persist quarantine so it's still quarantined if rebooted
			storage.makeTssQuarantineDurable();
		}
		tssInQuarantine = true;
	}

	StorageServerDisk storage;

	KeyRangeMap<Reference<ShardInfo>> shards;
	KeyRangeMap<SSBulkLoadMetadata> ssBulkLoadMetadataMap; // store the latest bulkload task on ranges
	uint64_t shardChangeCounter; // max( shards->changecounter )

	KeyRangeMap<bool> cachedRangeMap; // indicates if a key-range is being cached

	KeyRangeMap<std::vector<Reference<ChangeFeedInfo>>> keyChangeFeed;
	std::unordered_map<Key, Reference<ChangeFeedInfo>> uidChangeFeed;
	Deque<std::pair<std::vector<Key>, Version>> changeFeedVersions;
	std::map<UID, PromiseStream<Key>> changeFeedDestroys;
	std::set<Key> currentChangeFeeds;
	std::set<Key> fetchingChangeFeeds;
	std::unordered_map<NetworkAddress, std::unordered_map<UID, Version>> changeFeedClientVersions;
	std::unordered_map<Key, Version> changeFeedCleanupDurable;
	int64_t activeFeedQueries = 0;
	int64_t changeFeedMemoryBytes = 0;
	std::deque<std::pair<Version, int64_t>> feedMemoryBytesByVersion;

	// newestAvailableVersion[k]
	//   == invalidVersion -> k is unavailable at all versions
	//   <= storageVersion -> k is unavailable at all versions (but might be read anyway from storage if we are in the
	//   process of committing makeShardDurable)
	//   == v              -> k is readable (from storage+versionedData) @ [storageVersion,v], and not being updated
	//   when version increases
	//   == latestVersion  -> k is readable (from storage+versionedData) @ [storageVersion,version.get()], and thus
	//   stays available when version increases
	CoalescedKeyRangeMap<Version> newestAvailableVersion;

	CoalescedKeyRangeMap<Version> newestDirtyVersion; // Similar to newestAvailableVersion, but includes (only) keys
	                                                  // that were only partly available (due to cancelled fetchKeys)

	// The following are in rough order from newest to oldest
	Version lastTLogVersion, lastVersionWithData, restoredVersion, prevVersion;
	NotifiedVersion version;
	NotifiedVersion desiredOldestVersion; // We can increase oldestVersion (and then durableVersion) to this version
	                                      // when the disk permits
	NotifiedVersion oldestVersion; // See also storageVersion()
	NotifiedVersion durableVersion; // At least this version will be readable from storage after a power failure
	// In the event of the disk corruption, sqlite and redwood will either not recover, recover to durableVersion
	// but be unable to read some data, or they could lose the last commit. If we lose the last commit, the storage
	// might not be able to peek from the tlog (depending on when it sent the last pop). So this version just keeps
	// track of the version we committed to the storage engine before we did commit durableVersion.
	Version storageMinRecoverVersion = 0;
	Version rebootAfterDurableVersion;
	int8_t primaryLocality;
	NotifiedVersion knownCommittedVersion;

	Deque<std::pair<Version, Version>> recoveryVersionSkips;
	int64_t versionLag; // An estimate for how many versions it takes for the data to move from the logs to this storage
	                    // server

	Optional<UID> sourceTLogID; // the tLog from which the latest batch of versions were fetched

	ProtocolVersion logProtocol;

	Reference<ILogSystem> logSystem;
	Reference<ILogSystem::IPeekCursor> logCursor;

	// The version the cluster starts on. This value is not persisted and may
	// not be valid after a recovery.
	Version initialClusterVersion = 1;
	UID thisServerID;
	Optional<UID> tssPairID; // if this server is a tss, this is the id of its (ss) pair
	Optional<UID> ssPairID; // if this server is an ss, this is the id of its (tss) pair
	Optional<double> tssFaultInjectTime;
	bool tssInQuarantine;

	Key sk;
	Reference<AsyncVar<ServerDBInfo> const> db;
	Database cx;
	ActorCollection actors;

	CoalescedKeyRangeMap<bool, int64_t, KeyBytesMetric<int64_t>> byteSampleClears;
	AsyncVar<bool> byteSampleClearsTooLarge;
	Future<Void> byteSampleRecovery;
	Future<Void> durableInProgress;

	AsyncMap<Key, bool> watches;
	AsyncMap<int64_t, bool> tenantWatches;
	int64_t watchBytes;
	int64_t numWatches;
	AsyncVar<bool> noRecentUpdates;
	double lastUpdate;

	std::string folder;
	std::string checkpointFolder;
	std::string fetchedCheckpointFolder;
	std::string bulkDumpFolder;
	std::string bulkLoadFolder;

	// defined only during splitMutations()/addMutation()
	UpdateEagerReadInfo* updateEagerReads;

	FlowLock durableVersionLock;
	FlowLock fetchKeysParallelismLock;
	// Extra lock that prevents too much post-initial-fetch work from building up, such as mutation applying and change
	// feed tail fetching
	FlowLock fetchKeysParallelismChangeFeedLock;
	int64_t fetchKeysBytesBudget;
	AsyncVar<bool> fetchKeysBudgetUsed;
	int64_t fetchKeysTotalCommitBytes;
	std::vector<Promise<FetchInjectionInfo*>> readyFetchKeys;

	ThroughputLimiter fetchKeysLimiter;

	FlowLock serveFetchCheckpointParallelismLock;

	std::unordered_map<UID, std::shared_ptr<MoveInShard>> moveInShards;

	Reference<PriorityMultiLock> ssLock;
	std::vector<int> readPriorityRanks;

	Future<PriorityMultiLock::Lock> getReadLock(const Optional<ReadOptions>& options) {
		int readType = (int)(options.present() ? options.get().type : ReadType::NORMAL);
		readType = std::clamp<int>(readType, 0, readPriorityRanks.size() - 1);
		return ssLock->lock(readPriorityRanks[readType]);
	}

	FlowLock serveAuditStorageParallelismLock;

	FlowLock serveBulkDumpParallelismLock;

	int64_t instanceID;

	Promise<Void> otherError;
	Promise<Void> coreStarted;
	bool shuttingDown;

	Promise<Void> registerInterfaceAcceptingRequests;
	Future<Void> interfaceRegistered;

	bool behind;
	bool versionBehind;

	bool debug_inApplyUpdate;
	double debug_lastValidateTime;

	int64_t lastBytesInputEBrake;
	Version lastDurableVersionEBrake;

	int maxQueryQueue;
	int getAndResetMaxQueryQueueSize() {
		int val = maxQueryQueue;
		maxQueryQueue = 0;
		return val;
	}

	TransactionTagCounter transactionTagCounter;
	BusiestWriteTagContext busiestWriteTagContext;

	Optional<LatencyBandConfig> latencyBandConfig;

	Optional<EncryptionAtRestMode> encryptionMode;
	Reference<GetEncryptCipherKeysMonitor> getEncryptCipherKeysMonitor;

	struct Counters : CommonStorageCounters {

		Counter allQueries, systemKeyQueries, getKeyQueries, getValueQueries, getRangeQueries, getRangeSystemKeyQueries,
		    getRangeStreamQueries, lowPriorityQueries, rowsQueried, watchQueries, emptyQueries, feedRowsQueried,
		    feedBytesQueried, feedStreamQueries, rejectedFeedStreamQueries, feedVersionQueries;

		// counters related to getMappedRange queries
		Counter getMappedRangeBytesQueried, finishedGetMappedRangeSecondaryQueries, getMappedRangeQueries,
		    finishedGetMappedRangeQueries;

		// Bytes pulled from TLogs, it counts the size of the key value pairs, e.g., key-value pair ("a", "b") is
		// counted as 2 Bytes.
		Counter logicalBytesInput;
		// Bytes pulled from TLogs for moving-in shards, it counts the mutations sent to the moving-in shard during
		// Fetching and Waiting phases.
		Counter logicalBytesMoveInOverhead;
		// Bytes committed to the underlying storage engine by SS, it counts the size of key value pairs.
		Counter kvCommitLogicalBytes;
		// Count of all clearRange operations to the storage engine.
		Counter kvClearRanges;
		// Count of all clearRange operations on a singlekeyRange(key delete) to the storage engine.
		Counter kvClearSingleKey;
		// ClearRange operations issued by FDB, instead of from users, e.g., ClearRange operations to remove a shard
		// from a storage server, as in removeDataRange().
		Counter kvSystemClearRanges;
		// Bytes of the mutations that have been removed from memory because they durable. The counting is same as
		// bytesInput, instead of the actual bytes taken in the storages, so that (bytesInput - bytesDurable) can
		// reflect the current memory footprint of MVCC.
		Counter bytesDurable;
		// Count of all fetchKey clearRange operations to the storage engine.
		Counter kvClearRangesInFetchKeys;

		// Bytes fetched by fetchChangeFeed for data movements.
		Counter feedBytesFetched;

		Counter sampledBytesCleared;
		Counter atomicMutations, changeFeedMutations, changeFeedMutationsDurable;
		Counter updateBatches, updateVersions;
		Counter loops;
		Counter fetchWaitingMS, fetchWaitingCount, fetchExecutingMS, fetchExecutingCount;
		Counter readsRejected;
		Counter wrongShardServer;
		Counter fetchedVersions;
		Counter fetchesFromLogs;
		// The following counters measure how many of lookups in the getMappedRangeQueries are effective. "Miss"
		// means fallback if fallback is enabled, otherwise means failure (so that another layer could implement
		// fallback).
		Counter quickGetValueHit, quickGetValueMiss, quickGetKeyValuesHit, quickGetKeyValuesMiss;

		// The number of logical bytes returned from storage engine, in response to readRange operations.
		Counter kvScanBytes;
		// The number of logical bytes returned from storage engine, in response to readValue operations.
		Counter kvGetBytes;
		// The number of keys read from storage engine by eagerReads.
		Counter eagerReadsKeys;
		// The count of readValue operation to the storage engine.
		Counter kvGets;
		// The count of readValue operation to the storage engine.
		Counter kvScans;
		// The count of commit operation to the storage engine.
		Counter kvCommits;
		// The count of change feed reads that hit disk
		Counter changeFeedDiskReads;
		// The count of ChangeServerKeys actions.
		Counter changeServerKeysAssigned;
		Counter changeServerKeysUnassigned;

		// The count of 'set' inserted to pTree. The actual ptree.insert() number could be higher, because of the range
		// clear split, see metric pTreeClearSplits.
		Counter pTreeSets;
		// The count of clear range inserted to pTree
		Counter pTreeClears;
		// If set is within a range of clear, the clear is split. It's tracking the number of splits, the split could be
		// expensive.
		Counter pTreeClearSplits;

		std::unique_ptr<LatencySample> readLatencySample;
		std::unique_ptr<LatencySample> readKeyLatencySample;
		std::unique_ptr<LatencySample> readValueLatencySample;
		std::unique_ptr<LatencySample> readRangeLatencySample;
		std::unique_ptr<LatencySample> readVersionWaitSample;
		std::unique_ptr<LatencySample> readQueueWaitSample;
		std::unique_ptr<LatencySample> kvReadRangeLatencySample;
		std::unique_ptr<LatencySample> updateLatencySample;
		std::unique_ptr<LatencySample> updateEncryptionLatencySample;
		LatencyBands readLatencyBands;
		std::unique_ptr<LatencySample> mappedRangeSample; // Samples getMappedRange latency
		std::unique_ptr<LatencySample> mappedRangeRemoteSample; // Samples getMappedRange remote subquery latency
		std::unique_ptr<LatencySample> mappedRangeLocalSample; // Samples getMappedRange local subquery latency
		std::unique_ptr<LatencySample> ingestDurationLatencySample;

		explicit Counters(StorageServer* self)
		  : CommonStorageCounters("StorageServer", self->thisServerID.toString(), &self->metrics),
		    allQueries("QueryQueue", cc), systemKeyQueries("SystemKeyQueries", cc), getKeyQueries("GetKeyQueries", cc),
		    getValueQueries("GetValueQueries", cc), getRangeQueries("GetRangeQueries", cc),
		    getRangeSystemKeyQueries("GetRangeSystemKeyQueries", cc),
		    getMappedRangeQueries("GetMappedRangeQueries", cc), getRangeStreamQueries("GetRangeStreamQueries", cc),
		    lowPriorityQueries("LowPriorityQueries", cc), rowsQueried("RowsQueried", cc),
		    watchQueries("WatchQueries", cc), emptyQueries("EmptyQueries", cc), feedRowsQueried("FeedRowsQueried", cc),
		    feedBytesQueried("FeedBytesQueried", cc), feedStreamQueries("FeedStreamQueries", cc),
		    rejectedFeedStreamQueries("RejectedFeedStreamQueries", cc), feedVersionQueries("FeedVersionQueries", cc),
		    logicalBytesInput("LogicalBytesInput", cc), logicalBytesMoveInOverhead("LogicalBytesMoveInOverhead", cc),
		    kvCommitLogicalBytes("KVCommitLogicalBytes", cc), kvClearRanges("KVClearRanges", cc),
		    kvClearSingleKey("KVClearSingleKey", cc), kvSystemClearRanges("KVSystemClearRanges", cc),
		    bytesDurable("BytesDurable", cc), feedBytesFetched("FeedBytesFetched", cc),
		    sampledBytesCleared("SampledBytesCleared", cc), atomicMutations("AtomicMutations", cc),
		    changeFeedMutations("ChangeFeedMutations", cc),
		    changeFeedMutationsDurable("ChangeFeedMutationsDurable", cc), updateBatches("UpdateBatches", cc),
		    updateVersions("UpdateVersions", cc), loops("Loops", cc), fetchWaitingMS("FetchWaitingMS", cc),
		    fetchWaitingCount("FetchWaitingCount", cc), fetchExecutingMS("FetchExecutingMS", cc),
		    fetchExecutingCount("FetchExecutingCount", cc), readsRejected("ReadsRejected", cc),
		    wrongShardServer("WrongShardServer", cc), fetchedVersions("FetchedVersions", cc),
		    fetchesFromLogs("FetchesFromLogs", cc), quickGetValueHit("QuickGetValueHit", cc),
		    quickGetValueMiss("QuickGetValueMiss", cc), quickGetKeyValuesHit("QuickGetKeyValuesHit", cc),
		    quickGetKeyValuesMiss("QuickGetKeyValuesMiss", cc), kvScanBytes("KVScanBytes", cc),
		    kvGetBytes("KVGetBytes", cc), eagerReadsKeys("EagerReadsKeys", cc), kvGets("KVGets", cc),
		    kvScans("KVScans", cc), kvCommits("KVCommits", cc), changeFeedDiskReads("ChangeFeedDiskReads", cc),
		    getMappedRangeBytesQueried("GetMappedRangeBytesQueried", cc),
		    finishedGetMappedRangeQueries("FinishedGetMappedRangeQueries", cc),
		    finishedGetMappedRangeSecondaryQueries("FinishedGetMappedRangeSecondaryQueries", cc),
		    pTreeSets("PTreeSets", cc), pTreeClears("PTreeClears", cc), pTreeClearSplits("PTreeClearSplits", cc),
		    changeServerKeysAssigned("ChangeServerKeysAssigned", cc),
		    changeServerKeysUnassigned("ChangeServerKeysUnassigned", cc),
		    kvClearRangesInFetchKeys("KvClearRangesInFetchKeys", cc),
		    readLatencySample(std::make_unique<LatencySample>("ReadLatencyMetrics",
		                                                      self->thisServerID,
		                                                      SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                      SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readKeyLatencySample(std::make_unique<LatencySample>("GetKeyMetrics",
		                                                         self->thisServerID,
		                                                         SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                         SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readValueLatencySample(std::make_unique<LatencySample>("GetValueMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readRangeLatencySample(std::make_unique<LatencySample>("GetRangeMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readVersionWaitSample(std::make_unique<LatencySample>("ReadVersionWaitMetrics",
		                                                          self->thisServerID,
		                                                          SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                          SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readQueueWaitSample(std::make_unique<LatencySample>("ReadQueueWaitMetrics",
		                                                        self->thisServerID,
		                                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    kvReadRangeLatencySample(std::make_unique<LatencySample>("KVGetRangeMetrics",
		                                                             self->thisServerID,
		                                                             SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                             SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    updateLatencySample(std::make_unique<LatencySample>("UpdateLatencyMetrics",
		                                                        self->thisServerID,
		                                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    updateEncryptionLatencySample(
		        std::make_unique<LatencySample>("UpdateEncryptionLatencyMetrics",
		                                        self->thisServerID,
		                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readLatencyBands("ReadLatencyBands", self->thisServerID, SERVER_KNOBS->STORAGE_LOGGING_DELAY),
		    mappedRangeSample(std::make_unique<LatencySample>("GetMappedRangeMetrics",
		                                                      self->thisServerID,
		                                                      SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                      SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    mappedRangeRemoteSample(std::make_unique<LatencySample>("GetMappedRangeRemoteMetrics",
		                                                            self->thisServerID,
		                                                            SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                            SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    mappedRangeLocalSample(std::make_unique<LatencySample>("GetMappedRangeLocalMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    ingestDurationLatencySample(std::make_unique<LatencySample>("IngestDurationMetrics",
		                                                                self->thisServerID,
		                                                                SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                                SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)) {
			specialCounter(cc, "LastTLogVersion", [self]() { return self->lastTLogVersion; });
			specialCounter(cc, "Version", [self]() { return self->version.get(); });
			specialCounter(cc, "StorageVersion", [self]() { return self->storageVersion(); });
			specialCounter(cc, "DurableVersion", [self]() { return self->durableVersion.get(); });
			specialCounter(cc, "DesiredOldestVersion", [self]() { return self->desiredOldestVersion.get(); });
			specialCounter(cc, "VersionLag", [self]() { return self->versionLag; });
			specialCounter(cc, "LocalRate", [self] { return int64_t(self->currentRate() * 100); });

			specialCounter(
			    cc, "FetchKeysFetchActive", [self]() { return self->fetchKeysParallelismLock.activePermits(); });
			specialCounter(cc, "FetchKeysWaiting", [self]() { return self->fetchKeysParallelismLock.waiters(); });
			specialCounter(cc, "FetchKeysChangeFeedFetchActive", [self]() {
				return self->fetchKeysParallelismChangeFeedLock.activePermits();
			});
			specialCounter(cc, "FetchKeysFullFetchWaiting", [self]() {
				return self->fetchKeysParallelismChangeFeedLock.waiters();
			});
			specialCounter(cc, "ServeFetchCheckpointActive", [self]() {
				return self->serveFetchCheckpointParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeFetchCheckpointWaiting", [self]() {
				return self->serveFetchCheckpointParallelismLock.waiters();
			});
			specialCounter(cc, "ServeValidateStorageActive", [self]() {
				return self->serveAuditStorageParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeValidateStorageWaiting", [self]() {
				return self->serveAuditStorageParallelismLock.waiters();
			});
			specialCounter(
			    cc, "ServerBulkDumpActive", [self]() { return self->serveBulkDumpParallelismLock.activePermits(); });
			specialCounter(
			    cc, "ServerBulkDumpWaiting", [self]() { return self->serveBulkDumpParallelismLock.waiters(); });
			specialCounter(cc, "QueryQueueMax", [self]() { return self->getAndResetMaxQueryQueueSize(); });
			specialCounter(cc, "ActiveWatches", [self]() { return self->numWatches; });
			specialCounter(cc, "WatchBytes", [self]() { return self->watchBytes; });
			specialCounter(cc, "KvstoreSizeTotal", [self]() { return std::get<0>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreNodeTotal", [self]() { return std::get<1>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreInlineKey", [self]() { return std::get<2>(self->storage.getSize()); });
			specialCounter(cc, "ActiveChangeFeeds", [self]() { return self->uidChangeFeed.size(); });
			specialCounter(cc, "ActiveChangeFeedQueries", [self]() { return self->activeFeedQueries; });
			specialCounter(cc, "ChangeFeedMemoryBytes", [self]() { return self->changeFeedMemoryBytes; });
		}
	} counters;

	// Bytes read from storage engine when a storage server starts.
	int64_t bytesRestored = 0;

	Reference<EventCacheHolder> storageServerSourceTLogIDEventHolder;

	// Tenant metadata to manage connection to blob store for fetchKeys()
	BGTenantMap tenantData;

	std::shared_ptr<AccumulativeChecksumValidator> acsValidator = nullptr;

	std::shared_ptr<SSBulkLoadMetrics> bulkLoadMetrics = nullptr;

	StorageServer(IKeyValueStore* storage,
	              Reference<AsyncVar<ServerDBInfo> const> const& db,
	              StorageServerInterface const& ssi,
	              Reference<GetEncryptCipherKeysMonitor> encryptionMonitor)
	  : shardAware(false), locality(ssi.locality),
	    tlogCursorReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            TLOG_CURSOR_READS_LATENCY_HISTOGRAM,
	                                                            Histogram::Unit::milliseconds)),
	    ssVersionLockLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          SS_VERSION_LOCK_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    eagerReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                       EAGER_READS_LATENCY_HISTOGRAM,
	                                                       Histogram::Unit::milliseconds)),
	    fetchKeysPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  FETCH_KEYS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    tLogMsgsPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                 TLOG_MSGS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                 Histogram::Unit::milliseconds)),
	    storageUpdatesDurableLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  STORAGE_UPDATES_DURABLE_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    storageCommitLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          STORAGE_COMMIT_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    ssDurableVersionUpdateLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                   SS_DURABLE_VERSION_UPDATE_LATENCY_HISTOGRAM,
	                                                                   Histogram::Unit::milliseconds)),
	    readRangeBytesReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            SS_READ_RANGE_BYTES_RETURNED_HISTOGRAM,
	                                                            Histogram::Unit::bytes)),
	    readRangeBytesLimitHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                         SS_READ_RANGE_BYTES_LIMIT_HISTOGRAM,
	                                                         Histogram::Unit::bytes)),
	    readRangeKVPairsReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                              SS_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM,
	                                                              Histogram::Unit::bytes)),
	    tag(invalidTag), poppedAllAfter(std::numeric_limits<Version>::max()), cpuUsage(0.0), diskUsage(0.0),
	    storage(this, storage), shardChangeCounter(0), lastTLogVersion(0), lastVersionWithData(0), restoredVersion(0),
	    prevVersion(0), rebootAfterDurableVersion(std::numeric_limits<Version>::max()),
	    primaryLocality(tagLocalityInvalid), knownCommittedVersion(0), versionLag(0), logProtocol(0),
	    thisServerID(ssi.id()), tssInQuarantine(false), db(db), actors(false),
	    trackShardAssignmentMinVersion(invalidVersion), byteSampleClears(false, "\xff\xff\xff"_sr),
	    durableInProgress(Void()), watchBytes(0), numWatches(0), noRecentUpdates(false), lastUpdate(now()),
	    updateEagerReads(nullptr), fetchKeysParallelismLock(SERVER_KNOBS->FETCH_KEYS_PARALLELISM),
	    fetchKeysParallelismChangeFeedLock(SERVER_KNOBS->FETCH_KEYS_PARALLELISM_CHANGE_FEED),
	    fetchKeysBytesBudget(SERVER_KNOBS->STORAGE_FETCH_BYTES), fetchKeysBudgetUsed(false),
	    fetchKeysTotalCommitBytes(0), fetchKeysLimiter(SERVER_KNOBS->STORAGE_FETCH_KEYS_RATE_LIMIT),
	    serveFetchCheckpointParallelismLock(SERVER_KNOBS->SERVE_FETCH_CHECKPOINT_PARALLELISM),
	    ssLock(makeReference<PriorityMultiLock>(SERVER_KNOBS->STORAGE_SERVER_READ_CONCURRENCY,
	                                            SERVER_KNOBS->STORAGESERVER_READ_PRIORITIES)),
	    serveAuditStorageParallelismLock(SERVER_KNOBS->SERVE_AUDIT_STORAGE_PARALLELISM),
	    serveBulkDumpParallelismLock(SERVER_KNOBS->SS_SERVE_BULKDUMP_PARALLELISM),
	    instanceID(deterministicRandom()->randomUniqueID().first()), shuttingDown(false), behind(false),
	    versionBehind(false), debug_inApplyUpdate(false), debug_lastValidateTime(0), lastBytesInputEBrake(0),
	    lastDurableVersionEBrake(0), maxQueryQueue(0),
	    transactionTagCounter(ssi.id(),
	                          /*maxTagsTracked=*/SERVER_KNOBS->SS_THROTTLE_TAGS_TRACKED,
	                          /*minRateTracked=*/SERVER_KNOBS->MIN_TAG_READ_PAGES_RATE *
	                              CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE),
	    busiestWriteTagContext(ssi.id()), getEncryptCipherKeysMonitor(encryptionMonitor), counters(this),
	    storageServerSourceTLogIDEventHolder(
	        makeReference<EventCacheHolder>(ssi.id().toString() + "/StorageServerSourceTLogID")),
	    tenantData(db),
	    acsValidator(CLIENT_KNOBS->ENABLE_MUTATION_CHECKSUM && CLIENT_KNOBS->ENABLE_ACCUMULATIVE_CHECKSUM &&
	                         !SERVER_KNOBS->ENABLE_VERSION_VECTOR && !SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST
	                     ? std::make_shared<AccumulativeChecksumValidator>()
	                     : nullptr),
	    bulkLoadMetrics(std::make_shared<SSBulkLoadMetrics>()) {
		readPriorityRanks = parseStringToVector<int>(SERVER_KNOBS->STORAGESERVER_READTYPE_PRIORITY_MAP, ',');
		ASSERT(readPriorityRanks.size() > (int)ReadType::MAX);
		version.initMetric("StorageServer.Version"_sr, counters.cc.getId());
		oldestVersion.initMetric("StorageServer.OldestVersion"_sr, counters.cc.getId());
		durableVersion.initMetric("StorageServer.DurableVersion"_sr, counters.cc.getId());
		desiredOldestVersion.initMetric("StorageServer.DesiredOldestVersion"_sr, counters.cc.getId());

		newestAvailableVersion.insert(allKeys, invalidVersion);
		newestDirtyVersion.insert(allKeys, invalidVersion);
		if (storage->shardAware()) {
			addShard(ShardInfo::newShard(this, StorageServerShard::notAssigned(allKeys)));
		} else {
			addShard(ShardInfo::newNotAssigned(allKeys));
		}

		cx = openDBOnServer(db, TaskPriority::DefaultEndpoint, LockAware::True);

		this->storage.kvCommitLogicalBytes = &counters.kvCommitLogicalBytes;
		this->storage.kvClearRanges = &counters.kvClearRanges;
		this->storage.kvClearSingleKey = &counters.kvClearSingleKey;
		this->storage.kvGets = &counters.kvGets;
		this->storage.kvScans = &counters.kvScans;
		this->storage.kvCommits = &counters.kvCommits;
	}

	//~StorageServer() { fclose(log); }

	// Puts the given shard into shards.  The caller is responsible for adding shards
	//   for all ranges in shards.getAffectedRangesAfterInsertion(newShard->keys)), because these
	//   shards are invalidated by the call.
	void addShard(ShardInfo* newShard) {
		ASSERT(!newShard->range().empty());
		newShard->setChangeCounter(++shardChangeCounter);
		// TraceEvent("AddShard", this->thisServerID).detail("KeyBegin", newShard->keys.begin).detail("KeyEnd", newShard->keys.end).detail("State",newShard->isReadable() ? "Readable" : newShard->notAssigned() ? "NotAssigned" : "Adding").detail("Version", this->version.get());
		/*auto affected = shards.getAffectedRangesAfterInsertion( newShard->keys, Reference<ShardInfo>() );
		for(auto i = affected.begin(); i != affected.end(); ++i)
		    shards.insert( *i, Reference<ShardInfo>() );*/

		if (shardAware && newShard->notAssigned()) {
			auto sh = shards.intersectingRanges(newShard->range());
			for (auto it = sh.begin(); it != sh.end(); ++it) {
				if (it->value().isValid() && !it->value()->notAssigned()) {
					TraceEvent(SevVerbose, "StorageServerAddShardClear")
					    .detail("NewShardRange", newShard->range())
					    .detail("Range", it->value()->range())
					    .detail("ShardID", format("%016llx", it->value()->getShardId()))
					    .detail("NewShardID", format("%016llx", newShard->getDesiredShardId()))
					    .detail("NewShardActualID", format("%016llx", newShard->getShardId()));
				}
			}
		}

		Reference<ShardInfo> rShard(newShard);
		shards.insert(newShard->range(), rShard);
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 KeyRangeRef const& shard,
	                 UpdateEagerReadInfo* eagerReads);
	void setInitialVersion(Version ver) {
		version = ver;
		desiredOldestVersion = ver;
		oldestVersion = ver;
		durableVersion = ver;
		storageMinRecoverVersion = ver;
		lastVersionWithData = ver;
		restoredVersion = ver;

		mutableData().createNewVersion(ver);
		mutableData().forgetVersionsBefore(ver);
	}

	bool isTss() const { return tssPairID.present(); }

	bool isSSWithTSSPair() const { return ssPairID.present(); }

	void setSSWithTssPair(UID idOfTSS) { ssPairID = Optional<UID>(idOfTSS); }

	void clearSSWithTssPair() { ssPairID = Optional<UID>(); }

	// This is the maximum version that might be read from storage (the minimum version is durableVersion)
	Version storageVersion() const { return oldestVersion.get(); }

	bool isReadable(KeyRangeRef const& keys) const override {
		auto sh = shards.intersectingRanges(keys);
		for (auto i = sh.begin(); i != sh.end(); ++i)
			if (!i->value()->isReadable())
				return false;
		return true;
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRef const& key) {
		if (oldShardChangeCounter != shardChangeCounter && shards[key]->getChangeCounter() > oldShardChangeCounter) {
			CODE_PROBE(true, "shard change during getValueQ");
			throw wrong_shard_server();
		}
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRangeRef const& keys) {
		if (oldShardChangeCounter != shardChangeCounter) {
			auto sh = shards.intersectingRanges(keys);
			for (auto i = sh.begin(); i != sh.end(); ++i)
				if (i->value()->getChangeCounter() > oldShardChangeCounter) {
					CODE_PROBE(true, "shard change during range operation");
					throw wrong_shard_server();
				}
		}
	}

	Counter::Value queueSize() const { return counters.bytesInput.getValue() - counters.bytesDurable.getValue(); }

	// penalty used by loadBalance() to balance requests among SSes. We prefer SS with less write queue size.
	double getPenalty() const override {
		return std::max(std::max(1.0,
		                         (queueSize() - (SERVER_KNOBS->TARGET_BYTES_PER_STORAGE_SERVER -
		                                         2.0 * SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER)) /
		                             SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER),
		                (currentRate() < 1e-6 ? 1e6 : 1.0 / currentRate()));
	}

	// Normally the storage server prefers to serve read requests over making mutations
	// durable to disk. However, when the storage server falls to far behind on
	// making mutations durable, this function will change the priority to prefer writes.
	Future<Void> getQueryDelay() {
		if ((version.get() - durableVersion.get() > SERVER_KNOBS->LOW_PRIORITY_DURABILITY_LAG) ||
		    (queueSize() > SERVER_KNOBS->LOW_PRIORITY_STORAGE_QUEUE_BYTES)) {
			++counters.lowPriorityQueries;
			return delay(0, TaskPriority::LowPriorityRead);
		}
		return delay(0, TaskPriority::DefaultEndpoint);
	}

	template <class Reply>
	using isLoadBalancedReply = std::is_base_of<LoadBalancedReply, Reply>;

	template <class Reply>
	typename std::enable_if<isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double penalty) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		Reply reply;
		reply.error = err;
		reply.penalty = penalty;
		promise.send(reply);
	}

	template <class Reply>
	typename std::enable_if<!isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		promise.sendError(err);
	}

	template <class Request>
	bool shouldRead(const Request& request) {
		auto rate = currentRate();
		if (isTSSInQuarantine() || (rate < SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD &&
		                            deterministicRandom()->random01() >
		                                std::max(SERVER_KNOBS->STORAGE_DURABILITY_LAG_MIN_RATE,
		                                         rate / SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD))) {
			sendErrorWithPenalty(request.reply, server_overloaded(), getPenalty());
			++counters.readsRejected;
			return false;
		}
		return true;
	}

	template <class Request, class HandleFunction>
	Future<Void> readGuard(const Request& request, const HandleFunction& fun) {
		bool read = shouldRead(request);
		if (!read) {
			return Void();
		}
		return fun(this, request);
	}

	Version minFeedVersionForAddress(const NetworkAddress& addr) {
		auto& clientVersions = changeFeedClientVersions[addr];
		Version minVersion = version.get();
		for (auto& it : clientVersions) {
			/*fmt::print("SS {0} Blocked client {1} @ {2}\n",
			        thisServerID.toString().substr(0, 4),
			        it.first.toString().substr(0, 8),
			        it.second);*/
			minVersion = std::min(minVersion, it.second);
		}
		return minVersion;
	}

	// count in-memory change feed bytes towards storage queue size, for the purposes of memory management and
	// throttling
	void addFeedBytesAtVersion(int64_t bytes, Version version) {
		if (feedMemoryBytesByVersion.empty() || version != feedMemoryBytesByVersion.back().first) {
			ASSERT(feedMemoryBytesByVersion.empty() || version >= feedMemoryBytesByVersion.back().first);
			feedMemoryBytesByVersion.push_back({ version, 0 });
		}
		feedMemoryBytesByVersion.back().second += bytes;
		changeFeedMemoryBytes += bytes;
		if (SERVER_KNOBS->STORAGE_INCLUDE_FEED_STORAGE_QUEUE) {
			counters.bytesInput += bytes;
		}
	}

	void getSplitPoints(SplitRangeRequest const& req) override {
		try {
			checkTenantEntry(version.get(), req.tenantInfo, true);
			metrics.getSplitPoints(req, req.tenantInfo.prefix);
		} catch (Error& e) {
			req.reply.sendError(e);
		}
	}

	void maybeInjectTargetedRestart(Version v) {
		// inject an SS restart at most once per test
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    now() > g_simulator->injectTargetedSSRestartTime &&
		    rebootAfterDurableVersion == std::numeric_limits<Version>::max()) {
			CODE_PROBE(true, "Injecting SS targeted restart");
			TraceEvent("SimSSInjectTargetedRestart", thisServerID).detail("Version", v);
			rebootAfterDurableVersion = v;
			g_simulator->injectTargetedSSRestartTime = std::numeric_limits<double>::max();
		}
	}

	bool maybeInjectDelay() {
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation && now() > g_simulator->injectSSDelayTime) {
			CODE_PROBE(true, "Injecting SS targeted delay");
			TraceEvent("SimSSInjectDelay", thisServerID).log();
			g_simulator->injectSSDelayTime = std::numeric_limits<double>::max();
			return true;
		}
		return false;
	}

	Future<Void> waitMetricsTenantAware(const WaitMetricsRequest& req) override;

	void addActor(Future<Void> future) override { actors.add(future); }

	void getStorageMetrics(const GetStorageMetricsRequest& req) override {
		StorageBytes sb = storage.getStorageBytes();
		metrics.getStorageMetrics(req,
		                          sb,
		                          counters.bytesInput.getRate(),
		                          versionLag,
		                          lastUpdate,
		                          counters.bytesDurable.getValue(),
		                          counters.bytesInput.getValue(),
		                          bulkLoadMetrics->getOngoingTasks());
	}

	void getSplitMetrics(const SplitMetricsRequest& req) override { this->metrics.splitMetrics(req); }

	void getHotRangeMetrics(const ReadHotSubRangeRequest& req) override { this->metrics.getReadHotRanges(req); }

	int64_t getHotShardsMetrics(const KeyRange& range) override { return this->metrics.getHotShards(range); }

	// Used for recording shard assignment history for auditStorage
	std::vector<std::pair<Version, KeyRange>> shardAssignmentHistory;
	Version trackShardAssignmentMinVersion; // == invalidVersion means tracking stopped

	std::string printShardAssignmentHistory() {
		std::string toPrint = "";
		for (const auto& [version, range] : shardAssignmentHistory) {
			toPrint = toPrint + std::to_string(version) + " ";
		}
		return toPrint;
	}

	void startTrackShardAssignment(Version startVersion) {
		ASSERT(startVersion != invalidVersion);
		ASSERT(trackShardAssignmentMinVersion == invalidVersion);
		trackShardAssignmentMinVersion = startVersion;
		return;
	}

	void stopTrackShardAssignment() { trackShardAssignmentMinVersion = invalidVersion; }

	std::vector<std::pair<Version, KeyRangeRef>> getShardAssignmentHistory(Version early, Version later) {
		std::vector<std::pair<Version, KeyRangeRef>> res;
		for (const auto& shardAssignment : shardAssignmentHistory) {
			if (shardAssignment.first >= early && shardAssignment.first <= later) {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetOne", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first);
				res.push_back(shardAssignment);
			} else {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetSkip", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first)
				    .detail("EarlyVersion", early)
				    .detail("LaterVersion", later);
			}
		}
		TraceEvent(SevVerbose, "ShardAssignmentHistoryGetDone", thisServerID)
		    .detail("EarlyVersion", early)
		    .detail("LaterVersion", later)
		    .detail("HistoryTotalSize", shardAssignmentHistory.size())
		    .detail("HistoryTotal", printShardAssignmentHistory());
		return res;
	}
};

const StringRef StorageServer::CurrentRunningFetchKeys::emptyString = ""_sr;
const KeyRangeRef StorageServer::CurrentRunningFetchKeys::emptyKeyRange =
    KeyRangeRef(StorageServer::CurrentRunningFetchKeys::emptyString,
                StorageServer::CurrentRunningFetchKeys::emptyString);

// If and only if key:=value is in (storage+versionedData),    // NOT ACTUALLY: and key < allKeys.end,
//   and H(key) < |key+value|/bytesPerSample,
//     let sampledSize = max(|key+value|,bytesPerSample)
//     persistByteSampleKeys.begin()+key := sampledSize is in storage
//     (key,sampledSize) is in byteSample

// So P(key is sampled) * sampledSize == |key+value|

void StorageServer::byteSampleApplyMutation(MutationRef const& m, Version ver) {
	if (m.type == MutationRef::ClearRange)
		byteSampleApplyClear(KeyRangeRef(m.param1, m.param2), ver);
	else if (m.type == MutationRef::SetValue)
		byteSampleApplySet(KeyValueRef(m.param1, m.param2), ver);
	else
		ASSERT(false); // Mutation of unknown type modifying byte sample
}

// watchMap Operations
Reference<ServerWatchMetadata> StorageServer::getWatchMetadata(KeyRef key, int64_t tenantId) const {
	const WatchMapKey mapKey(tenantId, key);
	const auto it = watchMap.find(mapKey);
	if (it == watchMap.end())
		return Reference<ServerWatchMetadata>();
	return it->second;
}

KeyRef StorageServer::setWatchMetadata(Reference<ServerWatchMetadata> metadata) {
	KeyRef keyRef = metadata->key.contents();
	int64_t tenantId = metadata->tenantId;
	const WatchMapKey mapKey(tenantId, keyRef);

	watchMap[mapKey] = metadata;
	return keyRef;
}

void StorageServer::deleteWatchMetadata(KeyRef key, int64_t tenantId) {
	const WatchMapKey mapKey(tenantId, key);
	watchMap.erase(mapKey);
}

void StorageServer::clearWatchMetadata() {
	watchMap.clear();
}

#ifndef __INTEL_COMPILER
#pragma endregion
#endif

/////////////////////////////////// Validation ///////////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Validation
#endif
bool validateRange(StorageServer::VersionedData::ViewAtVersion const& view,
                   KeyRangeRef range,
                   Version version,
                   UID id,
                   Version minInsertVersion) {
	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion()

	//TraceEvent("ValidateRange", id).detail("KeyBegin", range.begin).detail("KeyEnd", range.end).detail("Version", version);
	KeyRef k;
	bool ok = true;
	bool kIsClear = false;
	auto i = view.lower_bound(range.begin);
	if (i != view.begin())
		--i;
	for (; i != view.end() && i.key() < range.end; ++i) {
		ASSERT(i.insertVersion() > minInsertVersion);
		if (kIsClear && i->isClearTo() ? i.key() <= k : i.key() < k) {
			TraceEvent(SevError, "InvalidRange", id)
			    .detail("Key1", k)
			    .detail("Key2", i.key())
			    .detail("Version", version);
			ok = false;
		}
		// ASSERT( i.key() >= k );
		kIsClear = i->isClearTo();
		k = kIsClear ? i->getEndKey() : i.key();
	}
	return ok;
}

void validate(StorageServer* data, bool force = false) {
	try {
		if (!data->shuttingDown && (force || (EXPENSIVE_VALIDATION))) {
			data->newestAvailableVersion.validateCoalesced();
			data->newestDirtyVersion.validateCoalesced();

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				TraceEvent(SevVerbose, "ValidateShard", data->thisServerID)
				    .detail("Range", s->range())
				    .detail("ShardID", format("%016llx", s->value()->getShardId()))
				    .detail("DesiredShardID", format("%016llx", s->value()->getDesiredShardId()))
				    .detail("ShardRange", s->value()->range())
				    .detail("ShardState", s->value()->debugDescribeState())
				    .log();
				ASSERT(s->value()->range() == s->range());
				ASSERT(!s->value()->range().empty());
				if (data->shardAware) {
					s->value()->validate();
				}
			}

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				if (s->value()->isReadable()) {
					auto ar = data->newestAvailableVersion.intersectingRanges(s->range());
					for (auto a = ar.begin(); a != ar.end(); ++a) {
						TraceEvent(SevVerbose, "ValidateShardReadable", data->thisServerID)
						    .detail("Range", s->range())
						    .detail("ShardRange", s->value()->range())
						    .detail("ShardState", s->value()->debugDescribeState())
						    .detail("AvailableRange", a->range())
						    .detail("AvailableVersion", a->value())
						    .log();
						ASSERT(a->value() == latestVersion);
					}
				}
			}

			// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
			// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
			//      contain later versions if applyUpdate is on the stack.
			ASSERT(data->data().getOldestVersion() == data->storageVersion());
			ASSERT(data->data().getLatestVersion() == data->version.get() ||
			       data->data().getLatestVersion() == data->version.get() + 1 ||
			       (data->debug_inApplyUpdate && data->data().getLatestVersion() > data->version.get()));

			auto latest = data->data().atLatest();

			// * Old shards are erased: versionedData.atLatest() has entries (sets or clear *begins*) only for keys in
			// readable or adding,transferred shards.
			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				ShardInfo* shard = s->value().getPtr();
				if (!shard->isInVersionedData()) {
					auto beginNext = latest.lower_bound(s->begin());
					auto endNext = latest.lower_bound(s->end());
					if (beginNext != endNext) {
						TraceEvent(SevError, "VF", data->thisServerID)
						    .detail("LastValidTime", data->debug_lastValidateTime)
						    .detail("KeyBegin", s->begin())
						    .detail("KeyEnd", s->end())
						    .detail("DbgState", shard->debugDescribeState())
						    .detail("FirstKey", beginNext.key())
						    .detail("LastKey", endNext != latest.end() ? endNext.key() : "End"_sr)
						    .detail("FirstInsertV", beginNext.insertVersion())
						    .detail("LastInsertV", endNext != latest.end() ? endNext.insertVersion() : invalidVersion);
					}
					ASSERT(beginNext == endNext);
				}

				if (shard->assigned() && data->shardAware) {
					TraceEvent(SevVerbose, "ValidateAssignedShard", data->thisServerID)
					    .detail("Range", shard->range())
					    .detailf("ShardID", "%016llx", shard->getShardId())
					    .detailf("DesiredShardID", "%016llx", shard->getDesiredShardId())
					    .detail("State", shard->debugDescribeState());
					ASSERT(shard->getShardId() != 0UL && shard->getDesiredShardId() != 0UL);
				}
			}

			// FIXME: do some change feed validation?

			latest.validate();
			validateRange(latest, allKeys, data->version.get(), data->thisServerID, data->durableVersion.get());

			data->debug_lastValidateTime = now();
		}
	} catch (...) {
		TraceEvent(SevError, "ValidationFailure", data->thisServerID)
		    .detail("LastValidTime", data->debug_lastValidateTime);
		throw;
	}
}
#ifndef __INTEL_COMPILER
#pragma endregion
#endif

void updateProcessStats(StorageServer* self) {
	if (g_network->isSimulated()) {
		// diskUsage and cpuUsage are not relevant in the simulator,
		// and relying on the actual values could break seed determinism
		self->cpuUsage = 100.0;
		self->diskUsage = 100.0;
		return;
	}

	SystemStatistics sysStats = getSystemStatistics();
	if (sysStats.initialized) {
		self->cpuUsage = 100 * sysStats.processCPUSeconds / sysStats.elapsed;
		self->diskUsage = 100 * std::max(0.0, (sysStats.elapsed - sysStats.processDiskIdleSeconds) / sysStats.elapsed);
	}
}

///////////////////////////////////// Queries /////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Queries
#endif

															#line 2766 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersionActor()
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionActorActor>
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActorState {
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActorState(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:WaitForVersion"_loc, spanContext)
															#line 2786 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionActorActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2232 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2803 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2240 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 2807 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 1;
															#line 2232 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 0, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2240 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 1, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2837 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2841 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2238 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2845 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2857 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2861 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2238 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2865 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2877 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2881 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2246 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2885 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2897 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2246 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2901 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 0, Void >::remove();
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3034 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForVersionActor()
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActor final : public Actor<Version>, public ActorCallback< WaitForVersionActorActor, 0, Void >, public ActorCallback< WaitForVersionActorActor, 1, Void >, public FastAllocated<WaitForVersionActorActor>, public WaitForVersionActorActorState<WaitForVersionActorActor> {
															#line 3039 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionActorActor>::operator new;
	using FastAllocated<WaitForVersionActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2734106421635830016UL, 1420926191567701248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionActorActor, 0, Void >;
friend struct ActorCallback< WaitForVersionActorActor, 1, Void >;
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActor(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 3057 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionActorActorState<WaitForVersionActorActor>(data, version, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17177306746367876352UL, 6989725333754748672UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionActor( StorageServer* const& data, Version const& version, SpanContext const& spanContext ) {
															#line 2229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionActorActor(data, version, spanContext));
															#line 3090 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2250 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// If the latest commit version that mutated the shard(s) being served by the specified storage
// server is below the client specified read version then do a read at the latest commit version
// of the storage server.
Version getRealReadVersion(VersionVector& ssLatestCommitVersions, Tag& tag, Version specifiedReadVersion) {
	Version realReadVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : specifiedReadVersion;
	ASSERT(realReadVersion <= specifiedReadVersion);
	return realReadVersion;
}

// Find the latest commit version of the given tag.
Version getLatestCommitVersion(VersionVector& ssLatestCommitVersions, Tag& tag) {
	Version commitVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : invalidVersion;
	return commitVersion;
}

Future<Version> waitForVersion(StorageServer* data, Version version, SpanContext spanContext) {
	if (version == latestVersion) {
		version = std::max(Version(1), data->version.get());
	}

	if (version < data->oldestVersion.get() || version <= 0) {
		// TraceEvent(SevDebug, "WFVThrow", data->thisServerID).detail("Version", version).detail("OldestVersion", data->oldestVersion.get());
		return transaction_too_old();
	} else if (version <= data->version.get()) {
		return version;
	}

	if ((data->behind || data->versionBehind) && version > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x").log();
	}
	return waitForVersionActor(data, version, spanContext);
}

Future<Version> waitForVersion(StorageServer* data,
                               Version commitVersion,
                               Version readVersion,
                               SpanContext spanContext) {
	ASSERT(commitVersion == invalidVersion || commitVersion < readVersion);

	if (commitVersion == invalidVersion) {
		return waitForVersion(data, readVersion, spanContext);
	}

	if (readVersion == latestVersion) {
		readVersion = std::max(Version(1), data->version.get());
	}

	if (readVersion < data->oldestVersion.get() || readVersion <= 0) {
		return transaction_too_old();
	} else {
		// It is correct to read any version between [commitVersion, readVersion],
		// because version vector guarantees no mutations between them.
		if (commitVersion < data->oldestVersion.get()) {
			if (data->version.get() < readVersion) {
				// Majority of the case, try using higher version to avoid
				// transaction_too_old error when oldestVersion advances.
				// BTW, any version in the range [oldestVersion, data->version.get()] is valid in this case.
				return data->version.get();
			} else {
				ASSERT(readVersion >= data->oldestVersion.get());
				return readVersion;
			}
		} else if (commitVersion <= data->version.get()) {
			return commitVersion;
		}
	}

	if ((data->behind || data->versionBehind) && commitVersion > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x");
	}
	return waitForVersionActor(data, std::max(commitVersion, data->oldestVersion.get()), spanContext);
}

															#line 3178 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionNoTooOldActor>
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActorState {
															#line 3185 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActorState(StorageServer* const& data,Version const& version) 
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 3194 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionNoTooOldActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2336 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 3209 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2337 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 3213 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2338 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version <= data->version.get())
															#line 3217 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2339 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3221 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
				this->~WaitForVersionNoTooOldActorState();
				static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2341 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2340 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2344 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 1;
															#line 2341 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 2344 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 3242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2342 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3265 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2342 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3277 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3289 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2346 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3293 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2350 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3297 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3305 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2346 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2350 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 0, Void >::remove();
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3442 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActor final : public Actor<Version>, public ActorCallback< WaitForVersionNoTooOldActor, 0, Void >, public ActorCallback< WaitForVersionNoTooOldActor, 1, Void >, public FastAllocated<WaitForVersionNoTooOldActor>, public WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor> {
															#line 3447 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionNoTooOldActor>::operator new;
	using FastAllocated<WaitForVersionNoTooOldActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13318222321209580032UL, 4822582592171690752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionNoTooOldActor, 0, Void >;
friend struct ActorCallback< WaitForVersionNoTooOldActor, 1, Void >;
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActor(StorageServer* const& data,Version const& version) 
															#line 3465 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14315589494631704320UL, 1805358750176616448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionNoTooOld");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionNoTooOld( StorageServer* const& data, Version const& version ) {
															#line 2334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionNoTooOldActor(data, version));
															#line 3498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2354 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 3503 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForMinVersion()
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForMinVersionActor>
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActorState {
															#line 3510 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActorState(StorageServer* const& data,Version const& version) 
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 3519 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitForMinVersionActorState() 
	{
		fdb_probe_actor_destroy("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2357 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 3534 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2358 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 3538 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2359 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version < data->oldestVersion.get() || version <= 0)
															#line 3542 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2360 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(data->oldestVersion.get()); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3546 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(data->oldestVersion.get());
				this->~WaitForMinVersionActorState();
				static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2361 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (version <= data->version.get())
															#line 3556 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2362 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3560 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
					this->~WaitForMinVersionActorState();
					static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 2365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2364 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3571 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2368 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3575 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 1;
															#line 2365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 0, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 2368 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 1, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 3582 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2366 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3605 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2366 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3617 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2369 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3629 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2370 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3633 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2374 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3637 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2369 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2370 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3649 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2374 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 0, Void >::remove();
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3782 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForMinVersion()
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActor final : public Actor<Version>, public ActorCallback< WaitForMinVersionActor, 0, Void >, public ActorCallback< WaitForMinVersionActor, 1, Void >, public FastAllocated<WaitForMinVersionActor>, public WaitForMinVersionActorState<WaitForMinVersionActor> {
															#line 3787 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForMinVersionActor>::operator new;
	using FastAllocated<WaitForMinVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17435970176551548416UL, 6239370310352859392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForMinVersionActor, 0, Void >;
friend struct ActorCallback< WaitForMinVersionActor, 1, Void >;
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActor(StorageServer* const& data,Version const& version) 
															#line 3805 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForMinVersionActorState<WaitForMinVersionActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3358323868325720064UL, 6852897307082981376UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForMinVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForMinVersionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForMinVersion( StorageServer* const& data, Version const& version ) {
															#line 2355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForMinVersionActor(data, version));
															#line 3838 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2378 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void StorageServer::checkTenantEntry(Version version, TenantInfo tenantInfo, bool lockAware) {
	if (tenantInfo.hasTenant()) {
		ASSERT(version == latestVersion || (version >= tenantMap.oldestVersion && version <= this->version.get()));
		auto view = tenantMap.at(version);
		auto itr = view.find(tenantInfo.tenantId);
		if (itr == view.end()) {
			TraceEvent(SevWarn, "StorageTenantNotFound", thisServerID)
			    .detail("Tenant", tenantInfo.tenantId)
			    .backtrace();
			CODE_PROBE(true, "Storage server tenant not found");
			throw tenant_not_found();
		} else if (!lockAware && itr->lockState == TenantAPI::TenantLockState::LOCKED) {
			CODE_PROBE(true, "Storage server access locked tenant without lock awareness");
			throw tenant_locked();
		}
	}
}

std::vector<StorageServerShard> StorageServer::getStorageServerShards(KeyRangeRef range) {
	std::vector<StorageServerShard> res;
	for (auto t : this->shards.intersectingRanges(range)) {
		res.push_back(t.value()->toStorageServerShard());
	}
	return res;
}

static Error dataMoveConflictError(const bool isTss) {
	if (isTss && g_network->isSimulated()) {
		// TSS data move conflicts can happen in both sim and prod, but in sim,
		// the sev40s cause failures of Joshua tests. We have been using please_reboot
		// as means to avoid sev40s, but semantically this is undesired because rebooting
		// will not fix/heal the TSS.
		// TODO: think of a proper TSS move conflict error that does not trigger
		// reboot but also avoids sev40. And throw that error regardless of sim or prod.
		return please_reboot();
	}
	return data_move_conflict();
}

std::shared_ptr<MoveInShard> StorageServer::getMoveInShard(const UID& dataMoveId,
                                                           const Version version,
                                                           const ConductBulkLoad conductBulkLoad) {
	for (auto& [id, moveInShard] : this->moveInShards) {
		if (moveInShard->dataMoveId() == dataMoveId && moveInShard->meta->createVersion == version) {
			return moveInShard;
		}
	}

	const UID id = deterministicRandom()->randomUniqueID();
	std::shared_ptr<MoveInShard> shard = std::make_shared<MoveInShard>(this, id, dataMoveId, version, conductBulkLoad);
	auto [it, inserted] = this->moveInShards.emplace(id, shard);
	ASSERT(inserted);
	TraceEvent(SevDebug, "SSNewMoveInShard", this->thisServerID)
	    .detail("MoveInShard", shard->toString())
	    .detail("ConductBulkLoad", conductBulkLoad);
	return shard;
}

															#line 3901 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getValueQ()
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetValueQActor>
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActorState {
															#line 3908 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActorState(StorageServer* const& data,GetValueRequest const& req) 
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2438 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 3919 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	~GetValueQActorState() 
	{
		fdb_probe_actor_destroy("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2439 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Span span("SS:getValue"_loc, req.spanContext);
															#line 3934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			try {
															#line 2444 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getValueQueries;
															#line 2445 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.allQueries;
															#line 2446 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.key.startsWith(systemKeys.begin))
															#line 3942 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2447 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					++data->counters.systemKeyQueries;
															#line 3946 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 2449 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 2454 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 2454 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3954 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 1;
															#line 2454 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 0, Void >*>(static_cast<GetValueQActor*>(this)));
															#line 3959 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2562 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, req.key.size() + resultSize);
															#line 2564 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 2566 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 2567 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySample->addMeasurement(duration);
															#line 2568 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readValueLatencySample->addMeasurement(duration);
															#line 2569 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 3998 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2570 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 2572 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes));
															#line 4004 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2575 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetValueQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetValueQActorState(); static_cast<GetValueQActor*>(this)->destroy(); return 0; }
															#line 4008 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetValueQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2555 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 4021 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2556 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4025 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2557 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 4029 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4046 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 4051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4062 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 4067 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2458 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 2459 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readQueueWaitSample->addMeasurement(queueWaitEnd - req.requestTime());
															#line 2461 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4155 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2462 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.DoRead");
															#line 4159 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = Optional<Value>();
															#line 2467 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 2468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, req.spanContext);
															#line 2468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4169 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 3;
															#line 2468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 2, Version >*>(static_cast<GetValueQActor*>(this)));
															#line 4174 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2469 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readVersionWaitSample->addMeasurement(g_network->timer() - queueWaitEnd);
															#line 2471 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4264 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2472 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterVersion");
															#line 4268 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2476 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkTenantEntry(version, req.tenantInfo, req.options.present() ? req.options.get().lockAware : false);
															#line 2477 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.tenantInfo.hasTenant())
															#line 4274 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2478 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.key = req.key.withPrefix(req.tenantInfo.prefix.get());
															#line 4278 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2480 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 2482 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!data->shards[req.key]->isReadable())
															#line 4284 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2484 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(wrong_shard_server(), loopDepth);
															#line 4288 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2487 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		path = 0;
															#line 2488 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto i = data->data().at(version).lastLessOrEqual(req.key);
															#line 2489 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (i && i->isValue() && i.key() == req.key)
															#line 4296 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			v = (Value)i->getValue();
															#line 2491 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			path = 1;
															#line 4302 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont7(loopDepth);
		}
		else
		{
															#line 2492 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!i || !i->isClearTo() || i->getEndKey() <= req.key)
															#line 4309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2493 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				path = 2;
															#line 2494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Optional<Value>> __when_expr_3 = data->storage.readValue(req.key, req.options);
															#line 2494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4317 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 4;
															#line 2494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 3, Optional<Value> >*>(static_cast<GetValueQActor*>(this)));
															#line 4322 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont12(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont4when1(Version const& __version,int loopDepth) 
	{
															#line 2468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 4337 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2505 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetValue", version, MutationRef(MutationRef::DebugKey, req.key, v.present() ? v.get() : "<null>"_sr), data->thisServerID);
															#line 2509 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetPath", version, MutationRef(MutationRef::DebugKey, req.key, path == 0 ? "0"_sr : path == 1 ? "1"_sr : "2"_sr), data->thisServerID);
															#line 2525 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (v.present())
															#line 4420 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2526 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.rowsQueried;
															#line 2527 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			resultSize = v.get().size();
															#line 2528 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.bytesQueried += resultSize;
															#line 4428 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2530 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 4434 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2533 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 4438 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2535 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = v.present() ? std::max((int64_t)(req.key.size() + v.get().size()), SERVER_KNOBS->EMPTY_READ_PENALTY) : SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 2538 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(req.key, bytesReadPerKSecond);
															#line 4444 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2541 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4448 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2542 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterRead");
															#line 4452 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2547 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto cached = data->cachedRangeMap[req.key];
															#line 2551 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetValueReply reply(v, cached);
															#line 2552 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.penalty = data->getPenalty();
															#line 2553 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 4462 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Optional<Value> const& vv,int loopDepth) 
	{
															#line 2495 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2497 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4479 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2499 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4485 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2501 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2502 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4491 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Optional<Value> && vv,int loopDepth) 
	{
															#line 2495 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2497 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4502 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2499 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4508 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2501 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2502 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4514 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> const& vv,int loopDepth) 
	{
		loopDepth = a_body1cont13(vv, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> && vv,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(vv), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 3, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueRequest req;
															#line 2438 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 2455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 2458 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 2466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<Value> v;
															#line 2468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2480 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 2487 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int path;
															#line 4625 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getValueQ()
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActor final : public Actor<Void>, public ActorCallback< GetValueQActor, 0, Void >, public ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetValueQActor, 2, Version >, public ActorCallback< GetValueQActor, 3, Optional<Value> >, public FastAllocated<GetValueQActor>, public GetValueQActorState<GetValueQActor> {
															#line 4630 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetValueQActor>::operator new;
	using FastAllocated<GetValueQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15736395631966835968UL, 3542452627688087040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValueQActor, 0, Void >;
friend struct ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetValueQActor, 2, Version >;
friend struct ActorCallback< GetValueQActor, 3, Optional<Value> >;
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActor(StorageServer* const& data,GetValueRequest const& req) 
															#line 4650 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetValueQActorState<GetValueQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17920039446934625280UL, 7083594275336435200UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValueQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValueQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetValueQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetValueQActor, 3, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getValueQ( StorageServer* const& data, GetValueRequest const& req ) {
															#line 2437 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetValueQActor(data, req));
															#line 4686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Pessimistic estimate the number of overhead bytes used by each
// watch. Watch key references are stored in an AsyncMap<Key,bool>, and actors
// must be kept alive until the watch is finished.
extern size_t WATCH_OVERHEAD_WATCHQ, WATCH_OVERHEAD_WATCHIMPL;

															#line 4696 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via watchWaitForValueChange()
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchWaitForValueChangeActor>
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActorState {
															#line 4703 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActorState(StorageServer* const& data,SpanContext const& parent,KeyRef const& key,int64_t const& tenantId) 
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parent(parent),
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   tenantId(tenantId),
															#line 2584 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanLocation("SS:watchWaitForValueChange"_loc),
															#line 2585 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span(spanLocation, parent),
															#line 2586 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   metadata(data->getWatchMetadata(key, tenantId))
															#line 4722 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	~WatchWaitForValueChangeActorState() 
	{
		fdb_probe_actor_destroy("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2587 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2588 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.Before");
															#line 4741 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2592 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			originalMetadataVersion = metadata->version;
															#line 2593 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(waitForVersionNoTooOld(data, metadata->version));
															#line 2593 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4749 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 1;
															#line 2593 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 0, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4754 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchWaitForValueChangeActorState();
		static_cast<WatchWaitForValueChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2594 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4777 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2595 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4781 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2599 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2600 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2601 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4789 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2602 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2604 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2605 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4799 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2594 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4808 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2595 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4812 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2599 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2600 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2601 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2602 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2604 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2605 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4830 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2607 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2608 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto view = data->tenantMap.at(latestVersion);
															#line 2609 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (view.find(tenantId) == view.end())
															#line 4928 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2610 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "Watched tenant removed");
															#line 2611 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(tenant_removed(), loopDepth);
															#line 4934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 2614 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			metadata = data->getWatchMetadata(key, tenantId);
															#line 2615 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			latest = data->version.get();
															#line 2616 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			options.debugID = metadata->debugID;
															#line 2618 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(latest >= minVersion && latest < data->data().latestVersion, "Starting watch loop with latestVersion > data->version", probe::decoration::rare);
															#line 2621 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetValueRequest getReq( span.context, TenantInfo(), metadata->key, latest, metadata->tags, options, VersionVector());
															#line 2623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue = getValueQ( data, getReq);
															#line 2625 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetValueReply> __when_expr_1 = getReq.reply.getFuture();
															#line 2625 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4953 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 2;
															#line 2625 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4958 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 2703 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2704 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4975 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2705 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4979 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2708 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = data->version.whenAtLeast(data->data().latestVersion);
															#line 2708 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4985 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 4;
															#line 2708 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 3, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2696 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() != error_code_transaction_too_old)
															#line 5000 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2697 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5004 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2700 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reading a watched key failed with transaction_too_old");
															#line 5008 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply const& reply,int loopDepth) 
	{
															#line 2626 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2628 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 5025 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2629 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2630 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 5031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2632 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 5035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2633 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 5039 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2636 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2643 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 5045 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2644 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 5049 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2652 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 5055 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2654 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 5059 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2655 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 5063 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2656 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 5073 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2659 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 5079 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 5083 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 2666 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 5090 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2667 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2668 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 5096 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2671 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2675 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 5102 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2677 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 5106 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2679 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 5110 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2681 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 5114 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2683 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 5118 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 5122 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2687 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 5126 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5132 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 5137 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply && reply,int loopDepth) 
	{
															#line 2626 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2628 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 5154 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2629 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2630 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 5160 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2632 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 5164 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2633 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 5168 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2636 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2643 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 5174 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2644 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 5178 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2652 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 5184 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2654 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 5188 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2655 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 5192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2656 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 5202 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2659 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 5208 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 5212 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 2666 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 5219 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2667 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2668 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 5225 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2671 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2675 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 5231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2677 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 5235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2679 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 5239 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2681 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 5243 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2683 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 5247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 5251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2687 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 5255 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5261 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 5266 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2692 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= watchBytes;
															#line 2693 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(e, loopDepth);
															#line 5365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont15(Void const& _,int loopDepth) 
	{
															#line 2690 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont15(Void && _,int loopDepth) 
	{
															#line 2690 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5388 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont15cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont20(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont20(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont20(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont20(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parent;
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef key;
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t tenantId;
															#line 2584 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Location spanLocation;
															#line 2585 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2586 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<ServerWatchMetadata> metadata;
															#line 2592 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version originalMetadataVersion;
															#line 2599 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version minVersion;
															#line 2600 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> watchFuture;
															#line 2604 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadOptions options;
															#line 2615 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version latest;
															#line 2623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> getValue;
															#line 2671 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t watchBytes;
															#line 5609 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via watchWaitForValueChange()
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActor final : public Actor<Version>, public ActorCallback< WatchWaitForValueChangeActor, 0, Void >, public ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >, public ActorCallback< WatchWaitForValueChangeActor, 2, Void >, public ActorCallback< WatchWaitForValueChangeActor, 3, Void >, public FastAllocated<WatchWaitForValueChangeActor>, public WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor> {
															#line 5614 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WatchWaitForValueChangeActor>::operator new;
	using FastAllocated<WatchWaitForValueChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5492273144312895488UL, 9677858225469669376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchWaitForValueChangeActor, 0, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 2, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 3, Void >;
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActor(StorageServer* const& data,SpanContext const& parent,KeyRef const& key,int64_t const& tenantId) 
															#line 5634 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>(data, parent, key, tenantId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(510625173218191360UL, 2956297564950870528UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchWaitForValueChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> watchWaitForValueChange( StorageServer* const& data, SpanContext const& parent, KeyRef const& key, int64_t const& tenantId ) {
															#line 2583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WatchWaitForValueChangeActor(data, parent, key, tenantId));
															#line 5670 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2711 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void checkCancelWatchImpl(StorageServer* data, WatchValueRequest req) {
	Reference<ServerWatchMetadata> metadata = data->getWatchMetadata(req.key.contents(), req.tenantInfo.tenantId);
	if (metadata.isValid() && metadata->versionPromise.getFutureReferenceCount() == 1) {
		// last watch timed out so cancel watch_impl and delete key from the map
		data->deleteWatchMetadata(req.key.contents(), req.tenantInfo.tenantId);
		metadata->watch_impl.cancel();
	}
}

															#line 5684 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via watchValueSendReply()
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchValueSendReplyActor>
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActorState {
															#line 5691 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActorState(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resp(resp),
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:watchValue"_loc, spanContext),
															#line 2726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   startTime(now())
															#line 5708 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	~WatchValueSendReplyActorState() 
	{
		fdb_probe_actor_destroy("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2727 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.watchQueries;
															#line 2728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->numWatches;
															#line 2729 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes += WATCH_OVERHEAD_WATCHQ;
															#line 2731 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 5729 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double timeoutDelay = -1;
															#line 2733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->noRecentUpdates.get())
															#line 5761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			timeoutDelay = std::max(CLIENT_KNOBS->FAST_WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5765 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2735 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!BUGGIFY)
															#line 5771 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2736 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				timeoutDelay = std::max(CLIENT_KNOBS->WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
		try {
															#line 2741 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_0 = resp;
															#line 2740 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5783 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 2749 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = timeoutDelay < 0 ? Never() : delay(timeoutDelay);
															#line 5787 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 2757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = data->noRecentUpdates.onChange();
															#line 5791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
			static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 1;
															#line 2741 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 0, Version >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2749 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 1, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 2, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 5800 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2760 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			checkCancelWatchImpl(data, req);
															#line 2762 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			--data->numWatches;
															#line 2764 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 5828 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2765 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2766 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 2767 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5838 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WatchValueSendReplyActorState();
			static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& ver,int loopDepth) 
	{
															#line 2743 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2744 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2745 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2746 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2747 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && ver,int loopDepth) 
	{
															#line 2743 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2744 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2745 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2746 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2747 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2751 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2752 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2753 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2754 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2751 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2752 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2753 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2754 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5930 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state > 0) static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 0;
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 0, Version >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 1, Void >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueRequest req;
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Version> resp;
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 6154 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via watchValueSendReply()
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActor final : public Actor<Void>, public ActorCallback< WatchValueSendReplyActor, 0, Version >, public ActorCallback< WatchValueSendReplyActor, 1, Void >, public ActorCallback< WatchValueSendReplyActor, 2, Void >, public FastAllocated<WatchValueSendReplyActor>, public WatchValueSendReplyActorState<WatchValueSendReplyActor> {
															#line 6159 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WatchValueSendReplyActor>::operator new;
	using FastAllocated<WatchValueSendReplyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(761334473073730560UL, 7879471509401638656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchValueSendReplyActor, 0, Version >;
friend struct ActorCallback< WatchValueSendReplyActor, 1, Void >;
friend struct ActorCallback< WatchValueSendReplyActor, 2, Void >;
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActor(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 6178 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   WatchValueSendReplyActorState<WatchValueSendReplyActor>(data, req, resp, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3738460070669966080UL, 521894726446841856UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchValueSendReply");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchValueSendReplyActor, 0, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> watchValueSendReply( StorageServer* const& data, WatchValueRequest const& req, Future<Version> const& resp, SpanContext const& spanContext ) {
															#line 2721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new WatchValueSendReplyActor(data, req, resp, spanContext));
															#line 6211 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2771 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Finds a checkpoint.
															#line 6217 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getCheckpointQ()
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetCheckpointQActor>
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActorState {
															#line 6224 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActorState(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 6233 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~GetCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->durableVersion.whenAtLeast(req.version + 1);
															#line 2775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6250 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetCheckpointQActor, 0, Void >*>(static_cast<GetCheckpointQActor*>(this)));
															#line 6255 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2777 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2781 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2782 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2783 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 6284 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2784 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2785 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6290 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2790 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2792 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6306 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2795 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2796 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6312 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
			}
															#line 2801 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6318 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2802 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6322 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2777 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2781 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2782 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2783 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 6344 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2784 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2785 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6350 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2790 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2792 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6366 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2795 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2796 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6372 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
			}
															#line 2801 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6378 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2802 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6382 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<GetCheckpointQActor*>(this)->ActorCallback< GetCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2810 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6473 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2805 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 6486 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2808 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(e);
															#line 6494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointRequest req;
															#line 6522 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getCheckpointQ()
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActor final : public Actor<Void>, public ActorCallback< GetCheckpointQActor, 0, Void >, public FastAllocated<GetCheckpointQActor>, public GetCheckpointQActorState<GetCheckpointQActor> {
															#line 6527 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetCheckpointQActor>::operator new;
	using FastAllocated<GetCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2698173643520802816UL, 10820883321478772992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetCheckpointQActor, 0, Void >;
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActor(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 6544 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetCheckpointQActorState<GetCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7171777638048396032UL, 5824655659692947712UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getCheckpointQ( StorageServer* const& self, GetCheckpointRequest const& req ) {
															#line 2773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetCheckpointQActor(self, req));
															#line 6577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2812 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Delete the checkpoint from disk, as well as all related persisted meta data.
															#line 6583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteCheckpointQ()
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class DeleteCheckpointQActor>
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActorState {
															#line 6590 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActorState(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   checkpoint(checkpoint)
															#line 6601 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~DeleteCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::Low);
															#line 2815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6618 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 0, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6648 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6664 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6669 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2819 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6766 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2819 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6783 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6789 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6794 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2830 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		persistCheckpointKey = Key(persistCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2831 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		pendingCheckpointKey = Key(persistPendingCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto& mLV = self->addVersionToMutationLog(self->data().getLatestVersion());
															#line 2833 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, pendingCheckpointKey, keyAfter(pendingCheckpointKey)));
															#line 2835 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, persistCheckpointKey, keyAfter(persistCheckpointKey)));
															#line 2837 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointEnd", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2839 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<DeleteCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteCheckpointQActorState(); static_cast<DeleteCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6896 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<DeleteCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2827 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 6909 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	CheckpointMetaData checkpoint;
															#line 2830 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key persistCheckpointKey;
															#line 2831 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key pendingCheckpointKey;
															#line 7029 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via deleteCheckpointQ()
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActor final : public Actor<Void>, public ActorCallback< DeleteCheckpointQActor, 0, Void >, public ActorCallback< DeleteCheckpointQActor, 1, Void >, public ActorCallback< DeleteCheckpointQActor, 2, Void >, public FastAllocated<DeleteCheckpointQActor>, public DeleteCheckpointQActorState<DeleteCheckpointQActor> {
															#line 7034 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<DeleteCheckpointQActor>::operator new;
	using FastAllocated<DeleteCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4313748965758850048UL, 2553995566312536064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteCheckpointQActor, 0, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 1, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 2, Void >;
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActor(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 7053 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteCheckpointQActorState<DeleteCheckpointQActor>(self, version, checkpoint),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7967604653483254528UL, 11896452891661470208UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> deleteCheckpointQ( StorageServer* const& self, Version const& version, CheckpointMetaData const& checkpoint ) {
															#line 2814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new DeleteCheckpointQActor(self, version, checkpoint));
															#line 7088 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2841 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointRequests.
															#line 7094 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchCheckpointQ()
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointQActor>
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActorState {
															#line 7101 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActorState(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 7110 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2844 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2848 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = nullptr;
															#line 2849 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalSize = 0;
															#line 2851 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2854 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const auto it = self->checkpoints.find(req.checkpointID);
															#line 2855 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 7135 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2856 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 2857 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2858 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7143 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~FetchCheckpointQActorState();
				static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 2862 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reader = newCheckpointReader(it->second, CheckpointAsKeyValues::False, deterministicRandom()->randomUniqueID());
															#line 2863 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = reader->init(req.token);
															#line 2863 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7156 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2863 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 0, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7161 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2897 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2897 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 5;
															#line 2897 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 4, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7197 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2874 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 7207 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2876 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("TotalSize", totalSize) .detail("Token", req.token);
															#line 7213 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1Catch2cont1(loopDepth);
			}
			else
			{
															#line 2881 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() != error_code_operation_obsolete)
															#line 7220 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2882 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarnAlways, "ServerFetchCheckpointFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2886 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (canReplyWith(e))
															#line 7226 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2887 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						req.reply.sendError(e);
															#line 7230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 2889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					err = e;
															#line 2890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (reader != nullptr)
															#line 7236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						StrictFuture<Void> __when_expr_3 = reader->close();
															#line 2891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_3.get(), loopDepth); };
						static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 4;
															#line 2891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 3, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1Catch2cont4(loopDepth);
					}
				}
				else
				{
					loopDepth = a_body1Catch2cont3(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2865 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7273 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2865 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7282 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 2866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = reader->nextChunk(CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7375 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7380 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
															#line 2867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = req.reply.onReady();
															#line 2867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7391 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 2, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7396 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 2866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data = __data;
															#line 7405 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2868 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2869 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2871 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2868 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2869 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2871 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7505 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1Catch2cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont3(int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont4(int loopDepth) 
	{
															#line 2893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 7601 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2898 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7696 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2898 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7708 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointRequest req;
															#line 2848 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2849 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t totalSize;
															#line 2866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Standalone<StringRef> data;
															#line 2889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Error err;
															#line 7803 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via fetchCheckpointQ()
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointQActor, 0, Void >, public ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >, public ActorCallback< FetchCheckpointQActor, 2, Void >, public ActorCallback< FetchCheckpointQActor, 3, Void >, public ActorCallback< FetchCheckpointQActor, 4, Void >, public FastAllocated<FetchCheckpointQActor>, public FetchCheckpointQActorState<FetchCheckpointQActor> {
															#line 7808 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FetchCheckpointQActor>::operator new;
	using FastAllocated<FetchCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13158055276133657600UL, 7639468450281519104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< FetchCheckpointQActor, 2, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 4, Void >;
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActor(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 7829 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   FetchCheckpointQActorState<FetchCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13937557724080409344UL, 5112112820403248640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointQ( StorageServer* const& self, FetchCheckpointRequest const& req ) {
															#line 2843 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointQActor(self, req));
															#line 7866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2900 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointKeyValuesRequest, reads local checkpoint and sends it to the client over wire.
															#line 7872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointKeyValuesQActor>
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActorState {
															#line 7879 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActorState(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 7888 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2903 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->serveFetchCheckpointParallelismLock.take(TaskPriority::DefaultYield);
															#line 2903 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7905 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 2903 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2904 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2913 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2914 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 7941 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2915 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 7949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2920 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2921 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 7961 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2923 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 7965 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2925 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2926 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 7973 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 7977 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7988 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2904 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2913 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2914 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 8011 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2915 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 8019 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2920 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2921 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 8031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2923 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 8035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2925 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2926 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 8043 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 8047 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8053 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8058 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2975 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter.reset();
															#line 2976 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reader->inUse())
															#line 8150 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2977 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders.erase(req.checkpointID);
															#line 2978 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2978 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8158 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont2when1(__when_expr_4.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 2978 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8163 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2958 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 8178 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2959 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2960 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "ServeFetchCheckpointKeyValuesEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 8184 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 2965 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarnAlways, "ServerFetchCheckpointKeyValuesFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2969 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (canReplyWith(e))
															#line 8192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2970 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(e);
															#line 8196 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2932 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 8215 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2932 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 8226 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1(int loopDepth) 
	{
															#line 2935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_2 = iter->nextBatch(CLIENT_KNOBS->REPLY_BYTE_LIMIT, CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 2935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1(int loopDepth) 
	{
															#line 2937 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.empty())
															#line 8333 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2938 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "FetchCheckpontKeyValuesReadRange", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("FirstReturnedKey", res.front().key) .detail("LastReturnedKey", res.back().key) .detail("Size", res.size());
															#line 8337 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "FetchCheckpontKeyValuesEmptyRange", self->thisServerID) .detail("CheckpointID", req.checkpointID);
															#line 8343 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = req.reply.onReady();
															#line 2948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8349 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 2948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8354 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 2935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = __res;
															#line 8363 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2951 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2952 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8448 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2955 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8452 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2951 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2952 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8467 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2955 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8471 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 2980 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 8555 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesRequest req;
															#line 2904 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 2920 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unique_ptr<ICheckpointIterator> iter;
															#line 2935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	RangeResult res;
															#line 8662 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >, public ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >, public FastAllocated<FetchCheckpointKeyValuesQActor>, public FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor> {
															#line 8667 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator new;
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6788367801790951680UL, 5541660875888030720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >;
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActor(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 8688 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12412750992388461056UL, 6903825642391152640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointKeyValuesQ( StorageServer* const& self, FetchCheckpointKeyValuesRequest const& req ) {
															#line 2902 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointKeyValuesQActor(self, req));
															#line 8725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2982 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 8730 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via overlappingChangeFeedsQ()
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class OverlappingChangeFeedsQActor>
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class OverlappingChangeFeedsQActorState {
															#line 8737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	OverlappingChangeFeedsQActorState(StorageServer* const& data,OverlappingChangeFeedsRequest const& req) 
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 8746 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this));

	}
	~OverlappingChangeFeedsQActorState() 
	{
		fdb_probe_actor_destroy("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2984 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 2984 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8763 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 1;
															#line 2984 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< OverlappingChangeFeedsQActor, 0, Void >*>(static_cast<OverlappingChangeFeedsQActor*>(this)));
															#line 8768 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~OverlappingChangeFeedsQActorState();
		static_cast<OverlappingChangeFeedsQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(waitForVersionNoTooOld(data, req.minVersion));
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8794 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 2;
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*>(static_cast<OverlappingChangeFeedsQActor*>(this)));
															#line 8799 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(waitForVersionNoTooOld(data, req.minVersion));
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 2;
															#line 2986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*>(static_cast<OverlappingChangeFeedsQActor*>(this)));
															#line 8822 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state > 0) static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 0;
		static_cast<OverlappingChangeFeedsQActor*>(this)->ActorCallback< OverlappingChangeFeedsQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< OverlappingChangeFeedsQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2994 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!data->isReadable(req.range))
															#line 8912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2995 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(wrong_shard_server());
															#line 2996 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<OverlappingChangeFeedsQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OverlappingChangeFeedsQActorState(); static_cast<OverlappingChangeFeedsQActor*>(this)->destroy(); return 0; }
															#line 8918 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<OverlappingChangeFeedsQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~OverlappingChangeFeedsQActorState();
			static_cast<OverlappingChangeFeedsQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2999 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version metadataWaitVersion = invalidVersion;
															#line 3001 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto ranges = data->keyChangeFeed.intersectingRanges(req.range);
															#line 3002 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::map<Key, std::tuple<KeyRange, Version, Version, Version>> rangeIds;
															#line 3003 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto r : ranges ) {
															#line 3004 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for( auto& it : r.value() ) {
															#line 3005 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!it->removing)
															#line 8936 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3009 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					metadataWaitVersion = std::max(metadataWaitVersion, it->metadataCreateVersion);
															#line 3013 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					Version stopVersion;
															#line 3014 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (it->stopVersion != MAX_VERSION && req.minVersion > it->stopVersion)
															#line 8944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3015 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						stopVersion = it->stopVersion;
															#line 3016 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						metadataWaitVersion = std::max(metadataWaitVersion, stopVersion);
															#line 8950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
					else
					{
															#line 3018 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						stopVersion = MAX_VERSION;
															#line 8956 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 3021 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					rangeIds[it->id] = std::tuple(it->range, it->emptyVersion, stopVersion, it->metadataVersion);
															#line 8960 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3022 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (it->destroyed && it->metadataVersion > metadataWaitVersion)
															#line 8966 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3025 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						CODE_PROBE(true, "Overlapping Change Feeds ensuring destroy isn't rolled back");
															#line 3026 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						metadataWaitVersion = it->metadataVersion;
															#line 8972 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 3030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply = OverlappingChangeFeedsReply();
															#line 3031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.feedMetadataVersion = data->version.get();
															#line 3032 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& it : rangeIds ) {
															#line 3033 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.feeds.push_back_deep(reply.arena, OverlappingChangeFeedEntry(it.first, std::get<0>(it.second), std::get<1>(it.second), std::get<2>(it.second), std::get<3>(it.second)));
															#line 3039 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "OverlappingChangeFeedEntry", data->thisServerID) .detail("MinVersion", req.minVersion) .detail("FeedID", it.first) .detail("Range", std::get<0>(it.second)) .detail("EmptyVersion", std::get<1>(it.second)) .detail("StopVersion", std::get<2>(it.second)) .detail("FeedMetadataVersion", std::get<3>(it.second));
															#line 8987 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3049 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadataWaitVersion != invalidVersion && metadataWaitVersion > data->desiredOldestVersion.get())
															#line 8991 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3050 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "overlapping change feeds waiting for metadata version to be safe from rollback");
															#line 3051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = data->desiredOldestVersion.whenAtLeast(metadataWaitVersion);
															#line 3051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8999 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 3;
															#line 3051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< OverlappingChangeFeedsQActor, 2, Void >*>(static_cast<OverlappingChangeFeedsQActor*>(this)));
															#line 9004 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2988 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 9019 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2989 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 9023 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(e);
															#line 2991 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<OverlappingChangeFeedsQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OverlappingChangeFeedsQActorState(); static_cast<OverlappingChangeFeedsQActor*>(this)->destroy(); return 0; }
															#line 9029 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<OverlappingChangeFeedsQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~OverlappingChangeFeedsQActorState();
			static_cast<OverlappingChangeFeedsQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state > 0) static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 0;
		static_cast<OverlappingChangeFeedsQActor*>(this)->ActorCallback< OverlappingChangeFeedsQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 3053 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 3054 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<OverlappingChangeFeedsQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OverlappingChangeFeedsQActorState(); static_cast<OverlappingChangeFeedsQActor*>(this)->destroy(); return 0; }
															#line 9149 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<OverlappingChangeFeedsQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~OverlappingChangeFeedsQActorState();
		static_cast<OverlappingChangeFeedsQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state > 0) static_cast<OverlappingChangeFeedsQActor*>(this)->actor_wait_state = 0;
		static_cast<OverlappingChangeFeedsQActor*>(this)->ActorCallback< OverlappingChangeFeedsQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< OverlappingChangeFeedsQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< OverlappingChangeFeedsQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	OverlappingChangeFeedsRequest req;
															#line 3030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	OverlappingChangeFeedsReply reply;
															#line 9250 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via overlappingChangeFeedsQ()
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class OverlappingChangeFeedsQActor final : public Actor<Void>, public ActorCallback< OverlappingChangeFeedsQActor, 0, Void >, public ActorCallback< OverlappingChangeFeedsQActor, 1, Void >, public ActorCallback< OverlappingChangeFeedsQActor, 2, Void >, public FastAllocated<OverlappingChangeFeedsQActor>, public OverlappingChangeFeedsQActorState<OverlappingChangeFeedsQActor> {
															#line 9255 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<OverlappingChangeFeedsQActor>::operator new;
	using FastAllocated<OverlappingChangeFeedsQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17139478567389500416UL, 15597892012890869248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< OverlappingChangeFeedsQActor, 0, Void >;
friend struct ActorCallback< OverlappingChangeFeedsQActor, 1, Void >;
friend struct ActorCallback< OverlappingChangeFeedsQActor, 2, Void >;
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	OverlappingChangeFeedsQActor(StorageServer* const& data,OverlappingChangeFeedsRequest const& req) 
															#line 9274 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   OverlappingChangeFeedsQActorState<OverlappingChangeFeedsQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17829927576781930496UL, 18314691442875751424UL);
		ActorExecutionContextHelper __helper(static_cast<OverlappingChangeFeedsQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("overlappingChangeFeedsQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("overlappingChangeFeedsQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< OverlappingChangeFeedsQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< OverlappingChangeFeedsQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< OverlappingChangeFeedsQActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> overlappingChangeFeedsQ( StorageServer* const& data, OverlappingChangeFeedsRequest const& req ) {
															#line 2983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new OverlappingChangeFeedsQActor(data, req));
															#line 9309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3056 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

MutationsAndVersionRef filterMutations(Arena& arena,
                                       EncryptedMutationsAndVersionRef const& m,
                                       KeyRange const& range,
                                       bool encrypted,
                                       int commonPrefixLength) {
	if (m.mutations.size() == 1 && m.mutations.back().param1 == lastEpochEndPrivateKey) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}

	Optional<VectorRef<MutationRef>> modifiedMutations;
	for (int i = 0; i < m.mutations.size(); i++) {
		if (m.mutations[i].type == MutationRef::SetValue) {
			bool inRange = range.begin.compareSuffix(m.mutations[i].param1, commonPrefixLength) <= 0 &&
			               m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0;
			if (modifiedMutations.present() && inRange) {
				modifiedMutations.get().push_back(
				    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
			}
			if (!modifiedMutations.present() && !inRange) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
		} else {
			ASSERT(m.mutations[i].type == MutationRef::ClearRange);
			// param1 < range.begin || param2 > range.end
			if (!modifiedMutations.present() &&
			    (m.mutations[i].param1.compareSuffix(range.begin, commonPrefixLength) < 0 ||
			     m.mutations[i].param2.compareSuffix(range.end, commonPrefixLength) > 0)) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
			if (modifiedMutations.present()) {
				// param1 < range.end && range.begin < param2
				if (m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0 &&
				    range.begin.compareSuffix(m.mutations[i].param2, commonPrefixLength) < 0) {
					StringRef clearBegin = m.mutations[i].param1;
					StringRef clearEnd = m.mutations[i].param2;
					bool modified = false;
					if (clearBegin.compareSuffix(range.begin, commonPrefixLength) < 0) {
						clearBegin = range.begin;
						modified = true;
					}
					if (range.end.compareSuffix(clearEnd, commonPrefixLength) < 0) {
						clearEnd = range.end;
						modified = true;
					}
					if (modified) {
						MutationRef clearMutation = MutationRef(MutationRef::ClearRange, clearBegin, clearEnd);
						if (encrypted && m.encrypted.present() && m.encrypted.get()[i].isEncrypted()) {
							clearMutation = clearMutation.encrypt(m.cipherKeys[i], arena, BlobCipherMetrics::TLOG);
						}
						modifiedMutations.get().push_back(arena, clearMutation);
					} else {
						modifiedMutations.get().push_back(
						    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
					}
				}
			}
		}
	}
	if (modifiedMutations.present()) {
		return MutationsAndVersionRef(modifiedMutations.get(), m.version, m.knownCommittedVersion);
	}
	if (!encrypted || !m.encrypted.present()) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}
	return MutationsAndVersionRef(m.encrypted.get(), m.version, m.knownCommittedVersion);
}

// set this for VERY verbose logs on change feed SS reads
#define DEBUG_CF_TRACE false

// To easily find if a change feed read missed data. Set the CF to the feedId, the key to the missing key, and the
// version to the version the mutation is missing at.
#define DO_DEBUG_CF_MISSING false
#define DEBUG_CF_MISSING_CF ""_sr
#define DEBUG_CF_MISSING_KEY ""_sr
#define DEBUG_CF_MISSING_VERSION invalidVersion
#define DEBUG_CF_MISSING(cfId, keyRange, beginVersion, lastVersion)                                                    \
	DO_DEBUG_CF_MISSING&& cfId.printable().substr(0, 6) ==                                                             \
	        DEBUG_CF_MISSING_CF&& keyRange.contains(DEBUG_CF_MISSING_KEY) &&                                           \
	    beginVersion <= DEBUG_CF_MISSING_VERSION&& lastVersion >= DEBUG_CF_MISSING_VERSION

// efficiently searches for the change feed mutation start point at begin version
static std::deque<Standalone<EncryptedMutationsAndVersionRef>>::const_iterator searchChangeFeedStart(
    std::deque<Standalone<EncryptedMutationsAndVersionRef>> const& mutations,
    Version beginVersion,
    bool atLatest) {

	if (mutations.empty() || beginVersion > mutations.back().version) {
		return mutations.end();
	} else if (beginVersion <= mutations.front().version) {
		return mutations.begin();
	}

	EncryptedMutationsAndVersionRef searchKey;
	searchKey.version = beginVersion;
	if (atLatest) {
		int jump = 1;
		// exponential search backwards, because atLatest means the new mutations are likely only at the very end
		auto lastEnd = mutations.end();
		auto currentEnd = mutations.end() - 1;
		while (currentEnd > mutations.begin()) {
			if (beginVersion >= currentEnd->version) {
				break;
			}
			lastEnd = currentEnd + 1;
			jump = std::min((int)(currentEnd - mutations.begin()), jump);
			currentEnd -= jump;
			jump <<= 1;
		}
		auto ret = std::lower_bound(currentEnd, lastEnd, searchKey, EncryptedMutationsAndVersionRef::OrderByVersion());
		// TODO REMOVE: for validation
		if (ret != mutations.end()) {
			if (ret->version < beginVersion) {
				fmt::print("ERROR: {0}) {1} < {2}\n", ret - mutations.begin(), ret->version, beginVersion);
			}
			ASSERT(ret->version >= beginVersion);
		}
		if (ret != mutations.begin()) {
			if ((ret - 1)->version >= beginVersion) {
				fmt::print("ERROR: {0}) {1} >= {2}\n", (ret - mutations.begin()) - 1, (ret - 1)->version, beginVersion);
			}
			ASSERT((ret - 1)->version < beginVersion);
		}
		return ret;
	} else {
		// binary search
		return std::lower_bound(
		    mutations.begin(), mutations.end(), searchKey, EncryptedMutationsAndVersionRef::OrderByVersion());
	}
}

// The normal read case for a change feed stream query is that it will first read the disk portion, which is at a lower
// version than the memory portion, and then will effectively switch to reading only the memory portion. The complexity
// lies in the fact that the feed does not know the switchover point ahead of time before reading from disk, and the
// switchover point is constantly changing as the SS persists the in-memory data to disk. As a result, the
// implementation first reads from memory, then reads from disk if necessary, then merges the result and potentially
// discards the in-memory read data if the disk data is large and behind the in-memory data. The goal of
// FeedDiskReadState is that we want to skip doing the full memory read if we still have a lot of disk reads to catch up
// on. In the DISK_CATCHUP phase, the feed query will read only the first row from memory, to
// determine if it's hit the switchover point, instead of reading (potentially) both in the normal phase.  We also want
// to default to the normal behavior at the start in case there is not a lot of disk data. This guarantees that if we
// somehow incorrectly went into DISK_CATCHUP when there wasn't much more data on disk, we only have one cycle of
// getChangeFeedMutations in the incorrect mode that returns a smaller result before switching to NORMAL mode.
//
// Put another way, the state transitions are:
//
// STARTING ->
//   DISK_CATCHUP (if after the first read, there is more disk data to read before the first memory data)
//   NORMAL (otherwise)
// DISK_CATCHUP ->
//   still DISK_CATCHUP (if there is still more disk data to read before the first memory data)
//   NORMAL (otherwise)
// NORMAL -> NORMAL (always)
enum FeedDiskReadState { STARTING, NORMAL, DISK_CATCHUP };

															#line 9479 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getChangeFeedMutations()
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetChangeFeedMutationsActor>
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetChangeFeedMutationsActorState {
															#line 9486 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetChangeFeedMutationsActorState(StorageServer* const& data,Reference<ChangeFeedInfo> const& feedInfo,ChangeFeedStreamRequest const& req,bool const& atLatest,bool const& doFilterMutations,int const& commonFeedPrefixLength,FeedDiskReadState* const& feedDiskReadState) 
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   feedInfo(feedInfo),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   atLatest(atLatest),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   doFilterMutations(doFilterMutations),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   commonFeedPrefixLength(commonFeedPrefixLength),
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   feedDiskReadState(feedDiskReadState),
															#line 3229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   reply(),
															#line 3230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   memoryReply(),
															#line 3231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   remainingLimitBytes(CLIENT_KNOBS->REPLY_BYTE_LIMIT),
															#line 3232 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   remainingDurableBytes(CLIENT_KNOBS->REPLY_BYTE_LIMIT),
															#line 3233 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   startVersion(data->version.get())
															#line 9515 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getChangeFeedMutations", reinterpret_cast<unsigned long>(this));

	}
	~GetChangeFeedMutationsActorState() 
	{
		fdb_probe_actor_destroy("getChangeFeedMutations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (DEBUG_CF_TRACE)
															#line 9530 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevDebug, "TraceChangeFeedMutationsBegin", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 9534 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3246 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->version.get() < req.begin)
															#line 9538 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(req.begin);
															#line 3247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9544 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 1;
															#line 3247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 0, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 9549 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetChangeFeedMutationsActorState();
		static_cast<GetChangeFeedMutationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3254 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 3255 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!data->isReadable(req.range))
															#line 9579 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3256 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(wrong_shard_server(), loopDepth);
															#line 9583 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3259 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (feedInfo->removing)
															#line 9587 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3260 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(unknown_change_feed(), loopDepth);
															#line 9591 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3265 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		dequeVersion = data->version.get();
															#line 3266 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		dequeKnownCommit = data->knownCommittedVersion.get();
															#line 3267 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		emptyVersion = feedInfo->emptyVersion;
															#line 3268 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		durableValidationVersion = std::min(data->durableVersion.get(), feedInfo->durableFetchVersion.get());
															#line 3269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastMemoryVersion = invalidVersion;
															#line 3270 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastMemoryKnownCommitted = invalidVersion;
															#line 3271 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version fetchStorageVersion = std::max(feedInfo->fetchVersion, feedInfo->durableFetchVersion.get());
															#line 3272 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		doValidation = EXPENSIVE_VALIDATION;
															#line 3274 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 9611 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3275 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "TraceChangeFeedMutationsDetails", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("AtLatest", atLatest) .detail("DequeVersion", dequeVersion) .detail("EmptyVersion", feedInfo->emptyVersion) .detail("StorageVersion", feedInfo->storageVersion) .detail("DurableVersion", feedInfo->durableVersion) .detail("FetchStorageVersion", fetchStorageVersion) .detail("FetchVersion", feedInfo->fetchVersion) .detail("DurableFetchVersion", feedInfo->durableFetchVersion.get()) .detail("DurableValidationVersion", durableValidationVersion) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 9615 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3294 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.end > emptyVersion + 1)
															#line 9619 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3295 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto it = searchChangeFeedStart(feedInfo->mutations, req.begin, atLatest);
															#line 3296 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != feedInfo->mutations.end();) {
															#line 3298 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (it->version >= req.end || it->version > dequeVersion || remainingLimitBytes <= 0)
															#line 9627 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
					break;
				}
															#line 3301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if ((*feedDiskReadState) == FeedDiskReadState::DISK_CATCHUP && !memoryReply.mutations.empty())
															#line 9633 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3303 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					remainingLimitBytes = -1;
															#line 9637 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
															#line 3308 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				remainingLimitBytes -= sizeof(MutationsAndVersionRef) + it->expectedSize();
															#line 3310 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				MutationsAndVersionRef m;
															#line 3311 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (doFilterMutations)
															#line 9646 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3312 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					m = filterMutations(memoryReply.arena, *it, req.range, req.encrypted, commonFeedPrefixLength);
															#line 9650 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3314 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					m = MutationsAndVersionRef(req.encrypted && it->encrypted.present() ? it->encrypted.get() : it->mutations, it->version, it->knownCommittedVersion);
															#line 9656 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (m.mutations.size())
															#line 9660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3320 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					memoryReply.arena.dependsOn(it->arena());
															#line 3321 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					memoryReply.mutations.push_back(memoryReply.arena, m);
															#line 9666 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				lastMemoryVersion = m.version;
															#line 3325 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				lastMemoryKnownCommitted = m.knownCommittedVersion;
															#line 3326 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				it++;
															#line 9674 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3330 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readDurable = feedInfo->durableVersion != invalidVersion && req.begin <= feedInfo->durableVersion;
															#line 3331 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readFetched = req.begin <= fetchStorageVersion && !atLatest;
															#line 3332 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		waitFetched = false;
															#line 3333 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.end > emptyVersion + 1 && (readDurable || readFetched))
															#line 9685 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (readFetched && req.begin <= feedInfo->fetchVersion)
															#line 9689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3335 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				waitFetched = true;
															#line 3339 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(req.begin <= feedInfo->fetchVersion);
															#line 3340 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "getChangeFeedMutations before fetched data durable");
															#line 3343 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_2 = feedInfo->durableFetchVersion.whenAtLeast(feedInfo->fetchVersion);
															#line 3343 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9701 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
				static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 3;
															#line 3343 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 2, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 9706 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont16(loopDepth);
			}
		}
		else
		{
															#line 3590 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply = memoryReply;
															#line 3591 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*feedDiskReadState = FeedDiskReadState::NORMAL;
															#line 3595 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if ((reply.mutations.empty() || reply.mutations.back().version < lastMemoryVersion) && remainingLimitBytes <= 0)
															#line 9722 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3597 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Memory feed adding empty version after memory filtered", probe::decoration::rare);
															#line 3598 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reply.mutations.push_back(reply.arena, MutationsAndVersionRef(lastMemoryVersion, lastMemoryKnownCommitted));
															#line 9728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9741 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 2;
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 1, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 9746 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 2;
															#line 3251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 1, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 9762 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 3602 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		bool gotAll = remainingLimitBytes > 0 && remainingDurableBytes > 0 && data->version.get() == startVersion;
															#line 3603 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version finalVersion = std::min(req.end - 1, dequeVersion);
															#line 3604 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if ((reply.mutations.empty() || reply.mutations.back().version < finalVersion) && remainingLimitBytes > 0 && remainingDurableBytes > 0)
															#line 9937 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3606 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Change feed adding empty version after empty results");
															#line 3607 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.mutations.push_back( reply.arena, MutationsAndVersionRef(finalVersion, finalVersion == dequeVersion ? dequeKnownCommit : 0));
															#line 3610 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->version.get() == startVersion)
															#line 9945 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3611 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				gotAll = true;
															#line 9949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3618 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!req.canReadPopped && req.begin <= feedInfo->emptyVersion)
															#line 9954 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Version minVersion = data->minFeedVersionForAddress(req.reply.getEndpoint().getPrimaryAddress());
															#line 3624 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			bool ok = atLatest && minVersion > feedInfo->emptyVersion;
															#line 3625 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(ok, "feed popped while valid read waiting");
															#line 3626 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(!ok, "feed popped while invalid read waiting");
															#line 3627 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!ok)
															#line 9966 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3628 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ChangeFeedMutationsPopped", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("EmptyVersion", feedInfo->emptyVersion) .detail("AtLatest", atLatest) .detail("MinVersionSent", minVersion);
															#line 9970 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3642 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (MUTATION_TRACKING_ENABLED)
															#line 9975 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3643 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for( auto& mutations : reply.mutations ) {
															#line 3644 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				for( auto& m : mutations.mutations ) {
															#line 3645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					DEBUG_MUTATION("ChangeFeedSSRead", mutations.version, m, data->thisServerID) .detail("ChangeFeedID", req.rangeID) .detail("StreamUID", req.id) .detail("ReqBegin", req.begin) .detail("ReqEnd", req.end) .detail("ReqRange", req.range);
															#line 9983 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
		}
															#line 3655 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_MISSING(req.rangeID, req.range, req.begin, reply.mutations.back().version) && !req.canReadPopped)
															#line 9989 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3656 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			bool foundVersion = false;
															#line 3657 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			bool foundKey = false;
															#line 3658 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for( auto& it : reply.mutations ) {
															#line 3659 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (it.version == DEBUG_CF_MISSING_VERSION)
															#line 9999 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					foundVersion = true;
															#line 3661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					for( auto& m : it.mutations ) {
															#line 3662 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (m.type == MutationRef::SetValue && m.param1 == DEBUG_CF_MISSING_KEY)
															#line 10007 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						{
															#line 3663 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							foundKey = true;
															#line 10011 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							break;
						}
					}
					break;
				}
			}
															#line 3670 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!foundVersion || !foundKey)
															#line 10020 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3671 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				fmt::print("ERROR: SS {0} CF {1} SQ {2} missing {3} @ {4} from request for [{5} - {6}) {7} - {8}\n", data->thisServerID.toString().substr(0, 4), req.rangeID.printable().substr(0, 6), req.id.toString().substr(0, 8), foundVersion ? "key" : "version", static_cast<int64_t>(DEBUG_CF_MISSING_VERSION), req.range.begin.printable(), req.range.end.printable(), req.begin, req.end);
															#line 3681 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				fmt::print("ERROR: {0} versions in response {1} - {2}:\n", reply.mutations.size(), reply.mutations.front().version, reply.mutations.back().version);
															#line 3685 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				for( auto& it : reply.mutations ) {
															#line 3686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					fmt::print("ERROR:    {0} ({1}){2}\n", it.version, it.mutations.size(), it.version == DEBUG_CF_MISSING_VERSION ? "<-------" : "");
															#line 10030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			else
			{
															#line 3692 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				fmt::print("DBG: SS {0} CF {1} SQ {2} correct @ {3} from request for [{4} - {5}) {6} - {7}\n", data->thisServerID.toString().substr(0, 4), req.rangeID.printable().substr(0, 6), req.id.toString().substr(0, 8), static_cast<int64_t>(DEBUG_CF_MISSING_VERSION), req.range.begin.printable(), req.range.end.printable(), req.begin, req.end);
															#line 10037 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3704 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.popVersion = feedInfo->emptyVersion + 1;
															#line 3706 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 10044 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3707 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "ChangeFeedMutationsDone", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("FirstVersion", reply.mutations.empty() ? invalidVersion : reply.mutations.front().version) .detail("LastVersion", reply.mutations.empty() ? invalidVersion : reply.mutations.back().version) .detail("PopVersion", reply.popVersion) .detail("Count", reply.mutations.size()) .detail("GotAll", gotAll) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 10048 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3724 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetChangeFeedMutationsActor*>(this)->SAV<std::pair<ChangeFeedStreamReply, bool>>::futures) { (void)(std::make_pair(reply, gotAll)); this->~GetChangeFeedMutationsActorState(); static_cast<GetChangeFeedMutationsActor*>(this)->destroy(); return 0; }
															#line 10052 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetChangeFeedMutationsActor*>(this)->SAV< std::pair<ChangeFeedStreamReply, bool> >::value()) std::pair<ChangeFeedStreamReply, bool>(std::make_pair(reply, gotAll));
		this->~GetChangeFeedMutationsActorState();
		static_cast<GetChangeFeedMutationsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont16(int loopDepth) 
	{
															#line 3348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_4 = data->getReadLock(req.options);
															#line 3348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10066 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont16when1(__when_expr_4.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 5;
															#line 3348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10071 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont17(Void const& _,int loopDepth) 
	{
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0);
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10082 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont17when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 4;
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 3, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10087 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont17(Void && _,int loopDepth) 
	{
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0);
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10098 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont17when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 4;
															#line 3345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 3, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10103 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont17(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont17(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont17cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1cont17cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1cont17when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont17cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont17when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont17cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont17when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont17when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont16cont1(int loopDepth) 
	{
															#line 3364 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(0);
															#line 3364 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10276 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont16cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 6;
															#line 3364 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 5, Void >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10281 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont16when1(PriorityMultiLock::Lock const& __ssReadLock,int loopDepth) 
	{
															#line 3348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ssReadLock = __ssReadLock;
															#line 10290 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont16cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont16when1(PriorityMultiLock::Lock && __ssReadLock,int loopDepth) 
	{
		ssReadLock = std::move(__ssReadLock);
		loopDepth = a_body1cont16cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont16when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont16when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont16cont2(Void const& _,int loopDepth) 
	{
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_6 = data->storage.readRange(KeyRangeRef(changeFeedDurableKey(req.rangeID, std::max(req.begin, emptyVersion)), changeFeedDurableKey(req.rangeID, req.end)), 1 << 30, remainingDurableBytes, req.options);
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10371 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont16cont2when1(__when_expr_6.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 7;
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10376 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont16cont2(Void && _,int loopDepth) 
	{
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_6 = data->storage.readRange(KeyRangeRef(changeFeedDurableKey(req.rangeID, std::max(req.begin, emptyVersion)), changeFeedDurableKey(req.rangeID, req.end)), 1 << 30, remainingDurableBytes, req.options);
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10387 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont16cont2when1(__when_expr_6.get(), loopDepth); };
		static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 7;
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10392 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont16cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont16cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont16cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont16cont3(int loopDepth) 
	{
															#line 3371 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ssReadLock.release();
															#line 3372 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += res.logicalSize();
															#line 3373 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.changeFeedDiskReads;
															#line 3375 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!req.range.empty())
															#line 10482 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3376 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->checkChangeCounter(changeCounter, req.range);
															#line 10486 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		decodedMutations = std::vector<std::pair<Standalone<VectorRef<MutationRef>>, Version>>();
															#line 3380 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::unordered_set<BlobCipherDetails> cipherDetails;
															#line 3381 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		decodedMutations.reserve(res.size());
															#line 3382 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& kv : res ) {
															#line 3383 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			decodedMutations.push_back(decodeChangeFeedDurableValue(kv.value));
															#line 3384 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (doFilterMutations || !req.encrypted)
															#line 10500 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3385 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				for( auto& m : decodedMutations.back().first ) {
															#line 3386 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(data->encryptionMode.present());
															#line 3387 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(!data->encryptionMode.get().isEncryptionEnabled() || m.isEncrypted() || isBackupLogMutation(m) || mutationForKey(m, lastEpochEndPrivateKey));
															#line 3389 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (m.isEncrypted())
															#line 10510 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3390 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						m.updateEncryptCipherDetails(cipherDetails);
															#line 10514 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 3396 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cipherMap = std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>();
															#line 3397 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (cipherDetails.size())
															#line 10523 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3398 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>> __when_expr_7 = GetEncryptCipherKeys<ServerDBInfo>::getEncryptCipherKeys( data->db, cipherDetails, BlobCipherMetrics::TLOG);
															#line 3398 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10529 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont16cont3when1(__when_expr_7.get(), loopDepth); };
			static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 8;
															#line 3398 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*>(static_cast<GetChangeFeedMutationsActor*>(this)));
															#line 10534 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont16cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont16cont2when1(RangeResult const& __res,int loopDepth) 
	{
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = __res;
															#line 10548 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont16cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont2when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1cont16cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont16cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont16cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont16cont4(int loopDepth) 
	{
															#line 3404 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int memoryVerifyIdx = 0;
															#line 3406 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version lastVersion = req.begin - 1;
															#line 3407 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version lastKnownCommitted = invalidVersion;
															#line 3408 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();i++) {
															#line 3409 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Key id;
															#line 3410 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Version version, knownCommittedVersion;
															#line 3411 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VectorRef<MutationRef>> mutations;
															#line 3412 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VectorRef<MutationRef>> encryptedMutations;
															#line 3413 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::vector<TextAndHeaderCipherKeys> cipherKeys;
															#line 3414 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::tie(id, version) = decodeChangeFeedDurableKey(res[i].key);
															#line 3415 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::tie(encryptedMutations, knownCommittedVersion) = decodedMutations[i];
															#line 3416 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			cipherKeys.resize(encryptedMutations.size());
															#line 3418 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (doFilterMutations || !req.encrypted)
															#line 10651 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3419 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				mutations.resize(mutations.arena(), encryptedMutations.size());
															#line 3420 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				for(int j = 0;j < encryptedMutations.size();j++) {
															#line 3421 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(data->encryptionMode.present());
															#line 3422 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(!data->encryptionMode.get().isEncryptionEnabled() || encryptedMutations[j].isEncrypted() || isBackupLogMutation(encryptedMutations[j]) || mutationForKey(encryptedMutations[j], lastEpochEndPrivateKey));
															#line 3425 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (encryptedMutations[j].isEncrypted())
															#line 10663 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3426 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						cipherKeys[j] = encryptedMutations[j].getCipherKeys(cipherMap);
															#line 3427 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						mutations[j] = encryptedMutations[j].decrypt(cipherKeys[j], mutations.arena(), BlobCipherMetrics::TLOG);
															#line 10669 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
					else
					{
															#line 3430 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						mutations[j] = encryptedMutations[j];
															#line 10675 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
			else
			{
															#line 3434 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				mutations = encryptedMutations;
															#line 10683 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3438 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;doValidation && memoryVerifyIdx < memoryReply.mutations.size() && version > memoryReply.mutations[memoryVerifyIdx].version;) {
															#line 3440 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.canReadPopped)
															#line 10689 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3443 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					memoryVerifyIdx++;
															#line 10693 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					continue;
				}
															#line 3450 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (waitFetched && feedInfo->emptyVersion > emptyVersion && memoryReply.mutations[memoryVerifyIdx].version <= feedInfo->emptyVersion)
															#line 10698 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3452 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					memoryVerifyIdx++;
															#line 10702 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					continue;
				}
				else
				{
															#line 3455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					fmt::print("ERROR: SS {0} CF {1} SQ {2} has mutation at {3} in memory but not on disk (next disk " "is {4}) (emptyVersion={5}, emptyBefore={6})!\n", data->thisServerID.toString().substr(0, 4), req.rangeID.printable().substr(0, 6), req.id.toString().substr(0, 8), memoryReply.mutations[memoryVerifyIdx].version, version, feedInfo->emptyVersion, emptyVersion);
															#line 3465 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					fmt::print("  Memory: ({})\n", memoryReply.mutations[memoryVerifyIdx].mutations.size());
															#line 3466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					for( auto& it : memoryReply.mutations[memoryVerifyIdx].mutations ) {
															#line 3467 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (it.type == MutationRef::SetValue)
															#line 10715 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						{
															#line 3468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							fmt::print("    {}=\n", it.param1.printable());
															#line 10719 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						}
						else
						{
															#line 3470 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							fmt::print("    {} - {}\n", it.param1.printable(), it.param2.printable());
															#line 10725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						}
					}
															#line 3473 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(false);
															#line 10730 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 3477 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			MutationsAndVersionRef m;
															#line 3478 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (doFilterMutations)
															#line 10737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3479 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				m = filterMutations(reply.arena, EncryptedMutationsAndVersionRef( mutations, encryptedMutations, cipherKeys, version, knownCommittedVersion), req.range, req.encrypted, commonFeedPrefixLength);
															#line 10741 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3486 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				m = MutationsAndVersionRef( req.encrypted ? encryptedMutations : mutations, version, knownCommittedVersion);
															#line 10747 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3489 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (m.mutations.size())
															#line 10751 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reply.arena.dependsOn(mutations.arena());
															#line 3491 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reply.arena.dependsOn(encryptedMutations.arena());
															#line 3492 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reply.mutations.push_back(reply.arena, m);
															#line 3494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (doValidation && memoryVerifyIdx < memoryReply.mutations.size() && version == memoryReply.mutations[memoryVerifyIdx].version)
															#line 10761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					memoryVerifyIdx++;
															#line 10765 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			else
			{
															#line 3500 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (doValidation && memoryVerifyIdx < memoryReply.mutations.size() && version == memoryReply.mutations[memoryVerifyIdx].version)
															#line 10772 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3502 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (version > durableValidationVersion)
															#line 10776 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3506 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						memoryVerifyIdx++;
															#line 10780 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
					else
					{
															#line 3508 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						fmt::print("ERROR: SS {0} CF {1} SQ {2} has mutation at {3} in memory but all filtered out on " "disk! (durable validation = {4})\n", data->thisServerID.toString().substr(0, 4), req.rangeID.printable().substr(0, 6), req.id.toString().substr(0, 8), version, durableValidationVersion);
															#line 3516 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						fmt::print("  Memory: ({})\n", memoryReply.mutations[memoryVerifyIdx].mutations.size());
															#line 3517 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						for( auto& it : memoryReply.mutations[memoryVerifyIdx].mutations ) {
															#line 3518 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							if (it.type == MutationRef::SetValue)
															#line 10792 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							{
															#line 3519 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
								fmt::print("    {}=\n", it.param1.printable().c_str());
															#line 10796 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							}
							else
							{
															#line 3521 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
								fmt::print("    {} - {}\n", it.param1.printable().c_str(), it.param2.printable().c_str());
															#line 10802 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							}
						}
															#line 3524 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						fmt::print("  Disk(pre-filter): ({})\n", mutations.size());
															#line 3525 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						for( auto& it : mutations ) {
															#line 3526 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							if (it.type == MutationRef::SetValue)
															#line 10811 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							{
															#line 3527 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
								fmt::print("    {}=\n", it.param1.printable().c_str());
															#line 10815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							}
							else
							{
															#line 3529 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
								fmt::print("    {} - {}\n", it.param1.printable().c_str(), it.param2.printable().c_str());
															#line 10821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
							}
						}
															#line 3532 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						ASSERT_WE_THINK(false);
															#line 10826 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
															#line 3535 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			remainingDurableBytes -= sizeof(KeyValueRef) + res[i].expectedSize();
															#line 3539 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			lastVersion = version;
															#line 3540 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			lastKnownCommitted = knownCommittedVersion;
															#line 10836 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3543 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if ((*feedDiskReadState) == FeedDiskReadState::STARTING || (*feedDiskReadState) == FeedDiskReadState::DISK_CATCHUP)
															#line 10840 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3545 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!memoryReply.mutations.empty() && !reply.mutations.empty() && reply.mutations.back().version < memoryReply.mutations.front().version && remainingDurableBytes <= 0)
															#line 10844 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3549 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				*feedDiskReadState = FeedDiskReadState::DISK_CATCHUP;
															#line 3550 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Feed switching to disk_catchup mode");
															#line 10850 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3553 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if ((*feedDiskReadState) == FeedDiskReadState::STARTING && BUGGIFY_WITH_PROB(0.001))
															#line 10856 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3554 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					*feedDiskReadState = FeedDiskReadState::DISK_CATCHUP;
															#line 3555 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "Feed forcing disk_catchup mode");
															#line 10862 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3558 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "Feed switching to normal mode");
															#line 3559 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					*feedDiskReadState = FeedDiskReadState::NORMAL;
															#line 10870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
		}
															#line 3564 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (remainingDurableBytes > 0)
															#line 10876 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3565 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.arena.dependsOn(memoryReply.arena);
															#line 3566 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto it = memoryReply.mutations.begin();
															#line 3567 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int totalCount = memoryReply.mutations.size();
															#line 3568 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != memoryReply.mutations.end() && it->version <= lastVersion;) {
															#line 3569 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++it;
															#line 3570 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--totalCount;
															#line 10890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3572 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.mutations.append(reply.arena, it, totalCount);
															#line 3575 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reply.mutations.empty())
															#line 10896 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3576 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (res.size() || (lastMemoryVersion != invalidVersion && remainingLimitBytes <= 0))
															#line 10900 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "Change feed adding empty version after disk + memory filtered");
															#line 3578 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (res.empty())
															#line 10906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3579 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						lastVersion = lastMemoryVersion;
															#line 3580 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						lastKnownCommitted = lastMemoryKnownCommitted;
															#line 10912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 3582 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					reply.mutations.push_back(reply.arena, MutationsAndVersionRef(lastVersion, lastKnownCommitted));
															#line 10916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
		}
		else
		{
															#line 3585 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reply.mutations.empty() || reply.mutations.back().version < lastVersion)
															#line 10924 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3586 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Change feed adding empty version after disk filtered");
															#line 3587 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reply.mutations.push_back(reply.arena, MutationsAndVersionRef(lastVersion, lastKnownCommitted));
															#line 10930 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont10(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& getCipherKeysResult,int loopDepth) 
	{
															#line 3401 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cipherMap = getCipherKeysResult;
															#line 10941 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont16cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont10(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && getCipherKeysResult,int loopDepth) 
	{
															#line 3401 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cipherMap = getCipherKeysResult;
															#line 10950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont16cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont3when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& getCipherKeysResult,int loopDepth) 
	{
		loopDepth = a_body1cont16cont10(getCipherKeysResult, loopDepth);

		return loopDepth;
	}
	int a_body1cont16cont3when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && getCipherKeysResult,int loopDepth) 
	{
		loopDepth = a_body1cont16cont10(std::move(getCipherKeysResult), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state > 0) static_cast<GetChangeFeedMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetChangeFeedMutationsActor*>(this)->ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont16cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont16cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<ChangeFeedInfo> feedInfo;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamRequest req;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool atLatest;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool doFilterMutations;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int commonFeedPrefixLength;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FeedDiskReadState* feedDiskReadState;
															#line 3229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamReply reply;
															#line 3230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamReply memoryReply;
															#line 3231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int remainingLimitBytes;
															#line 3232 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int remainingDurableBytes;
															#line 3233 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version startVersion;
															#line 3254 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 3265 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version dequeVersion;
															#line 3266 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version dequeKnownCommit;
															#line 3267 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version emptyVersion;
															#line 3268 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version durableValidationVersion;
															#line 3269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version lastMemoryVersion;
															#line 3270 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version lastMemoryKnownCommitted;
															#line 3272 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool doValidation;
															#line 3330 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool readDurable;
															#line 3331 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool readFetched;
															#line 3332 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool waitFetched;
															#line 3348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock ssReadLock;
															#line 3365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	RangeResult res;
															#line 3379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<std::pair<Standalone<VectorRef<MutationRef>>, Version>> decodedMutations;
															#line 3396 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> cipherMap;
															#line 11084 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getChangeFeedMutations()
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetChangeFeedMutationsActor final : public Actor<std::pair<ChangeFeedStreamReply, bool>>, public ActorCallback< GetChangeFeedMutationsActor, 0, Void >, public ActorCallback< GetChangeFeedMutationsActor, 1, Void >, public ActorCallback< GetChangeFeedMutationsActor, 2, Void >, public ActorCallback< GetChangeFeedMutationsActor, 3, Void >, public ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >, public ActorCallback< GetChangeFeedMutationsActor, 5, Void >, public ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >, public ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >, public FastAllocated<GetChangeFeedMutationsActor>, public GetChangeFeedMutationsActorState<GetChangeFeedMutationsActor> {
															#line 11089 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetChangeFeedMutationsActor>::operator new;
	using FastAllocated<GetChangeFeedMutationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7853662434979545856UL, 18364883722538494208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<ChangeFeedStreamReply, bool>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetChangeFeedMutationsActor, 0, Void >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 1, Void >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 2, Void >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 3, Void >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 5, Void >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >;
friend struct ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >;
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetChangeFeedMutationsActor(StorageServer* const& data,Reference<ChangeFeedInfo> const& feedInfo,ChangeFeedStreamRequest const& req,bool const& atLatest,bool const& doFilterMutations,int const& commonFeedPrefixLength,FeedDiskReadState* const& feedDiskReadState) 
															#line 11113 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<std::pair<ChangeFeedStreamReply, bool>>(),
		   GetChangeFeedMutationsActorState<GetChangeFeedMutationsActor>(data, feedInfo, req, atLatest, doFilterMutations, commonFeedPrefixLength, feedDiskReadState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6476611168701416960UL, 3240458038166094592UL);
		ActorExecutionContextHelper __helper(static_cast<GetChangeFeedMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getChangeFeedMutations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getChangeFeedMutations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 4, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 6, RangeResult >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< GetChangeFeedMutationsActor, 7, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<std::pair<ChangeFeedStreamReply, bool>> getChangeFeedMutations( StorageServer* const& data, Reference<ChangeFeedInfo> const& feedInfo, ChangeFeedStreamRequest const& req, bool const& atLatest, bool const& doFilterMutations, int const& commonFeedPrefixLength, FeedDiskReadState* const& feedDiskReadState ) {
															#line 3222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<std::pair<ChangeFeedStreamReply, bool>>(new GetChangeFeedMutationsActor(data, feedInfo, req, atLatest, doFilterMutations, commonFeedPrefixLength, feedDiskReadState));
															#line 11153 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Change feed stream must be sent an error as soon as it is moved away, or change feed can get incorrect results
															#line 11159 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via stopChangeFeedOnMove()
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class StopChangeFeedOnMoveActor>
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class StopChangeFeedOnMoveActorState {
															#line 11166 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StopChangeFeedOnMoveActorState(StorageServer* const& data,ChangeFeedStreamRequest const& req) 
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 11175 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this));

	}
	~StopChangeFeedOnMoveActorState() 
	{
		fdb_probe_actor_destroy("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3729 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto feed = data->uidChangeFeed.find(req.rangeID);
															#line 3730 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (feed == data->uidChangeFeed.end() || feed->second->removing)
															#line 11192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3731 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(unknown_change_feed());
															#line 3732 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<StopChangeFeedOnMoveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StopChangeFeedOnMoveActorState(); static_cast<StopChangeFeedOnMoveActor*>(this)->destroy(); return 0; }
															#line 11198 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<StopChangeFeedOnMoveActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~StopChangeFeedOnMoveActorState();
				static_cast<StopChangeFeedOnMoveActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 3734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			moved = Promise<Void>();
															#line 3735 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			feed->second->triggerOnMove(req.range, req.id, moved);
															#line 11208 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			try {
															#line 3737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = moved.getFuture();
															#line 3737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<StopChangeFeedOnMoveActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11214 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StopChangeFeedOnMoveActor*>(this)->actor_wait_state = 1;
															#line 3737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StopChangeFeedOnMoveActor, 0, Void >*>(static_cast<StopChangeFeedOnMoveActor*>(this)));
															#line 11219 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StopChangeFeedOnMoveActorState();
		static_cast<StopChangeFeedOnMoveActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3748 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		CODE_PROBE(true, "Change feed moved away cancelling queries");
															#line 3751 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(wrong_shard_server());
															#line 3752 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<StopChangeFeedOnMoveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StopChangeFeedOnMoveActorState(); static_cast<StopChangeFeedOnMoveActor*>(this)->destroy(); return 0; }
															#line 11252 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<StopChangeFeedOnMoveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StopChangeFeedOnMoveActorState();
		static_cast<StopChangeFeedOnMoveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3739 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(e.code() == error_code_operation_cancelled);
															#line 3742 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto feed = data->uidChangeFeed.find(req.rangeID);
															#line 3743 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (feed != data->uidChangeFeed.end())
															#line 11269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3744 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				feed->second->removeOnMoveTrigger(req.range, req.id);
															#line 11273 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3746 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<StopChangeFeedOnMoveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StopChangeFeedOnMoveActorState(); static_cast<StopChangeFeedOnMoveActor*>(this)->destroy(); return 0; }
															#line 11277 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<StopChangeFeedOnMoveActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~StopChangeFeedOnMoveActorState();
			static_cast<StopChangeFeedOnMoveActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StopChangeFeedOnMoveActor*>(this)->actor_wait_state > 0) static_cast<StopChangeFeedOnMoveActor*>(this)->actor_wait_state = 0;
		static_cast<StopChangeFeedOnMoveActor*>(this)->ActorCallback< StopChangeFeedOnMoveActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StopChangeFeedOnMoveActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopChangeFeedOnMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StopChangeFeedOnMoveActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopChangeFeedOnMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StopChangeFeedOnMoveActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StopChangeFeedOnMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamRequest req;
															#line 3734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Promise<Void> moved;
															#line 11397 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via stopChangeFeedOnMove()
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class StopChangeFeedOnMoveActor final : public Actor<Void>, public ActorCallback< StopChangeFeedOnMoveActor, 0, Void >, public FastAllocated<StopChangeFeedOnMoveActor>, public StopChangeFeedOnMoveActorState<StopChangeFeedOnMoveActor> {
															#line 11402 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<StopChangeFeedOnMoveActor>::operator new;
	using FastAllocated<StopChangeFeedOnMoveActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3572175154659010304UL, 17709854919011010816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StopChangeFeedOnMoveActor, 0, Void >;
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StopChangeFeedOnMoveActor(StorageServer* const& data,ChangeFeedStreamRequest const& req) 
															#line 11419 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   StopChangeFeedOnMoveActorState<StopChangeFeedOnMoveActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7825932792072467200UL, 4431126623240794112UL);
		ActorExecutionContextHelper __helper(static_cast<StopChangeFeedOnMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("stopChangeFeedOnMove");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("stopChangeFeedOnMove", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StopChangeFeedOnMoveActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> stopChangeFeedOnMove( StorageServer* const& data, ChangeFeedStreamRequest const& req ) {
															#line 3728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new StopChangeFeedOnMoveActor(data, req));
															#line 11452 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3754 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 11457 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via changeFeedStreamQ()
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ChangeFeedStreamQActor>
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ChangeFeedStreamQActorState {
															#line 11464 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamQActorState(StorageServer* const& data,ChangeFeedStreamRequest const& req) 
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 3756 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:getChangeFeedStream"_loc, req.spanContext),
															#line 3757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   atLatest(false),
															#line 3758 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   removeUID(false),
															#line 3759 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   feedDiskReadState(STARTING),
															#line 3760 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   blockedVersion(),
															#line 3761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   feedInfo(),
															#line 3762 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   streamEndReached(),
															#line 3763 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   doFilterMutations(),
															#line 3764 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   commonFeedPrefixLength()
															#line 11491 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("changeFeedStreamQ", reinterpret_cast<unsigned long>(this));

	}
	~ChangeFeedStreamQActorState() 
	{
		fdb_probe_actor_destroy("changeFeedStreamQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 3767 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.feedStreamQueries;
															#line 3772 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.canReadPopped && (data->activeFeedQueries >= SERVER_KNOBS->STORAGE_FEED_QUERY_HARD_LIMIT || (g_network->isSimulated() && BUGGIFY_WITH_PROB(0.005))))
															#line 11509 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3774 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(storage_too_many_feed_streams());
															#line 3775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					++data->counters.rejectedFeedStreamQueries;
															#line 3776 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!static_cast<ChangeFeedStreamQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedStreamQActorState(); static_cast<ChangeFeedStreamQActor*>(this)->destroy(); return 0; }
															#line 11517 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					new (&static_cast<ChangeFeedStreamQActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~ChangeFeedStreamQActorState();
					static_cast<ChangeFeedStreamQActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 3779 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->activeFeedQueries++;
															#line 3781 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.replyBufferSize <= 0)
															#line 11527 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3782 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.setByteLimit(SERVER_KNOBS->CHANGEFEEDSTREAM_LIMIT_BYTES);
															#line 11531 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3784 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.setByteLimit(std::min((int64_t)req.replyBufferSize, SERVER_KNOBS->CHANGEFEEDSTREAM_LIMIT_BYTES));
															#line 11537 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::SSSpilledChangeFeedReply);
															#line 3791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11543 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 1;
															#line 3791 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 0, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 11548 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeFeedStreamQActorState();
		static_cast<ChangeFeedStreamQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3968 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ChangeFeedStreamQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedStreamQActorState(); static_cast<ChangeFeedStreamQActor*>(this)->destroy(); return 0; }
															#line 11577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<ChangeFeedStreamQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ChangeFeedStreamQActorState();
		static_cast<ChangeFeedStreamQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3952 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->activeFeedQueries--;
															#line 3953 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto it = data->changeFeedClientVersions.find(req.reply.getEndpoint().getPrimaryAddress());
															#line 3954 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it != data->changeFeedClientVersions.end())
															#line 11594 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3955 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (removeUID)
															#line 11598 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3956 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					it->second.erase(req.id);
															#line 11602 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3958 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (it->second.empty())
															#line 11606 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3959 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					data->changeFeedClientVersions.erase(it);
															#line 11610 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 3962 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() != error_code_operation_obsolete)
															#line 11615 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3963 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!canReplyWith(e))
															#line 11619 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3964 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1Catch1(e, loopDepth);
															#line 11623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3965 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 11627 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 11643 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3794 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "TraceChangeFeedStreamStart", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 11647 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3805 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version checkTooOldVersion = (!req.canReadPopped || req.end == MAX_VERSION) ? req.begin : req.end;
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = success(waitForVersionNoTooOld(data, checkTooOldVersion));
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11655 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 2;
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 1, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 11660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 11669 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3794 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "TraceChangeFeedStreamStart", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 11673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3805 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version checkTooOldVersion = (!req.canReadPopped || req.end == MAX_VERSION) ? req.begin : req.end;
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = success(waitForVersionNoTooOld(data, checkTooOldVersion));
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11681 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 2;
															#line 3806 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 1, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 11686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 3809 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto feed = data->uidChangeFeed.find(req.rangeID);
															#line 3810 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (feed == data->uidChangeFeed.end() || feed->second->removing)
															#line 11772 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3811 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(unknown_change_feed());
															#line 3813 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(unknown_change_feed(), loopDepth);
															#line 11778 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedInfo = feed->second;
															#line 3817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		streamEndReached = (req.end == std::numeric_limits<Version>::max()) ? Never() : data->version.whenAtLeast(req.end);
															#line 3820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		doFilterMutations = !req.range.contains(feedInfo->range);
															#line 3821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		commonFeedPrefixLength = 0;
															#line 3822 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (doFilterMutations)
															#line 11790 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3823 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			commonFeedPrefixLength = commonPrefixLength(feedInfo->range.begin, feedInfo->range.end);
															#line 11794 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3827 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ChangeFeedStreamReply emptyInitialReply;
															#line 3828 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		MutationsAndVersionRef emptyInitialVersion;
															#line 3829 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		emptyInitialVersion.version = req.begin - 1;
															#line 3830 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		emptyInitialReply.mutations.push_back_deep(emptyInitialReply.arena, emptyInitialVersion);
															#line 3831 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(emptyInitialReply.atLatestVersion == false);
															#line 3832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(emptyInitialReply.minStreamVersion == invalidVersion);
															#line 3833 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(emptyInitialReply);
															#line 3835 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 11812 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3836 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "TraceChangeFeedStreamSentInitialEmpty", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("Version", req.begin - 1) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 11816 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3848 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 11820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 3809 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto feed = data->uidChangeFeed.find(req.rangeID);
															#line 3810 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (feed == data->uidChangeFeed.end() || feed->second->removing)
															#line 11831 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3811 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(unknown_change_feed());
															#line 3813 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(unknown_change_feed(), loopDepth);
															#line 11837 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedInfo = feed->second;
															#line 3817 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		streamEndReached = (req.end == std::numeric_limits<Version>::max()) ? Never() : data->version.whenAtLeast(req.end);
															#line 3820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		doFilterMutations = !req.range.contains(feedInfo->range);
															#line 3821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		commonFeedPrefixLength = 0;
															#line 3822 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (doFilterMutations)
															#line 11849 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3823 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			commonFeedPrefixLength = commonPrefixLength(feedInfo->range.begin, feedInfo->range.end);
															#line 11853 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3827 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ChangeFeedStreamReply emptyInitialReply;
															#line 3828 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		MutationsAndVersionRef emptyInitialVersion;
															#line 3829 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		emptyInitialVersion.version = req.begin - 1;
															#line 3830 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		emptyInitialReply.mutations.push_back_deep(emptyInitialReply.arena, emptyInitialVersion);
															#line 3831 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(emptyInitialReply.atLatestVersion == false);
															#line 3832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(emptyInitialReply.minStreamVersion == invalidVersion);
															#line 3833 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(emptyInitialReply);
															#line 3835 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (DEBUG_CF_TRACE)
															#line 11871 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3836 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "TraceChangeFeedStreamSentInitialEmpty", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("Version", req.begin - 1) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 11875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3848 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 11879 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1(int loopDepth) 
	{
															#line 3849 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Future<Void> onReady = req.reply.onReady();
															#line 3850 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atLatest && !onReady.isReady() && !removeUID)
															#line 11972 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3851 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->changeFeedClientVersions[req.reply.getEndpoint().getPrimaryAddress()][req.id] = blockedVersion.present() ? blockedVersion.get() : data->prevVersion;
															#line 3853 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (DEBUG_CF_TRACE)
															#line 11978 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3854 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevDebug, "TraceChangeFeedStreamBlockedOnReady", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("Version", blockedVersion.present() ? blockedVersion.get() : data->prevVersion) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 11982 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3865 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			removeUID = true;
															#line 11986 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = onReady;
															#line 3867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11992 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 3;
															#line 3867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 2, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 11997 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 3870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Future<std::pair<ChangeFeedStreamReply, bool>> feedReplyFuture = getChangeFeedMutations( data, feedInfo, req, atLatest, doFilterMutations, commonFeedPrefixLength, &feedDiskReadState);
															#line 3872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atLatest && !removeUID && !feedReplyFuture.isReady())
															#line 12008 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3873 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->changeFeedClientVersions[req.reply.getEndpoint().getPrimaryAddress()][req.id] = blockedVersion.present() ? blockedVersion.get() : data->prevVersion;
															#line 3875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			removeUID = true;
															#line 3876 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (DEBUG_CF_TRACE)
															#line 12016 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3877 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevDebug, "TraceChangeFeedStreamBlockedMutations", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("Version", blockedVersion.present() ? blockedVersion.get() : data->prevVersion) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 12020 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::pair<ChangeFeedStreamReply, bool>> __when_expr_3 = feedReplyFuture;
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12027 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 4;
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 12032 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 3870 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Future<std::pair<ChangeFeedStreamReply, bool>> feedReplyFuture = getChangeFeedMutations( data, feedInfo, req, atLatest, doFilterMutations, commonFeedPrefixLength, &feedDiskReadState);
															#line 3872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atLatest && !removeUID && !feedReplyFuture.isReady())
															#line 12043 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3873 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->changeFeedClientVersions[req.reply.getEndpoint().getPrimaryAddress()][req.id] = blockedVersion.present() ? blockedVersion.get() : data->prevVersion;
															#line 3875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			removeUID = true;
															#line 3876 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (DEBUG_CF_TRACE)
															#line 12051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3877 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevDebug, "TraceChangeFeedStreamBlockedMutations", data->thisServerID) .detail("FeedID", req.rangeID) .detail("StreamUID", req.id) .detail("Range", req.range) .detail("Begin", req.begin) .detail("End", req.end) .detail("CanReadPopped", req.canReadPopped) .detail("Version", blockedVersion.present() ? blockedVersion.get() : data->prevVersion) .detail("PeerAddr", req.reply.getEndpoint().getPrimaryAddress()) .detail("PeerAddress", req.reply.getEndpoint().getPrimaryAddress());
															#line 12055 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::pair<ChangeFeedStreamReply, bool>> __when_expr_3 = feedReplyFuture;
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12062 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 4;
															#line 3889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 12067 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6loopBody1cont4(std::pair<ChangeFeedStreamReply, bool> const& _feedReply,int loopDepth) 
	{
															#line 3890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ChangeFeedStreamReply feedReply = _feedReply.first;
															#line 3891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		bool gotAll = _feedReply.second;
															#line 3893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(feedReply.mutations.size() > 0);
															#line 3894 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.begin = feedReply.mutations.back().version + 1;
															#line 3895 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!atLatest && gotAll)
															#line 12159 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3896 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			atLatest = true;
															#line 12163 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3899 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto& clientVersions = data->changeFeedClientVersions[req.reply.getEndpoint().getPrimaryAddress()];
															#line 3906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version minVersion = removeUID ? data->version.get() : data->prevVersion;
															#line 3907 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (removeUID)
															#line 12171 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3908 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (gotAll || req.begin == req.end)
															#line 12175 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3909 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				clientVersions.erase(req.id);
															#line 3910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				removeUID = false;
															#line 12181 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				clientVersions[req.id] = feedReply.mutations.back().version;
															#line 12187 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& it : clientVersions ) {
															#line 3917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			minVersion = std::min(minVersion, it.second);
															#line 12194 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3919 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedReply.atLatestVersion = atLatest;
															#line 3920 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedReply.minStreamVersion = minVersion;
															#line 3922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.feedRowsQueried += feedReply.mutations.size();
															#line 3923 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.feedBytesQueried += feedReply.mutations.expectedSize();
															#line 3925 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(feedReply);
															#line 3926 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.begin == req.end)
															#line 12208 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3927 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->activeFeedQueries--;
															#line 3928 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(end_of_stream());
															#line 3929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ChangeFeedStreamQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedStreamQActorState(); static_cast<ChangeFeedStreamQActor*>(this)->destroy(); return 0; }
															#line 12216 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ChangeFeedStreamQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ChangeFeedStreamQActorState();
			static_cast<ChangeFeedStreamQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (gotAll)
															#line 12224 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3932 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			blockedVersion = Optional<Version>();
															#line 3933 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (feedInfo->removing)
															#line 12230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(unknown_change_feed());
															#line 3936 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch2(unknown_change_feed(), std::max(0, loopDepth - 1));
															#line 12236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3939 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = feedInfo->newMutations.onTrigger();
															#line 3938 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
															#line 3940 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = streamEndReached;
															#line 12246 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont4when2(__when_expr_5.get(), loopDepth); };
			static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 5;
															#line 3939 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 4, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 3940 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 5, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 12253 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 3948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			blockedVersion = feedReply.mutations.back().version;
															#line 12260 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont6loopBody1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6loopBody1cont4(std::pair<ChangeFeedStreamReply, bool> && _feedReply,int loopDepth) 
	{
															#line 3890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ChangeFeedStreamReply feedReply = _feedReply.first;
															#line 3891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		bool gotAll = _feedReply.second;
															#line 3893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(feedReply.mutations.size() > 0);
															#line 3894 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.begin = feedReply.mutations.back().version + 1;
															#line 3895 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!atLatest && gotAll)
															#line 12278 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3896 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			atLatest = true;
															#line 12282 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3899 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto& clientVersions = data->changeFeedClientVersions[req.reply.getEndpoint().getPrimaryAddress()];
															#line 3906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version minVersion = removeUID ? data->version.get() : data->prevVersion;
															#line 3907 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (removeUID)
															#line 12290 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3908 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (gotAll || req.begin == req.end)
															#line 12294 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3909 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				clientVersions.erase(req.id);
															#line 3910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				removeUID = false;
															#line 12300 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				clientVersions[req.id] = feedReply.mutations.back().version;
															#line 12306 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& it : clientVersions ) {
															#line 3917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			minVersion = std::min(minVersion, it.second);
															#line 12313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3919 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedReply.atLatestVersion = atLatest;
															#line 3920 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		feedReply.minStreamVersion = minVersion;
															#line 3922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.feedRowsQueried += feedReply.mutations.size();
															#line 3923 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.feedBytesQueried += feedReply.mutations.expectedSize();
															#line 3925 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(feedReply);
															#line 3926 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.begin == req.end)
															#line 12327 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3927 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->activeFeedQueries--;
															#line 3928 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(end_of_stream());
															#line 3929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ChangeFeedStreamQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedStreamQActorState(); static_cast<ChangeFeedStreamQActor*>(this)->destroy(); return 0; }
															#line 12335 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ChangeFeedStreamQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ChangeFeedStreamQActorState();
			static_cast<ChangeFeedStreamQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (gotAll)
															#line 12343 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3932 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			blockedVersion = Optional<Version>();
															#line 3933 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (feedInfo->removing)
															#line 12349 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(unknown_change_feed());
															#line 3936 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch2(unknown_change_feed(), std::max(0, loopDepth - 1));
															#line 12355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3939 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = feedInfo->newMutations.onTrigger();
															#line 3938 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12361 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
															#line 3940 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = streamEndReached;
															#line 12365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont4when2(__when_expr_5.get(), loopDepth); };
			static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 5;
															#line 3939 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 4, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 3940 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedStreamQActor, 5, Void >*>(static_cast<ChangeFeedStreamQActor*>(this)));
															#line 12372 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 3948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			blockedVersion = feedReply.mutations.back().version;
															#line 12379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont6loopBody1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(std::pair<ChangeFeedStreamReply, bool> const& _feedReply,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont4(_feedReply, loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(std::pair<ChangeFeedStreamReply, bool> && _feedReply,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont4(std::move(_feedReply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*,std::pair<ChangeFeedStreamReply, bool> const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*,std::pair<ChangeFeedStreamReply, bool> && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont6loopBody1cont7(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont14(int loopDepth) 
	{
															#line 3942 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (feedInfo->removing)
															#line 12470 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3943 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(unknown_change_feed());
															#line 3945 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(unknown_change_feed(), std::max(0, loopDepth - 1));
															#line 12476 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont6loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont4when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont4when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont14(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedStreamQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 4, Void >::remove();
		static_cast<ChangeFeedStreamQActor*>(this)->ActorCallback< ChangeFeedStreamQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6loopBody1cont4when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeFeedStreamQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6loopBody1cont4when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ChangeFeedStreamQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamRequest req;
															#line 3756 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool atLatest;
															#line 3758 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool removeUID;
															#line 3759 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FeedDiskReadState feedDiskReadState;
															#line 3760 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<Version> blockedVersion;
															#line 3761 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<ChangeFeedInfo> feedInfo;
															#line 3762 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> streamEndReached;
															#line 3763 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool doFilterMutations;
															#line 3764 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int commonFeedPrefixLength;
															#line 12649 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via changeFeedStreamQ()
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ChangeFeedStreamQActor final : public Actor<Void>, public ActorCallback< ChangeFeedStreamQActor, 0, Void >, public ActorCallback< ChangeFeedStreamQActor, 1, Void >, public ActorCallback< ChangeFeedStreamQActor, 2, Void >, public ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >, public ActorCallback< ChangeFeedStreamQActor, 4, Void >, public ActorCallback< ChangeFeedStreamQActor, 5, Void >, public FastAllocated<ChangeFeedStreamQActor>, public ChangeFeedStreamQActorState<ChangeFeedStreamQActor> {
															#line 12654 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ChangeFeedStreamQActor>::operator new;
	using FastAllocated<ChangeFeedStreamQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12586396288338447872UL, 12697421607560924416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeFeedStreamQActor, 0, Void >;
friend struct ActorCallback< ChangeFeedStreamQActor, 1, Void >;
friend struct ActorCallback< ChangeFeedStreamQActor, 2, Void >;
friend struct ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >;
friend struct ActorCallback< ChangeFeedStreamQActor, 4, Void >;
friend struct ActorCallback< ChangeFeedStreamQActor, 5, Void >;
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedStreamQActor(StorageServer* const& data,ChangeFeedStreamRequest const& req) 
															#line 12676 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   ChangeFeedStreamQActorState<ChangeFeedStreamQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2167020459532339200UL, 7047320365143205376UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedStreamQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeFeedStreamQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeFeedStreamQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeFeedStreamQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeFeedStreamQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ChangeFeedStreamQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ChangeFeedStreamQActor, 3, std::pair<ChangeFeedStreamReply, bool> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ChangeFeedStreamQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> changeFeedStreamQ( StorageServer* const& data, ChangeFeedStreamRequest const& req ) {
															#line 3755 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new ChangeFeedStreamQActor(data, req));
															#line 12713 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3970 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 12718 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via changeFeedVersionUpdateQ()
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ChangeFeedVersionUpdateQActor>
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ChangeFeedVersionUpdateQActorState {
															#line 12725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedVersionUpdateQActorState(StorageServer* const& data,ChangeFeedVersionUpdateRequest const& req) 
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 12734 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this));

	}
	~ChangeFeedVersionUpdateQActorState() 
	{
		fdb_probe_actor_destroy("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3972 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.feedVersionQueries;
															#line 3973 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(req.minVersion);
															#line 3973 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12753 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state = 1;
															#line 3973 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >*>(static_cast<ChangeFeedVersionUpdateQActor*>(this)));
															#line 12758 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeFeedVersionUpdateQActorState();
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12783 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state = 2;
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*>(static_cast<ChangeFeedVersionUpdateQActor*>(this)));
															#line 12788 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12799 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state = 2;
															#line 3974 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*>(static_cast<ChangeFeedVersionUpdateQActor*>(this)));
															#line 12804 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3975 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version minVersion = data->minFeedVersionForAddress(req.reply.getEndpoint().getPrimaryAddress());
															#line 3976 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(ChangeFeedVersionUpdateReply(minVersion));
															#line 3977 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ChangeFeedVersionUpdateQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedVersionUpdateQActorState(); static_cast<ChangeFeedVersionUpdateQActor*>(this)->destroy(); return 0; }
															#line 12892 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<ChangeFeedVersionUpdateQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ChangeFeedVersionUpdateQActorState();
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3975 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version minVersion = data->minFeedVersionForAddress(req.reply.getEndpoint().getPrimaryAddress());
															#line 3976 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(ChangeFeedVersionUpdateReply(minVersion));
															#line 3977 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ChangeFeedVersionUpdateQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeFeedVersionUpdateQActorState(); static_cast<ChangeFeedVersionUpdateQActor*>(this)->destroy(); return 0; }
															#line 12908 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<ChangeFeedVersionUpdateQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ChangeFeedVersionUpdateQActorState();
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state > 0) static_cast<ChangeFeedVersionUpdateQActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeFeedVersionUpdateQActor*>(this)->ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedVersionUpdateRequest req;
															#line 12995 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via changeFeedVersionUpdateQ()
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ChangeFeedVersionUpdateQActor final : public Actor<Void>, public ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >, public ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >, public FastAllocated<ChangeFeedVersionUpdateQActor>, public ChangeFeedVersionUpdateQActorState<ChangeFeedVersionUpdateQActor> {
															#line 13000 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ChangeFeedVersionUpdateQActor>::operator new;
	using FastAllocated<ChangeFeedVersionUpdateQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5685494874864482816UL, 1811867515824077824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >;
friend struct ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >;
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ChangeFeedVersionUpdateQActor(StorageServer* const& data,ChangeFeedVersionUpdateRequest const& req) 
															#line 13018 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   ChangeFeedVersionUpdateQActorState<ChangeFeedVersionUpdateQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4756634833664377088UL, 2218135851823152128UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeFeedVersionUpdateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeFeedVersionUpdateQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeFeedVersionUpdateQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeFeedVersionUpdateQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeFeedVersionUpdateQActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> changeFeedVersionUpdateQ( StorageServer* const& data, ChangeFeedVersionUpdateRequest const& req ) {
															#line 3971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new ChangeFeedVersionUpdateQActor(data, req));
															#line 13052 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3979 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

#ifdef NO_INTELLISENSE
size_t WATCH_OVERHEAD_WATCHQ =
    sizeof(WatchValueSendReplyActorState<WatchValueSendReplyActor>) + sizeof(WatchValueSendReplyActor);
size_t WATCH_OVERHEAD_WATCHIMPL =
    sizeof(WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>) + sizeof(WatchWaitForValueChangeActor);
#else
size_t WATCH_OVERHEAD_WATCHQ = 0; // only used in IDE so value is irrelevant
size_t WATCH_OVERHEAD_WATCHIMPL = 0;
#endif

															#line 13067 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getShardState_impl()
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardState_implActor>
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActorState {
															#line 13074 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 13083 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetShardState_implActorState() 
	{
		fdb_probe_actor_destroy("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3991 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.mode != GetShardStateRequest::NO_WAIT);
															#line 3993 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 13100 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardState_implActorState();
		static_cast<GetShardState_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3994 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Future<Void>> onChange;
															#line 3996 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto t : data->shards.intersectingRanges(req.keys) ) {
															#line 3997 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!t.value()->assigned())
															#line 13134 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3998 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				onChange.push_back(delay(SERVER_KNOBS->SHARD_READY_DELAY));
															#line 13138 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				break;
			}
															#line 4002 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::READABLE && !t.value()->isReadable())
															#line 13143 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4003 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 13147 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4004 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->readWrite.getFuture());
															#line 13151 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4006 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 4007 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->readWrite.getFuture());
															#line 13159 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 4011 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::FETCHING && !t.value()->isFetched())
															#line 13164 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4012 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 13168 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4013 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->fetchComplete.getFuture());
															#line 13172 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4015 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 4016 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->fetchComplete.getFuture());
															#line 13180 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
		}
															#line 4021 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!onChange.size())
															#line 13186 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4022 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetShardStateReply rep(data->version.get(), data->durableVersion.get());
															#line 4023 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.includePhysicalShard)
															#line 13192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4024 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				rep.shards = data->getStorageServerShards(req.keys);
															#line 13196 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4026 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(rep);
															#line 4027 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<GetShardState_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardState_implActorState(); static_cast<GetShardState_implActor*>(this)->destroy(); return 0; }
															#line 13202 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<GetShardState_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~GetShardState_implActorState();
			static_cast<GetShardState_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 4030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = waitForAll(onChange);
															#line 4030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13212 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 1;
															#line 4030 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 0, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 13217 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13228 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 13233 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13244 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 4031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 13249 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 13420 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getShardState_impl()
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActor final : public Actor<Void>, public ActorCallback< GetShardState_implActor, 0, Void >, public ActorCallback< GetShardState_implActor, 1, Void >, public FastAllocated<GetShardState_implActor>, public GetShardState_implActorState<GetShardState_implActor> {
															#line 13425 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetShardState_implActor>::operator new;
	using FastAllocated<GetShardState_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6350708669027531264UL, 5875642339875831552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardState_implActor, 0, Void >;
friend struct ActorCallback< GetShardState_implActor, 1, Void >;
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 13443 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetShardState_implActorState<GetShardState_implActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3570919601400010496UL, 14753609829273848064UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardState_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardState_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetShardState_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardState_impl( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 3990 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardState_implActor(data, req));
															#line 13477 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4034 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 13482 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getShardStateQ()
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardStateQActor>
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActorState {
															#line 13489 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 13498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	~GetShardStateQActorState() 
	{
		fdb_probe_actor_destroy("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4037 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = getShardState_impl(data, req);
															#line 4036 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetShardStateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13515 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 4038 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(g_network->isSimulated() ? 10 : 60);
															#line 13519 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<GetShardStateQActor*>(this)->actor_wait_state = 1;
															#line 4037 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 0, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 4038 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 1, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 13526 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4042 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetShardStateQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardStateQActorState(); static_cast<GetShardStateQActor*>(this)->destroy(); return 0; }
															#line 13549 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetShardStateQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 4039 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 13573 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 4039 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 13582 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardStateQActor*>(this)->actor_wait_state > 0) static_cast<GetShardStateQActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 0, Void >::remove();
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 13712 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getShardStateQ()
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActor final : public Actor<Void>, public ActorCallback< GetShardStateQActor, 0, Void >, public ActorCallback< GetShardStateQActor, 1, Void >, public FastAllocated<GetShardStateQActor>, public GetShardStateQActorState<GetShardStateQActor> {
															#line 13717 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetShardStateQActor>::operator new;
	using FastAllocated<GetShardStateQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(85989048690264576UL, 2759311728687776256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardStateQActor, 0, Void >;
friend struct ActorCallback< GetShardStateQActor, 1, Void >;
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 13735 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetShardStateQActorState<GetShardStateQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8443565892279057664UL, 11087380315085298688UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardStateQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardStateQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardStateQ( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 4035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardStateQActor(data, req));
															#line 13768 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4044 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRef addPrefix(KeyRef const& key, Optional<KeyRef> prefix, Arena& arena) {
	if (prefix.present()) {
		return key.withPrefix(prefix.get(), arena);
	} else {
		return key;
	}
}

KeyValueRef removePrefix(KeyValueRef const& src, Optional<KeyRef> prefix) {
	if (prefix.present()) {
		return KeyValueRef(src.key.removePrefix(prefix.get()), src.value);
	} else {
		return src;
	}
}

void merge(Arena& arena,
           VectorRef<KeyValueRef, VecSerStrategy::String>& output,
           VectorRef<KeyValueRef> const& vm_output,
           RangeResult const& base,
           int& vCount,
           int limit,
           bool stopAtEndOfBase,
           int& pos,
           int limitBytes,
           Optional<KeyRef> tenantPrefix)
// Combines data from base (at an older version) with sets from newer versions in [start, end) and appends the first (up
// to) |limit| rows to output If limit<0, base and output are in descending order, and start->key()>end->key(), but
// start is still inclusive and end is exclusive
{
	ASSERT(limit != 0);
	// Add a dependency of the new arena on the result from the KVS so that we don't have to copy any of the KVS
	// results.
	arena.dependsOn(base.arena());

	bool forward = limit > 0;
	if (!forward)
		limit = -limit;
	int adjustedLimit = limit + output.size();
	int accumulatedBytes = 0;
	KeyValueRef const* baseStart = base.begin();
	KeyValueRef const* baseEnd = base.end();
	while (baseStart != baseEnd && vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		if (forward ? baseStart->key < vm_output[pos].key : baseStart->key > vm_output[pos].key) {
			output.push_back(arena, removePrefix(*baseStart++, tenantPrefix));
		} else {
			output.push_back_deep(arena, removePrefix(vm_output[pos], tenantPrefix));
			if (baseStart->key == vm_output[pos].key)
				++baseStart;
			++pos;
			vCount--;
		}
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	while (baseStart != baseEnd && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		output.push_back(arena, removePrefix(*baseStart++, tenantPrefix));
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	if (!stopAtEndOfBase) {
		while (vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
			output.push_back_deep(arena, removePrefix(vm_output[pos], tenantPrefix));
			accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
			++pos;
			vCount--;
		}
	}
}

static inline void copyOptionalValue(Arena* a,
                                     GetValueReqAndResultRef& getValue,
                                     const Optional<Value>& optionalValue) {
	getValue.result = optionalValue.castTo<ValueRef>();
	if (optionalValue.present()) {
		a->dependsOn(optionalValue.get().arena());
	}
}
															#line 13849 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via quickGetValue()
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetValueActor>
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActorState {
															#line 13856 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActorState(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 4127 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValue(),
															#line 4128 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValueStart(g_network->timer())
															#line 13875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetValueActorState() 
	{
		fdb_probe_actor_destroy("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4129 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue.key = key;
															#line 4131 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->shards[key]->isReadable())
															#line 13892 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				try {
															#line 4134 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					GetValueRequest req(pOriginalReq->spanContext, pOriginalReq->tenantInfo, key, version, pOriginalReq->tags, pOriginalReq->options, VersionVector());
															#line 4144 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					data->actors.add(getValueQ(data, req));
															#line 4145 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					StrictFuture<GetValueReply> __when_expr_0 = req.reply.getFuture();
															#line 4145 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13903 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<QuickGetValueActor*>(this)->actor_wait_state = 1;
															#line 4145 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 0, GetValueReply >*>(static_cast<QuickGetValueActor*>(this)));
															#line 13908 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4160 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetValueMiss;
															#line 4161 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_VALUE_FALLBACK)
															#line 13944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4162 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<Reference<Tenant>> tenant = pOriginalReq->tenantInfo.hasTenant() ? makeReference<Tenant>(pOriginalReq->tenantInfo.tenantId) : Optional<Reference<Tenant>>();
															#line 4165 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx, tenant);
															#line 4166 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 4168 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 4169 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<Optional<Value>> valueFuture = tr.get(key, Snapshot::True);
															#line 4171 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = valueFuture;
															#line 4171 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13960 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetValueActor*>(this)->actor_wait_state = 2;
															#line 4171 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 1, Optional<Value> >*>(static_cast<QuickGetValueActor*>(this)));
															#line 13965 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 4177 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_value_miss(), loopDepth);
															#line 13972 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetValueReply const& reply,int loopDepth) 
	{
															#line 4146 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 14000 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4147 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 4148 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 4149 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 4150 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 4151 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 14012 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetValueReply && reply,int loopDepth) 
	{
															#line 4146 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 14026 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4147 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 4148 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 4149 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 4150 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 4151 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 14038 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 0, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& valueOption,int loopDepth) 
	{
															#line 4172 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 4173 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 4174 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 4175 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 14146 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && valueOption,int loopDepth) 
	{
															#line 4172 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 4173 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 4174 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 4175 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 14164 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> const& valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(valueOption, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> && valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(valueOption), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef key;
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 4127 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueReqAndResultRef getValue;
															#line 4128 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValueStart;
															#line 4165 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 14263 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via quickGetValue()
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActor final : public Actor<GetValueReqAndResultRef>, public ActorCallback< QuickGetValueActor, 0, GetValueReply >, public ActorCallback< QuickGetValueActor, 1, Optional<Value> >, public FastAllocated<QuickGetValueActor>, public QuickGetValueActorState<QuickGetValueActor> {
															#line 14268 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<QuickGetValueActor>::operator new;
	using FastAllocated<QuickGetValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17499939580815112960UL, 4848093098953475840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetValueReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetValueActor, 0, GetValueReply >;
friend struct ActorCallback< QuickGetValueActor, 1, Optional<Value> >;
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActor(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 14286 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetValueReqAndResultRef>(),
		   QuickGetValueActorState<QuickGetValueActor>(data, key, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13229342185383356416UL, 6029856400378886912UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetValueActor, 0, GetValueReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetValueActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetValueReqAndResultRef> quickGetValue( StorageServer* const& data, StringRef const& key, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 4121 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetValueReqAndResultRef>(new QuickGetValueActor(data, key, version, a, pOriginalReq));
															#line 14320 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4180 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// If limit>=0, it returns the first rows in the range (sorted ascending), otherwise the last rows (sorted descending).
// readRange has O(|result|) + O(log |data|) cost
															#line 14327 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via readRange()
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadRangeActor>
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActorState {
															#line 14334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActorState(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options,Optional<KeyRef> const& tenantPrefix) 
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   limit(limit),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pLimitBytes(pLimitBytes),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options),
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   tenantPrefix(tenantPrefix),
															#line 4191 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   result(),
															#line 4192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   view(data->data().at(version)),
															#line 4193 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCurrent(view.end()),
															#line 4194 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBegin(),
															#line 4195 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readEnd(),
															#line 4196 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBeginTemp(),
															#line 4197 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCount(0),
															#line 4198 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:readRange"_loc, parentSpan),
															#line 4199 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultLogicalSize(0),
															#line 4200 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   logicalSize(0),
															#line 4203 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultCache(),
															#line 4206 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pos(0)
															#line 14379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("readRange", reinterpret_cast<unsigned long>(this));

	}
	~ReadRangeActorState() 
	{
		fdb_probe_actor_destroy("readRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4209 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto containingRange = data->cachedRangeMap.rangeContaining(range.begin);
															#line 4210 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (containingRange.value() && containingRange->range().end >= range.end)
															#line 14396 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4213 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = true;
															#line 14400 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4215 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = false;
															#line 14406 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4218 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (limit >= 0)
															#line 14410 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4221 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLessOrEqual(range.begin);
															#line 4222 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() > range.begin)
															#line 14416 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4223 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = vCurrent->getEndKey();
															#line 14420 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4225 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = range.begin;
															#line 14426 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4228 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent)
															#line 14430 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4229 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (vCurrent.key() != readBegin)
															#line 14434 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						++vCurrent;
															#line 14438 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
				else
				{
															#line 4236 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					vCurrent = view.begin();
															#line 14445 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4238 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (EXPENSIVE_VALIDATION)
															#line 14449 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4239 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(vCurrent == view.lower_bound(readBegin));
															#line 14453 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 14457 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 4334 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLess(range.end);
															#line 4337 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() >= range.end)
															#line 14466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4338 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = vCurrent.key();
															#line 4339 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					--vCurrent;
															#line 14472 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4341 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = range.end;
															#line 14478 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4344 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 14482 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4426 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesReturnedHistogram->sample(resultLogicalSize);
															#line 4427 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeKVPairsReturnedHistogram->sample(result.data.size());
															#line 4430 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(result.data.size() == 0 || *pLimitBytes + result.data.end()[-1].expectedSize() + sizeof(KeyValueRef) > 0);
															#line 4431 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		result.more = limit == 0 || *pLimitBytes <= 0;
															#line 4432 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		result.version = version;
															#line 4433 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ReadRangeActor*>(this)->SAV<GetKeyValuesReply>::futures) { (void)(result); this->~ReadRangeActorState(); static_cast<ReadRangeActor*>(this)->destroy(); return 0; }
															#line 14516 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<ReadRangeActor*>(this)->SAV< GetKeyValuesReply >::value()) GetKeyValuesReply(std::move(result)); // state_var_RVO
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 4242 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit > 0 && *pLimitBytes > 0 && readBegin < range.end))
															#line 14541 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4243 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() >= readBegin);
															#line 4244 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 4247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 14551 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4248 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 14555 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4249 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 4250 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--b;
															#line 4251 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() < readBegin);
															#line 14563 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4255 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 4256 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() < range.end && !vCurrent->isClearTo() && vCount < limit && vSize < *pLimitBytes;) {
															#line 4259 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 4260 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize() - (tenantPrefix.present() ? tenantPrefix.get().size() : 0);
															#line 4262 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 4263 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 14577 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4268 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readEnd = vCurrent ? std::min(vCurrent.key(), range.end) : range.end;
															#line 4269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_0 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 4269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14586 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 1;
															#line 4269 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 0, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 14591 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 4271 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 4272 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 4273 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 4274 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 4276 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 4277 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 14623 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4278 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 4283 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 14629 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4288 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 4289 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 4299 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 4301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 4302 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 14641 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4305 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 14645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 14651 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4314 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 4319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 14657 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4323 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 14663 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 4326 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 4327 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 14671 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4329 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 14677 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 4271 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 4272 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 4273 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 4274 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 4276 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 4277 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 14699 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4278 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 4283 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 14705 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4288 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 4289 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 4299 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 4301 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 4302 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 14717 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4305 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 14721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 14727 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4314 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 4319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 14733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4323 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 14739 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4324 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 4326 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 4327 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 14747 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4329 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 14753 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
															#line 4344 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit < 0 && *pLimitBytes > 0 && readEnd > range.begin))
															#line 14853 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4345 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() < readEnd);
															#line 4346 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 4349 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 14863 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4350 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 14867 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4351 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 4352 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++b;
															#line 4353 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() >= readEnd);
															#line 14875 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4356 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			vCount = 0;
															#line 4357 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 4358 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() >= range.begin && !vCurrent->isClearTo() && vCount < -limit && vSize < *pLimitBytes;) {
															#line 4361 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 4362 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize() - (tenantPrefix.present() ? tenantPrefix.get().size() : 0);
															#line 4364 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 4365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 14891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4369 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readBegin = vCurrent ? std::max(vCurrent->isClearTo() ? vCurrent->getEndKey() : vCurrent.key(), range.begin) : range.begin;
															#line 4371 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 4371 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14900 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 2;
															#line 4371 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 1, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 14905 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break2(int loopDepth) 
	{
		try {
			return a_body1cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 4373 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 4374 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 4375 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 4376 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 4378 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 4379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 14937 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4380 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 4385 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 14943 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4388 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 4389 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 4399 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 4401 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 4402 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 14955 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4405 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 14959 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4409 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 14965 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4410 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 4415 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 14971 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4416 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 14977 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4417 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 4418 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 4419 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 14985 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4421 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 14991 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 4373 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 4374 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 4375 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 4376 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 4378 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 4379 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 15013 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4380 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 4385 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 15019 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4388 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 4389 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 4399 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 4401 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 4402 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 15031 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4405 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 15035 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4409 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 15041 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4410 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 4415 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 15047 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4416 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 15053 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4417 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 4418 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 4419 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 15061 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4421 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 15067 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int limit;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pLimitBytes;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<KeyRef> tenantPrefix;
															#line 4191 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply result;
															#line 4192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::ViewAtVersion view;
															#line 4193 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::iterator vCurrent;
															#line 4194 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readBegin;
															#line 4195 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readEnd;
															#line 4196 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key readBeginTemp;
															#line 4197 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int vCount;
															#line 4198 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 4199 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int resultLogicalSize;
															#line 4200 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int logicalSize;
															#line 4203 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	VectorRef<KeyValueRef> resultCache;
															#line 4206 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int pos;
															#line 15190 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via readRange()
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActor final : public Actor<GetKeyValuesReply>, public ActorCallback< ReadRangeActor, 0, RangeResult >, public ActorCallback< ReadRangeActor, 1, RangeResult >, public FastAllocated<ReadRangeActor>, public ReadRangeActorState<ReadRangeActor> {
															#line 15195 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ReadRangeActor>::operator new;
	using FastAllocated<ReadRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(304777456936001280UL, 4916795179294593024UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetKeyValuesReply>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadRangeActor, 0, RangeResult >;
friend struct ActorCallback< ReadRangeActor, 1, RangeResult >;
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActor(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options,Optional<KeyRef> const& tenantPrefix) 
															#line 15213 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetKeyValuesReply>(),
		   ReadRangeActorState<ReadRangeActor>(data, version, range, limit, pLimitBytes, parentSpan, options, tenantPrefix),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3635396024098275328UL, 3211122134102601216UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadRangeActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadRangeActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetKeyValuesReply> readRange( StorageServer* const& data, Version const& version, KeyRange const& range, int const& limit, int* const& pLimitBytes, SpanContext const& parentSpan, Optional<ReadOptions> const& options, Optional<KeyRef> const& tenantPrefix ) {
															#line 4183 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetKeyValuesReply>(new ReadRangeActor(data, version, range, limit, pLimitBytes, parentSpan, options, tenantPrefix));
															#line 15247 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4435 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 15252 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via findKey()
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FindKeyActor>
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActorState {
															#line 15259 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActorState(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   sel(sel),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOffset(pOffset),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 15278 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("findKey", reinterpret_cast<unsigned long>(this));

	}
	~FindKeyActorState() 
	{
		fdb_probe_actor_destroy("findKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4454 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version != latestVersion);
															#line 4455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(selectorInRange(sel, range) && version >= data->oldestVersion.get());
															#line 4458 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			forward = sel.offset > 0;
															#line 4459 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			sign = forward ? +1 : -1;
															#line 4460 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			skipEqualKey = sel.orEqual == forward;
															#line 4461 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			distance = forward ? sel.offset : 1 - sel.offset;
															#line 4462 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			span = Span("SS.findKey"_loc, parentSpan);
															#line 4466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = int();
															#line 4467 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (sel.offset <= 1 && sel.offset >= 0)
															#line 15309 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4468 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = std::numeric_limits<int>::max();
															#line 15313 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4470 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = (g_network->isSimulated() && g_simulator->tssMode == ISimulator::TSSMode::Disabled && BUGGIFY) ? SERVER_KNOBS->BUGGIFY_LIMIT_BYTES : SERVER_KNOBS->STORAGE_LIMIT_BYTES;
															#line 15319 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4474 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_0 = readRange(data, version, forward ? KeyRangeRef(sel.getKey(), range.end) : KeyRangeRef(range.begin, keyAfter(sel.getKey())), (distance + skipEqualKey) * sign, &maxBytes, span.context, options, {});
															#line 4474 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15325 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 1;
															#line 4474 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 15330 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FindKeyActorState();
		static_cast<FindKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4483 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 4487 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (more && !forward && rep.data.size() == 1)
															#line 15355 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4488 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reverse key selector returned only one result in range read");
															#line 4489 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = std::numeric_limits<int>::max();
															#line 4490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_1 = readRange( data, version, KeyRangeRef(range.begin, keyAfter(sel.getKey())), -2, &maxBytes, span.context, options, {});
															#line 4490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15365 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 2;
															#line 4490 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 15370 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& __rep,int loopDepth) 
	{
															#line 4474 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = __rep;
															#line 15384 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && __rep,int loopDepth) 
	{
		rep = std::move(__rep);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 4497 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int index = distance - 1;
															#line 4498 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (skipEqualKey && rep.data.size() && rep.data[0].key == sel.getKey())
															#line 15465 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4499 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++index;
															#line 15469 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4501 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (index < rep.data.size())
															#line 15473 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4502 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = 0;
															#line 4504 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 15479 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4505 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = std::max((int64_t)rep.data[index].key.size(), SERVER_KNOBS->EMPTY_READ_PENALTY);
															#line 4507 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 15485 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4510 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(rep.data[index].key); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 15489 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(rep.data[index].key);
			this->~FindKeyActorState();
			static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 4512 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 15499 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4513 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 4514 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 15505 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4518 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = index - rep.data.size() + 1;
															#line 4519 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!forward)
															#line 15511 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4520 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				*pOffset = -*pOffset;
															#line 15515 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4522 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (more)
															#line 15519 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4523 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Key selector read range had more results");
															#line 4525 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(rep.data.size());
															#line 4526 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				Key returnKey = forward ? keyAfter(rep.data.back().key) : rep.data.back().key;
															#line 4530 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(returnKey != sel.getKey());
															#line 4531 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(returnKey); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 15531 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(returnKey);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 4533 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(forward ? range.end : range.begin); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 15541 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(forward ? range.end : range.begin);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply const& rep2,int loopDepth) 
	{
															#line 4492 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 4493 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 4494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 15559 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply && rep2,int loopDepth) 
	{
															#line 4492 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 4493 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 4494 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 15572 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply const& rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(rep2, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply && rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(rep2), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 1, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeySelectorRef sel;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pOffset;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 4458 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool forward;
															#line 4459 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int sign;
															#line 4460 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool skipEqualKey;
															#line 4461 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int distance;
															#line 4462 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 4466 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int maxBytes;
															#line 4474 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply rep;
															#line 4483 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool more;
															#line 15682 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via findKey()
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActor final : public Actor<Key>, public ActorCallback< FindKeyActor, 0, GetKeyValuesReply >, public ActorCallback< FindKeyActor, 1, GetKeyValuesReply >, public FastAllocated<FindKeyActor>, public FindKeyActorState<FindKeyActor> {
															#line 15687 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FindKeyActor>::operator new;
	using FastAllocated<FindKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2058622005487182080UL, 13388808873940791808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FindKeyActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< FindKeyActor, 1, GetKeyValuesReply >;
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActor(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 15705 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Key>(),
		   FindKeyActorState<FindKeyActor>(data, sel, version, range, pOffset, parentSpan, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11946933499123123200UL, 11635880071562820096UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("findKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Key> findKey( StorageServer* const& data, KeySelectorRef const& sel, Version const& version, KeyRange const& range, int* const& pOffset, SpanContext const& parentSpan, Optional<ReadOptions> const& options ) {
															#line 4436 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new FindKeyActor(data, sel, version, range, pOffset, parentSpan, options));
															#line 15739 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4537 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRange getShardKeyRange(StorageServer* data, const KeySelectorRef& sel)
// Returns largest range such that the shard state isReadable and selectorInRange(sel, range) or wrong_shard_server if
// no such range exists
{
	auto i = sel.isBackward() ? data->shards.rangeContainingKeyBefore(sel.getKey())
	                          : data->shards.rangeContaining(sel.getKey());
	auto fullRange = data->shards.ranges();
	if (!i->value()->isReadable())
		throw wrong_shard_server();
	ASSERT(selectorInRange(sel, i->range()));
	Key begin, end;
	if (sel.isBackward()) {
		end = i->range().end;
		while (i != fullRange.begin() && i.value()->isReadable()) {
			begin = i->range().begin;
			--i;
		}
		if (i.value()->isReadable()) {
			begin = i->range().begin;
		}
	} else {
		begin = i->range().begin;
		while (i != fullRange.end() && i.value()->isReadable()) {
			end = i->range().end;
			++i;
		}
	}
	return KeyRangeRef(begin, end);
}

void maybeInjectConsistencyScanCorruption(UID thisServerID, GetKeyValuesRequest const& req, GetKeyValuesReply& reply) {
	if (g_simulator->consistencyScanState != ISimulator::SimConsistencyScanState::Enabled_InjectCorruption ||
	    !req.options.present() || !req.options.get().consistencyCheckStartVersion.present() ||
	    !g_simulator->consistencyScanCorruptRequestKey.present()) {
		return;
	}

	UID destination = req.reply.getEndpoint().token;

	ASSERT(g_simulator->consistencyScanInjectedCorruptionType.present() ==
	       g_simulator->consistencyScanInjectedCorruptionDestination.present());
	// if we already injected a corruption, reinject it if this request was a retransmit of the same one we corrupted
	// could also check that this storage sent the corruption but the reply endpoints should be globally unique so this
	// covers it
	if (g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	    (g_simulator->consistencyScanInjectedCorruptionDestination.get() != destination)) {
		return;
	}

	CODE_PROBE(true, "consistency check injecting corruption");
	CODE_PROBE(g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	               g_simulator->consistencyScanInjectedCorruptionDestination.get() == destination,
	           "consistency check re-injecting corruption after retransmit",
	           probe::decoration::rare);

	g_simulator->consistencyScanInjectedCorruptionDestination = destination;
	// FIXME: reinject same type of corruption once we enable other types

	// FIXME: code probe for each type?

	if (true /*deterministicRandom()->random01() < 0.3*/) {
		// flip more flag
		reply.more = !reply.more;
		g_simulator->consistencyScanInjectedCorruptionType = ISimulator::SimConsistencyScanCorruptionType::FlipMoreFlag;
	} else {
		// FIXME: weird memory issues when messing with actual response data, enable and figure out later
		ASSERT(false);
		// make deep copy of request, since some of the underlying memory can reference storage engine data directly
		GetKeyValuesReply copy = reply;
		reply = GetKeyValuesReply();
		reply.more = copy.more;
		reply.cached = copy.cached;
		reply.version = copy.version;
		reply.data.append_deep(reply.arena, copy.data.begin(), copy.data.size());

		if (reply.data.empty()) {
			// add row to empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::AddToEmpty;
			reply.data.push_back_deep(
			    reply.arena,
			    KeyValueRef(g_simulator->consistencyScanCorruptRequestKey.get(), "consistencyCheckCorruptValue"_sr));
		} else if (deterministicRandom()->coinflip() || reply.data.back().value.empty()) {
			// change value in non-empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::RemoveLastRow;
			reply.data.pop_back();
		} else {
			// chop off last byte of first value
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::ChangeFirstValue;

			reply.data[0].value = reply.data[0].value.substr(0, reply.data[0].value.size() - 1);
		}
	}

	TraceEvent(SevWarnAlways, "InjectedConsistencyScanCorruption", thisServerID)
	    .detail("CorruptionType", g_simulator->consistencyScanInjectedCorruptionType.get())
	    .detail("Version", req.version)
	    .detail("Count", reply.data.size());
}

															#line 15846 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getKeyValuesQ()
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetKeyValuesQActor>
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActorState {
															#line 15853 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActorState(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 4644 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:getKeyValues"_loc, req.spanContext),
															#line 4645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 15866 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4647 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getCurrentLineage()->modify(&TransactionLineage::txID) = req.spanContext.traceID;
															#line 4649 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.getRangeQueries;
															#line 4650 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.allQueries;
															#line 4651 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.begin.getKey().startsWith(systemKeys.begin))
															#line 15887 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4652 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.systemKeyQueries;
															#line 4653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getRangeSystemKeyQueries;
															#line 15893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4655 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 4660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 4660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15901 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 4660 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 0, Void >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 15906 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 15936 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15947 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 15952 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 4664 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 4665 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readQueueWaitSample->addMeasurement(queueWaitEnd - req.requestTime());
															#line 16038 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 4668 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 16042 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4669 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Before");
															#line 16046 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4672 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 4673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, span.context);
															#line 4673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 16054 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 4673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 2, Version >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 16059 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 16074 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 4851 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, resultSize);
															#line 4852 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 4854 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 4855 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySample->addMeasurement(duration);
															#line 4856 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readRangeLatencySample->addMeasurement(duration);
															#line 4857 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 16163 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4858 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 4860 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxSelectorOffset = data->latencyBandConfig.get().readConfig.maxKeySelectorOffset.orDefault(std::numeric_limits<int>::max());
															#line 4862 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes || abs(req.begin.offset) > maxSelectorOffset || abs(req.end.offset) > maxSelectorOffset));
															#line 16171 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4869 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetKeyValuesQActorState(); static_cast<GetKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 16175 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4846 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 16188 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4847 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 16192 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4848 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 16196 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 4674 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DisabledTraceEvent("VVV", data->thisServerID) .detail("Version", version) .detail("ReqVersion", req.version) .detail("Oldest", data->oldestVersion.get()) .detail("VV", req.ssLatestCommitVersions.toString()) .detail("DebugID", req.options.present() && req.options.get().debugID.present() ? req.options.get().debugID.get() : UID());
															#line 4682 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readVersionWaitSample->addMeasurement(g_network->timer() - queueWaitEnd);
															#line 4684 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkTenantEntry(version, req.tenantInfo, req.options.present() ? req.options.get().lockAware : false);
															#line 4685 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.tenantInfo.hasTenant())
															#line 16217 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4686 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin.setKeyUnlimited(req.begin.getKey().withPrefix(req.tenantInfo.prefix.get(), req.arena));
															#line 4687 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end.setKeyUnlimited(req.end.getKey().withPrefix(req.tenantInfo.prefix.get(), req.arena));
															#line 16223 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4690 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 4692 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		shard = getShardKeyRange(data, req.begin);
															#line 4694 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 16231 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4695 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterVersion");
															#line 16235 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4702 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!selectorInRange(req.end, shard) && !(req.end.isFirstGreaterOrEqual() && req.end.getKey() == shard.end))
															#line 16239 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4710 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 16243 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4713 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		KeyRangeRef searchRange = TenantAPI::clampRangeToTenant(shard, req.tenantInfo, req.arena);
															#line 4715 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		offset1 = 0;
															#line 4716 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		offset2 = int();
															#line 4717 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fBegin = req.begin.isFirstGreaterOrEqual() ? Future<Key>(req.begin.getKey()) : findKey(data, req.begin, version, searchRange, &offset1, span.context, req.options);
															#line 4721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fEnd = req.end.isFirstGreaterOrEqual() ? Future<Key>(req.end.getKey()) : findKey(data, req.end, version, searchRange, &offset2, span.context, req.options);
															#line 4725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_3 = fBegin;
															#line 4725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 16259 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 4725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 3, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 16264 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Version const& __version,int loopDepth) 
	{
															#line 4673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 16273 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 4726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_4 = fEnd;
															#line 4726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 16354 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont3Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 4726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 4, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 16359 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5when1(Key const& __begin,int loopDepth) 
	{
															#line 4725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		begin = __begin;
															#line 16368 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Key && __begin,int loopDepth) 
	{
		begin = std::move(__begin);
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 3, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 3, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 4728 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 16447 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4729 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterKeys");
															#line 16451 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if ((offset1 && offset1 != 1) || (offset2 && offset2 != 1))
															#line 16455 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4738 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "wrong_shard_server due to offset");
															#line 4754 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 16461 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4757 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (begin >= end)
															#line 16465 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4758 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 16469 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4759 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Send");
															#line 16473 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4763 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesReply none;
															#line 4764 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.version = version;
															#line 4765 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.more = false;
															#line 4766 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.penalty = data->getPenalty();
															#line 4768 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->checkChangeCounter(changeCounter, KeyRangeRef(std::min<KeyRef>(req.begin.getKey(), req.end.getKey()), std::max<KeyRef>(req.begin.getKey(), req.end.getKey())));
															#line 4772 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (g_network->isSimulated())
															#line 16487 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4773 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maybeInjectConsistencyScanCorruption(data->thisServerID, req, none);
															#line 16491 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4775 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(none);
															#line 16495 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont11cont1(loopDepth);
		}
		else
		{
															#line 4777 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			remainingLimitBytes = req.limitBytes;
															#line 4779 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			kvReadRange = g_network->timer();
															#line 4780 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_5 = readRange(data, version, KeyRangeRef(begin, end), req.limit, &remainingLimitBytes, span.context, req.options, req.tenantInfo.prefix);
															#line 4780 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 16508 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont3Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont11when1(__when_expr_5.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 6;
															#line 4780 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 16513 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont7when1(Key const& __end,int loopDepth) 
	{
															#line 4726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		end = __end;
															#line 16523 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Key && __end,int loopDepth) 
	{
		end = std::move(__end);
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 4, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 4, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11cont1(int loopDepth) 
	{
		loopDepth = a_body1cont11cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7(GetKeyValuesReply const& _r,int loopDepth) 
	{
															#line 4788 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 4789 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvReadRangeLatencySample->addMeasurement(duration);
															#line 4790 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 4792 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 16614 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 16618 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4797 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 4801 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 16624 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4802 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 4803 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.tenantInfo.prefix.present())
															#line 16630 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4804 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin.removePrefix(req.tenantInfo.prefix.get()) && r.data[i].key < end.removePrefix(req.tenantInfo.prefix.get()));
															#line 16634 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4807 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 16640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 4810 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 16645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 4815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 4816 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 16653 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4819 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 16657 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 4821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 16663 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 16667 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4825 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 4826 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(addPrefix(r.data[0].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 4828 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond( addPrefix(r.data[r.data.size() - 1].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 16675 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 4833 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 16681 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4834 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 16685 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4836 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 4838 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 4839 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 4840 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 4841 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 16697 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4842 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 16701 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7(GetKeyValuesReply && _r,int loopDepth) 
	{
															#line 4788 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 4789 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvReadRangeLatencySample->addMeasurement(duration);
															#line 4790 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 4792 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 16717 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4793 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 16721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4797 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 4801 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 16727 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4802 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 4803 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.tenantInfo.prefix.present())
															#line 16733 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4804 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin.removePrefix(req.tenantInfo.prefix.get()) && r.data[i].key < end.removePrefix(req.tenantInfo.prefix.get()));
															#line 16737 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4807 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 16743 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 4810 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 16748 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4814 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 4815 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 4816 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 16756 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4819 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 16760 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4820 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 4821 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 16766 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4824 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 16770 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4825 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 4826 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(addPrefix(r.data[0].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 4828 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond( addPrefix(r.data[r.data.size() - 1].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 16778 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4832 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 4833 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 16784 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4834 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 16788 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4836 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 4838 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 4839 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 4840 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 4841 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 16800 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4842 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 16804 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(GetKeyValuesReply const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(GetKeyValuesReply && _r,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont11cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesRequest req;
															#line 4644 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 4645 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 4661 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 4664 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 4673 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4690 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 4692 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange shard;
															#line 4715 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset1;
															#line 4716 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset2;
															#line 4717 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fBegin;
															#line 4721 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fEnd;
															#line 4725 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key begin;
															#line 4726 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key end;
															#line 4777 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int remainingLimitBytes;
															#line 4779 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double kvReadRange;
															#line 16932 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getKeyValuesQ()
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActor final : public Actor<Void>, public ActorCallback< GetKeyValuesQActor, 0, Void >, public ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetKeyValuesQActor, 2, Version >, public ActorCallback< GetKeyValuesQActor, 3, Key >, public ActorCallback< GetKeyValuesQActor, 4, Key >, public ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >, public FastAllocated<GetKeyValuesQActor>, public GetKeyValuesQActorState<GetKeyValuesQActor> {
															#line 16937 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetKeyValuesQActor>::operator new;
	using FastAllocated<GetKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9588981003018074112UL, 12857311097653527296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyValuesQActor, 0, Void >;
friend struct ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetKeyValuesQActor, 2, Version >;
friend struct ActorCallback< GetKeyValuesQActor, 3, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 4, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >;
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActor(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 16959 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetKeyValuesQActorState<GetKeyValuesQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15085295619130978816UL, 4562439640475096832UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 3, Key >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 4, Key >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getKeyValuesQ( StorageServer* const& data, GetKeyValuesRequest const& req ) {
															#line 4640 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetKeyValuesQActor(data, req));
															#line 16997 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4871 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 17002 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via quickGetKeyValues()
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetKeyValuesActor>
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActorState {
															#line 17009 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActorState(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   prefix(prefix),
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 4879 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getRange(),
															#line 4880 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValuesStart(g_network->timer())
															#line 17028 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetKeyValuesActorState() 
	{
		fdb_probe_actor_destroy("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4881 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.begin = firstGreaterOrEqual(KeyRef(*a, prefix));
															#line 4882 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.end = firstGreaterOrEqual(strinc(prefix, *a));
															#line 4883 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17047 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4884 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.Before");
															#line 17051 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			try {
															#line 4889 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				GetKeyValuesRequest req;
															#line 4890 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.spanContext = pOriginalReq->spanContext;
															#line 4891 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 4892 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.arena = *a;
															#line 4893 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.begin = getRange.begin;
															#line 4894 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.end = getRange.end;
															#line 4895 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.version = version;
															#line 4896 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.tenantInfo = pOriginalReq->tenantInfo;
															#line 4899 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limit = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT;
															#line 4900 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limitBytes = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT_BYTES;
															#line 4901 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 4903 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.tags = pOriginalReq->tags;
															#line 4904 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.ssLatestCommitVersions = VersionVector();
															#line 4909 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->actors.add(getKeyValuesQ(data, req));
															#line 4910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<GetKeyValuesReply> __when_expr_0 = req.reply.getFuture();
															#line 4910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 17086 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 1;
															#line 4910 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 17091 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4929 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetKeyValuesMiss;
															#line 4930 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_KEY_VALUES_FALLBACK)
															#line 17122 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4931 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<Reference<Tenant>> tenant = pOriginalReq->tenantInfo.hasTenant() ? makeReference<Tenant>(pOriginalReq->tenantInfo.tenantId) : Optional<Reference<Tenant>>();
															#line 4934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx, tenant);
															#line 4935 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 4936 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17132 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4937 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				tr.debugTransaction(pOriginalReq->options.get().debugID.get());
															#line 17136 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4940 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 4941 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<RangeResult> rangeResultFuture = tr.getRange(prefixRange(prefix), GetRangeLimits::ROW_LIMIT_UNLIMITED, Snapshot::True);
															#line 4944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = rangeResultFuture;
															#line 4944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17146 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 2;
															#line 4944 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 17151 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 4955 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_key_values_miss(), loopDepth);
															#line 17158 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply const& reply,int loopDepth) 
	{
															#line 4911 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 17180 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 4914 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 4915 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 4916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 4917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 4918 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17194 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4919 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 17198 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 17202 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply && reply,int loopDepth) 
	{
															#line 4911 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 17216 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4912 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 4914 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 4915 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 4916 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 4917 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 4918 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17230 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4919 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 17234 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4922 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 17238 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(RangeResult const& rangeResult,int loopDepth) 
	{
															#line 4945 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 4946 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 4947 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 4948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 4949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17348 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 17352 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4953 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 17356 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(RangeResult && rangeResult,int loopDepth) 
	{
															#line 4945 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 4946 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 4947 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 4948 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 4949 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 17376 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4950 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 17380 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4953 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 17384 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(rangeResult, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(rangeResult), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef prefix;
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 4879 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetRangeReqAndResultRef getRange;
															#line 4880 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValuesStart;
															#line 4934 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 17483 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via quickGetKeyValues()
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActor final : public Actor<GetRangeReqAndResultRef>, public ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >, public ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >, public FastAllocated<QuickGetKeyValuesActor>, public QuickGetKeyValuesActorState<QuickGetKeyValuesActor> {
															#line 17488 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<QuickGetKeyValuesActor>::operator new;
	using FastAllocated<QuickGetKeyValuesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9929413505407246848UL, 5193088986901632256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetRangeReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >;
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActor(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 17506 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetRangeReqAndResultRef>(),
		   QuickGetKeyValuesActorState<QuickGetKeyValuesActor>(data, prefix, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10745188868789411328UL, 14092707623558697728UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetKeyValues");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetRangeReqAndResultRef> quickGetKeyValues( StorageServer* const& data, StringRef const& prefix, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 4872 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetRangeReqAndResultRef>(new QuickGetKeyValuesActor(data, prefix, version, a, pOriginalReq));
															#line 17540 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4958 "/codebuild/output/src1965261092/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void unpackKeyTuple(Tuple** referenceTuple, Optional<Tuple>& keyTuple, KeyValueRef* keyValue) {
	if (!keyTuple.present()) {
		// May throw exception if the key is not parsable as a tuple.
		try {
			keyTuple = Tuple::unpack(keyValue->key);
		} catch (Error& e) {
			TraceEvent("KeyNotTuple").error(e).detail("Key", keyValue->key.printable());
			throw key_not_tuple();
		}
	}
	*referenceTuple = &keyTuple.get();
}

void unpackValueTuple(Tuple** referenceTuple, Optional<Tuple>& valueTuple, KeyValueRef* keyValue) {
	if (!valueTuple.present()) {
		// May throw exception if the value is not parsable as a tuple.
		try {
			valueTuple = Tuple::unpack(keyValue->value);
		} catch (Error& e) {
			TraceEvent("ValueNotTuple").error(e).detail("Value", keyValue->value.printable());
			throw value_not_tuple();
		}
	}
	*referenceTuple = &valueTuple.get();
}

bool unescapeLiterals(std::string& s, std::string before, std::string after) {
	bool escaped = false;
	size_t p = 0;
	while (true) {
		size_t found = s.find(before, p);
		if (found == std::string::npos) {
			break;
		}
		s.replace(found, before.length(), after);
		p = found + after.length();
		escaped = true;
	}
	return escaped;
}

bool singleKeyOrValue(const std::string& s, size_t sz) {
	// format would be {K[??]} or {V[??]}
	return sz > 5 && s[0] == '{' && (s[1] == 'K' || s[1] == 'V') && s[2] == '[' && s[sz - 2] == ']' && s[sz - 1] == '}';
}

bool rangeQuery(const std::string& s) {
	return s == "{...}";
}

// create a vector of Optional<Tuple>
// in case of a singleKeyOrValue, insert an empty Tuple to vector as placeholder
// in case of a rangeQuery, insert Optional.empty as placeholder
// in other cases, insert the correct Tuple to be used.
void preprocessMappedKey(Tuple& mappedKeyFormatTuple, std::vector<Optional<Tuple>>& vt, bool& isRangeQuery) {
	vt.reserve(mappedKeyFormatTuple.size());

	for (int i = 0; i < mappedKeyFormatTuple.size(); i++) {
		Tuple::ElementType type = mappedKeyFormatTuple.getType(i);
		if (type == Tuple::BYTES || type == Tuple::UTF8) {
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			bool escaped = unescapeLiterals(s, "{{", "{");
			escaped = unescapeLiterals(s, "}}", "}") || escaped;
			if (escaped) {
				vt.emplace_back(Tuple::makeTuple(s));
			} else if (singleKeyOrValue(s, sz)) {
				// when it is SingleKeyOrValue, insert an empty Tuple to vector as placeholder
				vt.emplace_back(Tuple());
			} else if (rangeQuery(s)) {
				if (i != mappedKeyFormatTuple.size() - 1) {
					// It must be the last element of the mapper tuple
					throw mapper_bad_range_decriptor();
				}
				// when it is rangeQuery, insert Optional.empty as placeholder
				vt.emplace_back(Optional<Tuple>());
				isRangeQuery = true;
			} else {
				Tuple t;
				t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
				vt.emplace_back(t);
			}
		} else {
			Tuple t;
			t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
			vt.emplace_back(t);
		}
	}
}

Key constructMappedKey(KeyValueRef* keyValue, std::vector<Optional<Tuple>>& vec, Tuple& mappedKeyFormatTuple) {
	// Lazily parse key and/or value to tuple because they may not need to be a tuple if not used.
	Optional<Tuple> keyTuple;
	Optional<Tuple> valueTuple;
	Tuple mappedKeyTuple;

	mappedKeyTuple.reserve(vec.size());

	for (int i = 0; i < vec.size(); i++) {
		if (!vec[i].present()) {
			// rangeQuery
			continue;
		}
		if (vec[i].get().size()) {
			mappedKeyTuple.append(vec[i].get());
		} else {
			// singleKeyOrValue is true
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			int idx;
			Tuple* referenceTuple;
			try {
				idx = std::stoi(s.substr(3, sz - 5));
			} catch (std::exception& e) {
				throw mapper_bad_index();
			}
			if (s[1] == 'K') {
				unpackKeyTuple(&referenceTuple, keyTuple, keyValue);
			} else if (s[1] == 'V') {
				unpackValueTuple(&referenceTuple, valueTuple, keyValue);
			} else {
				ASSERT(false);
				throw internal_error();
			}
			if (idx < 0 || idx >= referenceTuple->size()) {
				throw mapper_bad_