#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
/*
 * VersionedBTree.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbclient/RandomKeyValueUtils.h"
#include "fdbclient/Tuple.h"
#include "fdbrpc/DDSketch.h"
#include "fdbrpc/simulator.h"
#include "fdbserver/DeltaTree.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/IPager.h"
#include "fdbserver/IPageEncryptionKeyProvider.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/VersionedBTreeDebug.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "flow/ActorCollection.h"
#include "flow/Error.h"
#include "flow/FastRef.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/Histogram.h"
#include "flow/IAsyncFile.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/ObjectSerializer.h"
#include "flow/PriorityMultiLock.actor.h"
#include "flow/network.h"
#include "flow/serialize.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "fmt/format.h"

#include <boost/intrusive/list.hpp>
#include <cinttypes>
#include <limits>
#include <map>
#include <random>
#include <string>
#include <unordered_map>
#include <vector>

#include "flow/actorcompiler.h" // must be last include

using namespace std::string_view_literals;

// Returns a string where every line in lines is prefixed with prefix
std::string addPrefix(std::string prefix, std::string lines) {
	StringRef m = lines;
	std::string s;
	while (m.size() != 0) {
		StringRef line = m.eat("\n");
		s += prefix;
		s += ' ';
		s += line.toString();
		s += '\n';
	}
	return s;
}

// Some convenience functions for debugging to stringify various structures
// Classes can add compatibility by either specializing toString<T> or implementing
//   std::string toString() const;
template <typename T>
std::string toString(const T& o) {
	return o.toString();
}

std::string toString(const std::string& s) {
	return s;
}

std::string toString(StringRef s) {
	return s.printable();
}

std::string toString(LogicalPageID id) {
	if (id == invalidLogicalPageID) {
		return "LogicalPageID{invalid}";
	}
	return format("LogicalPageID{%u}", id);
}

std::string toString(Version v) {
	if (v == invalidVersion) {
		return "invalidVersion";
	}
	return format("@%" PRId64, v);
}

std::string toString(bool b) {
	return b ? "true" : "false";
}

template <typename T>
std::string toString(const Standalone<T>& s) {
	return toString((T)s);
}

template <typename T>
std::string toString(const T* begin, const T* end) {
	std::string r = "{";

	bool comma = false;
	while (begin != end) {
		if (comma) {
			r += ", ";
		} else {
			comma = true;
		}
		r += toString(*begin++);
	}

	r += "}";
	return r;
}

template <typename T>
std::string toString(const std::vector<T>& v) {
	return toString(&v.front(), &v.back() + 1);
}

template <typename T>
std::string toString(const VectorRef<T>& v) {
	return toString(v.begin(), v.end());
}

template <typename K, typename V>
std::string toString(const std::map<K, V>& m) {
	std::string r = "{";
	bool comma = false;
	for (const auto& [key, value] : m) {
		if (comma) {
			r += ", ";
		} else {
			comma = true;
		}
		r += toString(value);
		r += " ";
		r += toString(key);
	}
	r += "}\n";
	return r;
}

template <typename K, typename V>
std::string toString(const std::unordered_map<K, V>& u) {
	std::string r = "{";
	bool comma = false;
	for (const auto& n : u) {
		if (comma) {
			r += ", ";
		} else {
			comma = true;
		}
		r += toString(n.first);
		r += " => ";
		r += toString(n.second);
	}
	r += "}";
	return r;
}

template <typename T>
std::string toString(const Optional<T>& o) {
	if (o.present()) {
		return toString(o.get());
	}
	return "<not present>";
}

template <typename F, typename S>
std::string toString(const std::pair<F, S>& o) {
	return format("{%s, %s}", toString(o.first).c_str(), toString(o.second).c_str());
}

constexpr static int ioMinPriority = 0;
constexpr static int ioLeafPriority = 1;
constexpr static int ioMaxPriority = 3;

// A FIFO queue of T stored as a linked list of pages.
// Main operations are pop(), pushBack(), pushFront(), and flush().
//
// flush() will ensure all queue pages are written to the pager and move the unflushed
// pushFront()'d records onto the front of the queue, in FIFO order.
//
// pop() will only return records that have been flushed, and pops
// from the front of the queue.
//
// Each page contains some number of T items and a link to the next page and starting position on that page.
// When the queue is flushed, the last page in the chain is ended and linked to a newly allocated
// but not-yet-written-to pageID, which future writes after the flush will write to.
// Items pushed onto the front of the queue are written to a separate linked list until flushed,
// at which point that list becomes the new front of the queue.
//
// The write pattern is designed such that no page is ever expected to be valid after
// being written to or updated but not fsync'd.  This is why a new unused page is added
// to the queue, linked to by the last data page, before commit.  The new page can't be
// added and filled with data as part of the next commit because that would mean modifying
// the previous tail page to update its next link, which risks corrupting it and losing
// data that was not yet popped if that write is never fsync'd.
//
// While FIFOQueue stores data inside a pager, it also is used _by_ the pager to hold internal metadata
// such as free lists and logical page remapping activity.
// Therefore, FIFOQueue
//   - does not use versioned page reads
//   - does not use versioned page updates via atomicUpdatePage()
//   - does not update pages in-place
//   - writes to a Page ID only once before freeing it
//
// For clarity, FIFOQueue Page IDs are always typed as PhysicalPageID, as they are always
// physical page IDs since atomic page updates are not used.
//
// Requirements on T
//   - must be trivially copyable
//     OR have a specialization for FIFOQueueCodec<T>
//     OR have the following methods
//       // Deserialize from src into *this, return number of bytes from src consumed
//       int readFromBytes(const uint8_t *src);
//       // Return the size of *this serialized
//       int bytesNeeded() const;
//       // Serialize *this to dst, return number of bytes written to dst
//       int writeToBytes(uint8_t *dst) const;
//  - must be supported by toString(object) (see above)
template <typename T, typename Enable = void>
struct FIFOQueueCodec {
	static T readFromBytes(const uint8_t* src, int& bytesRead) {
		T x;
		bytesRead = x.readFromBytes(src);
		return x;
	}
	static int bytesNeeded(const T& x) { return x.bytesNeeded(); }
	static int writeToBytes(uint8_t* dst, const T& x) { return x.writeToBytes(dst); }
};

template <typename T>
struct FIFOQueueCodec<T, typename std::enable_if<std::is_trivially_copyable<T>::value>::type> {
	static_assert(std::is_trivially_copyable<T>::value);
	static T readFromBytes(const uint8_t* src, int& bytesRead) {
		bytesRead = sizeof(T);
		return *(T*)src;
	}
	static int bytesNeeded(const T& x) { return sizeof(T); }
	static int writeToBytes(uint8_t* dst, const T& x) {
		*(T*)dst = x;
		return sizeof(T);
	}
};

template <typename T, typename Codec = FIFOQueueCodec<T>>
class FIFOQueue {
public:
#pragma pack(push, 1)
	struct QueueState {
		constexpr static FileIdentifier file_identifier = 16482812;

		QueueID queueID;

		// First page in the linked-list queue structure
		PhysicalPageID headPageID;
		// Item offset representing the next item in the queue in the first page
		uint16_t headOffset;
		// Last page in the linked-list queue structure, where new entries will be written
		// Note there is no tailPageOffset, it is always 0 because the tail page has never been written
		PhysicalPageID tailPageID;

		uint32_t numPages;
		// numEntries could technically exceed the max page ID so it is 64 bits
		uint64_t numEntries;

		// State for queues that use pages in contiguous blocks of disk space called extents
		bool usesExtents;
		PhysicalPageID prevExtentEndPageID;
		bool tailPageNewExtent; // Tail page points to the start of a new extent

		std::string toString() const {
			return format("{queueID: %u head: %s:%d  tail: %s  numPages: %" PRId64 "  numEntries: %" PRId64
			              "  usesExtents:%d}",
			              queueID,
			              ::toString(headPageID).c_str(),
			              (int)headOffset,
			              ::toString(tailPageID).c_str(),
			              numPages,
			              numEntries,
			              usesExtents);
		}

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar,
			           queueID,
			           headPageID,
			           headOffset,
			           tailPageID,
			           numPages,
			           numEntries,
			           usesExtents,
			           prevExtentEndPageID,
			           tailPageNewExtent);
		}
	};

	struct QueuePage {
		// The next page of the queue after this one
		PhysicalPageID nextPageID;
		// The start offset of the next page
		uint16_t nextOffset;
		// The end offset of the current page's data entries
		uint16_t endOffset;
		// Current page within the extent
		PhysicalPageID extentCurPageID;
		// The end page within the extent
		PhysicalPageID extentEndPageID;
		uint16_t itemSpace;
		// Get pointer to data after page header
		const uint8_t* begin() const { return (const uint8_t*)(this + 1); }
		uint8_t* begin() { return (uint8_t*)(this + 1); }
	};
#pragma pack(pop)

	struct Cursor {
		// Queue mode
		enum Mode { NONE, POP, READONLY, WRITE };
		Mode mode;

		// Queue this cursor is accessing
		FIFOQueue* queue;

		// The current page and pageID being read or written to
		PhysicalPageID pageID;
		Reference<ArenaPage> page;

		// The first page ID to be written to the pager, if this cursor has written anything
		PhysicalPageID firstPageIDWritten;

		// Offset after QueuePage::begin() to next read from or write to
		int offset;

		// A read cursor will not read this page (or beyond)
		PhysicalPageID endPageID;

		// Page future and corresponding page ID for the expected next page to be used.  It may not
		// match the current page's next page link because queues can prepended with new front pages.
		Future<Reference<ArenaPage>> nextPageReader;
		PhysicalPageID nextPageID;

		// Future that represents all outstanding write operations previously issued
		// This exists because writing the queue returns void, not a future
		Future<Void> writeOperations;

		FlowMutex mutex;

		Cursor() : mode(NONE), mutex(true) {}

		// Initialize a cursor.
		void init(FIFOQueue* q = nullptr,
		          Mode m = NONE,
		          PhysicalPageID initialPageID = invalidPhysicalPageID,
		          bool initExtentInfo = true,
		          bool tailPageNewExtent = false,
		          PhysicalPageID endPage = invalidPhysicalPageID,
		          int readOffset = 0,
		          PhysicalPageID prevExtentEndPageID = invalidPhysicalPageID) {
			queue = q;
			mode = m;
			firstPageIDWritten = invalidPhysicalPageID;
			offset = readOffset;
			endPageID = endPage;
			page.clear();
			writeOperations = Void();

			if (mode == POP || mode == READONLY) {
				// If cursor is not pointed at the end page then start loading it.
				// The end page will not have been written to disk yet.
				pageID = initialPageID;
				if (pageID != endPageID) {
					// For extent based queues, we loads extents at a time during recovery
					if (queue->usesExtents && initExtentInfo)
						loadExtent();
					else
						startNextPageLoad(pageID);
				} else {
					nextPageID = invalidPhysicalPageID;
				}
			} else {
				pageID = invalidPhysicalPageID;
				ASSERT(mode == WRITE ||
				       (initialPageID == invalidPhysicalPageID && readOffset == 0 && endPage == invalidPhysicalPageID));
			}

			debug_printf("FIFOQueue::Cursor(%s) initialized\n", toString().c_str());

			if (mode == WRITE && initialPageID != invalidPhysicalPageID) {
				debug_printf("FIFOQueue::Cursor(%s) init. Adding new page %u\n", toString().c_str(), initialPageID);
				addNewPage(initialPageID, 0, true, initExtentInfo, tailPageNewExtent, prevExtentEndPageID);
			}
		}

		// Reset the read cursor (this is used only for extent based remap queue after recovering the remap
		// queue contents via fastpath extent reads)
		void resetRead() {
			ASSERT(mode == POP || mode == READONLY);
			page.clear();
			if (pageID != endPageID) {
				startNextPageLoad(pageID);
			}
		}

		// Since cursors can have async operations pending which modify their state they can't be copied cleanly
		Cursor(const Cursor& other) = delete;

		~Cursor() { cancel(); }

		// Cancel outstanding operations.  Further use of cursor is not allowed.
		void cancel() {
			nextPageReader.cancel();
			writeOperations.cancel();
		}

		// A read cursor can be initialized from a pop cursor
		void initReadOnly(const Cursor& c, bool readExtents = false) {
			ASSERT(c.mode == READONLY || c.mode == POP);
			init(c.queue, READONLY, c.pageID, readExtents, false, c.endPageID, c.offset);
		}

		std::string toString() const {
			if (mode == WRITE) {
				return format("{WriteCursor %s:%p pos=%s:%d rawEndOffset=%d}",
				              queue->name.c_str(),
				              this,
				              ::toString(pageID).c_str(),
				              offset,
				              page ? header()->endOffset : -1);
			}
			if (mode == POP || mode == READONLY) {
				return format("{ReadCursor %s:%p pos=%s:%d rawEndOffset=%d endPage=%s nextPage=%s}",
				              queue->name.c_str(),
				              this,
				              ::toString(pageID).c_str(),
				              offset,
				              (page && header()) ? header()->endOffset : -1,
				              ::toString(endPageID).c_str(),
				              ::toString(nextPageID).c_str());
			}
			ASSERT(mode == NONE);
			return format("{NullCursor=%p}", this);
		}

		// Returns true if the mutex cannot be immediately taken.
		bool isBusy() const { return !mutex.available(); }

		// Wait for all operations started before now to be ready, which is done by
		// obtaining and releasing the mutex.
		Future<Void> notBusy() {
			return isBusy() ? map(mutex.take(),
			                      [&](FlowMutex::Lock lock) {
				                      lock.release();
				                      return Void();
			                      })
			                : Void();
		}

		// Returns true if any items have been written to the last page
		bool pendingTailWrites() const { return mode == WRITE && offset != 0; }

		QueuePage* header() const { return ((QueuePage*)(page->mutateData())); }

		void setNext(PhysicalPageID pageID, int offset) {
			ASSERT(mode == WRITE);
			QueuePage* p = header();
			p->nextPageID = pageID;
			p->nextOffset = offset;
		}

		void startNextPageLoad(PhysicalPageID id) {
			nextPageID = id;
			debug_printf(
			    "FIFOQueue::Cursor(%s) loadPage start id=%s\n", toString().c_str(), ::toString(nextPageID).c_str());
			nextPageReader = queue->pager->readPage(
			    PagerEventReasons::MetaData, nonBtreeLevel, nextPageID, ioMaxPriority, true, false);
			if (!nextPageReader.isReady()) {
				nextPageReader = waitOrError(nextPageReader, queue->pagerError);
			}
		}

		Future<Void> loadExtent() {
			ASSERT(mode == POP | mode == READONLY);
			debug_printf("FIFOQueue::Cursor(%s) loadExtent\n", toString().c_str());
			return map(queue->pager->readExtent(pageID), [=](Reference<ArenaPage> p) {
				page = p;
				debug_printf("FIFOQueue::Cursor(%s) loadExtent done. Page: %p\n", toString().c_str(), page->rawData());
				return Void();
			});
		}

		void writePage() {
			ASSERT(mode == WRITE);
			debug_printf("FIFOQueue::Cursor(%s) writePage\n", toString().c_str());

			// Zero unused space within itemspace
			memset(header()->begin() + header()->endOffset, 0, header()->itemSpace - header()->endOffset);

			queue->pager->updatePage(
			    PagerEventReasons::MetaData, nonBtreeLevel, VectorRef<PhysicalPageID>(&pageID, 1), page);
			if (firstPageIDWritten == invalidPhysicalPageID) {
				firstPageIDWritten = pageID;
			}
		}

		// Link the current page to newPageID:newOffset and then write it to the pager.
		// The link destination could be a new page at the end of the queue, or the beginning of
		// an existing chain of queue pages.
		// If initializeNewPage is true a page buffer will be allocated for the new page and it will be initialized
		// as a new tail page.
		// if initializeExtentInfo is true in addition to initializeNewPage, update the extentEndPageID info
		// in the mew page being added using newExtentPage and prevExtentEndPageID parameters
		void addNewPage(PhysicalPageID newPageID,
		                int newOffset,
		                bool initializeNewPage,
		                bool initializeExtentInfo = false,
		                bool newExtentPage = false,
		                PhysicalPageID prevExtentEndPageID = invalidPhysicalPageID) {
			ASSERT(mode == WRITE);
			ASSERT(newPageID != invalidPhysicalPageID);
			debug_printf("FIFOQueue::Cursor(%s) Adding page %s initPage=%d initExtentInfo=%d newExtentPage=%d\n",
			             toString().c_str(),
			             ::toString(newPageID).c_str(),
			             initializeNewPage,
			             initializeExtentInfo,
			             newExtentPage);

			// Update existing page/newLastPageID and write, if it exists
			if (page) {
				setNext(newPageID, newOffset);
				debug_printf("FIFOQueue::Cursor(%s) Linked new page %s:%d\n",
				             toString().c_str(),
				             ::toString(newPageID).c_str(),
				             newOffset);
				writePage();
				prevExtentEndPageID = header()->extentEndPageID;
				if (pageID == prevExtentEndPageID)
					newExtentPage = true;
				debug_printf(
				    "FIFOQueue::Cursor(%s) Linked new page. pageID %u, newPageID %u,  prevExtentEndPageID %u\n",
				    toString().c_str(),
				    pageID,
				    newPageID,
				    prevExtentEndPageID);
			}

			pageID = newPageID;
			offset = newOffset;

			if (initializeNewPage) {
				debug_printf("FIFOQueue::Cursor(%s) Initializing new page. usesExtents: %d, initializeExtentInfo: %d\n",
				             toString().c_str(),
				             queue->usesExtents,
				             initializeExtentInfo);
				page = queue->pager->newPageBuffer();
				page->init(EncodingType::XXHash64,
				           queue->usesExtents ? PageType::QueuePageInExtent : PageType::QueuePageStandalone,
				           (uint8_t)queue->queueID);
				setNext(0, 0);
				auto p = header();
				ASSERT(newOffset == 0);
				p->endOffset = 0;
				p->itemSpace = page->dataSize() - sizeof(QueuePage);
				if (g_network->isSimulated() && deterministicRandom()->coinflip()) {
					// Randomly reduce available item space to cause more queue pages to be needed
					int reducedSpace = deterministicRandom()->randomInt(50, p->itemSpace);

					// Zero the eliminated space
					memset(header()->begin() + reducedSpace, 0, p->itemSpace - reducedSpace);

					p->itemSpace = reducedSpace;
				}

				// For extent based queue, update the index of current page within the extent
				if (queue->usesExtents) {
					debug_printf("FIFOQueue::Cursor(%s) Adding page %s init=%d pageCount % " PRId64 "\n",
					             toString().c_str(),
					             ::toString(newPageID).c_str(),
					             initializeNewPage,
					             queue->pager->getPageCount());
					p->extentCurPageID = newPageID;
					if (initializeExtentInfo) {
						int pagesPerExtent = queue->pagesPerExtent;
						if (newExtentPage) {
							p->extentEndPageID = newPageID + pagesPerExtent - 1;
							debug_printf("FIFOQueue::Cursor(%s) newExtentPage. newPageID %u, pagesPerExtent %d, "
							             "ExtentEndPageID: %s\n",
							             toString().c_str(),
							             newPageID,
							             pagesPerExtent,
							             ::toString(p->extentEndPageID).c_str());
						} else {
							p->extentEndPageID = prevExtentEndPageID;
							debug_printf("FIFOQueue::Cursor(%s) Copied ExtentEndPageID: %s\n",
							             toString().c_str(),
							             ::toString(p->extentEndPageID).c_str());
						}
					}
				} else {
					p->extentCurPageID = invalidPhysicalPageID;
					p->extentEndPageID = invalidPhysicalPageID;
				}
			} else {
				debug_printf("FIFOQueue::Cursor(%s) Clearing new page\n", toString().c_str());
				page.clear();
			}
		}

		// Write item to the next position in the current page or, if it won't fit, add a new page and write it there.
																	#line 634 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via write_impl()
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Write_implActor>
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Write_implActorState {
															#line 640 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Write_implActorState(Cursor* const& self,T const& item) 
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   item(item)
															#line 649 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("write_impl", reinterpret_cast<unsigned long>(this));

	}
	~Write_implActorState() 
	{
		fdb_probe_actor_destroy("write_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 633 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(self->mode == WRITE);
															#line 635 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			lock = FlowMutex::Lock();
															#line 636 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			mustWait = self->isBusy();
															#line 637 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			bytesNeeded = Codec::bytesNeeded(item);
															#line 638 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			needNewPage = self->pageID == invalidPhysicalPageID || self->offset + bytesNeeded > self->header()->itemSpace;
															#line 641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated())
															#line 674 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 644 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if ((self->offset > 0) && deterministicRandom()->random01() < 0.01)
															#line 678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 645 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					needNewPage = true;
															#line 682 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 649 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) write(%s) mustWait=%d needNewPage=%d\n", self->toString().c_str(), ::toString(item).c_str(), mustWait, needNewPage);
															#line 656 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (mustWait || needNewPage)
															#line 689 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<FlowMutex::Lock> __when_expr_0 = self->mutex.take();
															#line 657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<Write_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 695 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<Write_implActor*>(this)->actor_wait_state = 1;
															#line 657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Write_implActor, 0, FlowMutex::Lock >*>(static_cast<Write_implActor*>(this)));
															#line 700 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Write_implActorState();
		static_cast<Write_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (needNewPage)
															#line 728 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) write(%s) page is full, adding new page\n", self->toString().c_str(), ::toString(item).c_str());
															#line 681 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			newPageID = PhysicalPageID();
															#line 683 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->queue->usesExtents)
															#line 736 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 684 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				bool allocateNewExtent = false;
															#line 685 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->pageID != invalidPhysicalPageID)
															#line 742 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 686 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					auto praw = self->header();
															#line 687 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (praw->extentCurPageID < praw->extentEndPageID)
															#line 748 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					{
															#line 688 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						newPageID = praw->extentCurPageID + 1;
															#line 752 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
					else
					{
															#line 690 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						allocateNewExtent = true;
															#line 758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
				}
				else
				{
															#line 693 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					allocateNewExtent = true;
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
															#line 694 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (allocateNewExtent)
															#line 769 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 695 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					StrictFuture<PhysicalPageID> __when_expr_1 = self->queue->pager->newExtentPageID(self->queue->queueID);
															#line 695 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (static_cast<Write_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 775 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<Write_implActor*>(this)->actor_wait_state = 2;
															#line 695 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Write_implActor, 1, PhysicalPageID >*>(static_cast<Write_implActor*>(this)));
															#line 780 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont11(loopDepth);
				}
			}
			else
			{
															#line 699 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<PhysicalPageID> __when_expr_2 = self->queue->pager->newPageID();
															#line 699 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<Write_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 794 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
				static_cast<Write_implActor*>(this)->actor_wait_state = 3;
															#line 699 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Write_implActor, 2, PhysicalPageID >*>(static_cast<Write_implActor*>(this)));
															#line 799 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(FlowMutex::Lock const& _lock,int loopDepth) 
	{
															#line 658 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		lock = _lock;
															#line 663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (mustWait)
															#line 816 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 664 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			needNewPage = self->pageID == invalidPhysicalPageID || self->offset + bytesNeeded > self->header()->itemSpace;
															#line 666 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated())
															#line 822 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if ((self->offset > 0) && deterministicRandom()->random01() < 0.01)
															#line 826 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 670 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					needNewPage = true;
															#line 830 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(FlowMutex::Lock && _lock,int loopDepth) 
	{
															#line 658 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		lock = _lock;
															#line 663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (mustWait)
															#line 844 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 664 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			needNewPage = self->pageID == invalidPhysicalPageID || self->offset + bytesNeeded > self->header()->itemSpace;
															#line 666 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated())
															#line 850 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if ((self->offset > 0) && deterministicRandom()->random01() < 0.01)
															#line 854 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 670 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					needNewPage = true;
															#line 858 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(FlowMutex::Lock const& _lock,int loopDepth) 
	{
		loopDepth = a_body1cont4(_lock, loopDepth);

		return loopDepth;
	}
	int a_body1when1(FlowMutex::Lock && _lock,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_lock), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Write_implActor*>(this)->actor_wait_state > 0) static_cast<Write_implActor*>(this)->actor_wait_state = 0;
		static_cast<Write_implActor*>(this)->ActorCallback< Write_implActor, 0, FlowMutex::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< Write_implActor, 0, FlowMutex::Lock >*,FlowMutex::Lock const& value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Write_implActor, 0, FlowMutex::Lock >*,FlowMutex::Lock && value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Write_implActor, 0, FlowMutex::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont9(int loopDepth) 
	{
															#line 707 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf( "FIFOQueue::Cursor(%s) write(%s) writing\n", self->toString().c_str(), ::toString(item).c_str());
															#line 709 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto p = self->header();
															#line 710 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		Codec::writeToBytes(p->begin() + self->offset, item);
															#line 711 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->offset += bytesNeeded;
															#line 712 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		p->endOffset = self->offset;
															#line 713 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++self->queue->numEntries;
															#line 715 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (mustWait || needNewPage)
															#line 957 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 718 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			static int sinceYield = 0;
															#line 719 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (++sinceYield == 1000)
															#line 963 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 720 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				sinceYield = 0;
															#line 721 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_3 = delay(0);
															#line 721 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<Write_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont9when1(__when_expr_3.get(), loopDepth); };
				static_cast<Write_implActor*>(this)->actor_wait_state = 4;
															#line 721 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Write_implActor, 3, Void >*>(static_cast<Write_implActor*>(this)));
															#line 976 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont19(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont18(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
															#line 702 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->addNewPage(newPageID, 0, true, true);
															#line 704 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++self->queue->numPages;
															#line 997 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(PhysicalPageID const& newPID,int loopDepth) 
	{
															#line 696 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		newPageID = newPID;
															#line 1012 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(PhysicalPageID && newPID,int loopDepth) 
	{
															#line 696 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		newPageID = newPID;
															#line 1021 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(PhysicalPageID const& newPID,int loopDepth) 
	{
		loopDepth = a_body1cont16(newPID, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(PhysicalPageID && newPID,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(newPID), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Write_implActor*>(this)->actor_wait_state > 0) static_cast<Write_implActor*>(this)->actor_wait_state = 0;
		static_cast<Write_implActor*>(this)->ActorCallback< Write_implActor, 1, PhysicalPageID >::remove();

	}
	void a_callback_fire(ActorCallback< Write_implActor, 1, PhysicalPageID >*,PhysicalPageID const& value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Write_implActor, 1, PhysicalPageID >*,PhysicalPageID && value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Write_implActor, 1, PhysicalPageID >*,Error err) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont17(PhysicalPageID const& newPID,int loopDepth) 
	{
															#line 700 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		newPageID = newPID;
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont17(PhysicalPageID && newPID,int loopDepth) 
	{
															#line 700 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		newPageID = newPID;
															#line 1114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(PhysicalPageID const& newPID,int loopDepth) 
	{
		loopDepth = a_body1cont17(newPID, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(PhysicalPageID && newPID,int loopDepth) 
	{
		loopDepth = a_body1cont17(std::move(newPID), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Write_implActor*>(this)->actor_wait_state > 0) static_cast<Write_implActor*>(this)->actor_wait_state = 0;
		static_cast<Write_implActor*>(this)->ActorCallback< Write_implActor, 2, PhysicalPageID >::remove();

	}
	void a_callback_fire(ActorCallback< Write_implActor, 2, PhysicalPageID >*,PhysicalPageID const& value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Write_implActor, 2, PhysicalPageID >*,PhysicalPageID && value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Write_implActor, 2, PhysicalPageID >*,Error err) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont18(int loopDepth) 
	{
															#line 727 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Write_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Write_implActorState(); static_cast<Write_implActor*>(this)->destroy(); return 0; }
															#line 1198 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Write_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Write_implActorState();
		static_cast<Write_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont19(int loopDepth) 
	{
															#line 724 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		lock.release();
															#line 1210 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont20(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont20(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont20(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont20(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Write_implActor*>(this)->actor_wait_state > 0) static_cast<Write_implActor*>(this)->actor_wait_state = 0;
		static_cast<Write_implActor*>(this)->ActorCallback< Write_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Write_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Write_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Write_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Cursor* self;
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	T item;
															#line 635 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FlowMutex::Lock lock;
															#line 636 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool mustWait;
															#line 637 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int bytesNeeded;
															#line 638 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool needNewPage;
															#line 681 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PhysicalPageID newPageID;
															#line 1316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via write_impl()
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Write_implActor final : public Actor<Void>, public ActorCallback< Write_implActor, 0, FlowMutex::Lock >, public ActorCallback< Write_implActor, 1, PhysicalPageID >, public ActorCallback< Write_implActor, 2, PhysicalPageID >, public ActorCallback< Write_implActor, 3, Void >, public FastAllocated<Write_implActor>, public Write_implActorState<Write_implActor> {
															#line 1321 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Write_implActor>::operator new;
	using FastAllocated<Write_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16795426009057734400UL, 10335740687973240320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Write_implActor, 0, FlowMutex::Lock >;
friend struct ActorCallback< Write_implActor, 1, PhysicalPageID >;
friend struct ActorCallback< Write_implActor, 2, PhysicalPageID >;
friend struct ActorCallback< Write_implActor, 3, Void >;
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Write_implActor(Cursor* const& self,T const& item) 
															#line 1341 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   Write_implActorState<Write_implActor>(self, item),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("write_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9097952400270967040UL, 7491439827751099648UL);
		ActorExecutionContextHelper __helper(static_cast<Write_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("write_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("write_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Write_implActor, 0, FlowMutex::Lock >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Write_implActor, 1, PhysicalPageID >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Write_implActor, 2, PhysicalPageID >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Write_implActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> write_impl( Cursor* const& self, T const& item ) {
															#line 632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new Write_implActor(self, item));
															#line 1376 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 729 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

		void write(const T& item) {
			// Start the write.  It may complete immediately if no IO was being waited on
			Future<Void> w = write_impl(this, item);
			// If it didn't complete immediately, then store the future in operation
			if (!w.isReady()) {
				writeOperations = writeOperations && w;
			}
		}

		// If readNext() cannot complete immediately because it must wait for IO, it will route to here.
		// The purpose of this function is to serialize simultaneous readers on self while letting the
		// common case (>99.8% of the time) be handled with low overhead by the non-actor readNext() function.
		//
		// The mutex will be taken if locked is false.
		// The next page will be waited for if load is true.
		// Only mutex holders will wait on the page read.
																	#line 1397 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via waitThenReadNext()
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class WaitThenReadNextActor>
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class WaitThenReadNextActorState {
															#line 1403 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	WaitThenReadNextActorState(Cursor* const& self,Optional<T> const& inclusiveMaximum,FlowMutex::Lock* const& lock,bool const& load) 
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   inclusiveMaximum(inclusiveMaximum),
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   lock(lock),
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   load(load),
															#line 750 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   localLock()
															#line 1418 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("waitThenReadNext", reinterpret_cast<unsigned long>(this));

	}
	~WaitThenReadNextActorState() 
	{
		fdb_probe_actor_destroy("waitThenReadNext", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (lock == nullptr)
															#line 1433 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 754 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("FIFOQueue::Cursor(%s) waitThenReadNext locking mutex\n", self->toString().c_str());
															#line 755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<FlowMutex::Lock> __when_expr_0 = self->mutex.take();
															#line 755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1441 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 1;
															#line 755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >*>(static_cast<WaitThenReadNextActor*>(this)));
															#line 1446 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitThenReadNextActorState();
		static_cast<WaitThenReadNextActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 759 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (load)
															#line 1474 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 760 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) waitThenReadNext waiting for page load\n", self->toString().c_str());
															#line 762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(self->nextPageReader);
															#line 762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1482 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 2;
															#line 762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitThenReadNextActor, 1, Void >*>(static_cast<WaitThenReadNextActor*>(this)));
															#line 1487 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(FlowMutex::Lock const& newLock,int loopDepth) 
	{
															#line 756 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		localLock = newLock;
															#line 1501 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(FlowMutex::Lock && newLock,int loopDepth) 
	{
															#line 756 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		localLock = newLock;
															#line 1510 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(FlowMutex::Lock const& newLock,int loopDepth) 
	{
		loopDepth = a_body1cont2(newLock, loopDepth);

		return loopDepth;
	}
	int a_body1when1(FlowMutex::Lock && newLock,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(newLock), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state > 0) static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 0;
		static_cast<WaitThenReadNextActor*>(this)->ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >*,FlowMutex::Lock const& value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >*,FlowMutex::Lock && value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<T>> __when_expr_2 = self->readNext(inclusiveMaximum, &localLock);
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 3;
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitThenReadNextActor, 2, Optional<T> >*>(static_cast<WaitThenReadNextActor*>(this)));
															#line 1601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state > 0) static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 0;
		static_cast<WaitThenReadNextActor*>(this)->ActorCallback< WaitThenReadNextActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitThenReadNextActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 768 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (lock == nullptr)
															#line 1697 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 771 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			static int sinceYield = 0;
															#line 772 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (++sinceYield == 1000)
															#line 1703 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 773 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				sinceYield = 0;
															#line 774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_3 = delay(0);
															#line 774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1711 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
				static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 4;
															#line 774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitThenReadNextActor, 3, Void >*>(static_cast<WaitThenReadNextActor*>(this)));
															#line 1716 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(Optional<T> const& __result,int loopDepth) 
	{
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		result = __result;
															#line 1735 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Optional<T> && __result,int loopDepth) 
	{
		result = std::move(__result);
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state > 0) static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 0;
		static_cast<WaitThenReadNextActor*>(this)->ActorCallback< WaitThenReadNextActor, 2, Optional<T> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 2, Optional<T> >*,Optional<T> const& value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 2, Optional<T> >*,Optional<T> && value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitThenReadNextActor, 2, Optional<T> >*,Error err) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 781 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<WaitThenReadNextActor*>(this)->SAV<Optional<T>>::futures) { (void)(result); this->~WaitThenReadNextActorState(); static_cast<WaitThenReadNextActor*>(this)->destroy(); return 0; }
															#line 1814 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<WaitThenReadNextActor*>(this)->SAV< Optional<T> >::value()) Optional<T>(std::move(result)); // state_var_RVO
		this->~WaitThenReadNextActorState();
		static_cast<WaitThenReadNextActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 777 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("FIFOQueue::Cursor(%s) waitThenReadNext unlocking mutex\n", self->toString().c_str());
															#line 778 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		localLock.release();
															#line 1828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitThenReadNextActor*>(this)->actor_wait_state > 0) static_cast<WaitThenReadNextActor*>(this)->actor_wait_state = 0;
		static_cast<WaitThenReadNextActor*>(this)->ActorCallback< WaitThenReadNextActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitThenReadNextActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitThenReadNextActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Cursor* self;
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Optional<T> inclusiveMaximum;
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FlowMutex::Lock* lock;
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool load;
															#line 750 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FlowMutex::Lock localLock;
															#line 765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Optional<T> result;
															#line 1932 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via waitThenReadNext()
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class WaitThenReadNextActor final : public Actor<Optional<T>>, public ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >, public ActorCallback< WaitThenReadNextActor, 1, Void >, public ActorCallback< WaitThenReadNextActor, 2, Optional<T> >, public ActorCallback< WaitThenReadNextActor, 3, Void >, public FastAllocated<WaitThenReadNextActor>, public WaitThenReadNextActorState<WaitThenReadNextActor> {
															#line 1937 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<WaitThenReadNextActor>::operator new;
	using FastAllocated<WaitThenReadNextActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10453865537884106752UL, 15584701812979947008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<T>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >;
friend struct ActorCallback< WaitThenReadNextActor, 1, Void >;
friend struct ActorCallback< WaitThenReadNextActor, 2, Optional<T> >;
friend struct ActorCallback< WaitThenReadNextActor, 3, Void >;
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	WaitThenReadNextActor(Cursor* const& self,Optional<T> const& inclusiveMaximum,FlowMutex::Lock* const& lock,bool const& load) 
															#line 1957 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Optional<T>>(),
		   WaitThenReadNextActorState<WaitThenReadNextActor>(self, inclusiveMaximum, lock, load),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitThenReadNext", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14200081372596547072UL, 6426192591306499328UL);
		ActorExecutionContextHelper __helper(static_cast<WaitThenReadNextActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitThenReadNext");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitThenReadNext", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitThenReadNextActor, 0, FlowMutex::Lock >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitThenReadNextActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitThenReadNextActor, 2, Optional<T> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitThenReadNextActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Optional<T>> waitThenReadNext( Cursor* const& self, Optional<T> const& inclusiveMaximum, FlowMutex::Lock* const& lock, bool const& load ) {
															#line 746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Optional<T>>(new WaitThenReadNextActor(self, inclusiveMaximum, lock, load));
															#line 1992 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 783 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

		// Read the next item from the cursor, possibly moving to and waiting for a new page if the prior page was
		// exhausted. If the item is <= inclusiveMaximum, then return it after advancing the cursor to the next item.
		// Otherwise, return nothing and do not advance the cursor.
		// If locked is true, this call owns the mutex, which would have been locked by readNext() before a recursive
		// call. See waitThenReadNext() for more detail.
		Future<Optional<T>> readNext(const Optional<T>& inclusiveMaximum = {}, FlowMutex::Lock* lock = nullptr) {
			if ((mode != POP && mode != READONLY) || pageID == invalidPhysicalPageID || pageID == endPageID) {
				debug_printf("FIFOQueue::Cursor(%s) readNext returning nothing\n", toString().c_str());
				return Optional<T>();
			}

			// If we don't have a lock and the mutex isn't available then acquire it
			if (lock == nullptr && isBusy()) {
				return waitThenReadNext(this, inclusiveMaximum, lock, false);
			}

			// We now know pageID is valid and should be used, but page might not point to it yet
			if (!page) {
				debug_printf("FIFOQueue::Cursor(%s) loading\n", toString().c_str());

				// If the next pageID loading or loaded is not the page we should be reading then restart the load
				// nextPageID could be different because it could be invalid or it could be no longer relevant
				// if the previous commit added new pages to the front of the queue.
				if (pageID != nextPageID) {
					debug_printf("FIFOQueue::Cursor(%s) reloading\n", toString().c_str());
					startNextPageLoad(pageID);
				}

				if (!nextPageReader.isReady()) {
					return waitThenReadNext(this, inclusiveMaximum, lock, true);
				}

				page = nextPageReader.get();

				// Start loading the next page if it's not the end page
				auto p = header();
				if (p->nextPageID != endPageID) {
					startNextPageLoad(p->nextPageID);
				} else {
					// Prevent a future next page read from reusing the same result as page would have to be updated
					// before the queue would read it again
					nextPageID = invalidPhysicalPageID;
				}
			}
			auto p = header();
			debug_printf("FIFOQueue::Cursor(%s) readNext reading at current position\n", toString().c_str());
			ASSERT(offset < p->endOffset);
			int bytesRead;
			const T result = Codec::readFromBytes(p->begin() + offset, bytesRead);

			if (inclusiveMaximum.present() && inclusiveMaximum.get() < result) {
				debug_printf("FIFOQueue::Cursor(%s) not popping %s, exceeds upper bound %s\n",
				             toString().c_str(),
				             ::toString(result).c_str(),
				             ::toString(inclusiveMaximum.get()).c_str());

				return Optional<T>();
			}

			offset += bytesRead;
			if (mode == POP) {
				--queue->numEntries;
			}
			debug_printf("FIFOQueue::Cursor(%s) after read of %s\n", toString().c_str(), ::toString(result).c_str());
			ASSERT(offset <= p->endOffset);

			// If this page is exhausted, start reading the next page for the next readNext() to use, unless it's the
			// tail page
			if (offset == p->endOffset) {
				debug_printf("FIFOQueue::Cursor(%s) Page exhausted\n", toString().c_str());
				PhysicalPageID oldPageID = pageID;
				PhysicalPageID extentCurPageID = p->extentCurPageID;
				PhysicalPageID extentEndPageID = p->extentEndPageID;
				pageID = p->nextPageID;
				offset = p->nextOffset;

				// If pageID isn't the tail page and nextPageID isn't pageID then start loading the next page
				if (pageID != endPageID && nextPageID != pageID) {
					startNextPageLoad(pageID);
				}

				if (mode == POP) {
					--queue->numPages;
				}

				page.clear();
				debug_printf("FIFOQueue::Cursor(%s) readNext page exhausted, moved to new page\n", toString().c_str());
				if (mode == POP) {
					if (!queue->usesExtents) {
						// Freeing the old page must happen after advancing the cursor and clearing the page reference
						// because freePage() could cause a push onto a queue that causes a newPageID() call which could
						// pop() from this very same queue. Queue pages are freed at version 0 because they can be
						// reused after the next commit.
						queue->pager->freePage(oldPageID, 0);
					} else if (extentCurPageID == extentEndPageID) {
						// Figure out the beginning of the extent
						int pagesPerExtent = queue->pagesPerExtent;
						queue->pager->freeExtent(oldPageID - pagesPerExtent + 1);
					}
				}
			}

			debug_printf("FIFOQueue(%s) %s(inclusiveMaximum=%s) -> %s\n",
			             queue->name.c_str(),
			             (mode == POP ? "pop" : "peek"),
			             ::toString(inclusiveMaximum).c_str(),
			             ::toString(result).c_str());
			return Optional<T>(result);
		}
	};

public:
	FIFOQueue() : pager(nullptr) {}

	~FIFOQueue() { cancel(); }

	// Cancel outstanding operations.  Further use of queue is not allowed.
	void cancel() {
		headReader.cancel();
		tailWriter.cancel();
		headWriter.cancel();
		newTailPage.cancel();
	}

	FIFOQueue(const FIFOQueue& other) = delete;
	void operator=(const FIFOQueue& rhs) = delete;

	// Create a new queue at newPageID
	void create(IPager2* p, PhysicalPageID newPageID, std::string queueName, QueueID id, bool extent) {
		debug_printf("FIFOQueue(%s) create from page %s. usesExtents %d\n",
		             queueName.c_str(),
		             toString(newPageID).c_str(),
		             extent);
		pager = p;
		pagerError = pager->getError();
		name = queueName;
		queueID = id;
		numPages = 1;
		numEntries = 0;
		usesExtents = extent;
		tailPageNewExtent = false;
		prevExtentEndPageID = invalidPhysicalPageID;
		pagesPerExtent = pager->getPagesPerExtent();
		headReader.init(this, Cursor::POP, newPageID, false, false, newPageID, 0);
		tailWriter.init(this, Cursor::WRITE, newPageID, true, true);
		headWriter.init(this, Cursor::WRITE);
		newTailPage = invalidPhysicalPageID;
		debug_printf("FIFOQueue(%s) created\n", queueName.c_str());
	}

	// Load an existing queue from its queue state
	void recover(IPager2* p, const QueueState& qs, std::string queueName, bool loadExtents = true) {
		debug_printf("FIFOQueue(%s) recover from queue state %s\n", queueName.c_str(), qs.toString().c_str());
		pager = p;
		pagerError = pager->getError();
		name = queueName;
		queueID = qs.queueID;
		numPages = qs.numPages;
		numEntries = qs.numEntries;
		usesExtents = qs.usesExtents;
		tailPageNewExtent = qs.tailPageNewExtent;
		prevExtentEndPageID = qs.prevExtentEndPageID;
		pagesPerExtent = pager->getPagesPerExtent();
		headReader.init(this, Cursor::POP, qs.headPageID, loadExtents, false, qs.tailPageID, qs.headOffset);
		tailWriter.init(this,
		                Cursor::WRITE,
		                qs.tailPageID,
		                true,
		                qs.tailPageNewExtent,
		                invalidPhysicalPageID,
		                0,
		                qs.prevExtentEndPageID);
		headWriter.init(this, Cursor::WRITE);
		newTailPage = invalidPhysicalPageID;
		debug_printf("FIFOQueue(%s) recovered\n", queueName.c_str());
	}

	// Reset the head reader (this is used only for extent based remap queue after recovering the remap
	// queue contents via fastpath extent reads)
	void resetHeadReader() {
		headReader.resetRead();
		debug_printf("FIFOQueue(%s) read cursor reset\n", name.c_str());
	}

	// Fast path extent peekAll (this zooms through the queue reading extents at a time)
	// Output interface is a promise stream and one vector of results per extent found is sent to the promise stream
	// Once we are finished reading all the extents of the queue, end_of_stream() is sent to mark completion
																#line 2184 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via peekAll_ext()
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class PeekAll_extActor>
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PeekAll_extActorState {
															#line 2190 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PeekAll_extActorState(FIFOQueue* const& self,PromiseStream<Standalone<VectorRef<T>>> const& res) 
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   res(res),
															#line 972 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   c()
															#line 2201 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("peekAll_ext", reinterpret_cast<unsigned long>(this));

	}
	~PeekAll_extActorState() 
	{
		fdb_probe_actor_destroy("peekAll_ext", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 973 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			c.initReadOnly(self->headReader, true);
															#line 975 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) peekAllExt begin\n", c.toString().c_str());
															#line 976 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (c.pageID == invalidPhysicalPageID || c.pageID == c.endPageID)
															#line 2220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 977 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("FIFOQueue::Cursor(%s) peekAllExt returning nothing\n", c.toString().c_str());
															#line 978 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				res.sendError(end_of_stream());
															#line 979 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!static_cast<PeekAll_extActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PeekAll_extActorState(); static_cast<PeekAll_extActor*>(this)->destroy(); return 0; }
															#line 2228 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				new (&static_cast<PeekAll_extActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PeekAll_extActorState();
				static_cast<PeekAll_extActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 982 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			entriesRead = 0;
															#line 984 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 2238 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PeekAll_extActorState();
		static_cast<PeekAll_extActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 986 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!c.page)
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 987 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) peekAllExt going to Load Extent %s.\n", c.toString().c_str(), ::toString(c.pageID).c_str());
															#line 990 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = c.loadExtent();
															#line 990 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<PeekAll_extActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2276 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PeekAll_extActor*>(this)->actor_wait_state = 1;
															#line 990 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PeekAll_extActor, 0, Void >*>(static_cast<PeekAll_extActor*>(this)));
															#line 2281 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 994 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		results = Standalone<VectorRef<T>>();
															#line 995 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		results.reserve(results.arena(), self->pagesPerExtent * self->pager->getPhysicalPageSize() / sizeof(T));
															#line 998 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		pageIdx = 0;
															#line 999 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		;
															#line 2301 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = yield();
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<PeekAll_extActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2312 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PeekAll_extActor*>(this)->actor_wait_state = 2;
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PeekAll_extActor, 1, Void >*>(static_cast<PeekAll_extActor*>(this)));
															#line 2317 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = yield();
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<PeekAll_extActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2328 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PeekAll_extActor*>(this)->actor_wait_state = 2;
															#line 991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PeekAll_extActor, 1, Void >*>(static_cast<PeekAll_extActor*>(this)));
															#line 2333 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PeekAll_extActor*>(this)->actor_wait_state > 0) static_cast<PeekAll_extActor*>(this)->actor_wait_state = 0;
		static_cast<PeekAll_extActor*>(this)->ActorCallback< PeekAll_extActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PeekAll_extActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PeekAll_extActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PeekAll_extActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PeekAll_extActor*>(this)->actor_wait_state > 0) static_cast<PeekAll_extActor*>(this)->actor_wait_state = 0;
		static_cast<PeekAll_extActor*>(this)->ActorCallback< PeekAll_extActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PeekAll_extActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PeekAll_extActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PeekAll_extActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1001 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page = c.page->getSubPage(pageIdx++ * self->pager->getPhysicalPageSize(), self->pager->getLogicalPageSize());
															#line 1003 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("FIFOQueue::Cursor(%s) peekALLExt %s. Offset %d\n", c.toString().c_str(), toString(c.pageID).c_str(), c.pageID * self->pager->getPhysicalPageSize());
															#line 2519 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 1009 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadHeader(c.pageID);
															#line 1011 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadPayload(c.pageID);
															#line 2525 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1041 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		const QueuePage* p = (const QueuePage*)(page->data());
															#line 1042 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		int bytesRead;
															#line 1045 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for(;;) {
															#line 1046 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(c.offset < p->endOffset);
															#line 1047 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			T result = Codec::readFromBytes(p->begin() + c.offset, bytesRead);
															#line 1048 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf( "FIFOQueue::Cursor(%s) after read of %s\n", c.toString().c_str(), ::toString(result).c_str());
															#line 1050 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			results.push_back(results.arena(), result);
															#line 1051 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			entriesRead++;
															#line 1053 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			c.offset += bytesRead;
															#line 1054 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(c.offset <= p->endOffset);
															#line 1056 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (c.offset == p->endOffset)
															#line 2573 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1057 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				c.pageID = p->nextPageID;
															#line 1058 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				c.offset = p->nextOffset;
															#line 1059 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("FIFOQueue::Cursor(%s) peekAllExt page exhausted, moved to new page\n", c.toString().c_str());
															#line 1061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("FIFOQueue:: nextPageID=%s, extentCurPageID=%s, extentEndPageID=%s\n", ::toString(p->nextPageID).c_str(), ::toString(p->extentCurPageID).c_str(), ::toString(p->extentEndPageID).c_str());
															#line 2583 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				break;
			}
		}
															#line 1070 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (c.pageID == invalidPhysicalPageID || c.pageID == c.endPageID)
															#line 2589 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1071 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) peekAllExt Queue exhausted\n", c.toString().c_str());
															#line 1072 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			res.send(results);
															#line 1076 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (entriesRead != self->numEntries)
															#line 2597 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1077 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				Error e = internal_error();
															#line 1078 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				TraceEvent(SevError, "RedwoodQueueNumEntriesMisMatch") .error(e) .detail("EntriesRead", entriesRead) .detail("ExpectedEntries", self->numEntries);
															#line 1082 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 2605 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 1084 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			res.sendError(end_of_stream());
															#line 1085 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<PeekAll_extActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PeekAll_extActorState(); static_cast<PeekAll_extActor*>(this)->destroy(); return 0; }
															#line 2611 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<PeekAll_extActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PeekAll_extActorState();
			static_cast<PeekAll_extActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1089 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (p->extentCurPageID == p->extentEndPageID)
															#line 2619 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			c.page.clear();
															#line 1091 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) peekAllExt extent exhausted, moved to new extent\n", c.toString().c_str());
															#line 1095 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			res.send(results);
															#line 1096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->pager->releaseExtentReadLock();
															#line 2629 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1013 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			bool isInjected = false;
															#line 1014 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated())
															#line 2643 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1015 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				auto num4kBlocks = std::max(self->pager->getPhysicalPageSize() / 4096, 1);
															#line 1016 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				auto startBlock = (c.pageID * self->pager->getPhysicalPageSize()) / 4096;
															#line 1017 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				auto iter = g_simulator->corruptedBlocks.lower_bound( std::make_pair(self->pager->getName(), startBlock));
															#line 1019 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (iter->first == self->pager->getName() && iter->second < startBlock + num4kBlocks)
															#line 2653 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 1020 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					isInjected = true;
															#line 2657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 1023 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(isInjected ? SevWarnAlways : SevError, "RedwoodChecksumFailed") .error(e) .detail("PageID", c.pageID) .detail("PageSize", self->pager->getPhysicalPageSize()) .detail("Offset", c.pageID * self->pager->getPhysicalPageSize()) .detail("Filename", self->pager->getName());
															#line 1030 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue::Cursor(%s) peekALLExt getSubPage error=%s for %s. Offset %d ", c.toString().c_str(), e.what(), toString(c.pageID).c_str(), c.pageID * self->pager->getPhysicalPageSize());
															#line 1035 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (isInjected)
															#line 2666 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1036 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e.asInjectedFault(), std::max(0, loopDepth - 2));
															#line 2670 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 1038 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 2674 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FIFOQueue* self;
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PromiseStream<Standalone<VectorRef<T>>> res;
															#line 972 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Cursor c;
															#line 982 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int entriesRead;
															#line 994 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Standalone<VectorRef<T>> results;
															#line 998 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int pageIdx;
															#line 1001 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> page;
															#line 2711 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via peekAll_ext()
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PeekAll_extActor final : public Actor<Void>, public ActorCallback< PeekAll_extActor, 0, Void >, public ActorCallback< PeekAll_extActor, 1, Void >, public FastAllocated<PeekAll_extActor>, public PeekAll_extActorState<PeekAll_extActor> {
															#line 2716 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<PeekAll_extActor>::operator new;
	using FastAllocated<PeekAll_extActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1113358699680898560UL, 915780362318052864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PeekAll_extActor, 0, Void >;
friend struct ActorCallback< PeekAll_extActor, 1, Void >;
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PeekAll_extActor(FIFOQueue* const& self,PromiseStream<Standalone<VectorRef<T>>> const& res) 
															#line 2734 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   PeekAll_extActorState<PeekAll_extActor>(self, res),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("peekAll_ext", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14361256801945444864UL, 15986249875160335616UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_extActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("peekAll_ext");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("peekAll_ext", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PeekAll_extActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PeekAll_extActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> peekAll_ext( FIFOQueue* const& self, PromiseStream<Standalone<VectorRef<T>>> const& res ) {
															#line 971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new PeekAll_extActor(self, res));
															#line 2767 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1102 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> peekAllExt(PromiseStream<Standalone<VectorRef<T>>> resStream) { return peekAll_ext(this, resStream); }

																#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via peekAll_impl()
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class PeekAll_implActor>
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PeekAll_implActorState {
															#line 2780 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PeekAll_implActorState(FIFOQueue* const& self) 
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 1106 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   results(),
															#line 1107 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   c()
															#line 2791 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("peekAll_impl", reinterpret_cast<unsigned long>(this));

	}
	~PeekAll_implActorState() 
	{
		fdb_probe_actor_destroy("peekAll_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1108 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			c.initReadOnly(self->headReader);
															#line 1109 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			results.reserve(results.arena(), self->numEntries);
															#line 1111 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			sinceYield = 0;
															#line 1112 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 2812 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PeekAll_implActorState();
		static_cast<PeekAll_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1126 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<PeekAll_implActor*>(this)->SAV<Standalone<VectorRef<T>>>::futures) { (void)(results); this->~PeekAll_implActorState(); static_cast<PeekAll_implActor*>(this)->destroy(); return 0; }
															#line 2835 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<PeekAll_implActor*>(this)->SAV< Standalone<VectorRef<T>> >::value()) Standalone<VectorRef<T>>(std::move(results)); // state_var_RVO
		this->~PeekAll_implActorState();
		static_cast<PeekAll_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1113 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<T>> __when_expr_0 = c.readNext();
															#line 1113 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<PeekAll_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2856 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<PeekAll_implActor*>(this)->actor_wait_state = 1;
															#line 1113 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PeekAll_implActor, 0, Optional<T> >*>(static_cast<PeekAll_implActor*>(this)));
															#line 2861 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Optional<T> const& x,int loopDepth) 
	{
															#line 1114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!x.present())
															#line 2883 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		results.push_back(results.arena(), x.get());
															#line 1120 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (++sinceYield >= 100)
															#line 2891 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			sinceYield = 0;
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = yield();
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<PeekAll_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2899 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PeekAll_implActor*>(this)->actor_wait_state = 2;
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PeekAll_implActor, 1, Void >*>(static_cast<PeekAll_implActor*>(this)));
															#line 2904 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Optional<T> && x,int loopDepth) 
	{
															#line 1114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!x.present())
															#line 2918 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		results.push_back(results.arena(), x.get());
															#line 1120 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (++sinceYield >= 100)
															#line 2926 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			sinceYield = 0;
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = yield();
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<PeekAll_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2934 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PeekAll_implActor*>(this)->actor_wait_state = 2;
															#line 1122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PeekAll_implActor, 1, Void >*>(static_cast<PeekAll_implActor*>(this)));
															#line 2939 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<T> const& x,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(x, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<T> && x,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(x), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PeekAll_implActor*>(this)->actor_wait_state > 0) static_cast<PeekAll_implActor*>(this)->actor_wait_state = 0;
		static_cast<PeekAll_implActor*>(this)->ActorCallback< PeekAll_implActor, 0, Optional<T> >::remove();

	}
	void a_callback_fire(ActorCallback< PeekAll_implActor, 0, Optional<T> >*,Optional<T> const& value) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PeekAll_implActor, 0, Optional<T> >*,Optional<T> && value) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PeekAll_implActor, 0, Optional<T> >*,Error err) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PeekAll_implActor*>(this)->actor_wait_state > 0) static_cast<PeekAll_implActor*>(this)->actor_wait_state = 0;
		static_cast<PeekAll_implActor*>(this)->ActorCallback< PeekAll_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PeekAll_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PeekAll_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PeekAll_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FIFOQueue* self;
															#line 1106 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Standalone<VectorRef<T>> results;
															#line 1107 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Cursor c;
															#line 1111 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int sinceYield;
															#line 3125 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via peekAll_impl()
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PeekAll_implActor final : public Actor<Standalone<VectorRef<T>>>, public ActorCallback< PeekAll_implActor, 0, Optional<T> >, public ActorCallback< PeekAll_implActor, 1, Void >, public FastAllocated<PeekAll_implActor>, public PeekAll_implActorState<PeekAll_implActor> {
															#line 3130 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<PeekAll_implActor>::operator new;
	using FastAllocated<PeekAll_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12144025211552549888UL, 2917922391745133824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<VectorRef<T>>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PeekAll_implActor, 0, Optional<T> >;
friend struct ActorCallback< PeekAll_implActor, 1, Void >;
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PeekAll_implActor(FIFOQueue* const& self) 
															#line 3148 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Standalone<VectorRef<T>>>(),
		   PeekAll_implActorState<PeekAll_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("peekAll_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18004675872288453376UL, 15475843600805029376UL);
		ActorExecutionContextHelper __helper(static_cast<PeekAll_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("peekAll_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("peekAll_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PeekAll_implActor, 0, Optional<T> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PeekAll_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Standalone<VectorRef<T>>> peekAll_impl( FIFOQueue* const& self ) {
															#line 1105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Standalone<VectorRef<T>>>(new PeekAll_implActor(self));
															#line 3181 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1128 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Standalone<VectorRef<T>>> peekAll() { return peekAll_impl(this); }

																#line 3188 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via peek_impl()
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Peek_implActor>
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Peek_implActorState {
															#line 3194 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Peek_implActorState(FIFOQueue* const& self) 
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 1132 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   c()
															#line 3203 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("peek_impl", reinterpret_cast<unsigned long>(this));

	}
	~Peek_implActorState() 
	{
		fdb_probe_actor_destroy("peek_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			c.initReadOnly(self->headReader);
															#line 1134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Optional<T>> __when_expr_0 = c.readNext();
															#line 1134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Peek_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3222 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Peek_implActor*>(this)->actor_wait_state = 1;
															#line 1134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Peek_implActor, 0, Optional<T> >*>(static_cast<Peek_implActor*>(this)));
															#line 3227 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Peek_implActorState();
		static_cast<Peek_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<T> const& x,int loopDepth) 
	{
															#line 1135 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Peek_implActor*>(this)->SAV<Optional<T>>::futures) { (void)(x); this->~Peek_implActorState(); static_cast<Peek_implActor*>(this)->destroy(); return 0; }
															#line 3250 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Peek_implActor*>(this)->SAV< Optional<T> >::value()) Optional<T>(x);
		this->~Peek_implActorState();
		static_cast<Peek_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<T> && x,int loopDepth) 
	{
															#line 1135 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Peek_implActor*>(this)->SAV<Optional<T>>::futures) { (void)(x); this->~Peek_implActorState(); static_cast<Peek_implActor*>(this)->destroy(); return 0; }
															#line 3262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Peek_implActor*>(this)->SAV< Optional<T> >::value()) Optional<T>(x);
		this->~Peek_implActorState();
		static_cast<Peek_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<T> const& x,int loopDepth) 
	{
		loopDepth = a_body1cont1(x, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<T> && x,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(x), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Peek_implActor*>(this)->actor_wait_state > 0) static_cast<Peek_implActor*>(this)->actor_wait_state = 0;
		static_cast<Peek_implActor*>(this)->ActorCallback< Peek_implActor, 0, Optional<T> >::remove();

	}
	void a_callback_fire(ActorCallback< Peek_implActor, 0, Optional<T> >*,Optional<T> const& value) 
	{
		fdb_probe_actor_enter("peek_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Peek_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peek_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Peek_implActor, 0, Optional<T> >*,Optional<T> && value) 
	{
		fdb_probe_actor_enter("peek_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Peek_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peek_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Peek_implActor, 0, Optional<T> >*,Error err) 
	{
		fdb_probe_actor_enter("peek_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Peek_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("peek_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FIFOQueue* self;
															#line 1132 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Cursor c;
															#line 3349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via peek_impl()
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Peek_implActor final : public Actor<Optional<T>>, public ActorCallback< Peek_implActor, 0, Optional<T> >, public FastAllocated<Peek_implActor>, public Peek_implActorState<Peek_implActor> {
															#line 3354 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Peek_implActor>::operator new;
	using FastAllocated<Peek_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11164211004292278272UL, 2606363331860767232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<T>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Peek_implActor, 0, Optional<T> >;
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Peek_implActor(FIFOQueue* const& self) 
															#line 3371 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Optional<T>>(),
		   Peek_implActorState<Peek_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("peek_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8248347584181944320UL, 4593983314153380864UL);
		ActorExecutionContextHelper __helper(static_cast<Peek_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("peek_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("peek_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Peek_implActor, 0, Optional<T> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Optional<T>> peek_impl( FIFOQueue* const& self ) {
															#line 1131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Optional<T>>(new Peek_implActor(self));
															#line 3403 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1137 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Optional<T>> peek() { return peek_impl(this); }

	// Pop the next item on front of queue if it is <= inclusiveMaximum or if inclusiveMaximum is not present
	Future<Optional<T>> pop(Optional<T> inclusiveMaximum = {}) { return headReader.readNext(inclusiveMaximum); }

	QueueState getState() const {
		QueueState s;
		s.queueID = queueID;
		s.headOffset = headReader.offset;
		s.headPageID = headReader.pageID;
		s.tailPageID = tailWriter.pageID;
		s.numEntries = numEntries;
		s.numPages = numPages;
		s.usesExtents = usesExtents;
		s.tailPageNewExtent = tailPageNewExtent;
		s.prevExtentEndPageID = prevExtentEndPageID;

		debug_printf("FIFOQueue(%s) getState(): %s\n", name.c_str(), s.toString().c_str());
		return s;
	}

	void pushBack(const T& item) {
		debug_printf("FIFOQueue(%s) pushBack(%s)\n", name.c_str(), toString(item).c_str());
		tailWriter.write(item);
	}

	void pushFront(const T& item) {
		debug_printf("FIFOQueue(%s) pushFront(%s)\n", name.c_str(), toString(item).c_str());
		headWriter.write(item);
	}

	bool isBusy() const {
		return headWriter.isBusy() || headReader.isBusy() || tailWriter.isBusy() || !newTailPage.isReady();
	}

	// Wait until all previously started operations on each cursor are done and the new tail page is ready
	Future<Void> notBusy() {
		auto f = headWriter.notBusy() && headReader.notBusy() && tailWriter.notBusy() && ready(newTailPage);
		debug_printf("FIFOQueue(%s) notBusy future ready=%d\n", name.c_str(), f.isReady());
		return f;
	}

	// preFlush() prepares this queue to be flushed to disk, but doesn't actually do it so the queue can still
	// be pushed and popped after this operation. It returns whether or not any operations were pending or
	// started during execution.
	//
	// If one or more queues are used by their pager in newPageID() or freePage() operations, then preFlush()
	// must be called on each of them inside a loop that runs until each of the preFlush() calls have returned
	// false twice in a row.
	//
	// The reason for all this is that:
	//   - queue pop() can call pager->freePage() which can call push() on the same or another queue
	//   - queue push() can call pager->newPageID() which can call pop() on the same or another queue
	// This creates a circular dependency with 1 or more queues when those queues are used by the pager
	// to manage free page IDs.
																#line 3463 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via preFlush_impl()
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class PreFlush_implActor>
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PreFlush_implActorState {
															#line 3469 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PreFlush_implActorState(FIFOQueue* const& self) 
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 3476 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("preFlush_impl", reinterpret_cast<unsigned long>(this));

	}
	~PreFlush_implActorState() 
	{
		fdb_probe_actor_destroy("preFlush_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1194 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("FIFOQueue(%s) preFlush begin\n", self->name.c_str());
															#line 1195 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->notBusy();
															#line 1195 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<PreFlush_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3495 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PreFlush_implActor*>(this)->actor_wait_state = 1;
															#line 1195 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PreFlush_implActor, 0, Void >*>(static_cast<PreFlush_implActor*>(this)));
															#line 3500 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PreFlush_implActorState();
		static_cast<PreFlush_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1199 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		bool workPending = self->isBusy();
															#line 1201 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!workPending)
															#line 3525 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1211 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->newTailPage.isReady() && self->newTailPage.get() == invalidPhysicalPageID)
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1212 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->tailWriter.pendingTailWrites())
															#line 3533 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 1213 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("FIFOQueue(%s) preFlush starting to get new page ID\n", self->name.c_str());
															#line 1214 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (self->usesExtents)
															#line 3539 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					{
															#line 1215 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						if (self->tailWriter.pageID == invalidPhysicalPageID)
															#line 3543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
						{
															#line 1216 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->newTailPage = self->pager->newExtentPageID(self->queueID);
															#line 1217 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->tailPageNewExtent = true;
															#line 1218 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->prevExtentEndPageID = invalidPhysicalPageID;
															#line 3551 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
						}
						else
						{
															#line 1220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							auto p = self->tailWriter.header();
															#line 1221 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							debug_printf( "FIFOQueue(%s) newTailPage tailWriterPage %u extentCurPageID %u, extentEndPageID %u\n", self->name.c_str(), self->tailWriter.pageID, p->extentCurPageID, p->extentEndPageID);
															#line 1227 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							if (p->extentCurPageID < p->extentEndPageID)
															#line 3561 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							{
															#line 1228 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->newTailPage = p->extentCurPageID + 1;
															#line 1229 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->tailPageNewExtent = false;
															#line 1230 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->prevExtentEndPageID = p->extentEndPageID;
															#line 3569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							}
							else
							{
															#line 1232 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->newTailPage = self->pager->newExtentPageID(self->queueID);
															#line 1233 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->tailPageNewExtent = true;
															#line 1234 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->prevExtentEndPageID = invalidPhysicalPageID;
															#line 3579 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							}
						}
															#line 1237 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						debug_printf("FIFOQueue(%s) newTailPage tailPageNewExtent:%d prevExtentEndPageID: %u " "tailWriterPage %u\n", self->name.c_str(), self->tailPageNewExtent, self->prevExtentEndPageID, self->tailWriter.pageID);
															#line 3584 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
					else
					{
															#line 1244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->newTailPage = self->pager->newPageID();
															#line 3590 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
															#line 1245 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					workPending = true;
															#line 3594 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 1247 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (self->usesExtents)
															#line 3600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					{
															#line 1248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						auto p = self->tailWriter.header();
															#line 1249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->prevExtentEndPageID = p->extentEndPageID;
															#line 1250 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->tailPageNewExtent = false;
															#line 1251 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						debug_printf("FIFOQueue(%s) newTailPage tailPageNewExtent: %d prevExtentEndPageID: %u " "tailWriterPage %u\n", self->name.c_str(), self->tailPageNewExtent, self->prevExtentEndPageID, self->tailWriter.pageID);
															#line 3610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
				}
			}
		}
															#line 1262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("FIFOQueue(%s) preFlush returning %d\n", self->name.c_str(), workPending);
															#line 1263 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<PreFlush_implActor*>(this)->SAV<bool>::futures) { (void)(workPending); this->~PreFlush_implActorState(); static_cast<PreFlush_implActor*>(this)->destroy(); return 0; }
															#line 3619 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<PreFlush_implActor*>(this)->SAV< bool >::value()) bool(workPending);
		this->~PreFlush_implActorState();
		static_cast<PreFlush_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1199 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		bool workPending = self->isBusy();
															#line 1201 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!workPending)
															#line 3633 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 1211 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->newTailPage.isReady() && self->newTailPage.get() == invalidPhysicalPageID)
															#line 3637 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1212 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->tailWriter.pendingTailWrites())
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 1213 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("FIFOQueue(%s) preFlush starting to get new page ID\n", self->name.c_str());
															#line 1214 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (self->usesExtents)
															#line 3647 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					{
															#line 1215 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						if (self->tailWriter.pageID == invalidPhysicalPageID)
															#line 3651 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
						{
															#line 1216 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->newTailPage = self->pager->newExtentPageID(self->queueID);
															#line 1217 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->tailPageNewExtent = true;
															#line 1218 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							self->prevExtentEndPageID = invalidPhysicalPageID;
															#line 3659 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
						}
						else
						{
															#line 1220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							auto p = self->tailWriter.header();
															#line 1221 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							debug_printf( "FIFOQueue(%s) newTailPage tailWriterPage %u extentCurPageID %u, extentEndPageID %u\n", self->name.c_str(), self->tailWriter.pageID, p->extentCurPageID, p->extentEndPageID);
															#line 1227 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
							if (p->extentCurPageID < p->extentEndPageID)
															#line 3669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							{
															#line 1228 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->newTailPage = p->extentCurPageID + 1;
															#line 1229 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->tailPageNewExtent = false;
															#line 1230 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->prevExtentEndPageID = p->extentEndPageID;
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							}
							else
							{
															#line 1232 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->newTailPage = self->pager->newExtentPageID(self->queueID);
															#line 1233 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->tailPageNewExtent = true;
															#line 1234 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
								self->prevExtentEndPageID = invalidPhysicalPageID;
															#line 3687 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
							}
						}
															#line 1237 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						debug_printf("FIFOQueue(%s) newTailPage tailPageNewExtent:%d prevExtentEndPageID: %u " "tailWriterPage %u\n", self->name.c_str(), self->tailPageNewExtent, self->prevExtentEndPageID, self->tailWriter.pageID);
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
					else
					{
															#line 1244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->newTailPage = self->pager->newPageID();
															#line 3698 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
															#line 1245 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					workPending = true;
															#line 3702 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 1247 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (self->usesExtents)
															#line 3708 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					{
															#line 1248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						auto p = self->tailWriter.header();
															#line 1249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->prevExtentEndPageID = p->extentEndPageID;
															#line 1250 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						self->tailPageNewExtent = false;
															#line 1251 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
						debug_printf("FIFOQueue(%s) newTailPage tailPageNewExtent: %d prevExtentEndPageID: %u " "tailWriterPage %u\n", self->name.c_str(), self->tailPageNewExtent, self->prevExtentEndPageID, self->tailWriter.pageID);
															#line 3718 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					}
				}
			}
		}
															#line 1262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("FIFOQueue(%s) preFlush returning %d\n", self->name.c_str(), workPending);
															#line 1263 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<PreFlush_implActor*>(this)->SAV<bool>::futures) { (void)(workPending); this->~PreFlush_implActorState(); static_cast<PreFlush_implActor*>(this)->destroy(); return 0; }
															#line 3727 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<PreFlush_implActor*>(this)->SAV< bool >::value()) bool(workPending);
		this->~PreFlush_implActorState();
		static_cast<PreFlush_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PreFlush_implActor*>(this)->actor_wait_state > 0) static_cast<PreFlush_implActor*>(this)->actor_wait_state = 0;
		static_cast<PreFlush_implActor*>(this)->ActorCallback< PreFlush_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PreFlush_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PreFlush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PreFlush_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PreFlush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PreFlush_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PreFlush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("preFlush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FIFOQueue* self;
															#line 3812 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via preFlush_impl()
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class PreFlush_implActor final : public Actor<bool>, public ActorCallback< PreFlush_implActor, 0, Void >, public FastAllocated<PreFlush_implActor>, public PreFlush_implActorState<PreFlush_implActor> {
															#line 3817 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<PreFlush_implActor>::operator new;
	using FastAllocated<PreFlush_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7147130377842055680UL, 7632554017694741760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PreFlush_implActor, 0, Void >;
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PreFlush_implActor(FIFOQueue* const& self) 
															#line 3834 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<bool>(),
		   PreFlush_implActorState<PreFlush_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("preFlush_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15778326281340990720UL, 171956110384537600UL);
		ActorExecutionContextHelper __helper(static_cast<PreFlush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("preFlush_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("preFlush_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PreFlush_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<bool> preFlush_impl( FIFOQueue* const& self ) {
															#line 1193 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<bool>(new PreFlush_implActor(self));
															#line 3866 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1265 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<bool> preFlush() { return preFlush_impl(this); }

	void finishFlush() {
		debug_printf("FIFOQueue(%s) finishFlush start\n", name.c_str());
		ASSERT(!isBusy());
		bool initTailWriter = true;

		// If a new tail page was allocated, link the last page of the tail writer to it.
		if (newTailPage.get() != invalidPhysicalPageID) {
			tailWriter.addNewPage(newTailPage.get(), 0, false, false);
			// The flush sequence allocated a page and added it to the queue so increment numPages
			++numPages;

			// newPage() should be ready immediately since a pageID is being explicitly passed.
			ASSERT(!tailWriter.isBusy());

			newTailPage = invalidPhysicalPageID;
			initTailWriter = true;
		}

		// If the headWriter wrote anything, link its tail page to the headReader position and point the headReader
		// to the start of the headWriter
		if (headWriter.pendingTailWrites()) {
			headWriter.addNewPage(headReader.pageID, headReader.offset, false);
			headReader.pageID = headWriter.firstPageIDWritten;
			headReader.offset = 0;
			headReader.page.clear();
		}

		// Update headReader's end page to the new tail page
		headReader.endPageID = tailWriter.pageID;

		// Reset the write cursors
		debug_printf("FIFOQueue(%s) Reset tailWriter cursor. tailPageNewExtent: %d\n", name.c_str(), tailPageNewExtent);
		tailWriter.init(this,
		                Cursor::WRITE,
		                tailWriter.pageID,
		                initTailWriter /*false*/,
		                tailPageNewExtent,
		                invalidPhysicalPageID,
		                0,
		                prevExtentEndPageID);
		headWriter.init(this, Cursor::WRITE);

		debug_printf("FIFOQueue(%s) finishFlush end\n", name.c_str());
	}

																#line 3918 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via flush_impl()
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Flush_implActor>
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Flush_implActorState {
															#line 3924 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Flush_implActorState(FIFOQueue* const& self) 
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 3931 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("flush_impl", reinterpret_cast<unsigned long>(this));

	}
	~Flush_implActorState() 
	{
		fdb_probe_actor_destroy("flush_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1314 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 3946 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Flush_implActorState();
		static_cast<Flush_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1320 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->finishFlush();
															#line 1321 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Flush_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Flush_implActorState(); static_cast<Flush_implActor*>(this)->destroy(); return 0; }
															#line 3971 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Flush_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Flush_implActorState();
		static_cast<Flush_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<bool> __when_expr_0 = self->preFlush();
															#line 1315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Flush_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3992 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Flush_implActor*>(this)->actor_wait_state = 1;
															#line 1315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Flush_implActor, 0, bool >*>(static_cast<Flush_implActor*>(this)));
															#line 3997 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(bool const& notDone,int loopDepth) 
	{
															#line 1316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!notDone)
															#line 4019 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(bool && notDone,int loopDepth) 
	{
															#line 1316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!notDone)
															#line 4031 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool const& notDone,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(notDone, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool && notDone,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(notDone), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Flush_implActor*>(this)->actor_wait_state > 0) static_cast<Flush_implActor*>(this)->actor_wait_state = 0;
		static_cast<Flush_implActor*>(this)->ActorCallback< Flush_implActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< Flush_implActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("flush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Flush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Flush_implActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("flush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Flush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Flush_implActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("flush_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Flush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flush_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FIFOQueue* self;
															#line 4116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via flush_impl()
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Flush_implActor final : public Actor<Void>, public ActorCallback< Flush_implActor, 0, bool >, public FastAllocated<Flush_implActor>, public Flush_implActorState<Flush_implActor> {
															#line 4121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Flush_implActor>::operator new;
	using FastAllocated<Flush_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13520381717210287616UL, 14152736980014978048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Flush_implActor, 0, bool >;
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Flush_implActor(FIFOQueue* const& self) 
															#line 4138 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   Flush_implActorState<Flush_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flush_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15169802442430479872UL, 15109318566279382528UL);
		ActorExecutionContextHelper __helper(static_cast<Flush_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flush_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flush_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Flush_implActor, 0, bool >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> flush_impl( FIFOQueue* const& self ) {
															#line 1313 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new Flush_implActor(self));
															#line 4170 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1323 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> flush() { return flush_impl(this); }

	IPager2* pager;
	QueueID queueID;
	Future<Void> pagerError;

	int64_t numPages;
	int64_t numEntries;
	int pagesPerExtent;
	bool usesExtents;
	bool tailPageNewExtent;
	PhysicalPageID prevExtentEndPageID;

	Cursor headReader;
	Cursor tailWriter;
	Cursor headWriter;

	Future<PhysicalPageID> newTailPage;

	// For debugging
	std::string name;

	void toTraceEvent(TraceEvent& e, const char* prefix) const {
		e.detail(format("%sRecords", prefix), numEntries);
		e.detail(format("%sPages", prefix), numPages);
		e.detail(format("%sRecordsPerPage", prefix), numPages > 0 ? (double)numEntries / numPages : 0);
	}
};

int nextPowerOf2(uint32_t x) {
	return 1 << (32 - clz(x - 1));
}

struct RedwoodMetrics {
	constexpr static unsigned int btreeLevels = 5;
	static int maxRecordCount;

	struct EventReasonsArray {
		unsigned int eventReasons[(size_t)PagerEvents::MAXEVENTS][(size_t)PagerEventReasons::MAXEVENTREASONS];

		EventReasonsArray() { clear(); }
		void clear() { memset(eventReasons, 0, sizeof(eventReasons)); }

		void addEventReason(PagerEvents event, PagerEventReasons reason) {
			eventReasons[(size_t)event][(size_t)reason] += 1;
		}

		unsigned int getEventReason(PagerEvents event, PagerEventReasons reason) const {
			return eventReasons[(size_t)event][(size_t)reason];
		}

		std::string toString(unsigned int level, double elapsed) const {
			std::string result;

			const auto& pairs = (level == 0 ? L0PossibleEventReasonPairs : possibleEventReasonPairs);
			PagerEvents prevEvent = pairs.front().first;
			std::string lineStart = (level == 0) ? "" : "\t";

			for (const auto& p : pairs) {
				if (p.first != prevEvent) {
					result += "\n";
					result += lineStart;
				}

				std::string name =
				    format("%s%s", PagerEventsStrings[(int)p.first], PagerEventReasonsStrings[(int)p.second]);
				int count = getEventReason(p.first, p.second);
				result += format("%-15s %8u %8u/s  ", name.c_str(), count, int(count / elapsed));

				prevEvent = p.first;
			}

			return result;
		}

		void toTraceEvent(TraceEvent* t, unsigned int level) const {
			const auto& pairs = (level == 0 ? L0PossibleEventReasonPairs : possibleEventReasonPairs);
			for (const auto& p : pairs) {
				std::string name =
				    format(level == 0 ? "" : "L%d", level) +
				    format("%s%s", PagerEventsStrings[(int)p.first], PagerEventReasonsStrings[(int)p.second]);
				int count = getEventReason(p.first, p.second);
				t->detail(std::move(name), count);
			}
		}
	};

	// Metrics by level
	struct Level {
		struct Counters {
			unsigned int pageRead;
			unsigned int pageReadExt;
			unsigned int pageBuild;
			unsigned int pageBuildExt;
			unsigned int pageCommitStart;
			unsigned int pageModify;
			unsigned int pageModifyExt;
			unsigned int lazyClearRequeue;
			unsigned int lazyClearRequeueExt;
			unsigned int lazyClearFree;
			unsigned int lazyClearFreeExt;
			unsigned int forceUpdate;
			unsigned int detachChild;
			EventReasonsArray events;
		};
		Counters metrics;
		Reference<Histogram> buildFillPctSketch;
		Reference<Histogram> modifyFillPctSketch;
		Reference<Histogram> buildStoredPctSketch;
		Reference<Histogram> modifyStoredPctSketch;
		Reference<Histogram> buildItemCountSketch;
		Reference<Histogram> modifyItemCountSketch;

		Level() { metrics = {}; }

		void clear() { metrics = {}; }
	};

	struct metrics {
		unsigned int opSet;
		unsigned int opSetKeyBytes;
		unsigned int opSetValueBytes;
		unsigned int opClear;
		unsigned int opClearKey;
		unsigned int opCommit;
		unsigned int opGet;
		unsigned int opGetRange;
		unsigned int pagerDiskWrite;
		unsigned int pagerDiskRead;
		unsigned int pagerRemapFree;
		unsigned int pagerRemapCopy;
		unsigned int pagerRemapSkip;
		unsigned int pagerCacheHit;
		unsigned int pagerCacheMiss;
		unsigned int pagerProbeHit;
		unsigned int pagerProbeMiss;
		unsigned int pagerEvictUnhit;
		unsigned int pagerEvictFail;
		unsigned int btreeLeafPreload;
		unsigned int btreeLeafPreloadExt;
		unsigned int readRequestDecryptTimeNS;
	};

	RedwoodMetrics() {
		// All histograms have reset their buckets to 0 in the constructor.
		kvSizeWritten = Reference<Histogram>(
		    new Histogram(Reference<HistogramRegistry>(), "kvSize", "Written", Histogram::Unit::bytes));
		kvSizeReadByGet = Reference<Histogram>(
		    new Histogram(Reference<HistogramRegistry>(), "kvSize", "ReadByGet", Histogram::Unit::bytes));
		kvSizeReadByGetRange = Reference<Histogram>(
		    new Histogram(Reference<HistogramRegistry>(), "kvSize", "ReadByGetRange", Histogram::Unit::bytes));

		ioLock = nullptr;

		// These histograms are used for Btree events, hence level > 0
		unsigned int levelCounter = 0;
		for (RedwoodMetrics::Level& level : levels) {
			if (levelCounter > 0) {
				std::string levelString = "L" + std::to_string(levelCounter);
				level.buildFillPctSketch = Reference<Histogram>(new Histogram(
				    Reference<HistogramRegistry>(), "buildFillPct", levelString, Histogram::Unit::percentageLinear));
				level.modifyFillPctSketch = Reference<Histogram>(new Histogram(
				    Reference<HistogramRegistry>(), "modifyFillPct", levelString, Histogram::Unit::percentageLinear));
				level.buildStoredPctSketch = Reference<Histogram>(new Histogram(
				    Reference<HistogramRegistry>(), "buildStoredPct", levelString, Histogram::Unit::percentageLinear));
				level.modifyStoredPctSketch = Reference<Histogram>(new Histogram(
				    Reference<HistogramRegistry>(), "modifyStoredPct", levelString, Histogram::Unit::percentageLinear));
				level.buildItemCountSketch = Reference<Histogram>(new Histogram(Reference<HistogramRegistry>(),
				                                                                "buildItemCount",
				                                                                levelString,
				                                                                Histogram::Unit::countLinear,
				                                                                0,
				                                                                maxRecordCount));
				level.modifyItemCountSketch = Reference<Histogram>(new Histogram(Reference<HistogramRegistry>(),
				                                                                 "modifyItemCount",
				                                                                 levelString,
				                                                                 Histogram::Unit::countLinear,
				                                                                 0,
				                                                                 maxRecordCount));
			}
			++levelCounter;
		}
		clear();
	}

	void clear() {
		for (RedwoodMetrics::Level& level : levels) {
			level.clear();
		}
		metric = {};
		startTime = g_network ? now() : 0;
	}

	// btree levels and one extra level for non btree level.
	Level levels[btreeLevels + 1];
	metrics metric;
	// pointer to the priority multi lock used in pager
	PriorityMultiLock* ioLock;

	Reference<Histogram> kvSizeWritten;
	Reference<Histogram> kvSizeReadByGet;
	Reference<Histogram> kvSizeReadByGetRange;
	double startTime;

	// Return number of pages read or written, from cache or disk
	unsigned int pageOps() const {
		// All page reads are either a cache hit, probe hit, or a disk read
		return metric.pagerDiskWrite + metric.pagerDiskRead + metric.pagerCacheHit + metric.pagerProbeHit;
	}

	Level& level(unsigned int level) {
		// Valid levels are from 0 - btreeLevels
		// Level 0 is for operations that are not BTree level specific, as many of the metrics are the same
		// Level 0 - btreeLevels correspond to BTree node height, however heights above btreeLevels are combined
		//           into the level at btreeLevels
		return levels[std::min(level, btreeLevels)];
	}

	void updateMaxRecordCount(int maxRecords) {
		if (maxRecordCount != maxRecords) {
			maxRecordCount = maxRecords;
			for (int i = 1; i <= btreeLevels; ++i) {
				auto& level = levels[i];
				level.buildItemCountSketch->updateUpperBound(maxRecordCount);
				level.modifyItemCountSketch->updateUpperBound(maxRecordCount);
			}
		}
	}

	void logHistograms(double elapsed) {
		// All histograms have reset their buckets to 0 after writeToLog.
		kvSizeWritten->writeToLog(elapsed);
		kvSizeReadByGet->writeToLog(elapsed);
		kvSizeReadByGetRange->writeToLog(elapsed);
		unsigned int levelCounter = 0;
		for (RedwoodMetrics::Level& level : levels) {
			if (levelCounter > 0) {
				level.buildFillPctSketch->writeToLog(elapsed);
				level.modifyFillPctSketch->writeToLog(elapsed);
				level.buildStoredPctSketch->writeToLog(elapsed);
				level.modifyStoredPctSketch->writeToLog(elapsed);
				level.buildItemCountSketch->writeToLog(elapsed);
				level.modifyItemCountSketch->writeToLog(elapsed);
			}
			++levelCounter;
		}
	}

	// This will populate a trace event and/or a string with Redwood metrics.
	// The string is a reasonably well formatted page of information
	void getFields(TraceEvent* e, std::string* s = nullptr, bool skipZeroes = false);

	void getIOLockFields(TraceEvent* e, std::string* s = nullptr);

	std::string toString(bool clearAfter) {
		std::string s;
		getFields(nullptr, &s);
		getIOLockFields(nullptr, &s);

		if (clearAfter) {
			clear();
		}

		return s;
	}
};

// Using a global for Redwood metrics because a single process shouldn't normally have multiple storage engines
int RedwoodMetrics::maxRecordCount = 315;
RedwoodMetrics g_redwoodMetrics = {};
Future<Void> g_redwoodMetricsActor;

															#line 4447 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
namespace {
// This generated class is to be used only via redwoodHistogramsLogger()
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class RedwoodHistogramsLoggerActor>
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RedwoodHistogramsLoggerActorState {
															#line 4454 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RedwoodHistogramsLoggerActorState(double const& interval) 
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : interval(interval),
															#line 1597 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   currTime()
															#line 4463 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this));

	}
	~RedwoodHistogramsLoggerActorState() 
	{
		fdb_probe_actor_destroy("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1598 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 4478 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RedwoodHistogramsLoggerActorState();
		static_cast<RedwoodHistogramsLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1599 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		currTime = now();
															#line 1600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(interval);
															#line 1600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RedwoodHistogramsLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4512 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RedwoodHistogramsLoggerActor*>(this)->actor_wait_state = 1;
															#line 1600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >*>(static_cast<RedwoodHistogramsLoggerActor*>(this)));
															#line 4517 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double elapsed = now() - currTime;
															#line 1602 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.logHistograms(elapsed);
															#line 4528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double elapsed = now() - currTime;
															#line 1602 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.logHistograms(elapsed);
															#line 4539 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RedwoodHistogramsLoggerActor*>(this)->actor_wait_state > 0) static_cast<RedwoodHistogramsLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<RedwoodHistogramsLoggerActor*>(this)->ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodHistogramsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodHistogramsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodHistogramsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	double interval;
															#line 1597 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	double currTime;
															#line 4623 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via redwoodHistogramsLogger()
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RedwoodHistogramsLoggerActor final : public Actor<Void>, public ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >, public FastAllocated<RedwoodHistogramsLoggerActor>, public RedwoodHistogramsLoggerActorState<RedwoodHistogramsLoggerActor> {
															#line 4628 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<RedwoodHistogramsLoggerActor>::operator new;
	using FastAllocated<RedwoodHistogramsLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5717994401659501312UL, 4372732754550948608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >;
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RedwoodHistogramsLoggerActor(double const& interval) 
															#line 4645 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   RedwoodHistogramsLoggerActorState<RedwoodHistogramsLoggerActor>(interval),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10691872360246982912UL, 9540893634709859584UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodHistogramsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("redwoodHistogramsLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("redwoodHistogramsLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RedwoodHistogramsLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] Future<Void> redwoodHistogramsLogger( double const& interval ) {
															#line 1596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new RedwoodHistogramsLoggerActor(interval));
															#line 4678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1605 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

															#line 4683 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
namespace {
// This generated class is to be used only via redwoodMetricsLogger()
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class RedwoodMetricsLoggerActor>
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RedwoodMetricsLoggerActorState {
															#line 4690 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RedwoodMetricsLoggerActorState() 
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	{
															#line 4696 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		fdb_probe_actor_create("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this));

	}
	~RedwoodMetricsLoggerActorState() 
	{
		fdb_probe_actor_destroy("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1607 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			g_redwoodMetrics.clear();
															#line 1608 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			loggingFuture = redwoodHistogramsLogger(SERVER_KNOBS->REDWOOD_HISTOGRAM_INTERVAL);
															#line 1609 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 4714 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RedwoodMetricsLoggerActorState();
		static_cast<RedwoodMetricsLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->REDWOOD_METRICS_INTERVAL);
															#line 1610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RedwoodMetricsLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RedwoodMetricsLoggerActor*>(this)->actor_wait_state = 1;
															#line 1610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RedwoodMetricsLoggerActor, 0, Void >*>(static_cast<RedwoodMetricsLoggerActor*>(this)));
															#line 4751 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1612 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		TraceEvent e("RedwoodMetrics");
															#line 1613 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double elapsed = now() - g_redwoodMetrics.startTime;
															#line 1614 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("Elapsed", elapsed);
															#line 1615 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.getFields(&e);
															#line 1616 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.getIOLockFields(&e);
															#line 1617 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.clear();
															#line 4770 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1612 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		TraceEvent e("RedwoodMetrics");
															#line 1613 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double elapsed = now() - g_redwoodMetrics.startTime;
															#line 1614 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("Elapsed", elapsed);
															#line 1615 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.getFields(&e);
															#line 1616 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.getIOLockFields(&e);
															#line 1617 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.clear();
															#line 4789 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RedwoodMetricsLoggerActor*>(this)->actor_wait_state > 0) static_cast<RedwoodMetricsLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<RedwoodMetricsLoggerActor*>(this)->ActorCallback< RedwoodMetricsLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RedwoodMetricsLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodMetricsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RedwoodMetricsLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodMetricsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RedwoodMetricsLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodMetricsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1608 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Future<Void> loggingFuture;
															#line 4871 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via redwoodMetricsLogger()
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RedwoodMetricsLoggerActor final : public Actor<Void>, public ActorCallback< RedwoodMetricsLoggerActor, 0, Void >, public FastAllocated<RedwoodMetricsLoggerActor>, public RedwoodMetricsLoggerActorState<RedwoodMetricsLoggerActor> {
															#line 4876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<RedwoodMetricsLoggerActor>::operator new;
	using FastAllocated<RedwoodMetricsLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8497919599841351168UL, 12372894952746794752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RedwoodMetricsLoggerActor, 0, Void >;
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RedwoodMetricsLoggerActor() 
															#line 4893 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   RedwoodMetricsLoggerActorState<RedwoodMetricsLoggerActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2654850222193163520UL, 14584959921929314816UL);
		ActorExecutionContextHelper __helper(static_cast<RedwoodMetricsLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("redwoodMetricsLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("redwoodMetricsLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RedwoodMetricsLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] Future<Void> redwoodMetricsLogger(  ) {
															#line 1606 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new RedwoodMetricsLoggerActor());
															#line 4926 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1620 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

// Holds an index of recently used objects.
// ObjectType must have these methods
//
//   // Returns true iff the entry can be evicted
//   bool evictable() const;
//
//	 // Ready when object is safe to evict from cache
//   Future<Void> onEvictable() const;
//
//   // Ready when object destruction is safe
//   // Should cancel pending async operations that are safe to cancel when cache is being destroyed
//   Future<Void> cancel() const;
template <class IndexType, class ObjectType>
class ObjectCache : NonCopyable {
	struct Entry;
	typedef std::unordered_map<IndexType, Entry> CacheT;

	struct Entry : public boost::intrusive::list_base_hook<> {
		Entry() : hits(0), size(0) {}
		IndexType index;
		ObjectType item;
		int hits;
		int size;
		bool ownedByEvictor;
		CacheT* pCache;
	};

	typedef boost::intrusive::list<Entry> EvictionOrderT;

public:
	// Object evictor, manages the eviction order for one or more ObjectCaches
	// Not all objects tracked by the Evictor are in its evictionOrder, as ObjectCaches
	// using this Evictor can temporarily remove entries to an external order but they
	// must eventually give them back with moveIn() or remove them with reclaim().
	class Evictor : NonCopyable {
	public:
		Evictor(int64_t sizeLimit = 0) : sizeLimit(sizeLimit) {}

		// Evictors are normally singletons, either one per real process or one per virtual process in simulation
		static Evictor* getEvictor() {
			static Evictor nonSimEvictor;
			static std::map<NetworkAddress, Evictor> simEvictors;

			if (g_network->isSimulated()) {
				return &simEvictors[g_network->getLocalAddress()];
			} else {
				return &nonSimEvictor;
			}
		}

		// Move an entry to a different eviction order, stored outside of the Evictor,
		// but the entry size is still counted against the evictor
		void moveOut(Entry& e, EvictionOrderT& dest) {
			ASSERT(e.ownedByEvictor);
			dest.splice(dest.end(), evictionOrder, EvictionOrderT::s_iterator_to(e));
			e.ownedByEvictor = false;
			++movedOutCount;
		}

		// Move an entry to the back of the eviction order if it is in the eviction order
		void moveToBack(Entry& e) {
			ASSERT(e.ownedByEvictor);
			evictionOrder.splice(evictionOrder.end(), evictionOrder, EvictionOrderT::s_iterator_to(e));
		}

		// Move entire contents of an external eviction order containing entries whose size is part of
		// this Evictor to the front of its eviction order.
		void moveIn(EvictionOrderT& otherOrder) {
			for (auto& e : otherOrder) {
				ASSERT(!e.ownedByEvictor);
				e.ownedByEvictor = true;
				--movedOutCount;
			}
			evictionOrder.splice(evictionOrder.begin(), otherOrder);
		}

		// Add a new item to the back of the eviction order
		void addNew(Entry& e) {
			sizeUsed += e.size;
			evictionOrder.push_back(e);
			e.ownedByEvictor = true;
		}

		// Claim ownership of an entry, removing its size from the current size and removing it
		// from the eviction order if it exists there
		void reclaim(Entry& e) {
			sizeUsed -= e.size;
			// If e is in evictionOrder then remove it
			if (e.ownedByEvictor) {
				evictionOrder.erase(EvictionOrderT::s_iterator_to(e));
				e.ownedByEvictor = false;
			} else {
				// Otherwise, it wasn't so it had to be a movedOut item so decrement the count
				--movedOutCount;
			}
		}

		void trim(int additionalSpaceNeeded = 0) {
			int attemptsLeft = FLOW_KNOBS->MAX_EVICT_ATTEMPTS;
			// While the cache is too big, evict the oldest entry until the oldest entry can't be evicted.
			while (attemptsLeft-- > 0 && sizeUsed > (sizeLimit - reservedSize - additionalSpaceNeeded) &&
			       !evictionOrder.empty()) {
				Entry& toEvict = evictionOrder.front();

				debug_printf("Evictor count=%d sizeUsed=%" PRId64 " sizeLimit=%" PRId64 " sizePenalty=%" PRId64
				             " needed=%d  Trying to evict %s evictable %d\n",
				             (int)evictionOrder.size(),
				             sizeUsed,
				             sizeLimit,
				             reservedSize,
				             additionalSpaceNeeded,
				             ::toString(toEvict.index).c_str(),
				             toEvict.item.evictable());

				if (!toEvict.item.evictable()) {
					// shift the front to the back
					evictionOrder.shift_forward(1);
					++g_redwoodMetrics.metric.pagerEvictFail;
					break;
				} else {
					if (toEvict.hits == 0) {
						++g_redwoodMetrics.metric.pagerEvictUnhit;
					}
					sizeUsed -= toEvict.size;
					debug_printf("Evicting %s\n", ::toString(toEvict.index).c_str());
					evictionOrder.pop_front();
					toEvict.pCache->erase(toEvict.index);
				}
			}
		}

		int64_t getCountUsed() const { return evictionOrder.size() + movedOutCount; }
		int64_t getCountMoved() const { return movedOutCount; }
		int64_t getSizeUsed() const { return sizeUsed + reservedSize; }

		// Only to be used in tests at a point where all ObjectCache instances should be destroyed.
		bool empty() const { return reservedSize == 0 && sizeUsed == 0 && getCountUsed() == 0; }

		std::string toString() const {
			std::string s = format("Evictor {sizeLimit=%" PRId64 " sizeUsed=%" PRId64 " countUsed=%" PRId64
			                       " sizePenalty=%" PRId64 " movedOutCount=%" PRId64,
			                       sizeLimit,
			                       sizeUsed,
			                       getCountUsed(),
			                       reservedSize,
			                       movedOutCount);
			for (auto& entry : evictionOrder) {
				s += format("\n\tindex %s  size %d  evictable %d\n",
				            ::toString(entry.index).c_str(),
				            entry.size,
				            entry.item.evictable());
			}
			s += "}\n";
			return s;
		}

		// Any external data structures whose memory usage should be counted as part of the object cache
		// budget should add their usage to this total and keep it updated.
		int64_t reservedSize = 0;
		int64_t sizeLimit;

	private:
		EvictionOrderT evictionOrder;
		// Size of all entries in the eviction order or held in external eviction orders
		int64_t sizeUsed = 0;
		// Number of items that have been moveOut()'d to other evictionOrders and aren't back yet
		int64_t movedOutCount = 0;
	};

	ObjectCache(Evictor* evictor = nullptr) : pEvictor(evictor) {
		if (pEvictor == nullptr) {
			pEvictor = Evictor::getEvictor();
		}
	}

	Evictor& evictor() const { return *pEvictor; }

	int64_t getCount() const { return cache.size(); }

	void reserveCount(int count) { cache.reserve(count); }

	// Get the object for i if it exists, else return nullptr.
	// If the object exists, its eviction order will NOT change as this is not a cache hit.
	ObjectType* getIfExists(const IndexType& index) {
		auto i = cache.find(index);
		if (i != cache.end()) {
			++i->second.hits;
			return &i->second.item;
		}
		return nullptr;
	}

	// If index is in cache and not on the prioritized eviction order list, move it there.
	void prioritizeEviction(const IndexType& index) {
		auto i = cache.find(index);
		if (i != cache.end() && i->second.ownedByEvictor) {
			pEvictor->moveOut(i->second, prioritizedEvictions);
		}
	}

	// Get the object for i or create a new one.
	// After a get(), the object for i is the last in evictionOrder.
	// If noHit is set, do not consider this access to be cache hit if the object is present
	// If noMiss is set, do not consider this access to be a cache miss if the object is not present
	ObjectType& get(const IndexType& index, int size, bool noHit = false) {
		Entry& entry = cache[index];

		// If entry is linked into an evictionOrder
		if (entry.is_linked()) {
			// If this access is meant to be a hit
			if (!noHit) {
				++entry.hits;
				// If item eviction is not prioritized, move to end of eviction order
				if (entry.ownedByEvictor) {
					pEvictor->moveToBack(entry);
				}
			}
		} else {
			// Otherwise it was a cache miss

			// Finish initializing entry
			entry.index = index;
			entry.pCache = &cache;
			entry.hits = 0;
			entry.size = size;

			pEvictor->trim(entry.size);
			pEvictor->addNew(entry);
		}

		return entry.item;
	}

	// Clears the cache, saving the entries to second cache, then waits for each item to be evictable and evicts it.
																#line 5165 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via clear_impl()
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Clear_implActor>
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Clear_implActorState {
															#line 5171 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Clear_implActorState(ObjectCache* const& self,bool const& waitForSafeEviction) 
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   waitForSafeEviction(waitForSafeEviction)
															#line 5180 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("clear_impl", reinterpret_cast<unsigned long>(this));

	}
	~Clear_implActorState() 
	{
		fdb_probe_actor_destroy("clear_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1857 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for( auto& ie : self->cache ) {
															#line 1858 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->pEvictor->reclaim(ie.second);
															#line 5197 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 1863 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->prioritizedEvictions.clear();
															#line 1865 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			i = self->cache.begin();
															#line 1866 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 5205 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Clear_implActorState();
		static_cast<Clear_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1870 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->cache.clear();
															#line 1872 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Clear_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Clear_implActorState(); static_cast<Clear_implActor*>(this)->destroy(); return 0; }
															#line 5230 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Clear_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Clear_implActorState();
		static_cast<Clear_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1866 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!(i != self->cache.end()))
															#line 5249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1867 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_0 = waitForSafeEviction ? i->second.item.onEvictable() : i->second.item.cancel();
															#line 1867 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Clear_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5257 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Clear_implActor*>(this)->actor_wait_state = 1;
															#line 1867 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Clear_implActor, 0, Void >*>(static_cast<Clear_implActor*>(this)));
															#line 5262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1868 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++i;
															#line 5284 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1868 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++i;
															#line 5293 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Clear_implActor*>(this)->actor_wait_state > 0) static_cast<Clear_implActor*>(this)->actor_wait_state = 0;
		static_cast<Clear_implActor*>(this)->ActorCallback< Clear_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Clear_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Clear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Clear_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Clear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Clear_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Clear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ObjectCache* self;
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool waitForSafeEviction;
															#line 1865 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	typename CacheT::iterator i;
															#line 5379 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via clear_impl()
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Clear_implActor final : public Actor<Void>, public ActorCallback< Clear_implActor, 0, Void >, public FastAllocated<Clear_implActor>, public Clear_implActorState<Clear_implActor> {
															#line 5384 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Clear_implActor>::operator new;
	using FastAllocated<Clear_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17382168425685942272UL, 16477335398858735872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Clear_implActor, 0, Void >;
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Clear_implActor(ObjectCache* const& self,bool const& waitForSafeEviction) 
															#line 5401 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   Clear_implActorState<Clear_implActor>(self, waitForSafeEviction),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clear_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12698163740185442560UL, 17022253173317183488UL);
		ActorExecutionContextHelper __helper(static_cast<Clear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clear_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clear_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Clear_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> clear_impl( ObjectCache* const& self, bool const& waitForSafeEviction ) {
															#line 1855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new Clear_implActor(self, waitForSafeEviction));
															#line 5433 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1874 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> clear(bool waitForSafeEviction = false) { return clear_impl(this, waitForSafeEviction); }

	// Move the prioritized evictions queued to the front of the eviction order
	void flushPrioritizedEvictions() { pEvictor->moveIn(prioritizedEvictions); }

private:
	Evictor* pEvictor;
	CacheT cache;
	EvictionOrderT prioritizedEvictions;
};

															#line 5449 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
namespace {
// This generated class is to be used only via forwardError()
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class T, class ForwardErrorActor>
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ForwardErrorActorState {
															#line 5456 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ForwardErrorActorState(Future<T> const& f,Promise<Void> const& target) 
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : f(f),
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   target(target)
															#line 5465 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("forwardError", reinterpret_cast<unsigned long>(this));

	}
	~ForwardErrorActorState() 
	{
		fdb_probe_actor_destroy("forwardError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1889 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<T> __when_expr_0 = f;
															#line 1889 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<ForwardErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5483 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ForwardErrorActor*>(this)->actor_wait_state = 1;
															#line 1889 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ForwardErrorActor, 0, T >*>(static_cast<ForwardErrorActor*>(this)));
															#line 5488 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ForwardErrorActorState();
		static_cast<ForwardErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1892 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (e.code() != error_code_actor_cancelled && target.canBeSet())
															#line 5518 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 1893 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				target.sendError(e);
															#line 5522 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 1895 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 5526 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(T const& x,int loopDepth) 
	{
															#line 1890 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ForwardErrorActor*>(this)->SAV<T>::futures) { (void)(x); this->~ForwardErrorActorState(); static_cast<ForwardErrorActor*>(this)->destroy(); return 0; }
															#line 5540 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ForwardErrorActor*>(this)->SAV< T >::value()) T(x);
		this->~ForwardErrorActorState();
		static_cast<ForwardErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(T && x,int loopDepth) 
	{
															#line 1890 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ForwardErrorActor*>(this)->SAV<T>::futures) { (void)(x); this->~ForwardErrorActorState(); static_cast<ForwardErrorActor*>(this)->destroy(); return 0; }
															#line 5552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ForwardErrorActor*>(this)->SAV< T >::value()) T(x);
		this->~ForwardErrorActorState();
		static_cast<ForwardErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T const& x,int loopDepth) 
	{
		loopDepth = a_body1cont2(x, loopDepth);

		return loopDepth;
	}
	int a_body1when1(T && x,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(x), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ForwardErrorActor*>(this)->actor_wait_state > 0) static_cast<ForwardErrorActor*>(this)->actor_wait_state = 0;
		static_cast<ForwardErrorActor*>(this)->ActorCallback< ForwardErrorActor, 0, T >::remove();

	}
	void a_callback_fire(ActorCallback< ForwardErrorActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("forwardError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForwardErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forwardError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ForwardErrorActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("forwardError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForwardErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forwardError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ForwardErrorActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("forwardError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForwardErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forwardError", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Future<T> f;
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Promise<Void> target;
															#line 5639 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via forwardError()
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class T>
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ForwardErrorActor final : public Actor<T>, public ActorCallback< ForwardErrorActor<T>, 0, T >, public FastAllocated<ForwardErrorActor<T>>, public ForwardErrorActorState<T, ForwardErrorActor<T>> {
															#line 5646 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ForwardErrorActor<T>>::operator new;
	using FastAllocated<ForwardErrorActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8512944453639504896UL, 1132278896773386752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ForwardErrorActor<T>, 0, T >;
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ForwardErrorActor(Future<T> const& f,Promise<Void> const& target) 
															#line 5663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<T>(),
		   ForwardErrorActorState<T, ForwardErrorActor<T>>(f, target),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("forwardError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1301226801814355968UL, 15803518444079550976UL);
		ActorExecutionContextHelper __helper(static_cast<ForwardErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("forwardError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("forwardError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ForwardErrorActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class T>
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] Future<T> forwardError( Future<T> const& f, Promise<Void> const& target ) {
															#line 1886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<T>(new ForwardErrorActor<T>(f, target));
															#line 5698 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 1898 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

constexpr int initialVersion = invalidVersion;

class DWALPagerSnapshot;

// An implementation of IPager2 that supports atomicUpdate() of a page without forcing a change to new page ID.
// It does this internally mapping the original page ID to alternate page IDs by write version.
// The page id remaps are kept in memory and also logged to a "remap queue" which must be reloaded on cold start.
// To prevent the set of remaps from growing unboundedly, once a remap is old enough to be at or before the
// oldest pager version being maintained the remap can be "undone" by popping it from the remap queue,
// copying the alternate page ID's data over top of the original page ID's data, and deleting the remap from memory.
// This process basically describes a "Delayed" Write-Ahead-Log (DWAL) because the remap queue and the newly allocated
// alternate pages it references basically serve as a write ahead log for pages that will eventually be copied
// back to their original location once the original version is no longer needed.
class DWALPager final : public IPager2 {
public:
	typedef FIFOQueue<LogicalPageID> LogicalPageQueueT;
	typedef std::map<Version, LogicalPageID> VersionToPageMapT;
	typedef std::unordered_map<LogicalPageID, VersionToPageMapT> PageToVersionedMapT;
	struct PageCacheEntry {
		Future<Reference<ArenaPage>> readFuture;
		Future<Void> writeFuture;

		bool initialized() const { return readFuture.isValid(); }

		bool reading() const { return !readFuture.isReady(); }

		bool writing() const { return !writeFuture.isReady(); }

		bool evictable() const {
			// Don't evict if a page is still being read or written
			return !reading() && !writing();
		}

		// Entry is evictable when its write and read futures are ready, even if they are
		// errors, so any buffers they hold are no longer needed by the underlying file actors
		Future<Void> onEvictable() const { return ready(readFuture) && ready(writeFuture); }

		// Read and write futures are safe to cancel so just cancel them and return
		Future<Void> cancel() {
			writeFuture.cancel();
			readFuture.cancel();
			return Void();
		}
	};
	typedef ObjectCache<LogicalPageID, PageCacheEntry> PageCacheT;

	int64_t* getPageCachePenaltySource() override { return &pageCache.evictor().reservedSize; }

	constexpr static PhysicalPageID primaryHeaderPageID = 0;
	constexpr static PhysicalPageID backupHeaderPageID = 1;

#pragma pack(push, 1)
	struct DelayedFreePage {
		Version version;
		LogicalPageID pageID;

		bool operator<(const DelayedFreePage& rhs) const { return version < rhs.version; }

		std::string toString() const {
			return format("DelayedFreePage{%s @%" PRId64 "}", ::toString(pageID).c_str(), version);
		}
	};

	struct RemappedPage {
		enum Type { NONE = 'N', REMAP = 'R', FREE = 'F', DETACH = 'D' };
		RemappedPage(Version v = invalidVersion,
		             LogicalPageID o = invalidLogicalPageID,
		             LogicalPageID n = invalidLogicalPageID)
		  : version(v), originalPageID(o), newPageID(n) {}

		Version version;
		LogicalPageID originalPageID;
		LogicalPageID newPageID;

		static Type getTypeOf(LogicalPageID newPageID) {
			if (newPageID == invalidLogicalPageID) {
				return FREE;
			}
			if (newPageID == 0) {
				return DETACH;
			}
			return REMAP;
		}

		Type getType() const { return getTypeOf(newPageID); }

		bool operator<(const RemappedPage& rhs) const { return version < rhs.version; }

		std::string toString() const {
			return format("RemappedPage(%c: %s -> %s %s}",
			              getType(),
			              ::toString(originalPageID).c_str(),
			              ::toString(newPageID).c_str(),
			              ::toString(version).c_str());
		}
	};

	struct ExtentUsedListEntry {
		QueueID queueID;
		LogicalPageID extentID;

		bool operator<(const ExtentUsedListEntry& rhs) const { return queueID < rhs.queueID; }

		std::string toString() const {
			return format("ExtentUsedListEntry{%s @%s}", ::toString(extentID).c_str(), ::toString(queueID).c_str());
		}
	};

#pragma pack(pop)

	typedef FIFOQueue<DelayedFreePage> DelayedFreePageQueueT;
	typedef FIFOQueue<RemappedPage> RemapQueueT;
	typedef FIFOQueue<ExtentUsedListEntry> ExtentUsedListQueueT;

	// If the file already exists, pageSize might be different than desiredPageSize
	// Use pageCacheSizeBytes == 0 to use default from flow knobs
	// If memoryOnly is true, the pager will exist only in memory and once the cache is full writes will fail.
	// Note that ownership is not taken for keyProvider and it must outlive the pager
	DWALPager(int desiredPageSize,
	          int desiredExtentSize,
	          std::string filename,
	          int64_t pageCacheSizeBytes,
	          int64_t remapCleanupWindowBytes,
	          int concurrentExtentReads,
	          bool memoryOnly)
	  : ioLock(makeReference<PriorityMultiLock>(FLOW_KNOBS->MAX_OUTSTANDING, SERVER_KNOBS->REDWOOD_IO_PRIORITIES)),
	    pageCacheBytes(pageCacheSizeBytes), desiredPageSize(desiredPageSize), desiredExtentSize(desiredExtentSize),
	    filename(filename), memoryOnly(memoryOnly), remapCleanupWindowBytes(remapCleanupWindowBytes),
	    concurrentExtentReads(new FlowLock(concurrentExtentReads)) {

		// This sets the page cache size for all PageCacheT instances using the same evictor
		pageCache.evictor().sizeLimit = pageCacheBytes;

		g_redwoodMetrics.ioLock = ioLock.getPtr();
		if (!g_redwoodMetricsActor.isValid()) {
			g_redwoodMetricsActor = redwoodMetricsLogger();
		}

		commitFuture = Void();
		recoverFuture = forwardError(recover(this), errorPromise);
	}

	std::string getName() const override { return filename; }

	void setEncryptionKeyProvider(Reference<IPageEncryptionKeyProvider> kp) override {
		keyProvider = kp;
		keyProviderInitialized.send(Void());
	}

	void setPageSize(int size) {
		// Conservative maximum for number of records that can fit in this page size
		g_redwoodMetrics.updateMaxRecordCount(315.0 * size / 4096);

		logicalPageSize = size;
		// Physical page size is the total size of the smallest number of physical blocks needed to store
		// logicalPageSize bytes
		int blocks = 1 + ((logicalPageSize - 1) / smallestPhysicalBlock);
		physicalPageSize = blocks * smallestPhysicalBlock;
		header.pageSize = logicalPageSize;
	}

	void setExtentSize(int size) {
		// if the specified extent size is smaller than the physical page size, round it off to one physical page size
		// physical extent size has to be a multiple of physical page size
		if (size <= physicalPageSize) {
			pagesPerExtent = 1;
		} else {
			pagesPerExtent = 1 + ((size - 1) / physicalPageSize);
		}
		physicalExtentSize = pagesPerExtent * physicalPageSize;
		header.extentSize = size;
	}

	void updateHeaderPage() {
		Value serializedHeader = ObjectWriter::toValue(header, Unversioned());
		ASSERT(serializedHeader.size() <= headerPage->dataSize());
		serializedHeader.copyTo(headerPage->mutateData());

		// Set remaining header bytes to \xff
		memset(
		    headerPage->mutateData() + serializedHeader.size(), 0xff, headerPage->dataSize() - serializedHeader.size());
	}

	void updateLastCommittedHeader() {
		lastCommittedHeaderPage = headerPage->clone();
		lastCommittedHeader = header;
	}

																#line 5891 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via recover()
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class RecoverActor>
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RecoverActorState {
															#line 5897 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RecoverActorState(DWALPager* const& self) 
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 5904 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("recover", reinterpret_cast<unsigned long>(this));

	}
	~RecoverActorState() 
	{
		fdb_probe_actor_destroy("recover", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2088 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(!self->recoverFuture.isValid());
															#line 2090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			exists = false;
															#line 2092 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!self->memoryOnly)
															#line 5923 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2093 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				int64_t flags = IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_UNBUFFERED | IAsyncFile::OPEN_READWRITE | IAsyncFile::OPEN_LOCK;
															#line 2095 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				exists = fileExists(self->filename);
															#line 2096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->fileInitialized = exists;
															#line 2097 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!exists)
															#line 5933 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2098 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					flags |= IAsyncFile::OPEN_ATOMIC_WRITE_AND_CREATE | IAsyncFile::OPEN_CREATE;
															#line 5937 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
															#line 2100 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(self->pageFile, IAsyncFileSystem::filesystem()->open(self->filename, flags, 0644));
															#line 2100 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5943 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RecoverActor*>(this)->actor_wait_state = 1;
															#line 2100 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 0, Void >*>(static_cast<RecoverActor*>(this)));
															#line 5948 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2104 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->setPageSize(smallestPhysicalBlock);
															#line 2106 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		fileSize = 0;
															#line 2107 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (exists)
															#line 5980 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2108 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = store(fileSize, self->pageFile->size());
															#line 2108 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5986 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 2;
															#line 2108 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 1, Void >*>(static_cast<RecoverActor*>(this)));
															#line 5991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RecoverActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2111 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->fileExtension = Void();
															#line 2113 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf( "DWALPager(%s) recover exists=%d fileSize=%" PRId64 "\n", self->filename.c_str(), exists, fileSize);
															#line 2117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (exists)
															#line 6096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2118 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) recovering using existing file\n", self->filename.c_str());
															#line 2123 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (fileSize < (self->smallestPhysicalBlock * 2))
															#line 6102 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2124 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(storage_engine_not_initialized(), loopDepth);
															#line 6106 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 2127 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			recoveredBackupHeader = false;
															#line 6110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			try {
															#line 2131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_2 = store(self->headerPage, self->readHeaderPage(primaryHeaderPageID));
															#line 2131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1cont4Catch1(actor_cancelled(), loopDepth);
															#line 6116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont4Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
				static_cast<RecoverActor*>(this)->actor_wait_state = 3;
															#line 2131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 2, Void >*>(static_cast<RecoverActor*>(this)));
															#line 6121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont4Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 2289 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) creating new pager\n", self->filename.c_str());
															#line 2291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->headerPage = self->newPageBuffer();
															#line 2292 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->headerPage->init(EncodingType::XXHash64, PageType::HeaderPage, 0);
															#line 2295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->setPageSize(self->desiredPageSize);
															#line 2296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->filePageCount = 0;
															#line 2297 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->filePageCountPending = 0;
															#line 2301 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->setExtentSize(self->desiredExtentSize);
															#line 2304 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.formatVersion = PagerCommitHeader::FORMAT_VERSION;
															#line 2305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.committedVersion = initialVersion;
															#line 2306 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.oldestVersion = initialVersion;
															#line 2311 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.pageCount = 2;
															#line 2314 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.queueCount = 0;
															#line 2315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->freeList.create(self, self->newLastPageID(), "FreeList", self->newLastQueueID(), false);
															#line 2316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->delayedFreeList.create(self, self->newLastPageID(), "DelayedFreeList", self->newLastQueueID(), false);
															#line 2317 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentFreeList.create(self, self->newLastPageID(), "ExtentFreeList", self->newLastQueueID(), false);
															#line 2318 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.create(self, self->newLastPageID(), "ExtentUsedList", self->newLastQueueID(), false);
															#line 2319 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			LogicalPageID extID = self->newLastExtentID();
															#line 2320 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remapQueue.create(self, extID, "RemapQueue", self->newLastQueueID(), true);
															#line 2321 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.pushBack({ self->remapQueue.queueID, extID });
															#line 2326 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.freeList = self->freeList.getState();
															#line 2327 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.delayedFreeList = self->delayedFreeList.getState();
															#line 2328 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.extentFreeList = self->extentFreeList.getState();
															#line 2329 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.extentUsedList = self->extentUsedList.getState();
															#line 2330 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->header.remapQueue = self->remapQueue.getState();
															#line 2333 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->updateHeaderPage();
															#line 2337 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->updateLastCommittedHeader();
															#line 2338 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->addSnapshot(initialVersion, KeyRef());
															#line 2340 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remapCleanupFuture = Void();
															#line 6188 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RecoverActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 2343 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->memoryOnly)
															#line 6285 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2344 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_11 = store(fileSize, self->pageFile->size());
															#line 2344 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont6when1(__when_expr_11.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 11;
															#line 2344 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 11, Void >*>(static_cast<RecoverActor*>(this)));
															#line 6296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont15(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2157 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->headerPage.isValid())
															#line 6310 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			try {
															#line 2160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(self->headerPage, self->readHeaderPage(backupHeaderPageID));
															#line 2160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1cont7Catch1(actor_cancelled(), loopDepth);
															#line 6317 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont7Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
				static_cast<RecoverActor*>(this)->actor_wait_state = 4;
															#line 2160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 3, Void >*>(static_cast<RecoverActor*>(this)));
															#line 6322 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont7Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont7Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) Primary header read failed with %s\n", self->filename.c_str(), e.what());
															#line 2139 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			bool tryBackupHeader = (e.code() == error_code_page_header_version_not_supported || e.code() == error_code_page_encoding_not_supported || e.code() == error_code_page_decoding_failed || e.code() == error_code_page_header_checksum_failed);
															#line 2144 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarn, "RedwoodRecoveryErrorPrimaryHeaderFailed") .errorUnsuppressed(e) .detail("Filename", self->filename) .detail("PageID", primaryHeaderPageID) .detail("TryingBackupHeader", tryBackupHeader);
															#line 2151 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!tryBackupHeader)
															#line 6349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2152 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
			loopDepth = a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RecoverActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
															#line 2181 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header = ObjectReader::fromStringRef<PagerCommitHeader>(self->headerPage->dataAsStringRef(), Unversioned());
															#line 2184 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->header.formatVersion != PagerCommitHeader::FORMAT_VERSION)
															#line 6471 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2185 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			Error e = unsupported_format_version();
															#line 2186 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarnAlways, "RedwoodRecoveryFailedWrongVersion") .error(e) .detail("Filename", self->filename) .detail("Version", self->header.formatVersion) .detail("ExpectedVersion", PagerCommitHeader::FORMAT_VERSION);
															#line 2191 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 6479 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2194 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->setPageSize(self->header.pageSize);
															#line 2195 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->filePageCount = fileSize / self->physicalPageSize;
															#line 2196 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->filePageCountPending = self->filePageCount;
															#line 2198 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->logicalPageSize != self->desiredPageSize)
															#line 6489 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2199 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarnAlways, "RedwoodPageSizeMismatch") .detail("InstanceName", self->getName()) .detail("ExistingPageSize", self->logicalPageSize) .detail("DesiredPageSize", self->desiredPageSize);
															#line 6493 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2205 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->setExtentSize(self->header.extentSize);
															#line 2207 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.recover(self, self->header.freeList, "FreeListRecovered");
															#line 2208 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentFreeList.recover(self, self->header.extentFreeList, "ExtentFreeListRecovered");
															#line 2209 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.recover(self, self->header.delayedFreeList, "DelayedFreeListRecovered");
															#line 2210 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.recover(self, self->header.extentUsedList, "ExtentUsedListRecovered");
															#line 2211 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.recover(self, self->header.remapQueue, "RemapQueueRecovered");
															#line 2213 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) Queue recovery complete.\n", self->filename.c_str());
															#line 2217 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Standalone<VectorRef<ExtentUsedListEntry>>> __when_expr_4 = self->extentUsedList.peekAll();
															#line 2217 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6513 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont11when1(__when_expr_4.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 5;
															#line 2217 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >*>(static_cast<RecoverActor*>(this)));
															#line 6518 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2163 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) Backup header read failed with %s\n", self->filename.c_str(), e.what());
															#line 2164 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarn, "RedwoodRecoveryErrorBackupHeaderFailed") .error(e) .detail("Filename", self->filename) .detail("PageID", backupHeaderPageID);
															#line 2172 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated() && g_simulator->restarted)
															#line 6538 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2173 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e.asInjectedFault(), loopDepth);
															#line 6542 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 2176 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 6546 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont13(Void const& _,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		recoveredBackupHeader = true;
															#line 6560 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Void && _,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		recoveredBackupHeader = true;
															#line 6569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RecoverActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont13cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1(Standalone<VectorRef<ExtentUsedListEntry>> const& extents,int loopDepth) 
	{
															#line 2218 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) ExtentUsedList size: %d.\n", self->filename.c_str(), extents.size());
															#line 2219 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (extents.size() > 1)
															#line 6668 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			QueueID remapQueueID = self->remapQueue.queueID;
															#line 2221 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for(int i = 1;i < extents.size() - 1;i++) {
															#line 2222 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (extents[i].queueID == remapQueueID)
															#line 6676 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2223 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					LogicalPageID extID = extents[i].extentID;
															#line 2224 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("DWALPager Extents: ID: %s ", toString(extID).c_str());
															#line 2225 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->readExtent(extID);
															#line 6684 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
															#line 2233 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapStream = PromiseStream<Standalone<VectorRef<RemappedPage>>>();
															#line 2234 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = Future<Void>();
															#line 2235 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = self->remapQueue.peekAllExt(remapStream);
															#line 6694 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2237 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 6698 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1cont11cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1(Standalone<VectorRef<ExtentUsedListEntry>> && extents,int loopDepth) 
	{
															#line 2218 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) ExtentUsedList size: %d.\n", self->filename.c_str(), extents.size());
															#line 2219 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (extents.size() > 1)
															#line 6715 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			QueueID remapQueueID = self->remapQueue.queueID;
															#line 2221 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for(int i = 1;i < extents.size() - 1;i++) {
															#line 2222 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (extents[i].queueID == remapQueueID)
															#line 6723 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2223 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					LogicalPageID extID = extents[i].extentID;
															#line 2224 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("DWALPager Extents: ID: %s ", toString(extID).c_str());
															#line 2225 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->readExtent(extID);
															#line 6731 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
															#line 2233 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapStream = PromiseStream<Standalone<VectorRef<RemappedPage>>>();
															#line 2234 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = Future<Void>();
															#line 2235 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = self->remapQueue.peekAllExt(remapStream);
															#line 6741 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2237 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 6745 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1cont11cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11when1(Standalone<VectorRef<ExtentUsedListEntry>> const& extents,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(extents, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Standalone<VectorRef<ExtentUsedListEntry>> && extents,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(extents), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >*,Standalone<VectorRef<ExtentUsedListEntry>> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >*,Standalone<VectorRef<ExtentUsedListEntry>> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11cont4(int loopDepth) 
	{
															#line 2257 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) recovery complete. RemappedPagesMap: %s\n", self->filename.c_str(), toString(self->remappedPages).c_str());
															#line 2261 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) recovery complete. destroy extent cache\n", self->filename.c_str());
															#line 2262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_7 = self->extentCache.clear();
															#line 2262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont11cont4when1(__when_expr_7.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 7;
															#line 2262 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 7, Void >*>(static_cast<RecoverActor*>(this)));
															#line 6846 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2252 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 6856 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2253 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6860 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
			loopDepth = a_body1cont11cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont11cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1(int loopDepth) 
	{
															#line 2238 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto __when_expr_5 = remapStream.getFuture();
															#line 6883 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_5), FutureStream<Standalone<VectorRef<RemappedPage>>>>::value || std::is_same<decltype(__when_expr_5), ThreadFutureStream<Standalone<VectorRef<RemappedPage>>>>::value, "invalid type");
															#line 2237 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6887 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont11cont1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont11cont1loopBody1when1(__when_expr_5.pop(), loopDepth); };
															#line 2247 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_6 = remapRecoverActor;
															#line 6891 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont11cont1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont11cont1loopBody1when2(__when_expr_6.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 6;
															#line 2238 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >*>(static_cast<RecoverActor*>(this)));
															#line 2247 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 6, Void >*>(static_cast<RecoverActor*>(this)));
															#line 6898 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont11cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when1(Standalone<VectorRef<RemappedPage>> const& remaps,int loopDepth) 
	{
															#line 2239 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) recovery. remaps size: %d, queueEntries: %" PRId64 "\n", self->filename.c_str(), remaps.size(), self->remapQueue.numEntries);
															#line 2243 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for( auto& r : remaps ) {
															#line 2244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remappedPages[r.originalPageID][r.version] = r.newPageID;
															#line 6917 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when1(Standalone<VectorRef<RemappedPage>> && remaps,int loopDepth) 
	{
															#line 2239 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) recovery. remaps size: %d, queueEntries: %" PRId64 "\n", self->filename.c_str(), remaps.size(), self->remapQueue.numEntries);
															#line 2243 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for( auto& r : remaps ) {
															#line 2244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remappedPages[r.originalPageID][r.version] = r.newPageID;
															#line 6931 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = Never();
															#line 6941 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapRecoverActor = Never();
															#line 6950 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >::remove();
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >*,Standalone<VectorRef<RemappedPage>> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >*,Standalone<VectorRef<RemappedPage>> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RecoverActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont11cont9(Void const& _,int loopDepth) 
	{
															#line 2266 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (recoveredBackupHeader)
															#line 7080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_8 = self->writeHeaderPage(primaryHeaderPageID, self->headerPage);
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7086 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont11cont9when1(__when_expr_8.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 8;
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 8, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7091 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont9(Void && _,int loopDepth) 
	{
															#line 2266 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (recoveredBackupHeader)
															#line 7105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_8 = self->writeHeaderPage(primaryHeaderPageID, self->headerPage);
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7111 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont11cont9when1(__when_expr_8.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 8;
															#line 2268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 8, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< RecoverActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont11cont10(int loopDepth) 
	{
															#line 2280 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->updateLastCommittedHeader();
															#line 2281 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->addSnapshot(self->header.committedVersion, self->header.userCommitRecord);
															#line 2284 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.resetHeadReader();
															#line 2286 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupFuture = forwardError(remapCleanup(self), self->errorPromise);
															#line 7211 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11(Void const& _,int loopDepth) 
	{
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_9 = waitForAll(self->operations);
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7222 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont11cont11when1(__when_expr_9.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 9;
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 9, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7227 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont11(Void && _,int loopDepth) 
	{
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_9 = waitForAll(self->operations);
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7238 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont11cont11when1(__when_expr_9.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 9;
															#line 2271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 9, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7243 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< RecoverActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont11cont11cont1(Void const& _,int loopDepth) 
	{
															#line 2272 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_10 = self->pageFile->sync();
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7331 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont11cont11cont1when1(__when_expr_10.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 10;
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 10, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont11cont1(Void && _,int loopDepth) 
	{
															#line 2272 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_10 = self->pageFile->sync();
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont11cont11cont1when1(__when_expr_10.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 10;
															#line 2274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 10, Void >*>(static_cast<RecoverActor*>(this)));
															#line 7354 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< RecoverActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont11cont11cont2(Void const& _,int loopDepth) 
	{
															#line 2275 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) Header recovery complete.\n", self->filename.c_str());
															#line 7438 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11cont2(Void && _,int loopDepth) 
	{
															#line 2275 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) Header recovery complete.\n", self->filename.c_str());
															#line 7447 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont11cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< RecoverActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont15(int loopDepth) 
	{
															#line 2347 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		TraceEvent e(SevInfo, "RedwoodRecoveredPager");
															#line 2348 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("OpenedExisting", exists);
															#line 2349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->toTraceEvent(e);
															#line 2350 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.log();
															#line 2352 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->recoveryVersion = self->header.committedVersion;
															#line 2353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) recovered.  recoveryVersion=%" PRId64 " oldestVersion=%" PRId64 " logicalPageSize=%d physicalPageSize=%d headerPageCount=%" PRId64 " filePageCount=%" PRId64 "\n", self->filename.c_str(), self->recoveryVersion, self->header.oldestVersion, self->logicalPageSize, self->physicalPageSize, self->header.pageCount, self->filePageCount);
															#line 2363 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<RecoverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RecoverActorState(); static_cast<RecoverActor*>(this)->destroy(); return 0; }
															#line 7543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<RecoverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< RecoverActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 11);

	}
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool exists;
															#line 2106 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int64_t fileSize;
															#line 2127 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool recoveredBackupHeader;
															#line 2233 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PromiseStream<Standalone<VectorRef<RemappedPage>>> remapStream;
															#line 2234 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Future<Void> remapRecoverActor;
															#line 7650 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via recover()
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RecoverActor final : public Actor<Void>, public ActorCallback< RecoverActor, 0, Void >, public ActorCallback< RecoverActor, 1, Void >, public ActorCallback< RecoverActor, 2, Void >, public ActorCallback< RecoverActor, 3, Void >, public ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >, public ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >, public ActorCallback< RecoverActor, 6, Void >, public ActorCallback< RecoverActor, 7, Void >, public ActorCallback< RecoverActor, 8, Void >, public ActorCallback< RecoverActor, 9, Void >, public ActorCallback< RecoverActor, 10, Void >, public ActorCallback< RecoverActor, 11, Void >, public FastAllocated<RecoverActor>, public RecoverActorState<RecoverActor> {
															#line 7655 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<RecoverActor>::operator new;
	using FastAllocated<RecoverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2550929837541393408UL, 11057633041711127296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RecoverActor, 0, Void >;
friend struct ActorCallback< RecoverActor, 1, Void >;
friend struct ActorCallback< RecoverActor, 2, Void >;
friend struct ActorCallback< RecoverActor, 3, Void >;
friend struct ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >;
friend struct ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >;
friend struct ActorCallback< RecoverActor, 6, Void >;
friend struct ActorCallback< RecoverActor, 7, Void >;
friend struct ActorCallback< RecoverActor, 8, Void >;
friend struct ActorCallback< RecoverActor, 9, Void >;
friend struct ActorCallback< RecoverActor, 10, Void >;
friend struct ActorCallback< RecoverActor, 11, Void >;
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RecoverActor(DWALPager* const& self) 
															#line 7683 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   RecoverActorState<RecoverActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7334007120712012544UL, 4865582193762289920UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("recover");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RecoverActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RecoverActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RecoverActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RecoverActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< RecoverActor, 4, Standalone<VectorRef<ExtentUsedListEntry>> >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorSingleCallback< RecoverActor, 5, Standalone<VectorRef<RemappedPage>> >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< RecoverActor, 7, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< RecoverActor, 8, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< RecoverActor, 9, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< RecoverActor, 10, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< RecoverActor, 11, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> recover( DWALPager* const& self ) {
															#line 2087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new RecoverActor(self));
															#line 7725 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2365 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	void toTraceEvent(TraceEvent& e) const override {
		e.detail("FileName", filename.c_str());
		e.detail("LogicalFileSize", header.pageCount * physicalPageSize);
		e.detail("PhysicalFileSize", filePageCountPending * physicalPageSize);
		e.detail("CommittedVersion", header.committedVersion);
		e.detail("LogicalPageSize", logicalPageSize);
		e.detail("PhysicalPageSize", physicalPageSize);

		remapQueue.toTraceEvent(e, "RemapQueue");
		delayedFreeList.toTraceEvent(e, "FreeQueue");
		freeList.toTraceEvent(e, "DelayedFreeQueue");
		extentUsedList.toTraceEvent(e, "UsedExtentQueue");
		extentFreeList.toTraceEvent(e, "FreeExtentQueue");
		getStorageBytes().toTraceEvent(e);
	}

																#line 7746 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via extentCacheClear_impl()
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ExtentCacheClear_implActor>
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ExtentCacheClear_implActorState {
															#line 7752 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ExtentCacheClear_implActorState(DWALPager* const& self) 
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 7759 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("extentCacheClear_impl", reinterpret_cast<unsigned long>(this));

	}
	~ExtentCacheClear_implActorState() 
	{
		fdb_probe_actor_destroy("extentCacheClear_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->extentCache.clear();
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 7776 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<ExtentCacheClear_implActor*>(this)->actor_wait_state = 1;
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExtentCacheClear_implActor, 0, Void >*>(static_cast<ExtentCacheClear_implActor*>(this)));
															#line 7780 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<ExtentCacheClear_implActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExtentCacheClear_implActor*>(this)->actor_wait_state > 0) static_cast<ExtentCacheClear_implActor*>(this)->actor_wait_state = 0;
		static_cast<ExtentCacheClear_implActor*>(this)->ActorCallback< ExtentCacheClear_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExtentCacheClear_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtentCacheClear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExtentCacheClear_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtentCacheClear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExtentCacheClear_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExtentCacheClear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		delete static_cast<ExtentCacheClear_implActor*>(this);
															#line 7889 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		return 0;

		return loopDepth;
	}
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 7896 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via extentCacheClear_impl()
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ExtentCacheClear_implActor final : public Actor<void>, public ActorCallback< ExtentCacheClear_implActor, 0, Void >, public FastAllocated<ExtentCacheClear_implActor>, public ExtentCacheClear_implActorState<ExtentCacheClear_implActor> {
															#line 7901 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ExtentCacheClear_implActor>::operator new;
	using FastAllocated<ExtentCacheClear_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8289288674904024576UL, 11404219533729065472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< ExtentCacheClear_implActor, 0, Void >;
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ExtentCacheClear_implActor(DWALPager* const& self) 
															#line 7918 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<void>(),
		   ExtentCacheClear_implActorState<ExtentCacheClear_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8878323354082029312UL, 8227386504867395072UL);
		ActorExecutionContextHelper __helper(static_cast<ExtentCacheClear_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("extentCacheClear_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("extentCacheClear_impl", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
static void extentCacheClear_impl( DWALPager* const& self ) {
															#line 2382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	new ExtentCacheClear_implActor(self);
															#line 7941 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2383 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	void extentCacheClear() override { extentCacheClear_impl(this); }

	// get a list of used extents for a given extent based queue (for testing purpose)
																#line 7949 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via getUsedExtents_impl()
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class GetUsedExtents_implActor>
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class GetUsedExtents_implActorState {
															#line 7955 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	GetUsedExtents_implActorState(DWALPager* const& self,QueueID const& queueID) 
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   queueID(queueID),
															#line 2388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   extentIDs()
															#line 7966 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("getUsedExtents_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetUsedExtents_implActorState() 
	{
		fdb_probe_actor_destroy("getUsedExtents_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2389 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			extentIDs.reserve(extentIDs.arena(), self->extentUsedList.numEntries);
															#line 2392 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Standalone<VectorRef<ExtentUsedListEntry>>> __when_expr_0 = self->extentUsedList.peekAll();
															#line 2392 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<GetUsedExtents_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7985 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetUsedExtents_implActor*>(this)->actor_wait_state = 1;
															#line 2392 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >*>(static_cast<GetUsedExtents_implActor*>(this)));
															#line 7990 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetUsedExtents_implActorState();
		static_cast<GetUsedExtents_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Standalone<VectorRef<ExtentUsedListEntry>> const& extents,int loopDepth) 
	{
															#line 2393 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) ExtentUsedList size: %d.\n", self->filename.c_str(), extents.size());
															#line 2394 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (extents.size() > 1)
															#line 8015 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2395 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for(int i = 1;i < extents.size() - 1;i++) {
															#line 2396 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (extents[i].queueID == queueID)
															#line 8021 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2397 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					LogicalPageID extID = extents[i].extentID;
															#line 2398 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("DWALPager Extents: ID: %s ", toString(extID).c_str());
															#line 2399 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					extentIDs.push_back(extentIDs.arena(), extID);
															#line 8029 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
															#line 2403 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<GetUsedExtents_implActor*>(this)->SAV<Standalone<VectorRef<LogicalPageID>>>::futures) { (void)(extentIDs); this->~GetUsedExtents_implActorState(); static_cast<GetUsedExtents_implActor*>(this)->destroy(); return 0; }
															#line 8035 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<GetUsedExtents_implActor*>(this)->SAV< Standalone<VectorRef<LogicalPageID>> >::value()) Standalone<VectorRef<LogicalPageID>>(std::move(extentIDs)); // state_var_RVO
		this->~GetUsedExtents_implActorState();
		static_cast<GetUsedExtents_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Standalone<VectorRef<ExtentUsedListEntry>> && extents,int loopDepth) 
	{
															#line 2393 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) ExtentUsedList size: %d.\n", self->filename.c_str(), extents.size());
															#line 2394 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (extents.size() > 1)
															#line 8049 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2395 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for(int i = 1;i < extents.size() - 1;i++) {
															#line 2396 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (extents[i].queueID == queueID)
															#line 8055 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2397 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					LogicalPageID extID = extents[i].extentID;
															#line 2398 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("DWALPager Extents: ID: %s ", toString(extID).c_str());
															#line 2399 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					extentIDs.push_back(extentIDs.arena(), extID);
															#line 8063 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
															#line 2403 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<GetUsedExtents_implActor*>(this)->SAV<Standalone<VectorRef<LogicalPageID>>>::futures) { (void)(extentIDs); this->~GetUsedExtents_implActorState(); static_cast<GetUsedExtents_implActor*>(this)->destroy(); return 0; }
															#line 8069 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<GetUsedExtents_implActor*>(this)->SAV< Standalone<VectorRef<LogicalPageID>> >::value()) Standalone<VectorRef<LogicalPageID>>(std::move(extentIDs)); // state_var_RVO
		this->~GetUsedExtents_implActorState();
		static_cast<GetUsedExtents_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Standalone<VectorRef<ExtentUsedListEntry>> const& extents,int loopDepth) 
	{
		loopDepth = a_body1cont1(extents, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Standalone<VectorRef<ExtentUsedListEntry>> && extents,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(extents), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetUsedExtents_implActor*>(this)->actor_wait_state > 0) static_cast<GetUsedExtents_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetUsedExtents_implActor*>(this)->ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >::remove();

	}
	void a_callback_fire(ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >*,Standalone<VectorRef<ExtentUsedListEntry>> const& value) 
	{
		fdb_probe_actor_enter("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUsedExtents_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >*,Standalone<VectorRef<ExtentUsedListEntry>> && value) 
	{
		fdb_probe_actor_enter("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUsedExtents_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >*,Error err) 
	{
		fdb_probe_actor_enter("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetUsedExtents_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	QueueID queueID;
															#line 2388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Standalone<VectorRef<LogicalPageID>> extentIDs;
															#line 8158 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via getUsedExtents_impl()
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class GetUsedExtents_implActor final : public Actor<Standalone<VectorRef<LogicalPageID>>>, public ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >, public FastAllocated<GetUsedExtents_implActor>, public GetUsedExtents_implActorState<GetUsedExtents_implActor> {
															#line 8163 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<GetUsedExtents_implActor>::operator new;
	using FastAllocated<GetUsedExtents_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16592554913386148352UL, 5699793860849530624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<VectorRef<LogicalPageID>>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >;
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	GetUsedExtents_implActor(DWALPager* const& self,QueueID const& queueID) 
															#line 8180 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Standalone<VectorRef<LogicalPageID>>>(),
		   GetUsedExtents_implActorState<GetUsedExtents_implActor>(self, queueID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1853066824266135552UL, 12017888193915963648UL);
		ActorExecutionContextHelper __helper(static_cast<GetUsedExtents_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getUsedExtents_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getUsedExtents_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetUsedExtents_implActor, 0, Standalone<VectorRef<ExtentUsedListEntry>> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Standalone<VectorRef<LogicalPageID>>> getUsedExtents_impl( DWALPager* const& self, QueueID const& queueID ) {
															#line 2387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Standalone<VectorRef<LogicalPageID>>>(new GetUsedExtents_implActor(self, queueID));
															#line 8212 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2405 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Standalone<VectorRef<LogicalPageID>>> getUsedExtents(QueueID queueID) override {
		return getUsedExtents_impl(this, queueID);
	}

	void pushExtentUsedList(QueueID queueID, LogicalPageID extID) override {
		extentUsedList.pushBack({ queueID, extID });
	}

	// Allocate a new queueID
	QueueID newLastQueueID() override { return header.queueCount++; }

	Reference<ArenaPage> newPageBuffer(size_t blocks = 1) override {
		return makeReference<ArenaPage>(logicalPageSize * blocks, physicalPageSize * blocks);
	}

	int getPhysicalPageSize() const override { return physicalPageSize; }
	int getLogicalPageSize() const override { return logicalPageSize; }
	int getPagesPerExtent() const override { return pagesPerExtent; }

	// Get a new, previously available page ID.  The page will be considered in-use after the next commit
	// regardless of whether or not it was written to, until it is returned to the pager via freePage()
																#line 8238 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via newPageID_impl()
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class NewPageID_implActor>
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class NewPageID_implActorState {
															#line 8244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	NewPageID_implActorState(DWALPager* const& self) 
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 8251 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("newPageID_impl", reinterpret_cast<unsigned long>(this));

	}
	~NewPageID_implActorState() 
	{
		fdb_probe_actor_destroy("newPageID_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2429 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Optional<LogicalPageID>> __when_expr_0 = self->freeList.pop();
															#line 2429 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<NewPageID_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<NewPageID_implActor*>(this)->actor_wait_state = 1;
															#line 2429 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >*>(static_cast<NewPageID_implActor*>(this)));
															#line 8273 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~NewPageID_implActorState();
		static_cast<NewPageID_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<LogicalPageID> const& freePageID,int loopDepth) 
	{
															#line 2430 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freePageID.present())
															#line 8296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) newPageID() returning %s from free list\n", self->filename.c_str(), toString(freePageID.get()).c_str());
															#line 2434 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(freePageID.get()); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8302 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(freePageID.get());
			this->~NewPageID_implActorState();
			static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<DelayedFreePage>> __when_expr_1 = self->delayedFreeList.pop(DelayedFreePage{ self->effectiveOldestVersion(), 0 });
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<NewPageID_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8312 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<NewPageID_implActor*>(this)->actor_wait_state = 2;
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*>(static_cast<NewPageID_implActor*>(this)));
															#line 8317 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<LogicalPageID> && freePageID,int loopDepth) 
	{
															#line 2430 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freePageID.present())
															#line 8326 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) newPageID() returning %s from free list\n", self->filename.c_str(), toString(freePageID.get()).c_str());
															#line 2434 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(freePageID.get()); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8332 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(freePageID.get());
			this->~NewPageID_implActorState();
			static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<DelayedFreePage>> __when_expr_1 = self->delayedFreeList.pop(DelayedFreePage{ self->effectiveOldestVersion(), 0 });
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<NewPageID_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8342 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<NewPageID_implActor*>(this)->actor_wait_state = 2;
															#line 2439 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*>(static_cast<NewPageID_implActor*>(this)));
															#line 8347 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<LogicalPageID> const& freePageID,int loopDepth) 
	{
		loopDepth = a_body1cont1(freePageID, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<LogicalPageID> && freePageID,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(freePageID), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<NewPageID_implActor*>(this)->actor_wait_state > 0) static_cast<NewPageID_implActor*>(this)->actor_wait_state = 0;
		static_cast<NewPageID_implActor*>(this)->ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >::remove();

	}
	void a_callback_fire(ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >*,Optional<LogicalPageID> const& value) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >*,Optional<LogicalPageID> && value) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >*,Error err) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Optional<DelayedFreePage> const& delayedFreePageID,int loopDepth) 
	{
															#line 2441 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (delayedFreePageID.present())
															#line 8431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) newPageID() returning %s from delayed free list\n", self->filename.c_str(), toString(delayedFreePageID.get()).c_str());
															#line 2445 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(delayedFreePageID.get().pageID); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8437 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(delayedFreePageID.get().pageID);
			this->~NewPageID_implActorState();
			static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2449 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		LogicalPageID id = self->newLastPageID();
															#line 2450 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf( "DWALPager(%s) newPageID() returning %s at end of file\n", self->filename.c_str(), toString(id).c_str());
															#line 2452 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(id); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8449 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(id);
		this->~NewPageID_implActorState();
		static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<DelayedFreePage> && delayedFreePageID,int loopDepth) 
	{
															#line 2441 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (delayedFreePageID.present())
															#line 8461 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) newPageID() returning %s from delayed free list\n", self->filename.c_str(), toString(delayedFreePageID.get()).c_str());
															#line 2445 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(delayedFreePageID.get().pageID); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8467 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(delayedFreePageID.get().pageID);
			this->~NewPageID_implActorState();
			static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2449 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		LogicalPageID id = self->newLastPageID();
															#line 2450 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf( "DWALPager(%s) newPageID() returning %s at end of file\n", self->filename.c_str(), toString(id).c_str());
															#line 2452 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<NewPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(id); this->~NewPageID_implActorState(); static_cast<NewPageID_implActor*>(this)->destroy(); return 0; }
															#line 8479 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<NewPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(id);
		this->~NewPageID_implActorState();
		static_cast<NewPageID_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<DelayedFreePage> const& delayedFreePageID,int loopDepth) 
	{
		loopDepth = a_body1cont2(delayedFreePageID, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<DelayedFreePage> && delayedFreePageID,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(delayedFreePageID), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<NewPageID_implActor*>(this)->actor_wait_state > 0) static_cast<NewPageID_implActor*>(this)->actor_wait_state = 0;
		static_cast<NewPageID_implActor*>(this)->ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >::remove();

	}
	void a_callback_fire(ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*,Optional<DelayedFreePage> const& value) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*,Optional<DelayedFreePage> && value) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*,Error err) 
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 8564 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via newPageID_impl()
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class NewPageID_implActor final : public Actor<LogicalPageID>, public ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >, public ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >, public FastAllocated<NewPageID_implActor>, public NewPageID_implActorState<NewPageID_implActor> {
															#line 8569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<NewPageID_implActor>::operator new;
	using FastAllocated<NewPageID_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5564067029059281152UL, 10151896447647132672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<LogicalPageID>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >;
friend struct ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >;
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	NewPageID_implActor(DWALPager* const& self) 
															#line 8587 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<LogicalPageID>(),
		   NewPageID_implActorState<NewPageID_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("newPageID_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3366979944459341312UL, 2795710346137443584UL);
		ActorExecutionContextHelper __helper(static_cast<NewPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("newPageID_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("newPageID_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< NewPageID_implActor, 0, Optional<LogicalPageID> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< NewPageID_implActor, 1, Optional<DelayedFreePage> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<LogicalPageID> newPageID_impl( DWALPager* const& self ) {
															#line 2427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<LogicalPageID>(new NewPageID_implActor(self));
															#line 8620 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2453 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"


	// Grow the pager file by one page and return it
	LogicalPageID newLastPageID() {
		LogicalPageID id = header.pageCount;
		growPager(1);
		return id;
	}

	Future<LogicalPageID> newPageID() override { return newPageID_impl(this); }

	void growPager(int64_t pages) { header.pageCount += pages; }

	// Get a new, previously available extent and it's first page ID.  The page will be considered in-use after the next
	// commit regardless of whether or not it was written to, until it is returned to the pager via freePage()
																#line 8639 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via newExtentPageID_impl()
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class NewExtentPageID_implActor>
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class NewExtentPageID_implActorState {
															#line 8645 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	NewExtentPageID_implActorState(DWALPager* const& self,QueueID const& queueID) 
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   queueID(queueID)
															#line 8654 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("newExtentPageID_impl", reinterpret_cast<unsigned long>(this));

	}
	~NewExtentPageID_implActorState() 
	{
		fdb_probe_actor_destroy("newExtentPageID_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2470 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Optional<LogicalPageID>> __when_expr_0 = self->extentFreeList.pop();
															#line 2470 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<NewExtentPageID_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8671 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<NewExtentPageID_implActor*>(this)->actor_wait_state = 1;
															#line 2470 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >*>(static_cast<NewExtentPageID_implActor*>(this)));
															#line 8676 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~NewExtentPageID_implActorState();
		static_cast<NewExtentPageID_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<LogicalPageID> const& freeExtentID,int loopDepth) 
	{
															#line 2471 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeExtentID.present())
															#line 8699 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2472 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapQueue newExtentPageID() returning %s from free list\n", self->filename.c_str(), toString(freeExtentID.get()).c_str());
															#line 2475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.pushBack({ queueID, freeExtentID.get() });
															#line 2476 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.getState();
															#line 2477 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewExtentPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(freeExtentID.get()); this->~NewExtentPageID_implActorState(); static_cast<NewExtentPageID_implActor*>(this)->destroy(); return 0; }
															#line 8709 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewExtentPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(freeExtentID.get());
			this->~NewExtentPageID_implActorState();
			static_cast<NewExtentPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2481 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		LogicalPageID id = self->newLastExtentID();
															#line 2482 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) remapQueue newExtentPageID() returning %s at end of file\n", self->filename.c_str(), toString(id).c_str());
															#line 2485 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.pushBack({ queueID, id });
															#line 2486 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.getState();
															#line 2487 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<NewExtentPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(id); this->~NewExtentPageID_implActorState(); static_cast<NewExtentPageID_implActor*>(this)->destroy(); return 0; }
															#line 8725 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<NewExtentPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(id);
		this->~NewExtentPageID_implActorState();
		static_cast<NewExtentPageID_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<LogicalPageID> && freeExtentID,int loopDepth) 
	{
															#line 2471 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeExtentID.present())
															#line 8737 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2472 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapQueue newExtentPageID() returning %s from free list\n", self->filename.c_str(), toString(freeExtentID.get()).c_str());
															#line 2475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.pushBack({ queueID, freeExtentID.get() });
															#line 2476 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentUsedList.getState();
															#line 2477 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<NewExtentPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(freeExtentID.get()); this->~NewExtentPageID_implActorState(); static_cast<NewExtentPageID_implActor*>(this)->destroy(); return 0; }
															#line 8747 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<NewExtentPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(freeExtentID.get());
			this->~NewExtentPageID_implActorState();
			static_cast<NewExtentPageID_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2481 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		LogicalPageID id = self->newLastExtentID();
															#line 2482 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) remapQueue newExtentPageID() returning %s at end of file\n", self->filename.c_str(), toString(id).c_str());
															#line 2485 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.pushBack({ queueID, id });
															#line 2486 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.getState();
															#line 2487 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<NewExtentPageID_implActor*>(this)->SAV<LogicalPageID>::futures) { (void)(id); this->~NewExtentPageID_implActorState(); static_cast<NewExtentPageID_implActor*>(this)->destroy(); return 0; }
															#line 8763 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<NewExtentPageID_implActor*>(this)->SAV< LogicalPageID >::value()) LogicalPageID(id);
		this->~NewExtentPageID_implActorState();
		static_cast<NewExtentPageID_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<LogicalPageID> const& freeExtentID,int loopDepth) 
	{
		loopDepth = a_body1cont1(freeExtentID, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<LogicalPageID> && freeExtentID,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(freeExtentID), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<NewExtentPageID_implActor*>(this)->actor_wait_state > 0) static_cast<NewExtentPageID_implActor*>(this)->actor_wait_state = 0;
		static_cast<NewExtentPageID_implActor*>(this)->ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >::remove();

	}
	void a_callback_fire(ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >*,Optional<LogicalPageID> const& value) 
	{
		fdb_probe_actor_enter("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewExtentPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >*,Optional<LogicalPageID> && value) 
	{
		fdb_probe_actor_enter("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NewExtentPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >*,Error err) 
	{
		fdb_probe_actor_enter("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<NewExtentPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	QueueID queueID;
															#line 8850 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via newExtentPageID_impl()
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class NewExtentPageID_implActor final : public Actor<LogicalPageID>, public ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >, public FastAllocated<NewExtentPageID_implActor>, public NewExtentPageID_implActorState<NewExtentPageID_implActor> {
															#line 8855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<NewExtentPageID_implActor>::operator new;
	using FastAllocated<NewExtentPageID_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16478461426711525632UL, 10507358005514695680UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<LogicalPageID>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >;
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	NewExtentPageID_implActor(DWALPager* const& self,QueueID const& queueID) 
															#line 8872 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<LogicalPageID>(),
		   NewExtentPageID_implActorState<NewExtentPageID_implActor>(self, queueID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12285413946799175936UL, 12448560763938015744UL);
		ActorExecutionContextHelper __helper(static_cast<NewExtentPageID_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("newExtentPageID_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("newExtentPageID_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< NewExtentPageID_implActor, 0, Optional<LogicalPageID> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<LogicalPageID> newExtentPageID_impl( DWALPager* const& self, QueueID const& queueID ) {
															#line 2468 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<LogicalPageID>(new NewExtentPageID_implActor(self, queueID));
															#line 8904 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2489 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	// Grow the pager file by one extent and return it
	// We reserve all the pageIDs within the extent during this step
	// That translates to extentID being same as the return first pageID
	LogicalPageID newLastExtentID() {
		LogicalPageID id = header.pageCount;
		growPager(pagesPerExtent);
		return id;
	}

	Future<LogicalPageID> newExtentPageID(QueueID queueID) override { return newExtentPageID_impl(this, queueID); }

	// Write one block of a page of a physical page in the page file.  Futures returned must be allowed to complete.
																#line 8921 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via writePhysicalBlock()
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class WritePhysicalBlockActor>
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class WritePhysicalBlockActorState {
															#line 8927 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	WritePhysicalBlockActorState(DWALPager* const& self,Reference<ArenaPage> const& page,int const& blockNum,int const& blockSize,PhysicalPageID const& pageID,PagerEventReasons const& reason,unsigned int const& level,bool const& header) 
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   page(page),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   blockNum(blockNum),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   blockSize(blockSize),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageID(pageID),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   reason(reason),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   level(level),
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   header(header)
															#line 8948 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("writePhysicalBlock", reinterpret_cast<unsigned long>(this));

	}
	~WritePhysicalBlockActorState() 
	{
		fdb_probe_actor_destroy("writePhysicalBlock", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<PriorityMultiLock::Lock> __when_expr_0 = self->ioLock->lock(header ? ioMaxPriority : ioMinPriority);
															#line 2511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 8965 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 1;
															#line 2511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >*>(static_cast<WritePhysicalBlockActor*>(this)));
															#line 8969 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WritePhysicalBlockActorState();
		static_cast<WritePhysicalBlockActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2512 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++g_redwoodMetrics.metric.pagerDiskWrite;
															#line 2513 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		g_redwoodMetrics.level(level).metrics.events.addEventReason(PagerEvents::PageWrite, reason);
															#line 2514 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->memoryOnly)
															#line 8996 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!static_cast<WritePhysicalBlockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WritePhysicalBlockActorState(); static_cast<WritePhysicalBlockActor*>(this)->destroy(); return 0; }
															#line 9000 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			new (&static_cast<WritePhysicalBlockActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WritePhysicalBlockActorState();
			static_cast<WritePhysicalBlockActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2519 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (pageID >= self->filePageCount)
															#line 9008 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (pageID >= self->filePageCountPending)
															#line 9012 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2523 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->fileExtension = extendToCover(self, pageID, self->fileExtension);
															#line 9016 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 2527 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = self->fileExtension;
															#line 2527 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 9022 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 2;
															#line 2527 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WritePhysicalBlockActor, 1, Void >*>(static_cast<WritePhysicalBlockActor*>(this)));
															#line 9026 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(PriorityMultiLock::Lock const& __lock,int loopDepth) 
	{
															#line 2511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		lock = __lock;
															#line 9040 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(PriorityMultiLock::Lock && __lock,int loopDepth) 
	{
		lock = std::move(__lock);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state > 0) static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 0;
		static_cast<WritePhysicalBlockActor*>(this)->ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2531 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=writeBlock %s\n", self->filename.c_str(), toString(pageID).c_str());
															#line 2532 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->pageFile->write(page->rawData() + (blockNum * blockSize), blockSize, (int64_t)pageID * blockSize);
															#line 2532 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
															#line 9123 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 3;
															#line 2532 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WritePhysicalBlockActor, 2, Void >*>(static_cast<WritePhysicalBlockActor*>(this)));
															#line 9127 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state > 0) static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 0;
		static_cast<WritePhysicalBlockActor*>(this)->ActorCallback< WritePhysicalBlockActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WritePhysicalBlockActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2538 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<WritePhysicalBlockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WritePhysicalBlockActorState(); static_cast<WritePhysicalBlockActor*>(this)->destroy(); return 0; }
															#line 9223 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<WritePhysicalBlockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WritePhysicalBlockActorState();
		static_cast<WritePhysicalBlockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2538 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<WritePhysicalBlockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WritePhysicalBlockActorState(); static_cast<WritePhysicalBlockActor*>(this)->destroy(); return 0; }
															#line 9235 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<WritePhysicalBlockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WritePhysicalBlockActorState();
		static_cast<WritePhysicalBlockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state > 0) static_cast<WritePhysicalBlockActor*>(this)->actor_wait_state = 0;
		static_cast<WritePhysicalBlockActor*>(this)->ActorCallback< WritePhysicalBlockActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WritePhysicalBlockActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WritePhysicalBlockActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> page;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int blockNum;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int blockSize;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PhysicalPageID pageID;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PagerEventReasons reason;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	unsigned int level;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool header;
															#line 2511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PriorityMultiLock::Lock lock;
															#line 9336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via writePhysicalBlock()
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class WritePhysicalBlockActor final : public Actor<Void>, public ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >, public ActorCallback< WritePhysicalBlockActor, 1, Void >, public ActorCallback< WritePhysicalBlockActor, 2, Void >, public FastAllocated<WritePhysicalBlockActor>, public WritePhysicalBlockActorState<WritePhysicalBlockActor> {
															#line 9341 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<WritePhysicalBlockActor>::operator new;
	using FastAllocated<WritePhysicalBlockActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4539039663522364928UL, 15598005294190562304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WritePhysicalBlockActor, 0, PriorityMultiLock::Lock >;
friend struct ActorCallback< WritePhysicalBlockActor, 1, Void >;
friend struct ActorCallback< WritePhysicalBlockActor, 2, Void >;
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	WritePhysicalBlockActor(DWALPager* const& self,Reference<ArenaPage> const& page,int const& blockNum,int const& blockSize,PhysicalPageID const& pageID,PagerEventReasons const& reason,unsigned int const& level,bool const& header) 
															#line 9360 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   WritePhysicalBlockActorState<WritePhysicalBlockActor>(self, page, blockNum, blockSize, pageID, reason, level, header),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writePhysicalBlock", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11949412618454678784UL, 4346150413405985024UL);
		ActorExecutionContextHelper __helper(static_cast<WritePhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writePhysicalBlock");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writePhysicalBlock", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
static Future<Void> writePhysicalBlock( DWALPager* const& self, Reference<ArenaPage> const& page, int const& blockNum, int const& blockSize, PhysicalPageID const& pageID, PagerEventReasons const& reason, unsigned int const& level, bool const& header ) {
															#line 2502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new WritePhysicalBlockActor(self, page, blockNum, blockSize, pageID, reason, level, header));
															#line 9383 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2540 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

																#line 9388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via extendToCover()
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ExtendToCoverActor>
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ExtendToCoverActorState {
															#line 9394 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ExtendToCoverActorState(DWALPager* const& self,uint64_t const& pageID,Future<Void> const& previousExtension) 
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageID(pageID),
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   previousExtension(previousExtension),
															#line 2543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   newPageCount(pageID + SERVER_KNOBS->REDWOOD_PAGEFILE_GROWTH_SIZE_PAGES - (pageID % SERVER_KNOBS->REDWOOD_PAGEFILE_GROWTH_SIZE_PAGES))
															#line 9407 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("extendToCover", reinterpret_cast<unsigned long>(this));

	}
	~ExtendToCoverActorState() 
	{
		fdb_probe_actor_destroy("extendToCover", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2547 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->filePageCountPending = newPageCount;
															#line 2550 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = previousExtension;
															#line 2550 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ExtendToCoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9426 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExtendToCoverActor*>(this)->actor_wait_state = 1;
															#line 2550 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExtendToCoverActor, 0, Void >*>(static_cast<ExtendToCoverActor*>(this)));
															#line 9431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExtendToCoverActorState();
		static_cast<ExtendToCoverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->pageFile->truncate(newPageCount * self->physicalPageSize);
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ExtendToCoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9456 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ExtendToCoverActor*>(this)->actor_wait_state = 2;
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExtendToCoverActor, 1, Void >*>(static_cast<ExtendToCoverActor*>(this)));
															#line 9461 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->pageFile->truncate(newPageCount * self->physicalPageSize);
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ExtendToCoverActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9472 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ExtendToCoverActor*>(this)->actor_wait_state = 2;
															#line 2553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExtendToCoverActor, 1, Void >*>(static_cast<ExtendToCoverActor*>(this)));
															#line 9477 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExtendToCoverActor*>(this)->actor_wait_state > 0) static_cast<ExtendToCoverActor*>(this)->actor_wait_state = 0;
		static_cast<ExtendToCoverActor*>(this)->ActorCallback< ExtendToCoverActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExtendToCoverActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExtendToCoverActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExtendToCoverActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2556 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->filePageCount = newPageCount;
															#line 2558 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ExtendToCoverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExtendToCoverActorState(); static_cast<ExtendToCoverActor*>(this)->destroy(); return 0; }
															#line 9563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ExtendToCoverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExtendToCoverActorState();
		static_cast<ExtendToCoverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2556 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->filePageCount = newPageCount;
															#line 2558 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ExtendToCoverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExtendToCoverActorState(); static_cast<ExtendToCoverActor*>(this)->destroy(); return 0; }
															#line 9577 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ExtendToCoverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExtendToCoverActorState();
		static_cast<ExtendToCoverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ExtendToCoverActor*>(this)->actor_wait_state > 0) static_cast<ExtendToCoverActor*>(this)->actor_wait_state = 0;
		static_cast<ExtendToCoverActor*>(this)->ActorCallback< ExtendToCoverActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExtendToCoverActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ExtendToCoverActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ExtendToCoverActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	uint64_t pageID;
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Future<Void> previousExtension;
															#line 2543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int64_t newPageCount;
															#line 9668 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via extendToCover()
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ExtendToCoverActor final : public Actor<Void>, public ActorCallback< ExtendToCoverActor, 0, Void >, public ActorCallback< ExtendToCoverActor, 1, Void >, public FastAllocated<ExtendToCoverActor>, public ExtendToCoverActorState<ExtendToCoverActor> {
															#line 9673 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ExtendToCoverActor>::operator new;
	using FastAllocated<ExtendToCoverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8689354299987556096UL, 3828584883333479424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExtendToCoverActor, 0, Void >;
friend struct ActorCallback< ExtendToCoverActor, 1, Void >;
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ExtendToCoverActor(DWALPager* const& self,uint64_t const& pageID,Future<Void> const& previousExtension) 
															#line 9691 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   ExtendToCoverActorState<ExtendToCoverActor>(self, pageID, previousExtension),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("extendToCover", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8662340340325824256UL, 2028498679589298688UL);
		ActorExecutionContextHelper __helper(static_cast<ExtendToCoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("extendToCover");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("extendToCover", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExtendToCoverActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ExtendToCoverActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> extendToCover( DWALPager* const& self, uint64_t const& pageID, Future<Void> const& previousExtension ) {
															#line 2541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new ExtendToCoverActor(self, pageID, previousExtension));
															#line 9724 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2560 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	// All returned futures are added to the operations vector
	Future<Void> writePhysicalPage(PagerEventReasons reason,
	                               unsigned int level,
	                               Standalone<VectorRef<PhysicalPageID>> pageIDs,
	                               Reference<ArenaPage> page,
	                               bool header = false) {
		debug_printf("DWALPager(%s) op=%s %s encoding=%d ptr=%p\n",
		             filename.c_str(),
		             (header ? "writePhysicalHeader" : "writePhysicalPage"),
		             toString(pageIDs).c_str(),
		             page->getEncodingType(),
		             page->rawData());

		// Set metadata before prewrite so it's in the pre-encrypted page in cache if the page is encrypted
		// The actual next commit version is unknown, so the write version of a page is always the
		// last committed version + 1
		page->setWriteInfo(pageIDs.front(), this->getLastCommittedVersion() + 1);

		// Copy the page if preWrite will encrypt/modify the payload
		bool copy = page->isEncrypted();
		if (copy) {
			page = page->clone();
		}

		page->preWrite(pageIDs.front());

		int blockSize = header ? smallestPhysicalBlock : physicalPageSize;
		Future<Void> f;
		if (pageIDs.size() == 1) {
			f = writePhysicalBlock(this, page, 0, blockSize, pageIDs.front(), reason, level, header);
		} else {
			std::vector<Future<Void>> writers;
			for (int i = 0; i < pageIDs.size(); ++i) {
				Future<Void> p = writePhysicalBlock(this, page, i, blockSize, pageIDs[i], reason, level, header);
				writers.push_back(p);
			}
			f = waitForAll(writers);
		}

		operations.push_back(f);
		return f;
	}

	Future<Void> writeHeaderPage(PhysicalPageID pageID, Reference<ArenaPage> page) {
		return writePhysicalPage(
		    PagerEventReasons::MetaData, nonBtreeLevel, VectorRef<PhysicalPageID>(&pageID, 1), page, true);
	}

	void updatePage(PagerEventReasons reason,
	                unsigned int level,
	                Standalone<VectorRef<LogicalPageID>> pageIDs,
	                Reference<ArenaPage> data) override {
		// Get the cache entry for this page, without counting it as a cache hit as we're replacing its contents now
		// or as a cache miss because there is no benefit to the page already being in cache
		// Similarly, this does not count as a point lookup for reason.
		ASSERT(pageIDs.front() != invalidLogicalPageID);
		PageCacheEntry& cacheEntry = pageCache.get(pageIDs.front(), pageIDs.size() * physicalPageSize, true);
		debug_printf("DWALPager(%s) op=write %s cached=%d reading=%d writing=%d\n",
		             filename.c_str(),
		             toString(pageIDs).c_str(),
		             cacheEntry.initialized(),
		             cacheEntry.initialized() && cacheEntry.reading(),
		             cacheEntry.initialized() && cacheEntry.writing());

		// If the page is still being read then it's not also being written because a write places
		// the new content into readFuture when the write is launched, not when it is completed.
		// Read/write ordering is being enforced so waiting readers will not see the new write.  This
		// is necessary for remap erasure to work correctly since the oldest version of a page, located
		// at the original page ID, could have a pending read when that version is expired (after which
		// future reads of the version are not allowed) and the write of the next newest version over top
		// of the original page begins.
		if (!cacheEntry.initialized()) {
			cacheEntry.writeFuture = detach(writePhysicalPage(reason, level, pageIDs, data));
		} else if (cacheEntry.reading()) {
			// This is very unlikely, maybe impossible in the current pager use cases
			// Wait for the outstanding read to finish, then start the write
			cacheEntry.writeFuture = mapAsync(success(cacheEntry.readFuture),
			                                  [=](Void) { return writePhysicalPage(reason, level, pageIDs, data); });
		}

		// If the page is being written, wait for this write before issuing the new write to ensure the
		// writes happen in the correct order
		else if (cacheEntry.writing()) {
			// This is very unlikely, maybe impossible in the current pager use cases
			// Wait for the previous write to finish, then start new write
			cacheEntry.writeFuture =
			    mapAsync(cacheEntry.writeFuture, [=](Void) { return writePhysicalPage(reason, level, pageIDs, data); });
		} else {
			cacheEntry.writeFuture = detach(writePhysicalPage(reason, level, pageIDs, data));
		}

		// Always update the page contents immediately regardless of what happened above.
		cacheEntry.readFuture = data;
	}

	Future<LogicalPageID> atomicUpdatePage(PagerEventReasons reason,
	                                       unsigned int level,
	                                       LogicalPageID pageID,
	                                       Reference<ArenaPage> data,
	                                       Version v) override {
		debug_printf("DWALPager(%s) op=writeAtomic %s @%" PRId64 "\n", filename.c_str(), toString(pageID).c_str(), v);
		Future<LogicalPageID> f = map(newPageID(), [=](LogicalPageID newPageID) {
			updatePage(reason, level, VectorRef<LogicalPageID>(&newPageID, 1), data);
			// TODO:  Possibly limit size of remap queue since it must be recovered on cold start
			RemappedPage r{ v, pageID, newPageID };
			remapQueue.pushBack(r);
			auto& versionedMap = remappedPages[pageID];

			if (SERVER_KNOBS->REDWOOD_EVICT_UPDATED_PAGES) {
				// An update page is unlikely to have its old version read again soon, so prioritize its cache eviction
				// If the versioned map is empty for this page then the prior version of the page is at stored at the
				// PhysicalPageID pageID, otherwise it is the last mapped value in the version-ordered map.
				pageCache.prioritizeEviction(versionedMap.empty() ? pageID : versionedMap.rbegin()->second);
			}
			versionedMap[v] = newPageID;

			debug_printf("DWALPager(%s) pushed %s\n", filename.c_str(), RemappedPage(r).toString().c_str());
			return pageID;
		});

		return f;
	}

	// Free pageID as of version v.  This means that once the oldest readable pager snapshot is at version v, pageID is
	// not longer in use by any structure so it can be used to write new data.
	void freeUnmappedPage(PhysicalPageID pageID, Version v) {
		// If v is older than the oldest version still readable then mark pageID as free as of the next commit
		if (v < effectiveOldestVersion()) {
			debug_printf("DWALPager(%s) op=freeNow %s @%" PRId64 " oldestVersion=%" PRId64 "\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             v,
			             lastCommittedHeader.oldestVersion);
			freeList.pushBack(pageID);
		} else {
			// Otherwise add it to the delayed free list
			debug_printf("DWALPager(%s) op=freeLater %s @%" PRId64 " oldestVersion=%" PRId64 "\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             v,
			             lastCommittedHeader.oldestVersion);
			delayedFreeList.pushBack({ v, pageID });
		}

		// A freed page is unlikely to be read again soon so prioritize its cache eviction
		if (SERVER_KNOBS->REDWOOD_EVICT_UPDATED_PAGES) {
			pageCache.prioritizeEviction(pageID);
		}
	}

	LogicalPageID detachRemappedPage(LogicalPageID pageID, Version v) override {
		auto i = remappedPages.find(pageID);
		if (i == remappedPages.end()) {
			// Page is not remapped
			return invalidLogicalPageID;
		}

		// Get the page that id was most recently remapped to
		auto iLast = i->second.rbegin();
		LogicalPageID newID = iLast->second;
		ASSERT(RemappedPage::getTypeOf(newID) == RemappedPage::REMAP);

		// If the last change remap was also at v then change the remap to a delete, as it's essentially
		// the same as the original page being deleted at that version and newID being used from then on.
		if (iLast->first == v) {
			debug_printf("DWALPager(%s) op=detachDelete originalID=%s newID=%s @%" PRId64 " oldestVersion=%" PRId64
			             "\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             toString(newID).c_str(),
			             v,
			             lastCommittedHeader.oldestVersion);
			iLast->second = invalidLogicalPageID;
			remapQueue.pushBack(RemappedPage{ v, pageID, invalidLogicalPageID });
		} else {
			debug_printf("DWALPager(%s) op=detach originalID=%s newID=%s @%" PRId64 " oldestVersion=%" PRId64 "\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             toString(newID).c_str(),
			             v,
			             lastCommittedHeader.oldestVersion);
			// Mark id as converted to its last remapped location as of v
			i->second[v] = 0;
			remapQueue.pushBack(RemappedPage{ v, pageID, 0 });
		}
		return newID;
	}

	void freePage(LogicalPageID pageID, Version v) override {
		// If pageID has been remapped, then it can't be freed until all existing remaps for that page have been undone,
		// so queue it for later deletion during remap cleanup
		auto i = remappedPages.find(pageID);
		if (i != remappedPages.end()) {
			debug_printf("DWALPager(%s) op=freeRemapped %s @%" PRId64 " oldestVersion=%" PRId64 "\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             v,
			             lastCommittedHeader.oldestVersion);
			remapQueue.pushBack(RemappedPage{ v, pageID, invalidLogicalPageID });

			// A freed page is unlikely to be read again soon so prioritize its cache eviction
			if (SERVER_KNOBS->REDWOOD_EVICT_UPDATED_PAGES) {
				PhysicalPageID previousPhysicalPage = i->second.rbegin()->second;
				pageCache.prioritizeEviction(previousPhysicalPage);
			}

			i->second[v] = invalidLogicalPageID;
			return;
		}

		freeUnmappedPage(pageID, v);
	};

																#line 9942 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via freeExtent_impl()
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class FreeExtent_implActor>
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class FreeExtent_implActorState {
															#line 9948 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FreeExtent_implActorState(DWALPager* const& self,LogicalPageID const& pageID) 
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageID(pageID)
															#line 9957 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("freeExtent_impl", reinterpret_cast<unsigned long>(this));

	}
	~FreeExtent_implActorState() 
	{
		fdb_probe_actor_destroy("freeExtent_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2775 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->extentFreeList.pushBack(pageID);
															#line 2776 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Optional<ExtentUsedListEntry>> __when_expr_0 = self->extentUsedList.pop();
															#line 2776 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 9976 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<FreeExtent_implActor*>(this)->actor_wait_state = 1;
															#line 2776 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >*>(static_cast<FreeExtent_implActor*>(this)));
															#line 9980 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<FreeExtent_implActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<ExtentUsedListEntry> const& freeExtent,int loopDepth) 
	{
															#line 2778 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeExtent.present())
															#line 10002 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2779 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) freeExtentPageID() popped %s from used list\n", self->filename.c_str(), toString(freeExtent.get().extentID).c_str());
															#line 10006 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Optional<ExtentUsedListEntry> && freeExtent,int loopDepth) 
	{
															#line 2778 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeExtent.present())
															#line 10016 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2779 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) freeExtentPageID() popped %s from used list\n", self->filename.c_str(), toString(freeExtent.get().extentID).c_str());
															#line 10020 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<ExtentUsedListEntry> const& freeExtent,int loopDepth) 
	{
		loopDepth = a_body1cont1(freeExtent, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<ExtentUsedListEntry> && freeExtent,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(freeExtent), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FreeExtent_implActor*>(this)->actor_wait_state > 0) static_cast<FreeExtent_implActor*>(this)->actor_wait_state = 0;
		static_cast<FreeExtent_implActor*>(this)->ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >::remove();

	}
	void a_callback_fire(ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >*,Optional<ExtentUsedListEntry> const& value) 
	{
		fdb_probe_actor_enter("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FreeExtent_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >*,Optional<ExtentUsedListEntry> && value) 
	{
		fdb_probe_actor_enter("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FreeExtent_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >*,Error err) 
	{
		fdb_probe_actor_enter("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FreeExtent_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("freeExtent_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		delete static_cast<FreeExtent_implActor*>(this);
															#line 10105 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		return 0;

		return loopDepth;
	}
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	LogicalPageID pageID;
															#line 10114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via freeExtent_impl()
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class FreeExtent_implActor final : public Actor<void>, public ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >, public FastAllocated<FreeExtent_implActor>, public FreeExtent_implActorState<FreeExtent_implActor> {
															#line 10119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<FreeExtent_implActor>::operator new;
	using FastAllocated<FreeExtent_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15008458189904597248UL, 11708369033014074368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< FreeExtent_implActor, 0, Optional<ExtentUsedListEntry> >;
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FreeExtent_implActor(DWALPager* const& self,LogicalPageID const& pageID) 
															#line 10136 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<void>(),
		   FreeExtent_implActorState<FreeExtent_implActor>(self, pageID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("freeExtent_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5520210414864808704UL, 13568158073043096320UL);
		ActorExecutionContextHelper __helper(static_cast<FreeExtent_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("freeExtent_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("freeExtent_impl", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
static void freeExtent_impl( DWALPager* const& self, LogicalPageID const& pageID ) {
															#line 2774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	new FreeExtent_implActor(self, pageID);
															#line 10159 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2784 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	void freeExtent(LogicalPageID pageID) override { freeExtent_impl(this, pageID); }

																#line 10165 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via readPhysicalBlock()
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ReadPhysicalBlockActor>
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalBlockActorState {
															#line 10171 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalBlockActorState(DWALPager* const& self,Reference<ArenaPage> const& pageBuffer,int const& pageOffset,int const& blockSize,int64_t const& offset,int const& priority) 
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageBuffer(pageBuffer),
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageOffset(pageOffset),
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   blockSize(blockSize),
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   offset(offset),
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   priority(priority)
															#line 10188 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("readPhysicalBlock", reinterpret_cast<unsigned long>(this));

	}
	~ReadPhysicalBlockActorState() 
	{
		fdb_probe_actor_destroy("readPhysicalBlock", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<PriorityMultiLock::Lock> __when_expr_0 = self->ioLock->lock(std::min(priority, ioMaxPriority));
															#line 2792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 10205 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state = 1;
															#line 2792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >*>(static_cast<ReadPhysicalBlockActor*>(this)));
															#line 10209 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadPhysicalBlockActorState();
		static_cast<ReadPhysicalBlockActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2793 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++g_redwoodMetrics.metric.pagerDiskRead;
															#line 2794 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<int> __when_expr_1 = self->pageFile->read(pageBuffer->rawData() + pageOffset, blockSize, offset);
															#line 2794 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 10236 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state = 2;
															#line 2794 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalBlockActor, 1, int >*>(static_cast<ReadPhysicalBlockActor*>(this)));
															#line 10240 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(PriorityMultiLock::Lock const& __lock,int loopDepth) 
	{
															#line 2792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		lock = __lock;
															#line 10249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(PriorityMultiLock::Lock && __lock,int loopDepth) 
	{
		lock = std::move(__lock);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalBlockActor*>(this)->ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int const& bytes,int loopDepth) 
	{
															#line 2795 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalBlockActor*>(this)->SAV<int>::futures) { (void)(bytes); this->~ReadPhysicalBlockActorState(); static_cast<ReadPhysicalBlockActor*>(this)->destroy(); return 0; }
															#line 10328 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalBlockActor*>(this)->SAV< int >::value()) int(bytes);
		this->~ReadPhysicalBlockActorState();
		static_cast<ReadPhysicalBlockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(int && bytes,int loopDepth) 
	{
															#line 2795 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalBlockActor*>(this)->SAV<int>::futures) { (void)(bytes); this->~ReadPhysicalBlockActorState(); static_cast<ReadPhysicalBlockActor*>(this)->destroy(); return 0; }
															#line 10340 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalBlockActor*>(this)->SAV< int >::value()) int(bytes);
		this->~ReadPhysicalBlockActorState();
		static_cast<ReadPhysicalBlockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(int const& bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2(bytes, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int && bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bytes), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalBlockActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalBlockActor*>(this)->ActorCallback< ReadPhysicalBlockActor, 1, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalBlockActor, 1, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalBlockActor, 1, int >*,int && value) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadPhysicalBlockActor, 1, int >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> pageBuffer;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int pageOffset;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int blockSize;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int64_t offset;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int priority;
															#line 2792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PriorityMultiLock::Lock lock;
															#line 10437 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via readPhysicalBlock()
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalBlockActor final : public Actor<int>, public ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >, public ActorCallback< ReadPhysicalBlockActor, 1, int >, public FastAllocated<ReadPhysicalBlockActor>, public ReadPhysicalBlockActorState<ReadPhysicalBlockActor> {
															#line 10442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ReadPhysicalBlockActor>::operator new;
	using FastAllocated<ReadPhysicalBlockActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2546048436135301632UL, 13619721545528183808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadPhysicalBlockActor, 0, PriorityMultiLock::Lock >;
friend struct ActorCallback< ReadPhysicalBlockActor, 1, int >;
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalBlockActor(DWALPager* const& self,Reference<ArenaPage> const& pageBuffer,int const& pageOffset,int const& blockSize,int64_t const& offset,int const& priority) 
															#line 10460 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<int>(),
		   ReadPhysicalBlockActorState<ReadPhysicalBlockActor>(self, pageBuffer, pageOffset, blockSize, offset, priority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readPhysicalBlock", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16406985691053122048UL, 13557119125368433664UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalBlockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readPhysicalBlock");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readPhysicalBlock", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
static Future<int> readPhysicalBlock( DWALPager* const& self, Reference<ArenaPage> const& pageBuffer, int const& pageOffset, int const& blockSize, int64_t const& offset, int const& priority ) {
															#line 2786 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<int>(new ReadPhysicalBlockActor(self, pageBuffer, pageOffset, blockSize, offset, priority));
															#line 10483 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2797 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	// Read a physical page from the page file.  Note that header pages use a page size of smallestPhysicalBlock.
	// If the user chosen physical page size is larger, then there will be a gap of unused space after the header pages
	// and before the user-chosen sized pages.
																#line 10491 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via readPhysicalPage()
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ReadPhysicalPageActor>
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalPageActorState {
															#line 10497 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalPageActorState(DWALPager* const& self,PhysicalPageID const& pageID,int const& priority,bool const& header,PagerEventReasons const& reason) 
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageID(pageID),
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   priority(priority),
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   header(header),
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   reason(reason)
															#line 10512 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("readPhysicalPage", reinterpret_cast<unsigned long>(this));

	}
	~ReadPhysicalPageActorState() 
	{
		fdb_probe_actor_destroy("readPhysicalPage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2806 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(!self->memoryOnly);
															#line 2808 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page = header ? makeReference<ArenaPage>(smallestPhysicalBlock, smallestPhysicalBlock) : self->newPageBuffer();
															#line 2810 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) op=readPhysicalStart %s ptr=%p header=%d\n", self->filename.c_str(), toString(pageID).c_str(), page->rawData(), header);
															#line 2816 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<int> __when_expr_0 = readPhysicalBlock(self, page, 0, page->rawSize(), (int64_t)pageID * page->rawSize(), priority);
															#line 2816 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10535 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 1;
															#line 2816 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalPageActor, 0, int >*>(static_cast<ReadPhysicalPageActor*>(this)));
															#line 10540 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadPhysicalPageActorState();
		static_cast<ReadPhysicalPageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int const& readBytes,int loopDepth) 
	{
															#line 2818 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalDiskReadComplete %s ptr=%p bytes=%d\n", self->filename.c_str(), toString(pageID).c_str(), page->rawData(), readBytes);
															#line 10563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2825 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadHeader(pageID);
															#line 2826 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (page->isEncrypted())
															#line 10569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2827 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!self->keyProvider.isValid())
															#line 10573 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					StrictFuture<Void> __when_expr_1 = self->keyProviderInitialized.getFuture();
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 10579 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 2;
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalPageActor, 1, Void >*>(static_cast<ReadPhysicalPageActor*>(this)));
															#line 10584 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont4(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(int && readBytes,int loopDepth) 
	{
															#line 2818 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalDiskReadComplete %s ptr=%p bytes=%d\n", self->filename.c_str(), toString(pageID).c_str(), page->rawData(), readBytes);
															#line 10609 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2825 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadHeader(pageID);
															#line 2826 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (page->isEncrypted())
															#line 10615 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2827 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!self->keyProvider.isValid())
															#line 10619 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					StrictFuture<Void> __when_expr_1 = self->keyProviderInitialized.getFuture();
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 10625 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 2;
															#line 2828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalPageActor, 1, Void >*>(static_cast<ReadPhysicalPageActor*>(this)));
															#line 10630 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont4(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(int const& readBytes,int loopDepth) 
	{
		loopDepth = a_body1cont1(readBytes, loopDepth);

		return loopDepth;
	}
	int a_body1when1(int && readBytes,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(readBytes), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalPageActor*>(this)->ActorCallback< ReadPhysicalPageActor, 0, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 0, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 0, int >*,int && value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadPhysicalPageActor, 0, int >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2873 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalPageActor*>(this)->SAV<Reference<ArenaPage>>::futures) { (void)(page); this->~ReadPhysicalPageActorState(); static_cast<ReadPhysicalPageActor*>(this)->destroy(); return 0; }
															#line 10730 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalPageActor*>(this)->SAV< Reference<ArenaPage> >::value()) Reference<ArenaPage>(std::move(page)); // state_var_RVO
		this->~ReadPhysicalPageActorState();
		static_cast<ReadPhysicalPageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2851 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			Error err = e;
															#line 2852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated() && g_simulator->checkInjectedCorruption())
															#line 10745 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2853 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				err = err.asInjectedFault();
															#line 10749 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 2858 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(header ? SevWarnAlways : SevError, "RedwoodPageError") .error(err) .detail("Filename", self->filename.c_str()) .detail("PageID", pageID) .detail("PageSize", self->physicalPageSize) .detail("Offset", pageID * self->physicalPageSize);
															#line 2865 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) postread failed for %s with %s\n", self->filename.c_str(), toString(pageID).c_str(), err.what());
															#line 2870 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(err, loopDepth);
															#line 10757 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double decryptTime = 0;
															#line 2842 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->postReadPayload(pageID, &decryptTime);
															#line 2843 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (isReadRequest(reason))
															#line 10775 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2844 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			g_redwoodMetrics.metric.readRequestDecryptTimeNS += int64_t(decryptTime * 1e9);
															#line 10779 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2846 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalVerified %s ptr=%p\n", self->filename.c_str(), toString(pageID).c_str(), page->rawData());
															#line 10783 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2831 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->keyProvider->expectedEncodingType() != page->getEncodingType())
															#line 10792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2832 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarnAlways, "RedwoodBTreeUnexpectedNodeEncoding") .detail("PhysicalPageID", page->getPhysicalPageID()) .detail("EncodingTypeFound", page->getEncodingType()) .detail("EncodingTypeExpected", self->keyProvider->expectedEncodingType());
															#line 2836 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1cont1Catch1(unexpected_encoding_type(), loopDepth);
															#line 10798 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2838 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<ArenaPage::EncryptionKey> __when_expr_2 = self->keyProvider->getEncryptionKey(page->getEncodingHeader());
															#line 2838 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 10804 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 3;
															#line 2838 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >*>(static_cast<ReadPhysicalPageActor*>(this)));
															#line 10809 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2829 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(self->keyProvider.isValid());
															#line 10818 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2829 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(self->keyProvider.isValid());
															#line 10827 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalPageActor*>(this)->ActorCallback< ReadPhysicalPageActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadPhysicalPageActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
															#line 2839 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 10911 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
															#line 2839 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 10920 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
		loopDepth = a_body1cont7(k, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(k), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalPageActor*>(this)->ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey const& value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey && value) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PhysicalPageID pageID;
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int priority;
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool header;
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PagerEventReasons reason;
															#line 2808 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> page;
															#line 11025 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via readPhysicalPage()
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalPageActor final : public Actor<Reference<ArenaPage>>, public ActorCallback< ReadPhysicalPageActor, 0, int >, public ActorCallback< ReadPhysicalPageActor, 1, Void >, public ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >, public FastAllocated<ReadPhysicalPageActor>, public ReadPhysicalPageActorState<ReadPhysicalPageActor> {
															#line 11030 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ReadPhysicalPageActor>::operator new;
	using FastAllocated<ReadPhysicalPageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3868504474935246080UL, 4367732862147552256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<ArenaPage>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadPhysicalPageActor, 0, int >;
friend struct ActorCallback< ReadPhysicalPageActor, 1, Void >;
friend struct ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >;
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalPageActor(DWALPager* const& self,PhysicalPageID const& pageID,int const& priority,bool const& header,PagerEventReasons const& reason) 
															#line 11049 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Reference<ArenaPage>>(),
		   ReadPhysicalPageActorState<ReadPhysicalPageActor>(self, pageID, priority, header, reason),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readPhysicalPage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6101017109285837056UL, 11605873594164838400UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readPhysicalPage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readPhysicalPage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadPhysicalPageActor, 0, int >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadPhysicalPageActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadPhysicalPageActor, 2, ArenaPage::EncryptionKey >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Reference<ArenaPage>> readPhysicalPage( DWALPager* const& self, PhysicalPageID const& pageID, int const& priority, bool const& header, PagerEventReasons const& reason ) {
															#line 2801 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Reference<ArenaPage>>(new ReadPhysicalPageActor(self, pageID, priority, header, reason));
															#line 11083 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2875 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

																#line 11088 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via readPhysicalMultiPage()
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ReadPhysicalMultiPageActor>
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalMultiPageActorState {
															#line 11094 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalMultiPageActorState(DWALPager* const& self,Standalone<VectorRef<PhysicalPageID>> const& pageIDs,int const& priority,Optional<PagerEventReasons> const& reason) 
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageIDs(pageIDs),
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   priority(priority),
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   reason(reason)
															#line 11107 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this));

	}
	~ReadPhysicalMultiPageActorState() 
	{
		fdb_probe_actor_destroy("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2880 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(!self->memoryOnly);
															#line 2886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page = self->newPageBuffer(pageIDs.size());
															#line 2887 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) op=readPhysicalMultiStart %s ptr=%p\n", self->filename.c_str(), toString(pageIDs).c_str(), page->rawData());
															#line 2893 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			blockSize = self->physicalPageSize;
															#line 2894 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			std::vector<Future<int>> reads;
															#line 2895 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			for(int i = 0;i < pageIDs.size();++i) {
															#line 2896 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				reads.push_back( readPhysicalBlock(self, page, i * blockSize, blockSize, ((int64_t)pageIDs[i]) * blockSize, priority));
															#line 11134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 2900 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(reads);
															#line 2900 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11140 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 1;
															#line 2900 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalMultiPageActor, 0, Void >*>(static_cast<ReadPhysicalMultiPageActor*>(this)));
															#line 11145 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadPhysicalMultiPageActorState();
		static_cast<ReadPhysicalMultiPageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2902 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalMultiDiskReadsComplete %s ptr=%p bytes=%d\n", self->filename.c_str(), toString(pageIDs).c_str(), page->rawData(), pageIDs.size() * blockSize);
															#line 11168 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2909 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadHeader(pageIDs.front());
															#line 2910 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (page->isEncrypted())
															#line 11174 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2911 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!self->keyProvider.isValid())
															#line 11178 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					StrictFuture<Void> __when_expr_1 = self->keyProviderInitialized.getFuture();
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 11184 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 2;
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*>(static_cast<ReadPhysicalMultiPageActor*>(this)));
															#line 11189 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont5(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont4(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2902 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalMultiDiskReadsComplete %s ptr=%p bytes=%d\n", self->filename.c_str(), toString(pageIDs).c_str(), page->rawData(), pageIDs.size() * blockSize);
															#line 11214 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		try {
															#line 2909 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->postReadHeader(pageIDs.front());
															#line 2910 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (page->isEncrypted())
															#line 11220 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 2911 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (!self->keyProvider.isValid())
															#line 11224 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					StrictFuture<Void> __when_expr_1 = self->keyProviderInitialized.getFuture();
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 11230 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 2;
															#line 2912 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*>(static_cast<ReadPhysicalMultiPageActor*>(this)));
															#line 11235 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont5(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont4(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalMultiPageActor*>(this)->ActorCallback< ReadPhysicalMultiPageActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadPhysicalMultiPageActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2952 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalMultiPageActor*>(this)->SAV<Reference<ArenaPage>>::futures) { (void)(page); this->~ReadPhysicalMultiPageActorState(); static_cast<ReadPhysicalMultiPageActor*>(this)->destroy(); return 0; }
															#line 11335 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalMultiPageActor*>(this)->SAV< Reference<ArenaPage> >::value()) Reference<ArenaPage>(std::move(page)); // state_var_RVO
		this->~ReadPhysicalMultiPageActorState();
		static_cast<ReadPhysicalMultiPageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2938 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevError, "RedwoodPageError") .error(e) .detail("Filename", self->filename.c_str()) .detail("PageIDs", pageIDs) .detail("PageSize", self->physicalPageSize);
															#line 2944 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) postread failed for %s with %s\n", self->filename.c_str(), toString(pageIDs).c_str(), e.what());
															#line 2949 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 11352 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2925 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		double decryptTime = 0;
															#line 2926 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->postReadPayload(pageIDs.front(), &decryptTime);
															#line 2927 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (reason.present() && isReadRequest(reason.get()))
															#line 11370 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2928 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			g_redwoodMetrics.metric.readRequestDecryptTimeNS += int64_t(decryptTime * 1e9);
															#line 11374 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2931 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalVerified %s ptr=%p bytes=%d\n", self->filename.c_str(), toString(pageIDs).c_str(), page->rawData(), pageIDs.size() * blockSize);
															#line 11378 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2915 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->keyProvider->expectedEncodingType() != page->getEncodingType())
															#line 11387 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 2916 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			TraceEvent(SevWarnAlways, "RedwoodBTreeUnexpectedNodeEncoding") .detail("PhysicalPageID", page->getPhysicalPageID()) .detail("EncodingTypeFound", page->getEncodingType()) .detail("EncodingTypeExpected", self->keyProvider->expectedEncodingType());
															#line 2920 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			return a_body1cont1Catch1(unexpected_encoding_type(), loopDepth);
															#line 11393 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 2922 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<ArenaPage::EncryptionKey> __when_expr_2 = self->keyProvider->getEncryptionKey(page->getEncodingHeader());
															#line 2922 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 11399 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 3;
															#line 2922 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >*>(static_cast<ReadPhysicalMultiPageActor*>(this)));
															#line 11404 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2913 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(self->keyProvider.isValid());
															#line 11413 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2913 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(self->keyProvider.isValid());
															#line 11422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalMultiPageActor*>(this)->ActorCallback< ReadPhysicalMultiPageActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
															#line 2923 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 11506 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
															#line 2923 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 11515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
		loopDepth = a_body1cont8(k, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(k), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalMultiPageActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalMultiPageActor*>(this)->ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey const& value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey && value) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont13(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Standalone<VectorRef<PhysicalPageID>> pageIDs;
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int priority;
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Optional<PagerEventReasons> reason;
															#line 2886 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> page;
															#line 2893 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int blockSize;
															#line 11620 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via readPhysicalMultiPage()
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalMultiPageActor final : public Actor<Reference<ArenaPage>>, public ActorCallback< ReadPhysicalMultiPageActor, 0, Void >, public ActorCallback< ReadPhysicalMultiPageActor, 1, Void >, public ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >, public FastAllocated<ReadPhysicalMultiPageActor>, public ReadPhysicalMultiPageActorState<ReadPhysicalMultiPageActor> {
															#line 11625 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ReadPhysicalMultiPageActor>::operator new;
	using FastAllocated<ReadPhysicalMultiPageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2880206638603453440UL, 4699408781963930112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<ArenaPage>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadPhysicalMultiPageActor, 0, Void >;
friend struct ActorCallback< ReadPhysicalMultiPageActor, 1, Void >;
friend struct ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >;
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalMultiPageActor(DWALPager* const& self,Standalone<VectorRef<PhysicalPageID>> const& pageIDs,int const& priority,Optional<PagerEventReasons> const& reason) 
															#line 11644 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Reference<ArenaPage>>(),
		   ReadPhysicalMultiPageActorState<ReadPhysicalMultiPageActor>(self, pageIDs, priority, reason),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2933406179839034880UL, 1573163746616041216UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalMultiPageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readPhysicalMultiPage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readPhysicalMultiPage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadPhysicalMultiPageActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadPhysicalMultiPageActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadPhysicalMultiPageActor, 2, ArenaPage::EncryptionKey >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Reference<ArenaPage>> readPhysicalMultiPage( DWALPager* const& self, Standalone<VectorRef<PhysicalPageID>> const& pageIDs, int const& priority, Optional<PagerEventReasons> const& reason ) {
															#line 2876 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Reference<ArenaPage>>(new ReadPhysicalMultiPageActor(self, pageIDs, priority, reason));
															#line 11678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 2954 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Reference<ArenaPage>> readHeaderPage(PhysicalPageID pageID) {
		debug_printf("DWALPager(%s) readHeaderPage %s\n", filename.c_str(), toString(pageID).c_str());
		return readPhysicalPage(this, pageID, ioMaxPriority, true, PagerEventReasons::MetaData);
	}

	static bool isReadRequest(PagerEventReasons reason) {
		return reason == PagerEventReasons::PointRead || reason == PagerEventReasons::FetchRange ||
		       reason == PagerEventReasons::RangeRead || reason == PagerEventReasons::RangePrefetch;
	}

	// Reads the most recent version of pageID, either previously committed or written using updatePage()
	// in the current commit
	Future<Reference<ArenaPage>> readPage(PagerEventReasons reason,
	                                      unsigned int level,
	                                      PhysicalPageID pageID,
	                                      int priority,
	                                      bool cacheable,
	                                      bool noHit) override {
		// Use cached page if present, without triggering a cache hit.
		// Otherwise, read the page and return it but don't add it to the cache
		debug_printf("DWALPager(%s) op=read %s reason=%s  noHit=%d\n",
		             filename.c_str(),
		             toString(pageID).c_str(),
		             PagerEventReasonsStrings[(int)reason],
		             noHit);
		auto& eventReasons = g_redwoodMetrics.level(level).metrics.events;
		eventReasons.addEventReason(PagerEvents::CacheLookup, reason);
		if (!cacheable) {
			debug_printf("DWALPager(%s) op=readUncached %s\n", filename.c_str(), toString(pageID).c_str());
			PageCacheEntry* pCacheEntry = pageCache.getIfExists(pageID);
			if (pCacheEntry != nullptr) {
				++g_redwoodMetrics.metric.pagerProbeHit;
				debug_printf("DWALPager(%s) op=readUncachedHit %s\n", filename.c_str(), toString(pageID).c_str());
				return pCacheEntry->readFuture;
			}
			++g_redwoodMetrics.metric.pagerProbeMiss;
			debug_printf("DWALPager(%s) op=readUncachedMiss %s\n", filename.c_str(), toString(pageID).c_str());
			return forwardError(readPhysicalPage(this, pageID, priority, false, reason), errorPromise);
		}
		PageCacheEntry& cacheEntry = pageCache.get(pageID, physicalPageSize, noHit);
		debug_printf("DWALPager(%s) op=read %s cached=%d reading=%d writing=%d noHit=%d\n",
		             filename.c_str(),
		             toString(pageID).c_str(),
		             cacheEntry.initialized(),
		             cacheEntry.initialized() && cacheEntry.reading(),
		             cacheEntry.initialized() && cacheEntry.writing(),
		             noHit);
		if (!cacheEntry.initialized()) {
			debug_printf("DWALPager(%s) issuing actual read of %s\n", filename.c_str(), toString(pageID).c_str());
			cacheEntry.readFuture = forwardError(readPhysicalPage(this, pageID, priority, false, reason), errorPromise);
			cacheEntry.writeFuture = Void();

			++g_redwoodMetrics.metric.pagerCacheMiss;
			eventReasons.addEventReason(PagerEvents::CacheMiss, reason);
		} else {
			++g_redwoodMetrics.metric.pagerCacheHit;
			eventReasons.addEventReason(PagerEvents::CacheHit, reason);
		}
		return cacheEntry.readFuture;
	}

	Future<Reference<ArenaPage>> readMultiPage(PagerEventReasons reason,
	                                           unsigned int level,
	                                           VectorRef<PhysicalPageID> pageIDs,
	                                           int priority,
	                                           bool cacheable,
	                                           bool noHit) override {
		// Use cached page if present, without triggering a cache hit.
		// Otherwise, read the page and return it but don't add it to the cache
		debug_printf("DWALPager(%s) op=read %s reason=%s noHit=%d\n",
		             filename.c_str(),
		             toString(pageIDs).c_str(),
		             PagerEventReasonsStrings[(int)reason],
		             noHit);
		auto& eventReasons = g_redwoodMetrics.level(level).metrics.events;
		eventReasons.addEventReason(PagerEvents::CacheLookup, reason);
		if (!cacheable) {
			debug_printf("DWALPager(%s) op=readUncached %s\n", filename.c_str(), toString(pageIDs).c_str());
			PageCacheEntry* pCacheEntry = pageCache.getIfExists(pageIDs.front());
			if (pCacheEntry != nullptr) {
				++g_redwoodMetrics.metric.pagerProbeHit;
				debug_printf("DWALPager(%s) op=readUncachedHit %s\n", filename.c_str(), toString(pageIDs).c_str());
				return pCacheEntry->readFuture;
			}
			++g_redwoodMetrics.metric.pagerProbeMiss;
			debug_printf("DWALPager(%s) op=readUncachedMiss %s\n", filename.c_str(), toString(pageIDs).c_str());
			return forwardError(readPhysicalMultiPage(this, pageIDs, priority, reason), errorPromise);
		}

		PageCacheEntry& cacheEntry = pageCache.get(pageIDs.front(), pageIDs.size() * physicalPageSize, noHit);
		debug_printf("DWALPager(%s) op=read %s cached=%d reading=%d writing=%d noHit=%d\n",
		             filename.c_str(),
		             toString(pageIDs).c_str(),
		             cacheEntry.initialized(),
		             cacheEntry.initialized() && cacheEntry.reading(),
		             cacheEntry.initialized() && cacheEntry.writing(),
		             noHit);
		if (!cacheEntry.initialized()) {
			debug_printf("DWALPager(%s) issuing actual read of %s\n", filename.c_str(), toString(pageIDs).c_str());
			cacheEntry.readFuture = forwardError(readPhysicalMultiPage(this, pageIDs, priority, reason), errorPromise);
			cacheEntry.writeFuture = Void();

			++g_redwoodMetrics.metric.pagerCacheMiss;
			eventReasons.addEventReason(PagerEvents::CacheMiss, reason);
		} else {
			++g_redwoodMetrics.metric.pagerCacheHit;
			eventReasons.addEventReason(PagerEvents::CacheHit, reason);
		}
		return cacheEntry.readFuture;
	}

	PhysicalPageID getPhysicalPageID(LogicalPageID pageID, Version v) {
		auto i = remappedPages.find(pageID);

		if (i != remappedPages.end()) {
			auto j = i->second.upper_bound(v);
			if (j != i->second.begin()) {
				--j;
				debug_printf("DWALPager(%s) op=lookupRemapped %s @%" PRId64 " -> %s\n",
				             filename.c_str(),
				             toString(pageID).c_str(),
				             v,
				             toString(j->second).c_str());
				pageID = j->second;
				if (pageID == invalidLogicalPageID)
					debug_printf(
					    "DWALPager(%s) remappedPagesMap: %s\n", filename.c_str(), toString(remappedPages).c_str());

				ASSERT(pageID != invalidLogicalPageID);
			}
		} else {
			debug_printf("DWALPager(%s) op=lookupNotRemapped %s @%" PRId64 " (not remapped)\n",
			             filename.c_str(),
			             toString(pageID).c_str(),
			             v);
		}

		return (PhysicalPageID)pageID;
	}

	Future<Reference<ArenaPage>> readPageAtVersion(PagerEventReasons reason,
	                                               unsigned int level,
	                                               LogicalPageID logicalID,
	                                               int priority,
	                                               Version v,
	                                               bool cacheable,
	                                               bool noHit) {
		PhysicalPageID physicalID = getPhysicalPageID(logicalID, v);
		return readPage(reason, level, physicalID, priority, cacheable, noHit);
	}

	void releaseExtentReadLock() override { concurrentExtentReads->release(); }

	// Read the physical extent at given pageID
	// NOTE that we use the same interface (<ArenaPage>) for the extent as the page
																#line 11838 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via readPhysicalExtent()
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ReadPhysicalExtentActor>
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalExtentActorState {
															#line 11844 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalExtentActorState(DWALPager* const& self,PhysicalPageID const& pageID,int const& readSize = 0) 
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   pageID(pageID),
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   readSize(readSize)
															#line 11855 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("readPhysicalExtent", reinterpret_cast<unsigned long>(this));

	}
	~ReadPhysicalExtentActorState() 
	{
		fdb_probe_actor_destroy("readPhysicalExtent", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->concurrentExtentReads->take();
															#line 3114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11872 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 1;
															#line 3114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalExtentActor, 0, Void >*>(static_cast<ReadPhysicalExtentActor*>(this)));
															#line 11877 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadPhysicalExtentActorState();
		static_cast<ReadPhysicalExtentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(!self->memoryOnly);
															#line 3118 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (g_network->getCurrentTask() > TaskPriority::DiskRead)
															#line 11902 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::DiskRead);
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11908 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 2;
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalExtentActor, 1, Void >*>(static_cast<ReadPhysicalExtentActor*>(this)));
															#line 11913 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(!self->memoryOnly);
															#line 3118 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (g_network->getCurrentTask() > TaskPriority::DiskRead)
															#line 11929 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::DiskRead);
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11935 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 2;
															#line 3119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalExtentActor, 1, Void >*>(static_cast<ReadPhysicalExtentActor*>(this)));
															#line 11940 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalExtentActor*>(this)->ActorCallback< ReadPhysicalExtentActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadPhysicalExtentActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 3124 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (readSize == 0)
															#line 12029 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3125 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			readSize = self->physicalExtentSize;
															#line 12033 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3128 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		extent = makeReference<ArenaPage>(readSize, readSize);
															#line 3131 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto physicalReadSize = SERVER_KNOBS->REDWOOD_DEFAULT_EXTENT_READ_SIZE;
															#line 3132 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto parallelReads = readSize / physicalReadSize;
															#line 3133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto lastReadSize = readSize % physicalReadSize;
															#line 3135 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalExtentStart %s readSize %d offset %" PRId64 " physicalReadSize %d parallelReads %d\n", self->filename.c_str(), toString(pageID).c_str(), readSize, (int64_t)pageID * (self->physicalPageSize), physicalReadSize, parallelReads);
															#line 3147 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		std::vector<Future<int>> reads;
															#line 3148 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		int i;
															#line 3149 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		int64_t startOffset = (int64_t)pageID * (self->physicalPageSize);
															#line 3150 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		int64_t currentOffset;
															#line 3151 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for(i = 0;i < parallelReads;i++) {
															#line 3152 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			currentOffset = i * physicalReadSize;
															#line 3153 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) current offset %" PRId64 "\n", self->filename.c_str(), currentOffset);
															#line 3154 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			++g_redwoodMetrics.metric.pagerDiskRead;
															#line 3155 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			reads.push_back(self->readPhysicalBlock( self, extent, currentOffset, physicalReadSize, startOffset + currentOffset, ioMaxPriority));
															#line 12063 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (lastReadSize)
															#line 12067 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3161 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			currentOffset = i * physicalReadSize;
															#line 3162 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) iter %d current offset %" PRId64 " lastReadSize %d\n", self->filename.c_str(), i, currentOffset, lastReadSize);
															#line 3167 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			++g_redwoodMetrics.metric.pagerDiskRead;
															#line 3168 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			reads.push_back(self->readPhysicalBlock( self, extent, currentOffset, lastReadSize, startOffset + currentOffset, ioMaxPriority));
															#line 12077 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3173 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(reads);
															#line 3173 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12083 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 3;
															#line 3173 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadPhysicalExtentActor, 2, Void >*>(static_cast<ReadPhysicalExtentActor*>(this)));
															#line 12088 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalExtentActor*>(this)->ActorCallback< ReadPhysicalExtentActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadPhysicalExtentActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 3175 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalExtentComplete %s ptr=%p bytes=%d file offset=%d\n", self->filename.c_str(), toString(pageID).c_str(), extent->rawData(), readSize, (pageID * self->physicalPageSize));
															#line 3182 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalExtentActor*>(this)->SAV<Reference<ArenaPage>>::futures) { (void)(extent); this->~ReadPhysicalExtentActorState(); static_cast<ReadPhysicalExtentActor*>(this)->destroy(); return 0; }
															#line 12186 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalExtentActor*>(this)->SAV< Reference<ArenaPage> >::value()) Reference<ArenaPage>(std::move(extent)); // state_var_RVO
		this->~ReadPhysicalExtentActorState();
		static_cast<ReadPhysicalExtentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 3175 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) op=readPhysicalExtentComplete %s ptr=%p bytes=%d file offset=%d\n", self->filename.c_str(), toString(pageID).c_str(), extent->rawData(), readSize, (pageID * self->physicalPageSize));
															#line 3182 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ReadPhysicalExtentActor*>(this)->SAV<Reference<ArenaPage>>::futures) { (void)(extent); this->~ReadPhysicalExtentActorState(); static_cast<ReadPhysicalExtentActor*>(this)->destroy(); return 0; }
															#line 12200 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ReadPhysicalExtentActor*>(this)->SAV< Reference<ArenaPage> >::value()) Reference<ArenaPage>(std::move(extent)); // state_var_RVO
		this->~ReadPhysicalExtentActorState();
		static_cast<ReadPhysicalExtentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state > 0) static_cast<ReadPhysicalExtentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadPhysicalExtentActor*>(this)->ActorCallback< ReadPhysicalExtentActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadPhysicalExtentActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadPhysicalExtentActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PhysicalPageID pageID;
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int readSize;
															#line 3128 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> extent;
															#line 12291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via readPhysicalExtent()
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ReadPhysicalExtentActor final : public Actor<Reference<ArenaPage>>, public ActorCallback< ReadPhysicalExtentActor, 0, Void >, public ActorCallback< ReadPhysicalExtentActor, 1, Void >, public ActorCallback< ReadPhysicalExtentActor, 2, Void >, public FastAllocated<ReadPhysicalExtentActor>, public ReadPhysicalExtentActorState<ReadPhysicalExtentActor> {
															#line 12296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ReadPhysicalExtentActor>::operator new;
	using FastAllocated<ReadPhysicalExtentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1365234078905127680UL, 2410307220127239424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<ArenaPage>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadPhysicalExtentActor, 0, Void >;
friend struct ActorCallback< ReadPhysicalExtentActor, 1, Void >;
friend struct ActorCallback< ReadPhysicalExtentActor, 2, Void >;
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ReadPhysicalExtentActor(DWALPager* const& self,PhysicalPageID const& pageID,int const& readSize = 0) 
															#line 12315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Reference<ArenaPage>>(),
		   ReadPhysicalExtentActorState<ReadPhysicalExtentActor>(self, pageID, readSize),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readPhysicalExtent", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13313179281641170944UL, 8957613943258150656UL);
		ActorExecutionContextHelper __helper(static_cast<ReadPhysicalExtentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readPhysicalExtent");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readPhysicalExtent", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadPhysicalExtentActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadPhysicalExtentActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadPhysicalExtentActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Reference<ArenaPage>> readPhysicalExtent( DWALPager* const& self, PhysicalPageID const& pageID, int const& readSize = 0 ) {
															#line 3110 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Reference<ArenaPage>>(new ReadPhysicalExtentActor(self, pageID, readSize));
															#line 12349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3184 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Reference<ArenaPage>> readExtent(LogicalPageID pageID) override {
		debug_printf("DWALPager(%s) op=readExtent %s\n", filename.c_str(), toString(pageID).c_str());
		PageCacheEntry* pCacheEntry = extentCache.getIfExists(pageID);
		auto& eventReasons = g_redwoodMetrics.level(nonBtreeLevel).metrics.events;
		if (pCacheEntry != nullptr) {
			eventReasons.addEventReason(PagerEvents::CacheLookup, PagerEventReasons::MetaData);
			debug_printf("DWALPager(%s) Cache Entry exists for %s\n", filename.c_str(), toString(pageID).c_str());
			return pCacheEntry->readFuture;
		}
		eventReasons.addEventReason(PagerEvents::CacheLookup, PagerEventReasons::MetaData);

		LogicalPageID headPageID = header.remapQueue.headPageID;
		LogicalPageID tailPageID = header.remapQueue.tailPageID;
		int readSize = physicalExtentSize;
		bool headExt = false;
		bool tailExt = false;
		debug_printf("DWALPager(%s) #extentPages: %d, headPageID: %s, tailPageID: %s\n",
		             filename.c_str(),
		             pagesPerExtent,
		             toString(headPageID).c_str(),
		             toString(tailPageID).c_str());

		if (headPageID >= pageID && ((headPageID - pageID) < pagesPerExtent))
			headExt = true;
		if ((tailPageID - pageID) < pagesPerExtent)
			tailExt = true;
		if (headExt && tailExt) {
			readSize = (tailPageID - headPageID + 1) * physicalPageSize;
		} else if (headExt) {
			readSize = (pagesPerExtent - (headPageID - pageID)) * physicalPageSize;
		} else if (tailExt) {
			readSize = (tailPageID - pageID + 1) * physicalPageSize;
		}

		PageCacheEntry& cacheEntry = extentCache.get(pageID, 1);
		if (!cacheEntry.initialized()) {
			cacheEntry.writeFuture = Void();
			cacheEntry.readFuture =
			    forwardError(readPhysicalExtent(this, (PhysicalPageID)pageID, readSize), errorPromise);
			debug_printf("DWALPager(%s) Set the cacheEntry readFuture for page: %s\n",
			             filename.c_str(),
			             toString(pageID).c_str());

			++g_redwoodMetrics.metric.pagerCacheMiss;
			eventReasons.addEventReason(PagerEvents::CacheMiss, PagerEventReasons::MetaData);
			eventReasons.addEventReason(PagerEvents::CacheLookup, PagerEventReasons::MetaData);
		} else {
			++g_redwoodMetrics.metric.pagerCacheHit;
			eventReasons.addEventReason(PagerEvents::CacheHit, PagerEventReasons::MetaData);
			eventReasons.addEventReason(PagerEvents::CacheLookup, PagerEventReasons::MetaData);
		}
		return cacheEntry.readFuture;
	}

	// Get snapshot as of the most recent committed version of the pager
	Reference<IPagerSnapshot> getReadSnapshot(Version v) override;
	void addSnapshot(Version version, KeyRef meta) {
		if (snapshots.empty()) {
			oldestSnapshotVersion = version;
		} else {
			ASSERT(snapshots.back().version != version);
		}

		snapshots.push_back({ version, makeReference<DWALPagerSnapshot>(this, meta, version) });
	}

	// Set the pending oldest versiont to keep as of the next commit
	void setOldestReadableVersion(Version v) override {
		ASSERT(v >= header.oldestVersion);
		ASSERT(v <= header.committedVersion);
		header.oldestVersion = v;
		expireSnapshots(v);
	};

	// Get the oldest *readable* version, which is not the same as the oldest retained version as the version
	// returned could have been set as the oldest version in the pending commit
	Version getOldestReadableVersion() const override { return header.oldestVersion; };

	// Calculate the *effective* oldest version, which can be older than the one set in the last commit since we
	// are allowing active snapshots to temporarily delay page reuse.
	Version effectiveOldestVersion() { return std::min(lastCommittedHeader.oldestVersion, oldestSnapshotVersion); }

																#line 12436 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via removeRemapEntry()
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class RemoveRemapEntryActor>
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RemoveRemapEntryActorState {
															#line 12442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemoveRemapEntryActorState(DWALPager* const& self,RemappedPage const& p,Version const& oldestRetainedVersion) 
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   p(p),
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   oldestRetainedVersion(oldestRetainedVersion),
															#line 3269 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   iPageMapPair(self->remappedPages.find(p.originalPageID))
															#line 12455 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("removeRemapEntry", reinterpret_cast<unsigned long>(this));

	}
	~RemoveRemapEntryActorState() 
	{
		fdb_probe_actor_destroy("removeRemapEntry", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3271 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(iPageMapPair != self->remappedPages.end());
															#line 3272 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(!iPageMapPair->second.empty());
															#line 3273 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			iVersionPagePair = iPageMapPair->second.find(p.version);
															#line 3274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(iVersionPagePair != iPageMapPair->second.end());
															#line 3276 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			RemappedPage::Type firstType = p.getType();
															#line 3277 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			secondType = RemappedPage::Type();
															#line 3278 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			bool secondAfterOldestRetainedVersion = false;
															#line 3279 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			deleteAtSameVersion = false;
															#line 3280 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (p.newPageID == iVersionPagePair->second)
															#line 12486 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3281 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				auto nextEntry = iVersionPagePair;
															#line 3282 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				++nextEntry;
															#line 3283 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (nextEntry == iPageMapPair->second.end())
															#line 12494 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 3284 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					secondType = RemappedPage::NONE;
															#line 12498 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 3286 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					secondType = RemappedPage::getTypeOf(nextEntry->second);
															#line 3287 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					secondAfterOldestRetainedVersion = nextEntry->first > oldestRetainedVersion;
															#line 12506 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
			else
			{
															#line 3290 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				ASSERT(iVersionPagePair->second == invalidLogicalPageID);
															#line 3291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				secondType = RemappedPage::FREE;
															#line 3292 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				deleteAtSameVersion = true;
															#line 12517 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 3294 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(firstType == RemappedPage::REMAP || secondType == RemappedPage::NONE);
															#line 3332 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			freeNewID = (firstType == RemappedPage::REMAP && secondType != RemappedPage::DETACH && !deleteAtSameVersion);
															#line 3334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			copyNewToOriginal = (firstType == RemappedPage::REMAP && (secondAfterOldestRetainedVersion || secondType == RemappedPage::NONE));
															#line 3336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			freeOriginalID = (firstType == RemappedPage::FREE || firstType == RemappedPage::DETACH);
															#line 3338 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup %s secondType=%c mapEntry=%s oldestRetainedVersion=%" PRId64 " \n", self->filename.c_str(), p.toString().c_str(), secondType, ::toString(*iVersionPagePair).c_str(), oldestRetainedVersion);
															#line 3345 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (copyNewToOriginal)
															#line 12531 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3346 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (g_network->isSimulated())
															#line 12535 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 3347 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					ASSERT(!self->remapDestinationsSimOnly.contains(p.originalPageID));
															#line 3348 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->remapDestinationsSimOnly.insert(p.originalPageID);
															#line 12541 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
															#line 3350 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("DWALPager(%s) remapCleanup copy %s\n", self->filename.c_str(), p.toString().c_str());
															#line 3353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Reference<ArenaPage>> __when_expr_0 = self->readPage(PagerEventReasons::MetaData, nonBtreeLevel, p.newPageID, ioLeafPriority, false, true);
															#line 3353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<RemoveRemapEntryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12549 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RemoveRemapEntryActor*>(this)->actor_wait_state = 1;
															#line 3353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >*>(static_cast<RemoveRemapEntryActor*>(this)));
															#line 12554 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 3360 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (firstType == RemappedPage::REMAP)
															#line 12561 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 3361 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					++g_redwoodMetrics.metric.pagerRemapSkip;
															#line 12565 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveRemapEntryActorState();
		static_cast<RemoveRemapEntryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3367 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!deleteAtSameVersion)
															#line 12590 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3368 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf( "DWALPager(%s) remapCleanup deleting map entry %s\n", self->filename.c_str(), p.toString().c_str());
															#line 3371 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			iVersionPagePair = iPageMapPair->second.erase(iVersionPagePair);
															#line 3374 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (iPageMapPair->second.empty())
															#line 12598 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3375 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf( "DWALPager(%s) remapCleanup deleting empty map %s\n", self->filename.c_str(), p.toString().c_str());
															#line 3377 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->remappedPages.erase(iPageMapPair);
															#line 12604 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
			else
			{
															#line 3378 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (freeNewID && secondType == RemappedPage::NONE && iVersionPagePair != iPageMapPair->second.end() && RemappedPage::getTypeOf(iVersionPagePair->second) == RemappedPage::DETACH)
															#line 12610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 3384 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					freeNewID = false;
															#line 12614 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
		}
															#line 3388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeNewID)
															#line 12620 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3389 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup freeNew %s %s\n", self->filename.c_str(), p.toString().c_str(), toString(self->getLastCommittedVersion()).c_str());
															#line 3407 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->freeUnmappedPage(p.newPageID, self->getLastCommittedVersion() + 1);
															#line 3408 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			++g_redwoodMetrics.metric.pagerRemapFree;
															#line 12628 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3411 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (freeOriginalID)
															#line 12632 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3412 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup freeOriginal %s\n", self->filename.c_str(), p.toString().c_str());
															#line 3415 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->freeUnmappedPage(p.originalPageID, 0);
															#line 3416 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			++g_redwoodMetrics.metric.pagerRemapFree;
															#line 12640 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3419 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<RemoveRemapEntryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveRemapEntryActorState(); static_cast<RemoveRemapEntryActor*>(this)->destroy(); return 0; }
															#line 12644 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<RemoveRemapEntryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveRemapEntryActorState();
		static_cast<RemoveRemapEntryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Reference<ArenaPage> const& data,int loopDepth) 
	{
															#line 3357 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->updatePage( PagerEventReasons::MetaData, nonBtreeLevel, VectorRef<LogicalPageID>(&p.originalPageID, 1), data);
															#line 3359 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++g_redwoodMetrics.metric.pagerRemapCopy;
															#line 12658 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Reference<ArenaPage> && data,int loopDepth) 
	{
															#line 3357 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->updatePage( PagerEventReasons::MetaData, nonBtreeLevel, VectorRef<LogicalPageID>(&p.originalPageID, 1), data);
															#line 3359 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++g_redwoodMetrics.metric.pagerRemapCopy;
															#line 12669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<ArenaPage> const& data,int loopDepth) 
	{
		loopDepth = a_body1cont6(data, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<ArenaPage> && data,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(data), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveRemapEntryActor*>(this)->actor_wait_state > 0) static_cast<RemoveRemapEntryActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveRemapEntryActor*>(this)->ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >*,Reference<ArenaPage> const& value) 
	{
		fdb_probe_actor_enter("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveRemapEntryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >*,Reference<ArenaPage> && value) 
	{
		fdb_probe_actor_enter("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveRemapEntryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >*,Error err) 
	{
		fdb_probe_actor_enter("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveRemapEntryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeRemapEntry", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemappedPage p;
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Version oldestRetainedVersion;
															#line 3269 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	PageToVersionedMapT::iterator iPageMapPair;
															#line 3273 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	VersionToPageMapT::iterator iVersionPagePair;
															#line 3277 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemappedPage::Type secondType;
															#line 3279 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool deleteAtSameVersion;
															#line 3332 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool freeNewID;
															#line 3334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool copyNewToOriginal;
															#line 3336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool freeOriginalID;
															#line 12769 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via removeRemapEntry()
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RemoveRemapEntryActor final : public Actor<Void>, public ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >, public FastAllocated<RemoveRemapEntryActor>, public RemoveRemapEntryActorState<RemoveRemapEntryActor> {
															#line 12774 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<RemoveRemapEntryActor>::operator new;
	using FastAllocated<RemoveRemapEntryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14028049526474828544UL, 1842870702787064832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >;
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemoveRemapEntryActor(DWALPager* const& self,RemappedPage const& p,Version const& oldestRetainedVersion) 
															#line 12791 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveRemapEntryActorState<RemoveRemapEntryActor>(self, p, oldestRetainedVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeRemapEntry", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2881951226262710272UL, 2185007865796532480UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveRemapEntryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeRemapEntry");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeRemapEntry", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveRemapEntryActor, 0, Reference<ArenaPage> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> removeRemapEntry( DWALPager* const& self, RemappedPage const& p, Version const& oldestRetainedVersion ) {
															#line 3267 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new RemoveRemapEntryActor(self, p, oldestRetainedVersion));
															#line 12823 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3421 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

																#line 12828 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via remapCleanup()
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class RemapCleanupActor>
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RemapCleanupActorState {
															#line 12834 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemapCleanupActorState(DWALPager* const& self) 
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 3423 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   tasks(true),
															#line 3424 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   signal()
															#line 12845 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("remapCleanup", reinterpret_cast<unsigned long>(this));

	}
	~RemapCleanupActorState() 
	{
		fdb_probe_actor_destroy("remapCleanup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3425 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			tasks.add(signal.getFuture());
															#line 3427 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remapCleanupStop = false;
															#line 3431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			oldestRetainedVersion = self->effectiveOldestVersion();
															#line 3434 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			cutoff = RemappedPage(oldestRetainedVersion);
															#line 3437 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			double toleranceRatio = BUGGIFY ? deterministicRandom()->randomInt(0, 10) / 100.0 : SERVER_KNOBS->REDWOOD_REMAP_CLEANUP_TOLERANCE_RATIO;
															#line 3440 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			uint64_t remapCleanupWindowEntries = static_cast<uint64_t>(self->remapCleanupWindowBytes / self->header.pageSize);
															#line 3442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			minRemapEntries = static_cast<uint64_t>(remapCleanupWindowEntries * (1.0 - toleranceRatio));
															#line 3443 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			maxRemapEntries = static_cast<uint64_t>(remapCleanupWindowEntries * (1.0 + toleranceRatio));
															#line 3445 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup oldestRetainedVersion=%" PRId64 " remapCleanupWindowBytes=%" PRId64 " pageSize=%" PRIu32 " minRemapEntries=%" PRId64 " maxRemapEntries=%" PRId64 " items=%" PRId64 "\n", self->filename.c_str(), oldestRetainedVersion, self->remapCleanupWindowBytes, self->header.pageSize, minRemapEntries, maxRemapEntries, self->remapQueue.numEntries);
															#line 3456 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (g_network->isSimulated())
															#line 12878 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3457 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->remapDestinationsSimOnly.clear();
															#line 12882 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 3460 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			sinceYield = 0;
															#line 3461 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 12888 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemapCleanupActorState();
		static_cast<RemapCleanupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3502 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) remapCleanup stopped stopSignal=%d remap=%" PRId64 " free=%" PRId64 " delayedFree=%" PRId64 "\n", self->filename.c_str(), self->remapCleanupStop, self->remapQueue.numEntries, self->freeList.numEntries, self->delayedFreeList.numEntries);
															#line 3509 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		signal.send(Void());
															#line 3510 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tasks.getResult();
															#line 3510 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RemapCleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12917 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<RemapCleanupActor*>(this)->actor_wait_state = 3;
															#line 3510 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RemapCleanupActor, 2, Void >*>(static_cast<RemapCleanupActor*>(this)));
															#line 12922 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3464 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		int64_t remainingEntries = self->remapQueue.numEntries;
															#line 3465 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (remainingEntries <= minRemapEntries || (self->remapCleanupStop && remainingEntries <= maxRemapEntries))
															#line 12940 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3467 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup finished remainingEntries=%" PRId64 " minRemapEntries=%" PRId64 " maxRemapEntries=%" PRId64, self->filename.c_str(), remainingEntries, minRemapEntries, maxRemapEntries);
															#line 12944 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<RemappedPage>> __when_expr_0 = self->remapQueue.pop(cutoff);
															#line 3475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<RemapCleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12951 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemapCleanupActor*>(this)->actor_wait_state = 1;
															#line 3475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >*>(static_cast<RemapCleanupActor*>(this)));
															#line 12956 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 3476 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) remapCleanup popped %s items=%" PRId64 "\n", self->filename.c_str(), ::toString(p).c_str(), self->remapQueue.numEntries);
															#line 3482 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!p.present())
															#line 12980 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3483 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) remapCleanup pop failed cutoffVer=%" PRId64 " items=%" PRId64 "\n", self->filename.c_str(), cutoff.version, self->remapQueue.numEntries);
															#line 12984 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3490 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		Future<Void> task = removeRemapEntry(self, p.get(), oldestRetainedVersion);
															#line 3491 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!task.isReady())
															#line 12991 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3492 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			tasks.add(task);
															#line 12995 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3496 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (++sinceYield >= 100)
															#line 12999 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3497 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			sinceYield = 0;
															#line 3498 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_1 = yield();
															#line 3498 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<RemapCleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13007 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemapCleanupActor*>(this)->actor_wait_state = 2;
															#line 3498 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemapCleanupActor, 1, Void >*>(static_cast<RemapCleanupActor*>(this)));
															#line 13012 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<RemappedPage> const& __p,int loopDepth) 
	{
															#line 3475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		p = __p;
															#line 13026 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<RemappedPage> && __p,int loopDepth) 
	{
		p = std::move(__p);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemapCleanupActor*>(this)->actor_wait_state > 0) static_cast<RemapCleanupActor*>(this)->actor_wait_state = 0;
		static_cast<RemapCleanupActor*>(this)->ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >::remove();

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >*,Optional<RemappedPage> const& value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >*,Optional<RemappedPage> && value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >*,Error err) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemapCleanupActor*>(this)->actor_wait_state > 0) static_cast<RemapCleanupActor*>(this)->actor_wait_state = 0;
		static_cast<RemapCleanupActor*>(this)->ActorCallback< RemapCleanupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemapCleanupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<RemapCleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemapCleanupActorState(); static_cast<RemapCleanupActor*>(this)->destroy(); return 0; }
															#line 13198 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<RemapCleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemapCleanupActorState();
		static_cast<RemapCleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3511 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<RemapCleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemapCleanupActorState(); static_cast<RemapCleanupActor*>(this)->destroy(); return 0; }
															#line 13210 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<RemapCleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemapCleanupActorState();
		static_cast<RemapCleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RemapCleanupActor*>(this)->actor_wait_state > 0) static_cast<RemapCleanupActor*>(this)->actor_wait_state = 0;
		static_cast<RemapCleanupActor*>(this)->ActorCallback< RemapCleanupActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RemapCleanupActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RemapCleanupActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3423 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ActorCollection tasks;
															#line 3424 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Promise<Void> signal;
															#line 3431 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Version oldestRetainedVersion;
															#line 3434 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemappedPage cutoff;
															#line 3442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	uint64_t minRemapEntries;
															#line 3443 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	uint64_t maxRemapEntries;
															#line 3460 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int sinceYield;
															#line 3475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Optional<RemappedPage> p;
															#line 13311 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via remapCleanup()
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class RemapCleanupActor final : public Actor<Void>, public ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >, public ActorCallback< RemapCleanupActor, 1, Void >, public ActorCallback< RemapCleanupActor, 2, Void >, public FastAllocated<RemapCleanupActor>, public RemapCleanupActorState<RemapCleanupActor> {
															#line 13316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<RemapCleanupActor>::operator new;
	using FastAllocated<RemapCleanupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2477910167469886464UL, 2175225312397297152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >;
friend struct ActorCallback< RemapCleanupActor, 1, Void >;
friend struct ActorCallback< RemapCleanupActor, 2, Void >;
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	RemapCleanupActor(DWALPager* const& self) 
															#line 13335 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   RemapCleanupActorState<RemapCleanupActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remapCleanup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17254336442698638080UL, 18277987376887644160UL);
		ActorExecutionContextHelper __helper(static_cast<RemapCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remapCleanup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remapCleanup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemapCleanupActor, 0, Optional<RemappedPage> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemapCleanupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RemapCleanupActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> remapCleanup( DWALPager* const& self ) {
															#line 3422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new RemapCleanupActor(self));
															#line 13369 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3513 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	// Flush all queues so they have no operations pending.
																#line 13375 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via flushQueues()
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class FlushQueuesActor>
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class FlushQueuesActorState {
															#line 13381 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FlushQueuesActorState(DWALPager* const& self) 
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 13388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("flushQueues", reinterpret_cast<unsigned long>(this));

	}
	~FlushQueuesActorState() 
	{
		fdb_probe_actor_destroy("flushQueues", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3516 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(self->remapCleanupFuture.isReady());
															#line 3519 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->remapQueue.flush();
															#line 3519 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13407 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlushQueuesActor*>(this)->actor_wait_state = 1;
															#line 3519 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 0, Void >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13412 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlushQueuesActorState();
		static_cast<FlushQueuesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->extentFreeList.flush();
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13437 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 2;
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 1, Void >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13442 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->extentFreeList.flush();
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13453 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 2;
															#line 3520 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 1, Void >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13458 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state > 0) static_cast<FlushQueuesActor*>(this)->actor_wait_state = 0;
		static_cast<FlushQueuesActor*>(this)->ActorCallback< FlushQueuesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlushQueuesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->extentUsedList.flush();
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13544 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 3;
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 2, Void >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13549 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->extentUsedList.flush();
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13560 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 3;
															#line 3521 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 2, Void >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13565 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state > 0) static_cast<FlushQueuesActor*>(this)->actor_wait_state = 0;
		static_cast<FlushQueuesActor*>(this)->ActorCallback< FlushQueuesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlushQueuesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3526 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		clear = 0;
															#line 3527 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		;
															#line 13651 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3526 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		clear = 0;
															#line 3527 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		;
															#line 13662 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state > 0) static_cast<FlushQueuesActor*>(this)->actor_wait_state = 0;
		static_cast<FlushQueuesActor*>(this)->ActorCallback< FlushQueuesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlushQueuesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3546 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.finishFlush();
															#line 3547 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.finishFlush();
															#line 3549 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<FlushQueuesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlushQueuesActorState(); static_cast<FlushQueuesActor*>(this)->destroy(); return 0; }
															#line 13750 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<FlushQueuesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlushQueuesActorState();
		static_cast<FlushQueuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 3528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<bool> __when_expr_3 = self->freeList.preFlush();
															#line 3528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13771 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 4;
															#line 3528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 3, bool >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13776 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<bool> __when_expr_4 = self->delayedFreeList.preFlush();
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13800 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlushQueuesActor*>(this)->actor_wait_state = 5;
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlushQueuesActor, 4, bool >*>(static_cast<FlushQueuesActor*>(this)));
															#line 13805 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(bool const& __freeBusy,int loopDepth) 
	{
															#line 3528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		freeBusy = __freeBusy;
															#line 13814 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(bool && __freeBusy,int loopDepth) 
	{
		freeBusy = std::move(__freeBusy);
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state > 0) static_cast<FlushQueuesActor*>(this)->actor_wait_state = 0;
		static_cast<FlushQueuesActor*>(this)->ActorCallback< FlushQueuesActor, 3, bool >::remove();

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 3, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 3, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlushQueuesActor, 3, bool >*,Error err) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont3loopBody1cont2(int loopDepth) 
	{
															#line 3530 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) flushQueues freeBusy=%d delayedFreeBusy=%d\n", self->filename.c_str(), freeBusy, delayedFreeBusy);
															#line 3538 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!freeBusy && !delayedFreeBusy)
															#line 13895 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3539 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (++clear == 2)
															#line 13899 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
				return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		else
		{
															#line 3543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			clear = 0;
															#line 13908 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(bool const& __delayedFreeBusy,int loopDepth) 
	{
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		delayedFreeBusy = __delayedFreeBusy;
															#line 13918 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(bool && __delayedFreeBusy,int loopDepth) 
	{
		delayedFreeBusy = std::move(__delayedFreeBusy);
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlushQueuesActor*>(this)->actor_wait_state > 0) static_cast<FlushQueuesActor*>(this)->actor_wait_state = 0;
		static_cast<FlushQueuesActor*>(this)->ActorCallback< FlushQueuesActor, 4, bool >::remove();

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 4, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlushQueuesActor, 4, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlushQueuesActor, 4, bool >*,Error err) 
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3526 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int clear;
															#line 3528 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool freeBusy;
															#line 3529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool delayedFreeBusy;
															#line 14001 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via flushQueues()
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class FlushQueuesActor final : public Actor<Void>, public ActorCallback< FlushQueuesActor, 0, Void >, public ActorCallback< FlushQueuesActor, 1, Void >, public ActorCallback< FlushQueuesActor, 2, Void >, public ActorCallback< FlushQueuesActor, 3, bool >, public ActorCallback< FlushQueuesActor, 4, bool >, public FastAllocated<FlushQueuesActor>, public FlushQueuesActorState<FlushQueuesActor> {
															#line 14006 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<FlushQueuesActor>::operator new;
	using FastAllocated<FlushQueuesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10570852516793620736UL, 175364509033591040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlushQueuesActor, 0, Void >;
friend struct ActorCallback< FlushQueuesActor, 1, Void >;
friend struct ActorCallback< FlushQueuesActor, 2, Void >;
friend struct ActorCallback< FlushQueuesActor, 3, bool >;
friend struct ActorCallback< FlushQueuesActor, 4, bool >;
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	FlushQueuesActor(DWALPager* const& self) 
															#line 14027 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   FlushQueuesActorState<FlushQueuesActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flushQueues", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11696366400212896512UL, 16375013173057342720UL);
		ActorExecutionContextHelper __helper(static_cast<FlushQueuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flushQueues");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flushQueues", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlushQueuesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlushQueuesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlushQueuesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlushQueuesActor, 3, bool >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlushQueuesActor, 4, bool >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> flushQueues( DWALPager* const& self ) {
															#line 3515 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new FlushQueuesActor(self));
															#line 14063 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3551 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

																#line 14068 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via commit_impl()
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Commit_implActor>
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Commit_implActorState {
															#line 14074 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Commit_implActorState(DWALPager* const& self,Version const& v,Value const& commitRecord) 
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   v(v),
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   commitRecord(commitRecord)
															#line 14085 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("commit_impl", reinterpret_cast<unsigned long>(this));

	}
	~Commit_implActorState() 
	{
		fdb_probe_actor_destroy("commit_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3553 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) commit begin %s\n", self->filename.c_str(), ::toString(v).c_str());
															#line 3557 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->writeHeaderPage(backupHeaderPageID, self->lastCommittedHeaderPage);
															#line 3560 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remapCleanupStop = true;
															#line 3561 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->remapCleanupFuture;
															#line 3561 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14108 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Commit_implActor*>(this)->actor_wait_state = 1;
															#line 3561 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 0, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14113 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Commit_implActorState();
		static_cast<Commit_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = flushQueues(self);
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14138 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Commit_implActor*>(this)->actor_wait_state = 2;
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 1, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14143 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = flushQueues(self);
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14154 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Commit_implActor*>(this)->actor_wait_state = 2;
															#line 3563 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 1, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14159 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3565 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.committedVersion = v;
															#line 3566 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.remapQueue = self->remapQueue.getState();
															#line 3567 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.extentFreeList = self->extentFreeList.getState();
															#line 3568 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.extentUsedList = self->extentUsedList.getState();
															#line 3569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.freeList = self->freeList.getState();
															#line 3570 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.delayedFreeList = self->delayedFreeList.getState();
															#line 3573 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) waiting for outstanding writes\n", self->filename.c_str());
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(self->operations);
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14259 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<Commit_implActor*>(this)->actor_wait_state = 3;
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 2, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14264 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3565 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.committedVersion = v;
															#line 3566 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.remapQueue = self->remapQueue.getState();
															#line 3567 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.extentFreeList = self->extentFreeList.getState();
															#line 3568 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.extentUsedList = self->extentUsedList.getState();
															#line 3569 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.freeList = self->freeList.getState();
															#line 3570 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.delayedFreeList = self->delayedFreeList.getState();
															#line 3573 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) waiting for outstanding writes\n", self->filename.c_str());
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(self->operations);
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14289 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<Commit_implActor*>(this)->actor_wait_state = 3;
															#line 3574 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 2, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14294 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3575 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 3576 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) Syncing\n", self->filename.c_str());
															#line 3584 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->memoryOnly && self->fileInitialized)
															#line 14382 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_3 = self->pageFile->sync();
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<Commit_implActor*>(this)->actor_wait_state = 4;
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 3, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14393 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3575 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 3576 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) Syncing\n", self->filename.c_str());
															#line 3584 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->memoryOnly && self->fileInitialized)
															#line 14411 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_3 = self->pageFile->sync();
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14417 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<Commit_implActor*>(this)->actor_wait_state = 4;
															#line 3585 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 3, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14422 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3592 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->header.userCommitRecord = commitRecord;
															#line 3593 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->updateHeaderPage();
															#line 3596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->writeHeaderPage(primaryHeaderPageID, self->headerPage);
															#line 3596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14517 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<Commit_implActor*>(this)->actor_wait_state = 5;
															#line 3596 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 4, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14522 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 3586 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) commit version %" PRId64 " sync 1\n", self->filename.c_str(), self->header.committedVersion);
															#line 14531 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 3586 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) commit version %" PRId64 " sync 1\n", self->filename.c_str(), self->header.committedVersion);
															#line 14540 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 3600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->memoryOnly)
															#line 14624 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_5 = self->pageFile->sync();
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14630 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont7when1(__when_expr_5.get(), loopDepth); };
			static_cast<Commit_implActor*>(this)->actor_wait_state = 6;
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 5, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14635 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 3600 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->memoryOnly)
															#line 14649 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_5 = self->pageFile->sync();
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Commit_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14655 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont7when1(__when_expr_5.get(), loopDepth); };
			static_cast<Commit_implActor*>(this)->actor_wait_state = 6;
															#line 3601 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< Commit_implActor, 5, Void >*>(static_cast<Commit_implActor*>(this)));
															#line 14660 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 3615 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->updateLastCommittedHeader();
															#line 3616 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->addSnapshot(v, self->header.userCommitRecord);
															#line 3620 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->expireSnapshots(self->header.oldestVersion);
															#line 3623 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupFuture = forwardError(remapCleanup(self), self->errorPromise);
															#line 3626 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->pageCache.flushPrioritizedEvictions();
															#line 3628 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Commit_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Commit_implActorState(); static_cast<Commit_implActor*>(this)->destroy(); return 0; }
															#line 14759 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Commit_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Commit_implActorState();
		static_cast<Commit_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 3602 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) commit version %" PRId64 " sync 2\n", self->filename.c_str(), self->header.committedVersion);
															#line 3608 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->fileInitialized)
															#line 14773 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3609 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->fileInitialized = true;
															#line 3610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) Atomic file creation complete.\n", self->filename.c_str());
															#line 14779 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 3602 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) commit version %" PRId64 " sync 2\n", self->filename.c_str(), self->header.committedVersion);
															#line 3608 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!self->fileInitialized)
															#line 14791 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3609 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->fileInitialized = true;
															#line 3610 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("DWALPager(%s) Atomic file creation complete.\n", self->filename.c_str());
															#line 14797 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<Commit_implActor*>(this)->actor_wait_state > 0) static_cast<Commit_implActor*>(this)->actor_wait_state = 0;
		static_cast<Commit_implActor*>(this)->ActorCallback< Commit_implActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< Commit_implActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< Commit_implActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Version v;
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Value commitRecord;
															#line 14884 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via commit_impl()
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Commit_implActor final : public Actor<Void>, public ActorCallback< Commit_implActor, 0, Void >, public ActorCallback< Commit_implActor, 1, Void >, public ActorCallback< Commit_implActor, 2, Void >, public ActorCallback< Commit_implActor, 3, Void >, public ActorCallback< Commit_implActor, 4, Void >, public ActorCallback< Commit_implActor, 5, Void >, public FastAllocated<Commit_implActor>, public Commit_implActorState<Commit_implActor> {
															#line 14889 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Commit_implActor>::operator new;
	using FastAllocated<Commit_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9192959100290778624UL, 18334299379551600128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Commit_implActor, 0, Void >;
friend struct ActorCallback< Commit_implActor, 1, Void >;
friend struct ActorCallback< Commit_implActor, 2, Void >;
friend struct ActorCallback< Commit_implActor, 3, Void >;
friend struct ActorCallback< Commit_implActor, 4, Void >;
friend struct ActorCallback< Commit_implActor, 5, Void >;
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Commit_implActor(DWALPager* const& self,Version const& v,Value const& commitRecord) 
															#line 14911 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   Commit_implActorState<Commit_implActor>(self, v, commitRecord),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("commit_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2738407717099012608UL, 14419500805409879808UL);
		ActorExecutionContextHelper __helper(static_cast<Commit_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("commit_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("commit_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Commit_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Commit_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Commit_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Commit_implActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< Commit_implActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< Commit_implActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> commit_impl( DWALPager* const& self, Version const& v, Value const& commitRecord ) {
															#line 3552 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new Commit_implActor(self, v, commitRecord));
															#line 14948 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3630 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> commit(Version v, Value commitRecord) override {
		// Can't have more than one commit outstanding.
		ASSERT(commitFuture.isReady());
		ASSERT(v > lastCommittedHeader.committedVersion);
		commitFuture = forwardError(commit_impl(this, v, commitRecord), errorPromise);
		return commitFuture;
	}

	Value getCommitRecord() const override { return lastCommittedHeader.userCommitRecord; }

																#line 14963 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via shutdown()
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ShutdownActor>
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ShutdownActorState {
															#line 14969 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ShutdownActorState(DWALPager* const& self,bool const& dispose) 
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   dispose(dispose)
															#line 14978 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("shutdown", reinterpret_cast<unsigned long>(this));

	}
	~ShutdownActorState() 
	{
		fdb_probe_actor_destroy("shutdown", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3644 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->errorPromise.canBeSet())
															#line 14993 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3645 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("DWALPager(%s) shutdown sending error\n", self->filename.c_str());
															#line 3646 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->errorPromise.sendError(actor_cancelled());
															#line 14999 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 3648 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 3648 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 15005 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			static_cast<ShutdownActor*>(this)->actor_wait_state = 1;
															#line 3648 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 0, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15009 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<ShutdownActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3651 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown kill ioLock\n", self->filename.c_str());
															#line 3652 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->ioLock->halt();
															#line 3654 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel recovery\n", self->filename.c_str());
															#line 3655 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->recoverFuture.cancel();
															#line 3656 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel commit\n", self->filename.c_str());
															#line 3657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->commitFuture.cancel();
															#line 3658 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel remap\n", self->filename.c_str());
															#line 3659 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupFuture.cancel();
															#line 3660 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown kill file extension\n", self->filename.c_str());
															#line 3661 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->fileExtension.cancel();
															#line 3663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel operations\n", self->filename.c_str());
															#line 3664 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for( auto& f : self->operations ) {
															#line 3665 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			f.cancel();
															#line 15055 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3667 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 3669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel queues\n", self->filename.c_str());
															#line 3670 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.cancel();
															#line 3671 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.cancel();
															#line 3672 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.cancel();
															#line 3673 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentFreeList.cancel();
															#line 3674 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.cancel();
															#line 3676 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown destroy page cache\n", self->filename.c_str());
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->extentCache.clear();
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 15077 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<ShutdownActor*>(this)->actor_wait_state = 2;
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 1, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15081 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3651 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown kill ioLock\n", self->filename.c_str());
															#line 3652 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->ioLock->halt();
															#line 3654 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel recovery\n", self->filename.c_str());
															#line 3655 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->recoverFuture.cancel();
															#line 3656 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel commit\n", self->filename.c_str());
															#line 3657 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->commitFuture.cancel();
															#line 3658 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel remap\n", self->filename.c_str());
															#line 3659 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupFuture.cancel();
															#line 3660 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown kill file extension\n", self->filename.c_str());
															#line 3661 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->fileExtension.cancel();
															#line 3663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel operations\n", self->filename.c_str());
															#line 3664 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for( auto& f : self->operations ) {
															#line 3665 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			f.cancel();
															#line 15114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3667 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->operations.clear();
															#line 3669 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown cancel queues\n", self->filename.c_str());
															#line 3670 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.cancel();
															#line 3671 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.cancel();
															#line 3672 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.cancel();
															#line 3673 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentFreeList.cancel();
															#line 3674 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.cancel();
															#line 3676 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown destroy page cache\n", self->filename.c_str());
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->extentCache.clear();
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 15136 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<ShutdownActor*>(this)->actor_wait_state = 2;
															#line 3677 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 1, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15140 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ShutdownActor*>(this)->actor_wait_state > 0) static_cast<ShutdownActor*>(this)->actor_wait_state = 0;
		static_cast<ShutdownActor*>(this)->ActorCallback< ShutdownActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ShutdownActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->pageCache.clear();
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
															#line 15226 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<ShutdownActor*>(this)->actor_wait_state = 3;
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 2, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15230 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->pageCache.clear();
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
															#line 15241 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		static_cast<ShutdownActor*>(this)->actor_wait_state = 3;
															#line 3678 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 2, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15245 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ShutdownActor*>(this)->actor_wait_state > 0) static_cast<ShutdownActor*>(this)->actor_wait_state = 0;
		static_cast<ShutdownActor*>(this)->ActorCallback< ShutdownActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ShutdownActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 3680 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown remappedPagesMap: %s\n", self->filename.c_str(), toString(self->remappedPages).c_str());
															#line 3685 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->pageFile.clear();
															#line 3686 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (dispose)
															#line 15333 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3687 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!self->memoryOnly)
															#line 15337 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3688 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("DWALPager(%s) shutdown deleting file\n", self->filename.c_str());
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_3 = ready(IAsyncFileSystem::filesystem()->incrementalDeleteFile(self->filename, true));
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
															#line 15345 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				static_cast<ShutdownActor*>(this)->actor_wait_state = 4;
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 3, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15349 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 3680 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager(%s) shutdown remappedPagesMap: %s\n", self->filename.c_str(), toString(self->remappedPages).c_str());
															#line 3685 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->pageFile.clear();
															#line 3686 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (dispose)
															#line 15372 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3687 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!self->memoryOnly)
															#line 15376 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 3688 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				debug_printf("DWALPager(%s) shutdown deleting file\n", self->filename.c_str());
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<Void> __when_expr_3 = ready(IAsyncFileSystem::filesystem()->incrementalDeleteFile(self->filename, true));
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
															#line 15384 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				static_cast<ShutdownActor*>(this)->actor_wait_state = 4;
															#line 3692 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ShutdownActor, 3, Void >*>(static_cast<ShutdownActor*>(this)));
															#line 15388 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ShutdownActor*>(this)->actor_wait_state > 0) static_cast<ShutdownActor*>(this)->actor_wait_state = 0;
		static_cast<ShutdownActor*>(this)->ActorCallback< ShutdownActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ShutdownActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 3696 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		Promise<Void> closedPromise = self->closedPromise;
															#line 3697 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		delete self;
															#line 3698 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		closedPromise.send(Void());
															#line 15486 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ShutdownActor*>(this)->actor_wait_state > 0) static_cast<ShutdownActor*>(this)->actor_wait_state = 0;
		static_cast<ShutdownActor*>(this)->ActorCallback< ShutdownActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ShutdownActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ShutdownActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		delete static_cast<ShutdownActor*>(this);
															#line 15588 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		return 0;

		return loopDepth;
	}
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	bool dispose;
															#line 15597 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via shutdown()
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ShutdownActor final : public Actor<void>, public ActorCallback< ShutdownActor, 0, Void >, public ActorCallback< ShutdownActor, 1, Void >, public ActorCallback< ShutdownActor, 2, Void >, public ActorCallback< ShutdownActor, 3, Void >, public FastAllocated<ShutdownActor>, public ShutdownActorState<ShutdownActor> {
															#line 15602 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ShutdownActor>::operator new;
	using FastAllocated<ShutdownActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14832663904492344832UL, 3476074116129387776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< ShutdownActor, 0, Void >;
friend struct ActorCallback< ShutdownActor, 1, Void >;
friend struct ActorCallback< ShutdownActor, 2, Void >;
friend struct ActorCallback< ShutdownActor, 3, Void >;
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ShutdownActor(DWALPager* const& self,bool const& dispose) 
															#line 15622 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<void>(),
		   ShutdownActorState<ShutdownActor>(self, dispose),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("shutdown", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3873605778738212608UL, 11672863135119866112UL);
		ActorExecutionContextHelper __helper(static_cast<ShutdownActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("shutdown");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("shutdown", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
void shutdown( DWALPager* const& self, bool const& dispose ) {
															#line 3641 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	new ShutdownActor(self, dispose);
															#line 15645 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3700 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	void dispose() override { shutdown(this, true); }

	void close() override { shutdown(this, false); }

	Future<Void> getError() const override { return errorPromise.getFuture(); }

	Future<Void> onClosed() const override { return closedPromise.getFuture(); }

	StorageBytes getStorageBytes() const override {
		int64_t free;
		int64_t total;
		if (memoryOnly) {
			total = pageCache.evictor().sizeLimit;
			free = pageCache.evictor().getSizeUsed();
		} else {
			g_network->getDiskBytes(parentDirectory(filename), free, total);
		}

		// Size of redwood data file.  Note that filePageCountPending is used here instead of filePageCount.  This is
		// because it is always >= filePageCount and accounts for file size changes which will complete soon.
		int64_t pagerPhysicalSize = filePageCountPending * physicalPageSize;

		// Size of the pager, which can be less than the data file size.  All pages within this size are either in use
		// in a data structure or accounted for in one of the pager's free page lists.
		int64_t pagerLogicalSize = header.pageCount * physicalPageSize;

		// It is not exactly known how many pages on the delayed free list are usable as of right now.  It could be
		// known, if each commit delayed entries that were freeable were shuffled from the delayed free queue to the
		// free queue, but this doesn't seem necessary.

		// Amount of space taken up by all of the items in the free lists
		int64_t reusablePageSpace = (freeList.numEntries + delayedFreeList.numEntries) * physicalPageSize;
		// Amount of space taken up by the free list queues themselves, as if we were to pop and use
		// items on the free lists the space the items are stored in would also become usable
		int64_t reusableQueueSpace = (freeList.numPages + delayedFreeList.numPages) * physicalPageSize;

		// Pager slack is the space at the end of the pager's logical size until the end of the pager file's size.
		// These pages will be used if needed without growing the file size.
		int64_t reusablePagerSlackSpace = pagerPhysicalSize - pagerLogicalSize;

		int64_t reusable = reusablePageSpace + reusableQueueSpace + reusablePagerSlackSpace;

		// Space currently in used by old page versions have have not yet been freed due to the remap cleanup window.
		int64_t temp = remapQueue.numEntries * physicalPageSize;

		return StorageBytes(free, total, pagerPhysicalSize, free + reusable, temp);
	}

	int64_t getPageCacheCount() override { return pageCache.getCount(); }
	int64_t getPageCount() override { return header.pageCount; }
	int64_t getExtentCacheCount() override { return extentCache.getCount(); }

																#line 15702 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via getUserPageCount_cleanup()
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class GetUserPageCount_cleanupActor>
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class GetUserPageCount_cleanupActorState {
															#line 15708 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	GetUserPageCount_cleanupActorState(DWALPager* const& self) 
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 15715 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this));

	}
	~GetUserPageCount_cleanupActorState() 
	{
		fdb_probe_actor_destroy("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->remapCleanupFuture;
															#line 3755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15732 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state = 1;
															#line 3755 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetUserPageCount_cleanupActor, 0, Void >*>(static_cast<GetUserPageCount_cleanupActor*>(this)));
															#line 15737 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetUserPageCount_cleanupActorState();
		static_cast<GetUserPageCount_cleanupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = flushQueues(self);
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state = 2;
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*>(static_cast<GetUserPageCount_cleanupActor*>(this)));
															#line 15767 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = flushQueues(self);
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15778 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state = 2;
															#line 3758 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*>(static_cast<GetUserPageCount_cleanupActor*>(this)));
															#line 15783 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state > 0) static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state = 0;
		static_cast<GetUserPageCount_cleanupActor*>(this)->ActorCallback< GetUserPageCount_cleanupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetUserPageCount_cleanupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetUserPageCount_cleanupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetUserPageCount_cleanupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3760 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager getUserPageCount_cleanup\n");
															#line 3761 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.getState();
															#line 3762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.getState();
															#line 3763 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentFreeList.getState();
															#line 3764 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.getState();
															#line 3765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.getState();
															#line 3766 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<GetUserPageCount_cleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetUserPageCount_cleanupActorState(); static_cast<GetUserPageCount_cleanupActor*>(this)->destroy(); return 0; }
															#line 15879 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<GetUserPageCount_cleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetUserPageCount_cleanupActorState();
		static_cast<GetUserPageCount_cleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3760 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("DWALPager getUserPageCount_cleanup\n");
															#line 3761 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeList.getState();
															#line 3762 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->delayedFreeList.getState();
															#line 3763 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentFreeList.getState();
															#line 3764 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->extentUsedList.getState();
															#line 3765 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapQueue.getState();
															#line 3766 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<GetUserPageCount_cleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetUserPageCount_cleanupActorState(); static_cast<GetUserPageCount_cleanupActor*>(this)->destroy(); return 0; }
															#line 15903 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<GetUserPageCount_cleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetUserPageCount_cleanupActorState();
		static_cast<GetUserPageCount_cleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state > 0) static_cast<GetUserPageCount_cleanupActor*>(this)->actor_wait_state = 0;
		static_cast<GetUserPageCount_cleanupActor*>(this)->ActorCallback< GetUserPageCount_cleanupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 15988 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via getUserPageCount_cleanup()
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class GetUserPageCount_cleanupActor final : public Actor<Void>, public ActorCallback< GetUserPageCount_cleanupActor, 0, Void >, public ActorCallback< GetUserPageCount_cleanupActor, 1, Void >, public FastAllocated<GetUserPageCount_cleanupActor>, public GetUserPageCount_cleanupActorState<GetUserPageCount_cleanupActor> {
															#line 15993 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<GetUserPageCount_cleanupActor>::operator new;
	using FastAllocated<GetUserPageCount_cleanupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10696300963344209664UL, 8955621552772552960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetUserPageCount_cleanupActor, 0, Void >;
friend struct ActorCallback< GetUserPageCount_cleanupActor, 1, Void >;
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	GetUserPageCount_cleanupActor(DWALPager* const& self) 
															#line 16011 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   GetUserPageCount_cleanupActorState<GetUserPageCount_cleanupActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6188854379200205824UL, 12806750536355350528UL);
		ActorExecutionContextHelper __helper(static_cast<GetUserPageCount_cleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getUserPageCount_cleanup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getUserPageCount_cleanup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetUserPageCount_cleanupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetUserPageCount_cleanupActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> getUserPageCount_cleanup( DWALPager* const& self ) {
															#line 3753 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new GetUserPageCount_cleanupActor(self));
															#line 16044 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3768 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	// Get the number of pages in use by the pager's user
	Future<int64_t> getUserPageCount() override {
		return map(getUserPageCount_cleanup(this), [=](Void) {
			int64_t userPages =
			    header.pageCount - 2 - freeList.numPages - freeList.numEntries - delayedFreeList.numPages -
			    delayedFreeList.numEntries - ((((remapQueue.numPages - 1) / pagesPerExtent) + 1) * pagesPerExtent) -
			    extentFreeList.numPages - (pagesPerExtent * extentFreeList.numEntries) - extentUsedList.numPages;

			debug_printf("DWALPager(%s) userPages=%" PRId64 " totalPageCount=%" PRId64 " freeQueuePages=%" PRId64
			             " freeQueueCount=%" PRId64 " delayedFreeQueuePages=%" PRId64 " delayedFreeQueueCount=%" PRId64
			             " remapQueuePages=%" PRId64 " remapQueueCount=%" PRId64 "\n",
			             filename.c_str(),
			             userPages,
			             header.pageCount,
			             freeList.numPages,
			             freeList.numEntries,
			             delayedFreeList.numPages,
			             delayedFreeList.numEntries,
			             remapQueue.numPages,
			             remapQueue.numEntries);
			return userPages;
		});
	}

	Future<Void> init() override { return recoverFuture; }

	Version getLastCommittedVersion() const override { return lastCommittedHeader.committedVersion; }

private:
	~DWALPager() {}

	// Try to expire snapshots up to but not including v, but do not expire any snapshots that are in use.
	void expireSnapshots(Version v);

	// Header is the format of page 0 of the database
	struct PagerCommitHeader {
		constexpr static FileIdentifier file_identifier = 11836690;
		constexpr static unsigned int FORMAT_VERSION = 10;

		uint16_t formatVersion;
		uint32_t queueCount;
		uint32_t pageSize;
		int64_t pageCount;
		uint32_t extentSize;
		Version committedVersion;
		Version oldestVersion;
		Value userCommitRecord;
		FIFOQueue<LogicalPageID>::QueueState freeList;
		FIFOQueue<LogicalPageID>::QueueState extentFreeList; // free list for extents
		FIFOQueue<ExtentUsedListEntry>::QueueState extentUsedList; // in-use list for extents
		FIFOQueue<DelayedFreePage>::QueueState delayedFreeList;
		FIFOQueue<RemappedPage>::QueueState remapQueue;

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar,
			           formatVersion,
			           queueCount,
			           pageSize,
			           pageCount,
			           extentSize,
			           committedVersion,
			           oldestVersion,
			           userCommitRecord,
			           freeList,
			           extentFreeList,
			           extentUsedList,
			           delayedFreeList,
			           remapQueue);
		}
	};

																#line 16121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via clearRemapQueue_impl()
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ClearRemapQueue_implActor>
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ClearRemapQueue_implActorState {
															#line 16127 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ClearRemapQueue_implActorState(DWALPager* const& self) 
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 16134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this));

	}
	~ClearRemapQueue_implActorState() 
	{
		fdb_probe_actor_destroy("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3843 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->commitFuture;
															#line 3843 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 16151 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state = 1;
															#line 3843 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClearRemapQueue_implActor, 0, Void >*>(static_cast<ClearRemapQueue_implActor*>(this)));
															#line 16156 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClearRemapQueue_implActorState();
		static_cast<ClearRemapQueue_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3846 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapCleanupWindowBytes = self->remapCleanupWindowBytes;
															#line 3847 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupWindowBytes = 0;
															#line 3851 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt = 0;
															#line 3852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt = 0;
															#line 16185 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3846 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		remapCleanupWindowBytes = self->remapCleanupWindowBytes;
															#line 3847 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->remapCleanupWindowBytes = 0;
															#line 3851 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt = 0;
															#line 3852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt = 0;
															#line 16200 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state > 0) static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state = 0;
		static_cast<ClearRemapQueue_implActor*>(this)->ActorCallback< ClearRemapQueue_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearRemapQueue_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClearRemapQueue_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClearRemapQueue_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 3856 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(self->remapQueue.numEntries == 0);
															#line 3859 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (remapCleanupWindowBytes != 0)
															#line 16286 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 3860 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->remapCleanupWindowBytes = remapCleanupWindowBytes;
															#line 16290 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 3862 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		TraceEvent e("RedwoodClearRemapQueue");
															#line 3863 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->toTraceEvent(e);
															#line 3864 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.log();
															#line 3865 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<ClearRemapQueue_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ClearRemapQueue_implActorState(); static_cast<ClearRemapQueue_implActor*>(this)->destroy(); return 0; }
															#line 16300 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<ClearRemapQueue_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ClearRemapQueue_implActorState();
		static_cast<ClearRemapQueue_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 3852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!(attempt < 2))
															#line 16319 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3853 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->setOldestReadableVersion(self->getLastCommittedVersion());
															#line 3854 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->commit(self->getLastCommittedVersion() + 1, self->getCommitRecord());
															#line 3854 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16329 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state = 2;
															#line 3854 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClearRemapQueue_implActor, 1, Void >*>(static_cast<ClearRemapQueue_implActor*>(this)));
															#line 16334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 3852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt++;
															#line 16356 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 3852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		attempt++;
															#line 16365 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state > 0) static_cast<ClearRemapQueue_implActor*>(this)->actor_wait_state = 0;
		static_cast<ClearRemapQueue_implActor*>(this)->ActorCallback< ClearRemapQueue_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearRemapQueue_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ClearRemapQueue_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ClearRemapQueue_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	DWALPager* self;
															#line 3846 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int64_t remapCleanupWindowBytes;
															#line 3851 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int attempt;
															#line 16451 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via clearRemapQueue_impl()
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class ClearRemapQueue_implActor final : public Actor<Void>, public ActorCallback< ClearRemapQueue_implActor, 0, Void >, public ActorCallback< ClearRemapQueue_implActor, 1, Void >, public FastAllocated<ClearRemapQueue_implActor>, public ClearRemapQueue_implActorState<ClearRemapQueue_implActor> {
															#line 16456 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<ClearRemapQueue_implActor>::operator new;
	using FastAllocated<ClearRemapQueue_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5742816625449159680UL, 16755361683693363200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClearRemapQueue_implActor, 0, Void >;
friend struct ActorCallback< ClearRemapQueue_implActor, 1, Void >;
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	ClearRemapQueue_implActor(DWALPager* const& self) 
															#line 16474 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   ClearRemapQueue_implActorState<ClearRemapQueue_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3762574589026699520UL, 11027026041023317760UL);
		ActorExecutionContextHelper __helper(static_cast<ClearRemapQueue_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clearRemapQueue_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clearRemapQueue_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClearRemapQueue_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ClearRemapQueue_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> clearRemapQueue_impl( DWALPager* const& self ) {
															#line 3841 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new ClearRemapQueue_implActor(self));
															#line 16507 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 3867 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> clearRemapQueue() override { return clearRemapQueue_impl(this); }

private:
	// Physical page sizes will always be a multiple of 4k because AsyncFileNonDurable requires
	// this in simulation, and it also makes sense for current SSDs.
	// Allowing a smaller 'logical' page size is very useful for testing.
	constexpr static int smallestPhysicalBlock = 4096;
	int physicalPageSize;
	int logicalPageSize; // In simulation testing it can be useful to use a small logical page size

	// Extents are multi-page blocks used by the FIFO queues
	int physicalExtentSize;
	int pagesPerExtent;

	Reference<IPageEncryptionKeyProvider> keyProvider;
	Promise<Void> keyProviderInitialized;

	Reference<PriorityMultiLock> ioLock;

	int64_t pageCacheBytes;

	// The header will be written to / read from disk as a smallestPhysicalBlock sized chunk.
	Reference<ArenaPage> headerPage;
	PagerCommitHeader header;

	Reference<ArenaPage> lastCommittedHeaderPage;
	PagerCommitHeader lastCommittedHeader;

	// Pages - pages known to be in the file, truncations complete to that size
	int64_t filePageCount;
	// Pages that will be in file once fileExtension is ready
	int64_t filePageCountPending;
	// Future representing the end of all pending truncations
	Future<Void> fileExtension;

	int desiredPageSize;
	int desiredExtentSize;

	Version recoveryVersion;
	std::string filename;
	bool memoryOnly;
	bool fileInitialized = false;

	PageCacheT pageCache;

	// The extent cache isn't a normal cache, it isn't allowed to evict things.  It is populated
	// during recovery with remap queue extents and then cleared.
	PageCacheT::Evictor extentCacheDummyEvictor{ std::numeric_limits<int64_t>::max() };
	PageCacheT extentCache{ &extentCacheDummyEvictor };

	Promise<Void> closedPromise;
	Promise<Void> errorPromise;
	Future<Void> commitFuture;

	// The operations vector is used to hold all disk writes made by the Pager, but could also hold
	// other operations that need to be waited on before a commit can finish.
	std::vector<Future<Void>> operations;

	Future<Void> recoverFuture;
	Future<Void> remapCleanupFuture;
	bool remapCleanupStop;

	Reference<IAsyncFile> pageFile;

	LogicalPageQueueT freeList;

	// The delayed free list will be approximately in Version order.
	// TODO: Make this an ordered container some day.
	DelayedFreePageQueueT delayedFreeList;

	RemapQueueT remapQueue;
	LogicalPageQueueT extentFreeList;
	ExtentUsedListQueueT extentUsedList;
	uint64_t remapCleanupWindowBytes;
	Reference<FlowLock> concurrentExtentReads;
	std::unordered_set<PhysicalPageID> remapDestinationsSimOnly;

	struct SnapshotEntry {
		Version version;
		Reference<DWALPagerSnapshot> snapshot;
	};

	struct SnapshotEntryLessThanVersion {
		bool operator()(Version v, const SnapshotEntry& snapshot) { return v < snapshot.version; }

		bool operator()(const SnapshotEntry& snapshot, Version v) { return snapshot.version < v; }
	};

	// TODO: Better data structure
	PageToVersionedMapT remappedPages;

	// Readable snapshots in version order
	std::deque<SnapshotEntry> snapshots;
	Version oldestSnapshotVersion;
};

// Prevents pager from reusing freed pages from version until the snapshot is destroyed
class DWALPagerSnapshot : public IPagerSnapshot, public ReferenceCounted<DWALPagerSnapshot> {
public:
	DWALPagerSnapshot(DWALPager* pager, Key meta, Version version) : pager(pager), version(version), metaKey(meta) {}
	~DWALPagerSnapshot() override {}

	Future<Reference<const ArenaPage>> getPhysicalPage(PagerEventReasons reason,
	                                                   unsigned int level,
	                                                   LogicalPageID pageID,
	                                                   int priority,
	                                                   bool cacheable,
	                                                   bool noHit) override {

		return map(pager->readPageAtVersion(reason, level, pageID, priority, version, cacheable, noHit),
		           [=](Reference<ArenaPage> p) { return Reference<const ArenaPage>(std::move(p)); });
	}

	Future<Reference<const ArenaPage>> getMultiPhysicalPage(PagerEventReasons reason,
	                                                        unsigned int level,
	                                                        VectorRef<PhysicalPageID> pageIDs,
	                                                        int priority,
	                                                        bool cacheable,
	                                                        bool noHit) override {

		return map(pager->readMultiPage(reason, level, pageIDs, priority, cacheable, noHit),
		           [=](Reference<ArenaPage> p) { return Reference<const ArenaPage>(std::move(p)); });
	}

	Key getMetaKey() const override { return metaKey; }

	Version getVersion() const override { return version; }

	void addref() override { ReferenceCounted<DWALPagerSnapshot>::addref(); }

	void delref() override { ReferenceCounted<DWALPagerSnapshot>::delref(); }

	DWALPager* pager;
	Version version;
	Key metaKey;
};

void DWALPager::expireSnapshots(Version v) {
	debug_printf("DWALPager(%s) expiring snapshots through %" PRId64 " snapshot count %d\n",
	             filename.c_str(),
	             v,
	             (int)snapshots.size());

	// While there is more than one snapshot and the front snapshot is older than v and has no other reference holders
	while (snapshots.size() > 1 && snapshots.front().version < v && snapshots.front().snapshot->isSoleOwner()) {
		debug_printf("DWALPager(%s) expiring snapshot for %" PRId64 " soleOwner=%d\n",
		             filename.c_str(),
		             snapshots.front().version,
		             snapshots.front().snapshot->isSoleOwner());

		// Expire the snapshot and update the oldest snapshot version
		snapshots.pop_front();
		oldestSnapshotVersion = snapshots.front().version;
	}
}

Reference<IPagerSnapshot> DWALPager::getReadSnapshot(Version v) {
	auto i = std::upper_bound(snapshots.begin(), snapshots.end(), v, SnapshotEntryLessThanVersion());
	if (i == snapshots.begin()) {
		throw version_invalid();
	}
	--i;
	return i->snapshot;
}

// TODO: Move this to a flow header once it is mature.
struct SplitStringRef {
	StringRef a;
	StringRef b;

	SplitStringRef(StringRef a = StringRef(), StringRef b = StringRef()) : a(a), b(b) {}

	SplitStringRef(Arena& arena, const SplitStringRef& toCopy) : a(toStringRef(arena)), b() {}

	SplitStringRef prefix(int len) const {
		if (len <= a.size()) {
			return SplitStringRef(a.substr(0, len));
		}
		len -= a.size();
		return SplitStringRef(a, b.substr(0, len));
	}

	StringRef toStringRef(Arena& arena) const {
		StringRef c = makeString(size(), arena);
		memcpy(mutateString(c), a.begin(), a.size());
		memcpy(mutateString(c) + a.size(), b.begin(), b.size());
		return c;
	}

	Standalone<StringRef> toStringRef() const {
		Arena a;
		return Standalone<StringRef>(toStringRef(a), a);
	}

	int size() const { return a.size() + b.size(); }

	int expectedSize() const { return size(); }

	std::string toString() const { return format("%s%s", a.toString().c_str(), b.toString().c_str()); }

	std::string toHexString() const { return format("%s%s", a.toHexString().c_str(), b.toHexString().c_str()); }

	struct const_iterator {
		const uint8_t* ptr;
		const uint8_t* end{ nullptr };
		const uint8_t* next{ nullptr };

		inline bool operator==(const const_iterator& rhs) const { return ptr == rhs.ptr; }
		inline bool operator!=(const const_iterator& rhs) const { return !(*this == rhs); }

		inline const_iterator& operator++() {
			++ptr;
			if (ptr == end) {
				ptr = next;
			}
			return *this;
		}

		inline const_iterator& operator+(int n) {
			ptr += n;
			if (ptr >= end) {
				ptr = next + (ptr - end);
			}
			return *this;
		}

		inline uint8_t operator*() const { return *ptr; }
	};

	inline const_iterator begin() const { return { a.begin(), a.end(), b.begin() }; }

	inline const_iterator end() const { return { b.end() }; }

	template <typename StringT>
	int compare(const StringT& rhs) const {
		auto j = begin();
		auto k = rhs.begin();
		auto jEnd = end();
		auto kEnd = rhs.end();

		while (j != jEnd && k != kEnd) {
			int cmp = *j - *k;
			if (cmp != 0) {
				return cmp;
			}
		}

		// If we've reached the end of *this, then values are equal if rhs is also exhausted, otherwise *this is less
		// than rhs
		if (j == jEnd) {
			return k == kEnd ? 0 : -1;
		}

		return 1;
	}
};

// A BTree node link is a list of LogicalPageID's whose contents should be concatenated together.
typedef VectorRef<LogicalPageID> BTreeNodeLinkRef;
typedef Standalone<BTreeNodeLinkRef> BTreeNodeLink;

constexpr LogicalPageID maxPageID = (LogicalPageID)-1;

std::string toString(BTreeNodeLinkRef id) {
	return std::string("BTreePageID") + toString(id.begin(), id.end());
}

struct RedwoodRecordRef {
	typedef uint8_t byte;

	RedwoodRecordRef(KeyRef key = KeyRef(), Optional<ValueRef> value = {}) : key(key), value(value) {}

	RedwoodRecordRef(Arena& arena, const RedwoodRecordRef& toCopy) : key(arena, toCopy.key) {
		if (toCopy.value.present()) {
			value = ValueRef(arena, toCopy.value.get());
		}
	}

	typedef KeyRef Partial;

	void updateCache(Optional<Partial>& cache, Arena& arena) const { cache = KeyRef(arena, key); }

	KeyValueRef toKeyValueRef() const { return KeyValueRef(key, value.get()); }

	// RedwoodRecordRefs are used for both internal and leaf pages of the BTree.
	// Boundary records in internal pages are made from leaf records.
	// These functions make creating and working with internal page records more convenient.
	inline BTreeNodeLinkRef getChildPage() const {
		ASSERT(value.present());
		return BTreeNodeLinkRef((LogicalPageID*)value.get().begin(), value.get().size() / sizeof(LogicalPageID));
	}

	inline void setChildPage(BTreeNodeLinkRef id) {
		value = ValueRef((const uint8_t*)id.begin(), id.size() * sizeof(LogicalPageID));
	}

	inline void setChildPage(Arena& arena, BTreeNodeLinkRef id) {
		value = ValueRef(arena, (const uint8_t*)id.begin(), id.size() * sizeof(LogicalPageID));
	}

	inline RedwoodRecordRef withPageID(BTreeNodeLinkRef id) const {
		return RedwoodRecordRef(key, ValueRef((const uint8_t*)id.begin(), id.size() * sizeof(LogicalPageID)));
	}

	inline RedwoodRecordRef withoutValue() const { return RedwoodRecordRef(key); }

	inline RedwoodRecordRef withMaxPageID() const {
		return RedwoodRecordRef(key, StringRef((uint8_t*)&maxPageID, sizeof(maxPageID)));
	}

	// Truncate (key, version, part) tuple to len bytes.
	void truncate(int len) {
		ASSERT(len <= key.size());
		key = key.substr(0, len);
	}

	// Find the common key prefix between two records, assuming that the first skipLen bytes are the same
	inline int getCommonPrefixLen(const RedwoodRecordRef& other, int skipLen = 0) const {
		return skipLen + commonPrefixLength(key, other.key, skipLen);
	}

	// Compares and orders by key, version, chunk.total, chunk.start, value
	// This is the same order that delta compression uses for prefix borrowing
	int compare(const RedwoodRecordRef& rhs, int skip = 0) const {
		int keySkip = std::min(skip, key.size());
		int cmp = key.compareSuffix(rhs.key, keySkip);

		if (cmp == 0) {
			cmp = value.compare(rhs.value);
		}
		return cmp;
	}

	bool sameUserKey(const StringRef& k, int skipLen) const {
		// Keys are the same if the sizes are the same and either the skipLen is longer or the non-skipped suffixes are
		// the same.
		return (key.size() == k.size()) && (key.substr(skipLen) == k.substr(skipLen));
	}

	bool sameExceptValue(const RedwoodRecordRef& rhs, int skipLen = 0) const { return sameUserKey(rhs.key, skipLen); }

	// TODO: Use SplitStringRef (unless it ends up being slower)
	KeyRef key;
	Optional<ValueRef> value;

	int expectedSize() const { return key.expectedSize() + value.expectedSize(); }
	int kvBytes() const { return expectedSize(); }

#pragma pack(push, 1)
	struct Delta {

		uint8_t flags;

		// Four field sizing schemes ranging from 3 to 8 bytes, with 3 being the most common.
		union {
			struct {
				uint8_t prefixLength;
				uint8_t suffixLength;
				uint8_t valueLength;
			} LengthFormat0;

			struct {
				uint8_t prefixLength;
				uint8_t suffixLength;
				uint16_t valueLength;
			} LengthFormat1;

			struct {
				uint8_t prefixLength;
				uint8_t suffixLength;
				uint32_t valueLength;
			} LengthFormat2;

			struct {
				uint16_t prefixLength;
				uint16_t suffixLength;
				uint32_t valueLength;
			} LengthFormat3;
		};

		constexpr static int LengthFormatSizes[] = { sizeof(LengthFormat0),
			                                         sizeof(LengthFormat1),
			                                         sizeof(LengthFormat2),
			                                         sizeof(LengthFormat3) };

		// Serialized Format
		//
		// Flags - 1 byte
		//    1 bit - borrow source is prev ancestor (otherwise next ancestor)
		//    1 bit - item is deleted
		//    1 bit - has value (different from a zero-length value, which is still a value)
		//    3 unused bits
		//    2 bits - length fields format
		//
		// Length fields using 3 to 8 bytes total depending on length fields format
		//
		// Byte strings
		//    Value bytes
		//    Key suffix bytes

		enum EFlags {
			PREFIX_SOURCE_PREV = 0x80,
			IS_DELETED = 0x40,
			HAS_VALUE = 0x20,
			// 3 unused bits
			LENGTHS_FORMAT = 0x03
		};

		// Figure out which length format must be used for the given lengths
		static inline int determineLengthFormat(int prefixLength, int suffixLength, int valueLength) {
			// Large prefix or suffix length, which should be rare, is format 3
			if (prefixLength > 0xFF || suffixLength > 0xFF) {
				return 3;
			} else if (valueLength < 0x100) {
				return 0;
			} else if (valueLength < 0x10000) {
				return 1;
			} else {
				return 2;
			}
		}

		// Large prefix or suffix length, which should be rare, is format 3
		byte* data() const {
			switch (flags & LENGTHS_FORMAT) {
			case 0:
				return (byte*)(&LengthFormat0 + 1);
			case 1:
				return (byte*)(&LengthFormat1 + 1);
			case 2:
				return (byte*)(&LengthFormat2 + 1);
			case 3:
			default:
				return (byte*)(&LengthFormat3 + 1);
			}
		}

		int getKeyPrefixLength() const {
			switch (flags & LENGTHS_FORMAT) {
			case 0:
				return LengthFormat0.prefixLength;
			case 1:
				return LengthFormat1.prefixLength;
			case 2:
				return LengthFormat2.prefixLength;
			case 3:
			default:
				return LengthFormat3.prefixLength;
			}
		}

		int getKeySuffixLength() const {
			switch (flags & LENGTHS_FORMAT) {
			case 0:
				return LengthFormat0.suffixLength;
			case 1:
				return LengthFormat1.suffixLength;
			case 2:
				return LengthFormat2.suffixLength;
			case 3:
			default:
				return LengthFormat3.suffixLength;
			}
		}

		int getValueLength() const {
			switch (flags & LENGTHS_FORMAT) {
			case 0:
				return LengthFormat0.valueLength;
			case 1:
				return LengthFormat1.valueLength;
			case 2:
				return LengthFormat2.valueLength;
			case 3:
			default:
				return LengthFormat3.valueLength;
			}
		}

		StringRef getKeySuffix() const { return StringRef(data() + getValueLength(), getKeySuffixLength()); }

		StringRef getValue() const { return StringRef(data(), getValueLength()); }

		bool hasValue() const { return flags & HAS_VALUE; }

		void setPrefixSource(bool val) {
			if (val) {
				flags |= PREFIX_SOURCE_PREV;
			} else {
				flags &= ~PREFIX_SOURCE_PREV;
			}
		}

		bool getPrefixSource() const { return flags & PREFIX_SOURCE_PREV; }

		void setDeleted(bool val) {
			if (val) {
				flags |= IS_DELETED;
			} else {
				flags &= ~IS_DELETED;
			}
		}

		bool getDeleted() const { return flags & IS_DELETED; }

		// DeltaTree interface
		RedwoodRecordRef apply(const RedwoodRecordRef& base, Arena& arena) const {
			int keyPrefixLen = getKeyPrefixLength();
			int keySuffixLen = getKeySuffixLength();
			int valueLen = hasValue() ? getValueLength() : 0;
			byte* pData = data();

			StringRef k;
			// If there is a key suffix, reconstitute the complete key into a contiguous string
			if (keySuffixLen > 0) {
				k = makeString(keyPrefixLen + keySuffixLen, arena);
				memcpy(mutateString(k), base.key.begin(), keyPrefixLen);
				memcpy(mutateString(k) + keyPrefixLen, pData + valueLen, keySuffixLen);
			} else {
				// Otherwise just reference the base key's memory
				k = base.key.substr(0, keyPrefixLen);
			}

			return RedwoodRecordRef(k, hasValue() ? ValueRef(pData, valueLen) : Optional<ValueRef>());
		}

		// DeltaTree interface
		RedwoodRecordRef apply(const Partial& cache) {
			return RedwoodRecordRef(cache, hasValue() ? Optional<ValueRef>(getValue()) : Optional<ValueRef>());
		}

		RedwoodRecordRef apply(Arena& arena, const Partial& baseKey, Optional<Partial>& cache) {
			int keyPrefixLen = getKeyPrefixLength();
			int keySuffixLen = getKeySuffixLength();
			int valueLen = hasValue() ? getValueLength() : 0;
			byte* pData = data();

			StringRef k;
			// If there is a key suffix, reconstitute the complete key into a contiguous string
			if (keySuffixLen > 0) {
				k = makeString(keyPrefixLen + keySuffixLen, arena);
				memcpy(mutateString(k), baseKey.begin(), keyPrefixLen);
				memcpy(mutateString(k) + keyPrefixLen, pData + valueLen, keySuffixLen);
			} else {
				// Otherwise just reference the base key's memory
				k = baseKey.substr(0, keyPrefixLen);
			}
			cache = k;

			return RedwoodRecordRef(k, hasValue() ? ValueRef(pData, valueLen) : Optional<ValueRef>());
		}

		RedwoodRecordRef apply(Arena& arena, const RedwoodRecordRef& base, Optional<Partial>& cache) {
			return apply(arena, base.key, cache);
		}

		int size() const {
			int size = 1;
			switch (flags & LENGTHS_FORMAT) {
			case 0:
				return size + sizeof(LengthFormat0) + LengthFormat0.suffixLength + LengthFormat0.valueLength;
			case 1:
				return size + sizeof(LengthFormat1) + LengthFormat1.suffixLength + LengthFormat1.valueLength;
			case 2:
				return size + sizeof(LengthFormat2) + LengthFormat2.suffixLength + LengthFormat2.valueLength;
			case 3:
			default:
				return size + sizeof(LengthFormat3) + LengthFormat3.suffixLength + LengthFormat3.valueLength;
			}
		}

		std::string toString() const {
			std::string flagString = " ";
			if (flags & PREFIX_SOURCE_PREV) {
				flagString += "PrefixSource|";
			}
			if (flags & IS_DELETED) {
				flagString += "IsDeleted|";
			}
			if (hasValue()) {
				flagString += "HasValue|";
			}
			int lengthFormat = flags & LENGTHS_FORMAT;

			int prefixLen = getKeyPrefixLength();
			int keySuffixLen = getKeySuffixLength();
			int valueLen = getValueLength();

			return format("lengthFormat: %d  totalDeltaSize: %d  flags: %s  prefixLen: %d  keySuffixLen: %d  "
			              "valueLen %d  raw: %s",
			              lengthFormat,
			              size(),
			              flagString.c_str(),
			              prefixLen,
			              keySuffixLen,
			              valueLen,
			              StringRef((const uint8_t*)this, size()).toHexString().c_str());
		}
	};

	// Using this class as an alternative for Delta enables reading a DeltaTree2<RecordRef> while only decoding
	// its values, so the Reader does not require the original prev/next ancestors.
	struct DeltaValueOnly : Delta {
		RedwoodRecordRef apply(const RedwoodRecordRef& base, Arena& arena) const {
			return RedwoodRecordRef(KeyRef(), hasValue() ? Optional<ValueRef>(getValue()) : Optional<ValueRef>());
		}

		RedwoodRecordRef apply(const Partial& cache) {
			return RedwoodRecordRef(KeyRef(), hasValue() ? Optional<ValueRef>(getValue()) : Optional<ValueRef>());
		}

		RedwoodRecordRef apply(Arena& arena, const RedwoodRecordRef& base, Optional<Partial>& cache) {
			cache = KeyRef();
			return RedwoodRecordRef(KeyRef(), hasValue() ? Optional<ValueRef>(getValue()) : Optional<ValueRef>());
		}
	};
#pragma pack(pop)

	bool operator==(const RedwoodRecordRef& rhs) const { return compare(rhs) == 0; }

	bool operator!=(const RedwoodRecordRef& rhs) const { return compare(rhs) != 0; }

	bool operator<(const RedwoodRecordRef& rhs) const { return compare(rhs) < 0; }

	bool operator>(const RedwoodRecordRef& rhs) const { return compare(rhs) > 0; }

	bool operator<=(const RedwoodRecordRef& rhs) const { return compare(rhs) <= 0; }

	bool operator>=(const RedwoodRecordRef& rhs) const { return compare(rhs) >= 0; }

	// Worst case overhead means to assume that either the prefix length or the suffix length
	// could contain the full key size
	int deltaSize(const RedwoodRecordRef& base, int skipLen, bool worstCaseOverhead) const {
		int prefixLen = getCommonPrefixLen(base, skipLen);
		int keySuffixLen = key.size() - prefixLen;
		int valueLen = value.present() ? value.get().size() : 0;

		int formatType;
		if (worstCaseOverhead) {
			formatType = Delta::determineLengthFormat(key.size(), key.size(), valueLen);
		} else {
			formatType = Delta::determineLengthFormat(prefixLen, keySuffixLen, valueLen);
		}

		return 1 + Delta::LengthFormatSizes[formatType] + keySuffixLen + valueLen;
	}

	// commonPrefix between *this and base can be passed if known
	int writeDelta(Delta& d, const RedwoodRecordRef& base, int keyPrefixLen = -1) const {
		d.flags = value.present() ? Delta::HAS_VALUE : 0;

		if (keyPrefixLen < 0) {
			keyPrefixLen = getCommonPrefixLen(base, 0);
		}

		StringRef keySuffix = key.substr(keyPrefixLen);
		int valueLen = value.present() ? value.get().size() : 0;

		int formatType = Delta::determineLengthFormat(keyPrefixLen, keySuffix.size(), valueLen);
		d.flags |= formatType;

		switch (formatType) {
		case 0:
			d.LengthFormat0.prefixLength = keyPrefixLen;
			d.LengthFormat0.suffixLength = keySuffix.size();
			d.LengthFormat0.valueLength = valueLen;
			break;
		case 1:
			d.LengthFormat1.prefixLength = keyPrefixLen;
			d.LengthFormat1.suffixLength = keySuffix.size();
			d.LengthFormat1.valueLength = valueLen;
			break;
		case 2:
			d.LengthFormat2.prefixLength = keyPrefixLen;
			d.LengthFormat2.suffixLength = keySuffix.size();
			d.LengthFormat2.valueLength = valueLen;
			break;
		case 3:
		default:
			d.LengthFormat3.prefixLength = keyPrefixLen;
			d.LengthFormat3.suffixLength = keySuffix.size();
			d.LengthFormat3.valueLength = valueLen;
			break;
		}

		uint8_t* wptr = d.data();

		// Write value bytes
		if (valueLen > 0) {
			wptr = value.get().copyTo(wptr);
		}

		// Write key suffix string
		wptr = keySuffix.copyTo(wptr);

		return wptr - (uint8_t*)&d;
	}

	static std::string kvformat(StringRef s, int hexLimit = -1) {
		bool hex = false;

		for (auto c : s) {
			if (!isprint(c)) {
				hex = true;
				break;
			}
		}

		return hex ? s.toHexString(hexLimit) : s.toString();
	}

	std::string toString(bool leaf = true) const {
		std::string r;
		r += format("'%s' => ", key.printable().c_str());
		if (value.present()) {
			if (leaf) {
				r += format("'%s'", kvformat(value.get()).c_str());
			} else {
				r += format("[%s]", ::toString(getChildPage()).c_str());
			}
		} else {
			r += "(absent)";
		}
		return r;
	}
};

struct BTreePage {
	typedef DeltaTree2<RedwoodRecordRef> BinaryTree;
	typedef DeltaTree2<RedwoodRecordRef, RedwoodRecordRef::DeltaValueOnly> ValueTree;

#pragma pack(push, 1)
	struct {
		// treeOffset allows for newer versions to have additional fields but older code to read them
		uint8_t treeOffset;
		uint8_t height;
		uint32_t kvBytes;
	};

#pragma pack(pop)

	void init(unsigned int height, unsigned int kvBytes) {
		treeOffset = sizeof(BTreePage);
		this->height = height;
		this->kvBytes = kvBytes;
	}

	int size() const { return treeOffset + tree()->size(); }

	uint8_t* treeBuffer() const { return (uint8_t*)this + treeOffset; }
	BinaryTree* tree() { return (BinaryTree*)treeBuffer(); }
	BinaryTree* tree() const { return (BinaryTree*)treeBuffer(); }
	ValueTree* valueTree() const { return (ValueTree*)treeBuffer(); }

	bool isLeaf() const { return height == 1; }

	std::string toString(const char* context,
	                     BTreeNodeLinkRef id,
	                     Version ver,
	                     const RedwoodRecordRef& lowerBound,
	                     const RedwoodRecordRef& upperBound) const {
		std::string r;
		r += format("BTreePage op=%s %s @%" PRId64
		            " ptr=%p height=%d count=%d kvBytes=%d\n  lowerBound: %s\n  upperBound: %s\n",
		            context,
		            ::toString(id).c_str(),
		            ver,
		            this,
		            height,
		            (int)tree()->numItems,
		            (int)kvBytes,
		            lowerBound.toString(false).c_str(),
		            upperBound.toString(false).c_str());
		try {
			if (tree()->numItems > 0) {
				// This doesn't use the cached reader for the page because it is only for debugging purposes,
				// a cached reader may not exist
				BinaryTree::Cursor c(makeReference<BinaryTree::DecodeCache>(lowerBound, upperBound), tree());

				c.moveFirst();
				ASSERT(c.valid());

				do {
					r += "  ";
					r += c.get().toString(height == 1);

					// Out of range entries are annotated but can actually be valid, as they can be the result of
					// subtree deletion followed by incremental insertions of records in the deleted range being added
					// to an adjacent subtree which is logically expanded encompass the deleted range but still is using
					// the original subtree boundaries as DeltaTree2 boundaries.
					bool tooLow = c.get().withoutValue() < lowerBound.withoutValue();
					bool tooHigh = c.get().withoutValue() >= upperBound.withoutValue();
					if (tooLow || tooHigh) {
						if (tooLow) {
							r += " (below decode lower bound)";
						}
						if (tooHigh) {
							r += " (at or above decode upper bound)";
						}
					}
					r += "\n";

				} while (c.moveNext());
			}
		} catch (Error& e) {
			debug_printf("BTreePage::toString ERROR: %s\n", e.what());
			debug_printf("BTreePage::toString partial result: %s\n", r.c_str());
			throw;
		}

		// All appends to r end in a linefeed, remove the final one.
		r.resize(r.size() - 1);
		return r;
	}
};

struct BoundaryRefAndPage {
	Standalone<RedwoodRecordRef> lowerBound;
	Reference<ArenaPage> firstPage;
	std::vector<Reference<ArenaPage>> extPages;

	std::string toString() const {
		return format("[%s, %d pages]", lowerBound.toString().c_str(), extPages.size() + (firstPage ? 1 : 0));
	}
};

// DecodeBoundaryVerifier provides simulation-only verification of DeltaTree boundaries between
// reads and writes by using a static structure to track boundaries used during DeltaTree generation
// for all writes and updates across cold starts and virtual process restarts.
class DecodeBoundaryVerifier {
	struct DecodeBoundaries {
		Key lower;
		Key upper;
		unsigned int height;
		Optional<int64_t> domainId;
		bool empty() const { return lower.empty() && upper.empty(); }
	};

	typedef std::map<Version, DecodeBoundaries> BoundariesByVersion;
	std::unordered_map<LogicalPageID, BoundariesByVersion> boundariesByPageID;
	int boundarySampleSize = 1000;
	int boundaryPopulation = 0;
	Reference<IPageEncryptionKeyProvider> keyProvider;

public:
	std::vector<Key> boundarySamples;

	// Sample rate of pages to be scanned to verify if all entries in the page meet domain prefix requirement.
	double domainPrefixScanProbability = 0.01;
	uint64_t domainPrefixScanCount = 0;

	static DecodeBoundaryVerifier* getVerifier(std::string name) {
		static std::map<std::string, DecodeBoundaryVerifier> verifiers;
		// Only use verifier in a non-restarted simulation so that all page writes are captured
		if (g_network->isSimulated() && !g_simulator->restarted) {
			return &verifiers[name];
		}
		return nullptr;
	}

	void setKeyProvider(Reference<IPageEncryptionKeyProvider> kp) { keyProvider = kp; }

	void sampleBoundary(Key b) {
		if (boundaryPopulation <= boundarySampleSize) {
			boundarySamples.push_back(b);
		} else if (deterministicRandom()->random01() < ((double)boundarySampleSize / boundaryPopulation)) {
			boundarySamples[deterministicRandom()->randomInt(0, boundarySampleSize)] = b;
		}
		++boundaryPopulation;
	}

	Key getSample() const {
		if (boundarySamples.empty()) {
			return Key();
		}
		return deterministicRandom()->randomChoice(boundarySamples);
	}

	bool update(BTreeNodeLinkRef id,
	            Version v,
	            Key lowerBound,
	            Key upperBound,
	            unsigned int height,
	            Optional<int64_t> domainId) {
		sampleBoundary(lowerBound);
		sampleBoundary(upperBound);
		debug_printf("decodeBoundariesUpdate %s %s '%s' to '%s', %u, %s\n",
		             ::toString(id).c_str(),
		             ::toString(v).c_str(),
		             lowerBound.printable().c_str(),
		             upperBound.printable().c_str(),
		             height,
		             Traceable<decltype(domainId)>::toString(domainId).c_str());

		if (domainId.present()) {
			ASSERT(keyProvider && keyProvider->enableEncryptionDomain());
			if (!keyProvider->keyFitsInDomain(domainId.get(), lowerBound, true)) {
				fprintf(stderr,
				        "Page lower bound not in domain: %s %s, domain id %s, lower bound '%s'\n",
				        ::toString(id).c_str(),
				        ::toString(v).c_str(),
				        ::toString(domainId).c_str(),
				        lowerBound.printable().c_str());
				return false;
			}
		}

		auto& b = boundariesByPageID[id.front()][v];
		ASSERT(b.empty());
		b = { lowerBound, upperBound, height, domainId };
		return true;
	}

	bool verify(LogicalPageID id,
	            Version v,
	            Key lowerBound,
	            Key upperBound,
	            Optional<int64_t> domainId,
	            BTreePage::BinaryTree::Cursor& cursor) {
		auto i = boundariesByPageID.find(id);
		ASSERT(i != boundariesByPageID.end());
		ASSERT(!i->second.empty());

		auto b = i->second.upper_bound(v);
		--b;
		if (b->second.lower != lowerBound || b->second.upper != upperBound) {
			fprintf(stderr,
			        "Boundary mismatch on %s %s\nUsing:\n\t'%s'\n\t'%s'\nWritten %s:\n\t'%s'\n\t'%s'\n",
			        ::toString(id).c_str(),
			        ::toString(v).c_str(),
			        lowerBound.printable().c_str(),
			        upperBound.printable().c_str(),
			        ::toString(b->first).c_str(),
			        b->second.lower.printable().c_str(),
			        b->second.upper.printable().c_str());
			return false;
		}

		if (!b->second.domainId.present()) {
			ASSERT(!keyProvider || !keyProvider->enableEncryptionDomain());
			ASSERT(!domainId.present());
		} else {
			ASSERT(keyProvider->enableEncryptionDomain());
			if (b->second.domainId != domainId) {
				fprintf(stderr,
				        "Page encrypted with incorrect domain: %s %s, using %s, written %s\n",
				        ::toString(id).c_str(),
				        ::toString(v).c_str(),
				        ::toString(domainId).c_str(),
				        ::toString(b->second.domainId).c_str());
				return false;
			}
			ASSERT(domainId.present());
			auto checkKeyFitsInDomain = [&]() -> bool {
				if (!keyProvider->keyFitsInDomain(domainId.get(), cursor.get().key, b->second.height > 1)) {
					fprintf(stderr,
					        "Encryption domain mismatch on %s, %s, domain: %s, key %s\n",
					        ::toString(id).c_str(),
					        ::toString(v).c_str(),
					        ::toString(domainId).c_str(),
					        cursor.get().key.printable().c_str());
					return false;
				}
				return true;
			};
			cursor.moveFirst();
			if (cursor.valid() && !checkKeyFitsInDomain()) {
				return false;
			}
			cursor.moveLast();
			if (cursor.valid() && !checkKeyFitsInDomain()) {
				return false;
			}
		}

		return true;
	}

	void updatePageId(Version v, LogicalPageID oldID, LogicalPageID newID) {
		auto& old = boundariesByPageID[oldID];
		ASSERT(!old.empty());
		auto i = old.end();
		--i;
		boundariesByPageID[newID][v] = i->second;
		debug_printf("decodeBoundariesUpdate copy %s %s to %s '%s' to '%s'\n",
		             ::toString(v).c_str(),
		             ::toString(oldID).c_str(),
		             ::toString(newID).c_str(),
		             i->second.lower.printable().c_str(),
		             i->second.upper.printable().c_str());
	}

	void removeAfterVersion(Version version) {
		auto i = boundariesByPageID.begin();
		while (i != boundariesByPageID.end()) {
			auto v = i->second.upper_bound(version);
			while (v != i->second.end()) {
				debug_printf("decodeBoundariesUpdate remove %s %s '%s' to '%s'\n",
				             ::toString(v->first).c_str(),
				             ::toString(i->first).c_str(),
				             v->second.lower.printable().c_str(),
				             v->second.upper.printable().c_str());
				v = i->second.erase(v);
			}

			if (i->second.empty()) {
				debug_printf("decodeBoundariesUpdate remove empty map for %s\n", ::toString(i->first).c_str());
				i = boundariesByPageID.erase(i);
			} else {
				++i;
			}
		}
	}
};

class VersionedBTree {
public:
	// The first possible internal record possible in the tree
	static RedwoodRecordRef dbBegin;
	// A record which is greater than the last possible record in the tree
	static RedwoodRecordRef dbEnd;

	struct LazyClearQueueEntry {
		uint8_t height;
		Version version;
		BTreeNodeLink pageID;

		bool operator<(const LazyClearQueueEntry& rhs) const { return version < rhs.version; }

		int readFromBytes(const uint8_t* src) {
			height = *(uint8_t*)src;
			src += sizeof(uint8_t);
			version = *(Version*)src;
			src += sizeof(Version);
			int count = *src++;
			pageID = BTreeNodeLinkRef((LogicalPageID*)src, count);
			return bytesNeeded();
		}

		int bytesNeeded() const {
			return sizeof(uint8_t) + sizeof(Version) + 1 + (pageID.size() * sizeof(LogicalPageID));
		}

		int writeToBytes(uint8_t* dst) const {
			*(uint8_t*)dst = height;
			dst += sizeof(uint8_t);
			*(Version*)dst = version;
			dst += sizeof(Version);
			*dst++ = pageID.size();
			memcpy(dst, pageID.begin(), pageID.size() * sizeof(LogicalPageID));
			return bytesNeeded();
		}

		std::string toString() const { return format("{%s @%" PRId64 "}", ::toString(pageID).c_str(), version); }
	};

	typedef FIFOQueue<LazyClearQueueEntry> LazyClearQueueT;

	struct ParentInfo {
		ParentInfo() {
			count = 0;
			bits = 0;
		}
		void clear() {
			count = 0;
			bits = 0;
		}

		static uint32_t mask(LogicalPageID id) { return 1 << (id & 31); }

		void pageUpdated(LogicalPageID child) {
			auto m = mask(child);
			if ((bits & m) == 0) {
				bits |= m;
				++count;
			}
		}

		bool maybeUpdated(LogicalPageID child) const { return (mask(child) & bits) != 0; }

		uint32_t bits;
		int count;
	};

	typedef std::unordered_map<LogicalPageID, ParentInfo> ParentInfoMapT;

	struct BTreeCommitHeader {
		constexpr static FileIdentifier file_identifier = 10847329;
		constexpr static unsigned int FORMAT_VERSION = 17;

		// Maximum size of the root pointer
		constexpr static int maxRootPointerSize = 3000 / sizeof(LogicalPageID);

		// This serves as the format version for the entire tree, individual pages will not be versioned
		uint32_t formatVersion;
		EncodingType encodingType;
		uint8_t height;
		LazyClearQueueT::QueueState lazyDeleteQueue;
		BTreeNodeLink root;
		EncryptionAtRestMode encryptionMode = EncryptionAtRestMode::DISABLED; // since 7.3

		std::string toString() {
			return format("{formatVersion=%d  height=%d  root=%s  lazyDeleteQueue=%s encryptionMode=%s}",
			              (int)formatVersion,
			              (int)height,
			              ::toString(root).c_str(),
			              lazyDeleteQueue.toString().c_str(),
			              encryptionMode.toString().c_str());
		}

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar, formatVersion, encodingType, height, lazyDeleteQueue, root, encryptionMode);
		}
	};

	// All async opts on the btree are based on pager reads, writes, and commits, so
	// we can mostly forward these next few functions to the pager
	Future<Void> getError() const { return m_errorPromise.getFuture() || m_pager->getError(); }

	Future<Void> onClosed() const { return m_pager->onClosed(); }

	void close_impl(bool dispose) {
		auto* pager = m_pager;
		delete this;
		if (dispose)
			pager->dispose();
		else
			pager->close();
	}

	void dispose() { return close_impl(true); }

	void close() { return close_impl(false); }

	StorageBytes getStorageBytes() const { return m_pager->getStorageBytes(); }

	// Set key to value as of the next commit
	// The new value is not readable until after the next commit is completed.
	void set(KeyValueRef keyValue) {
		++m_mutationCount;
		++g_redwoodMetrics.metric.opSet;
		g_redwoodMetrics.metric.opSetKeyBytes += keyValue.key.size();
		g_redwoodMetrics.metric.opSetValueBytes += keyValue.value.size();
		m_pBuffer->insert(keyValue.key).mutation().setBoundaryValue(m_pBuffer->copyToArena(keyValue.value));
	}

	void clear(KeyRangeRef clearedRange) {
		++m_mutationCount;
		ASSERT(!clearedRange.empty());
		// Optimization for single key clears to create just one mutation boundary instead of two
		if (clearedRange.singleKeyRange()) {
			++g_redwoodMetrics.metric.opClear;
			++g_redwoodMetrics.metric.opClearKey;
			m_pBuffer->insert(clearedRange.begin).mutation().clearBoundary();
			return;
		}
		++g_redwoodMetrics.metric.opClear;
		MutationBuffer::iterator iBegin = m_pBuffer->insert(clearedRange.begin);
		MutationBuffer::iterator iEnd = m_pBuffer->insert(clearedRange.end);

		iBegin.mutation().clearAll();
		++iBegin;
		m_pBuffer->erase(iBegin, iEnd);
	}

	void setOldestReadableVersion(Version v) { m_newOldestVersion = v; }

	Version getOldestReadableVersion() const { return m_pager->getOldestReadableVersion(); }

	Version getLastCommittedVersion() const { return m_pager->getLastCommittedVersion(); }

	// VersionedBTree takes ownership of pager
	VersionedBTree(IPager2* pager,
	               std::string name,
	               UID logID,
	               Reference<AsyncVar<ServerDBInfo> const> db,
	               Optional<EncryptionAtRestMode> expectedEncryptionMode,
	               EncodingType encodingType = EncodingType::MAX_ENCODING_TYPE,
	               Reference<IPageEncryptionKeyProvider> keyProvider = {},
	               Reference<GetEncryptCipherKeysMonitor> encryptionMonitor = {})
	  : m_pager(pager), m_db(db), m_expectedEncryptionMode(expectedEncryptionMode), m_encodingType(encodingType),
	    m_enforceEncodingType(false), m_keyProvider(keyProvider), m_encryptionMonitor(encryptionMonitor),
	    m_pBuffer(nullptr), m_mutationCount(0), m_name(name), m_logID(logID),
	    m_pBoundaryVerifier(DecodeBoundaryVerifier::getVerifier(name)) {
		m_pDecodeCacheMemory = m_pager->getPageCachePenaltySource();
		m_lazyClearActor = 0;
		m_init = init_impl(this);
		m_latestCommit = m_init;
	}

	Future<EncryptionAtRestMode> encryptionMode() { return m_encryptionMode.getFuture(); }

																#line 17705 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via makeEmptyRoot()
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class MakeEmptyRootActor>
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class MakeEmptyRootActorState {
															#line 17711 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	MakeEmptyRootActorState(VersionedBTree* const& self) 
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self),
															#line 5062 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		   page(self->m_pager->newPageBuffer())
															#line 17720 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("makeEmptyRoot", reinterpret_cast<unsigned long>(this));

	}
	~MakeEmptyRootActorState() 
	{
		fdb_probe_actor_destroy("makeEmptyRoot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 5063 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			page->init(self->m_encodingType, PageType::BTreeNode, 1);
															#line 5064 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (page->isEncrypted())
															#line 17737 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5065 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				StrictFuture<ArenaPage::EncryptionKey> __when_expr_0 = self->m_keyProvider->getLatestDefaultEncryptionKey();
															#line 5065 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (static_cast<MakeEmptyRootActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17743 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<MakeEmptyRootActor*>(this)->actor_wait_state = 1;
															#line 5065 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >*>(static_cast<MakeEmptyRootActor*>(this)));
															#line 17748 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MakeEmptyRootActorState();
		static_cast<MakeEmptyRootActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 5069 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		BTreePage* btpage = (BTreePage*)page->mutateData();
															#line 5070 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		btpage->init(1, 0);
															#line 5071 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		btpage->tree()->build(page->dataSize() - sizeof(BTreePage), nullptr, nullptr, nullptr, nullptr);
															#line 5072 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<MakeEmptyRootActor*>(this)->SAV<Reference<ArenaPage>>::futures) { (void)(page); this->~MakeEmptyRootActorState(); static_cast<MakeEmptyRootActor*>(this)->destroy(); return 0; }
															#line 17782 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<MakeEmptyRootActor*>(this)->SAV< Reference<ArenaPage> >::value()) Reference<ArenaPage>(std::move(page)); // state_var_RVO
		this->~MakeEmptyRootActorState();
		static_cast<MakeEmptyRootActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
															#line 5066 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 17794 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
															#line 5066 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->encryptionKey = k;
															#line 17803 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(ArenaPage::EncryptionKey const& k,int loopDepth) 
	{
		loopDepth = a_body1cont2(k, loopDepth);

		return loopDepth;
	}
	int a_body1when1(ArenaPage::EncryptionKey && k,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(k), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MakeEmptyRootActor*>(this)->actor_wait_state > 0) static_cast<MakeEmptyRootActor*>(this)->actor_wait_state = 0;
		static_cast<MakeEmptyRootActor*>(this)->ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >::remove();

	}
	void a_callback_fire(ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey const& value) 
	{
		fdb_probe_actor_enter("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeEmptyRootActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >*,ArenaPage::EncryptionKey && value) 
	{
		fdb_probe_actor_enter("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeEmptyRootActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >*,Error err) 
	{
		fdb_probe_actor_enter("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakeEmptyRootActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeEmptyRoot", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	VersionedBTree* self;
															#line 5062 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<ArenaPage> page;
															#line 17887 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via makeEmptyRoot()
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class MakeEmptyRootActor final : public Actor<Reference<ArenaPage>>, public ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >, public FastAllocated<MakeEmptyRootActor>, public MakeEmptyRootActorState<MakeEmptyRootActor> {
															#line 17892 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<MakeEmptyRootActor>::operator new;
	using FastAllocated<MakeEmptyRootActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15986219520771402496UL, 4552967140114319616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<ArenaPage>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >;
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	MakeEmptyRootActor(VersionedBTree* const& self) 
															#line 17909 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Reference<ArenaPage>>(),
		   MakeEmptyRootActorState<MakeEmptyRootActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("makeEmptyRoot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13359752408764136192UL, 17936091595489778688UL);
		ActorExecutionContextHelper __helper(static_cast<MakeEmptyRootActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("makeEmptyRoot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("makeEmptyRoot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MakeEmptyRootActor, 0, ArenaPage::EncryptionKey >*)0, actor_cancelled()); break;
		}

	}
};
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Reference<ArenaPage>> makeEmptyRoot( VersionedBTree* const& self ) {
															#line 5061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Reference<ArenaPage>>(new MakeEmptyRootActor(self));
															#line 17941 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 5074 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	void toTraceEvent(TraceEvent& e) const {
		m_pager->toTraceEvent(e);
		m_lazyClearQueue.toTraceEvent(e, "LazyClearQueue");
	}

																#line 17951 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via incrementalLazyClear()
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class IncrementalLazyClearActor>
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class IncrementalLazyClearActorState {
															#line 17957 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	IncrementalLazyClearActorState(VersionedBTree* const& self) 
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 17964 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("incrementalLazyClear", reinterpret_cast<unsigned long>(this));

	}
	~IncrementalLazyClearActorState() 
	{
		fdb_probe_actor_destroy("incrementalLazyClear", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 5081 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT(self->m_lazyClearActor.isReady());
															#line 5082 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_lazyClearStop = false;
															#line 5085 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			snapshot = self->m_pager->getReadSnapshot(self->m_pager->getLastCommittedVersion());
															#line 5087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			freedPages = 0;
															#line 5089 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			;
															#line 17987 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IncrementalLazyClearActorState();
		static_cast<IncrementalLazyClearActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 5176 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: freed %d pages, %s has %" PRId64 " entries\n", freedPages, self->m_lazyClearQueue.name.c_str(), self->m_lazyClearQueue.numEntries);
															#line 5180 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<IncrementalLazyClearActor*>(this)->SAV<int>::futures) { (void)(freedPages); this->~IncrementalLazyClearActorState(); static_cast<IncrementalLazyClearActor*>(this)->destroy(); return 0; }
															#line 18012 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<IncrementalLazyClearActor*>(this)->SAV< int >::value()) int(std::move(freedPages)); // state_var_RVO
		this->~IncrementalLazyClearActorState();
		static_cast<IncrementalLazyClearActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 5090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		toPop = SERVER_KNOBS->REDWOOD_LAZY_CLEAR_BATCH_SIZE_PAGES;
															#line 5091 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		entries = std::vector<std::pair<LazyClearQueueEntry, Future<Reference<const ArenaPage>>>>();
															#line 5092 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		entries.reserve(toPop);
															#line 5095 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		;
															#line 18037 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 5115 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		i = int();
															#line 5116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		i = 0;
															#line 18061 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 5095 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!(toPop > 0))
															#line 18077 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 5096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Optional<LazyClearQueueEntry>> __when_expr_0 = self->m_lazyClearQueue.pop();
															#line 5096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 18085 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state = 1;
															#line 5096 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >*>(static_cast<IncrementalLazyClearActor*>(this)));
															#line 18090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<LazyClearQueueEntry> const& q,int loopDepth) 
	{
															#line 5097 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: popped %s\n", toString(q).c_str());
															#line 5098 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!q.present())
															#line 18114 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 5103 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		entries.emplace_back(q.get(), self->readPage(self, PagerEventReasons::LazyClear, q.get().height, snapshot.getPtr(), q.get().pageID, ioLeafPriority, true, false));
															#line 5112 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		--toPop;
															#line 18122 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<LazyClearQueueEntry> && q,int loopDepth) 
	{
															#line 5097 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: popped %s\n", toString(q).c_str());
															#line 5098 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!q.present())
															#line 18133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 5103 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		entries.emplace_back(q.get(), self->readPage(self, PagerEventReasons::LazyClear, q.get().height, snapshot.getPtr(), q.get().pageID, ioLeafPriority, true, false));
															#line 5112 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		--toPop;
															#line 18141 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<LazyClearQueueEntry> const& q,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(q, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<LazyClearQueueEntry> && q,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(q), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state > 0) static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state = 0;
		static_cast<IncrementalLazyClearActor*>(this)->ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >::remove();

	}
	void a_callback_fire(ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >*,Optional<LazyClearQueueEntry> const& value) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >*,Optional<LazyClearQueueEntry> && value) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >*,Error err) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 5170 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (toPop > 0 || (freedPages >= SERVER_KNOBS->REDWOOD_LAZY_CLEAR_MIN_PAGES && self->m_lazyClearStop) || (freedPages >= SERVER_KNOBS->REDWOOD_LAZY_CLEAR_MAX_PAGES))
															#line 18225 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 5116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!(i < entries.size()))
															#line 18244 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 5117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Reference<const ArenaPage>> __when_expr_1 = entries[i].second;
															#line 5117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 18252 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state = 2;
															#line 5117 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >*>(static_cast<IncrementalLazyClearActor*>(this)));
															#line 18257 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Reference<const ArenaPage> const& p,int loopDepth) 
	{
															#line 5118 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		const LazyClearQueueEntry& entry = entries[i].first;
															#line 5119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		const BTreePage& btPage = *(const BTreePage*)p->data();
															#line 5120 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(btPage.height == entry.height);
															#line 5121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto& metrics = g_redwoodMetrics.level(entry.height).metrics;
															#line 5123 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: processing %s\n", toString(entry).c_str());
															#line 5126 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(entry.height > 1);
															#line 5130 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		BTreePage::ValueTree::Cursor c(makeReference<BTreePage::ValueTree::DecodeCache>(dbBegin, dbEnd), btPage.valueTree());
															#line 5132 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(c.moveFirst());
															#line 5133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		Version v = entry.version;
															#line 5134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for(;1;) {
															#line 5135 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (c.get().value.present())
															#line 18299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5136 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				BTreeNodeLinkRef btChildPageID = c.get().getChildPage();
															#line 5138 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (entry.height == 2)
															#line 18305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5139 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("LazyClear: freeing leaf child %s\n", toString(btChildPageID).c_str());
															#line 5140 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->freeBTreePage(1, btChildPageID, v);
															#line 5141 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					freedPages += btChildPageID.size();
															#line 5142 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearFree += 1;
															#line 5143 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearFreeExt += (btChildPageID.size() - 1);
															#line 18317 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 5146 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("LazyClear: queuing child %s\n", toString(btChildPageID).c_str());
															#line 5147 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->m_lazyClearQueue.pushFront( LazyClearQueueEntry{ (uint8_t)(entry.height - 1), v, btChildPageID });
															#line 5149 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearRequeue += 1;
															#line 5150 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearRequeueExt += (btChildPageID.size() - 1);
															#line 18329 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 5153 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!c.moveNext())
															#line 18334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
				break;
			}
		}
															#line 5159 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: freeing queue entry %s\n", toString(entry.pageID).c_str());
															#line 5160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeBTreePage(entry.height, entry.pageID, v);
															#line 5161 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		freedPages += entry.pageID.size();
															#line 5162 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		metrics.lazyClearFree += 1;
															#line 5163 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		metrics.lazyClearFreeExt += entry.pageID.size() - 1;
															#line 5116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++i;
															#line 18351 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Reference<const ArenaPage> && p,int loopDepth) 
	{
															#line 5118 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		const LazyClearQueueEntry& entry = entries[i].first;
															#line 5119 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		const BTreePage& btPage = *(const BTreePage*)p->data();
															#line 5120 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(btPage.height == entry.height);
															#line 5121 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		auto& metrics = g_redwoodMetrics.level(entry.height).metrics;
															#line 5123 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: processing %s\n", toString(entry).c_str());
															#line 5126 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(entry.height > 1);
															#line 5130 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		BTreePage::ValueTree::Cursor c(makeReference<BTreePage::ValueTree::DecodeCache>(dbBegin, dbEnd), btPage.valueTree());
															#line 5132 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		ASSERT(c.moveFirst());
															#line 5133 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		Version v = entry.version;
															#line 5134 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		for(;1;) {
															#line 5135 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (c.get().value.present())
															#line 18380 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5136 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				BTreeNodeLinkRef btChildPageID = c.get().getChildPage();
															#line 5138 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (entry.height == 2)
															#line 18386 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5139 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("LazyClear: freeing leaf child %s\n", toString(btChildPageID).c_str());
															#line 5140 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->freeBTreePage(1, btChildPageID, v);
															#line 5141 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					freedPages += btChildPageID.size();
															#line 5142 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearFree += 1;
															#line 5143 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearFreeExt += (btChildPageID.size() - 1);
															#line 18398 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 5146 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					debug_printf("LazyClear: queuing child %s\n", toString(btChildPageID).c_str());
															#line 5147 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->m_lazyClearQueue.pushFront( LazyClearQueueEntry{ (uint8_t)(entry.height - 1), v, btChildPageID });
															#line 5149 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearRequeue += 1;
															#line 5150 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					metrics.lazyClearRequeueExt += (btChildPageID.size() - 1);
															#line 18410 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 5153 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!c.moveNext())
															#line 18415 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
				break;
			}
		}
															#line 5159 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("LazyClear: freeing queue entry %s\n", toString(entry.pageID).c_str());
															#line 5160 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->freeBTreePage(entry.height, entry.pageID, v);
															#line 5161 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		freedPages += entry.pageID.size();
															#line 5162 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		metrics.lazyClearFree += 1;
															#line 5163 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		metrics.lazyClearFreeExt += entry.pageID.size() - 1;
															#line 5116 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		++i;
															#line 18432 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Reference<const ArenaPage> const& p,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(p, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Reference<const ArenaPage> && p,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(std::move(p), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state > 0) static_cast<IncrementalLazyClearActor*>(this)->actor_wait_state = 0;
		static_cast<IncrementalLazyClearActor*>(this)->ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >::remove();

	}
	void a_callback_fire(ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >*,Reference<const ArenaPage> const& value) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >*,Reference<const ArenaPage> && value) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >*,Error err) 
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	VersionedBTree* self;
															#line 5085 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Reference<IPagerSnapshot> snapshot;
															#line 5087 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int freedPages;
															#line 5090 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int toPop;
															#line 5091 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	std::vector<std::pair<LazyClearQueueEntry, Future<Reference<const ArenaPage>>>> entries;
															#line 5115 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	int i;
															#line 18524 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via incrementalLazyClear()
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class IncrementalLazyClearActor final : public Actor<int>, public ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >, public ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >, public FastAllocated<IncrementalLazyClearActor>, public IncrementalLazyClearActorState<IncrementalLazyClearActor> {
															#line 18529 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<IncrementalLazyClearActor>::operator new;
	using FastAllocated<IncrementalLazyClearActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12423951304950818816UL, 6324796194701120000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >;
friend struct ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >;
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	IncrementalLazyClearActor(VersionedBTree* const& self) 
															#line 18547 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<int>(),
		   IncrementalLazyClearActorState<IncrementalLazyClearActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("incrementalLazyClear", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4568414028304052480UL, 13805437888623186688UL);
		ActorExecutionContextHelper __helper(static_cast<IncrementalLazyClearActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("incrementalLazyClear");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("incrementalLazyClear", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IncrementalLazyClearActor, 0, Optional<LazyClearQueueEntry> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IncrementalLazyClearActor, 1, Reference<const ArenaPage> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<int> incrementalLazyClear( VersionedBTree* const& self ) {
															#line 5080 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<int>(new IncrementalLazyClearActor(self));
															#line 18580 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 5182 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	void checkOrUpdateEncodingType(const std::string& event,
	                               const EncryptionAtRestMode& encryptionMode,
	                               EncodingType& encodingType) {
		EncodingType expectedEncodingType = EncodingType::MAX_ENCODING_TYPE;
		if (encryptionMode == EncryptionAtRestMode::DISABLED) {
			expectedEncodingType = EncodingType::XXHash64;
		} else {
			expectedEncodingType = FLOW_KNOBS->ENCRYPT_HEADER_AUTH_TOKEN_ENABLED ? EncodingType::AESEncryptionWithAuth
			                                                                     : EncodingType::AESEncryption;
		}
		// Randomly enable XOR encryption in simulation. Also ignore encoding type mismatch if XOR encryption is set but
		// default encoding is expected.
		if (encodingType == EncodingType::MAX_ENCODING_TYPE) {
			encodingType = expectedEncodingType;
			if (encodingType == EncodingType::XXHash64 && g_network->isSimulated() && m_logID.hash() % 2 == 0) {
				encodingType = EncodingType::XOREncryption_TestOnly;
			}
		} else if (encodingType != expectedEncodingType) {
			// In simulation we could enable xor encryption for testing. Ignore encoding type mismatch in such a case.
			if (!(g_network->isSimulated() && encodingType == EncodingType::XOREncryption_TestOnly &&
			      expectedEncodingType == EncodingType::XXHash64)) {
				TraceEvent(SevWarnAlways, "RedwoodBTreeMismatchEncryptionModeAndEncodingType")
				    .detail("InstanceName", m_pager->getName())
				    .detail("Event", event)
				    .detail("EncryptionMode", encryptionMode)
				    .detail("ExpectedEncodingType", expectedEncodingType)
				    .detail("ActualEncodingType", encodingType);
				throw encrypt_mode_mismatch();
			}
		}
	}

	void initEncryptionKeyProvider() {
		if (!m_keyProvider.isValid()) {
			switch (m_encodingType) {
			case EncodingType::XXHash64:
				m_keyProvider = makeReference<NullEncryptionKeyProvider>();
				break;
			case EncodingType::XOREncryption_TestOnly:
				m_keyProvider = makeReference<XOREncryptionKeyProvider_TestOnly>(m_name);
				break;
			case EncodingType::AESEncryption:
				ASSERT(m_expectedEncryptionMode.present());
				ASSERT(m_db.isValid());
				m_keyProvider = makeReference<AESEncryptionKeyProvider<AESEncryption>>(
				    m_db, m_expectedEncryptionMode.get(), m_encryptionMonitor);
				break;
			case EncodingType::AESEncryptionWithAuth:
				ASSERT(m_expectedEncryptionMode.present());
				ASSERT(m_db.isValid());
				m_keyProvider = makeReference<AESEncryptionKeyProvider<AESEncryptionWithAuth>>(
				    m_db, m_expectedEncryptionMode.get(), m_encryptionMonitor);
				break;
			default:
				ASSERT(false);
			}
		} else {
			ASSERT_EQ(m_encodingType, m_keyProvider->expectedEncodingType());
		}
		m_pager->setEncryptionKeyProvider(m_keyProvider);
		if (m_pBoundaryVerifier != nullptr) {
			m_pBoundaryVerifier->setKeyProvider(m_keyProvider);
		}
	}

																#line 18650 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via init_impl()
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class Init_implActor>
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Init_implActorState {
															#line 18656 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Init_implActorState(VersionedBTree* const& self) 
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		 : self(self)
															#line 18663 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
	{
		fdb_probe_actor_create("init_impl", reinterpret_cast<unsigned long>(this));

	}
	~Init_implActorState() 
	{
		fdb_probe_actor_destroy("init_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 5249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->m_pager->init();
															#line 5249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18680 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Init_implActor*>(this)->actor_wait_state = 1;
															#line 5249 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 0, Void >*>(static_cast<Init_implActor*>(this)));
															#line 18685 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Init_implActorState();
		static_cast<Init_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 5250 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_pBuffer.reset(new MutationBuffer());
															#line 5252 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_blockSize = self->m_pager->getLogicalPageSize();
															#line 5253 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_newOldestVersion = self->m_pager->getOldestReadableVersion();
															#line 5255 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("Recovered pager to version %" PRId64 ", oldest version is %" PRId64 "\n", self->getLastCommittedVersion(), self->m_newOldestVersion);
															#line 5260 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->m_pBoundaryVerifier != nullptr)
															#line 18716 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 5261 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_pBoundaryVerifier->removeAfterVersion(self->getLastCommittedVersion());
															#line 18720 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 5264 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		btreeHeader = self->m_pager->getCommitRecord();
															#line 5265 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (btreeHeader.size() == 0)
															#line 18726 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 5268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!self->m_expectedEncryptionMode.present())
															#line 18730 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				Error err = storage_engine_not_initialized();
															#line 5279 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (g_network->isSimulated() && g_simulator->restarted)
															#line 18736 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5280 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					err = err.asInjectedFault();
															#line 18740 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
															#line 5282 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(err, loopDepth);
															#line 18744 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5285 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_encryptionMode.send(self->m_expectedEncryptionMode.get());
															#line 5286 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->checkOrUpdateEncodingType("NewBTree", self->m_expectedEncryptionMode.get(), self->m_encodingType);
															#line 5287 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->initEncryptionKeyProvider();
															#line 5288 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_enforceEncodingType = isEncodingTypeEncrypted(self->m_encodingType);
															#line 5290 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.formatVersion = BTreeCommitHeader::FORMAT_VERSION;
															#line 5291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.encodingType = self->m_encodingType;
															#line 5292 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.height = 1;
															#line 5293 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.encryptionMode = self->m_expectedEncryptionMode.get();
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<LogicalPageID> __when_expr_1 = self->m_pager->newPageID();
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18766 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<Init_implActor*>(this)->actor_wait_state = 2;
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 1, LogicalPageID >*>(static_cast<Init_implActor*>(this)));
															#line 18771 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 5312 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header = ObjectReader::fromStringRef<BTreeCommitHeader>(btreeHeader, Unversioned());
															#line 5314 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_header.formatVersion != BTreeCommitHeader::FORMAT_VERSION)
															#line 18780 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				Error e = unsupported_format_version();
															#line 5316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				TraceEvent(SevWarn, "RedwoodBTreeVersionUnsupported") .error(e) .detail("Version", self->m_header.formatVersion) .detail("ExpectedVersion", BTreeCommitHeader::FORMAT_VERSION);
															#line 5320 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 18788 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5323 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_expectedEncryptionMode.present())
															#line 18792 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5324 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->m_header.encryptionMode != self->m_expectedEncryptionMode.get())
															#line 18796 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5325 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					TraceEvent(SevWarnAlways, "RedwoodBTreeEncryptionModeMismatched") .detail("InstanceName", self->m_pager->getName()) .detail("ExpectedEncryptionMode", self->m_expectedEncryptionMode) .detail("StoredEncryptionMode", self->m_header.encryptionMode);
															#line 5329 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					return a_body1Catch1(encrypt_mode_mismatch(), loopDepth);
															#line 18802 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 5331 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->m_expectedEncryptionMode = self->m_header.encryptionMode;
															#line 18808 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
			else
			{
															#line 5334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->m_expectedEncryptionMode = self->m_header.encryptionMode;
															#line 18815 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_encryptionMode.send(self->m_header.encryptionMode);
															#line 5338 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT_NE(EncodingType::MAX_ENCODING_TYPE, self->m_header.encodingType);
															#line 5339 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_encodingType == EncodingType::MAX_ENCODING_TYPE)
															#line 18823 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5340 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->m_encodingType = self->m_header.encodingType;
															#line 18827 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
			else
			{
															#line 5341 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->m_encodingType != self->m_header.encodingType)
															#line 18833 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5342 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					TraceEvent(SevWarn, "RedwoodBTreeUnexpectedEncodingType") .detail("InstanceName", self->m_pager->getName()) .detail("UsingEncodingType", self->m_encodingType) .detail("ExistingEncodingType", self->m_header.encodingType);
															#line 5346 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					return a_body1Catch1(unexpected_encoding_type(), loopDepth);
															#line 18839 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 5350 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->checkOrUpdateEncodingType("ExistingBTree", self->m_header.encryptionMode, self->m_encodingType);
															#line 5351 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->initEncryptionKeyProvider();
															#line 5352 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_enforceEncodingType = isEncodingTypeEncrypted(self->m_encodingType);
															#line 5354 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_lazyClearQueue.recover(self->m_pager, self->m_header.lazyDeleteQueue, "LazyClearQueueRecovered");
															#line 5355 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("BTree recovered.\n");
															#line 18852 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 5250 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_pBuffer.reset(new MutationBuffer());
															#line 5252 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_blockSize = self->m_pager->getLogicalPageSize();
															#line 5253 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_newOldestVersion = self->m_pager->getOldestReadableVersion();
															#line 5255 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("Recovered pager to version %" PRId64 ", oldest version is %" PRId64 "\n", self->getLastCommittedVersion(), self->m_newOldestVersion);
															#line 5260 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (self->m_pBoundaryVerifier != nullptr)
															#line 18870 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 5261 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_pBoundaryVerifier->removeAfterVersion(self->getLastCommittedVersion());
															#line 18874 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		}
															#line 5264 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		btreeHeader = self->m_pager->getCommitRecord();
															#line 5265 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (btreeHeader.size() == 0)
															#line 18880 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		{
															#line 5268 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (!self->m_expectedEncryptionMode.present())
															#line 18884 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5274 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				Error err = storage_engine_not_initialized();
															#line 5279 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (g_network->isSimulated() && g_simulator->restarted)
															#line 18890 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5280 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					err = err.asInjectedFault();
															#line 18894 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
															#line 5282 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(err, loopDepth);
															#line 18898 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5285 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_encryptionMode.send(self->m_expectedEncryptionMode.get());
															#line 5286 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->checkOrUpdateEncodingType("NewBTree", self->m_expectedEncryptionMode.get(), self->m_encodingType);
															#line 5287 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->initEncryptionKeyProvider();
															#line 5288 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_enforceEncodingType = isEncodingTypeEncrypted(self->m_encodingType);
															#line 5290 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.formatVersion = BTreeCommitHeader::FORMAT_VERSION;
															#line 5291 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.encodingType = self->m_encodingType;
															#line 5292 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.height = 1;
															#line 5293 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header.encryptionMode = self->m_expectedEncryptionMode.get();
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			StrictFuture<LogicalPageID> __when_expr_1 = self->m_pager->newPageID();
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18920 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<Init_implActor*>(this)->actor_wait_state = 2;
															#line 5295 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 1, LogicalPageID >*>(static_cast<Init_implActor*>(this)));
															#line 18925 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 5312 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_header = ObjectReader::fromStringRef<BTreeCommitHeader>(btreeHeader, Unversioned());
															#line 5314 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_header.formatVersion != BTreeCommitHeader::FORMAT_VERSION)
															#line 18934 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5315 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				Error e = unsupported_format_version();
															#line 5316 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				TraceEvent(SevWarn, "RedwoodBTreeVersionUnsupported") .error(e) .detail("Version", self->m_header.formatVersion) .detail("ExpectedVersion", BTreeCommitHeader::FORMAT_VERSION);
															#line 5320 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 18942 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5323 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_expectedEncryptionMode.present())
															#line 18946 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5324 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->m_header.encryptionMode != self->m_expectedEncryptionMode.get())
															#line 18950 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5325 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					TraceEvent(SevWarnAlways, "RedwoodBTreeEncryptionModeMismatched") .detail("InstanceName", self->m_pager->getName()) .detail("ExpectedEncryptionMode", self->m_expectedEncryptionMode) .detail("StoredEncryptionMode", self->m_header.encryptionMode);
															#line 5329 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					return a_body1Catch1(encrypt_mode_mismatch(), loopDepth);
															#line 18956 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
				else
				{
															#line 5331 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					self->m_expectedEncryptionMode = self->m_header.encryptionMode;
															#line 18962 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
			else
			{
															#line 5334 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->m_expectedEncryptionMode = self->m_header.encryptionMode;
															#line 18969 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
															#line 5336 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_encryptionMode.send(self->m_header.encryptionMode);
															#line 5338 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			ASSERT_NE(EncodingType::MAX_ENCODING_TYPE, self->m_header.encodingType);
															#line 5339 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			if (self->m_encodingType == EncodingType::MAX_ENCODING_TYPE)
															#line 18977 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			{
															#line 5340 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				self->m_encodingType = self->m_header.encodingType;
															#line 18981 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			}
			else
			{
															#line 5341 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
				if (self->m_encodingType != self->m_header.encodingType)
															#line 18987 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				{
															#line 5342 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					TraceEvent(SevWarn, "RedwoodBTreeUnexpectedEncodingType") .detail("InstanceName", self->m_pager->getName()) .detail("UsingEncodingType", self->m_encodingType) .detail("ExistingEncodingType", self->m_header.encodingType);
															#line 5346 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
					return a_body1Catch1(unexpected_encoding_type(), loopDepth);
															#line 18993 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
				}
			}
															#line 5350 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->checkOrUpdateEncodingType("ExistingBTree", self->m_header.encryptionMode, self->m_encodingType);
															#line 5351 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->initEncryptionKeyProvider();
															#line 5352 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_enforceEncodingType = isEncodingTypeEncrypted(self->m_encodingType);
															#line 5354 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			self->m_lazyClearQueue.recover(self->m_pager, self->m_header.lazyDeleteQueue, "LazyClearQueueRecovered");
															#line 5355 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
			debug_printf("BTree recovered.\n");
															#line 19006 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Init_implActor*>(this)->actor_wait_state > 0) static_cast<Init_implActor*>(this)->actor_wait_state = 0;
		static_cast<Init_implActor*>(this)->ActorCallback< Init_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Init_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Init_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Init_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 5357 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_lazyClearActor = 0;
															#line 5359 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		TraceEvent e(SevInfo, "RedwoodRecoveredBTree");
															#line 5360 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("FileName", self->m_name);
															#line 5361 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("OpenedExisting", btreeHeader.size() != 0);
															#line 5362 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.detail("LatestVersion", self->m_pager->getLastCommittedVersion());
															#line 5363 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_lazyClearQueue.toTraceEvent(e, "LazyClearQueue");
															#line 5364 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		e.log();
															#line 5366 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("Recovered btree at version %" PRId64 ": %s\n", self->m_pager->getLastCommittedVersion(), self->m_header.toString().c_str());
															#line 5370 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (!static_cast<Init_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Init_implActorState(); static_cast<Init_implActor*>(this)->destroy(); return 0; }
															#line 19107 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		new (&static_cast<Init_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Init_implActorState();
		static_cast<Init_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(LogicalPageID const& id,int loopDepth) 
	{
															#line 5296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_header.root = BTreeNodeLinkRef((LogicalPageID*)&id, 1);
															#line 5297 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("new root %s\n", toString(self->m_header.root).c_str());
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Reference<ArenaPage>> __when_expr_2 = makeEmptyRoot(self);
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19125 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<Init_implActor*>(this)->actor_wait_state = 3;
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*>(static_cast<Init_implActor*>(this)));
															#line 19130 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(LogicalPageID && id,int loopDepth) 
	{
															#line 5296 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_header.root = BTreeNodeLinkRef((LogicalPageID*)&id, 1);
															#line 5297 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("new root %s\n", toString(self->m_header.root).c_str());
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<Reference<ArenaPage>> __when_expr_2 = makeEmptyRoot(self);
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19145 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<Init_implActor*>(this)->actor_wait_state = 3;
															#line 5299 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*>(static_cast<Init_implActor*>(this)));
															#line 19150 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(LogicalPageID const& id,int loopDepth) 
	{
		loopDepth = a_body1cont4(id, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(LogicalPageID && id,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(id), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Init_implActor*>(this)->actor_wait_state > 0) static_cast<Init_implActor*>(this)->actor_wait_state = 0;
		static_cast<Init_implActor*>(this)->ActorCallback< Init_implActor, 1, LogicalPageID >::remove();

	}
	void a_callback_fire(ActorCallback< Init_implActor, 1, LogicalPageID >*,LogicalPageID const& value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Init_implActor, 1, LogicalPageID >*,LogicalPageID && value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Init_implActor, 1, LogicalPageID >*,Error err) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(Reference<ArenaPage> const& page,int loopDepth) 
	{
															#line 5302 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->setLogicalPageInfo(self->m_header.root.front(), invalidLogicalPageID);
															#line 5303 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_pager->updatePage(PagerEventReasons::MetaData, nonBtreeLevel, self->m_header.root, page);
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<LogicalPageID> __when_expr_3 = self->m_pager->newPageID();
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19240 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<Init_implActor*>(this)->actor_wait_state = 4;
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 3, LogicalPageID >*>(static_cast<Init_implActor*>(this)));
															#line 19245 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7(Reference<ArenaPage> && page,int loopDepth) 
	{
															#line 5302 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		page->setLogicalPageInfo(self->m_header.root.front(), invalidLogicalPageID);
															#line 5303 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_pager->updatePage(PagerEventReasons::MetaData, nonBtreeLevel, self->m_header.root, page);
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		StrictFuture<LogicalPageID> __when_expr_3 = self->m_pager->newPageID();
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		if (static_cast<Init_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19260 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<Init_implActor*>(this)->actor_wait_state = 4;
															#line 5305 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Init_implActor, 3, LogicalPageID >*>(static_cast<Init_implActor*>(this)));
															#line 19265 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Reference<ArenaPage> const& page,int loopDepth) 
	{
		loopDepth = a_body1cont7(page, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Reference<ArenaPage> && page,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(page), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Init_implActor*>(this)->actor_wait_state > 0) static_cast<Init_implActor*>(this)->actor_wait_state = 0;
		static_cast<Init_implActor*>(this)->ActorCallback< Init_implActor, 2, Reference<ArenaPage> >::remove();

	}
	void a_callback_fire(ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*,Reference<ArenaPage> const& value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*,Reference<ArenaPage> && value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*,Error err) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(LogicalPageID const& newQueuePage,int loopDepth) 
	{
															#line 5306 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_lazyClearQueue.create( self->m_pager, newQueuePage, "LazyClearQueue", self->m_pager->newLastQueueID(), false);
															#line 5308 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_header.lazyDeleteQueue = self->m_lazyClearQueue.getState();
															#line 5310 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("BTree created (but not committed)\n");
															#line 19353 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(LogicalPageID && newQueuePage,int loopDepth) 
	{
															#line 5306 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_lazyClearQueue.create( self->m_pager, newQueuePage, "LazyClearQueue", self->m_pager->newLastQueueID(), false);
															#line 5308 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		self->m_header.lazyDeleteQueue = self->m_lazyClearQueue.getState();
															#line 5310 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
		debug_printf("BTree created (but not committed)\n");
															#line 19366 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(LogicalPageID const& newQueuePage,int loopDepth) 
	{
		loopDepth = a_body1cont8(newQueuePage, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(LogicalPageID && newQueuePage,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(newQueuePage), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Init_implActor*>(this)->actor_wait_state > 0) static_cast<Init_implActor*>(this)->actor_wait_state = 0;
		static_cast<Init_implActor*>(this)->ActorCallback< Init_implActor, 3, LogicalPageID >::remove();

	}
	void a_callback_fire(ActorCallback< Init_implActor, 3, LogicalPageID >*,LogicalPageID const& value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Init_implActor, 3, LogicalPageID >*,LogicalPageID && value) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Init_implActor, 3, LogicalPageID >*,Error err) 
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	VersionedBTree* self;
															#line 5264 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Value btreeHeader;
															#line 19450 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
};
// This generated class is to be used only via init_impl()
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
class Init_implActor final : public Actor<Void>, public ActorCallback< Init_implActor, 0, Void >, public ActorCallback< Init_implActor, 1, LogicalPageID >, public ActorCallback< Init_implActor, 2, Reference<ArenaPage> >, public ActorCallback< Init_implActor, 3, LogicalPageID >, public FastAllocated<Init_implActor>, public Init_implActorState<Init_implActor> {
															#line 19455 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
public:
	using FastAllocated<Init_implActor>::operator new;
	using FastAllocated<Init_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15396319455609139456UL, 3750481061287151872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Init_implActor, 0, Void >;
friend struct ActorCallback< Init_implActor, 1, LogicalPageID >;
friend struct ActorCallback< Init_implActor, 2, Reference<ArenaPage> >;
friend struct ActorCallback< Init_implActor, 3, LogicalPageID >;
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	Init_implActor(VersionedBTree* const& self) 
															#line 19475 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
		 : Actor<Void>(),
		   Init_implActorState<Init_implActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1539219769978647296UL, 12543635476165692928UL);
		ActorExecutionContextHelper __helper(static_cast<Init_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Init_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Init_implActor, 1, LogicalPageID >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Init_implActor, 2, Reference<ArenaPage> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Init_implActor, 3, LogicalPageID >*)0, actor_cancelled()); break;
		}

	}
};
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
[[nodiscard]] static Future<Void> init_impl( VersionedBTree* const& self ) {
															#line 5248 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
	return Future<Void>(new Init_implActor(self));
															#line 19510 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
}

#line 5372 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"

	Future<Void> init() { return m_init; }

	virtual ~VersionedBTree() {
		// DecodeBoundaryVerifier objects outlive simulated processes.
		// Thus, if we did not clear the key providers here, each DecodeBoundaryVerifier object might
		// maintain references to untracked peers through its key provider. This would result in
		// errors when FlowTransport::removePeerReference is called to remove a peer that is no
		// longer tracked by FlowTransport::transport().
		if (m_pBoundaryVerifier != nullptr) {
			m_pBoundaryVerifier->setKeyProvider(Reference<IPageEncryptionKeyProvider>());
		}

		// This probably shouldn't be called directly (meaning deleting an instance directly) but it should be safe,
		// it will cancel init and commit and leave the pager alive but with potentially an incomplete set of
		// uncommitted writes so it should not be committed.
		m_latestCommit.cancel();
		m_lazyClearActor.cancel();
		m_init.cancel();
	}

	Future<Void> commit(Version v) {
		// Replace latest commit with a new one which waits on the old one
		m_latestCommit = commit_impl(this, v, m_latestCommit);
		return m_latestCommit;
	}

	// Clear all btree data, allow pager remap to fully process its queue, and verify final
	// page counts in pager and queues.
																#line 19543 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/build_output/fdbserver/VersionedBTree.actor.g.cpp"
// This generated class is to be used only via clearAllAndCheckSanity_impl()
															#line 5401 "/codebuild/output/src2177420633/src/github.com/apple/foundationdb/fdbserver/VersionedBTree.actor.cpp"
template <class ClearAllAndCheckSanity_implActor>
															#line 5401 "/codebuild/output/src2177420633/src/github.com