#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
/*
 * AsyncFileWriteChecker.actor.h
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#if defined(NO_INTELLISENSE) && !defined(ASYNC_FILE_WRITE_CHECKER_ACTOR_G_H)
#define ASYNC_FILE_WRITE_CHECKER_ACTOR_G_H
#include "fdbrpc/AsyncFileWriteChecker.actor.g.h"
#elif !defined(ASYNC_FILE_WRITE_CHECKER_ACTOR_H)
#define ASYNC_FILE_WRITE_CHECKER_ACTOR_H

#include "flow/IAsyncFile.h"
#include "crc32/crc32c.h"

#if VALGRIND
#include <memcheck.h>
#endif

#include "flow/actorcompiler.h"
static double millisecondsPerSecond = 1000;

// this class does checksum for the wrapped IAsyncFile in read and writes opertions.
// it maintains a dynamic data structure to store the recently written page and its checksum.
// it has an actor to continuously read and verify checksums for the recently written page,
// and also deletes the corresponding entry upon a successful to avoid using too much memory.
class AsyncFileWriteChecker : public IAsyncFile, public ReferenceCounted<AsyncFileWriteChecker> {
public:
	void addref() override { ReferenceCounted<AsyncFileWriteChecker>::addref(); }
	void delref() override { ReferenceCounted<AsyncFileWriteChecker>::delref(); }

	virtual StringRef getClassName() override { return "AsyncFileWriteChecker"_sr; }

	// For read() and write(), the data buffer must remain valid until the future is ready
	Future<int> read(void* data, int length, int64_t offset) override {
		// Lambda must hold a reference to this to keep it alive until after the read
		auto self = Reference<AsyncFileWriteChecker>::addRef(this);
		return map(m_f->read(data, length, offset), [self, data, offset](int r) {
			self->updateChecksumHistory(false, offset, r, (uint8_t*)data);
			return r;
		});
	}
	Future<Void> readZeroCopy(void** data, int* length, int64_t offset) override {
		// Lambda must hold a reference to this to keep it alive until after the read
		auto self = Reference<AsyncFileWriteChecker>::addRef(this);
		return map(m_f->readZeroCopy(data, length, offset), [self, data, length, offset](Void r) {
			self->updateChecksumHistory(false, offset, *length, (uint8_t*)data);
			return r;
		});
	}

	Future<Void> write(void const* data, int length, int64_t offset) override {
		auto pages = updateChecksumHistory(true, offset, length, (uint8_t*)data);
		auto self = Reference<AsyncFileWriteChecker>::addRef(this);
		return map(m_f->write(data, length, offset), [self, pages](Void r) {
			for (uint32_t page : pages) {
				self->writing.erase(page);
			}
			return r;
		});
	}

	Future<Void> truncate(int64_t size) override {
		auto self = Reference<AsyncFileWriteChecker>::addRef(this);
		return map(m_f->truncate(size), [self, size](Void r) {
			int maxFullPage = size / checksumHistoryPageSize;
			int oldSize = self->lru.size();
			self->lru.truncate(maxFullPage);
			checksumHistoryBudget.get() += oldSize - self->lru.size();
			return r;
		});
	}

	Future<Void> sync() override {
		auto self = Reference<AsyncFileWriteChecker>::addRef(this);
		return map(m_f->sync(), [self](Void r) {
			self->syncedTime = AsyncFileWriteChecker::transformTime(now());
			return r;
		});
	}

	Future<Void> flush() override { return m_f->flush(); }
	Future<int64_t> size() const override { return m_f->size(); }
	std::string getFilename() const override { return m_f->getFilename(); }
	void releaseZeroCopy(void* data, int length, int64_t offset) override {
		return m_f->releaseZeroCopy(data, length, offset);
	}
	int64_t debugFD() const override { return m_f->debugFD(); }

	struct WriteInfo {
		WriteInfo() : checksum(0), timestamp(0) {}
		uint32_t checksum;
		uint64_t timestamp; // keep a precision of ms
	};

	static uint64_t transformTime(double unixTime) { return (uint64_t)(unixTime * millisecondsPerSecond); }

	class LRU {
	private:
		uint64_t step;
		std::string fileName;
		std::map<uint64_t, uint32_t> stepToKey;
		std::map<uint32_t, uint64_t> keyToStep; // std::map is to support ::truncate
		std::unordered_map<uint32_t, AsyncFileWriteChecker::WriteInfo> pageContents;

	public:
		LRU(std::string _fileName) {
			step = 0;
			fileName = _fileName;
		}

		void update(uint32_t page, AsyncFileWriteChecker::WriteInfo writeInfo) {
			if (keyToStep.find(page) != keyToStep.end()) {
				// remove its old entry in stepToKey
				stepToKey.erase(keyToStep[page]);
			}
			keyToStep[page] = step;
			stepToKey[step] = page;
			pageContents[page] = writeInfo;
			step++;
		}

		void truncate(uint32_t page) {
			auto it = keyToStep.lower_bound(page);
			// iterate through keyToStep, to find corresponding entries in stepToKey
			while (it != keyToStep.end()) {
				uint64_t step = it->second;
				auto next = it;
				next++;
				keyToStep.erase(it);
				stepToKey.erase(step);
				it = next;
			}
		}

		uint32_t randomPage() {
			if (keyToStep.size() == 0) {
				return 0;
			}
			auto it = keyToStep.begin();
			std::advance(it, deterministicRandom()->randomInt(0, (int)keyToStep.size()));
			return it->first;
		}

		int size() { return keyToStep.size(); }

		bool exist(uint32_t page) { return keyToStep.find(page) != keyToStep.end(); }

		AsyncFileWriteChecker::WriteInfo find(uint32_t page) {
			auto it = keyToStep.find(page);
			if (it == keyToStep.end()) {
				TraceEvent(SevError, "LRUCheckerTryFindingPageNotExist")
				    .detail("FileName", fileName)
				    .detail("Page", page)
				    .log();
				return AsyncFileWriteChecker::WriteInfo();
			}
			return pageContents[page];
		}

		uint32_t leastRecentlyUsedPage() {
			if (stepToKey.size() == 0) {
				return 0;
			}
			return stepToKey.begin()->second;
		}

		void remove(uint32_t page) {
			if (keyToStep.find(page) == keyToStep.end()) {
				return;
			}
			pageContents.erase(page);
			stepToKey.erase(keyToStep[page]);
			keyToStep.erase(page);
		}
	};

	AsyncFileWriteChecker(Reference<IAsyncFile> f) : m_f(f), lru(f->getFilename()) {
		// Initialize the static history budget the first time (and only the first time) a file is opened.
		if (!checksumHistoryBudget.present()) {
			checksumHistoryBudget = FLOW_KNOBS->PAGE_WRITE_CHECKSUM_HISTORY;
		}
		pageBuffer = (void*)new char[checksumHistoryPageSize];
		totalCheckedSucceed = 0;
		totalCheckedFail = 0;
		lru = LRU(m_f->getFilename());
		checksumWorker = AsyncFileWriteChecker::sweep(this);
		checksumLogger = runChecksumLogger(this);
	}

	~AsyncFileWriteChecker() override {
		checksumHistoryBudget.get() += lru.size();
		delete[] reinterpret_cast<char*>(pageBuffer);
	}

private:
	Reference<IAsyncFile> m_f;
	Future<Void> checksumWorker;
	Future<Void> checksumLogger;
	LRU lru;
	void* pageBuffer;
	uint64_t totalCheckedFail, totalCheckedSucceed;
	// transform from unixTime(double) to uint64_t, to retain ms precision.
	uint64_t syncedTime;
	// to avoid concurrent operation, so that the continuous reader will skip a page if it is being written
	std::unordered_set<uint32_t> writing;
	// This is the most page checksum history blocks we will use across all files.
	static Optional<int> checksumHistoryBudget;
	static int checksumHistoryPageSize;

																#line 226 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
// This generated class is to be used only via sweep()
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
template <class SweepActor>
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
class SweepActorState {
															#line 232 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
public:
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	SweepActorState(AsyncFileWriteChecker* const& self) 
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		 : self(self)
															#line 239 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
	{
		fdb_probe_actor_create("sweep", reinterpret_cast<unsigned long>(this));

	}
	~SweepActorState() 
	{
		fdb_probe_actor_destroy("sweep", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 225 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
			;
															#line 254 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SweepActorState();
		static_cast<SweepActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 228 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		page = self->lru.leastRecentlyUsedPage();
															#line 229 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		;
															#line 286 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 234 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		int64_t offset = page * checksumHistoryPageSize;
															#line 236 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		StrictFuture<Void> __when_expr_1 = success(self->read(self->pageBuffer, checksumHistoryPageSize, offset));
															#line 236 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		if (static_cast<SweepActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 299 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SweepActor*>(this)->actor_wait_state = 2;
															#line 236 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SweepActor, 1, Void >*>(static_cast<SweepActor*>(this)));
															#line 304 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 229 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		if (!(self->writing.find(page) != self->writing.end() || page == 0))
															#line 320 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 231 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->ASYNC_FILE_WRITE_CHEKCER_CHECKING_DELAY);
															#line 231 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		if (static_cast<SweepActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 328 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<SweepActor*>(this)->actor_wait_state = 1;
															#line 231 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SweepActor, 0, Void >*>(static_cast<SweepActor*>(this)));
															#line 333 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		return a_body1loopBody1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SweepActor*>(this)->actor_wait_state > 0) static_cast<SweepActor*>(this)->actor_wait_state = 0;
		static_cast<SweepActor*>(this)->ActorCallback< SweepActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SweepActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SweepActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SweepActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SweepActor*>(this)->actor_wait_state > 0) static_cast<SweepActor*>(this)->actor_wait_state = 0;
		static_cast<SweepActor*>(this)->ActorCallback< SweepActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SweepActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SweepActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SweepActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	AsyncFileWriteChecker* self;
															#line 228 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	uint32_t page;
															#line 529 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
};
// This generated class is to be used only via sweep()
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
class SweepActor final : public Actor<Void>, public ActorCallback< SweepActor, 0, Void >, public ActorCallback< SweepActor, 1, Void >, public FastAllocated<SweepActor>, public SweepActorState<SweepActor> {
															#line 534 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
public:
	using FastAllocated<SweepActor>::operator new;
	using FastAllocated<SweepActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2197393322214228992UL, 8245099897361947392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SweepActor, 0, Void >;
friend struct ActorCallback< SweepActor, 1, Void >;
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	SweepActor(AsyncFileWriteChecker* const& self) 
															#line 552 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		 : Actor<Void>(),
		   SweepActorState<SweepActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("sweep", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18397400345488559104UL, 7725592744864208384UL);
		ActorExecutionContextHelper __helper(static_cast<SweepActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("sweep");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("sweep", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SweepActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SweepActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
[[nodiscard]] Future<Void> sweep( AsyncFileWriteChecker* const& self ) {
															#line 224 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	return Future<Void>(new SweepActor(self));
															#line 585 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
}

#line 239 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"

																#line 590 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
// This generated class is to be used only via runChecksumLogger()
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
template <class RunChecksumLoggerActor>
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
class RunChecksumLoggerActorState {
															#line 596 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
public:
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	RunChecksumLoggerActorState(AsyncFileWriteChecker* const& self) 
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		 : self(self),
															#line 241 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		   delayDuration(FLOW_KNOBS->ASYNC_FILE_WRITE_CHEKCER_LOGGING_INTERVAL)
															#line 605 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
	{
		fdb_probe_actor_create("runChecksumLogger", reinterpret_cast<unsigned long>(this));

	}
	~RunChecksumLoggerActorState() 
	{
		fdb_probe_actor_destroy("runChecksumLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 242 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
			;
															#line 620 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunChecksumLoggerActorState();
		static_cast<RunChecksumLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 243 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		StrictFuture<Void> __when_expr_0 = delay(delayDuration);
															#line 243 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		if (static_cast<RunChecksumLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 652 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunChecksumLoggerActor*>(this)->actor_wait_state = 1;
															#line 243 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunChecksumLoggerActor, 0, Void >*>(static_cast<RunChecksumLoggerActor*>(this)));
															#line 657 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 245 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		TraceEvent("AsyncFileWriteChecker") .detail("Delay", delayDuration) .detail("Filename", self->getFilename()) .detail("TotalCheckedSucceed", self->totalCheckedSucceed) .detail("TotalCheckedFail", self->totalCheckedFail) .detail("CurrentSize", self->lru.size());
															#line 666 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 245 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
		TraceEvent("AsyncFileWriteChecker") .detail("Delay", delayDuration) .detail("Filename", self->getFilename()) .detail("TotalCheckedSucceed", self->totalCheckedSucceed) .detail("TotalCheckedFail", self->totalCheckedFail) .detail("CurrentSize", self->lru.size());
															#line 675 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunChecksumLoggerActor*>(this)->actor_wait_state > 0) static_cast<RunChecksumLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<RunChecksumLoggerActor*>(this)->ActorCallback< RunChecksumLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunChecksumLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunChecksumLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunChecksumLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunChecksumLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunChecksumLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunChecksumLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runChecksumLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	AsyncFileWriteChecker* self;
															#line 241 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	double delayDuration;
															#line 759 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
};
// This generated class is to be used only via runChecksumLogger()
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
class RunChecksumLoggerActor final : public Actor<Void>, public ActorCallback< RunChecksumLoggerActor, 0, Void >, public FastAllocated<RunChecksumLoggerActor>, public RunChecksumLoggerActorState<RunChecksumLoggerActor> {
															#line 764 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
public:
	using FastAllocated<RunChecksumLoggerActor>::operator new;
	using FastAllocated<RunChecksumLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3048013111283189760UL, 9030036534208282112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunChecksumLoggerActor, 0, Void >;
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	RunChecksumLoggerActor(AsyncFileWriteChecker* const& self) 
															#line 781 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
		 : Actor<Void>(),
		   RunChecksumLoggerActorState<RunChecksumLoggerActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runChecksumLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1563772722638463488UL, 1855537222147633664UL);
		ActorExecutionContextHelper __helper(static_cast<RunChecksumLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runChecksumLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runChecksumLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunChecksumLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
[[nodiscard]] Future<Void> runChecksumLogger( AsyncFileWriteChecker* const& self ) {
															#line 240 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"
	return Future<Void>(new RunChecksumLoggerActor(self));
															#line 813 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/build_output/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.g.h"
}

#line 253 "/codebuild/output/src332181838/src/github.com/apple/foundationdb/fdbrpc/include/fdbrpc/AsyncFileWriteChecker.actor.h"

	// return true if there are still remaining valid synced pages to check, otherwise false
	// this method removes the page entry from checksum history upon a successful check
	bool verifyChecksum(uint32_t page, uint32_t checksum, uint8_t* start) {
		if (!lru.exist(page)) {
			// it has already been verified succesfully and removed by checksumWorker
			return true;
		}
		WriteInfo history = lru.find(page);
		// only verify checksum for pages have been synced
		if (history.timestamp < syncedTime) {
			if (history.checksum != checksum) {
				TraceEvent(SevError, "AsyncFileLostWriteDetected")
				    .error(checksum_failed())
				    .detail("Filename", getFilename())
				    .detail("PageNumber", page)
				    .detail("Size", lru.size())
				    .detail("Start", (long)start)
				    .detail("ChecksumOfPage", checksum)
				    .detail("ChecksumHistory", history.checksum)
				    .detail("SyncedTime", syncedTime / millisecondsPerSecond)
				    .detail("LastWriteTime", history.timestamp / millisecondsPerSecond);
				totalCheckedFail += 1;
			} else {
				checksumHistoryBudget.get() += 1;
				lru.remove(page);
				totalCheckedSucceed += 1;
			}
			return true;
		} else {
			return false;
		}
	}

	// Update or check checksum(s) in history for any full pages covered by this operation
	// return the updated pages when updateChecksum is true
	std::vector<uint32_t> updateChecksumHistory(bool updateChecksum, int64_t offset, int len, uint8_t* buf) {
		std::vector<uint32_t> pages;
		// Check or set each full block in the the range
		// page number starts at 1, as we use 0 to indicate invalid page
		uint32_t page = offset / checksumHistoryPageSize + 1; // First page number
		int slack = offset % checksumHistoryPageSize; // Bytes after most recent page boundary
		uint8_t* start = buf; // Position in buffer to start checking from
		// If offset is not page-aligned, move to next page and adjust start
		if (slack != 0) {
			++page;
			start += (checksumHistoryPageSize - slack);
		}
		uint32_t startPage = page;
		uint32_t pageEnd = (offset + len) / checksumHistoryPageSize; // Last page plus 1
		while (page < pageEnd) {
			uint32_t checksum = crc32c_append(0xab12fd93, start, checksumHistoryPageSize);
#if VALGRIND
			// It's possible we'll read or write a page where not all of the data is defined, but the checksum of the
			// page is still valid
			VALGRIND_MAKE_MEM_DEFINED_IF_ADDRESSABLE(&checksum, sizeof(uint32_t));
#endif
			// when updateChecksum is true, just update the stored sum and skip checking
			if (updateChecksum) {
				writing.insert(page);
				pages.push_back(page);
				WriteInfo history;
				if (!lru.exist(page)) {
					if (checksumHistoryBudget.get() > 0) {
						checksumHistoryBudget.get() -= 1;
					} else {
						TraceEvent("SkippedPagesDuringUpdateChecksum")
						    .detail("Filename", getFilename())
						    .detail("StartPage", startPage)
						    .detail("CheckedPage", page)
						    .detail("TotalPage", pageEnd);
						break;
					}
				}
				history.timestamp = AsyncFileWriteChecker::transformTime(now());
				history.checksum = checksum;
				lru.update(page, history);
			} else {
				if (!verifyChecksum(page, checksum, start)) {
					break;
				}
			}
			start += checksumHistoryPageSize;
			++page;
		}
		return pages;
	}
};

#include "flow/unactorcompiler.h"
#endif