#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
/*
 * Net2.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "boost/asio/buffer.hpp"
#include "boost/asio/ip/address.hpp"
#include "boost/system/system_error.hpp"
#include "flow/Arena.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/swift.h"
#include "flow/swift_concurrency_hooks.h"
#include <algorithm>
#include <memory>
#include <string_view>
#ifndef BOOST_SYSTEM_NO_LIB
#define BOOST_SYSTEM_NO_LIB
#endif
#ifndef BOOST_DATE_TIME_NO_LIB
#define BOOST_DATE_TIME_NO_LIB
#endif
#ifndef BOOST_REGEX_NO_LIB
#define BOOST_REGEX_NO_LIB
#endif
#include <boost/asio.hpp>
#include "boost/asio/ssl.hpp"
#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <boost/range.hpp>
#include <boost/algorithm/string/join.hpp>
#include "flow/network.h"
#include "flow/IThreadPool.h"

#include "flow/IAsyncFile.h"
#include "flow/ActorCollection.h"
#include "flow/TaskQueue.h"
#include "flow/ThreadHelper.actor.h"
#include "flow/ChaosMetrics.h"
#include "flow/TDMetric.actor.h"
#include "flow/AsioReactor.h"
#include "flow/Profiler.h"
#include "flow/ProtocolVersion.h"
#include "flow/SendBufferIterator.h"
#include "flow/TLSConfig.actor.h"
#include "flow/WatchFile.actor.h"
#include "flow/genericactors.actor.h"
#include "flow/Util.h"
#include "flow/UnitTest.h"
#include "flow/ScopeExit.h"
#include "flow/IUDPSocket.h"
#include "flow/IConnection.h"

#ifdef ADDRESS_SANITIZER
#include <sanitizer/lsan_interface.h>
#endif

#ifdef WIN32
#include <mmsystem.h>
#endif
#include "flow/actorcompiler.h" // This must be the last #include.

// Defined to track the stack limit
extern "C" intptr_t g_stackYieldLimit;
intptr_t g_stackYieldLimit = 0;

using namespace boost::asio::ip;

#if defined(__linux__) || defined(__FreeBSD__)
#include <execinfo.h>

std::atomic<int64_t> net2RunLoopIterations(0);
std::atomic<int64_t> net2RunLoopSleeps(0);

volatile size_t net2backtraces_max = 10000;
volatile void** volatile net2backtraces = nullptr;
volatile size_t net2backtraces_offset = 0;
volatile bool net2backtraces_overflow = false;
volatile int net2backtraces_count = 0;

volatile void** other_backtraces = nullptr;
sigset_t sigprof_set;

void initProfiling() {
	net2backtraces = new volatile void*[net2backtraces_max];
	other_backtraces = new volatile void*[net2backtraces_max];

	// According to folk wisdom, calling this once before setting up the signal handler makes
	// it async signal safe in practice :-/
	backtrace(const_cast<void**>(other_backtraces), net2backtraces_max);

	sigemptyset(&sigprof_set);
	sigaddset(&sigprof_set, SIGPROF);
}
#endif

template<> struct Descriptor<struct SlowTask> {
	static StringRef typeName() { return "SlowTask"_sr; }
	typedef SlowTask type;
	struct clocksDescriptor {
		static StringRef name() { return "clocks"_sr; }
		static StringRef typeName() { return "int64_t"_sr; }
		static StringRef comment() { return " clocks"_sr; }
		typedef int64_t type;
		static inline type get(SlowTask& from);
	};
	struct durationDescriptor {
		static StringRef name() { return "duration"_sr; }
		static StringRef typeName() { return "int64_t"_sr; }
		static StringRef comment() { return " ns"_sr; }
		typedef int64_t type;
		static inline type get(SlowTask& from);
	};
	struct priorityDescriptor {
		static StringRef name() { return "priority"_sr; }
		static StringRef typeName() { return "int64_t"_sr; }
		static StringRef comment() { return " priority level"_sr; }
		typedef int64_t type;
		static inline type get(SlowTask& from);
	};
	struct numYieldsDescriptor {
		static StringRef name() { return "numYields"_sr; }
		static StringRef typeName() { return "int64_t"_sr; }
		static StringRef comment() { return " count"_sr; }
		typedef int64_t type;
		static inline type get(SlowTask& from);
	};
	typedef std::tuple<clocksDescriptor,durationDescriptor,priorityDescriptor,numYieldsDescriptor> fields;
	typedef make_index_sequence_impl<0, index_sequence<>, std::tuple_size<fields>::value>::type field_indexes;
};
struct SlowTask {
	int64_t clocks; // clocks
	int64_t duration; // ns
	int64_t priority; // priority level
	int64_t numYields; // count
};
int64_t Descriptor<SlowTask>::clocksDescriptor::get(SlowTask& from) { return from.clocks; }
int64_t Descriptor<SlowTask>::durationDescriptor::get(SlowTask& from) { return from.duration; }
int64_t Descriptor<SlowTask>::priorityDescriptor::get(SlowTask& from) { return from.priority; }
int64_t Descriptor<SlowTask>::numYieldsDescriptor::get(SlowTask& from) { return from.numYields; }
#line 116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"


namespace N2 { // No indent, it's the whole file

class Peer;
class Connection;

// Outlives main
Net2* g_net2 = nullptr;

thread_local INetwork* thread_network = nullptr;

class Net2 final : public INetwork, public INetworkConnections {

private:
	void updateStarvationTracker(struct NetworkMetrics::PriorityStats& binStats,
	                             TaskPriority priority,
	                             TaskPriority lastPriority,
	                             double now);

public:
	Net2(const TLSConfig& tlsConfig, bool useThreadPool, bool useMetrics);
	void initTLS(ETLSInitState targetState) override;
	void run() override;
	void initMetrics() override;

	// INetworkConnections interface
	Future<Reference<IConnection>> connect(NetworkAddress toAddr, tcp::socket* existingSocket = nullptr) override;
	Future<Reference<IConnection>> connectExternal(NetworkAddress toAddr) override;
	Future<Reference<IUDPSocket>> createUDPSocket(NetworkAddress toAddr) override;
	Future<Reference<IUDPSocket>> createUDPSocket(bool isV6) override;
	// The mock DNS methods should only be used in simulation.
	void addMockTCPEndpoint(const std::string& host,
	                        const std::string& service,
	                        const std::vector<NetworkAddress>& addresses) override {
		throw operation_failed();
	}
	// The mock DNS methods should only be used in simulation.
	void removeMockTCPEndpoint(const std::string& host, const std::string& service) override {
		throw operation_failed();
	}
	void parseMockDNSFromString(const std::string& s) override { throw operation_failed(); }
	std::string convertMockDNSToString() override { throw operation_failed(); }

	Future<std::vector<NetworkAddress>> resolveTCPEndpoint(const std::string& host,
	                                                       const std::string& service) override;
	Future<std::vector<NetworkAddress>> resolveTCPEndpointWithDNSCache(const std::string& host,
	                                                                   const std::string& service) override;
	std::vector<NetworkAddress> resolveTCPEndpointBlocking(const std::string& host,
	                                                       const std::string& service) override;
	std::vector<NetworkAddress> resolveTCPEndpointBlockingWithDNSCache(const std::string& host,
	                                                                   const std::string& service) override;
	Reference<IListener> listen(NetworkAddress localAddr) override;

	// INetwork interface
	double now() const override { return currentTime; };
	double timer() override { return ::timer(); };
	double timer_monotonic() override { return ::timer_monotonic(); };
	Future<Void> delay(double seconds, TaskPriority taskId) override;
	Future<Void> orderedDelay(double seconds, TaskPriority taskId) override;
	void _swiftEnqueue(void* task) override;
	Future<class Void> yield(TaskPriority taskID) override;
	bool check_yield(TaskPriority taskId) override;
	TaskPriority getCurrentTask() const override { return currentTaskID; }
	void setCurrentTask(TaskPriority taskID) override {
		currentTaskID = taskID;
		priorityMetric = (int64_t)taskID;
	}
	void onMainThread(Promise<Void>&& signal, TaskPriority taskID) override;
	bool isOnMainThread() const override { return thread_network == this; }
	void stop() override {
		if (thread_network == this)
			stopImmediately();
		else
			onMainThreadVoid([this] { this->stopImmediately(); });
	}
	void addStopCallback(std::function<void()> fn) override {
		if (thread_network == this)
			stopCallbacks.emplace_back(std::move(fn));
		else
			onMainThreadVoid([this, fn] { this->stopCallbacks.emplace_back(std::move(fn)); });
	}

	bool isSimulated() const override { return false; }
	THREAD_HANDLE startThread(THREAD_FUNC_RETURN (*func)(void*), void* arg, int stackSize, const char* name) override;

	void getDiskBytes(std::string const& directory, int64_t& free, int64_t& total) override;
	bool isAddressOnThisHost(NetworkAddress const& addr) const override;
	void updateNow() { currentTime = timer_monotonic(); }

	flowGlobalType global(int id) const override { return (globals.size() > id) ? globals[id] : nullptr; }
	void setGlobal(size_t id, flowGlobalType v) override {
		ASSERT(id < globals.size());
		globals[id] = v;
	}

	ProtocolVersion protocolVersion() const override { return currentProtocolVersion(); }

	std::vector<flowGlobalType> globals;

	const TLSConfig& getTLSConfig() const override { return tlsConfig; }

	bool checkRunnable() override;

#ifdef ENABLE_SAMPLING
	ActorLineageSet& getActorLineageSet() override;
#endif

	bool useThreadPool;

	// private:

	ASIOReactor reactor;
	AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>> sslContextVar;
	Reference<IThreadPool> sslHandshakerPool;
	int sslHandshakerThreadsStarted;
	int sslPoolHandshakesInProgress;
	TLSConfig tlsConfig;
	Reference<TLSPolicy> activeTlsPolicy;
	Future<Void> backgroundCertRefresh;
	ETLSInitState tlsInitializedState;

	INetworkConnections* network; // initially this, but can be changed

	int64_t tscBegin, tscEnd;
	double taskBegin;
	TaskPriority currentTaskID;
	TDMetricCollection tdmetrics;
	MetricCollection metrics;
	ChaosMetrics chaosMetrics;
	// we read now() from a different thread. On Intel, reading a double is atomic anyways, but on other platforms it's
	// not. For portability this should be atomic
	std::atomic<double> currentTime;
	// May be accessed off the network thread, e.g. by onMainThread
	std::atomic<bool> stopped;
	mutable std::map<IPAddress, bool> addressOnHostCache;

#ifdef ENABLE_SAMPLING
	ActorLineageSet actorLineageSet;
#endif

	std::atomic<bool> started;

	uint64_t numYields;

	NetworkMetrics::PriorityStats* lastPriorityStats;

	struct PromiseTask final : public FastAllocated<PromiseTask> {
		Promise<Void> promise;
		swift::Job* _Nullable swiftJob = nullptr;
		PromiseTask() {}
		explicit PromiseTask(Promise<Void>&& promise) noexcept : promise(std::move(promise)) {}
		explicit PromiseTask(swift::Job* swiftJob) : swiftJob(swiftJob) {}

		void operator()() {
#ifdef WITH_SWIFT
			if (auto job = swiftJob) {
				swift_job_run(job, ExecutorRef::generic());
			} else {
				promise.send(Void());
			}
#else
			promise.send(Void());
#endif
			delete this;
		}
	};

	TaskQueue<PromiseTask> taskQueue;

	void checkForSlowTask(int64_t tscBegin, int64_t tscEnd, double duration, TaskPriority priority);
	bool check_yield(TaskPriority taskId, int64_t tscNow);
	void trackAtPriority(TaskPriority priority, double now);
	void stopImmediately() {
#ifdef ADDRESS_SANITIZER
		// Do leak check before intentionally leaking a bunch of memory
		__lsan_do_leak_check();
#endif
		stopped = true;
		taskQueue.clear();
	}

	Future<Void> timeOffsetLogger;
	Future<Void> logTimeOffset();

	Int64MetricHandle bytesReceived;
	Int64MetricHandle udpBytesReceived;
	Int64MetricHandle countWriteProbes;
	Int64MetricHandle countReadProbes;
	Int64MetricHandle countReads;
	Int64MetricHandle countUDPReads;
	Int64MetricHandle countWouldBlock;
	Int64MetricHandle countWrites;
	Int64MetricHandle countUDPWrites;
	Int64MetricHandle countRunLoop;
	Int64MetricHandle countTasks;
	Int64MetricHandle countYields;
	Int64MetricHandle countYieldBigStack;
	Int64MetricHandle countYieldCalls;
	Int64MetricHandle countYieldCallsTrue;
	Int64MetricHandle countASIOEvents;
	Int64MetricHandle countRunLoopProfilingSignals;
	Int64MetricHandle countTLSPolicyFailures;
	Int64MetricHandle priorityMetric;
	DoubleMetricHandle countLaunchTime;
	DoubleMetricHandle countReactTime;
	BoolMetricHandle awakeMetric;

	EventMetricHandle<SlowTask> slowTaskMetric;

	std::vector<std::string> blobCredentialFiles;
	Optional<std::string> proxy;
	std::vector<std::function<void()>> stopCallbacks;
};

static boost::asio::ip::address tcpAddress(IPAddress const& n) {
	if (n.isV6()) {
		return boost::asio::ip::address_v6(n.toV6());
	} else {
		return boost::asio::ip::address_v4(n.toV4());
	}
}

static IPAddress toIPAddress(boost::asio::ip::address const& addr) {
	if (addr.is_v4()) {
		return IPAddress(addr.to_v4().to_uint());
	} else {
		return IPAddress(addr.to_v6().to_bytes());
	}
}

static tcp::endpoint tcpEndpoint(NetworkAddress const& n) {
	return tcp::endpoint(tcpAddress(n.ip), n.port);
}

static udp::endpoint udpEndpoint(NetworkAddress const& n) {
	return udp::endpoint(tcpAddress(n.ip), n.port);
}

class BindPromise {
	Promise<Void> p;
	std::variant<const char*, AuditedEvent> errContext;
	UID errID;
	NetworkAddress peerAddr;

public:
	BindPromise(const char* errContext, UID errID) : errContext(errContext), errID(errID) {}
	BindPromise(AuditedEvent auditedEvent, UID errID) : errContext(auditedEvent), errID(errID) {}
	BindPromise(BindPromise const& r) : p(r.p), errContext(r.errContext), errID(r.errID), peerAddr(r.peerAddr) {}
	BindPromise(BindPromise&& r) noexcept
	  : p(std::move(r.p)), errContext(r.errContext), errID(r.errID), peerAddr(r.peerAddr) {}

	Future<Void> getFuture() const { return p.getFuture(); }

	NetworkAddress getPeerAddr() const { return peerAddr; }

	void setPeerAddr(const NetworkAddress& addr) { peerAddr = addr; }

	void operator()(const boost::system::error_code& error, size_t bytesWritten = 0) {
		try {
			if (error) {
				// Log the error...
				{
					std::optional<TraceEvent> traceEvent;
					if (std::holds_alternative<AuditedEvent>(errContext))
						traceEvent.emplace(SevWarn, std::get<AuditedEvent>(errContext), errID);
					else
						traceEvent.emplace(SevWarn, std::get<const char*>(errContext), errID);
					TraceEvent& evt = *traceEvent;
					evt.suppressFor(1.0).detail("ErrorCode", error.value()).detail("Message", error.message());
					// There is no function in OpenSSL to use to check if an error code is from OpenSSL,
					// but all OpenSSL errors have a non-zero "library" code set in bits 24-32, and linux
					// error codes should never go that high.
					if (error.value() >= (1 << 24L)) {
						evt.detail("WhichMeans", TLSPolicy::ErrorString(error));
					}

					if (peerAddr.isValid()) {
						evt.detail("PeerAddr", peerAddr);
						evt.detail("PeerAddress", peerAddr);
					}
				}

				p.sendError(connection_failed());
			} else
				p.send(Void());
		} catch (Error& e) {
			p.sendError(e);
		} catch (...) {
			p.sendError(unknown_error());
		}
	}
};

class Connection final : public IConnection, ReferenceCounted<Connection> {
public:
	void addref() override { ReferenceCounted<Connection>::addref(); }
	void delref() override { ReferenceCounted<Connection>::delref(); }

	void close() override { closeSocket(); }

	explicit Connection(boost::asio::io_service& io_service)
	  : id(nondeterministicRandom()->randomUniqueID()), socket(io_service) {}

	// This is not part of the IConnection interface, because it is wrapped by INetwork::connect()
																#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via connect()
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ConnectActor>
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActorState {
															#line 469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActorState(boost::asio::io_service* const& ios,NetworkAddress const& addr) 
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : ios(ios),
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   addr(addr),
															#line 422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   self(new Connection(*ios))
															#line 480 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("connect", reinterpret_cast<unsigned long>(this));

	}
	~ConnectActorState() 
	{
		fdb_probe_actor_destroy("connect", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->peer_address = addr;
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			try {
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto to = tcpEndpoint(addr);
															#line 427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				BindPromise p("N2_ConnectError", self->id);
															#line 428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				Future<Void> onConnected = p.getFuture();
															#line 429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				self->socket.async_connect(to, std::move(p));
															#line 431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = onConnected;
															#line 431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (static_cast<ConnectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ConnectActor*>(this)->actor_wait_state = 1;
															#line 431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectActor, 0, Void >*>(static_cast<ConnectActor*>(this)));
															#line 514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectActorState();
		static_cast<ConnectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->init();
															#line 433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(self); this->~ConnectActorState(); static_cast<ConnectActor*>(this)->destroy(); return 0; }
															#line 562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(self)); // state_var_RVO
		this->~ConnectActorState();
		static_cast<ConnectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->init();
															#line 433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(self); this->~ConnectActorState(); static_cast<ConnectActor*>(this)->destroy(); return 0; }
															#line 576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(self)); // state_var_RVO
		this->~ConnectActorState();
		static_cast<ConnectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectActor*>(this)->actor_wait_state > 0) static_cast<ConnectActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectActor*>(this)->ActorCallback< ConnectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	boost::asio::io_service* ios;
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	NetworkAddress addr;
															#line 422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<Connection> self;
															#line 665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via connect()
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActor final : public Actor<Reference<IConnection>>, public ActorCallback< ConnectActor, 0, Void >, public FastAllocated<ConnectActor>, public ConnectActorState<ConnectActor> {
															#line 670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ConnectActor>::operator new;
	using FastAllocated<ConnectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7188116270322820608UL, 10227479270083352576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IConnection>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectActor, 0, Void >;
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActor(boost::asio::io_service* const& ios,NetworkAddress const& addr) 
															#line 687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Reference<IConnection>>(),
		   ConnectActorState<ConnectActor>(ios, addr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6918112119896711168UL, 6671612149293629184UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Reference<IConnection>> connect( boost::asio::io_service* const& ios, NetworkAddress const& addr ) {
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Reference<IConnection>>(new ConnectActor(ios, addr));
															#line 719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

	// This is not part of the IConnection interface, because it is wrapped by IListener::accept()
	void accept(NetworkAddress peerAddr) {
		this->peer_address = peerAddr;
		init();
	}

	Future<Void> acceptHandshake() override { return Void(); }

	Future<Void> connectHandshake() override { return Void(); }

	// returns when write() can write at least one byte
	Future<Void> onWritable() override {
		++g_net2->countWriteProbes;
		BindPromise p("N2_WriteProbeError", id);
		auto f = p.getFuture();
		socket.async_write_some(boost::asio::null_buffers(), std::move(p));
		return f;
	}

	// returns when read() can read at least one byte
	Future<Void> onReadable() override {
		++g_net2->countReadProbes;
		BindPromise p("N2_ReadProbeError", id);
		auto f = p.getFuture();
		socket.async_read_some(boost::asio::null_buffers(), std::move(p));
		return f;
	}

	// Reads as many bytes as possible from the read buffer into [begin,end) and returns the number of bytes read (might
	// be 0)
	int read(uint8_t* begin, uint8_t* end) override {
		boost::system::error_code err;
		++g_net2->countReads;
		size_t toRead = end - begin;
		size_t size = socket.read_some(boost::asio::mutable_buffers_1(begin, toRead), err);
		g_net2->bytesReceived += size;
		//TraceEvent("ConnRead", this->id).detail("Bytes", size);
		if (err) {
			if (err == boost::asio::error::would_block) {
				++g_net2->countWouldBlock;
				return 0;
			}
			onReadError(err);
			throw connection_failed();
		}
		ASSERT(size); // If the socket is closed, we expect an 'eof' error, not a zero return value

		return size;
	}

	// Writes as many bytes as possible from the given SendBuffer chain into the write buffer and returns the number of
	// bytes written (might be 0)
	int write(SendBuffer const* data, int limit) override {
		boost::system::error_code err;
		++g_net2->countWrites;

		size_t sent = socket.write_some(
		    boost::iterator_range<SendBufferIterator>(SendBufferIterator(data, limit), SendBufferIterator()), err);

		if (err) {
			// Since there was an error, sent's value can't be used to infer that the buffer has data and the limit is
			// positive so check explicitly.
			ASSERT(limit > 0);
			bool notEmpty = false;
			for (auto p = data; p; p = p->next)
				if (p->bytes_written - p->bytes_sent > 0) {
					notEmpty = true;
					break;
				}
			ASSERT(notEmpty);

			if (err == boost::asio::error::would_block) {
				++g_net2->countWouldBlock;
				return 0;
			}
			onWriteError(err);
			throw connection_failed();
		}

		ASSERT(sent); // Make sure data was sent, and also this check will fail if the buffer chain was empty or the
		              // limit was not > 0.
		return sent;
	}

	NetworkAddress getPeerAddress() const override { return peer_address; }

	bool hasTrustedPeer() const override { return true; }

	UID getDebugID() const override { return id; }

	tcp::socket& getSocket() override { return socket; }

private:
	UID id;
	tcp::socket socket;
	NetworkAddress peer_address;

	void init() {
		// Socket settings that have to be set after connect or accept succeeds
		socket.non_blocking(true);
		if (FLOW_KNOBS->FLOW_TCP_NODELAY & 1) {
			socket.set_option(boost::asio::ip::tcp::no_delay(true));
		}
		if (FLOW_KNOBS->FLOW_TCP_QUICKACK & 1) {
#ifdef __linux__
			socket.set_option(boost::asio::detail::socket_option::boolean<IPPROTO_TCP, TCP_QUICKACK>(true));
#else
			TraceEvent(SevWarn, "N2_InitWarn").detail("Message", "TCP_QUICKACK not supported");
#endif
		}
		platform::setCloseOnExec(socket.native_handle());
	}

	void closeSocket() {
		boost::system::error_code error;
		socket.close(error);
		if (error)
			TraceEvent(SevWarn, "N2_CloseError", id)
			    .suppressFor(1.0)
			    .detail("PeerAddr", peer_address)
			    .detail("PeerAddress", peer_address)
			    .detail("ErrorCode", error.value())
			    .detail("Message", error.message());
	}

	void onReadError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_ReadError", id)
		    .suppressFor(1.0)
		    .detail("PeerAddr", peer_address)
		    .detail("PeerAddress", peer_address)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}
	void onWriteError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_WriteError", id)
		    .suppressFor(1.0)
		    .detail("PeerAddr", peer_address)
		    .detail("PeerAddress", peer_address)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}
};

class ReadPromise {
	Promise<int> p;
	const char* errContext;
	UID errID;
	std::shared_ptr<udp::endpoint> endpoint = nullptr;

public:
	ReadPromise(const char* errContext, UID errID) : errContext(errContext), errID(errID) {}
	ReadPromise(ReadPromise const& other) = default;
	ReadPromise(ReadPromise&& other) : p(std::move(other.p)), errContext(other.errContext), errID(other.errID) {}

	std::shared_ptr<udp::endpoint>& getEndpoint() { return endpoint; }

	Future<int> getFuture() { return p.getFuture(); }
	void operator()(const boost::system::error_code& error, size_t bytesWritten) {
		try {
			if (error) {
				TraceEvent evt(SevWarn, errContext, errID);
				evt.suppressFor(1.0).detail("ErrorCode", error.value()).detail("Message", error.message());
				p.sendError(connection_failed());
			} else {
				p.send(int(bytesWritten));
			}
		} catch (Error& e) {
			p.sendError(e);
		} catch (...) {
			p.sendError(unknown_error());
		}
	}
};

class UDPSocket : public IUDPSocket, ReferenceCounted<UDPSocket> {
	UID id;
	Optional<NetworkAddress> toAddress;
	udp::socket socket;
	bool isPublic = false;

public:
																#line 907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via connect()
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ConnectActor1>
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActor1State {
															#line 913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActor1State(boost::asio::io_service* const& io_service,Optional<NetworkAddress> const& toAddress,bool const& isV6) 
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : io_service(io_service),
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   toAddress(toAddress),
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   isV6(isV6),
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   self(new UDPSocket(*io_service, toAddress, isV6))
															#line 926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("connect", reinterpret_cast<unsigned long>(this));

	}
	~ConnectActor1State() 
	{
		fdb_probe_actor_destroy("connect", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(!toAddress.present() || toAddress.get().ip.isV6() == isV6);
															#line 629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (!toAddress.present())
															#line 943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (!static_cast<ConnectActor1*>(this)->SAV<Reference<IUDPSocket>>::futures) { (void)(self); this->~ConnectActor1State(); static_cast<ConnectActor1*>(this)->destroy(); return 0; }
															#line 947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				new (&static_cast<ConnectActor1*>(this)->SAV< Reference<IUDPSocket> >::value()) Reference<IUDPSocket>(std::move(self)); // state_var_RVO
				this->~ConnectActor1State();
				static_cast<ConnectActor1*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (toAddress.present())
															#line 956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 634 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					auto to = udpEndpoint(toAddress.get());
															#line 635 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					BindPromise p("N2_UDPConnectError", self->id);
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					Future<Void> onConnected = p.getFuture();
															#line 637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					self->socket.async_connect(to, std::move(p));
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					StrictFuture<Void> __when_expr_0 = onConnected;
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					if (static_cast<ConnectActor1*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<ConnectActor1*>(this)->actor_wait_state = 1;
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectActor1, 0, Void >*>(static_cast<ConnectActor1*>(this)));
															#line 975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont3(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectActor1State();
		static_cast<ConnectActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 1012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 641 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->init();
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectActor1*>(this)->SAV<Reference<IUDPSocket>>::futures) { (void)(self); this->~ConnectActor1State(); static_cast<ConnectActor1*>(this)->destroy(); return 0; }
															#line 1028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectActor1*>(this)->SAV< Reference<IUDPSocket> >::value()) Reference<IUDPSocket>(std::move(self)); // state_var_RVO
		this->~ConnectActor1State();
		static_cast<ConnectActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectActor1*>(this)->actor_wait_state > 0) static_cast<ConnectActor1*>(this)->actor_wait_state = 0;
		static_cast<ConnectActor1*>(this)->ActorCallback< ConnectActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	boost::asio::io_service* io_service;
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Optional<NetworkAddress> toAddress;
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	bool isV6;
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<UDPSocket> self;
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via connect()
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActor1 final : public Actor<Reference<IUDPSocket>>, public ActorCallback< ConnectActor1, 0, Void >, public FastAllocated<ConnectActor1>, public ConnectActor1State<ConnectActor1> {
															#line 1136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ConnectActor1>::operator new;
	using FastAllocated<ConnectActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7188116270322820608UL, 10227479270083352576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IUDPSocket>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectActor1, 0, Void >;
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActor1(boost::asio::io_service* const& io_service,Optional<NetworkAddress> const& toAddress,bool const& isV6) 
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Reference<IUDPSocket>>(),
		   ConnectActor1State<ConnectActor1>(io_service, toAddress, isV6),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5041636254048564480UL, 17126502678794650112UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectActor1, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Reference<IUDPSocket>> connect( boost::asio::io_service* const& io_service, Optional<NetworkAddress> const& toAddress, bool const& isV6 ) {
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Reference<IUDPSocket>>(new ConnectActor1(io_service, toAddress, isV6));
															#line 1185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

	void close() override { closeSocket(); }

	Future<int> receive(uint8_t* begin, uint8_t* end) override {
		++g_net2->countUDPReads;
		ReadPromise p("N2_UDPReadError", id);
		auto res = p.getFuture();
		socket.async_receive(boost::asio::mutable_buffer(begin, end - begin), std::move(p));
		return fmap(
		    [](int bytes) {
			    g_net2->udpBytesReceived += bytes;
			    return bytes;
		    },
		    res);
	}

	Future<int> receiveFrom(uint8_t* begin, uint8_t* end, NetworkAddress* sender) override {
		++g_net2->countUDPReads;
		ReadPromise p("N2_UDPReadFromError", id);
		p.getEndpoint() = std::make_shared<udp::endpoint>();
		auto endpoint = p.getEndpoint().get();
		auto res = p.getFuture();
		socket.async_receive_from(boost::asio::mutable_buffer(begin, end - begin), *endpoint, std::move(p));
		return fmap(
		    [endpoint, sender](int bytes) {
			    if (sender) {
				    sender->port = endpoint->port();
				    sender->ip = toIPAddress(endpoint->address());
			    }
			    g_net2->udpBytesReceived += bytes;
			    return bytes;
		    },
		    res);
	}

	Future<int> send(uint8_t const* begin, uint8_t const* end) override {
		++g_net2->countUDPWrites;
		ReadPromise p("N2_UDPWriteError", id);
		auto res = p.getFuture();
		socket.async_send(boost::asio::const_buffer(begin, end - begin), std::move(p));
		return res;
	}

	Future<int> sendTo(uint8_t const* begin, uint8_t const* end, NetworkAddress const& peer) override {
		++g_net2->countUDPWrites;
		ReadPromise p("N2_UDPWriteError", id);
		auto res = p.getFuture();
		udp::endpoint toEndpoint = udpEndpoint(peer);
		socket.async_send_to(boost::asio::const_buffer(begin, end - begin), toEndpoint, std::move(p));
		return res;
	}

	void bind(NetworkAddress const& addr) override {
		boost::system::error_code ec;
		socket.bind(udpEndpoint(addr), ec);
		if (ec) {
			Error x;
			if (ec.value() == EADDRINUSE)
				x = address_in_use();
			else if (ec.value() == EADDRNOTAVAIL)
				x = invalid_local_address();
			else
				x = bind_failed();
			TraceEvent(SevWarnAlways, "Net2UDPBindError").error(x);
			throw x;
		}
		isPublic = true;
	}

	UID getDebugID() const override { return id; }

	void addref() override { ReferenceCounted<UDPSocket>::addref(); }
	void delref() override { ReferenceCounted<UDPSocket>::delref(); }

	NetworkAddress localAddress() const override {
		auto endpoint = socket.local_endpoint();
		return NetworkAddress(toIPAddress(endpoint.address()), endpoint.port(), isPublic, false);
	}

	boost::asio::ip::udp::socket::native_handle_type native_handle() override { return socket.native_handle(); }

private:
	UDPSocket(boost::asio::io_service& io_service, Optional<NetworkAddress> toAddress, bool isV6)
	  : id(nondeterministicRandom()->randomUniqueID()), socket(io_service, isV6 ? udp::v6() : udp::v4()) {}

	void closeSocket() {
		boost::system::error_code error;
		socket.close(error);
		if (error)
			TraceEvent(SevWarn, "N2_CloseError", id)
			    .suppressFor(1.0)
			    .detail("ErrorCode", error.value())
			    .detail("Message", error.message());
	}

	void onReadError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_UDPReadError", id)
		    .suppressFor(1.0)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}
	void onWriteError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_UDPWriteError", id)
		    .suppressFor(1.0)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}

	void init() {
		socket.non_blocking(true);
		platform::setCloseOnExec(socket.native_handle());
	}
};

class Listener final : public IListener, ReferenceCounted<Listener> {
	boost::asio::io_context& io_service;
	NetworkAddress listenAddress;
	tcp::acceptor acceptor;

public:
	Listener(boost::asio::io_context& io_service, NetworkAddress listenAddress)
	  : io_service(io_service), listenAddress(listenAddress), acceptor(io_service, tcpEndpoint(listenAddress)) {
		// when port 0 is passed in, a random port will be opened
		// set listenAddress as the address with the actual port opened instead of port 0
		if (listenAddress.port == 0) {
			this->listenAddress =
			    NetworkAddress::parse(acceptor.local_endpoint().address().to_string().append(":").append(
			        std::to_string(acceptor.local_endpoint().port())));
		}
		platform::setCloseOnExec(acceptor.native_handle());
	}

	void addref() override { ReferenceCounted<Listener>::addref(); }
	void delref() override { ReferenceCounted<Listener>::delref(); }

	// Returns one incoming connection when it is available
	Future<Reference<IConnection>> accept() override { return doAccept(this); }

	NetworkAddress getListenAddress() const override { return listenAddress; }

private:
																#line 1332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via doAccept()
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class DoAcceptActor>
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptActorState {
															#line 1338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptActorState(Listener* const& self) 
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   conn(new Connection(self->io_service)),
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   peer_endpoint()
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("doAccept", reinterpret_cast<unsigned long>(this));

	}
	~DoAcceptActorState() 
	{
		fdb_probe_actor_destroy("doAccept", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				BindPromise p("N2_AcceptError", UID());
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto f = p.getFuture();
															#line 797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				self->acceptor.async_accept(conn->getSocket(), peer_endpoint, std::move(p));
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = f;
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (static_cast<DoAcceptActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoAcceptActor*>(this)->actor_wait_state = 1;
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoAcceptActor, 0, Void >*>(static_cast<DoAcceptActor*>(this)));
															#line 1378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoAcceptActorState();
		static_cast<DoAcceptActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			conn->close();
															#line 806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 1410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		auto peer_address = peer_endpoint.address().is_v6() ? IPAddress(peer_endpoint.address().to_v6().to_bytes()) : IPAddress(peer_endpoint.address().to_v4().to_ulong());
															#line 801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		conn->accept(NetworkAddress(peer_address, peer_endpoint.port()));
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<DoAcceptActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(conn); this->~DoAcceptActorState(); static_cast<DoAcceptActor*>(this)->destroy(); return 0; }
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<DoAcceptActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(conn)); // state_var_RVO
		this->~DoAcceptActorState();
		static_cast<DoAcceptActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		auto peer_address = peer_endpoint.address().is_v6() ? IPAddress(peer_endpoint.address().to_v6().to_bytes()) : IPAddress(peer_endpoint.address().to_v4().to_ulong());
															#line 801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		conn->accept(NetworkAddress(peer_address, peer_endpoint.port()));
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<DoAcceptActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(conn); this->~DoAcceptActorState(); static_cast<DoAcceptActor*>(this)->destroy(); return 0; }
															#line 1444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<DoAcceptActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(conn)); // state_var_RVO
		this->~DoAcceptActorState();
		static_cast<DoAcceptActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoAcceptActor*>(this)->actor_wait_state > 0) static_cast<DoAcceptActor*>(this)->actor_wait_state = 0;
		static_cast<DoAcceptActor*>(this)->ActorCallback< DoAcceptActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoAcceptActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoAcceptActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoAcceptActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Listener* self;
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<Connection> conn;
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	tcp::acceptor::endpoint_type peer_endpoint;
															#line 1533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via doAccept()
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptActor final : public Actor<Reference<IConnection>>, public ActorCallback< DoAcceptActor, 0, Void >, public FastAllocated<DoAcceptActor>, public DoAcceptActorState<DoAcceptActor> {
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<DoAcceptActor>::operator new;
	using FastAllocated<DoAcceptActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9356202762953707264UL, 14099876164327578368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IConnection>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoAcceptActor, 0, Void >;
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptActor(Listener* const& self) 
															#line 1555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Reference<IConnection>>(),
		   DoAcceptActorState<DoAcceptActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16397608747409512704UL, 446929050777440256UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doAccept");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoAcceptActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Reference<IConnection>> doAccept( Listener* const& self ) {
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Reference<IConnection>>(new DoAcceptActor(self));
															#line 1587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
};

typedef boost::asio::ssl::stream<boost::asio::ip::tcp::socket&> ssl_socket;

struct SSLHandshakerThread final : IThreadPoolReceiver {
	SSLHandshakerThread() {}
	void init() override {}

	struct Handshake final : TypedAction<SSLHandshakerThread, Handshake> {
		Handshake(ssl_socket& socket, ssl_socket::handshake_type type) : socket(socket), type(type) {}
		double getTimeEstimate() const override { return 0.001; }

		std::string getPeerAddress() const {
			std::ostringstream o;
			boost::system::error_code ec;
			auto addr = socket.lowest_layer().remote_endpoint(ec);
			o << (!ec.failed() ? addr.address().to_string() : std::string_view("0.0.0.0"));
			return std::move(o).str();
		}

		ThreadReturnPromise<Void> done;
		ssl_socket& socket;
		ssl_socket::handshake_type type;
		boost::system::error_code err;
	};

	void action(Handshake& h) {
		try {
			h.socket.next_layer().non_blocking(false, h.err);
			if (!h.err.failed()) {
				h.socket.handshake(h.type, h.err);
			}
			if (!h.err.failed()) {
				h.socket.next_layer().non_blocking(true, h.err);
			}
			if (h.err.failed()) {
				TraceEvent(SevWarn,
				           h.type == ssl_socket::handshake_type::client ? "N2_ConnectHandshakeError"_audit
				                                                        : "N2_AcceptHandshakeError"_audit)
				    .detail("PeerAddr", h.getPeerAddress())
				    .detail("PeerAddress", h.getPeerAddress())
				    .detail("ErrorCode", h.err.value())
				    .detail("ErrorMsg", h.err.message().c_str())
				    .detail("BackgroundThread", true);
				h.done.sendError(connection_failed());
			} else {
				h.done.send(Void());
			}
		} catch (...) {
			TraceEvent(SevWarn,
			           h.type == ssl_socket::handshake_type::client ? "N2_ConnectHandshakeUnknownError"_audit
			                                                        : "N2_AcceptHandshakeUnknownError"_audit)
			    .detail("PeerAddr", h.getPeerAddress())
			    .detail("PeerAddress", h.getPeerAddress())
			    .detail("BackgroundThread", true);
			h.done.sendError(connection_failed());
		}
	}
};

class SSLConnection final : public IConnection, ReferenceCounted<SSLConnection> {
public:
	void addref() override { ReferenceCounted<SSLConnection>::addref(); }
	void delref() override { ReferenceCounted<SSLConnection>::delref(); }

	void close() override { closeSocket(); }

	explicit SSLConnection(boost::asio::io_service& io_service,
	                       Reference<ReferencedObject<boost::asio::ssl::context>> context)
	  : id(nondeterministicRandom()->randomUniqueID()), socket(io_service), ssl_sock(socket, context->mutate()),
	    sslContext(context), has_trusted_peer(false) {}

	explicit SSLConnection(Reference<ReferencedObject<boost::asio::ssl::context>> context, tcp::socket* existingSocket)
	  : id(nondeterministicRandom()->randomUniqueID()), socket(std::move(*existingSocket)),
	    ssl_sock(socket, context->mutate()), sslContext(context) {}

	// This is not part of the IConnection interface, because it is wrapped by INetwork::connect()
																#line 1668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via connect()
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ConnectActor2>
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActor2State {
															#line 1674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActor2State(boost::asio::io_service* const& ios,Reference<ReferencedObject<boost::asio::ssl::context>> const& context,NetworkAddress const& addr,tcp::socket* const& existingSocket = nullptr) 
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : ios(ios),
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   context(context),
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   addr(addr),
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   existingSocket(existingSocket)
															#line 1687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("connect", reinterpret_cast<unsigned long>(this));

	}
	~ConnectActor2State() 
	{
		fdb_probe_actor_destroy("connect", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			std::pair<IPAddress, uint16_t> peerIP = std::make_pair(addr.ip, addr.port);
															#line 891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			auto iter(g_network->networkInfo.serverTLSConnectionThrottler.find(peerIP));
															#line 892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (iter != g_network->networkInfo.serverTLSConnectionThrottler.end())
															#line 1706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (now() < iter->second.second)
															#line 1710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					if (iter->second.first >= FLOW_KNOBS->TLS_CLIENT_CONNECTION_THROTTLE_ATTEMPTS)
															#line 1714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					{
															#line 895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						TraceEvent("TLSOutgoingConnectionThrottlingWarning").suppressFor(1.0).detail("PeerIP", addr);
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						if (static_cast<ConnectActor2*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
						if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
						static_cast<ConnectActor2*>(this)->actor_wait_state = 1;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectActor2, 0, Void >*>(static_cast<ConnectActor2*>(this)));
															#line 1727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont3(loopDepth);
					}
				}
				else
				{
															#line 900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					g_network->networkInfo.serverTLSConnectionThrottler.erase(peerIP);
															#line 1739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectActor2State();
		static_cast<ConnectActor2*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (existingSocket != nullptr)
															#line 1768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		{
															#line 905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			Reference<SSLConnection> self(new SSLConnection(context, existingSocket));
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->peer_address = addr;
															#line 907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->init();
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (!static_cast<ConnectActor2*>(this)->SAV<Reference<IConnection>>::futures) { (void)(self); this->~ConnectActor2State(); static_cast<ConnectActor2*>(this)->destroy(); return 0; }
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			new (&static_cast<ConnectActor2*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(self);
			this->~ConnectActor2State();
			static_cast<ConnectActor2*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self = Reference<SSLConnection>(new SSLConnection(*ios, context));
															#line 912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->peer_address = addr;
															#line 1788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			auto to = tcpEndpoint(self->peer_address);
															#line 915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			BindPromise p("N2_ConnectError", self->id);
															#line 916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			Future<Void> onConnected = p.getFuture();
															#line 917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->socket.async_connect(to, std::move(p));
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_1 = onConnected;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ConnectActor2*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ConnectActor2*>(this)->actor_wait_state = 2;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectActor2, 1, Void >*>(static_cast<ConnectActor2*>(this)));
															#line 1807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1Catch1(connection_failed(), loopDepth);
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1Catch1(connection_failed(), loopDepth);
															#line 1842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectActor2*>(this)->actor_wait_state > 0) static_cast<ConnectActor2*>(this)->actor_wait_state = 0;
		static_cast<ConnectActor2*>(this)->ActorCallback< ConnectActor2, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectActor2, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectActor2, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectActor2, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1Catch1(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 1928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->init();
															#line 921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectActor2*>(this)->SAV<Reference<IConnection>>::futures) { (void)(self); this->~ConnectActor2State(); static_cast<ConnectActor2*>(this)->destroy(); return 0; }
															#line 1944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectActor2*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(self)); // state_var_RVO
		this->~ConnectActor2State();
		static_cast<ConnectActor2*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->init();
															#line 921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectActor2*>(this)->SAV<Reference<IConnection>>::futures) { (void)(self); this->~ConnectActor2State(); static_cast<ConnectActor2*>(this)->destroy(); return 0; }
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectActor2*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(self)); // state_var_RVO
		this->~ConnectActor2State();
		static_cast<ConnectActor2*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectActor2*>(this)->actor_wait_state > 0) static_cast<ConnectActor2*>(this)->actor_wait_state = 0;
		static_cast<ConnectActor2*>(this)->ActorCallback< ConnectActor2, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectActor2, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectActor2, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectActor2, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	boost::asio::io_service* ios;
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<ReferencedObject<boost::asio::ssl::context>> context;
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	NetworkAddress addr;
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	tcp::socket* existingSocket;
															#line 911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> self;
															#line 2051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via connect()
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectActor2 final : public Actor<Reference<IConnection>>, public ActorCallback< ConnectActor2, 0, Void >, public ActorCallback< ConnectActor2, 1, Void >, public FastAllocated<ConnectActor2>, public ConnectActor2State<ConnectActor2> {
															#line 2056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ConnectActor2>::operator new;
	using FastAllocated<ConnectActor2>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7188116270322820608UL, 10227479270083352576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IConnection>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectActor2, 0, Void >;
friend struct ActorCallback< ConnectActor2, 1, Void >;
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectActor2(boost::asio::io_service* const& ios,Reference<ReferencedObject<boost::asio::ssl::context>> const& context,NetworkAddress const& addr,tcp::socket* const& existingSocket = nullptr) 
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Reference<IConnection>>(),
		   ConnectActor2State<ConnectActor2>(ios, context, addr, existingSocket),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15786448584903880704UL, 2626871032733114624UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectActor2*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectActor2, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectActor2, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Reference<IConnection>> connect( boost::asio::io_service* const& ios, Reference<ReferencedObject<boost::asio::ssl::context>> const& context, NetworkAddress const& addr, tcp::socket* const& existingSocket = nullptr ) {
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Reference<IConnection>>(new ConnectActor2(ios, context, addr, existingSocket));
															#line 2107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

	// This is not part of the IConnection interface, because it is wrapped by IListener::accept()
	void accept(NetworkAddress peerAddr) {
		this->peer_address = peerAddr;
		init();
	}

																#line 2118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via doAcceptHandshake()
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class DoAcceptHandshakeActor>
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptHandshakeActorState {
															#line 2124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptHandshakeActorState(Reference<SSLConnection> const& self,Promise<Void> const& connected) 
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   connected(connected),
															#line 936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   holder()
															#line 2135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("doAcceptHandshake", reinterpret_cast<unsigned long>(this));

	}
	~DoAcceptHandshakeActorState() 
	{
		fdb_probe_actor_destroy("doAcceptHandshake", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				Future<Void> onHandshook;
															#line 940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				ConfigureSSLStream(N2::g_net2->activeTlsPolicy, self->ssl_sock, self->peer_address, [conn = self.getPtr()](bool verifyOk) { conn->has_trusted_peer = verifyOk; });
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (N2::g_net2->sslPoolHandshakesInProgress < N2::g_net2->sslHandshakerThreadsStarted)
															#line 2155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					holder = Hold(&N2::g_net2->sslPoolHandshakesInProgress);
															#line 948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					auto handshake = new SSLHandshakerThread::Handshake(self->ssl_sock, boost::asio::ssl::stream_base::server);
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					onHandshook = handshake->done.getFuture();
															#line 951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					N2::g_net2->sslHandshakerPool->post(handshake);
															#line 2165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
				else
				{
															#line 954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					BindPromise p("N2_AcceptHandshakeError"_audit, self->id);
															#line 955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					p.setPeerAddr(self->getPeerAddress());
															#line 956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					onHandshook = p.getFuture();
															#line 957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					self->ssl_sock.async_handshake(boost::asio::ssl::stream_base::server, std::move(p));
															#line 2177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
															#line 959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = onHandshook;
															#line 959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state = 1;
															#line 959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoAcceptHandshakeActor, 0, Void >*>(static_cast<DoAcceptHandshakeActor*>(this)));
															#line 2187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<DoAcceptHandshakeActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			connected.sendError(connection_failed());
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::Handshake);
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state = 2;
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoAcceptHandshakeActor, 1, Void >*>(static_cast<DoAcceptHandshakeActor*>(this)));
															#line 2245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::Handshake);
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 2256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state = 2;
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoAcceptHandshakeActor, 1, Void >*>(static_cast<DoAcceptHandshakeActor*>(this)));
															#line 2260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state > 0) static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state = 0;
		static_cast<DoAcceptHandshakeActor*>(this)->ActorCallback< DoAcceptHandshakeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoAcceptHandshakeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoAcceptHandshakeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoAcceptHandshakeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		connected.send(Void());
															#line 2344 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		connected.send(Void());
															#line 2353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state > 0) static_cast<DoAcceptHandshakeActor*>(this)->actor_wait_state = 0;
		static_cast<DoAcceptHandshakeActor*>(this)->ActorCallback< DoAcceptHandshakeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoAcceptHandshakeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoAcceptHandshakeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoAcceptHandshakeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		delete static_cast<DoAcceptHandshakeActor*>(this);
															#line 2450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		return 0;

		return loopDepth;
	}
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> self;
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Promise<Void> connected;
															#line 936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Hold<int> holder;
															#line 2461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via doAcceptHandshake()
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptHandshakeActor final : public Actor<void>, public ActorCallback< DoAcceptHandshakeActor, 0, Void >, public ActorCallback< DoAcceptHandshakeActor, 1, Void >, public FastAllocated<DoAcceptHandshakeActor>, public DoAcceptHandshakeActorState<DoAcceptHandshakeActor> {
															#line 2466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<DoAcceptHandshakeActor>::operator new;
	using FastAllocated<DoAcceptHandshakeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16667718383922977024UL, 13445425546397253632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< DoAcceptHandshakeActor, 0, Void >;
friend struct ActorCallback< DoAcceptHandshakeActor, 1, Void >;
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptHandshakeActor(Reference<SSLConnection> const& self,Promise<Void> const& connected) 
															#line 2484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<void>(),
		   DoAcceptHandshakeActorState<DoAcceptHandshakeActor>(self, connected),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doAcceptHandshake", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4967735093768019200UL, 17175562138694802432UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doAcceptHandshake");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doAcceptHandshake", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
static void doAcceptHandshake( Reference<SSLConnection> const& self, Promise<Void> const& connected ) {
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	new DoAcceptHandshakeActor(self, connected);
															#line 2507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

																#line 2512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via acceptHandshakeWrapper()
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class AcceptHandshakeWrapperActor>
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class AcceptHandshakeWrapperActorState {
															#line 2518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	AcceptHandshakeWrapperActorState(Reference<SSLConnection> const& self) 
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   peerIP()
															#line 2527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this));

	}
	~AcceptHandshakeWrapperActorState() 
	{
		fdb_probe_actor_destroy("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			peerIP = std::make_pair(self->getPeerAddress().ip, static_cast<uint16_t>(0));
															#line 971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			auto iter(g_network->networkInfo.serverTLSConnectionThrottler.find(peerIP));
															#line 972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (iter != g_network->networkInfo.serverTLSConnectionThrottler.end())
															#line 2546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (now() < iter->second.second)
															#line 2550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					if (iter->second.first >= FLOW_KNOBS->TLS_SERVER_CONNECTION_THROTTLE_ATTEMPTS)
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					{
															#line 975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						TraceEvent("TLSIncomingConnectionThrottlingWarning") .suppressFor(1.0) .detail("PeerIP", peerIP.first.toString());
															#line 978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						StrictFuture<Void> __when_expr_0 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
						if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
						static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 1;
															#line 978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
						__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 0, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 2567 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont3(loopDepth);
					}
				}
				else
				{
															#line 983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					g_network->networkInfo.serverTLSConnectionThrottler.erase(peerIP);
															#line 2579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AcceptHandshakeWrapperActorState();
		static_cast<AcceptHandshakeWrapperActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = g_network->networkInfo.handshakeLock->take();
															#line 987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 2;
															#line 987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 1, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 2615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->closeSocket();
															#line 980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1Catch1(connection_failed(), loopDepth);
															#line 2638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->closeSocket();
															#line 980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1Catch1(connection_failed(), loopDepth);
															#line 2648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state > 0) static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 0;
		static_cast<AcceptHandshakeWrapperActor*>(this)->ActorCallback< AcceptHandshakeWrapperActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AcceptHandshakeWrapperActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		releaser = FlowLock::Releaser(*g_network->networkInfo.handshakeLock);
															#line 990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> connected;
															#line 991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doAcceptHandshake(self, connected);
															#line 2735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_2 = connected.getFuture();
															#line 993 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1cont7Catch1(actor_cancelled(), loopDepth);
															#line 2741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont7Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont7when1(__when_expr_2.get(), loopDepth); };
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 2745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont7Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when2(__when_expr_3.get(), loopDepth); };
			static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 3;
															#line 994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 3, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 2752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont7Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		releaser = FlowLock::Releaser(*g_network->networkInfo.handshakeLock);
															#line 990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> connected;
															#line 991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doAcceptHandshake(self, connected);
															#line 2771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_2 = connected.getFuture();
															#line 993 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1cont7Catch1(actor_cancelled(), loopDepth);
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont7Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont7when1(__when_expr_2.get(), loopDepth); };
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 2781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont7Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when2(__when_expr_3.get(), loopDepth); };
			static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 3;
															#line 994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AcceptHandshakeWrapperActor, 3, Void >*>(static_cast<AcceptHandshakeWrapperActor*>(this)));
															#line 2788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont7Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state > 0) static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 0;
		static_cast<AcceptHandshakeWrapperActor*>(this)->ActorCallback< AcceptHandshakeWrapperActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AcceptHandshakeWrapperActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 2879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto iter(g_network->networkInfo.serverTLSConnectionThrottler.find(peerIP));
															#line 1004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (iter != g_network->networkInfo.serverTLSConnectionThrottler.end())
															#line 2885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 1005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					iter->second.first++;
															#line 2889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
				else
				{
															#line 1007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					g_network->networkInfo.serverTLSConnectionThrottler[peerIP] = std::make_pair(0, now() + FLOW_KNOBS->TLS_SERVER_CONNECTION_THROTTLE_TIMEOUT);
															#line 2895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
			}
															#line 1012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 1013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<AcceptHandshakeWrapperActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AcceptHandshakeWrapperActorState(); static_cast<AcceptHandshakeWrapperActor*>(this)->destroy(); return 0; }
															#line 2916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<AcceptHandshakeWrapperActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AcceptHandshakeWrapperActorState();
		static_cast<AcceptHandshakeWrapperActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<AcceptHandshakeWrapperActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AcceptHandshakeWrapperActorState(); static_cast<AcceptHandshakeWrapperActor*>(this)->destroy(); return 0; }
															#line 2928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<AcceptHandshakeWrapperActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AcceptHandshakeWrapperActorState();
		static_cast<AcceptHandshakeWrapperActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7when2(Void const& _,int loopDepth) 
	{
															#line 998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1cont7Catch1(connection_failed(), loopDepth);
															#line 2940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont7when2(Void && _,int loopDepth) 
	{
															#line 998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1cont7Catch1(connection_failed(), loopDepth);
															#line 2948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state > 0) static_cast<AcceptHandshakeWrapperActor*>(this)->actor_wait_state = 0;
		static_cast<AcceptHandshakeWrapperActor*>(this)->ActorCallback< AcceptHandshakeWrapperActor, 2, Void >::remove();
		static_cast<AcceptHandshakeWrapperActor*>(this)->ActorCallback< AcceptHandshakeWrapperActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< AcceptHandshakeWrapperActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< AcceptHandshakeWrapperActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> self;
															#line 969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::pair<IPAddress, uint16_t> peerIP;
															#line 988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowLock::Releaser releaser;
															#line 3079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via acceptHandshakeWrapper()
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class AcceptHandshakeWrapperActor final : public Actor<Void>, public ActorCallback< AcceptHandshakeWrapperActor, 0, Void >, public ActorCallback< AcceptHandshakeWrapperActor, 1, Void >, public ActorCallback< AcceptHandshakeWrapperActor, 2, Void >, public ActorCallback< AcceptHandshakeWrapperActor, 3, Void >, public FastAllocated<AcceptHandshakeWrapperActor>, public AcceptHandshakeWrapperActorState<AcceptHandshakeWrapperActor> {
															#line 3084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<AcceptHandshakeWrapperActor>::operator new;
	using FastAllocated<AcceptHandshakeWrapperActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15556563232789517056UL, 6595221896298125056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AcceptHandshakeWrapperActor, 0, Void >;
friend struct ActorCallback< AcceptHandshakeWrapperActor, 1, Void >;
friend struct ActorCallback< AcceptHandshakeWrapperActor, 2, Void >;
friend struct ActorCallback< AcceptHandshakeWrapperActor, 3, Void >;
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	AcceptHandshakeWrapperActor(Reference<SSLConnection> const& self) 
															#line 3104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   AcceptHandshakeWrapperActorState<AcceptHandshakeWrapperActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6378667139452327424UL, 2738152998914196736UL);
		ActorExecutionContextHelper __helper(static_cast<AcceptHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("acceptHandshakeWrapper");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("acceptHandshakeWrapper", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AcceptHandshakeWrapperActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AcceptHandshakeWrapperActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AcceptHandshakeWrapperActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Void> acceptHandshakeWrapper( Reference<SSLConnection> const& self ) {
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new AcceptHandshakeWrapperActor(self));
															#line 3138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

	Future<Void> acceptHandshake() override { return acceptHandshakeWrapper(Reference<SSLConnection>::addRef(this)); }

																#line 3145 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via doConnectHandshake()
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class DoConnectHandshakeActor>
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoConnectHandshakeActorState {
															#line 3151 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoConnectHandshakeActorState(Reference<SSLConnection> const& self,Promise<Void> const& connected) 
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   connected(connected),
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   holder()
															#line 3162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("doConnectHandshake", reinterpret_cast<unsigned long>(this));

	}
	~DoConnectHandshakeActorState() 
	{
		fdb_probe_actor_destroy("doConnectHandshake", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				Future<Void> onHandshook;
															#line 1024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				ConfigureSSLStream(N2::g_net2->activeTlsPolicy, self->ssl_sock, self->peer_address, [conn = self.getPtr()](bool verifyOk) { conn->has_trusted_peer = verifyOk; });
															#line 1030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (N2::g_net2->sslPoolHandshakesInProgress < N2::g_net2->sslHandshakerThreadsStarted)
															#line 3182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 1031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					holder = Hold(&N2::g_net2->sslPoolHandshakesInProgress);
															#line 1032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					auto handshake = new SSLHandshakerThread::Handshake(self->ssl_sock, boost::asio::ssl::stream_base::client);
															#line 1034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					onHandshook = handshake->done.getFuture();
															#line 1035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					N2::g_net2->sslHandshakerPool->post(handshake);
															#line 3192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
				else
				{
															#line 1038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					BindPromise p("N2_ConnectHandshakeError"_audit, self->id);
															#line 1039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					p.setPeerAddr(self->getPeerAddress());
															#line 1040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					onHandshook = p.getFuture();
															#line 1041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					self->ssl_sock.async_handshake(boost::asio::ssl::stream_base::client, std::move(p));
															#line 3204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
															#line 1043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = onHandshook;
															#line 1043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 3210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state = 1;
															#line 1043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoConnectHandshakeActor, 0, Void >*>(static_cast<DoConnectHandshakeActor*>(this)));
															#line 3214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<DoConnectHandshakeActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 1047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			connected.sendError(connection_failed());
															#line 3251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::Handshake);
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state = 2;
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoConnectHandshakeActor, 1, Void >*>(static_cast<DoConnectHandshakeActor*>(this)));
															#line 3272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::Handshake);
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state = 2;
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoConnectHandshakeActor, 1, Void >*>(static_cast<DoConnectHandshakeActor*>(this)));
															#line 3287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state > 0) static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state = 0;
		static_cast<DoConnectHandshakeActor*>(this)->ActorCallback< DoConnectHandshakeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoConnectHandshakeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoConnectHandshakeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoConnectHandshakeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		connected.send(Void());
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		connected.send(Void());
															#line 3380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state > 0) static_cast<DoConnectHandshakeActor*>(this)->actor_wait_state = 0;
		static_cast<DoConnectHandshakeActor*>(this)->ActorCallback< DoConnectHandshakeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoConnectHandshakeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoConnectHandshakeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoConnectHandshakeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		delete static_cast<DoConnectHandshakeActor*>(this);
															#line 3477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		return 0;

		return loopDepth;
	}
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> self;
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Promise<Void> connected;
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Hold<int> holder;
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via doConnectHandshake()
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoConnectHandshakeActor final : public Actor<void>, public ActorCallback< DoConnectHandshakeActor, 0, Void >, public ActorCallback< DoConnectHandshakeActor, 1, Void >, public FastAllocated<DoConnectHandshakeActor>, public DoConnectHandshakeActorState<DoConnectHandshakeActor> {
															#line 3493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<DoConnectHandshakeActor>::operator new;
	using FastAllocated<DoConnectHandshakeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5138030323399713024UL, 10944970532516105984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< DoConnectHandshakeActor, 0, Void >;
friend struct ActorCallback< DoConnectHandshakeActor, 1, Void >;
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoConnectHandshakeActor(Reference<SSLConnection> const& self,Promise<Void> const& connected) 
															#line 3511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<void>(),
		   DoConnectHandshakeActorState<DoConnectHandshakeActor>(self, connected),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doConnectHandshake", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5485981565538822144UL, 17863232542333579008UL);
		ActorExecutionContextHelper __helper(static_cast<DoConnectHandshakeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doConnectHandshake");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doConnectHandshake", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
static void doConnectHandshake( Reference<SSLConnection> const& self, Promise<Void> const& connected ) {
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	new DoConnectHandshakeActor(self, connected);
															#line 3534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

																#line 3539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via connectHandshakeWrapper()
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ConnectHandshakeWrapperActor>
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectHandshakeWrapperActorState {
															#line 3545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectHandshakeWrapperActorState(Reference<SSLConnection> const& self) 
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self)
															#line 3552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this));

	}
	~ConnectHandshakeWrapperActorState() 
	{
		fdb_probe_actor_destroy("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_0 = g_network->networkInfo.handshakeLock->take();
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state = 1;
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ConnectHandshakeWrapperActor, 0, Void >*>(static_cast<ConnectHandshakeWrapperActor*>(this)));
															#line 3574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ConnectHandshakeWrapperActorState();
		static_cast<ConnectHandshakeWrapperActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		releaser = FlowLock::Releaser(*g_network->networkInfo.handshakeLock);
															#line 1056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> connected;
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doConnectHandshake(self, connected);
															#line 3601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_1 = connected.getFuture();
															#line 1059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state = 2;
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*>(static_cast<ConnectHandshakeWrapperActor*>(this)));
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectHandshakeWrapperActor, 2, Void >*>(static_cast<ConnectHandshakeWrapperActor*>(this)));
															#line 3618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		releaser = FlowLock::Releaser(*g_network->networkInfo.handshakeLock);
															#line 1056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> connected;
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doConnectHandshake(self, connected);
															#line 3637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_1 = connected.getFuture();
															#line 1059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(FLOW_KNOBS->CONNECTION_MONITOR_TIMEOUT);
															#line 3647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state = 2;
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*>(static_cast<ConnectHandshakeWrapperActor*>(this)));
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ConnectHandshakeWrapperActor, 2, Void >*>(static_cast<ConnectHandshakeWrapperActor*>(this)));
															#line 3654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state > 0) static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectHandshakeWrapperActor*>(this)->ActorCallback< ConnectHandshakeWrapperActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ConnectHandshakeWrapperActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 3745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				std::pair<IPAddress, uint16_t> peerIP = std::make_pair(self->peer_address.ip, self->peer_address.port);
															#line 1071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto iter(g_network->networkInfo.serverTLSConnectionThrottler.find(peerIP));
															#line 1072 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (iter != g_network->networkInfo.serverTLSConnectionThrottler.end())
															#line 3753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 1073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					iter->second.first++;
															#line 3757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
				else
				{
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					g_network->networkInfo.serverTLSConnectionThrottler[peerIP] = std::make_pair(0, now() + FLOW_KNOBS->TLS_CLIENT_CONNECTION_THROTTLE_TIMEOUT);
															#line 3763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
			}
															#line 1079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			self->closeSocket();
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
															#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectHandshakeWrapperActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectHandshakeWrapperActorState(); static_cast<ConnectHandshakeWrapperActor*>(this)->destroy(); return 0; }
															#line 3784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectHandshakeWrapperActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConnectHandshakeWrapperActorState();
		static_cast<ConnectHandshakeWrapperActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
															#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ConnectHandshakeWrapperActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ConnectHandshakeWrapperActorState(); static_cast<ConnectHandshakeWrapperActor*>(this)->destroy(); return 0; }
															#line 3796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ConnectHandshakeWrapperActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ConnectHandshakeWrapperActorState();
		static_cast<ConnectHandshakeWrapperActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when2(Void const& _,int loopDepth) 
	{
															#line 1064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1cont1Catch1(connection_failed(), loopDepth);
															#line 3808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1when2(Void && _,int loopDepth) 
	{
															#line 1064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		return a_body1cont1Catch1(connection_failed(), loopDepth);
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state > 0) static_cast<ConnectHandshakeWrapperActor*>(this)->actor_wait_state = 0;
		static_cast<ConnectHandshakeWrapperActor*>(this)->ActorCallback< ConnectHandshakeWrapperActor, 1, Void >::remove();
		static_cast<ConnectHandshakeWrapperActor*>(this)->ActorCallback< ConnectHandshakeWrapperActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ConnectHandshakeWrapperActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ConnectHandshakeWrapperActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> self;
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowLock::Releaser releaser;
															#line 3945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via connectHandshakeWrapper()
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ConnectHandshakeWrapperActor final : public Actor<Void>, public ActorCallback< ConnectHandshakeWrapperActor, 0, Void >, public ActorCallback< ConnectHandshakeWrapperActor, 1, Void >, public ActorCallback< ConnectHandshakeWrapperActor, 2, Void >, public FastAllocated<ConnectHandshakeWrapperActor>, public ConnectHandshakeWrapperActorState<ConnectHandshakeWrapperActor> {
															#line 3950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ConnectHandshakeWrapperActor>::operator new;
	using FastAllocated<ConnectHandshakeWrapperActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16759534233905758720UL, 695394429508301056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ConnectHandshakeWrapperActor, 0, Void >;
friend struct ActorCallback< ConnectHandshakeWrapperActor, 1, Void >;
friend struct ActorCallback< ConnectHandshakeWrapperActor, 2, Void >;
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ConnectHandshakeWrapperActor(Reference<SSLConnection> const& self) 
															#line 3969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   ConnectHandshakeWrapperActorState<ConnectHandshakeWrapperActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7260785384656561920UL, 8689602154264559616UL);
		ActorExecutionContextHelper __helper(static_cast<ConnectHandshakeWrapperActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connectHandshakeWrapper");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connectHandshakeWrapper", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ConnectHandshakeWrapperActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ConnectHandshakeWrapperActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Void> connectHandshakeWrapper( Reference<SSLConnection> const& self ) {
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new ConnectHandshakeWrapperActor(self));
															#line 4002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

	Future<Void> connectHandshake() override { return connectHandshakeWrapper(Reference<SSLConnection>::addRef(this)); }

	// returns when write() can write at least one byte
	Future<Void> onWritable() override {
		++g_net2->countWriteProbes;
		BindPromise p("N2_WriteProbeError", id);
		auto f = p.getFuture();
		socket.async_write_some(boost::asio::null_buffers(), std::move(p));
		return f;
	}

	// returns when read() can read at least one byte
	Future<Void> onReadable() override {
		++g_net2->countReadProbes;
		BindPromise p("N2_ReadProbeError", id);
		auto f = p.getFuture();
		socket.async_read_some(boost::asio::null_buffers(), std::move(p));
		return f;
	}

	// Reads as many bytes as possible from the read buffer into [begin,end) and returns the number of bytes read (might
	// be 0)
	int read(uint8_t* begin, uint8_t* end) override {
		boost::system::error_code err;
		++g_net2->countReads;
		size_t toRead = end - begin;
		size_t size = ssl_sock.read_some(boost::asio::mutable_buffers_1(begin, toRead), err);
		g_net2->bytesReceived += size;
		//TraceEvent("ConnRead", this->id).detail("Bytes", size);
		if (err) {
			if (err == boost::asio::error::would_block) {
				++g_net2->countWouldBlock;
				return 0;
			}
			onReadError(err);
			throw connection_failed();
		}
		ASSERT(size); // If the socket is closed, we expect an 'eof' error, not a zero return value

		return size;
	}

	// Writes as many bytes as possible from the given SendBuffer chain into the write buffer and returns the number of
	// bytes written (might be 0)
	int write(SendBuffer const* data, int limit) override {
#ifdef __APPLE__
		// For some reason, writing ssl_sock with more than 2016 bytes when socket is writeable sometimes results in a
		// broken pipe error.
		limit = std::min(limit, 2016);
#endif
		boost::system::error_code err;
		++g_net2->countWrites;

		size_t sent = ssl_sock.write_some(
		    boost::iterator_range<SendBufferIterator>(SendBufferIterator(data, limit), SendBufferIterator()), err);

		if (err) {
			// Since there was an error, sent's value can't be used to infer that the buffer has data and the limit is
			// positive so check explicitly.
			ASSERT(limit > 0);
			bool notEmpty = false;
			for (auto p = data; p; p = p->next)
				if (p->bytes_written - p->bytes_sent > 0) {
					notEmpty = true;
					break;
				}
			ASSERT(notEmpty);

			if (err == boost::asio::error::would_block) {
				++g_net2->countWouldBlock;
				return 0;
			}
			onWriteError(err);
			throw connection_failed();
		}

		ASSERT(sent); // Make sure data was sent, and also this check will fail if the buffer chain was empty or the
		              // limit was not > 0.
		return sent;
	}

	NetworkAddress getPeerAddress() const override { return peer_address; }

	bool hasTrustedPeer() const override { return has_trusted_peer; }

	UID getDebugID() const override { return id; }

	tcp::socket& getSocket() override { return socket; }

	ssl_socket& getSSLSocket() { return ssl_sock; }

private:
	UID id;
	tcp::socket socket;
	ssl_socket ssl_sock;
	NetworkAddress peer_address;
	Reference<ReferencedObject<boost::asio::ssl::context>> sslContext;
	bool has_trusted_peer;

	void init() {
		// Socket settings that have to be set after connect or accept succeeds
		socket.non_blocking(true);
		socket.set_option(boost::asio::ip::tcp::no_delay(true));
		platform::setCloseOnExec(socket.native_handle());
	}

	void closeSocket() {
		boost::system::error_code cancelError;
		socket.cancel(cancelError);
		boost::system::error_code closeError;
		socket.close(closeError);
		boost::system::error_code shutdownError;
		ssl_sock.shutdown(shutdownError);
	}

	void onReadError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_ReadError", id)
		    .suppressFor(1.0)
		    .detail("PeerAddr", peer_address)
		    .detail("PeerAddress", peer_address)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}
	void onWriteError(const boost::system::error_code& error) {
		TraceEvent(SevWarn, "N2_WriteError", id)
		    .suppressFor(1.0)
		    .detail("PeerAddr", peer_address)
		    .detail("PeerAddress", peer_address)
		    .detail("ErrorCode", error.value())
		    .detail("Message", error.message());
		closeSocket();
	}
};

class SSLListener final : public IListener, ReferenceCounted<SSLListener> {
	boost::asio::io_context& io_service;
	NetworkAddress listenAddress;
	tcp::acceptor acceptor;
	AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* contextVar;

public:
	SSLListener(boost::asio::io_context& io_service,
	            AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* contextVar,
	            NetworkAddress listenAddress)
	  : io_service(io_service), listenAddress(listenAddress), acceptor(io_service, tcpEndpoint(listenAddress)),
	    contextVar(contextVar) {
		// when port 0 is passed in, a random port will be opened
		// set listenAddress as the address with the actual port opened instead of port 0
		if (listenAddress.port == 0) {
			this->listenAddress = NetworkAddress::parse(acceptor.local_endpoint()
			                                                .address()
			                                                .to_string()
			                                                .append(":")
			                                                .append(std::to_string(acceptor.local_endpoint().port()))
			                                                .append(listenAddress.isTLS() ? ":tls" : ""));
		}
		platform::setCloseOnExec(acceptor.native_handle());
	}

	void addref() override { ReferenceCounted<SSLListener>::addref(); }
	void delref() override { ReferenceCounted<SSLListener>::delref(); }

	// Returns one incoming connection when it is available
	Future<Reference<IConnection>> accept() override { return doAccept(this); }

	NetworkAddress getListenAddress() const override { return listenAddress; }

private:
																#line 4176 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via doAccept()
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class DoAcceptActor1>
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptActor1State {
															#line 4182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptActor1State(SSLListener* const& self) 
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   conn(new SSLConnection(self->io_service, self->contextVar->get())),
															#line 1255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   peer_endpoint()
															#line 4193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("doAccept", reinterpret_cast<unsigned long>(this));

	}
	~DoAcceptActor1State() 
	{
		fdb_probe_actor_destroy("doAccept", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				BindPromise p("N2_AcceptError", UID());
															#line 1258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto f = p.getFuture();
															#line 1259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				self->acceptor.async_accept(conn->getSocket(), peer_endpoint, std::move(p));
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = f;
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (static_cast<DoAcceptActor1*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoAcceptActor1*>(this)->actor_wait_state = 1;
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoAcceptActor1, 0, Void >*>(static_cast<DoAcceptActor1*>(this)));
															#line 4222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoAcceptActor1State();
		static_cast<DoAcceptActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 1268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			conn->close();
															#line 1269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 4254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		auto peer_address = peer_endpoint.address().is_v6() ? IPAddress(peer_endpoint.address().to_v6().to_bytes()) : IPAddress(peer_endpoint.address().to_v4().to_ulong());
															#line 1264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		conn->accept(NetworkAddress(peer_address, peer_endpoint.port(), false, true));
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<DoAcceptActor1*>(this)->SAV<Reference<IConnection>>::futures) { (void)(conn); this->~DoAcceptActor1State(); static_cast<DoAcceptActor1*>(this)->destroy(); return 0; }
															#line 4272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<DoAcceptActor1*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(conn)); // state_var_RVO
		this->~DoAcceptActor1State();
		static_cast<DoAcceptActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		auto peer_address = peer_endpoint.address().is_v6() ? IPAddress(peer_endpoint.address().to_v6().to_bytes()) : IPAddress(peer_endpoint.address().to_v4().to_ulong());
															#line 1264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		conn->accept(NetworkAddress(peer_address, peer_endpoint.port(), false, true));
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<DoAcceptActor1*>(this)->SAV<Reference<IConnection>>::futures) { (void)(conn); this->~DoAcceptActor1State(); static_cast<DoAcceptActor1*>(this)->destroy(); return 0; }
															#line 4288 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<DoAcceptActor1*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(std::move(conn)); // state_var_RVO
		this->~DoAcceptActor1State();
		static_cast<DoAcceptActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoAcceptActor1*>(this)->actor_wait_state > 0) static_cast<DoAcceptActor1*>(this)->actor_wait_state = 0;
		static_cast<DoAcceptActor1*>(this)->ActorCallback< DoAcceptActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoAcceptActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoAcceptActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoAcceptActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	SSLListener* self;
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<SSLConnection> conn;
															#line 1255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	tcp::acceptor::endpoint_type peer_endpoint;
															#line 4377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via doAccept()
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class DoAcceptActor1 final : public Actor<Reference<IConnection>>, public ActorCallback< DoAcceptActor1, 0, Void >, public FastAllocated<DoAcceptActor1>, public DoAcceptActor1State<DoAcceptActor1> {
															#line 4382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<DoAcceptActor1>::operator new;
	using FastAllocated<DoAcceptActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9356202762953707264UL, 14099876164327578368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IConnection>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoAcceptActor1, 0, Void >;
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	DoAcceptActor1(SSLListener* const& self) 
															#line 4399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Reference<IConnection>>(),
		   DoAcceptActor1State<DoAcceptActor1>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doAccept", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16158557830277718272UL, 17507257483778108416UL);
		ActorExecutionContextHelper __helper(static_cast<DoAcceptActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doAccept");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doAccept", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoAcceptActor1, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Reference<IConnection>> doAccept( SSLListener* const& self ) {
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Reference<IConnection>>(new DoAcceptActor1(self));
															#line 4431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
};

// 5MB for loading files into memory

Net2::Net2(const TLSConfig& tlsConfig, bool useThreadPool, bool useMetrics)
  : globals(enumGlobal::COUNT), useThreadPool(useThreadPool), reactor(this),
    sslContextVar({ ReferencedObject<boost::asio::ssl::context>::from(
        boost::asio::ssl::context(boost::asio::ssl::context::tls)) }),
    sslHandshakerThreadsStarted(0), sslPoolHandshakesInProgress(0), tlsConfig(tlsConfig),
    tlsInitializedState(ETLSInitState::NONE), network(this), tscBegin(0), tscEnd(0), taskBegin(0),
    currentTaskID(TaskPriority::DefaultYield), stopped(false), started(false), numYields(0),
    lastPriorityStats(nullptr) {
	// Until run() is called, yield() will always yield
	TraceEvent("Net2Starting").log();

	// Set the global members
	if (useMetrics) {
		setGlobal(INetwork::enTDMetrics, (flowGlobalType)&tdmetrics);
	}
	if (FLOW_KNOBS->ENABLE_CHAOS_FEATURES) {
		setGlobal(INetwork::enChaosMetrics, (flowGlobalType)&chaosMetrics);
	}
	setGlobal(INetwork::enMetrics, (flowGlobalType)&metrics);
	setGlobal(INetwork::enNetworkConnections, (flowGlobalType)network);
	setGlobal(INetwork::enASIOService, (flowGlobalType)&reactor.ios);
	setGlobal(INetwork::enBlobCredentialFiles, &blobCredentialFiles);
	setGlobal(INetwork::enProxy, &proxy);

#ifdef __linux__
	setGlobal(INetwork::enEventFD, (flowGlobalType)N2::ASIOReactor::newEventFD(reactor));
#endif

	updateNow();
}

															#line 4470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via reloadCertificatesOnChange()
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ReloadCertificatesOnChangeActor>
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ReloadCertificatesOnChangeActorState {
															#line 4476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ReloadCertificatesOnChangeActorState(TLSConfig const& config,std::function<void()> const& onPolicyFailure,AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* const& contextVar,Reference<TLSPolicy>* const& policy) 
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : config(config),
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   onPolicyFailure(onPolicyFailure),
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   contextVar(contextVar),
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   policy(policy)
															#line 4489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this));

	}
	~ReloadCertificatesOnChangeActorState() 
	{
		fdb_probe_actor_destroy("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (FLOW_KNOBS->TLS_CERT_REFRESH_DELAY_SECONDS <= 0)
															#line 4504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (!static_cast<ReloadCertificatesOnChangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReloadCertificatesOnChangeActorState(); static_cast<ReloadCertificatesOnChangeActor*>(this)->destroy(); return 0; }
															#line 4508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				new (&static_cast<ReloadCertificatesOnChangeActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ReloadCertificatesOnChangeActorState();
				static_cast<ReloadCertificatesOnChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			;
															#line 4516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReloadCertificatesOnChangeActorState();
		static_cast<ReloadCertificatesOnChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		mismatches = 0;
															#line 1323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		fileChanged = AsyncTrigger();
															#line 1324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		lifetimes = std::vector<Future<Void>>();
															#line 1325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		const int& intervalSeconds = FLOW_KNOBS->TLS_CERT_REFRESH_DELAY_SECONDS;
															#line 1326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		lifetimes.push_back(watchFileForChanges( config.getCertificatePathSync(), &fileChanged, &intervalSeconds, "TLSCertificateRefreshStatError"));
															#line 1328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		lifetimes.push_back( watchFileForChanges(config.getKeyPathSync(), &fileChanged, &intervalSeconds, "TLSKeyRefreshStatError"));
															#line 1330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		lifetimes.push_back( watchFileForChanges(config.getCAPathSync(), &fileChanged, &intervalSeconds, "TLSCARefreshStatError"));
															#line 1332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		;
															#line 4553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (IAsyncFileSystem::filesystem() != nullptr)
															#line 4569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 1;
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >*>(static_cast<ReloadCertificatesOnChangeActor*>(this)));
															#line 4582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state > 0) static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReloadCertificatesOnChangeActor*>(this)->ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = fileChanged.onTrigger();
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 2;
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >*>(static_cast<ReloadCertificatesOnChangeActor*>(this)));
															#line 4705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		TraceEvent("TLSCertificateRefreshBegin").log();
															#line 4714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<LoadedTLSConfig> __when_expr_2 = config.loadAsync();
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4720 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 3;
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*>(static_cast<ReloadCertificatesOnChangeActor*>(this)));
															#line 4725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		TraceEvent("TLSCertificateRefreshBegin").log();
															#line 4740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		try {
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<LoadedTLSConfig> __when_expr_2 = config.loadAsync();
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 3;
															#line 1337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*>(static_cast<ReloadCertificatesOnChangeActor*>(this)));
															#line 4751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state > 0) static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReloadCertificatesOnChangeActor*>(this)->ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1345 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4848 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 1346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 4852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			}
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			mismatches++;
															#line 1350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			TraceEvent(SevWarn, "TLSCertificateRefreshMismatch").error(e).detail("mismatches", mismatches);
															#line 4858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(LoadedTLSConfig const& loaded,int loopDepth) 
	{
															#line 1338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		boost::asio::ssl::context context(boost::asio::ssl::context::tls);
															#line 1339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		ConfigureSSLContext(loaded, context);
															#line 1340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		*policy = makeReference<TLSPolicy>(loaded, onPolicyFailure);
															#line 1341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		TraceEvent(SevInfo, "TLSCertificateRefreshSucceeded").log();
															#line 1342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		mismatches = 0;
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		contextVar->set(ReferencedObject<boost::asio::ssl::context>::from(std::move(context)));
															#line 4883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(LoadedTLSConfig && loaded,int loopDepth) 
	{
															#line 1338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		boost::asio::ssl::context context(boost::asio::ssl::context::tls);
															#line 1339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		ConfigureSSLContext(loaded, context);
															#line 1340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		*policy = makeReference<TLSPolicy>(loaded, onPolicyFailure);
															#line 1341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		TraceEvent(SevInfo, "TLSCertificateRefreshSucceeded").log();
															#line 1342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		mismatches = 0;
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		contextVar->set(ReferencedObject<boost::asio::ssl::context>::from(std::move(context)));
															#line 4902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(LoadedTLSConfig const& loaded,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loaded, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(LoadedTLSConfig && loaded,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(loaded), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state > 0) static_cast<ReloadCertificatesOnChangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReloadCertificatesOnChangeActor*>(this)->ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >::remove();

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*,LoadedTLSConfig const& value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*,LoadedTLSConfig && value) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*,Error err) 
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	TLSConfig config;
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::function<void()> onPolicyFailure;
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* contextVar;
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Reference<TLSPolicy>* policy;
															#line 1322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	int mismatches;
															#line 1323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	AsyncTrigger fileChanged;
															#line 1324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::vector<Future<Void>> lifetimes;
															#line 5009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via reloadCertificatesOnChange()
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ReloadCertificatesOnChangeActor final : public Actor<Void>, public ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >, public ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >, public ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >, public FastAllocated<ReloadCertificatesOnChangeActor>, public ReloadCertificatesOnChangeActorState<ReloadCertificatesOnChangeActor> {
															#line 5014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ReloadCertificatesOnChangeActor>::operator new;
	using FastAllocated<ReloadCertificatesOnChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13998607754601233664UL, 17459494140935577856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >;
friend struct ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >;
friend struct ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >;
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ReloadCertificatesOnChangeActor(TLSConfig const& config,std::function<void()> const& onPolicyFailure,AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* const& contextVar,Reference<TLSPolicy>* const& policy) 
															#line 5033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   ReloadCertificatesOnChangeActorState<ReloadCertificatesOnChangeActor>(config, onPolicyFailure, contextVar, policy),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1057131585964832512UL, 16691980449039810048UL);
		ActorExecutionContextHelper __helper(static_cast<ReloadCertificatesOnChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("reloadCertificatesOnChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("reloadCertificatesOnChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReloadCertificatesOnChangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReloadCertificatesOnChangeActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReloadCertificatesOnChangeActor, 2, LoadedTLSConfig >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<Void> reloadCertificatesOnChange( TLSConfig const& config, std::function<void()> const& onPolicyFailure, AsyncVar<Reference<ReferencedObject<boost::asio::ssl::context>>>* const& contextVar, Reference<TLSPolicy>* const& policy ) {
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new ReloadCertificatesOnChangeActor(config, onPolicyFailure, contextVar, policy));
															#line 5067 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

void Net2::initTLS(ETLSInitState targetState) {
	if (tlsInitializedState >= targetState) {
		return;
	}
	// Any target state must be higher than NONE so if the current state is NONE
	// then initialize the TLS config
	if (tlsInitializedState == ETLSInitState::NONE) {
		auto onPolicyFailure = [this]() { this->countTLSPolicyFailures++; };
		try {
			boost::asio::ssl::context newContext(boost::asio::ssl::context::tls);
			const LoadedTLSConfig& loaded = tlsConfig.loadSync();
			TraceEvent("Net2TLSConfig")
			    .detail("CAPath", tlsConfig.getCAPathSync())
			    .detail("CertificatePath", tlsConfig.getCertificatePathSync())
			    .detail("KeyPath", tlsConfig.getKeyPathSync())
			    .detail("HasPassword", !loaded.getPassword().empty())
			    .detail("VerifyPeers", boost::algorithm::join(loaded.getVerifyPeers(), "|"))
			    .detail("DisablePlainTextConnection", tlsConfig.getDisablePlainTextConnection());
			auto loadedTlsConfig = tlsConfig.loadSync();
			ConfigureSSLContext(loadedTlsConfig, newContext);
			activeTlsPolicy = makeReference<TLSPolicy>(loadedTlsConfig, onPolicyFailure);
			sslContextVar.set(ReferencedObject<boost::asio::ssl::context>::from(std::move(newContext)));
		} catch (Error& e) {
			TraceEvent("Net2TLSInitError").error(e);
		}
		backgroundCertRefresh =
		    reloadCertificatesOnChange(tlsConfig, onPolicyFailure, &sslContextVar, &activeTlsPolicy);
	}

	// If a TLS connection is actually going to be used then start background threads if configured
	if (targetState > ETLSInitState::CONFIG) {
		int threadsToStart;
		switch (targetState) {
		case ETLSInitState::CONNECT:
			threadsToStart = FLOW_KNOBS->TLS_CLIENT_HANDSHAKE_THREADS;
			break;
		case ETLSInitState::LISTEN:
			threadsToStart = FLOW_KNOBS->TLS_SERVER_HANDSHAKE_THREADS;
			break;
		default:
			threadsToStart = 0;
		};
		threadsToStart -= sslHandshakerThreadsStarted;

		if (threadsToStart > 0) {
			if (sslHandshakerThreadsStarted == 0) {
#if defined(__linux__)
				if (mallopt(M_ARENA_MAX, FLOW_KNOBS->TLS_MALLOC_ARENA_MAX) != 1) {
					TraceEvent(SevWarn, "TLSMallocSetMaxArenasFailure")
					    .detail("MaxArenas", FLOW_KNOBS->TLS_MALLOC_ARENA_MAX);
				};
#endif
				sslHandshakerPool = createGenericThreadPool(FLOW_KNOBS->TLS_HANDSHAKE_THREAD_STACKSIZE);
			}

			for (int i = 0; i < threadsToStart; ++i) {
				++sslHandshakerThreadsStarted;
				sslHandshakerPool->addThread(new SSLHandshakerThread(), "fdb-ssl-connect");
			}
		}
	}

	tlsInitializedState = targetState;
}

															#line 5137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via logTimeOffset()
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class Net2_LogTimeOffsetActor>
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class Net2_LogTimeOffsetActorState {
															#line 5143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Net2_LogTimeOffsetActorState() 
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	{
															#line 5149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		fdb_probe_actor_create("logTimeOffset", reinterpret_cast<unsigned long>(this));

	}
	~Net2_LogTimeOffsetActorState() 
	{
		fdb_probe_actor_destroy("logTimeOffset", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			;
															#line 5163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Net2_LogTimeOffsetActorState();
		static_cast<Net2_LogTimeOffsetActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		double processTime = timer_monotonic();
															#line 1423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		double systemTime = timer();
															#line 1424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		TraceEvent("ProcessTimeOffset") .detailf("ProcessTime", "%lf", processTime) .detailf("SystemTime", "%lf", systemTime) .detailf("OffsetFromSystemTime", "%lf", processTime - systemTime);
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_0 = ::delay(FLOW_KNOBS->TIME_OFFSET_LOGGING_INTERVAL);
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<Net2_LogTimeOffsetActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Net2_LogTimeOffsetActor*>(this)->actor_wait_state = 1;
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Net2_LogTimeOffsetActor, 0, Void >*>(static_cast<Net2_LogTimeOffsetActor*>(this)));
															#line 5206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Net2_LogTimeOffsetActor*>(this)->actor_wait_state > 0) static_cast<Net2_LogTimeOffsetActor*>(this)->actor_wait_state = 0;
		static_cast<Net2_LogTimeOffsetActor*>(this)->ActorCallback< Net2_LogTimeOffsetActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Net2_LogTimeOffsetActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Net2_LogTimeOffsetActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Net2_LogTimeOffsetActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Net2_LogTimeOffsetActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Net2_LogTimeOffsetActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Net2_LogTimeOffsetActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logTimeOffset", reinterpret_cast<unsigned long>(this), 0);

	}
};
// This generated class is to be used only via logTimeOffset()
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class Net2_LogTimeOffsetActor final : public Actor<Void>, public ActorCallback< Net2_LogTimeOffsetActor, 0, Void >, public FastAllocated<Net2_LogTimeOffsetActor>, public Net2_LogTimeOffsetActorState<Net2_LogTimeOffsetActor> {
															#line 5302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<Net2_LogTimeOffsetActor>::operator new;
	using FastAllocated<Net2_LogTimeOffsetActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12780802771962062592UL, 3842910134975517696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Net2_LogTimeOffsetActor, 0, Void >;
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Net2_LogTimeOffsetActor() 
															#line 5319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   Net2_LogTimeOffsetActorState<Net2_LogTimeOffsetActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("logTimeOffset", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16010764156899767808UL, 14083495797646494208UL);
		ActorExecutionContextHelper __helper(static_cast<Net2_LogTimeOffsetActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("logTimeOffset");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("logTimeOffset", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Net2_LogTimeOffsetActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] Future<Void> Net2::logTimeOffset(  ) {
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new Net2_LogTimeOffsetActor());
															#line 5351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

void Net2::initMetrics() {
	bytesReceived.init("Net2.BytesReceived"_sr);
	countWriteProbes.init("Net2.CountWriteProbes"_sr);
	countReadProbes.init("Net2.CountReadProbes"_sr);
	countReads.init("Net2.CountReads"_sr);
	countWouldBlock.init("Net2.CountWouldBlock"_sr);
	countWrites.init("Net2.CountWrites"_sr);
	countRunLoop.init("Net2.CountRunLoop"_sr);
	countTasks.init("Net2.CountTasks"_sr);
	countYields.init("Net2.CountYields"_sr);
	countYieldBigStack.init("Net2.CountYieldBigStack"_sr);
	countYieldCalls.init("Net2.CountYieldCalls"_sr);
	countASIOEvents.init("Net2.CountASIOEvents"_sr);
	countYieldCallsTrue.init("Net2.CountYieldCallsTrue"_sr);
	countRunLoopProfilingSignals.init("Net2.CountRunLoopProfilingSignals"_sr);
	countTLSPolicyFailures.init("Net2.CountTLSPolicyFailures"_sr);
	priorityMetric.init("Net2.Priority"_sr);
	awakeMetric.init("Net2.Awake"_sr);
	slowTaskMetric.init("Net2.SlowTask"_sr);
	countLaunchTime.init("Net2.CountLaunchTime"_sr);
	countReactTime.init("Net2.CountReactTime"_sr);
	taskQueue.initMetrics();
}

bool Net2::checkRunnable() {
	return !started.exchange(true);
}

#ifdef ENABLE_SAMPLING
ActorLineageSet& Net2::getActorLineageSet() {
	return actorLineageSet;
}
#endif

void Net2::run() {
	TraceEvent::setNetworkThread();
	TraceEvent("Net2Running").log();
	thread_network = this;

	unsigned int tasksSinceReact = 0;

#ifdef WIN32
	if (timeBeginPeriod(1) != TIMERR_NOERROR)
		TraceEvent(SevError, "TimeBeginPeriodError").log();
#endif

	timeOffsetLogger = logTimeOffset();
	const char* flow_profiler_enabled = getenv("FLOW_PROFILER_ENABLED");
	if (flow_profiler_enabled != nullptr && *flow_profiler_enabled != '\0') {
		// The empty string check is to allow running `FLOW_PROFILER_ENABLED= ./fdbserver` to force disabling flow
		// profiling at startup.
		startProfiling(this);
	}

	// Get the address to the launch function
	typedef void (*runCycleFuncPtr)();
	runCycleFuncPtr runFunc = reinterpret_cast<runCycleFuncPtr>(
	    reinterpret_cast<flowGlobalType>(g_network->global(INetwork::enRunCycleFunc)));

	started.store(true);
	double nnow = timer_monotonic();

	while (!stopped) {
		FDB_TRACE_PROBE(run_loop_begin);
		++countRunLoop;

		if (runFunc) {
			tscBegin = timestampCounter();
			taskBegin = nnow;
			trackAtPriority(TaskPriority::RunCycleFunction, taskBegin);
			runFunc();
			double taskEnd = timer_monotonic();
			trackAtPriority(TaskPriority::RunLoop, taskEnd);
			countLaunchTime += taskEnd - taskBegin;
			checkForSlowTask(tscBegin, timestampCounter(), taskEnd - taskBegin, TaskPriority::RunCycleFunction);
		}

		double sleepTime = 0;
		if (taskQueue.canSleep()) {
			sleepTime = 1e99;
			double sleepStart = timer_monotonic();
			sleepTime = taskQueue.getSleepTime(sleepStart);
			if (sleepTime > 0) {
#if defined(__linux__)
				// notify the run loop monitoring thread that we have gone idle
				net2RunLoopSleeps.fetch_add(1);
#endif

				trackAtPriority(TaskPriority::Zero, sleepStart);
				awakeMetric = false;
				priorityMetric = 0;
				reactor.sleep(sleepTime);
				awakeMetric = true;
			}
		}

		tscBegin = timestampCounter();
		taskBegin = timer_monotonic();
		trackAtPriority(TaskPriority::ASIOReactor, taskBegin);
		reactor.react();
		tasksSinceReact = 0;

		updateNow();
		double now = this->currentTime;
		trackAtPriority(TaskPriority::RunLoop, now);

		countReactTime += now - taskBegin;
		checkForSlowTask(tscBegin, timestampCounter(), now - taskBegin, TaskPriority::ASIOReactor);

		if ((now - nnow) > FLOW_KNOBS->SLOW_LOOP_CUTOFF &&
		    nondeterministicRandom()->random01() < (now - nnow) * FLOW_KNOBS->SLOW_LOOP_SAMPLING_RATE)
			TraceEvent("SomewhatSlowRunLoopTop").detail("Elapsed", now - nnow);

		taskQueue.processReadyTimers(now);

		taskQueue.processThreadReady();

		tscBegin = timestampCounter();
		tscEnd = tscBegin + FLOW_KNOBS->TSC_YIELD_TIME;
		taskBegin = timer_monotonic();
		numYields = 0;
		TaskPriority minTaskID = TaskPriority::Max;
		[[maybe_unused]] int queueSize = taskQueue.getNumReadyTasks();

		FDB_TRACE_PROBE(run_loop_tasks_start, queueSize);
		while (taskQueue.hasReadyTask()) {
			++countTasks;
			currentTaskID = taskQueue.getReadyTaskID();
			priorityMetric = static_cast<int64_t>(currentTaskID);
			PromiseTask* task = taskQueue.getReadyTask();
			taskQueue.popReadyTask();

			try {
				++tasksSinceReact;
				(*task)();
			} catch (Error& e) {
				TraceEvent(SevError, "TaskError").error(e);
			} catch (...) {
				TraceEvent(SevError, "TaskError").error(unknown_error());
			}

			if (currentTaskID < minTaskID) {
				trackAtPriority(currentTaskID, taskBegin);
				minTaskID = currentTaskID;
			}

			// attempt to empty out the IO backlog
			if (tasksSinceReact >= FLOW_KNOBS->TASKS_PER_REACTOR_CHECK) {
				if (runFunc) {
					runFunc();
				}
				reactor.react();
				tasksSinceReact = 0;
			}

			double tscNow = timestampCounter();
			double newTaskBegin = timer_monotonic();
			if (check_yield(TaskPriority::Max, tscNow)) {
				checkForSlowTask(tscBegin, tscNow, newTaskBegin - taskBegin, currentTaskID);
				taskBegin = newTaskBegin;
				FDB_TRACE_PROBE(run_loop_yield);
				++countYields;
				break;
			}

			taskBegin = newTaskBegin;
			tscBegin = tscNow;
		}

		trackAtPriority(TaskPriority::RunLoop, taskBegin);

		queueSize = taskQueue.getNumReadyTasks();
		FDB_TRACE_PROBE(run_loop_done, queueSize);

#if defined(__linux__)
		if (FLOW_KNOBS->RUN_LOOP_PROFILING_INTERVAL > 0) {
			sigset_t orig_set;
			pthread_sigmask(SIG_BLOCK, &sigprof_set, &orig_set);

			size_t other_offset = net2backtraces_offset;
			bool was_overflow = net2backtraces_overflow;
			int signal_count = net2backtraces_count;

			countRunLoopProfilingSignals += signal_count;

			if (other_offset) {
				volatile void** _traces = net2backtraces;
				net2backtraces = other_backtraces;
				other_backtraces = _traces;

				net2backtraces_offset = 0;
			}

			net2backtraces_overflow = false;
			net2backtraces_count = 0;

			pthread_sigmask(SIG_SETMASK, &orig_set, nullptr);

			if (was_overflow) {
				TraceEvent("Net2RunLoopProfilerOverflow")
				    .detail("SignalsReceived", signal_count)
				    .detail("BackTraceHarvested", other_offset != 0);
			}
			if (other_offset) {
				size_t iter_offset = 0;
				while (iter_offset < other_offset) {
					ProfilingSample* ps = (ProfilingSample*)(other_backtraces + iter_offset);
					TraceEvent(SevWarn, "Net2RunLoopTrace")
					    .detailf("TraceTime", "%.6f", ps->timestamp)
					    .detail("Trace", platform::format_backtrace(ps->frames, ps->length));
					iter_offset += ps->length + 2;
				}
			}

			// notify the run loop monitoring thread that we are making progress
			net2RunLoopIterations.fetch_add(1);
		}
#endif
		nnow = timer_monotonic();

		if ((nnow - now) > FLOW_KNOBS->SLOW_LOOP_CUTOFF &&
		    nondeterministicRandom()->random01() < (nnow - now) * FLOW_KNOBS->SLOW_LOOP_SAMPLING_RATE)
			TraceEvent("SomewhatSlowRunLoopBottom")
			    .detail("Elapsed", nnow - now); // This includes the time spent running tasks
	}

	for (auto& fn : stopCallbacks) {
		fn();
	}

#ifdef WIN32
	timeEndPeriod(1);
#endif
} // Net2::run

// Updates the PriorityStats found in NetworkMetrics
void Net2::updateStarvationTracker(struct NetworkMetrics::PriorityStats& binStats,
                                   TaskPriority priority,
                                   TaskPriority lastPriority,
                                   double now) {

	// Busy -> idle at binStats.priority
	if (binStats.priority > priority && binStats.priority <= lastPriority) {
		binStats.active = false;
		binStats.duration += now - binStats.windowedTimer;
		binStats.maxDuration = std::max(binStats.maxDuration, now - binStats.timer);
	}

	// Idle -> busy at binStats.priority
	else if (binStats.priority <= priority && binStats.priority > lastPriority) {
		binStats.active = true;
		binStats.timer = now;
		binStats.windowedTimer = now;
	}
}

// Update both vectors of starvation trackers (one that updates every 5s and the other every 1s)
void Net2::trackAtPriority(TaskPriority priority, double now) {
	if (lastPriorityStats == nullptr || priority != lastPriorityStats->priority) {
		// Start tracking current priority
		auto activeStatsItr = networkInfo.metrics.activeTrackers.try_emplace(priority, priority);
		activeStatsItr.first->second.active = true;
		activeStatsItr.first->second.windowedTimer = now;

		if (lastPriorityStats != nullptr) {
			// Stop tracking previous priority
			lastPriorityStats->active = false;
			lastPriorityStats->duration += now - lastPriorityStats->windowedTimer;
		}

		// Update starvation trackers
		TaskPriority lastPriority = (lastPriorityStats == nullptr) ? TaskPriority::Zero : lastPriorityStats->priority;
		for (auto& binStats : networkInfo.metrics.starvationTrackers) {
			if (binStats.priority > lastPriority && binStats.priority > priority) {
				break;
			}
			updateStarvationTracker(binStats, priority, lastPriority, now);
		}

		// Update starvation trackers for network busyness
		updateStarvationTracker(networkInfo.metrics.starvationTrackerNetworkBusyness, priority, lastPriority, now);

		lastPriorityStats = &activeStatsItr.first->second;
	}
}

void Net2::checkForSlowTask(int64_t tscBegin, int64_t tscEnd, double duration, TaskPriority priority) {
	int64_t elapsed = tscEnd - tscBegin;
	if (elapsed > FLOW_KNOBS->TSC_YIELD_TIME && tscBegin > 0) {
		int i = std::min<double>(NetworkMetrics::SLOW_EVENT_BINS - 1, log(elapsed / 1e6) / log(2.));
		++networkInfo.metrics.countSlowEvents[i];
		int64_t warnThreshold = g_network->isSimulated() ? 10e9 : 500e6;

		// printf("SlowTask: %d, %d yields\n", (int)(elapsed/1e6), numYields);

		slowTaskMetric->clocks = elapsed;
		slowTaskMetric->duration = (int64_t)(duration * 1e9);
		slowTaskMetric->priority = static_cast<int64_t>(priority);
		slowTaskMetric->numYields = numYields;
		slowTaskMetric->log();

		double sampleRate = std::min(1.0, (elapsed > warnThreshold) ? 1.0 : elapsed / 10e9);
		double slowTaskProfilingLogInterval =
		    std::max(FLOW_KNOBS->RUN_LOOP_PROFILING_INTERVAL, FLOW_KNOBS->SLOWTASK_PROFILING_LOG_INTERVAL);
		if (slowTaskProfilingLogInterval > 0 && duration > slowTaskProfilingLogInterval) {
			sampleRate = 1; // Always include slow task events that could show up in our slow task profiling.
		}

		if (!DEBUG_DETERMINISM && (nondeterministicRandom()->random01() < sampleRate))
			TraceEvent(elapsed > warnThreshold ? SevWarnAlways : SevInfo, "SlowTask")
			    .detail("TaskID", priority)
			    .detail("MClocks", elapsed / 1e6)
			    .detail("Duration", duration)
			    .detail("SampleRate", sampleRate)
			    .detail("NumYields", numYields);
	}
}

bool Net2::check_yield(TaskPriority taskID, int64_t tscNow) {
	// SOMEDAY: Yield if there are lots of higher priority tasks queued?
	if ((g_stackYieldLimit) && ((intptr_t)&taskID < g_stackYieldLimit)) {
		++countYieldBigStack;
		return true;
	}

	taskQueue.processThreadReady();

	if (taskID == TaskPriority::DefaultYield)
		taskID = currentTaskID;
	if (taskQueue.hasReadyTask() && taskQueue.getReadyTaskPriority() > int64_t(taskID) << 32) {
		return true;
	}

	if (tscNow < tscBegin) {
		return true;
	}

	if (tscNow > tscEnd) {
		++numYields;
		return true;
	}

	return false;
}

bool Net2::check_yield(TaskPriority taskID) {
	if (numYields > 0) {
		++numYields;
		return true;
	}

	return check_yield(taskID, timestampCounter());
}

Future<class Void> Net2::yield(TaskPriority taskID) {
	++countYieldCalls;
	if (taskID == TaskPriority::DefaultYield)
		taskID = currentTaskID;
	if (check_yield(taskID)) {
		++countYieldCallsTrue;
		return delay(0, taskID);
	}
	g_network->setCurrentTask(taskID);
	return Void();
}

// TODO: can we wrap our swift task and insert it in here?
Future<Void> Net2::delay(double seconds, TaskPriority taskId) {
	if (seconds >= 4e12) // Intervals that overflow an int64_t in microseconds (more than 100,000 years) are treated
	                     // as infinite
		return Never();

	PromiseTask* t = new PromiseTask;
	if (seconds <= 0.) {
		taskQueue.addReady(taskId, t);
	} else {
		double at = now() + seconds;
		taskQueue.addTimer(at, taskId, t);
	}
	return t->promise.getFuture();
}

Future<Void> Net2::orderedDelay(double seconds, TaskPriority taskId) {
	// The regular delay already provides the required ordering property
	return delay(seconds, taskId);
}

void Net2::_swiftEnqueue(void* _job) {
#ifdef WITH_SWIFT
	swift::Job* job = (swift::Job*)_job;
	TaskPriority priority = swift_priority_to_net2(job->getPriority());
	PromiseTask* t = new PromiseTask(job);
	taskQueue.addReady(priority, t);
#endif
}

void Net2::onMainThread(Promise<Void>&& signal, TaskPriority taskID) {
	if (stopped)
		return;
	PromiseTask* p = new PromiseTask(std::move(signal));
	if (taskQueue.addReadyThreadSafe(isOnMainThread(), taskID, p)) {
		reactor.wake();
	}
}

THREAD_HANDLE Net2::startThread(THREAD_FUNC_RETURN (*func)(void*), void* arg, int stackSize, const char* name) {
	return ::startThread(func, arg, stackSize, name);
}

Future<Reference<IConnection>> Net2::connect(NetworkAddress toAddr, tcp::socket* existingSocket) {
	if (toAddr.isTLS()) {
		initTLS(ETLSInitState::CONNECT);
		return SSLConnection::connect(&this->reactor.ios, this->sslContextVar.get(), toAddr, existingSocket);
	}

	if (tlsConfig.getDisablePlainTextConnection()) {
		TraceEvent(SevError, "PlainTextConnectionDisabled").detail("toAddr", toAddr);
		throw connection_failed();
	}

	return Connection::connect(&this->reactor.ios, toAddr);
}

Future<Reference<IConnection>> Net2::connectExternal(NetworkAddress toAddr) {
	return connect(toAddr);
}

Future<Reference<IUDPSocket>> Net2::createUDPSocket(NetworkAddress toAddr) {
	return UDPSocket::connect(&reactor.ios, toAddr, toAddr.ip.isV6());
}

Future<Reference<IUDPSocket>> Net2::createUDPSocket(bool isV6) {
	return UDPSocket::connect(&reactor.ios, Optional<NetworkAddress>(), isV6);
}

															#line 5791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
// This generated class is to be used only via resolveTCPEndpoint_impl()
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class ResolveTCPEndpoint_implActor>
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ResolveTCPEndpoint_implActorState {
															#line 5797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ResolveTCPEndpoint_implActorState(Net2* const& self,std::string const& host,std::string const& service) 
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : self(self),
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   host(host),
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   service(service),
															#line 1870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		   tcpResolver(self->reactor.ios)
															#line 5810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this));

	}
	~ResolveTCPEndpoint_implActorState() 
	{
		fdb_probe_actor_destroy("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			Promise<std::vector<NetworkAddress>> promise;
															#line 1872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			result = promise.getFuture();
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			tcpResolver.async_resolve( host, service, [promise](const boost::system::error_code& ec, tcp::resolver::iterator iter) { if (ec) { promise.sendError(lookup_failed()); return; } std::vector<NetworkAddress> addrs; tcp::resolver::iterator end; while (iter != end) { auto endpoint = iter->endpoint(); auto addr = endpoint.address(); if (addr.is_v6()) { if (!addr.is_loopback()) { addrs.emplace_back(IPAddress(addr.to_v6().to_bytes()), endpoint.port()); } } else { addrs.emplace_back(addr.to_v4().to_ulong(), endpoint.port()); } ++iter; } if (addrs.empty()) { promise.sendError(lookup_failed()); } else { promise.send(addrs); } });
															#line 5829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			try {
															#line 1906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				StrictFuture<Void> __when_expr_0 = ready(result);
															#line 1906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (static_cast<ResolveTCPEndpoint_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ResolveTCPEndpoint_implActor*>(this)->actor_wait_state = 1;
															#line 1906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >*>(static_cast<ResolveTCPEndpoint_implActor*>(this)));
															#line 5840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolveTCPEndpoint_implActorState();
		static_cast<ResolveTCPEndpoint_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		tcpResolver.cancel();
															#line 1914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		std::vector<NetworkAddress> ret = result.get();
															#line 1915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		self->dnsCache.add(host, service, ret);
															#line 1917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<ResolveTCPEndpoint_implActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(ret); this->~ResolveTCPEndpoint_implActorState(); static_cast<ResolveTCPEndpoint_implActor*>(this)->destroy(); return 0; }
															#line 5875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<ResolveTCPEndpoint_implActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(ret);
		this->~ResolveTCPEndpoint_implActorState();
		static_cast<ResolveTCPEndpoint_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (e.code() == error_code_lookup_failed)
															#line 5888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			{
															#line 1909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				self->dnsCache.remove(host, service);
															#line 5892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			}
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 5896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolveTCPEndpoint_implActor*>(this)->actor_wait_state > 0) static_cast<ResolveTCPEndpoint_implActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveTCPEndpoint_implActor*>(this)->ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveTCPEndpoint_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveTCPEndpoint_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveTCPEndpoint_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Net2* self;
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::string host;
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::string service;
															#line 1870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	tcp::resolver tcpResolver;
															#line 1872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Future<std::vector<NetworkAddress>> result;
															#line 6016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via resolveTCPEndpoint_impl()
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class ResolveTCPEndpoint_implActor final : public Actor<std::vector<NetworkAddress>>, public ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >, public FastAllocated<ResolveTCPEndpoint_implActor>, public ResolveTCPEndpoint_implActorState<ResolveTCPEndpoint_implActor> {
															#line 6021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<ResolveTCPEndpoint_implActor>::operator new;
	using FastAllocated<ResolveTCPEndpoint_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16740558687066822912UL, 13772017461584468480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<NetworkAddress>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >;
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	ResolveTCPEndpoint_implActor(Net2* const& self,std::string const& host,std::string const& service) 
															#line 6038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<std::vector<NetworkAddress>>(),
		   ResolveTCPEndpoint_implActorState<ResolveTCPEndpoint_implActor>(self, host, service),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17241569023256932352UL, 1390472597874904064UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveTCPEndpoint_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolveTCPEndpoint_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolveTCPEndpoint_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolveTCPEndpoint_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
[[nodiscard]] static Future<std::vector<NetworkAddress>> resolveTCPEndpoint_impl( Net2* const& self, std::string const& host, std::string const& service ) {
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<std::vector<NetworkAddress>>(new ResolveTCPEndpoint_implActor(self, host, service));
															#line 6070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}

#line 1919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

Future<std::vector<NetworkAddress>> Net2::resolveTCPEndpoint(const std::string& host, const std::string& service) {
	return resolveTCPEndpoint_impl(this, host, service);
}

Future<std::vector<NetworkAddress>> Net2::resolveTCPEndpointWithDNSCache(const std::string& host,
                                                                         const std::string& service) {
	if (FLOW_KNOBS->ENABLE_COORDINATOR_DNS_CACHE) {
		Optional<std::vector<NetworkAddress>> cache = dnsCache.find(host, service);
		if (cache.present()) {
			return cache.get();
		}
	}
	return resolveTCPEndpoint_impl(this, host, service);
}

std::vector<NetworkAddress> Net2::resolveTCPEndpointBlocking(const std::string& host, const std::string& service) {
	tcp::resolver tcpResolver(reactor.ios);
	try {
		auto iter = tcpResolver.resolve(host, service);
		decltype(iter) end;
		std::vector<NetworkAddress> addrs;
		while (iter != end) {
			auto endpoint = iter->endpoint();
			auto addr = endpoint.address();
			if (addr.is_v6()) {
				addrs.emplace_back(IPAddress(addr.to_v6().to_bytes()), endpoint.port());
			} else {
				addrs.emplace_back(addr.to_v4().to_ulong(), endpoint.port());
			}
			++iter;
		}
		if (addrs.empty()) {
			throw lookup_failed();
		}
		return addrs;
	} catch (...) {
		dnsCache.remove(host, service);
		throw lookup_failed();
	}
}

std::vector<NetworkAddress> Net2::resolveTCPEndpointBlockingWithDNSCache(const std::string& host,
                                                                         const std::string& service) {
	if (FLOW_KNOBS->ENABLE_COORDINATOR_DNS_CACHE) {
		Optional<std::vector<NetworkAddress>> cache = dnsCache.find(host, service);
		if (cache.present()) {
			return cache.get();
		}
	}
	return resolveTCPEndpointBlocking(host, service);
}

bool Net2::isAddressOnThisHost(NetworkAddress const& addr) const {
	auto it = addressOnHostCache.find(addr.ip);
	if (it != addressOnHostCache.end())
		return it->second;

	if (addressOnHostCache.size() > 50000)
		addressOnHostCache.clear(); // Bound cache memory; should not really happen

	try {
		boost::asio::io_service ioService;
		boost::asio::ip::udp::socket socket(ioService);
		boost::asio::ip::udp::endpoint endpoint(tcpAddress(addr.ip), 1);
		socket.connect(endpoint);
		bool local = addr.ip.isV6() ? socket.local_endpoint().address().to_v6().to_bytes() == addr.ip.toV6()
		                            : socket.local_endpoint().address().to_v4().to_ulong() == addr.ip.toV4();
		socket.close();
		if (local)
			TraceEvent(SevInfo, "AddressIsOnHost").detail("Address", addr);
		return addressOnHostCache[addr.ip] = local;
	} catch (boost::system::system_error e) {
		TraceEvent(SevWarnAlways, "IsAddressOnHostError")
		    .detail("Address", addr)
		    .detail("ErrDesc", e.what())
		    .detail("ErrCode", e.code().value());
		return addressOnHostCache[addr.ip] = false;
	}
}

Reference<IListener> Net2::listen(NetworkAddress localAddr) {
	try {
		if (localAddr.isTLS()) {
			initTLS(ETLSInitState::LISTEN);
			return Reference<IListener>(new SSLListener(reactor.ios, &this->sslContextVar, localAddr));
		}
		return Reference<IListener>(new Listener(reactor.ios, localAddr));
	} catch (boost::system::system_error const& e) {
		Error x;
		if (e.code().value() == EADDRINUSE)
			x = address_in_use();
		else if (e.code().value() == EADDRNOTAVAIL)
			x = invalid_local_address();
		else
			x = bind_failed();
		TraceEvent("Net2ListenError").error(x).detail("Message", e.what());
		throw x;
	} catch (std::exception const& e) {
		Error x = unknown_error();
		TraceEvent("Net2ListenError").error(x).detail("Message", e.what());
		throw x;
	} catch (Error& e) {
		TraceEvent("Net2ListenError").error(e);
		throw e;
	} catch (...) {
		Error x = unknown_error();
		TraceEvent("Net2ListenError").error(x);
		throw x;
	}
}

void Net2::getDiskBytes(std::string const& directory, int64_t& free, int64_t& total) {
	return ::getDiskBytes(directory, free, total);
}

#ifdef __linux__
#include <sys/prctl.h>
#include <pthread.h>
#include <sched.h>
#endif

ASIOReactor::ASIOReactor(Net2* net) : do_not_stop(ios), network(net), firstTimer(ios) {
#ifdef __linux__
	// Reactor flags are used only for experimentation, and are platform-specific
	if (FLOW_KNOBS->REACTOR_FLAGS & 1) {
		prctl(PR_SET_TIMERSLACK, 1, 0, 0, 0);
		printf("Set timerslack to 1ns\n");
	}

	if (FLOW_KNOBS->REACTOR_FLAGS & 2) {
		int ret;
		pthread_t this_thread = pthread_self();
		struct sched_param params;
		params.sched_priority = sched_get_priority_max(SCHED_FIFO);
		ret = pthread_setschedparam(this_thread, SCHED_FIFO, &params);
		if (ret != 0)
			printf("Error setting priority (%d %d)\n", ret, errno);
		else
			printf("Set scheduler mode to SCHED_FIFO\n");
	}
#endif
}

void ASIOReactor::sleep(double sleepTime) {
	if (sleepTime > FLOW_KNOBS->BUSY_WAIT_THRESHOLD) {
		if (FLOW_KNOBS->REACTOR_FLAGS & 4) {
#ifdef __linux
			timespec tv;
			tv.tv_sec = 0;
			tv.tv_nsec = 20000;
			nanosleep(&tv, nullptr);
#endif
		} else {
			sleepTime -= FLOW_KNOBS->BUSY_WAIT_THRESHOLD;
			if (sleepTime < 4e12) {
				this->firstTimer.expires_from_now(boost::posix_time::microseconds(int64_t(sleepTime * 1e6)));
				this->firstTimer.async_wait(&nullWaitHandler);
			}
			setProfilingEnabled(0); // The following line generates false positives for slow task profiling
			ios.run_one();
			setProfilingEnabled(1);
			this->firstTimer.cancel();
		}
		++network->countASIOEvents;
	} else if (sleepTime > 0) {
		if (!(FLOW_KNOBS->REACTOR_FLAGS & 8))
			threadYield();
	}
}

void ASIOReactor::react() {
	while (ios.poll_one())
		++network->countASIOEvents; // Make this a task?
}

void ASIOReactor::wake() {
	ios.post(nullCompletionHandler);
}

} // namespace N2

SendBufferIterator::SendBufferIterator(SendBuffer const* p, int limit) : p(p), limit(limit) {
	ASSERT(limit > 0);
}

void SendBufferIterator::operator++() {
	limit -= p->bytes_written - p->bytes_sent;
	if (limit > 0)
		p = p->next;
	else
		p = nullptr;
}

boost::asio::const_buffer SendBufferIterator::operator*() const {
	return boost::asio::const_buffer(p->data() + p->bytes_sent, std::min(limit, p->bytes_written - p->bytes_sent));
}

INetwork* newNet2(const TLSConfig& tlsConfig, bool useThreadPool, bool useMetrics) {
	try {
		N2::g_net2 = new N2::Net2(tlsConfig, useThreadPool, useMetrics);
	} catch (boost::system::system_error e) {
		TraceEvent("Net2InitError").detail("Message", e.what());
		throw unknown_error();
	} catch (std::exception const& e) {
		TraceEvent("Net2InitError").detail("Message", e.what());
		throw unknown_error();
	}

	return N2::g_net2;
}

struct TestGVR {
	Standalone<StringRef> key;
	int64_t version;
	Optional<std::pair<UID, UID>> debugID;
	Promise<Optional<Standalone<StringRef>>> reply;

	TestGVR() {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, key, version, debugID, reply);
	}
};

template <class F>
THREAD_HANDLE startThreadF(F&& func) {
	struct Thing {
		F f;
		Thing(F&& f) : f(std::move(f)) {}
		THREAD_FUNC start(void* p) {
			Thing* self = (Thing*)p;
			self->f();
			delete self;
			THREAD_RETURN;
		}
	};
	Thing* t = new Thing(std::move(func));
	return g_network->startThread(Thing::start, t);
}

															#line 6316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2161()
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class FlowTestCase2161Actor>
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2161ActorState {
															#line 6323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2161ActorState(UnitTestParameters const& params) 
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : params(params)
															#line 6330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2161", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2161ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2161", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ThreadSafeQueue<int> tq;
															#line 2163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(!tq.pop().present());
															#line 2164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.canSleep());
															#line 2166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.push(1) == true);
															#line 2167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(!tq.canSleep());
															#line 2168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(!tq.canSleep());
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.push(2) == false);
															#line 2170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.push(3) == false);
															#line 2172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.pop().get() == 1);
															#line 2173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.pop().get() == 2);
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.push(4) == false);
															#line 2175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.pop().get() == 3);
															#line 2176 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.pop().get() == 4);
															#line 2177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(!tq.pop().present());
															#line 2178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ASSERT(tq.canSleep());
															#line 2179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (!static_cast<FlowTestCase2161Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2161ActorState(); static_cast<FlowTestCase2161Actor*>(this)->destroy(); return 0; }
															#line 6375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			new (&static_cast<FlowTestCase2161Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2161ActorState();
			static_cast<FlowTestCase2161Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2161ActorState();
		static_cast<FlowTestCase2161Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	UnitTestParameters params;
															#line 6399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2161()
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2161Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2161Actor>, public FlowTestCase2161ActorState<FlowTestCase2161Actor> {
															#line 6404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2161Actor>::operator new;
	using FastAllocated<FlowTestCase2161Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15930129998857346560UL, 10882994601232036608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2161Actor(UnitTestParameters const& params) 
															#line 6420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2161ActorState<FlowTestCase2161Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2161", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15518088519112975104UL, 17609684797972893184UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2161Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2161");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2161", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
static Future<Void> flowTestCase2161( UnitTestParameters const& params ) {
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new FlowTestCase2161Actor(params));
															#line 6452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2161, "flow/Net2/ThreadSafeQueue/Interface")

#line 2181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

// A helper struct used by queueing tests which use multiple threads.
struct QueueTestThreadState {
	QueueTestThreadState(int threadId, int toProduce) : threadId(threadId), toProduce(toProduce) {}
	int threadId;
	THREAD_HANDLE handle;
	int toProduce;
	int produced = 0;
	Promise<Void> doneProducing;
	int consumed = 0;

	static int valueToThreadId(int value) { return value >> 20; }
	int elementValue(int index) { return index + (threadId << 20); }
	int nextProduced() { return elementValue(produced++); }
	int nextConsumed() { return elementValue(consumed++); }
	void checkDone() {
		ASSERT_EQ(produced, toProduce);
		ASSERT_EQ(consumed, produced);
	}
};

															#line 6478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2202()
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class FlowTestCase2202Actor>
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2202ActorState {
															#line 6485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2202ActorState(UnitTestParameters const& params) 
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : params(params)
															#line 6492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2202", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2202ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2202", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			noUnseed = true;
															#line 2207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			ThreadSafeQueue<int> queue;
															#line 2208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread = { QueueTestThreadState(0, 1000000), QueueTestThreadState(1, 100000), QueueTestThreadState(2, 1000000) };
															#line 2211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			doneProducing = std::vector<Future<Void>>();
															#line 2213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			int total = 0;
															#line 2214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			for(int t = 0;t < perThread.size();++t) {
															#line 2215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto& s = perThread[t];
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				doneProducing.push_back(s.doneProducing.getFuture());
															#line 2217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				total += s.toProduce;
															#line 2218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				s.handle = startThreadF([&queue, &s]() { printf("Thread%d\n", s.threadId); int nextYield = 0; while (s.produced < s.toProduce) { queue.push(s.nextProduced()); if (nextYield-- == 0) { std::this_thread::yield(); nextYield = nondeterministicRandom()->randomInt(0, 100); } } printf("T%dDone\n", s.threadId); s.doneProducing.send(Void()); });
															#line 6525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			}
															#line 2232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			int consumed = 0;
															#line 2233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			for(;consumed < total;) {
															#line 2234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				Optional<int> element = queue.pop();
															#line 2235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if (element.present())
															#line 6535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 2236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					int v = element.get();
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					auto& s = perThread[QueueTestThreadState::valueToThreadId(v)];
															#line 2238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					++consumed;
															#line 2239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					ASSERT(v == s.nextConsumed());
															#line 6545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
				else
				{
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					std::this_thread::yield();
															#line 6551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
															#line 2243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				if ((consumed & 3) == 0)
															#line 6555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				{
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
					queue.canSleep();
															#line 6559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
				}
			}
															#line 2247 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(doneProducing);
															#line 2247 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state = 1;
															#line 2247 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2202Actor, 0, Void >*>(static_cast<FlowTestCase2202Actor*>(this)));
															#line 6571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2202ActorState();
		static_cast<FlowTestCase2202Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> signal;
															#line 2251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doneConsuming = signal.getFuture();
															#line 2252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		g_network->onMainThread(std::move(signal), TaskPriority::DefaultOnMainThread);
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = doneConsuming;
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6602 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state = 2;
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2202Actor, 1, Void >*>(static_cast<FlowTestCase2202Actor*>(this)));
															#line 6607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> signal;
															#line 2251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doneConsuming = signal.getFuture();
															#line 2252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		g_network->onMainThread(std::move(signal), TaskPriority::DefaultOnMainThread);
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = doneConsuming;
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state = 2;
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2202Actor, 1, Void >*>(static_cast<FlowTestCase2202Actor*>(this)));
															#line 6629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2202Actor*>(this)->ActorCallback< FlowTestCase2202Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2202Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2202Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2202Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 2255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		for(int t = 0;t < perThread.size();++t) {
															#line 2256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			waitThread(perThread[t].handle);
															#line 2257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread[t].checkDone();
															#line 6717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
															#line 2259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<FlowTestCase2202Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2202ActorState(); static_cast<FlowTestCase2202Actor*>(this)->destroy(); return 0; }
															#line 6721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<FlowTestCase2202Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2202ActorState();
		static_cast<FlowTestCase2202Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 2255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		for(int t = 0;t < perThread.size();++t) {
															#line 2256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			waitThread(perThread[t].handle);
															#line 2257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread[t].checkDone();
															#line 6737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
															#line 2259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<FlowTestCase2202Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2202ActorState(); static_cast<FlowTestCase2202Actor*>(this)->destroy(); return 0; }
															#line 6741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<FlowTestCase2202Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2202ActorState();
		static_cast<FlowTestCase2202Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2202Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2202Actor*>(this)->ActorCallback< FlowTestCase2202Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2202Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2202Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2202Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	UnitTestParameters params;
															#line 2208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::vector<QueueTestThreadState> perThread;
															#line 2211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::vector<Future<Void>> doneProducing;
															#line 2251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Future<Void> doneConsuming;
															#line 6832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2202()
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2202Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2202Actor, 0, Void >, public ActorCallback< FlowTestCase2202Actor, 1, Void >, public FastAllocated<FlowTestCase2202Actor>, public FlowTestCase2202ActorState<FlowTestCase2202Actor> {
															#line 6837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2202Actor>::operator new;
	using FastAllocated<FlowTestCase2202Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3941875254796113408UL, 8639239959075068672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2202Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2202Actor, 1, Void >;
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2202Actor(UnitTestParameters const& params) 
															#line 6855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2202ActorState<FlowTestCase2202Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2202", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2871377471312605440UL, 17417846351857059584UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2202Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2202");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2202", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2202Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2202Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
static Future<Void> flowTestCase2202( UnitTestParameters const& params ) {
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new FlowTestCase2202Actor(params));
															#line 6889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2202, "flow/Net2/ThreadSafeQueue/Threaded")

#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

															#line 6895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2262()
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
template <class FlowTestCase2262Actor>
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2262ActorState {
															#line 6902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2262ActorState(UnitTestParameters const& params) 
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		 : params(params)
															#line 6909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2262", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2262ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2262", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			noUnseed = true;
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread = { QueueTestThreadState(0, 1000000), QueueTestThreadState(1, 100000), QueueTestThreadState(2, 1000000) };
															#line 2269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			doneProducing = std::vector<Future<Void>>();
															#line 2270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			for(int t = 0;t < perThread.size();++t) {
															#line 2271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				auto& s = perThread[t];
															#line 2272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				doneProducing.push_back(s.doneProducing.getFuture());
															#line 2273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
				s.handle = startThreadF([&s]() { int nextYield = 0; while (s.produced < s.toProduce) { if (nextYield-- == 0) { std::this_thread::yield(); nextYield = nondeterministicRandom()->randomInt(0, 100); } int v = s.nextProduced(); onMainThreadVoid([&s, v]() { ASSERT_EQ(v, s.nextConsumed()); }); } s.doneProducing.send(Void()); });
															#line 6936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			}
															#line 2286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(doneProducing);
															#line 2286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			if (static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state = 1;
															#line 2286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2262Actor, 0, Void >*>(static_cast<FlowTestCase2262Actor*>(this)));
															#line 6947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2262ActorState();
		static_cast<FlowTestCase2262Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> signal;
															#line 2290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doneConsuming = signal.getFuture();
															#line 2291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		g_network->onMainThread(std::move(signal), TaskPriority::DefaultOnMainThread);
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = doneConsuming;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state = 2;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2262Actor, 1, Void >*>(static_cast<FlowTestCase2262Actor*>(this)));
															#line 6983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		Promise<Void> signal;
															#line 2290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		doneConsuming = signal.getFuture();
															#line 2291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		g_network->onMainThread(std::move(signal), TaskPriority::DefaultOnMainThread);
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		StrictFuture<Void> __when_expr_1 = doneConsuming;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7000 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state = 2;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2262Actor, 1, Void >*>(static_cast<FlowTestCase2262Actor*>(this)));
															#line 7005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2262Actor*>(this)->ActorCallback< FlowTestCase2262Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2262Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2262Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2262Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		for(int t = 0;t < perThread.size();++t) {
															#line 2295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			waitThread(perThread[t].handle);
															#line 2296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread[t].checkDone();
															#line 7093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
															#line 2298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<FlowTestCase2262Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2262ActorState(); static_cast<FlowTestCase2262Actor*>(this)->destroy(); return 0; }
															#line 7097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<FlowTestCase2262Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2262ActorState();
		static_cast<FlowTestCase2262Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		for(int t = 0;t < perThread.size();++t) {
															#line 2295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			waitThread(perThread[t].handle);
															#line 2296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
			perThread[t].checkDone();
															#line 7113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		}
															#line 2298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
		if (!static_cast<FlowTestCase2262Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2262ActorState(); static_cast<FlowTestCase2262Actor*>(this)->destroy(); return 0; }
															#line 7117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		new (&static_cast<FlowTestCase2262Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2262ActorState();
		static_cast<FlowTestCase2262Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2262Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2262Actor*>(this)->ActorCallback< FlowTestCase2262Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2262Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2262Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2262Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	UnitTestParameters params;
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::vector<QueueTestThreadState> perThread;
															#line 2269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	std::vector<Future<Void>> doneProducing;
															#line 2290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	Future<Void> doneConsuming;
															#line 7208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2262()
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
class FlowTestCase2262Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2262Actor, 0, Void >, public ActorCallback< FlowTestCase2262Actor, 1, Void >, public FastAllocated<FlowTestCase2262Actor>, public FlowTestCase2262ActorState<FlowTestCase2262Actor> {
															#line 7213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2262Actor>::operator new;
	using FastAllocated<FlowTestCase2262Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6727017279375666432UL, 12871913940834238720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2262Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2262Actor, 1, Void >;
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	FlowTestCase2262Actor(UnitTestParameters const& params) 
															#line 7231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2262ActorState<FlowTestCase2262Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2262", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4706423270079984384UL, 1923059049817263104UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2262Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2262");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2262", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2262Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2262Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
static Future<Void> flowTestCase2262( UnitTestParameters const& params ) {
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"
	return Future<Void>(new FlowTestCase2262Actor(params));
															#line 7265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/flow/Net2.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2262, "noSim/flow/Net2/onMainThreadFIFO")

#line 2300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/flow/Net2.actor.cpp"

void net2_test() {
	/*
	g_network = newNet2();  // for promise serialization below

	Endpoint destination;

	printf("  Used: %lld\n", FastAllocator<4096>::getTotalMemory());

	char junk[100];

	double before = timer();

	std::vector<TestGVR> reqs;
	reqs.reserve( 10000 );

	int totalBytes = 0;
	for(int j=0; j<1000; j++) {
	    UnsentPacketQueue unsent;
	    ReliablePacketList reliable;

	    reqs.resize(10000);
	    for(int i=0; i<10000; i++) {
	        TestGVR &req = reqs[i];
	        req.key = "Foobar"_sr;

	        SerializeSource<TestGVR> what(req);

	        SendBuffer* pb = unsent.getWriteBuffer();
	        ReliablePacket* rp = new ReliablePacket;  // 0

	        PacketWriter wr(pb,rp,AssumeVersion(g_network->protocolVersion()));
	        //BinaryWriter wr;
	        SplitBuffer packetLen;
	        uint32_t len = 0;
	        wr.writeAhead(sizeof(len), &packetLen);
	        wr << destination.token;
	        //req.reply.getEndpoint();
	        what.serializePacketWriter(wr);
	        //wr.serializeBytes(junk, 43);

	        unsent.setWriteBuffer(wr.finish());
	        len = wr.size() - sizeof(len);
	        packetLen.write(&len, sizeof(len));

	        //totalBytes += wr.getLength();
	        totalBytes += wr.size();

	        if (rp) reliable.insert(rp);
	    }
	    reqs.clear();
	    unsent.discardAll();
	    reliable.discardAll();
	}

	printf("SimSend x 1Kx10K: %0.2f sec\n", timer()-before);
	printf("  Bytes: %d\n", totalBytes);
	printf("  Used: %lld\n", FastAllocator<4096>::getTotalMemory());
	*/
};
