#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
/*
 * BlobGranuleCorrectnessWorkload.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cmath>
#include <map>
#include <string>
#include <utility>
#include <vector>

#include "fmt/format.h"
#include "fdbclient/BlobGranuleReader.actor.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/TenantManagement.actor.h"
#include "fdbclient/Tuple.h"
#include "fdbserver/BlobGranuleServerCommon.actor.h"
#include "fdbserver/BlobGranuleValidation.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "flow/Arena.h"
#include "flow/IRandom.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"

#include "flow/actorcompiler.h" // This must be the last #include.

#define BGW_DEBUG true
#define BGW_TUPLE_KEY_SIZE 2

struct WriteData {
	Version writeVersion;
	Version clearVersion;
	int32_t val;
	int16_t valLength;

	// start as MAX_VERSION while uncommitted/uncleared so that they're ignored by concurrent readers
	explicit WriteData(int32_t val, int16_t valLength)
	  : writeVersion(MAX_VERSION), clearVersion(MAX_VERSION), val(val), valLength(valLength) {}

	// loading existing write data from the database
	explicit WriteData(Version writeVersion, Version clearVersion, int32_t val, int16_t valLength)
	  : writeVersion(writeVersion), clearVersion(clearVersion), val(val), valLength(valLength) {}
};

struct KeyData {
	int nextClearIdx;
	std::vector<WriteData> writes;
};

static std::vector<int> targetValSizes = { 40, 100, 500 };

struct ThreadData : ReferenceCounted<ThreadData>, NonCopyable {
	// directory info
	int32_t directoryID;
	KeyRange directoryRange;
	TenantName tenantName;
	Reference<Tenant> tenant;
	TenantMapEntry tenantEntry;
	Reference<BlobConnectionProvider> bstore;

	// key + value gen data
	// in vector for efficient random selection
	std::vector<uint32_t> usedKeys;
	// by key for tracking data
	std::map<uint32_t, KeyData> keyData;

	std::deque<Version> writeVersions;

	// randomized parameters that can be different per directory
	int targetByteRate;
	bool nextKeySequential;
	int16_t targetValLength;
	double reuseKeyProb;
	int targetIDsPerKey;
	uint32_t nextSeqKey = 0;

	// communication between workers
	Promise<Void> firstWriteSuccessful;
	Version minSuccessfulReadVersion = MAX_VERSION;

	Future<Void> summaryClient;
	Future<Void> forceFlushingClient;
	Promise<Void> triggerSummaryComplete;

	// stats
	int64_t errors = 0;
	int64_t mismatches = 0;
	int64_t reads = 0;
	int64_t timeTravelReads = 0;
	int64_t timeTravelTooOld = 0;
	int64_t rowsRead = 0;
	int64_t bytesRead = 0;
	int64_t rowsWritten = 0;
	int64_t bytesWritten = 0;

	ThreadData(uint32_t directoryID, int64_t targetByteRate)
	  : directoryID(directoryID), targetByteRate(targetByteRate) {
		tenantName = StringRef(std::to_string(directoryID));

		targetByteRate *= (0.5 + deterministicRandom()->random01());

		targetValLength = deterministicRandom()->randomChoice(targetValSizes);
		targetValLength *= (0.5 + deterministicRandom()->random01());

		nextKeySequential = deterministicRandom()->random01() < 0.5;
		reuseKeyProb = 0.1 + (deterministicRandom()->random01() * 0.8);
		targetIDsPerKey = 1 + deterministicRandom()->randomInt(10, 100);

		if (BGW_DEBUG) {
			fmt::print("Directory {0} initialized with the following parameters:\n", directoryID);
			fmt::print("  targetByteRate={0}\n", targetByteRate);
			fmt::print("  targetValLength={0}\n", targetValLength);
			fmt::print("  nextKeySequential={0}\n", nextKeySequential);
			fmt::print("  reuseKeyProb={0}\n", reuseKeyProb);
			fmt::print("  targetIDsPerKey={0}\n", targetIDsPerKey);
		}
	}

	Future<Void> openTenant(Database const& cx) {
		tenant = makeReference<Tenant>(cx, tenantName);
		return tenant->ready();
	}

	// randomly reopen tenant and do not wait for it to be ready, to test races
	void maybeReopenTenant(Database const& cx) {
		if (BUGGIFY_WITH_PROB(0.01)) {
			openTenant(cx);
		}
	}

	// TODO could make keys variable length?
	Key getKey(uint32_t key, uint32_t id) {
		std::stringstream ss;
		ss << std::setw(32) << std::setfill('0') << id;
		if (g_network->isSimulated() && g_simulator->dataAtRestPlaintextMarker.present()) {
			ss << g_simulator->dataAtRestPlaintextMarker.get();
		}

		Standalone<StringRef> str(ss.str());
		Tuple::UserTypeStr udt(0x41, str);
		return Tuple::makeTuple((int64_t)key, udt).pack();
	}

	void validateGranuleBoundary(Key k, Key e, Key lastKey) {
		if (k == allKeys.begin || k == allKeys.end) {
			return;
		}

		// Fully formed tuples are inserted. The expectation is boundaries should be a
		// sub-tuple of the inserted key.
		Tuple t = Tuple::unpackUserType(k, true);
		if (SERVER_KNOBS->BG_KEY_TUPLE_TRUNCATE_OFFSET) {
			Tuple t2;
			try {
				t2 = Tuple::unpackUserType(lastKey);
			} catch (Error& e) {
				// Ignore being unable to parse lastKey as it may be a dummy key.
			}

			if (t2.size() > 0 && t.getInt(0) != t2.getInt(0)) {
				if (t.size() > BGW_TUPLE_KEY_SIZE - SERVER_KNOBS->BG_KEY_TUPLE_TRUNCATE_OFFSET) {
					fmt::print("Tenant: {0}, K={1}, E={2}, LK={3}. {4} != {5}\n",
					           tenantEntry.prefix.printable(),
					           k.printable(),
					           e.printable(),
					           lastKey.printable(),
					           t.getInt(0),
					           t2.getInt(0));
				}
				ASSERT(t.size() <= BGW_TUPLE_KEY_SIZE - SERVER_KNOBS->BG_KEY_TUPLE_TRUNCATE_OFFSET);
			}
		}
	}
};

// For debugging mismatches on what data should be and why
// set mismatch to true, dir id and key id to the directory and key id that are wrong, and rv to read version that read
// the wrong value
#define DEBUG_MISMATCH false
#define DEBUG_DIR_ID 0
#define DEBUG_KEY_ID 0
#define DEBUG_RV invalidVersion

#define DEBUG_KEY_OP(dirId, keyId) BGW_DEBUG&& DEBUG_MISMATCH&& dirId == DEBUG_DIR_ID&& DEBUG_KEY_ID == keyId
#define DEBUG_READ_OP(dirId, rv) BGW_DEBUG&& DEBUG_MISMATCH&& dirId == DEBUG_DIR_ID&& rv == DEBUG_RV

const std::string BG_ENCRYPTION_AT_REST_MARKER_STRING = "Expecto..Patronum...BlobGranule";
/*
 * This is a stand-alone workload designed to validate blob granule correctness.
 * By enabling distinct ranges and writing to those parts of the key space, we can control what parts of the key space
 * are written to blob, and can validate that the granule data is correct at any desired version.
 */
struct BlobGranuleCorrectnessWorkload : TestWorkload {
	static constexpr auto NAME = "BlobGranuleCorrectnessWorkload";
	bool doSetup;
	double testDuration;

	// parameters global across all clients
	int64_t targetByteRate;
	bool doMergeCheckAtEnd;
	bool doForceFlushing;

	std::vector<Reference<ThreadData>> directories;
	std::vector<Future<Void>> clients;
	DatabaseConfiguration config;

	BlobGranuleCorrectnessWorkload(WorkloadContext const& wcx) : TestWorkload(wcx) {
		doSetup = !clientId; // only do this on the "first" client
		testDuration = getOption(options, "testDuration"_sr, 120.0);

		// randomize global test settings based on shared parameter to get similar workload across tests, but then vary
		// different parameters within those constraints
		int64_t randomness = sharedRandomNumber;

		doMergeCheckAtEnd = randomness % 10 == 0;
		randomness /= 10;

		if (g_network->isSimulated() && g_simulator->willRestart) {
			doMergeCheckAtEnd = false;
		}

		// randomize between low and high directory count
		int64_t targetDirectories = 1 + (randomness % 8);
		randomness /= 8;

		doForceFlushing = (randomness % 4);
		randomness /= 4;

		int64_t targetMyDirectories =
		    (targetDirectories / clientCount) + ((targetDirectories % clientCount > clientId) ? 1 : 0);

		if (g_network->isSimulated() && g_simulator->restarted) {
			// load directories later
			targetMyDirectories = 0;
		}

		if (targetMyDirectories > 0) {
			int myDirectories = 1;
			if (targetMyDirectories > 1) {
				myDirectories = deterministicRandom()->randomInt(1, 2 * targetMyDirectories + 1);
			}

			// anywhere from 2 delta files per second to 1 delta file every 2 seconds, spread across all directories
			int denom = std::min(clientCount, (int)targetDirectories);
			targetByteRate = 2 * SERVER_KNOBS->BG_DELTA_FILE_TARGET_BYTES / (1 + (randomness % 4)) / denom;
			randomness /= 4;

			// either do equal across all of my directories, or skewed
			bool skewed = myDirectories > 1 && deterministicRandom()->random01() < 0.4;
			int skewMultiplier;
			if (skewed) {
				// first directory has 1/2, second has 1/4, third has 1/8, etc...
				skewMultiplier = 2;
				targetByteRate /= 2;
			} else {
				skewMultiplier = 1;
				targetByteRate /= myDirectories;
			}
			for (int i = 0; i < myDirectories; i++) {
				// set up directory with its own randomness
				uint32_t dirId = i * clientCount + clientId;
				if (BGW_DEBUG) {
					fmt::print("Client {0}/{1} creating directory {2}\n", clientId, clientCount, dirId);
				}
				directories.push_back(makeReference<ThreadData>(dirId, targetByteRate));
				targetByteRate /= skewMultiplier;
			}
		}
	}

																#line 292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via setUpTenant()
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class SetUpTenantActor>
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class SetUpTenantActorState {
															#line 298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	SetUpTenantActorState(Database const& cx,TenantName const& tenantName) 
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : cx(cx),
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tenantName(tenantName)
															#line 307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("setUpTenant", reinterpret_cast<unsigned long>(this));

	}
	~SetUpTenantActorState() 
	{
		fdb_probe_actor_destroy("setUpTenant", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (BGW_DEBUG)
															#line 322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				fmt::print("Setting up blob granule range for tenant {0}\n", tenantName.printable());
															#line 326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Optional<TenantMapEntry>> __when_expr_0 = TenantAPI::createTenant(cx.getReference(), tenantName);
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<SetUpTenantActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetUpTenantActor*>(this)->actor_wait_state = 1;
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >*>(static_cast<SetUpTenantActor*>(this)));
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetUpTenantActorState();
		static_cast<SetUpTenantActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<TenantMapEntry> const& entry,int loopDepth) 
	{
															#line 296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(entry.present());
															#line 298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Set up blob granule range for tenant {0}: {1}\n", tenantName.printable(), entry.get().prefix.printable());
															#line 366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<SetUpTenantActor*>(this)->SAV<TenantMapEntry>::futures) { (void)(entry.get()); this->~SetUpTenantActorState(); static_cast<SetUpTenantActor*>(this)->destroy(); return 0; }
															#line 370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<SetUpTenantActor*>(this)->SAV< TenantMapEntry >::value()) TenantMapEntry(entry.get());
		this->~SetUpTenantActorState();
		static_cast<SetUpTenantActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<TenantMapEntry> && entry,int loopDepth) 
	{
															#line 296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(entry.present());
															#line 298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Set up blob granule range for tenant {0}: {1}\n", tenantName.printable(), entry.get().prefix.printable());
															#line 388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<SetUpTenantActor*>(this)->SAV<TenantMapEntry>::futures) { (void)(entry.get()); this->~SetUpTenantActorState(); static_cast<SetUpTenantActor*>(this)->destroy(); return 0; }
															#line 392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<SetUpTenantActor*>(this)->SAV< TenantMapEntry >::value()) TenantMapEntry(entry.get());
		this->~SetUpTenantActorState();
		static_cast<SetUpTenantActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<TenantMapEntry> const& entry,int loopDepth) 
	{
		loopDepth = a_body1cont1(entry, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<TenantMapEntry> && entry,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(entry), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetUpTenantActor*>(this)->actor_wait_state > 0) static_cast<SetUpTenantActor*>(this)->actor_wait_state = 0;
		static_cast<SetUpTenantActor*>(this)->ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >::remove();

	}
	void a_callback_fire(ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> const& value) 
	{
		fdb_probe_actor_enter("setUpTenant", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetUpTenantActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setUpTenant", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> && value) 
	{
		fdb_probe_actor_enter("setUpTenant", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetUpTenantActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setUpTenant", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >*,Error err) 
	{
		fdb_probe_actor_enter("setUpTenant", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetUpTenantActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setUpTenant", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	TenantName tenantName;
															#line 479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via setUpTenant()
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class SetUpTenantActor final : public Actor<TenantMapEntry>, public ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >, public FastAllocated<SetUpTenantActor>, public SetUpTenantActorState<SetUpTenantActor> {
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<SetUpTenantActor>::operator new;
	using FastAllocated<SetUpTenantActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11540015375471717888UL, 15080139460117310464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<TenantMapEntry>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >;
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	SetUpTenantActor(Database const& cx,TenantName const& tenantName) 
															#line 501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<TenantMapEntry>(),
		   SetUpTenantActorState<SetUpTenantActor>(cx, tenantName),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setUpTenant", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5618268116584690432UL, 14843378931619308800UL);
		ActorExecutionContextHelper __helper(static_cast<SetUpTenantActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setUpTenant");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setUpTenant", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetUpTenantActor, 0, Optional<TenantMapEntry> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<TenantMapEntry> setUpTenant( Database const& cx, TenantName const& tenantName ) {
															#line 290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<TenantMapEntry>(new SetUpTenantActor(cx, tenantName));
															#line 533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

	Future<Void> setup(Database const& cx) override { return _setup(cx, this); }

	std::pair<uint32_t, uint32_t> parseKey(const KeyRef& key) {
		Tuple t = Tuple::unpackUserType(key, true);
		uint32_t k = t.getInt(0);
		Tuple::UserTypeStr userType = t.getUserType(1);
		std::stringstream ss(userType.str.toString());
		// ss.seekg(32); // skip first 32 zeroes?
		uint32_t id;
		ss >> id;

		return { k, id };
	}

	uint32_t parseVal(const ValueRef& val) {
		uint32_t v;
		sscanf(val.toString().substr(0, 8).c_str(), "%08x", &v);
		return v;
	}

	// because we don't have write versions for all of previous data, don't time travel back past restart and set all
	// write/clear versions to the read version of the data
																#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via loadPreviousDirectoryData()
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class LoadPreviousDirectoryDataActor>
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class LoadPreviousDirectoryDataActorState {
															#line 566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	LoadPreviousDirectoryDataActorState(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData),
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   key(),
															#line 333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   id(),
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   val(),
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tr(cx, threadData->tenant),
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   keyRange(normalKeys),
															#line 339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   gotEOS(false),
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   totalRows(0),
															#line 341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   lastKey(-1),
															#line 342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   lastKeyData(threadData->keyData.end())
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this));

	}
	~LoadPreviousDirectoryDataActorState() 
	{
		fdb_probe_actor_destroy("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 344 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Loading previous directory data for {0}\n", threadData->directoryID);
															#line 346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPreviousDirectoryDataActorState();
		static_cast<LoadPreviousDirectoryDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->nextSeqKey = key + 1;
															#line 412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Found {0} rows for  previous directory {1}\n", totalRows, threadData->directoryID);
															#line 414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<LoadPreviousDirectoryDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPreviousDirectoryDataActorState(); static_cast<LoadPreviousDirectoryDataActor*>(this)->destroy(); return 0; }
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<LoadPreviousDirectoryDataActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPreviousDirectoryDataActorState();
		static_cast<LoadPreviousDirectoryDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Version> __when_expr_0 = tr.getReadVersion();
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 1;
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >*>(static_cast<LoadPreviousDirectoryDataActor*>(this)));
															#line 666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (gotEOS)
															#line 694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Error reading range for [{0} - {1}): {2}\n", keyRange.begin.printable(), keyRange.end.printable(), e.name());
															#line 395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() == error_code_end_of_stream)
															#line 717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				gotEOS = true;
															#line 721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
			}
			else
			{
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
				static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 3;
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >*>(static_cast<LoadPreviousDirectoryDataActor*>(this)));
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Dir {0}: RV={1}\n", threadData->directoryID, ver);
															#line 351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		results = PromiseStream<Standalone<RangeResultRef>>();
															#line 352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		stream = tr.getRangeStream(results, keyRange, GetRangeLimits());
															#line 353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& __ver,int loopDepth) 
	{
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ver = __ver;
															#line 766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && __ver,int loopDepth) 
	{
		ver = std::move(__ver);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousDirectoryDataActor*>(this)->ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		auto __when_expr_1 = results.getFuture();
															#line 852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<Standalone<RangeResultRef>>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<Standalone<RangeResultRef>>>::value, "invalid type");
															#line 354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 2;
															#line 354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >*>(static_cast<LoadPreviousDirectoryDataActor*>(this)));
															#line 861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(Standalone<RangeResultRef> const& res,int loopDepth) 
	{
															#line 355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		totalRows += res.size();
															#line 356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for( auto& it : res ) {
															#line 357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			std::tie(key, id) = self->parseKey(it.key);
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (key != lastKey)
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				auto insert = threadData->keyData.insert({ key, KeyData() });
															#line 361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				ASSERT(insert.second);
															#line 362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKeyData = insert.first;
															#line 363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKeyData->second.nextClearIdx = id;
															#line 364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->usedKeys.push_back(key);
															#line 367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				for(int clearedId = 0;clearedId < id;clearedId++) {
															#line 368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					lastKeyData->second.writes.emplace_back(ver, ver, 0, 20);
															#line 892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKey = key;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			val = self->parseVal(it.value);
															#line 376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Dir {0}: ({1}, {2}) = {3}\n", threadData->directoryID, key, id, val);
															#line 379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			lastKeyData->second.writes.emplace_back(ver, MAX_VERSION, val, it.value.size());
															#line 904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!res.empty())
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			keyRange = KeyRangeRef(keyAfter(res.back().key), keyRange.end);
															#line 912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		else
		{
															#line 386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print( "Empty range for [{0} - {1})\n", keyRange.begin.printable(), keyRange.end.printable());
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(Standalone<RangeResultRef> && res,int loopDepth) 
	{
															#line 355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		totalRows += res.size();
															#line 356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for( auto& it : res ) {
															#line 357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			std::tie(key, id) = self->parseKey(it.key);
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (key != lastKey)
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				auto insert = threadData->keyData.insert({ key, KeyData() });
															#line 361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				ASSERT(insert.second);
															#line 362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKeyData = insert.first;
															#line 363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKeyData->second.nextClearIdx = id;
															#line 364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->usedKeys.push_back(key);
															#line 367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				for(int clearedId = 0;clearedId < id;clearedId++) {
															#line 368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					lastKeyData->second.writes.emplace_back(ver, ver, 0, 20);
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				lastKey = key;
															#line 954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			val = self->parseVal(it.value);
															#line 376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Dir {0}: ({1}, {2}) = {3}\n", threadData->directoryID, key, id, val);
															#line 379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			lastKeyData->second.writes.emplace_back(ver, MAX_VERSION, val, it.value.size());
															#line 962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!res.empty())
															#line 966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			keyRange = KeyRangeRef(keyAfter(res.back().key), keyRange.end);
															#line 970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		else
		{
															#line 386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print( "Empty range for [{0} - {1})\n", keyRange.begin.printable(), keyRange.end.printable());
															#line 976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Standalone<RangeResultRef> const& res,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Standalone<RangeResultRef> && res,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousDirectoryDataActor*>(this)->ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >::remove();

	}
	void a_callback_fire(ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >*,Standalone<RangeResultRef> const& value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >*,Standalone<RangeResultRef> && value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousDirectoryDataActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousDirectoryDataActor*>(this)->ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t key;
															#line 333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t id;
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t val;
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction tr;
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	KeyRange keyRange;
															#line 339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	bool gotEOS;
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int64_t totalRows;
															#line 341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t lastKey;
															#line 342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::map<uint32_t, KeyData>::iterator lastKeyData;
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Version ver;
															#line 351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	PromiseStream<Standalone<RangeResultRef>> results;
															#line 352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Future<Void> stream;
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via loadPreviousDirectoryData()
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class LoadPreviousDirectoryDataActor final : public Actor<Void>, public ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >, public ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >, public ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >, public FastAllocated<LoadPreviousDirectoryDataActor>, public LoadPreviousDirectoryDataActorState<LoadPreviousDirectoryDataActor> {
															#line 1185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<LoadPreviousDirectoryDataActor>::operator new;
	using FastAllocated<LoadPreviousDirectoryDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2128048195898615552UL, 10635034822630652672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >;
friend struct ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >;
friend struct ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >;
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	LoadPreviousDirectoryDataActor(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPreviousDirectoryDataActorState<LoadPreviousDirectoryDataActor>(self, cx, threadData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6464009575162079232UL, 12099411638126846208UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousDirectoryDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPreviousDirectoryData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPreviousDirectoryData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPreviousDirectoryDataActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< LoadPreviousDirectoryDataActor, 1, Standalone<RangeResultRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPreviousDirectoryDataActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> loadPreviousDirectoryData( BlobGranuleCorrectnessWorkload* const& self, Database const& cx, Reference<ThreadData> const& threadData ) {
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new LoadPreviousDirectoryDataActor(self, cx, threadData));
															#line 1238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 1243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via loadPreviousTenants()
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class LoadPreviousTenantsActor>
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class LoadPreviousTenantsActorState {
															#line 1249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	LoadPreviousTenantsActorState(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,BGTenantMap* const& tenantData) 
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tenantData(tenantData),
															#line 418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   allTenants(),
															#line 419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tr(cx),
															#line 420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   i()
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPreviousTenants", reinterpret_cast<unsigned long>(this));

	}
	~LoadPreviousTenantsActorState() 
	{
		fdb_probe_actor_destroy("loadPreviousTenants", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 1281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPreviousTenantsActorState();
		static_cast<LoadPreviousTenantsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Loaded {0} previous tenants\n", allTenants.size());
															#line 442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		blobbifiedTenants = std::vector<std::pair<int64_t, TenantMapEntry>>();
															#line 443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i = 0;
															#line 1308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>>> __when_expr_0 = TenantMetadata::tenantMap().getRange(&tr, {}, {}, CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER + 1);
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 1;
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >*>(static_cast<LoadPreviousTenantsActor*>(this)));
															#line 1336 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 2;
															#line 435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousTenantsActor, 1, Void >*>(static_cast<LoadPreviousTenantsActor*>(this)));
															#line 1378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> const& tenantList,int loopDepth) 
	{
															#line 428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(tenantList.results.size() <= CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER && !tenantList.more);
															#line 430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(!tenantList.results.empty());
															#line 432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		allTenants = tenantList.results;
															#line 1397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> && tenantList,int loopDepth) 
	{
															#line 428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(tenantList.results.size() <= CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER && !tenantList.more);
															#line 430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(!tenantList.results.empty());
															#line 432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		allTenants = tenantList.results;
															#line 1410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> const& tenantList,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(tenantList, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> && tenantList,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(tenantList), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousTenantsActor*>(this)->ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >*,KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> const& value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >*,KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> && value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousTenantsActor*>(this)->ActorCallback< LoadPreviousTenantsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPreviousTenantsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenantData->addTenants(blobbifiedTenants);
															#line 451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(!blobbifiedTenants.empty());
															#line 454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Found {0} previous blobbified tenants\n", blobbifiedTenants.size());
															#line 458 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		int myDirectories = 0;
															#line 460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for(i = self->clientId;i < blobbifiedTenants.size();i += self->clientCount) {
															#line 461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			myDirectories++;
															#line 1591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Client {0} loading {1} directories\n", self->clientId, myDirectories);
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (myDirectories == 0)
															#line 1597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (!static_cast<LoadPreviousTenantsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPreviousTenantsActorState(); static_cast<LoadPreviousTenantsActor*>(this)->destroy(); return 0; }
															#line 1601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			new (&static_cast<LoadPreviousTenantsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~LoadPreviousTenantsActorState();
			static_cast<LoadPreviousTenantsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		int denom = std::min(self->clientCount, (int)myDirectories);
															#line 471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		targetByteRate = 2 * SERVER_KNOBS->BG_DELTA_FILE_TARGET_BYTES / deterministicRandom()->randomInt(1, 5) / denom;
															#line 475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		bool skewed = myDirectories > 1 && deterministicRandom()->random01() < 0.4;
															#line 476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		skewMultiplier = int();
															#line 477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (skewed)
															#line 1617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			skewMultiplier = 2;
															#line 480 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			targetByteRate /= 2;
															#line 1623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		else
		{
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			skewMultiplier = 1;
															#line 483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			targetByteRate /= myDirectories;
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 486 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		dataLoaders = std::vector<Future<Void>>();
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i = self->clientId;
															#line 1637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!(i < allTenants.size()))
															#line 1653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		Reference<Tenant> tenant = makeReference<Tenant>(allTenants[i].first);
															#line 445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Standalone<VectorRef<KeyRangeRef>>> __when_expr_2 = cx->listBlobbifiedRanges(normalKeys, 1, tenant);
															#line 445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 3;
															#line 445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >*>(static_cast<LoadPreviousTenantsActor*>(this)));
															#line 1668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Standalone<VectorRef<KeyRangeRef>> const& blobbifiedRanges,int loopDepth) 
	{
															#line 446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!blobbifiedRanges.empty())
															#line 1690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			blobbifiedTenants.push_back(allTenants[i]);
															#line 1694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i++;
															#line 1698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Standalone<VectorRef<KeyRangeRef>> && blobbifiedRanges,int loopDepth) 
	{
															#line 446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!blobbifiedRanges.empty())
															#line 1707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			blobbifiedTenants.push_back(allTenants[i]);
															#line 1711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i++;
															#line 1715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Standalone<VectorRef<KeyRangeRef>> const& blobbifiedRanges,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(blobbifiedRanges, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Standalone<VectorRef<KeyRangeRef>> && blobbifiedRanges,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(blobbifiedRanges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousTenantsActor*>(this)->ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> const& value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> && value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(dataLoaders);
															#line 505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont3when1(__when_expr_4.get(), loopDepth); };
		static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 5;
															#line 505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousTenantsActor, 4, Void >*>(static_cast<LoadPreviousTenantsActor*>(this)));
															#line 1806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!(i < blobbifiedTenants.size()))
															#line 1822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		int dirId = atoi(blobbifiedTenants[i].second.tenantName.printable().c_str());
															#line 489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 1830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Client {0}/{1} re-creating directory {2}\n", self->clientId, self->clientCount, dirId);
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData = makeReference<ThreadData>(dirId, targetByteRate);
															#line 494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_3 = threadData->openTenant(cx);
															#line 494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 4;
															#line 494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousTenantsActor, 3, Void >*>(static_cast<LoadPreviousTenantsActor*>(this)));
															#line 1847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		auto& tenantEntry = blobbifiedTenants[i].second;
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->tenantEntry = tenantEntry;
															#line 497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->directoryRange = KeyRangeRef(tenantEntry.prefix, tenantEntry.prefix.withSuffix(normalKeys.end));
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories.push_back(threadData);
															#line 500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		dataLoaders.push_back(self->loadPreviousDirectoryData(self, cx, threadData));
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		targetByteRate /= skewMultiplier;
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i += self->clientCount;
															#line 1881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		auto& tenantEntry = blobbifiedTenants[i].second;
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->tenantEntry = tenantEntry;
															#line 497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->directoryRange = KeyRangeRef(tenantEntry.prefix, tenantEntry.prefix.withSuffix(normalKeys.end));
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories.push_back(threadData);
															#line 500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		dataLoaders.push_back(self->loadPreviousDirectoryData(self, cx, threadData));
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		targetByteRate /= skewMultiplier;
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		i += self->clientCount;
															#line 1902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousTenantsActor*>(this)->ActorCallback< LoadPreviousTenantsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< LoadPreviousTenantsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Client {0}/{1} recreated {2} directories\n", self->clientId, self->clientCount, dataLoaders.size());
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<LoadPreviousTenantsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPreviousTenantsActorState(); static_cast<LoadPreviousTenantsActor*>(this)->destroy(); return 0; }
															#line 1988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<LoadPreviousTenantsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPreviousTenantsActorState();
		static_cast<LoadPreviousTenantsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Client {0}/{1} recreated {2} directories\n", self->clientId, self->clientCount, dataLoaders.size());
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<LoadPreviousTenantsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPreviousTenantsActorState(); static_cast<LoadPreviousTenantsActor*>(this)->destroy(); return 0; }
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<LoadPreviousTenantsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPreviousTenantsActorState();
		static_cast<LoadPreviousTenantsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousTenantsActor*>(this)->ActorCallback< LoadPreviousTenantsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< LoadPreviousTenantsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< LoadPreviousTenantsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BGTenantMap* tenantData;
															#line 418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<std::pair<int64_t, TenantMapEntry>> allTenants;
															#line 419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction tr;
															#line 420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int i;
															#line 442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<std::pair<int64_t, TenantMapEntry>> blobbifiedTenants;
															#line 471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int targetByteRate;
															#line 476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int skewMultiplier;
															#line 486 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<Future<Void>> dataLoaders;
															#line 492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 2107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via loadPreviousTenants()
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class LoadPreviousTenantsActor final : public Actor<Void>, public ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >, public ActorCallback< LoadPreviousTenantsActor, 1, Void >, public ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >, public ActorCallback< LoadPreviousTenantsActor, 3, Void >, public ActorCallback< LoadPreviousTenantsActor, 4, Void >, public FastAllocated<LoadPreviousTenantsActor>, public LoadPreviousTenantsActorState<LoadPreviousTenantsActor> {
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<LoadPreviousTenantsActor>::operator new;
	using FastAllocated<LoadPreviousTenantsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14797259112319071232UL, 13657057170939702272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >;
friend struct ActorCallback< LoadPreviousTenantsActor, 1, Void >;
friend struct ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >;
friend struct ActorCallback< LoadPreviousTenantsActor, 3, Void >;
friend struct ActorCallback< LoadPreviousTenantsActor, 4, Void >;
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	LoadPreviousTenantsActor(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,BGTenantMap* const& tenantData) 
															#line 2133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPreviousTenantsActorState<LoadPreviousTenantsActor>(self, cx, tenantData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPreviousTenants", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1408123992786710272UL, 16547784132044564224UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPreviousTenants");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPreviousTenants", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPreviousTenantsActor, 0, KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPreviousTenantsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPreviousTenantsActor, 2, Standalone<VectorRef<KeyRangeRef>> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< LoadPreviousTenantsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< LoadPreviousTenantsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> loadPreviousTenants( BlobGranuleCorrectnessWorkload* const& self, Database const& cx, BGTenantMap* const& tenantData ) {
															#line 417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new LoadPreviousTenantsActor(self, cx, tenantData));
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via initializeTenants()
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class InitializeTenantsActor>
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class InitializeTenantsActorState {
															#line 2180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	InitializeTenantsActorState(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,BGTenantMap* const& tenantData) 
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tenantData(tenantData),
															#line 513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   directoryIdx(0),
															#line 514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   tenants()
															#line 2195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("initializeTenants", reinterpret_cast<unsigned long>(this));

	}
	~InitializeTenantsActorState() 
	{
		fdb_probe_actor_destroy("initializeTenants", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 2210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitializeTenantsActorState();
		static_cast<InitializeTenantsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenantData->addTenants(tenants);
															#line 529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<InitializeTenantsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitializeTenantsActorState(); static_cast<InitializeTenantsActor*>(this)->destroy(); return 0; }
															#line 2235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<InitializeTenantsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitializeTenantsActorState();
		static_cast<InitializeTenantsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!(directoryIdx < self->directories.size()))
															#line 2254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<TenantMapEntry> __when_expr_0 = self->setUpTenant(cx, self->directories[directoryIdx]->tenantName);
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 1;
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >*>(static_cast<InitializeTenantsActor*>(this)));
															#line 2267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->directories[directoryIdx]->openTenant(cx);
															#line 519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 2;
															#line 519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitializeTenantsActor, 1, Void >*>(static_cast<InitializeTenantsActor*>(this)));
															#line 2296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(TenantMapEntry const& __tenantEntry,int loopDepth) 
	{
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenantEntry = __tenantEntry;
															#line 2305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(TenantMapEntry && __tenantEntry,int loopDepth) 
	{
		tenantEntry = std::move(__tenantEntry);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state > 0) static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<InitializeTenantsActor*>(this)->ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >::remove();

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >*,TenantMapEntry const& value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >*,TenantMapEntry && value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >*,Error err) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->tenantEntry = tenantEntry;
															#line 521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->directoryRange = KeyRangeRef(tenantEntry.prefix, tenantEntry.prefix.withSuffix(normalKeys.end));
															#line 523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenants.push_back({ self->directories[directoryIdx]->tenant->id(), tenantEntry });
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<bool> __when_expr_2 = cx->blobbifyRange(self->directories[directoryIdx]->directoryRange);
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 3;
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitializeTenantsActor, 2, bool >*>(static_cast<InitializeTenantsActor*>(this)));
															#line 2397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->tenantEntry = tenantEntry;
															#line 521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->directoryRange = KeyRangeRef(tenantEntry.prefix, tenantEntry.prefix.withSuffix(normalKeys.end));
															#line 523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenants.push_back({ self->directories[directoryIdx]->tenant->id(), tenantEntry });
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<bool> __when_expr_2 = cx->blobbifyRange(self->directories[directoryIdx]->directoryRange);
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 3;
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitializeTenantsActor, 2, bool >*>(static_cast<InitializeTenantsActor*>(this)));
															#line 2419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state > 0) static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<InitializeTenantsActor*>(this)->ActorCallback< InitializeTenantsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitializeTenantsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(bool const& _success,int loopDepth) 
	{
															#line 525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(_success);
															#line 516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx++;
															#line 2505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(bool && _success,int loopDepth) 
	{
															#line 525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(_success);
															#line 516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx++;
															#line 2516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(bool const& _success,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_success, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(bool && _success,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_success), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitializeTenantsActor*>(this)->actor_wait_state > 0) static_cast<InitializeTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<InitializeTenantsActor*>(this)->ActorCallback< InitializeTenantsActor, 2, bool >::remove();

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 2, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitializeTenantsActor, 2, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitializeTenantsActor, 2, bool >*,Error err) 
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BGTenantMap* tenantData;
															#line 513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int directoryIdx;
															#line 514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<std::pair<int64_t, TenantMapEntry>> tenants;
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	TenantMapEntry tenantEntry;
															#line 2608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via initializeTenants()
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class InitializeTenantsActor final : public Actor<Void>, public ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >, public ActorCallback< InitializeTenantsActor, 1, Void >, public ActorCallback< InitializeTenantsActor, 2, bool >, public FastAllocated<InitializeTenantsActor>, public InitializeTenantsActorState<InitializeTenantsActor> {
															#line 2613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<InitializeTenantsActor>::operator new;
	using FastAllocated<InitializeTenantsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2509503593992579328UL, 9908105185262414848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >;
friend struct ActorCallback< InitializeTenantsActor, 1, Void >;
friend struct ActorCallback< InitializeTenantsActor, 2, bool >;
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	InitializeTenantsActor(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,BGTenantMap* const& tenantData) 
															#line 2632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   InitializeTenantsActorState<InitializeTenantsActor>(self, cx, tenantData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initializeTenants", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12618106252002230528UL, 12262811807427804928UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initializeTenants");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initializeTenants", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitializeTenantsActor, 0, TenantMapEntry >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitializeTenantsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitializeTenantsActor, 2, bool >*)0, actor_cancelled()); break;
		}

	}
};
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> initializeTenants( BlobGranuleCorrectnessWorkload* const& self, Database const& cx, BGTenantMap* const& tenantData ) {
															#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new InitializeTenantsActor(self, cx, tenantData));
															#line 2666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 2671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via _setup()
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class _setupActor>
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class _setupActorState {
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	_setupActorState(Database const& cx,BlobGranuleCorrectnessWorkload* const& self) 
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : cx(cx),
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   self(self)
															#line 2686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (self->doSetup)
															#line 2701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<Void> __when_expr_0 = success(ManagementAPI::changeConfig(cx.getReference(), "blob_granules_enabled=1", true));
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Void >*>(static_cast<_setupActor*>(this)));
															#line 2712 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<DatabaseConfiguration> __when_expr_1 = getDatabaseConfiguration(cx);
															#line 538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, DatabaseConfiguration >*>(static_cast<_setupActor*>(this)));
															#line 2747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(DatabaseConfiguration const& dbConfig,int loopDepth) 
	{
															#line 539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (g_network->isSimulated() && dbConfig.encryptionAtRestMode.isEncryptionEnabled())
															#line 2843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			TraceEvent("EncryptionAtRestPlainTextMarkerCheckEnabled") .detail("EncryptionMode", dbConfig.encryptionAtRestMode.toString()) .detail("DataAtRestMarker", BG_ENCRYPTION_AT_REST_MARKER_STRING);
															#line 543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			g_simulator->dataAtRestPlaintextMarker = BG_ENCRYPTION_AT_REST_MARKER_STRING;
															#line 2849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenantData = BGTenantMap(self->dbInfo);
															#line 548 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (g_network->isSimulated() && g_simulator->restarted)
															#line 2855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_2 = self->loadPreviousTenants(self, cx, &tenantData);
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 3;
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 2, Void >*>(static_cast<_setupActor*>(this)));
															#line 2866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_3 = self->initializeTenants(self, cx, &tenantData);
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when2(__when_expr_3.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 4;
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 3, Void >*>(static_cast<_setupActor*>(this)));
															#line 2880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont3(DatabaseConfiguration && dbConfig,int loopDepth) 
	{
															#line 539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (g_network->isSimulated() && dbConfig.encryptionAtRestMode.isEncryptionEnabled())
															#line 2890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			TraceEvent("EncryptionAtRestPlainTextMarkerCheckEnabled") .detail("EncryptionMode", dbConfig.encryptionAtRestMode.toString()) .detail("DataAtRestMarker", BG_ENCRYPTION_AT_REST_MARKER_STRING);
															#line 543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			g_simulator->dataAtRestPlaintextMarker = BG_ENCRYPTION_AT_REST_MARKER_STRING;
															#line 2896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tenantData = BGTenantMap(self->dbInfo);
															#line 548 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (g_network->isSimulated() && g_simulator->restarted)
															#line 2902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_2 = self->loadPreviousTenants(self, cx, &tenantData);
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 3;
															#line 549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 2, Void >*>(static_cast<_setupActor*>(this)));
															#line 2913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_3 = self->initializeTenants(self, cx, &tenantData);
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when2(__when_expr_3.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 4;
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 3, Void >*>(static_cast<_setupActor*>(this)));
															#line 2927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1when1(DatabaseConfiguration const& dbConfig,int loopDepth) 
	{
		loopDepth = a_body1cont3(dbConfig, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(DatabaseConfiguration && dbConfig,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(dbConfig), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 1, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _setupActor, 1, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		data = Reference<GranuleTenantData>();
															#line 556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx = 0;
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx = 0;
															#line 3016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _setupActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< _setupActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< _setupActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 564 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 3199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!(directoryIdx < self->directories.size()))
															#line 3218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_4 = store(data, tenantData.getDataForGranule(self->directories[directoryIdx]->directoryRange));
															#line 558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when1(__when_expr_4.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 5;
															#line 558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 4, Void >*>(static_cast<_setupActor*>(this)));
															#line 3231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_5 = data->bstoreLoaded.getFuture();
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 6;
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 5, Void >*>(static_cast<_setupActor*>(this)));
															#line 3260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_5 = data->bstoreLoaded.getFuture();
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 6;
															#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 5, Void >*>(static_cast<_setupActor*>(this)));
															#line 3276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< _setupActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< _setupActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont4loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(0);
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1cont3when1(__when_expr_6.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 7;
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 6, Void >*>(static_cast<_setupActor*>(this)));
															#line 3367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(0);
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1cont3when1(__when_expr_6.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 7;
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 6, Void >*>(static_cast<_setupActor*>(this)));
															#line 3383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< _setupActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< _setupActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont4loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->bstore = data->bstore;
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx++;
															#line 3469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->directories[directoryIdx]->bstore = data->bstore;
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		directoryIdx++;
															#line 3480 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< _setupActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< _setupActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BGTenantMap tenantData;
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<GranuleTenantData> data;
															#line 556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int directoryIdx;
															#line 3570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via _setup()
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Void >, public ActorCallback< _setupActor, 1, DatabaseConfiguration >, public ActorCallback< _setupActor, 2, Void >, public ActorCallback< _setupActor, 3, Void >, public ActorCallback< _setupActor, 4, Void >, public ActorCallback< _setupActor, 5, Void >, public ActorCallback< _setupActor, 6, Void >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 3575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7782538122686296576UL, 15054017437640794368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Void >;
friend struct ActorCallback< _setupActor, 1, DatabaseConfiguration >;
friend struct ActorCallback< _setupActor, 2, Void >;
friend struct ActorCallback< _setupActor, 3, Void >;
friend struct ActorCallback< _setupActor, 4, Void >;
friend struct ActorCallback< _setupActor, 5, Void >;
friend struct ActorCallback< _setupActor, 6, Void >;
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	_setupActor(Database const& cx,BlobGranuleCorrectnessWorkload* const& self) 
															#line 3598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _setupActor, 1, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _setupActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< _setupActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< _setupActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< _setupActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< _setupActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> _setup( Database const& cx, BlobGranuleCorrectnessWorkload* const& self ) {
															#line 532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new _setupActor(cx, self));
															#line 3636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

	// handle retries + errors
	// It's ok to reset the transaction here because its read version is only used for reading the granule mapping from
	// the system keyspace
																#line 3644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via doGrv()
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class DoGrvActor>
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class DoGrvActorState {
															#line 3650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	DoGrvActorState(Transaction* const& tr) 
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : tr(tr)
															#line 3657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("doGrv", reinterpret_cast<unsigned long>(this));

	}
	~DoGrvActorState() 
	{
		fdb_probe_actor_destroy("doGrv", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 3672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoGrvActorState();
		static_cast<DoGrvActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Version> __when_expr_0 = tr->getReadVersion();
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<DoGrvActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DoGrvActor*>(this)->actor_wait_state = 1;
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoGrvActor, 0, Version >*>(static_cast<DoGrvActor*>(this)));
															#line 3710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr->onError(e);
															#line 576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<DoGrvActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DoGrvActor*>(this)->actor_wait_state = 2;
															#line 576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoGrvActor, 1, Void >*>(static_cast<DoGrvActor*>(this)));
															#line 3739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version const& readVersion,int loopDepth) 
	{
															#line 574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<DoGrvActor*>(this)->SAV<Version>::futures) { (void)(readVersion); this->~DoGrvActorState(); static_cast<DoGrvActor*>(this)->destroy(); return 0; }
															#line 3754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<DoGrvActor*>(this)->SAV< Version >::value()) Version(readVersion);
		this->~DoGrvActorState();
		static_cast<DoGrvActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version && readVersion,int loopDepth) 
	{
															#line 574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<DoGrvActor*>(this)->SAV<Version>::futures) { (void)(readVersion); this->~DoGrvActorState(); static_cast<DoGrvActor*>(this)->destroy(); return 0; }
															#line 3766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<DoGrvActor*>(this)->SAV< Version >::value()) Version(readVersion);
		this->~DoGrvActorState();
		static_cast<DoGrvActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(readVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(readVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoGrvActor*>(this)->actor_wait_state > 0) static_cast<DoGrvActor*>(this)->actor_wait_state = 0;
		static_cast<DoGrvActor*>(this)->ActorCallback< DoGrvActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< DoGrvActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoGrvActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoGrvActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoGrvActor*>(this)->actor_wait_state > 0) static_cast<DoGrvActor*>(this)->actor_wait_state = 0;
		static_cast<DoGrvActor*>(this)->ActorCallback< DoGrvActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoGrvActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoGrvActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoGrvActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction* tr;
															#line 3938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via doGrv()
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class DoGrvActor final : public Actor<Version>, public ActorCallback< DoGrvActor, 0, Version >, public ActorCallback< DoGrvActor, 1, Void >, public FastAllocated<DoGrvActor>, public DoGrvActorState<DoGrvActor> {
															#line 3943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<DoGrvActor>::operator new;
	using FastAllocated<DoGrvActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4224705647315507456UL, 14060395440459582208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoGrvActor, 0, Version >;
friend struct ActorCallback< DoGrvActor, 1, Void >;
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	DoGrvActor(Transaction* const& tr) 
															#line 3961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Version>(),
		   DoGrvActorState<DoGrvActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doGrv", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1994116863933768704UL, 13279750342320377088UL);
		ActorExecutionContextHelper __helper(static_cast<DoGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doGrv");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doGrv", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoGrvActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoGrvActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Version> doGrv( Transaction* const& tr ) {
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Version>(new DoGrvActor(tr));
															#line 3994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 3999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via waitFirstSnapshot()
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class WaitFirstSnapshotActor>
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class WaitFirstSnapshotActorState {
															#line 4005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	WaitFirstSnapshotActorState(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData,bool const& doSetup) 
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData),
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   doSetup(doSetup)
															#line 4018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("waitFirstSnapshot", reinterpret_cast<unsigned long>(this));

	}
	~WaitFirstSnapshotActorState() 
	{
		fdb_probe_actor_destroy("waitFirstSnapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 4033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitFirstSnapshotActorState();
		static_cast<WaitFirstSnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tr = Transaction(cx, threadData->tenant);
															#line 4063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		try {
															#line 589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Version> __when_expr_0 = self->doGrv(&tr);
															#line 589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 1;
															#line 589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitFirstSnapshotActor, 0, Version >*>(static_cast<WaitFirstSnapshotActor*>(this)));
															#line 4074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 4096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 4100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() != error_code_blob_granule_transaction_too_old)
															#line 4104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
				static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 3;
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitFirstSnapshotActor, 2, Void >*>(static_cast<WaitFirstSnapshotActor*>(this)));
															#line 4115 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<Void> __when_expr_3 = delay(1.0);
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when2(__when_expr_3.get(), loopDepth); };
				static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 4;
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitFirstSnapshotActor, 3, Void >*>(static_cast<WaitFirstSnapshotActor*>(this)));
															#line 4129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version const& rv,int loopDepth) 
	{
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		readVersion = rv;
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>>> __when_expr_1 = readFromBlob( cx, threadData->bstore, normalKeys , 0, readVersion, threadData->tenant);
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 2;
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*>(static_cast<WaitFirstSnapshotActor*>(this)));
															#line 4154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version && rv,int loopDepth) 
	{
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		readVersion = rv;
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>>> __when_expr_1 = readFromBlob( cx, threadData->bstore, normalKeys , 0, readVersion, threadData->tenant);
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 2;
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*>(static_cast<WaitFirstSnapshotActor*>(this)));
															#line 4172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& rv,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(rv, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && rv,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(rv), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WaitFirstSnapshotActor*>(this)->ActorCallback< WaitFirstSnapshotActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitFirstSnapshotActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Directory {0} got {1} RV {2}\n", threadData->directoryID, doSetup ? "initial" : "final", readVersion);
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->minSuccessfulReadVersion = readVersion;
															#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<WaitFirstSnapshotActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitFirstSnapshotActorState(); static_cast<WaitFirstSnapshotActor*>(this)->destroy(); return 0; }
															#line 4260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<WaitFirstSnapshotActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitFirstSnapshotActorState();
		static_cast<WaitFirstSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Directory {0} got {1} RV {2}\n", threadData->directoryID, doSetup ? "initial" : "final", readVersion);
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->minSuccessfulReadVersion = readVersion;
															#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<WaitFirstSnapshotActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitFirstSnapshotActorState(); static_cast<WaitFirstSnapshotActor*>(this)->destroy(); return 0; }
															#line 4276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<WaitFirstSnapshotActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitFirstSnapshotActorState();
		static_cast<WaitFirstSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(blob, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(blob), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WaitFirstSnapshotActor*>(this)->ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,Error err) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WaitFirstSnapshotActor*>(this)->ActorCallback< WaitFirstSnapshotActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitFirstSnapshotActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WaitFirstSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WaitFirstSnapshotActor*>(this)->ActorCallback< WaitFirstSnapshotActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitFirstSnapshotActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitFirstSnapshotActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	bool doSetup;
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction tr;
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Version readVersion;
															#line 4551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via waitFirstSnapshot()
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class WaitFirstSnapshotActor final : public Actor<Void>, public ActorCallback< WaitFirstSnapshotActor, 0, Version >, public ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >, public ActorCallback< WaitFirstSnapshotActor, 2, Void >, public ActorCallback< WaitFirstSnapshotActor, 3, Void >, public FastAllocated<WaitFirstSnapshotActor>, public WaitFirstSnapshotActorState<WaitFirstSnapshotActor> {
															#line 4556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<WaitFirstSnapshotActor>::operator new;
	using FastAllocated<WaitFirstSnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2014783604650488576UL, 2332875132540745984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitFirstSnapshotActor, 0, Version >;
friend struct ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >;
friend struct ActorCallback< WaitFirstSnapshotActor, 2, Void >;
friend struct ActorCallback< WaitFirstSnapshotActor, 3, Void >;
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	WaitFirstSnapshotActor(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData,bool const& doSetup) 
															#line 4576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   WaitFirstSnapshotActorState<WaitFirstSnapshotActor>(self, cx, threadData, doSetup),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16633058889432242432UL, 1828694609578432512UL);
		ActorExecutionContextHelper __helper(static_cast<WaitFirstSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitFirstSnapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitFirstSnapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitFirstSnapshotActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitFirstSnapshotActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitFirstSnapshotActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitFirstSnapshotActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> waitFirstSnapshot( BlobGranuleCorrectnessWorkload* const& self, Database const& cx, Reference<ThreadData> const& threadData, bool const& doSetup ) {
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new WaitFirstSnapshotActor(self, cx, threadData, doSetup));
															#line 4611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

	void logKey(Optional<Key> key) {
		if (!key.present()) {
			fmt::print("<missing>\n");
		} else {
			uint32_t k, id;
			std::tie(k, id) = parseKey(key.get());
			fmt::print("({0}, {1}) : {2}\n", k, id, key.get().printable());
		}
	}

	void logMismatch(Reference<ThreadData> threadData,
	                 const Optional<Key>& lastMatching,
	                 const Optional<Key>& expectedKey,
	                 const Optional<Key>& blobKey,
	                 const Optional<Value>& expectedValue,
	                 const Optional<Value>& blobValue,
	                 uint32_t startKey,
	                 uint32_t endKey,
	                 Version beginVersion,
	                 Version readVersion,
	                 const std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>>& blob) {
		threadData->mismatches++;
		if (!BGW_DEBUG) {
			return;
		}

		TraceEvent ev(SevError, "BGMismatch");
		ev.detail("DirectoryID", format("%08x", threadData->directoryID))
		    .detail("RangeStart", format("%08x", startKey))
		    .detail("RangeEnd", format("%08x", endKey))
		    .detail("BeginVersion", beginVersion)
		    .detail("Version", readVersion);
		fmt::print("Found mismatch! Request for dir {0} [{1} - {2}) @ {3} - {4}\n",
		           format("%08x", threadData->directoryID),
		           format("%08x", startKey),
		           format("%08x", endKey),
		           beginVersion,
		           readVersion);
		if (lastMatching.present()) {
			fmt::print("    last correct: ");
			logKey(lastMatching);
		}
		if (expectedValue.present() || blobValue.present()) {
			// value mismatch
			ASSERT(blobKey.present());
			ASSERT(blobKey == expectedKey);
			fmt::print("  Value mismatch for {0}.\n    Expected={1}\n    Actual={2}\n",
			           blobKey.get().printable(),
			           expectedValue.get().printable(),
			           blobValue.get().printable());
		} else {
			// key mismatch
			fmt::print("    Expected Key: ");
			logKey(expectedKey);
			fmt::print("      Actual Key: ");
			logKey(blobKey);
		}

		fmt::print("Chunks: {0}\n", blob.second.size());
		for (auto& chunk : blob.second) {
			fmt::print("[{0} - {1})\n", chunk.keyRange.begin.printable(), chunk.keyRange.end.printable());

			fmt::print("  SnapshotFile:\n    {}\n",
			           chunk.snapshotFile.present() ? chunk.snapshotFile.get().toString().c_str() : "<none>");
			fmt::print("  DeltaFiles:\n");
			for (auto& df : chunk.deltaFiles) {
				fmt::print("    {}\n", df.toString());
			}
			fmt::print("  Deltas: ({})", chunk.newDeltas.size());
			if (chunk.newDeltas.size() > 0) {
				fmt::print(" with version [{0} - {1}]",
				           chunk.newDeltas[0].version,
				           chunk.newDeltas[chunk.newDeltas.size() - 1].version);
			}
			fmt::print("  IncludedVersion: {}\n", chunk.includedVersion);
		}
		printf("\n");
	}

	Value genVal(uint32_t val, uint16_t valLen) {
		std::string v(valLen, 'x');
		auto valFormatted = format("%08x", val);
		ASSERT(valFormatted.size() <= v.size());

		for (int i = 0; i < valFormatted.size(); i++) {
			v[i] = valFormatted[i];
		}
		// copy into an arena
		// TODO do this in original arena? a bit more efficient that way
		Arena a;
		return Standalone<StringRef>(StringRef(a, v), a);
	}

	bool validateValue(const Value& v, uint32_t val, uint16_t valLen) {
		if (v.size() != valLen) {
			return false;
		}
		// check for correct value portion
		auto valFormatted = format("%08x", val);
		ASSERT(valFormatted.size() <= v.size());
		if (v.substr(0, valFormatted.size()) != valFormatted) {
			return false;
		}
		// check for corruption
		for (int i = valFormatted.size(); i < v.size(); i++) {
			if (v[i] != 'x') {
				return false;
			}
		}
		return true;
	}

	bool validateResult(Reference<ThreadData> threadData,
	                    std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> blob,
	                    int startKeyInclusive,
	                    int endKeyExclusive,
	                    Version beginVersion,
	                    Version readVersion) {
		auto checkIt = threadData->keyData.lower_bound(startKeyInclusive);
		if (checkIt != threadData->keyData.end() && checkIt->first < startKeyInclusive) {
			checkIt++;
		}
		int resultIdx = 0;
		Optional<Key> lastMatching;
		if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
			fmt::print("DBG READ: [{0} - {1}) @ {2} ({3} rows)\n",
			           format("%08x", startKeyInclusive),
			           format("%08x", endKeyExclusive),
			           readVersion,
			           blob.first.size());
		}

		// because each chunk could be separately collapsed or not if we set beginVersion, we have to track it by chunk
		KeyRangeMap<Version> beginVersionByChunk;
		beginVersionByChunk.insert(normalKeys, 0);
		int beginCollapsed = 0;
		int beginNotCollapsed = 0;
		Key lastBeginKey = ""_sr;
		for (auto& chunk : blob.second) {
			KeyRange beginVersionRange;
			if (chunk.tenantPrefix.present()) {
				beginVersionRange = KeyRangeRef(chunk.keyRange.begin.removePrefix(chunk.tenantPrefix.get()),
				                                chunk.keyRange.end.removePrefix(chunk.tenantPrefix.get()));
			} else {
				beginVersionRange = chunk.keyRange;
			}

			if (!chunk.snapshotFile.present()) {
				ASSERT(beginVersion > 0);
				ASSERT(chunk.snapshotVersion == invalidVersion);
				beginCollapsed++;

				beginVersionByChunk.insert(beginVersionRange, beginVersion);
			} else {
				ASSERT(chunk.snapshotVersion != invalidVersion);
				if (beginVersion > 0) {
					beginNotCollapsed++;
				}
			}

			// Validate boundary alignment.
			threadData->validateGranuleBoundary(beginVersionRange.begin, beginVersionRange.end, lastBeginKey);
			lastBeginKey = beginVersionRange.begin;
		}
		CODE_PROBE(
		    beginCollapsed > 0, "BGCorrectness got collapsed request with beginVersion > 0", probe::decoration::rare);
		CODE_PROBE(beginNotCollapsed > 0,
		           "BGCorrectness got un-collapsed request with beginVersion > 0",
		           probe::decoration::rare);
		CODE_PROBE(beginCollapsed > 0 && beginNotCollapsed > 0,
		           "BGCorrectness got both collapsed and uncollapsed in the same request!",
		           probe::decoration::rare);

		while (checkIt != threadData->keyData.end() && checkIt->first < endKeyExclusive) {
			uint32_t key = checkIt->first;
			if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
				fmt::print("DBG READ:   Key {0}\n", format("%08x", key));
			}

			// TODO could binary search this to find clearVersion if it gets long
			int idIdx = 0;
			for (; idIdx < checkIt->second.writes.size() && checkIt->second.writes[idIdx].clearVersion <= readVersion;
			     idIdx++) {
				// iterate until we find the oldest tag that should have not been cleared
				/*if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
				    fmt::print(
				        "DBG READ:     Skip ID {0} cleared @ {1}\n", idIdx, checkIt->second.writes[idIdx].clearVersion);
				}*/
			}
			for (; idIdx < checkIt->second.writes.size() && checkIt->second.writes[idIdx].writeVersion <= readVersion;
			     idIdx++) {
				Key nextKeyShouldBe = threadData->getKey(key, idIdx);
				Version keyBeginVersion = beginVersionByChunk.rangeContaining(nextKeyShouldBe).cvalue();
				if (keyBeginVersion > checkIt->second.writes[idIdx].writeVersion) {
					if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
						fmt::print("DBG READ:     Skip ID {0} written @ {1} < beginVersion {2}\n",
						           idIdx,
						           checkIt->second.writes[idIdx].clearVersion,
						           keyBeginVersion);
					}
					continue;
				}
				if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
					fmt::print("DBG READ:     Checking ID {0} ({1}) written @ {2}\n",
					           format("%08x", idIdx),
					           idIdx,
					           checkIt->second.writes[idIdx].writeVersion);
				}
				if (resultIdx >= blob.first.size()) {
					// missing at end!!
					logMismatch(threadData,
					            lastMatching,
					            nextKeyShouldBe,
					            Optional<Key>(),
					            Optional<Value>(),
					            Optional<Value>(),
					            startKeyInclusive,
					            endKeyExclusive,
					            beginVersion,
					            readVersion,
					            blob);
					return false;
				}

				if (nextKeyShouldBe != blob.first[resultIdx].key) {
					// key mismatch!
					if (DEBUG_READ_OP(threadData->directoryID, readVersion)) {
						printf("key mismatch!\n");
					}
					logMismatch(threadData,
					            lastMatching,
					            nextKeyShouldBe,
					            blob.first[resultIdx].key,
					            Optional<Value>(),
					            Optional<Value>(),
					            startKeyInclusive,
					            endKeyExclusive,
					            beginVersion,
					            readVersion,
					            blob);
					return false;
				} else if (!validateValue(blob.first[resultIdx].value,
				                          checkIt->second.writes[idIdx].val,
				                          checkIt->second.writes[idIdx].valLength)) {
					logMismatch(threadData,
					            lastMatching,
					            nextKeyShouldBe,
					            blob.first[resultIdx].key,
					            genVal(checkIt->second.writes[idIdx].val, checkIt->second.writes[idIdx].valLength),
					            blob.first[resultIdx].value,
					            startKeyInclusive,
					            endKeyExclusive,
					            beginVersion,
					            readVersion,
					            blob);
					return false;
					// value mismatch for same key
				} else {
					lastMatching = nextKeyShouldBe;
				}
				resultIdx++;
			}
			checkIt++;
		}

		if (resultIdx < blob.first.size()) {
			// blob has extra stuff!!
			logMismatch(threadData,
			            lastMatching,
			            Optional<Key>(),
			            blob.first[resultIdx].key,
			            Optional<Value>(),
			            Optional<Value>(),
			            startKeyInclusive,
			            endKeyExclusive,
			            beginVersion,
			            readVersion,
			            blob);
			return false;
		}

		return true;
	}

																#line 4900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via readWorker()
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class ReadWorkerActor>
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class ReadWorkerActorState {
															#line 4906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	ReadWorkerActorState(BlobGranuleCorrectnessWorkload* const& self,Future<Void> const& firstSnapshot,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   firstSnapshot(firstSnapshot),
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData),
															#line 900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   last(now()),
															#line 901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   targetBytesReadPerQuery(SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES * 2.0 / deterministicRandom()->randomInt(1, 11)),
															#line 905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   targetReadBytesPerSec(threadData->targetByteRate * 4)
															#line 4925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("readWorker", reinterpret_cast<unsigned long>(this));

	}
	~ReadWorkerActorState() 
	{
		fdb_probe_actor_destroy("readWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(targetReadBytesPerSec > 0);
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			beginVersion = Version();
															#line 909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			readVersion = Version();
															#line 910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			range = KeyRange();
															#line 912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			TraceEvent("BlobGranuleCorrectnessReaderStart").log();
															#line 913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (BGW_DEBUG)
															#line 4950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				printf("BGW read thread starting\n");
															#line 4954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_0 = firstSnapshot;
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadWorkerActor*>(this)->actor_wait_state = 1;
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 0, Void >*>(static_cast<ReadWorkerActor*>(this)));
															#line 4965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadWorkerActorState();
		static_cast<ReadWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_1 = threadData->firstWriteSuccessful.getFuture();
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadWorkerActor*>(this)->actor_wait_state = 2;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 1, Void >*>(static_cast<ReadWorkerActor*>(this)));
															#line 4995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_1 = threadData->firstWriteSuccessful.getFuture();
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadWorkerActor*>(this)->actor_wait_state = 2;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 1, Void >*>(static_cast<ReadWorkerActor*>(this)));
															#line 5011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state > 0) static_cast<ReadWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<ReadWorkerActor*>(this)->ActorCallback< ReadWorkerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadWorkerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleCorrectnessReaderReady").log();
															#line 922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 5097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			printf("BGW read thread ready\n");
															#line 5101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 5105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleCorrectnessReaderReady").log();
															#line 922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 5116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			printf("BGW read thread ready\n");
															#line 5120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 5124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state > 0) static_cast<ReadWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<ReadWorkerActor*>(this)->ActorCallback< ReadWorkerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadWorkerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
		try {
															#line 931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			int startKeyIdx = deterministicRandom()->randomInt(0, threadData->usedKeys.size());
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			startKey = threadData->usedKeys[startKeyIdx];
															#line 933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			auto endKeyIt = threadData->keyData.find(startKey);
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(endKeyIt != threadData->keyData.end());
															#line 937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			endKey = uint32_t();
															#line 938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (deterministicRandom()->random01() < 0.01)
															#line 5226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				endKey = startKey + 1;
															#line 5230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
			else
			{
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int targetQueryBytes = (deterministicRandom()->randomInt(1, 20) * targetBytesReadPerQuery) / 10;
															#line 942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int estimatedQueryBytes = 0;
															#line 943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				for(;estimatedQueryBytes < targetQueryBytes && endKeyIt != threadData->keyData.end();endKeyIt++) {
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					estimatedQueryBytes += (1 + endKeyIt->second.writes.size() - endKeyIt->second.nextClearIdx) * threadData->targetValLength;
															#line 5242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (endKeyIt == threadData->keyData.end())
															#line 5246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					endKey = std::numeric_limits<uint32_t>::max();
															#line 5250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
				else
				{
															#line 953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					endKey = endKeyIt->first;
															#line 5256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
			}
															#line 957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			range = KeyRangeRef(threadData->getKey(startKey, 0), threadData->getKey(endKey, 0));
															#line 960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(threadData->writeVersions.back() >= threadData->minSuccessfulReadVersion);
															#line 961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			size_t readVersionIdx;
															#line 963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (deterministicRandom()->random01() < 0.5)
															#line 5267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->reads++;
															#line 965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				readVersionIdx = threadData->writeVersions.size() - 1;
															#line 966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				readVersion = threadData->writeVersions.back();
															#line 5275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
			else
			{
															#line 968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->timeTravelReads++;
															#line 969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				size_t startIdx = 0;
															#line 970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				for(;;) {
															#line 971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					readVersionIdx = deterministicRandom()->randomInt(startIdx, threadData->writeVersions.size());
															#line 972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					readVersion = threadData->writeVersions[readVersionIdx];
															#line 973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					if (readVersion >= threadData->minSuccessfulReadVersion)
															#line 5291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
					{
						break;
					}
					else
					{
															#line 976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
						startIdx = readVersionIdx + 1;
															#line 5299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
					}
				}
			}
															#line 982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			beginVersion = 0;
															#line 983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (deterministicRandom()->random01() < 0.5)
															#line 5307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int startIdx = 0;
															#line 985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int endIdxExclusive = readVersionIdx + 1;
															#line 988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				size_t beginVersionIdx = (size_t)std::sqrt( deterministicRandom()->randomInt(startIdx * startIdx, endIdxExclusive * endIdxExclusive));
															#line 990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				beginVersion = threadData->writeVersions[beginVersionIdx];
															#line 5317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 993 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			threadData->maybeReopenTenant(cx);
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>>> __when_expr_2 = readFromBlob(cx, threadData->bstore, range, beginVersion, readVersion, threadData->tenant);
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ReadWorkerActor*>(this)->actor_wait_state = 3;
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*>(static_cast<ReadWorkerActor*>(this)));
															#line 5330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 5352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (e.code() == error_code_blob_granule_transaction_too_old)
															#line 5360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (BGW_DEBUG)
															#line 5364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					fmt::print("ERROR: TTO for [{0} - {1}) @ {2} for tenant {3}\n", range.begin.printable(), range.end.printable(), readVersion, printable(threadData->tenant->description()));
															#line 5368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 1016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->timeTravelTooOld++;
															#line 5372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
			else
			{
															#line 1018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->errors++;
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (BGW_DEBUG)
															#line 5380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					printf("BGWorkload got unexpected error %s\n", e.name());
															#line 5384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
			}
			loopDepth = a_body1cont3loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->validateResult(threadData, blob, startKey, endKey, beginVersion, readVersion);
															#line 999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		int resultBytes = blob.first.expectedSize();
															#line 1000 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->rowsRead += blob.first.size();
															#line 1001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->bytesRead += resultBytes;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_3 = poisson(&last, (resultBytes + 1) / targetReadBytesPerSec);
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadWorkerActor*>(this)->actor_wait_state = 4;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 3, Void >*>(static_cast<ReadWorkerActor*>(this)));
															#line 5416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		self->validateResult(threadData, blob, startKey, endKey, beginVersion, readVersion);
															#line 999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		int resultBytes = blob.first.expectedSize();
															#line 1000 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->rowsRead += blob.first.size();
															#line 1001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->bytesRead += resultBytes;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_3 = poisson(&last, (resultBytes + 1) / targetReadBytesPerSec);
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadWorkerActor*>(this)->actor_wait_state = 4;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadWorkerActor, 3, Void >*>(static_cast<ReadWorkerActor*>(this)));
															#line 5440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(blob, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(std::move(blob), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state > 0) static_cast<ReadWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<ReadWorkerActor*>(this)->ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,Error err) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3loopBody1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadWorkerActor*>(this)->actor_wait_state > 0) static_cast<ReadWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<ReadWorkerActor*>(this)->ActorCallback< ReadWorkerActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadWorkerActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadWorkerActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont3loopBody1cont14cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Future<Void> firstSnapshot;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	double last;
															#line 901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	double targetBytesReadPerQuery;
															#line 905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	double targetReadBytesPerSec;
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Version beginVersion;
															#line 909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Version readVersion;
															#line 910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	KeyRange range;
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t startKey;
															#line 937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t endKey;
															#line 5644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via readWorker()
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class ReadWorkerActor final : public Actor<Void>, public ActorCallback< ReadWorkerActor, 0, Void >, public ActorCallback< ReadWorkerActor, 1, Void >, public ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >, public ActorCallback< ReadWorkerActor, 3, Void >, public FastAllocated<ReadWorkerActor>, public ReadWorkerActorState<ReadWorkerActor> {
															#line 5649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<ReadWorkerActor>::operator new;
	using FastAllocated<ReadWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1775271521349658368UL, 10924911600931528192UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadWorkerActor, 0, Void >;
friend struct ActorCallback< ReadWorkerActor, 1, Void >;
friend struct ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >;
friend struct ActorCallback< ReadWorkerActor, 3, Void >;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	ReadWorkerActor(BlobGranuleCorrectnessWorkload* const& self,Future<Void> const& firstSnapshot,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 5669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   ReadWorkerActorState<ReadWorkerActor>(self, firstSnapshot, cx, threadData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3771596646280737536UL, 599502687565927936UL);
		ActorExecutionContextHelper __helper(static_cast<ReadWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadWorkerActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadWorkerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadWorkerActor, 2, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadWorkerActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> readWorker( BlobGranuleCorrectnessWorkload* const& self, Future<Void> const& firstSnapshot, Database const& cx, Reference<ThreadData> const& threadData ) {
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new ReadWorkerActor(self, firstSnapshot, cx, threadData));
															#line 5704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 1026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 5709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via writeWorker()
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class WriteWorkerActor>
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class WriteWorkerActorState {
															#line 5715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	WriteWorkerActorState(BlobGranuleCorrectnessWorkload* const& self,Future<Void> const& firstSnapshot,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   firstSnapshot(firstSnapshot),
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData),
															#line 1032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   last(now()),
															#line 1033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   keysPerQuery(100),
															#line 1036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   nextVal(0)
															#line 5734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("writeWorker", reinterpret_cast<unsigned long>(this));

	}
	~WriteWorkerActorState() 
	{
		fdb_probe_actor_destroy("writeWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			TraceEvent("BlobGranuleCorrectnessWriterStart").log();
															#line 1040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_0 = firstSnapshot;
															#line 1040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<WriteWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteWorkerActor*>(this)->actor_wait_state = 1;
															#line 1040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteWorkerActor, 0, Void >*>(static_cast<WriteWorkerActor*>(this)));
															#line 5758 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteWorkerActorState();
		static_cast<WriteWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleCorrectnessWriterReady").log();
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 5783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleCorrectnessWriterReady").log();
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 5794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteWorkerActor*>(this)->actor_wait_state > 0) static_cast<WriteWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<WriteWorkerActor*>(this)->ActorCallback< WriteWorkerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteWorkerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tr = Transaction(cx, threadData->tenant);
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		keyAndIdToWrite = std::vector<std::tuple<uint32_t, uint32_t, uint32_t, uint16_t>>();
															#line 1049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		keyAndIdToClear = std::vector<std::pair<uint32_t, uint32_t>>();
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		queryKeys = keysPerQuery * (0.1 + deterministicRandom()->random01() * 1.8);
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for(int i = 0;i < queryKeys;i++) {
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			uint32_t key;
															#line 1055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (threadData->keyData.empty() || deterministicRandom()->random01() > threadData->reuseKeyProb)
															#line 5897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (threadData->nextKeySequential)
															#line 5901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					key = threadData->nextSeqKey;
															#line 1059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					++threadData->nextSeqKey;
															#line 5907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
				else
				{
															#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					key = std::numeric_limits<uint32_t>::max();
															#line 1062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					for(;key == std::numeric_limits<uint32_t>::max() || threadData->keyData.find(key) != threadData->keyData.end();) {
															#line 1065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
						key = deterministicRandom()->randomUInt32() / 2;
															#line 5917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
					}
				}
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->usedKeys.push_back(key);
															#line 1071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->keyData.insert({ key, KeyData() });
															#line 5924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
			else
			{
															#line 1073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int keyIdx = deterministicRandom()->randomInt(0, threadData->usedKeys.size());
															#line 1074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				key = threadData->usedKeys[keyIdx];
															#line 5932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			auto keyIt = threadData->keyData.find(key);
															#line 1078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(keyIt != threadData->keyData.end());
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			int unclearedIds = keyIt->second.writes.size() - keyIt->second.nextClearIdx;
															#line 1083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			double probAddId = (threadData->targetIDsPerKey * 2.0 - unclearedIds) / threadData->targetIDsPerKey;
															#line 1084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (deterministicRandom()->random01() < probAddId || keyIt->second.nextClearIdx == keyIt->second.writes.size())
															#line 5944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int32_t val = nextVal++;
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				int16_t valLen = (0.5 + deterministicRandom()->random01()) * threadData->targetValLength;
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (valLen < 10)
															#line 5952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					valLen = 10;
															#line 5956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 1092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				uint32_t nextId = keyIt->second.writes.size();
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				keyIt->second.writes.push_back(WriteData(val, valLen));
															#line 1095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				keyAndIdToWrite.push_back(std::tuple(key, nextId, val, valLen));
															#line 5964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
			else
			{
															#line 1097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				uint32_t idToClear = keyIt->second.nextClearIdx++;
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				keyAndIdToClear.push_back(std::pair(key, idToClear));
															#line 5972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
		}
															#line 1102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		txnBytes = int64_t();
															#line 1103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		;
															#line 5979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Writer committed @ {0}\n", commitVersion);
															#line 1136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for( auto& it : keyAndIdToWrite ) {
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			uint32_t key = std::get<0>(it);
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			uint32_t id = std::get<1>(it);
															#line 1139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			auto keyIt = threadData->keyData.find(key);
															#line 1140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(keyIt != threadData->keyData.end());
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			keyIt->second.writes[id].writeVersion = commitVersion;
															#line 1143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (DEBUG_KEY_OP(threadData->directoryID, key))
															#line 6004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				fmt::print("DBG: {0} WRITE {1} = {2}:{3}\n", commitVersion, format("%08x/%08x/%08x", threadData->directoryID, key, id), std::get<2>(it), std::get<3>(it));
															#line 6008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
		}
															#line 1152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for( auto& it : keyAndIdToClear ) {
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			auto keyIt = threadData->keyData.find(it.first);
															#line 1154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(keyIt != threadData->keyData.end());
															#line 1155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			keyIt->second.writes[it.second].clearVersion = commitVersion;
															#line 1156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (DEBUG_KEY_OP(threadData->directoryID, it.first))
															#line 6021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				fmt::print("DBG: {0} CLEAR {1}\n", commitVersion, format("%08x/%08x/%08x", threadData->directoryID, it.first, it.second));
															#line 6025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
		}
															#line 1163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->writeVersions.push_back(commitVersion);
															#line 1165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (threadData->firstWriteSuccessful.canBeSet())
															#line 6032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			threadData->firstWriteSuccessful.send(Void());
															#line 6036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 1169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->rowsWritten += queryKeys;
															#line 1170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		threadData->bytesWritten += txnBytes;
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_3 = poisson(&last, (txnBytes + 1.0) / threadData->targetByteRate);
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<WriteWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteWorkerActor*>(this)->actor_wait_state = 4;
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteWorkerActor, 3, Void >*>(static_cast<WriteWorkerActor*>(this)));
															#line 6051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			for( auto& it : keyAndIdToWrite ) {
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (DEBUG_KEY_OP(threadData->directoryID, std::get<0>(it)))
															#line 6070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1108 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					fmt::print("DBG: {0} PREWRITE ({1}, {2}) = {3}:{4}\n", threadData->directoryID, std::get<0>(it), std::get<1>(it), std::get<2>(it), std::get<3>(it));
															#line 6074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 1115 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				Value v = self->genVal(std::get<2>(it), std::get<3>(it));
															#line 1116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				tr.set(threadData->getKey(std::get<0>(it), std::get<1>(it)), v);
															#line 6080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			for( auto& it : keyAndIdToClear ) {
															#line 1119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				tr.clear(singleKeyRange(threadData->getKey(it.first, it.second)));
															#line 6086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			txnBytes = tr.getSize();
															#line 1122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 1122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<WriteWorkerActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WriteWorkerActor*>(this)->actor_wait_state = 2;
															#line 1122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteWorkerActor, 1, Void >*>(static_cast<WriteWorkerActor*>(this)));
															#line 6099 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Writer error {0}\n", e.name());
															#line 1126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<WriteWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1cont1loopBody1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<WriteWorkerActor*>(this)->actor_wait_state = 3;
															#line 1126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteWorkerActor, 2, Void >*>(static_cast<WriteWorkerActor*>(this)));
															#line 6143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteWorkerActor*>(this)->actor_wait_state > 0) static_cast<WriteWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<WriteWorkerActor*>(this)->ActorCallback< WriteWorkerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteWorkerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteWorkerActor*>(this)->actor_wait_state > 0) static_cast<WriteWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<WriteWorkerActor*>(this)->ActorCallback< WriteWorkerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteWorkerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont11(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteWorkerActor*>(this)->actor_wait_state > 0) static_cast<WriteWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<WriteWorkerActor*>(this)->ActorCallback< WriteWorkerActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteWorkerActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteWorkerActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Future<Void> firstSnapshot;
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 1032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	double last;
															#line 1033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int keysPerQuery;
															#line 1036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	uint32_t nextVal;
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction tr;
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<std::tuple<uint32_t, uint32_t, uint32_t, uint16_t>> keyAndIdToWrite;
															#line 1049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<std::pair<uint32_t, uint32_t>> keyAndIdToClear;
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int queryKeys;
															#line 1102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int64_t txnBytes;
															#line 6439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via writeWorker()
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class WriteWorkerActor final : public Actor<Void>, public ActorCallback< WriteWorkerActor, 0, Void >, public ActorCallback< WriteWorkerActor, 1, Void >, public ActorCallback< WriteWorkerActor, 2, Void >, public ActorCallback< WriteWorkerActor, 3, Void >, public FastAllocated<WriteWorkerActor>, public WriteWorkerActorState<WriteWorkerActor> {
															#line 6444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<WriteWorkerActor>::operator new;
	using FastAllocated<WriteWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9122339168916059392UL, 17602799041825991424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteWorkerActor, 0, Void >;
friend struct ActorCallback< WriteWorkerActor, 1, Void >;
friend struct ActorCallback< WriteWorkerActor, 2, Void >;
friend struct ActorCallback< WriteWorkerActor, 3, Void >;
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	WriteWorkerActor(BlobGranuleCorrectnessWorkload* const& self,Future<Void> const& firstSnapshot,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 6464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   WriteWorkerActorState<WriteWorkerActor>(self, firstSnapshot, cx, threadData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6156249598888385280UL, 17815502008178127872UL);
		ActorExecutionContextHelper __helper(static_cast<WriteWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteWorkerActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteWorkerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteWorkerActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteWorkerActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> writeWorker( BlobGranuleCorrectnessWorkload* const& self, Future<Void> const& firstSnapshot, Database const& cx, Reference<ThreadData> const& threadData ) {
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new WriteWorkerActor(self, firstSnapshot, cx, threadData));
															#line 6499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 1176 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

	Future<Void> start(Database const& cx) override {
		clients.reserve(3 * directories.size());
		for (auto& it : directories) {
			// Wait for blob worker to initialize snapshot before starting test for that range
			Future<Void> start = waitFirstSnapshot(this, cx, it, true);
			it->summaryClient = validateGranuleSummaries(cx, normalKeys, it->tenant, it->triggerSummaryComplete);
			if (doForceFlushing && deterministicRandom()->random01() < 0.25) {
				it->forceFlushingClient =
				    validateForceFlushing(cx, it->directoryRange, testDuration, it->triggerSummaryComplete);
			} else {
				it->forceFlushingClient = Future<Void>(Void());
			}
			clients.push_back(timeout(writeWorker(this, start, cx, it), testDuration, Void()));
			clients.push_back(timeout(readWorker(this, start, cx, it), testDuration, Void()));
		}
		return delay(testDuration);
	}

																#line 6522 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via checkTenantRanges()
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class CheckTenantRangesActor>
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class CheckTenantRangesActorState {
															#line 6528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	CheckTenantRangesActorState(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : self(self),
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   cx(cx),
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData)
															#line 6539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("checkTenantRanges", reinterpret_cast<unsigned long>(this));

	}
	~CheckTenantRangesActorState() 
	{
		fdb_probe_actor_destroy("checkTenantRanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			;
															#line 6554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckTenantRangesActorState();
		static_cast<CheckTenantRangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		tr = Transaction(cx, threadData->tenant);
															#line 6584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		try {
															#line 1203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Standalone<VectorRef<KeyRangeRef>>> __when_expr_0 = tr.getBlobGranuleRanges(normalKeys, 1000000);
															#line 1203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckTenantRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckTenantRangesActor*>(this)->actor_wait_state = 1;
															#line 1203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*>(static_cast<CheckTenantRangesActor*>(this)));
															#line 6595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckTenantRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckTenantRangesActor*>(this)->actor_wait_state = 2;
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckTenantRangesActor, 1, Void >*>(static_cast<CheckTenantRangesActor*>(this)));
															#line 6624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Standalone<VectorRef<KeyRangeRef>> const& ranges,int loopDepth) 
	{
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.size() >= 1 && ranges.size() < 1000000);
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.front().begin == normalKeys.begin);
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.back().end == normalKeys.end);
															#line 1207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for(int i = 0;i < ranges.size() - 1;i++) {
															#line 1208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(ranges[i].end == ranges[i + 1].begin);
															#line 6647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<CheckTenantRangesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckTenantRangesActorState(); static_cast<CheckTenantRangesActor*>(this)->destroy(); return 0; }
															#line 6651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<CheckTenantRangesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckTenantRangesActorState();
		static_cast<CheckTenantRangesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Standalone<VectorRef<KeyRangeRef>> && ranges,int loopDepth) 
	{
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.size() >= 1 && ranges.size() < 1000000);
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.front().begin == normalKeys.begin);
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(ranges.back().end == normalKeys.end);
															#line 1207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		for(int i = 0;i < ranges.size() - 1;i++) {
															#line 1208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			ASSERT(ranges[i].end == ranges[i + 1].begin);
															#line 6671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<CheckTenantRangesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckTenantRangesActorState(); static_cast<CheckTenantRangesActor*>(this)->destroy(); return 0; }
															#line 6675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<CheckTenantRangesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckTenantRangesActorState();
		static_cast<CheckTenantRangesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Standalone<VectorRef<KeyRangeRef>> const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Standalone<VectorRef<KeyRangeRef>> && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckTenantRangesActor*>(this)->actor_wait_state > 0) static_cast<CheckTenantRangesActor*>(this)->actor_wait_state = 0;
		static_cast<CheckTenantRangesActor*>(this)->ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> const& value) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> && value) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckTenantRangesActor*>(this)->actor_wait_state > 0) static_cast<CheckTenantRangesActor*>(this)->actor_wait_state = 0;
		static_cast<CheckTenantRangesActor*>(this)->ActorCallback< CheckTenantRangesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckTenantRangesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckTenantRangesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckTenantRangesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 1201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Transaction tr;
															#line 6853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via checkTenantRanges()
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class CheckTenantRangesActor final : public Actor<Void>, public ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >, public ActorCallback< CheckTenantRangesActor, 1, Void >, public FastAllocated<CheckTenantRangesActor>, public CheckTenantRangesActorState<CheckTenantRangesActor> {
															#line 6858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<CheckTenantRangesActor>::operator new;
	using FastAllocated<CheckTenantRangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1490851252099600896UL, 3218161661990317056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >;
friend struct ActorCallback< CheckTenantRangesActor, 1, Void >;
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	CheckTenantRangesActor(BlobGranuleCorrectnessWorkload* const& self,Database const& cx,Reference<ThreadData> const& threadData) 
															#line 6876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<Void>(),
		   CheckTenantRangesActorState<CheckTenantRangesActor>(self, cx, threadData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkTenantRanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7950637522308152320UL, 17818570868787621120UL);
		ActorExecutionContextHelper __helper(static_cast<CheckTenantRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkTenantRanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkTenantRanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckTenantRangesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckTenantRangesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<Void> checkTenantRanges( BlobGranuleCorrectnessWorkload* const& self, Database const& cx, Reference<ThreadData> const& threadData ) {
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<Void>(new CheckTenantRangesActor(self, cx, threadData));
															#line 6909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 1216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 6914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via checkDirectory()
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class CheckDirectoryActor>
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class CheckDirectoryActorState {
															#line 6920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	CheckDirectoryActorState(Database const& cx,BlobGranuleCorrectnessWorkload* const& self,Reference<ThreadData> const& threadData) 
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : cx(cx),
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   self(self),
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   threadData(threadData)
															#line 6931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("checkDirectory", reinterpret_cast<unsigned long>(this));

	}
	~CheckDirectoryActorState() 
	{
		fdb_probe_actor_destroy("checkDirectory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (threadData->triggerSummaryComplete.canBeSet())
															#line 6946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				threadData->triggerSummaryComplete.send(Void());
															#line 6950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			result = true;
															#line 1225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			finalRowsValidated = int();
															#line 1226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (threadData->writeVersions.empty())
															#line 6958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (BGW_DEBUG)
															#line 6962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					fmt::print("Directory {0} doing final availability check\n", threadData->directoryID);
															#line 6966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 1233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<Void> __when_expr_0 = self->waitFirstSnapshot(self, cx, threadData, false);
															#line 1233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 1;
															#line 1233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 0, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 6977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				readVersion = threadData->writeVersions.back();
															#line 1238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (BGW_DEBUG)
															#line 6986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				{
															#line 1239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
					fmt::print("Directory {0} doing final data check @ {1}\n", threadData->directoryID, readVersion);
															#line 6990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				}
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				StrictFuture<std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>>> __when_expr_1 = readFromBlob( cx, threadData->bstore, normalKeys , 0, readVersion, threadData->tenant);
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
				static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 2;
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckDirectoryActorState();
		static_cast<CheckDirectoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (BGW_DEBUG)
															#line 7025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			fmt::print("Directory {0} checking tenant ranges\n", threadData->directoryID);
															#line 7029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->checkTenantRanges(self, cx, threadData);
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 4;
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 3, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
															#line 1243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		result = self->validateResult(threadData, blob, 0, std::numeric_limits<uint32_t>::max(), 0, readVersion);
															#line 1244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		finalRowsValidated = blob.first.size();
															#line 1248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (result)
															#line 7140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (BGW_DEBUG)
															#line 7144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				fmt::print("Directory {0} doing final availability check after data check\n", threadData->directoryID);
															#line 7148 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_2 = self->waitFirstSnapshot(self, cx, threadData, false);
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont5when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 3;
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 2, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7159 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
															#line 1243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		result = self->validateResult(threadData, blob, 0, std::numeric_limits<uint32_t>::max(), 0, readVersion);
															#line 1244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		finalRowsValidated = blob.first.size();
															#line 1248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (result)
															#line 7177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (BGW_DEBUG)
															#line 7181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			{
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				fmt::print("Directory {0} doing final availability check after data check\n", threadData->directoryID);
															#line 7185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_2 = self->waitFirstSnapshot(self, cx, threadData, false);
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont5when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 3;
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 2, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when2(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& blob,int loopDepth) 
	{
		loopDepth = a_body1cont5(blob, loopDepth);

		return loopDepth;
	}
	int a_body1when2(std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && blob,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(blob), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 1262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		initialCheck = result;
															#line 1263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		result &= threadData->mismatches == 0 && (threadData->timeTravelTooOld == 0);
															#line 1265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Blob Granule Workload Directory {0} {1}:\n", threadData->directoryID, result ? "passed" : "failed");
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  Final granule check {0}successful\n", initialCheck ? "" : "un");
															#line 1267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} Rows read in final check\n", finalRowsValidated);
															#line 1268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} mismatches\n", threadData->mismatches);
															#line 1269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} time travel too old\n", threadData->timeTravelTooOld);
															#line 1270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} errors\n", threadData->errors);
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} rows written\n", threadData->rowsWritten);
															#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} bytes written\n", threadData->bytesWritten);
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} unique keys\n", threadData->usedKeys.size());
															#line 1274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} real-time reads\n", threadData->reads);
															#line 1275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} time travel reads\n", threadData->timeTravelReads);
															#line 1276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} rows read\n", threadData->rowsRead);
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} bytes read\n", threadData->bytesRead);
															#line 1280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleWorkloadChecked").detail("Directory", threadData->directoryID).detail("Result", result);
															#line 1283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(result);
															#line 1285 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (self->clientId == 0 && SERVER_KNOBS->BG_ENABLE_MERGING && self->doMergeCheckAtEnd)
															#line 7412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			CODE_PROBE(true, "BGCorrectness clearing database and awaiting merge", probe::decoration::rare);
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_4 = clearAndAwaitMerge(cx, threadData->directoryRange);
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont10when1(__when_expr_4.get(), loopDepth); };
			static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 5;
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 4, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 1262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		initialCheck = result;
															#line 1263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		result &= threadData->mismatches == 0 && (threadData->timeTravelTooOld == 0);
															#line 1265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("Blob Granule Workload Directory {0} {1}:\n", threadData->directoryID, result ? "passed" : "failed");
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  Final granule check {0}successful\n", initialCheck ? "" : "un");
															#line 1267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} Rows read in final check\n", finalRowsValidated);
															#line 1268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} mismatches\n", threadData->mismatches);
															#line 1269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} time travel too old\n", threadData->timeTravelTooOld);
															#line 1270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} errors\n", threadData->errors);
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} rows written\n", threadData->rowsWritten);
															#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} bytes written\n", threadData->bytesWritten);
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} unique keys\n", threadData->usedKeys.size());
															#line 1274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} real-time reads\n", threadData->reads);
															#line 1275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} time travel reads\n", threadData->timeTravelReads);
															#line 1276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} rows read\n", threadData->rowsRead);
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		fmt::print("  {} bytes read\n", threadData->bytesRead);
															#line 1280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		TraceEvent("BlobGranuleWorkloadChecked").detail("Directory", threadData->directoryID).detail("Result", result);
															#line 1283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		ASSERT(result);
															#line 1285 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (self->clientId == 0 && SERVER_KNOBS->BG_ENABLE_MERGING && self->doMergeCheckAtEnd)
															#line 7473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			CODE_PROBE(true, "BGCorrectness clearing database and awaiting merge", probe::decoration::rare);
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<Void> __when_expr_4 = clearAndAwaitMerge(cx, threadData->directoryRange);
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont10when1(__when_expr_4.get(), loopDepth); };
			static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 5;
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 4, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7486 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont10cont1(int loopDepth) 
	{
															#line 1291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_5 = threadData->summaryClient && threadData->forceFlushingClient;
															#line 1291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 6;
															#line 1291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< CheckDirectoryActor, 5, Void >*>(static_cast<CheckDirectoryActor*>(this)));
															#line 7582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont10cont3(Void const& _,int loopDepth) 
	{
															#line 1293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<CheckDirectoryActor*>(this)->SAV<bool>::futures) { (void)(result); this->~CheckDirectoryActorState(); static_cast<CheckDirectoryActor*>(this)->destroy(); return 0; }
															#line 7678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<CheckDirectoryActor*>(this)->SAV< bool >::value()) bool(std::move(result)); // state_var_RVO
		this->~CheckDirectoryActorState();
		static_cast<CheckDirectoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont3(Void && _,int loopDepth) 
	{
															#line 1293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<CheckDirectoryActor*>(this)->SAV<bool>::futures) { (void)(result); this->~CheckDirectoryActorState(); static_cast<CheckDirectoryActor*>(this)->destroy(); return 0; }
															#line 7690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<CheckDirectoryActor*>(this)->SAV< bool >::value()) bool(std::move(result)); // state_var_RVO
		this->~CheckDirectoryActorState();
		static_cast<CheckDirectoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<CheckDirectoryActor*>(this)->actor_wait_state > 0) static_cast<CheckDirectoryActor*>(this)->actor_wait_state = 0;
		static_cast<CheckDirectoryActor*>(this)->ActorCallback< CheckDirectoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< CheckDirectoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< CheckDirectoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Reference<ThreadData> threadData;
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	bool result;
															#line 1225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	int finalRowsValidated;
															#line 1237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Version readVersion;
															#line 1262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	bool initialCheck;
															#line 7787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via checkDirectory()
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class CheckDirectoryActor final : public Actor<bool>, public ActorCallback< CheckDirectoryActor, 0, Void >, public ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >, public ActorCallback< CheckDirectoryActor, 2, Void >, public ActorCallback< CheckDirectoryActor, 3, Void >, public ActorCallback< CheckDirectoryActor, 4, Void >, public ActorCallback< CheckDirectoryActor, 5, Void >, public FastAllocated<CheckDirectoryActor>, public CheckDirectoryActorState<CheckDirectoryActor> {
															#line 7792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<CheckDirectoryActor>::operator new;
	using FastAllocated<CheckDirectoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7775909698435064320UL, 13346007995470932992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckDirectoryActor, 0, Void >;
friend struct ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >;
friend struct ActorCallback< CheckDirectoryActor, 2, Void >;
friend struct ActorCallback< CheckDirectoryActor, 3, Void >;
friend struct ActorCallback< CheckDirectoryActor, 4, Void >;
friend struct ActorCallback< CheckDirectoryActor, 5, Void >;
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	CheckDirectoryActor(Database const& cx,BlobGranuleCorrectnessWorkload* const& self,Reference<ThreadData> const& threadData) 
															#line 7814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<bool>(),
		   CheckDirectoryActorState<CheckDirectoryActor>(cx, self, threadData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkDirectory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8656377057068110080UL, 8097237935806647296UL);
		ActorExecutionContextHelper __helper(static_cast<CheckDirectoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkDirectory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkDirectory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckDirectoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckDirectoryActor, 1, std::pair<RangeResult, Standalone<VectorRef<BlobGranuleChunkRef>>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckDirectoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CheckDirectoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CheckDirectoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< CheckDirectoryActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<bool> checkDirectory( Database const& cx, BlobGranuleCorrectnessWorkload* const& self, Reference<ThreadData> const& threadData ) {
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<bool>(new CheckDirectoryActor(cx, self, threadData));
															#line 7851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

																#line 7856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
// This generated class is to be used only via _check()
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
template <class _checkActor>
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class _checkActorState {
															#line 7862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	_checkActorState(Database const& cx,BlobGranuleCorrectnessWorkload* const& self) 
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		 : cx(cx),
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   self(self),
															#line 1298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		   results()
															#line 7873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
	{
		fdb_probe_actor_create("_check", reinterpret_cast<unsigned long>(this));

	}
	~_checkActorState() 
	{
		fdb_probe_actor_destroy("_check", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			for( auto& it : self->directories ) {
															#line 1301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
				results.push_back(self->checkDirectory(cx, self, it));
															#line 7890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			}
															#line 1303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			allSuccessful = true;
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			RangeForbody1Iterator0 = std::begin(results);
															#line 7896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		checkFeedCleanupFuture = Future<Void>();
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (self->clientId == 0)
															#line 7921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			checkFeedCleanupFuture = checkFeedCleanup(cx, BGW_DEBUG);
															#line 7925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
		else
		{
															#line 1314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			checkFeedCleanupFuture = Future<Void>(Void());
															#line 7931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		}
															#line 1316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		StrictFuture<Void> __when_expr_1 = checkFeedCleanupFuture;
															#line 1316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 2;
															#line 1316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 1, Void >*>(static_cast<_checkActor*>(this)));
															#line 7942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!(RangeForbody1Iterator0 != std::end(results)))
															#line 7958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		auto& f = *RangeForbody1Iterator0;
															#line 7964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		{
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			StrictFuture<bool> __when_expr_0 = f;
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 1;
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 0, bool >*>(static_cast<_checkActor*>(this)));
															#line 7975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		++RangeForbody1Iterator0;
															#line 7998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(bool const& dirSuccess,int loopDepth) 
	{
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		allSuccessful &= dirSuccess;
															#line 8007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(bool && dirSuccess,int loopDepth) 
	{
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		allSuccessful &= dirSuccess;
															#line 8016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool const& dirSuccess,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(dirSuccess, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool && dirSuccess,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(dirSuccess), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _checkActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<_checkActor*>(this)->SAV<bool>::futures) { (void)(allSuccessful); this->~_checkActorState(); static_cast<_checkActor*>(this)->destroy(); return 0; }
															#line 8100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<_checkActor*>(this)->SAV< bool >::value()) bool(std::move(allSuccessful)); // state_var_RVO
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
		if (!static_cast<_checkActor*>(this)->SAV<bool>::futures) { (void)(allSuccessful); this->~_checkActorState(); static_cast<_checkActor*>(this)->destroy(); return 0; }
															#line 8112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		new (&static_cast<_checkActor*>(this)->SAV< bool >::value()) bool(std::move(allSuccessful)); // state_var_RVO
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _checkActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Database cx;
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	BlobGranuleCorrectnessWorkload* self;
															#line 1298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	std::vector<Future<bool>> results;
															#line 1303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	bool allSuccessful;
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	decltype(std::begin(std::declval<std::vector<Future<bool>>>())) RangeForbody1Iterator0;
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	Future<Void> checkFeedCleanupFuture;
															#line 8207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
};
// This generated class is to be used only via _check()
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
class _checkActor final : public Actor<bool>, public ActorCallback< _checkActor, 0, bool >, public ActorCallback< _checkActor, 1, Void >, public FastAllocated<_checkActor>, public _checkActorState<_checkActor> {
															#line 8212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
public:
	using FastAllocated<_checkActor>::operator new;
	using FastAllocated<_checkActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6946956553676046848UL, 17960886005040762368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _checkActor, 0, bool >;
friend struct ActorCallback< _checkActor, 1, Void >;
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	_checkActor(Database const& cx,BlobGranuleCorrectnessWorkload* const& self) 
															#line 8230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
		 : Actor<bool>(),
		   _checkActorState<_checkActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18414574916690025728UL, 14115196402046968064UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_check");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _checkActor, 0, bool >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _checkActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
[[nodiscard]] Future<bool> _check( Database const& cx, BlobGranuleCorrectnessWorkload* const& self ) {
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"
	return Future<bool>(new _checkActor(cx, self));
															#line 8263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.g.cpp"
}

#line 1319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/workloads/BlobGranuleCorrectnessWorkload.actor.cpp"

	Future<bool> check(Database const& cx) override { return _check(cx, this); }
	void getMetrics(std::vector<PerfMetric>& m) override {}
};

WorkloadFactory<BlobGranuleCorrectnessWorkload> BlobGranuleCorrectnessWorkloadFactory;
