#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
/*
 * DataDistribution.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Audit.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DDSharedContext.h"
#include "fdbserver/DDTeamCollection.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TenantCache.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/MockDataDistributor.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const std::string ddServerBulkDumpFolder = "ddBulkDumpFiles";
static const std::string ddServerBulkLoadFolder = "ddBulkLoadFiles";

DataMoveType getDataMoveTypeFromDataMoveId(const UID& dataMoveId) {
	bool assigned, emptyRange;
	DataMoveType dataMoveType;
	DataMovementReason dataMoveReason;
	decodeDataMoveId(dataMoveId, assigned, emptyRange, dataMoveType, dataMoveReason);
	return dataMoveType;
}

void RelocateShard::setParentRange(KeyRange const& parent) {
	ASSERT(reason == RelocateReason::WRITE_SPLIT || reason == RelocateReason::SIZE_SPLIT);
	parent_range = parent;
}

Optional<KeyRange> RelocateShard::getParentRange() const {
	return parent_range;
}

ShardSizeBounds ShardSizeBounds::shardSizeBoundsBeforeTrack() {
	return ShardSizeBounds{ .max = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                           .iosPerKSecond = StorageMetrics::infinity,
		                                           .bytesReadPerKSecond = StorageMetrics::infinity,
		                                           .opsReadPerKSecond = StorageMetrics::infinity },
		                    .min = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = 0,
		                                           .iosPerKSecond = 0,
		                                           .bytesReadPerKSecond = 0,
		                                           .opsReadPerKSecond = 0 },
		                    .permittedError = StorageMetrics{ .bytes = -1,
		                                                      .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                                      .iosPerKSecond = StorageMetrics::infinity,
		                                                      .bytesReadPerKSecond = StorageMetrics::infinity,
		                                                      .opsReadPerKSecond = StorageMetrics::infinity } };
}

namespace {

std::set<int> const& normalDDQueueErrors() {
	static std::set<int> s{ error_code_movekeys_conflict,
		                    error_code_broken_promise,
		                    error_code_data_move_cancelled,
		                    error_code_data_move_dest_team_not_found };
	return s;
}

} // anonymous namespace

enum class DDAuditContext : uint8_t {
	INVALID = 0,
	RESUME = 1,
	LAUNCH = 2,
	RETRY = 3,
};

struct DDAudit {
	DDAudit(AuditStorageState coreState)
	  : coreState(coreState), actors(true), foundError(false), auditStorageAnyChildFailed(false), retryCount(0),
	    cancelled(false), overallCompleteDoAuditCount(0), overallIssuedDoAuditCount(0), overallSkippedDoAuditCount(0),
	    remainingBudgetForAuditTasks(SERVER_KNOBS->CONCURRENT_AUDIT_TASK_COUNT_MAX), context(DDAuditContext::INVALID) {}

	AuditStorageState coreState;
	ActorCollection actors;
	Future<Void> auditActor;
	bool foundError;
	int retryCount;
	bool auditStorageAnyChildFailed;
	bool cancelled; // use to cancel any actor beyond auditActor
	int64_t overallIssuedDoAuditCount;
	int64_t overallCompleteDoAuditCount;
	int64_t overallSkippedDoAuditCount;
	AsyncVar<int> remainingBudgetForAuditTasks;
	DDAuditContext context;
	std::unordered_set<UID> serversFinishedSSShardAudit; // dedicated to ssshard

	inline void setAuditRunActor(Future<Void> actor) { auditActor = actor; }
	inline Future<Void> getAuditRunActor() { return auditActor; }

	inline void setDDAuditContext(DDAuditContext context_) { this->context = context_; }
	inline DDAuditContext getDDAuditContext() const { return context; }

	// auditActor and actors are guaranteed to deliver a cancel signal
	void cancel() {
		auditActor.cancel();
		actors.clear(true);
		cancelled = true;
	}

	bool isCancelled() const { return cancelled; }
};

void DataMove::validateShard(const DDShardInfo& shard, KeyRangeRef range, int priority) {
	if (!valid) {
		if (shard.hasDest && shard.destId != anonymousShardId) {
			TraceEvent(SevError, "DataMoveValidationError")
			    .detail("Range", range)
			    .detail("Reason", "DataMoveMissing")
			    .detail("DestID", shard.destId)
			    .detail("ShardPrimaryDest", describe(shard.primaryDest))
			    .detail("ShardRemoteDest", describe(shard.remoteDest));
		}
		return;
	}

	if (this->meta.ranges.empty()) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeEmpty")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!this->meta.ranges.front().contains(range)) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeMismatch")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!shard.hasDest) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "ShardMissingDest")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest));
		cancelled = true;
		return;
	}

	if (shard.destId != this->meta.id) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveIDMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardMoveID", shard.destId);
		cancelled = true;
		return;
	}

	if (!std::equal(
	        this->primaryDest.begin(), this->primaryDest.end(), shard.primaryDest.begin(), shard.primaryDest.end()) ||
	    !std::equal(
	        this->remoteDest.begin(), this->remoteDest.end(), shard.remoteDest.begin(), shard.remoteDest.end())) {
		TraceEvent(g_network->isSimulated() ? SevWarn : SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveDestMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest))
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		cancelled = true;
	}
}

Future<Void> StorageWiggler::onCheck() const {
	return delay(MIN_ON_CHECK_DELAY_SEC);
}

// add server to wiggling queue
void StorageWiggler::addServer(const UID& serverId, const StorageMetadataType& metadata) {
	// std::cout << "size: " << pq_handles.size() << " add " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	ASSERT(!pq_handles.contains(serverId));
	pq_handles[serverId] = wiggle_pq.emplace(metadata, serverId);
}

void StorageWiggler::removeServer(const UID& serverId) {
	// std::cout << "size: " << pq_handles.size() << " remove " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	if (contains(serverId)) { // server haven't been popped
		auto handle = pq_handles.at(serverId);
		pq_handles.erase(serverId);
		wiggle_pq.erase(handle);
	}
}

void StorageWiggler::updateMetadata(const UID& serverId, const StorageMetadataType& metadata) {
	//	std::cout << "size: " << pq_handles.size() << " update " << serverId.toString()
	//	          << " DC: " << teamCollection->isPrimary() << std::endl;
	auto handle = pq_handles.at(serverId);
	if ((*handle).first == metadata) {
		return;
	}
	wiggle_pq.update(handle, std::make_pair(metadata, serverId));
}

bool StorageWiggler::necessary(const UID& serverId, const StorageMetadataType& metadata) const {
	return metadata.wrongConfiguredForWiggle ||
	       (now() - metadata.createdTime > SERVER_KNOBS->DD_STORAGE_WIGGLE_MIN_SS_AGE_SEC);
}

Optional<UID> StorageWiggler::getNextServerId(bool necessaryOnly) {
	if (!wiggle_pq.empty()) {
		auto [metadata, id] = wiggle_pq.top();
		if (necessaryOnly && !necessary(id, metadata)) {
			return {};
		}
		wiggle_pq.pop();
		pq_handles.erase(id);
		return Optional<UID>(id);
	}
	return Optional<UID>();
}

Future<Void> StorageWiggler::resetStats() {
	metrics.reset();
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.resetStorageWiggleMetrics(tr, PrimaryRegion(teamCollection->isPrimary()), metrics);
	    });
}

Future<Void> StorageWiggler::restoreStats() {
	auto readFuture = wiggleData.storageWiggleMetrics(PrimaryRegion(teamCollection->isPrimary()))
	                      .getD(teamCollection->dbContext().getReference(), Snapshot::False, metrics);
	return store(metrics, readFuture);
}

Future<Void> StorageWiggler::startWiggle() {
	metrics.last_wiggle_start = StorageMetadataType::currentTime();
	if (shouldStartNewRound()) {
		metrics.last_round_start = metrics.last_wiggle_start;
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

Future<Void> StorageWiggler::finishWiggle() {
	metrics.last_wiggle_finish = StorageMetadataType::currentTime();
	metrics.finished_wiggle += 1;
	auto duration = metrics.last_wiggle_finish - metrics.last_wiggle_start;
	metrics.smoothed_wiggle_duration.setTotal((double)duration);

	if (shouldFinishRound()) {
		metrics.last_round_finish = metrics.last_wiggle_finish;
		metrics.finished_round += 1;
		duration = metrics.last_round_finish - metrics.last_round_start;
		metrics.smoothed_round_duration.setTotal((double)duration);
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

															#line 318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via remoteRecovered()
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoteRecoveredActor>
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActorState {
															#line 325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActorState(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : db(db)
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	~RemoteRecoveredActorState() 
	{
		fdb_probe_actor_destroy("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDTrackerStarting").log();
															#line 318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoteRecoveredActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoteRecoveredActorState(); static_cast<RemoteRecoveredActor*>(this)->destroy(); return 0; }
															#line 372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoteRecoveredActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(db->get().recoveryState < RecoveryState::ALL_LOGS_RECRUITED))
															#line 391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDTrackerStarting").detail("RecoveryState", (int)db->get().recoveryState);
															#line 320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 1;
															#line 320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoteRecoveredActor, 0, Void >*>(static_cast<RemoteRecoveredActor*>(this)));
															#line 406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state > 0) static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 0;
		static_cast<RemoteRecoveredActor*>(this)->ActorCallback< RemoteRecoveredActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoteRecoveredActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via remoteRecovered()
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActor final : public Actor<Void>, public ActorCallback< RemoteRecoveredActor, 0, Void >, public FastAllocated<RemoteRecoveredActor>, public RemoteRecoveredActorState<RemoteRecoveredActor> {
															#line 518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoteRecoveredActor>::operator new;
	using FastAllocated<RemoteRecoveredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5358891194445973760UL, 10017581095877282304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoteRecoveredActor, 0, Void >;
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActor(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoteRecoveredActorState<RemoteRecoveredActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17462346244829468672UL, 10124612030083111680UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remoteRecovered");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoteRecoveredActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> remoteRecovered( Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoteRecoveredActor(db));
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Ensures that the serverKeys key space is properly coalesced
// This method is only used for testing and is not implemented in a manner that is safe for large databases
															#line 575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via debugCheckCoalescing()
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DebugCheckCoalescingActor>
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActorState {
															#line 582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActorState(Database const& cx) 
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : cx(cx),
															#line 328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	~DebugCheckCoalescingActorState() 
	{
		fdb_probe_actor_destroy("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 1;
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 3;
															#line 351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 2, Void >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = int();
															#line 335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __serverList,int loopDepth) 
	{
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		serverList = __serverList;
															#line 701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __serverList,int loopDepth) 
	{
		serverList = std::move(__serverList);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DoneCheckingCoalescing").log();
															#line 349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DebugCheckCoalescingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DebugCheckCoalescingActorState(); static_cast<DebugCheckCoalescingActor*>(this)->destroy(); return 0; }
															#line 782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DebugCheckCoalescingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < serverList.size()))
															#line 801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 336 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		id = decodeServerListValue(serverList[i].value).id();
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, serverKeysPrefixFor(id), allKeys);
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 2;
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult const& ranges,int loopDepth) 
	{
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult && ranges,int loopDepth) 
	{
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult serverList;
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 336 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID id;
															#line 1050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via debugCheckCoalescing()
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActor final : public Actor<Void>, public ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 2, Void >, public FastAllocated<DebugCheckCoalescingActor>, public DebugCheckCoalescingActorState<DebugCheckCoalescingActor> {
															#line 1055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DebugCheckCoalescingActor>::operator new;
	using FastAllocated<DebugCheckCoalescingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5286848793743505664UL, 9881245728283070720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 2, Void >;
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActor(Database const& cx) 
															#line 1074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DebugCheckCoalescingActorState<DebugCheckCoalescingActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13012747114799893760UL, 10690014687488268032UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("debugCheckCoalescing");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> debugCheckCoalescing( Database const& cx ) {
															#line 327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DebugCheckCoalescingActor(cx));
															#line 1109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DataDistributor;
void runAuditStorage(
    Reference<DataDistributor> self,
    AuditStorageState auditStates,
    int retryCount,
    DDAuditContext context,
    Optional<std::unordered_set<UID>> serversFinishedSSShardAudit = Optional<std::unordered_set<UID>>());
															#line 1121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorageCore( Reference<DataDistributor> const& self, UID const& auditID, AuditType const& auditType, int const& currentRetryCount );

#line 367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<UID> launchAudit( Reference<DataDistributor> const& self, KeyRange const& auditRange, AuditType const& auditType, KeyValueStoreType const& auditStorageEngineType );

#line 371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorage( Reference<DataDistributor> const& self, TriggerAuditRequest const& req );

#line 372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
void loadAndDispatchAudit(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit);
															#line 1134 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorageServerShard( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditStorageShardOnServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi );

#line 377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorage( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& auditRange );

#line 384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1158 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditOnStorageServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi, AuditStorageRequest const& req );

#line 391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> skipAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& rangeToSkip );

#line 394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void runBulkLoadTaskAsync(Reference<DataDistributor> self, KeyRange range, UID taskId, bool restart);
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self );

#line 397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DDBulkLoadJobManager {
	BulkLoadJobState jobState;
	BulkLoadTransportMethod jobTransportMethod;
	std::shared_ptr<BulkLoadManifestFileMap> manifestEntryMap;
	std::string manifestLocalTempFolder;
	bool allTaskSubmitted = false;

	DDBulkLoadJobManager() = default;
	DDBulkLoadJobManager(const BulkLoadJobState& jobState, const std::string& manifestLocalTempFolder)
	  : jobState(jobState), manifestLocalTempFolder(manifestLocalTempFolder), allTaskSubmitted(false) {
		manifestEntryMap = std::make_shared<BulkLoadManifestFileMap>();
	}

	bool isValid() const { return jobState.isValid(); }
};

struct DDBulkDumpJobManager {
	BulkDumpState jobState;
	std::map<Key, BulkLoadManifest> taskManifestMap;

	DDBulkDumpJobManager() = default;
	DDBulkDumpJobManager(const BulkDumpState& jobState) : jobState(jobState) {}

	bool isValid() const { return jobState.isValid(); }
};

struct DataDistributor : NonCopyable, ReferenceCounted<DataDistributor> {
public:
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
	Reference<DDSharedContext> context;
	UID ddId;
	PromiseStream<Future<Void>> addActor;

	// State initialized when bootstrap
	Reference<IDDTxnProcessor> txnProcessor;
	MoveKeysLock& lock; // reference to context->lock
	DatabaseConfiguration& configuration; // reference to context->configuration
	std::vector<Optional<Key>> primaryDcId;
	std::vector<Optional<Key>> remoteDcIds;
	Reference<InitialDataDistribution> initData;

	Reference<EventCacheHolder> initialDDEventHolder;
	Reference<EventCacheHolder> movingDataEventHolder;
	Reference<EventCacheHolder> totalDataInFlightEventHolder;
	Reference<EventCacheHolder> totalDataInFlightRemoteEventHolder;

	// Optional components that can be set after ::init(). They're optional when test, but required for DD being
	// fully-functional.
	DDTeamCollection* teamCollection;
	Reference<ShardsAffectedByTeamFailure> shardsAffectedByTeamFailure;
	// consumer is a yield stream from producer. The RelocateShard is pushed into relocationProducer and popped from
	// relocationConsumer (by DDQueue)
	PromiseStream<RelocateShard> relocationProducer, relocationConsumer;
	PromiseStream<BulkLoadShardRequest> triggerShardBulkLoading;
	Reference<PhysicalShardCollection> physicalShardCollection;
	Reference<BulkLoadTaskCollection> bulkLoadTaskCollection;

	Promise<Void> initialized;

	std::unordered_map<AuditType, std::unordered_map<UID, std::shared_ptr<DDAudit>>> audits;
	FlowLock auditStorageHaLaunchingLock;
	FlowLock auditStorageReplicaLaunchingLock;
	FlowLock auditStorageLocationMetadataLaunchingLock;
	FlowLock auditStorageSsShardLaunchingLock;
	Promise<Void> auditStorageInitialized;
	bool auditStorageInitStarted;

	Optional<Reference<TenantCache>> ddTenantCache;

	// monitor DD configuration change
	Promise<Version> configChangeWatching;
	Future<Void> onConfigChange;

	ActorCollection bulkLoadActors;
	bool bulkLoadEnabled = false;
	ParallelismLimitor bulkLoadParallelismLimitor;
	ParallelismLimitor bulkLoadEngineParallelismLimitor;
	std::string bulkLoadFolder;

	Optional<DDBulkLoadJobManager> bulkLoadJobManager;

	bool bulkDumpEnabled = false;
	ParallelismLimitor bulkDumpParallelismLimitor;
	std::string folder;
	std::string bulkDumpFolder;

	DDBulkDumpJobManager bulkDumpJobManager;

	DataDistributor(Reference<AsyncVar<ServerDBInfo> const> const& db,
	                UID id,
	                Reference<DDSharedContext> context,
	                std::string folder)
	  : dbInfo(db), context(context), ddId(id), txnProcessor(nullptr), lock(context->lock),
	    configuration(context->configuration), initialDDEventHolder(makeReference<EventCacheHolder>("InitialDD")),
	    movingDataEventHolder(makeReference<EventCacheHolder>("MovingData")),
	    totalDataInFlightEventHolder(makeReference<EventCacheHolder>("TotalDataInFlight")),
	    totalDataInFlightRemoteEventHolder(makeReference<EventCacheHolder>("TotalDataInFlightRemote")),
	    teamCollection(nullptr), bulkLoadTaskCollection(nullptr), auditStorageHaLaunchingLock(1),
	    auditStorageReplicaLaunchingLock(1), auditStorageLocationMetadataLaunchingLock(1),
	    auditStorageSsShardLaunchingLock(1), auditStorageInitStarted(false), bulkLoadActors(false),
	    bulkLoadEnabled(false), bulkLoadParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM),
	    bulkLoadEngineParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM), bulkDumpEnabled(false),
	    bulkDumpParallelismLimitor(SERVER_KNOBS->DD_BULKDUMP_PARALLELISM), folder(folder) {
		if (!folder.empty()) {
			bulkDumpFolder = abspath(joinPath(folder, ddServerBulkDumpFolder));
			// TODO(BulkDump): clear this folder in the presence of crash
			bulkLoadFolder = abspath(joinPath(folder, ddServerBulkLoadFolder));
			// TODO(BulkLoad): clear this folder in the presence of crash
		}
	}

	// bootstrap steps

	Future<Void> takeMoveKeysLock() { return store(lock, txnProcessor->takeMoveKeysLock(ddId)); }

	Future<Void> loadDatabaseConfiguration() { return store(configuration, txnProcessor->getDatabaseConfiguration()); }

	Future<Void> updateReplicaKeys() {
		return txnProcessor->updateReplicaKeys(primaryDcId, remoteDcIds, configuration);
	}

	Future<Void> loadInitialDataDistribution() {
		return store(initData,
		             txnProcessor->getInitialDataDistribution(
		                 ddId,
		                 lock,
		                 configuration.usableRegions > 1 ? remoteDcIds : std::vector<Optional<Key>>(),
		                 context->ddEnabledState.get(),
		                 SkipDDModeCheck::False));
	}

	void initDcInfo() {
		primaryDcId.clear();
		remoteDcIds.clear();
		const std::vector<RegionInfo>& regions = configuration.regions;
		if (configuration.regions.size() > 0) {
			primaryDcId.push_back(regions[0].dcId);
		}
		if (configuration.regions.size() > 1) {
			remoteDcIds.push_back(regions[1].dcId);
		}
	}

	Future<Void> waitDataDistributorEnabled() const {
		return txnProcessor->waitForDataDistributionEnabled(context->ddEnabledState.get());
	}

	// Resume in-memory audit instances and issue background audit metadata cleanup
	void resumeAuditStorage(Reference<DataDistributor> self, std::vector<AuditStorageState> auditStates) {
		for (const auto& auditState : auditStates) {
			if (auditState.getPhase() != AuditPhase::Running) {
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "WrongAuditStateToResume")
				    .detail("AuditState", auditState.toString());
				return;
			}
			if (self->audits.contains(auditState.getType()) &&
			    self->audits[auditState.getType()].contains(auditState.id)) {
				// Ignore any RUNNING auditState with an alive audit
				// instance in DD audits map
				continue;
			}
			runAuditStorage(self, auditState, 0, DDAuditContext::RESUME);
			TraceEvent(SevInfo, "AuditStorageResumed", self->ddId)
			    .detail("AuditID", auditState.id)
			    .detail("AuditType", auditState.getType())
			    .detail("AuditState", auditState.toString());
		}
		return;
	}

																#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via initAuditStorage()
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitAuditStorageActor>
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActorState {
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActorState(Reference<DataDistributor> const& self) 
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 1356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	~InitAuditStorageActorState() 
	{
		fdb_probe_actor_destroy("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->auditStorageInitStarted = true;
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			MoveKeyLockInfo lockInfo;
															#line 571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.myOwner = self->lock.myOwner;
															#line 572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevOwner = self->lock.prevOwner;
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevWrite = self->lock.prevWrite;
															#line 574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<std::vector<AuditStorageState>> __when_expr_0 = initAuditMetadata(self->txnProcessor->context(), lockInfo, self->context->isDDEnabled(), self->ddId, SERVER_KNOBS->PERSIST_FINISH_AUDIT_COUNT);
															#line 574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 1;
															#line 574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*>(static_cast<InitAuditStorageActor*>(this)));
															#line 1388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
															#line 580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
															#line 580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(auditStatesToResume, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(auditStatesToResume), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state > 0) static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 0;
		static_cast<InitAuditStorageActor*>(this)->ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >::remove();

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> const& value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> && value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,Error err) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via initAuditStorage()
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActor final : public Actor<Void>, public ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >, public FastAllocated<InitAuditStorageActor>, public InitAuditStorageActorState<InitAuditStorageActor> {
															#line 1521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitAuditStorageActor>::operator new;
	using FastAllocated<InitAuditStorageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11947021153667747584UL, 5044388840274500864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActor(Reference<DataDistributor> const& self) 
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitAuditStorageActorState<InitAuditStorageActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8855224104647952640UL, 5194562846350463232UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initAuditStorage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> initAuditStorage( Reference<DataDistributor> const& self ) {
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitAuditStorageActor(self));
															#line 1570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 1575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class WaitUntilDataDistributorExitSecurityModeActor>
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActorState {
															#line 1581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActorState(Reference<DataDistributor> const& self) 
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(self->txnProcessor->context())
															#line 1590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	~WaitUntilDataDistributorExitSecurityModeActorState() 
	{
		fdb_probe_actor_destroy("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 1605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitUntilDataDistributorExitSecurityModeActorState();
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->DD_ENABLED_CHECK_DELAY, TaskPriority::DataDistribution);
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 1;
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1696 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 4;
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1800 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& mode,int loopDepth) 
	{
															#line 594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && mode,int loopDepth) 
	{
															#line 594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> const& mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(mode, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> && mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(mode), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActor final : public Actor<Void>, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >, public FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>, public WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor> {
															#line 2180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator new;
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10174712273687357184UL, 6039049890480785920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >;
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActor(Reference<DataDistributor> const& self) 
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4834519635010961920UL, 15563594834600371456UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitUntilDataDistributorExitSecurityMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> waitUntilDataDistributorExitSecurityMode( Reference<DataDistributor> const& self ) {
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new WaitUntilDataDistributorExitSecurityModeActor(self));
															#line 2235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Initialize the required internal states of DataDistributor from system metadata. It's necessary before
	// DataDistributor start working. Doesn't include initialization of optional components, like TenantCache, DDQueue,
	// Tracker, TeamCollection. The components should call its own ::init methods.
																#line 2243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via init()
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitActor>
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActorState {
															#line 2249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActorState(Reference<DataDistributor> const& self) 
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 2256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("init", reinterpret_cast<unsigned long>(this));

	}
	~InitActorState() 
	{
		fdb_probe_actor_destroy("init", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 2271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitActorState();
		static_cast<InitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitActorState(); static_cast<InitActor*>(this)->destroy(); return 0; }
															#line 2294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitActorState();
		static_cast<InitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->waitDataDistributorEnabled();
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 1;
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitActor, 0, Void >*>(static_cast<InitActor*>(this)));
															#line 2320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2503 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributorExitSecurityMode").log();
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->loadDatabaseConfiguration();
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 4;
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< InitActor, 3, Void >*>(static_cast<InitActor*>(this)));
															#line 2621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< InitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< InitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 720 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 2992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 3001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 3015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 720 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 3030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< InitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 3112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via init()
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActor final : public Actor<Void>, public ActorCallback< InitActor, 0, Void >, public ActorCallback< InitActor, 1, Void >, public ActorCallback< InitActor, 2, Void >, public ActorCallback< InitActor, 3, Void >, public ActorCallback< InitActor, 4, Void >, public ActorCallback< InitActor, 5, Void >, public FastAllocated<InitActor>, public InitActorState<InitActor> {
															#line 3117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitActor>::operator new;
	using FastAllocated<InitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2956690947713847552UL, 15501434131580385280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitActor, 0, Void >;
friend struct ActorCallback< InitActor, 1, Void >;
friend struct ActorCallback< InitActor, 2, Void >;
friend struct ActorCallback< InitActor, 3, Void >;
friend struct ActorCallback< InitActor, 4, Void >;
friend struct ActorCallback< InitActor, 5, Void >;
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActor(Reference<DataDistributor> const& self) 
															#line 3139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitActorState<InitActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1770802396446596864UL, 4776746127134105600UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< InitActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< InitActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< InitActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> init( Reference<DataDistributor> const& self ) {
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitActor(self));
															#line 3176 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoveDataMoveTombstoneBackgroundActor>
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActorState {
															#line 3187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActorState(Reference<DataDistributor> const& self) 
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentID()
															#line 3196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	~RemoveDataMoveTombstoneBackgroundActorState() 
	{
		fdb_probe_actor_destroy("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				cx = openDBOnServer(self->dbInfo, TaskPriority::DefaultEndpoint, LockAware::True);
															#line 734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr = Transaction(cx);
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				;
															#line 3216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveDataMoveTombstoneBackgroundActorState(); static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->destroy(); return 0; }
															#line 3245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "RemoveDataMoveTombstoneError", self->ddId) .errorUnsuppressed(e) .detail("CurrentDataMoveID", currentID);
															#line 3266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			for( UID& dataMoveID : self->initData->toCleanDataMoveTombstone ) {
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				currentID = dataMoveID;
															#line 741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr.clear(dataMoveKeyFor(currentID));
															#line 742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "RemoveDataMoveTombstone", self->ddId).detail("DataMoveID", currentID);
															#line 3305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 1;
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 2;
															#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID currentID;
															#line 733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 3564 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActor final : public Actor<Void>, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >, public FastAllocated<RemoveDataMoveTombstoneBackgroundActor>, public RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor> {
															#line 3569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator new;
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3120403552972470528UL, 2577689287817630720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >;
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >;
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActor(Reference<DataDistributor> const& self) 
															#line 3587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929663207261195520UL, 2170269584154920960UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeDataMoveTombstoneBackground");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> removeDataMoveTombstoneBackground( Reference<DataDistributor> const& self ) {
															#line 730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoveDataMoveTombstoneBackgroundActor(self));
															#line 3620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 762 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromShards()
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromShardsActor>
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActorState {
															#line 3631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActorState(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   traceShard(traceShard)
															#line 3640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromShardsActorState() 
	{
		fdb_probe_actor_destroy("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 3655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for(int i = 0;i < self->initData->shards.size() - 1;i++) {
															#line 767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					const DDShardInfo& iShard = self->initData->shards[i];
															#line 768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					KeyRangeRef keys = KeyRangeRef(iShard.key, self->initData->shards[i + 1].key);
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					teams.emplace_back(iShard.primarySrc, true);
															#line 771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (self->configuration.usableRegions > 1)
															#line 3669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.emplace_back(iShard.remoteSrc, false);
															#line 3673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
															#line 774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->physicalShardCollection->initPhysicalShardCollection(keys, teams, iShard.srcId.first(), 0);
															#line 3677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundaries = std::vector<Key>();
															#line 779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadIsEnabled(self->initData->bulkLoadMode))
															#line 3684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevInfo, "DDInitCustomRangeConfigDisabledByBulkLoadMode", self->ddId);
															#line 3688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for( auto it : self->initData->userRangeConfig->ranges() ) {
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					auto range = it->range();
															#line 785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					customBoundaries.push_back(range.begin);
															#line 786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDInitCustomRangeConfig", self->ddId) .detail("Range", KeyRangeRef(range.begin, range.end)) .detail("Config", it->value());
															#line 3700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			shard = 0;
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary = 0;
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			overreplicatedCount = 0;
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 3711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromShardsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromShardsActorState(); static_cast<ResumeFromShardsActor*>(this)->destroy(); return 0; }
															#line 3734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromShardsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(shard < self->initData->shards.size() - 1))
															#line 3753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DDShardInfo& iShard = self->initData->shards[shard];
															#line 797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<KeyRangeRef> ranges;
															#line 799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key beginKey = iShard.key;
															#line 800 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key endKey = self->initData->shards[shard + 1].key;
															#line 801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] <= beginKey;) {
															#line 802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] < endKey;) {
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ranges.push_back(KeyRangeRef(beginKey, customBoundaries[customBoundary]));
															#line 806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = customBoundaries[customBoundary];
															#line 807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ranges.push_back(KeyRangeRef(beginKey, endKey));
															#line 811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.primarySrc, true));
															#line 813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.usableRegions > 1)
															#line 3789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.remoteSrc, false));
															#line 3793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 817 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int r = 0;r < ranges.size();r++) {
															#line 818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto& keys = ranges[r];
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->defineShard(keys);
															#line 821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->initData->userRangeConfig->rangeContaining(keys.begin);
															#line 822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			int customReplicas = std::max(self->configuration.storageTeamSize, it->value().replicationFactor.orDefault(0));
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT_WE_THINK(KeyRangeRef(it->range().begin, it->range().end).contains(keys));
															#line 826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bool unhealthy = iShard.primarySrc.size() != customReplicas;
															#line 827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && self->configuration.usableRegions > 1)
															#line 3811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				unhealthy = iShard.remoteSrc.size() != customReplicas;
															#line 3815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && iShard.primarySrc.size() > self->configuration.storageTeamSize)
															#line 3819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (++overreplicatedCount > SERVER_KNOBS->DD_MAX_SHARDS_ON_LARGE_TEAMS)
															#line 3823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					unhealthy = true;
															#line 3827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (traceShard)
															#line 3832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDInitShard", self->ddId) .detail("Keys", keys) .detail("PrimarySrc", describe(iShard.primarySrc)) .detail("RemoteSrc", describe(iShard.remoteSrc)) .detail("PrimaryDest", describe(iShard.primaryDest)) .detail("RemoteDest", describe(iShard.remoteDest)) .detail("SrcID", iShard.srcId) .detail("DestID", iShard.destId) .detail("CustomReplicas", customReplicas) .detail("StorageTeamSize", self->configuration.storageTeamSize) .detail("Unhealthy", unhealthy) .detail("Overreplicated", overreplicatedCount);
															#line 3836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(keys, teams);
															#line 852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if ((ddLargeTeamEnabled() && (unhealthy || r > 0)) || (iShard.hasDest && iShard.destId == anonymousShardId))
															#line 3842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				DataMovementReason reason = DataMovementReason::RECOVER_MOVE;
															#line 858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (unhealthy)
															#line 3848 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 859 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					reason = DataMovementReason::TEAM_UNHEALTHY;
															#line 3852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 860 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (r > 0)
															#line 3858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						reason = DataMovementReason::SPLIT_SHARD;
															#line 3862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
				}
															#line 863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(RelocateShard(keys, reason, RelocateReason::OTHER));
															#line 3867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::DataDistribution);
															#line 867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 1;
															#line 867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromShardsActor, 0, Void >*>(static_cast<ResumeFromShardsActor*>(this)));
															#line 3879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromShardsActor*>(this)->ActorCallback< ResumeFromShardsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromShardsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool traceShard;
															#line 778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Key> customBoundaries;
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int shard;
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int customBoundary;
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int overreplicatedCount;
															#line 4002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromShards()
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActor final : public Actor<Void>, public ActorCallback< ResumeFromShardsActor, 0, Void >, public FastAllocated<ResumeFromShardsActor>, public ResumeFromShardsActorState<ResumeFromShardsActor> {
															#line 4007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromShardsActor>::operator new;
	using FastAllocated<ResumeFromShardsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15792597540319737088UL, 6347219941067527936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromShardsActor, 0, Void >;
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActor(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 4024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromShardsActorState<ResumeFromShardsActor>(self, traceShard),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17921411212936075776UL, 15009917662088599808UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromShards");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromShardsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromShards( Reference<DataDistributor> const& self, bool const& traceShard ) {
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromShardsActor(self, traceShard));
															#line 4056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// TODO: unit test needed
																#line 4062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromDataMoves()
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromDataMovesActor>
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActorState {
															#line 4068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart),
															#line 874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   it(self->initData->dataMoveMap.ranges().begin())
															#line 4079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromDataMovesActorState() 
	{
		fdb_probe_actor_destroy("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 1;
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 0, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 4217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->addActor.send(self->removeDataMoveTombstoneBackground(self));
															#line 4221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromDataMovesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromDataMovesActorState(); static_cast<ResumeFromDataMovesActor*>(this)->destroy(); return 0; }
															#line 4225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromDataMovesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(it != self->initData->dataMoveMap.ranges().end()))
															#line 4244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DataMoveMetaData& meta = it.value()->meta;
															#line 880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		DataMoveType dataMoveType = getDataMoveTypeFromDataMoveId(meta.id);
															#line 881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.ranges.empty())
															#line 4254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "EmptyDataMoveRange", self->ddId).detail("DataMoveMetaData", meta.toString());
															#line 4258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1cont1continue1(loopDepth); // continue
		}
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.bulkLoadTaskState.present())
															#line 4263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.dataMoveId = meta.id;
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.cancelled = true;
															#line 889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->relocationProducer.send(rs);
															#line 892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMove", self->ddId) .detail("Reason", "DDInit") .detail("DataMove", meta.toString());
															#line 4275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		else
		{
															#line 895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (dataMoveType == DataMoveType::LOGICAL_BULKLOAD || dataMoveType == DataMoveType::PHYSICAL_BULKLOAD)
															#line 4282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.dataMoveId = meta.id;
															#line 900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.cancelled = true;
															#line 901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(rs);
															#line 902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMoveForWrongType", self->ddId) .detail("Reason", "WrongTypeWhenDDInit") .detail("DataMoveType", dataMoveType) .detail("DataMove", meta.toString());
															#line 4294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1cont1loopBody1cont5(loopDepth);
			}
			else
			{
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (it.value()->isCancelled() || (it.value()->valid && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA))
															#line 4301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.dataMoveId = meta.id;
															#line 910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.cancelled = true;
															#line 911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->relocationProducer.send(rs);
															#line 912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent("DDInitScheduledCancelDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 4313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = a_body1cont1loopBody1cont7(loopDepth);
				}
				else
				{
															#line 913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (it.value()->valid)
															#line 4320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						TraceEvent(SevDebug, "DDInitFoundDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						ASSERT(meta.ranges.front() == it.range());
															#line 917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMoveId = meta.id;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMove = it.value();
															#line 920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->primaryDest, true));
															#line 922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (!rs.dataMove->remoteDest.empty())
															#line 4338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->remoteDest, false));
															#line 4342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						}
															#line 928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->restartShardTracker.send(rs.keys);
															#line 929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->defineShard(rs.keys);
															#line 933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->moveShard(rs.keys, teams);
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->relocationProducer.send(rs);
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						StrictFuture<Void> __when_expr_1 = yield(TaskPriority::DataDistribution);
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
						static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 2;
															#line 935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 1, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont1loopBody1cont9(loopDepth);
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1continue1(int loopDepth) 
	{
															#line 878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRangeMap<std::shared_ptr<DataMove>>::iterator it;
															#line 4516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromDataMoves()
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActor final : public Actor<Void>, public ActorCallback< ResumeFromDataMovesActor, 0, Void >, public ActorCallback< ResumeFromDataMovesActor, 1, Void >, public FastAllocated<ResumeFromDataMovesActor>, public ResumeFromDataMovesActorState<ResumeFromDataMovesActor> {
															#line 4521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromDataMovesActor>::operator new;
	using FastAllocated<ResumeFromDataMovesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1144678587562388480UL, 9354163553398956288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromDataMovesActor, 0, Void >;
friend struct ActorCallback< ResumeFromDataMovesActor, 1, Void >;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 4539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromDataMovesActorState<ResumeFromDataMovesActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929780632836526848UL, 14642014520803479552UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromDataMoves");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromDataMoves( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromDataMovesActor(self, readyToStart));
															#line 4572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Resume inflight relocations from the previous DD
	// TODO: The initialDataDistribution is unused once resumeRelocations,
	// DataDistributionTracker::trackInitialShards, and DDTeamCollection::init are done. In the future, we can release
	// the object to save memory usage if it turns out to be a problem.
	Future<Void> resumeRelocations() {
		ASSERT(shardsAffectedByTeamFailure); // has to be allocated
		Future<Void> shardsReady = resumeFromShards(Reference<DataDistributor>::addRef(this), g_network->isSimulated());
		return resumeFromDataMoves(Reference<DataDistributor>::addRef(this), shardsReady);
	}

	Future<Void> pollMoveKeysLock() const {
		return txnProcessor->pollMoveKeysLock(lock, context->ddEnabledState.get());
	}

	Future<bool> isDataDistributionEnabled() const {
		return txnProcessor->isDataDistributionEnabled(context->ddEnabledState.get());
	}

	Future<Void> removeKeysFromFailedServer(const UID& serverID, const std::vector<UID>& teamForDroppedRange) const {
		return txnProcessor->removeKeysFromFailedServer(
		    serverID, teamForDroppedRange, lock, context->ddEnabledState.get());
	}

	Future<Void> removeStorageServer(const UID& serverID, const Optional<UID>& tssPairID = Optional<UID>()) const {
		return txnProcessor->removeStorageServer(serverID, tssPairID, lock, context->ddEnabledState.get());
	}

	Future<Void> initDDConfigWatch();

	Future<Void> initTenantCache();
};

Future<Void> DataDistributor::initDDConfigWatch() {
	if (txnProcessor->isMocked()) {
		onConfigChange = Never();
		return Void();
	}
	onConfigChange = map(DDConfiguration().trigger.onChange(
	                         SystemDBWriteLockedNow(txnProcessor->context().getReference()), {}, configChangeWatching),
	                     [](Version v) {
		                     CODE_PROBE(true, "DataDistribution change detected");
		                     TraceEvent("DataDistributionConfigChanged").detail("ChangeVersion", v);
		                     throw dd_config_changed();
		                     return Void();
	                     });

	return success(configChangeWatching.getFuture());
}

Future<Void> DataDistributor::initTenantCache() {
	// SOMEDAY: support tenant cache in MockDD
	ASSERT(!txnProcessor->isMocked());
	ddTenantCache = makeReference<TenantCache>(txnProcessor->context(), ddId);
	return ddTenantCache.get()->build();
}

inline void addAuditToAuditMap(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit) {
	AuditType auditType = audit->coreState.getType();
	UID auditID = audit->coreState.id;
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "addAuditToAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(!self->audits[auditType].contains(auditID));
	self->audits[auditType][auditID] = audit;
	return;
}

inline std::shared_ptr<DDAudit> getAuditFromAuditMap(Reference<DataDistributor> self,
                                                     AuditType auditType,
                                                     UID auditID) {
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "getAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	return self->audits[auditType][auditID];
}

inline void removeAuditFromAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	std::shared_ptr<DDAudit> audit = self->audits[auditType][auditID];
	audit->cancel();
	self->audits[auditType].erase(auditID);
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "removeAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	return;
}

inline bool auditExistInAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	return self->audits.contains(auditType) && self->audits[auditType].contains(auditID);
}

inline bool existAuditInAuditMapForType(Reference<DataDistributor> self, AuditType auditType) {
	return self->audits.contains(auditType) && !self->audits[auditType].empty();
}

inline std::unordered_map<UID, std::shared_ptr<DDAudit>> getAuditsForType(Reference<DataDistributor> self,
                                                                          AuditType auditType) {
	ASSERT(self->audits.contains(auditType));
	return self->audits[auditType];
}

// Periodically check and log the physicalShard status; clean up empty physicalShard;
															#line 4683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MonitorPhysicalShardStatusActor>
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActorState {
															#line 4690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActorState(Reference<PhysicalShardCollection> const& self) 
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 4697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	~MonitorPhysicalShardStatusActorState() 
	{
		fdb_probe_actor_destroy("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD);
															#line 1055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorPhysicalShardStatusActorState();
		static_cast<MonitorPhysicalShardStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->cleanUpPhysicalShardCollection();
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->logPhysicalShardCollection();
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->PHYSICAL_SHARD_METRICS_DELAY);
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 1;
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*>(static_cast<MonitorPhysicalShardStatusActor*>(this)));
															#line 4757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state > 0) static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorPhysicalShardStatusActor*>(this)->ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<PhysicalShardCollection> self;
															#line 4851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActor final : public Actor<Void>, public ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >, public FastAllocated<MonitorPhysicalShardStatusActor>, public MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor> {
															#line 4856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator new;
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5301810338833235200UL, 6220587859209929216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >;
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActor(Reference<PhysicalShardCollection> const& self) 
															#line 4873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8017156993063803648UL, 8326163650824365056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorPhysicalShardStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> monitorPhysicalShardStatus( Reference<PhysicalShardCollection> const& self ) {
															#line 1052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MonitorPhysicalShardStatusActor(self));
															#line 4906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// This actor must be a singleton
															#line 4912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via prepareDataMigration()
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PrepareDataMigrationActor>
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActorState {
															#line 4919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActorState(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : req(req),
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   context(context),
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(cx)
															#line 4930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	~PrepareDataMigrationActorState() 
	{
		fdb_probe_actor_destroy("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<Version, Tag>> __when_expr_0 = addStorageServer(cx, req.ssi);
															#line 1068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 1;
															#line 1068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 4953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(context->ddEnabledState->trySetEnabled(req.requesterID));
															#line 1085 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PrepareDataMigrationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrepareDataMigrationActorState(); static_cast<PrepareDataMigrationActor*>(this)->destroy(); return 0; }
															#line 4984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PrepareDataMigrationActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 5001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(e);
															#line 5005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
															#line 1069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
															#line 1069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(verAndTag, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(verAndTag), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareBlobRestoreRequest req;
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDSharedContext> context;
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 5239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via prepareDataMigration()
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActor final : public Actor<Void>, public ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >, public ActorCallback< PrepareDataMigrationActor, 1, Void >, public FastAllocated<PrepareDataMigrationActor>, public PrepareDataMigrationActorState<PrepareDataMigrationActor> {
															#line 5244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PrepareDataMigrationActor>::operator new;
	using FastAllocated<PrepareDataMigrationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13444184362283140096UL, 15852689743265661952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >;
friend struct ActorCallback< PrepareDataMigrationActor, 1, Void >;
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActor(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 5262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PrepareDataMigrationActorState<PrepareDataMigrationActor>(req, context, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12945933996581036800UL, 2706979800331646720UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("prepareDataMigration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> prepareDataMigration( PrepareBlobRestoreRequest const& req, Reference<DDSharedContext> const& context, Database const& cx ) {
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PrepareDataMigrationActor(req, context, cx));
															#line 5296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 5301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ServeBlobMigratorRequestsActor>
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActorState {
															#line 5308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActorState(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tracker(tracker),
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   queue(queue)
															#line 5319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	~ServeBlobMigratorRequestsActorState() 
	{
		fdb_probe_actor_destroy("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->initialized.getFuture();
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5336 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 1;
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ServeBlobMigratorRequestsActorState();
		static_cast<ServeBlobMigratorRequestsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		auto __when_expr_1 = self->context->interface.prepareBlobRestoreReq.getFuture();
															#line 5464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<PrepareBlobRestoreRequest>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<PrepareBlobRestoreRequest>>::value, "invalid type");
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 2;
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
															#line 1094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5486 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
															#line 1094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(req, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(req), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributionTracker> tracker;
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDQueue> queue;
															#line 5649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActor final : public Actor<Void>, public ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >, public ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >, public FastAllocated<ServeBlobMigratorRequestsActor>, public ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor> {
															#line 5654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator new;
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5410873966625105664UL, 6170737959497336064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >;
friend struct ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >;
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActor(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 5672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor>(self, tracker, queue),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13290839411857998080UL, 673675850101827840UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("serveBlobMigratorRequests");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> serveBlobMigratorRequests( Reference<DataDistributor> const& self, Reference<DataDistributionTracker> const& tracker, Reference<DDQueue> const& queue ) {
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ServeBlobMigratorRequestsActor(self, tracker, queue));
															#line 5706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Trigger a task on range based on the current bulk load task metadata
															#line 5712 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class TriggerBulkLoadTaskActor>
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActorState {
															#line 5719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId)
															#line 5730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~TriggerBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1123 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 5745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Database cx = self->txnProcessor->context();
															#line 1125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr = Transaction(cx);
															#line 1126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState = BulkLoadTaskState();
															#line 5779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 1128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistTriggerStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 5820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1148 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1148 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState newBulkLoadTaskState;
															#line 6391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActor final : public Actor<std::pair<BulkLoadTaskState, Version>>, public ActorCallback< TriggerBulkLoadTaskActor, 0, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 1, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 2, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 3, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 4, Void >, public FastAllocated<TriggerBulkLoadTaskActor>, public TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor> {
															#line 6396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<TriggerBulkLoadTaskActor>::operator new;
	using FastAllocated<TriggerBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2672929486027271936UL, 5849023838383258368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BulkLoadTaskState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TriggerBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 4, Void >;
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 6417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<std::pair<BulkLoadTaskState, Version>>(),
		   TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9998265325154424320UL, 14265371869313678848UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("triggerBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<std::pair<BulkLoadTaskState, Version>> triggerBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<std::pair<BulkLoadTaskState, Version>>(new TriggerBulkLoadTaskActor(self, taskRange, taskId));
															#line 6454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// TODO(BulkLoad): add reason to persist
															#line 6460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via failBulkLoadTask()
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FailBulkLoadTaskActor>
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActorState {
															#line 6467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cancelledDataMovePriority(cancelledDataMovePriority),
															#line 1172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState()
															#line 6486 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~FailBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 6501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FailBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FailBulkLoadTaskActorState(); static_cast<FailBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FailBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 0, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 6590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistErrorStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 4, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int cancelledDataMovePriority;
															#line 1172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 7151 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via failBulkLoadTask()
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< FailBulkLoadTaskActor, 0, Void >, public ActorCallback< FailBulkLoadTaskActor, 1, Void >, public ActorCallback< FailBulkLoadTaskActor, 2, Void >, public ActorCallback< FailBulkLoadTaskActor, 3, Void >, public ActorCallback< FailBulkLoadTaskActor, 4, Void >, public FastAllocated<FailBulkLoadTaskActor>, public FailBulkLoadTaskActorState<FailBulkLoadTaskActor> {
															#line 7156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FailBulkLoadTaskActor>::operator new;
	using FastAllocated<FailBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9093224181926140416UL, 10242658219310788096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FailBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 4, Void >;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 7177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FailBulkLoadTaskActorState<FailBulkLoadTaskActor>(self, taskRange, taskId, cancelledDataMovePriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5276853651913060352UL, 273265509463260672UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("failBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> failBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId, int const& cancelledDataMovePriority ) {
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FailBulkLoadTaskActor(self, taskRange, taskId, cancelledDataMovePriority));
															#line 7214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulk load task is guaranteed to be either complete or overwritten by another task
// When a bulk load task is trigged, the range traffic is turned off atomically
// If the task completes, the task re-enables the traffic atomically
															#line 7222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via doBulkLoadTask()
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DoBulkLoadTaskActor>
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActorState {
															#line 7229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   completeAck(),
															#line 1213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   triggeredBulkLoadTask(),
															#line 1214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   commitVersion(invalidVersion),
															#line 1215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 7248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~DoBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadEngineParallelismLimitor.canStart());
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadEngineParallelismLimitor.incrementTaskCounter();
															#line 7265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<BulkLoadTaskState, Version>> __when_expr_0 = triggerBulkLoadTask(self, range, taskId);
															#line 1220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadEngineParallelismLimitor.decrementTaskCounter();
															#line 1299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DoBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBulkLoadTaskActorState(); static_cast<DoBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DoBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1284 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 7320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1285 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(e) .detail("Phase", "Error") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 7330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(triggeredBulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(triggeredBulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ack.unretrievableError)
															#line 7481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "See unretrievable error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_2 = failBulkLoadTask(self, range, taskId, ack.dataMovePriority);
															#line 1252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 7491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 2, Void >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont3Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Task complete") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck const& __ack,int loopDepth) 
	{
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ack = __ack;
															#line 7519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck && __ack,int loopDepth) 
	{
		ack = std::move(__ack);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& failTaskError,int loopDepth=0) 
	{
		try {
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_actor_cancelled)
															#line 7611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(failTaskError) .detail("Phase", "Failed to mark task error. Error should be bulkload_task_outdated") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_movekeys_conflict)
															#line 7621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(failTaskError.code() == error_code_bulkload_task_outdated);
															#line 7629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<BulkLoadAck> completeAck;
															#line 1213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState triggeredBulkLoadTask;
															#line 1214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Version commitVersion;
															#line 1215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadAck ack;
															#line 7775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via doBulkLoadTask()
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >, public ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >, public ActorCallback< DoBulkLoadTaskActor, 2, Void >, public FastAllocated<DoBulkLoadTaskActor>, public DoBulkLoadTaskActorState<DoBulkLoadTaskActor> {
															#line 7780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DoBulkLoadTaskActor>::operator new;
	using FastAllocated<DoBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10416721834491933952UL, 17915870509640622848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >;
friend struct ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >;
friend struct ActorCallback< DoBulkLoadTaskActor, 2, Void >;
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 7799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DoBulkLoadTaskActorState<DoBulkLoadTaskActor>(self, range, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9618644142710713856UL, 6256719351634756608UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> doBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& taskId ) {
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DoBulkLoadTaskActor(self, range, taskId));
															#line 7834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class EraseBulkLoadTaskActor>
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActorState {
															#line 7846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask()
															#line 7863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~EraseBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<EraseBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EraseBulkLoadTaskActorState(); static_cast<EraseBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<EraseBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(bulkLoadTask, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Acknowledged }));
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 0, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskEraseStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 7975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 3, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 8435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< EraseBulkLoadTaskActor, 0, Void >, public ActorCallback< EraseBulkLoadTaskActor, 1, Void >, public ActorCallback< EraseBulkLoadTaskActor, 2, Void >, public ActorCallback< EraseBulkLoadTaskActor, 3, Void >, public FastAllocated<EraseBulkLoadTaskActor>, public EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor> {
															#line 8440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<EraseBulkLoadTaskActor>::operator new;
	using FastAllocated<EraseBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12033003604358617600UL, 12824559098665684224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EraseBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 3, Void >;
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 8460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(445300323391651072UL, 14768580555042305792UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("eraseBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> eraseBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new EraseBulkLoadTaskActor(self, taskRange, taskId));
															#line 8496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 8501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadTasksActor>
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActorState {
															#line 8508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActorState(Reference<DataDistributor> const& self) 
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginKey(normalKeys.begin),
															#line 1351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   endKey(normalKeys.end),
															#line 1352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeToRead(),
															#line 1353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   i(0),
															#line 1356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   result(),
															#line 1358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadActors(),
															#line 1359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range()
															#line 8535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadTasksActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = waitForAll(bulkLoadActors);
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 7;
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 8596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			result.clear();
															#line 1364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store( result, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead, SERVER_KNOBS->DD_BULKLOAD_TASK_METADATA_READ_SIZE));
															#line 1364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 1;
															#line 1364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 8649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 8653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 6;
															#line 1418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1413 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = result.back().key;
															#line 8776 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < result.size() - 1))
															#line 8792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (result[i].value.empty())
															#line 8798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3continue1(loopDepth); // continue
		}
															#line 1372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		range = Standalone(KeyRangeRef(result[i].key, result[i + 1].key));
															#line 1373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[i].value);
															#line 1374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 8808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 1375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 2;
															#line 1375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3continue1(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (range != bulkLoadTaskState.getRange())
															#line 8855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.1);
															#line 1380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 3;
															#line 1380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont6(int loopDepth) 
	{
															#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.phase == BulkLoadPhase::Submitted || bulkLoadTaskState.phase == BulkLoadPhase::Triggered || bulkLoadTaskState.phase == BulkLoadPhase::Running)
															#line 8967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3loopBody1cont6loopHead1(loopDepth);
		}
		else
		{
															#line 1399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadTaskState.phase == BulkLoadPhase::Acknowledged)
															#line 8978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskClearMetadata", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_4 = eraseBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId());
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
				static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 5;
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (bulkLoadTaskState.phase == BulkLoadPhase::Error)
															#line 8998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevWarnAlways, "DDBulkLoadTaskUnretriableError", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 9002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 1410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(bulkLoadTaskState.phase == BulkLoadPhase::Complete);
															#line 9008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3loopBody1cont9(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 9107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(int loopDepth) 
	{
															#line 1393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSchedule", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("Phase", bulkLoadTaskState.phase);
															#line 1397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadActors.push_back( doBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId()));
															#line 9118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadEngineParallelismLimitor.canStart())
															#line 9134 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->bulkLoadEngineParallelismLimitor.waitUntilCounterChanged();
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont6loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 4;
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 9147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult result;
															#line 1358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> bulkLoadActors;
															#line 1359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 9566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >, public FastAllocated<ScheduleBulkLoadTasksActor>, public ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor> {
															#line 9571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator new;
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15823755891477853952UL, 7531093034296865536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >;
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActor(Reference<DataDistributor> const& self) 
															#line 9594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14856297450692458752UL, 7460955458194361088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadTasks");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self ) {
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadTasksActor(self));
															#line 9633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 9638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadTaskCoreActor>
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActorState {
															#line 9645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 9654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadTaskCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 1426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 1;
															#line 1426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 0, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadTaskCoreActorState();
		static_cast<BulkLoadTaskCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = scheduleBulkLoadTasks(self);
															#line 1432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 3;
															#line 1432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 2, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 1442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 4;
															#line 1442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 3, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadTaskCoreError", self->ddId).errorUnsuppressed(e);
															#line 1438 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 9964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 1427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 10173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActor final : public Actor<Void>, public ActorCallback< BulkLoadTaskCoreActor, 0, Void >, public ActorCallback< BulkLoadTaskCoreActor, 1, Void >, public ActorCallback< BulkLoadTaskCoreActor, 2, Void >, public ActorCallback< BulkLoadTaskCoreActor, 3, Void >, public FastAllocated<BulkLoadTaskCoreActor>, public BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor> {
															#line 10178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadTaskCoreActor>::operator new;
	using FastAllocated<BulkLoadTaskCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8357674444347790336UL, 6088104955584429312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadTaskCoreActor, 0, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 1, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 2, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 3, Void >;
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 10198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12210363843676168704UL, 8182787864321598208UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadTaskCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadTaskCore( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadTaskCoreActor(self, readyToStart));
															#line 10234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadJob()
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadJobActor>
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActorState {
															#line 10246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadJobActor, 0, Void >*>(static_cast<CheckBulkLoadJobActor*>(this)));
															#line 10279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1453 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1455 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1453 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1455 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadJobActor*>(this)->ActorCallback< CheckBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadJob()
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActor final : public Actor<Void>, public ActorCallback< CheckBulkLoadJobActor, 0, Void >, public FastAllocated<CheckBulkLoadJobActor>, public CheckBulkLoadJobActorState<CheckBulkLoadJobActor> {
															#line 10468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadJobActor>::operator new;
	using FastAllocated<CheckBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4673233018438503936UL, 16276302004461967872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadJobActor, 0, Void >;
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   CheckBulkLoadJobActorState<CheckBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6430116811539734272UL, 4354220408088791808UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> checkBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new CheckBulkLoadJobActor(tr, jobId));
															#line 10518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via getBulkLoadJob()
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class GetBulkLoadJobActor>
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActorState {
															#line 10530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~GetBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBulkLoadJobActor, 0, Void >*>(static_cast<GetBulkLoadJobActor*>(this)));
															#line 10563 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<GetBulkLoadJobActor*>(this)->ActorCallback< GetBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via getBulkLoadJob()
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActor final : public Actor<BulkLoadJobState>, public ActorCallback< GetBulkLoadJobActor, 0, Void >, public FastAllocated<GetBulkLoadJobActor>, public GetBulkLoadJobActorState<GetBulkLoadJobActor> {
															#line 10752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<GetBulkLoadJobActor>::operator new;
	using FastAllocated<GetBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9112178853689254912UL, 1259415344281533696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadJobState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBulkLoadJobActor, 0, Void >;
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadJobState>(),
		   GetBulkLoadJobActorState<GetBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17307579102949951488UL, 17075247084345564672UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadJobState> getBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadJobState>(new GetBulkLoadJobActor(tr, jobId));
															#line 10802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Find task metadata for a bulk load job with jobId and input range
															#line 10808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobFindTaskActor>
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActorState {
															#line 10815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   logId(logId),
															#line 1489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 10834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobFindTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 10849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(bulkLoadTaskState); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(std::move(bulkLoadTaskState)); // state_var_RVO
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobId);
															#line 1495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 1;
															#line 1495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 3;
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& result,int loopDepth) 
	{
															#line 1499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && result,int loopDepth) 
	{
															#line 1499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID logId;
															#line 1489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 11313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActor final : public Actor<Optional<BulkLoadTaskState>>, public ActorCallback< BulkLoadJobFindTaskActor, 0, Void >, public ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >, public ActorCallback< BulkLoadJobFindTaskActor, 2, Void >, public FastAllocated<BulkLoadJobFindTaskActor>, public BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor> {
															#line 11318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobFindTaskActor>::operator new;
	using FastAllocated<BulkLoadJobFindTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8590569849630314752UL, 1647906308709256960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<BulkLoadTaskState>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobFindTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 2, Void >;
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 11337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Optional<BulkLoadTaskState>>(),
		   BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor>(self, range, jobId, logId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3995601487305067776UL, 1783749224615820032UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobFindTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Optional<BulkLoadTaskState>> bulkLoadJobFindTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& jobId, UID const& logId ) {
															#line 1485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Optional<BulkLoadTaskState>>(new BulkLoadJobFindTaskActor(self, range, jobId, logId));
															#line 11372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Submit a bulkload task for the given jobId
															#line 11378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobSubmitTaskActor>
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActorState {
															#line 11385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(manifests),
															#line 1528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(jobId, manifests)
															#line 11402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobSubmitTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 11417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV<BulkLoadTaskState>::futures) { (void)(bulkLoadTask); this->~BulkLoadJobSubmitTaskActorState(); static_cast<BulkLoadJobSubmitTaskActor*>(this)->destroy(); return 0; }
															#line 11440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV< BulkLoadTaskState >::value()) BulkLoadTaskState(std::move(bulkLoadTask)); // state_var_RVO
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 1;
															#line 1535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 4;
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11526 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11547 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 11930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActor final : public Actor<BulkLoadTaskState>, public ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >, public FastAllocated<BulkLoadJobSubmitTaskActor>, public BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor> {
															#line 11935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator new;
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15578561993400140032UL, 5459391786865018368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadTaskState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >;
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 11955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadTaskState>(),
		   BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor>(self, jobId, manifests),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9698732467664213248UL, 5402256740808584704UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobSubmitTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadTaskState> bulkLoadJobSubmitTask( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadManifestSet const& manifests ) {
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadTaskState>(new BulkLoadJobSubmitTaskActor(self, jobId, manifests));
															#line 11991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// If the task reaches unretrievable error, return true; otherwise, return false.
															#line 11997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobWaitUntilTaskCompleteOrErrorActor>
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActorState {
															#line 12004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(bulkLoadTask)
															#line 12015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(bulkLoadTask.isValid());
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentTask = BulkLoadTaskState();
															#line 1563 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 12038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentTask, getBulkLoadTask(&tr, bulkLoadTask.getRange(), bulkLoadTask.getTaskId(), { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running, BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }));
															#line 1565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_JOB_MONITOR_PERIOD_SEC);
															#line 1592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12115 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12134 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState currentTask;
															#line 12480 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActor final : public Actor<Void>, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>, public BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor> {
															#line 12485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17121605347527014144UL, 11215134223756778240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >;
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 12504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>(self, jobId, bulkLoadTask),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16093638368250768128UL, 12615896230081500416UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobWaitUntilTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobWaitUntilTaskCompleteOrError( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadTaskState const& bulkLoadTask ) {
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobWaitUntilTaskCompleteOrErrorActor(self, jobId, bulkLoadTask));
															#line 12539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a list of manifestEntry, create a bulkload task.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 12547 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobNewTaskActor>
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActorState {
															#line 12554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRoot(jobRoot),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestLocalTempFolder(manifestLocalTempFolder),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntries(manifestEntries),
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(),
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 12581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobNewTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!manifestEntries.empty());
															#line 12596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(manifests, getBulkLoadManifestMetadataFromEntry( manifestEntries, manifestLocalTempFolder, jobTransportMethod, jobRoot, self->ddId));
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12602 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 1;
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 12653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 12659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 12663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string manifestLocalTempFolder;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobNewTaskActor, 0, Void >, public ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobNewTaskActor, 2, Void >, public FastAllocated<BulkLoadJobNewTaskActor>, public BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor> {
															#line 13068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobNewTaskActor>::operator new;
	using FastAllocated<BulkLoadJobNewTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12542373784865013504UL, 17667374456378049536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobNewTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 2, Void >;
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActor(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 13087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor>(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8655289167937893888UL, 2026549218517418240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobNewTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobNewTask( Reference<DataDistributor> const& self, UID const& jobId, std::string const& jobRoot, BulkLoadTransportMethod const& jobTransportMethod, std::string const& manifestLocalTempFolder, std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries, Promise<Void> const& errorOut ) {
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobNewTaskActor(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut));
															#line 13122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a bulkload task range, find the task and wait until the task is complete or error.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 13130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobMonitorTaskActor>
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActorState {
															#line 13137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 13156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobMonitorTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadParallelismLimitor.canStart());
															#line 1675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.incrementTaskCounter();
															#line 13173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Optional<BulkLoadTaskState>> __when_expr_0 = bulkLoadJobFindTask(self, taskRange, jobId, self->ddId);
															#line 1678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 1;
															#line 1678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 13230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskMonitorError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 13238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 13242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13288 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >, public FastAllocated<BulkLoadJobMonitorTaskActor>, public BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor> {
															#line 13546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator new;
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16683151934654658560UL, 14641439290897120000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >;
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActor(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 13564 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor>(self, jobId, taskRange, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3763012194873441280UL, 18076434267272298240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobMonitorTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobMonitorTask( Reference<DataDistributor> const& self, UID const& jobId, KeyRange const& taskRange, Promise<Void> const& errorOut ) {
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobMonitorTaskActor(self, jobId, taskRange, errorOut));
															#line 13598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 13603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PersistBulkLoadJobTaskCountActor>
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActorState {
															#line 13610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActorState(Reference<DataDistributor> const& self) 
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 13621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	~PersistBulkLoadJobTaskCountActorState() 
	{
		fdb_probe_actor_destroy("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1728 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = jobState.getJobId();
															#line 1731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = jobState.getJobRange();
															#line 1732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(jobState.getTaskCount().present());
															#line 1733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			taskCount = jobState.getTaskCount().get();
															#line 1734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentJobState = BulkLoadJobState();
															#line 1735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 13650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 1;
															#line 1737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 4;
															#line 1761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 14021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 14032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	uint64_t taskCount;
															#line 1734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 14215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActor final : public Actor<Void>, public ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >, public FastAllocated<PersistBulkLoadJobTaskCountActor>, public PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor> {
															#line 14220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator new;
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15204873911993204224UL, 17041499047204954624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >;
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActor(Reference<DataDistributor> const& self) 
															#line 14240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10651365252717723648UL, 4800485421425282816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistBulkLoadJobTaskCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> persistBulkLoadJobTaskCount( Reference<DataDistributor> const& self ) {
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PersistBulkLoadJobTaskCountActor(self));
															#line 14276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Remove the bulkload job metadata from the range map metadata.
// Add the job metadata to the history metadata with the error message.
															#line 14283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MoveErrorBulkLoadJobToHistoryActor>
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActorState {
															#line 14290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActorState(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage(errorMessage),
															#line 1770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentJobState()
															#line 14305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	~MoveErrorBulkLoadJobToHistoryActorState() 
	{
		fdb_probe_actor_destroy("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = self->bulkLoadJobManager.get().jobState.getJobId();
															#line 1775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 14324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveErrorBulkLoadJobToHistoryActorState(); static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->destroy(); return 0; }
															#line 14347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 1;
															#line 1777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14438 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14762 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 1770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 1774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 15049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActor final : public Actor<Void>, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >, public FastAllocated<MoveErrorBulkLoadJobToHistoryActor>, public MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor> {
															#line 15054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator new;
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1836476354631573248UL, 7498283660366949888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >;
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActor(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 15076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor>(self, errorMessage),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10076677940119608320UL, 17212560479686694656UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("moveErrorBulkLoadJobToHistory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> moveErrorBulkLoadJobToHistory( Reference<DataDistributor> const& self, std::string const& errorMessage ) {
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MoveErrorBulkLoadJobToHistoryActor(self, errorMessage));
															#line 15114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Download the job manifest file from the remoteJobManifestFilePath to the localJobManifestFilePath.
// Build the bulkload manifest range map based on the localJobManifestFilePath file content.
// For any failure, we mark the job metadata as error and move the metadata to the history.
															#line 15122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FetchBulkLoadTaskManifestEntryMapActor>
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActorState {
															#line 15129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActorState(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   localJobManifestFilePath(localJobManifestFilePath),
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   remoteJobManifestFilePath(remoteJobManifestFilePath)
															#line 15142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	~FetchBulkLoadTaskManifestEntryMapActorState() 
	{
		fdb_probe_actor_destroy("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1800 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid() && self->bulkLoadJobManager.get().manifestEntryMap->empty());
															#line 1802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginTime = now();
															#line 1803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 15161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!fileExists(abspath(localJobManifestFilePath)))
															#line 15165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDBulkLoadJobManagerDownloadJobManifest", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath);
															#line 1810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					StrictFuture<Void> __when_expr_0 = downloadBulkLoadJobManifestFile( jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath, self->ddId);
															#line 1810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 1;
															#line 1810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBulkLoadTaskManifestEntryMapActorState(); static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->destroy(); return 0; }
															#line 15212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 15225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 15229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			err = e;
															#line 1838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string errorMessage = "Failed to build job-manifest map with error code " + std::to_string(err.code()) + ". The remote file path is " + remoteJobManifestFilePath + ". The local file path is " + localJobManifestFilePath + ". The transport method is " + convertBulkLoadTransportMethodToString(jobTransportMethod) + ".";
															#line 1849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = moveErrorBulkLoadJobToHistory(self, errorMessage);
															#line 1849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 3;
															#line 1849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = getBulkLoadJobFileManifestEntryFromJobManifestFile( localJobManifestFilePath, jobRange, self->ddId, self->bulkLoadJobManager.get().manifestEntryMap);
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 2;
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2cont1(Void const& _,int loopDepth) 
	{
															#line 1850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont1(Void && _,int loopDepth) 
	{
															#line 1850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string localJobManifestFilePath;
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string remoteJobManifestFilePath;
															#line 1802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Error err;
															#line 15585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActor final : public Actor<Void>, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >, public FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>, public FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor> {
															#line 15590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator new;
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16796091164750673920UL, 424488081484819200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >;
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActor(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 15609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor>(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7119443845485172480UL, 13876208275314761984UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBulkLoadTaskManifestEntryMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> fetchBulkLoadTaskManifestEntryMap( Reference<DataDistributor> const& self, BulkLoadTransportMethod const& jobTransportMethod, std::string const& localJobManifestFilePath, std::string const& remoteJobManifestFilePath ) {
															#line 1796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FetchBulkLoadTaskManifestEntryMapActor(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath));
															#line 15644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1860 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 15649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadJobActor>
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActorState {
															#line 15656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActorState(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntry()
															#line 15667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors = std::vector<Future<Void>>();
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 15694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(actors);
															#line 1957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 1957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, jobState.getJobRange().end));
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15743 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 1954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		res = __res;
															#line 15816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (beginKey == jobState.getJobRange().end)
															#line 15895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.get().allTaskSubmitted = true;
															#line 15899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < res.size() - 1))
															#line 15917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!res[i].value.empty())
															#line 15923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			task = decodeBulkLoadTaskState(res[i].value);
															#line 1883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (task.isValid())
															#line 15929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.getJobId() != self->bulkLoadJobManager.get().jobState.getJobId())
															#line 15933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					return a_body1loopBody1Catch1(bulkload_task_outdated(), std::max(0, loopDepth - 1));
															#line 15937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
															#line 1888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(self->bulkLoadJobManager.get().manifestEntryMap->find(task.getRange().begin) != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.onAnyPhase( { BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }))
															#line 15943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(task.getRange().end == res[i + 1].key);
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					beginKey = task.getRange().end;
															#line 15949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					return a_body1loopBody1cont2continue1(loopDepth); // continue
				}
				else
				{
															#line 1896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (task.onAnyPhase({ BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }))
															#line 15956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 1901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (self->bulkLoadJobManager.get().allTaskSubmitted)
															#line 15960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 1903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							;
															#line 15964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
							loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
						}
						else
						{
							loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);
						}
					}
					else
					{
															#line 1917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						UNREACHABLE();
															#line 15976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = a_body1loopBody1cont2loopBody1cont7(loopDepth);
					}
				}
			}
			else
			{
				loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2continue1(int loopDepth) 
	{
															#line 1877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i].key);
															#line 1926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont8(int loopDepth) 
	{
															#line 1913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(task.getRange().end == res[i + 1].key);
															#line 1914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = task.getRange().end;
															#line 16050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont2continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont9(int loopDepth) 
	{
															#line 1910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back( bulkLoadJobMonitorTask(self, task.getJobId(), task.getRange(), errorOut));
															#line 16059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 1904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadParallelismLimitor.canStart())
															#line 16075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->bulkLoadParallelismLimitor.waitUntilCounterChanged();
															#line 1907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 1907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2loopBody1cont13(int loopDepth) 
	{
															#line 1946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i + 1].key);
															#line 1877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < res[i + 1].key))
															#line 16215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;manifestEntries.size() < SERVER_KNOBS->MANIFEST_COUNT_MAX_PER_BULKLOAD_TASK && beginKey < res[i + 1].key;) {
															#line 1930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->bulkLoadJobManager.get().manifestEntryMap->find(beginKey);
															#line 1931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(it != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntry = it->second;
															#line 1933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntries.push_back(manifestEntry);
															#line 1934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = manifestEntry.getEndKey();
															#line 16233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!manifestEntries.empty());
															#line 1937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobNewTask(self, jobState.getJobId(), jobState.getJobRoot(), jobState.getTransportMethod(), self->bulkLoadJobManager.get().manifestLocalTempFolder, manifestEntries, errorOut));
															#line 1944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_TASK_SUBMISSION_INTERVAL_SEC);
															#line 1944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 1944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFileManifestEntry manifestEntry;
															#line 1864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> actors;
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult res;
															#line 1876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState task;
															#line 16574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >, public ActorCallback< ScheduleBulkLoadJobActor, 1, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 2, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 3, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 4, Void >, public FastAllocated<ScheduleBulkLoadJobActor>, public ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor> {
															#line 16579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadJobActor>::operator new;
	using FastAllocated<ScheduleBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1857104814440181248UL, 18378498564117568768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 4, Void >;
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActor(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 16600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor>(self, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4684338048225246976UL, 14035353769110992896UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadJob( Reference<DataDistributor> const& self, Promise<Void> const& errorOut ) {
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadJobActor(self, errorOut));
															#line 16637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 16642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadTaskCompleteOrErrorActor>
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActorState {
															#line 16649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self) 
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 16660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 1968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 1969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = KeyRange();
															#line 1970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 1971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 16689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16712 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(true);
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 16731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 1974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobState.getJobId());
															#line 1975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16749 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 2013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 17002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 17006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 17011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 17017 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 17026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 1969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 17224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActor final : public Actor<bool>, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>, public CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor> {
															#line 17229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16498808218316919808UL, 4543740576974717952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >;
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActor(Reference<DataDistributor> const& self) 
															#line 17248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<bool>(),
		   CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13835388531052272896UL, 2426800946393579264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<bool> checkBulkLoadTaskCompleteOrError( Reference<DataDistributor> const& self ) {
															#line 1961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<bool>(new CheckBulkLoadTaskCompleteOrErrorActor(self));
															#line 17283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 17288 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FinalizeBulkLoadJobActor>
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActorState {
															#line 17295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActorState(Reference<DataDistributor> const& self) 
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 2021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 17306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~FinalizeBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2022 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 2025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 2026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = Optional<Key>();
															#line 2027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 2028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobCompleteRange = KeyRange();
															#line 2030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = false;
															#line 2031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			i = 0;
															#line 2032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			allFinish = false;
															#line 2033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 17343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FinalizeBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinalizeBulkLoadJobActorState(); static_cast<FinalizeBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 17366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FinalizeBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 17385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.reset();
															#line 2036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId());
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 2109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 9;
															#line 2109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
															#line 2038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && currentJobState,int loopDepth) 
	{
															#line 2038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(currentJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(currentJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 2079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (hasError)
															#line 17668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setErrorPhase("BulkLoadTask Error.");
															#line 17672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 2082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setCompletePhase();
															#line 17678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 2084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(lastKey.present());
															#line 2085 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobCompleteRange = KeyRangeRef(jobState.getJobRange().begin, lastKey.get());
															#line 2086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 2086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 2086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < bulkLoadTaskResult.size() - 1))
															#line 17709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 2043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 2044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!existTask.isValid())
															#line 17719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1loopBody1cont4continue1(loopDepth); // continue
		}
															#line 2048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getRange() == KeyRangeRef(bulkLoadTaskResult[i].key, bulkLoadTaskResult[i + 1].key));
															#line 2053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getJobId() == jobState.getJobId());
															#line 2054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (existTask.phase == BulkLoadPhase::Error)
															#line 17732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerStopClearMetadata", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = true;
															#line 17738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);
		}
		else
		{
															#line 2064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete && existTask.phase != BulkLoadPhase::Acknowledged)
															#line 17745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerWrongTaskPhase", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 17751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = setBulkLoadFinalizeTransaction(&tr, existTask.getRange(), existTask.getTaskId());
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17762 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4continue1(int loopDepth) 
	{
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 17785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
															#line 2076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr.commit();
															#line 2096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_7.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18066 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast