#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
/*
 * DataDistributionQueue.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <limits>
#include <numeric>
#include <utility>
#include <vector>

#include "fdbserver/DataDistributionTeam.h"
#include "flow/ActorCollection.h"
#include "flow/Buggify.h"
#include "flow/FastRef.h"
#include "flow/Trace.h"
#include "fdbrpc/sim_validation.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbrpc/simulator.h"
#include "fdbserver/DDTxnProcessor.h"
#include "flow/DebugTrace.h"
#include "fdbserver/DDRelocationQueue.h"
#include "flow/actorcompiler.h" // This must be the last #include.

#define WORK_FULL_UTILIZATION 10000 // This is not a knob; it is a fixed point scaling factor!

typedef Reference<IDataDistributionTeam> ITeamRef;
typedef std::pair<ITeamRef, ITeamRef> SrcDestTeamPair;

inline bool isDataMovementForDiskBalancing(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_OVERUTILIZED_TEAM;
}

inline bool isDataMovementForReadBalancing(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM;
}

inline bool isDataMovementForMountainChopper(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_OVERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM;
}

// FIXME: Always use DataMovementReason to invoke these functions.
inline bool isValleyFillerPriority(int priority) {
	return priority == SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM ||
	       priority == SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM;
}

inline bool isDataMovementForValleyFiller(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM;
}

typedef std::map<DataMovementReason, int> DmReasonPriorityMapping;
typedef std::map<int, DataMovementReason> PriorityDmReasonMapping;
std::pair<const DmReasonPriorityMapping*, const PriorityDmReasonMapping*> buildPriorityMappings() {
	static DmReasonPriorityMapping reasonPriority{
		{ DataMovementReason::INVALID, -1 },
		{ DataMovementReason::RECOVER_MOVE, SERVER_KNOBS->PRIORITY_RECOVER_MOVE },
		{ DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM },
		{ DataMovementReason::REBALANCE_OVERUTILIZED_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM },
		{ DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM },
		{ DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM },
		{ DataMovementReason::PERPETUAL_STORAGE_WIGGLE, SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE },
		{ DataMovementReason::TEAM_HEALTHY, SERVER_KNOBS->PRIORITY_TEAM_HEALTHY },
		{ DataMovementReason::TEAM_CONTAINS_UNDESIRED_SERVER, SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER },
		{ DataMovementReason::TEAM_REDUNDANT, SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT },
		{ DataMovementReason::MERGE_SHARD, SERVER_KNOBS->PRIORITY_MERGE_SHARD },
		{ DataMovementReason::POPULATE_REGION, SERVER_KNOBS->PRIORITY_POPULATE_REGION },
		{ DataMovementReason::TEAM_UNHEALTHY, SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY },
		{ DataMovementReason::TEAM_2_LEFT, SERVER_KNOBS->PRIORITY_TEAM_2_LEFT },
		{ DataMovementReason::TEAM_1_LEFT, SERVER_KNOBS->PRIORITY_TEAM_1_LEFT },
		{ DataMovementReason::TEAM_FAILED, SERVER_KNOBS->PRIORITY_TEAM_FAILED },
		{ DataMovementReason::TEAM_0_LEFT, SERVER_KNOBS->PRIORITY_TEAM_0_LEFT },
		{ DataMovementReason::SPLIT_SHARD, SERVER_KNOBS->PRIORITY_SPLIT_SHARD },
		{ DataMovementReason::ENFORCE_MOVE_OUT_OF_PHYSICAL_SHARD,
		  SERVER_KNOBS->PRIORITY_ENFORCE_MOVE_OUT_OF_PHYSICAL_SHARD },
		{ DataMovementReason::REBALANCE_STORAGE_QUEUE, SERVER_KNOBS->PRIORITY_REBALANCE_STORAGE_QUEUE },
		{ DataMovementReason::ASSIGN_EMPTY_RANGE, -2 }, // dummy reason, no corresponding actual data move
		{ DataMovementReason::SEED_SHARD_SERVER, -3 }, // dummy reason, no corresponding actual data move
		{ DataMovementReason::NUMBER_OF_REASONS, -4 }, // dummy reason, no corresponding actual data move
	};

	static PriorityDmReasonMapping priorityReason;
	if (priorityReason.empty()) { // only build once
		for (const auto& [r, p] : reasonPriority) {
			priorityReason[p] = r;
		}
		// Don't allow 2 priorities value being the same.
		if (priorityReason.size() != reasonPriority.size()) {
			TraceEvent(SevError, "DuplicateDataMovementPriority").log();
			ASSERT(false);
		}
	}

	return std::make_pair(&reasonPriority, &priorityReason);
}

// Return negative priority for invalid or dummy reasons
int dataMovementPriority(DataMovementReason reason) {
	auto [reasonPriority, _] = buildPriorityMappings();
	return reasonPriority->at(reason);
}

DataMovementReason priorityToDataMovementReason(int priority) {
	auto [_, priorityReason] = buildPriorityMappings();
	return priorityReason->at(priority);
}

RelocateData::RelocateData()
  : priority(-1), boundaryPriority(-1), healthPriority(-1), reason(RelocateReason::OTHER), startTime(-1),
    dataMoveId(anonymousShardId), workFactor(0), wantsNewServers(false), cancellable(false),
    interval("QueuedRelocation") {};

RelocateData::RelocateData(RelocateShard const& rs)
  : parent_range(rs.getParentRange()), keys(rs.keys), priority(rs.priority),
    boundaryPriority(isBoundaryPriority(rs.priority) ? rs.priority : -1),
    healthPriority(isHealthPriority(rs.priority) ? rs.priority : -1), reason(rs.reason), dmReason(rs.moveReason),
    startTime(now()), randomId(rs.traceId.isValid() ? rs.traceId : deterministicRandom()->randomUniqueID()),
    dataMoveId(rs.dataMoveId), workFactor(0),
    wantsNewServers(isDataMovementForMountainChopper(rs.moveReason) || isDataMovementForValleyFiller(rs.moveReason) ||
                    rs.moveReason == DataMovementReason::SPLIT_SHARD ||
                    rs.moveReason == DataMovementReason::TEAM_REDUNDANT ||
                    rs.moveReason == DataMovementReason::REBALANCE_STORAGE_QUEUE),
    cancellable(true), interval("QueuedRelocation", randomId), dataMove(rs.dataMove) {
	if (dataMove != nullptr) {
		this->src.insert(this->src.end(), dataMove->meta.src.begin(), dataMove->meta.src.end());
	}
}

bool RelocateData::isRestore() const {
	return this->dataMove != nullptr;
}

// Note: C++ standard library uses the Compare operator, uniqueness is determined by !comp(a, b) && !comp(b, a).
// So operator == and != is not used by std::set<RelocateData, std::greater<RelocateData>>
bool RelocateData::operator>(const RelocateData& rhs) const {
	if (priority != rhs.priority) {
		return priority > rhs.priority;
	} else if (startTime != rhs.startTime) {
		return startTime < rhs.startTime;
	} else if (randomId != rhs.randomId) {
		return randomId > rhs.randomId;
	} else if (keys.begin != rhs.keys.begin) {
		return keys.begin < rhs.keys.begin;
	} else {
		return keys.end < rhs.keys.end;
	}
}

bool RelocateData::operator==(const RelocateData& rhs) const {
	return priority == rhs.priority && boundaryPriority == rhs.boundaryPriority &&
	       healthPriority == rhs.healthPriority && reason == rhs.reason && keys == rhs.keys &&
	       startTime == rhs.startTime && workFactor == rhs.workFactor && src == rhs.src &&
	       completeSources == rhs.completeSources && wantsNewServers == rhs.wantsNewServers && randomId == rhs.randomId;
}
bool RelocateData::operator!=(const RelocateData& rhs) const {
	return !(*this == rhs);
}
Optional<KeyRange> RelocateData::getParentRange() const {
	return parent_range;
}

class ParallelTCInfo final : public ReferenceCounted<ParallelTCInfo>, public IDataDistributionTeam {
	std::vector<Reference<IDataDistributionTeam>> teams;
	std::vector<UID> tempServerIDs;

	template <typename NUM>
	NUM sum(std::function<NUM(IDataDistributionTeam const&)> func) const {
		NUM result = 0;
		for (const auto& team : teams) {
			result += func(*team);
		}
		return result;
	}

	template <class T>
	std::vector<T> collect(std::function<std::vector<T>(IDataDistributionTeam const&)> func) const {
		std::vector<T> result;

		for (const auto& team : teams) {
			std::vector<T> newItems = func(*team);
			result.insert(result.end(), newItems.begin(), newItems.end());
		}
		return result;
	}

	bool any(std::function<bool(IDataDistributionTeam const&)> func) const {
		for (const auto& team : teams) {
			if (func(*team)) {
				return true;
			}
		}
		return false;
	}

public:
	ParallelTCInfo() = default;
	explicit ParallelTCInfo(ParallelTCInfo const& info) : teams(info.teams), tempServerIDs(info.tempServerIDs) {};

	void addTeam(Reference<IDataDistributionTeam> team) { teams.push_back(team); }

	void clear() { teams.clear(); }

	bool all(std::function<bool(IDataDistributionTeam const&)> func) const {
		return !any([func](IDataDistributionTeam const& team) { return !func(team); });
	}

	std::vector<StorageServerInterface> getLastKnownServerInterfaces() const override {
		return collect<StorageServerInterface>(
		    [](IDataDistributionTeam const& team) { return team.getLastKnownServerInterfaces(); });
	}

	int size() const override {
		int totalSize = 0;
		for (auto it = teams.begin(); it != teams.end(); it++) {
			totalSize += (*it)->size();
		}
		return totalSize;
	}

	std::vector<UID> const& getServerIDs() const override {
		static std::vector<UID> tempServerIDs;
		tempServerIDs.clear();
		for (const auto& team : teams) {
			std::vector<UID> const& childIDs = team->getServerIDs();
			tempServerIDs.insert(tempServerIDs.end(), childIDs.begin(), childIDs.end());
		}
		return tempServerIDs;
	}

	void addDataInFlightToTeam(int64_t delta) override {
		for (auto& team : teams) {
			team->addDataInFlightToTeam(delta);
		}
	}

	void addReadInFlightToTeam(int64_t delta) override {
		for (auto& team : teams) {
			team->addReadInFlightToTeam(delta);
		}
	}

	int64_t getDataInFlightToTeam() const override {
		return sum<int64_t>([](IDataDistributionTeam const& team) { return team.getDataInFlightToTeam(); });
	}

	int64_t getLoadBytes(bool includeInFlight = true, double inflightPenalty = 1.0) const override {
		return sum<int64_t>([includeInFlight, inflightPenalty](IDataDistributionTeam const& team) {
			return team.getLoadBytes(includeInFlight, inflightPenalty);
		});
	}

	int64_t getReadInFlightToTeam() const override {
		return sum<int64_t>([](IDataDistributionTeam const& team) { return team.getReadInFlightToTeam(); });
	}

	double getReadLoad(bool includeInFlight = true, double inflightPenalty = 1.0) const override {
		return sum<double>([includeInFlight, inflightPenalty](IDataDistributionTeam const& team) {
			return team.getReadLoad(includeInFlight, inflightPenalty);
		});
	}

	double getAverageCPU() const override {
		return sum<double>([](IDataDistributionTeam const& team) { return team.getAverageCPU(); }) / teams.size();
	}

	bool hasLowerCpu(double cpuThreshold) const override {
		return all([cpuThreshold](IDataDistributionTeam const& team) { return team.hasLowerCpu(cpuThreshold); });
	}

	Optional<int64_t> getLongestStorageQueueSize() const override {
		int64_t maxQueueSize = 0;
		for (const auto& team : teams) {
			Optional<int64_t> queueSize = team->getLongestStorageQueueSize();
			if (!queueSize.present()) {
				return Optional<int64_t>();
			}
			maxQueueSize = std::max(maxQueueSize, queueSize.get());
		}
		return maxQueueSize;
	}

	Optional<int> getMaxOngoingBulkLoadTaskCount() const override {
		int maxOngoingBulkLoadTaskCount = 0;
		for (const auto& team : teams) {
			Optional<int> ongoingBulkLoadTaskCount = team->getMaxOngoingBulkLoadTaskCount();
			if (!ongoingBulkLoadTaskCount.present()) {
				// If a SS tracker cannot get the metrics from the SS, it is possible that this SS has some healthy
				// issue. So, return an empty result to avoid choosing this server.
				return Optional<int>();
			}
			maxOngoingBulkLoadTaskCount = std::max(maxOngoingBulkLoadTaskCount, ongoingBulkLoadTaskCount.get());
		}
		return maxOngoingBulkLoadTaskCount;
	}

	int64_t getMinAvailableSpace(bool includeInFlight = true) const override {
		int64_t result = std::numeric_limits<int64_t>::max();
		for (const auto& team : teams) {
			result = std::min(result, team->getMinAvailableSpace(includeInFlight));
		}
		return result;
	}

	double getMinAvailableSpaceRatio(bool includeInFlight = true) const override {
		double result = std::numeric_limits<double>::max();
		for (const auto& team : teams) {
			result = std::min(result, team->getMinAvailableSpaceRatio(includeInFlight));
		}
		return result;
	}

	bool hasHealthyAvailableSpace(double minRatio) const override {
		return all([minRatio](IDataDistributionTeam const& team) { return team.hasHealthyAvailableSpace(minRatio); });
	}

	Future<Void> updateStorageMetrics() override {
		std::vector<Future<Void>> futures;

		for (auto& team : teams) {
			futures.push_back(team->updateStorageMetrics());
		}
		return waitForAll(futures);
	}

	bool isOptimal() const override {
		return all([](IDataDistributionTeam const& team) { return team.isOptimal(); });
	}

	bool isWrongConfiguration() const override {
		return any([](IDataDistributionTeam const& team) { return team.isWrongConfiguration(); });
	}
	void setWrongConfiguration(bool wrongConfiguration) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setWrongConfiguration(wrongConfiguration);
		}
	}

	bool isHealthy() const override {
		return all([](IDataDistributionTeam const& team) { return team.isHealthy(); });
	}

	void setHealthy(bool h) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setHealthy(h);
		}
	}

	int getPriority() const override {
		int priority = 0;
		for (auto it = teams.begin(); it != teams.end(); it++) {
			priority = std::max(priority, (*it)->getPriority());
		}
		return priority;
	}

	void setPriority(int p) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setPriority(p);
		}
	}
	void addref() const override { ReferenceCounted<ParallelTCInfo>::addref(); }
	void delref() const override { ReferenceCounted<ParallelTCInfo>::delref(); }

	void addServers(const std::vector<UID>& servers) override {
		ASSERT(!teams.empty());
		teams[0]->addServers(servers);
	}

	std::string getTeamID() const override {
		std::string id;
		for (int i = 0; i < teams.size(); i++) {
			auto const& team = teams[i];
			id += (i == teams.size() - 1) ? team->getTeamID() : format("%s, ", team->getTeamID().c_str());
		}
		return id;
	}
};

bool Busyness::canLaunch(int prio, int work) const {
	ASSERT(prio > 0 && prio < 1000);
	return ledger[prio / 100] <= WORK_FULL_UTILIZATION - work; // allow for rounding errors in double division
}

void Busyness::addWork(int prio, int work) {
	ASSERT(prio > 0 && prio < 1000);
	for (int i = 0; i <= (prio / 100); i++)
		ledger[i] += work;
}

void Busyness::removeWork(int prio, int work) {
	addWork(prio, -work);
}

std::string Busyness::toString() {
	std::string result;
	for (int i = 1; i < ledger.size();) {
		int j = i + 1;
		while (j < ledger.size() && ledger[i] == ledger[j])
			j++;
		if (i != 1)
			result += ", ";
		result += i + 1 == j ? format("%03d", i * 100) : format("%03d/%03d", i * 100, (j - 1) * 100);
		result += format("=%1.02f (%d/%d)", (float)ledger[i] / WORK_FULL_UTILIZATION, ledger[i], WORK_FULL_UTILIZATION);
		i = j;
	}
	return result;
}

double adjustRelocationParallelismForSrc(double srcParallelism) {
	double res = srcParallelism;
	if (SERVER_KNOBS->ENABLE_CONSERVATIVE_RELOCATION_WHEN_REPLICA_CONSISTENCY_CHECK &&
	    SERVER_KNOBS->ENABLE_REPLICA_CONSISTENCY_CHECK_ON_DATA_MOVEMENT &&
	    srcParallelism >= 1.0 + SERVER_KNOBS->DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS) {
		// DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS is the number of extra replicas that the destination
		// servers will read from the source team.
		res = res / (1.0 + SERVER_KNOBS->DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS);
	}
	ASSERT(res > 0);
	return res;
}

// find the "workFactor" for this, were it launched now
int getSrcWorkFactor(RelocateData const& relocation, int singleRegionTeamSize) {
	// RELOCATION_PARALLELISM_PER_SOURCE_SERVER is the number of concurrent replications that can be launched on a
	// single storage server at a time, given the team size is 1 --- only this storage server is available to serve
	// fetchKey read requests from the dest team.
	// The real parallelism is adjusted by the number of source servers of a source team that can serve
	// fetchKey requests.
	// When ENABLE_REPLICA_CONSISTENCY_CHECK_ON_DATA_MOVEMENT is enabled, the fetchKeys on
	// destination servers will read from DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS + 1 replicas from the source team
	// (suppose the team size is large enough). As a result it is possible that the source team can be overloaded by the
	// fetchKey read requests. This is especially true when the shard split data movements are launched. So, we
	// introduce ENABLE_CONSERVATIVE_RELOCATION_WHEN_REPLICA_CONSISTENCY_CHECK knob to adjust the relocation parallelism
	// accordingly. The adjustment is to reduce the relocation parallelism by a factor of
	// (1 + DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS).
	if (relocation.bulkLoadTask.present())
		return 0;
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	         relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(2 * SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE)
		// we want to set PRIORITY_PERPETUAL_STORAGE_WIGGLE to a reasonably large value
		// to make this parallelism take effect
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->WIGGLING_RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.priority == SERVER_KNOBS->PRIORITY_MERGE_SHARD)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->MERGE_RELOCATION_PARALLELISM_PER_TEAM);
	else { // for now we assume that any message at a lower priority can best be assumed to have a full team left for
		   // work
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(singleRegionTeamSize *
		                                         SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	}
}

int getDestWorkFactor() {
	// Work of moving a shard is even across destination servers
	return WORK_FULL_UTILIZATION / SERVER_KNOBS->RELOCATION_PARALLELISM_PER_DEST_SERVER;
}

// Data movement's resource control: Do not overload servers used for the RelocateData
// return true if servers are not too busy to launch the relocation
// This ensure source servers will not be overloaded.
bool canLaunchSrc(RelocateData& relocation,
                  int teamSize,
                  int singleRegionTeamSize,
                  std::map<UID, Busyness>& busymap,
                  std::vector<RelocateData> cancellableRelocations) {
	// assert this has not already been launched
	ASSERT(relocation.workFactor == 0);
	ASSERT(relocation.src.size() != 0);
	ASSERT(teamSize >= singleRegionTeamSize);

	// Blob migrator is backed by s3 so it can allow unlimited data movements
	if (relocation.src.size() == 1 && BlobMigratorInterface::isBlobMigrator(relocation.src.back())) {
		return true;
	} else if (relocation.bulkLoadTask.present()) {
		// workFactor for bulk load task on source is always 0, therefore, we can safely launch
		// the data move with a bulk load task
		return true;
	}

	// find the "workFactor" for this, were it launched now
	int workFactor = getSrcWorkFactor(relocation, singleRegionTeamSize);
	int neededServers = std::min<int>(relocation.src.size(), teamSize - singleRegionTeamSize + 1);
	if (SERVER_KNOBS->USE_OLD_NEEDED_SERVERS) {
		neededServers = std::max(1, (int)relocation.src.size() - teamSize + 1);
	}
	// see if each of the SS can launch this task
	for (int i = 0; i < relocation.src.size(); i++) {
		// For each source server for this relocation, copy and modify its busyness to reflect work that WOULD be
		// cancelled
		auto busyCopy = busymap[relocation.src[i]];
		for (int j = 0; j < cancellableRelocations.size(); j++) {
			auto& servers = cancellableRelocations[j].src;
			if (std::count(servers.begin(), servers.end(), relocation.src[i]))
				busyCopy.removeWork(cancellableRelocations[j].priority, cancellableRelocations[j].workFactor);
		}
		// Use this modified busyness to check if this relocation could be launched
		if (busyCopy.canLaunch(relocation.priority, workFactor)) {
			--neededServers;
			if (neededServers == 0)
				return true;
		}
	}

	return false;
}

// candidateTeams is a vector containing one team per datacenter, the team(s) DD is planning on moving the shard to.
bool canLaunchDest(const std::vector<std::pair<Reference<IDataDistributionTeam>, bool>>& candidateTeams,
                   int priority,
                   std::map<UID, Busyness>& busymapDest) {
	// fail switch if this is causing issues
	if (SERVER_KNOBS->RELOCATION_PARALLELISM_PER_DEST_SERVER <= 0) {
		return true;
	}
	int workFactor = getDestWorkFactor();
	for (auto& [team, _] : candidateTeams) {
		for (UID id : team->getServerIDs()) {
			if (!busymapDest[id].canLaunch(priority, workFactor)) {
				return false;
			}
		}
	}
	return true;
}

// update busyness for each server
void launch(RelocateData& relocation, std::map<UID, Busyness>& busymap, int singleRegionTeamSize) {
	// if we are here this means that we can launch and should adjust all the work the servers can do
	relocation.workFactor = getSrcWorkFactor(relocation, singleRegionTeamSize);
	for (int i = 0; i < relocation.src.size(); i++)
		busymap[relocation.src[i]].addWork(relocation.priority, relocation.workFactor);
}

void launchDest(RelocateData& relocation,
                const std::vector<std::pair<Reference<IDataDistributionTeam>, bool>>& candidateTeams,
                std::map<UID, Busyness>& destBusymap) {
	ASSERT(relocation.completeDests.empty());
	int destWorkFactor = getDestWorkFactor();
	for (auto& [team, _] : candidateTeams) {
		for (UID id : team->getServerIDs()) {
			relocation.completeDests.push_back(id);
			destBusymap[id].addWork(relocation.priority, destWorkFactor);
		}
	}
}
void completeDest(RelocateData const& relocation, std::map<UID, Busyness>& destBusymap) {
	int destWorkFactor = getDestWorkFactor();
	for (UID id : relocation.completeDests) {
		destBusymap[id].removeWork(relocation.priority, destWorkFactor);
	}
}

void complete(RelocateData const& relocation, std::map<UID, Busyness>& busymap, std::map<UID, Busyness>& destBusymap) {
	ASSERT(relocation.bulkLoadTask.present() || relocation.workFactor > 0);
	for (int i = 0; i < relocation.src.size(); i++)
		busymap[relocation.src[i]].removeWork(relocation.priority, relocation.workFactor);

	completeDest(relocation, destBusymap);
}

// Cancels in-flight data moves intersecting with range.
															#line 594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
[[nodiscard]] Future<Void> cancelDataMove( class DDQueue* const& self, KeyRange const& range, const DDEnabledState* const& ddEnabledState );

#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
[[nodiscard]] Future<Void> dataDistributionRelocator( class DDQueue* const& self, RelocateData const& rd, Future<Void> const& prevCleanup, const DDEnabledState* const& ddEnabledState );

#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via getSourceServersForRange()
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSourceServersForRangeActor>
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSourceServersForRangeActorState {
															#line 611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSourceServersForRangeActorState(DDQueue* const& self,RelocateData const& input,PromiseStream<RelocateData> const& output,Reference<FlowLock> const& fetchLock) 
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   input(input),
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   output(output),
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   fetchLock(fetchLock)
															#line 624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("getSourceServersForRange", reinterpret_cast<unsigned long>(this));

	}
	~GetSourceServersForRangeActorState() 
	{
		fdb_probe_actor_destroy("getSourceServersForRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (input.priority == SERVER_KNOBS->PRIORITY_MERGE_SHARD)
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(0.5, TaskPriority::DataDistributionVeryLow);
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 1;
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 0, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = delay(0.0001, TaskPriority::DataDistributionLaunch);
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
				static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 2;
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 1, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = fetchLock->take(TaskPriority::DataDistributionLaunch);
															#line 611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 3;
															#line 611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 2, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		releaser = FlowLock::Releaser(*fetchLock);
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<IDDTxnProcessor::SourceServers> __when_expr_3 = self->txnProcessor->getSourceServersForRange(input.keys);
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 4;
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		releaser = FlowLock::Releaser(*fetchLock);
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<IDDTxnProcessor::SourceServers> __when_expr_3 = self->txnProcessor->getSourceServersForRange(input.keys);
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 4;
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(IDDTxnProcessor::SourceServers const& res,int loopDepth) 
	{
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.src = std::move(res.srcServers);
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.completeSources = std::move(res.completeSources);
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		output.send(input);
															#line 618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSourceServersForRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetSourceServersForRangeActorState(); static_cast<GetSourceServersForRangeActor*>(this)->destroy(); return 0; }
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<GetSourceServersForRangeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(IDDTxnProcessor::SourceServers && res,int loopDepth) 
	{
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.src = std::move(res.srcServers);
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.completeSources = std::move(res.completeSources);
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		output.send(input);
															#line 618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSourceServersForRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetSourceServersForRangeActorState(); static_cast<GetSourceServersForRangeActor*>(this)->destroy(); return 0; }
															#line 1013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<GetSourceServersForRangeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(IDDTxnProcessor::SourceServers const& res,int loopDepth) 
	{
		loopDepth = a_body1cont5(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(IDDTxnProcessor::SourceServers && res,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,IDDTxnProcessor::SourceServers const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,IDDTxnProcessor::SourceServers && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData input;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> output;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<FlowLock> fetchLock;
															#line 612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowLock::Releaser releaser;
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via getSourceServersForRange()
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSourceServersForRangeActor final : public Actor<Void>, public ActorCallback< GetSourceServersForRangeActor, 0, Void >, public ActorCallback< GetSourceServersForRangeActor, 1, Void >, public ActorCallback< GetSourceServersForRangeActor, 2, Void >, public ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >, public FastAllocated<GetSourceServersForRangeActor>, public GetSourceServersForRangeActorState<GetSourceServersForRangeActor> {
															#line 1111 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<GetSourceServersForRangeActor>::operator new;
	using FastAllocated<GetSourceServersForRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13039137379639594240UL, 15530960983242423552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSourceServersForRangeActor, 0, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 1, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 2, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSourceServersForRangeActor(DDQueue* const& self,RelocateData const& input,PromiseStream<RelocateData> const& output,Reference<FlowLock> const& fetchLock) 
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   GetSourceServersForRangeActorState<GetSourceServersForRangeActor>(self, input, output, fetchLock),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7022398268476599552UL, 5262681254722997504UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSourceServersForRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> getSourceServersForRange( DDQueue* const& self, RelocateData const& input, PromiseStream<RelocateData> const& output, Reference<FlowLock> const& fetchLock ) {
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new GetSourceServersForRangeActor(self, input, output, fetchLock));
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

DDQueue::DDQueue(DDQueueInitParams const& params)
  : IDDRelocationQueue(), distributorId(params.id), lock(params.lock), txnProcessor(params.db),
    teamCollections(params.teamCollections), shardsAffectedByTeamFailure(params.shardsAffectedByTeamFailure),
    physicalShardCollection(params.physicalShardCollection), bulkLoadTaskCollection(params.bulkLoadTaskCollection),
    getAverageShardBytes(params.getAverageShardBytes),
    startMoveKeysParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    finishMoveKeysParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    cleanUpDataMoveParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    fetchSourceLock(new FlowLock(SERVER_KNOBS->DD_FETCH_SOURCE_PARALLELISM)), activeRelocations(0),
    queuedRelocations(0), bytesWritten(0), teamSize(params.teamSize), singleRegionTeamSize(params.singleRegionTeamSize),
    output(params.relocationProducer), input(params.relocationConsumer), getShardMetrics(params.getShardMetrics),
    getTopKMetrics(params.getTopKMetrics), lastInterval(0), suppressIntervals(0),
    rawProcessingUnhealthy(new AsyncVar<bool>(false)), rawProcessingWiggle(new AsyncVar<bool>(false)),
    unhealthyRelocations(0), movedKeyServersEventHolder(makeReference<EventCacheHolder>("MovedKeyServers")),
    moveReusePhysicalShard(0), moveCreateNewPhysicalShard(0),
    retryFindDstReasonCount(static_cast<int>(RetryFindDstReason::NumberOfTypes), 0),
    moveBytesRate(SERVER_KNOBS->DD_TRACE_MOVE_BYTES_AVERAGE_INTERVAL) {}

void DDQueue::startRelocation(int priority, int healthPriority) {
	// Although PRIORITY_TEAM_REDUNDANT has lower priority than split and merge shard movement,
	// we must count it into unhealthyRelocations; because team removers relies on unhealthyRelocations to
	// ensure a team remover will not start before the previous one finishes removing a team and move away data
	// NOTE: split and merge shard have higher priority. If they have to wait for unhealthyRelocations = 0,
	// deadlock may happen: split/merge shard waits for unhealthyRelocations, while blocks team_redundant.
	if (healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT || healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		unhealthyRelocations++;
		rawProcessingUnhealthy->set(true);
	}
	if (healthPriority == SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE) {
		rawProcessingWiggle->set(true);
	}
	priority_relocations[priority]++;
}

void DDQueue::finishRelocation(int priority, int healthPriority) {
	if (healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT || healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		unhealthyRelocations--;
		ASSERT(unhealthyRelocations >= 0);
		if (unhealthyRelocations == 0) {
			rawProcessingUnhealthy->set(false);
		}
	}
	priority_relocations[priority]--;
	if (priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE] == 0) {
		rawProcessingWiggle->set(false);
	}
}

void DDQueue::validate() {
	if (EXPENSIVE_VALIDATION) {
		for (auto it = fetchingSourcesQueue.begin(); it != fetchingSourcesQueue.end(); ++it) {
			// relocates in the fetching queue do not have src servers yet.
			if (it->src.size())
				TraceEvent(SevError, "DDQueueValidateError1")
				    .detail("Problem", "relocates in the fetching queue do not have src servers yet");

			// relocates in the fetching queue do not have a work factor yet.
			if (it->workFactor != 0.0)
				TraceEvent(SevError, "DDQueueValidateError2")
				    .detail("Problem", "relocates in the fetching queue do not have a work factor yet");

			// relocates in the fetching queue are in the queueMap.
			auto range = queueMap.rangeContaining(it->keys.begin);
			if (range.value() != *it || range.range() != it->keys)
				TraceEvent(SevError, "DDQueueValidateError3")
				    .detail("Problem", "relocates in the fetching queue are in the queueMap");
		}

		/*
		for( auto it = queue.begin(); it != queue.end(); ++it ) {
		    for( auto rdit = it->second.begin(); rdit != it->second.end(); ++rdit ) {
		        // relocates in the queue are in the queueMap exactly.
		        auto range = queueMap.rangeContaining( rdit->keys.begin );
		        if( range.value() != *rdit || range.range() != rdit->keys )
		            TraceEvent(SevError, "DDQueueValidateError4").detail("Problem", "relocates in the queue are in the queueMap exactly")
		            .detail("RangeBegin", range.range().begin)
		            .detail("RangeEnd", range.range().end)
		            .detail("RelocateBegin2", range.value().keys.begin)
		            .detail("RelocateEnd2", range.value().keys.end)
		            .detail("RelocateStart", range.value().startTime)
		            .detail("MapStart", rdit->startTime)
		            .detail("RelocateWork", range.value().workFactor)
		            .detail("MapWork", rdit->workFactor)
		            .detail("RelocateSrc", range.value().src.size())
		            .detail("MapSrc", rdit->src.size())
		            .detail("RelocatePrio", range.value().priority)
		            .detail("MapPrio", rdit->priority);

		        // relocates in the queue have src servers
		        if( !rdit->src.size() )
		            TraceEvent(SevError, "DDQueueValidateError5").detail("Problem", "relocates in the queue have src servers");

		        // relocates in the queue do not have a work factor yet.
		        if( rdit->workFactor != 0.0 )
		            TraceEvent(SevError, "DDQueueValidateError6").detail("Problem", "relocates in the queue do not have a work factor yet");

		        bool contains = false;
		        for( int i = 0; i < rdit->src.size(); i++ ) {
		            if( rdit->src[i] == it->first ) {
		                contains = true;
		                break;
		            }
		        }
		        if( !contains )
		            TraceEvent(SevError, "DDQueueValidateError7").detail("Problem", "queued relocate data does not include ss under which its filed");
		    }
		}*/

		auto inFlightRanges = inFlight.ranges();
		for (auto it = inFlightRanges.begin(); it != inFlightRanges.end(); ++it) {
			for (int i = 0; i < it->value().src.size(); i++) {
				// each server in the inFlight map is in the busymap
				if (!busymap.contains(it->value().src[i]))
					TraceEvent(SevError, "DDQueueValidateError8")
					    .detail("Problem", "each server in the inFlight map is in the busymap");

				// relocate data that is inFlight is not also in the queue
				if (queue[it->value().src[i]].contains(it->value()))
					TraceEvent(SevError, "DDQueueValidateError9")
					    .detail("Problem", "relocate data that is inFlight is not also in the queue");
			}

			for (int i = 0; i < it->value().completeDests.size(); i++) {
				// each server in the inFlight map is in the dest busymap
				if (!destBusymap.contains(it->value().completeDests[i]))
					TraceEvent(SevError, "DDQueueValidateError10")
					    .detail("Problem", "each server in the inFlight map is in the destBusymap");
			}

			// in flight relocates have source servers
			if (it->value().startTime != -1 && !it->value().src.size())
				TraceEvent(SevError, "DDQueueValidateError11")
				    .detail("Problem", "in flight relocates have source servers");

			if (inFlightActors.liveActorAt(it->range().begin)) {
				// the key range in the inFlight map matches the key range in the RelocateData message
				if (it->value().keys != it->range())
					TraceEvent(SevError, "DDQueueValidateError12")
					    .detail("Problem",
					            "the key range in the inFlight map matches the key range in the RelocateData message");
			} else if (it->value().cancellable) {
				TraceEvent(SevError, "DDQueueValidateError13")
				    .detail("Problem", "key range is cancellable but not in flight!")
				    .detail("Range", it->range());
			}
		}

		for (auto it = busymap.begin(); it != busymap.end(); ++it) {
			for (int i = 0; i < it->second.ledger.size() - 1; i++) {
				if (it->second.ledger[i] < it->second.ledger[i + 1])
					TraceEvent(SevError, "DDQueueValidateError14")
					    .detail("Problem", "ascending ledger problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValueA", it->second.ledger[i])
					    .detail("LedgerValueB", it->second.ledger[i + 1]);
				if (it->second.ledger[i] < 0.0)
					TraceEvent(SevError, "DDQueueValidateError15")
					    .detail("Problem", "negative ascending problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValue", it->second.ledger[i]);
			}
		}

		for (auto it = destBusymap.begin(); it != destBusymap.end(); ++it) {
			for (int i = 0; i < it->second.ledger.size() - 1; i++) {
				if (it->second.ledger[i] < it->second.ledger[i + 1])
					TraceEvent(SevError, "DDQueueValidateError16")
					    .detail("Problem", "ascending ledger problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValueA", it->second.ledger[i])
					    .detail("LedgerValueB", it->second.ledger[i + 1]);
				if (it->second.ledger[i] < 0.0)
					TraceEvent(SevError, "DDQueueValidateError17")
					    .detail("Problem", "negative ascending problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValue", it->second.ledger[i]);
			}
		}

		std::set<RelocateData, std::greater<RelocateData>> queuedRelocationsMatch;
		for (auto it = queue.begin(); it != queue.end(); ++it)
			queuedRelocationsMatch.insert(it->second.begin(), it->second.end());
		ASSERT(queuedRelocations == queuedRelocationsMatch.size() + fetchingSourcesQueue.size());

		int testActive = 0;
		for (auto it = priority_relocations.begin(); it != priority_relocations.end(); ++it)
			testActive += it->second;
		ASSERT(activeRelocations + queuedRelocations == testActive);
	}
}

void DDQueue::queueRelocation(RelocateShard rs, std::set<UID>& serversToLaunchFrom) {
	//TraceEvent("QueueRelocationBegin").detail("Begin", rd.keys.begin).detail("End", rd.keys.end);

	// remove all items from both queues that are fully contained in the new relocation (i.e. will be overwritten)
	RelocateData rd(rs);
	bool hasHealthPriority = RelocateData::isHealthPriority(rd.priority);
	bool hasBoundaryPriority = RelocateData::isBoundaryPriority(rd.priority);

	auto ranges = queueMap.intersectingRanges(rd.keys);
	for (auto r = ranges.begin(); r != ranges.end(); ++r) {
		RelocateData& rrs = r->value();

		auto fetchingSourcesItr = fetchingSourcesQueue.find(rrs);
		bool foundActiveFetching = fetchingSourcesItr != fetchingSourcesQueue.end();
		std::set<RelocateData, std::greater<RelocateData>>* firstQueue;
		std::set<RelocateData, std::greater<RelocateData>>::iterator firstRelocationItr;
		bool foundActiveRelocation = false;

		if (!foundActiveFetching && rrs.src.size()) {
			firstQueue = &queue[rrs.src[0]];
			firstRelocationItr = firstQueue->find(rrs);
			foundActiveRelocation = firstRelocationItr != firstQueue->end();
		}

		// If there is a queued job that wants data relocation which we are about to cancel/modify,
		//  make sure that we keep the relocation intent for the job that we queue up
		if (foundActiveFetching || foundActiveRelocation) {
			rd.wantsNewServers |= rrs.wantsNewServers;
			rd.startTime = std::min(rd.startTime, rrs.startTime);
			if (!hasHealthPriority) {
				rd.healthPriority = std::max(rd.healthPriority, rrs.healthPriority);
			}
			if (!hasBoundaryPriority) {
				rd.boundaryPriority = std::max(rd.boundaryPriority, rrs.boundaryPriority);
			}
			rd.priority = std::max(rd.priority, std::max(rd.boundaryPriority, rd.healthPriority));
		}

		if (rd.keys.contains(rrs.keys)) {
			if (foundActiveFetching)
				fetchingSourcesQueue.erase(fetchingSourcesItr);
			else if (foundActiveRelocation) {
				firstQueue->erase(firstRelocationItr);
				for (int i = 1; i < rrs.src.size(); i++)
					queue[rrs.src[i]].erase(rrs);
			}
		}

		if (foundActiveFetching || foundActiveRelocation) {
			serversToLaunchFrom.insert(rrs.src.begin(), rrs.src.end());
			/*TraceEvent(rrs.interval.end(), mi.id()).detail("Result","Cancelled")
			    .detail("WasFetching", foundActiveFetching).detail("Contained", rd.keys.contains( rrs.keys ));*/
			queuedRelocations--;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rrs.dataMoveId)
			    .detail("RandomID", rrs.randomId)
			    .detail("Total", queuedRelocations);
			finishRelocation(rrs.priority, rrs.healthPriority);
		}
	}

	// determine the final state of the relocations map
	auto affectedQueuedItems = queueMap.getAffectedRangesAfterInsertion(rd.keys, rd);

	// put the new request into the global map of requests (modifies the ranges already present)
	queueMap.insert(rd.keys, rd);

	// cancel all the getSourceServers actors that intersect the new range that we will be getting
	getSourceActors.cancel(KeyRangeRef(affectedQueuedItems.front().begin, affectedQueuedItems.back().end));

	// update fetchingSourcesQueue and the per-server queue based on truncated ranges after insertion, (re-)launch
	// getSourceServers
	auto queueMapItr = queueMap.rangeContaining(affectedQueuedItems[0].begin);

	// Put off erasing elements from fetchingSourcesQueue
	std::set<RelocateData, std::greater<RelocateData>> delayDelete;
	for (int r = 0; r < affectedQueuedItems.size(); ++r, ++queueMapItr) {
		// ASSERT(queueMapItr->value() == queueMap.rangeContaining(affectedQueuedItems[r].begin)->value());
		RelocateData& rrs = queueMapItr->value();

		if (rrs.src.size() == 0 && (rrs.keys == rd.keys || fetchingSourcesQueue.contains(rrs))) {
			if (rrs.keys != rd.keys) {
				delayDelete.insert(rrs);
			}

			rrs.keys = affectedQueuedItems[r];
			rrs.interval = TraceInterval("QueuedRelocation", rrs.randomId); // inherit the old randomId

			DebugRelocationTraceEvent(rrs.interval.begin(), distributorId)
			    .detail("KeyBegin", rrs.keys.begin)
			    .detail("KeyEnd", rrs.keys.end)
			    .detail("Priority", rrs.priority)
			    .detail("WantsNewServers", rrs.wantsNewServers);

			queuedRelocations++;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rrs.dataMoveId)
			    .detail("RandomID", rrs.randomId)
			    .detail("Total", queuedRelocations);
			startRelocation(rrs.priority, rrs.healthPriority);

			fetchingSourcesQueue.insert(rrs);
			getSourceActors.insert(rrs.keys,
			                       getSourceServersForRange(this, rrs, fetchSourceServersComplete, fetchSourceLock));
		} else {
			RelocateData newData(rrs);
			newData.keys = affectedQueuedItems[r];
			ASSERT(rrs.src.size() || rrs.startTime == -1);

			bool foundActiveRelocation = false;
			for (int i = 0; i < rrs.src.size(); i++) {
				auto& serverQueue = queue[rrs.src[i]];

				if (serverQueue.erase(rrs) > 0) {
					if (!foundActiveRelocation) {
						newData.interval = TraceInterval("QueuedRelocation", rrs.randomId); // inherit the old randomId

						DebugRelocationTraceEvent(newData.interval.begin(), distributorId)
						    .detail("KeyBegin", newData.keys.begin)
						    .detail("KeyEnd", newData.keys.end)
						    .detail("Priority", newData.priority)
						    .detail("WantsNewServers", newData.wantsNewServers);

						queuedRelocations++;
						TraceEvent(SevVerbose, "QueuedRelocationsChanged")
						    .detail("DataMoveID", newData.dataMoveId)
						    .detail("RandomID", newData.randomId)
						    .detail("Total", queuedRelocations);
						startRelocation(newData.priority, newData.healthPriority);
						foundActiveRelocation = true;
					}

					serverQueue.insert(newData);
				} else
					break;
			}

			// We update the keys of a relocation even if it is "dead" since it helps validate()
			rrs.keys = affectedQueuedItems[r];
			rrs.interval = newData.interval;
		}
	}

	for (auto it : delayDelete) {
		fetchingSourcesQueue.erase(it);
	}
	DebugRelocationTraceEvent("ReceivedRelocateShard", distributorId)
	    .detail("KeyBegin", rd.keys.begin)
	    .detail("KeyEnd", rd.keys.end)
	    .detail("Priority", rd.priority)
	    .detail("AffectedRanges", affectedQueuedItems.size());
}

void DDQueue::completeSourceFetch(const RelocateData& results) {
	ASSERT(fetchingSourcesQueue.contains(results));

	// logRelocation( results, "GotSourceServers" );

	fetchingSourcesQueue.erase(results);
	queueMap.insert(results.keys, results);
	for (int i = 0; i < results.src.size(); i++) {
		queue[results.src[i]].insert(results);
	}
	updateLastAsSource(results.src);
	serverCounter.increaseForTeam(results.src, results.reason, ServerCounter::CountType::QueuedSource);
}

void DDQueue::logRelocation(const RelocateData& rd, const char* title) {
	std::string busyString;
	for (int i = 0; i < rd.src.size() && i < teamSize * 2; i++)
		busyString += describe(rd.src[i]) + " - (" + busymap[rd.src[i]].toString() + "); ";

	TraceEvent(title, distributorId)
	    .detail("KeyBegin", rd.keys.begin)
	    .detail("KeyEnd", rd.keys.end)
	    .detail("Priority", rd.priority)
	    .detail("WorkFactor", rd.workFactor)
	    .detail("SourceServerCount", rd.src.size())
	    .detail("SourceServers", describe(rd.src, teamSize * 2))
	    .detail("SourceBusyness", busyString);
}

void DDQueue::launchQueuedWork(KeyRange keys, const DDEnabledState* ddEnabledState) {
	// combine all queued work in the key range and check to see if there is anything to launch
	std::set<RelocateData, std::greater<RelocateData>> combined;
	auto f = queueMap.intersectingRanges(keys);
	for (auto it = f.begin(); it != f.end(); ++it) {
		if (it->value().src.size() && queue[it->value().src[0]].contains(it->value()))
			combined.insert(it->value());
	}
	launchQueuedWork(combined, ddEnabledState);
}

void DDQueue::launchQueuedWork(const std::set<UID>& serversToLaunchFrom, const DDEnabledState* ddEnabledState) {
	// combine all work from the source servers to see if there is anything new to launch
	std::set<RelocateData, std::greater<RelocateData>> combined;
	for (auto id : serversToLaunchFrom) {
		auto& queuedWork = queue[id];
		auto it = queuedWork.begin();
		for (int j = 0; j < teamSize && it != queuedWork.end(); j++) {
			combined.insert(*it);
			++it;
		}
	}
	launchQueuedWork(combined, ddEnabledState);
}

void DDQueue::launchQueuedWork(RelocateData launchData, const DDEnabledState* ddEnabledState) {
	// check a single RelocateData to see if it can be launched
	std::set<RelocateData, std::greater<RelocateData>> combined;
	combined.insert(launchData);
	launchQueuedWork(combined, ddEnabledState);
}

DataMoveType newDataMoveType(bool doBulkLoading) {
	DataMoveType type = DataMoveType::LOGICAL;
	if (deterministicRandom()->random01() < SERVER_KNOBS->DD_PHYSICAL_SHARD_MOVE_PROBABILITY) {
		type = DataMoveType::PHYSICAL;
	}
	if (type != DataMoveType::PHYSICAL && SERVER_KNOBS->ENABLE_PHYSICAL_SHARD_MOVE_EXPERIMENT) {
		type = DataMoveType::PHYSICAL_EXP;
	}
	if (doBulkLoading) {
		if (type == DataMoveType::LOGICAL) {
			type = DataMoveType::LOGICAL_BULKLOAD;
		} else if (type == DataMoveType::PHYSICAL || type == DataMoveType::PHYSICAL_EXP) {
			type = DataMoveType::PHYSICAL_BULKLOAD;
		} else {
			UNREACHABLE();
		}
	}
	return type;
}

bool runPendingBulkLoadTaskWithRelocateData(DDQueue* self, RelocateData& rd) {
	bool doBulkLoading = false;
	Optional<DDBulkLoadEngineTask> task = self->bulkLoadTaskCollection->getTaskByRange(rd.keys);
	if (task.present() && task.get().coreState.onAnyPhase({ BulkLoadPhase::Triggered, BulkLoadPhase::Running })) {
		rd.bulkLoadTask = task.get();
		doBulkLoading = true;
	}
	if (doBulkLoading) {
		try {
			self->bulkLoadTaskCollection->startTask(rd.bulkLoadTask.get().coreState);
		} catch (Error& e) {
			ASSERT_WE_THINK(e.code() == error_code_bulkload_task_outdated);
			if (e.code() == error_code_bulkload_task_outdated) {
				TraceEvent(SevError, "DDBulkLoadTaskOutdatedWhenStartRelocator", self->distributorId)
				    .detail("NewDataMoveID", rd.dataMoveId)
				    .detail("NewDataMovePriority", rd.priority)
				    .detail("NewDataMoveRange", rd.keys)
				    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
				    .detail("TaskJobID", rd.bulkLoadTask.get().coreState.getJobId())
				    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange())
				    .detail("TaskDataMoveID",
				            rd.bulkLoadTask.get().coreState.getDataMoveId().present()
				                ? rd.bulkLoadTask.get().coreState.getDataMoveId().get().toString()
				                : "");
				throw movekeys_conflict();
			} else {
				throw e;
			}
		}
	}
	return doBulkLoading;
}

// For each relocateData rd in the queue, check if there exist inflight relocate data whose keyrange is overlapped
// with rd. If there exist, cancel them by cancelling their actors and reducing the src servers' busyness of those
// canceled inflight relocateData. Launch the relocation for the rd.
void DDQueue::launchQueuedWork(std::set<RelocateData, std::greater<RelocateData>> combined,
                               const DDEnabledState* ddEnabledState) {
	[[maybe_unused]] int startedHere = 0;
	double startTime = now();
	// kick off relocators from items in the queue as need be
	auto it = combined.begin();
	for (; it != combined.end(); it++) {
		RelocateData rd(*it);

		// If having a bulk load task overlapping the rd range,
		// attach bulk load task to the input rd if rd is not a data move
		// for unhealthy. Make the bulk load task visible on the global task map
		bool doBulkLoading = runPendingBulkLoadTaskWithRelocateData(this, rd);
		if (doBulkLoading) {
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskLaunchingDataMove", this->distributorId)
			    .detail("NewDataMoveId", rd.dataMoveId)
			    .detail("NewDataMovePriority", rd.priority)
			    .detail("NewDataMoveRange", rd.keys)
			    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
			    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId());
		}

		// Check if there is an inflight shard that is overlapped with the queued relocateShard (rd)
		bool overlappingInFlight = false;
		auto intersectingInFlight = inFlight.intersectingRanges(rd.keys);
		for (auto it = intersectingInFlight.begin(); it != intersectingInFlight.end(); ++it) {
			if (fetchKeysComplete.contains(it->value()) && inFlightActors.liveActorAt(it->range().begin) &&
			    !rd.keys.contains(it->range()) && it->value().priority >= rd.priority &&
			    rd.healthPriority < SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY) {

				DebugRelocationTraceEvent("OverlappingInFlight", distributorId)
				    .detail("KeyBegin", it->value().keys.begin)
				    .detail("KeyEnd", it->value().keys.end)
				    .detail("Priority", it->value().priority);

				overlappingInFlight = true;
				break;
			}
		}

		if (overlappingInFlight) {
			ASSERT(!rd.isRestore());
			// logRelocation( rd, "SkippingOverlappingInFlight" );
			continue;
		}

		// Because the busyness of a server is decreased when a superseding relocation is issued, we
		//  need to consider what the busyness of a server WOULD be if
		auto containedRanges = inFlight.containedRanges(rd.keys);
		std::vector<RelocateData> cancellableRelocations;
		for (auto it = containedRanges.begin(); it != containedRanges.end(); ++it) {
			if (it.value().cancellable) {
				cancellableRelocations.push_back(it->value());
			}
		}

		// Data movement avoids overloading source servers in moving data.
		// SOMEDAY: the list of source servers may be outdated since they were fetched when the work was put in the
		// queue
		// FIXME: we need spare capacity even when we're just going to be cancelling work via TEAM_HEALTHY
		if (!rd.isRestore() && !canLaunchSrc(rd, teamSize, singleRegionTeamSize, busymap, cancellableRelocations)) {
			// logRelocation( rd, "SkippingQueuedRelocation" );
			if (rd.bulkLoadTask.present()) {
				TraceEvent(SevError, "DDBulkLoadTaskDelayedByBusySrc", this->distributorId)
				    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
				    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
				    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
			}
			continue;
		}

		// From now on, the source servers for the RelocateData rd have enough resource to move the data away,
		// because they do not have too much inflight data movement.

		// logRelocation( rd, "LaunchingRelocation" );
		DebugRelocationTraceEvent(rd.interval.end(), distributorId).detail("Result", "Success");

		if (!rd.isRestore()) {
			queuedRelocations--;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rd.dataMoveId)
			    .detail("RandomID", rd.randomId)
			    .detail("Total", queuedRelocations);
			finishRelocation(rd.priority, rd.healthPriority);

			// now we are launching: remove this entry from the queue of all the src servers
			for (size_t i = 0; i < rd.src.size(); i++) {
				const auto result = queue[rd.src[i]].erase(rd);
				ASSERT(result);
			}
		}

		// If there is a job in flight that wants data relocation which we are about to cancel/modify,
		//     make sure that we keep the relocation intent for the job that we launch
		auto f = inFlight.intersectingRanges(rd.keys);
		for (auto it = f.begin(); it != f.end(); ++it) {
			if (inFlightActors.liveActorAt(it->range().begin)) {
				rd.wantsNewServers |= it->value().wantsNewServers;
			}
		}
		startedHere++;

		// update both inFlightActors and inFlight key range maps, cancelling deleted RelocateShards
		std::vector<KeyRange> ranges;
		inFlightActors.getRangesAffectedByInsertion(rd.keys, ranges);
		inFlightActors.cancel(KeyRangeRef(ranges.front().begin, ranges.back().end));
		// The cancelDataMove feature assumes inFlightActors are immediately cancelled.
		// If this is not true, multiple inflightActors can have overlapped range,
		// which leads to conflicts of moving keys

		Future<Void> fCleanup =
		    SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA ? cancelDataMove(this, rd.keys, ddEnabledState) : Void();

		inFlight.insert(rd.keys, rd);
		for (int r = 0; r < ranges.size(); r++) {
			RelocateData& rrs = inFlight.rangeContaining(ranges[r].begin)->value();
			rrs.keys = ranges[r];
			if (rrs.bulkLoadTask.present() && rrs.bulkLoadTask.get().coreState.getRange() != rrs.keys) {
				// The new bulk load data move partially overwrites an old bulk load data move.
				// In this case, the old bulk load task is cancelled.
				// For the range that is not covered by the new data move, drop the bulk load task and
				// run it as a normal data move.
				ASSERT(rrs.bulkLoadTask.get().coreState.getRange().contains(rrs.keys));
				rrs.bulkLoadTask.reset();
			}
			if (rd.keys == ranges[r] && rd.isRestore()) {
				ASSERT(rd.dataMove != nullptr);
				ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
				rrs.dataMoveId = rd.dataMove->meta.id;
			} else {
				ASSERT_WE_THINK(!rd.isRestore()); // Restored data move should not overlap.
				// TODO(psm): The shard id is determined by DD.
				rrs.dataMove.reset();
				if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA) {
					if (SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD) {
						rrs.dataMoveId = UID();
					} else if (rrs.bulkLoadTask.present()) {
						// We have to decide this after prevCleanup completes.
						// For details, see the comment in dataDistributionRelocator.
						rrs.dataMoveId = UID();
					} else {
						DataMoveType dataMoveType = newDataMoveType(rrs.bulkLoadTask.present());
						rrs.dataMoveId = newDataMoveId(
						    deterministicRandom()->randomUInt64(), AssignEmptyRange::False, dataMoveType, rrs.dmReason);
						TraceEvent(SevInfo, "NewDataMoveWithRandomDestID", this->distributorId)
						    .detail("DataMoveID", rrs.dataMoveId.toString())
						    .detail("TrackID", rrs.randomId)
						    .detail("Range", rrs.keys)
						    .detail("Reason", rrs.reason.toString())
						    .detail("DataMoveType", dataMoveType)
						    .detail("DataMoveReason", static_cast<int>(rrs.dmReason));
					}
				} else {
					rrs.dataMoveId = anonymousShardId;
					TraceEvent(SevInfo, "NewDataMoveWithAnonymousDestID", this->distributorId)
					    .detail("DataMoveID", rrs.dataMoveId.toString())
					    .detail("TrackID", rrs.randomId)
					    .detail("Range", rrs.keys)
					    .detail("Reason", rrs.reason.toString())
					    .detail("DataMoveReason", static_cast<int>(rrs.dmReason));
				}
			}

			launch(rrs, busymap, singleRegionTeamSize);
			activeRelocations++;
			TraceEvent(SevVerbose, "InFlightRelocationChange")
			    .detail("Launch", rrs.dataMoveId)
			    .detail("Total", activeRelocations);
			startRelocation(rrs.priority, rrs.healthPriority);
			// Start the actor that relocates data in the rrs.keys
			inFlightActors.insert(rrs.keys, dataDistributionRelocator(this, rrs, fCleanup, ddEnabledState));
		}

		// logRelocation( rd, "LaunchedRelocation" );
	}
	if (now() - startTime > .001 && deterministicRandom()->random01() < 0.001)
		TraceEvent(SevWarnAlways, "LaunchingQueueSlowx1000").detail("Elapsed", now() - startTime);

	/*if( startedHere > 0 ) {
	    TraceEvent("StartedDDRelocators", distributorId)
	        .detail("QueueSize", queuedRelocations)
	        .detail("StartedHere", startedHere)
	        .detail("ActiveRelocations", activeRelocations);
	} */

	validate();
}

int DDQueue::getHighestPriorityRelocation() const {
	int highestPriority{ 0 };
	for (const auto& [priority, count] : priority_relocations) {
		if (count > 0) {
			highestPriority = std::max(highestPriority, priority);
		}
	}
	return highestPriority;
}

// return true if the servers are throttled as source for read rebalance
bool DDQueue::timeThrottle(const std::vector<UID>& ids) const {
	return std::any_of(ids.begin(), ids.end(), [this](const UID& id) {
		if (this->lastAsSource.contains(id)) {
			return (now() - this->lastAsSource.at(id)) * SERVER_KNOBS->READ_REBALANCE_SRC_PARALLELISM <
			       SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;
		}
		return false;
	});
}

void DDQueue::updateLastAsSource(const std::vector<UID>& ids, double t) {
	for (auto& id : ids)
		lastAsSource[id] = t;
}

// Schedules cancellation of a data move.
void DDQueue::enqueueCancelledDataMove(UID dataMoveId, KeyRange range, const DDEnabledState* ddEnabledState) {
	ASSERT(!txnProcessor->isMocked()); // the mock implementation currently doesn't support data move
	std::vector<Future<Void>> cleanup;
	auto f = this->dataMoves.intersectingRanges(range);
	for (auto it = f.begin(); it != f.end(); ++it) {
		if (it->value().isValid()) {
			TraceEvent(SevError, "DDEnqueueCancelledDataMoveConflict", this->distributorId)
			    .detail("DataMoveID", dataMoveId)
			    .detail("CancelledRange", range)
			    .detail("ConflictingDataMoveID", it->value().id)
			    .detail("ConflictingRange", KeyRangeRef(it->range().begin, it->range().end));
			return;
		}
	}

	DDQueue::DDDataMove dataMove(dataMoveId);
	dataMove.cancel = cleanUpDataMove(txnProcessor->context(),
	                                  dataMoveId,
	                                  this->lock,
	                                  &this->cleanUpDataMoveParallelismLock,
	                                  range,
	                                  ddEnabledState,
	                                  this->addBackgroundCleanUpDataMoveActor);
	this->dataMoves.insert(range, dataMove);
	TraceEvent(SevInfo, "DDEnqueuedCancelledDataMove", this->distributorId)
	    .detail("DataMoveID", dataMoveId)
	    .detail("Range", range);
}

Future<Void> DDQueue::periodicalRefreshCounter() {
	auto f = [this]() {
		serverCounter.traceAll(distributorId);
		serverCounter.clear();
	};
	return recurring(f, SERVER_KNOBS->DD_QUEUE_COUNTER_REFRESH_INTERVAL);
}

int DDQueue::getUnhealthyRelocationCount() const {
	return unhealthyRelocations;
}

// Cancel existing relocation if exists, and serialize all concurrent relocations
															#line 1898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via cancelDataMove()
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class CancelDataMoveActor>
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class CancelDataMoveActorState {
															#line 1905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	CancelDataMoveActorState(class DDQueue* const& self,KeyRange const& range,const DDEnabledState* const& ddEnabledState) 
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   range(range),
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddEnabledState(ddEnabledState),
															#line 1348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   cleanup(),
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   lastObservedDataMoves()
															#line 1920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("cancelDataMove", reinterpret_cast<unsigned long>(this));

	}
	~CancelDataMoveActorState() 
	{
		fdb_probe_actor_destroy("cancelDataMove", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				lastObservedDataMoves.clear();
															#line 1353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto f = self->dataMoves.intersectingRanges(range);
															#line 1354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(auto it = f.begin();it != f.end();++it) {
															#line 1355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (!it->value().isValid())
															#line 1942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
						continue;
					}
															#line 1358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					KeyRange keys = KeyRangeRef(it->range().begin, it->range().end);
															#line 1359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DDQueueCancelDataMove", self->distributorId) .detail("DataMoveID", it->value().id) .detail("DataMoveRange", keys) .detail("Range", range);
															#line 1363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (!it->value().cancel.isValid())
															#line 1952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 1364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						it->value().cancel = cleanUpDataMove(self->txnProcessor->context(), it->value().id, self->lock, &self->cleanUpDataMoveParallelismLock, keys, ddEnabledState, self->addBackgroundCleanUpDataMoveActor);
															#line 1956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
															#line 1372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					lastObservedDataMoves.push_back(std::make_pair(keys, it->value().id));
															#line 1373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					cleanup.push_back(it->value().cancel);
															#line 1962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 1376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_0 = waitForAll(cleanup);
															#line 1376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<CancelDataMoveActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CancelDataMoveActor*>(this)->actor_wait_state = 1;
															#line 1376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CancelDataMoveActor, 0, Void >*>(static_cast<CancelDataMoveActor*>(this)));
															#line 1973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CancelDataMoveActorState();
		static_cast<CancelDataMoveActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<CancelDataMoveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CancelDataMoveActorState(); static_cast<CancelDataMoveActor*>(this)->destroy(); return 0; }
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<CancelDataMoveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CancelDataMoveActorState();
		static_cast<CancelDataMoveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<KeyRange> toResetRanges;
															#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto observedDataMove : lastObservedDataMoves ) {
															#line 1385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto f = self->dataMoves.intersectingRanges(observedDataMove.first);
															#line 1386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(auto it = f.begin();it != f.end();++it) {
															#line 1387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (it->value().id != observedDataMove.second)
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DataMoveWrittenByConcurrentDataMove", self->distributorId) .detail("Range", range) .detail("OldRange", observedDataMove.first) .detail("LastObservedDataMoveID", observedDataMove.second) .detail("CurrentDataMoveID", it->value().id);
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(!it->value().isValid() || (it->value().cancel.isValid() && it->value().cancel.isReady()));
															#line 1395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					toResetRanges.push_back(Standalone(it->range()));
															#line 2049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
															#line 1399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& toResetRange : toResetRanges ) {
															#line 1400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(toResetRange, DDQueue::DDDataMove());
															#line 2057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<KeyRange> toResetRanges;
															#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto observedDataMove : lastObservedDataMoves ) {
															#line 1385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto f = self->dataMoves.intersectingRanges(observedDataMove.first);
															#line 1386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(auto it = f.begin();it != f.end();++it) {
															#line 1387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (it->value().id != observedDataMove.second)
															#line 2075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DataMoveWrittenByConcurrentDataMove", self->distributorId) .detail("Range", range) .detail("OldRange", observedDataMove.first) .detail("LastObservedDataMoveID", observedDataMove.second) .detail("CurrentDataMoveID", it->value().id);
															#line 2079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(!it->value().isValid() || (it->value().cancel.isValid() && it->value().cancel.isReady()));
															#line 1395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					toResetRanges.push_back(Standalone(it->range()));
															#line 2087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
															#line 1399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& toResetRange : toResetRanges ) {
															#line 1400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(toResetRange, DDQueue::DDDataMove());
															#line 2095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CancelDataMoveActor*>(this)->actor_wait_state > 0) static_cast<CancelDataMoveActor*>(this)->actor_wait_state = 0;
		static_cast<CancelDataMoveActor*>(this)->ActorCallback< CancelDataMoveActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CancelDataMoveActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CancelDataMoveActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CancelDataMoveActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	class DDQueue* self;
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange range;
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const DDEnabledState* ddEnabledState;
															#line 1348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<Future<Void>> cleanup;
															#line 1349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<std::pair<KeyRange, UID>> lastObservedDataMoves;
															#line 2199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via cancelDataMove()
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class CancelDataMoveActor final : public Actor<Void>, public ActorCallback< CancelDataMoveActor, 0, Void >, public FastAllocated<CancelDataMoveActor>, public CancelDataMoveActorState<CancelDataMoveActor> {
															#line 2204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<CancelDataMoveActor>::operator new;
	using FastAllocated<CancelDataMoveActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2657712718045360640UL, 3322790634596070144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CancelDataMoveActor, 0, Void >;
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	CancelDataMoveActor(class DDQueue* const& self,KeyRange const& range,const DDEnabledState* const& ddEnabledState) 
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   CancelDataMoveActorState<CancelDataMoveActor>(self, range, ddEnabledState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7738710457594625024UL, 3228702489229399296UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cancelDataMove");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CancelDataMoveActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> cancelDataMove( class DDQueue* const& self, KeyRange const& range, const DDEnabledState* const& ddEnabledState ) {
															#line 1347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new CancelDataMoveActor(self, range, ddEnabledState));
															#line 2254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 1409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

static std::string destServersString(std::vector<std::pair<Reference<IDataDistributionTeam>, bool>> const& bestTeams) {
	std::stringstream ss;

	for (auto& tc : bestTeams) {
		for (const auto& id : tc.first->getServerIDs()) {
			ss << id.toString() << " ";
		}
	}

	return std::move(ss).str();
}

void traceRelocateDecision(TraceEvent& ev, const UID& pairId, const RelocateDecision& decision) {
	ev.detail("PairId", pairId)
	    .detail("Priority", decision.rd.priority)
	    .detail("KeyBegin", decision.rd.keys.begin)
	    .detail("KeyEnd", decision.rd.keys.end)
	    .detail("Reason", decision.rd.reason.toString())
	    .detail("SourceServers", describe(decision.rd.src))
	    .detail("DestinationTeam", describe(decision.destIds))
	    .detail("ExtraIds", describe(decision.extraIds));
	if (SERVER_KNOBS->DD_ENABLE_VERBOSE_TRACING) {
		// StorageMetrics is the rd shard's metrics, e.g., bytes and write bandwidth
		ev.detail("StorageMetrics", decision.metrics.toString());
	}

	if (decision.rd.reason == RelocateReason::WRITE_SPLIT) {
		// tell if the splitter acted as expected for write bandwidth splitting
		// SOMEDAY: trace the source team write bytes if necessary
		ev.detail("ShardWriteBytes", decision.metrics.bytesWrittenPerKSecond)
		    .detail("ParentShardWriteBytes", decision.parentMetrics.get().bytesWrittenPerKSecond);
	} else if (decision.rd.reason == RelocateReason::SIZE_SPLIT) {
		ev.detail("ShardSize", decision.metrics.bytes).detail("ParentShardSize", decision.parentMetrics.get().bytes);
	}
}

static int nonOverlappedServerCount(const std::vector<UID>& srcIds, const std::vector<UID>& destIds) {
	std::unordered_set<UID> srcSet{ srcIds.begin(), srcIds.end() };
	int count = 0;
	for (int i = 0; i < destIds.size(); i++) {
		if (!srcSet.contains(destIds[i])) {
			count++;
		}
	}
	return count;
}

void validateBulkLoadRelocateData(const RelocateData& rd, const std::vector<UID>& destIds, UID logId) {
	if (rd.keys != rd.bulkLoadTask.get().coreState.getRange()) {
		TraceEvent(SevError, "DDBulkLoadTaskLaunchFailed", logId)
		    .detail("Reason", "Wrong data move range")
		    .detail("DataMovePriority", rd.priority)
		    .detail("DataMoveId", rd.dataMoveId)
		    .detail("RelocatorRange", rd.keys)
		    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
		    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
		    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
		throw movekeys_conflict();
		// Very important invariant. If this error appears, check the logic
	}
	for (const auto& destId : destIds) {
		if (std::find(rd.src.begin(), rd.src.end(), destId) != rd.src.end()) {
			// In this case, getTeam has to select src as dest when remote team collection is not ready
			// This is not expected
			TraceEvent(SevError, "DDBulkLoadEngineTaskLaunchFailed", logId)
			    .detail("Reason", "Conflict src and destd due to remote recovery")
			    .detail("DataMovePriority", rd.priority)
			    .detail("DataMoveId", rd.dataMoveId)
			    .detail("RelocatorRange", rd.keys)
			    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
			    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
			    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
			throw movekeys_conflict();
		}
	}
	return;
}

// With probability, set wantTrueBestIfMoveout true for teamUnhealthy data moves and teamRedundant data moves only.
// This flag takes effect in getTeam. When the flag is set true, DD always getBestTeam for teamRedundant data moves and
// do getBestTeam for a teamRedundant data move if the data move decides to move data out of a SS.
bool getWantTrueBestIfMoveout(int priority) {
	if (priority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY) {
		return deterministicRandom()->random01() <
		       SERVER_KNOBS->PROBABILITY_TEAM_UNHEALTHY_DATAMOVE_CHOOSE_TRUE_BEST_DEST;
	} else if (priority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		return deterministicRandom()->random01() <
		       SERVER_KNOBS->PROBABILITY_TEAM_REDUNDANT_DATAMOVE_CHOOSE_TRUE_BEST_DEST;
	} else {
		return false;
	}
}

// This actor relocates the specified keys to a good place.
// The inFlightActor key range map stores the actor for each RelocateData
															#line 2354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via dataDistributionRelocator()
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class DataDistributionRelocatorActor>
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class DataDistributionRelocatorActorState {
															#line 2361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataDistributionRelocatorActorState(DDQueue* const& self,RelocateData const& rd,Future<Void> const& prevCleanup,const DDEnabledState* const& ddEnabledState) 
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rd(rd),
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   prevCleanup(prevCleanup),
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddEnabledState(ddEnabledState),
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   errorOut(self->error),
															#line 1510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   relocateShardInterval("RelocateShard", rd.randomId),
															#line 1511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   dataTransferComplete(self->dataTransferComplete),
															#line 1512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   relocationComplete(self->relocationComplete),
															#line 1513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   signalledTransferComplete(false),
															#line 1514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   distributorId(self->distributorId),
															#line 1515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   healthyDestinations(),
															#line 1517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyHealthy(false),
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   allHealthy(true),
															#line 1519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyWithSource(false),
															#line 1520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyDestOverloaded(false),
															#line 1521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destOverloadedCount(0),
															#line 1522 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   stuckCount(0),
															#line 1523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   bestTeams(),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   startTime(now()),
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destIds(),
															#line 1526 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   wantTrueBest(isValleyFillerPriority(rd.priority)),
															#line 1527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   wantTrueBestIfMoveout(getWantTrueBestIfMoveout(rd.priority)),
															#line 1528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   debugID(deterministicRandom()->randomUInt64()),
															#line 1529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   enableShardMove(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD),
															#line 1533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   doBulkLoading(rd.bulkLoadTask.present())
															#line 2416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("dataDistributionRelocator", reinterpret_cast<unsigned long>(this));

	}
	~DataDistributionRelocatorActorState() 
	{
		fdb_probe_actor_destroy("dataDistributionRelocator", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (now() - self->lastInterval < 1.0)
															#line 2432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					relocateShardInterval.severity = SevDebug;
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->suppressIntervals++;
															#line 2438 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 1541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(relocateShardInterval.begin(), distributorId) .detail("KeyBegin", rd.keys.begin) .detail("KeyEnd", rd.keys.end) .detail("Priority", rd.priority) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("SuppressedEventCount", self->suppressIntervals);
															#line 1548 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (relocateShardInterval.severity != SevDebug)
															#line 2444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->lastInterval = now();
															#line 1550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->suppressIntervals = 0;
															#line 2450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 1553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 1555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.range() == rd.keys);
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.value().randomId == rd.randomId);
															#line 1557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.value().dataMoveId == rd.dataMoveId);
															#line 1558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					inFlightRange.value().cancellable = false;
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					StrictFuture<Void> __when_expr_0 = prevCleanup;
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 1;
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 0, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DataDistributionRelocatorActorState();
		static_cast<DataDistributionRelocatorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			err = e;
															#line 2411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(relocateShardInterval.end(), distributorId) .errorUnsuppressed(err) .detail("Duration", now() - startTime);
															#line 2414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (now() - startTime > 600)
															#line 2514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "RelocateShardTooLong") .errorUnsuppressed(err) .detail("Duration", now() - startTime) .detail("Dest", describe(destIds)) .detail("Src", describe(rd.src));
															#line 2518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 2522 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				dataTransferComplete.send(rd);
															#line 2526 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			relocationComplete.send(rd);
															#line 2426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading && e.code() != error_code_actor_cancelled && e.code() != error_code_movekeys_conflict)
															#line 2532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskRelocatorFailed", self->distributorId) .errorUnsuppressed(e) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
															#line 2536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (err.code() == error_code_data_move_dest_team_not_found)
															#line 2540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_14 = cancelDataMove(self, rd.keys, ddEnabledState);
															#line 2435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1Catch1(__when_expr_14.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_14.get(), loopDepth); };
				static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 13;
															#line 2435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 14, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 2439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (err.code() != error_code_actor_cancelled && err.code() != error_code_data_move_cancelled)
															#line 2558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (errorOut.canBeSet())
															#line 2562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 2441 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						errorOut.sendError(err);
															#line 2566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
				loopDepth = a_body1Catch2cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		parentMetrics = Optional<StorageMetrics>();
															#line 1650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		metrics = StorageMetrics();
															#line 1652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Future<StorageMetrics> metricsF = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(rd.keys)));
															#line 1654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rd.getParentRange().present())
															#line 2590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<StorageMetrics> parentMetricsF = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(rd.getParentRange().get())));
															#line 1657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_3 = store(metrics, metricsF) && store(parentMetrics, parentMetricsF);
															#line 1657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 4;
															#line 1657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 3, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(metrics, metricsF);
															#line 1659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1cont2when2(__when_expr_4.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 5;
															#line 1659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 4, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 2627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			tr = Transaction(self->txnProcessor->context());
															#line 1569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 2633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont5loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 2647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			tr = Transaction(self->txnProcessor->context());
															#line 1569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 2653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont5loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto f = self->dataMoves.intersectingRanges(rd.keys);
															#line 1626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(auto it = f.begin();it != f.end();++it) {
															#line 1627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			KeyRangeRef kr(it->range().begin, it->range().end);
															#line 1628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			const UID mId = it->value().id;
															#line 1629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (mId.isValid() && mId != rd.dataMoveId)
															#line 2750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent("DDRelocatorConflictingDataMove", distributorId) .detail("CurrentDataMoveID", rd.dataMoveId) .detail("DataMoveID", mId) .detail("Range", kr);
															#line 2754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
															#line 1636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rd.isRestore() || !SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 2759 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 2763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(rd.dataMoveId.isValid());
															#line 2767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 1640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(rd.keys, DDQueue::DDDataMove(rd.dataMoveId));
															#line 2771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		DataMoveType dataMoveType = newDataMoveType(doBulkLoading);
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		rd.dataMoveId = newDataMoveId( deterministicRandom()->randomUInt64(), AssignEmptyRange::False, dataMoveType, rd.dmReason);
															#line 1615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskNewDataMoveID", self->distributorId) .detail("DataMoveID", rd.dataMoveId.toString()) .detail("TrackID", rd.randomId) .detail("Range", rd.keys) .detail("Priority", rd.priority) .detail("DataMoveType", dataMoveType) .detail("DoBulkLoading", doBulkLoading) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 2785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
		try {
															#line 1571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<BulkLoadTaskState> __when_expr_1 = getBulkLoadTask(&tr, rd.bulkLoadTask.get().coreState.getRange(), rd.bulkLoadTask.get().coreState.getTaskId(), { BulkLoadPhase::Triggered, BulkLoadPhase::Running });
															#line 1571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont5loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont5loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 2;
															#line 1571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 2844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.bulkLoadTask.get().completeAck.canBeSet())
															#line 2848 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					rd.bulkLoadTask.get().completeAck.sendError(bulkload_task_outdated());
															#line 2852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 1594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				doBulkLoading = false;
															#line 1601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskFallbackToNormalDataMove", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
															#line 2858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 3;
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 2, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2(BulkLoadTaskState const& currentBulkLoadTaskState,int loopDepth) 
	{
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDataMoveLaunched", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("ExistingJobID", currentBulkLoadTaskState.getJobId()) .detail("ExistingTaskID", currentBulkLoadTaskState.getTaskId()) .detail("ExistingTaskRange", currentBulkLoadTaskState.getRange());
															#line 2885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2(BulkLoadTaskState && currentBulkLoadTaskState,int loopDepth) 
	{
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDataMoveLaunched", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("ExistingJobID", currentBulkLoadTaskState.getJobId()) .detail("ExistingTaskID", currentBulkLoadTaskState.getTaskId()) .detail("ExistingTaskRange", currentBulkLoadTaskState.getRange());
															#line 2894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(BulkLoadTaskState const& currentBulkLoadTaskState,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2(currentBulkLoadTaskState, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(BulkLoadTaskState && currentBulkLoadTaskState,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2(std::move(currentBulkLoadTaskState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,BulkLoadTaskState const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,BulkLoadTaskState && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont14(int loopDepth) 
	{
															#line 1662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		excludedDstPhysicalShards = std::unordered_set<uint64_t>();
															#line 1664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(rd.src.size());
															#line 1665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont14loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont14loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1(int loopDepth) 
	{
															#line 1666 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destOverloadedCount = 0;
															#line 1667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		stuckCount = 0;
															#line 1668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		physicalShardIDCandidate = UID().first();
															#line 1669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		forceToUseNewPhysicalShard = false;
															#line 1670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont14loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1(int loopDepth) 
	{
															#line 2007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 3276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!rd.isRestore())
															#line 3280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first());
															#line 2013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (self->physicalShardCollection->physicalShardExists(physicalShardIDCandidate))
															#line 3286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->moveReusePhysicalShard++;
															#line 3290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 2016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->moveCreateNewPhysicalShard++;
															#line 3296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 2018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				rd.dataMoveId = newDataMoveId( physicalShardIDCandidate, AssignEmptyRange::False, newDataMoveType(doBulkLoading), rd.dmReason);
															#line 2020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevInfo, "NewDataMoveWithPhysicalShard") .detail("DataMoveID", rd.dataMoveId.toString()) .detail("Reason", rd.reason.toString()) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 2025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inFlightRange.value().dataMoveId = rd.dataMoveId;
															#line 2026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto f = self->dataMoves.intersectingRanges(rd.keys);
															#line 2027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(auto it = f.begin();it != f.end();++it) {
															#line 2028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					KeyRangeRef kr(it->range().begin, it->range().end);
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					const UID mId = it->value().id;
															#line 2030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (mId.isValid() && mId != rd.dataMoveId)
															#line 3316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 2031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						TraceEvent("DDRelocatorConflictingDataMoveAfterGetTeam", distributorId) .detail("CurrentDataMoveID", rd.dataMoveId) .detail("DataMoveID", mId) .detail("Range", kr);
															#line 3320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataMoves.insert(rd.keys, DDQueue::DDDataMove(rd.dataMoveId));
															#line 3325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rd.dataMoveId.first() != UID().first());
															#line 2040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto dataMoveRange = self->dataMoves.rangeContaining(rd.keys.begin);
															#line 2041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(dataMoveRange.value().id == rd.dataMoveId);
															#line 3333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 2046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(inFlightRange.range() == rd.keys);
															#line 2047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(inFlightRange.value().randomId == rd.randomId);
															#line 2048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		inFlightRange.value().cancellable = false;
															#line 2050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destIds.clear();
															#line 2051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyIds = std::vector<UID>();
															#line 2052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		extraIds = std::vector<UID>();
															#line 2053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destinationTeams = std::vector<ShardsAffectedByTeamFailure::Team>();
															#line 2055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < bestTeams.size();i++) {
															#line 2056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto& serverIds = bestTeams[i].first->getServerIDs();
															#line 2057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destinationTeams.push_back(ShardsAffectedByTeamFailure::Team(serverIds, i == 0));
															#line 2060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (allHealthy && anyWithSource && !bestTeams[i].second && !doBulkLoading)
															#line 3359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				int idx = deterministicRandom()->randomInt(0, serverIds.size());
															#line 2070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destIds.push_back(serverIds[idx]);
															#line 2071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				healthyIds.push_back(serverIds[idx]);
															#line 2072 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(int j = 0;j < serverIds.size();j++) {
															#line 2073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (j != idx)
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						extraIds.push_back(serverIds[j]);
															#line 3375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
															#line 2077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				healthyDestinations.addTeam(bestTeams[i].first);
															#line 3380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destIds.insert(destIds.end(), serverIds.begin(), serverIds.end());
															#line 2080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (bestTeams[i].first->isHealthy())
															#line 3388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					healthyIds.insert(healthyIds.end(), serverIds.begin(), serverIds.end());
															#line 2082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					healthyDestinations.addTeam(bestTeams[i].first);
															#line 3394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
															#line 2088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			validateBulkLoadRelocateData(rd, destIds, self->distributorId);
															#line 3404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevInfo, "DDRelocatorGotDestTeam", self->distributorId) .detail("KeyBegin", rd.keys.begin) .detail("KeyEnd", rd.keys.end) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("SrcIds", describe(rd.src)) .detail("DestId", describe(destIds)) .detail("BulkLoadTaskID", doBulkLoading ? rd.bulkLoadTask.get().coreState.getTaskId().toString() : "");
															#line 2101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		totalIds = 0;
															#line 2102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& destTeam : destinationTeams ) {
															#line 2103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			totalIds += destTeam.servers.size();
															#line 3414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (totalIds < self->teamSize)
															#line 3418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarn, "IncorrectDestTeamSize") .suppressFor(1.0) .detail("ExpectedTeamSize", self->teamSize) .detail("DestTeamSize", totalIds);
															#line 3422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!rd.isRestore())
															#line 3426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(rd.keys, destinationTeams);
															#line 3430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addDataInFlightToTeam(+metrics.bytes);
															#line 2118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addReadInFlightToTeam(+metrics.readLoadKSecond());
															#line 2122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchDest(rd, bestTeams, self->destBusymap);
															#line 2123 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for( const auto& [team, _] : bestTeams ) {
															#line 2125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const UID& ssid : team->getServerIDs() ) {
															#line 2126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->bulkLoadTaskCollection->busyMap.addTask(ssid);
															#line 3448 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
															#line 2131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent ev(relocateShardInterval.severity, "RelocateShardHasDestination", distributorId);
															#line 2132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		RelocateDecision decision{ rd, destIds, extraIds, metrics, parentMetrics };
															#line 2133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceRelocateDecision(ev, relocateShardInterval.pairID, decision);
															#line 2135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(rd.src, rd.reason, DDQueue::ServerCounter::LaunchedSource);
															#line 2136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(destIds, rd.reason, DDQueue::ServerCounter::LaunchedDest);
															#line 2137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(extraIds, rd.reason, DDQueue::ServerCounter::LaunchedDest);
															#line 2139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		error = success();
															#line 2140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		dataMovementComplete = Promise<Void>();
															#line 2142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::unique_ptr<MoveKeysParams> params;
															#line 2143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 3472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, dataMovementComplete, &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, doBulkLoading ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 3476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2159 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, dataMovementComplete, &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, doBulkLoading ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 3482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 2175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		try {
															#line 2179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 3492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont14loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1(int loopDepth) 
	{
															#line 1671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		tciIndex = 0;
															#line 1672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		foundTeams = true;
															#line 1673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeamReady = false;
															#line 1674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = false;
															#line 1675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		allHealthy = true;
															#line 1676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyWithSource = false;
															#line 1677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyDestOverloaded = false;
															#line 1678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.clear();
															#line 1680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont14loopBody1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyDestOverloaded = !canLaunchDest(bestTeams, rd.priority, self->destBusymap);
															#line 1914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyDestOverloaded = false;
															#line 3558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (foundTeams && anyHealthy && !anyDestOverloaded)
															#line 3562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rd.completeDests.empty());
															#line 3566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (foundTeams)
															#line 3571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!anyHealthy)
															#line 3575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]++;
															#line 3579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 1926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (anyDestOverloaded)
															#line 3585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]++;
															#line 3589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
															#line 1931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (anyDestOverloaded)
															#line 3595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "Destination overloaded throttled move");
															#line 1933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destOverloadedCount++;
															#line 1934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(destOverloadedCount > 50 ? SevInfo : SevDebug, "DestSSBusy", distributorId) .suppressFor(1.0) .detail("TraceID", rd.randomId) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("IsRestore", rd.isRestore()) .detail("Priority", rd.priority) .detail("StuckCount", stuckCount) .detail("DestOverloadedCount", destOverloadedCount) .detail("TeamCollectionId", tciIndex) .detail("AnyDestOverloaded", anyDestOverloaded) .detail("NumOfTeamCollections", self->teamCollections.size()) .detail("Servers", destServersString(bestTeams));
															#line 1946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove)
															#line 3605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.isRestore() && destOverloadedCount > 50)
															#line 3609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
															#line 1951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(SERVER_KNOBS->DEST_OVERLOADED_DELAY, TaskPriority::DataDistributionLaunch);
															#line 1951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), std::max(0, loopDepth - 2)); else return a_body1cont14loopBody1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 8;
															#line 1951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 7, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "did not find a healthy destination team on the first attempt");
															#line 1954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			stuckCount++;
															#line 1955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(stuckCount > 50 ? SevWarnAlways : SevWarn, "BestTeamStuck", distributorId) .suppressFor(1.0) .detail("TraceID", rd.randomId) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("IsRestore", rd.isRestore()) .detail("Priority", rd.priority) .detail("StuckCount", stuckCount) .detail("DestOverloadedCount", destOverloadedCount) .detail("TeamCollectionId", tciIndex) .detail("AnyDestOverloaded", anyDestOverloaded) .detail("NumOfTeamCollections", self->teamCollections.size()) .detail("ConductingBulkLoad", doBulkLoading);
															#line 1967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading && stuckCount == 50)
															#line 3638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(rd.bulkLoadTask.present());
															#line 1969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskStuck", self->distributorId) .detail("TraceID", rd.randomId) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("DataMoveID", rd.dataMoveId) .detail("Reason", rd.reason.toString()) .detail("Priority", rd.priority) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.bulkLoadTask.get().completeAck.canBeSet())
															#line 3646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					rd.bulkLoadTask.get().completeAck.send( BulkLoadAck( true, rd.priority));
															#line 1982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
															#line 1988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.isRestore() && stuckCount > 50)
															#line 3657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1989 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 1991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_8 = delay(SERVER_KNOBS->BEST_TEAM_STUCK_DELAY, TaskPriority::DataDistributionLaunch);
															#line 1991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), std::max(0, loopDepth - 2)); else return a_body1cont14loopBody1loopBody1cont1when2(__when_expr_8.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 9;
															#line 1991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 8, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1(int loopDepth) 
	{
															#line 1680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!(tciIndex < self->teamCollections.size()))
															#line 3689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && rd.isRestore())
															#line 3695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto req = GetTeamRequest(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest);
															#line 1683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.keys = rd.keys;
															#line 1684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> fbestTeam = brokenPromiseToNever(self->teamCollections[tciIndex].getTeam.getReply(req));
															#line 1686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeamReady = fbestTeam.isReady();
															#line 1687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> __when_expr_5 = fbestTeam;
															#line 1687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 3));
															#line 3709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 3)); else return a_body1cont14loopBody1loopBody1loopBody1when1(__when_expr_5.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 6;
															#line 1687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			double inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_HEALTHY;
															#line 1725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT)
															#line 3723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_UNHEALTHY;
															#line 3727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 1728 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT)
															#line 3731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_ONE_LEFT;
															#line 3735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 1733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TeamSelect destTeamSelect;
															#line 1734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!rd.wantsNewServers)
															#line 3741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destTeamSelect = TeamSelect::WANT_COMPLETE_SRCS;
															#line 3745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 1736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wantTrueBest)
															#line 3751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					destTeamSelect = TeamSelect::WANT_TRUE_BEST;
															#line 3755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 1739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					destTeamSelect = TeamSelect::ANY;
															#line 3761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
															#line 1741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			PreferLowerReadUtil preferLowerReadTeam = SERVER_KNOBS->DD_PREFER_LOW_READ_UTIL_TEAM || rd.reason == RelocateReason::REBALANCE_READ ? PreferLowerReadUtil::True : PreferLowerReadUtil::False;
															#line 1745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto req = GetTeamRequest(destTeamSelect, PreferLowerDiskUtil::True, TeamMustHaveShards::False, preferLowerReadTeam, PreferWithinShardLimit::True, ForReadBalance(rd.reason == RelocateReason::REBALANCE_READ), inflightPenalty, rd.keys);
															#line 1754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.src = rd.src;
															#line 1755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.completeSources = rd.completeSources;
															#line 1756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.storageQueueAware = SERVER_KNOBS->ENABLE_STORAGE_QUEUE_AWARE_TEAM_SELECTION;
															#line 1757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.findTeamForBulkLoad = doBulkLoading;
															#line 1758 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.wantTrueBestIfMoveout = wantTrueBestIfMoveout;
															#line 1760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove && tciIndex == 1)
															#line 3780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 1763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				std::pair<Optional<ShardsAffectedByTeamFailure::Team>, bool> remoteTeamWithPhysicalShard = self->physicalShardCollection->tryGetAvailableRemoteTeamWith( physicalShardIDCandidate, metrics, debugID);
															#line 1766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!remoteTeamWithPhysicalShard.second)
															#line 3788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 3794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (remoteTeamWithPhysicalShard.first.present())
															#line 3799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1776 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					req = GetTeamRequest(remoteTeamWithPhysicalShard.first.get().servers);
															#line 1777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					req.keys = rd.keys;
															#line 3805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
															#line 1784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> fbestTeam = brokenPromiseToNever(self->teamCollections[tciIndex].getTeam.getReply(req));
															#line 1786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeamReady = fbestTeam.isReady();
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> __when_expr_6 = fbestTeam;
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 3));
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), std::max(0, loopDepth - 3)); else return a_body1cont14loopBody1loopBody1loopBody1when2(__when_expr_6.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 7;
															#line 1787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		tciIndex++;
															#line 3844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont14loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont3(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
															#line 1688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 3853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady") .detail("TraceID", rd.randomId) .detail("TeamCollectionIndex", tciIndex) .detail("RestoreDataMoveForDest", describe(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest));
															#line 1697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present() || !bestTeam.first.get()->isHealthy())
															#line 3866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = true;
															#line 1709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "Restore") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 3885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont3(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
															#line 1688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 3895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady") .detail("TraceID", rd.randomId) .detail("TeamCollectionIndex", tciIndex) .detail("RestoreDataMoveForDest", describe(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest));
															#line 1697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present() || !bestTeam.first.get()->isHealthy())
															#line 3908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = true;
															#line 1709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "Restore") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 3927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when1(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont3(bestTeam, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when1(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont3(std::move(bestTeam), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont14loopBody1loopBody1loopBody1cont8(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
															#line 1788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 4012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorBestTeamReceived", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("BestTeamReady", bestTeamReady);
															#line 4016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 4020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady");
															#line 1802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present())
															#line 4033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.get()->isHealthy())
															#line 4044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			allHealthy = false;
															#line 4048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 1818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyHealthy = true;
															#line 4054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (bestTeam.second)
															#line 4058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyWithSource = true;
															#line 4062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1825 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4066 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 1 && !forceToUseNewPhysicalShard)
															#line 4070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool minAvailableSpaceRatio = bestTeam.first.get()->getMinAvailableSpaceRatio(true);
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (minAvailableSpaceRatio < SERVER_KNOBS->TARGET_AVAILABLE_SPACE_RATIO)
															#line 4076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]++;
															#line 1836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!bestTeam.first.get()->isHealthy())
															#line 4087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]++;
															#line 1849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
			}
															#line 1854 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), true);
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 0)
															#line 4101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(foundTeams);
															#line 1863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ShardsAffectedByTeamFailure::Team primaryTeam = ShardsAffectedByTeamFailure::Team(bestTeams[0].first->getServerIDs(), true);
															#line 1866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (forceToUseNewPhysicalShard)
															#line 4109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					physicalShardIDCandidate = self->physicalShardCollection->generateNewPhysicalShardID(debugID);
															#line 4113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 1870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					Optional<uint64_t> candidate = self->physicalShardCollection->trySelectAvailablePhysicalShardFor( primaryTeam, metrics, excludedDstPhysicalShards, debugID);
															#line 1873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (candidate.present())
															#line 4121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						physicalShardIDCandidate = candidate.get();
															#line 4125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
					else
					{
															#line 1876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						if (wantTrueBest)
															#line 4133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						{
															#line 1882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
							forceToUseNewPhysicalShard = true;
															#line 4137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						}
															#line 1884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						foundTeams = false;
															#line 4141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
					}
				}
															#line 1888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 4147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		else
		{
															#line 1892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "New") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 4160 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont8(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
															#line 1788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 4171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorBestTeamReceived", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("BestTeamReady", bestTeamReady);
															#line 4175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 4179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady");
															#line 1802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present())
															#line 4192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.get()->isHealthy())
															#line 4203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			allHealthy = false;
															#line 4207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 1818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyHealthy = true;
															#line 4213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (bestTeam.second)
															#line 4217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyWithSource = true;
															#line 4221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 1825 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 1 && !forceToUseNewPhysicalShard)
															#line 4229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool minAvailableSpaceRatio = bestTeam.first.get()->getMinAvailableSpaceRatio(true);
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (minAvailableSpaceRatio < SERVER_KNOBS->TARGET_AVAILABLE_SPACE_RATIO)
															#line 4235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]++;
															#line 1836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!bestTeam.first.get()->isHealthy())
															#line 4246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]++;
															#line 1849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
			}
															#line 1854 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), true);
															#line 1861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 0)
															#line 4260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(foundTeams);
															#line 1863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ShardsAffectedByTeamFailure::Team primaryTeam = ShardsAffectedByTeamFailure::Team(bestTeams[0].first->getServerIDs(), true);
															#line 1866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (forceToUseNewPhysicalShard)
															#line 4268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					physicalShardIDCandidate = self->physicalShardCollection->generateNewPhysicalShardID(debugID);
															#line 4272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
				else
				{
															#line 1870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					Optional<uint64_t> candidate = self->physicalShardCollection->trySelectAvailablePhysicalShardFor( primaryTeam, metrics, excludedDstPhysicalShards, debugID);
															#line 1873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (candidate.present())
															#line 4280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					{
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						physicalShardIDCandidate = candidate.get();
															#line 4284 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
					else
					{
															#line 1876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						if (wantTrueBest)
															#line 4292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						{
															#line 1882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
							forceToUseNewPhysicalShard = true;
															#line 4296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						}
															#line 1884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						foundTeams = false;
															#line 4300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
						return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
					}
				}
															#line 1888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 4306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		else
		{
															#line 1892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "New") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 4319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when2(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont8(bestTeam, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when2(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont8(std::move(bestTeam), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont14loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 1996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 1997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (destOverloadedCount + stuckCount > 20)
															#line 4409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 1998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]++;
															#line 1999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				forceToUseNewPhysicalShard = true;
															#line 4415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			excludedDstPhysicalShards.insert(physicalShardIDCandidate);
															#line 4419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont14loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont14loopBody1loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont14loopBody1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont14loopBody1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont14loopBody1cont2(int loopDepth) 
	{
															#line 2263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (error.code() != error_code_move_to_removed_server)
															#line 4603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!error.code())
															#line 4607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
				try {
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					StrictFuture<Void> __when_expr_12 = healthyDestinations .updateStorageMetrics();
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont14loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1cont14loopBody1cont2Catch1(__when_expr_12.getError(), loopDepth); else return a_body1cont14loopBody1cont2when1(__when_expr_12.get(), loopDepth); };
					static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 11;
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 12, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1cont14loopBody1cont2Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont14loopBody1cont2Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont14loopBody1cont26(loopDepth);
			}
		}
		else
		{
															#line 2379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "move to removed server", probe::decoration::rare);
															#line 2380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyDestinations.addDataInFlightToTeam(-metrics.bytes);
															#line 2381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto readLoad = metrics.readLoadKSecond();
															#line 2382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto& destinationRef = healthyDestinations;
															#line 2383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->noErrorActors.add( trigger([destinationRef, readLoad]() mutable { destinationRef.addReadInFlightToTeam(-readLoad); }, delay(SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL)));
															#line 2387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				completeDest(rd, self->destBusymap);
															#line 4651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rd.completeDests.clear();
															#line 2394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorError") .errorUnsuppressed(error) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 2399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 4667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
			}
															#line 2406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_13 = delay(SERVER_KNOBS->RETRY_RELOCATESHARD_DELAY, TaskPriority::DataDistributionLaunch);
															#line 2406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1Catch2(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont2when2(__when_expr_13.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 12;
															#line 2406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 13, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			error = e;
															#line 4691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont14loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont24(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont24cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 2181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_9 = doMoveKeys;
															#line 2180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont14loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when1(__when_expr_9.get(), loopDepth); };
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_10 = pollHealth;
															#line 4725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when2(__when_expr_10.get(), loopDepth); };
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_11 = signalledTransferComplete ? Never() : dataMovementComplete.getFuture();
															#line 4729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when3(__when_expr_11.get(), loopDepth); };
		static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 10;
															#line 2181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 9, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 10, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 11, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1cont24(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont14loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 2182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (extraIds.size())
															#line 4766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destIds.insert(destIds.end(), extraIds.begin(), extraIds.end());
															#line 2184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyIds.insert(healthyIds.end(), extraIds.begin(), extraIds.end());
															#line 2185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			extraIds.clear();
															#line 2186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(totalIds == destIds.size());
															#line 2187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::unique_ptr<MoveKeysParams> params;
															#line 2188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 4794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->fetchKeysComplete.insert(rd);
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto ranges = self->dataMoves.getAffectedRangesAfterInsertion(rd.keys);
															#line 2226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (ranges.size() == 1 && static_cast<KeyRange>(ranges[0]) == rd.keys && ranges[0].value.id == rd.dataMoveId && !ranges[0].value.cancel.isValid())
															#line 4808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->dataMoves.insert(rd.keys, DDQueue::DDDataMove());
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevVerbose, "DequeueDataMoveOnSuccess", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveRange", rd.keys);
															#line 4814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
			return a_body1cont14loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 2182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (extraIds.size())
															#line 4827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destIds.insert(destIds.end(), extraIds.begin(), extraIds.end());
															#line 2184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyIds.insert(healthyIds.end(), extraIds.begin(), extraIds.end());
															#line 2185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			extraIds.clear();
															#line 2186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(totalIds == destIds.size());
															#line 2187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::unique_ptr<MoveKeysParams> params;
															#line 2188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4841 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 4855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->fetchKeysComplete.insert(rd);
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto ranges = self->dataMoves.getAffectedRangesAfterInsertion(rd.keys);
															#line 2226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (ranges.size() == 1 && static_cast<KeyRange>(ranges[0]) == rd.keys && ranges[0].value.id == rd.dataMoveId && !ranges[0].value.cancel.isValid())
															#line 4869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->dataMoves.insert(rd.keys, DDQueue::DDDataMove());
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevVerbose, "DequeueDataMoveOnSuccess", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveRange", rd.keys);
															#line 4875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
			return a_body1cont14loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!healthyDestinations.isHealthy())
															#line 4888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataTransferComplete.send(rd);
															#line 4898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 4903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!healthyDestinations.isHealthy())
															#line 4912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataTransferComplete.send(rd);
															#line 4922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 4927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when3(Void const& _,int loopDepth) 
	{
															#line 2249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.insert(rd);
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!signalledTransferComplete)
															#line 4938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			signalledTransferComplete = true;
															#line 2252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataTransferComplete.send(rd);
															#line 4944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when3(Void && _,int loopDepth) 
	{
															#line 2249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.insert(rd);
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!signalledTransferComplete)
															#line 4956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			signalledTransferComplete = true;
															#line 2252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataTransferComplete.send(rd);
															#line 4962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 9, Void >::remove();
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 10, Void >::remove();
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont14loopBody1cont24cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont25(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont14loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26(int loopDepth) 
	{
															#line 2274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addDataInFlightToTeam(-metrics.bytes);
															#line 2275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto readLoad = metrics.readLoadKSecond();
															#line 2280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& destinationRef = healthyDestinations;
															#line 2281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( trigger([destinationRef, readLoad]() mutable { destinationRef.addReadInFlightToTeam(-readLoad); }, delay(SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL)));
															#line 2286 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!error.code())
															#line 5178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(relocateShardInterval.end(), distributorId) .detail("Duration", now() - startTime) .detail("Result", "Success");
															#line 2290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DataMoveStats", distributorId) .detail("Duration", now() - startTime) .detail("Bytes", metrics.bytes) .detail("Rate", static_cast<double>(metrics.bytes) / (now() - startTime)) .detail("Reason", rd.reason.toString()) .detail("DataMoveReason", static_cast<int>(rd.dmReason)) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveType", getDataMoveTypeFromDataMoveId(rd.dataMoveId));
															#line 2298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (now() - startTime > 600)
															#line 5186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "RelocateShardTooLong") .detail("Duration", now() - startTime) .detail("Dest", describe(destIds)) .detail("Src", describe(rd.src));
															#line 5190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.keys.begin == keyServersPrefix)
															#line 5194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent("MovedKeyServerKeys") .detail("Dest", describe(destIds)) .trackLatest(self->movedKeyServersEventHolder->trackingKey);
															#line 5198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 5202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				dataTransferComplete.send(rd);
															#line 5208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			const int nonOverlappingCount = nonOverlappedServerCount(rd.completeSources, destIds);
															#line 2319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->bytesWritten += metrics.bytes;
															#line 2320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveBytesRate.addSample(metrics.bytes * nonOverlappingCount);
															#line 2321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->shardsAffectedByTeamFailure->finishMove(rd.keys);
															#line 2322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			relocationComplete.send(rd);
															#line 2324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove)
															#line 5222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				std::vector<ShardsAffectedByTeamFailure::Team> selectedTeams;
															#line 2327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(int i = 0;i < bestTeams.size();i++) {
															#line 2328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					auto serverIds = bestTeams[i].first->getServerIDs();
															#line 2329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					selectedTeams.push_back(ShardsAffectedByTeamFailure::Team(serverIds, i == 0));
															#line 5232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
															#line 2333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->physicalShardCollection->updatePhysicalShardCollection( rd.keys, rd.isRestore(), selectedTeams, rd.dataMoveId.first(), metrics, debugID);
															#line 5236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 5240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 5248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
				try {
															#line 2344 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->bulkLoadTaskCollection->terminateTask(rd.bulkLoadTask.get().coreState);
															#line 2345 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorComplete", self->distributorId) .detail("Dests", describe(destIds)) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 5256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					loopDepth = a_body1cont14loopBody1cont26cont12(loopDepth);
				}
				catch (Error& error) {
					loopDepth = a_body1cont14loopBody1cont26Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont14loopBody1cont26Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont14loopBody1cont26cont2(loopDepth);
			}
		}
		else
		{
															#line 2365 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 5274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorError") .errorUnsuppressed(error) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 2370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 5284 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
					}
				}
			}
															#line 2376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch2(error, std::max(0, loopDepth - 1));
															#line 5290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont27(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			error = e;
															#line 5306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont14loopBody1cont27(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont28(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont28(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont14loopBody1cont28cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont27(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont2(int loopDepth) 
	{
															#line 2363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<DataDistributionRelocatorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DataDistributionRelocatorActorState(); static_cast<DataDistributionRelocatorActor*>(this)->destroy(); return 0; }
															#line 5421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<DataDistributionRelocatorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DataDistributionRelocatorActorState();
		static_cast<DataDistributionRelocatorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont8(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont26cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26Catch1(const Error& bulkLoadError,int loopDepth=0) 
	{
		try {
															#line 2351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT_WE_THINK(bulkLoadError.code() == error_code_bulkload_task_outdated);
															#line 2352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (bulkLoadError.code() != error_code_bulkload_task_outdated)
															#line 5442 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				return a_body1Catch2(bulkLoadError, std::max(0, loopDepth - 1));
															#line 5446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorCompleteButOutdated", self->distributorId) .detail("Dests", describe(destIds)) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 5450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1cont14loopBody1cont26cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont26cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 13, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont14loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont14loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	int a_body1Catch2cont1(int loopDepth) 
	{
															#line 2444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 5565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont5(Void const& _,int loopDepth) 
	{
															#line 2436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevWarnAlways, "RelocateShardCancelDataMoveTeamNotFound") .detail("Src", describe(rd.src)) .detail("DataMoveMetaData", rd.dataMove != nullptr ? rd.dataMove->meta.toString() : "Empty");
															#line 5573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont5(Void && _,int loopDepth) 
	{
															#line 2436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevWarnAlways, "RelocateShardCancelDataMoveTeamNotFound") .detail("Src", describe(rd.src)) .detail("DataMoveMetaData", rd.dataMove != nullptr ? rd.dataMove->meta.toString() : "Empty");
															#line 5582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 14, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData rd;
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> prevCleanup;
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const DDEnabledState* ddEnabledState;
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Promise<Void> errorOut;
															#line 1510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceInterval relocateShardInterval;
															#line 1511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> dataTransferComplete;
															#line 1512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> relocationComplete;
															#line 1513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool signalledTransferComplete;
															#line 1514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	UID distributorId;
															#line 1515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	ParallelTCInfo healthyDestinations;
															#line 1517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyHealthy;
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool allHealthy;
															#line 1519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyWithSource;
															#line 1520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyDestOverloaded;
															#line 1521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int destOverloadedCount;
															#line 1522 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int stuckCount;
															#line 1523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<std::pair<Reference<IDataDistributionTeam>, bool>> bestTeams;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double startTime;
															#line 1525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> destIds;
															#line 1526 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	WantTrueBest wantTrueBest;
															#line 1527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	WantTrueBestIfMoveout wantTrueBestIfMoveout;
															#line 1528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	uint64_t debugID;
															#line 1529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool enableShardMove;
															#line 1533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool doBulkLoading;
															#line 1568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Transaction tr;
															#line 1649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Optional<StorageMetrics> parentMetrics;
															#line 1650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	StorageMetrics metrics;
															#line 1662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::unordered_set<uint64_t> excludedDstPhysicalShards;
															#line 1668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	uint64_t physicalShardIDCandidate;
															#line 1669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool forceToUseNewPhysicalShard;
															#line 1671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int tciIndex;
															#line 1672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool foundTeams;
															#line 1673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool bestTeamReady;
															#line 2051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> healthyIds;
															#line 2052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> extraIds;
															#line 2053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<ShardsAffectedByTeamFailure::Team> destinationTeams;
															#line 2101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int totalIds;
															#line 2139 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Error error;
															#line 2140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Promise<Void> dataMovementComplete;
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> doMoveKeys;
															#line 2175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> pollHealth;
															#line 2410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Error err;
															#line 5748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via dataDistributionRelocator()
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class DataDistributionRelocatorActor final : public Actor<Void>, public ActorCallback< DataDistributionRelocatorActor, 0, Void >, public ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >, public ActorCallback< DataDistributionRelocatorActor, 2, Void >, public ActorCallback< DataDistributionRelocatorActor, 3, Void >, public ActorCallback< DataDistributionRelocatorActor, 4, Void >, public ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >, public ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >, public ActorCallback< DataDistributionRelocatorActor, 7, Void >, public ActorCallback< DataDistributionRelocatorActor, 8, Void >, public ActorCallback< DataDistributionRelocatorActor, 9, Void >, public ActorCallback< DataDistributionRelocatorActor, 10, Void >, public ActorCallback< DataDistributionRelocatorActor, 11, Void >, public ActorCallback< DataDistributionRelocatorActor, 12, Void >, public ActorCallback< DataDistributionRelocatorActor, 13, Void >, public ActorCallback< DataDistributionRelocatorActor, 14, Void >, public FastAllocated<DataDistributionRelocatorActor>, public DataDistributionRelocatorActorState<DataDistributionRelocatorActor> {
															#line 5753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<DataDistributionRelocatorActor>::operator new;
	using FastAllocated<DataDistributionRelocatorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14103592050235720192UL, 5780745182796113920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DataDistributionRelocatorActor, 0, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >;
friend struct ActorCallback< DataDistributionRelocatorActor, 2, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 3, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 4, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >;
friend struct ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >;
friend struct ActorCallback< DataDistributionRelocatorActor, 7, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 8, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 9, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 10, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 11, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 12, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 13, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 14, Void >;
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataDistributionRelocatorActor(DDQueue* const& self,RelocateData const& rd,Future<Void> const& prevCleanup,const DDEnabledState* const& ddEnabledState) 
															#line 5784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   DataDistributionRelocatorActorState<DataDistributionRelocatorActor>(self, rd, prevCleanup, ddEnabledState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2631274672844690944UL, 4050065294653027072UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("dataDistributionRelocator");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 12, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 13, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 14, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> dataDistributionRelocator( DDQueue* const& self, RelocateData const& rd, Future<Void> const& prevCleanup, const DDEnabledState* const& ddEnabledState ) {
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new DataDistributionRelocatorActor(self, rd, prevCleanup, ddEnabledState));
															#line 5829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

inline double getWorstCpu(const HealthMetrics& metrics, const std::vector<UID>& ids) {
	double cpu = 0;
	for (auto& id : ids) {
		if (metrics.storageStats.contains(id)) {
			cpu = std::max(cpu, metrics.storageStats.at(id).cpuUsage);
		} else {
			// assume the server is too busy to report its stats
			cpu = std::max(cpu, 100.0);
			break;
		}
	}
	return cpu;
}

// Move the shard with the top K highest read density of sourceTeam's to destTeam if sourceTeam has much more read
// load than destTeam
															#line 5850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via rebalanceReadLoad()
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RebalanceReadLoadActor>
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceReadLoadActorState {
															#line 5857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceReadLoadActorState(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam> const& sourceTeam,Reference<IDataDistributionTeam> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveReason(moveReason),
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   sourceTeam(sourceTeam),
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destTeam(destTeam),
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   primary(primary),
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 5874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("rebalanceReadLoad", reinterpret_cast<unsigned long>(this));

	}
	~RebalanceReadLoadActorState() 
	{
		fdb_probe_actor_destroy("rebalanceReadLoad", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (g_network->isSimulated() && g_simulator->speedUpSimulation)
															#line 5889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("CancelingDueToSimulationSpeedup", true);
															#line 2472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("ShardsInSource", shards.size());
															#line 2481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shards.size() <= 1)
															#line 5907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "NoShardOnSource");
															#line 2483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (self->timeThrottle(sourceTeam->getServerIDs()))
															#line 5921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "SourceTeamThrottle");
															#line 2492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto srcLoad = sourceTeam->getReadLoad(false), destLoad = destTeam->getReadLoad();
															#line 2496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SrcReadBandwidth", srcLoad).detail("DestReadBandwidth", destLoad);
															#line 2499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((1.0 - SERVER_KNOBS->READ_REBALANCE_DIFF_FRAC) * srcLoad <= destLoad)
															#line 5939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "TeamTooSimilar");
															#line 2501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int topK = std::max(1, std::min(int(0.1 * shards.size()), SERVER_KNOBS->READ_REBALANCE_SHARD_TOPK));
															#line 2505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthMetrics = self->txnProcessor->getHealthMetrics(true);
															#line 2506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req = GetTopKMetricsRequest(shards, topK, (srcLoad - destLoad) * SERVER_KNOBS->READ_REBALANCE_MAX_SHARD_FRAC, std::min(srcLoad / shards.size(), SERVER_KNOBS->READ_REBALANCE_MIN_READ_BYTES_KS));
															#line 2510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<GetTopKMetricsReply> __when_expr_0 = brokenPromiseToNever(self->getTopKMetrics.getReply(req));
															#line 2510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 1;
															#line 2510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*>(static_cast<RebalanceReadLoadActor*>(this)));
															#line 5966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = ready(healthMetrics);
															#line 2511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 2;
															#line 2511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RebalanceReadLoadActor, 1, Void >*>(static_cast<RebalanceReadLoadActor*>(this)));
															#line 5996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(GetTopKMetricsReply const& __reply,int loopDepth) 
	{
															#line 2510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		reply = __reply;
															#line 6005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetTopKMetricsReply && __reply,int loopDepth) 
	{
		reply = std::move(__reply);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state > 0) static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceReadLoadActor*>(this)->ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,GetTopKMetricsReply const& value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,GetTopKMetricsReply && value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto cpu = getWorstCpu(healthMetrics.get(), sourceTeam->getServerIDs());
															#line 2513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (cpu < SERVER_KNOBS->READ_REBALANCE_CPU_THRESHOLD)
															#line 6086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "LowReadLoad").detail("WorstSrcCpu", cpu);
															#line 2515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& metricsList = reply.shardMetrics;
															#line 2520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		deterministicRandom()->randomShuffle(metricsList);
															#line 2521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("MinReadLoad", reply.minReadLoad).detail("MaxReadLoad", reply.maxReadLoad);
															#line 2523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (metricsList.empty())
															#line 6106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoEligibleShards");
															#line 2525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& [shard, metrics] = metricsList[0];
															#line 2529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("ShardReadBandwidth", metrics.bytesReadPerKSecond) .detail("ShardReadOps", metrics.opsReadPerKSecond);
															#line 2533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shard == shards[i])
															#line 6128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(shard, moveReason, RelocateReason::REBALANCE_READ, traceId));
															#line 2539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto serverIds = sourceTeam->getServerIDs();
															#line 2542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->updateLastAsSource(serverIds);
															#line 2544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( serverIds, RelocateReason::REBALANCE_READ, DDQueue::ServerCounter::ProposedSource);
															#line 2546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto cpu = getWorstCpu(healthMetrics.get(), sourceTeam->getServerIDs());
															#line 2513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (cpu < SERVER_KNOBS->READ_REBALANCE_CPU_THRESHOLD)
															#line 6169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "LowReadLoad").detail("WorstSrcCpu", cpu);
															#line 2515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& metricsList = reply.shardMetrics;
															#line 2520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		deterministicRandom()->randomShuffle(metricsList);
															#line 2521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("MinReadLoad", reply.minReadLoad).detail("MaxReadLoad", reply.maxReadLoad);
															#line 2523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (metricsList.empty())
															#line 6189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoEligibleShards");
															#line 2525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& [shard, metrics] = metricsList[0];
															#line 2529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("ShardReadBandwidth", metrics.bytesReadPerKSecond) .detail("ShardReadOps", metrics.opsReadPerKSecond);
															#line 2533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shard == shards[i])
															#line 6211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(shard, moveReason, RelocateReason::REBALANCE_READ, traceId));
															#line 2539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto serverIds = sourceTeam->getServerIDs();
															#line 2542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->updateLastAsSource(serverIds);
															#line 2544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( serverIds, RelocateReason::REBALANCE_READ, DDQueue::ServerCounter::ProposedSource);
															#line 2546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state > 0) static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceReadLoadActor*>(this)->ActorCallback< RebalanceReadLoadActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason moveReason;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> sourceTeam;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> destTeam;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool primary;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<KeyRange> shards;
															#line 2505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<HealthMetrics> healthMetrics;
															#line 2506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTopKMetricsRequest req;
															#line 2510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTopKMetricsReply reply;
															#line 6341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via rebalanceReadLoad()
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceReadLoadActor final : public Actor<bool>, public ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >, public ActorCallback< RebalanceReadLoadActor, 1, Void >, public FastAllocated<RebalanceReadLoadActor>, public RebalanceReadLoadActorState<RebalanceReadLoadActor> {
															#line 6346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<RebalanceReadLoadActor>::operator new;
	using FastAllocated<RebalanceReadLoadActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15931676592777635584UL, 11740538997370116096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >;
friend struct ActorCallback< RebalanceReadLoadActor, 1, Void >;
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceReadLoadActor(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam> const& sourceTeam,Reference<IDataDistributionTeam> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 6364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<bool>(),
		   RebalanceReadLoadActorState<RebalanceReadLoadActor>(self, moveReason, sourceTeam, destTeam, primary, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(257547752678163200UL, 2881630330149849344UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("rebalanceReadLoad");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RebalanceReadLoadActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<bool> rebalanceReadLoad( DDQueue* const& self, DataMovementReason const& moveReason, Reference<IDataDistributionTeam> const& sourceTeam, Reference<IDataDistributionTeam> const& destTeam, bool const& primary, TraceEvent* const& traceEvent ) {
															#line 2464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new RebalanceReadLoadActor(self, moveReason, sourceTeam, destTeam, primary, traceEvent));
															#line 6398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

// Move a random shard from sourceTeam if sourceTeam has much more data than provided destTeam
															#line 6404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via rebalanceTeams()
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RebalanceTeamsActor>
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceTeamsActorState {
															#line 6411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceTeamsActorState(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam const> const& sourceTeam,Reference<IDataDistributionTeam const> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveReason(moveReason),
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   sourceTeam(sourceTeam),
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destTeam(destTeam),
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   primary(primary),
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 6428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("rebalanceTeams", reinterpret_cast<unsigned long>(this));

	}
	~RebalanceTeamsActorState() 
	{
		fdb_probe_actor_destroy("rebalanceTeams", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (g_network->isSimulated() && g_simulator->speedUpSimulation)
															#line 6443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("CancelingDueToSimulationSpeedup", true);
															#line 2562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceTeamsActorState();
				static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Promise<int64_t> req;
															#line 2566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->getAverageShardBytes.send(req);
															#line 2568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<int64_t> __when_expr_0 = req.getFuture();
															#line 2568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 1;
															#line 2568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RebalanceTeamsActor, 0, int64_t >*>(static_cast<RebalanceTeamsActor*>(this)));
															#line 6468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RebalanceTeamsActorState();
		static_cast<RebalanceTeamsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("AverageShardBytes", averageShardBytes).detail("ShardsInSource", shards.size());
															#line 2574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!shards.size())
															#line 6495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoShardOnSource");
															#line 2576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceTeamsActorState();
			static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		moveShard = KeyRange();
															#line 2580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		metrics = StorageMetrics();
															#line 2581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries = 0;
															#line 2582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 6515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int64_t const& __averageShardBytes,int loopDepth) 
	{
															#line 2568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		averageShardBytes = __averageShardBytes;
															#line 6524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int64_t && __averageShardBytes,int loopDepth) 
	{
		averageShardBytes = std::move(__averageShardBytes);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state > 0) static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceTeamsActor*>(this)->ActorCallback< RebalanceTeamsActor, 0, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		int64_t sourceBytes = sourceTeam->getLoadBytes(false);
															#line 2597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		int64_t destBytes = destTeam->getLoadBytes();
															#line 2599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool sourceAndDestTooSimilar = sourceBytes - destBytes <= 3 * std::max<int64_t>(SERVER_KNOBS->MIN_SHARD_BYTES, metrics.bytes);
															#line 2601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SourceBytes", sourceBytes) .detail("DestBytes", destBytes) .detail("ShardBytes", metrics.bytes) .detail("SourceAndDestTooSimilar", sourceAndDestTooSimilar);
															#line 2606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceAndDestTooSimilar || metrics.bytes == 0)
															#line 6611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", sourceAndDestTooSimilar ? "TeamTooSimilar" : "ShardZeroSize");
															#line 2608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceTeamsActorState();
			static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (moveShard == shards[i])
															#line 6629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(moveShard, moveReason, RelocateReason::REBALANCE_DISK, traceId));
															#line 2618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( sourceTeam->getServerIDs(), RelocateReason::REBALANCE_DISK, DDQueue::ServerCounter::ProposedSource);
															#line 2622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6641 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceTeamsActorState();
				static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceTeamsActorState();
		static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 2582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!(retries < SERVER_KNOBS->REBALANCE_MAX_RETRIES))
															#line 6671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2583 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		testShard = deterministicRandom()->randomChoice(shards);
															#line 2584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<StorageMetrics> __when_expr_1 = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(testShard)));
															#line 2584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 2;
															#line 2584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*>(static_cast<RebalanceTeamsActor*>(this)));
															#line 6686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(StorageMetrics const& testMetrics,int loopDepth) 
	{
															#line 2586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (testMetrics.bytes > metrics.bytes)
															#line 6708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			moveShard = testShard;
															#line 2588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			metrics = testMetrics;
															#line 2589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (metrics.bytes > averageShardBytes)
															#line 6716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
															#line 2593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries++;
															#line 6723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(StorageMetrics && testMetrics,int loopDepth) 
	{
															#line 2586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (testMetrics.bytes > metrics.bytes)
															#line 6732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			moveShard = testShard;
															#line 2588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			metrics = testMetrics;
															#line 2589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (metrics.bytes > averageShardBytes)
															#line 6740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
															#line 2593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries++;
															#line 6747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(StorageMetrics const& testMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(testMetrics, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(StorageMetrics && testMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(testMetrics), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state > 0) static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceTeamsActor*>(this)->ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,StorageMetrics const& value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,StorageMetrics && value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason moveReason;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam const> sourceTeam;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam const> destTeam;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool primary;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int64_t averageShardBytes;
															#line 2569 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<KeyRange> shards;
															#line 2579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange moveShard;
															#line 2580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	StorageMetrics metrics;
															#line 2581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int retries;
															#line 2583 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange testShard;
															#line 6851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via rebalanceTeams()
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceTeamsActor final : public Actor<bool>, public ActorCallback< RebalanceTeamsActor, 0, int64_t >, public ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >, public FastAllocated<RebalanceTeamsActor>, public RebalanceTeamsActorState<RebalanceTeamsActor> {
															#line 6856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<RebalanceTeamsActor>::operator new;
	using FastAllocated<RebalanceTeamsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5425735919675171328UL, 12511827423552781568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RebalanceTeamsActor, 0, int64_t >;
friend struct ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >;
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceTeamsActor(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam const> const& sourceTeam,Reference<IDataDistributionTeam const> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 6874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<bool>(),
		   RebalanceTeamsActorState<RebalanceTeamsActor>(self, moveReason, sourceTeam, destTeam, primary, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(846430350490556672UL, 3554085506117065472UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("rebalanceTeams");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RebalanceTeamsActor, 0, int64_t >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] static Future<bool> rebalanceTeams( DDQueue* const& self, DataMovementReason const& moveReason, Reference<IDataDistributionTeam const> const& sourceTeam, Reference<IDataDistributionTeam const> const& destTeam, bool const& primary, TraceEvent* const& traceEvent ) {
															#line 2554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new RebalanceTeamsActor(self, moveReason, sourceTeam, destTeam, primary, traceEvent));
															#line 6908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 6913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via getSrcDestTeams()
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSrcDestTeamsActor>
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSrcDestTeamsActorState {
															#line 6920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSrcDestTeamsActorState(DDQueue* const& self,int const& teamCollectionIndex,GetTeamRequest const& srcReq,GetTeamRequest const& destReq,int const& priority,TraceEvent* const& traceEvent) 
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   teamCollectionIndex(teamCollectionIndex),
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   srcReq(srcReq),
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destReq(destReq),
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   priority(priority),
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 6937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("getSrcDestTeams", reinterpret_cast<unsigned long>(this));

	}
	~GetSrcDestTeamsActorState() 
	{
		fdb_probe_actor_destroy("getSrcDestTeams", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<ITeamRef>, bool>> __when_expr_0 = brokenPromiseToNever(self->teamCollections[teamCollectionIndex].getTeam.getReply(destReq));
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 1;
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*>(static_cast<GetSrcDestTeamsActor*>(this)));
															#line 6959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSrcDestTeamsActorState();
		static_cast<GetSrcDestTeamsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("DestTeam", printable(randomTeam.first.mapRef(&IDataDistributionTeam::getDesc)));
															#line 2641 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (randomTeam.first.present())
															#line 6984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<ITeamRef>, bool>> __when_expr_1 = brokenPromiseToNever(self->teamCollections[teamCollectionIndex].getTeam.getReply(srcReq));
															#line 2642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 2;
															#line 2642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*>(static_cast<GetSrcDestTeamsActor*>(this)));
															#line 6995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(std::pair<Optional<ITeamRef>, bool> const& __randomTeam,int loopDepth) 
	{
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		randomTeam = __randomTeam;
															#line 7009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<Optional<ITeamRef>, bool> && __randomTeam,int loopDepth) 
	{
		randomTeam = std::move(__randomTeam);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state > 0) static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<GetSrcDestTeamsActor*>(this)->ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> const& value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> && value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSrcDestTeamsActor*>(this)->SAV<SrcDestTeamPair>::futures) { (void)({}); this->~GetSrcDestTeamsActorState(); static_cast<GetSrcDestTeamsActor*>(this)->destroy(); return 0; }
															#line 7088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<GetSrcDestTeamsActor*>(this)->SAV< SrcDestTeamPair >::value()) SrcDestTeamPair({});
		this->~GetSrcDestTeamsActorState();
		static_cast<GetSrcDestTeamsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SourceTeam", printable(loadedTeam.first.mapRef(&IDataDistributionTeam::getDesc)));
															#line 2647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (loadedTeam.first.present())
															#line 7102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSrcDestTeamsActor*>(this)->SAV<SrcDestTeamPair>::futures) { (void)(std::make_pair(loadedTeam.first.get(), randomTeam.first.get())); this->~GetSrcDestTeamsActorState(); static_cast<GetSrcDestTeamsActor*>(this)->destroy(); return 0; }
															#line 7106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<GetSrcDestTeamsActor*>(this)->SAV< SrcDestTeamPair >::value()) SrcDestTeamPair(std::make_pair(loadedTeam.first.get(), randomTeam.first.get()));
			this->~GetSrcDestTeamsActorState();
			static_cast<GetSrcDestTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::pair<Optional<ITeamRef>, bool> const& __loadedTeam,int loopDepth) 
	{
															#line 2642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		loadedTeam = __loadedTeam;
															#line 7120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::pair<Optional<ITeamRef>, bool> && __loadedTeam,int loopDepth) 
	{
		loadedTeam = std::move(__loadedTeam);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state > 0) static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<GetSrcDestTeamsActor*>(this)->ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> const& value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> && value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int teamCollectionIndex;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTeamRequest srcReq;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTeamRequest destReq;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int priority;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::pair<Optional<ITeamRef>, bool> randomTeam;
															#line 2642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::pair<Optional<ITeamRef>, bool> loadedTeam;
															#line 7211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via getSrcDestTeams()
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSrcDestTeamsActor final : public Actor<SrcDestTeamPair>, public ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >, public ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >, public FastAllocated<GetSrcDestTeamsActor>, public GetSrcDestTeamsActorState<GetSrcDestTeamsActor> {
															#line 7216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<GetSrcDestTeamsActor>::operator new;
	using FastAllocated<GetSrcDestTeamsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(438100092973839616UL, 3620207231842956800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<SrcDestTeamPair>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >;
friend struct ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >;
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSrcDestTeamsActor(DDQueue* const& self,int const& teamCollectionIndex,GetTeamRequest const& srcReq,GetTeamRequest const& destReq,int const& priority,TraceEvent* const& traceEvent) 
															#line 7234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<SrcDestTeamPair>(),
		   GetSrcDestTeamsActorState<GetSrcDestTeamsActor>(self, teamCollectionIndex, srcReq, destReq, priority, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17622014136999606784UL, 6190347938397110016UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSrcDestTeams");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<SrcDestTeamPair> getSrcDestTeams( DDQueue* const& self, int const& teamCollectionIndex, GetTeamRequest const& srcReq, GetTeamRequest const& destReq, int const& priority, TraceEvent* const& traceEvent ) {
															#line 2630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<SrcDestTeamPair>(new GetSrcDestTeamsActor(self, teamCollectionIndex, srcReq, destReq, priority, traceEvent));
															#line 7268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

Future<SrcDestTeamPair> DDQueue::getSrcDestTeams(const int& teamCollectionIndex,
                                                 const GetTeamRequest& srcReq,
                                                 const GetTeamRequest& destReq,
                                                 const int& priority,
                                                 TraceEvent* traceEvent) {
	return ::getSrcDestTeams(this, teamCollectionIndex, srcReq, destReq, priority, traceEvent);
}
Future<bool> DDQueue::rebalanceReadLoad(DataMovementReason moveReason,
                                        Reference<IDataDistributionTeam> sourceTeam,
                                        Reference<IDataDistributionTeam> destTeam,
                                        bool primary,
                                        TraceEvent* traceEvent) {
	return ::rebalanceReadLoad(this, moveReason, sourceTeam, destTeam, primary, traceEvent);
}

Future<bool> DDQueue::rebalanceTeams(DataMovementReason moveReason,
                                     Reference<const IDataDistributionTeam> sourceTeam,
                                     Reference<const IDataDistributionTeam> destTeam,
                                     bool primary,
                                     TraceEvent* traceEvent) {
	return ::rebalanceTeams(this, moveReason, sourceTeam, destTeam, primary, traceEvent);
}

															#line 7296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via getSkipRebalanceValue()
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSkipRebalanceValueActor>
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSkipRebalanceValueActorState {
															#line 7303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSkipRebalanceValueActorState(Reference<IDDTxnProcessor> const& txnProcessor,bool const& readRebalance) 
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : txnProcessor(txnProcessor),
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   readRebalance(readRebalance)
															#line 7312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this));

	}
	~GetSkipRebalanceValueActorState() 
	{
		fdb_probe_actor_destroy("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = txnProcessor->readRebalanceDDIgnoreKey();
															#line 2678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state = 1;
															#line 2678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*>(static_cast<GetSkipRebalanceValueActor*>(this)));
															#line 7334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& val,int loopDepth) 
	{
															#line 2680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!val.present())
															#line 7357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(false); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(false);
			this->~GetSkipRebalanceValueActorState();
			static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool skipCurrentLoop = false;
															#line 2685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (val.get().size() > 0 && val.get() != "on"_sr)
															#line 7371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int ddIgnore = BinaryReader::fromStringRef<uint8_t>(val.get(), Unversioned());
															#line 2687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 7377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_READ) > 0;
															#line 7381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_DISK) > 0;
															#line 7387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		else
		{
															#line 2693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			skipCurrentLoop = true;
															#line 7394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2696 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(skipCurrentLoop); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(skipCurrentLoop);
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && val,int loopDepth) 
	{
															#line 2680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!val.present())
															#line 7410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(false); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(false);
			this->~GetSkipRebalanceValueActorState();
			static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool skipCurrentLoop = false;
															#line 2685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (val.get().size() > 0 && val.get() != "on"_sr)
															#line 7424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int ddIgnore = BinaryReader::fromStringRef<uint8_t>(val.get(), Unversioned());
															#line 2687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 7430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_READ) > 0;
															#line 7434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_DISK) > 0;
															#line 7440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		else
		{
															#line 2693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			skipCurrentLoop = true;
															#line 7447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2696 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(skipCurrentLoop); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(skipCurrentLoop);
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont1(val, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state > 0) static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetSkipRebalanceValueActor*>(this)->ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDDTxnProcessor> txnProcessor;
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool readRebalance;
															#line 7538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via getSkipRebalanceValue()
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSkipRebalanceValueActor final : public Actor<bool>, public ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >, public FastAllocated<GetSkipRebalanceValueActor>, public GetSkipRebalanceValueActorState<GetSkipRebalanceValueActor> {
															#line 7543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<GetSkipRebalanceValueActor>::operator new;
	using FastAllocated<GetSkipRebalanceValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6631503669513792000UL, 15389203450246391808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >;
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSkipRebalanceValueActor(Reference<IDDTxnProcessor> const& txnProcessor,bool const& readRebalance) 
															#line 7560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<bool>(),
		   GetSkipRebalanceValueActorState<GetSkipRebalanceValueActor>(txnProcessor, readRebalance),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9255883138487682304UL, 8872371140409258752UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSkipRebalanceValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<bool> getSkipRebalanceValue( Reference<IDDTxnProcessor> const& txnProcessor, bool const& readRebalance ) {
															#line 2677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new GetSkipRebalanceValueActor(txnProcessor, readRebalance));
															#line 7593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 7598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via BgDDLoadRebalance()
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class BgDDLoadRebalanceActor>
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class BgDDLoadRebalanceActorState {
															#line 7605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	BgDDLoadRebalanceActorState(DDQueue* const& self,int const& teamCollectionIndex,DataMovementReason const& reason) 
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   teamCollectionIndex(teamCollectionIndex),
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   reason(reason),
															#line 2700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   resetCount(0),
															#line 2701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   lastRead(0),
															#line 2702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   skipCurrentLoop(false),
															#line 2703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   readRebalance(isDataMovementForReadBalancing(reason)),
															#line 2704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveType(isDataMovementForMountainChopper(reason) ? "BgDDMountainChopper" : "BgDDValleyFiller"),
															#line 2706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddPriority(dataMovementPriority(reason)),
															#line 2707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   mcMove(isDataMovementForMountainChopper(reason)),
															#line 2708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   preferLowerReadTeam(readRebalance || SERVER_KNOBS->DD_PREFER_LOW_READ_UTIL_TEAM ? PreferLowerReadUtil::True : PreferLowerReadUtil::False),
															#line 2711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rebalancePollingInterval(0)
															#line 7634 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this));

	}
	~BgDDLoadRebalanceActorState() 
	{
		fdb_probe_actor_destroy("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 7649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BgDDLoadRebalanceActorState();
		static_cast<BgDDLoadRebalanceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		moved = false;
															#line 2715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = Reference<IDataDistributionTeam>();
															#line 2716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = Reference<IDataDistributionTeam>();
															#line 2719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(rebalancePollingInterval, TaskPriority::DataDistributionLaunch);
															#line 2719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 1;
															#line 2719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 0, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 2721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((now() - lastRead) > SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL)
															#line 7702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = store(skipCurrentLoop, getSkipRebalanceValue(self->txnProcessor, readRebalance));
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 2;
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 1, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 2721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((now() - lastRead) > SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL)
															#line 7734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = store(skipCurrentLoop, getSkipRebalanceValue(self->txnProcessor, readRebalance));
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 2;
															#line 2722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 1, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("RebalanceMoveError", self->distributorId).detail("MoveType", moveType).errorUnsuppressed(e);
															#line 2804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 7849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 2726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (skipCurrentLoop)
															#line 7863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rebalancePollingInterval = std::max(rebalancePollingInterval, SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL);
															#line 2729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DDRebalancePaused", self->distributorId) .suppressFor(5.0) .detail("MoveType", moveType) .detail("Reason", "Disabled");
															#line 7869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 2736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (self->priority_relocations[ddPriority] >= SERVER_KNOBS->DD_REBALANCE_PARALLELISM)
															#line 7874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rebalancePollingInterval = std::min(rebalancePollingInterval * 2, SERVER_KNOBS->BG_REBALANCE_MAX_POLLING_INTERVAL);
															#line 2739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DDRebalancePaused", self->distributorId) .suppressFor(5.0) .detail("MoveType", moveType) .detail("Reason", "DataMoveLimitReached") .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("PollingInterval", rebalancePollingInterval);
															#line 7880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 2748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		rebalancePollingInterval = std::max(rebalancePollingInterval / 2, SERVER_KNOBS->BG_REBALANCE_POLLING_INTERVAL);
															#line 2751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent = TraceEvent(mcMove ? "MountainChopperSample" : "ValleyFillerSample", self->distributorId);
															#line 2752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent.suppressFor(5.0);
															#line 2753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		GetTeamRequest srcReq = GetTeamRequest(mcMove ? TeamSelect::WANT_TRUE_BEST : TeamSelect::ANY, PreferLowerDiskUtil::False, TeamMustHaveShards::True, PreferLowerReadUtil::False, PreferWithinShardLimit::False, ForReadBalance(readRebalance));
															#line 2760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		GetTeamRequest destReq = GetTeamRequest(!mcMove ? TeamSelect::WANT_TRUE_BEST : TeamSelect::ANY, PreferLowerDiskUtil::True, TeamMustHaveShards::False, preferLowerReadTeam, PreferWithinShardLimit::False, ForReadBalance(readRebalance));
															#line 2766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		getTeamFuture = self->getSrcDestTeams(teamCollectionIndex, srcReq, destReq, ddPriority, &traceEvent);
															#line 2768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = ready(getTeamFuture);
															#line 2768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 3;
															#line 2768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 2, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		lastRead = now();
															#line 7913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		lastRead = now();
															#line 7922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = getTeamFuture.get().first;
															#line 2770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = getTeamFuture.get().second;
															#line 2773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceTeam.isValid() && destTeam.isValid())
															#line 8010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 8014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(moved,self->rebalanceReadLoad( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 4;
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 3, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_4 = store(moved,self->rebalanceTeams( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_4.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 5;
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 4, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = getTeamFuture.get().first;
															#line 2770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = getTeamFuture.get().second;
															#line 2773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceTeam.isValid() && destTeam.isValid())
															#line 8058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 8062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(moved,self->rebalanceReadLoad( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 4;
															#line 2775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 3, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_4 = store(moved,self->rebalanceTeams( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_4.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 5;
															#line 2777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 4, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent.detail("Moved", moved).log();
															#line 2783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (moved)
															#line 8179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			resetCount = 0;
															#line 2785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(mcMove ? "MountainChopperMoved" : "ValleyFillerMoved", self->distributorId) .suppressFor(5.0) .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("PollingInterval", rebalancePollingInterval);
															#line 8185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			++resetCount;
															#line 2791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (resetCount > 30)
															#line 8193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				rebalancePollingInterval = SERVER_KNOBS->BG_REBALANCE_MAX_POLLING_INTERVAL;
															#line 8197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 2794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(mcMove ? "MountainChopperSkipped" : "ValleyFillerSkipped", self->distributorId) .suppressFor(5.0) .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("ResetCount", resetCount) .detail("PollingInterval", rebalancePollingInterval);
															#line 8201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int teamCollectionIndex;
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason reason;
															#line 2700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int resetCount;
															#line 2701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double lastRead;
															#line 2702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool skipCurrentLoop;
															#line 2703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const bool readRebalance;
															#line 2704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const std::string moveType;
															#line 2706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int ddPriority;
															#line 2707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool mcMove;
															#line 2708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PreferLowerReadUtil preferLowerReadTeam;
															#line 2711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double rebalancePollingInterval;
															#line 2714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool moved;
															#line 2715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> sourceTeam;
															#line 2716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> destTeam;
															#line 2751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent traceEvent;
															#line 2766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<SrcDestTeamPair> getTeamFuture;
															#line 8434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via BgDDLoadRebalance()
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class BgDDLoadRebalanceActor final : public Actor<Void>, public ActorCallback< BgDDLoadRebalanceActor, 0, Void >, public ActorCallback< BgDDLoadRebalanceActor, 1, Void >, public ActorCallback< BgDDLoadRebalanceActor, 2, Void >, public ActorCallback< BgDDLoadRebalanceActor, 3, Void >, public ActorCallback< BgDDLoadRebalanceActor, 4, Void >, public FastAllocated<BgDDLoadRebalanceActor>, public BgDDLoadRebalanceActorState<BgDDLoadRebalanceActor> {
															#line 8439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<BgDDLoadRebalanceActor>::operator new;
	using FastAllocated<BgDDLoadRebalanceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1773179383083207424UL, 7466356012032892928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BgDDLoadRebalanceActor, 0, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 1, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 2, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 3, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 4, Void >;
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	BgDDLoadRebalanceActor(DDQueue* const& self,int const& teamCollectionIndex,DataMovementReason const& reason) 
															#line 8460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   BgDDLoadRebalanceActorState<BgDDLoadRebalanceActor>(self, teamCollectionIndex, reason),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16598836187423810560UL, 4740218205161676032UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("BgDDLoadRebalance");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> BgDDLoadRebalance( DDQueue* const& self, int const& teamCollectionIndex, DataMovementReason const& reason ) {
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new BgDDLoadRebalanceActor(self, teamCollectionIndex, reason));
															#line 8497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 2808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

struct DDQueueImpl {
																#line 8503 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
// This generated class is to be used only via run()
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RunActor>
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RunActorState {
															#line 8509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RunActorState(Reference<DDQueue> const& self,Reference<AsyncVar<bool>> const& processingUnhealthy,Reference<AsyncVar<bool>> const& processingWiggle,FutureStream<Promise<int>> const& getUnhealthyRelocationCount) 
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   processingUnhealthy(processingUnhealthy),
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   processingWiggle(processingWiggle),
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   getUnhealthyRelocationCount(getUnhealthyRelocationCount),
															#line 2815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   serversToLaunchFrom(),
															#line 2816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   keysToLaunchFrom(),
															#line 2817 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   launchData(),
															#line 2818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   recordMetrics(delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL)),
															#line 2820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddQueueFutures(),
															#line 2822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rangesComplete(),
															#line 2823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   launchQueuedWorkTimeout(Never()),
															#line 2824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   onCleanUpDataMoveActorError(actorCollection(self->addBackgroundCleanUpDataMoveActor.getFuture()))
															#line 8538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("run", reinterpret_cast<unsigned long>(this));

	}
	~RunActorState() 
	{
		fdb_probe_actor_destroy("run", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->teamCollections.size();i++) {
															#line 2828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_OVERUTILIZED_TEAM));
															#line 2830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM));
															#line 2832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 8559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM));
															#line 2835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM));
															#line 8565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
															#line 2839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(delayedAsyncVar(self->rawProcessingUnhealthy, processingUnhealthy, 0));
															#line 2840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(delayedAsyncVar(self->rawProcessingWiggle, processingWiggle, 0));
															#line 2841 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(self->periodicalRefreshCounter());
															#line 8574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			try {
															#line 2844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				;
															#line 8578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunActorState();
		static_cast<RunActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (e.code() != error_code_broken_promise && e.code() != error_code_movekeys_conflict && e.code() != error_code_data_move_cancelled && e.code() != error_code_data_move_dest_team_not_found)
															#line 8608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 3009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevError, "DataDistributionQueueError", self->distributorId).error(e);
															#line 8612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
															#line 3010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->validate();
															#line 2849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (launchData.startTime != -1)
															#line 8639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(launchData, self->ddEnabledState);
															#line 2852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchData = RelocateData();
															#line 8645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!keysToLaunchFrom.empty())
															#line 8651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2854 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->launchQueuedWork(keysToLaunchFrom, self->ddEnabledState);
															#line 2855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				keysToLaunchFrom = KeyRangeRef();
															#line 8657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
															#line 2858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(launchData.startTime == -1 && keysToLaunchFrom.empty());
															#line 2861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_0 = self->input;
															#line 8664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<RelocateShard>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<RelocateShard>>::value, "invalid type");
															#line 2860 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
															#line 2875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = launchQueuedWorkTimeout;
															#line 8672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 2880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_2 = self->fetchSourceServersComplete.getFuture();
															#line 8676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.pop(), loopDepth); };
															#line 2885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_3 = self->dataTransferComplete.getFuture();
															#line 8681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when4(__when_expr_3.pop(), loopDepth); };
															#line 2891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_4 = self->relocationComplete.getFuture();
															#line 8686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_4), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_4), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when5(__when_expr_4.pop(), loopDepth); };
															#line 2909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_5 = rangesComplete.getFuture();
															#line 8691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_5), FutureStream<KeyRange>>::value || std::is_same<decltype(__when_expr_5), ThreadFutureStream<KeyRange>>::value, "invalid type");
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when6(__when_expr_5.pop(), loopDepth); };
															#line 2912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_6 = recordMetrics;
															#line 8696 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when7(__when_expr_6.get(), loopDepth); };
															#line 2996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_7 = self->error.getFuture();
															#line 8700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when8(__when_expr_7.get(), loopDepth); };
															#line 2997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_8 = waitForAll(ddQueueFutures);
															#line 8704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when9(__when_expr_8.get(), loopDepth); };
															#line 2998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_9 = getUnhealthyRelocationCount;
															#line 8708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_9), FutureStream<Promise<int>>>::value || std::is_same<decltype(__when_expr_9), ThreadFutureStream<Promise<int>>>::value, "invalid type");
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when10(__when_expr_9.pop(), loopDepth); };
															#line 3001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_10 = onCleanUpDataMoveActorError;
															#line 8713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when11(__when_expr_10.get(), loopDepth); };
		static_cast<RunActor*>(this)->actor_wait_state = 1;
															#line 2861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 0, RelocateShard >*>(static_cast<RunActor*>(this)));
															#line 2875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunActor, 1, Void >*>(static_cast<RunActor*>(this)));
															#line 2880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 2, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 3, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 4, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 5, KeyRange >*>(static_cast<RunActor*>(this)));
															#line 2912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RunActor, 6, Void >*>(static_cast<RunActor*>(this)));
															#line 2996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< RunActor, 7, Void >*>(static_cast<RunActor*>(this)));
															#line 2997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< RunActor, 8, Void >*>(static_cast<RunActor*>(this)));
															#line 2998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 9, Promise<int> >*>(static_cast<RunActor*>(this)));
															#line 3001 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< RunActor, 10, Void >*>(static_cast<RunActor*>(this)));
															#line 8738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RelocateShard const& rs,int loopDepth) 
	{
															#line 2862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rs.isRestore())
															#line 8753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMove != nullptr);
															#line 2864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMoveId.isValid());
															#line 2865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(RelocateData(rs), self->ddEnabledState);
															#line 8761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rs.cancelled)
															#line 8767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->enqueueCancelledDataMove(rs.dataMoveId, rs.keys, self->ddEnabledState);
															#line 8771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool wasEmpty = serversToLaunchFrom.empty();
															#line 2870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->queueRelocation(rs, serversToLaunchFrom);
															#line 2871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wasEmpty && !serversToLaunchFrom.empty())
															#line 8781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RelocateShard && rs,int loopDepth) 
	{
															#line 2862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rs.isRestore())
															#line 8797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMove != nullptr);
															#line 2864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMoveId.isValid());
															#line 2865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(RelocateData(rs), self->ddEnabledState);
															#line 8805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		else
		{
															#line 2866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rs.cancelled)
															#line 8811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			{
															#line 2867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->enqueueCancelledDataMove(rs.dataMoveId, rs.keys, self->ddEnabledState);
															#line 8815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
			else
			{
															#line 2869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool wasEmpty = serversToLaunchFrom.empty();
															#line 2870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->queueRelocation(rs, serversToLaunchFrom);
															#line 2871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wasEmpty && !serversToLaunchFrom.empty())
															#line 8825 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				{
															#line 2872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->launchQueuedWork(serversToLaunchFrom, self->ddEnabledState);
															#line 2877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom = std::set<UID>();
															#line 2878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchQueuedWorkTimeout = Never();
															#line 8845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->launchQueuedWork(serversToLaunchFrom, self->ddEnabledState);
															#line 2877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom = std::set<UID>();
															#line 2878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchQueuedWorkTimeout = Never();
															#line 8858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(RelocateData const& results,int loopDepth) 
	{
															#line 2882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->completeSourceFetch(results);
															#line 2883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchData = results;
															#line 8869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(RelocateData && results,int loopDepth) 
	{
															#line 2882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->completeSourceFetch(results);
															#line 2883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchData = results;
															#line 8880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(RelocateData const& done,int loopDepth) 
	{
															#line 2886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		complete(done, self->busymap, self->destBusymap);
															#line 2887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (serversToLaunchFrom.empty() && !done.src.empty())
															#line 8891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom.insert(done.src.begin(), done.src.end());
															#line 8899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(RelocateData && done,int loopDepth) 
	{
															#line 2886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		complete(done, self->busymap, self->destBusymap);
															#line 2887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (serversToLaunchFrom.empty() && !done.src.empty())
															#line 8910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 2889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom.insert(done.src.begin(), done.src.end());
															#line 8918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(RelocateData const& done,int loopDepth) 
	{
															#line 2892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->activeRelocations--;
															#line 2893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevVerbose, "InFlightRelocationChange") .detail("Complete", done.dataMoveId) .detail("IsRestore", done.isRestore()) .detail("Total", self->activeRelocations);
															#line 2897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->finishRelocation(done.priority, done.healthPriority);
															#line 2898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.erase(done);
															#line 2900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( tag(delay(0, TaskPriority::DataDistributionLaunch), done.keys, rangesComplete));
															#line 2902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (g_network->isSimulated() && debug_isCheckRelocationDuration() && now() - done.startTime > 60)
															#line 8937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarnAlways, "RelocationDurationTooLong") .detail("Duration", now() - done.startTime);
															#line 2906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			debug_setCheckRelocationDuration(false);
															#line 8943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(RelocateData && done,int loopDepth) 
	{
															#line 2892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->activeRelocations--;
															#line 2893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevVerbose, "InFlightRelocationChange") .detail("Complete", done.dataMoveId) .detail("IsRestore", done.isRestore()) .detail("Total", self->activeRelocations);
															#line 2897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->finishRelocation(done.priority, done.healthPriority);
															#line 2898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.erase(done);
															#line 2900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( tag(delay(0, TaskPriority::DataDistributionLaunch), done.keys, rangesComplete));
															#line 2902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (g_network->isSimulated() && debug_isCheckRelocationDuration() && now() - done.startTime > 60)
															#line 8963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarnAlways, "RelocationDurationTooLong") .detail("Duration", now() - done.startTime);
															#line 2906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			debug_setCheckRelocationDuration(false);
															#line 8969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(KeyRange const& done,int loopDepth) 
	{
															#line 2910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		keysToLaunchFrom = done;
															#line 8979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(KeyRange && done,int loopDepth) 
	{
															#line 2910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		keysToLaunchFrom = done;
															#line 8988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void const& _,int loopDepth) 
	{
															#line 2913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Promise<int64_t> req;
															#line 2914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->getAverageShardBytes.send(req);
															#line 2916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		recordMetrics = delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 2918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto const highestPriorityRelocation = self->getHighestPriorityRelocation();
															#line 2920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent("MovingData", self->distributorId) .detail("InFlight", self->activeRelocations) .detail("InQueue", self->queuedRelocations) .detail("AverageShardSize", req.getFuture().isReady() ? req.getFuture().get() : -1) .detail("UnhealthyRelocations", self->unhealthyRelocations) .detail("HighestPriority", highestPriorityRelocation) .detail("BytesWritten", self->moveBytesRate.getTotal()) .detail("BytesWrittenAverageRate", self->moveBytesRate.getAverage()) .detail("PriorityRecoverMove", self->priority_relocations[SERVER_KNOBS->PRIORITY_RECOVER_MOVE]) .detail("PriorityRebalanceUnderutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM]) .detail("PriorityRebalanceOverutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM]) .detail("PriorityRebalanceReadUnderutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM]) .detail("PriorityRebalanceReadOverutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM]) .detail("PriorityStorageWiggle", self->priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE]) .detail("PriorityTeamHealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_HEALTHY]) .detail("PriorityTeamContainsUndesiredServer", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER]) .detail("PriorityTeamRedundant", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT]) .detail("PriorityMergeShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_MERGE_SHARD]) .detail("PriorityPopulateRegion", self->priority_relocations[SERVER_KNOBS->PRIORITY_POPULATE_REGION]) .detail("PriorityTeamUnhealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY]) .detail("PriorityTeam2Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_2_LEFT]) .detail("PriorityTeam1Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_1_LEFT]) .detail("PriorityTeam0Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_0_LEFT]) .detail("PrioritySplitShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_SPLIT_SHARD]) .trackLatest("MovingData");
															#line 2961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 9007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("PhysicalShardMoveStats") .detail("MoveCreateNewPhysicalShard", self->moveCreateNewPhysicalShard) .detail("MoveReusePhysicalShard", self->moveReusePhysicalShard) .detail( "RemoteBestTeamNotReady", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]) .detail( "PrimaryNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam]) .detail("RemoteNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]) .detail("RemoteTeamIsFull", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]) .detail( "RemoteTeamIsNotHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]) .detail("UnknownForceNew", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::UnknownForceNew]) .detail("NoAnyHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]) .detail("DstOverloaded", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]) .detail("NoAvailablePhysicalShard", self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]) .detail("RetryLimitReached", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]);
															#line 2989 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveCreateNewPhysicalShard = 0;
															#line 2990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveReusePhysicalShard = 0;
															#line 2991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->retryFindDstReasonCount.size();++i) {
															#line 2992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[i] = 0;
															#line 9019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void && _,int loopDepth) 
	{
															#line 2913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Promise<int64_t> req;
															#line 2914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->getAverageShardBytes.send(req);
															#line 2916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		recordMetrics = delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 2918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto const highestPriorityRelocation = self->getHighestPriorityRelocation();
															#line 2920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent("MovingData", self->distributorId) .detail("InFlight", self->activeRelocations) .detail("InQueue", self->queuedRelocations) .detail("AverageShardSize", req.getFuture().isReady() ? req.getFuture().get() : -1) .detail("UnhealthyRelocations", self->unhealthyRelocations) .detail("HighestPriority", highestPriorityRelocation) .detail("BytesWritten", self->moveBytesRate.getTotal()) .detail("BytesWrittenAverageRate", self->moveBytesRate.getAverage()) .detail("PriorityRecoverMove", self->priority_relocations[SERVER_KNOBS->PRIORITY_RECOVER_MOVE]) .detail("PriorityRebalanceUnderutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM]) .detail("PriorityRebalanceOverutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM]) .detail("PriorityRebalanceReadUnderutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM]) .detail("PriorityRebalanceReadOverutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM]) .detail("PriorityStorageWiggle", self->priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE]) .detail("PriorityTeamHealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_HEALTHY]) .detail("PriorityTeamContainsUndesiredServer", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER]) .detail("PriorityTeamRedundant", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT]) .detail("PriorityMergeShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_MERGE_SHARD]) .detail("PriorityPopulateRegion", self->priority_relocations[SERVER_KNOBS->PRIORITY_POPULATE_REGION]) .detail("PriorityTeamUnhealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY]) .detail("PriorityTeam2Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_2_LEFT]) .detail("PriorityTeam1Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_1_LEFT]) .detail("PriorityTeam0Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_0_LEFT]) .detail("PrioritySplitShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_SPLIT_SHARD]) .trackLatest("MovingData");
															#line 2961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 9040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		{
															#line 2962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("PhysicalShardMoveStats") .detail("MoveCreateNewPhysicalShard", self->moveCreateNewPhysicalShard) .detail("MoveReusePhysicalShard", self->moveReusePhysicalShard) .detail( "RemoteBestTeamNotReady", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]) .detail( "PrimaryNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam]) .detail("RemoteNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]) .detail("RemoteTeamIsFull", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]) .detail( "RemoteTeamIsNotHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]) .detail("UnknownForceNew", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::UnknownForceNew]) .detail("NoAnyHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]) .detail("DstOverloaded", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]) .detail("NoAvailablePhysicalShard", self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]) .detail("RetryLimitReached", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]);
															#line 2989 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveCreateNewPhysicalShard = 0;
															#line 2990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveReusePhysicalShard = 0;
															#line 2991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->retryFindDstReasonCount.size();++i) {
															#line 2992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[i] = 0;
															#line 9052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when10(Promise<int> const& r,int loopDepth) 
	{
															#line 2999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		r.send(self->getUnhealthyRelocationCount());
															#line 9087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when10(Promise<int> && r,int loopDepth) 
	{
															#line 2999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		r.send(self->getUnhealthyRelocationCount());
															#line 9096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 0, RelocateShard >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 1, Void >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 2, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 3, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 4, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 5, KeyRange >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 6, Void >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 7, Void >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 8, Void >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 9, Promise<int> >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 0, RelocateShard >*,RelocateShard const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 0, RelocateShard >*,RelocateShard && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 0, RelocateShard >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 2, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 2, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 2, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 3, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 3, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 3, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 4, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 4, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 4, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 5, KeyRange >*,KeyRange const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 5, KeyRange >*,KeyRange && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 5, KeyRange >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RunActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RunActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RunActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< RunActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RunActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< RunActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< RunActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 9, Promise<int> >*,Promise<int> const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 9, Promise<int> >*,Promise<int> && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 9, Promise<int> >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< RunActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< RunActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< RunActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<DDQueue> self;
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<AsyncVar<bool>> processingUnhealthy;
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<AsyncVar<bool>> processingWiggle;
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FutureStream<Promise<int>> getUnhealthyRelocationCount;
															#line 2815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::set<UID> serversToLaunchFrom;
															#line 2816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange keysToLaunchFrom;
															#line 2817 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData launchData;
															#line 2818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> recordMetrics;
															#line 2820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<Future<Void>> ddQueueFutures;
															#line 2822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<KeyRange> rangesComplete;
															#line 2823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> launchQueuedWorkTimeout;
															#line 2824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> onCleanUpDataMoveActorError;
															#line 9780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via run()
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RunActor final : public Actor<Void>, public ActorSingleCallback< RunActor, 0, RelocateShard >, public ActorCallback< RunActor, 1, Void >, public ActorSingleCallback< RunActor, 2, RelocateData >, public ActorSingleCallback< RunActor, 3, RelocateData >, public ActorSingleCallback< RunActor, 4, RelocateData >, public ActorSingleCallback< RunActor, 5, KeyRange >, public ActorCallback< RunActor, 6, Void >, public ActorCallback< RunActor, 7, Void >, public ActorCallback< RunActor, 8, Void >, public ActorSingleCallback< RunActor, 9, Promise<int> >, public ActorCallback< RunActor, 10, Void >, public FastAllocated<RunActor>, public RunActorState<RunActor> {
															#line 9785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<RunActor>::operator new;
	using FastAllocated<RunActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12979701890728581888UL, 8963873464624640768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< RunActor, 0, RelocateShard >;
friend struct ActorCallback< RunActor, 1, Void >;
friend struct ActorSingleCallback< RunActor, 2, RelocateData >;
friend struct ActorSingleCallback< RunActor, 3, RelocateData >;
friend struct ActorSingleCallback< RunActor, 4, RelocateData >;
friend struct ActorSingleCallback< RunActor, 5, KeyRange >;
friend struct ActorCallback< RunActor, 6, Void >;
friend struct ActorCallback< RunActor, 7, Void >;
friend struct ActorCallback< RunActor, 8, Void >;
friend struct ActorSingleCallback< RunActor, 9, Promise<int> >;
friend struct ActorCallback< RunActor, 10, Void >;
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RunActor(Reference<DDQueue> const& self,Reference<AsyncVar<bool>> const& processingUnhealthy,Reference<AsyncVar<bool>> const& processingWiggle,FutureStream<Promise<int>> const& getUnhealthyRelocationCount) 
															#line 9812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   RunActorState<RunActor>(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(633236942609902848UL, 16810589115111353856UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("run");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< RunActor, 0, RelocateShard >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] static Future<Void> run( Reference<DDQueue> const& self, Reference<AsyncVar<bool>> const& processingUnhealthy, Reference<AsyncVar<bool>> const& processingWiggle, FutureStream<Promise<int>> const& getUnhealthyRelocationCount ) {
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new RunActor(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount));
															#line 9844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}

#line 3013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
};

Future<Void> DDQueue::run(Reference<DDQueue> self,
                          Reference<AsyncVar<bool>> processingUnhealthy,
                          Reference<AsyncVar<bool>> processingWiggle,
                          FutureStream<Promise<int>> getUnhealthyRelocationCount,
                          const DDEnabledState* ddEnabledState) {
	self->ddEnabledState = ddEnabledState;
	return DDQueueImpl::run(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount);
}
															#line 9858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase3023()
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class FlowTestCase3023Actor>
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class FlowTestCase3023ActorState {
															#line 9865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowTestCase3023ActorState(UnitTestParameters const& params) 
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : params(params),
															#line 3024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   duration(2.5 * SERVER_KNOBS->DD_QUEUE_COUNTER_REFRESH_INTERVAL),
															#line 3025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   self(),
															#line 3026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   counterFuture(self.periodicalRefreshCounter()),
															#line 3027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   finishFuture(delay(duration))
															#line 9880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase3023", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase3023ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase3023", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::cout << "Start trace counter unit test for " << duration << "s ...\n";
															#line 3029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 9897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase3023ActorState();
		static_cast<FlowTestCase3023Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::cout << "Finished.";
															#line 3046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<FlowTestCase3023Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3023ActorState(); static_cast<FlowTestCase3023Actor*>(this)->destroy(); return 0; }
															#line 9922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		new (&static_cast<FlowTestCase3023Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3023ActorState();
		static_cast<FlowTestCase3023Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_0 = counterFuture;
															#line 3029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<FlowTestCase3023Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 3031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = finishFuture;
															#line 9947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 3034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayJittered(2.0);
															#line 9951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3023Actor*>(this)->actor_wait_state = 1;
															#line 3030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3023Actor, 0, Void >*>(static_cast<FlowTestCase3023Actor*>(this)));
															#line 3031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3023Actor, 1, Void >*>(static_cast<FlowTestCase3023Actor*>(this)));
															#line 3034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3023Actor, 2, Void >*>(static_cast<FlowTestCase3023Actor*>(this)));
															#line 9960 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
															#line 3035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<UID> team(3);
															#line 3036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < team.size();++i) {
															#line 3037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			team[i] = UID(deterministicRandom()->randomInt(1, 400), 0);
															#line 10016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 3039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto reason = RelocateReason(deterministicRandom()->randomInt(0, RelocateReason::typeCount()));
															#line 3040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto countType = DDQueue::ServerCounter::randomCountType();
															#line 3041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self.serverCounter.increaseForTeam(team, reason, countType);
															#line 3042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(self.serverCounter.get(team[0], reason, countType));
															#line 10026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
															#line 3035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<UID> team(3);
															#line 3036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < team.size();++i) {
															#line 3037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			team[i] = UID(deterministicRandom()->randomInt(1, 400), 0);
															#line 10039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		}
															#line 3039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto reason = RelocateReason(deterministicRandom()->randomInt(0, RelocateReason::typeCount()));
															#line 3040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto countType = DDQueue::ServerCounter::randomCountType();
															#line 3041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self.serverCounter.increaseForTeam(team, reason, countType);
															#line 3042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(self.serverCounter.get(team[0], reason, countType));
															#line 10049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase3023Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3023Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3023Actor*>(this)->ActorCallback< FlowTestCase3023Actor, 0, Void >::remove();
		static_cast<FlowTestCase3023Actor*>(this)->ActorCallback< FlowTestCase3023Actor, 1, Void >::remove();
		static_cast<FlowTestCase3023Actor*>(this)->ActorCallback< FlowTestCase3023Actor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase3023Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase3023Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3023Actor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase3023Actor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	UnitTestParameters params;
															#line 3024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double duration;
															#line 3025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue self;
															#line 3026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> counterFuture;
															#line 3027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> finishFuture;
															#line 10243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase3023()
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class FlowTestCase3023Actor final : public Actor<Void>, public ActorCallback< FlowTestCase3023Actor, 0, Void >, public ActorCallback< FlowTestCase3023Actor, 1, Void >, public ActorCallback< FlowTestCase3023Actor, 2, Void >, public FastAllocated<FlowTestCase3023Actor>, public FlowTestCase3023ActorState<FlowTestCase3023Actor> {
															#line 10248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase3023Actor>::operator new;
	using FastAllocated<FlowTestCase3023Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5280816509044913152UL, 14296357956948299008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase3023Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase3023Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase3023Actor, 2, Void >;
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowTestCase3023Actor(UnitTestParameters const& params) 
															#line 10267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase3023ActorState<FlowTestCase3023Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase3023", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3476441535982592UL, 9406683258368359680UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3023Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase3023");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase3023", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase3023Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
static Future<Void> flowTestCase3023( UnitTestParameters const& params ) {
															#line 3023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new FlowTestCase3023Actor(params));
															#line 10300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbserver/DDRelocationQueue.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase3023, "/DataDistribution/DDQueue/ServerCounterTrace")

#line 3048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
