#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
/*
 * HTTP.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbrpc/HTTP.h"
#include "fdbrpc/simulator.h"

#include "flow/IRandom.h"
#include "flow/Net2Packet.h"
#include "flow/Trace.h"
#include "md5/md5.h"
#include "libb64/encode.h"
#include "flow/Knobs.h"
#include <cctype>
#include "flow/IConnection.h"
#include <unordered_map>

#include "flow/actorcompiler.h" // has to be last include

namespace HTTP {

// AWS V4 headers require this encoding for its signature calculation
std::string awsV4URIEncode(const std::string& s, bool encodeSlash) {
	std::string o;
	o.reserve(s.size() * 3);
	char buf[4];
	for (auto c : s) {
		if (std::isalnum(c) || c == '-' || c == '_' || c == '.' || c == '~')
			o.append(&c, 1);
		else if (c == '/')
			o.append(encodeSlash ? "%2F" : "/");
		else {
			sprintf(buf, "%%%.02X", c);
			o.append(buf);
		}
	}
	return o;
}

std::string urlEncode(const std::string& s) {
	std::string o;
	o.reserve(s.size() * 3);
	char buf[4];
	for (auto c : s)
		if (std::isalnum(c) || c == '?' || c == '/' || c == '-' || c == '_' || c == '.' || c == ',' || c == ':')
			o.append(&c, 1);
		else {
			sprintf(buf, "%%%.02X", c);
			o.append(buf);
		}
	return o;
}

template <typename T>
std::string ResponseBase<T>::getCodeDescription() {
	if (code == HTTP_STATUS_CODE_OK) {
		return "OK";
	} else if (code == HTTP_STATUS_CODE_CREATED) {
		return "Created";
	} else if (code == HTTP_STATUS_CODE_ACCEPTED) {
		return "Accepted";
	} else if (code == HTTP_STATUS_CODE_NO_CONTENT) {
		return "No Content";
	} else if (code == HTTP_STATUS_CODE_UNAUTHORIZED) {
		return "Unauthorized";
	} else if (code == HTTP_STATUS_CODE_NOT_ACCEPTABLE) {
		return "Not Acceptable";
	} else if (code == HTTP_STATUS_CODE_TIMEOUT) {
		return "Timeout";
	} else if (code == HTTP_STATUS_CODE_TOO_MANY_REQUESTS) {
		return "Too Many Requests";
	} else if (code == HTTP_STATUS_CODE_INTERNAL_SERVER_ERROR) {
		return "Internal Server Error";
	} else if (code == HTTP_STATUS_CODE_BAD_GATEWAY) {
		return "Bad Gateway";
	} else if (code == HTTP_STATUS_CODE_SERVICE_UNAVAILABLE) {
		return "Service Unavailable";
	} else if (code == HTTP_STATUS_CODE_GATEWAY_TIMEOUT) {
		return "Gateway Timeout";
	} else {
		throw internal_error();
	}
}

std::string computeMD5Sum(std::string content) {
	MD5_CTX sum;
	::MD5_Init(&sum);
	::MD5_Update(&sum, content.data(), content.size());
	std::string sumBytes;
	sumBytes.resize(16);
	::MD5_Final((unsigned char*)sumBytes.data(), &sum);
	std::string sumStr = base64::encoder::from_string(sumBytes);
	sumStr.resize(sumStr.size() - 1);
	return sumStr;
}

bool verifyMD5(HTTPData<std::string>* data, bool fail_if_header_missing, Optional<std::string> content_sum) {
	auto i = data->headers.find("Content-MD5");
	if (i != data->headers.end()) {
		// If a content sum is not provided, calculate one from the response content
		if (!content_sum.present()) {
			content_sum = computeMD5Sum(data->content);
		}
		return i->second == content_sum.get();
	}
	return !fail_if_header_missing;
}

std::string IncomingResponse::toString() const {
	std::string r = fmt::format("Response Code: {0}\n", code);
	r += fmt::format("Response ContentLen: {0}\n", data.contentLen);
	for (auto h : data.headers)
		r += fmt::format("Response Header: {0}: {1}\n", h.first, h.second);
	r.append("-- RESPONSE CONTENT--\n");
	// Limit the length of the response content to 1024 bytes for logging.
	// No one wants 40MB of content dumped to the console.
	int maxLen = 1024;
	if (data.content.size() > maxLen) {
		r.append(data.content.substr(0, maxLen));
		r.append("...\n");
	} else {
		r.append(data.content);
	}
	r.append("\n--------\n");
	return r;
}

void writeHeaders(HTTP::Headers const& headers, PacketWriter& writer) {
	for (auto h : headers) {
		writer.serializeBytes(h.first);
		writer.serializeBytes(": "_sr);
		writer.serializeBytes(h.second);
		writer.serializeBytes("\r\n"_sr);
	}
	writer.serializeBytes("\r\n"_sr);
}

PacketBuffer* writeRequestHeader(Reference<OutgoingRequest> req, PacketBuffer* dest) {
	PacketWriter writer(dest, nullptr, Unversioned());
	writer.serializeBytes(req->verb);
	writer.serializeBytes(" ", 1);
	writer.serializeBytes(req->resource);
	writer.serializeBytes(" HTTP/1.1\r\n"_sr);

	writeHeaders(req->data.headers, writer);

	return writer.finish();
}

PacketBuffer* writeResponseHeader(Reference<OutgoingResponse> response, PacketBuffer* dest) {
	PacketWriter writer(dest, nullptr, Unversioned());
	writer.serializeBytes("HTTP/1.1 "_sr);
	writer.serializeBytes(std::to_string(response->code));
	writer.serializeBytes(" ", 1);
	writer.serializeBytes(response->getCodeDescription());
	writer.serializeBytes("\r\n"_sr);

	writeHeaders(response->data.headers, writer);

	return writer.finish();
}

															#line 181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via writeResponse()
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class WriteResponseActor>
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class WriteResponseActorState {
															#line 187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	WriteResponseActorState(Reference<IConnection> const& conn,Reference<OutgoingResponse> const& response) 
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   response(response)
															#line 196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("writeResponse", reinterpret_cast<unsigned long>(this));

	}
	~WriteResponseActorState() 
	{
		fdb_probe_actor_destroy("writeResponse", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			ASSERT(response.isValid());
															#line 183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			response->data.headers["Content-Length"] = std::to_string(response->data.contentLen);
															#line 184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			PacketBuffer* pFirst = PacketBuffer::create();
															#line 185 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			PacketBuffer* pLast = writeResponseHeader(response, pFirst);
															#line 187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			response->data.content->prependWriteBuffer(pFirst, pLast);
															#line 188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteResponseActorState();
		static_cast<WriteResponseActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int trySend = FLOW_KNOBS->HTTP_SEND_SIZE;
															#line 190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if ((!g_network->isSimulated() || !g_simulator->speedUpSimulation) && BUGGIFY_WITH_PROB(0.01))
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			trySend = deterministicRandom()->randomInt(1, 10);
															#line 257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = conn->write(response->data.content->getUnsent(), trySend);
															#line 194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		response->data.content->sent(len);
															#line 195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (response->data.content->empty())
															#line 265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<WriteResponseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteResponseActorState(); static_cast<WriteResponseActor*>(this)->destroy(); return 0; }
															#line 269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<WriteResponseActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WriteResponseActorState();
			static_cast<WriteResponseActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_0 = conn->onWritable();
															#line 199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<WriteResponseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WriteResponseActor*>(this)->actor_wait_state = 1;
															#line 199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteResponseActor, 0, Void >*>(static_cast<WriteResponseActor*>(this)));
															#line 284 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = yield(TaskPriority::WriteSocket);
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<WriteResponseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteResponseActor*>(this)->actor_wait_state = 2;
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteResponseActor, 1, Void >*>(static_cast<WriteResponseActor*>(this)));
															#line 300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = yield(TaskPriority::WriteSocket);
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<WriteResponseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteResponseActor*>(this)->actor_wait_state = 2;
															#line 200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteResponseActor, 1, Void >*>(static_cast<WriteResponseActor*>(this)));
															#line 316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteResponseActor*>(this)->actor_wait_state > 0) static_cast<WriteResponseActor*>(this)->actor_wait_state = 0;
		static_cast<WriteResponseActor*>(this)->ActorCallback< WriteResponseActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteResponseActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteResponseActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteResponseActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteResponseActor*>(this)->actor_wait_state > 0) static_cast<WriteResponseActor*>(this)->actor_wait_state = 0;
		static_cast<WriteResponseActor*>(this)->ActorCallback< WriteResponseActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteResponseActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteResponseActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteResponseActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<OutgoingResponse> response;
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via writeResponse()
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class WriteResponseActor final : public Actor<Void>, public ActorCallback< WriteResponseActor, 0, Void >, public ActorCallback< WriteResponseActor, 1, Void >, public FastAllocated<WriteResponseActor>, public WriteResponseActorState<WriteResponseActor> {
															#line 492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<WriteResponseActor>::operator new;
	using FastAllocated<WriteResponseActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2210326668557916416UL, 323641627301880576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteResponseActor, 0, Void >;
friend struct ActorCallback< WriteResponseActor, 1, Void >;
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	WriteResponseActor(Reference<IConnection> const& conn,Reference<OutgoingResponse> const& response) 
															#line 510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   WriteResponseActorState<WriteResponseActor>(conn, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeResponse", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10580219571093262848UL, 18415521016041965568UL);
		ActorExecutionContextHelper __helper(static_cast<WriteResponseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeResponse");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeResponse", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteResponseActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteResponseActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> writeResponse( Reference<IConnection> const& conn, Reference<OutgoingResponse> const& response ) {
															#line 179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new WriteResponseActor(conn, response));
															#line 543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// Read at least 1 bytes from conn and up to maxlen in a single read, append read data into *buf
// Returns the number of bytes read.
															#line 550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_into_string()
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_into_stringActor>
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_into_stringActorState {
															#line 556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_into_stringActorState(Reference<IConnection> const& conn,std::string* const& buf,int const& maxlen) 
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(buf),
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   maxlen(maxlen)
															#line 567 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_into_string", reinterpret_cast<unsigned long>(this));

	}
	~Read_into_stringActorState() 
	{
		fdb_probe_actor_destroy("read_into_string", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_into_stringActorState();
		static_cast<Read_into_stringActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 209 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int originalSize = buf->size();
															#line 213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		buf->resize(originalSize + maxlen);
															#line 214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		uint8_t* wptr = (uint8_t*)buf->data() + originalSize;
															#line 215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = conn->read(wptr, wptr + maxlen);
															#line 216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		buf->resize(originalSize + len);
															#line 219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (len > 0)
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<Read_into_stringActor*>(this)->SAV<int>::futures) { (void)(len); this->~Read_into_stringActorState(); static_cast<Read_into_stringActor*>(this)->destroy(); return 0; }
															#line 626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<Read_into_stringActor*>(this)->SAV< int >::value()) int(len);
			this->~Read_into_stringActorState();
			static_cast<Read_into_stringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_0 = conn->onReadable();
															#line 223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_into_stringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Read_into_stringActor*>(this)->actor_wait_state = 1;
															#line 223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_into_stringActor, 0, Void >*>(static_cast<Read_into_stringActor*>(this)));
															#line 641 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::ReadSocket);
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_into_stringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_into_stringActor*>(this)->actor_wait_state = 2;
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_into_stringActor, 1, Void >*>(static_cast<Read_into_stringActor*>(this)));
															#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::ReadSocket);
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_into_stringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_into_stringActor*>(this)->actor_wait_state = 2;
															#line 224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_into_stringActor, 1, Void >*>(static_cast<Read_into_stringActor*>(this)));
															#line 673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_into_stringActor*>(this)->actor_wait_state > 0) static_cast<Read_into_stringActor*>(this)->actor_wait_state = 0;
		static_cast<Read_into_stringActor*>(this)->ActorCallback< Read_into_stringActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_into_stringActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_into_stringActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_into_stringActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Read_into_stringActor*>(this)->actor_wait_state > 0) static_cast<Read_into_stringActor*>(this)->actor_wait_state = 0;
		static_cast<Read_into_stringActor*>(this)->ActorCallback< Read_into_stringActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_into_stringActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Read_into_stringActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Read_into_stringActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string* buf;
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int maxlen;
															#line 846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_into_string()
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_into_stringActor final : public Actor<int>, public ActorCallback< Read_into_stringActor, 0, Void >, public ActorCallback< Read_into_stringActor, 1, Void >, public FastAllocated<Read_into_stringActor>, public Read_into_stringActorState<Read_into_stringActor> {
															#line 851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_into_stringActor>::operator new;
	using FastAllocated<Read_into_stringActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16887148442126027776UL, 11182969943461066240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_into_stringActor, 0, Void >;
friend struct ActorCallback< Read_into_stringActor, 1, Void >;
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_into_stringActor(Reference<IConnection> const& conn,std::string* const& buf,int const& maxlen) 
															#line 869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<int>(),
		   Read_into_stringActorState<Read_into_stringActor>(conn, buf, maxlen),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_into_string", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5498250837005894144UL, 8411832560289904640UL);
		ActorExecutionContextHelper __helper(static_cast<Read_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_into_string");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_into_string", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_into_stringActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Read_into_stringActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<int> read_into_string( Reference<IConnection> const& conn, std::string* const& buf, int const& maxlen ) {
															#line 206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<int>(new Read_into_stringActor(conn, buf, maxlen));
															#line 902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// Returns the position of delim within buf, relative to pos.  If delim is not found, continues to read from conn until
// either it is found or the connection ends, at which point connection_failed is thrown and buf contains
// everything that was read up to that point.
															#line 910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_delimited_into_string()
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_delimited_into_stringActor>
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_delimited_into_stringActorState {
															#line 916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_delimited_into_stringActorState(Reference<IConnection> const& conn,const char* const& delim,std::string* const& buf,size_t const& pos) 
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   delim(delim),
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(buf),
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(pos),
															#line 235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   sPos(pos),
															#line 236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   lookBack(strlen(delim) - 1)
															#line 933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_delimited_into_string", reinterpret_cast<unsigned long>(this));

	}
	~Read_delimited_into_stringActorState() 
	{
		fdb_probe_actor_destroy("read_delimited_into_string", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			ASSERT(lookBack >= 0);
															#line 239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_delimited_into_stringActorState();
		static_cast<Read_delimited_into_stringActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		size_t endPos = buf->find(delim, sPos);
															#line 241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (endPos != std::string::npos)
															#line 982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<Read_delimited_into_stringActor*>(this)->SAV<size_t>::futures) { (void)(endPos - pos); this->~Read_delimited_into_stringActorState(); static_cast<Read_delimited_into_stringActor*>(this)->destroy(); return 0; }
															#line 986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<Read_delimited_into_stringActor*>(this)->SAV< size_t >::value()) size_t(endPos - pos);
			this->~Read_delimited_into_stringActorState();
			static_cast<Read_delimited_into_stringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (buf->size() >= lookBack)
															#line 994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			sPos = buf->size() - lookBack;
															#line 998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_0 = success(read_into_string(conn, buf, FLOW_KNOBS->HTTP_READ_SIZE));
															#line 248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_delimited_into_stringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Read_delimited_into_stringActor*>(this)->actor_wait_state = 1;
															#line 248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_delimited_into_stringActor, 0, Void >*>(static_cast<Read_delimited_into_stringActor*>(this)));
															#line 1009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_delimited_into_stringActor*>(this)->actor_wait_state > 0) static_cast<Read_delimited_into_stringActor*>(this)->actor_wait_state = 0;
		static_cast<Read_delimited_into_stringActor*>(this)->ActorCallback< Read_delimited_into_stringActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_delimited_into_stringActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_delimited_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_delimited_into_stringActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_delimited_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_delimited_into_stringActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_delimited_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_delimited_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	const char* delim;
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string* buf;
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t pos;
															#line 235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t sPos;
															#line 236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int lookBack;
															#line 1113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_delimited_into_string()
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_delimited_into_stringActor final : public Actor<size_t>, public ActorCallback< Read_delimited_into_stringActor, 0, Void >, public FastAllocated<Read_delimited_into_stringActor>, public Read_delimited_into_stringActorState<Read_delimited_into_stringActor> {
															#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_delimited_into_stringActor>::operator new;
	using FastAllocated<Read_delimited_into_stringActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9022642847084879872UL, 3358465681755697152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<size_t>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_delimited_into_stringActor, 0, Void >;
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_delimited_into_stringActor(Reference<IConnection> const& conn,const char* const& delim,std::string* const& buf,size_t const& pos) 
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<size_t>(),
		   Read_delimited_into_stringActorState<Read_delimited_into_stringActor>(conn, delim, buf, pos),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_delimited_into_string", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3695670155312176128UL, 5186133134809679872UL);
		ActorExecutionContextHelper __helper(static_cast<Read_delimited_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_delimited_into_string");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_delimited_into_string", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_delimited_into_stringActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<size_t> read_delimited_into_string( Reference<IConnection> const& conn, const char* const& delim, std::string* const& buf, size_t const& pos ) {
															#line 231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<size_t>(new Read_delimited_into_stringActor(conn, delim, buf, pos));
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// Reads from conn (as needed) until there are at least len bytes starting at pos in buf
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_fixed_into_string()
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_fixed_into_stringActor>
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_fixed_into_stringActorState {
															#line 1179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_fixed_into_stringActorState(Reference<IConnection> const& conn,int const& len,std::string* const& buf,size_t const& pos) 
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   len(len),
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(buf),
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(pos),
															#line 254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   stop_size(pos + len)
															#line 1194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_fixed_into_string", reinterpret_cast<unsigned long>(this));

	}
	~Read_fixed_into_stringActorState() 
	{
		fdb_probe_actor_destroy("read_fixed_into_string", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 1209 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_fixed_into_stringActorState();
		static_cast<Read_fixed_into_stringActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<Read_fixed_into_stringActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_fixed_into_stringActorState(); static_cast<Read_fixed_into_stringActor*>(this)->destroy(); return 0; }
															#line 1232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<Read_fixed_into_stringActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Read_fixed_into_stringActorState();
		static_cast<Read_fixed_into_stringActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!(buf->size() < stop_size))
															#line 1251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_0 = success(read_into_string(conn, buf, FLOW_KNOBS->HTTP_READ_SIZE));
															#line 256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_fixed_into_stringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Read_fixed_into_stringActor*>(this)->actor_wait_state = 1;
															#line 256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_fixed_into_stringActor, 0, Void >*>(static_cast<Read_fixed_into_stringActor*>(this)));
															#line 1264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_fixed_into_stringActor*>(this)->actor_wait_state > 0) static_cast<Read_fixed_into_stringActor*>(this)->actor_wait_state = 0;
		static_cast<Read_fixed_into_stringActor*>(this)->ActorCallback< Read_fixed_into_stringActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_fixed_into_stringActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_fixed_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_fixed_into_stringActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_fixed_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_fixed_into_stringActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_fixed_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_fixed_into_string", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int len;
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string* buf;
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t pos;
															#line 254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int stop_size;
															#line 1379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_fixed_into_string()
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_fixed_into_stringActor final : public Actor<Void>, public ActorCallback< Read_fixed_into_stringActor, 0, Void >, public FastAllocated<Read_fixed_into_stringActor>, public Read_fixed_into_stringActorState<Read_fixed_into_stringActor> {
															#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_fixed_into_stringActor>::operator new;
	using FastAllocated<Read_fixed_into_stringActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10091896891012955136UL, 9323576814103911424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_fixed_into_stringActor, 0, Void >;
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_fixed_into_stringActor(Reference<IConnection> const& conn,int const& len,std::string* const& buf,size_t const& pos) 
															#line 1401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   Read_fixed_into_stringActorState<Read_fixed_into_stringActor>(conn, len, buf, pos),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_fixed_into_string", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6724135353552150528UL, 6435673751594239744UL);
		ActorExecutionContextHelper __helper(static_cast<Read_fixed_into_stringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_fixed_into_string");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_fixed_into_string", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_fixed_into_stringActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> read_fixed_into_string( Reference<IConnection> const& conn, int const& len, std::string* const& buf, size_t const& pos ) {
															#line 253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new Read_fixed_into_stringActor(conn, len, buf, pos));
															#line 1433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

															#line 1438 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_http_response_headers()
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_http_response_headersActor>
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_response_headersActorState {
															#line 1444 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_response_headersActorState(Reference<IConnection> const& conn,Headers* const& headers,std::string* const& buf,size_t* const& pos) 
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   headers(headers),
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(buf),
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(pos)
															#line 1457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_http_response_headers", reinterpret_cast<unsigned long>(this));

	}
	~Read_http_response_headersActorState() 
	{
		fdb_probe_actor_destroy("read_http_response_headers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 1472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_http_response_headersActorState();
		static_cast<Read_http_response_headersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<size_t> __when_expr_0 = read_delimited_into_string(conn, "\r\n", buf, *pos);
															#line 267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_http_response_headersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<Read_http_response_headersActor*>(this)->actor_wait_state = 1;
															#line 267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_http_response_headersActor, 0, size_t >*>(static_cast<Read_http_response_headersActor*>(this)));
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(size_t const& lineLen,int loopDepth) 
	{
															#line 269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (lineLen == 0)
															#line 1518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += 2;
															#line 272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<Read_http_response_headersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_response_headersActorState(); static_cast<Read_http_response_headersActor*>(this)->destroy(); return 0; }
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<Read_http_response_headersActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~Read_http_response_headersActorState();
			static_cast<Read_http_response_headersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int nameEnd = -1, valueStart = -1, valueEnd = -1;
															#line 276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = -1;
															#line 278 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string name, value;
															#line 289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (sscanf(buf->c_str() + *pos, "%*[^:]%n:%*[ \t]%n", &nameEnd, &valueStart) >= 0 && valueStart > 0)
															#line 1538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			name = std::string(buf->substr(*pos, nameEnd));
															#line 292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += valueStart;
															#line 1544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevError, "HTTPReadHeadersMalformed").detail("Buffer", *buf).detail("Pos", *pos);
															#line 296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 1552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (sscanf(buf->c_str() + *pos, "%*[^\r]%n%*1[\r]%*1[\n]%n", &valueEnd, &len) >= 0 && len > 0)
															#line 1556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			value = std::string(buf->substr(*pos, valueEnd));
															#line 302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += len;
															#line 1562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (sscanf(buf->c_str() + *pos, "%*1[\r]%*1[\n]%n", &len) >= 0 && len > 0)
															#line 1568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				*pos += len;
															#line 1572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
			else
			{
															#line 308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevError, "HTTPReadHeadersMalformed").detail("Buffer", *buf).detail("Pos", *pos);
															#line 309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 1580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}
															#line 312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		(*headers)[name] = value;
															#line 1585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(size_t && lineLen,int loopDepth) 
	{
															#line 269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (lineLen == 0)
															#line 1594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += 2;
															#line 272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<Read_http_response_headersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_response_headersActorState(); static_cast<Read_http_response_headersActor*>(this)->destroy(); return 0; }
															#line 1600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<Read_http_response_headersActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~Read_http_response_headersActorState();
			static_cast<Read_http_response_headersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int nameEnd = -1, valueStart = -1, valueEnd = -1;
															#line 276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = -1;
															#line 278 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string name, value;
															#line 289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (sscanf(buf->c_str() + *pos, "%*[^:]%n:%*[ \t]%n", &nameEnd, &valueStart) >= 0 && valueStart > 0)
															#line 1614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			name = std::string(buf->substr(*pos, nameEnd));
															#line 292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += valueStart;
															#line 1620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevError, "HTTPReadHeadersMalformed").detail("Buffer", *buf).detail("Pos", *pos);
															#line 296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 1628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (sscanf(buf->c_str() + *pos, "%*[^\r]%n%*1[\r]%*1[\n]%n", &valueEnd, &len) >= 0 && len > 0)
															#line 1632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			value = std::string(buf->substr(*pos, valueEnd));
															#line 302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos += len;
															#line 1638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (sscanf(buf->c_str() + *pos, "%*1[\r]%*1[\n]%n", &len) >= 0 && len > 0)
															#line 1644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				*pos += len;
															#line 1648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
			else
			{
															#line 308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevError, "HTTPReadHeadersMalformed").detail("Buffer", *buf).detail("Pos", *pos);
															#line 309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 1656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}
															#line 312 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		(*headers)[name] = value;
															#line 1661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(size_t const& lineLen,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(lineLen, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(size_t && lineLen,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(lineLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_http_response_headersActor*>(this)->actor_wait_state > 0) static_cast<Read_http_response_headersActor*>(this)->actor_wait_state = 0;
		static_cast<Read_http_response_headersActor*>(this)->ActorCallback< Read_http_response_headersActor, 0, size_t >::remove();

	}
	void a_callback_fire(ActorCallback< Read_http_response_headersActor, 0, size_t >*,size_t const& value) 
	{
		fdb_probe_actor_enter("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_response_headersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_http_response_headersActor, 0, size_t >*,size_t && value) 
	{
		fdb_probe_actor_enter("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_response_headersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_http_response_headersActor, 0, size_t >*,Error err) 
	{
		fdb_probe_actor_enter("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_response_headersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response_headers", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Headers* headers;
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string* buf;
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t* pos;
															#line 1749 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_http_response_headers()
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_response_headersActor final : public Actor<Void>, public ActorCallback< Read_http_response_headersActor, 0, size_t >, public FastAllocated<Read_http_response_headersActor>, public Read_http_response_headersActorState<Read_http_response_headersActor> {
															#line 1754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_http_response_headersActor>::operator new;
	using FastAllocated<Read_http_response_headersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7059565595714502656UL, 6718673074104396544UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_http_response_headersActor, 0, size_t >;
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_response_headersActor(Reference<IConnection> const& conn,Headers* const& headers,std::string* const& buf,size_t* const& pos) 
															#line 1771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   Read_http_response_headersActorState<Read_http_response_headersActor>(conn, headers, buf, pos),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_http_response_headers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16089228226944314368UL, 14219718383745876480UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_response_headersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_http_response_headers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_http_response_headers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_http_response_headersActor, 0, size_t >*)0, actor_cancelled()); break;
		}

	}
};
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> read_http_response_headers( Reference<IConnection> const& conn, Headers* const& headers, std::string* const& buf, size_t* const& pos ) {
															#line 261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new Read_http_response_headersActor(conn, headers, buf, pos));
															#line 1803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// buf has the http header line in it at *pos offset.
// FIXME: should this throw a different error for http requests? Or should we rename http_bad_response to
// http_bad_<something>?
															#line 1811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via readHTTPData()
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class ReadHTTPDataActor>
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class ReadHTTPDataActorState {
															#line 1817 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	ReadHTTPDataActorState(HTTPData<std::string>* const& r,Reference<IConnection> const& conn,std::string* const& buf,size_t* const& pos,bool const& content_optional,bool const& skipCheckMD5) 
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : r(r),
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   conn(conn),
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(buf),
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(pos),
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   content_optional(content_optional),
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   skipCheckMD5(skipCheckMD5)
															#line 1834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("readHTTPData", reinterpret_cast<unsigned long>(this));

	}
	~ReadHTTPDataActorState() 
	{
		fdb_probe_actor_destroy("readHTTPData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->headers.clear();
															#line 328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<Void> __when_expr_0 = read_http_response_headers(conn, &r->headers, buf, pos);
															#line 328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 1;
															#line 328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 0, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 1858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadHTTPDataActorState();
		static_cast<ReadHTTPDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		auto i = r->headers.find("Content-Length");
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (i != r->headers.end())
															#line 1883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->contentLen = strtoll(i->second.c_str(), NULL, 10);
															#line 1887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->contentLen = -1;
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		transferEncoding = std::string();
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		i = r->headers.find("Transfer-Encoding");
															#line 339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (i != r->headers.end())
															#line 1901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			transferEncoding = i->second;
															#line 1905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.clear();
															#line 347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (content_optional && *pos == buf->size())
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<ReadHTTPDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadHTTPDataActorState(); static_cast<ReadHTTPDataActor*>(this)->destroy(); return 0; }
															#line 1915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<ReadHTTPDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadHTTPDataActorState();
			static_cast<ReadHTTPDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (r->contentLen >= 0)
															#line 1923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->content = buf->substr(*pos);
															#line 356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos = 0;
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<Void> __when_expr_1 = read_fixed_into_string(conn, r->contentLen, &r->content, *pos);
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 2;
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 1, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (transferEncoding == "chunked")
															#line 1945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				r->content = buf->substr(*pos);
															#line 373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				*pos = 0;
															#line 375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				;
															#line 1953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				loopDepth = a_body1cont1loopHead1(loopDepth);
			}
			else
			{
															#line 423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevWarn, "HTTPUnknownContentScheme") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 1962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		auto i = r->headers.find("Content-Length");
															#line 331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (i != r->headers.end())
															#line 1974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->contentLen = strtoll(i->second.c_str(), NULL, 10);
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		else
		{
															#line 334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->contentLen = -1;
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		transferEncoding = std::string();
															#line 338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		i = r->headers.find("Transfer-Encoding");
															#line 339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (i != r->headers.end())
															#line 1992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			transferEncoding = i->second;
															#line 1996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.clear();
															#line 347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (content_optional && *pos == buf->size())
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<ReadHTTPDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadHTTPDataActorState(); static_cast<ReadHTTPDataActor*>(this)->destroy(); return 0; }
															#line 2006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<ReadHTTPDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadHTTPDataActorState();
			static_cast<ReadHTTPDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (r->contentLen >= 0)
															#line 2014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->content = buf->substr(*pos);
															#line 356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pos = 0;
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<Void> __when_expr_1 = read_fixed_into_string(conn, r->contentLen, &r->content, *pos);
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 2;
															#line 359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 1, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (transferEncoding == "chunked")
															#line 2036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				r->content = buf->substr(*pos);
															#line 373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				*pos = 0;
															#line 375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				;
															#line 2044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				loopDepth = a_body1cont1loopHead1(loopDepth);
			}
			else
			{
															#line 423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevWarn, "HTTPUnknownContentScheme") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (r->content.size() > 0)
															#line 2138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (skipCheckMD5)
															#line 2142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				if (!static_cast<ReadHTTPDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadHTTPDataActorState(); static_cast<ReadHTTPDataActor*>(this)->destroy(); return 0; }
															#line 2146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				new (&static_cast<ReadHTTPDataActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ReadHTTPDataActorState();
				static_cast<ReadHTTPDataActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!HTTP::verifyMD5(r, false))
															#line 2154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevWarn, "HTTPMD5Mismatch") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2160 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}
															#line 443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<ReadHTTPDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadHTTPDataActorState(); static_cast<ReadHTTPDataActor*>(this)->destroy(); return 0; }
															#line 2165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<ReadHTTPDataActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReadHTTPDataActorState();
		static_cast<ReadHTTPDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (r->content.size() != r->contentLen)
															#line 2177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPContentLengthMismatch") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (r->content.size() != r->contentLen)
															#line 2193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPContentLengthMismatch") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont10(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
															#line 406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->contentLen = *pos;
															#line 409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_5 = read_http_response_headers(conn, &r->headers, &r->content, pos);
															#line 409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont11when1(__when_expr_5.get(), loopDepth); };
		static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 6;
															#line 409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 5, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		{
															#line 378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<size_t> __when_expr_2 = read_delimited_into_string(conn, "\r\n", &r->content, *pos);
															#line 378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 3;
															#line 378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 2, size_t >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		{
															#line 396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<size_t> __when_expr_4 = read_delimited_into_string(conn, "\r\n", &r->content, *pos);
															#line 396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2349 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 5;
															#line 396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 4, size_t >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(size_t const& lineLen,int loopDepth) 
	{
															#line 379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		chunkLen = strtol(r->content.substr(*pos, lineLen).c_str(), nullptr, 16);
															#line 383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(*pos, lineLen + 2);
															#line 386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (chunkLen == 0)
															#line 2368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_3 = read_fixed_into_string(conn, chunkLen, &r->content, *pos);
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 4;
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 3, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(size_t && lineLen,int loopDepth) 
	{
															#line 379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		chunkLen = strtol(r->content.substr(*pos, lineLen).c_str(), nullptr, 16);
															#line 383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(*pos, lineLen + 2);
															#line 386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (chunkLen == 0)
															#line 2394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_3 = read_fixed_into_string(conn, chunkLen, &r->content, *pos);
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 4;
															#line 390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadHTTPDataActor, 3, Void >*>(static_cast<ReadHTTPDataActor*>(this)));
															#line 2407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(size_t const& lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(lineLen, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(size_t && lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(lineLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 2, size_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 2, size_t >*,size_t const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 2, size_t >*,size_t && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 2, size_t >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		*pos += chunkLen;
															#line 2491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		*pos += chunkLen;
															#line 2500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(size_t const& lineLen,int loopDepth) 
	{
															#line 397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (lineLen != 0)
															#line 2590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 2594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(*pos, 2);
															#line 2598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(size_t && lineLen,int loopDepth) 
	{
															#line 397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (lineLen != 0)
															#line 2607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 2611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(*pos, 2);
															#line 2615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(size_t const& lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(lineLen, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(size_t && lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(std::move(lineLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 4, size_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 4, size_t >*,size_t const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 4, size_t >*,size_t && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 4, size_t >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11cont1(Void const& _,int loopDepth) 
	{
															#line 412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (*pos != r->content.size())
															#line 2699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 413 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPContentLengthMismatch2") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2705 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(r->contentLen);
															#line 2709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1(Void && _,int loopDepth) 
	{
															#line 412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (*pos != r->content.size())
															#line 2718 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 413 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPContentLengthMismatch2") .detail("ContentLength", r->contentLen) .detail("ContentSize", r->content.size());
															#line 416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r->content.erase(r->contentLen);
															#line 2728 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ReadHTTPDataActor*>(this)->actor_wait_state > 0) static_cast<ReadHTTPDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadHTTPDataActor*>(this)->ActorCallback< ReadHTTPDataActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ReadHTTPDataActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ReadHTTPDataActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	HTTPData<std::string>* r;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string* buf;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t* pos;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	bool content_optional;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	bool skipCheckMD5;
															#line 337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string transferEncoding;
															#line 379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int chunkLen;
															#line 2824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via readHTTPData()
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class ReadHTTPDataActor final : public Actor<Void>, public ActorCallback< ReadHTTPDataActor, 0, Void >, public ActorCallback< ReadHTTPDataActor, 1, Void >, public ActorCallback< ReadHTTPDataActor, 2, size_t >, public ActorCallback< ReadHTTPDataActor, 3, Void >, public ActorCallback< ReadHTTPDataActor, 4, size_t >, public ActorCallback< ReadHTTPDataActor, 5, Void >, public FastAllocated<ReadHTTPDataActor>, public ReadHTTPDataActorState<ReadHTTPDataActor> {
															#line 2829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<ReadHTTPDataActor>::operator new;
	using FastAllocated<ReadHTTPDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4530736372839140608UL, 8424631535554642176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadHTTPDataActor, 0, Void >;
friend struct ActorCallback< ReadHTTPDataActor, 1, Void >;
friend struct ActorCallback< ReadHTTPDataActor, 2, size_t >;
friend struct ActorCallback< ReadHTTPDataActor, 3, Void >;
friend struct ActorCallback< ReadHTTPDataActor, 4, size_t >;
friend struct ActorCallback< ReadHTTPDataActor, 5, Void >;
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	ReadHTTPDataActor(HTTPData<std::string>* const& r,Reference<IConnection> const& conn,std::string* const& buf,size_t* const& pos,bool const& content_optional,bool const& skipCheckMD5) 
															#line 2851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   ReadHTTPDataActorState<ReadHTTPDataActor>(r, conn, buf, pos, content_optional, skipCheckMD5),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readHTTPData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1318857027048510208UL, 4056033421168738048UL);
		ActorExecutionContextHelper __helper(static_cast<ReadHTTPDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readHTTPData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readHTTPData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 2, size_t >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 4, size_t >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ReadHTTPDataActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> readHTTPData( HTTPData<std::string>* const& r, Reference<IConnection> const& conn, std::string* const& buf, size_t* const& pos, bool const& content_optional, bool const& skipCheckMD5 ) {
															#line 319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new ReadHTTPDataActor(r, conn, buf, pos, content_optional, skipCheckMD5));
															#line 2888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// Reads an HTTP request from a network connection
// If the connection fails while being read the exception will emitted
// If the response is not parsable or complete in some way, http_bad_response will be thrown
															#line 2896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_http_request()
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_http_requestActor>
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_requestActorState {
															#line 2902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_requestActorState(Reference<HTTP::IncomingRequest> const& r,Reference<IConnection> const& conn) 
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : r(r),
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   conn(conn),
															#line 450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(),
															#line 451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(0)
															#line 2915 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_http_request", reinterpret_cast<unsigned long>(this));

	}
	~Read_http_requestActorState() 
	{
		fdb_probe_actor_destroy("read_http_request", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<size_t> __when_expr_0 = read_delimited_into_string(conn, "\r\n", &buf, pos);
															#line 454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<Read_http_requestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Read_http_requestActor*>(this)->actor_wait_state = 1;
															#line 454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_http_requestActor, 0, size_t >*>(static_cast<Read_http_requestActor*>(this)));
															#line 2937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_http_requestActorState();
		static_cast<Read_http_requestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(size_t const& lineLen,int loopDepth) 
	{
															#line 457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string requestLine = buf.substr(0, lineLen);
															#line 458 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::stringstream ss(requestLine);
															#line 461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> r->verb;
															#line 462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 2966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestVerbNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> r->resource;
															#line 472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 2978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestResourceNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string httpVersion;
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> httpVersion;
															#line 483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 2992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestHTTPVersionNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 2998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss && !ss.eof())
															#line 3002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestExtraData").detail("Buffer", buf).detail("Pos", pos).detail("LineLen", lineLen);
															#line 493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		float version;
															#line 497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sscanf(httpVersion.c_str(), "HTTP/%f", &version);
															#line 498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (version < 1.1)
															#line 3016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestHTTPVersionLessThan1_1") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 503 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3022 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		pos += lineLen + 2;
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = readHTTPData(&r->data, conn, &buf, &pos, false, false);
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_http_requestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_http_requestActor*>(this)->actor_wait_state = 2;
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_http_requestActor, 1, Void >*>(static_cast<Read_http_requestActor*>(this)));
															#line 3035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(size_t && lineLen,int loopDepth) 
	{
															#line 457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string requestLine = buf.substr(0, lineLen);
															#line 458 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::stringstream ss(requestLine);
															#line 461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> r->verb;
															#line 462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 3050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestVerbNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> r->resource;
															#line 472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 3062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestResourceNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		std::string httpVersion;
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		ss >> httpVersion;
															#line 483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss.fail())
															#line 3076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestHTTPVersionNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (ss && !ss.eof())
															#line 3086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestExtraData").detail("Buffer", buf).detail("Pos", pos).detail("LineLen", lineLen);
															#line 493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		float version;
															#line 497 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sscanf(httpVersion.c_str(), "HTTP/%f", &version);
															#line 498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (version < 1.1)
															#line 3100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPRequestHTTPVersionLessThan1_1") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen);
															#line 503 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		pos += lineLen + 2;
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = readHTTPData(&r->data, conn, &buf, &pos, false, false);
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_http_requestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_http_requestActor*>(this)->actor_wait_state = 2;
															#line 509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_http_requestActor, 1, Void >*>(static_cast<Read_http_requestActor*>(this)));
															#line 3119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(size_t const& lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1(lineLen, loopDepth);

		return loopDepth;
	}
	int a_body1when1(size_t && lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(lineLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_http_requestActor*>(this)->actor_wait_state > 0) static_cast<Read_http_requestActor*>(this)->actor_wait_state = 0;
		static_cast<Read_http_requestActor*>(this)->ActorCallback< Read_http_requestActor, 0, size_t >::remove();

	}
	void a_callback_fire(ActorCallback< Read_http_requestActor, 0, size_t >*,size_t const& value) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_http_requestActor, 0, size_t >*,size_t && value) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_http_requestActor, 0, size_t >*,Error err) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<Read_http_requestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_requestActorState(); static_cast<Read_http_requestActor*>(this)->destroy(); return 0; }
															#line 3203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<Read_http_requestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Read_http_requestActorState();
		static_cast<Read_http_requestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<Read_http_requestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_requestActorState(); static_cast<Read_http_requestActor*>(this)->destroy(); return 0; }
															#line 3215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<Read_http_requestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Read_http_requestActorState();
		static_cast<Read_http_requestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Read_http_requestActor*>(this)->actor_wait_state > 0) static_cast<Read_http_requestActor*>(this)->actor_wait_state = 0;
		static_cast<Read_http_requestActor*>(this)->ActorCallback< Read_http_requestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_http_requestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Read_http_requestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Read_http_requestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<HTTP::IncomingRequest> r;
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 450 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string buf;
															#line 451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t pos;
															#line 3306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_http_request()
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_requestActor final : public Actor<Void>, public ActorCallback< Read_http_requestActor, 0, size_t >, public ActorCallback< Read_http_requestActor, 1, Void >, public FastAllocated<Read_http_requestActor>, public Read_http_requestActorState<Read_http_requestActor> {
															#line 3311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_http_requestActor>::operator new;
	using FastAllocated<Read_http_requestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5065797244885774848UL, 11726265518004779520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_http_requestActor, 0, size_t >;
friend struct ActorCallback< Read_http_requestActor, 1, Void >;
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_requestActor(Reference<HTTP::IncomingRequest> const& r,Reference<IConnection> const& conn) 
															#line 3329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   Read_http_requestActorState<Read_http_requestActor>(r, conn),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_http_request", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5287104334665317376UL, 1540967142947874048UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_requestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_http_request");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_http_request", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_http_requestActor, 0, size_t >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Read_http_requestActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> read_http_request( Reference<HTTP::IncomingRequest> const& r, Reference<IConnection> const& conn ) {
															#line 449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new Read_http_requestActor(r, conn));
															#line 3362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

Future<Void> HTTP::IncomingRequest::read(Reference<IConnection> conn, bool header_only) {
	return read_http_request(Reference<HTTP::IncomingRequest>::addRef(this), conn);
}

Future<Void> HTTP::OutgoingResponse::write(Reference<IConnection> conn) {
	return writeResponse(conn, Reference<HTTP::OutgoingResponse>::addRef(this));
}

void HTTP::OutgoingResponse::reset() {
	data.headers = HTTP::Headers();
	data.content->discardAll();
	data.contentLen = 0;
}

// Reads an HTTP response from a network connection
// If the connection fails while being read the exception will emitted
// If the response is not parsable or complete in some way, http_bad_response will be thrown
															#line 3384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via read_http_response()
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class Read_http_responseActor>
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_responseActorState {
															#line 3390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_responseActorState(Reference<HTTP::IncomingResponse> const& r,Reference<IConnection> const& conn,bool const& header_only) 
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : r(r),
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   conn(conn),
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   header_only(header_only),
															#line 533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   buf(),
															#line 534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pos(0)
															#line 3405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("read_http_response", reinterpret_cast<unsigned long>(this));

	}
	~Read_http_responseActorState() 
	{
		fdb_probe_actor_destroy("read_http_response", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<size_t> __when_expr_0 = read_delimited_into_string(conn, "\r\n", &buf, pos);
															#line 537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<Read_http_responseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Read_http_responseActor*>(this)->actor_wait_state = 1;
															#line 537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Read_http_responseActor, 0, size_t >*>(static_cast<Read_http_responseActor*>(this)));
															#line 3427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Read_http_responseActorState();
		static_cast<Read_http_responseActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(size_t const& lineLen,int loopDepth) 
	{
															#line 539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int reachedEnd = -1;
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sscanf(buf.c_str() + pos, "HTTP/%f %d%n", &r->version, &r->code, &reachedEnd);
															#line 541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (reachedEnd < 0)
															#line 3454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPResponseCodeNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen) .log();
															#line 547 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		pos += lineLen + 2;
															#line 553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		bool skipCheckMD5 = r->code == 206 && FLOW_KNOBS->HTTP_RESPONSE_SKIP_VERIFY_CHECKSUM_FOR_PARTIAL_CONTENT;
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = readHTTPData(&r->data, conn, &buf, &pos, header_only, skipCheckMD5);
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_http_responseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_http_responseActor*>(this)->actor_wait_state = 2;
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_http_responseActor, 1, Void >*>(static_cast<Read_http_responseActor*>(this)));
															#line 3475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(size_t && lineLen,int loopDepth) 
	{
															#line 539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int reachedEnd = -1;
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sscanf(buf.c_str() + pos, "HTTP/%f %d%n", &r->version, &r->code, &reachedEnd);
															#line 541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (reachedEnd < 0)
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevWarn, "HTTPResponseCodeNotFound") .detail("Buffer", buf) .detail("Pos", pos) .detail("LineLen", lineLen) .log();
															#line 547 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 3494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		pos += lineLen + 2;
															#line 553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		bool skipCheckMD5 = r->code == 206 && FLOW_KNOBS->HTTP_RESPONSE_SKIP_VERIFY_CHECKSUM_FOR_PARTIAL_CONTENT;
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = readHTTPData(&r->data, conn, &buf, &pos, header_only, skipCheckMD5);
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<Read_http_responseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Read_http_responseActor*>(this)->actor_wait_state = 2;
															#line 555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Read_http_responseActor, 1, Void >*>(static_cast<Read_http_responseActor*>(this)));
															#line 3509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(size_t const& lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1(lineLen, loopDepth);

		return loopDepth;
	}
	int a_body1when1(size_t && lineLen,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(lineLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Read_http_responseActor*>(this)->actor_wait_state > 0) static_cast<Read_http_responseActor*>(this)->actor_wait_state = 0;
		static_cast<Read_http_responseActor*>(this)->ActorCallback< Read_http_responseActor, 0, size_t >::remove();

	}
	void a_callback_fire(ActorCallback< Read_http_responseActor, 0, size_t >*,size_t const& value) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Read_http_responseActor, 0, size_t >*,size_t && value) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Read_http_responseActor, 0, size_t >*,Error err) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<Read_http_responseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_responseActorState(); static_cast<Read_http_responseActor*>(this)->destroy(); return 0; }
															#line 3593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<Read_http_responseActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Read_http_responseActorState();
		static_cast<Read_http_responseActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<Read_http_responseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Read_http_responseActorState(); static_cast<Read_http_responseActor*>(this)->destroy(); return 0; }
															#line 3605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<Read_http_responseActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Read_http_responseActorState();
		static_cast<Read_http_responseActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Read_http_responseActor*>(this)->actor_wait_state > 0) static_cast<Read_http_responseActor*>(this)->actor_wait_state = 0;
		static_cast<Read_http_responseActor*>(this)->ActorCallback< Read_http_responseActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Read_http_responseActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Read_http_responseActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Read_http_responseActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	bool header_only;
															#line 533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string buf;
															#line 534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	size_t pos;
															#line 3698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via read_http_response()
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class Read_http_responseActor final : public Actor<Void>, public ActorCallback< Read_http_responseActor, 0, size_t >, public ActorCallback< Read_http_responseActor, 1, Void >, public FastAllocated<Read_http_responseActor>, public Read_http_responseActorState<Read_http_responseActor> {
															#line 3703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<Read_http_responseActor>::operator new;
	using FastAllocated<Read_http_responseActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17881169942267724544UL, 12225319027303241216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Read_http_responseActor, 0, size_t >;
friend struct ActorCallback< Read_http_responseActor, 1, Void >;
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Read_http_responseActor(Reference<HTTP::IncomingResponse> const& r,Reference<IConnection> const& conn,bool const& header_only) 
															#line 3721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   Read_http_responseActorState<Read_http_responseActor>(r, conn, header_only),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read_http_response", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2330803063780665088UL, 18144549331055198208UL);
		ActorExecutionContextHelper __helper(static_cast<Read_http_responseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read_http_response");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read_http_response", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Read_http_responseActor, 0, size_t >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Read_http_responseActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> read_http_response( Reference<HTTP::IncomingResponse> const& r, Reference<IConnection> const& conn, bool const& header_only ) {
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new Read_http_responseActor(r, conn, header_only));
															#line 3754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 559 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

Future<Void> HTTP::IncomingResponse::read(Reference<IConnection> conn, bool header_only) {
	return read_http_response(Reference<HTTP::IncomingResponse>::addRef(this), conn, header_only);
}

// Do a request, get a Response.
// Request content is provided as UnsentPacketQueue in req, which will be depleted as bytes are sent but the queue
// itself must live for the life of this actor and be destroyed by the caller
// TODO:  pSent is very hackish, do something better.
															#line 3767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via doRequestActor()
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class DoRequestActorActor>
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class DoRequestActorActorState {
															#line 3773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	DoRequestActorActorState(Reference<IConnection> const& conn,Reference<OutgoingRequest> const& request,Reference<IRateControl> const& sendRate,int64_t* const& pSent,Reference<IRateControl> const& recvRate) 
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   request(request),
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   sendRate(sendRate),
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   pSent(pSent),
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   recvRate(recvRate),
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   event(SevDebug, "HTTPRequest"),
															#line 577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   requestIDHeader(FLOW_KNOBS->HTTP_REQUEST_ID_HEADER),
															#line 579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   earlyResponse(false),
															#line 580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   total_sent(0),
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   send_start()
															#line 3798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("doRequestActor", reinterpret_cast<unsigned long>(this));

	}
	~DoRequestActorActorState() 
	{
		fdb_probe_actor_destroy("doRequestActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 583 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("DebugID", conn->getDebugID());
															#line 584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("RemoteAddress", conn->getPeerAddress());
															#line 585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("Verb", request->verb);
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("Resource", request->resource);
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("RequestContentLen", request->data.contentLen);
															#line 3821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			try {
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				requestID = std::string();
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				if (!requestIDHeader.empty())
															#line 3827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				{
															#line 592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					requestID = deterministicRandom()->randomUniqueID().toString();
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					requestID = requestID.insert(20, "-");
															#line 594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					requestID = requestID.insert(16, "-");
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					requestID = requestID.insert(12, "-");
															#line 596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					requestID = requestID.insert(8, "-");
															#line 598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					request->data.headers[requestIDHeader] = requestID;
															#line 599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					event.detail("RequestIDSent", requestID);
															#line 3843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				}
															#line 601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				request->data.headers["Content-Length"] = std::to_string(request->data.contentLen);
															#line 604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				PacketBuffer* pFirst = PacketBuffer::create();
															#line 605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				PacketBuffer* pLast = writeRequestHeader(request, pFirst);
															#line 607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				request->data.content->prependWriteBuffer(pFirst, pLast);
															#line 609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 1)
															#line 3855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				{
															#line 610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					fmt::print("[{}] HTTP starting {} {} ContentLen:{}\n", conn->getDebugID().toString(), request->verb, request->resource, request->data.contentLen);
															#line 3859 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				}
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 2)
															#line 3863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				{
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					for( auto h : request->data.headers ) {
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
						fmt::print("Request Header: {}: {}\n", h.first, h.second);
															#line 3869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
					}
				}
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				r = Reference<HTTP::IncomingResponse>(new HTTP::IncomingResponse());
															#line 621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				event.detail("IsHeaderOnlyResponse", request->isHeaderOnlyResponse());
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				responseReading = r->read(conn, request->isHeaderOnlyResponse());
															#line 623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				event.detail("ResponseReading", responseReading.isReady());
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				send_start = timer();
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				;
															#line 3884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoRequestActorActorState();
		static_cast<DoRequestActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			double elapsed = timer() - send_start;
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 0 && e.code() != error_code_http_bad_request_id)
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				fmt::print("[{}] HTTP *ERROR*={} early={}, time={}s {} {} contentLen={} [{} out]\n", conn->getDebugID().toString(), e.name(), earlyResponse, elapsed, request->verb, request->resource, request->data.contentLen, total_sent);
															#line 3920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.errorUnsuppressed(e);
															#line 751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_3 = responseReading;
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 4;
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 3, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 3947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (responseReading.isReady())
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 634 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent("HTTPRequestConectionClosing") .detail("Ready", responseReading.isReady()) .detail("Error", responseReading.isError()) .detail("QueueEmpty", request->data.content->empty()) .detail("Verb", request->verb) .detail("TotalSent", total_sent) .detail("ContentLen", request->data.contentLen) .log();
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			conn->close();
															#line 643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			r->data.headers["Connection"] = "close";
															#line 644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			earlyResponse = true;
															#line 3973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		trySend = FLOW_KNOBS->HTTP_SEND_SIZE;
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if ((!g_network->isSimulated() || !g_simulator->speedUpSimulation) && BUGGIFY_WITH_PROB(0.01))
															#line 3980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			trySend = deterministicRandom()->randomInt(1, 10);
															#line 3984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_0 = sendRate->getAllowance(trySend);
															#line 652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 1;
															#line 652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 0, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 3995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = conn->write(request->data.content->getUnsent(), trySend);
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (pSent != nullptr)
															#line 4019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pSent += len;
															#line 4023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sendRate->returnUnused(trySend - len);
															#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		total_sent += len;
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		request->data.content->sent(len);
															#line 659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (request->data.content->empty())
															#line 4033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = conn->onWritable();
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 2;
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 1, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 4046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		int len = conn->write(request->data.content->getUnsent(), trySend);
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (pSent != nullptr)
															#line 4057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			*pSent += len;
															#line 4061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		sendRate->returnUnused(trySend - len);
															#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		total_sent += len;
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		request->data.content->sent(len);
															#line 659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (request->data.content->empty())
															#line 4071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_1 = conn->onWritable();
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 2;
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 1, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 4084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state > 0) static_cast<DoRequestActorActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequestActorActor*>(this)->ActorCallback< DoRequestActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoRequestActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::WriteSocket);
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 3;
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 2, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 4175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::WriteSocket);
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<DoRequestActorActor*>(this)->actor_wait_state = 3;
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoRequestActorActor, 2, Void >*>(static_cast<DoRequestActorActor*>(this)));
															#line 4191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state > 0) static_cast<DoRequestActorActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequestActorActor*>(this)->ActorCallback< DoRequestActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoRequestActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state > 0) static_cast<DoRequestActorActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequestActorActor*>(this)->ActorCallback< DoRequestActorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoRequestActorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		double elapsed = timer() - send_start;
															#line 670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("ResponseCode", r->code);
															#line 671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("ResponseContentLen", r->data.contentLen);
															#line 672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("Elapsed", elapsed);
															#line 673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("RequestIDHeader", requestIDHeader);
															#line 675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		Optional<Error> err;
															#line 676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!requestIDHeader.empty())
															#line 4374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			std::string responseID;
															#line 678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			auto iid = r->data.headers.find(requestIDHeader);
															#line 679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (iid != r->data.headers.end())
															#line 4382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				responseID = iid->second;
															#line 4386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("RequestIDReceived", responseID);
															#line 686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			bool serverError = r->code >= 500 && r->code < 600;
															#line 690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (requestID != responseID && (!serverError || !responseID.empty()))
															#line 4394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				event.detail("RequestIDMismatch", true);
															#line 692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				err = http_bad_request_id();
															#line 694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevWarn, "HTTPRequestFailedIDMismatch") .error(err.get()) .detail("DebugID", conn->getDebugID()) .detail("RemoteAddress", conn->getPeerAddress()) .detail("Verb", request->verb) .detail("Resource", request->resource) .detail("RequestContentLen", request->data.contentLen) .detail("ResponseCode", r->code) .detail("ResponseContentLen", r->data.contentLen) .detail("RequestIDSent", requestID) .detail("RequestIDReceived", responseID);
															#line 4402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}
															#line 708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 0)
															#line 4407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			fmt::print("[{0}] HTTP {1}code={2} early={3}, time={4} {5} {6} contentLen={7} [{8} out, response content " "len {9}]\n", conn->getDebugID().toString(), (err.present() ? format("*ERROR*=%s ", err.get().name()).c_str() : ""), r->code, earlyResponse, elapsed, request->verb, request->resource, request->data.contentLen, total_sent, r->data.contentLen);
															#line 4411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 2)
															#line 4415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			fmt::print("[{}] HTTP RESPONSE:  {} {}\n{}\n", conn->getDebugID().toString(), request->verb, request->resource, r->toString());
															#line 4419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (err.present())
															#line 4423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch2(err.get(), loopDepth);
															#line 4427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<DoRequestActorActor*>(this)->SAV<Reference<HTTP::IncomingResponse>>::futures) { (void)(r); this->~DoRequestActorActorState(); static_cast<DoRequestActorActor*>(this)->destroy(); return 0; }
															#line 4431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<DoRequestActorActor*>(this)->SAV< Reference<HTTP::IncomingResponse> >::value()) Reference<HTTP::IncomingResponse>(std::move(r)); // state_var_RVO
		this->~DoRequestActorActorState();
		static_cast<DoRequestActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		double elapsed = timer() - send_start;
															#line 670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("ResponseCode", r->code);
															#line 671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("ResponseContentLen", r->data.contentLen);
															#line 672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("Elapsed", elapsed);
															#line 673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("RequestIDHeader", requestIDHeader);
															#line 675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		Optional<Error> err;
															#line 676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!requestIDHeader.empty())
															#line 4455 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			std::string responseID;
															#line 678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			auto iid = r->data.headers.find(requestIDHeader);
															#line 679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (iid != r->data.headers.end())
															#line 4463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				responseID = iid->second;
															#line 4467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("RequestIDReceived", responseID);
															#line 686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			bool serverError = r->code >= 500 && r->code < 600;
															#line 690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (requestID != responseID && (!serverError || !responseID.empty()))
															#line 4475 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				event.detail("RequestIDMismatch", true);
															#line 692 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				err = http_bad_request_id();
															#line 694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				TraceEvent(SevWarn, "HTTPRequestFailedIDMismatch") .error(err.get()) .detail("DebugID", conn->getDebugID()) .detail("RemoteAddress", conn->getPeerAddress()) .detail("Verb", request->verb) .detail("Resource", request->resource) .detail("RequestContentLen", request->data.contentLen) .detail("ResponseCode", r->code) .detail("ResponseContentLen", r->data.contentLen) .detail("RequestIDSent", requestID) .detail("RequestIDReceived", responseID);
															#line 4483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
		}
															#line 708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 0)
															#line 4488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 709 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			fmt::print("[{0}] HTTP {1}code={2} early={3}, time={4} {5} {6} contentLen={7} [{8} out, response content " "len {9}]\n", conn->getDebugID().toString(), (err.present() ? format("*ERROR*=%s ", err.get().name()).c_str() : ""), r->code, earlyResponse, elapsed, request->verb, request->resource, request->data.contentLen, total_sent, r->data.contentLen);
															#line 4492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (FLOW_KNOBS->HTTP_VERBOSE_LEVEL > 2)
															#line 4496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			fmt::print("[{}] HTTP RESPONSE:  {} {}\n{}\n", conn->getDebugID().toString(), request->verb, request->resource, r->toString());
															#line 4500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (err.present())
															#line 4504 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch2(err.get(), loopDepth);
															#line 4508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<DoRequestActorActor*>(this)->SAV<Reference<HTTP::IncomingResponse>>::futures) { (void)(r); this->~DoRequestActorActorState(); static_cast<DoRequestActorActor*>(this)->destroy(); return 0; }
															#line 4512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<DoRequestActorActor*>(this)->SAV< Reference<HTTP::IncomingResponse> >::value()) Reference<HTTP::IncomingResponse>(std::move(r)); // state_var_RVO
		this->~DoRequestActorActorState();
		static_cast<DoRequestActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DoRequestActorActor*>(this)->actor_wait_state > 0) static_cast<DoRequestActorActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequestActorActor*>(this)->ActorCallback< DoRequestActorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DoRequestActorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DoRequestActorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<OutgoingRequest> request;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IRateControl> sendRate;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int64_t* pSent;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IRateControl> recvRate;
															#line 573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	TraceEvent event;
															#line 577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string requestIDHeader;
															#line 579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	bool earlyResponse;
															#line 580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int total_sent;
															#line 581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	double send_start;
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string requestID;
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Future<Void> responseReading;
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int trySend;
															#line 4623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via doRequestActor()
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class DoRequestActorActor final : public Actor<Reference<HTTP::IncomingResponse>>, public ActorCallback< DoRequestActorActor, 0, Void >, public ActorCallback< DoRequestActorActor, 1, Void >, public ActorCallback< DoRequestActorActor, 2, Void >, public ActorCallback< DoRequestActorActor, 3, Void >, public FastAllocated<DoRequestActorActor>, public DoRequestActorActorState<DoRequestActorActor> {
															#line 4628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<DoRequestActorActor>::operator new;
	using FastAllocated<DoRequestActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9309630098459170304UL, 10383924507637832704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<HTTP::IncomingResponse>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoRequestActorActor, 0, Void >;
friend struct ActorCallback< DoRequestActorActor, 1, Void >;
friend struct ActorCallback< DoRequestActorActor, 2, Void >;
friend struct ActorCallback< DoRequestActorActor, 3, Void >;
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	DoRequestActorActor(Reference<IConnection> const& conn,Reference<OutgoingRequest> const& request,Reference<IRateControl> const& sendRate,int64_t* const& pSent,Reference<IRateControl> const& recvRate) 
															#line 4648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Reference<HTTP::IncomingResponse>>(),
		   DoRequestActorActorState<DoRequestActorActor>(conn, request, sendRate, pSent, recvRate),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doRequestActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3601371249093706496UL, 1975182758072853248UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequestActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doRequestActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doRequestActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoRequestActorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoRequestActorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoRequestActorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DoRequestActorActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Reference<HTTP::IncomingResponse>> doRequestActor( Reference<IConnection> const& conn, Reference<OutgoingRequest> const& request, Reference<IRateControl> const& sendRate, int64_t* const& pSent, Reference<IRateControl> const& recvRate ) {
															#line 568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Reference<HTTP::IncomingResponse>>(new DoRequestActorActor(conn, request, sendRate, pSent, recvRate));
															#line 4683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

// IDE build didn't like the actor conversion i guess
Future<Reference<IncomingResponse>> doRequest(Reference<IConnection> conn,
                                              Reference<OutgoingRequest> request,
                                              Reference<IRateControl> sendRate,
                                              int64_t* pSent,
                                              Reference<IRateControl> recvRate) {
	return doRequestActor(conn, request, sendRate, pSent, recvRate);
}

															#line 4697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via sendProxyConnectRequest()
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class SendProxyConnectRequestActor>
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class SendProxyConnectRequestActorState {
															#line 4703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	SendProxyConnectRequestActorState(Reference<IConnection> const& conn,std::string const& remoteHost,std::string const& remoteService) 
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : conn(conn),
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   remoteHost(remoteHost),
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   remoteService(remoteService),
															#line 767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   requestTimeout(60),
															#line 768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   maxTries(FLOW_KNOBS->RESTCLIENT_CONNECT_TRIES),
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   thisTry(1),
															#line 770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   nextRetryDelay(2.0),
															#line 771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   sendReceiveRate(makeReference<Unlimited>()),
															#line 772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   bytes_sent(0),
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   empty_packet_queue(),
															#line 775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   req(makeReference<HTTP::OutgoingRequest>())
															#line 4730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this));

	}
	~SendProxyConnectRequestActorState() 
	{
		fdb_probe_actor_destroy("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 776 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->verb = HTTP_VERB_CONNECT;
															#line 777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->resource = remoteHost + ":" + remoteService;
															#line 778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->data.content = &empty_packet_queue;
															#line 779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->data.contentLen = 0;
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->data.headers["Host"] = req->resource;
															#line 781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->data.headers["Accept"] = "application/xml";
															#line 782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			req->data.headers["Proxy-Connection"] = "Keep-Alive";
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			;
															#line 4759 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SendProxyConnectRequestActorState();
		static_cast<SendProxyConnectRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		err = Optional<Error>();
															#line 787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 4791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		try {
															#line 790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			Future<Reference<HTTP::IncomingResponse>> f = HTTP::doRequest(conn, req, sendReceiveRate, &bytes_sent, sendReceiveRate);
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_0 = timeoutError(f, requestTimeout);
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state = 1;
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >*>(static_cast<SendProxyConnectRequestActor*>(this)));
															#line 4804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!err.present() && r->code == 200)
															#line 4819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent(SevDebug, "ProxyRequestSuccess") .detail("RemoteHost", remoteHost) .detail("RemoteService", remoteService);
															#line 807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (!static_cast<SendProxyConnectRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendProxyConnectRequestActorState(); static_cast<SendProxyConnectRequestActor*>(this)->destroy(); return 0; }
															#line 4825 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			new (&static_cast<SendProxyConnectRequestActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~SendProxyConnectRequestActorState();
			static_cast<SendProxyConnectRequestActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		bool retryable = err.present() || r->code == 500 || r->code == 502 || r->code == 503 || r->code == 429;
															#line 814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		retryable = retryable && (thisTry < maxTries);
															#line 816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		TraceEvent event(SevWarn, retryable ? "ProxyConnectCommandFailedRetryable" : "ProxyConnectCommandFailed");
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (err.present())
															#line 4839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.errorUnsuppressed(err.get());
															#line 4843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.suppressFor(60);
															#line 823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!err.present())
															#line 4849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("ResponseCode", r->code);
															#line 4853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		event.detail("ThisTry", thisTry);
															#line 831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!r || r->code != 429)
															#line 4859 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			++thisTry;
															#line 4863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}
															#line 835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		double delay = nextRetryDelay;
															#line 837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		nextRetryDelay = std::min(nextRetryDelay * 2, 60.0);
															#line 839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (retryable)
															#line 4871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		{
															#line 841 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (r)
															#line 4875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				auto iRetryAfter = r->data.headers.find("Retry-After");
															#line 843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				if (iRetryAfter != r->data.headers.end())
															#line 4881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				{
															#line 844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					event.detail("RetryAfterHeader", iRetryAfter->second);
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					char* pEnd;
															#line 846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					double retryAfter = strtod(iRetryAfter->second.c_str(), &pEnd);
															#line 847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					if (*pEnd)
															#line 4891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
					{
															#line 849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
						retryAfter = 300;
															#line 4895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
					}
															#line 851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
					delay = std::max(delay, retryAfter);
															#line 4899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
				}
			}
															#line 856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			event.detail("RetryDelay", delay);
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ::delay(delay);
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state = 2;
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SendProxyConnectRequestActor, 1, Void >*>(static_cast<SendProxyConnectRequestActor*>(this)));
															#line 4913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (r && r->code == 406)
															#line 4920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_not_accepted(), std::max(0, loopDepth - 1));
															#line 4924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (r && r->code == 401)
															#line 4928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(http_auth_failed(), std::max(0, loopDepth - 1));
															#line 4932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			return a_body1Catch1(connection_failed(), std::max(0, loopDepth - 1));
															#line 4936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			TraceEvent("ProxyRequestFailed").errorUnsuppressed(e);
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			{
															#line 797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 4952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			}
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			err = e;
															#line 4956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r = _r;
															#line 4971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		r = _r;
															#line 4980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_r, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state > 0) static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendProxyConnectRequestActor*>(this)->ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state > 0) static_cast<SendProxyConnectRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendProxyConnectRequestActor*>(this)->ActorCallback< SendProxyConnectRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SendProxyConnectRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SendProxyConnectRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SendProxyConnectRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> conn;
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string remoteHost;
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string remoteService;
															#line 767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int requestTimeout;
															#line 768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int maxTries;
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int thisTry;
															#line 770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	double nextRetryDelay;
															#line 771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IRateControl> sendReceiveRate;
															#line 772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	int64_t bytes_sent;
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	UnsentPacketQueue empty_packet_queue;
															#line 775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<HTTP::OutgoingRequest> req;
															#line 785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Optional<Error> err;
															#line 787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 5192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via sendProxyConnectRequest()
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class SendProxyConnectRequestActor final : public Actor<Void>, public ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >, public ActorCallback< SendProxyConnectRequestActor, 1, Void >, public FastAllocated<SendProxyConnectRequestActor>, public SendProxyConnectRequestActorState<SendProxyConnectRequestActor> {
															#line 5197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<SendProxyConnectRequestActor>::operator new;
	using FastAllocated<SendProxyConnectRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6561619212260902400UL, 5642170005425207296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< SendProxyConnectRequestActor, 1, Void >;
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	SendProxyConnectRequestActor(Reference<IConnection> const& conn,std::string const& remoteHost,std::string const& remoteService) 
															#line 5215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Void>(),
		   SendProxyConnectRequestActorState<SendProxyConnectRequestActor>(conn, remoteHost, remoteService),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1064934949979343616UL, 12075563473874751488UL);
		ActorExecutionContextHelper __helper(static_cast<SendProxyConnectRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("sendProxyConnectRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("sendProxyConnectRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SendProxyConnectRequestActor, 0, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SendProxyConnectRequestActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Void> sendProxyConnectRequest( Reference<IConnection> const& conn, std::string const& remoteHost, std::string const& remoteService ) {
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Void>(new SendProxyConnectRequestActor(conn, remoteHost, remoteService));
															#line 5248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

															#line 5253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
// This generated class is to be used only via proxyConnectImpl()
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
template <class ProxyConnectImplActor>
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class ProxyConnectImplActorState {
															#line 5259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	ProxyConnectImplActorState(std::string const& remoteHost,std::string const& remoteService,std::string const& proxyHost,std::string const& proxyService) 
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		 : remoteHost(remoteHost),
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   remoteService(remoteService),
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   proxyHost(proxyHost),
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		   proxyService(proxyService)
															#line 5272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
	{
		fdb_probe_actor_create("proxyConnectImpl", reinterpret_cast<unsigned long>(this));

	}
	~ProxyConnectImplActorState() 
	{
		fdb_probe_actor_destroy("proxyConnectImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			StrictFuture<NetworkAddress> __when_expr_0 = map(INetworkConnections::net()->resolveTCPEndpoint(remoteHost, remoteService), [=](std::vector<NetworkAddress> const& addresses) -> NetworkAddress { NetworkAddress addr = addresses[deterministicRandom()->randomInt(0, addresses.size())]; addr.fromHostname = true; addr.flags = NetworkAddress::FLAG_TLS; return addr; });
															#line 877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 1;
															#line 877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >*>(static_cast<ProxyConnectImplActor*>(this)));
															#line 5294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProxyConnectImplActorState();
		static_cast<ProxyConnectImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Reference<IConnection>> __when_expr_1 = INetworkConnections::net()->connect(proxyHost, proxyService);
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 2;
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >*>(static_cast<ProxyConnectImplActor*>(this)));
															#line 5324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(NetworkAddress const& __remoteEndpoint,int loopDepth) 
	{
															#line 877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		remoteEndpoint = __remoteEndpoint;
															#line 5333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(NetworkAddress && __remoteEndpoint,int loopDepth) 
	{
		remoteEndpoint = std::move(__remoteEndpoint);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state > 0) static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 0;
		static_cast<ProxyConnectImplActor*>(this)->ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >::remove();

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >*,NetworkAddress const& value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >*,NetworkAddress && value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >*,Error err) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Void> __when_expr_2 = sendProxyConnectRequest(connection, remoteHost, remoteService);
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 3;
															#line 886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ProxyConnectImplActor, 2, Void >*>(static_cast<ProxyConnectImplActor*>(this)));
															#line 5419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<IConnection> const& __connection,int loopDepth) 
	{
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		connection = __connection;
															#line 5428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<IConnection> && __connection,int loopDepth) 
	{
		connection = std::move(__connection);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state > 0) static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 0;
		static_cast<ProxyConnectImplActor*>(this)->ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		boost::asio::ip::tcp::socket socket = std::move(connection->getSocket());
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Reference<IConnection>> __when_expr_3 = INetworkConnections::net()->connect(remoteEndpoint, &socket);
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 4;
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*>(static_cast<ProxyConnectImplActor*>(this)));
															#line 5516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		boost::asio::ip::tcp::socket socket = std::move(connection->getSocket());
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		StrictFuture<Reference<IConnection>> __when_expr_3 = INetworkConnections::net()->connect(remoteEndpoint, &socket);
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5529 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 4;
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*>(static_cast<ProxyConnectImplActor*>(this)));
															#line 5534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state > 0) static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 0;
		static_cast<ProxyConnectImplActor*>(this)->ActorCallback< ProxyConnectImplActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ProxyConnectImplActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Reference<IConnection> const& remoteConnection,int loopDepth) 
	{
															#line 889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<ProxyConnectImplActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(remoteConnection); this->~ProxyConnectImplActorState(); static_cast<ProxyConnectImplActor*>(this)->destroy(); return 0; }
															#line 5618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<ProxyConnectImplActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(remoteConnection);
		this->~ProxyConnectImplActorState();
		static_cast<ProxyConnectImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Reference<IConnection> && remoteConnection,int loopDepth) 
	{
															#line 889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
		if (!static_cast<ProxyConnectImplActor*>(this)->SAV<Reference<IConnection>>::futures) { (void)(remoteConnection); this->~ProxyConnectImplActorState(); static_cast<ProxyConnectImplActor*>(this)->destroy(); return 0; }
															#line 5630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		new (&static_cast<ProxyConnectImplActor*>(this)->SAV< Reference<IConnection> >::value()) Reference<IConnection>(remoteConnection);
		this->~ProxyConnectImplActorState();
		static_cast<ProxyConnectImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(Reference<IConnection> const& remoteConnection,int loopDepth) 
	{
		loopDepth = a_body1cont4(remoteConnection, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Reference<IConnection> && remoteConnection,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(remoteConnection), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ProxyConnectImplActor*>(this)->actor_wait_state > 0) static_cast<ProxyConnectImplActor*>(this)->actor_wait_state = 0;
		static_cast<ProxyConnectImplActor*>(this)->ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string remoteHost;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string remoteService;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string proxyHost;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	std::string proxyService;
															#line 877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	NetworkAddress remoteEndpoint;
															#line 885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	Reference<IConnection> connection;
															#line 5725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
};
// This generated class is to be used only via proxyConnectImpl()
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
class ProxyConnectImplActor final : public Actor<Reference<IConnection>>, public ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >, public ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >, public ActorCallback< ProxyConnectImplActor, 2, Void >, public ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >, public FastAllocated<ProxyConnectImplActor>, public ProxyConnectImplActorState<ProxyConnectImplActor> {
															#line 5730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
public:
	using FastAllocated<ProxyConnectImplActor>::operator new;
	using FastAllocated<ProxyConnectImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6896580679843920128UL, 13026731930350272256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<IConnection>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >;
friend struct ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >;
friend struct ActorCallback< ProxyConnectImplActor, 2, Void >;
friend struct ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >;
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	ProxyConnectImplActor(std::string const& remoteHost,std::string const& remoteService,std::string const& proxyHost,std::string const& proxyService) 
															#line 5750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
		 : Actor<Reference<IConnection>>(),
		   ProxyConnectImplActorState<ProxyConnectImplActor>(remoteHost, remoteService, proxyHost, proxyService),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("proxyConnectImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14471179542079253760UL, 2015523430403604224UL);
		ActorExecutionContextHelper __helper(static_cast<ProxyConnectImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("proxyConnectImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("proxyConnectImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProxyConnectImplActor, 0, NetworkAddress >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ProxyConnectImplActor, 1, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ProxyConnectImplActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ProxyConnectImplActor, 3, Reference<IConnection> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
[[nodiscard]] Future<Reference<IConnection>> proxyConnectImpl( std::string const& remoteHost, std::string const& remoteService, std::string const& proxyHost, std::string const& proxyService ) {
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"
	return Future<Reference<IConnection>>(new ProxyConnectImplActor(remoteHost, remoteService, proxyHost, proxyService));
															#line 5785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbrpc/HTTP.actor.g.cpp"
}

#line 891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbrpc/HTTP.actor.cpp"

Future<Reference<IConnection>> proxyConnect(const std::string& remoteHost,
                                            const std::string& remoteService,
                                            const std::string& proxyHost,
                                            const std::string& proxyService) {
	return proxyConnectImpl(remoteHost, remoteService, proxyHost, proxyService);
}

} // namespace HTTP
